C
C________________________________________________________________
C
C      FUNCTION  F _ R P E R P
C________________________________________________________________
C
C FUNCTION F_RPERP CALCULATES THE PERPENDICULAR-POLARIZATION
C COMPONENT OF REFLECTIVITY (SQUARED AMPLITUDE OF FRESNEL
C REFLECTION COEFFICIENT) AS A FUNCTION OF INCIDENCE ANGLE
C (THETA) AND THE COMPLEX INDICIES OF REFRACTION (N1, N2),
C ACCORDING TO EQUATION (4.44) ON PAGE 60 IN "THEORY OF
C REFLECTANCE AND EMITTANCE SPECTROSCOPY" 2ND EDITION, BY Bruce
C Hapke, 2012.  A COMPANION EQUATION, (4.45), PROVIDES THE PHASE
C OF THE REFLECTED RAY, AND IS OPTIONALLY COMPUTED BY THIS
C FUNCTION ROUTINE.  (ALL REFERENCES TO EQUATIONS IN THE NOTES
C AND DESCRIPTIONS OF THIS FUNCTION ROUTINE ARE ALSO TO Hapke,
C 2012 UNLESS OTHERWISE NOTED).
C
C
C NOTES:
C
C * THETA IS THE ANGLE BETWEEN THE INCIDENT RAY AND THE NORMAL TO
C THE REFLECTING SURFACE.
C
C * APPARENTLY IN THE NOTATIONAL ARENA OF REFLECTANCE
C SPECTROSCOPY, PRIME- AND DOUBLE-PRIME MARKS CAN REFER TO A
C NUMBER OF DIFFERING THINGS, DEPENDING UPON THE CONTEXT.  FOR
C EXAMPLE, THE PERPENDICULAR-POLARIZATION COMPONENT IS DESIGNATED
C BY A SINGLE PRIME; THE PARALLEL A DOUBLE PRIME.  WHEREAS, THETA
C IS THE ANGLE OF INCIDENCE, THETA-PRIME IS THE ANGLE OF
C TRANSMITTANCE, AND THETA-DOUBLE-PRIME IS THE ANGLE OF
C REFLECTANCE.
C
C * IF THE INDICIES OF REFRACTION ARE BOTH REAL, THE COMPLEX
C REFLECTIVITY COMPUTED BY (4.44) REDUCES TO A REAL-ONLY
C REFLECTIVITY SHOWN BY (4.37), WHICH IS IDENTICAL TO THE
C EXPRESSION FOR f' GIVEN IN Kerker's 1969 Se INTEGRAL EQUATION
C FOR COLLIMATED LIGHT INCIDENT ON THE SURFACE OF A LARGE SPHERE.
C THIS EQUATION APPEARS AS (21) ON PAGE 3046 OF Hapke's PAPER,
C "BIDIRECTIONAL REFLECTANCE SPECTROSCOPY", in JOURNAL OF
C GEOPHYSICAL RESEARCH, VOL. 86, NO. B4, PAGES 3039-3054, APRIL
C 10, 1981.
C
C * WHEN THE INDICIES OF REFRACTION CONTAIN NON-ZERO IMAGINARY
C COMPONENTS AND THE INCIDENT RAY IS UNPOLARIZED, THE PHASE OF
C THE PERPENDICULAR POLARIZATION (THIS FUNCTION) MAY DIFFER FROM
C THE PHASE OF THE PARALLEL-POLARIZATION (FUNCTION F_RPARA)
C RESULTING IN THE REFLECTED LIGHT BEING PARTIALLY CIRCULARLY
C POLARIZED.  THIS DOES NOT HAPPEN WHEN THE INDICIES OF
C REFRACTION ARE BOTH REAL-ONLY.
C
C * WHEN THE INDICIES OF REFRACTION ARE COMPLEX, CONTAINING BOTH
C REAL AND IMAGINARY PARTS, THE REFLECTIVITY (WHETHER
C PERPENDICULAR OR PARALLEL), IS NOT COMPLETELY ANALOGOUS TO THE
C REAL-ONLY CASE, BECAUSE IT IS ONLY THE REAL COMPONENT OF THE
C REFLECTIVITY THAT DRIVES SNELL'S LAW IN DETERMINING THE ANGLE
C OF TRANSMITTANCE (SEE Happke's EQUATION (4.43)).  PERHAPS THIS
C DRIVES OTHER THINGS, AS WELL, IN THE GENERALLIZED RADIATIVE
C TRANSFER EQUATION.  FOR INSTANCE, IS IT THE WHOLE VALUE OF
C REFLECTIVITY THAT IS USED IN THE CALCULATION OF Se?  OR IS IT
C SIMPLY THE SQUARED REAL COMPONENT OF THE REFLECTION COEFFICIENT
C AMPLITUDE?
C
C * THE NOTATION USED BY THIS FUNCTION FOR INDEX OF REFRACTION:
C
C     n1 = n1r + i(n1i)  (MEDIUM OF INCIDENT LIGHT)
C     n2 = n2r + i(n2i)  (MEDIUM OF TRANSMITTED LIGHT)
C
C HOWEVER, AN ALTERNATIVE NOTATION NOT USED HERE:
C
C     m = n(1-jk) WHERE n = REAL PART, AND nk = IMAGINARY PART
C
C
C FUNCTION:
C   F_RPERP - REAL*8.  PERPENDICULAR-POLARIZATION COMPONENT OF
C             REFLECTIVITY (SQUARED AMPLITUDE OF FRESNEL
C             REFLECTION COEFFICIENT).
C
C INPUT ARGUMENTS:
C   THETA  - REAL*8.  ANGLE <RADIANS> BETWEEN THE INCIDENT RAY
C            AND THE NORMAL TO THE REFLECTING SURFACE.
C   LNOP   - INTEGER*4.  IN THE INTEREST OF SAVING CPU, LNOP
C            DIRECTS WHETHER TO CALCULATE TAN(PHI):
C                 LNOP =  0 - CALCULATE TANPHI.
C                 LNOP <> 0 - DO NOT CALCULATE TANPHI.
C   N1R    - REAL*8.  THE REFRACTIVE INDEX IN MEDIUM 1
C            (INCIDENT SIDE), REAL PART.
C   N1I    - REAL*8.  THE REFRACTIVE INDEX IN MEDIUM 1
C            (INCIDENT SIDE), IMAGINARY PART.
C   N2R    - REAL*8.  THE REFRACTIVE INDEX IN MEDIUM 2
C            (TRANSMITTING SIDE), REAL PART.
C   N2I    - REAL*8.  THE REFRACTIVE INDEX IN MEDIUM 2
C            (TRANSMITTING SIDE), IMAGINARY PART.
C
C INPUT/OUTPUT ARGUMENTS:
C   LNON   - INTEGER*4.  IN THE INTEREST OF SAVING CPU, LNON
C            DIRECTS WHETHER TO CALCULATE NR AND NI:
C
C               LNON =  0 - CALCULATE NR AND NI (MAKING THEM
C                  OUTPUT ARGUMENTS) USING N1R, N1I, N2R, AND N2I
C                  ACCORDING TO EQNS (4.5) (4.6).  AFTER
C                  CALCULATING NR AND NI, UPON RETURN FROM THIS
C                  FUNCTION, LNON WILL BE SET TO 1, SO THAT THE
C                  CALCULATED VALUES OF NR AND NI WILL IN EFFECT
C                  AUTOMATICALLY BE USED ON THE SUBSEQUENT
C                  FUNCTION CALLS.
C               LNON <> 0 - DO NOT CALCULATE NR AND NI.  INSTEAD,
C                  USE THE VALUES OF NR AND NI PASSED TO THIS
C                  FUNCTION.  IN THIS CASE, THE VALUES OF N1R,
C                  N1I, N2R, AND N2I WILL BE IGNORED.
C
C   NR     - REAL*8.  GIVEN BY (4.5), THE REAL PART OF THE
C            RELATIVE REFRACTIVE INDEX.  NR WILL BE AN OUTPUT
C            ARGUMENT IF LNON=0.  IT WILL BE AN INPUT ARGUMENT IF
C            LNON<>0.
C   NI     - REAL*8.  GIVEN BY (4.6), THE IMAGINARY PART OF THE
C            RELATIVE REFRACTIVE INDEX.  NI WILL BE AN OUTPUT
C            ARGUMENT IF LNON=0.  IT WILL BE AN INPUT ARGUMENT IF
C            LNON<>0.
C
C OUTPUT ARGUMENTS:
C   G1     - REAL*8. FIRST PRECALCULATED PORTION OF (4.44), THE
C            SQUARED VALUE OF WHICH IS GIVEN IN (4.48).  THIS
C            VALUE IS RECALCULATED WITH EACH CALL TO THIS
C            FUNCTION, AND IS THEREFORE AN OUTPUT ARGUMENT. 
C            FUNCTION F_RPARA REQUIRES G1 AS AN INPUT ARGUMENT,
C            AND THEREFORE THIS FUNCTION MUST BE CALLED FIRST.
C   G2     - REAL*8. SECOND PRECALCULATED PORTION OF (4.44), THE
C            SQUARED VALUE OF WHICH IS GIVEN IN (4.49).  G2
C            OPERATES THE SAME WAY AS G1.
C   SINT   - REAL*8. SINE(THETA), PROVIDED AS AN OUTPUT ARGUMENT
C            FOR SUBSEQUENT INTEGRAL CALCULATIONS.
C   COST   - REAL*8. COSINE(THETA), PROVIDED AS AN OUTPUT FOR
C            SUBSEQUENT INTEGRAL CALCULATIONS AND FOR SUBSEQUENT
C            INPUT TO F_RPARA.
C
C   TANPHI - REAL*8.  TANGENT OF PHASE OF FRESNEL REFLECTION
C            COEFFICIENT).
C
C
C
C FUNCTION F_RPERP WRITTEN BY ROB BRACKEN, USGS.
C DIGITAL Visual Fortran, F90 version 5.0, DEC copyright 1997.
C VERSION 1.0, 20130621 ( ORIGINAL CODE
C                                USING EQUATIONS AS CITED ABOVE).
C
C

      function f_Rperp(theta,lnop ,n1r,n1i,n2r,n2i    ,lnon,nr,ni
     &                                  ,g1,g2,sint,cost ,tanphi)
C
C     FUNCTION
      real*8 f_Rperp
C
C     INPUT ARGUMENTS
      real*8 theta
      integer*4 lnop
      real*8 n1r,n1i,n2r,n2i
C
C     INPUT/OUTPUT ARGUMENTS
      integer*4 lnon
      real*8 nr,ni
C
C     OUTPUT ARGUMENTS
      real*8 g1,g2,sint,cost
      real*8 tanphi
C
C     LOCAL DECLARATIONS
      real*8 x,rxxyy,r
C
C     PI CONVERSION FACTORS
      real*8 PIE,D2R,R2D
      parameter( PIE = 3.1415 92653 58979 32384 62643d0 )
      parameter( D2R = PIE/180.d0 , R2D = 180.d0/PIE)
C
C
C
C FIND NR & NI
C
      if(lnon.eq.0) then
        nr=(n1r*n2r+n1i*n2i)/(n1r*n1r+n1i*n1i)
        ni=(n1r*n2i-n2r*n1i)/(n1r*n1r+n1i*n1i)
        lnon=1
      endif
C
C
C CHECK VALUES OF NR AND NI
C
      if(nr.eq.1.d0.and.ni.eq.0.d0) then
C
C       DEGENERATIVE CASE WITH NO REFLECTION
        r=0.d0
        if(theta.eq.pie/2) r=1.d0
        tanphi=0.d0
        goto 990
      endif
C
C
C FIND G1, G2, AND COS(THETA)
C
      sint=dsin(theta)
      x=nr*nr-ni*ni-sint*sint
      rxxyy=dsqrt(x*x+4*nr*nr*ni*ni)
C
      g1=dsqrt((rxxyy+x)*0.5d0)
      g2=dsqrt((rxxyy-x)*0.5d0)
      cost=dcos(theta)
C
C
C FIND PERPENDICULAR-POLARIZATION COMPONENT OF REFLECTIVITY
C
      r=((cost-g1)*(cost-g1)+g2*g2)/
     &  ((cost+g1)*(cost+g1)+g2*g2)
C
C
C FIND TAN(PHI), WHERE PHI IS PHASE ANGLE OF THE REFLECTED RAY
C
C NOTE:  EQN (4.45) HAS A COS(SCRIPTED THETA) IN THE DENOMINATOR,
C RATHER THAN THE COS(THETA) THAT APPEARS EVERYWHERE ELSE.  SINCE
C THE SCRIPTED THETA DOES NOT APPEAR ANYWHERE IN THE PRECEDING
C PAGES OF THE BOOK, I ASSUME IT TO BE A MISPRINT, AND THE
C INTENDED MEANING TO BE COS(THETA).
C
      if(lnop.eq.0) then
        tanphi=2*g2*cost/(g1*g1+g2*g2-cost*cost)
      endif
C
C
C
C EXIT PROCEDURE
C
  990 f_rperp=r
      return
      end
