C
C________________________________________________________________
C
C      FUNCTION  F _ R P A R A
C________________________________________________________________
C
C FUNCTION F_RPARA CALCULATES THE PARALLEL-POLARIZATION COMPONENT
C OF REFLECTIVITY (SQUARED AMPLITUDE OF FRESNEL REFLECTION
C COEFFICIENT) AS A FUNCTION OF INCIDENCE ANGLE (THETA) AND THE
C COMPLEX INDICIES OF REFRACTION (N1, N2), ACCORDING TO EQUATION
C (4.46) ON PAGE 60 IN "THEORY OF REFLECTANCE AND EMITTANCE
C SPECTROSCOPY" 2ND EDITION, BY Bruce Hapke, 2012.  A COMPANION
C EQUATION, (4.47), PROVIDES THE PHASE OF THE REFLECTED RAY, AND
C IS OPTIONALLY COMPUTED BY THIS FUNCTION ROUTINE.  (ALL
C REFERENCES TO EQUATIONS IN THE NOTES AND DESCRIPTIONS OF THIS
C FUNCTION ROUTINE ARE ALSO TO Hapke, 2012 UNLESS OTHERWISE
C NOTED).
C
C
C NOTES:
C
C * THETA IS THE ANGLE BETWEEN THE INCIDENT RAY AND THE NORMAL TO
C THE REFLECTING SURFACE.
C
C * APPARENTLY IN THE NOTATIONAL ARENA OF REFLECTANCE
C SPECTROSCOPY, PRIME- AND DOUBLE-PRIME MARKS CAN REFER TO A
C NUMBER OF DIFFERING THINGS, DEPENDING UPON THE CONTEXT.  FOR
C EXAMPLE, THE PERPENDICULAR-POLARIZATION COMPONENT IS DESIGNATED
C BY A SINGLE PRIME; THE PARALLEL A DOUBLE PRIME.  WHEREAS, THETA
C IS THE ANGLE OF INCIDENCE, THETA-PRIME IS THE ANGLE OF
C TRANSMITTANCE, AND THETA-DOUBLE-PRIME IS THE ANGLE OF
C REFLECTANCE.
C
C * IF THE INDICIES OF REFRACTION ARE BOTH REAL, THE COMPLEX
C REFLECTIVITY COMPUTED BY (4.46) REDUCES TO A REAL-ONLY
C REFLECTIVITY SHOWN BY (4.38), WHICH IS IDENTICAL TO THE
C EXPRESSION FOR f" GIVEN IN Kerker's 1969 Se INTEGRAL EQUATION
C FOR COLLIMATED LIGHT INCIDENT ON THE SURFACE OF A LARGE SPHERE.
C THIS EQUATION APPEARS AS (21) ON PAGE 3046 OF Hapke's PAPER,
C "BIDIRECTIONAL REFLECTANCE SPECTROSCOPY", in JOURNAL OF
C GEOPHYSICAL RESEARCH, VOL. 86, NO. B4, PAGES 3039-3054, APRIL
C 10, 1981.
C
C * WHEN THE INDICIES OF REFRACTION CONTAIN NON-ZERO IMAGINARY
C COMPONENTS AND THE INCIDENT RAY IS UNPOLARIZED, THE PHASE OF
C THE PERPENDICULAR POLARIZATION (THIS FUNCTION) MAY DIFFER FROM
C THE PHASE OF THE PARALLEL-POLARIZATION (FUNCTION F_RPARA)
C RESULTING IN THE REFLECTED LIGHT BEING PARTIALLY CIRCULARLY
C POLARIZED.  THIS DOES NOT HAPPEN WHEN THE INDICIES OF
C REFRACTION ARE BOTH REAL-ONLY.
C
C * WHEN THE INDICIES OF REFRACTION ARE COMPLEX, CONTAINING BOTH
C REAL AND IMAGINARY PARTS, THE REFLECTIVITY (WHETHER
C PERPENDICULAR OR PARALLEL), IS NOT COMPLETELY ANALOGOUS TO THE
C REAL-ONLY CASE, BECAUSE IT IS ONLY THE REAL COMPONENT OF THE
C REFLECTIVITY THAT DRIVES SNELL'S LAW IN DETERMINING THE ANGLE
C OF TRANSMITTANCE (SEE Happke's EQUATION (4.43)).  PERHAPS THIS
C DRIVES OTHER THINGS, AS WELL, IN THE GENERALLIZED RADIATIVE
C TRANSFER EQUATION.  FOR INSTANCE, IS IT THE WHOLE VALUE OF
C REFLECTIVITY THAT IS USED IN THE CALCULATION OF Se?  OR IS IT
C SIMPLY THE SQUARED REAL COMPONENT OF THE REFLECTION COEFFICIENT
C AMPLITUDE?
C
C * THE NOTATION USED BY THIS FUNCTION FOR INDEX OF REFRACTION:
C
C     n1 = n1r + i(n1i)  (MEDIUM OF INCIDENT LIGHT)
C     n2 = n2r + i(n2i)  (MEDIUM OF TRANSMITTED LIGHT)
C
C HOWEVER, AN ALTERNATIVE NOTATION NOT USED HERE:
C
C     m = n(1-jk) WHERE n = REAL PART, AND nk = IMAGINARY PART
C
C
C
C FUNCTION:
C   F_RPARA - REAL*8.  PARALLEL-POLARIZATION COMPONENT OF
C             REFLECTIVITY (SQUARED AMPLITUDE OF FRESNEL
C             REFLECTION COEFFICIENT).
C
C INPUT ARGUMENTS:
C   THETA  - REAL*8.  ANGLE <RADIANS> BETWEEN THE INCIDENT RAY
C            AND THE NORMAL TO THE REFLECTING SURFACE.
C   LNOP   - INTEGER*4.  IN THE INTEREST OF SAVING CPU, LNOP
C            DIRECTS WHETHER TO CALCULATE TAN(PHI):
C                 LNOP =  0 - CALCULATE TANPHI.
C                 LNOP <> 0 - DO NOT CALCULATE TANPHI.
C   NR     - REAL*8.  GIVEN BY (4.5), THE REAL PART OF THE
C            RELATIVE REFRACTIVE INDEX.  THE VALUE OF NR IS
C            CALCULATED BY F_RPERP.
C   NI     - REAL*8.  GIVEN BY (4.6), THE IMAGINARY PART OF THE
C            RELATIVE REFRACTIVE INDEX.  THE VALUE OF NI IS
C            CALCULATED BY F_RPERP.
C   G1     - REAL*8. FIRST PRECALCULATED PORTION OF (4.46), THE
C            SQUARED VALUE OF WHICH IS GIVEN IN (4.48).  THIS
C            VALUE MUST BE RECALCULATED EACH TIME THETA, NR, OR
C            NI IS CHANGED.  FUNCTION F_RPERP PERFORMS THIS
C            CALCULATION AND SUPPLIES G1 AS AN OUTPUT ARGUMENT. 
C            THEREFORE F_RPERP MUST BE CALLED IMMEDIATELY PRIOR
C            TO CALLING THIS FUNCTION.
C   G2     - REAL*8. SECOND PRECALCULATED PORTION OF (4.46), THE
C            SQUARED VALUE OF WHICH IS GIVEN IN (4.49).  G2
C            OPERATES THE SAME WAY AS G1.
C   SINT   - REAL*8. SINE(THETA) IS PROVIDED AS AN OUTPUT
C            ARGUMENT FROM F_RPERP.
C   COST   - REAL*8. COSINE(THETA) IS PROVIDED AS AN OUTPUT
C            ARGUMENT FROM F_RPERP.
C
C OUTPUT ARGUMENT:
C   TANPHI - REAL*8.  TANGENT OF PHASE OF FRESNEL REFLECTION
C            COEFFICIENT).
C
C
C
C FUNCTION F_RPARA WRITTEN BY ROB BRACKEN, USGS.
C DIGITAL Visual Fortran, F90 version 5.0, DEC copyright 1997.
C VERSION 1.0, 20130621 ( ORIGINAL CODE
C                                USING EQUATIONS AS CITED ABOVE).
C
C

      function f_Rpara(theta,lnop  ,nr,ni  ,g1,g2,sint,cost
     &                                                   ,tanphi)
C
C     FUNCTION
      real*8 f_Rpara
C
C     INPUT ARGUMENTS
      real*8 theta
      integer*4 lnop
      real*8 nr,ni
      real*8 g1,g2,sint,cost
C
C     OUTPUT ARGUMENT
      real*8 tanphi
C
C     LOCAL DECLARATIONS
      real*8 r
C
C     PI CONVERSION FACTORS
      real*8 PIE,D2R,R2D
      parameter( PIE = 3.1415 92653 58979 32384 62643d0 )
      parameter( D2R = PIE/180.d0 , R2D = 180.d0/PIE)
C
C
C
C CHECK VALUES OF NR AND NI
C
      if(nr.eq.1.d0.and.ni.eq.0.d0) then
C
C       DEGENERATIVE CASE WITH NO REFLECTION
        r=0.d0
        if(theta.eq.pie/2) r=1.d0
        tanphi=0.d0
        goto 990
      endif
C
C
C FIND PARALLEL-POLARIZATION COMPONENT OF REFLECTIVITY
C
C NOTE:  EQN (4.46) CONTAINS AN OBVIOUS TYPO WHERE THE LAST G2 IN
C THE DENOMINATOR IS SHOWN AS G*G.  THIS WOULD BE INCONSISTENT
C WITH THE FORM AND BALANCE OF THE REST OF THE EQUATIONS, AND
C THEREFORE IT APPEARS TO BE A TYPO.
C
      r=(((nr*nr-ni*ni)*cost-g1)**2+(2*nr*ni*cost-g2)**2)/
     &  (((nr*nr-ni*ni)*cost+g1)**2+(2*nr*ni*cost+g2)**2)
C
C
C FIND TAN(PHI), WHERE PHI IS PHASE ANGLE OF THE REFLECTED RAY
C ACCORDING TO EQN (4.47)
C
C NOTE:  THE EXPONENT IN THE DENOMINATOR OF EQN (4.47) LOOKS
C CONSPICUOUS.  IN VIEW OF THE FACT THAT THE EQUATIONS IN THIS
C BOOK APPEAR TO CONTAIN MANY TYPO'S, THIS TERM MAY NOT ACTUALLY
C BE SQUARED.
C
      if(lnop.eq.0) then
        tanphi=2*cost*(2*nr*ni*g1-(nr*nr-ni*ni)*g2)/
     &        ((nr*nr+ni*ni)**2*cost*cost-(g1*g1+g2*g2))
      endif
C
C
C
C EXIT PROCEDURE
C
  990 f_rpara=r
      return
      end
