#include <iostream>
#include "global.h"
#include "random.h"
#include "problem.h"
#include "mutation.h"

using namespace std;

static bool weakDom(IntVec const &v, IntVec const &w) { // true if v weakly dominates w
	size_t d = v.size();
	for (size_t i = 0; i < d; i++) {
		if (v[i] > w[i]) return(false);
	}
	return(true);
}
static bool dom(IntVec const &v, IntVec const &w) { // true if v dominates w
	size_t d = v.size();
	size_t cnt = 0;
	for (size_t i = 0; i < d; i++) {
		if (v[i] > w[i]) return(false);
		if (v[i] < w[i]) cnt++;
	}
	return(cnt > 0);
}
static int32_t distance(Pop_t const &pop, int32_t delta) { // distance to Pareto front
	int32_t dist = INT32_MAX;
	for (size_t i = 0; i < pop.size(); i++) {
		IntVec f = pop[i].second;
		int32_t d = f[0] + f[1] - delta;
		if (d < dist) dist = d;
	}
	return dist;
}
static bool has_found_PF(Pop_t const &pop, int32_t delta, int &noOfOptInd) {
	IntSet indexSet;
	for (size_t i = 0; i < pop.size(); i++) {
		IntVec f = pop[i].second;
		int32_t dist = f[0] + f[1] - delta;
		if (dist > 0) continue;
		indexSet.insert(f[0]);
	}
	noOfOptInd = static_cast<int32_t>(indexSet.size());
	return(noOfOptInd == delta + 1);
}

static void copyPop(Pop_t& pnew, Pop_t& pold) {
	for (size_t i = 0; i < pold.size(); i++)
		pnew.push_back(pold[i]);
}
static void copyPop(Pop_t& pnew, Pop_t& pold, IntSet &ignore) {
	if (ignore.size() == 0) copyPop(pnew, pold);
	else {
		auto it = ignore.begin();
		for (size_t i = 0; i < pold.size(); i++) {		
			if (it != ignore.end() && i == *it) {
				it++;
				continue;
			}
			pnew.push_back(pold[i]);
		}
	}
}


int main(int argc, char** argv)
{

	// caution: the two flags below can be overwritten by optional command line arguments

	bool redo = false; // if true, mutation is repeated until first nonzero mutation vector
	                   // if false, offspring with zero mutation vectors are not evaluated (but counted) ### TODO

	bool nina = false; // if true, step size adaptation counts all nondominated offspring (=nina)
	                   // if false, step size adaptation counts only dominating offspring (=ni)

	int32_t d = 2; // number of objectives (fixed at the moment)

	size_t maxFE = 1000000;	// maximum number of function evaluations
	size_t cntFE = 0;

	//environment parameters:
	//	pid   [int]   problem id
	//	xid   [int]   starting point id
	//	s0    [float] initial step size > 0
	//	S | F [char]  subdim or fulldim mutation (p = 1/n or p = 1)
	//	w     [int]   window size factor
	//	ps    [float] relative success frequency
	//	c-    [float] reduction factor; in (0, 1)
	//	c+    [float] enlargement factor; > 0
	//      redo  [int]   0: false / 1: true    default: false
	//      nina  [int]   0: false / 1: true    default: false


	if (argc < 9 || argc == 10 || argc > 11) {
		cerr << "usage: " << argv[0] << " pid xid s0 (S|F) w ps c- c+ [redo nina]" << endl;
		exit(1);
	}
	
	int pid = atoi(argv[1]);        // problem id
	int xid = atoi(argv[2]);        // id of starting point
	double s = atof(argv[3]);       // initial mean step size

	enum class MutType { SubDim, FullDim };
	MutType mutType;
	string mutspec = argv[4];
	if (mutspec.size() == 1) {
		if (mutspec[0] == 'S') mutType = MutType::SubDim;
		else if (mutspec[0] == 'F') mutType = MutType::FullDim;
		else {
			cerr << argv[0] << ": invalid mutation specification" << endl; exit(1);
		}
	}
	else {
		cerr << argv[0] << ": invalid mutation specification" << endl; exit(1);
	}
	
	int w = atoi(argv[5]);	    // window size factor
	double ps = atof(argv[6]);  // relative success frequency
	double cm = atof(argv[7]);  // c-: reduction factor
	double cp = atof(argv[8]);  // c+: enlargement factor

	if (w < 1 || ps < 0 || ps > 1 || cm <= 0 || cm > 1 || cp < 1) {
		cerr << argv[0] << ": check your hyperparams!" << endl;
		exit(1);
	}
	if (argc == 11) { // read optional params 'redo' and 'nina'
		string str = argv[9];
		redo = (str == "true" || str == "TRUE" || str == "T" || str == "1");
		str = argv[10];
		nina = (str == "true" || str == "TRUE" || str == "T" || str == "1");
	}


	// define problem
    BOP bop(pid);
	int32_t n = bop.getDim();
	int32_t delta = bop.getSizePF()-1;
	w *= n; // absolute window size

	Mutation *mut = nullptr;
	if (mutType == MutType::SubDim) mut = new MutationLaplaceSubspace();
	else mut = new MutationLaplaceFullspace();

	double ss = (mutType == MutType::SubDim) ? s : s / n;
	double q = 1 - ss / (1 + sqrt(1 + ss * ss));

	// set starting point
	IntVec x = bop.get_x(xid);

	// initial individual			
	IntVec fx = { bop.f1(x), bop.f2(x) };
	cntFE++;
	Ind_t ind = { x, fx };
	Pop_t pop[2];
	size_t ff = 0; // flipflop
	pop[ff] = { ind };
	pop[1-ff].clear();

	int32_t na = 0; // number of accepted offspring
	int32_t ni = 0; // number of improving offspring

	Irnd irnd;	// RNG for selecting parent

	// 1st reporting
	int noOfOptInd = 0;
	bool found_PF = has_found_PF(pop[ff], delta, noOfOptInd);
	int32_t dist = distance(pop[ff], delta);
	cout << cntFE << " " << dist << " " << s << " " << pop[ff].size() << " " << noOfOptInd << endl;

	int32_t cntRedo = 0;
	while (cntFE <= maxFE) {
		int32_t mu = static_cast<int32_t>(pop[ff].size());
		size_t k = irnd.rnd(0, mu - 1);
		ind = pop[ff][k];
		IntVec y(ind.first);	// y is copy of randomly chosen individual
		if (redo) {
			do {
				mut->mutate(q, y);
				if (y == ind.first) cntRedo++; // debug
			} while (y == ind.first);
			
		}
		else {
			mut->mutate(q, y);
			if (y == ind.first) { // no change by mutation
				//cntFE++;
				//continue; // don't evaluate but increment FE
				//### caution: can affect adaptation this way!
			}
		}
		IntVec fy{ bop.f1(y), bop.f2(y) }; // evaluate y
		cntFE++;
		Ind_t kid{ y, fy };

		// selection stage 1
		IntSet E, B, W, I;
		for (int32_t i = 0; i < mu; i++) {
			Ind_t z = pop[ff][i];
			IntVec fz = z.second;
			if (fz == fy) E.insert(i);			// fz equal fy
			else if (dom(fz, fy)) B.insert(i);  // fz better than fy
			else if (dom(fy, fz)) W.insert(i);  // fy better than fz
			else I.insert(i);					// fy and fz incomparable
		}
		// selection stage 2
		if (E.size() > 0) {		// remark: E can contain at most a single element
			copyPop(pop[1 - ff], pop[ff]);
			int32_t k = *E.begin();
			if (pop[1 - ff][k].first != kid.first) na = na + 1; // only count if different x!
			pop[1 - ff][k] = kid;
			
		}
		else if (W.size() > 0) {
			copyPop(pop[1 - ff], pop[ff], W);
			pop[1 - ff].push_back(kid);
			ni = ni + 1;
		}
		else if (B.size() > 0) {
			copyPop(pop[1 - ff], pop[ff]);
		}
		else {
			copyPop(pop[1 - ff], pop[ff]);
			pop[1 - ff].push_back(kid);
			na = na + 1;
		}


		// tidy up
		ff = 1 - ff; // now pop[ff] is the current population
		pop[1 - ff].clear(); // delete old population

		// step size adaptation
		if (cntFE % w == 0) { 
			if (nina) ni += na;
			double sp = static_cast<double>(ni) / w;	
			s *= ((sp > ps) ? cp : cm);
			ni = na = 0; // reset counters
			// update q
			if (mutType == MutType::SubDim) {
				if (s < 1) s = 1;
				ss = s;
			}
			else {
				if (s / n < 1) s = n;
				ss = s / n;
			}
			q = 1 - ss / (1 + sqrt(1 + ss * ss));
		}

		// reporting
		noOfOptInd = 0;
		found_PF = has_found_PF(pop[ff], delta, noOfOptInd);
		dist = distance(pop[ff], delta);
		//cout << ni << " " << na << " : ";
		cout << cntFE << " " << dist << " " << ss << " " << mu << " " << noOfOptInd << endl;

		if (found_PF) break;
	} 

	return 0;
}
