/*
 * Copyright (C) 1999-2011 University of Connecticut Health Center
 *
 * Licensed under the MIT License (the "License").
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *  http://www.opensource.org/licenses/mit-license.php
 */

package cbit.vcell.model;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyVetoException;
import java.beans.VetoableChangeListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.vcell.util.*;
import org.vcell.util.Issue.IssueCategory;
import org.vcell.util.Issue.IssueSource;
import org.vcell.util.IssueContext.ContextType;

import cbit.vcell.matrix.RationalNumber;
import cbit.vcell.model.Model.ElectricalTopology;
import cbit.vcell.model.Model.ModelParameter;
import cbit.vcell.parser.AbstractNameScope;
import cbit.vcell.parser.Expression;
import cbit.vcell.parser.ExpressionBindingException;
import cbit.vcell.parser.ExpressionException;
import cbit.vcell.parser.NameScope;
import cbit.vcell.parser.SymbolTable;
import cbit.vcell.parser.SymbolTableEntry;
import cbit.vcell.parser.VCUnitEvaluator;
import cbit.vcell.units.VCUnitDefinition;
import cbit.vcell.units.VCUnitException;
import net.sourceforge.interval.ia_math.RealInterval;
import org.vcell.util.document.Identifiable;

/**
 * This class was generated by a SmartGuide.
 * 
 */
@SuppressWarnings("serial")
public abstract class Kinetics implements ModelProcessDynamics, Matchable, PropertyChangeListener, VetoableChangeListener, 
java.io.Serializable, IssueSource, Relatable
{
	private final static Logger logger = LogManager.getLogger(Kinetics.class);

	public static final String PROPERTY_NAME_KINETICS_PARAMETERS = "kineticsParameters";
	private String name;
	private ReactionStep reactionStep;
	protected transient java.beans.PropertyChangeSupport propertyChange;
	protected transient java.beans.VetoableChangeSupport vetoPropertyChange;
	private Kinetics.KineticsParameter[] fieldKineticsParameters = new KineticsParameter[0];
	private Kinetics.UnresolvedParameter[] fieldUnresolvedParameters = new UnresolvedParameter[0];
	private Kinetics.KineticsProxyParameter[] fieldProxyParameters = new KineticsProxyParameter[0];
	private transient boolean bReading = false;
	private transient boolean bResolvingUnits = false;
	public transient boolean bRefreshingUnits = false;
	
	// !! HACK ALERT!!
	// For older models that did have GeneralTotalKinetics: need this to bring the params into GeneralLumpedKinetics 
	public static String GTK_TotalRate_oldname = "total rate";
	public static String GTK_AssumedCompartmentSize_oldname = "assumed compartment size";
	public static String GTK_ReactionRate_oldname = "reaction rate";
	public static String GTK_CurrentDensity_oldname = "inward current density";

	public static String observedForwardRate_oldname = "observed forward rate";
	public static String observedReverseRate_oldname = "observed reverse rate";

	public static final int ROLE_NotARole		= -1;
//	public static final int ROLE_Kcat			= -2;
	
	public static final int ROLE_UserDefined	= 0;
	public static final int ROLE_ReactionRate	= 1;
	public static final int ROLE_CurrentDensity	= 2;
	public static final int ROLE_KForward		= 3;
	public static final int ROLE_KReverse		= 4;
	public static final int ROLE_Km				= 5;
	public static final int ROLE_Vmax			= 6;
	public static final int ROLE_KmFwd			= 7;
	public static final int ROLE_VmaxFwd		= 8;
	public static final int ROLE_KmRev			= 9;
	public static final int ROLE_VmaxRev		= 10;
	public static final int ROLE_Permeability	= 11;
	public static final int ROLE_Conductivity	= 12;
	public static final int ROLE_NetChargeValence	= 13;
	public static final int ROLE_CarrierChargeValence	= 14;
	public static final int ROLE_LumpedReactionRate = 15;
	public static final int ROLE_LumpedCurrent	= 16;
	public static final int ROLE_ElectricalUnitFactor = 17;
	
	// spatial stochastic-related roles
	public static final int ROLE_Binding_Radius  			= 18;
	public static final int ROLE_KOn  						= 19;
	public static final int ROLE_Diffusion_Reactant1 		= 20;
	public static final int ROLE_Diffusion_Reactant2  		= 21;
	public static final int ROLE_Concentration_Reactant1 	= 22;
	public static final int ROLE_Concentration_Reactant2	= 23;
	
	// not used anymore, for parsing only
	public static final int ROLE_ChargeValence				= 24;
	public static final int ROLE_Kcat						= 25;
	

	public static final int NUM_ROLES		= 26;

	
	/**
	 * These default descriptions are used by the XML persistence layer.
	 * 
	 */
	private static final String RoleDescs[] = {
		"user defined",
		"reaction rate",
		"inward current density",
		"forward rate constant",
		"reverse rate constant",
		"Km (1/2 max)",
		"max reaction rate",
		"Km forward",
		"max forward rate",
		"Km reverese",
		"max reverse rate",
		"permeability",
		"conductivity",
		"net charge valence",
		"flux carrier valence",
		"lumped reaction rate",
		"lumped current",
		"electrical unit factor",
		"binding radius",
		"reaction forward rate",
		"diffusion reactant1",
		"diffusion reactant2",
		"concentration reactant1",
		"concentration reactant2",
		// no used anymore, for parsing only
		"charge",
	};


	private static final String RoleTags[] = {
		"no tag",
		VCMODL.ReactionRate,
		VCMODL.CurrentDensity,
		VCMODL.ForwardRate,
		VCMODL.ReverseRate,
		VCMODL.Km,
		VCMODL.Vmax,
		VCMODL.KmFwd,
		VCMODL.VmaxFwd,
		VCMODL.KmRev,
		VCMODL.VmaxRev,
		VCMODL.Permeability,
		VCMODL.Conductivity,
		VCMODL.NetChargeValence,
		VCMODL.CarrierChargeValence,
		VCMODL.LumpedReactionRate,
		VCMODL.LumpedCurrent,
		VCMODL.ElectricalUnitFactor,
		VCMODL.Binding_Radius,
		VCMODL.Kon,
		VCMODL.Diffusion_Reactant1,
		VCMODL.Diffusion_Reactant2,
		VCMODL.Concentration_Reactant1,
		VCMODL.Concentration_Reactant2,
		// past end of list, old tags with equivalent parameters 
		// which is determined by isEquivalentRoleFromKineticsVCMLTokens()
		VCMODL.ChargeValence_OLD,
	};

	private static final String DefaultNames[] = {
		null,
		"J",
		"I",
		"Kf",
		"Kr",
		"Km",
		"Vmax",
		"KmFwd",
		"VmaxFwd",
		"KmRev",
		"VmaxRev",
		"P",
		"C",
		"netValence",
		"carrierValence",
		"LumpedJ",
		"LumpedI",
		"unitFactor",
		"Binding_Radius",
		"Kon",
		"Diffusion_Reactant1",
		"Diffusion_Reactant2",
		"Concentration_Reactant1",
		"Concentration_Reactant2",
		// not used
		"valence"
	};

	private static final RealInterval[] bounds = {
		null, // user defined
		null, // reaction rate
		null, // current density
		new RealInterval(0,Double.POSITIVE_INFINITY), // Kf
		new RealInterval(0,Double.POSITIVE_INFINITY), // Kr
		new RealInterval(0,Double.POSITIVE_INFINITY), // Km
		new RealInterval(0,Double.POSITIVE_INFINITY), // Vmax
		new RealInterval(0,Double.POSITIVE_INFINITY), // KmFwd
		new RealInterval(0,Double.POSITIVE_INFINITY), // VmaxFwd
		new RealInterval(0,Double.POSITIVE_INFINITY), // KmRev
		new RealInterval(0,Double.POSITIVE_INFINITY), // VmaxRev
		new RealInterval(0,Double.POSITIVE_INFINITY), // Permeability
		new RealInterval(0,Double.POSITIVE_INFINITY), // Conductivity
		null,   // net charge valence
		null,   // charge carrier valence
		null,   // lumped rate
		null,	// lumped current
		null,	// electrical unit factor
		new RealInterval(0,Double.POSITIVE_INFINITY), // Binding radius
		new RealInterval(0,Double.POSITIVE_INFINITY), // KOn
		new RealInterval(0,Double.POSITIVE_INFINITY), // Diff_reactant1
		new RealInterval(0,Double.POSITIVE_INFINITY), // Diff_reactant2
		new RealInterval(0,Double.POSITIVE_INFINITY), // Conc_reactant1
		new RealInterval(0,Double.POSITIVE_INFINITY), // Conc_reactant2
		// not used
		null,	// valence
	};
	
	private static final Logger lg = LogManager.getLogger(Kinetics.class);


	public class KineticsParameter extends Parameter implements ExpressionContainer, IssueSource, Identifiable {
		
		private String fieldParameterName = null;
		private Expression fieldParameterExpression = null;
		private int fieldParameterRole = ROLE_UserDefined;
		private VCUnitDefinition fieldUnitDefinition = null;

		protected KineticsParameter(String argName, Expression expression, int argRole, VCUnitDefinition unitDefinition) {
			if (argName == null){
				throw new IllegalArgumentException("parameter name is null");
			}
			if (argName.length()<1){
				throw new IllegalArgumentException("parameter name is zero length");
			}
			this.fieldParameterName = argName;
			this.fieldParameterExpression = expression;
			if (argRole >= 0 && argRole < NUM_ROLES){
				this.fieldParameterRole = argRole;
				setDescription(Kinetics.RoleDescs[argRole]);
			}else{
				throw new IllegalArgumentException("parameter 'role' = "+argRole+" is out of range");
			}
			this.fieldUnitDefinition = unitDefinition;
		}


		@Override
		public boolean compareEqual(Matchable obj) {
			if (!(obj instanceof KineticsParameter)){
				return false;
			}
			KineticsParameter kp = (KineticsParameter)obj;
			if (!super.compareEqual0(kp)){
				return false;
			}
			if (fieldParameterRole != kp.fieldParameterRole){
				return false;
			}
			
			return true;
		}

		@Override
		public boolean relate(Relatable obj, RelationVisitor rv) {
			if (!(obj instanceof KineticsParameter)){
				return false;
			}
			KineticsParameter kp = (KineticsParameter)obj;
			if (!super.relate0(kp, rv)){
				return false;
			}
			if (!rv.relate(fieldParameterRole, kp.fieldParameterRole)){
				return false;
			}

			return true;
		}



		public boolean isRegenerated(){
			return !isExpressionEditable();
		}
		public boolean isExpressionEditable(){
			//
			// don't allow direct editing of rates and currents if they are generated
			//
			if (getRole() == ROLE_ReactionRate){
				// only allow editing "ReactionRate" for GeneralKinetics
				if (Kinetics.this instanceof GeneralKinetics){
					return true;
				}else{
					return false;
				}
			}
			if (getRole() == ROLE_LumpedReactionRate){
				// only allow editing "LumpedReactionRate" for GeneralLumpedKinetics
				if (Kinetics.this instanceof GeneralLumpedKinetics){
					return true;
				}else{
					return false;
				}
			}
			if (getRole() == ROLE_CurrentDensity){
				// only allow editing "CurrentDensity" for GeneralCurrentKinetics
				if (Kinetics.this instanceof GeneralCurrentKinetics){
					return true;
				}else{
					return false;
				}
			}
			if (getRole() == ROLE_LumpedCurrent){
				// only allow editing "Current" for GeneralCurrentLumpedKinetics
				if (Kinetics.this instanceof GeneralCurrentLumpedKinetics){
					return true;
				}else{
					return false;
				}
			}
			
			if (getRole() == ROLE_Binding_Radius){
				// only allow editing "Current" for GeneralCurrentLumpedKinetics
				if (Kinetics.this instanceof Macroscopic_IRRKinetics){
					return false;
				}else{
					return true;
				}
			}
			
			if (getRole() == ROLE_KOn){
				// only allow editing "Current" for GeneralCurrentLumpedKinetics
				if (Kinetics.this instanceof Microscopic_IRRKinetics){
					return false;
				}else{
					return true;
				}
			}
			
			if (getRole() == ROLE_NetChargeValence || getRole() == ROLE_CarrierChargeValence){
				if (reactionStep.getPhysicsOptions() == ReactionStep.PHYSICS_MOLECULAR_ONLY){
					return false;
				}
			}
			
			if (getRole() == ROLE_ElectricalUnitFactor){
				return false;
			}
			
			return true;
		}
		
		public boolean isUnitEditable(){
			//
			// don't allow direct editing of rates and currents if they are generated
			//
			if (getRole() == ROLE_UserDefined){
				return true;
			}else{
				return false;
			}
		}

		public boolean isNameEditable(){
			return true;
		}

		public double getConstantValue() throws ExpressionException {
			return this.fieldParameterExpression.evaluateConstant();
		}      


		public Expression getExpression() {
			return this.fieldParameterExpression;
		}


		public int getIndex() {
			return -1;
		}


		public String getName(){ 
			return this.fieldParameterName; 
		}   

		public NameScope getNameScope() {
			if (Kinetics.this.reactionStep!=null){
				return Kinetics.this.reactionStep.getNameScope();
			}else{
				return null;
			}
		}

		public Kinetics getKinetics() {
			return Kinetics.this;
		}

		public int getRole() {
			return this.fieldParameterRole;
		}

		public VCUnitDefinition getUnitDefinition() {
			return fieldUnitDefinition;
		}

		public void setUnitDefinition(VCUnitDefinition unitDefinition) {
			VCUnitDefinition oldValue = fieldUnitDefinition;
			fieldUnitDefinition = unitDefinition;
			if (oldValue==unitDefinition){
				return;
			}
			if (unitDefinition!=null && unitDefinition.compareEqual(oldValue)){
				return;
			}
			super.firePropertyChange("unitDefinition", oldValue, unitDefinition);
		}
		
		public void setExpression(Expression expression) {
			Expression oldValue = fieldParameterExpression;
			fieldParameterExpression = expression;
			super.firePropertyChange(PROPERTYNAME_EXPRESSION, oldValue, expression);
		}
		public void setName(java.lang.String name) throws java.beans.PropertyVetoException {
			String oldValue = fieldParameterName;
			super.fireVetoableChange(PROPERTYNAME_NAME, oldValue, name);
			try {
				renameParameter(oldValue, name);
			} catch (ExpressionException e) {
				throw new RuntimeException("failed to change kinetic parameter "+oldValue+" to "+name+": "+e.getMessage(),e);
			}
			super.firePropertyChange(PROPERTYNAME_NAME, oldValue, name);
		}
		private void replaceNameLocal(String name) {
			// used rather than setName() to avoid unnecessary propogation of an alternate name during parsing.
			fieldParameterName = name;
		}
		
		public boolean isDescriptionEditable() {
			return false;
		}


		@Override
		public String getDescription() {
			if (getRole() == ROLE_NetChargeValence && reactionStep.getStructure() instanceof Membrane && reactionStep.getModel() != null){
				Membrane membrane = (Membrane)reactionStep.getStructure();
				Feature positiveFeature = reactionStep.getModel().getElectricalTopology().getPositiveFeature(membrane);
				if (positiveFeature != null){
					return "net charge into '"+positiveFeature.getName()+"'";
				}
			}
			return super.getDescription();
		}

	}

	public class KineticsProxyParameter extends ProxyParameter {

		public KineticsProxyParameter(SymbolTableEntry target){
			super(target);
		}
		
		public NameScope getNameScope() {
			if (Kinetics.this.reactionStep!=null){
				return Kinetics.this.reactionStep.getNameScope();
			}else{
				return null;
			}
		}

		@Override
		public boolean compareEqual(Matchable obj) {
			if (!(obj instanceof KineticsProxyParameter)){
				return false;
			}
			KineticsProxyParameter other = (KineticsProxyParameter)obj;
			if (getTarget() instanceof Matchable &&
					other.getTarget() instanceof Matchable &&
					Compare.isEqual((Matchable)getTarget(), (Matchable)other.getTarget())){
				return true;
			}else{
				return false;
			}
		}

		@Override
		public boolean relate(Relatable obj, RelationVisitor rv) {
			if (!(obj instanceof KineticsProxyParameter)){
				return false;
			}
			KineticsProxyParameter other = (KineticsProxyParameter)obj;
			if (getTarget() instanceof Relatable &&
					other.getTarget() instanceof Relatable &&
					((Relatable)getTarget()).relate((Relatable)other.getTarget(), rv)){
				return true;
			}else{
				return false;
			}
		}

		
		@Override
		public String getDescription() {
			if (getTarget() instanceof SpeciesContext) {
				return "Species Concentration";
			} else {
				return super.getDescription();
			}
		}

		@Override
		public void targetPropertyChange(PropertyChangeEvent evt) {
			super.targetPropertyChange(evt);
			if (evt.getPropertyName().equals("name")){
				String oldName = (String)evt.getOldValue();
				String newName = (String)evt.getNewValue();
				try {
					onProxyParameterNameChange(oldName, newName);
					
					//
					// clean up dangling parameters (those not reachable from the 'required' parameters).
					//
					cleanupParameters();
					
				}catch (ModelException | ExpressionException | PropertyVetoException e1){
					logger.error("property change failed", e1);
				}
			} 
		} 
		
	}
	
	public class UnresolvedParameter extends Parameter implements IssueSource {
		
		private String fieldParameterName = null;

		protected UnresolvedParameter(String argName) {
			if (argName == null){
				throw new IllegalArgumentException("parameter name is null");
			}
			if (argName.length()<1){
				throw new IllegalArgumentException("parameter name is zero length");
			}
			this.fieldParameterName = argName;
			setDescription("unresolved");
		}

		@Override
		public boolean compareEqual(Matchable obj) {
			if (!(obj instanceof UnresolvedParameter)){
				return false;
			}
			UnresolvedParameter up = (UnresolvedParameter)obj;
			if (!super.compareEqual0(up)){
				return false;
			}
			return true;
		}

		@Override
		public boolean relate(Relatable obj, RelationVisitor rv) {
			if (!(obj instanceof UnresolvedParameter)){
				return false;
			}
			UnresolvedParameter up = (UnresolvedParameter)obj;
			if (!super.relate0(up, rv)){
				return false;
			}
			return true;
		}

		public double getConstantValue() throws ExpressionException {
			throw new ExpressionException("no expression defined for UnresolvedParameter '"+fieldParameterName+"'");
		}      

		public Expression getExpression() {
			return null;
		}

		public boolean isExpressionEditable(){
			return false;
		}

		public boolean isNameEditable(){
			return false;
		}

		public boolean isUnitEditable(){
			return false;
		}

		public int getIndex() {
			return -1;
		}

		public String getName(){ 
			return this.fieldParameterName; 
		}

		public VCUnitDefinition getUnitDefinition() {
			return getReactionStep().getModel().getUnitSystem().getInstance_TBD();
		}

		public NameScope getNameScope() {
			if (Kinetics.this.reactionStep!=null){
				return Kinetics.this.reactionStep.getNameScope();
			}else{
				return null;
			}
		}
		
		public void setExpression(Expression exp){
			throw new RuntimeException("cannot set Expression on an UnresolvedParameter");
		}

		public void setUnitDefinition(VCUnitDefinition unit){
			throw new RuntimeException("cannot set unit on an UnresolvedParameter");
		}

		public void setName(java.lang.String name) throws java.beans.PropertyVetoException {
			String oldValue = fieldParameterName;
			super.fireVetoableChange("name", oldValue, name);
			fieldParameterName = name;
			super.firePropertyChange("name", oldValue, name);
		}

	}	

/**
 * Kinetics constructor comment.
 * @param name java.lang.String
 */
protected Kinetics(String name, ReactionStep reactionStep) {
	
	if (reactionStep.getModel() == null) {
		throw new RuntimeException("Model cannot be null");
	}
	
	this.name = name;
	this.reactionStep = reactionStep;
	addVetoableChangeListener(this);
	addPropertyChangeListener(this);
	reactionStep.addPropertyChangeListener(this);
	reactionStep.setKinetics(this);
	ReactionParticipant reactionParticipants[] = reactionStep.getReactionParticipants();
	for (int i = 0; i < reactionParticipants.length; i++){
		reactionParticipants[i].addPropertyChangeListener(this);
	}
}


protected void addKineticsParameter(KineticsParameter newParameter) throws PropertyVetoException {
	//
	// make copy of old array, and add to end
	//
	int oldLength = 0;
	if (fieldKineticsParameters!=null){
		oldLength = getKineticsParameters().length;
	}
	KineticsParameter newKineticsParameters[] = new KineticsParameter[oldLength+1];
	if (fieldKineticsParameters!=null){
		System.arraycopy(fieldKineticsParameters,0,newKineticsParameters,0,oldLength);
	}
	newKineticsParameters[oldLength] = newParameter;

	//
	// set new value of indexed property (kineticsParameters)
	//
	setKineticsParameters(newKineticsParameters);
}


public void addKineticsParameters(KineticsParameter[] newparameters) throws PropertyVetoException {
	ArrayList<KineticsParameter> newArrayList  = new ArrayList<KineticsParameter>();

	//check the new parameters are not repeated
	for (int i = 0; i < newparameters.length; i++){
		KineticsParameter oldParam = this.getKineticsParameter(newparameters[i].getName());
		
		if ( oldParam ==null ) {
			//if it is brand new just add it
			newArrayList.add(newparameters[i]);
		} else {
			//if it already exists, reuse it but update the content
			oldParam.setExpression(newparameters[i].getExpression());
			newArrayList.add(oldParam);
		}
	}

	//Check that there are no left parameters...
	KineticsParameter[] oldArray = this.getKineticsParameters();

	for (int i = 0; i < oldArray.length ; i++){
		if ( !newArrayList.contains(oldArray[i]) ) {
			newArrayList.add(oldArray[i]);
		}
	}

	//put it back into an array
	KineticsParameter[] allNewParameters = new KineticsParameter[newArrayList.size()];
	allNewParameters = (KineticsParameter[])newArrayList.toArray(allNewParameters);	
	//save the whole new array into this kinetics
	this.setKineticsParameters(allNewParameters);
}


/**
 * The addPropertyChangeListener method was generated to support the propertyChange field.
 */
public synchronized void addPropertyChangeListener(java.beans.PropertyChangeListener listener) {
	getPropertyChange().addPropertyChangeListener(listener);
}


/**
 * Insert the method's description here.
 * Creation date: (9/22/2003 9:51:49 AM)
 * @param parameterName java.lang.String
 */
public UnresolvedParameter addUnresolvedParameter(String parameterName) {
	if (getKineticsParameter(parameterName)!=null){
		throw new RuntimeException("local parameter '"+parameterName+"' already exists");
	}
	if (getProxyParameter(parameterName)!=null){
		throw new RuntimeException("referenced external symbol '"+parameterName+"' already exists");
	}
	if (getUnresolvedParameter(parameterName)!=null){
		throw new RuntimeException("unresolved parameter '"+parameterName+"' already exists");
	}
	UnresolvedParameter newParameter = new UnresolvedParameter(parameterName);
	UnresolvedParameter[] newUnresolvedParameters = ArrayUtils.addElement(fieldUnresolvedParameters,newParameter);
	setUnresolvedParameters(newUnresolvedParameters);
	
	return newParameter;
}


public KineticsProxyParameter addProxyParameter(SymbolTableEntry symbolTableEntry) {
	if (getKineticsParameter(symbolTableEntry.getName())!=null){
		throw new RuntimeException("local parameter '"+symbolTableEntry.getName()+"' already exists");
	}
	if (getProxyParameter(symbolTableEntry.getName())!=null){
		throw new RuntimeException("referenced external symbol '"+symbolTableEntry.getName()+"' already exists");
	}
	if (getUnresolvedParameter(symbolTableEntry.getName())!=null){
		removeUnresolvedParameter(getUnresolvedParameter(symbolTableEntry.getName()));
	}
	KineticsProxyParameter newProxyParameter = new KineticsProxyParameter(symbolTableEntry);
	KineticsProxyParameter[] newProxyParameters = ArrayUtils.addElement(fieldProxyParameters,newProxyParameter);
	setProxyParameters(newProxyParameters);
	return newProxyParameter;
}


/**
 * Insert the method's description here.
 * Creation date: (9/22/2003 9:51:49 AM)
 * @param parameterName java.lang.String
 */
public KineticsParameter addUserDefinedKineticsParameter(String parameterName, Expression expression, VCUnitDefinition unit) throws PropertyVetoException {
	if (getKineticsParameter(parameterName)!=null){
		throw new RuntimeException("parameter '"+parameterName+"' already exists");
	}
	if (getUnresolvedParameter(parameterName)!=null){
		removeUnresolvedParameter(getUnresolvedParameter(parameterName));
	}
	if (getProxyParameter(parameterName)!=null){
		removeProxyParameter(getProxyParameter(parameterName));
	}
	KineticsParameter newKineticsParameter = new KineticsParameter(parameterName,expression,ROLE_UserDefined, unit);
	KineticsParameter[] newKineticsParameters = ArrayUtils.addElement(fieldKineticsParameters,newKineticsParameter);
	setKineticsParameters(newKineticsParameters);
	return newKineticsParameter;
}

/**
 * convertParameterType : Used to convert a parameter from model (global) to local or vice versa depending on 'bConvertToLocal'
 * @param param : the parameter to be converted (could be kinetic parameter or kineticProxyParameter)
 * @param bConvertToGlobal : <true> => convert model to local parameter; <false> => convert local to model parameter
 */
public void convertParameterType(Parameter param, boolean bConvertToGlobal) throws PropertyVetoException, ExpressionBindingException {
	Expression expression = param.getExpression();
	if (!bConvertToGlobal) {
		// need to convert model parameter (the proxyparam/global) to local (kinetics) parameter 
		if (!(param instanceof KineticsProxyParameter)) {
			throw new RuntimeException("Parameter : \'" + param.getName() + "\' is not a proxy (global) parameter, cannot convert it to a local kinetics parameter.");
		} else {
			// first remove proxy param, 
			removeProxyParameter((KineticsProxyParameter)param);
			// then add it as kinetic param
			if (expression != null) {
				Expression newExpr = new Expression(expression);
				newExpr.bindExpression(getReactionStep());
				addKineticsParameter(new KineticsParameter(param.getName(), newExpr, Kinetics.ROLE_UserDefined, param.getUnitDefinition()));
			} 
		}
	} else {
		// need to convert local (the kinetics parameter) to model (proxy) parameter
		if (!(param instanceof KineticsParameter)) {
			throw new RuntimeException("Parameter : \'" + param.getName() + "\' is not a local kinetics parameter, cannot convert it to a global (proxy) parameter.");
		} else {
			// first check if kinetic param is the 'authoritative' kinetic parame; if so, cannot remove it.
			// else remove kinetic param - should have already been checked in parameterTableModel
			if (((KineticsParameter)param).getRole() != Kinetics.ROLE_UserDefined) {
				throw new RuntimeException("Cannot convert pre-defined kinetic parameter : \'" + param.getName() + "\' to global parameter.");
			}
			// First add param as a model parameter, if it is not already present
			ModelParameter mp = getReactionStep().getModel().getModelParameter(param.getName());
			if (mp == null) {
				Model model = getReactionStep().getModel();
				Expression newExpr = new Expression(expression);
				newExpr.bindExpression(model);
				model.addModelParameter(model.new ModelParameter(param.getName(), newExpr, Model.ROLE_UserDefined, param.getUnitDefinition()));
			}
			// Then remove param as a kinetic param (if 'param' is a model param, it is automatically added as a (proxy/global) param, 
			// since it is present in the reaction rate equn.
			removeKineticsParameter((KineticsParameter)param);
		}
	}
}

/**
 * The addVetoableChangeListener method was generated to support the vetoPropertyChange field.
 */
public synchronized void addVetoableChangeListener(java.beans.VetoableChangeListener listener) {
	getVetoPropertyChange().addVetoableChangeListener(listener);
}


/**
 * This method was created by a SmartGuide.
 * @param symbolTable cbit.vcell.parser.SymbolTable
 */
public void bind(SymbolTable symbolTable) throws ExpressionBindingException {
	setProxyParameters(new KineticsProxyParameter[0]);
	for (int i = 0; i < getKineticsParameters().length; i++){
		getKineticsParameters()[i].getExpression().bindExpression(symbolTable);
	}
	try {
		cleanupParameters();
		//resolveUndefinedUnits();
	}catch (PropertyVetoException | ModelException | ExpressionException e){
		throw new RuntimeException(e.getMessage(), e);
	}
}


/**
 * This method was created in VisualAge.
 */
private final void cleanupParameters() throws ModelException, ExpressionException, PropertyVetoException {
	if (bReading){
		return;
	}
	//
	// for each parameter, see if it is used, if not delete it
	//
	if (fieldKineticsParameters != null){
		for (int i=0;i<fieldKineticsParameters.length;i++){
			if (fieldKineticsParameters[i].getRole()==ROLE_UserDefined && !isReferenced(fieldKineticsParameters[i],0)){
				removeKineticsParameter(fieldKineticsParameters[i]);
				i--;
			}
		}
	}
	if (fieldUnresolvedParameters != null){
		for (int i=0;i<fieldUnresolvedParameters.length;i++){
			if (!isReferenced(fieldUnresolvedParameters[i],0)){
				removeUnresolvedParameter(fieldUnresolvedParameters[i]);
				i--;
			}
		}
	}
	if (fieldProxyParameters != null){
		for (int i=0;i<fieldProxyParameters.length;i++){
			if (!isReferenced(fieldProxyParameters[i],0)){
				removeProxyParameter(fieldProxyParameters[i]);
				i--;
			}
		}
	}

	if (fieldKineticsParameters!=null) {
		for (int i = 0; i < fieldKineticsParameters.length; i++){
			Expression exp = fieldKineticsParameters[i].getExpression();
			if (exp!=null){
				try {
					exp.bindExpression(reactionStep);
				}catch (ExpressionBindingException e){
					logger.error("error binding expression '"+exp.infix()+"': "+e.getMessage(), e);
				}
			}
		}
	}
//	resolveUndefinedUnits();
}


protected boolean compareEqual0(Kinetics kinetics) {

	if (!Compare.isEqual(name,kinetics.name)){
		return false;
	}
	if (!Compare.isEqualOrNull(fieldKineticsParameters,kinetics.fieldKineticsParameters)){
		return false;
	}
	
	return true;
}


protected boolean relate0(Kinetics kinetics, RelationVisitor rv) {

	if (!rv.relate(name,kinetics.name)){
		return false;
	}
	if (!rv.relateOrNull(fieldKineticsParameters,kinetics.fieldKineticsParameters)){
		return false;
	}

	return true;
}


/**
 * The firePropertyChange method was generated to support the propertyChange field.
 */
public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
	getPropertyChange().firePropertyChange(propertyName, oldValue, newValue);
}


/**
 * The fireVetoableChange method was generated to support the vetoPropertyChange field.
 */
public void fireVetoableChange(java.lang.String propertyName, java.lang.Object oldValue, java.lang.Object newValue) throws java.beans.PropertyVetoException {
	getVetoPropertyChange().fireVetoableChange(propertyName, oldValue, newValue);
}


public final void fromTokens(String kinetics_vcmlStr) throws ExpressionException, PropertyVetoException {
	try {
		reading(true);
		ArrayList<KineticsParameter> localKineticParameters = getKineticsParametersFromTokens(kinetics_vcmlStr);
		
		// remove any existing proxyParameters and unresolvedParameters (the list of localKineticParameters, when bound, will re-create the required proxy parameters
		fieldProxyParameters = new KineticsProxyParameter[0];
		fieldUnresolvedParameters = new UnresolvedParameter[0];
		
		// set kinetic parameters from the localParameters
		fieldKineticsParameters = localKineticParameters.toArray(new KineticsParameter[localKineticParameters.size()]);
		
		// bind all kinetic parameters (to automatically create proxy (and unresolved?) parameters
		bind(reactionStep);
		
	} finally {
		reading(false);
	}
	
	
}




private ArrayList<KineticsParameter> getKineticsParametersFromTokens(String kinetics_vcmlStr) throws ExpressionException, PropertyVetoException {
	//
	//  old format (version 1) (still supported for reading)
	//
	//	Kinetics GeneralCurrentKinetics {
	//      Parameter a 10;
	//      Parameter b 3;
	//      Parameter c d/2;
	//      Parameter d 5;
	//      CurrentDensity a+b/c;
	//  }
	//
	//
	//  new format (version 2) (deprecated, incompatible with Version 1, still supported for reading)
	//
	//	Kinetics GeneralCurrentKinetics {
	//      CurrentDensity 'currentDensity'
	//		Parameter currentDensity a+b/c;
	//      Parameter a 10;
	//      Parameter b 3;
	//      Parameter c d/2;
	//      Parameter d 5;
	//  }
	//
	//  latest format (version 3), which is backward compatable with version 1
	//  ParameterVCMLTokens (such as "CurrentDensity") have simple expressions 
	//  that always consist of only the requiredIdentifier (e.g.  currentDensity; )
	//
	//	Kinetics GeneralCurrentKinetics {
	//		Parameter currentDensity a+b/c;
	//      Parameter a 10;
	//      Parameter b 3;
	//      Parameter c d/2;
	//      Parameter d 5;
	//      CurrentDensity currentDensity;
	//  }
	//
	//	Kinetics GeneralCurrentKinetics {
	//		Parameter currentDensity a+b/c; [uM]
	//      Parameter a 10; [s-1]
	//      Parameter b 3; [m-1]
	//      Parameter c d/2; [uM.s-1]
	//      Parameter d 5; [s]
	//      CurrentDensity currentDensity; [s]
	//  }
	//
	//

	
	// first pass : 
	// protect predefined parameters
	ArrayList<KineticsParameter> localParameters = new ArrayList<Kinetics.KineticsParameter>();
	KineticsParameter[] predefinedKineticParameters = getKineticsParameters();
	for (KineticsParameter kp : predefinedKineticParameters){
		if (kp.getRole() != ROLE_UserDefined){
			localParameters.add(kp);
		}
	}
	
	
	CommentStringTokenizer tokens = new CommentStringTokenizer(kinetics_vcmlStr);
	@SuppressWarnings("unused")
	String kineticsKeywordToken = tokens.nextToken();
	@SuppressWarnings("unused")
	String kineticsNameToken = tokens.nextToken();
	@SuppressWarnings("unused")
	String beginBraceToken = tokens.nextToken(); // read "{"
	ModelUnitSystem unitSystem = reactionStep.getModel().getUnitSystem();
	
	HashMap<String,String> symbolRenamings = new HashMap<String, String>();

	while (tokens.hasMoreTokens()){
		String firstTokenOfLine = tokens.nextToken();
		if (firstTokenOfLine.equalsIgnoreCase(VCMODL.EndBlock)){
			break;
		}
		if (firstTokenOfLine.equalsIgnoreCase(VCMODL.Fast)){
			//setFast(true);
			continue;
		}
		if (firstTokenOfLine.equalsIgnoreCase(VCMODL.Parameter)){
			String name = tokens.nextToken();
			Expression exp = new Expression(tokens.readToSemicolon());
			String unitsString = tokens.nextToken();
			VCUnitDefinition unitDef = unitSystem.getInstance_TBD();
			if (unitsString.startsWith("[")){
				while (!unitsString.endsWith("]")){
					String tempToken = tokens.nextToken();
					unitsString = unitsString + " " + tempToken;
				}
				//
				// now string starts with '[' and ends with ']'
				//
				unitDef = unitSystem.getInstance(unitsString.substring(1,unitsString.length()-1));
			}else{
				tokens.pushToken(unitsString);  // read too far, put it back in the stream.
			}
			localParameters.add(new KineticsParameter(name,exp,ROLE_UserDefined,unitDef)); // could have two parameters with same name (one reserved, one user defined)
			continue;
		}
		//
		// not written as a parameter, try a requiredIdentifier
		//
		if (this instanceof GeneralLumpedKinetics) {
			if (firstTokenOfLine.equals(VCMODL.ReactionRate) || firstTokenOfLine.equals(VCMODL.CurrentDensity) || firstTokenOfLine.equals(VCMODL.AssumedCompartmentSize_oldname)){
				firstTokenOfLine = tokens.nextToken();
				continue;
			} else if (firstTokenOfLine.equals(VCMODL.TotalRate_oldname)) {
				firstTokenOfLine = VCMODL.LumpedReactionRate;
			}
		}
		//
		// assume that this line is a reserved ROLE
		//
		KineticsParameter parameterForRole = null;
		for (int i = 0; i < RoleTags.length; i++){
			if (firstTokenOfLine.equalsIgnoreCase(RoleTags[i])){
				//
				// get the parameter with this reserved role
				//
				for (KineticsParameter kp : localParameters){
					if (kp.getRole() == i){
						parameterForRole = kp;
						break;
					}
				}
				if (parameterForRole==null){
					for (KineticsParameter kp : localParameters){
						if (isEquivalentRoleFromKineticsVCMLTokens(i, kp.getRole())){
							parameterForRole = kp;
							break;
						}
					}
				}
				// may need to generate on the fly during database reading.
				if (parameterForRole == null && i == ROLE_ElectricalUnitFactor){
					KineticsParameter unitFactorParm = new KineticsParameter(getDefaultParameterName(ROLE_ElectricalUnitFactor), new Expression(1.0), ROLE_ElectricalUnitFactor, null);
					addKineticsParameter(unitFactorParm);
					parameterForRole = unitFactorParm;
				}
				// be forgiving while parsing charge parameter variants from the database - to support legacy models.
				if (parameterForRole == null && (i == ROLE_ChargeValence || i == ROLE_NetChargeValence || i == ROLE_CarrierChargeValence) ){
					parameterForRole = getChargeValenceParameter();
				}
				if (parameterForRole == null) {
					throw new RuntimeException("parameter for role "+RoleTags[i]+" not found in kinetic law "+this.getKineticsDescription().getName());
				}	
			}
		}

		String nextTokenAfterRole = tokens.nextToken();
		if (nextTokenAfterRole.endsWith("'") && nextTokenAfterRole.startsWith("'")){
			//
			// if requiredIdentifier name is present (delimited by single quotes)
			// use it as the user-supplied name for that required parameter
			//
			//     e.g. CurrentDensity 'currentDensity'
			//
			String parmName = nextTokenAfterRole.substring(1,nextTokenAfterRole.length()-1);
			if (!parameterForRole.getName().equals(parmName)){
				symbolRenamings.put(parameterForRole.getName(),parmName);
			}
		}else{
			//
			// else if a non-trivial expression, then use the default name for the requiredParameter.
			//      if a single identifier expression, then use that identifier as the requiredParameter name.
			//
			
			// first token (already popped) must be part of the expression (need to push it back).
			//
			//     e.g. CurrentDensity a+b/c;
			//
			tokens.pushToken(nextTokenAfterRole); // put back expression token for subsequent parsing.
			Expression exp = new Expression(tokens.readToSemicolon());
			
			//
			// find out if expression refers to another parameter declaration (expression of type "paramName;").
			//
			//     e.g. CurrentDensity I2;
			//
			String[] symbols = exp.getSymbols();
			boolean bIsSingleId = false;
			if (symbols != null && symbols.length==1){
				if (exp.compareEqual(new Expression(symbols[0]))){
					bIsSingleId = true;
				}
			}
			
			//
			// get unit definition (optional)
			//
			String unitsString = tokens.nextToken();
			VCUnitDefinition unitDef = unitSystem.getInstance_TBD();
			if (unitsString.startsWith("[")){
				while (!unitsString.endsWith("]")){
					String tempToken = tokens.nextToken();
					unitsString = unitsString + " " + tempToken;
				}
				//
				// now string starts with '[' and ends with ']'
				//
				unitDef = unitSystem.getInstance(unitsString.substring(1,unitsString.length()-1));
			}else{
				tokens.pushToken(unitsString); // from the next line put it back in the stream.
			}
			
			if (!bIsSingleId){
				//
				// expression is real, not just a reference to a parameter defined on another line
				// set expression and unit
				//
				parameterForRole.setExpression(exp);
				if (unitDef!=null && !unitDef.isTBD()){
					parameterForRole.setUnitDefinition(unitDef);
				}
			}else{
				//
				// expression is just a reference to a parameter defined on another line (e.g. Kf_012;)
				// rename the reserved parameter to that name
				//
				// NOTE: COULD CREATE A CONFLICT ... TWO PARAMETERS WITH THE SAME NAME (RESOLVE LATER)
				//
				String parmName = symbols[0];
				if (!parameterForRole.getName().equals(parmName)){
					symbolRenamings.put(parameterForRole.getName(),parmName);
				}
			}
		}
	}
	
	/*
	 * rename reserved parameter names and update expressions to reflect new names
	 * typical renaming example of kinetics_vcmlStr, Kf, Kr renamed to K1, K2
	 * 
	 Kinetics MassActionKinetics { 
		Parameter J ((K1 * s0) - (K2 * s1)); [uM.s-1]
		Parameter K1 (g0 / g1); [s-1]
		Parameter K2 (g1 / g0); [s-1]
		Rate J;
		ForwardRate K1;
		ReverseRate K2;
	} 
	 ... or ...
	Kinetics MassActionKinetics {
		ForwardRate 'Kf_PIP2_PLC'
		ReverseRate 'Kr_PIP2_PLC'
		Parameter Kf_PIP2_PLC kf_PIP2PLC;
		Parameter Kr_PIP2_PLC kr_PIP2PLC;
		Parameter kf_PIP2PLC 200.0;
		Parameter kr_PIP2PLC (Kd_PIP2PLC * kf_PIP2PLC);
		Parameter Kd_PIP2PLC 2.0;
	}
	 */
	for (String origSymbol : symbolRenamings.keySet()){
		String newSymbol = symbolRenamings.get(origSymbol);
		KineticsParameter newParam = null;
		for (KineticsParameter kp : localParameters){
			if (kp.getName().equals(newSymbol)){
				newParam = kp;
			}
		}
		for (KineticsParameter kp : localParameters){
			if (kp.getName().equals(origSymbol)){
				kp.replaceNameLocal(newSymbol);
				if (newParam != null && kp.getExpression().isZero()){
					kp.setExpression(new Expression(newParam.getExpression()));
				}
			}
			if (kp.getExpression().hasSymbol(origSymbol)){
				kp.getExpression().substituteInPlace(new Expression(origSymbol), new Expression(newSymbol));
			}
		}
	}
	
	//
	// merge parameters with same name (one is UserDefined, the other is Reserved role)
	//
	boolean bDirty = true;
	while (bDirty) {
		bDirty = false;
		for (KineticsParameter kp1 : localParameters){
			KineticsParameter kp_withSameName = null;
			for (KineticsParameter kp2 : localParameters){
				if (kp1!=kp2 && kp1.getName().equals(kp2.getName())){
					kp_withSameName = kp2;
				}
			}
			if (kp_withSameName!=null){
				// copy expression and unit from UserDefined Parameter to the Reserved Parmaeter and remove the user defined parameter
				KineticsParameter userParameter = null;
				KineticsParameter reservedParameter = null;
				if (kp1.getRole()==ROLE_UserDefined && kp_withSameName.getRole()!=ROLE_UserDefined){
					userParameter = kp1;
					reservedParameter = kp_withSameName;
				}else if (kp_withSameName.getRole()==ROLE_UserDefined && kp1.getRole()!=ROLE_UserDefined){
					reservedParameter = kp1;
					userParameter = kp_withSameName;
				}else{
					throw new RuntimeException("found two parameters with same name '"+kp1.getName()+"' in reaction '"+reactionStep.getName()+"', not able to reconcile");
				}
				reservedParameter.setExpression(userParameter.getExpression());
				if (userParameter.getUnitDefinition()==null || userParameter.getUnitDefinition().isTBD()){
					reservedParameter.setUnitDefinition(userParameter.getUnitDefinition());
				}
				localParameters.remove(userParameter);

				bDirty = true;
				break;
			}
		}
	}
	return localParameters;
}

private boolean isEquivalentRoleFromKineticsVCMLTokens(int role1, int role2) {
	if ( (role1 == ROLE_KmFwd && role2 == ROLE_Km) || (role2 == ROLE_KmFwd && role1 == ROLE_Km) ) {
		return true;
	}
	if ( (role1 == ROLE_VmaxFwd && role2 == ROLE_Vmax) || (role2 == ROLE_VmaxFwd && role1 == ROLE_Vmax) ) {
		return true;
	}
	if ( (reactionStep instanceof SimpleReaction) && 
			 ((role1 == ROLE_NetChargeValence && role2 == ROLE_ChargeValence) || 
			  (role2 == ROLE_ChargeValence && role1 == ROLE_NetChargeValence))){
			return true;
	}
	if ( (reactionStep instanceof FluxReaction) && 
			 ((role1 == ROLE_CarrierChargeValence && role2 == ROLE_ChargeValence) || 
			  (role2 == ROLE_ChargeValence && role1 == ROLE_CarrierChargeValence))){
			return true;
	}
	return false;
}

/**
 * Insert the method's description here.
 * Creation date: (5/12/2004 2:53:13 PM)
 */
public void gatherIssues(IssueContext issueContext, List<Issue> issueList) {
	issueContext = issueContext.newChildContext(ContextType.ModelProcessDynamics,this);
	//
	// for each user unresolved parameter, make an issue
	//
	for (int i = 0; fieldUnresolvedParameters!=null && i < fieldUnresolvedParameters.length; i++){
		issueList.add(new Issue(fieldUnresolvedParameters[i],issueContext, IssueCategory.UnresolvedParameter,"Unresolved parameter '"+fieldUnresolvedParameters[i].getName()+"' in reaction '"+reactionStep.getName()+"'",Issue.SEVERITY_ERROR));
	}
	//
	// for each user defined parameter, see if it is used, if not make an issue
	//
	for (int i=0;fieldKineticsParameters!=null && i<fieldKineticsParameters.length;i++){
		if (fieldKineticsParameters[i].getRole()==ROLE_UserDefined){
			try {
				if (!isReferenced(fieldKineticsParameters[i],0)){
					issueList.add(new Issue(fieldKineticsParameters[i],issueContext, IssueCategory.KineticsUnreferencedParameter,"Unreferenced Kinetic Parameter '"+fieldKineticsParameters[i].getName()+"' in reaction '"+reactionStep.getName()+"'",Issue.SEVERITY_WARNING));
				}
			}catch (ExpressionException e){
				issueList.add(new Issue(fieldKineticsParameters[i],issueContext,IssueCategory.KineticsExpressionError, "error resolving expression " + e.getMessage(),Issue.SEVERITY_WARNING));
			}catch (ModelException e){
				issueList.add(new Issue(getReactionStep(),issueContext,IssueCategory.CyclicDependency,"cyclic dependency in the parameter definitions", Issue.SEVERITY_ERROR));
			}
		}
	}

	//
	// check for use of symbol bindings that are species contexts that are not reaction participants
	//
	if (fieldKineticsParameters!=null) {
		for (KineticsParameter kineticsParameter : fieldKineticsParameters){
			if (kineticsParameter.getExpression()==null){
				issueList.add(new Issue(kineticsParameter,issueContext,IssueCategory.KineticsExpressionMissing,"expression is missing",Issue.SEVERITY_INFO));
			}else{
				Expression exp = kineticsParameter.getExpression();
				String symbols[] = exp.getSymbols();
				String issueMessagePrefix = "Kinetic parameter '" + kineticsParameter.getName() + "' in reaction '" + getReactionStep().getName() + "' ";
				if (symbols!=null) { 
					for (int j = 0; j < symbols.length; j++){
						SymbolTableEntry ste = exp.getSymbolBinding(symbols[j]);
						if (ste instanceof KineticsProxyParameter) {
							ste = ((KineticsProxyParameter) ste).getTarget();
						}
						if (ste == null) {
							issueList.add(new Issue(kineticsParameter,issueContext,IssueCategory.KineticsExpressionUndefinedSymbol, issueMessagePrefix + "references undefined symbol '"+symbols[j]+"'",Issue.SEVERITY_ERROR));
						} else if (ste instanceof SpeciesContext) {
							if (!getReactionStep().getModel().contains((SpeciesContext)ste)) {
								issueList.add(new Issue(kineticsParameter,issueContext,IssueCategory.KineticsExpressionUndefinedSymbol, issueMessagePrefix + "references undefined species '"+symbols[j]+"'",Issue.SEVERITY_ERROR));
							}
							if (reactionStep.countNumReactionParticipants((SpeciesContext)ste) == 0){
								issueList.add(new Issue(kineticsParameter,issueContext,IssueCategory.KineticsExpressionNonParticipantSymbol, issueMessagePrefix + "references species context '"+symbols[j]+"', but it is not a reactant/product/catalyst of this reaction",Issue.SEVERITY_WARNING));
							}
						} else if (ste instanceof ModelParameter) {
							if (!getReactionStep().getModel().contains((ModelParameter)ste)) {
								issueList.add(new Issue(kineticsParameter,issueContext,IssueCategory.KineticsExpressionUndefinedSymbol, issueMessagePrefix + "references undefined global parameter '"+symbols[j]+"'",Issue.SEVERITY_ERROR));
							}
						}
					}
				}
			}
		}
		
		// looking for local param which masks a global and issueing a warning
		for (KineticsParameter kineticsParameter : fieldKineticsParameters){
			String name = kineticsParameter.getName();
			SymbolTableEntry ste = getReactionStep().getNameScope().getExternalEntry(name, getReactionStep());
			String steName;
			if(ste != null) {
				if(ste instanceof Displayable) {
					steName = ((Displayable) ste).getDisplayType() + " " + ste.getName();
				} else {
					steName = ste.getClass().getSimpleName() + " " + ste.getName();
				}
				String msg = steName + " is overriden by a local parameter " + name + " in reaction " + getReactionStep().getName();
				issueList.add(new Issue(kineticsParameter,issueContext,IssueCategory.Identifiers,msg ,Issue.SEVERITY_WARNING));
			}
		}
	}

	try {
		//
		// determine unit consistency for each expression
		//
		ModelUnitSystem modelUnitSystem = getReactionStep().getModel().getUnitSystem();
		VCUnitEvaluator unitEvaluator = new VCUnitEvaluator(modelUnitSystem);
		for (int i = 0; i < fieldKineticsParameters.length; i++){
			try {
				VCUnitDefinition paramUnitDef = fieldKineticsParameters[i].getUnitDefinition();
				VCUnitDefinition expUnitDef = unitEvaluator.getUnitDefinition(fieldKineticsParameters[i].getExpression());
				if (paramUnitDef == null){
					issueList.add(new Issue(fieldKineticsParameters[i], issueContext, IssueCategory.Units,"defined unit is null",Issue.SEVERITY_WARNING));
				}else if (paramUnitDef.isTBD()){
					issueList.add(new Issue(fieldKineticsParameters[i], issueContext, IssueCategory.Units,"undefined unit " + modelUnitSystem.getInstance_TBD().getSymbol(),Issue.SEVERITY_WARNING));
				}else if (expUnitDef == null){
					issueList.add(new Issue(fieldKineticsParameters[i], issueContext, IssueCategory.Units,"computed unit is null",Issue.SEVERITY_WARNING));
				}else if (paramUnitDef.isTBD() || (!paramUnitDef.isEquivalent(expUnitDef) && !expUnitDef.isTBD())){
					issueList.add(new Issue(fieldKineticsParameters[i], issueContext, IssueCategory.Units,"inconsistent units, defined=["+fieldKineticsParameters[i].getUnitDefinition().getSymbol()+"], computed=["+expUnitDef.getSymbol()+"]",Issue.SEVERITY_WARNING));
				}
			}catch (VCUnitException e){
				issueList.add(new Issue(fieldKineticsParameters[i], issueContext, IssueCategory.Units, e.getMessage(),Issue.SEVERITY_WARNING));
			}catch (ExpressionException e){
				issueList.add(new Issue(fieldKineticsParameters[i],issueContext,IssueCategory.Units, e.getMessage(),Issue.SEVERITY_WARNING));
			}
		}
	}catch (Throwable e){
		issueList.add(new Issue(getReactionStep(),issueContext,IssueCategory.Units,"unexpected exception: "+e.getMessage(),Issue.SEVERITY_INFO));
	}

	//
	// add constraints (simpleBounds) for predefined parameters
	//
	for (int i = 0; i < fieldKineticsParameters.length; i++){
		RealInterval simpleBounds = bounds[fieldKineticsParameters[i].getRole()];
		if (simpleBounds!=null){
			String parmName = reactionStep.getNameScope().getName()+"."+fieldKineticsParameters[i].getName();
			issueList.add(new SimpleBoundsIssue(fieldKineticsParameters[i], issueContext, simpleBounds, "parameter "+parmName+": must be within "+simpleBounds.toString()));
		}
	}
	
}


/**
 * Insert the method's description here.
 * Creation date: (10/16/2003 7:22:36 AM)
 * @return java.lang.String
 * @param role int
 */
public String getDefaultParameterDesc(int role) {
	if (role < 0 || role >= NUM_ROLES){
		throw new IllegalArgumentException("role out of range, "+role);
	}
//	return DefaultNames[role]+"_"+TokenMangler.fixTokenStrict(getReactionStep().getName());
	return RoleDescs[role];
}


/**
 * Insert the method's description here.
 * Creation date: (10/16/2003 7:22:36 AM)
 * @return java.lang.String
 * @param role int
 */
String getDefaultParameterName(int role) {
	if (role < 0 || role >= NUM_ROLES){
		throw new IllegalArgumentException("role out of range, "+role);
	}
//	return DefaultNames[role]+"_"+TokenMangler.fixTokenStrict(getReactionStep().getName());
	return DefaultNames[role];
}


/**
 * Insert the method's description here.
 * Creation date: (8/5/2002 5:51:50 PM)
 * @return cbit.vcell.model.KineticsDescription
 */
public abstract KineticsDescription getKineticsDescription();


public KineticsParameter getKineticsParameter(String pName){
	if (fieldKineticsParameters == null){
		return null;
	}
	for (int i=0;i<fieldKineticsParameters.length;i++){
		KineticsParameter parm = fieldKineticsParameters[i];
		if (pName.equals(parm.getName())){
			return parm;
		}
	}
	return null;
}   

public KineticsProxyParameter getProxyParameter(String pName){
	if (fieldProxyParameters == null){
		return null;
	}
	for (int i=0;i<fieldProxyParameters.length;i++){
		KineticsProxyParameter parm = fieldProxyParameters[i];
		if (pName.equals(parm.getName())){
			return parm;
		}
	}
	return null;
}   


/**
 * Gets the kineticsParameters index property (cbit.vcell.model.KineticsParameter) value.
 * @return The kineticsParameters property value.
 * @see #setKineticsParameters
 */
public KineticsParameter getKineticsParameterFromRole(int role) {
	for (int i = 0; i < fieldKineticsParameters.length; i++){
		if (fieldKineticsParameters[i].getRole() == role){
			return fieldKineticsParameters[i];
		}
	}
	return null;
}


/**
 * Gets the kineticsParameters property (cbit.vcell.model.KineticsParameter[]) value.
 * @return The kineticsParameters property value.
 * @see #setKineticsParameters
 */
public KineticsParameter[] getKineticsParameters() {
	return fieldKineticsParameters;
}

public KineticsProxyParameter[] getProxyParameters() {
	return fieldProxyParameters;
}

protected boolean hasOutwardFlux(){
	if (getReactionStep()!=null && getReactionStep() instanceof FluxReaction && getReactionStep().getModel()!=null && getReactionStep().getStructure() instanceof Membrane){
		Membrane membrane = (Membrane)getReactionStep().getStructure();
		Reactant r = null;
		Product p = null;
		for (ReactionParticipant rp : getReactionStep().getReactionParticipants()){
			if (rp instanceof Reactant){
				r = (Reactant)rp;
			}
			if (rp instanceof Product){
				p = (Product)rp;
			}
		}
		Feature insideFeature = getReactionStep().getModel().getElectricalTopology().getPositiveFeature(membrane);
		Feature outsideFeature = getReactionStep().getModel().getElectricalTopology().getNegativeFeature(membrane);
		if (insideFeature!=null && outsideFeature!=null && r!=null && p!=null){
			if (r.getStructure() == insideFeature && p.getStructure() == outsideFeature){ // outward molecular flux
				return true;
			}
		}
	}
	return false;
}


/**
 * Gets the kineticsParameters index property (cbit.vcell.model.KineticsParameter) value.
 * @return The kineticsParameters property value.
 * @param index The index value into the property array.
 * @see #setKineticsParameters
 */
public KineticsParameter getKineticsParameters(int index) {
	return getKineticsParameters()[index];
}


   public final String getName() 
   { 
	  return name; 
   }   

	public static int getParamRoleFromDefaultDesc(String paramDesc) {

		int paramRole = -1;
		
		if (paramDesc == null || paramDesc.length() == 0) {
			throw new IllegalArgumentException("Invalid value for parameter description: " + paramDesc); 
		}
		for (int i = 0; i < RoleDescs.length; i++) {
			if (RoleDescs[i].equals(paramDesc)) {
				paramRole = i;
			}	
		}

		if (paramDesc.equals(Kinetics.GTK_TotalRate_oldname)) {
			paramRole = ROLE_LumpedReactionRate;
		} 
		if (paramDesc.equals(Kinetics.GTK_AssumedCompartmentSize_oldname)) {
			paramRole = ROLE_UserDefined;
		}
		if (paramDesc.equals(Kinetics.observedForwardRate_oldname)) {
			paramRole = ROLE_KForward;
		}
		if (paramDesc.equals(Kinetics.observedReverseRate_oldname)) {
			paramRole = ROLE_KReverse;
		}

		if (paramRole == -1) {
			throw new IllegalArgumentException("Parameter description: " + paramDesc + " is not valid.");
		}
		return paramRole;
	}


/**
 * Accessor for the propertyChange field.
 */
protected java.beans.PropertyChangeSupport getPropertyChange() {
	if (propertyChange == null) {
		propertyChange = new java.beans.PropertyChangeSupport(this);
	};
	return propertyChange;
}


/**
 * This method was created by a SmartGuide.
 * @return cbit.vcell.model.ReactionStep
 */
public ReactionStep getReactionStep() {
	return reactionStep;
}


public UnresolvedParameter getUnresolvedParameter(String pName){
	if (fieldUnresolvedParameters == null){
		return null;
	}
	for (int i=0;i<fieldUnresolvedParameters.length;i++){
		UnresolvedParameter parm = fieldUnresolvedParameters[i];
		if (pName.equals(parm.getName())){
			return parm;
		}
	}
	return null;
}   


/**
 * Gets the unresolvedParameters property (cbit.vcell.model.UnresolvedParameter[]) value.
 * @return The unresolvedParameters property value.
 * @see #setUnresolvedParameters
 */
public UnresolvedParameter[] getUnresolvedParameters() {
	return fieldUnresolvedParameters;
}


/**
 * Gets the unresolvedParameters index property (cbit.vcell.model.UnresolvedParameter) value.
 * @return The unresolvedParameters property value.
 * @param index The index value into the property array.
 * @see #setUnresolvedParameters
 */
public UnresolvedParameter getUnresolvedParameters(int index) {
	return getUnresolvedParameters()[index];
}


/**
 * This method was created in VisualAge.
 * @return java.lang.String
 */
public String getVCML() {
	java.io.StringWriter stringWriter = new java.io.StringWriter();
	java.io.PrintWriter pw = new java.io.PrintWriter(stringWriter);
	writeTokens(pw);
	pw.flush();
	pw.close();
	return stringWriter.getBuffer().toString();
}


/**
 * Accessor for the vetoPropertyChange field.
 */
protected java.beans.VetoableChangeSupport getVetoPropertyChange() {
	if (vetoPropertyChange == null) {
		vetoPropertyChange = new java.beans.VetoableChangeSupport(this);
	};
	return vetoPropertyChange;
}


private boolean isReferenced(Parameter parm, int level) throws ModelException, ExpressionException {
	//
	// check for unbounded recursion (level > 10)
	//
	if (level >= 10){
		throw new ModelException("there is a loop in the parameter definitions");
	}

	//
	// if reactionStep is null (building from database), then it is OK
	//
	if (reactionStep==null){
		return true;
	}
	
	//
	// if this parameter is same as a reaction participant, then it should be removed
	//
//	if (reactionStep.getReactionParticipantFromSymbol(parm.getName())!=null){
//		return false;
//	}

	//
	// if this unresolved parameter is same as another parameter, then it should be removed
	// note: that external identifiers have precedence over UndefinedParameters in ReactionStep.getEntry()
	//       so getEntry() will only return a UnresolvedParameter if there is no alternative.
	//
	if ((parm instanceof Kinetics.UnresolvedParameter) && parm != reactionStep.getEntry(parm.getName())){
		return false;
	}

	////
	//// if parameter is referenced in rate expression, then it is OK
	////
	//String symbols[] = getRequiredIdentifiers();
	//if (symbols != null){
		//for (int j=0;j<symbols.length;j++){
			//if (symbols[j].equals(parm.getName())){
				//return true;
			//}
		//}
	//}
	if (parm instanceof KineticsParameter && ((KineticsParameter)parm).getRole() != ROLE_UserDefined){
		return true;
	}

	//
	// else, if parameter is referenced in another parameter's expression, continue with that expression
	//
	if (fieldKineticsParameters != null){
		for (int i=0;i<fieldKineticsParameters.length;i++){
			Parameter parentParm = fieldKineticsParameters[i];
			Expression exp = parentParm.getExpression();
			String[] symbols = exp.getSymbols();
			if (symbols!=null){
				for (int j=0;j<symbols.length;j++){
					if (AbstractNameScope.getStrippedIdentifier(symbols[j]).equals(parm.getName())){
						if (isReferenced(parentParm,level+1)){
							return true;
						}
					}
				}
			}
		}
	}
	return false;
}


public final void electricalTopologyChanged(ElectricalTopology electricalTopology) {
	/**
	 * Kinetics doesn't listen directly to Model, ReactionStep listens to Model and relays this event. 
	 */
	try {
		updateGeneratedExpressions();
		refreshUnits();
		cleanupParameters();
	}catch (Exception e){
		logger.error(e.getMessage(), e);
	}
}


/**
 * Insert the method's description here.
 * Creation date: (2/15/01 1:56:56 PM)
 * @param event java.beans.PropertyChangeEvent
 */
public void propertyChange(PropertyChangeEvent event) {
	try {
		if (event.getSource() instanceof ReactionStep && event.getPropertyName().equals("physicsOptions")){
			updateGeneratedExpressions();
			refreshUnits();
			cleanupParameters();
		}
		if (event.getSource() instanceof ReactionStep && event.getPropertyName().equals("structure")){
			updateGeneratedExpressions();
			refreshUnits();
			cleanupParameters();
		}
		if (event.getSource() instanceof ReactionStep && event.getPropertyName().equals(ReactionStep.PROPERTY_NAME_REACTION_PARTICIPANTS)){
			ReactionParticipant oldValues[] = (ReactionParticipant[])event.getOldValue();
			for (int i = 0; oldValues!=null && i < oldValues.length; i++){
				oldValues[i].removePropertyChangeListener(this);
			}
			ReactionParticipant newValues[] = (ReactionParticipant[])event.getNewValue();
			for (int i = 0; newValues != null && i < newValues.length; i++){
				newValues[i].addPropertyChangeListener(this);
			}
			updateGeneratedExpressions();
			refreshUnits();
			cleanupParameters();
		}
		if (event.getSource() == this && event.getPropertyName().equals(PROPERTY_NAME_KINETICS_PARAMETERS)){
			KineticsParameter oldValues[] = (KineticsParameter[])event.getOldValue();
			for (int i = 0; oldValues!=null && i < oldValues.length; i++){
				oldValues[i].removePropertyChangeListener(this);
			}
			KineticsParameter newValues[] = (KineticsParameter[])event.getNewValue();
			for (int i = 0; newValues != null && i < newValues.length; i++){
				newValues[i].addPropertyChangeListener(this);
			}
			refreshUnits();
			cleanupParameters();
		}
		if (event.getSource() instanceof ReactionParticipant){
			updateGeneratedExpressions();
			if (event.getPropertyName().equals("name")){
				String oldName = (String)event.getOldValue();
				String newName = (String)event.getNewValue();
				onProxyParameterNameChange(oldName,newName);
			}
			refreshUnits();
			cleanupParameters();
		}
		if (event.getSource() instanceof KineticsParameter){
			refreshUnits();
			cleanupParameters();
		}
	}catch (Throwable e){
		logger.error(e.getMessage(), e);
	}
}


/**
 * Insert the method's description here.
 * Creation date: (5/11/2004 6:19:00 PM)
 * @param argReading boolean
 */
public void reading(boolean argReading) {
	if (argReading == bReading){
		throw new RuntimeException("flag conflict");
	}
	this.bReading = argReading;
	if (!bReading){
		resolveUndefinedUnits();
	}
}


/**
 * Insert the method's description here.
 * Creation date: (5/24/01 4:05:36 PM)
 */
public void refreshDependencies() {
	removePropertyChangeListener(this);
	removeVetoableChangeListener(this);
	addPropertyChangeListener(this);
	addVetoableChangeListener(this);
	reactionStep.removePropertyChangeListener(this);
	reactionStep.addPropertyChangeListener(this);
	ReactionParticipant reactionParticipants[] = reactionStep.getReactionParticipants();
	for (int i = 0; i < reactionParticipants.length; i++){
		reactionParticipants[i].removePropertyChangeListener(this);
		reactionParticipants[i].addPropertyChangeListener(this);
	}
	for (int i = 0; i < fieldKineticsParameters.length; i++){
		fieldKineticsParameters[i].removePropertyChangeListener(this);
		fieldKineticsParameters[i].addPropertyChangeListener(this);
	}
	try {
		updateGeneratedExpressions();
	}catch (ExpressionException e){
	}catch (PropertyVetoException e){
	}
	
	// if modelUnitSystem was null for some reason while reading fromTokens(), a temp vcUnitSystem was used to parse units which can be re-bound within refreshUnits()
	refreshUnits();
	//resolveUndefinedUnits();
}


/**
 * Insert the method's description here.
 * Creation date: (3/31/2004 3:55:44 PM)
 */
protected abstract void refreshUnits();


void removeAllUnresolvedParameters() {
	setUnresolvedParameters(new UnresolvedParameter[0]);
}


protected void removeKineticsParameter(KineticsParameter parameter) throws PropertyVetoException {
	KineticsParameter[] newKineticsParameters = ArrayUtils.removeFirstInstanceOfElement(fieldKineticsParameters,parameter);
	setKineticsParameters(newKineticsParameters);
}

/**
 * removeUserDefinedKineticParam : wrapper for 'removeKineticsParameter' to be able to remove user-defined kinetics parameter
 * 		from ModelParameterPanel.
 */
public void removeUserDefinedKineticParam(Parameter parameter) throws PropertyVetoException {
	if (parameter instanceof KineticsParameter && ((KineticsParameter)parameter).getRole() == ROLE_UserDefined) {
		removeKineticsParameter((KineticsParameter)parameter);
	} else {
		throw new RuntimeException("The kinetic parameter to be removed is not user-defined; Cannot remove it from reaction \'" + getReactionStep().getName() + "\'.");
	}
}


/**
 * The removePropertyChangeListener method was generated to support the propertyChange field.
 */
public synchronized void removePropertyChangeListener(java.beans.PropertyChangeListener listener) {
	getPropertyChange().removePropertyChangeListener(listener);
}


/**
 * Insert the method's description here.
 * Creation date: (9/22/2003 9:51:49 AM)
 * @param parameter cbit.vcell.model.UnsolvedParameter
 */
protected void removeUnresolvedParameter(UnresolvedParameter parameter) {
	for (int i = 0; i < fieldUnresolvedParameters.length; i++){
		if (fieldUnresolvedParameters[i] == parameter){
			UnresolvedParameter[] newUnresolvedParameters = ArrayUtils.removeFirstInstanceOfElement(fieldUnresolvedParameters,parameter);
			setUnresolvedParameters(newUnresolvedParameters);
			return;
		}
	}
	throw new RuntimeException("UnresolvedParameter '"+parameter.getName()+"' not found");
}



protected void removeProxyParameter(KineticsProxyParameter parameter) {
	for (int i = 0; i < fieldProxyParameters.length; i++){
		if (fieldProxyParameters[i] == parameter){
			KineticsProxyParameter[] newProxyParameters = ArrayUtils.removeFirstInstanceOfElement(fieldProxyParameters,parameter);
			setProxyParameters(newProxyParameters);
			return;
		}
	}
	throw new RuntimeException("UnresolvedParameter '"+parameter.getName()+"' not found");
}


void removeUnresolvedParameters(SymbolTable symbolTable) {
	Kinetics.UnresolvedParameter[] unresolvedParms = fieldUnresolvedParameters.clone();
	for (int i = 0; i < unresolvedParms.length; i++){
		SymbolTableEntry ste = symbolTable.getEntry(unresolvedParms[i].getName());
		if (ste != unresolvedParms[i]){
			unresolvedParms = ArrayUtils.removeFirstInstanceOfElement(unresolvedParms,unresolvedParms[i]);
			i--;
		}
	}
	setUnresolvedParameters(unresolvedParms);
}


/**
 * The removeVetoableChangeListener method was generated to support the vetoPropertyChange field.
 */
public synchronized void removeVetoableChangeListener(java.beans.VetoableChangeListener listener) {
	getVetoPropertyChange().removeVetoableChangeListener(listener);
}


/**
 * Insert the method's description here.
 * Creation date: (5/24/01 4:05:36 PM)
 */
public void renameParameter(String oldName, String newName) throws ExpressionException, java.beans.PropertyVetoException{
	if (oldName==null || newName==null){
		throw new RuntimeException("renameParameter from '"+oldName+"' to '"+newName+"', nulls are not allowed");
	}

	if (oldName.equals(newName)){
		logger.info("rename parameter called with same name: "+oldName);
	}
	KineticsParameter parameter = getKineticsParameter(oldName);
	if (parameter!=null){
		//
		// must change name in KineticsParameter directly
		// then change all references to this name in the other parameter's expressions.
		//
		KineticsParameter newParameters[] = (KineticsParameter[])fieldKineticsParameters.clone();
		//
		// replaces parameter with name 'oldName' with new parameter with name 'newName' and original expression.
		//
		for (int i = 0; i < newParameters.length; i++){
			if (newParameters[i] == parameter){
				newParameters[i] = new KineticsParameter(newName,parameter.getExpression(),parameter.getRole(),parameter.getUnitDefinition());
			}
		}
		//
		// go through all parameters' expressions and replace references to 'oldName' with 'newName'
		//
		for (int i = 0; i < newParameters.length; i++){
			Expression exp = newParameters[i].getExpression();
			if (exp.getSymbolBinding(oldName) != null) {
				Expression newExp = new Expression(exp);
			
				newExp.substituteInPlace(new Expression(oldName), new Expression(newName));
				//
				// if expression changed, create a new KineticsParameter
				//
				newParameters[i] = new KineticsParameter(newParameters[i].getName(),newExp,newParameters[i].getRole(),newParameters[i].getUnitDefinition());
			}
		}
		setKineticsParameters(newParameters);

		// 
		// rebind all expressions
		//
		for (int i = 0; i < fieldKineticsParameters.length; i++){
			fieldKineticsParameters[i].getExpression().bindExpression(getReactionStep());
		}

		//
		// clean up dangling parameters (those not reachable from the 'required' parameters).
		//
		try {
			cleanupParameters();
		}catch (ModelException e){
			throw new RuntimeException(e.getMessage(), e);
		}
	}
}


/**
 * Insert the method's description here.
 * Creation date: (11/29/2006 3:33:33 PM)
 */
public void resolveCurrentWithStructure(Structure structure) throws PropertyVetoException{
	
	ModelUnitSystem modelUnitSystem = reactionStep.getModel().getUnitSystem();
	if(structure instanceof Feature && this.getKineticsParameterFromRole(Kinetics.ROLE_CurrentDensity) != null){
		this.removeKineticsParameter(this.getKineticsParameterFromRole(Kinetics.ROLE_CurrentDensity));
		
	}else if(structure instanceof Feature && this.getKineticsParameterFromRole(Kinetics.ROLE_LumpedCurrent) != null){
		this.removeKineticsParameter(this.getKineticsParameterFromRole(Kinetics.ROLE_LumpedCurrent));
		
	}else if(structure instanceof Membrane && this.getKineticsParameterFromRole(Kinetics.ROLE_CurrentDensity) == null){
		String pname = this.getDefaultParameterName(Kinetics.ROLE_CurrentDensity);
		Kinetics.KineticsParameter currentParm = new Kinetics.KineticsParameter(pname,new Expression(0.0),Kinetics.ROLE_CurrentDensity, modelUnitSystem.getCurrentDensityUnit());
		addKineticsParameter(currentParm);
		
	}else if(structure instanceof Membrane && this.getKineticsParameterFromRole(Kinetics.ROLE_LumpedCurrent) == null){
		String pname = this.getDefaultParameterName(Kinetics.ROLE_LumpedCurrent);
		Kinetics.KineticsParameter currentParm = new Kinetics.KineticsParameter(pname,new Expression(0.0),Kinetics.ROLE_LumpedCurrent, modelUnitSystem.getCurrentUnit());
		addKineticsParameter(currentParm);
	}
	
	
}


/**
 * Insert the method's description here.
 * Creation date: (4/13/2004 3:09:21 PM)
 */
public void resolveUndefinedUnits() {
	//
	// try to fix units for UserDefined parameters
	//
	if (!bResolvingUnits){
		bResolvingUnits = true;
		try {
			if (getReactionStep().getModel() != null) {
				ModelUnitSystem modelUnitSystem = getReactionStep().getModel().getUnitSystem();
				boolean bAnyTBDUnits = false;
				for (int i=0;i<fieldKineticsParameters.length;i++){
					if (fieldKineticsParameters[i].getUnitDefinition()==null){
						return; // not ready to resolve units yet
					}else if (fieldKineticsParameters[i].getUnitDefinition().isTBD()){
						bAnyTBDUnits = true;
					}
				}
				//
				// try to resolve TBD units (will fail if units are inconsistent) ... but these errors are collected in Kinetics.getIssues().
				//
				if (bAnyTBDUnits){
					VCUnitEvaluator unitEvaluator = new VCUnitEvaluator(modelUnitSystem);
					VCUnitDefinition vcUnitDefinitions[] = unitEvaluator.suggestUnitDefinitions(fieldKineticsParameters);
					for (int i = 0; i < fieldKineticsParameters.length; i++){
						if (!fieldKineticsParameters[i].getUnitDefinition().isEquivalent(vcUnitDefinitions[i])){
							fieldKineticsParameters[i].setUnitDefinition(vcUnitDefinitions[i]);
						}
					}
				}
			}
		}catch (Exception e){
			if (lg.isWarnEnabled()) {
				lg.warn("Kinetics.resolveUndefinedUnits(): EXCEPTION: "+e.getMessage());
			}
		}finally{
			bResolvingUnits = false;
		}
	}
}


/**
 * Sets the kineticsParameters property (cbit.vcell.model.KineticsParameter[]) value.
 * @param kineticsParameters The new value for the property.
 * @exception java.beans.PropertyVetoException The exception description.
 * @see #getKineticsParameters
 */
void setKineticsParameters(KineticsParameter[] kineticsParameters) throws java.beans.PropertyVetoException {
	KineticsParameter[] oldValue = fieldKineticsParameters;
	fireVetoableChange(PROPERTY_NAME_KINETICS_PARAMETERS, oldValue, kineticsParameters);
	fieldKineticsParameters = kineticsParameters;

	firePropertyChange(PROPERTY_NAME_KINETICS_PARAMETERS, oldValue, kineticsParameters);
}


/**
 * Sets the kineticsParameters index property (cbit.vcell.model.KineticsParameter[]) value.
 * @param index The index value into the property array.
 * @param kineticsParameters The new value for the property.
 * @see #getKineticsParameters
 */
void setKineticsParameters(int index, KineticsParameter kineticsParameters) {
	KineticsParameter oldValue = fieldKineticsParameters[index];
	fieldKineticsParameters[index] = kineticsParameters;
	if (oldValue != null && !oldValue.equals(kineticsParameters)) {
		firePropertyChange(PROPERTY_NAME_KINETICS_PARAMETERS, null, fieldKineticsParameters);
	};
}


/**
 * This method was created by a SmartGuide.
 * @param name java.lang.String
 */
void setName(String name) {
	this.name = name;
}


public void setParameterValue(KineticsParameter parm, Expression exp) throws ExpressionException, PropertyVetoException {
	Parameter p = getKineticsParameter(parm.getName());
	if (p != parm){
		throw new RuntimeException("parameter "+parm.getName()+" not found");
	}
	Expression oldExpression = parm.getExpression();
	boolean bBound = false;
	try {
		KineticsParameter[] newKineticsParameters = (KineticsParameter[])fieldKineticsParameters.clone();
		//KineticsProxyParameter newProxyParameters[] = (KineticsProxyParameter[])fieldProxyParameters.clone();
		String symbols[] = exp.getSymbols();
		ModelUnitSystem modelUnitSystem = getReactionStep().getModel().getUnitSystem();
		for (int i = 0; symbols!=null && i < symbols.length; i++){
			SymbolTableEntry ste = reactionStep.getEntry(symbols[i]);
			if (ste==null){
				newKineticsParameters = ArrayUtils.addElement(newKineticsParameters,new KineticsParameter(symbols[i],new Expression(0.0),ROLE_UserDefined, modelUnitSystem.getInstance_TBD()));
			}
		}
		parm.setExpression(exp);
		setKineticsParameters(newKineticsParameters);
		//setProxyParameters(newProxyParameters);
		exp.bindExpression(reactionStep);
		bBound = true;
	}finally{
		try {
			if (!bBound){
				parm.setExpression(oldExpression);
			}
			cleanupParameters();
		}catch (ModelException | PropertyVetoException e){
			throw new RuntimeException(e.getMessage(), e);
		}
	}
}

/**
 * Sets the unresolvedParameters property (cbit.vcell.model.UnresolvedParameter[]) value.
 * @param unresolvedParameters The new value for the property.
 * @see #getUnresolvedParameters
 */
private void setUnresolvedParameters(UnresolvedParameter[] unresolvedParameters) {
	UnresolvedParameter[] oldValue = fieldUnresolvedParameters;
	fieldUnresolvedParameters = unresolvedParameters;

	firePropertyChange("unresolvedParameters", oldValue, unresolvedParameters);
}

private void setProxyParameters(KineticsProxyParameter[] proxyParameters) {
	KineticsProxyParameter[] oldValue = fieldProxyParameters;
	fieldProxyParameters = proxyParameters;

	firePropertyChange("proxyParameters", oldValue, proxyParameters);
}


/**
 * This method was created by a SmartGuide.
 * @return java.lang.String
 */
public String toString() {
	return getClass().getName() + "@" + Integer.toHexString(hashCode());
}


/**
 * updates the rate parameter expression and the current parameter expression based on reactants/products/modifiers/charge valence/membrane polarity
 */
protected abstract void updateGeneratedExpressions() throws ExpressionException, PropertyVetoException;


/**
 * Insert the method's description here.
 * Creation date: (5/23/00 10:41:22 PM)
 * @param e java.beans.PropertyChangeEvent
 */
public void vetoableChange(PropertyChangeEvent e) throws PropertyVetoException {
}


public final void writeTokens(java.io.PrintWriter pw) {
	//
	//  old format (version 1) (still supported for reading)
	//
	//	Kinetics GeneralCurrentKinetics {
	//      Parameter a 10;
	//      Parameter b 3;
	//      Parameter c d/2;
	//      Parameter d 5;
	//      CurrentDensity a+b/c;
	//  }
	//
	//
	//  new format (version 2) (deprecated, incompatible with Version 1, still supported for reading)
	//
	//	Kinetics GeneralCurrentKinetics {
	//      CurrentDensity 'currentDensity'
	//		Parameter currentDensity a+b/c;
	//      Parameter a 10;
	//      Parameter b 3;
	//      Parameter c d/2;
	//      Parameter d 5;
	//  }
	//
	//  latest format (version 3), which is backward compatable with version 1
	//  ParameterVCMLTokens (such as "CurrentDensity") have simple expressions 
	//  that always consist of only the requiredIdentifier (e.g.  currentDensity; )
	//
	//	Kinetics GeneralCurrentKinetics {
	//		Parameter currentDensity a+b/c;
	//      Parameter a 10;
	//      Parameter b 3;
	//      Parameter c d/2;
	//      Parameter d 5;
	//      CurrentDensity currentDensity;
	//  }
	//
	//

	pw.println("\t\t"+VCMODL.Kinetics+" "+getKineticsDescription().getVCMLKineticsName()+" "+VCMODL.BeginBlock+" ");

	KineticsParameter parameters[] = getKineticsParameters();
	if (parameters!=null){
		for (int i=0;i<parameters.length;i++){
			KineticsParameter parm = parameters[i];
			VCUnitDefinition unit = parm.getUnitDefinition();
			if (unit == null) {
				pw.println("\t\t\t"+VCMODL.Parameter+" "+parm.getName()+" "+parm.getExpression().infix() + ";" );
			} else {
				pw.println("\t\t\t"+VCMODL.Parameter+" "+parm.getName()+" "+parm.getExpression().infix() + ";" +
					       " [" +  unit.getSymbol() + "]");
			}
		}
	}

	//
	// write primary tokens as simple expressions (e.g. parmName; )
	//
	for (int i = 0; i < parameters.length; i++){
		KineticsParameter parm = parameters[i];
		if (parm.getRole() != ROLE_UserDefined){
			pw.println("\t\t\t"+RoleTags[parm.getRole()]+" "+parm.getName()+";");
		}
	}
	
	
	pw.println("\t\t"+VCMODL.EndBlock+" ");
}

public final String writeTokensWithReplacingProxyParams(Hashtable<String, Expression> parmExprHash) {
	//
	//  old format (version 1) (still supported for reading)
	//
	//	Kinetics GeneralCurrentKinetics {
	//      Parameter a 10;
	//      Parameter b 3;
	//      Parameter c d/2;
	//      Parameter d 5;
	//      CurrentDensity a+b/c;
	//  }
	//
	//
	//  new format (version 2) (deprecated, incompatible with Version 1, still supported for reading)
	//
	//	Kinetics GeneralCurrentKinetics {
	//      CurrentDensity 'currentDensity'
	//		Parameter currentDensity a+b/c;
	//      Parameter a 10;
	//      Parameter b 3;
	//      Parameter c d/2;
	//      Parameter d 5;
	//  }
	//
	//  latest format (version 3), which is backward compatable with version 1
	//  ParameterVCMLTokens (such as "CurrentDensity") have simple expressions 
	//  that always consist of only the requiredIdentifier (e.g.  currentDensity; )
	//
	//	Kinetics GeneralCurrentKinetics {
	//		Parameter currentDensity a+b/c;
	//      Parameter a 10;
	//      Parameter b 3;
	//      Parameter c d/2;
	//      Parameter d 5;
	//      CurrentDensity currentDensity;
	//  }
	//
	//

	java.io.StringWriter stringWriter = new java.io.StringWriter();
	java.io.PrintWriter pw = new java.io.PrintWriter(stringWriter);

	pw.println("\t\t"+VCMODL.Kinetics+" "+getKineticsDescription().getVCMLKineticsName()+" "+VCMODL.BeginBlock+" ");

	KineticsParameter parameters[] = getKineticsParameters();
	if (parameters!=null){
		for (int i=0;i<parameters.length;i++){
			KineticsParameter parm = parameters[i];
			Expression paramExpr = parmExprHash.get(parm.getName());
			VCUnitDefinition unit = parm.getUnitDefinition();
			if (unit == null) {
				pw.println("\t\t\t"+VCMODL.Parameter+" "+parm.getName()+" "+paramExpr.infix() + ";" );
			} else {
				pw.println("\t\t\t"+VCMODL.Parameter+" "+parm.getName()+" "+paramExpr.infix() + ";" +
				       " [" +  unit.getSymbol() + "]");
			}
		}
	}

	//
	// write primary tokens as simple expressions (e.g. parmName; )
	//
	for (int i = 0; i < parameters.length; i++){
		KineticsParameter parm = parameters[i];
		if (parm.getRole() != ROLE_UserDefined){
			pw.println("\t\t\t"+RoleTags[parm.getRole()]+" "+parm.getName()+";");
		}
	}
	
	
	pw.println("\t\t"+VCMODL.EndBlock+" ");
	
	pw.flush();
	pw.close();
	return stringWriter.getBuffer().toString();

}

protected Expression getSymbolExpression(SymbolTableEntry ste) throws ExpressionBindingException {
	SymbolTableEntry entry = getReactionStep().getEntry(ste.getName());
	return new Expression(entry, getReactionStep().getNameScope());
}

public abstract KineticsParameter getAuthoritativeParameter();

public final KineticsParameter getChargeValenceParameter() {
	KineticsParameter netChargeValenceParameter = getKineticsParameterFromRole(ROLE_NetChargeValence);
	KineticsParameter carrierChargeValenceParameter = getKineticsParameterFromRole(ROLE_CarrierChargeValence);
	if (netChargeValenceParameter!=null && carrierChargeValenceParameter!=null){
		throw new RuntimeException("both netChargeValence and carrierChargeValence parameters are defined");
	}
	if (netChargeValenceParameter!=null){
		return netChargeValenceParameter;
	}
	if (carrierChargeValenceParameter!=null){
		return carrierChargeValenceParameter;
	}
	return null;
};

public final Expression getElectricalUnitFactor(VCUnitDefinition unitFactor) throws PropertyVetoException {
	if (unitFactor.isEquivalent(getReactionStep().getModel().getUnitSystem().getInstance_DIMENSIONLESS())){
		return null;
	}
	
	RationalNumber factor = unitFactor.getDimensionlessScale();
	KineticsParameter kineticsParameter = getKineticsParameterFromRole(ROLE_ElectricalUnitFactor);
	if (kineticsParameter == null){
		kineticsParameter = new KineticsParameter(DefaultNames[ROLE_ElectricalUnitFactor],new Expression(factor),ROLE_ElectricalUnitFactor, unitFactor);
		addKineticsParameter(kineticsParameter);
	}else{
		kineticsParameter.setExpression(new Expression(factor));
		kineticsParameter.setUnitDefinition(unitFactor);
	}
	return new Expression(kineticsParameter,null);
}


private void onProxyParameterNameChange(String oldName, String newName) throws PropertyVetoException, ModelException, ExpressionException {
	//
	// go through all parameters' expressions and replace references to 'oldName' with 'newName'
	//
	KineticsParameter newParameters[] = null;
	for (int i = 0; i < fieldKineticsParameters.length; i++){ 
		Expression exp = fieldKineticsParameters[i].getExpression();
		
		if (exp.getSymbolBinding(oldName) != null) {
			if (newParameters == null) {
				newParameters = fieldKineticsParameters.clone();
			}
			exp = exp.renameBoundSymbols(getReactionStep().getNameScope());		
			newParameters[i] = new KineticsParameter(newParameters[i].getName(),exp,newParameters[i].getRole(),newParameters[i].getUnitDefinition());
		}
	}
	if (newParameters != null) {
		setKineticsParameters(newParameters);
	}
}


public void setReactionStep(ReactionStep reactionStep) {
	this.reactionStep = reactionStep;
}

}
