/*
 * Copyright (C) 1999-2011 University of Connecticut Health Center
 *
 * Licensed under the MIT License (the "License").
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *  http://www.opensource.org/licenses/mit-license.php
 */

package cbit.vcell.geometry;

import java.beans.PropertyChangeListener;
import java.beans.PropertyVetoException;
import java.beans.VetoableChangeListener;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.TreeSet;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.vcell.util.Compare;
import org.vcell.util.Coordinate;
import org.vcell.util.Extent;
import org.vcell.util.ISize;
import org.vcell.util.Issue;
import org.vcell.util.Issue.IssueCategory;
import org.vcell.util.IssueContext;
import org.vcell.util.Matchable;
import org.vcell.util.Origin;
import org.vcell.util.PropertyChangeListenerProxyVCell;
import org.vcell.util.State;
import org.vcell.util.VCellThreadChecker;
import org.vcell.util.document.Version;

import cbit.image.ImageException;
import cbit.image.ThumbnailImage;
import cbit.image.VCImage;
import cbit.image.VCImageUncompressed;
import cbit.image.VCPixelClass;
import cbit.vcell.parser.Expression;
import cbit.vcell.parser.ExpressionException;
/**
 * This interface was generated by a SmartGuide.
 * 
 */
@SuppressWarnings("serial")
public class GeometrySpec implements Matchable, PropertyChangeListener, VetoableChangeListener, Serializable {

	public static final String PROPERTY_NAME_SAMPLED_IMAGE = "sampledImage";
	public static final String PROPERTY_NAME_THUMBNAIL_IMAGE = "thumbnailImage";
	public static final String PROPERTY_NAME_GEOMETRY_NAME = "geometryName";

	public final static int IMAGE_SIZE_LIMIT =  4000000;
	
	public static final String ORIGIN_PROPERTY = "origin";
	public static final String EXTENT_PROPERTY = "extent";
	
	private static final Logger lg = LogManager.getLogger(GeometrySpec.class);
	
	private VCImage vcImage = null;
	private transient byte[] uncompressedPixels = null;
	private transient State<VCImage> sampledImage = new State<VCImage>(null);
	private transient State<ThumbnailImage> thumbnailImage = new State<ThumbnailImage>(null);
	
	//
	// zero dimension is a point geometry object
	//
	private int dimension = 0;
	private Extent extent = new Extent(10,10,10);
	private Origin origin = new Origin(0,0,0);
//	private Vector subVolumeList = new Vector();
	//private Vector curveTypeList = new Vector();
	protected transient java.beans.PropertyChangeSupport propertyChange;
	protected transient java.beans.VetoableChangeSupport vetoPropertyChange;
	private SubVolume[] subVolumes = new SubVolume[0];
	private FilamentGroup filamentGroup = new FilamentGroup();
	
public GeometrySpec(Version aVersion, int aDimension) {
	addVetoableChangeListener(this);
	addPropertyChangeListener(this);
	this.dimension = aDimension;
	if (dimension==0){
		try {
			setSubVolumes(new SubVolume[] { new CompartmentSubVolume(null,0) });
		}catch (PropertyVetoException e){
			lg.error("PropertyVetoException in Geometry(Version,int=0): this should never happen", e);
		}
	}
}


public GeometrySpec(Version aVersion, VCImage aVCImage) {
	addVetoableChangeListener(this);
	addPropertyChangeListener(this);
	try {
		setImage(aVCImage);
	}catch (PropertyVetoException e){
		lg.error(e);
		throw new RuntimeException(e.getMessage());
	}
}


public GeometrySpec(GeometrySpec geometrySpec){
	addVetoableChangeListener(this);
	addPropertyChangeListener(this);
	if(geometrySpec.vcImage != null){
		try {
			this.vcImage = new VCImageUncompressed(geometrySpec.vcImage);
		} catch (ImageException e) {
			lg.error(e);
			throw new RuntimeException(e.getMessage(), e);
		}
	}
	this.dimension = geometrySpec.dimension;
	try {
		SubVolume[] newSubvolumes = new SubVolume[geometrySpec.subVolumes.length];
		for (int i = 0; i < newSubvolumes.length; i++) {
			if(geometrySpec.subVolumes[i] instanceof ImageSubVolume){
				ImageSubVolume oldImagesuImageSubVolume = (ImageSubVolume)(geometrySpec.subVolumes[i]);
				newSubvolumes[i] = new ImageSubVolume(oldImagesuImageSubVolume);
				VCPixelClass newVCPixelClass = this.vcImage.getPixelClassFromName(oldImagesuImageSubVolume.getPixelClass().getPixelClassName());
				((ImageSubVolume)newSubvolumes[i]).setPixelClass(newVCPixelClass);
			}else if(geometrySpec.subVolumes[i] instanceof AnalyticSubVolume){
				newSubvolumes[i] = new AnalyticSubVolume((AnalyticSubVolume)(geometrySpec.subVolumes[i]));
			}else if(geometrySpec.subVolumes[i] instanceof CSGObject){
				newSubvolumes[i] = new CSGObject((CSGObject)(geometrySpec.subVolumes[i]));
			}else if(geometrySpec.subVolumes[i] instanceof CompartmentSubVolume){
				newSubvolumes[i] = new CompartmentSubVolume(geometrySpec.subVolumes[i].getKey(), geometrySpec.subVolumes[i].getHandle());
			}else{
				throw new RuntimeException("Unknown SubVolume "+geometrySpec.subVolumes[i]);
			}
		}
		setSubVolumes(newSubvolumes);
	}catch (PropertyVetoException e){
		lg.error(e);
		throw new RuntimeException(e.getMessage());
	}
	//this.curveTypeList = (Vector)geometry.curveTypeList.clone();
	this.extent = geometrySpec.getExtent(); // Extent is immutable
	this.origin = geometrySpec.getOrigin(); // Origin is immutable
}

public GeometrySpec(String aName, int aDimension) {
	addVetoableChangeListener(this);
	addPropertyChangeListener(this);
	this.dimension = aDimension;
	if (dimension==0){
		try {
			setSubVolumes(new SubVolume[] { new CompartmentSubVolume(null,0) });
		}catch (PropertyVetoException e){
			lg.error("PropertyVetoException in Geometry(String,int=0): this should never happen", e);
		}
	}
}


public GeometrySpec(String aName, VCImage aVCImage) {
	addVetoableChangeListener(this);
	addPropertyChangeListener(this);
	try {
		setImage(aVCImage);
	}catch (PropertyVetoException e){
		lg.error(e);
		throw new RuntimeException(e.getMessage());
	}
}


/**
 * The addPropertyChangeListener method was generated to support the propertyChange field.
 */
public synchronized void addPropertyChangeListener(java.beans.PropertyChangeListener listener) {
	PropertyChangeListenerProxyVCell.addProxyListener(getPropertyChange(), listener);
}


public void addSubVolume(AnalyticSubVolume subVolume) throws PropertyVetoException {
	addSubVolume(subVolume, false);
}
/**
 * This method was created in VisualAge.
 * @param subVolume cbit.vcell.geometry.SubVolume
 */
public void addSubVolume(AnalyticSubVolume subVolume,boolean bFront) throws PropertyVetoException {
	addAnalyticSubVolumeOrCSGObject(subVolume, bFront);
}

public void addSubVolume(CSGObject csgObject, boolean bFront) throws PropertyVetoException {
	addAnalyticSubVolumeOrCSGObject(csgObject, bFront);
}

private void addAnalyticSubVolumeOrCSGObject(SubVolume subVolume, boolean bFront) throws PropertyVetoException {
	if (!(subVolume instanceof AnalyticSubVolume || subVolume instanceof CSGObject)) {
		throw new IllegalArgumentException("Only AnalyticSubVolume or CSGObject can added");
	}
	
	if (getSubVolumeIndex(subVolume) != -1){
		throw new IllegalArgumentException("subdomain "+subVolume+" cannot be added, already exists");
	}

	//
	// add after last analytic subvolume (but before imageSubVolumes)
	//
	int firstImageIndex = -1;
	for (int i = 0; i< subVolumes.length; i++){
		if (subVolumes[i] instanceof ImageSubVolume){
			firstImageIndex = i;
			break;
		}
	}
	
	SubVolume newArray[] = new SubVolume[subVolumes.length+1];
	
	if (firstImageIndex == -1){
		//
		// no image subvolumes
		//
		if(bFront){
			newArray[0] = subVolume;
			if (subVolumes.length>0){
				System.arraycopy(subVolumes,0,newArray,1, subVolumes.length);
			}
		}else{
			//add to end of analytics
			// copy first N elements
			if (subVolumes.length>0){
				System.arraycopy(subVolumes,0,newArray,0, subVolumes.length);
			}
			// add new element to end
			newArray[subVolumes.length] = subVolume;
		}
		
	}else{
	//
	// first imageSubVolume at index 'firstImageIndex'
	// insert subVolume at this index (before first ImageSubVolume) 
	// and push all of the imageSubVolumes back
	//
		int newIndex = 0;
		for (int i = 0; i< subVolumes.length; i++){
			if(bFront){
				if (i == 0){
					newArray[newIndex++] = subVolume;
				}				
			}else{
				if (i == firstImageIndex){
					newArray[newIndex++] = subVolume;
				}				
			}
			newArray[newIndex++] = subVolumes[i];
		}
	}	

	subVolume.setHandle(getFreeSubVolumeHandle());
	setSubVolumes(newArray);	
		
/*
	if (!subVolumeList.contains(subVolume)){
		for (int i=0;i<subVolumeList.size();i++){
			SubVolume sv = (SubVolume)subVolumeList.elementAt(i);
			if (sv instanceof ImageSubVolume){
				if (subVolume.getHandle()>=0){
					throw new RuntimeException("subVolume already has handle set");
				}
				subVolume.setHandle(getFreeSubVolumeHandle());
				insertSubVolumeAt0(subVolume,i);
				setChanged();
				notifyObservers(subVolume);
				return;
			}
		}
		subVolume.setHandle(getFreeSubVolumeHandle());
		addSubVolume0(subVolume);
		setChanged();
		notifyObservers(subVolume);
	}
*/
}


/**
 * The addVetoableChangeListener method was generated to support the vetoPropertyChange field.
 */
public synchronized void addVetoableChangeListener(java.beans.VetoableChangeListener listener) {
	getVetoPropertyChange().addVetoableChangeListener(listener);
}

public void bringForward(AnalyticSubVolume subVolume) throws PropertyVetoException {
	bringForwardAnalyticSubVolumeOrCSGObject(subVolume);
}

public void bringForward(CSGObject subVolume) throws PropertyVetoException {
	bringForwardAnalyticSubVolumeOrCSGObject(subVolume);
}

/**
 * This method was created in VisualAge.
 * @param subVolume cbit.vcell.geometry.SubVolume
 */
private void bringForwardAnalyticSubVolumeOrCSGObject(SubVolume subVolume) throws PropertyVetoException {

	if (!(subVolume instanceof AnalyticSubVolume || subVolume instanceof CSGObject)) {
		throw new IllegalArgumentException("Only AnalyticSubVolume or CSGObject can be brought forward");
	}
	
	//
	// copy existing array
	//
	SubVolume newArray[] = (SubVolume[]) subVolumes.clone();
	
	//
	// find index of subVolume in newArray
	//
	int index = -1;
	for (int i=0;i<newArray.length;i++){
		if (newArray[i] == subVolume){
			index = i;
		}
	}

	if (index == -1){
		throw new IllegalArgumentException("subdomain "+subVolume+" not found");
	}

	//
	// if not already the first AnalyticSubVolume or CSGObject, then swap with neighbor at index-1
	//
	if ((index > 0) && (newArray[index-1] instanceof AnalyticSubVolume || newArray[index-1] instanceof CSGObject)){
		SubVolume tempSubVolume = newArray[index-1];
		newArray[index-1] = subVolume;
		newArray[index] = tempSubVolume;
	
		setSubVolumes(newArray);
	}
}


/**
 * This method was created in VisualAge.
 * @return boolean
 * @param object java.lang.Object
 */
public boolean compareEqual(Matchable object) {
	if (object == null || !(object instanceof GeometrySpec)){
		return false;
	}
	GeometrySpec geometrySpec = (GeometrySpec)object;
	
	if (dimension != geometrySpec.dimension){
		return false;
	}
	if (!Compare.isEqual(getExtent().getAsClipped(dimension), geometrySpec.getExtent().getAsClipped(dimension))){
		return false;
	}

	if (!Compare.isEqual(getOrigin().getAsClipped(dimension), geometrySpec.getOrigin().getAsClipped(dimension))){
		return false;
	}
	
	//
	// if only one is null, bad
	//
	if ((vcImage == null && geometrySpec.vcImage != null) || (vcImage != null && geometrySpec.vcImage == null)) {
		return false;
	}
	if (vcImage != null && geometrySpec.vcImage != null && !vcImage.compareEqual(geometrySpec.vcImage, dimension, true)){
		return false;
	}

	if (subVolumes.length != geometrySpec.subVolumes.length){
		return false;
	}

	for (int i = 0; i< subVolumes.length; i++){
		if (subVolumes[i] instanceof ImageSubVolume isv1 && geometrySpec.subVolumes[i] instanceof ImageSubVolume isv2){
            if (!isv1.compareEqual(isv2, VCPixelClass::compareEqualIgnoreNames)){
				return false;
			}
		}else {
			if (!subVolumes[i].compareEqual(geometrySpec.subVolumes[i])) {
				return false;
			}
		}
	}
	//
	// FilamentGroup
	//
	if(!Compare.isEqual(filamentGroup,geometrySpec.filamentGroup)){
		return false;
	}
	return true;
}


/**
 * This method was created in VisualAge.
 * @return cbit.image.VCImage
 */
public VCImage createSampledImage(ISize sampleSize) throws GeometryException, ImageException, ExpressionException {
	
	VCellThreadChecker.checkCpuIntensiveInvocation();
	
	byte handles[] = new byte[sampleSize.getX()*sampleSize.getY()*sampleSize.getZ()];
	for (int i=0;i<handles.length;i++){
		handles[i] = -1;
	}
	//
	// compartmental
	//
	if (dimension==0){
		handles[0] = 0;
	}
	
	//
	// if image exists, preload with pixels (translated into handles)
	//
	if (vcImage!=null){
		//
		// make lookup table of pixel values ----> handles
		//
		byte handleLookupTable[] = new byte[256];
		for (int i = 0; i< subVolumes.length; i++){
			SubVolume sv = subVolumes[i];
			if (sv instanceof ImageSubVolume){
				ImageSubVolume isv = (ImageSubVolume)sv;
				handleLookupTable[isv.getPixelValue()] = (byte)isv.getHandle();
			}
		}
		
		byte vciPixels[] = vcImage.getPixels();
		if (vcImage.getNumX()==sampleSize.getX() && vcImage.getNumY()==sampleSize.getY() && vcImage.getNumZ()==sampleSize.getZ()){
			//
			// image and sample same size, just translate pixels to handles
			//
			for (int j=0;j<vciPixels.length;j++){
				handles[j] = handleLookupTable[0xff&(int)vciPixels[j]];
			}
		}else{
			//
			// image and sample same size, just translate pixels to handles
			//
			double deltaX = (sampleSize.getX()>1)?(getExtent().getX()/(sampleSize.getX()-1)):0;
			double deltaY = (sampleSize.getY()>1)?(getExtent().getY()/(sampleSize.getY()-1)):0;
			double deltaZ = (sampleSize.getZ()>1)?(getExtent().getZ()/(sampleSize.getZ()-1)):0;
			double ox = getOrigin().getX();
			double oy = getOrigin().getY();
			double oz = getOrigin().getZ();
			int handleIndex = 0;
			for (int k = 0; k < sampleSize.getZ(); k++){
				double coordZ = oz + deltaZ*k;
				for (int j = 0; j < sampleSize.getY(); j++){
					double coordY = oy + deltaY*j;
					for (int i = 0; i < sampleSize.getX(); i++){
						double coordX = ox + deltaX*i;
						int imageIndex = getImageIndex(coordX,coordY,coordZ);
						handles[handleIndex++] = handleLookupTable[0xff&(int)vciPixels[imageIndex]];
					}
				}
			}
		}
	}
	
	//
	// go through AnalyticSubVolumes to overlay over ImageSubVolumes
	//
	int displayIndex = 0;
	//double deltaX = (sampleSize.getX()>1)?(getExtent().getX()/(sampleSize.getX()-1)):0;
	//double deltaY = (sampleSize.getY()>1)?(getExtent().getY()/(sampleSize.getY()-1)):0;
	//double deltaZ = (sampleSize.getZ()>1)?(getExtent().getZ()/(sampleSize.getZ()-1)):0;
	double ox = getOrigin().getX();
	double oy = getOrigin().getY();
	double oz = getOrigin().getZ();
	int numX = sampleSize.getX();
	int numY = sampleSize.getY();
	int numZ = sampleSize.getZ();
	
	//
	// rebind x,y,z,t and set Index values
	//
	Enumeration<SubVolume> enumASV = getAnalyticOrCSGSubVolumes();
	while (enumASV.hasMoreElements()){
		SubVolume sv = enumASV.nextElement();
		if (sv instanceof AnalyticSubVolume){
			((AnalyticSubVolume)sv).rebind();
		}
	}

	if (getNumAnalyticOrCSGSubVolumes()>0){
		for (int k=0;k<sampleSize.getZ();k++){
			double unit_z = (numZ>1) ? ((double)k)/(numZ-1):0.5;
			double coordZ = (numZ>1) ? oz + extent.getZ() * unit_z : 0;
			for (int j=0;j<sampleSize.getY();j++){
				double unit_y = (numY>1) ? ((double)j)/(numY-1):0.5;
				double coordY = (numY>1) ? oy + extent.getY() * unit_y : 0;
				for (int i=0;i<sampleSize.getX();i++){
					double unit_x = (numX>1)?((double)i)/(numX-1):0.5;
					double coordX = ox + extent.getX() * unit_x;
					SubVolume subVolume = getAnalyticOrCSGSubVolume(coordX,coordY,coordZ);
					if (subVolume!=null){
						handles[displayIndex] = (byte)subVolume.getHandle();
					} else {
						AnalyticSubVolume backgroundSubVol = new AnalyticSubVolume(getFreeSubVolumeName("background"), new Expression(1.0));
						try {
							this.addAnalyticSubVolumeOrCSGObject(backgroundSubVol, false);
						} catch (PropertyVetoException e) {
							lg.error(e);
							throw new GeometryException("Issue adding a background subvolume to geometry : " + e.getMessage());
						}
						handles[displayIndex] = (byte)backgroundSubVol.getHandle();
					}
					displayIndex++;
				}
			}
		}
	}

	return new VCImageUncompressed(null,handles,getExtent(), sampleSize.getX(),sampleSize.getY(),sampleSize.getZ());
}

/**
 * Insert the method's description here.
 * Creation date: (8/8/00 5:17:17 PM)
 * @return boolean
 * @param curve cbit.vcell.geometry.Curve
 */
private boolean curveSatisfyGeometryConstraints(Curve curve) {
	int master = -1;
	for (double i = 0; i < 1.0; i += .01) {
		double x = curve.getX(i); 
		double y = curve.getY(i); 
		double z = curve.getZ(i); 
		int handle = -1; 
		try {
			handle = getSubVolume(x, y, z).getHandle();
		} catch (Throwable e) {
			throw new RuntimeException(e.toString());
		}
		if (i == 0) {
			master = handle;
		}
		if (handle != master) {
			return false; 
		}
	}
	return true;
}


/**
 * The firePropertyChange method was generated to support the propertyChange field.
 */
public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
	getPropertyChange().firePropertyChange(propertyName, oldValue, newValue);
}


/**
 * The fireVetoableChange method was generated to support the vetoPropertyChange field.
 */
public void fireVetoableChange(java.lang.String propertyName, java.lang.Object oldValue, java.lang.Object newValue) throws java.beans.PropertyVetoException {
	getVetoPropertyChange().fireVetoableChange(propertyName, oldValue, newValue);
}


/**
 * This method was created by a SmartGuide.
 * @return int
 * @param x double
 * @param y double
 */
private SubVolume getAnalyticOrCSGSubVolume(double x, double y, double z) throws GeometryException, ImageException, ExpressionException {
	for (int i = 0; i< subVolumes.length; i++){
		SubVolume subVolume = subVolumes[i];
		if (subVolume instanceof AnalyticSubVolume || subVolume instanceof CSGObject){
			if (subVolume.isInside(x,y,z,this)){
				return subVolume;
			}
		}
	}
	return null;
}

/**
 * This method was created by a SmartGuide.
 * @return int[]
 */
public Enumeration<SubVolume> getAnalyticOrCSGSubVolumes() {
	Vector<SubVolume> analSubVolList = new Vector<SubVolume>();
	for (int i = 0; i< subVolumes.length; i++){
		SubVolume sv = subVolumes[i];
		if (sv instanceof AnalyticSubVolume || sv instanceof CSGObject){
			analSubVolList.addElement(sv);
		}
	}
	return analSubVolList.elements();
}


/**
 * This method was created in VisualAge.
 * @return cbit.image.VCImage
 */
public ISize getDefaultSampledImageSize() {

	//
	// determine size of sampled image
	//
	ISize sampleSize = null;
	if (getImage()!=null){
		
		// if image exists, then use that as the sampled image size
		
		sampleSize = new ISize(getImage().getNumX(),getImage().getNumY(),getImage().getNumZ());
	}else{
		//
		// if no image, then use that as the sampled image size
		//
		int dim = getDimension();
		switch (dim){
			case 0:{
				sampleSize = new ISize(1,1,1);
				break;
			}
			case 1:{
				int total = 50;	
				sampleSize = calulateResetSamplingSize(dim, getExtent(), total);
				break;
			}
			case 2:{
				long max2D = 101*101;
				sampleSize = calulateResetSamplingSize(dim, extent, max2D);
				break;
			}
			case 3:{
				long max3D = 101*101*101;
				sampleSize = calulateResetSamplingSize(dim, extent, max3D);
				break;
			}
		}
	}

	return sampleSize;
}

public static ISize calulateResetSamplingSize(int dim, Extent extent, long total) {
	long numX = 1;
	long numY = 1;
	long numZ = 1;
	
	switch (dim){
		case 1:{
			numX = total;
			numY = 1;
			numZ = 1;
			break;
		}
		case 2:{
			//
			// choose so that the aspect ratio is correct
			//
			//    so x * y = total    and x/y = extentX/extentY = xyRatio
			//
			//    thus  y = sqrt(total/xyRatio);
			//          x = total/y;
			//
			double aspectRatio = extent.getX()/extent.getY();
			numY = Math.max(3, Math.round(Math.sqrt(total/aspectRatio)));
			numX = Math.max(3, Math.round(total/numY));
			numZ = 1;
			break;
		}
		case 3:{
			//
			// choose so that the aspect ratio is correct
			//
			//       x * y * z = total    
			//           z / x = extentZ/extentX = zxRatio
			//           z / y = extentZ/extentY = zyRatio
			//
			//    thus  z = pow(total*zxRatio*zyRatio,1/3)
			//          x = z/zxRatio;
			//          y = z/zyRatio;
			//
			double aspectRatioZX = extent.getZ()/extent.getX();
			double aspectRatioZY = extent.getZ()/extent.getY();
			numZ = Math.max(3, Math.round(Math.pow(total*aspectRatioZX*aspectRatioZY,1.0/3.0)));
			numX = Math.max(3, Math.round(numZ/aspectRatioZX));
			numY = Math.max(3, Math.round(numZ/aspectRatioZY));
			break;
		}
	}
	
	ISize samplingSize = new ISize((int)numX,(int)numY,(int)numZ);
	return samplingSize;
} 

/**
 * This method was created by a SmartGuide.
 * @return int
 */
public int getDimension() {
	return dimension;
}


/**
 * This method was created by a SmartGuide.
 * @return double
 */
public Extent getExtent() {
	if (vcImage!=null){
		return vcImage.getExtent();
	}else{
		return extent;
	}
}	


public FilamentGroup getFilamentGroup() {
	if(getDimension() == 0){
		throw new RuntimeException("FilamentGroup undefined for 0 dimesnion geometry");
	}
	return filamentGroup;
}


private int getFreeSubVolumeHandle() {
	int count=0;
	while (getSubVolume(count)!=null){
		count++;
	}
	return count;
}


public String getFreeSubVolumeName() {
	return getFreeSubVolumeName("subdomain");
}


private String getFreeSubVolumeName(String featureName) {
	int count=0;
	while (getSubVolume(featureName+count)!=null){
		count++;
	}
	return featureName+count;
}


public VCImage getImage() {
	return vcImage;
}


/**
 * This method was created in VisualAge.
 * @return int
 * @param x double
 * @param y double
 * @param z double
 */
int getImageIndex(double x, double y, double z) throws GeometryException {
	if (vcImage==null){
		throw new GeometryException("image not defined in geometry");
	}
	int i = (int)(((vcImage.getNumX()-1)*(x-getOrigin().getX())/getExtent().getX())+0.5);
	int j = (int)(((vcImage.getNumY()-1)*(y-getOrigin().getY())/getExtent().getY())+0.5);
	int k = (int)(((vcImage.getNumZ()-1)*(z-getOrigin().getZ())/getExtent().getZ())+0.5);
	int index = i + j*vcImage.getNumX() + k*vcImage.getNumX()*vcImage.getNumY();
	return index;
}


public ImageSubVolume getImageSubVolumeFromPixelValue(int pixelValue) {
	if (getImage() == null){
		throw new RuntimeException("Geometry doesn't have an image");
	}
	for (int i = 0; i< subVolumes.length; i++){
		SubVolume subVolume = subVolumes[i];
		if (subVolume instanceof ImageSubVolume){
			ImageSubVolume isv = (ImageSubVolume)subVolume;
			if (isv.getPixelValue()==pixelValue){
				return isv;
			}
		}
	}
	return null;
}


/**
 * This method was created in VisualAge.
 * @return int
 */
public int getNumAnalyticOrCSGSubVolumes() {
	int count = 0;
	for (int i = 0; i< subVolumes.length; i++){
		if (subVolumes[i] instanceof AnalyticSubVolume || subVolumes[i] instanceof CSGObject){
			count++;
		}
	}
	return count;
}


/**
 * This method was created in VisualAge.
 * @return int
 */
public int getNumSubVolumes() {
	return subVolumes.length;
}


/**
 * This method was created by a SmartGuide.
 * @return double
 */
public Origin getOrigin() {
	return origin;
}


/**
 * Accessor for the propertyChange field.
 */
protected java.beans.PropertyChangeSupport getPropertyChange() {
	if (propertyChange == null) {
		propertyChange = new java.beans.PropertyChangeSupport(this);
	};
	return propertyChange;
}

public State<VCImage> getSampledImage() {
	if (sampledImage==null){
		sampledImage = new State<VCImage>(null);
		sampledImage.setDirty();
	}
	return sampledImage;
}

public State<ThumbnailImage> getThumbnailImage() {
	if (thumbnailImage==null){
		thumbnailImage = new State<ThumbnailImage>(null);
		thumbnailImage.setDirty();
	}
	return thumbnailImage;
}
/**
 * This method was created in VisualAge.
 * @return cbit.image.VCImage
 */
void updateSampledImage(GeometryThumbnailImageFactory geometryThumbnailImageFactory) throws GeometryException, ImageException, ExpressionException {

	if (getSampledImage().isDirty()){
		ISize sampleSize = getDefaultSampledImageSize();
		getSampledImage().setValue(createSampledImage(sampleSize));
		getThumbnailImage().setValue(createThumbnailImage(geometryThumbnailImageFactory));
	}
}

void fireAll() {
	firePropertyChange(PROPERTY_NAME_SAMPLED_IMAGE, getSampledImage().getOldValue(), getSampledImage().getCurrentValue());
	firePropertyChange(PROPERTY_NAME_THUMBNAIL_IMAGE, getThumbnailImage().getOldValue(), getThumbnailImage().getCurrentValue());
}

private ThumbnailImage createThumbnailImage(GeometryThumbnailImageFactory geometryThumbnailImageFactory) throws ImageException {
	return geometryThumbnailImageFactory.getThumbnailImage(this);
}

/**
 * This method was created by a SmartGuide.
 * @return int
 * @param x double
 * @param y double
 */
public SubVolume getSubVolume(double x, double y, double z) throws GeometryException, ImageException, ExpressionException {
	for (int i = 0; i< subVolumes.length; i++){
		if (subVolumes[i].isInside(x,y,z,this)){
			return subVolumes[i];
		}
	}
	if (vcImage!=null){
		int index = getImageIndex(x,y,z);
		//
		// get pixel value
		//
		int pixel = 0xff&(int)getUncompressedPixels()[index];
		System.out.println("there was no subVolume defined for pixel value <"+pixel+"> at imageIndex("+index+"), coord=("+x+","+y+","+z+")");
	}
	return null;	
}

public SubVolume getSubVolume(int handle) {
	for (int i = 0; i< subVolumes.length; i++){
		if (subVolumes[i].getHandle() == handle){
			return subVolumes[i];
		}
	}
	return null;
}


public SubVolume getSubVolume(String name) {
	for (int i = 0; i< subVolumes.length; i++){
		if (subVolumes[i].getName().equals(name)){
			return subVolumes[i];
		}
	}
	return null;
}


public int getSubVolumeIndex(SubVolume subVolume) {
	for (int i = 0; i< subVolumes.length; i++){
		if (subVolumes[i] == subVolume){
			return i;
		}
	}
	return -1;
}


/**
 * Gets the subVolumes property (cbit.vcell.geometry.SubVolume[]) value.
 * @return The subVolumes property value.
 * @see #setSubVolumes
 */
public SubVolume[] getSubVolumes() {
	return subVolumes;
}


/**
 * Gets the subVolumes index property (cbit.vcell.geometry.SubVolume) value.
 * @return The subVolumes property value.
 * @param index The index value into the property array.
 * @see #setSubVolumes
 */
public SubVolume getSubVolumes(int index) {
	return getSubVolumes()[index];
}


/**
 * This method was created in VisualAge.
 * @return byte[]
 */
byte[] getUncompressedPixels() throws cbit.image.ImageException {
	if (uncompressedPixels==null){
		if (getImage()!=null){
			uncompressedPixels = getImage().getPixels();
		}
	}
	return uncompressedPixels;
}


/**
 * Accessor for the vetoPropertyChange field.
 */
protected java.beans.VetoableChangeSupport getVetoPropertyChange() {
	if (vetoPropertyChange == null) {
		vetoPropertyChange = new java.beans.VetoableChangeSupport(this);
	};
	return vetoPropertyChange;
}

public boolean hasImage() {
	return (getImage() != null);
}


public boolean isCurveValid(Curve curve) {
	if (!curve.isValid()) {
		return false;
	}
	if (!curve.isInside(getOrigin(), getExtent(), new Coordinate(0, 0, 0))) {
		return false;
	}
	if (!curveSatisfyGeometryConstraints(curve)) {
		return false;
	}
	return true;
}

public void propertyChange(java.beans.PropertyChangeEvent event) {
	if (event.getSource() == this && event.getPropertyName().equals("subVolumes")) {
		SubVolume oldSubVolumes[] = (SubVolume[])event.getOldValue();
		SubVolume newSubVolumes[] = (SubVolume[])event.getNewValue();
		if (!Compare.isEqualStrict(oldSubVolumes,newSubVolumes)){  // ignore if just a change of instances
			getSampledImage().setDirty();
			getThumbnailImage().setDirty();
		}
	}
	if (event.getSource() == this && (event.getPropertyName().equals("extent") || event.getPropertyName().equals("origin"))){
		Matchable oldExtentOrOrigin = (Matchable)event.getOldValue();
		Matchable newExtentOrOrigin = (Matchable)event.getNewValue();
		if (!Compare.isEqual(oldExtentOrOrigin,newExtentOrOrigin)){
			getSampledImage().setDirty();
			getThumbnailImage().setDirty();
		}
	}
	if (event.getSource() instanceof AnalyticSubVolume && event.getPropertyName().equals("expression")) {
		Expression oldExpression = (Expression)event.getOldValue();
		Expression newExpression = (Expression)event.getNewValue();
		if (!Compare.isEqual(oldExpression,newExpression)) {
			getSampledImage().setDirty();
			getThumbnailImage().setDirty();
		}
	}
	if (event.getSource() instanceof CSGObject && event.getPropertyName().equals(CSGObject.PROPERTY_NAME_ROOT)) {
		getSampledImage().setDirty();
		getThumbnailImage().setDirty();		
	}
}

/**
 * This method was created in VisualAge.
 */
public void refreshDependencies() {
	//
	// Geometry Listens to itself
	//
	removePropertyChangeListener(this);
	removeVetoableChangeListener(this);
	addPropertyChangeListener(this);
	addVetoableChangeListener(this);

	//
	// Geometry Listens to it's subVolumes
	//
	for (int i = 0; i< subVolumes.length; i++){
		SubVolume sv = subVolumes[i];
		sv.removePropertyChangeListener(this);
		sv.removeVetoableChangeListener(this);
		sv.addPropertyChangeListener(this);
		sv.addVetoableChangeListener(this);
		if (sv instanceof AnalyticSubVolume){
			try {
				((AnalyticSubVolume)sv).rebind();
			}catch (ExpressionException e){
				lg.error(e);
			}
		}
	}
}


/**
 * This method was created in VisualAge.
 * @param subVolume cbit.vcell.geometry.SubVolume
 */
public void removeAnalyticSubVolume(AnalyticSubVolume subVolume) throws PropertyVetoException {
	removeAnalyticSubVolumeOrCSGObject(subVolume);
}

private void removeAnalyticSubVolumeOrCSGObject(SubVolume subVolume) throws PropertyVetoException {
	if (!(subVolume instanceof AnalyticSubVolume || subVolume instanceof CSGObject)) {
		throw new IllegalArgumentException("Only AnalyticSubVolume or CSGObject can deleted");
	}
	
	int subVolumeIndex = getSubVolumeIndex(subVolume);
	if (subVolumeIndex == -1){
		throw new IllegalArgumentException("subdomain "+subVolume+" cannot be removed, it doesn't belong to this Geometry");
	}

	SubVolume newArray[] = new SubVolume[subVolumes.length-1];
	
	int newIndex = 0;
	for (int i = 0; i< subVolumes.length; i++){
		if (i != subVolumeIndex){
			newArray[newIndex++] = subVolumes[i];
		}
	}

	setSubVolumes(newArray);
}

public void removeCSGObject(CSGObject csgObject) throws PropertyVetoException {
	removeAnalyticSubVolumeOrCSGObject(csgObject);
}

/**
 * The removePropertyChangeListener method was generated to support the propertyChange field.
 */
public synchronized void removePropertyChangeListener(java.beans.PropertyChangeListener listener) {
	PropertyChangeListenerProxyVCell.removeProxyListener(getPropertyChange(), listener);
	getPropertyChange().removePropertyChangeListener(listener);
}


/**
 * The removeVetoableChangeListener method was generated to support the vetoPropertyChange field.
 */
public synchronized void removeVetoableChangeListener(java.beans.VetoableChangeListener listener) {
	getVetoPropertyChange().removeVetoableChangeListener(listener);
}


/**
 * This method was created in VisualAge.
 * @param subVolume cbit.vcell.geometry.SubVolume
 */
public void sendBackward(AnalyticSubVolume subVolume) throws PropertyVetoException {
	sendBackwardAnalyticSubVolumeOrCSGObject(subVolume);
}

public void sendBackward(CSGObject subVolume) throws PropertyVetoException {
	sendBackwardAnalyticSubVolumeOrCSGObject(subVolume);
}

private void sendBackwardAnalyticSubVolumeOrCSGObject(SubVolume subVolume) throws PropertyVetoException {
	if (!(subVolume instanceof AnalyticSubVolume || subVolume instanceof CSGObject)) {
		throw new IllegalArgumentException("Only AnalyticSubVolume or CSGObject can be moved back");
	}
	//
	// copy existing array
	//
	SubVolume newArray[] = (SubVolume[]) subVolumes.clone();
	
	//
	// find index of subVolume in newArray
	//
	int index = -1;
	for (int i=0;i<newArray.length;i++){
		if (newArray[i] == subVolume){
			index = i;
		}
	}

	if (index == -1){
		throw new IllegalArgumentException("subdomain "+subVolume+" not found");
	}

	//
	// if not already the last AnalyticSubVolume or CSGObject, then swap with neighbor at index+1
	//
	if ((index < (newArray.length-1)) && (newArray[index+1] instanceof AnalyticSubVolume || newArray[index+1] instanceof CSGObject)){
		SubVolume tempSubVolume = newArray[index+1];
		newArray[index+1] = subVolume;
		newArray[index] = tempSubVolume;

		setSubVolumes(newArray);
	}
}


public void setExtent(Extent aExtent) throws PropertyVetoException {
	if (!this.extent.compareEqual(aExtent)){
		Extent oldExtent = this.extent;
		fireVetoableChange("extent",oldExtent,extent);
		
		//
		// if image exists, then synchronize the image sizes
		//
		if (vcImage != null){
			vcImage.setExtent(aExtent);
		}
		
		this.extent = aExtent;

		firePropertyChange("extent",oldExtent,getExtent());
	}

	return;
}


public void setImage(VCImage image) throws PropertyVetoException {
	
	VCImage oldImage = this.vcImage;
	fireVetoableChange("image",oldImage,image);
	//
	//This used to be a PRIVATE method.  It was made public to allow the DbGeomDriver to save Images on the fly.
	//
	
	if (this.vcImage != image){
		uncompressedPixels = null;
		if (image!=null){
			setExtent(image.getExtent());
			if (image.getNumY()==1 && image.getNumZ()==1){
				dimension = 1;
			}else if (image.getNumZ()==1){
				dimension = 2;
			}else{
				dimension = 3;
			}
		}
		this.vcImage = image;

		SubVolume newImageSubVolumes[] = new SubVolume[vcImage.getNumPixelClasses()];
		int svCount = 0;
		cbit.image.VCPixelClass vcPixelClasses[] = vcImage.getPixelClasses();
		for (int i = 0; i < vcPixelClasses.length; i++){
			ImageSubVolume isv = new ImageSubVolume(null,vcPixelClasses[i],svCount);
			//
			// if this is just refreshing the list of subvolumes, make sure previous subvolume names and handles are preserved.
			//
			if (getSubVolumes()!=null && getSubVolumes().length>i && getSubVolumes().length == vcPixelClasses.length){
				isv.setName(getSubVolumes()[i].getName());
				isv.setHandle(getSubVolumes()[i].getHandle());
				if(isv.compareEqual(getSubVolumes()[i])){
					isv = (ImageSubVolume)getSubVolumes()[i];
					isv.setPixelClass(vcPixelClasses[i]);
				}
			}else{
				isv.setHandle(svCount);
			}
			newImageSubVolumes[svCount++] = isv;
		}
		//
		//merge existing analytic subvolumes with the new image subvolumes
		//
		SubVolume[] allSubVolumes = new SubVolume[newImageSubVolumes.length+getNumAnalyticOrCSGSubVolumes()];
		Enumeration<SubVolume> analyticOrCSGSubVolumeEnum = getAnalyticOrCSGSubVolumes();
		svCount = 0;
		TreeSet<Integer> analyticSubVolHandlesTreeSet = new TreeSet<Integer>();
		while(analyticOrCSGSubVolumeEnum.hasMoreElements()){
			allSubVolumes[svCount] = analyticOrCSGSubVolumeEnum.nextElement();
			analyticSubVolHandlesTreeSet.add(allSubVolumes[svCount].getHandle());
			svCount++;
		}
		for (int i = 0; i < newImageSubVolumes.length; i++) {
			allSubVolumes[svCount] = newImageSubVolumes[i];
			if(analyticSubVolHandlesTreeSet.contains(allSubVolumes[svCount].getHandle())){
				throw new RuntimeException("Duplicate Subvolume handles found while setting new Image");
			}
			svCount++;
		}
		
		setSubVolumes(allSubVolumes);
	}
	firePropertyChange("image",oldImage,image);
}

public void setOrigin(Origin aOrigin) {
	if (!this.origin.compareEqual(aOrigin)){
		Origin oldOrigin = this.origin;
		this.origin = aOrigin;
		firePropertyChange("origin",oldOrigin,origin);
	}
	return;
}

public void setSubVolumes(SubVolume[] subVolumes) throws java.beans.PropertyVetoException {
	SubVolume[] oldSubVolumes = this.subVolumes;
	//Check instance change
//	if(!BeanUtils.checkFullyEqual(oldSubVolumes, subVolumes)){
//		System.out.println("GeometrySpec.setSubVolumes has different instances of same old and new subvolumes");
//	}
	fireVetoableChange("subVolumes", oldSubVolumes, subVolumes);
	this.subVolumes = subVolumes;
	for (int i=0;i<oldSubVolumes.length;i++){
		oldSubVolumes[i].removePropertyChangeListener(this);
		oldSubVolumes[i].removeVetoableChangeListener(this);
	}
	for (int i=0;i<subVolumes.length;i++){
		this.subVolumes[i].addPropertyChangeListener(this);
		this.subVolumes[i].addVetoableChangeListener(this);
	}
	firePropertyChange("subVolumes", oldSubVolumes, subVolumes);
}

public void gatherIssues(IssueContext issueContext, Geometry geometry,List<Issue> issueVector) {
	//
	// from SimulationContext, expecting issueSource to be a GeometryContext
	// from a MathModel, expecting issueSource to be a geometry.
	//
	if (getDimension() > 0) {
		VCImage argSampledImage = getSampledImage().getCurrentValue();
		
		if (argSampledImage == null) {
			return;
		}
		try {
			//
			// make sure that each sample was assigned to a SubVolume
			//
			int count = 0;
			byte samples[] = argSampledImage.getPixels();
			for (int i = 0; i < samples.length; i++){
				if (samples[i] == -1){
					count++;
				}
			}
			if (count>0){
				String errorMessage = "Invalid Geometry - " + count + " of "+samples.length + " samples of geometry domain didn't map to any subdomain";
				Issue issue = new Issue(geometry, issueContext, IssueCategory.SubVolumeVerificationError, errorMessage, Issue.SEVERITY_ERROR);
				issueVector.add(issue);
			}
			//
			// make sure that each subvolume is resolved in the geometry
			//
			ArrayList<SubVolume> missingSubVolumeList = new ArrayList<SubVolume>();
			SubVolume subVolumes[] = getSubVolumes();
			for (int i=0;i<subVolumes.length;i++){
				if (argSampledImage.getPixelClassFromPixelValue(subVolumes[i].getHandle())==null){
					missingSubVolumeList.add(subVolumes[i]);
				}
			}
			if (missingSubVolumeList.size()>0){
				for (int i = 0; i < missingSubVolumeList.size(); i++){
					String errorMessage = "Subdomain '" + missingSubVolumeList.get(i).getName() + "' is not resolved in geometry domain";
					Issue issue = new Issue(geometry, issueContext, IssueCategory.SubVolumeVerificationError, errorMessage, Issue.SEVERITY_ERROR);
					issueVector.add(issue);
				}
			}	
		} catch (Exception ex) {
			Issue issue = new Issue(geometry, issueContext, IssueCategory.SubVolumeVerificationError, ex.getMessage(), Issue.SEVERITY_ERROR);
			issueVector.add(issue);
		}
	}
}

/**
 * Insert the method's description here.
 * Creation date: (6/3/00 9:58:08 AM)
 * @param event java.beans.PropertyChangeEvent
 */
public void vetoableChange(java.beans.PropertyChangeEvent event) throws PropertyVetoException {

	if (event.getSource() == this && event.getPropertyName().equals("image")){
		if (event.getNewValue() != null){
			VCImage newVCImage = (VCImage)event.getNewValue();
			if (newVCImage.getNumXYZ() > IMAGE_SIZE_LIMIT){
				//throw new PropertyVetoException("image size "+newVCImage.getNumXYZ()+" pixels exceeded limit of "+IMAGE_SIZE_LIMIT,event);
				if (lg.isWarnEnabled()) {
					lg.warn("WARNING: image size "+newVCImage.getNumXYZ()+" pixels exceeded limit of "+IMAGE_SIZE_LIMIT);				
				}
			}
		}
	}
	
	if (event.getSource() == this && event.getPropertyName().equals("subVolumes")){
		SubVolume subVolumes[] = (SubVolume[])event.getNewValue();
		

		//
		// add subvolumes 
		// (handles must be unique and non-negative, and analytic subvolumes must be first in array)
		//
		boolean bFoundImageSubVolume = false;
		for (int i=0;i<subVolumes.length;i++){
			SubVolume sv = subVolumes[i];
			//
			// verify that handles are non-negative
			//
			if (sv.getHandle()<0){
				throw new PropertyVetoException("subdomain handle="+sv.getHandle()+" must be non-negative",event);
			}
			
			if (sv.getName() == null || sv.getName().length() < 1){
				throw new PropertyVetoException("Subdomain name cannot be null or blank : ",event);
			}

			//
			// verify that handles and names are unique
			//
			for (int j=i+1;j<subVolumes.length;j++){
				if (subVolumes[j].getHandle() == subVolumes[i].getHandle()){
					throw new PropertyVetoException("subdomain handle="+sv.getHandle()+" has already been used in geometry",event);
				}
				if (subVolumes[j].getName().equals(subVolumes[i].getName())){
					throw new PropertyVetoException("subdomain name="+sv.getName()+" has already been used in geometry",event);
				}
			}
			//
			// verify that all analytic subVolumes come first in array
			//
			if (sv instanceof ImageSubVolume){
				bFoundImageSubVolume = true;
			}else if (sv instanceof AnalyticSubVolume){
				if (bFoundImageSubVolume){
					throw new PropertyVetoException("subdomains are out of order, all analytic subdomains must preceed all image subdomains",event);
				}
			}
		}
		
		//
		// check for applicability (regarding dimension)
		//

		//
		// check for applicability (regarding images)
		// image must be set first in constructor
		//
		for (int i=0;i<subVolumes.length;i++){
			if (subVolumes[i] instanceof ImageSubVolume){
				if (vcImage == null){
					throw new PropertyVetoException("adding image subdomain, an image must be set first",event);
				}
			}
		}
		//
		// check for uniqueness of name and handles 
		// check for handle values of (0..N-1)
		//
	}else if (event.getSource() == this && event.getPropertyName().equals("extent")){
		Extent newExtent = (Extent)event.getNewValue();
		if (newExtent==null){
			throw new PropertyVetoException("extent cannot be null",event);
		}
		if (newExtent.getX()<=0 || newExtent.getY()<=0.0 || newExtent.getZ()<=0){
			throw new PropertyVetoException("extent must be positive in X,Y,Z",event);
		}
	}else if (event.getSource() instanceof SubVolume && event.getPropertyName().equals("name")){
		String newName = (String)event.getNewValue();
		if (newName==null || newName.length()<1){
			throw new PropertyVetoException("subdomain name must be at least one character",event);
		}
		//
		// check if name already used
		//
		SubVolume sv = getSubVolume(newName);
		if (sv != null && sv != event.getSource()){
			throw new PropertyVetoException("A subdomain with name '" + newName + "' already exists.", event);
		}
	}
}


/**
 * fire {@link #PROPERTY_NAME_GEOMETRY_NAME} event if names different
 * @param oldName 
 * @param newName 
 */
public void geometryNameChanged(String oldName, String newName) {
	if (!StringUtils.equals(oldName, newName)) {
		firePropertyChange(PROPERTY_NAME_GEOMETRY_NAME, oldName, newName);
	}
}

}

