package org.vcell.wizard;

import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Hashtable;

import cbit.vcell.client.task.AsynchClientTask;
import cbit.vcell.client.task.ClientTaskDispatcher;
/**
 * Copyright 2008, 2010 Oracle and/or its affiliates. All rights reserved. Use is subject to license terms.
Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
 

    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    * Neither the name of Oracle Corporation nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.

 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 
You acknowledge that this software is not designed, licensed or intended for use in the design, construction, operation or maintenance of any nuclear facility. 
 */
/**
 * This class is responsible for reacting to events generated by pushing any of the
 * three buttons, 'Next', 'back', and 'Cancel.' Based on what button is pressed,
 * the controller will update the model to show a new panel and reset the state of
 * the buttons as necessary.
 */
/**
 * Modified by Tracy Li for VCell usage.
 */
public class WizardController implements ActionListener {
    
    private Wizard wizard;
    
    /**
     * This constructor accepts a reference to the Wizard component that created it,
     * which it uses to update the button components and access the WizardModel.
?     */    
    public WizardController(Wizard w) {
        wizard = w;
    }

    /**
     * Calling method for the action listener. This class listens for actions
     * performed by the buttons in the Wizard class, and calls methods below to determine
     * the action.
     */    
    public void actionPerformed(java.awt.event.ActionEvent evt) {
        
        if (evt.getActionCommand().equals(Wizard.CANCEL_BUTTON_ACTION_COMMAND))
            cancelButtonPressed();
        else if (evt.getActionCommand().equals(Wizard.BACK_BUTTON_ACTION_COMMAND))
            backButtonPressed();
        else if (evt.getActionCommand().equals(Wizard.NEXT_BUTTON_ACTION_COMMAND))
            nextButtonPressed();
        
    }
    
    private void cancelButtonPressed() 
    {
    	ArrayList<AsynchClientTask> totalTasks = new ArrayList<AsynchClientTask>();
    	//add pre tasks
    	WizardPanelDescriptor descriptor = wizard.getModel().getCurrentPanelDescriptor();
    	totalTasks.addAll(descriptor.preCancelProcess());
    	//add close panel task
    	AsynchClientTask aTask1 = new AsynchClientTask("", AsynchClientTask.TASKTYPE_SWING_BLOCKING) 
		{
			public void run(Hashtable<String, Object> hashTable) throws Exception
			{
				try
				{
					wizard.close(Wizard.CANCEL_RETURN_CODE);
				}catch (Exception e){
					e.printStackTrace(System.out);
				}
			}
		};
        totalTasks.add(aTask1);
        //add post tasks
        totalTasks.addAll(descriptor.postCancelProcess());
        //dispatch tasks
        dispatchTasks(totalTasks, descriptor);
    }

    private void nextButtonPressed() {
    	ArrayList<AsynchClientTask> totalTasks = new ArrayList<AsynchClientTask>();
    	//add pre tasks
        WizardModel model = wizard.getModel();
        WizardPanelDescriptor descriptor = model.getCurrentPanelDescriptor();
        totalTasks.addAll(descriptor.preNextProcess());
        
        //add display panel task(if next button pressed) or close task(if finish button pressed)
        final String nextPanelDescriptorID = descriptor.getNextPanelDescriptorID();
        AsynchClientTask aTask1 = new AsynchClientTask("", AsynchClientTask.TASKTYPE_SWING_BLOCKING) 
		{
			public void run(Hashtable<String, Object> hashTable) throws Exception
			{
				try
				{
			        if (nextPanelDescriptorID.equals(Wizard.FINISH.getPanelDescriptorIdentifier())) {
			            wizard.close(Wizard.FINISH_RETURN_CODE);
			        } else {        
			            wizard.setCurrentPanel(nextPanelDescriptorID);
			        }
				}catch (Exception e){
					e.printStackTrace(System.out);
				}
			}
		};
        totalTasks.add(aTask1);
        //add post tasks
        totalTasks.addAll(descriptor.postNextProcess());
        //dispatch tasks
        dispatchTasks(totalTasks, descriptor);
    }

	private void dispatchTasks(ArrayList<AsynchClientTask> totalTasks, WizardPanelDescriptor descriptor) {
		AsynchClientTask[] taskArray = new AsynchClientTask[totalTasks.size()];
        if(descriptor.isProgressPopupShown())
        {
        	ClientTaskDispatcher.dispatch(wizard.getDialog(), new Hashtable<String, Object>(), totalTasks.toArray(taskArray), descriptor.isTaskProgressKnown(), false, null, true);
        }
        else
        {
        	ClientTaskDispatcher.dispatch(wizard.getDialog(), new Hashtable<String, Object>(), totalTasks.toArray(taskArray));
        }
	}

    private void backButtonPressed() {
    	ArrayList<AsynchClientTask> totalTasks = new ArrayList<AsynchClientTask>();
    	//add pre tasks
        WizardModel model = wizard.getModel();
        WizardPanelDescriptor descriptor = model.getCurrentPanelDescriptor();
        totalTasks.addAll(descriptor.preBackProcess());
        //add display panel task(if next descriptor is not null)
        final String backPanelDescriptorID = descriptor.getBackPanelDescriptorID();
        AsynchClientTask aTask1 = new AsynchClientTask("", AsynchClientTask.TASKTYPE_SWING_BLOCKING) 
		{
			public void run(Hashtable<String, Object> hashTable) throws Exception
			{
				try
				{
					if(backPanelDescriptorID != null)
			        {
			        	wizard.setCurrentPanel(backPanelDescriptorID);
			        }
				}catch (Exception e){
					e.printStackTrace(System.out);
				}
			}
		};
        totalTasks.add(aTask1);
        //add post tasks
        totalTasks.addAll(descriptor.postBackProcess());
        //dispatch tasks
        dispatchTasks(totalTasks, descriptor);
    }

    
    void resetButtonsToPanelRules() {
    
        //  Reset the buttons to support the original panel rules,
        //  including whether the next or back buttons are enabled or
        //  disabled, or if the panel is finished.
        WizardModel model = wizard.getModel();
        WizardPanelDescriptor descriptor = model.getCurrentPanelDescriptor();
        
        model.setCancelButtonText(Wizard.CANCEL_TEXT);
        
        //  If the panel has another panel behind it, enable
        //  the back button. Otherwise, disable it.
        model.setBackButtonText(Wizard.BACK_TEXT);
        if (descriptor.getBackPanelDescriptorID() != null)
            model.setBackButtonEnabled(Boolean.TRUE);
        else
            model.setBackButtonEnabled(Boolean.FALSE);

        //  If the panel has one or more panels in front of it,
        //  enable the next button. Otherwise, disable it.
        if (descriptor.getNextPanelDescriptorID() != null)
            model.setNextFinishButtonEnabled(Boolean.TRUE);
        else
            model.setNextFinishButtonEnabled(Boolean.FALSE);
 
        //  If the panel is the last panel in the wizard, change
        //  the Next button to Finish. Otherwise, set the text back to Next.
        if (descriptor.getNextPanelDescriptorID().equals(Wizard.FINISH.getPanelDescriptorIdentifier())) {
            model.setNextFinishButtonText(Wizard.FINISH_TEXT);
        } else {
            model.setNextFinishButtonText(Wizard.NEXT_TEXT);
        }
    }
    
}
