package org.vcell.model.rbm.gui;

import java.awt.Component;
import java.awt.Container;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.beans.PropertyChangeListener;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.Border;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import cbit.vcell.client.BioModelWindowManager;
import cbit.vcell.client.RequestManager;
import cbit.vcell.client.desktop.DocumentWindow;
import cbit.vcell.client.desktop.biomodel.ApplicationSpecificationsPanel;
import cbit.vcell.client.desktop.biomodel.BioModelEditor;
import cbit.vcell.client.desktop.biomodel.IssueManager;
import cbit.vcell.client.desktop.biomodel.SelectionManager;
import cbit.vcell.client.desktop.biomodel.SelectionManager.ActiveViewID;
import cbit.vcell.client.desktop.biomodel.SimulationConsolePanel;
import cbit.vcell.mapping.SimulationContext;
import cbit.vcell.model.Model;
import cbit.vcell.model.Model.RbmModelContainer;
import cbit.vcell.model.SpeciesContext;
import cbit.vcell.solvers.ApplicationMessage;
import org.vcell.util.gui.GeneralGuiUtils;

// we should use WindowBuilder Plugin (add it to Eclipse IDE) to speed up panel design
// can choose absolute layout and place everything exactly as we see fit
@SuppressWarnings("serial")
public class NetworkFreePanel extends JPanel implements ApplicationSpecificationsPanel.Specifier {

    private EventHandler eventHandler = new EventHandler();
    private SimulationContext fieldSimulationContext;
    private IssueManager fieldIssueManager;
    private SelectionManager fieldSelectionManager;

    private JLabel speciesLabel;
    private JLabel futureSpeciesLabel;
    private JLabel speciesMoleculesLabel;
    private JLabel reactionsLabel;
    private JLabel rulesLabel;
    private JLabel futureRulesLabel;
    private JLabel molecularTypesLabel;
    private JLabel futureMolecularTypesLabel;
    private JLabel observablesLabel;
    private JLabel futureObservablesLabel;
    private JLabel estimatedMemoryLabel;        // keep it, at times we may want to bring back in use the mem estimator
    private JLabel rateWarningLabel;

    private static String infoText = "<html>"
            + "A rule-based  model is generated by conversion of  "
            + "existing Molecules, Reaction Rules, Species consisting of Molecules and Observables to the new BioModel.<br> "
            + "For every Species that had no molecular details, a new Molecule and a new Observable will be generated, "
            + "and every ordinary Reaction will be converted to Reaction Rule. "
            + "</html>";

    private JButton createModelButton;

    private class EventHandler implements FocusListener, ActionListener, PropertyChangeListener {

        public void actionPerformed(ActionEvent e) {
            if (e.getSource() == getCreateModelButton()) {
                createModel();
            }
        }

        public void focusGained(FocusEvent e) {
        }

        public void focusLost(FocusEvent e) {
        }

        public void propertyChange(java.beans.PropertyChangeEvent event) {
            if (event.getSource() instanceof Model && event.getPropertyName().equals(RbmModelContainer.PROPERTY_NAME_MOLECULAR_TYPE_LIST)) {
                refreshInterface();
            } else if (event.getSource() instanceof Model && event.getPropertyName().equals(RbmModelContainer.PROPERTY_NAME_OBSERVABLE_LIST)) {
                refreshInterface();
            } else if (event.getSource() instanceof Model && event.getPropertyName().equals(RbmModelContainer.PROPERTY_NAME_REACTION_RULE_LIST)) {
                refreshInterface();
            } else if (event.getSource() instanceof Model && event.getPropertyName().equals(Model.PROPERTY_NAME_SPECIES_CONTEXTS)) {
                if (event.getOldValue() instanceof SpeciesContext[]) {
                    SpeciesContext[] oldValue = (SpeciesContext[]) event.getOldValue();
                    for (SpeciesContext sc : oldValue) {
                        sc.removePropertyChangeListener(eventHandler);
                    }
                }
                if (event.getNewValue() instanceof SpeciesContext[]) {
                    SpeciesContext[] newValue = (SpeciesContext[]) event.getNewValue();
                    for (SpeciesContext sc : newValue) {
                        sc.addPropertyChangeListener(eventHandler);
                    }
                }
                refreshInterface();
            } else if (event.getSource() instanceof Model && event.getPropertyName().equals(Model.PROPERTY_NAME_REACTION_STEPS)) {
                refreshInterface();
            } else if (event.getSource() instanceof SpeciesContext && event.getPropertyName().equals(SpeciesContext.PROPERTY_NAME_SPECIES_PATTERN)) {
                refreshInterface();
            } else {
                System.out.println("NetworkFreePanel: " + event.getSource() + ", " + event.getPropertyName() + "");
            }
        }
    }

    ApplicationMessage getApplicationMessage() {
        return new ApplicationMessage(ApplicationMessage.PROGRESS_MESSAGE, 2, 1, "some progress error", "some progress message");
    }

    public NetworkFreePanel() {
        super();
        initialize();
    }

    @Override
    public ActiveViewID getActiveView() {
        return ActiveViewID.network_free_setting;
    }

    /**
     * no-op
     */
    @Override
    public void setSearchText(String s) {

    }

    private JButton getCreateModelButton() {
        if (createModelButton == null) {
            createModelButton = new javax.swing.JButton(" Create new Rule-Based VCell BioModel ");
            createModelButton.setName("CreateModelButton");
        }
        return createModelButton;
    }

    private void initialize() {
        speciesLabel = new JLabel("");
        speciesMoleculesLabel = new JLabel("");
        futureSpeciesLabel = new JLabel("");
        reactionsLabel = new JLabel("");
        rulesLabel = new JLabel("");
        futureRulesLabel = new JLabel("");
        molecularTypesLabel = new JLabel("");
        futureMolecularTypesLabel = new JLabel("");
        observablesLabel = new JLabel("");
        futureObservablesLabel = new JLabel("");
        rateWarningLabel = new JLabel("");
        estimatedMemoryLabel = new JLabel("");

        Border loweredEtchedBorder = BorderFactory.createEtchedBorder(EtchedBorder.LOWERED);
        Border loweredBevelBorder = BorderFactory.createLoweredBevelBorder();

        JPanel mainPanel = new JPanel();
        mainPanel.setLayout(new GridBagLayout());
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 1;
        gbc.weighty = 1;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.insets = new Insets(5, 3, 2, 1);
        add(mainPanel, gbc);

        JPanel left = new JPanel();
        TitledBorder titleLeft = BorderFactory.createTitledBorder(loweredEtchedBorder, " Original Physiology ");
        titleLeft.setTitleJustification(TitledBorder.LEFT);
        titleLeft.setTitlePosition(TitledBorder.TOP);
        left.setBorder(titleLeft);
        left.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 0.5;
        gbc.weighty = 1.0;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.insets = new Insets(5, 2, 2, 3);    //  top, left, bottom, right
        mainPanel.add(left, gbc);

        JPanel right = new JPanel();
        TitledBorder titleRight = BorderFactory.createTitledBorder(loweredEtchedBorder, " Created Rule-Based Physiology");
        titleRight.setTitleJustification(TitledBorder.LEFT);
        titleRight.setTitlePosition(TitledBorder.TOP);
        right.setBorder(titleRight);
        right.setLayout(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.weightx = 0.5;
        gbc.weighty = 1.0;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.insets = new Insets(5, 2, 2, 3);    //  top, left, bottom, right
        mainPanel.add(right, gbc);

        // ================================================= Populating the left group box ===============
        int gridy = 0;
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 1;
        gbc.weighty = 1;            // fake cell used for filling all the vertical empty space
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(new JLabel(""), gbc);

        gridy++;
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);    //  top, left, bottom, right
        left.add(new JLabel("Species without molecular structure: "), gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(speciesLabel, gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 2;
        gbc.gridy = gridy;
        gbc.weightx = 1.0;
        gbc.gridwidth = 2;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(new JLabel(""), gbc);

        gridy++;    // -----------------------------------
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(new JLabel("Reactions: "), gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(reactionsLabel, gbc);

        gridy++;    // -----------------------------------
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(new JLabel("Species with molecular structure: "), gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(speciesMoleculesLabel, gbc);

        gridy++;    // -----------------------------------
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(new JLabel("Reaction Rules: "), gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(rulesLabel, gbc);

        gridy++;    // -----------------------------------
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(new JLabel("Molecules: "), gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        left.add(molecularTypesLabel, gbc);

        gridy++;    // -----------------------------------
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 10, 10);
        left.add(new JLabel("Observables: "), gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 10, 10);
        left.add(observablesLabel, gbc);

        // ============================================= Populating the right group box =================
        gridy = 0;
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 1;
        gbc.gridwidth = 2;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.anchor = GridBagConstraints.EAST;
        gbc.insets = new Insets(4, 4, 2, 10);
        right.add(new JLabel(infoText), gbc);

        gridy++;
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 1;
        gbc.gridwidth = 2;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.anchor = GridBagConstraints.EAST;
        gbc.insets = new Insets(2, 4, 2, 10);
        right.add(rateWarningLabel, gbc);

        gridy++;
        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
//		gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.anchor = GridBagConstraints.EAST;
        gbc.insets = new Insets(10, 4, 4, 10);        //  top, left, bottom, right
        right.add(getCreateModelButton(), gbc);

        gridy++;
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weighty = 1;            // fake cell used for filling all the vertical empty space
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        right.add(new JLabel(""), gbc);

        gridy++;    // ----------------------------
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);    //  top, left, bottom, right
        right.add(new JLabel("Species with molecular structure: "), gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 14, 4, 10);
        right.add(futureSpeciesLabel, gbc);

        gridy++;    // --------------------------------
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        right.add(new JLabel("Reaction Rules: "), gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 14, 4, 10);
        right.add(futureRulesLabel, gbc);

        gridy++;    // --------------------------------
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 4, 10);
        right.add(new JLabel("Molecules: "), gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 14, 4, 10);
        right.add(futureMolecularTypesLabel, gbc);

        gridy++;    // ---------------------------------
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 4, 10, 10);
        right.add(new JLabel("Observables: "), gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = gridy;
        gbc.weightx = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(4, 14, 10, 10);
        right.add(futureObservablesLabel, gbc);

//		gridy++;	// ---------------------------------
//		gbc = new GridBagConstraints();
//		gbc.gridx = 0;
//		gbc.gridy = gridy;
//		gbc.weightx = 0;
//		gbc.anchor = GridBagConstraints.WEST;
//		gbc.insets = new Insets(4, 4, 4, 10);
//		right.add(new JLabel("Estimated Memory: "), gbc);
//
//		gbc = new GridBagConstraints();
//		gbc.gridx = 1;
//		gbc.gridy = gridy;
//		gbc.weightx = 0;
//		gbc.anchor = GridBagConstraints.WEST;
//		gbc.insets = new Insets(4, 14, 4, 10);
//		right.add(estimatedMemoryLabel, gbc);


        getCreateModelButton().addActionListener(eventHandler);
        addFocusListener(eventHandler);
    }

    public void setSimulationContext(SimulationContext simulationContext) {
        if (simulationContext == null) {
            return;
        }
        if (fieldSimulationContext != null) {
            fieldSimulationContext.removePropertyChangeListener(eventHandler);
        }
        fieldSimulationContext = simulationContext;
        fieldSimulationContext.addPropertyChangeListener(eventHandler);

        Model m = fieldSimulationContext.getModel();
        if (m != null) {
            m.removePropertyChangeListener(eventHandler);
            m.addPropertyChangeListener(eventHandler);
            for (SpeciesContext sc : m.getSpeciesContexts()) {
                sc.removePropertyChangeListener(eventHandler);
                sc.addPropertyChangeListener(eventHandler);
            }
        }
        refreshInterface();
    }

    public SimulationContext getSimulationContext() {
        return fieldSimulationContext;
    }

    public void setSelectionManager(SelectionManager selectionManager) {
        fieldSelectionManager = selectionManager;
    }

    public void setIssueManager(IssueManager issueManager) {
        fieldIssueManager = issueManager;
    }

    public void refreshInterface() {
        int s1 = 0;
        int s2 = 0;
        for (SpeciesContext sc : fieldSimulationContext.getModel().getSpeciesContexts()) {
            if (sc.hasSpeciesPattern()) {
                s1++;
            } else {
                s2++;
            }
        }
        int s3 = fieldSimulationContext.getModel().getNumSpeciesContexts();

        int r1 = fieldSimulationContext.getModel().getNumReactions();
        int r2 = fieldSimulationContext.getModel().getRbmModelContainer().getReactionRuleList().size();

        int m1 = fieldSimulationContext.getModel().getRbmModelContainer().getMolecularTypeList().size();
        int o1 = fieldSimulationContext.getModel().getRbmModelContainer().getObservableList().size();


		/*	
		Map <String, Integer >moleculesSites = new LinkedHashMap<>();	// key = molecular type name, value = adjusted number of states
		for(MolecularType mt : fieldSimulationContext.getModel().getRbmModelContainer().getMolecularTypeList()) {
			int em = 1;		// we assume size one for molecules with no sites
			if(mt.getComponentList().size() > 0) {
				em = mt.getComponentList().size();	// each site has by default size one, even if there are no states defined
				for(MolecularComponent mc : mt.getComponentList()) {
					int numCSD = 0;
					if(mc.getComponentStateDefinitions().size() > 0) {
						numCSD = mc.getComponentStateDefinitions().size();
					}
					em += numCSD;
				}
			}
			// molecule size = 1 + #sites + sum(#states for each site)
			moleculesSites.put(mt.getName(),  em);
		}
		
		Map <String, Double >totals = new LinkedHashMap<>();	// key = molecular type name, value = total number of states
		SpeciesContextSpec[] scsList = fieldSimulationContext.getReactionContext().getSpeciesContextSpecs();
		boolean usingConcentration = true;
		boolean evaluatesToNumber = true;
		for(SpeciesContextSpec scs : scsList) {
			SpeciesContext sc = scs.getSpeciesContext();
			String scName = sc.getName();
			double count = 0;		// number of molecules of this species
			SpeciesContextSpecParameter countParameter = scs.getInitialCountParameter();
			SpeciesContextSpecParameter concentParameter = scs.getInitialConcentrationParameter();
			if(countParameter != null && countParameter.getExpression() != null) {
				try {
//					Expression exp = countParameter.getExpression().flatten();
					count = countParameter.getConstantValue();
//					count = exp.evaluateConstant();
				} catch (ExpressionException e) {
//					e.printStackTrace();
					System.out.println("NetworkFreePanel: Count parameter for species '" + scName + "' is not a constant, defaulting to '1'!");
					evaluatesToNumber = false;
					count = 1;		// we just put something in there
					break;
				}
			} else if(concentParameter != null && concentParameter.getExpression() != null) {
				// TODO: compute num molecules starting from concentration
				// we compute count from concentration
				System.out.println("NetworkFreePanel: Using concentration for species '" + scName + "' is not implemented yet, defaulting to '1'!");
				usingConcentration = false;
				count = 1;
				break;
			} else {
				// don't know what to do, we just initialize to one
				System.out.println("NetworkFreePanel: Unable to compute Count for species '" + scName + "', defaulting to '1'!");
				usingConcentration = false;
				count = 1;
				break;
			}
			
			// for each molecular type, we multiply the number of states with the number of species it's present in
			// note that different species may contain the same molecular type in their species pattern, more than once
			if(!sc.hasSpeciesPattern()) {
				// plain molecular type, with no sites or states, name of the species is the name of the molecule
				int numStates = 1;
				// if we don't have a molecule for this species we just assume a size of 1
				if(moleculesSites.containsKey(scName)) {
					numStates = moleculesSites.get(scName);
				} else {
					System.out.println("NetworkFreePanel: Plain species '" + scName + "' not found in the MolecularTypes map!");
				}
				double totalCount = count * numStates;
				if(totals.containsKey(scName)) {
					totalCount += totals.get(scName);
				}
				totals.put(scName, totalCount);
			} else {	// we have a species pattern, we update the number of states for each molecular type
				SpeciesPattern sp = sc.getSpeciesPattern();
				for(MolecularTypePattern mtp : sp.getMolecularTypePatterns()) {
					int numStates = 1;
					String key = mtp.getMolecularType().getName();
					if(moleculesSites.containsKey(key)) {
						numStates = moleculesSites.get(key);
					} else {
						System.out.println("NetworkFreePanel: MolecularType '" + key + "' not found in the MolecularTypes map!");
					}
					double totalCount = count * numStates;
					if(totals.containsKey(key)) {
						totalCount += totals.get(key);
					}
					totals.put(key, totalCount);
				}
			}
		}
		double totalCount = 0;
		for(Map.Entry<String,Double> entry : totals.entrySet()) {
			String name = entry.getKey();
			Double count = entry.getValue();
			System.out.println(name + ", " + count);
			totalCount += count;
		}
		long rounded = Math.round(1800 * totalCount / new Double(1e6));
		String strRounded = "";
		if(rounded > 1) {
			strRounded = rounded + "";
		} else {
			strRounded = (1800 * totalCount / new Double(1e6)) + "";
		}
		if(usingConcentration == false) {
			estimatedMemoryLabel.setText("'Count' units must be used.");
		} else if(evaluatesToNumber == false) {
			estimatedMemoryLabel.setText("The number of molecules must be a constant.");
		} else {
			estimatedMemoryLabel.setText(strRounded + " MB, (1800 * " + totalCount + " / 1e6)");
		}

		*/

        speciesLabel.setText(s2 + "");
        speciesMoleculesLabel.setText(s1 + "");
        futureSpeciesLabel.setText(s1 + s2 + "");
        reactionsLabel.setText(r1 + "");
        rulesLabel.setText(r2 + "");
        futureRulesLabel.setText((r1 + r2) + "");
        molecularTypesLabel.setText(m1 + "");
        observablesLabel.setText(o1 + "");
        futureMolecularTypesLabel.setText((m1 + s2) + "");
        futureObservablesLabel.setText((o1 + s2) + "");

//		if(s1 > 0) {
        rateWarningLabel.setText("<html><font color=#8C001A>" + SimulationContext.rateWarning + "</font></html>");
//		} else {
//			rateWarningLabel.setText("");
//		}

        createModelButton.setEnabled(true);
    }

    private void createModel() {

        DocumentWindow dw = (DocumentWindow) GeneralGuiUtils.findTypeParentOfComponent(this, DocumentWindow.class);
        BioModelWindowManager bmwm = (BioModelWindowManager) (dw.getTopLevelWindowManager());
        RequestManager rm = dw.getTopLevelWindowManager().getRequestManager();

        rm.createBioModelFromApplication(bmwm, "Test", fieldSimulationContext);
    }

    @Override        // for testing/debugging purposes
    public void setVisible(boolean bVisible) {
        super.setVisible(bVisible);
    }

    public void activateConsole() {
        boolean found = false;
        Container parent = getParent();
        while (parent != null) {
            parent = parent.getParent();
            if (parent instanceof BioModelEditor) {
                found = true;
                break;
            }
        }
        if (found) {
            System.out.println("Parent Found");
            BioModelEditor e = (BioModelEditor) parent;
//			e.getRightBottomTabbedPane().setSelectedComponent(e.getSimulationConsolePanel());
            Component[] cList = e.getRightBottomTabbedPane().getComponents();
            for (Component c : cList) {
                if (c instanceof SimulationConsolePanel) {
                    e.getRightBottomTabbedPane().setSelectedComponent(c);
                    break;
                }
            }
        }
    }
}
