#' ***************************************************************** #
#' ~~ STEP 03 CALC. CLIMATE NOVELTY.
#' ~~ This code calculates fractional novelty of a variable between two time periods.
#' ~~ We use annual bioclimate variables derived from the microclimate model.
#' ~~ You will need a set of bioclimate variables for a baseline period and one for the period of 'novelty'
#' ~~ Code is provided for an example tile.
#' ***************************************************************** #
#' ~~ Code written by B.T Trew on 5th August 2022

#' ~~ Required libraries:
require(terra)
library(sfsmisc)

# Function to calculate novelty:
novel <- function(x1, x2) {
  # define limits of a common grid, adding a buffer so that tails aren't cut off:
  lower <- min(c(x1, x2)) - 1 
  upper <- max(c(x1, x2)) + 1
  # generate kernel densities"
  if(is.na(upper) == FALSE & is.na(lower) == FALSE){
    da <- density(x1, from=lower, to=upper)
    db <- density(x2, from=lower, to=upper)
    d <- data.frame(x=da$x, a=da$y, b=db$y)
    # calculate intersection densities:
    d$w <- pmin(d$a, d$b)
    # integrate areas under curves:
    total <- integrate.xy(d$x, d$a) + integrate.xy(d$x, d$b)
    intersection <- integrate.xy(d$x, d$w)
    # compute overlap coefficient:
    overlap <- 2 * intersection / total
    novelty <- 1 - overlap
  } else { 
    novelty <- NA
  }
  return(novelty)
}

#' ~~ Set up input data folders (examples provided in zip folder).
pathtodata <-"C:/Out/bioclims/"
pathout <- "C:/Out"

#' ~~ Select years of interest to study:
yr_seq <- seq(1990,2019,1)
b1 <- yr_seq[1] # 1990: start of baseline timeframe
b2 <- yr_seq[15] # 2004: end of baseline timeframe
r1 <- yr_seq[16] # 2005: start of recent timeframe
r2 <- yr_seq[30] # 2019: end of recent timeframe

#' ~~ For each bioclimatic variable (eg. 1-19) calculate novelty:
varnames <- c("bio1", "bio2","bio3","bio4","bio5","bio6","bio7")
for (n in 1:length(varnames)){
  var <- varnames[n]
  files <- list.files(path = pathtodata, pattern = paste0(var,"_",".*.tif$"), full.names = T)
  # Baseline timeframe:
  p1 <- rast(files[b1:b2])
  a1 <- as.array(p1)
  # Recent timeframe:
  p2 <- rast(files[r1:r2])
  a2 <-  as.array(p2)
  
  #' Create empty array for novelty results:
  nvl <- array(NA, dim = c(nrow(a1), ncol(a1)))
  #' Loop through each array value using the novelty function:
  for (a in 1:nrow(nvl)){
    for (b in 1:ncol(nvl)){
      print(paste0("row: ", a, "; col: ", b))
      x1 <- a1[a,b,]
      x2 <- a2[a,b,]
      if(is.na(x1[1]) == FALSE){
        nvl[a,b] <- novel(x1,x2)
      }
    }
  } # end loop.
  
  #' convert to raster:
  r <- t(raster(nvl))
  r <- flip(r, direction='y')
  crs(r) <- crs(p1[[1]])
  extent(r) <- extent(p1[[1]])
  writeRaster(r, filename = paste0(pathout,"/Novelty_",var,".tif"), overwrite = TRUE)
}