z#' ***************************************************************** #
#' ~~ STEP 02 MICROCLIMATE MODELLING.
#' ~~ This code (1) processes vegetation, elevation and soil parameters required to drive the microclimate model,
#' ~~ (2) runs the microclimf model @ 5cm above the ground, 
#' ~~ and (3) derives yearly bioclimatic variables from the output.
#' ~~ Please see the README file for web addresses of full open access datasets used. 
#' ~~ Code is provided for an example tile. We do not recommend trying to run this process at large scales.
#' ***************************************************************** #
#' ~~ Code written by B.T Trew on 5th August 2022
#' ~~ Required libraries:
require(microclimf) # https://github.com/ilyamaclean/microclimf
require(terra)
require(abind)
require(elevatr) # https://github.com/ilyamaclean/climvars
require(ncdf4)

#' ~~ Set up input data folders (examples provided in zip folder).
#' ~~ Input data needs to be cropped to your tile template and re projected to a meter projection system (We used EPSG:3857)
pathtodata <-"revisions/high_res_modelling/"
pathtolai <- paste0(pathtodata,"lai/")
pathtohabitat <- paste0(pathtodata,"habitats/")
pathtohgt <- paste0(pathtodata,"hgts/")
out <- paste0(pathtodata,"micro_out/")

tile_no <- "01"
tile <- rast(paste0(pathtodata,"tile_",tile_no,".tif"))

#' Digital elevation model:
d <- elevatr::get_elev_raster(locations = raster::raster(tile), prj = crs(tile), z = 12)

#' Habitat types (Annual time steps):
#' https://www.esa-landcover-cci.org/
#' Note: Habitat type needs to be reclassified to match microclimf habitat classifications 
#' see: ?microclimf::vegpfromhab
hab <- paste0(pathtohabitat,"C3S-LC-L4-LCCS-Map-300m-P1Y-2019-v2.1.1.nc")
hab2 <- raster::brick(hab)
crs_info <- crs(hab2)
ext_info <- ext(hab2)
habnc <- rast(hab)
ext(habnc) <- ext_info
crs(habnc) <- crs_info
habitats <- crop(habnc$lccs_class,tile)

#' reclassify to match microclimf:
m <- c(10,13, 11,7, 12, 6, 
       20, 13, 
       30, 15, 
       40, 15,
       50, 2,
       60, 4, 61, 4,62, 4,
       70,1, 71,1,72,1,
       80,3,81,3,82,3,
       90, 5,
       100, 6,110, 7,
       120, 7, 121, 7,122, 7,
       130, 11,
       140, 16,
       150, 16,151, 16, 152, 16,153, 16,
       160, 12,
       170, 12,
       180, 12, 
       190, 14, 
       200, 16,201, 16, 202, 16,
       210, 12, 
       220, 16)
rclmat <- matrix(m, ncol=2, byrow=TRUE)
hab_rc <- classify(habitats, rclmat)

#' *Optional* Vegetation height (Annual time steps):
#' https://webmap.ornl.gov/ogc/
hgt <- rast(paste0(pathtohgt,"hgt_2019.tif"))
hgt <- crop(hgt,tile)
hgt <- resample(hgt, hab_rc, method = "bilinear")
#hgt <- project(hgt, "EPSG:3395")

#hab_rc <- project(hab_rc, "EPSG:3395", method = "near")

#' *Optional* Leaf Area Index (Monthly time steps):
#' https://www.ncei.noaa.gov/data/avhrr-land-leaf-area-index-and-fapar/
lai_list <- list.files(path = paste0(pathtodata,"lai/tile_",tile_no), pattern = "c_gls_LAI300-LAI_", recursive = T, full.names = T)
lai_stk <- rast(lai_list)
#lai_stk <- project(lai_stk, "EPSG:3395")
lai_stk <- resample(lai_stk, hgt)
yr_stk <- rast()
st <- 1
for(i in 1:12){
    en <- st+2
    mm <- mean(lai_stk[[st:en]])
    yr_stk <- c(yr_stk,mm)
    st <- en+1
}
lai_a <- terra::as.array(yr_stk)

#' *Optional* Calculate PAI (Plant Area Index)
#' Calculate 20% of maximum LAI to account for PAI
pai_multi <- apply(lai_a, c(1,2),FUN = max) * 0.2
#' Add 20% to all LAI values to calculate PAI
months <- seq(from = 1, to = 12, by = 1)
PAI_stack <- list()
for (i in 1:length(months)){
  a <- lai_a[,,i]+pai_multi
  PAI_stack[[i]] <- a
}
PAI <- abind(PAI_stack, along=3)
PAI[is.na(PAI)] <- 0
#dem <- project(rast(d), "EPSG:3395")
dem <- resample(rast(d), hgt)

#' Process vegetation parameters for microclimf:
vegp <- vegpfromhab(hab_rc, hgt, PAI, clump0 = F)

#' Soil Type:
#' #' See <?soilcfromtype> for more details.
#' https://www.soilgrids.org
soil <- hab_rc
soil[] <- 1

#' Note: groundr layer is optional and can be replaced by a single numeric integer.
soilc <- soilcfromtype(soil, 0.15)

#' ~~ Model Microclimate >>>
#' ~~ Read in the parameters processed in '01_ClimateDate.R':
r <- rast(paste0(pathtodata,"era5_processed/tile_",tile_no,"/tile_",tile_no,".tif"))

for(i in 1991:2019){
  year <- i
  
  climdat <- readRDS(paste0(pathtodata,"era5_processed/tile_",tile_no,"/climdat_",year,".RDS"))
  precip <- readRDS(paste0(pathtodata,"era5_processed/tile_",tile_no,"/rainfall_",year,".RDS"))
  
  m1 <- microclimf::runpointmodela(climarray = climdat, precarray = precip, tme = climdat$tme, reqhgt = 0.05,
                                   vegp = vegp, soilc = soilc, windhgt = 2)
  
  micro <- microclimf::runbioclim(weather = climdat, precip = precip, tme = climdat$obs_time, 
                                  reqhgt = 0.05, vegp = vegp, soilc = soilc, 
                                  dtm = dem, dtmc = r, temp = "air",hourly = F, 
                                  altcorrect = 2, windhgt = 2, runchecks = T)
  
  micro_agg <- aggregate(micro$BIO1, 16.7)
  
  outpath <- paste0(out, "tile_", tile_no, "/")
  dir.create(outpath)
  
  writeRaster(micro, paste0(outpath, "/micro_fine_",year,".tif"))
  writeRaster(micro_agg, paste0(outpath, "/micro_coarse_",year,".tif"))
  
  small <- values(micro$BIO1)
  big <- values(micro_agg$BIO1)
  mean(small, na.rm = T)
  mean(big, na.rm = T)
  
}
