#' *****************************************************************
#' ~~ STEP 01 DOWNLOADING & PROCESSING HOURLY CLIMATE DATA
#' ~~ This code processes ERA5 reanalysis climate data ready for use with the microclimf package.
#' ~~ https://cds.climate.copernicus.eu/cdsapp#!/dataset/reanalysis-era5-single-levels
#' ~~ This code runs as a single small tile (We do not recommend running this code at large scales)
#' ~~ For example, split the standard era5 global tiles (eg. tile 14 out of 18 tiles) into ~ 500 smaller tiles. 
#' ~~ Note: This depends largely on the resolution of your vegetation parameters (example is 5km)
#' ~~ We recommend using the mcera5 package to download the hourly climate data: 
#' ~~ https://github.com/dklinges9/mcera5
#' *****************************************************************
#' ~~ Code written by B.T Trew on 5th August 2022

#' ~~ Required libraries:
require(terra)
require(raster)
require(mcera5) # https://github.com/dklinges9/mcera5
require(ncdf4)
require(microclima) # https://github.com/ilyamaclean/microclima
require(microclimf) # https://github.com/ilyamaclean/microclimf
require(ecmwfr)
require(lutz)
require(lubridate)

#' ~~ Set up input data folders (examples provided in zip folder).
pathtodata <-"C:/Data/"
pathtoera5 <- paste0(pathtodata,"era5/")

#' The code runs in annual time steps. Please define a year of interest.
#' Note: You will need to run this for multiple years  for two different time frames to calculate novelty.
year <- 2019

#' Download Hourly Climate Data >>>
#' ERA5 is the fifth generation ECMWF reanalysis for the global climate and weather for the past 8 decades. 
#' Data is available from 1940 onwards. 
#' You will need a registered account to access the data. You can find your UID and API Key in your user profile.
uid <- "your user id here"
cds_api_key <- "your API key here"
ecmwfr::wf_set_key(user = uid,
                   key = cds_api_key,
                   service = "cds")

# Define the spatial extent for your tile:
xmn <- 18.125
xmx <- 22.875
ymn <- -1.625
ymx <- 1.875

# Define the temporal extent of the run:
st_time <- lubridate::ymd(paste0(year,":01:01"))
en_time <- lubridate::ymd(paste0(year,":12:31"))

file_prefix <- paste0("era5_reanalysis_",yr)
req <- build_era5_request(xmin = xmn, xmax = xmx,
                          ymin = ymn, ymax = ymx,
                          start_time = st_time,
                          end_time = en_time,
                          outfile_name = file_prefix)

request_era5(request = req, uid = uid, out_path = pathtoera5)
#' Note: The request will appear in your account online and will take some time to run.

#' Process Hourly Climate Data >>>
file <- paste0(pathtoera5,"era5_reanalysis_",year,".nc")
clim <- nc_open(file)

#' create a template to crop input dataset to for step 2: '02_VegParms.R'
test <- raster::brick(file, varname = "t2m")
t_array <- as.array(test[[1]])
ext_r <- ext(raster::extent(test))
r <- rast(t_array, crs = "EPSG:4326", ext = ext_r)

#' Get coordinates & time:
lons <- ncdf4::ncvar_get(clim, varid = "longitude")
lats <- ncdf4::ncvar_get(clim, varid = "latitude")
time <- ncdf4::ncvar_get(clim, "time")

x_dim <- length(lons)
y_dim <- length(lats)
z_dim <- length(time)

#' Assign a local timezone:
tmz <- lutz::tz_lookup_coords(lats[length(lats)/2], lons[length(lons)/2], method = 'fast')

origin <- as.POSIXlt("1900-01-01 00:00:00", tz = "UTC")
UTC_tme <- origin + as.difftime(time, units = "hours")
UTC_tme <- as.POSIXlt(UTC_tme, tz = "UTC")
local_tme <- lubridate::with_tz(UTC_tme, tzone = tmz)

jd <- microctools::jday(tme = UTC_tme)
lt <- local_tme$hour + local_tme$min/60 + local_tme$sec/3600

#' Create empty climate variable arrays:
#' These are 3-D arrays with time (hours) in the 3rd dimension.
t_a <- array(data = NA, c(y_dim, x_dim, z_dim))
t_sh <- array(data = NA, c(y_dim, x_dim, z_dim))
t_pa <- array(data = NA, c(y_dim, x_dim, z_dim))
t_ws <- array(data = NA, c(y_dim, x_dim, z_dim))
t_wd <- array(data = NA, c(y_dim, x_dim, z_dim))
t_se <- array(data = NA, c(y_dim, x_dim, z_dim))
t_nl <- array(data = NA, c(y_dim, x_dim, z_dim))
t_ul <- array(data = NA, c(y_dim, x_dim, z_dim))
t_dl <- array(data = NA, c(y_dim, x_dim, z_dim))
t_rd <- array(data = NA, c(y_dim, x_dim, z_dim))
t_rdf <- array(data = NA, c(y_dim, x_dim, z_dim))
t_sz <- array(data = NA, c(y_dim, x_dim, z_dim))
p_a <- array(data = NA, c(y_dim, x_dim, length(local_tme)/24)) # note: rainfall recorded daily.

#' Fill empty arrays with processed era5 data:
#' Use the 'extract_clim' function from the mcera5 package to convert era5 data to microclimf-ready data. 
for(i in 1:y_dim){ # for each row in the new array.
  for(j in 1:x_dim){ # for each column in the new array.
    long <- lons[j]
    lat <- lats[i]
    climate <- extract_clim(file, long, lat, start_time = UTC_tme[1], end_time = UTC_tme[length(UTC_tme)])
    t_a[i,j,] <- climate$temperature
    t_sh[i,j,] <- climate$humidity
    t_pa[i,j,] <- climate$pressure
    t_ws[i,j,] <- climate$windspeed
    t_wd[i,j,] <- climate$winddir
    t_se[i,j,] <- climate$emissivity
    t_nl[i,j,] <- climate$netlong
    t_ul[i,j,] <- climate$uplong
    t_dl[i,j,] <- climate$downlong
    t_rd[i,j,] <- climate$rad_dni
    t_rdf[i,j,] <- climate$rad_dif
    t_sz[i,j,] <- climate$szenith
  } # end column j
} # end row i

#' Repeat for daily rainfall:
#' Use the 'extract_precip' function from the mcera5 package to convert era5 data to microclimf-ready data. 
for(i in 1:y_dim){ # for each row in the new array.
  for(j in 1:x_dim){ # for each column in the new array.
    long <- lons[j]
    lat <- lats[i]
    precip <- extract_precip(file, long, lat, start_time = UTC_tme[1], end_time = UTC_tme[length(UTC_tme)])
    p_a[i,j,] <- precip
  } # end column j
} # end row i


#' Additional processing for microclimf inputs:
#' Calculate Solar Index...
si <- array(data = NA, dim = c(y_dim, x_dim, z_dim))
for(a in 1:nrow(si)){
  for(b in 1:ncol(si)){
    x <- lons[b]
    y <- lats[a]
    s = microclima::siflat(lubridate::hour(local_tme), y, x, jd)    
    si[a,b,] <- s
  }
}  
#' Calculate Global Horizontal Irradiance (GHI) from Direct Normal Irradiance (DNI)
#' and convert units from MJh/m^2 to kWh/m^2
raddr <- (t_rd * si)/0.0036
difrad <- t_rdf/0.0036

#' Cap diffuse radiation data (Cannot be less than 0) 
difrad[difrad < 0] <- 0

#' Calculate shortwave radiation:
#' Sum Global Horizontal Irradiance (GHI) and Diffuse Radiation. 
swrad <- raddr + difrad

#' Cap shortwave radiation between 0 > sw < 1350 (lower than the solar constant)
swrad[swrad < 0] <- 0
swrad[swrad > 1350] <- 1350

#' Calculate relative humidity
#' Using specific humidity, temperature and pressure. 
t_rh <- array(data = NA, c(y_dim, x_dim, z_dim))
for(i in 1:nrow(t_rh)){ 
  for(j in 1:ncol(t_rh)){ 
    rh <- microclima::humidityconvert(t_sh[i,j,],intype = "specific", tc = t_a[i,j,], p = t_pa[i,j,])
    rh <- rh$relative
    rh[rh > 100] <- 100
    t_rh[i,j,] <- rh
  } 
} 

#' Convert pressure untis from Pa to kPa:
t_pr <- t_pa/1000


#' Create final climate data set to drive microclimate model:
#' Note: keep the nomenclature as shown here for microclimf, see microclimf::climdat for example names.
climdat <- list(tme = local_tme, obs_time = UTC_tme, 
                temp = t_a, relhum = t_rh, 
                pres = t_pr, swrad = swrad,
                difrad = difrad, skyem = t_se,
                windspeed = t_ws, winddir = t_wd)

#' Save data:
pathout <- "C:/Out/"
saveRDS(climdat, paste0(pathout,"climdat_",year,".RDS"))
saveRDS(p_a, paste0(pathout,"rainfall_",year,".RDS"))
tile_no <- "01"
writeRaster(r, paste0(pathout,"tile_",tile_no,".tif"))



