"""
Last updated on 2024 March 30

@author: Shunya Kaneki (AIST)
"""

import numpy as np
import re as re
import os
import glob
import copy
import pandas as pd
from scipy import stats
import matplotlib.pyplot as plt
import matplotlib.patches as patches


def numericalSort(value):
    numbers = re.compile(r'(\d+)')
    parts = numbers.split(value)
    parts[1::2] = map(int, parts[1::2])
    return parts

def index(list,num):
    out = np.abs(np.asarray(list)-num).argmin()
    return out

def idxmax(list,nbsint,Pran):
    out = np.argmax(nbsint[index(list,Pran[0]):index(list,Pran[1])])
    return out+index(list,Pran[0])

def fwhml(fitsft,fitint,liml,amp,x0):
    tmpsft = fitsft[index(fitsft,liml):index(fitsft,x0)]
    tmpint = fitint[index(fitsft,liml):index(fitsft,x0)]
    return np.abs(tmpsft[index(tmpint,amp/2)]-x0)

def fwhmr(fitsft,fitint,limr,amp,x0):
    tmpsft = fitsft[index(fitsft,x0):index(fitsft,limr)]
    tmpint = fitint[index(fitsft,x0):index(fitsft,limr)]
    return np.abs(tmpsft[index(tmpint,amp/2)]-x0)

def tempR2(R2slice):
    a, b, c = 671.8, -716.9, 393.6
    R2ave = np.average(R2slice)
    T = a+b*R2ave+c*R2ave*R2ave
    n = int(8)
    sigma = 4.209
    x0 = np.array([1,R2ave,R2ave**2]).reshape(-1,1)
    x0T = np.transpose(x0)
    XXinv = np.array([[0.813958,-7.07328,11.4752],[-7.07328,83.4335,-149.253],[11.4752,-149.253,282.54]])
    aaa = np.dot(x0T,XXinv)
    bbb = np.dot(aaa,x0)
    sigmaR2 = np.std(R2slice,ddof=1)/np.sqrt(len(R2slice))
    Tse = np.sqrt((sigma**2)*(1+bbb)+((b+2*c*R2ave)**2)*(sigmaR2**2))
    TerPI95 = stats.t.ppf(1-0.05/2,n-3)*Tse
    return np.c_[np.round(np.average(R2slice),3),np.round(np.std(R2slice),3),np.round(np.std(R2slice,ddof=1)/np.sqrt(len(R2slice)),3),
                 int(np.round(T,0)),int(np.round(Tse,0)),int(np.round(TerPI95,0))]

def Rhist(R1,R2,R1slice,R2slice,mkdir):
    R1bin = np.linspace(0,3,31)
    R2bin = np.linspace(0,1,21)
    
    fig = plt.figure(figsize=(14,7),dpi=200)
    ax1 = fig.add_subplot(1,2,1)
    ax2 = fig.add_subplot(1,2,2)
    
    ax1.hist(R1,color='black',bins=R1bin,weights=np.ones_like(R1)/len(R1))
    ax1.set(xlim=(R1bin[0],R1bin[-1]),ylim=(0,1),
            xticks=(np.linspace(R1bin[0],R1bin[-1],7)),yticks=(np.linspace(0,1,11)),
            xlabel='R1 ratio',ylabel='Frequency')
    ax1.vlines(x=np.round(np.average(R1),3),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax1.add_patch(patches.Rectangle((np.round(np.average(R1),3)-2*np.round(np.std(R1),3),0),4*np.round(np.std(R1),3),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax1.text(0.05,0.9,'mean: {}\nSD: {}'.format(np.round(np.average(R1),3),np.round(np.std(R1),3)),transform=ax1.transAxes)
    
    ax2.hist(R2,color='black',bins=R2bin,weights=np.ones_like(R2)/len(R2))
    ax2.set(xlim=(R2bin[0],R2bin[-1]),ylim=(0,1),
            xticks=(np.linspace(R2bin[0],R2bin[-1],11)),yticks=(np.linspace(0,1,11)),
            xlabel='R2 ratio',ylabel='Frequency')
    ax2.vlines(x=np.round(np.average(R2),3),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax2.add_patch(patches.Rectangle((np.round(np.average(R2),3)-2*np.round(np.std(R2),3),0),4*np.round(np.std(R2),3),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax2.text(0.05,0.9,'mean: {}\nSD: {}'.format(np.round(np.average(R2),3),np.round(np.std(R2),3)),transform=ax2.transAxes)
    
    plt.suptitle('Histogram of R1 and R2 ratios before eliminating outliers',fontsize=20)
    plt.savefig(mkdir.imgdir+'/hist/Rhist_before.pdf')
    plt.savefig(mkdir.imgdir+'/hist/Rhist_before.png')
    plt.close()
    
    fig = plt.figure(figsize=(14,7),dpi=200)
    ax1 = fig.add_subplot(1,2,1)
    ax2 = fig.add_subplot(1,2,2)
    
    ax1.hist(R1slice,color='black',bins=R1bin,weights=np.ones_like(R1slice)/len(R1slice))
    ax1.set(xlim=(R1bin[0],R1bin[-1]),ylim=(0,1),
            xticks=(np.linspace(R1bin[0],R1bin[-1],7)),yticks=(np.linspace(0,1,11)),
            xlabel='R1 ratio',ylabel='Frequency')
    ax1.vlines(x=np.round(np.average(R1slice),3),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax1.add_patch(patches.Rectangle((np.round(np.average(R1slice),3)-2*np.round(np.std(R1slice),3),0),4*np.round(np.std(R1slice),3),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax1.text(0.05,0.9,'mean: {}\nSD: {}'.format(np.round(np.average(R1slice),3),np.round(np.std(R1slice),3)),transform=ax1.transAxes)

    ax2.hist(R2slice,color='black',bins=R2bin,weights=np.ones_like(R2slice)/len(R2slice))
    ax2.set(xlim=(R2bin[0],R2bin[-1]),ylim=(0,1),
            xticks=(np.linspace(R2bin[0],R2bin[-1],11)),yticks=(np.linspace(0,1,11)),
            xlabel='R2 ratio',ylabel='Frequency')
    ax2.vlines(x=np.round(np.average(R2slice),3),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax2.add_patch(patches.Rectangle((np.round(np.average(R2slice),3)-2*np.round(np.std(R2slice),3),0),4*np.round(np.std(R2slice),3),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax2.text(0.05,0.9,'mean: {}\nSD: {}'.format(np.round(np.average(R2slice),3),np.round(np.std(R2slice),3)),transform=ax2.transAxes)

    plt.suptitle('Histogram of R1 and R2 ratios after eliminating outliers',fontsize=20)
    plt.savefig(mkdir.imgdir+'/hist/Rhist_after.pdf')
    plt.savefig(mkdir.imgdir+'/hist/Rhist_after.png')
    plt.close()

def omghist(D1omg,Gomg,D1omgslice,Gomgslice,mkdir):
    D1bin = np.linspace(1320,1380,31)
    Gbin = np.linspace(1570,1600,31)
    
    fig = plt.figure(figsize=(14,7),dpi=200)
    ax1 = fig.add_subplot(1,2,1)
    ax2 = fig.add_subplot(1,2,2)
    
    ax1.hist(D1omg,color='black',bins=D1bin,weights=np.ones_like(D1omg)/len(D1omg))
    ax1.set(xlim=(D1bin[0],D1bin[-1]),ylim=(0,1),
            xticks=(np.linspace(D1bin[0],D1bin[-1],13)),yticks=(np.linspace(0,1,11)),
            xlabel='D1-band center [/cm]',ylabel='Frequency')
    ax1.vlines(x=np.round(np.average(D1omg),1),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax1.add_patch(patches.Rectangle((np.round(np.average(D1omg),1)-2*np.round(np.std(D1omg),1),0),4*np.round(np.std(D1omg),1),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax1.text(0.05,0.9,'mean: {} [/cm]\nSD: {} [/cm]'.format(np.round(np.average(D1omg),1),np.round(np.std(D1omg),1)),transform=ax1.transAxes)
    
    ax2.hist(Gomg,color='black',bins=Gbin,weights=np.ones_like(Gomg)/len(Gomg))
    ax2.set(xlim=(Gbin[0],Gbin[-1]),ylim=(0,1),
            xticks=(np.linspace(Gbin[0],Gbin[-1],7)),yticks=(np.linspace(0,1,11)),
            xlabel='G-band center [/cm]',ylabel='Frequency')
    ax2.vlines(x=np.round(np.average(Gomg),1),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax2.add_patch(patches.Rectangle((np.round(np.average(Gomg),1)-2*np.round(np.std(Gomg),1),0),4*np.round(np.std(Gomg),1),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax2.text(0.05,0.9,'mean: {} [/cm]\nSD: {} [/cm]'.format(np.round(np.average(Gomg),1),np.round(np.std(Gomg),1)),transform=ax2.transAxes)
    
    plt.suptitle('Histogram of center positions of the D1- and G-bands before eliminating outliers',fontsize=20)
    plt.savefig(mkdir.imgdir+'/hist/omghist_before.pdf')
    plt.savefig(mkdir.imgdir+'/hist/omghist_before.png')
    plt.close()
    
    fig = plt.figure(figsize=(14,7),dpi=200)
    ax1 = fig.add_subplot(1,2,1)
    ax2 = fig.add_subplot(1,2,2)
    
    ax1.hist(D1omgslice,color='black',bins=D1bin,weights=np.ones_like(D1omgslice)/len(D1omgslice))
    ax1.set(xlim=(D1bin[0],D1bin[-1]),ylim=(0,1),
            xticks=(np.linspace(D1bin[0],D1bin[-1],13)),yticks=(np.linspace(0,1,11)),
            xlabel='D1-band center [/cm]',ylabel='Frequency')
    ax1.vlines(x=np.round(np.average(D1omgslice),1),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax1.add_patch(patches.Rectangle((np.round(np.average(D1omgslice),1)-2*np.round(np.std(D1omgslice),1),0),4*np.round(np.std(D1omgslice),1),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax1.text(0.05,0.9,'mean: {} [/cm]\nSD: {} [/cm]'.format(np.round(np.average(D1omgslice),1),np.round(np.std(D1omgslice),1)),transform=ax1.transAxes)

    ax2.hist(Gomgslice,color='black',bins=Gbin,weights=np.ones_like(Gomgslice)/len(Gomgslice))
    ax2.set(xlim=(Gbin[0],Gbin[-1]),ylim=(0,1),
            xticks=(np.linspace(Gbin[0],Gbin[-1],7)),yticks=(np.linspace(0,1,11)),
            xlabel='G-band center [/cm]',ylabel='Frequency')
    ax2.vlines(x=np.round(np.average(Gomgslice),1),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax2.add_patch(patches.Rectangle((np.round(np.average(Gomgslice),1)-2*np.round(np.std(Gomgslice),1),0),4*np.round(np.std(Gomgslice),1),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax2.text(0.05,0.9,'mean: {} [/cm]\nSD: {} [/cm]'.format(np.round(np.average(Gomgslice),1),np.round(np.std(Gomgslice),1)),transform=ax2.transAxes)

    plt.suptitle('Histogram of center positions of the D1- and G-bands after eliminating outliers',fontsize=20)
    plt.savefig(mkdir.imgdir+'/hist/omghist_after.pdf')
    plt.savefig(mkdir.imgdir+'/hist/omghist_after.png')
    plt.close()

def FWHMhist(D1fwhm,Gfwhm,D1fwhmslice,Gfwhmslice,mkdir):
    D1bin = np.linspace(0,100,21)
    Gbin = np.linspace(0,70,29)
    
    fig = plt.figure(figsize=(14,7),dpi=200)
    ax1 = fig.add_subplot(1,2,1)
    ax2 = fig.add_subplot(1,2,2)
    
    ax1.hist(D1fwhm,color='black',bins=D1bin,weights=np.ones_like(D1fwhm)/len(D1fwhm))
    ax1.set(xlim=(D1bin[0],D1bin[-1]),ylim=(0,1),
            xticks=(np.linspace(D1bin[0],D1bin[-1],11)),yticks=(np.linspace(0,1,11)),
            xlabel='D1-FWHM [/cm]',ylabel='Frequency')
    ax1.vlines(x=np.round(np.average(D1fwhm),1),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax1.add_patch(patches.Rectangle((np.round(np.average(D1fwhm),1)-2*np.round(np.std(D1fwhm),1),0),4*np.round(np.std(D1fwhm),1),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax1.text(0.05,0.9,'mean: {} [/cm]\nSD: {} [/cm]'.format(np.round(np.average(D1fwhm),1),np.round(np.std(D1fwhm),1)),transform=ax1.transAxes)
    
    ax2.hist(Gfwhm,color='black',bins=Gbin,weights=np.ones_like(Gfwhm)/len(Gfwhm))
    ax2.set(xlim=(Gbin[0],Gbin[-1]),ylim=(0,1),
            xticks=(np.linspace(Gbin[0],Gbin[-1],15)),yticks=(np.linspace(0,1,11)),
            xlabel='G-FWHM [/cm]',ylabel='Frequency')
    ax2.vlines(x=np.round(np.average(Gfwhm),1),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax2.add_patch(patches.Rectangle((np.round(np.average(Gfwhm),1)-2*np.round(np.std(Gfwhm),1),0),4*np.round(np.std(Gfwhm),1),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax2.text(0.05,0.9,'mean: {} [/cm]\nSD: {} [/cm]'.format(np.round(np.average(Gfwhm),1),np.round(np.std(Gfwhm),1)),transform=ax2.transAxes)
    
    plt.suptitle('Histogram of FWHM of the D1- and G-bands before eliminating outliers',fontsize=20)
    plt.savefig(mkdir.imgdir+'/hist/FWHMhist_before.pdf')
    plt.savefig(mkdir.imgdir+'/hist/FWHMhist_before.png')
    plt.close()
    
    fig = plt.figure(figsize=(14,7),dpi=200)
    ax1 = fig.add_subplot(1,2,1)
    ax2 = fig.add_subplot(1,2,2)
    
    ax1.hist(D1fwhmslice,color='black',bins=D1bin,weights=np.ones_like(D1fwhmslice)/len(D1fwhmslice))
    ax1.set(xlim=(D1bin[0],D1bin[-1]),ylim=(0,1),
            xticks=(np.linspace(D1bin[0],D1bin[-1],11)),yticks=(np.linspace(0,1,11)),
            xlabel='D1-FWHM [/cm]',ylabel='Frequency')
    ax1.vlines(x=np.round(np.average(D1fwhmslice),1),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax1.add_patch(patches.Rectangle((np.round(np.average(D1fwhmslice),1)-2*np.round(np.std(D1fwhmslice),1),0),4*np.round(np.std(D1fwhmslice),1),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax1.text(0.05,0.9,'mean: {} [/cm]\nSD: {} [/cm]'.format(np.round(np.average(D1fwhmslice),1),np.round(np.std(D1fwhmslice),1)),transform=ax1.transAxes)

    ax2.hist(Gfwhmslice,color='black',bins=Gbin,weights=np.ones_like(Gfwhmslice)/len(Gfwhmslice))
    ax2.set(xlim=(Gbin[0],Gbin[-1]),ylim=(0,1),
            xticks=(np.linspace(Gbin[0],Gbin[-1],15)),yticks=(np.linspace(0,1,11)),
            xlabel='G-FWHM [/cm]',ylabel='Frequency')
    ax2.vlines(x=np.round(np.average(Gfwhmslice),1),ymin=0,ymax=1,linestyle='dashed',color='black',lw=1)
    ax2.add_patch(patches.Rectangle((np.round(np.average(Gfwhmslice),1)-2*np.round(np.std(Gfwhmslice),1),0),4*np.round(np.std(Gfwhmslice),1),1,fc='gray',fill=True,alpha=0.5,zorder=0))
    ax2.text(0.05,0.9,'mean: {} [/cm]\nSD: {} [/cm]'.format(np.round(np.average(Gfwhmslice),1),np.round(np.std(Gfwhmslice),1)),transform=ax2.transAxes)

    plt.suptitle('Histogram of FWHM of the D1- and G-bands after eliminating outliers',fontsize=20)
    plt.savefig(mkdir.imgdir+'/hist/FWHMhist_after.pdf')
    plt.savefig(mkdir.imgdir+'/hist/FWHMhist_after.png')
    plt.close()

class dload():
    def __init__(self, file,dtype):
        if dtype=='txt':
            dat = np.loadtxt(file)
        elif dtype=='csv' or dtype=='CSV':
            dat = np.loadtxt(file,delimiter=',')
        else:
            print('Warning: Data type MUST be either .txt or .csv (.CSV).')
        
        if dat[-1,0]<1750 or dat[0,0]>1100:
            print('Warning: Raman shift does not cover the range 1100-1750 [/cm].')
        else:
            pass
        
        self.dat = dat

class mkdir():
    def __init__(self):
        if not os.path.exists('./result'):
            os.makedirs('./result')
            os.makedirs('./result/spec')
            os.makedirs('./result/image')
            os.makedirs('./result/param')
            os.makedirs('./result/image/fit')
            os.makedirs('./result/image/hist')
            os.makedirs('./result/param/initial')
            os.makedirs('./result/param/final')
        else:
            pass
        
        self.specdir = './result/spec'
        self.prmdir = './result/param'
        self.imgdir = './result/image'

class analysis():
    def __init__(self, mkdir):
        smpl, R1, R2, D1omg, Gomg, D1fwhm, Gfwhm = [], [], [], [], [], [], []
        prmdir = mkdir.prmdir+'/final/'
        files = sorted(glob.glob(os.path.join(prmdir,'*.csv')),key=numericalSort)
        i = int(0)
        for file in files:
            if i==0:
                prmold = pd.read_csv(file)
                ID = str(file.lstrip(mkdir.prmdir+'/final/').rstrip('_fin.csv'))
                smpl.append(ID)
                prm = prmold.rename(index={0:ID})
                prmsum = copy.copy(prm)
                R1 = np.append(R1,prm['R1'].values[0])
                R2 = np.append(R2,prm['R2'].values[0])
                D1omg = np.append(D1omg,prm['D1x0'].values[0])
                Gomg = np.append(Gomg,prm['Gx0'].values[0])
                D1fwhm = np.append(D1fwhm,prm['D1fwhm'].values[0])
                Gfwhm = np.append(Gfwhm,prm['Gfwhm'].values[0])
                i += 1
            else:
                prmold = pd.read_csv(file)
                ID = str(file.lstrip(mkdir.prmdir+'/final/').rstrip('_fin.csv'))
                smpl.append(ID)
                prm = prmold.rename(index={0:ID})
                prmsum = pd.concat([prmsum,prm])
                R1 = np.append(R1,prm['R1'].values[0])
                R2 = np.append(R2,prm['R2'].values[0])
                D1omg = np.append(D1omg,prm['D1x0'].values[0])
                Gomg = np.append(Gomg,prm['Gx0'].values[0])
                D1fwhm = np.append(D1fwhm,prm['D1fwhm'].values[0])
                Gfwhm = np.append(Gfwhm,prm['Gfwhm'].values[0])
        
        prmsum.to_csv('./result/param/all_fin.csv')
        
        R1sig = np.std(R1)
        R2sig = np.std(R2)
        D1omgsig = np.std(D1omg)
        Gomgsig = np.std(Gomg)
        D1fwhmsig = np.std(D1fwhm)
        Gfwhmsig = np.std(Gfwhm)
        R1ave = np.average(R1)
        R2ave = np.average(R2)
        D1omgave = np.average(D1omg)
        Gomgave = np.average(Gomg)
        D1fwhmave = np.average(D1fwhm)
        Gfwhmave = np.average(Gfwhm)

        R1slice = np.delete(R1,np.where(np.abs(R1-R1ave)>2*R1sig))
        R2slice = np.delete(R2,np.where(np.abs(R2-R2ave)>2*R2sig))
        D1omgslice = np.delete(D1omg,np.where(np.abs(D1omg-D1omgave)>2*D1omgsig))
        Gomgslice = np.delete(Gomg,np.where(np.abs(Gomg-Gomgave)>2*Gomgsig))
        D1fwhmslice = np.delete(D1fwhm,np.where(np.abs(D1fwhm-D1fwhmave)>2*D1fwhmsig))
        Gfwhmslice = np.delete(Gfwhm,np.where(np.abs(Gfwhm-Gfwhmave)>2*Gfwhmsig))
        
        parR1 = np.c_[np.round(np.average(R1slice),3),np.round(np.std(R1slice),3),
                      np.round(np.std(R1slice,ddof=1)/np.sqrt(len(R1slice)),3),len(R1slice)]
        parR2 = np.c_[np.round(np.average(R2slice),3),np.round(np.std(R2slice),3),
                      np.round(np.std(R2slice,ddof=1)/np.sqrt(len(R2slice)),3),len(R2slice)]
        parD1omg = np.c_[np.round(np.average(D1omgslice),1),np.round(np.std(D1omgslice),1),
                         np.round(np.std(D1omgslice,ddof=1)/np.sqrt(len(D1omgslice)),1),len(D1omgslice)]
        parGomg = np.c_[np.round(np.average(Gomgslice),1),np.round(np.std(Gomgslice),1),
                        np.round(np.std(Gomgslice,ddof=1)/np.sqrt(len(Gomgslice)),1),len(Gomgslice)]
        parD1fwhm = np.c_[np.round(np.average(D1fwhmslice),1),np.round(np.std(D1fwhmslice),1),
                          np.round(np.std(D1fwhmslice,ddof=1)/np.sqrt(len(D1fwhmslice)),1),len(D1fwhmslice)]
        parGfwhm = np.c_[np.round(np.average(Gfwhmslice),1),np.round(np.std(Gfwhmslice),1),
                         np.round(np.std(Gfwhmslice,ddof=1)/np.sqrt(len(Gfwhmslice)),1),len(Gfwhmslice)]
        header = ['mean','SD','SE','No']
        index = ['R1ratio','R2ratio','D1omg[/cm]','Gomg[/cm]','D1fwhm[/cm]','Gfwhm[/cm]']

        df = pd.DataFrame(np.vstack((parR1,parR2,parD1omg,parGomg,parD1fwhm,parGfwhm)),index=index,columns=header)
        df.to_csv('./result/param.csv')

        df = pd.DataFrame(tempR2(R2slice),index=['sample'],columns=['R2ave','R2SD','R2SE','Tave[°C]','Tse[°C]','TerPI95[°C]'])
        df.to_csv('./result/temp.csv')
        
        Rhist(R1,R2,R1slice,R2slice,mkdir)
        omghist(D1omg,Gomg,D1omgslice,Gomgslice,mkdir)
        FWHMhist(D1fwhm,Gfwhm,D1fwhmslice,Gfwhmslice,mkdir)
        
        print('\nTemperature: '+str(tempR2(R2slice)[0][3])+' degree C')
        
        if tempR2(R2slice)[0][0]>0.57:
            print('\nWarning: CM maturity may be too low! Better try the code of Kaneki & Kouketsu (2022)!')
        else:
            pass