"""
Last updated on 2023 August 31

@author: Shunya Kaneki (AIST)
"""

import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import curve_fit
from utilmod import index as idx
from utilmod import fwhml, fwhmr

def lorentz(x,x0,fwhm,A):
    gam = fwhm/2
    return A/(np.pi*gam*(1+((x-x0)/gam)**2))
def gauss(x,x0,fwhm,A):
    sig = fwhm/2/np.sqrt(2*np.log(2))
    return A/np.sqrt(2*np.pi*(sig**2))*np.exp(-((x-x0)**2)/(2*(sig**2)))
def pv(x,x0,fwhm,eta,A):
    return eta*lorentz(x,x0,fwhm,A)+(1-eta)*gauss(x,x0,fwhm,A)

class fit():
    def __init__(self, sft,nbint):
        
        tol = 1e-8
        
        D1ran, Gran, D2ran, D3ran = [1320,1380], [1570,1600], [1610,1630], [1500,1540]
        fac = 0.9
        
        self.fitsft = sft[idx(sft,1100):idx(sft,1750)]
        self.fitint = nbint[idx(sft,1100):idx(sft,1750)]
        
        self.D1ampini = float(self.fitint[self.fitint==self.fitint[idx(self.fitsft,D1ran[0]):idx(self.fitsft,D1ran[1])].max()])
        self.D1x0ini = float(self.fitsft[idx(self.fitsft,D1ran[0]):idx(self.fitsft,D1ran[1])][self.fitint[idx(self.fitsft,D1ran[0]):idx(self.fitsft,D1ran[1])]==self.D1ampini])
        if self.D1ampini>=0:
            self.D1fwhmini = fwhml(self.fitsft,self.fitint,1200,self.D1ampini,self.D1x0ini)+fwhmr(self.fitsft,self.fitint,1500,self.D1ampini,self.D1x0ini)
        else:
            self.D1ampini = 0
            self.D1fwhmini = 10
        if self.D1fwhmini == 0:
            self.D1fwhmini = 10
        else:
            pass
        self.D1etaini = 1
        self.D1Aini = (self.D1ampini*self.D1fwhmini)/(2/np.pi*self.D1etaini+2*(1-self.D1etaini)*np.sqrt(np.log(2)/np.pi))
        
        self.Gampini = float(self.fitint[self.fitint==self.fitint[idx(self.fitsft,Gran[0]):idx(self.fitsft,Gran[1])].max()])
        self.Gx0ini = float(self.fitsft[idx(self.fitsft,Gran[0]):idx(self.fitsft,Gran[1])][self.fitint[idx(self.fitsft,Gran[0]):idx(self.fitsft,Gran[1])]==self.Gampini])
        if self.Gampini>=0:
            self.Gfwhmini = fwhml(self.fitsft,self.fitint,1500,self.Gampini,self.Gx0ini)*2
        else:
            self.Gampini = 0
            self.Gfwhmini = 10
        if self.Gfwhmini == 0:
            self.Gfwhmini = 10
        else:
            pass
        self.Getaini = 1
        self.GAini = (self.Gampini*self.Gfwhmini)/(2/np.pi*self.Getaini+2*(1-self.Getaini)*np.sqrt(np.log(2)/np.pi))
        
        self.D1intini = pv(self.fitsft,self.D1x0ini,self.D1fwhmini,self.D1etaini,self.D1Aini)
        self.Gintini = pv(self.fitsft,self.Gx0ini,self.Gfwhmini,self.Getaini,self.GAini)
        
        self.owint = self.fitint-self.D1intini-self.Gintini
        
        self.D2ampini = float(self.owint[self.owint==self.owint[idx(self.fitsft,D2ran[0]):idx(self.fitsft,D2ran[1])].max()])
        self.D2x0ini = float(self.fitsft[idx(self.fitsft,D2ran[0]):idx(self.fitsft,D2ran[1])][self.owint[idx(self.fitsft,D2ran[0]):idx(self.fitsft,D2ran[1])]==self.D2ampini])
        if self.D2ampini>=0:
            self.D2fwhmini = fwhmr(self.fitsft,self.owint,1700,self.D2ampini,self.D2x0ini)*2
        else:
            self.D2ampini = 0
            self.D2fwhmini = 10
        if self.D2fwhmini == 0:
            self.D2fwhmini = 10
        else:
            pass
        self.D2etaini = 0.5
        self.D2Aini = (fac*self.D2ampini*self.D2fwhmini)/(2/np.pi*self.D2etaini+2*(1-self.D2etaini)*np.sqrt(np.log(2)/np.pi))
        
        self.D3ampini = float(self.owint[self.owint==self.owint[idx(self.fitsft,D3ran[0]):idx(self.fitsft,D3ran[1])].max()])
        self.D3x0ini = float(self.fitsft[idx(self.fitsft,D3ran[0]):idx(self.fitsft,D3ran[1])][self.owint[idx(self.fitsft,D3ran[0]):idx(self.fitsft,D3ran[1])]==self.D3ampini])
        if self.D3ampini>=0:
            pass
        else:
            self.D3ampini = 0
        self.D3fwhmini = 130
        self.D3etaini = 0.5
        self.D3Aini = (fac*self.D3ampini*self.D3fwhmini)/(2/np.pi*self.D3etaini+2*(1-self.D3etaini)*np.sqrt(np.log(2)/np.pi))
        
        self.D2intini = pv(self.fitsft,self.D2x0ini,self.D2fwhmini,self.D2etaini,self.D2Aini)
        self.D3intini = pv(self.fitsft,self.D3x0ini,self.D3fwhmini,self.D3etaini,self.D3Aini)
        
        self.fitini = self.D1intini+self.Gintini+self.D2intini+self.D3intini
        self.resini = self.fitint-self.fitini
        
        self.R1ini = self.D1ampini/self.Gampini
        self.R2ini = self.D1Aini/(self.GAini+self.D1Aini+self.D2Aini)
        
        self.r2ini = 1-(((self.fitini-self.fitint)**2).sum())/(((self.fitint-np.average(self.fitint))**2).sum())
        
        self.prmini = np.array([self.D1x0ini,self.D1fwhmini,self.D1etaini,self.D1Aini,
                                self.Gx0ini,self.Gfwhmini,self.Getaini,self.GAini,
                                self.D2x0ini,self.D2fwhmini,self.D2etaini,self.D2Aini,
                                self.D3x0ini,self.D3fwhmini,self.D3etaini,self.D3Aini])
        
        def func(x,*prm):
            D1int = pv(x,prm[0],prm[1],prm[2],prm[3])
            Gint = pv(x,prm[4],prm[5],prm[6],prm[7])
            D2int = pv(x,prm[8],prm[9],prm[10],prm[11])
            D3int = pv(x,prm[12],prm[13],prm[14],prm[15])
            return D1int+Gint+D2int+D3int
        
        prmlim = ((1300,0,0,0,1550,0,0,0,1600,0,0,0,1450,0,0,0),
                  (1400,np.inf,1,np.inf,1600,np.inf,1,np.inf,1650,np.inf,1,np.inf,1550,np.inf,1,np.inf))
        
        self.prm, self.cov = curve_fit(func,self.fitsft,self.fitint,p0=self.prmini,method='trf',
                                       bounds=prmlim,ftol=tol,xtol=tol,gtol=tol,maxfev=10000)
        
        self.D1x0 = self.prm[0]
        self.D1fwhm = self.prm[1]
        self.D1eta = self.prm[2]
        self.D1A = self.prm[3]
        self.D1amp = self.D1A/self.D1fwhm*(2/np.pi*self.D1eta+2*(1-self.D1eta)*np.sqrt(np.log(2)/np.pi))
        
        self.Gx0 = self.prm[4]
        self.Gfwhm = self.prm[5]
        self.Geta = self.prm[6]
        self.GA = self.prm[7]
        self.Gamp = self.GA/self.Gfwhm*(2/np.pi*self.Geta+2*(1-self.Geta)*np.sqrt(np.log(2)/np.pi))
        
        self.D2x0 = self.prm[8]
        self.D2fwhm = self.prm[9]
        self.D2eta = self.prm[10]
        self.D2A = self.prm[11]
        self.D2amp = self.D2A/self.D2fwhm*(2/np.pi*self.D2eta+2*(1-self.D2eta)*np.sqrt(np.log(2)/np.pi))
                
        self.D3x0 = self.prm[12]
        self.D3fwhm = self.prm[13]
        self.D3eta = self.prm[14]
        self.D3A = self.prm[15]
        self.D3amp = self.D3A/self.D3fwhm*(2/np.pi*self.D3eta+2*(1-self.D3eta)*np.sqrt(np.log(2)/np.pi))
        
        self.D1int = pv(self.fitsft,self.D1x0,self.D1fwhm,self.D1eta,self.D1A)
        self.Gint = pv(self.fitsft,self.Gx0,self.Gfwhm,self.Geta,self.GA)
        self.D2int = pv(self.fitsft,self.D2x0,self.D2fwhm,self.D2eta,self.D2A)
        self.D3int = pv(self.fitsft,self.D3x0,self.D3fwhm,self.D3eta,self.D3A)
        
        self.fit = self.D1int+self.Gint+self.D2int+self.D3int
        self.res = self.fitint-self.fit
        
        self.R1 = self.D1amp/self.Gamp
        self.R2 = self.D1A/(self.GA+self.D1A+self.D2A)
        
        self.r2 = 1-(((self.fit-self.fitint)**2).sum())/(((self.fitint-np.average(self.fitint))**2).sum())

def savefig(file,dtype,datadir,mkdir,fitting,figurePDF=True,figurePNG=True):
    if figurePDF and figurePNG:
        c0 = 'gray'
        c1 = 'cyan'
        c2 = 'lime'
        c3 = 'wheat'
        c5 = 'black'
        c6 = 'purple'
        ls1 = 'solid'
        lw1 = 1
        lw2 = 0.75
        imgdir = mkdir.imgdir
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.fitint,lw=lw1,color=c0,linestyle=ls1,label='measured')
        plt.plot(fitting.fitsft,fitting.D1intini,lw=lw2,color=c1,linestyle=ls1,label='D1ini')
        plt.plot(fitting.fitsft,fitting.Gintini,lw=lw2,color=c6,linestyle=ls1,label='Gini')
        plt.legend()
        plt.xlim(1100,1750)
        plt.ylim(-0.1,1.1)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', D1- and G-bands, initial')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_D1Gini.pdf')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_D1Gini.png')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.owint,lw=lw1,color=c0,linestyle=ls1,label='measured-D1ini-Gini')
        plt.plot(fitting.fitsft,fitting.D2intini,lw=lw2,color=c2,linestyle=ls1,label='D2ini')
        plt.plot(fitting.fitsft,fitting.D3intini,lw=lw2,color=c3,linestyle=ls1,label='D3ini')
        plt.legend()
        plt.xlim(1100,1750)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', D2- and D3-bands, initial')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_D2D3ini.pdf')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_D2D3ini.png')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.fitint,lw=lw1,color=c0,linestyle=ls1,label='measured')
        plt.plot(fitting.fitsft,fitting.D1intini,lw=lw2,color=c1,linestyle=ls1,label='D1ini')
        plt.plot(fitting.fitsft,fitting.Gintini,lw=lw2,color=c6,linestyle=ls1,label='Gini')
        plt.plot(fitting.fitsft,fitting.D2intini,lw=lw2,color=c2,linestyle=ls1,label='D2ini')
        plt.plot(fitting.fitsft,fitting.D3intini,lw=lw2,color=c3,linestyle=ls1,label='D3ini')
        plt.plot(fitting.fitsft,fitting.fitini,lw=lw1,color=c5,linestyle=ls1,label='initial fit')
        plt.legend()
        plt.xlim(1100,1750)
        plt.ylim(-0.1,1.1)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', initial fit, r2={}'.format(np.round(fitting.r2ini,5)))
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_inifit.pdf')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_inifit.png')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.resini,lw=lw1,color=c0,linestyle=ls1)
        plt.xlim(1100,1750)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', initial residue')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_iniresidue.pdf')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_iniresidue.png')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.fitint,lw=lw1,color=c0,linestyle=ls1,label='measured')
        plt.plot(fitting.fitsft,fitting.D1int,lw=lw2,color=c1,linestyle=ls1,label='D1')
        plt.plot(fitting.fitsft,fitting.Gint,lw=lw2,color=c6,linestyle=ls1,label='G')
        plt.plot(fitting.fitsft,fitting.D2int,lw=lw2,color=c2,linestyle=ls1,label='D2')
        plt.plot(fitting.fitsft,fitting.D3int,lw=lw2,color=c3,linestyle=ls1,label='D3')
        plt.plot(fitting.fitsft,fitting.fit,lw=lw1,color=c5,linestyle=ls1,label='fitted')
        plt.legend()
        plt.xlim(1100,1750)
        plt.ylim(-0.1,1.1)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', fitted, r2={}'.format(np.round(fitting.r2,5)))
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_fit.pdf')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_fit.png')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.res,lw=lw1,color=c0,linestyle=ls1)
        plt.xlim(1100,1750)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', residue')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_residue.pdf')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_residue.png')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.hist(fitting.res,color=c0,bins=20)
        plt.xlabel('Residue')
        plt.ylabel('Count')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', residue histogram')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_residue_hist.pdf')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_residue_hist.png')
        plt.close()
    
    elif figurePDF and not figurePNG:
        c0 = 'gray'
        c1 = 'cyan'
        c2 = 'lime'
        c3 = 'wheat'
        c5 = 'black'
        c6 = 'purple'
        ls1 = 'solid'
        lw1 = 1
        lw2 = 0.75
        imgdir = mkdir.imgdir
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.fitint,lw=lw1,color=c0,linestyle=ls1,label='measured')
        plt.plot(fitting.fitsft,fitting.D1intini,lw=lw2,color=c1,linestyle=ls1,label='D1ini')
        plt.plot(fitting.fitsft,fitting.Gintini,lw=lw2,color=c6,linestyle=ls1,label='Gini')
        plt.legend()
        plt.xlim(1100,1750)
        plt.ylim(-0.1,1.1)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', D1- and G-bands, initial')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_D1Gini.pdf')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.owint,lw=lw1,color=c0,linestyle=ls1,label='measured-D1ini-Gini')
        plt.plot(fitting.fitsft,fitting.D2intini,lw=lw2,color=c2,linestyle=ls1,label='D2ini')
        plt.plot(fitting.fitsft,fitting.D3intini,lw=lw2,color=c3,linestyle=ls1,label='D3ini')
        plt.legend()
        plt.xlim(1100,1750)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', D2- and D3-bands, initial')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_D2D3ini.pdf')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.fitint,lw=lw1,color=c0,linestyle=ls1,label='measured')
        plt.plot(fitting.fitsft,fitting.D1intini,lw=lw2,color=c1,linestyle=ls1,label='D1ini')
        plt.plot(fitting.fitsft,fitting.Gintini,lw=lw2,color=c6,linestyle=ls1,label='Gini')
        plt.plot(fitting.fitsft,fitting.D2intini,lw=lw2,color=c2,linestyle=ls1,label='D2ini')
        plt.plot(fitting.fitsft,fitting.D3intini,lw=lw2,color=c3,linestyle=ls1,label='D3ini')
        plt.plot(fitting.fitsft,fitting.fitini,lw=lw1,color=c5,linestyle=ls1,label='initial fit')
        plt.legend()
        plt.xlim(1100,1750)
        plt.ylim(-0.1,1.1)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', initial fit, r2={}'.format(np.round(fitting.r2ini,5)))
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_inifit.pdf')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.resini,lw=lw1,color=c0,linestyle=ls1)
        plt.xlim(1100,1750)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', initial residue')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_iniresidue.pdf')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.fitint,lw=lw1,color=c0,linestyle=ls1,label='measured')
        plt.plot(fitting.fitsft,fitting.D1int,lw=lw2,color=c1,linestyle=ls1,label='D1')
        plt.plot(fitting.fitsft,fitting.Gint,lw=lw2,color=c6,linestyle=ls1,label='G')
        plt.plot(fitting.fitsft,fitting.D2int,lw=lw2,color=c2,linestyle=ls1,label='D2')
        plt.plot(fitting.fitsft,fitting.D3int,lw=lw2,color=c3,linestyle=ls1,label='D3')
        plt.plot(fitting.fitsft,fitting.fit,lw=lw1,color=c5,linestyle=ls1,label='fitted')
        plt.legend()
        plt.xlim(1100,1750)
        plt.ylim(-0.1,1.1)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', fitted, r2={}'.format(np.round(fitting.r2,5)))
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_fit.pdf')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.plot(fitting.fitsft,fitting.res,lw=lw1,color=c0,linestyle=ls1)
        plt.xlim(1100,1750)
        plt.xlabel('Raman shift [/cm]')
        plt.ylabel('Normalized intensity')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', residue')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_residue.pdf')
        plt.close()
        
        plt.figure(figsize=(7,7),dpi=200)
        plt.hist(fitting.res,color=c0,bins=20)
        plt.xlabel('Residue')
        plt.ylabel('Count')
        plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', residue histogram')
        plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_residue_hist.pdf')
        plt.close()
    
    elif not figurePDF and figurePNG:
       c0 = 'gray'
       c1 = 'cyan'
       c2 = 'lime'
       c3 = 'wheat'
       c5 = 'black'
       c6 = 'purple'
       ls1 = 'solid'
       lw1 = 1
       lw2 = 0.75
       imgdir = mkdir.imgdir
       
       plt.figure(figsize=(7,7),dpi=200)
       plt.plot(fitting.fitsft,fitting.fitint,lw=lw1,color=c0,linestyle=ls1,label='measured')
       plt.plot(fitting.fitsft,fitting.D1intini,lw=lw2,color=c1,linestyle=ls1,label='D1ini')
       plt.plot(fitting.fitsft,fitting.Gintini,lw=lw2,color=c6,linestyle=ls1,label='Gini')
       plt.legend()
       plt.xlim(1100,1750)
       plt.ylim(-0.1,1.1)
       plt.xlabel('Raman shift [/cm]')
       plt.ylabel('Normalized intensity')
       plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', D1- and G-bands, initial')
       plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_D1Gini.png')
       plt.close()
       
       plt.figure(figsize=(7,7),dpi=200)
       plt.plot(fitting.fitsft,fitting.owint,lw=lw1,color=c0,linestyle=ls1,label='measured-D1ini-Gini')
       plt.plot(fitting.fitsft,fitting.D2intini,lw=lw2,color=c2,linestyle=ls1,label='D2ini')
       plt.plot(fitting.fitsft,fitting.D3intini,lw=lw2,color=c3,linestyle=ls1,label='D3ini')
       plt.legend()
       plt.xlim(1100,1750)
       plt.xlabel('Raman shift [/cm]')
       plt.ylabel('Normalized intensity')
       plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', D2- and D3-bands, initial')
       plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_D2D3ini.png')
       plt.close()
       
       plt.figure(figsize=(7,7),dpi=200)
       plt.plot(fitting.fitsft,fitting.fitint,lw=lw1,color=c0,linestyle=ls1,label='measured')
       plt.plot(fitting.fitsft,fitting.D1intini,lw=lw2,color=c1,linestyle=ls1,label='D1ini')
       plt.plot(fitting.fitsft,fitting.Gintini,lw=lw2,color=c6,linestyle=ls1,label='Gini')
       plt.plot(fitting.fitsft,fitting.D2intini,lw=lw2,color=c2,linestyle=ls1,label='D2ini')
       plt.plot(fitting.fitsft,fitting.D3intini,lw=lw2,color=c3,linestyle=ls1,label='D3ini')
       plt.plot(fitting.fitsft,fitting.fitini,lw=lw1,color=c5,linestyle=ls1,label='initial fit')
       plt.legend()
       plt.xlim(1100,1750)
       plt.ylim(-0.1,1.1)
       plt.xlabel('Raman shift [/cm]')
       plt.ylabel('Normalized intensity')
       plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', initial fit, r2={}'.format(np.round(fitting.r2ini,5)))
       plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_inifit.png')
       plt.close()
       
       plt.figure(figsize=(7,7),dpi=200)
       plt.plot(fitting.fitsft,fitting.resini,lw=lw1,color=c0,linestyle=ls1)
       plt.xlim(1100,1750)
       plt.xlabel('Raman shift [/cm]')
       plt.ylabel('Normalized intensity')
       plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', initial residue')
       plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_iniresidue.png')
       plt.close()
       
       plt.figure(figsize=(7,7),dpi=200)
       plt.plot(fitting.fitsft,fitting.fitint,lw=lw1,color=c0,linestyle=ls1,label='measured')
       plt.plot(fitting.fitsft,fitting.D1int,lw=lw2,color=c1,linestyle=ls1,label='D1')
       plt.plot(fitting.fitsft,fitting.Gint,lw=lw2,color=c6,linestyle=ls1,label='G')
       plt.plot(fitting.fitsft,fitting.D2int,lw=lw2,color=c2,linestyle=ls1,label='D2')
       plt.plot(fitting.fitsft,fitting.D3int,lw=lw2,color=c3,linestyle=ls1,label='D3')
       plt.plot(fitting.fitsft,fitting.fit,lw=lw1,color=c5,linestyle=ls1,label='fitted')
       plt.legend()
       plt.xlim(1100,1750)
       plt.ylim(-0.1,1.1)
       plt.xlabel('Raman shift [/cm]')
       plt.ylabel('Normalized intensity')
       plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', fitted, r2={}'.format(np.round(fitting.r2,5)))
       plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_fit.png')
       plt.close()
       
       plt.figure(figsize=(7,7),dpi=200)
       plt.plot(fitting.fitsft,fitting.res,lw=lw1,color=c0,linestyle=ls1)
       plt.xlim(1100,1750)
       plt.xlabel('Raman shift [/cm]')
       plt.ylabel('Normalized intensity')
       plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', residue')
       plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_residue.png')
       plt.close()
       
       plt.figure(figsize=(7,7),dpi=200)
       plt.hist(fitting.res,color=c0,bins=20)
       plt.xlabel('Residue')
       plt.ylabel('Count')
       plt.title(file.lstrip(datadir).rstrip('.'+dtype)+', residue histogram')
       plt.savefig(imgdir+'/fit/'+file.lstrip(datadir).rstrip('.'+dtype)+'_residue_hist.png')
       plt.close()

def savespec(file,dtype,datadir,mkdir,fitting,specdat=True):
    if specdat:
        specdir = mkdir.specdir
        specvalue = np.c_[fitting.fitsft,fitting.fitint,fitting.D1intini,fitting.Gintini,fitting.D2intini,fitting.D3intini,fitting.fitini,fitting.D1int,fitting.Gint,fitting.D2int,fitting.D3int,fitting.fit]
        csvname = specdir+'/'+file.lstrip(datadir).rstrip('.'+dtype)+'_fit.csv'
        np.savetxt(csvname,specvalue,delimiter=',',comments='',header='shift,bg+norm,D1ini,Gini,D2ini,D3ini,inifit,D1,G,D2,D3,fit')
    elif not specdat:
        pass

def saveprm(file,dtype,datadir,mkdir,fitting):
    prmdir = mkdir.prmdir
    prmvalueini = np.c_[fitting.R1ini,fitting.R2ini,
                        fitting.D1x0ini,fitting.D1fwhmini,fitting.D1etaini,fitting.D1Aini,fitting.D1ampini,
                        fitting.Gx0ini,fitting.Gfwhmini,fitting.Getaini,fitting.GAini,fitting.Gampini,
                        fitting.D2x0ini,fitting.D2fwhmini,fitting.D2etaini,fitting.D2Aini,fitting.D2ampini,
                        fitting.D3x0ini,fitting.D3fwhmini,fitting.D3etaini,fitting.D3Aini,fitting.D3ampini,fitting.r2ini]
    prmvaluefin = np.c_[fitting.R1,fitting.R2,
                        fitting.D1x0,fitting.D1fwhm,fitting.D1eta,fitting.D1A,fitting.D1amp,
                        fitting.Gx0,fitting.Gfwhm,fitting.Geta,fitting.GA,fitting.Gamp,
                        fitting.D2x0,fitting.D2fwhm,fitting.D2eta,fitting.D2A,fitting.D2amp,
                        fitting.D3x0,fitting.D3fwhm,fitting.D3eta,fitting.D3A,fitting.D3amp,fitting.r2]
    
    csvnameini = prmdir+'/initial/'+file.lstrip(datadir).rstrip('.'+dtype)+'_ini.csv'
    csvnamefin = prmdir+'/final/'+file.lstrip(datadir).rstrip('.'+dtype)+'_fin.csv'
    headerini = 'R1ini,R2ini,D1x0ini,D1fwhmini,D1etaini,D1Aini,D1ampini,Gx0ini,Gfwhmini,Getaini,GAini,Gampini,D2x0ini,D2fwhmini,D2etaini,D2Aini,D2ampini,D3x0ini,D3fwhmini,D3etaini,D3Aini,D3ampini,r2ini'
    headerfin = 'R1,R2,D1x0,D1fwhm,D1eta,D1A,D1amp,Gx0,Gfwhm,Geta,GA,Gamp,D2x0,D2fwhm,D2eta,D2A,D2amp,D3x0,D3fwhm,D3eta,D3A,D3amp,r2'
    np.savetxt(csvnameini,prmvalueini,delimiter=',',comments='',header=headerini)
    np.savetxt(csvnamefin,prmvaluefin,delimiter=',',comments='',header=headerfin)