import requests
import json
from argparse import ArgumentParser
from datetime import datetime

# base_url = 'https://sandbox.zenodo.org/api/'
base_url = 'https://zenodo.org/api/'


def __get_latest_url(last_version_id, access_token):
    request_url = f"{base_url}deposit/depositions/{last_version_id}?access_token={access_token}"
    response = requests.get(request_url)
    response.raise_for_status()
    data = response.json()

    if not data['submitted']:
        raise Exception('There is already a draft!')

    return data['links']['newversion']


def __get_last_version_id(global_id, access_token):
    request_url = f"{base_url}deposit/depositions?access_token={access_token}"
    response = requests.get(request_url)
    response.raise_for_status()
    data = response.json()

    for entry in data:
        if entry['conceptrecid'] == global_id:
            return entry['id']

    return None


def __create_new_version(new_version_url, access_token):
    response = requests.post(new_version_url, params={'access_token': access_token})
    response.raise_for_status()
    data = response.json()
    return data


def __update_metadata(draft_url, access_token, data):
    request_url = f"{draft_url}?access_token={access_token}"
    headers = {"Content-Type": "application/json"}

    response = requests.put(request_url, data=json.dumps(data), headers=headers)
    response.raise_for_status()


def __delete_files(draft_url, files, access_token):
    for file in files:
        url = f'{draft_url}/files/{file["id"]}'
        response = requests.delete(url, params={'access_token': access_token})
        response.raise_for_status()


def __upload_file(draft_url, access_token, filename, file_path):
    request_url = f'{draft_url}/files?access_token={access_token}'
    data = {'name': filename}
    files = {'file': open(file_path, 'rb')}
    response = requests.post(request_url, data=data, files=files)
    response.raise_for_status()


def __publish(draft_url, access_token):
    request_url = f'{draft_url}/actions/publish'
    response = requests.post(request_url, params={'access_token': access_token})
    response.raise_for_status()


def main():
    argument_parser = ArgumentParser()

    argument_parser.add_argument("--access-token", dest="access_token",
                                 help="The access token", required=True)
    argument_parser.add_argument("--id", dest="global_id",
                                 help="The global id", required=True)
    argument_parser.add_argument("--title", dest="title",
                                 help="The title", required=True)
    argument_parser.add_argument("--version", dest="version",
                                 help="The version", required=True)
    argument_parser.add_argument("--file-path", dest="file_path",
                                 help="The file path", required=True)

    args = argument_parser.parse_args()
    access_token = args.access_token
    global_id = args.global_id
    title = args.title
    version = args.version
    file_path = args.file_path

    if version.startswith('v.'):
        version = version[2:]
    elif version.startswith('v'):
        version = version[1:]

    latest_version_id = __get_last_version_id(global_id, access_token)

    if not latest_version_id:
        raise Exception(f'ID {global_id} not found!')

    new_version_url = __get_latest_url(latest_version_id, access_token)
    new_version_data = __create_new_version(new_version_url, access_token)

    draft_url = new_version_data['links']['latest_draft']
    updated_metadata = {'metadata': new_version_data['metadata']}

    updated_metadata['metadata']['title'] = f'{title} v.{version}'
    updated_metadata['metadata']['version'] = version

    __update_metadata(draft_url, access_token, updated_metadata)
    __delete_files(draft_url, new_version_data['files'], access_token)

    filename = f'{title}-v.{version}.zip'

    __upload_file(draft_url, access_token, filename, file_path)
    __publish(draft_url, access_token)


if __name__ == '__main__':
    main()
