from unittest import TestCase
from unittest.mock import Mock

from quid.match.Match import Match
from quid.match.MatchSpan import MatchSpan

from proquo.core.ProQuoLm import ProQuoLm
import numpy as np


class ProQuoLmTestCase(TestCase):

    def test_one_word_exact_match(self):

        source_text = 'This is some text for testing'
        target_text = 'And this is a "text" with a quote'

        quid_matches = []

        model_mock = Mock()
        vectorizer_mock = Mock()
        prediction_mock = Mock()

        logits = np.array([[-1.74764, 0.88853]])
        prediction_mock.logits = logits
        vectorizer_mock.vectorize.return_value = None
        model_mock.predict.return_value = prediction_mock

        proquo = ProQuoLm(model_mock, vectorizer_mock, '"', '"')
        matches = proquo.compare(source_text, target_text, quid_matches)

        self.assertEqual(1, len(matches))
        self.assertEqual(13, matches[0].source_span.start)
        self.assertEqual(17, matches[0].source_span.end)

    def test_one_word_fuzzy_match(self):

        source_text = 'This is some testword for testing'
        target_text = 'And this is a "test-word" with a quote'

        quid_matches = []

        model_mock = Mock()
        vectorizer_mock = Mock()
        prediction_mock = Mock()

        logits = np.array([[-1.74764, 0.88853]])
        prediction_mock.logits = logits
        vectorizer_mock.vectorize.return_value = None
        model_mock.predict.return_value = prediction_mock

        proquo = ProQuoLm(model_mock, vectorizer_mock, '"', '"')
        matches = proquo.compare(source_text, target_text, quid_matches)

        self.assertEqual(1, len(matches))
        self.assertEqual(13, matches[0].source_span.start)
        self.assertEqual(21, matches[0].source_span.end)

    def test_multi_word_exact_match(self):

        source_text = 'This is some double word for testing'
        target_text = 'And this is a "double word" with a quote'

        source_span = MatchSpan(13, 24, '')
        target_span = MatchSpan(15, 26, '')

        quid_matches = [Match(source_span, target_span)]

        model_mock = Mock()
        vectorizer_mock = Mock()
        prediction_mock = Mock()

        logits = np.array([[-1.74764, 0.88853]])
        prediction_mock.logits = logits
        vectorizer_mock.vectorize.return_value = None
        model_mock.predict.return_value = prediction_mock

        proquo = ProQuoLm(model_mock, vectorizer_mock, '"', '"')
        matches = proquo.compare(source_text, target_text, quid_matches)

        self.assertEqual(1, len(matches))
        self.assertEqual(13, matches[0].source_span.start)
        self.assertEqual(24, matches[0].source_span.end)

    def test_multi_word_fuzzy_match(self):

        source_text = 'This is some double word for testing'
        target_text = 'And this is a "double words" with a quote'

        source_span = MatchSpan(13, 24, '')
        target_span = MatchSpan(15, 27, '')

        quid_matches = [Match(source_span, target_span)]

        model_mock = Mock()
        vectorizer_mock = Mock()
        prediction_mock = Mock()

        logits = np.array([[-1.74764, 0.88853]])
        prediction_mock.logits = logits
        vectorizer_mock.vectorize.return_value = None
        model_mock.predict.return_value = prediction_mock

        proquo = ProQuoLm(model_mock, vectorizer_mock, '"', '"')
        matches = proquo.compare(source_text, target_text, quid_matches)

        self.assertEqual(1, len(matches))
        self.assertEqual(13, matches[0].source_span.start)
        self.assertEqual(24, matches[0].source_span.end)

    def test_quotation_mark_detection(self):

        source_text = 'This is some words for testing'
        target_text = 'And this is a ‘word’s’ with a quote'

        quid_matches = []

        model_mock = Mock()
        vectorizer_mock = Mock()
        prediction_mock = Mock()

        logits = np.array([[-1.74764, 0.88853]])
        prediction_mock.logits = logits
        vectorizer_mock.vectorize.return_value = None
        model_mock.predict.return_value = prediction_mock

        proquo = ProQuoLm(model_mock, vectorizer_mock)
        matches = proquo.compare(source_text, target_text, quid_matches)

        self.assertEqual(1, len(matches))
        self.assertEqual(13, matches[0].source_span.start)
        self.assertEqual(18, matches[0].source_span.end)

        self.assertEqual(15, matches[0].target_span.start)
        self.assertEqual(21, matches[0].target_span.end)
