import unittest

from nacc import redcap2nacc
from nacc.uds3.tip.builder import build_uds3_tip_form


class option():
    flag = 'tip'
    iorf = False
    cv = False
    csf = False
    lbd = False
    tip = True
    tfp = False
    tfp3 = False
    ftld = False
    ivp = False
    fvp = False


class TestBlankRulesForTIP(unittest.TestCase):
    '''
    These tests are designed to run TIP data fields (generated below
    the tests here) through the check_blanks function for the UDS3 module.
    The DED for the TIP module saw some updates to the grammar of existing
    blanking rules, and so this suite of tests is mostly concerned with making
    sure the updated versions of old rules / "special cases" are functioning
    properly.
    '''

    def setUp(self):
        self.options = option()

    def test_for_filled_when_ruled_blank(self):
        '''
        Have it look for the langa4 error to see that general blanking rules
        are working (langa4 also comes before the variable (a4sub)
        it's dependent on)
        '''
        record = make_filled_form()
        record['tip_a4sub'] = '0'
        record['tip_langa4'] = '1'
        fpacket = build_uds3_tip_form(record)
        warnings = []

        warnings = redcap2nacc.check_blanks(fpacket, self.options)
        expected = ["LANGA4 in form Z1X is '1' with length '1', but should be"
                    " blank: 'Blank if #5b. A4SUB = 0 (No)'."]
        self.assertEqual(warnings, expected)

    def test_for_single_blanking_rule_returned(self):
        '''
        Have it make sure that only one error is returned from a list of
        rules when not working with special cases
        '''
        record = make_filled_form()
        record['tip_cvhatt'] = '9'
        record['tip_hattmult'] = '0'
        fpacket = build_uds3_tip_form(record)
        warnings = []

        warnings = redcap2nacc.check_blanks(fpacket, self.options)
        expected = ["HATTMULT in form A5  is '0' with length '1', but"
                    " should be blank: 'Blank if #2a. CVHATT = 9 (Unknown)'."]
        self.assertEqual(warnings, expected)

    def test_for_special_case_rey1rec(self):
        '''
        Have it make sure that with options.tip active, the blank parsing can
        access the TIP-specific special cases.
        '''
        record = make_filled_form()
        record['tip_rey1rec'] = '88'
        record['tip_rey1int'] = '1'
        fpacket = build_uds3_tip_form(record)
        warnings = []

        warnings = redcap2nacc.check_blanks(fpacket, self.options)
        expected = ["REY1INT in form C2  is '1 ' with length '2', but"
                    " should be blank: 'Blank if 4a REY1REC 88, 95-98'."]
        self.assertEqual(warnings, expected)

    def test_for_special_case_mocalanx(self):
        '''
        Have it test a special case where there are two possible conditions
        for the blanking rule to be violated. In this example, one condition
        for the rule violation should also be blank.
        '''
        record = make_filled_form()
        record['tip_mocacomp'] = '0'
        record['tip_mocalan'] = '1'
        record['tip_mocalanx'] = 'a'
        fpacket = build_uds3_tip_form(record)
        warnings = []

        warnings = redcap2nacc.check_blanks(fpacket, self.options)
        expected = ["MOCALAN in form C2  is '1' with length '1', but should be"
                    " blank: 'Blank if 1a MOCACOMP = 0 (No)'.",
                    "MOCALANX in form C2  is 'a                               "
                    "                            ' with length '60', but"
                    " should be blank: 'Blank if 1a MOCACOMP = 0 (No) or 1b"
                    " MOCALAN = English (1) or Spanish (2)'."]
        self.assertEqual(warnings, expected)


def make_filled_form() -> dict:
    return {
        # Headers
        'ptid': '1',
        'adcid': '2',
        'visitmo': '1',
        'visitday': '1',
        'visityr': '2020',
        'visitnum': '1',
        'initials': '',
        # filled forms
        'tip_langt1': '',
        'tip_langa1': '',
        'tip_langa2': '',
        'tip_langa3': '',
        'tip_a3sub': '',
        'tip_a3not': '',
        'tip_langa4': '',
        'tip_a4sub': '',
        'tip_a4not': '',
        'tip_langa5': '',
        'tip_langb1': '',
        'tip_langb4': '',
        'tip_langb5': '',
        'tip_b5sub': '',
        'tip_b5not': '',
        'tip_langb6': '',
        'tip_b6sub': '',
        'tip_b6not': '',
        'tip_langb7': '',
        'tip_b7sub': '',
        'tip_b7not': '',
        'tip_langb8': '',
        'tip_b8sub': '',
        'tip_b8not': '',
        'tip_langb9': '',
        'tip_langc2t': '',
        'tip_langd1': '',
        'tip_langd2': '',
        'tip_langcls': '',
        'tip_clssub': '',
        'tip_z1x_complete': '2',
        'tip_telcog': '',
        'tip_telill': '',
        'tip_telhome': '',
        'tip_telrefu': '',
        'tip_telcov': '',
        'tip_telothr': '',
        'tip_telothrx': '',
        'tip_telmod': '',
        'tip_telinper': '',
        'tip_telmile': '',
        'tip_reason': '',
        'tip_refersc': '',
        'tip_learned': '',
        'tip_prestat': '',
        'tip_prespart': '',
        'tip_sourcenw': '',
        'tip_birthmo': '',
        'tip_birthyr': '',
        'tip_sex': '',
        'tip_hispanic': '',
        'tip_hispor': '',
        'tip_hisporx': '',
        'tip_race': '',
        'tip_racex': '',
        'tip_racesec': '',
        'tip_racesecx': '',
        'tip_raceter': '',
        'tip_raceterx': '',
        'tip_primlang': '',
        'tip_primlanx': '',
        'tip_educ': '',
        'tip_maristat': '',
        'tip_livsitua': '',
        'tip_independ': '',
        'tip_residenc': '',
        'tip_zip': '',
        'tip_handed': '',
        'tip_inbirmo': '',
        'tip_inbiryr': '',
        'tip_insex': '',
        'tip_inhisp': '',
        'tip_inhispor': '',
        'tip_inhispox': '',
        'tip_inrace': '',
        'tip_inracex': '',
        'tip_inrasec': '',
        'tip_inrasecx': '',
        'tip_inrater': '',
        'tip_inraterx': '',
        'tip_ineduc': '',
        'tip_inrelto': '',
        'tip_inknown': '',
        'tip_inlivwth': '',
        'tip_invisits': '',
        'tip_incalls': '',
        'tip_inrely': '',
        'tip_afffamm': '',
        'tip_fadmut': '',
        'tip_fadmutx': '',
        'tip_fadmuso': '',
        'tip_fadmusox': '',
        'tip_fftdmut': '',
        'tip_fftdmutx': '',
        'tip_fftdmuso': '',
        'tip_fftdmusx': '',
        'tip_fothmut': '',
        'tip_fothmutx': '',
        'tip_fothmuso': '',
        'tip_fothmusx': '',
        'tip_mommob': '',
        'tip_momyob': '',
        'tip_momdage': '',
        'tip_momneur': '',
        'tip_momprdx': '',
        'tip_mommoe': '',
        'tip_momageo': '',
        'tip_dadmob': '',
        'tip_dadyob': '',
        'tip_daddage': '',
        'tip_dadneur': '',
        'tip_dadprdx': '',
        'tip_dadmoe': '',
        'tip_dadageo': '',
        'tip_sibs': '',
        'tip_sib1mob': '',
        'tip_sib1yob': '',
        'tip_sib1agd': '',
        'tip_sib1neu': '',
        'tip_sib1pdx': '',
        'tip_sib1moe': '',
        'tip_sib1ago': '',
        'tip_sib2mob': '',
        'tip_sib2yob': '',
        'tip_sib2agd': '',
        'tip_sib2neu': '',
        'tip_sib2pdx': '',
        'tip_sib2moe': '',
        'tip_sib2ago': '',
        'tip_sib3mob': '',
        'tip_sib3yob': '',
        'tip_sib3agd': '',
        'tip_sib3neu': '',
        'tip_sib3pdx': '',
        'tip_sib3moe': '',
        'tip_sib3ago': '',
        'tip_sib4mob': '',
        'tip_sib4yob': '',
        'tip_sib4agd': '',
        'tip_sib4neu': '',
        'tip_sib4pdx': '',
        'tip_sib4moe': '',
        'tip_sib4ago': '',
        'tip_sib5mob': '',
        'tip_sib5yob': '',
        'tip_sib5agd': '',
        'tip_sib5neu': '',
        'tip_sib5pdx': '',
        'tip_sib5moe': '',
        'tip_sib5ago': '',
        'tip_sib6mob': '',
        'tip_sib6yob': '',
        'tip_sib6agd': '',
        'tip_sib6neu': '',
        'tip_sib6pdx': '',
        'tip_sib6moe': '',
        'tip_sib6ago': '',
        'tip_sib7mob': '',
        'tip_sib7yob': '',
        'tip_sib7agd': '',
        'tip_sib7neu': '',
        'tip_sib7pdx': '',
        'tip_sib7moe': '',
        'tip_sib7ago': '',
        'tip_sib8mob': '',
        'tip_sib8yob': '',
        'tip_sib8agd': '',
        'tip_sib8neu': '',
        'tip_sib8pdx': '',
        'tip_sib8moe': '',
        'tip_sib8ago': '',
        'tip_sib9mob': '',
        'tip_sib9yob': '',
        'tip_sib9agd': '',
        'tip_sib9neu': '',
        'tip_sib9pdx': '',
        'tip_sib9moe': '',
        'tip_sib9ago': '',
        'tip_sib10mob': '',
        'tip_sib10yob': '',
        'tip_sib10agd': '',
        'tip_sib10neu': '',
        'tip_sib10pdx': '',
        'tip_sib10moe': '',
        'tip_sib10ago': '',
        'tip_sib11mob': '',
        'tip_sib11yob': '',
        'tip_sib11agd': '',
        'tip_sib11neu': '',
        'tip_sib11pdx': '',
        'tip_sib11moe': '',
        'tip_sib11ago': '',
        'tip_sib12mob': '',
        'tip_sib12yob': '',
        'tip_sib12agd': '',
        'tip_sib12neu': '',
        'tip_sib12pdx': '',
        'tip_sib12moe': '',
        'tip_sib12ago': '',
        'tip_sib13mob': '',
        'tip_sib13yob': '',
        'tip_sib13agd': '',
        'tip_sib13neu': '',
        'tip_sib13pdx': '',
        'tip_sib13moe': '',
        'tip_sib13ago': '',
        'tip_sib14mob': '',
        'tip_sib14yob': '',
        'tip_sib14agd': '',
        'tip_sib14neu': '',
        'tip_sib14pdx': '',
        'tip_sib14moe': '',
        'tip_sib14ago': '',
        'tip_sib15mob': '',
        'tip_sib15yob': '',
        'tip_sib15agd': '',
        'tip_sib15neu': '',
        'tip_sib15pdx': '',
        'tip_sib15moe': '',
        'tip_sib15ago': '',
        'tip_sib16mob': '',
        'tip_sib16yob': '',
        'tip_sib16agd': '',
        'tip_sib16neu': '',
        'tip_sib16pdx': '',
        'tip_sib16moe': '',
        'tip_sib16ago': '',
        'tip_sib17mob': '',
        'tip_sib17yob': '',
        'tip_sib17agd': '',
        'tip_sib17neu': '',
        'tip_sib17pdx': '',
        'tip_sib17moe': '',
        'tip_sib17ago': '',
        'tip_sib18mob': '',
        'tip_sib18yob': '',
        'tip_sib18agd': '',
        'tip_sib18neu': '',
        'tip_sib18pdx': '',
        'tip_sib18moe': '',
        'tip_sib18ago': '',
        'tip_sib19mob': '',
        'tip_sib19yob': '',
        'tip_sib19agd': '',
        'tip_sib19neu': '',
        'tip_sib19pdx': '',
        'tip_sib19moe': '',
        'tip_sib19ago': '',
        'tip_sib20mob': '',
        'tip_sib20yob': '',
        'tip_sib20agd': '',
        'tip_sib20neu': '',
        'tip_sib20pdx': '',
        'tip_sib20moe': '',
        'tip_sib20ago': '',
        'tip_kids': '',
        'tip_kid1mob': '',
        'tip_kid1yob': '',
        'tip_kid1agd': '',
        'tip_kid1neu': '',
        'tip_kid1pdx': '',
        'tip_kid1moe': '',
        'tip_kid1ago': '',
        'tip_kid2mob': '',
        'tip_kid2yob': '',
        'tip_kid2agd': '',
        'tip_kid2neu': '',
        'tip_kid2pdx': '',
        'tip_kid2moe': '',
        'tip_kid2ago': '',
        'tip_kid3mob': '',
        'tip_kid3yob': '',
        'tip_kid3agd': '',
        'tip_kid3neu': '',
        'tip_kid3pdx': '',
        'tip_kid3moe': '',
        'tip_kid3ago': '',
        'tip_kid4mob': '',
        'tip_kid4yob': '',
        'tip_kid4agd': '',
        'tip_kid4neu': '',
        'tip_kid4pdx': '',
        'tip_kid4moe': '',
        'tip_kid4ago': '',
        'tip_kid5mob': '',
        'tip_kid5yob': '',
        'tip_kid5agd': '',
        'tip_kid5neu': '',
        'tip_kid5pdx': '',
        'tip_kid5moe': '',
        'tip_kid5ago': '',
        'tip_kid6mob': '',
        'tip_kid6yob': '',
        'tip_kid6agd': '',
        'tip_kid6neu': '',
        'tip_kid6pdx': '',
        'tip_kid6moe': '',
        'tip_kid6ago': '',
        'tip_kid7mob': '',
        'tip_kid7yob': '',
        'tip_kid7agd': '',
        'tip_kid7neu': '',
        'tip_kid7pdx': '',
        'tip_kid7moe': '',
        'tip_kid7ago': '',
        'tip_kid8mob': '',
        'tip_kid8yob': '',
        'tip_kid8agd': '',
        'tip_kid8neu': '',
        'tip_kid8pdx': '',
        'tip_kid8moe': '',
        'tip_kid8ago': '',
        'tip_kid9mob': '',
        'tip_kid9yob': '',
        'tip_kid9agd': '',
        'tip_kid9neu': '',
        'tip_kid9pdx': '',
        'tip_kid9moe': '',
        'tip_kid9ago': '',
        'tip_kid10mob': '',
        'tip_kid10yob': '',
        'tip_kid10agd': '',
        'tip_kid10neu': '',
        'tip_kid10pdx': '',
        'tip_kid10moe': '',
        'tip_kid10ago': '',
        'tip_kid11mob': '',
        'tip_kid11yob': '',
        'tip_kid11agd': '',
        'tip_kid11neu': '',
        'tip_kid11pdx': '',
        'tip_kid11moe': '',
        'tip_kid11ago': '',
        'tip_kid12mob': '',
        'tip_kid12yob': '',
        'tip_kid12agd': '',
        'tip_kid12neu': '',
        'tip_kid12pdx': '',
        'tip_kid12moe': '',
        'tip_kid12ago': '',
        'tip_kid13mob': '',
        'tip_kid13yob': '',
        'tip_kid13agd': '',
        'tip_kid13neu': '',
        'tip_kid13pdx': '',
        'tip_kid13moe': '',
        'tip_kid13ago': '',
        'tip_kid14mob': '',
        'tip_kid14yob': '',
        'tip_kid14agd': '',
        'tip_kid14neu': '',
        'tip_kid14pdx': '',
        'tip_kid14moe': '',
        'tip_kid14ago': '',
        'tip_kid15mob': '',
        'tip_kid15yob': '',
        'tip_kid15agd': '',
        'tip_kid15neu': '',
        'tip_kid15pdx': '',
        'tip_kid15moe': '',
        'tip_kid15ago': '',
        'tip_anymeds': '',
        'tip_tobac30': '',
        'tip_smokyrs': '',
        'tip_tobac100': '',
        'tip_packsper': '',
        'tip_quitsmok': '',
        'tip_alcoccas': '',
        'tip_alcfreq': '',
        'tip_cvhatt': '',
        'tip_hattmult': '',
        'tip_hattyear': '',
        'tip_cvafib': '',
        'tip_cvangio': '',
        'tip_cvbypass': '',
        'tip_cvpacdef': '',
        'tip_cvchf': '',
        'tip_cvangina': '',
        'tip_cvhvalve': '',
        'tip_cvothr': '',
        'tip_cvothrx': '',
        'tip_cbstroke': '',
        'tip_strokmul': '',
        'tip_strokyr': '',
        'tip_cbtia': '',
        'tip_tiamult': '',
        'tip_tiayear': '',
        'tip_pd': '',
        'tip_pdyr': '',
        'tip_pdothr': '',
        'tip_pdothryr': '',
        'tip_seizures': '',
        'tip_tbi': '',
        'tip_tbibrief': '',
        'tip_tbiexten': '',
        'tip_tbiwolos': '',
        'tip_tbiyear': '',
        'tip_diabetes': '',
        'tip_diabtype': '',
        'tip_hyperten': '',
        'tip_hypercho': '',
        'tip_b12def': '',
        'tip_thyroid': '',
        'tip_arthrit': '',
        'tip_arthtype': '',
        'tip_arthtypx': '',
        'tip_arthupex': '',
        'tip_arthloex': '',
        'tip_arthspin': '',
        'tip_arthunk': '',
        'tip_incontu': '',
        'tip_incontf': '',
        'tip_apnea': '',
        'tip_rbd': '',
        'tip_insomn': '',
        'tip_othsleep': '',
        'tip_othsleex': '',
        'tip_alcohol': '',
        'tip_abusothr': '',
        'tip_abusx': '',
        'tip_ptsd': '',
        'tip_bipolar': '',
        'tip_schiz': '',
        'tip_dep2yrs': '',
        'tip_depothr': '',
        'tip_anxiety': '',
        'tip_ocd': '',
        'tip_npsydev': '',
        'tip_psycdis': '',
        'tip_psycdisx': '',
        'tip_vision': '',
        'tip_viscorr': '',
        'tip_viswcorr': '',
        'tip_hearing': '',
        'tip_hearaid': '',
        'tip_hearwaid': '',
        'tip_memory': '',
        'tip_orient': '',
        'tip_judgment': '',
        'tip_commun': '',
        'tip_homehobb': '',
        'tip_perscare': '',
        'tip_cdrsum': '',
        'tip_cdrglob': '',
        'tip_comport': '',
        'tip_cdrlang': '',
        'tip_npiqinf': '',
        'tip_npiqinfx': '',
        'tip_del': '',
        'tip_delsev': '',
        'tip_hall': '',
        'tip_hallsev': '',
        'tip_agit': '',
        'tip_agitsev': '',
        'tip_depd': '',
        'tip_depdsev': '',
        'tip_anx': '',
        'tip_anxsev': '',
        'tip_elat': '',
        'tip_elatsev': '',
        'tip_apa': '',
        'tip_apasev': '',
        'tip_disn': '',
        'tip_disnsev': '',
        'tip_irr': '',
        'tip_irrsev': '',
        'tip_mot': '',
        'tip_motsev': '',
        'tip_nite': '',
        'tip_nitesev': '',
        'tip_app': '',
        'tip_appsev': '',
        'tip_nogds': '',
        'tip_satis': '',
        'tip_dropact': '',
        'tip_empty': '',
        'tip_bored': '',
        'tip_spirits': '',
        'tip_afraid': '',
        'tip_happy': '',
        'tip_helpless': '',
        'tip_stayhome': '',
        'tip_memprob': '',
        'tip_wondrful': '',
        'tip_wrthless': '',
        'tip_energy': '',
        'tip_hopeless': '',
        'tip_better': '',
        'tip_gds': '',
        'tip_bills': '',
        'tip_taxes': '',
        'tip_shopping': '',
        'tip_games': '',
        'tip_stove': '',
        'tip_mealprep': '',
        'tip_events': '',
        'tip_payattn': '',
        'tip_remdates': '',
        'tip_travel': '',
        'tip_normexam': '',
        'tip_parksign': '',
        'tip_resttrl': '',
        'tip_resttrr': '',
        'tip_slowingl': '',
        'tip_slowingr': '',
        'tip_rigidl': '',
        'tip_rigidr': '',
        'tip_brady': '',
        'tip_parkgait': '',
        'tip_postinst': '',
        'tip_cvdsigns': '',
        'tip_cortdef': '',
        'tip_sivdfind': '',
        'tip_cvdmotl': '',
        'tip_cvdmotr': '',
        'tip_cortvisl': '',
        'tip_cortvisr': '',
        'tip_somatl': '',
        'tip_somatr': '',
        'tip_postcort': '',
        'tip_pspcbs': '',
        'tip_eyepsp': '',
        'tip_dyspsp': '',
        'tip_axialpsp': '',
        'tip_gaitpsp': '',
        'tip_apraxsp': '',
        'tip_apraxl': '',
        'tip_apraxr': '',
        'tip_cortsenl': '',
        'tip_cortsenr': '',
        'tip_ataxl': '',
        'tip_ataxr': '',
        'tip_alienlml': '',
        'tip_alienlmr': '',
        'tip_dystonl': '',
        'tip_dystonr': '',
        'tip_myocllt': '',
        'tip_myoclrt': '',
        'tip_alsfind': '',
        'tip_gaitnph': '',
        'tip_othneur': '',
        'tip_othneurx': '',
        'tip_decsub': '',
        'tip_decin': '',
        'tip_decclcog': '',
        'tip_cogmem': '',
        'tip_cogori': '',
        'tip_cogjudg': '',
        'tip_coglang': '',
        'tip_cogvis': '',
        'tip_cogattn': '',
        'tip_cogfluc': '',
        'tip_cogflago': '',
        'tip_cogothr': '',
        'tip_cogothrx': '',
        'tip_cogfpred': '',
        'tip_cogfprex': '',
        'tip_cogmode': '',
        'tip_cogmodex': '',
        'tip_decage': '',
        'tip_decclbe': '',
        'tip_beapathy': '',
        'tip_bedep': '',
        'tip_bevhall': '',
        'tip_bevwell': '',
        'tip_bevhago': '',
        'tip_beahall': '',
        'tip_bedel': '',
        'tip_bedisin': '',
        'tip_beirrit': '',
        'tip_beagit': '',
        'tip_beperch': '',
        'tip_berem': '',
        'tip_beremago': '',
        'tip_beanx': '',
        'tip_beothr': '',
        'tip_beothrx': '',
        'tip_befpred': '',
        'tip_befpredx': '',
        'tip_bemode': '',
        'tip_bemodex': '',
        'tip_beage': '',
        'tip_decclmot': '',
        'tip_mogait': '',
        'tip_mofalls': '',
        'tip_motrem': '',
        'tip_moslow': '',
        'tip_mofrst': '',
        'tip_momode': '',
        'tip_momodex': '',
        'tip_momopark': '',
        'tip_parkage': '',
        'tip_momoals': '',
        'tip_alsage': '',
        'tip_moage': '',
        'tip_course': '',
        'tip_frstchg': '',
        'tip_lbdeval': '',
        'tip_ftldeval': '',
        'tip_modcomm': '',
        'tip_mocacomp': '',
        'tip_mocareas': '',
        'tip_mocalan': '',
        'tip_mocalanx': '',
        'tip_mocahear': '',
        'tip_mocbtots': '',
        'tip_mocadigi': '',
        'tip_mocalett': '',
        'tip_mocaser7': '',
        'tip_mocarepe': '',
        'tip_mocaflue': '',
        'tip_mocaabst': '',
        'tip_mocarecn': '',
        'tip_mocarecc': '',
        'tip_mocarecr': '',
        'tip_mocaordt': '',
        'tip_mocaormo': '',
        'tip_mocaoryr': '',
        'tip_mocaordy': '',
        'tip_mocaorpl': '',
        'tip_mocaorct': '',
        'tip_npsylan': '',
        'tip_npsylanx': '',
        'tip_craftvrs': '',
        'tip_crafturs': '',
        'tip_rey1rec': '',
        'tip_rey1int': '',
        'tip_rey2rec': '',
        'tip_rey2int': '',
        'tip_rey3rec': '',
        'tip_rey3int': '',
        'tip_rey4rec': '',
        'tip_rey4int': '',
        'tip_rey5rec': '',
        'tip_rey5int': '',
        'tip_rey6rec': '',
        'tip_rey6int': '',
        'tip_digforct': '',
        'tip_digforsl': '',
        'tip_digbacct': '',
        'tip_digbacls': '',
        'tip_otraila': '',
        'tip_otrlarr': '',
        'tip_otrlali': '',
        'tip_otrailb': '',
        'tip_otrlbrr': '',
        'tip_otrlbli': '',
        'tip_craftdvr': '',
        'tip_craftdre': '',
        'tip_craftdti': '',
        'tip_craftcue': '',
        'tip_animals': '',
        'tip_veg': '',
        'tip_udsverfc': '',
        'tip_udsverfn': '',
        'tip_udsvernf': '',
        'tip_udsverlc': '',
        'tip_udsverlr': '',
        'tip_udsverln': '',
        'tip_udsvertn': '',
        'tip_udsverte': '',
        'tip_udsverti': '',
        'tip_reydrec': '',
        'tip_reydint': '',
        'tip_reytcor': '',
        'tip_reyfpos': '',
        'tip_vnttotw': '',
        'tip_vntpcnc': '',
        'tip_cogstat': '',
        'tip_respval': '',
        'tip_resphear': '',
        'tip_respdist': '',
        'tip_respintr': '',
        'tip_respdisn': '',
        'tip_respfatg': '',
        'tip_respemot': '',
        'tip_respasst': '',
        'tip_respoth': '',
        'tip_respothx': '',
        'tip_dxmethod': '',
        'tip_normcog': '',
        'tip_demented': '',
        'tip_amndem': '',
        'tip_pca': '',
        'tip_ppasyn': '',
        'tip_ppasynt': '',
        'tip_ftdsyn': '',
        'tip_lbdsyn': '',
        'tip_namndem': '',
        'tip_mciamem': '',
        'tip_mciaplus': '',
        'tip_mciaplan': '',
        'tip_mciapatt': '',
        'tip_mciapex': '',
        'tip_mciapvis': '',
        'tip_mcinon1': '',
        'tip_mcin1lan': '',
        'tip_mcin1att': '',
        'tip_mcin1ex': '',
        'tip_mcin1vis': '',
        'tip_mcinon2': '',
        'tip_mcin2lan': '',
        'tip_mcin2att': '',
        'tip_mcin2ex': '',
        'tip_mcin2vis': '',
        'tip_impnomci': '',
        'tip_amylpet': '',
        'tip_amylcsf': '',
        'tip_fdgad': '',
        'tip_hippatr': '',
        'tip_taupetad': '',
        'tip_csftau': '',
        'tip_fdgftld': '',
        'tip_tpetftld': '',
        'tip_mrftld': '',
        'tip_datscan': '',
        'tip_othbiom': '',
        'tip_othbiomx': '',
        'tip_imaglinf': '',
        'tip_imaglac': '',
        'tip_imagmach': '',
        'tip_imagmich': '',
        'tip_imagmwmh': '',
        'tip_imagewmh': '',
        'tip_admut': '',
        'tip_ftldmut': '',
        'tip_othmut': '',
        'tip_othmutx': '',
        'tip_alzdis': '',
        'tip_alzdisif': '',
        'tip_lbdis': '',
        'tip_lbdif': '',
        'tip_park': '',
        'tip_msa': '',
        'tip_msaif': '',
        'tip_psp': '',
        'tip_pspif': '',
        'tip_cort': '',
        'tip_cortif': '',
        'tip_ftldmo': '',
        'tip_ftldmoif': '',
        'tip_ftldnos': '',
        'tip_ftldnoif': '',
        'tip_ftldsubt': '',
        'tip_ftldsubx': '',
        'tip_cvd': '',
        'tip_cvdif': '',
        'tip_prevstk': '',
        'tip_strokdec': '',
        'tip_stkimag': '',
        'tip_infnetw': '',
        'tip_infwmh': '',
        'tip_esstrem': '',
        'tip_esstreif': '',
        'tip_downs': '',
        'tip_downsif': '',
        'tip_hunt': '',
        'tip_huntif': '',
        'tip_prion': '',
        'tip_prionif': '',
        'tip_brninj': '',
        'tip_brninjif': '',
        'tip_brnincte': '',
        'tip_hyceph': '',
        'tip_hycephif': '',
        'tip_epilep': '',
        'tip_epilepif': '',
        'tip_neop': '',
        'tip_neopif': '',
        'tip_neopstat': '',
        'tip_hiv': '',
        'tip_hivif': '',
        'tip_othcog': '',
        'tip_othcogif': '',
        'tip_othcogx': '',
        'tip_dep': '',
        'tip_depif': '',
        'tip_deptreat': '',
        'tip_bipoldx': '',
        'tip_bipoldif': '',
        'tip_schizop': '',
        'tip_schizoif': '',
        'tip_anxiet': '',
        'tip_anxietif': '',
        'tip_delir': '',
        'tip_delirif': '',
        'tip_ptsddx': '',
        'tip_ptsddxif': '',
        'tip_othpsy': '',
        'tip_othpsyif': '',
        'tip_othpsyx': '',
        'tip_alcdem': '',
        'tip_alcdemif': '',
        'tip_alcabuse': '',
        'tip_impsub': '',
        'tip_impsubif': '',
        'tip_dysill': '',
        'tip_dysillif': '',
        'tip_meds': '',
        'tip_medsif': '',
        'tip_cogoth': '',
        'tip_cogothif': '',
        'tip_cogothx': '',
        'tip_cogoth2': '',
        'tip_cogoth2f': '',
        'tip_cogoth2x': '',
        'tip_cogoth3': '',
        'tip_cogoth3f': '',
        'tip_cogoth3x': '',
        'tip_cancer': '',
        'tip_cancsite': '',
        'tip_diabet': '',
        'tip_myoinf': '',
        'tip_conghrt': '',
        'tip_afibrill': '',
        'tip_hypert': '',
        'tip_angina': '',
        'tip_hypchol': '',
        'tip_vb12def': '',
        'tip_thydis': '',
        'tip_arth': '',
        'tip_artype': '',
        'tip_artypex': '',
        'tip_artupex': '',
        'tip_artloex': '',
        'tip_artspin': '',
        'tip_artunkn': '',
        'tip_urineinc': '',
        'tip_bowlinc': '',
        'tip_sleepap': '',
        'tip_remdis': '',
        'tip_hyposom': '',
        'tip_sleepoth': '',
        'tip_sleepotx': '',
        'tip_angiocp': '',
        'tip_angiopci': '',
        'tip_pacemake': '',
        'tip_hvalve': '',
        'tip_antienc': '',
        'tip_antiencx': '',
        'tip_othcond': '',
        'tip_othcondx': ''
    }


if __name__ == "__main__":
    unittest.main()
