import unittest

from nacc import redcap2nacc
from nacc.uds3 import blanks
from nacc.uds3.tfp.v3_2.builder import build_uds3_tfp_new_form


class option():
    flag = 'tfp'
    iorf = False
    cv = False
    csf = False
    lbd = False
    tip = False
    tfp = True
    tfp3 = False
    ftld = False
    ivp = False
    fvp = False


class TestBlankRulesForTFP(unittest.TestCase):
    '''
    These tests are designed to run TFP data fields (generated below
    the tests here) through the check_blanks function for the UDS3 module.
    It is mostly concerned with making sure the "special cases" are functioning
    properly. There should be overlap with the ivp and fvp blanks.
    '''

    def setUp(self):
        self.options = option()

    def test_for_filled_when_ruled_blank(self):
        '''
        Have it look for the langa4 error to see that general blanking rules
        are working (langa4 also comes before the variable (a4sub)
        it's dependent on)
        '''
        record = make_filled_form()
        record['tele_a4sub'] = '0'
        record['tele_langa4'] = '1'
        fpacket = build_uds3_tfp_new_form(record)
        warnings = []

        warnings = redcap2nacc.check_blanks(fpacket, self.options)
        expected = ["LANGA4 in form Z1X is '1' with length '1', but should be"
                    " blank: 'Blank if Question 5b A4SUB = 0 (No)'."]
        self.assertEqual(warnings, expected)

    def test_for_single_blanking_rule_returned(self):
        '''
        Have it make sure that only one error is returned from a list of
        rules when not working with special cases (special cases need a
        fix in a later feature branch)
        '''
        record = make_filled_form()
        record['tele_mocalan'] = '2'
        record['tele_mocalanx'] = 'a'
        fpacket = build_uds3_tfp_new_form(record)
        warnings = []

        warnings = redcap2nacc.check_blanks(fpacket, self.options)
        expected = ["MOCALANX in form C2  is 'a                               "
                    "                            ' with length '60', but"
                    " should be blank: 'Blank if 1b MOCALAN = 2'."]
        self.assertEqual(warnings, expected)

    def test_for_special_case_FTLDSUBT(self):
        '''
        FTLDSUBT in D1 should be blank if all of the parts of 14 is not marked
        as '1'
        '''
        record = make_filled_form()
        record['tele_psp'] = ''
        record['tele_cort'] = ''
        record['tele_ftldmo'] = ''
        record['tele_ftldnos'] = ''
        record['tele_ftldsubt'] = '1'
        fpacket = build_uds3_tfp_new_form(record)
        warnings = []

        warnings = redcap2nacc.check_blanks(fpacket, self.options)
        expected = ["FTLDSUBT in form D1  is '1' with length '1', but should"
                    " be blank: 'Blank if Question 14a PSP ne 1 and Question"
                    " 14b CORT ne 1 and Question 14c FTLDMO ne 1 and 14d"
                    " FTLDNOS ne 1'."]
        self.assertEqual(warnings, expected)

    def test_multiple_rules_returned(self):
        '''
        Form C2 has several fields that are defaulted to 0, but should be
        blank if RESPVAL is '1'.
        '''
        record = make_filled_form()
        record['tele_respval'] = '1'
        record['tele_resphear'] = '0'
        record['tele_respdist'] = '0'
        record['tele_respintr'] = '0'
        record['tele_respdisn'] = '0'
        record['tele_respfatg'] = '0'
        record['tele_respemot'] = '0'
        record['tele_respasst'] = '0'
        record['tele_respoth'] = '0'
        fpacket = build_uds3_tfp_new_form(record)
        warnings = []

        warnings = redcap2nacc.check_blanks(fpacket, self.options)
        expected = ["RESPHEAR in form C2  is '0' with length '1', but should be blank: 'Blank if 14a RESPVAL = 1'.", "RESPDIST in form C2  is '0' with length '1', but should be blank: 'Blank if 14a RESPVAL = 1'.", "RESPINTR in form C2  is '0' with length '1', but should be blank: 'Blank if 14a RESPVAL = 1'.", "RESPDISN in form C2  is '0' with length '1', but should be blank: 'Blank if 14a RESPVAL = 1'.", "RESPFATG in form C2  is '0' with length '1', but should be blank: 'Blank if 14a RESPVAL = 1'.", "RESPEMOT in form C2  is '0' with length '1', but should be blank: 'Blank if 14a RESPVAL = 1'.", "RESPASST in form C2  is '0' with length '1', but should be blank: 'Blank if 14a RESPVAL = 1'.", "RESPOTH in form C2  is '0' with length '1', but should be blank: 'Blank if 14a RESPVAL = 1'."]
        self.assertEqual(warnings, expected)

    def test_set_zeros_to_blanks(self):
        '''
        To follow up with the last test, the function within uds3.blanks.py
        "set_zeros_to_blanks" should clear up these errors.
        '''
        record = make_filled_form()
        record['tele_respval'] = '1'
        record['tele_resphear'] = '0'
        record['tele_respdist'] = '0'
        record['tele_respintr'] = '0'
        record['tele_respdisn'] = '0'
        record['tele_respfatg'] = '0'
        record['tele_respemot'] = '0'
        record['tele_respasst'] = '0'
        record['tele_respoth'] = '0'
        fpacket = build_uds3_tfp_new_form(record)
        warnings = []

        blanks.set_zeros_to_blanks(fpacket)

        warnings = redcap2nacc.check_blanks(fpacket, self.options)
        expected = []

        self.assertEqual(warnings, expected)


def make_filled_form() -> dict:
    return {
        # Headers
        'ptid': '1',
        'adcid': '2',
        'visitmo': '1',
        'visitday': '1',
        'visityr': '2020',
        'visitnum': '3',
        'initials': '',
        # filled forms
        'tele_langt1': '',
        'tele_langa1': '',
        'tele_langa2': '',
        'tele_a2sub': '',
        'tele_a2not': '',
        'tele_langa3': '',
        'tele_a3sub': '',
        'tele_langa4': '',
        'tele_a4sub': '',
        'tele_a4not': '',
        'tele_langb1': '',
        'tele_b1sub': '',
        'tele_b1not': '',
        'tele_langb4': '',
        'tele_langb5': '',
        'tele_b5sub': '',
        'tele_b5not': '',
        'tele_langb6': '',
        'tele_b6sub': '',
        'tele_b6not': '',
        'tele_langb7': '',
        'tele_b7sub': '',
        'tele_b7not': '',
        'tele_langb9': '',
        'tele_langc2': '',
        'tele_c2sub': '1',
        'tele_c2not': '',
        'tele_langd1': '',
        'tele_langd2': '',
        'tele_langa3a': '',
        'tele_ftda3afs': '',
        'tele_ftda3afr': '',
        'tele_langb3f': '',
        'tele_langb9f': '',
        'tele_langc1f': '',
        'tele_langc2f': '',
        'tele_langc3f': '',
        'tele_langc4f': '',
        'tele_ftdc4fs': '',
        'tele_ftdc4fr': '',
        'tele_ftdc5fs': '',
        'tele_ftdc5fr': '',
        'tele_ftdc6fs': '',
        'tele_ftdc6fr': '',
        'tele_lange2f': '',
        'tele_lange3f': '',
        'tele_langcls': '',
        'tele_clssub': '',
        'tvp_z1x_checklist_complete': '2',
        'telcog': '',
        'telill': '',
        'telhome': '',
        'telrefu': '',
        'telcov': '',
        'telothr': '',
        'telothrx': '',
        'telmod': '',
        'telinper': '',
        'telmile': '',
        'tele_birthmo': '',
        'tele_birthyr': '',
        'tele_maristat': '',
        'tele_sex': '',
        'tele_livsitua': '',
        'tele_independ': '',
        'tele_residenc': '',
        'tele_zip': '',
        'tele_inbirmo': '',
        'tele_inbiryr': '',
        'tele_insex': '',
        'tele_newinf': '',
        'tele_inhisp': '',
        'tele_inhispor': '',
        'tele_inhispox': '',
        'tele_inrace': '',
        'tele_inracex': '',
        'tele_inrasec': '',
        'tele_inrasecx': '',
        'tele_inrater': '',
        'tele_inraterx': '',
        'tele_ineduc': '',
        'tele_inrelto': '',
        'tele_inknown': '',
        'tele_inlivwth': '',
        'tele_invisits': '',
        'tele_incalls': '',
        'tele_inrely': '',
        'tele_nwinfmut': '',
        'tele_fadmut': '',
        'tele_fadmutx': '',
        'tele_fadmuso': '',
        'tele_fadmusox': '',
        'tele_fftdmut': '',
        'tele_fftdmutx': '',
        'tele_fftdmuso': '',
        'tele_fftdmusx': '',
        'tele_fothmut': '',
        'tele_fothmutx': '',
        'tele_fothmuso': '',
        'tele_fothmusx': '',
        'tele_nwinfpar': '',
        'tele_mommob': '',
        'tele_momyob': '',
        'tele_momdage': '',
        'tele_momneur': '',
        'tele_momprdx': '',
        'tele_mommoe': '',
        'tele_momageo': '',
        'tele_dadmob': '',
        'tele_dadyob': '',
        'tele_daddage': '',
        'tele_dadneur': '',
        'tele_dadprdx': '',
        'tele_dadmoe': '',
        'tele_dadageo': '',
        'tele_sibs': '',
        'tele_nwinfsib': '',
        'tele_sib1mob': '',
        'tele_sib1yob': '',
        'tele_sib1agd': '',
        'tele_sib1neu': '',
        'tele_sib1pdx': '',
        'tele_sib1moe': '',
        'tele_sib1ago': '',
        'tele_sib2mob': '',
        'tele_sib2yob': '',
        'tele_sib2agd': '',
        'tele_sib2neu': '',
        'tele_sib2pdx': '',
        'tele_sib2moe': '',
        'tele_sib2ago': '',
        'tele_sib3mob': '',
        'tele_sib3yob': '',
        'tele_sib3agd': '',
        'tele_sib3neu': '',
        'tele_sib3pdx': '',
        'tele_sib3moe': '',
        'tele_sib3ago': '',
        'tele_sib4mob': '',
        'tele_sib4yob': '',
        'tele_sib4agd': '',
        'tele_sib4neu': '',
        'tele_sib4pdx': '',
        'tele_sib4moe': '',
        'tele_sib4ago': '',
        'tele_sib5mob': '',
        'tele_sib5yob': '',
        'tele_sib5agd': '',
        'tele_sib5neu': '',
        'tele_sib5pdx': '',
        'tele_sib5moe': '',
        'tele_sib5ago': '',
        'tele_sib6mob': '',
        'tele_sib6yob': '',
        'tele_sib6agd': '',
        'tele_sib6neu': '',
        'tele_sib6pdx': '',
        'tele_sib6moe': '',
        'tele_sib6ago': '',
        'tele_sib7mob': '',
        'tele_sib7yob': '',
        'tele_sib7agd': '',
        'tele_sib7neu': '',
        'tele_sib7pdx': '',
        'tele_sib7moe': '',
        'tele_sib7ago': '',
        'tele_sib8mob': '',
        'tele_sib8yob': '',
        'tele_sib8agd': '',
        'tele_sib8neu': '',
        'tele_sib8pdx': '',
        'tele_sib8moe': '',
        'tele_sib8ago': '',
        'tele_sib9mob': '',
        'tele_sib9yob': '',
        'tele_sib9agd': '',
        'tele_sib9neu': '',
        'tele_sib9pdx': '',
        'tele_sib9moe': '',
        'tele_sib9ago': '',
        'tele_sib10mob': '',
        'tele_sib10yob': '',
        'tele_sib10agd': '',
        'tele_sib10neu': '',
        'tele_sib10pdx': '',
        'tele_sib10moe': '',
        'tele_sib10ago': '',
        'tele_sib11mob': '',
        'tele_sib11yob': '',
        'tele_sib11agd': '',
        'tele_sib11neu': '',
        'tele_sib11pdx': '',
        'tele_sib11moe': '',
        'tele_sib11ago': '',
        'tele_sib12mob': '',
        'tele_sib12yob': '',
        'tele_sib12agd': '',
        'tele_sib12neu': '',
        'tele_sib12pdx': '',
        'tele_sib12moe': '',
        'tele_sib12ago': '',
        'tele_sib13mob': '',
        'tele_sib13yob': '',
        'tele_sib13agd': '',
        'tele_sib13neu': '',
        'tele_sib13pdx': '',
        'tele_sib13moe': '',
        'tele_sib13ago': '',
        'tele_sib14mob': '',
        'tele_sib14yob': '',
        'tele_sib14agd': '',
        'tele_sib14neu': '',
        'tele_sib14pdx': '',
        'tele_sib14moe': '',
        'tele_sib14ago': '',
        'tele_sib15mob': '',
        'tele_sib15yob': '',
        'tele_sib15agd': '',
        'tele_sib15neu': '',
        'tele_sib15pdx': '',
        'tele_sib15moe': '',
        'tele_sib15ago': '',
        'tele_sib16mob': '',
        'tele_sib16yob': '',
        'tele_sib16agd': '',
        'tele_sib16neu': '',
        'tele_sib16pdx': '',
        'tele_sib16moe': '',
        'tele_sib16ago': '',
        'tele_sib17mob': '',
        'tele_sib17yob': '',
        'tele_sib17agd': '',
        'tele_sib17neu': '',
        'tele_sib17pdx': '',
        'tele_sib17moe': '',
        'tele_sib17ago': '',
        'tele_sib18mob': '',
        'tele_sib18yob': '',
        'tele_sib18agd': '',
        'tele_sib18neu': '',
        'tele_sib18pdx': '',
        'tele_sib18moe': '',
        'tele_sib18ago': '',
        'tele_sib19mob': '',
        'tele_sib19yob': '',
        'tele_sib19agd': '',
        'tele_sib19neu': '',
        'tele_sib19pdx': '',
        'tele_sib19moe': '',
        'tele_sib19ago': '',
        'tele_sib20mob': '',
        'tele_sib20yob': '',
        'tele_sib20agd': '',
        'tele_sib20neu': '',
        'tele_sib20pdx': '',
        'tele_sib20moe': '',
        'tele_sib20ago': '',
        'tele_kids': '',
        'tele_nwinfkid': '',
        'tele_kid1mob': '',
        'tele_kid1yob': '',
        'tele_kid1agd': '',
        'tele_kid1neu': '',
        'tele_kid1pdx': '',
        'tele_kid1moe': '',
        'tele_kid1ago': '',
        'tele_kid2mob': '',
        'tele_kid2yob': '',
        'tele_kid2agd': '',
        'tele_kid2neu': '',
        'tele_kid2pdx': '',
        'tele_kid2moe': '',
        'tele_kid2ago': '',
        'tele_kid3mob': '',
        'tele_kid3yob': '',
        'tele_kid3agd': '',
        'tele_kid3neu': '',
        'tele_kid3pdx': '',
        'tele_kid3moe': '',
        'tele_kid3ago': '',
        'tele_kid4mob': '',
        'tele_kid4yob': '',
        'tele_kid4agd': '',
        'tele_kid4neu': '',
        'tele_kid4pdx': '',
        'tele_kid4moe': '',
        'tele_kid4ago': '',
        'tele_kid5mob': '',
        'tele_kid5yob': '',
        'tele_kid5agd': '',
        'tele_kid5neu': '',
        'tele_kid5pdx': '',
        'tele_kid5moe': '',
        'tele_kid5ago': '',
        'tele_kid6mob': '',
        'tele_kid6yob': '',
        'tele_kid6agd': '',
        'tele_kid6neu': '',
        'tele_kid6pdx': '',
        'tele_kid6moe': '',
        'tele_kid6ago': '',
        'tele_kid7mob': '',
        'tele_kid7yob': '',
        'tele_kid7agd': '',
        'tele_kid7neu': '',
        'tele_kid7pdx': '',
        'tele_kid7moe': '',
        'tele_kid7ago': '',
        'tele_kid8mob': '',
        'tele_kid8yob': '',
        'tele_kid8agd': '',
        'tele_kid8neu': '',
        'tele_kid8pdx': '',
        'tele_kid8moe': '',
        'tele_kid8ago': '',
        'tele_kid9mob': '',
        'tele_kid9yob': '',
        'tele_kid9agd': '',
        'tele_kid9neu': '',
        'tele_kid9pdx': '',
        'tele_kid9moe': '',
        'tele_kid9ago': '',
        'tele_kid10mob': '',
        'tele_kid10yob': '',
        'tele_kid10agd': '',
        'tele_kid10neu': '',
        'tele_kid10pdx': '',
        'tele_kid10moe': '',
        'tele_kid10ago': '',
        'tele_kid11mob': '',
        'tele_kid11yob': '',
        'tele_kid11agd': '',
        'tele_kid11neu': '',
        'tele_kid11pdx': '',
        'tele_kid11moe': '',
        'tele_kid11ago': '',
        'tele_kid12mob': '',
        'tele_kid12yob': '',
        'tele_kid12agd': '',
        'tele_kid12neu': '',
        'tele_kid12pdx': '',
        'tele_kid12moe': '',
        'tele_kid12ago': '',
        'tele_kid13mob': '',
        'tele_kid13yob': '',
        'tele_kid13agd': '',
        'tele_kid13neu': '',
        'tele_kid13pdx': '',
        'tele_kid13moe': '',
        'tele_kid13ago': '',
        'tele_kid14mob': '',
        'tele_kid14yob': '',
        'tele_kid14agd': '',
        'tele_kid14neu': '',
        'tele_kid14pdx': '',
        'tele_kid14moe': '',
        'tele_kid14ago': '',
        'tele_kid15mob': '',
        'tele_kid15yob': '',
        'tele_kid15agd': '',
        'tele_kid15neu': '',
        'tele_kid15pdx': '',
        'tele_kid15moe': '',
        'tele_kid15ago': '',
        'tele_anymeds': '',
        'tele_height': '',
        'tele_weight': '',
        'tele_bpsys': '',
        'tele_bpdias': '',
        'tele_hrate': '',
        'tele_vision': '',
        'tele_viscorr': '',
        'tele_viswcorr': '',
        'tele_hearing': '',
        'tele_hearaid': '',
        'tele_hearwaid': '',
        'tele_memory': '',
        'tele_orient': '',
        'tele_judgment': '',
        'tele_commun': '',
        'tele_homehobb': '',
        'tele_perscare': '',
        'tele_cdrsum': '',
        'tele_cdrglob': '',
        'tele_comport': '',
        'tele_cdrlang': '',
        'tele_npiqinf': '',
        'tele_npiqinfx': '',
        'tele_del': '',
        'tele_delsev': '',
        'tele_hall': '',
        'tele_hallsev': '',
        'tele_agit': '',
        'tele_agitsev': '',
        'tele_depd': '',
        'tele_depdsev': '',
        'tele_anx': '',
        'tele_anxsev': '',
        'tele_elat': '',
        'tele_elatsev': '',
        'tele_apa': '',
        'tele_apasev': '',
        'tele_disn': '',
        'tele_disnsev': '',
        'tele_irr': '',
        'tele_irrsev': '',
        'tele_mot': '',
        'tele_motsev': '',
        'tele_nite': '',
        'tele_nitesev': '',
        'tele_app': '',
        'tele_appsev': '',
        'tele_nogds': '',
        'tele_satis': '',
        'tele_dropact': '',
        'tele_empty': '',
        'tele_bored': '',
        'tele_spirits': '',
        'tele_afraid': '',
        'tele_happy': '',
        'tele_helpless': '',
        'tele_stayhome': '',
        'tele_memprob': '',
        'tele_wondrful': '',
        'tele_wrthless': '',
        'tele_energy': '',
        'tele_hopeless': '',
        'tele_better': '',
        'tele_gds': '',
        'tele_bills': '',
        'tele_taxes': '',
        'tele_shopping': '',
        'tele_games': '',
        'tele_stove': '',
        'tele_mealprep': '',
        'tele_events': '',
        'tele_payattn': '',
        'tele_remdates': '',
        'tele_travel': '',
        'tele_decsub': '',
        'tele_decin': '',
        'tele_decclcog': '',
        'tele_cogmem': '',
        'tele_cogori': '',
        'tele_cogjudg': '',
        'tele_coglang': '',
        'tele_cogvis': '',
        'tele_cogattn': '',
        'tele_cogfluc': '',
        'tele_cogflago': '',
        'tele_cogothr': '',
        'tele_cogothrx': '',
        'tele_cogfpred': '',
        'tele_cogfprex': '',
        'tele_cogmode': '',
        'tele_cogmodex': '',
        'tele_decage': '',
        'tele_decclbe': '',
        'tele_beapathy': '',
        'tele_bedep': '',
        'tele_bevhall': '',
        'tele_bevwell': '',
        'tele_bevhago': '',
        'tele_beahall': '',
        'tele_bedel': '',
        'tele_bedisin': '',
        'tele_beirrit': '',
        'tele_beagit': '',
        'tele_beperch': '',
        'tele_berem': '',
        'tele_beremago': '',
        'tele_beanx': '',
        'tele_beothr': '',
        'tele_beothrx': '',
        'tele_befpred': '',
        'tele_befpredx': '',
        'tele_bemode': '',
        'tele_bemodex': '',
        'tele_beage': '',
        'tele_decclmot': '',
        'tele_mogait': '',
        'tele_mofalls': '',
        'tele_motrem': '',
        'tele_moslow': '',
        'tele_mofrst': '',
        'tele_momode': '',
        'tele_momodex': '',
        'tele_momopark': '',
        'tele_parkage': '',
        'tele_momoals': '',
        'tele_alsage': '',
        'tele_moage': '',
        'tele_course': '',
        'tele_frstchg': '',
        'tele_lbdeval': '',
        'tele_ftldeval': '',
        'tele_modcomm': '',
        'tele_mocacomp': '',
        'tele_mocareas': '',
        'tele_mocalan': '',
        'tele_mocalanx': '',
        'tele_mocahear': '',
        'tele_mocbtots': '',
        'tele_mocadigi': '',
        'tele_mocalett': '',
        'tele_mocaser7': '',
        'tele_mocarepe': '',
        'tele_mocaflue': '',
        'tele_mocaabst': '',
        'tele_mocarecn': '',
        'tele_mocarecc': '',
        'tele_mocarecr': '',
        'tele_mocaordt': '',
        'tele_mocaormo': '',
        'tele_mocaoryr': '',
        'tele_mocaordy': '',
        'tele_mocaorpl': '',
        'tele_mocaorct': '',
        'tele_npsylan': '',
        'tele_npsylanx': '',
        'tele_craftvrs': '',
        'tele_crafturs': '',
        'tele_rey1rec': '',
        'tele_rey1int': '',
        'tele_rey2rec': '',
        'tele_rey2int': '',
        'tele_rey3rec': '',
        'tele_rey3int': '',
        'tele_rey4rec': '',
        'tele_rey4int': '',
        'tele_rey5rec': '',
        'tele_rey5int': '',
        'tele_rey6rec': '',
        'tele_rey6int': '',
        'tele_digforct': '',
        'tele_digforsl': '',
        'tele_digbacct': '',
        'tele_digbacls': '',
        'tele_otraila': '',
        'tele_otrlarr': '',
        'tele_otrlali': '',
        'tele_otrailb': '',
        'tele_otrlbrr': '',
        'tele_otrlbli': '',
        'tele_craftdvr': '',
        'tele_craftdre': '',
        'tele_craftdti': '',
        'tele_craftcue': '',
        'tele_animals': '',
        'tele_veg': '',
        'tele_udsverfc': '',
        'tele_udsverfn': '',
        'tele_udsvernf': '',
        'tele_udsverlc': '',
        'tele_udsverlr': '',
        'tele_udsverln': '',
        'tele_udsvertn': '',
        'tele_udsverte': '',
        'tele_udsverti': '',
        'tele_reydrec': '',
        'tele_reydint': '',
        'tele_reytcor': '',
        'tele_reyfpos': '',
        'tele_vnttotw': '',
        'tele_vntpcnc': '',
        'tele_cogstat': '',
        'tele_respval': '',
        'tele_resphear': '',
        'tele_respdist': '',
        'tele_respintr': '',
        'tele_respdisn': '',
        'tele_respfatg': '',
        'tele_respemot': '',
        'tele_respasst': '',
        'tele_respoth': '',
        'tele_respothx': '',
        'tele_dxmethod': '',
        'tele_normcog': '',
        'tele_demented': '',
        'tele_amndem': '',
        'tele_pca': '',
        'tele_ppasyn': '',
        'tele_ppasynt': '',
        'tele_ftdsyn': '',
        'tele_lbdsyn': '',
        'tele_namndem': '',
        'tele_mciamem': '',
        'tele_mciaplus': '',
        'tele_mciaplan': '',
        'tele_mciapatt': '',
        'tele_mciapex': '',
        'tele_mciapvis': '',
        'tele_mcinon1': '',
        'tele_mcin1lan': '',
        'tele_mcin1att': '',
        'tele_mcin1ex': '',
        'tele_mcin1vis': '',
        'tele_mcinon2': '',
        'tele_mcin2lan': '',
        'tele_mcin2att': '',
        'tele_mcin2ex': '',
        'tele_mcin2vis': '',
        'tele_impnomci': '',
        'tele_amylpet': '',
        'tele_amylcsf': '',
        'tele_fdgad': '',
        'tele_hippatr': '',
        'tele_taupetad': '',
        'tele_csftau': '',
        'tele_fdgftld': '',
        'tele_tpetftld': '',
        'tele_mrftld': '',
        'tele_datscan': '',
        'tele_othbiom': '',
        'tele_othbiomx': '',
        'tele_imaglinf': '',
        'tele_imaglac': '',
        'tele_imagmach': '',
        'tele_imagmich': '',
        'tele_imagmwmh': '',
        'tele_imagewmh': '',
        'tele_admut': '',
        'tele_ftldmut': '',
        'tele_othmut': '',
        'tele_othmutx': '',
        'tele_alzdis': '',
        'tele_alzdisif': '',
        'tele_lbdis': '',
        'tele_lbdif': '',
        'tele_park': '',
        'tele_msa': '',
        'tele_msaif': '',
        'tele_psp': '',
        'tele_pspif': '',
        'tele_cort': '',
        'tele_cortif': '',
        'tele_ftldmo': '',
        'tele_ftldmoif': '',
        'tele_ftldnos': '',
        'tele_ftldnoif': '',
        'tele_ftldsubt': '',
        'tele_ftldsubx': '',
        'tele_cvd': '',
        'tele_cvdif': '',
        'tele_prevstk': '',
        'tele_strokdec': '',
        'tele_stkimag': '',
        'tele_infnetw': '',
        'tele_infwmh': '',
        'tele_esstrem': '',
        'tele_esstreif': '',
        'tele_downs': '',
        'tele_downsif': '',
        'tele_hunt': '',
        'tele_huntif': '',
        'tele_prion': '',
        'tele_prionif': '',
        'tele_brninj': '',
        'tele_brninjif': '',
        'tele_brnincte': '',
        'tele_hyceph': '',
        'tele_hycephif': '',
        'tele_epilep': '',
        'tele_epilepif': '',
        'tele_neop': '',
        'tele_neopif': '',
        'tele_neopstat': '',
        'tele_hiv': '',
        'tele_hivif': '',
        'tele_othcog': '',
        'tele_othcogif': '',
        'tele_othcogx': '',
        'tele_dep': '',
        'tele_depif': '',
        'tele_deptreat': '',
        'tele_bipoldx': '',
        'tele_bipoldif': '',
        'tele_schizop': '',
        'tele_schizoif': '',
        'tele_anxiet': '',
        'tele_anxietif': '',
        'tele_delir': '',
        'tele_delirif': '',
        'tele_ptsddx': '',
        'tele_ptsddxif': '',
        'tele_othpsy': '',
        'tele_othpsyif': '',
        'tele_othpsyx': '',
        'tele_alcdem': '',
        'tele_alcdemif': '',
        'tele_alcabuse': '',
        'tele_impsub': '',
        'tele_impsubif': '',
        'tele_dysill': '',
        'tele_dysillif': '',
        'tele_meds': '',
        'tele_medsif': '',
        'tele_cogoth': '',
        'tele_cogothif': '',
        'tele_cogothx': '',
        'tele_cogoth2': '',
        'tele_cogoth2f': '',
        'tele_cogoth2x': '',
        'tele_cogoth3': '',
        'tele_cogoth3f': '',
        'tele_cogoth3x': '',
        'tele_cancer': '',
        'tele_cancsite': '',
        'tele_diabet': '',
        'tele_myoinf': '',
        'tele_conghrt': '',
        'tele_afibrill': '',
        'tele_hypert': '',
        'tele_angina': '',
        'tele_hypchol': '',
        'tele_vb12def': '',
        'tele_thydis': '',
        'tele_arth': '',
        'tele_artype': '',
        'tele_artypex': '',
        'tele_artupex': '',
        'tele_artloex': '',
        'tele_artspin': '',
        'tele_artunkn': '',
        'tele_urineinc': '',
        'tele_bowlinc': '',
        'tele_sleepap': '',
        'tele_remdis': '',
        'tele_hyposom': '',
        'tele_sleepoth': '',
        'tele_sleepotx': '',
        'tele_angiocp': '',
        'tele_angiopci': '',
        'tele_pacemake': '',
        'tele_hvalve': '',
        'tele_antienc': '',
        'tele_antiencx': '',
        'tele_othcond': '',
        'tele_othcondx': '',
    }


if __name__ == "__main__":
    unittest.main()
