###############################################################################
# Copyright 2015-2023 University of Florida. All rights reserved.
# This file is part of UF CTS-IT's NACCulator project.
# Use of this source code is governed by the license found in the LICENSE file.
###############################################################################

import sys

from nacc.uds3.tip import forms as tip_forms
from nacc.uds3 import packet as tip_packet


def build_uds3_tip_form(record, err=sys.stderr):
    """
    Populates a Packet object with a record of REDCap CSV-read data (list of
    TIP Form objects)
    """
    packet = tip_packet.Packet()

    # Set up the forms
    add_z1x(record, packet)
    add_t1(record, packet)
    add_a1(record, packet)
    add_a2(record, packet)
    try:
        z1x_complete = record['tip_z1x_checklist_complete']
    except KeyError:
        try:
            z1x_complete = record['tip_z1x_complete']
        except KeyError:
            z1x_complete = '0'
    if z1x_complete in ['1', '2']:
        try:
            if record['tip_a3sub'] == '1':
                add_a3(record, packet)
        except KeyError:
            pass
        try:
            if record['tip_a4sub'] == '1':
                add_a4(record, packet)
        except KeyError:
            pass
        add_a5(record, packet)
        add_b1(record, packet)
        add_b4(record, packet)
        try:
            if record['tip_b5sub'] == '1':
                add_b5(record, packet)
        except KeyError:
            pass
        try:
            if record['tip_b6sub'] == '1':
                add_b6(record, packet)
        except KeyError:
            pass
        try:
            if record['tip_b7sub'] == '1':
                add_b7(record, packet)
        except KeyError:
            pass
        try:
            if record['tip_b8sub'] == '1':
                add_b8(record, packet)
        except KeyError:
            pass

    add_b9(record, packet)
    add_c2t(record, packet)
    add_d1(record, packet)
    add_d2(record, packet)
    update_header(record, packet)

    return packet


def add_z1x(record, packet):
    # Forms T1, A1, A2, A5, B1, B4, B9, C2T, D1, and D2 are all REQUIRED.
    z1x = tip_forms.FormZ1X()
    z1x.LANGT1  = record['tip_LANGT1'.lower()]
    z1x.LANGA1  = record['tip_LANGA1'.lower()]
    z1x.LANGA2  = record['tip_LANGA2'.lower()]
    z1x.LANGA3  = record['tip_LANGA3'.lower()]
    z1x.A3SUB   = record['tip_A3SUB'.lower()]
    z1x.A3NOT   = record['tip_A3NOT'.lower()]
    z1x.LANGA4  = record['tip_LANGA4'.lower()]
    z1x.A4SUB   = record['tip_A4SUB'.lower()]
    z1x.A4NOT   = record['tip_A4NOT'.lower()]
    z1x.LANGA5  = record['tip_LANGA5'.lower()]
    z1x.LANGB1  = record['tip_LANGB1'.lower()]
    z1x.LANGB4  = record['tip_LANGB4'.lower()]
    z1x.LANGB5  = record['tip_LANGB5'.lower()]
    z1x.B5SUB   = record['tip_B5SUB'.lower()]
    z1x.B5NOT   = record['tip_B5NOT'.lower()]
    z1x.LANGB6  = record['tip_LANGB6'.lower()]
    z1x.B6SUB   = record['tip_B6SUB'.lower()]
    z1x.B6NOT   = record['tip_B6NOT'.lower()]
    z1x.LANGB7  = record['tip_LANGB7'.lower()]
    z1x.B7SUB   = record['tip_B7SUB'.lower()]
    z1x.B7NOT   = record['tip_B7NOT'.lower()]
    z1x.LANGB8  = record['tip_LANGB8'.lower()]
    z1x.B8SUB   = record['tip_B8SUB'.lower()]
    z1x.B8NOT   = record['tip_B8NOT'.lower()]
    z1x.LANGB9  = record['tip_LANGB9'.lower()]
    z1x.LANGC2T  = record['tip_LANGC2T'.lower()]
    z1x.LANGD1  = record['tip_LANGD1'.lower()]
    z1x.LANGD2  = record['tip_LANGD2'.lower()]
    z1x.LANGCLS = record['tip_LANGCLS'.lower()]
    z1x.CLSSUB  = record['tip_CLSSUB'.lower()]
    packet.insert(0, z1x)


def add_t1(record, packet):
    t1 = tip_forms.FormT1()
    t1.TELCOG   = record['tip_TELCOG'.lower()]
    t1.TELILL   = record['tip_TELILL'.lower()]
    t1.TELHOME  = record['tip_TELHOME'.lower()]
    t1.TELREFU  = record['tip_TELREFU'.lower()]
    t1.TELCOV   = record['tip_TELCOV'.lower()]
    t1.TELOTHR  = record['tip_TELOTHR'.lower()]
    t1.TELOTHRX = record['tip_TELOTHRX'.lower()]
    t1.TELMOD   = record['tip_TELMOD'.lower()]
    t1.TELINPER = record['tip_TELINPER'.lower()]
    t1.TELMILE  = record['tip_TELMILE'.lower()]
    packet.append(t1)


def add_a1(record, packet):
    a1 = tip_forms.FormA1()
    a1.REASON   = record['tip_REASON'.lower()]
    a1.REFERSC  = record['tip_REFERSC'.lower()]
    a1.LEARNED  = record['tip_LEARNED'.lower()]
    a1.PRESTAT  = record['tip_PRESTAT'.lower()]
    a1.PRESPART = record['tip_PRESPART'.lower()]
    a1.SOURCENW = record['tip_SOURCENW'.lower()]
    a1.BIRTHMO  = record['tip_BIRTHMO'.lower()]
    a1.BIRTHYR  = record['tip_BIRTHYR'.lower()]
    a1.SEX      = record['tip_SEX'.lower()]
    a1.HISPANIC = record['tip_HISPANIC'.lower()]
    a1.HISPOR   = record['tip_HISPOR'.lower()]
    a1.HISPORX  = record['tip_HISPORX'.lower()]
    a1.RACE     = record['tip_RACE'.lower()]
    a1.RACEX    = record['tip_RACEX'.lower()]
    a1.RACESEC  = record['tip_RACESEC'.lower()]
    a1.RACESECX = record['tip_RACESECX'.lower()]
    a1.RACETER  = record['tip_RACETER'.lower()]
    a1.RACETERX = record['tip_RACETERX'.lower()]
    a1.PRIMLANG = record['tip_PRIMLANG'.lower()]
    a1.PRIMLANX = record['tip_PRIMLANX'.lower()]
    a1.EDUC     = record['tip_EDUC'.lower()]
    a1.MARISTAT = record['tip_MARISTAT'.lower()]
    a1.LIVSITUA = record['tip_LIVSITUA'.lower()]
    a1.INDEPEND = record['tip_INDEPEND'.lower()]
    a1.RESIDENC = record['tip_RESIDENC'.lower()]
    a1.ZIP      = record['tip_ZIP'.lower()]
    a1.HANDED   = record['tip_HANDED'.lower()]
    packet.append(a1)


def add_a2(record, packet):
    a2 = tip_forms.FormA2()
    a2.INBIRMO   = record['tip_inbirmo']
    a2.INBIRYR   = record['tip_inbiryr']
    a2.INSEX     = record['tip_insex']
    a2.INHISP    = record['tip_inhisp']
    a2.INHISPOR  = record['tip_inhispor']
    a2.INHISPOX  = record['tip_inhispox']
    a2.INRACE    = record['tip_inrace']
    a2.INRACEX   = record['tip_inracex']
    a2.INRASEC   = record['tip_inrasec']
    a2.INRASECX  = record['tip_inrasecx']
    a2.INRATER   = record['tip_inrater']
    a2.INRATERX  = record['tip_inraterx']
    a2.INEDUC    = record['tip_ineduc']
    a2.INRELTO   = record['tip_inrelto']
    a2.INKNOWN   = record['tip_inknown']
    a2.INLIVWTH  = record['tip_inlivwth']
    a2.INVISITS  = record['tip_invisits']
    a2.INCALLS   = record['tip_incalls']
    a2.INRELY    = record['tip_inrely']
    packet.append(a2)


def add_a3(record, packet):
    a3 = tip_forms.FormA3()
    a3.AFFFAMM   = record['tip_afffamm']
    a3.FADMUT    = record['tip_fadmut']
    a3.FADMUTX   = record['tip_fadmutx']
    a3.FADMUSO   = record['tip_fadmuso']
    a3.FADMUSOX  = record['tip_fadmusox']
    a3.FFTDMUT   = record['tip_fftdmut']
    a3.FFTDMUTX  = record['tip_fftdmutx']
    a3.FFTDMUSO  = record['tip_fftdmuso']
    a3.FFTDMUSX  = record['tip_fftdmusx']
    a3.FOTHMUT   = record['tip_fothmut']
    a3.FOTHMUTX  = record['tip_fothmutx']
    a3.FOTHMUSO  = record['tip_fothmuso']
    a3.FOTHMUSX  = record['tip_fothmusx']
    a3.MOMMOB    = record['tip_mommob']
    a3.MOMYOB    = record['tip_momyob']
    a3.MOMDAGE   = record['tip_momdage']
    a3.MOMNEUR   = record['tip_momneur']
    a3.MOMPRDX   = record['tip_momprdx']
    a3.MOMMOE    = record['tip_mommoe']
    a3.MOMAGEO   = record['tip_momageo']
    a3.DADMOB    = record['tip_dadmob']
    a3.DADYOB    = record['tip_dadyob']
    a3.DADDAGE   = record['tip_daddage']
    a3.DADNEUR   = record['tip_dadneur']
    a3.DADPRDX   = record['tip_dadprdx']
    a3.DADMOE    = record['tip_dadmoe']
    a3.DADAGEO   = record['tip_dadageo']
    a3.SIBS      = record['tip_sibs']
    a3.SIB1MOB   = record['tip_sib1mob']
    a3.SIB1YOB   = record['tip_sib1yob']
    a3.SIB1AGD   = record['tip_sib1agd']
    a3.SIB1NEU   = record['tip_sib1neu']
    a3.SIB1PDX   = record['tip_sib1pdx']
    a3.SIB1MOE   = record['tip_sib1moe']
    a3.SIB1AGO   = record['tip_sib1ago']
    a3.SIB2MOB   = record['tip_sib2mob']
    a3.SIB2YOB   = record['tip_sib2yob']
    a3.SIB2AGD   = record['tip_sib2agd']
    a3.SIB2NEU   = record['tip_sib2neu']
    a3.SIB2PDX   = record['tip_sib2pdx']
    a3.SIB2MOE   = record['tip_sib2moe']
    a3.SIB2AGO   = record['tip_sib2ago']
    a3.SIB3MOB   = record['tip_sib3mob']
    a3.SIB3YOB   = record['tip_sib3yob']
    a3.SIB3AGD   = record['tip_sib3agd']
    a3.SIB3NEU   = record['tip_sib3neu']
    a3.SIB3PDX   = record['tip_sib3pdx']
    a3.SIB3MOE   = record['tip_sib3moe']
    a3.SIB3AGO   = record['tip_sib3ago']
    a3.SIB4MOB   = record['tip_sib4mob']
    a3.SIB4YOB   = record['tip_sib4yob']
    a3.SIB4AGD   = record['tip_sib4agd']
    a3.SIB4NEU   = record['tip_sib4neu']
    a3.SIB4PDX   = record['tip_sib4pdx']
    a3.SIB4MOE   = record['tip_sib4moe']
    a3.SIB4AGO   = record['tip_sib4ago']
    a3.SIB5MOB   = record['tip_sib5mob']
    a3.SIB5YOB   = record['tip_sib5yob']
    a3.SIB5AGD   = record['tip_sib5agd']
    a3.SIB5NEU   = record['tip_sib5neu']
    a3.SIB5PDX   = record['tip_sib5pdx']
    a3.SIB5MOE   = record['tip_sib5moe']
    a3.SIB5AGO   = record['tip_sib5ago']
    a3.SIB6MOB   = record['tip_sib6mob']
    a3.SIB6YOB   = record['tip_sib6yob']
    a3.SIB6AGD   = record['tip_sib6agd']
    a3.SIB6NEU   = record['tip_sib6neu']
    a3.SIB6PDX   = record['tip_sib6pdx']
    a3.SIB6MOE   = record['tip_sib6moe']
    a3.SIB6AGO   = record['tip_sib6ago']
    a3.SIB7MOB   = record['tip_sib7mob']
    a3.SIB7YOB   = record['tip_sib7yob']
    a3.SIB7AGD   = record['tip_sib7agd']
    a3.SIB7NEU   = record['tip_sib7neu']
    a3.SIB7PDX   = record['tip_sib7pdx']
    a3.SIB7MOE   = record['tip_sib7moe']
    a3.SIB7AGO   = record['tip_sib7ago']
    a3.SIB8MOB   = record['tip_sib8mob']
    a3.SIB8YOB   = record['tip_sib8yob']
    a3.SIB8AGD   = record['tip_sib8agd']
    a3.SIB8NEU   = record['tip_sib8neu']
    a3.SIB8PDX   = record['tip_sib8pdx']
    a3.SIB8MOE   = record['tip_sib8moe']
    a3.SIB8AGO   = record['tip_sib8ago']
    a3.SIB9MOB   = record['tip_sib9mob']
    a3.SIB9YOB   = record['tip_sib9yob']
    a3.SIB9AGD   = record['tip_sib9agd']
    a3.SIB9NEU   = record['tip_sib9neu']
    a3.SIB9PDX   = record['tip_sib9pdx']
    a3.SIB9MOE   = record['tip_sib9moe']
    a3.SIB9AGO   = record['tip_sib9ago']
    a3.SIB10MOB  = record['tip_sib10mob']
    a3.SIB10YOB  = record['tip_sib10yob']
    a3.SIB10AGD  = record['tip_sib10agd']
    a3.SIB10NEU  = record['tip_sib10neu']
    a3.SIB10PDX  = record['tip_sib10pdx']
    a3.SIB10MOE  = record['tip_sib10moe']
    a3.SIB10AGO  = record['tip_sib10ago']
    a3.SIB11MOB  = record['tip_sib11mob']
    a3.SIB11YOB  = record['tip_sib11yob']
    a3.SIB11AGD  = record['tip_sib11agd']
    a3.SIB11NEU  = record['tip_sib11neu']
    a3.SIB11PDX  = record['tip_sib11pdx']
    a3.SIB11MOE  = record['tip_sib11moe']
    a3.SIB11AGO  = record['tip_sib11ago']
    a3.SIB12MOB  = record['tip_sib12mob']
    a3.SIB12YOB  = record['tip_sib12yob']
    a3.SIB12AGD  = record['tip_sib12agd']
    a3.SIB12NEU  = record['tip_sib12neu']
    a3.SIB12PDX  = record['tip_sib12pdx']
    a3.SIB12MOE  = record['tip_sib12moe']
    a3.SIB12AGO  = record['tip_sib12ago']
    a3.SIB13MOB  = record['tip_sib13mob']
    a3.SIB13YOB  = record['tip_sib13yob']
    a3.SIB13AGD  = record['tip_sib13agd']
    a3.SIB13NEU  = record['tip_sib13neu']
    a3.SIB13PDX  = record['tip_sib13pdx']
    a3.SIB13MOE  = record['tip_sib13moe']
    a3.SIB13AGO  = record['tip_sib13ago']
    a3.SIB14MOB  = record['tip_sib14mob']
    a3.SIB14YOB  = record['tip_sib14yob']
    a3.SIB14AGD  = record['tip_sib14agd']
    a3.SIB14NEU  = record['tip_sib14neu']
    a3.SIB14PDX  = record['tip_sib14pdx']
    a3.SIB14MOE  = record['tip_sib14moe']
    a3.SIB14AGO  = record['tip_sib14ago']
    a3.SIB15MOB  = record['tip_sib15mob']
    a3.SIB15YOB  = record['tip_sib15yob']
    a3.SIB15AGD  = record['tip_sib15agd']
    a3.SIB15NEU  = record['tip_sib15neu']
    a3.SIB15PDX  = record['tip_sib15pdx']
    a3.SIB15MOE  = record['tip_sib15moe']
    a3.SIB15AGO  = record['tip_sib15ago']
    a3.SIB16MOB  = record['tip_sib16mob']
    a3.SIB16YOB  = record['tip_sib16yob']
    a3.SIB16AGD  = record['tip_sib16agd']
    a3.SIB16NEU  = record['tip_sib16neu']
    a3.SIB16PDX  = record['tip_sib16pdx']
    a3.SIB16MOE  = record['tip_sib16moe']
    a3.SIB16AGO  = record['tip_sib16ago']
    a3.SIB17MOB  = record['tip_sib17mob']
    a3.SIB17YOB  = record['tip_sib17yob']
    a3.SIB17AGD  = record['tip_sib17agd']
    a3.SIB17NEU  = record['tip_sib17neu']
    a3.SIB17PDX  = record['tip_sib17pdx']
    a3.SIB17MOE  = record['tip_sib17moe']
    a3.SIB17AGO  = record['tip_sib17ago']
    a3.SIB18MOB  = record['tip_sib18mob']
    a3.SIB18YOB  = record['tip_sib18yob']
    a3.SIB18AGD  = record['tip_sib18agd']
    a3.SIB18NEU  = record['tip_sib18neu']
    a3.SIB18PDX  = record['tip_sib18pdx']
    a3.SIB18MOE  = record['tip_sib18moe']
    a3.SIB18AGO  = record['tip_sib18ago']
    a3.SIB19MOB  = record['tip_sib19mob']
    a3.SIB19YOB  = record['tip_sib19yob']
    a3.SIB19AGD  = record['tip_sib19agd']
    a3.SIB19NEU  = record['tip_sib19neu']
    a3.SIB19PDX  = record['tip_sib19pdx']
    a3.SIB19MOE  = record['tip_sib19moe']
    a3.SIB19AGO  = record['tip_sib19ago']
    a3.SIB20MOB  = record['tip_sib20mob']
    a3.SIB20YOB  = record['tip_sib20yob']
    a3.SIB20AGD  = record['tip_sib20agd']
    a3.SIB20NEU  = record['tip_sib20neu']
    a3.SIB20PDX  = record['tip_sib20pdx']
    a3.SIB20MOE  = record['tip_sib20moe']
    a3.SIB20AGO  = record['tip_sib20ago']
    a3.KIDS      = record['tip_kids']
    a3.KID1MOB   = record['tip_kid1mob']
    a3.KID1YOB   = record['tip_kid1yob']
    a3.KID1AGD   = record['tip_kid1agd']
    a3.KID1NEU   = record['tip_kid1neu']
    a3.KID1PDX   = record['tip_kid1pdx']
    a3.KID1MOE   = record['tip_kid1moe']
    a3.KID1AGO   = record['tip_kid1ago']
    a3.KID2MOB   = record['tip_kid2mob']
    a3.KID2YOB   = record['tip_kid2yob']
    a3.KID2AGD   = record['tip_kid2agd']
    a3.KID2NEU   = record['tip_kid2neu']
    a3.KID2PDX   = record['tip_kid2pdx']
    a3.KID2MOE   = record['tip_kid2moe']
    a3.KID2AGO   = record['tip_kid2ago']
    a3.KID3MOB   = record['tip_kid3mob']
    a3.KID3YOB   = record['tip_kid3yob']
    a3.KID3AGD   = record['tip_kid3agd']
    a3.KID3NEU   = record['tip_kid3neu']
    a3.KID3PDX   = record['tip_kid3pdx']
    a3.KID3MOE   = record['tip_kid3moe']
    a3.KID3AGO   = record['tip_kid3ago']
    a3.KID4MOB   = record['tip_kid4mob']
    a3.KID4YOB   = record['tip_kid4yob']
    a3.KID4AGD   = record['tip_kid4agd']
    a3.KID4NEU   = record['tip_kid4neu']
    a3.KID4PDX   = record['tip_kid4pdx']
    a3.KID4MOE   = record['tip_kid4moe']
    a3.KID4AGO   = record['tip_kid4ago']
    a3.KID5MOB   = record['tip_kid5mob']
    a3.KID5YOB   = record['tip_kid5yob']
    a3.KID5AGD   = record['tip_kid5agd']
    a3.KID5NEU   = record['tip_kid5neu']
    a3.KID5PDX   = record['tip_kid5pdx']
    a3.KID5MOE   = record['tip_kid5moe']
    a3.KID5AGO   = record['tip_kid5ago']
    a3.KID6MOB   = record['tip_kid6mob']
    a3.KID6YOB   = record['tip_kid6yob']
    a3.KID6AGD   = record['tip_kid6agd']
    a3.KID6NEU   = record['tip_kid6neu']
    a3.KID6PDX   = record['tip_kid6pdx']
    a3.KID6MOE   = record['tip_kid6moe']
    a3.KID6AGO   = record['tip_kid6ago']
    a3.KID7MOB   = record['tip_kid7mob']
    a3.KID7YOB   = record['tip_kid7yob']
    a3.KID7AGD   = record['tip_kid7agd']
    a3.KID7NEU   = record['tip_kid7neu']
    a3.KID7PDX   = record['tip_kid7pdx']
    a3.KID7MOE   = record['tip_kid7moe']
    a3.KID7AGO   = record['tip_kid7ago']
    a3.KID8MOB   = record['tip_kid8mob']
    a3.KID8YOB   = record['tip_kid8yob']
    a3.KID8AGD   = record['tip_kid8agd']
    a3.KID8NEU   = record['tip_kid8neu']
    a3.KID8PDX   = record['tip_kid8pdx']
    a3.KID8MOE   = record['tip_kid8moe']
    a3.KID8AGO   = record['tip_kid8ago']
    a3.KID9MOB   = record['tip_kid9mob']
    a3.KID9YOB   = record['tip_kid9yob']
    a3.KID9AGD   = record['tip_kid9agd']
    a3.KID9NEU   = record['tip_kid9neu']
    a3.KID9PDX   = record['tip_kid9pdx']
    a3.KID9MOE   = record['tip_kid9moe']
    a3.KID9AGO   = record['tip_kid9ago']
    a3.KID10MOB  = record['tip_kid10mob']
    a3.KID10YOB  = record['tip_kid10yob']
    a3.KID10AGD  = record['tip_kid10agd']
    a3.KID10NEU  = record['tip_kid10neu']
    a3.KID10PDX  = record['tip_kid10pdx']
    a3.KID10MOE  = record['tip_kid10moe']
    a3.KID10AGO  = record['tip_kid10ago']
    a3.KID11MOB  = record['tip_kid11mob']
    a3.KID11YOB  = record['tip_kid11yob']
    a3.KID11AGD  = record['tip_kid11agd']
    a3.KID11NEU  = record['tip_kid11neu']
    a3.KID11PDX  = record['tip_kid11pdx']
    a3.KID11MOE  = record['tip_kid11moe']
    a3.KID11AGO  = record['tip_kid11ago']
    a3.KID12MOB  = record['tip_kid12mob']
    a3.KID12YOB  = record['tip_kid12yob']
    a3.KID12AGD  = record['tip_kid12agd']
    a3.KID12NEU  = record['tip_kid12neu']
    a3.KID12PDX  = record['tip_kid12pdx']
    a3.KID12MOE  = record['tip_kid12moe']
    a3.KID12AGO  = record['tip_kid12ago']
    a3.KID13MOB  = record['tip_kid13mob']
    a3.KID13YOB  = record['tip_kid13yob']
    a3.KID13AGD  = record['tip_kid13agd']
    a3.KID13NEU  = record['tip_kid13neu']
    a3.KID13PDX  = record['tip_kid13pdx']
    a3.KID13MOE  = record['tip_kid13moe']
    a3.KID13AGO  = record['tip_kid13ago']
    a3.KID14MOB  = record['tip_kid14mob']
    a3.KID14YOB  = record['tip_kid14yob']
    a3.KID14AGD  = record['tip_kid14agd']
    a3.KID14NEU  = record['tip_kid14neu']
    a3.KID14PDX  = record['tip_kid14pdx']
    a3.KID14MOE  = record['tip_kid14moe']
    a3.KID14AGO  = record['tip_kid14ago']
    a3.KID15MOB  = record['tip_kid15mob']
    a3.KID15YOB  = record['tip_kid15yob']
    a3.KID15AGD  = record['tip_kid15agd']
    a3.KID15NEU  = record['tip_kid15neu']
    a3.KID15PDX  = record['tip_kid15pdx']
    a3.KID15MOE  = record['tip_kid15moe']
    a3.KID15AGO  = record['tip_kid15ago']
    packet.append(a3)


def add_a4(record, packet):
    # Form A4D and A4G are special in that our REDCap implementation (FVP A4)
    # combines them by asking if the subject is taking any medications (which
    # corresponds to A4G.ANYMEDS), then has 50 fields to specify each
    # medication used, which we turn each one into a FormA4D object.
    a4g = tip_forms.FormA4G()
    a4g.ANYMEDS = record['tip_anymeds']
    packet.append(a4g)

    for i in range(1, 51):
            key = 'tip_drugid_' + str(i)
            if record[key]:
                a4d = tip_forms.FormA4D()
                a4d.DRUGID = record[key]
                packet.append(a4d)


def add_a5(record, packet):
    a5 = tip_forms.FormA5()
    a5.TOBAC30  = record['tip_TOBAC30'.lower()]
    a5.TOBAC100 = record['tip_TOBAC100'.lower()]
    a5.SMOKYRS  = record['tip_SMOKYRS'.lower()]
    a5.PACKSPER = record['tip_PACKSPER'.lower()]
    a5.QUITSMOK = record['tip_QUITSMOK'.lower()]
    a5.ALCOCCAS = record['tip_ALCOCCAS'.lower()]
    a5.ALCFREQ  = record['tip_ALCFREQ'.lower()]
    a5.CVHATT   = record['tip_CVHATT'.lower()]
    a5.HATTMULT = record['tip_HATTMULT'.lower()]
    a5.HATTYEAR = record['tip_HATTYEAR'.lower()]
    a5.CVAFIB   = record['tip_CVAFIB'.lower()]
    a5.CVANGIO  = record['tip_CVANGIO'.lower()]
    a5.CVBYPASS = record['tip_CVBYPASS'.lower()]
    a5.CVPACDEF = record['tip_CVPACDEF'.lower()]
    a5.CVCHF    = record['tip_CVCHF'.lower()]
    a5.CVANGINA = record['tip_CVANGINA'.lower()]
    a5.CVHVALVE = record['tip_CVHVALVE'.lower()]
    a5.CVOTHR   = record['tip_CVOTHR'.lower()]
    a5.CVOTHRX  = record['tip_CVOTHRX'.lower()]
    a5.CBSTROKE = record['tip_CBSTROKE'.lower()]
    a5.STROKMUL = record['tip_STROKMUL'.lower()]
    a5.STROKYR  = record['tip_STROKYR'.lower()]
    a5.CBTIA    = record['tip_CBTIA'.lower()]
    a5.TIAMULT  = record['tip_TIAMULT'.lower()]
    a5.TIAYEAR  = record['tip_TIAYEAR'.lower()]
    a5.PD       = record['tip_PD'.lower()]
    a5.PDYR     = record['tip_PDYR'.lower()]
    a5.PDOTHR   = record['tip_PDOTHR'.lower()]
    a5.PDOTHRYR = record['tip_PDOTHRYR'.lower()]
    a5.SEIZURES = record['tip_SEIZURES'.lower()]
    a5.TBI      = record['tip_TBI'.lower()]
    a5.TBIBRIEF = record['tip_TBIBRIEF'.lower()]
    a5.TBIEXTEN = record['tip_TBIEXTEN'.lower()]
    a5.TBIWOLOS = record['tip_TBIWOLOS'.lower()]
    a5.TBIYEAR  = record['tip_TBIYEAR'.lower()]
    a5.DIABETES = record['tip_DIABETES'.lower()]
    a5.DIABTYPE = record['tip_DIABTYPE'.lower()]
    a5.HYPERTEN = record['tip_HYPERTEN'.lower()]
    a5.HYPERCHO = record['tip_HYPERCHO'.lower()]
    a5.B12DEF   = record['tip_B12DEF'.lower()]
    a5.THYROID  = record['tip_THYROID'.lower()]
    a5.ARTHRIT  = record['tip_ARTHRIT'.lower()]
    a5.ARTHTYPE = record['tip_ARTHTYPE'.lower()]
    a5.ARTHTYPX = record['tip_ARTHTYPX'.lower()]
    a5.ARTHUPEX = record['tip_ARTHUPEX'.lower()]
    a5.ARTHLOEX = record['tip_ARTHLOEX'.lower()]
    a5.ARTHSPIN = record['tip_ARTHSPIN'.lower()]
    a5.ARTHUNK  = record['tip_ARTHUNK'.lower()]
    a5.INCONTU  = record['tip_INCONTU'.lower()]
    a5.INCONTF  = record['tip_INCONTF'.lower()]
    a5.APNEA    = record['tip_APNEA'.lower()]
    a5.RBD      = record['tip_RBD'.lower()]
    a5.INSOMN   = record['tip_INSOMN'.lower()]
    a5.OTHSLEEP = record['tip_OTHSLEEP'.lower()]
    a5.OTHSLEEX = record['tip_OTHSLEEX'.lower()]
    a5.ALCOHOL  = record['tip_ALCOHOL'.lower()]
    a5.ABUSOTHR = record['tip_ABUSOTHR'.lower()]
    a5.ABUSX    = record['tip_ABUSX'.lower()]
    a5.PTSD     = record['tip_PTSD'.lower()]
    a5.BIPOLAR  = record['tip_BIPOLAR'.lower()]
    a5.SCHIZ    = record['tip_SCHIZ'.lower()]
    a5.DEP2YRS  = record['tip_DEP2YRS'.lower()]
    a5.DEPOTHR  = record['tip_DEPOTHR'.lower()]
    a5.ANXIETY  = record['tip_ANXIETY'.lower()]
    a5.OCD      = record['tip_OCD'.lower()]
    a5.NPSYDEV  = record['tip_NPSYDEV'.lower()]
    a5.PSYCDIS  = record['tip_PSYCDIS'.lower()]
    a5.PSYCDISX = record['tip_PSYCDISX'.lower()]
    packet.append(a5)


def add_b1(record, packet):
    b1 = tip_forms.FormB1()
    b1.VISION   = record['tip_VISION'.lower()]
    b1.VISCORR  = record['tip_VISCORR'.lower()]
    b1.VISWCORR = record['tip_VISWCORR'.lower()]
    b1.HEARING  = record['tip_HEARING'.lower()]
    b1.HEARAID  = record['tip_HEARAID'.lower()]
    b1.HEARWAID = record['tip_HEARWAID'.lower()]
    packet.append(b1)


def add_b4(record, packet):
    b4 = tip_forms.FormB4()
    b4.MEMORY    = record['tip_memory']
    b4.ORIENT    = record['tip_orient']
    b4.JUDGMENT  = record['tip_judgment']
    b4.COMMUN    = record['tip_commun']
    b4.HOMEHOBB  = record['tip_homehobb']
    b4.PERSCARE  = record['tip_perscare']
    b4.CDRSUM    = record['tip_cdrsum']
    b4.CDRGLOB   = record['tip_cdrglob']
    b4.COMPORT   = record['tip_comport']
    b4.CDRLANG   = record['tip_cdrlang']
    packet.append(b4)


def add_b5(record, packet):
    b5 = tip_forms.FormB5()
    b5.NPIQINF   = record['tip_npiqinf']
    b5.NPIQINFX  = record['tip_npiqinfx']
    b5.DEL       = record['tip_del']
    b5.DELSEV    = record['tip_delsev']
    b5.HALL      = record['tip_hall']
    b5.HALLSEV   = record['tip_hallsev']
    b5.AGIT      = record['tip_agit']
    b5.AGITSEV   = record['tip_agitsev']
    b5.DEPD      = record['tip_depd']
    b5.DEPDSEV   = record['tip_depdsev']
    b5.ANX       = record['tip_anx']
    b5.ANXSEV    = record['tip_anxsev']
    b5.ELAT      = record['tip_elat']
    b5.ELATSEV   = record['tip_elatsev']
    b5.APA       = record['tip_apa']
    b5.APASEV    = record['tip_apasev']
    b5.DISN      = record['tip_disn']
    b5.DISNSEV   = record['tip_disnsev']
    b5.IRR       = record['tip_irr']
    b5.IRRSEV    = record['tip_irrsev']
    b5.MOT       = record['tip_mot']
    b5.MOTSEV    = record['tip_motsev']
    b5.NITE      = record['tip_nite']
    b5.NITESEV   = record['tip_nitesev']
    b5.APP       = record['tip_app']
    b5.APPSEV    = record['tip_appsev']
    packet.append(b5)


def add_b6(record, packet):
    b6 = tip_forms.FormB6()
    b6.NOGDS     = record['tip_nogds']
    b6.SATIS     = record['tip_satis']
    b6.DROPACT   = record['tip_dropact']
    b6.EMPTY     = record['tip_empty']
    b6.BORED     = record['tip_bored']
    b6.SPIRITS   = record['tip_spirits']
    b6.AFRAID    = record['tip_afraid']
    b6.HAPPY     = record['tip_happy']
    b6.HELPLESS  = record['tip_helpless']
    b6.STAYHOME  = record['tip_stayhome']
    b6.MEMPROB   = record['tip_memprob']
    b6.WONDRFUL  = record['tip_wondrful']
    b6.WRTHLESS  = record['tip_wrthless']
    b6.ENERGY    = record['tip_energy']
    b6.HOPELESS  = record['tip_hopeless']
    b6.BETTER    = record['tip_better']
    b6.GDS       = record['tip_gds']
    packet.append(b6)


def add_b7(record, packet):
    b7 = tip_forms.FormB7()
    b7.BILLS     = record['tip_bills']
    b7.TAXES     = record['tip_taxes']
    b7.SHOPPING  = record['tip_shopping']
    b7.GAMES     = record['tip_games']
    b7.STOVE     = record['tip_stove']
    b7.MEALPREP  = record['tip_mealprep']
    b7.EVENTS    = record['tip_events']
    b7.PAYATTN   = record['tip_payattn']
    b7.REMDATES  = record['tip_remdates']
    b7.TRAVEL    = record['tip_travel']
    packet.append(b7)


def add_b8(record, packet):
    b8 = tip_forms.FormB8()
    b8.NORMEXAM = record['tip_NORMEXAM'.lower()]
    b8.PARKSIGN = record['tip_PARKSIGN'.lower()]
    b8.RESTTRL  = record['tip_RESTTRL'.lower()]
    b8.RESTTRR  = record['tip_RESTTRR'.lower()]
    b8.SLOWINGL = record['tip_SLOWINGL'.lower()]
    b8.SLOWINGR = record['tip_SLOWINGR'.lower()]
    b8.RIGIDL   = record['tip_RIGIDL'.lower()]
    b8.RIGIDR   = record['tip_RIGIDR'.lower()]
    b8.BRADY    = record['tip_BRADY'.lower()]
    b8.PARKGAIT = record['tip_PARKGAIT'.lower()]
    b8.POSTINST = record['tip_POSTINST'.lower()]
    b8.CVDSIGNS = record['tip_CVDSIGNS'.lower()]
    b8.CORTDEF  = record['tip_CORTDEF'.lower()]
    b8.SIVDFIND = record['tip_SIVDFIND'.lower()]
    b8.CVDMOTL  = record['tip_CVDMOTL'.lower()]
    b8.CVDMOTR  = record['tip_CVDMOTR'.lower()]
    b8.CORTVISL = record['tip_CORTVISL'.lower()]
    b8.CORTVISR = record['tip_CORTVISR'.lower()]
    b8.SOMATL   = record['tip_SOMATL'.lower()]
    b8.SOMATR   = record['tip_SOMATR'.lower()]
    b8.POSTCORT = record['tip_POSTCORT'.lower()]
    b8.PSPCBS   = record['tip_PSPCBS'.lower()]
    b8.EYEPSP   = record['tip_EYEPSP'.lower()]
    b8.DYSPSP   = record['tip_DYSPSP'.lower()]
    b8.AXIALPSP = record['tip_AXIALPSP'.lower()]
    b8.GAITPSP  = record['tip_GAITPSP'.lower()]
    b8.APRAXSP  = record['tip_APRAXSP'.lower()]
    b8.APRAXL   = record['tip_APRAXL'.lower()]
    b8.APRAXR   = record['tip_APRAXR'.lower()]
    b8.CORTSENL = record['tip_CORTSENL'.lower()]
    b8.CORTSENR = record['tip_CORTSENR'.lower()]
    b8.ATAXL    = record['tip_ATAXL'.lower()]
    b8.ATAXR    = record['tip_ATAXR'.lower()]
    b8.ALIENLML = record['tip_ALIENLML'.lower()]
    b8.ALIENLMR = record['tip_ALIENLMR'.lower()]
    b8.DYSTONL  = record['tip_DYSTONL'.lower()]
    b8.DYSTONR  = record['tip_DYSTONR'.lower()]
    b8.MYOCLLT  = record['tip_MYOCLLT'.lower()]
    b8.MYOCLRT  = record['tip_MYOCLRT'.lower()]
    b8.ALSFIND  = record['tip_ALSFIND'.lower()]
    b8.GAITNPH  = record['tip_GAITNPH'.lower()]
    b8.OTHNEUR  = record['tip_OTHNEUR'.lower()]
    b8.OTHNEURX = record['tip_OTHNEURX'.lower()]
    packet.append(b8)


def add_b9(record, packet):
    b9 = tip_forms.FormB9()
    b9.DECSUB    = record['tip_decsub']
    b9.DECIN     = record['tip_decin']
    b9.DECCLCOG  = record['tip_decclcog']
    b9.COGMEM    = record['tip_cogmem']
    b9.COGORI    = record['tip_cogori']
    b9.COGJUDG   = record['tip_cogjudg']
    b9.COGLANG   = record['tip_coglang']
    b9.COGVIS    = record['tip_cogvis']
    b9.COGATTN   = record['tip_cogattn']
    b9.COGFLUC   = record['tip_cogfluc']
    b9.COGFLAGO  = record['tip_cogflago']
    b9.COGOTHR   = record['tip_cogothr']
    b9.COGOTHRX  = record['tip_cogothrx']
    b9.COGFPRED  = record['tip_cogfpred']
    b9.COGFPREX  = record['tip_cogfprex']
    b9.COGMODE   = record['tip_cogmode']
    b9.COGMODEX  = record['tip_cogmodex']
    b9.DECAGE    = record['tip_decage']
    b9.DECCLBE   = record['tip_decclbe']
    b9.BEAPATHY  = record['tip_beapathy']
    b9.BEDEP     = record['tip_bedep']
    b9.BEVHALL   = record['tip_bevhall']
    b9.BEVWELL   = record['tip_bevwell']
    b9.BEVHAGO   = record['tip_bevhago']
    b9.BEAHALL   = record['tip_beahall']
    b9.BEDEL     = record['tip_bedel']
    b9.BEDISIN   = record['tip_bedisin']
    b9.BEIRRIT   = record['tip_beirrit']
    b9.BEAGIT    = record['tip_beagit']
    b9.BEPERCH   = record['tip_beperch']
    b9.BEREM     = record['tip_berem']
    b9.BEREMAGO  = record['tip_beremago']
    b9.BEANX     = record['tip_beanx']
    b9.BEOTHR    = record['tip_beothr']
    b9.BEOTHRX   = record['tip_beothrx']
    b9.BEFPRED   = record['tip_befpred']
    b9.BEFPREDX  = record['tip_befpredx']
    b9.BEMODE    = record['tip_bemode']
    b9.BEMODEX   = record['tip_bemodex']
    b9.BEAGE     = record['tip_beage']
    b9.DECCLMOT  = record['tip_decclmot']
    b9.MOGAIT    = record['tip_mogait']
    b9.MOFALLS   = record['tip_mofalls']
    b9.MOTREM    = record['tip_motrem']
    b9.MOSLOW    = record['tip_moslow']
    b9.MOFRST    = record['tip_mofrst']
    b9.MOMODE    = record['tip_momode']
    b9.MOMODEX   = record['tip_momodex']
    b9.MOMOPARK  = record['tip_momopark']
    b9.PARKAGE   = record['tip_parkage']
    b9.MOMOALS   = record['tip_momoals']
    b9.ALSAGE    = record['tip_alsage']
    b9.MOAGE     = record['tip_moage']
    b9.COURSE    = record['tip_course']
    b9.FRSTCHG   = record['tip_frstchg']
    b9.LBDEVAL   = record['tip_lbdeval']
    b9.FTLDEVAL  = record['tip_ftldeval']
    packet.append(b9)


def add_c2t(record, packet):
    c2 = tip_forms.FormC2()
    c2.MODCOMM  = record['tip_modcomm']
    c2.MOCACOMP = record['tip_mocacomp']
    c2.MOCAREAS = record['tip_mocareas']
    c2.MOCALAN  = record['tip_mocalan']
    c2.MOCALANX = record['tip_mocalanx']
    c2.MOCAHEAR = record['tip_mocahear']
    c2.MOCBTOTS = record['tip_mocbtots']
    c2.MOCADIGI = record['tip_mocadigi']
    c2.MOCALETT = record['tip_mocalett']
    c2.MOCASER7 = record['tip_mocaser7']
    c2.MOCAREPE = record['tip_mocarepe']
    c2.MOCAFLUE = record['tip_mocaflue']
    c2.MOCAABST = record['tip_mocaabst']
    c2.MOCARECN = record['tip_mocarecn']
    c2.MOCARECC = record['tip_mocarecc']
    c2.MOCARECR = record['tip_mocarecr']
    c2.MOCAORDT = record['tip_mocaordt']
    c2.MOCAORMO = record['tip_mocaormo']
    c2.MOCAORYR = record['tip_mocaoryr']
    c2.MOCAORDY = record['tip_mocaordy']
    c2.MOCAORPL = record['tip_mocaorpl']
    c2.MOCAORCT = record['tip_mocaorct']
    c2.NPSYLAN  = record['tip_npsylan']
    c2.NPSYLANX = record['tip_npsylanx']
    c2.CRAFTVRS = record['tip_craftvrs']
    c2.CRAFTURS = record['tip_crafturs']
    c2.REY1REC  = record['tip_rey1rec']
    c2.REY1INT  = record['tip_rey1int']
    c2.REY2REC  = record['tip_rey2rec']
    c2.REY2INT  = record['tip_rey2int']
    c2.REY3REC  = record['tip_rey3rec']
    c2.REY3INT  = record['tip_rey3int']
    c2.REY4REC  = record['tip_rey4rec']
    c2.REY4INT  = record['tip_rey4int']
    c2.REY5REC  = record['tip_rey5rec']
    c2.REY5INT  = record['tip_rey5int']
    c2.REY6REC  = record['tip_rey6rec']
    c2.REY6INT  = record['tip_rey6int']
    c2.DIGFORCT = record['tip_digforct']
    c2.DIGFORSL = record['tip_digforsl']
    c2.DIGBACCT = record['tip_digbacct']
    c2.DIGBACLS = record['tip_digbacls']
    c2.OTRAILA  = record['tip_otraila']
    c2.OTRLARR  = record['tip_otrlarr']
    c2.OTRLALI  = record['tip_otrlali']
    c2.OTRAILB  = record['tip_otrailb']
    c2.OTRLBRR  = record['tip_otrlbrr']
    c2.OTRLBLI  = record['tip_otrlbli']
    c2.CRAFTDVR = record['tip_craftdvr']
    c2.CRAFTDRE = record['tip_craftdre']
    c2.CRAFTDTI = record['tip_craftdti']
    c2.CRAFTCUE = record['tip_craftcue']
    c2.ANIMALS  = record['tip_animals']
    c2.VEG      = record['tip_veg']
    c2.UDSVERFC = record['tip_udsverfc']
    c2.UDSVERFN = record['tip_udsverfn']
    c2.UDSVERNF = record['tip_udsvernf']
    c2.UDSVERLC = record['tip_udsverlc']
    c2.UDSVERLR = record['tip_udsverlr']
    c2.UDSVERLN = record['tip_udsverln']
    c2.UDSVERTN = record['tip_udsvertn']
    c2.UDSVERTE = record['tip_udsverte']
    c2.UDSVERTI = record['tip_udsverti']
    c2.REYDREC  = record['tip_reydrec']
    c2.REYDINT  = record['tip_reydint']
    c2.REYTCOR  = record['tip_reytcor']
    c2.REYFPOS  = record['tip_reyfpos']
    c2.VNTTOTW  = record['tip_vnttotw']
    c2.VNTPCNC  = record['tip_vntpcnc']
    c2.COGSTAT  = record['tip_cogstat']
    c2.RESPVAL  = record['tip_respval']
    c2.RESPHEAR = record['tip_resphear']
    c2.RESPDIST = record['tip_respdist']
    c2.RESPINTR = record['tip_respintr']
    c2.RESPDISN = record['tip_respdisn']
    c2.RESPFATG = record['tip_respfatg']
    c2.RESPEMOT = record['tip_respemot']
    c2.RESPASST = record['tip_respasst']
    c2.RESPOTH  = record['tip_respoth']
    c2.RESPOTHX = record['tip_respothx']
    packet.append(c2)


def add_d1(record, packet):
    d1 = tip_forms.FormD1()
    d1.DXMETHOD  = record['tip_dxmethod']
    d1.NORMCOG   = record['tip_normcog']
    d1.DEMENTED  = record['tip_demented']
    d1.AMNDEM    = record['tip_amndem']
    d1.PCA       = record['tip_pca']
    d1.PPASYN    = record['tip_ppasyn']
    d1.PPASYNT   = record['tip_ppasynt']
    d1.FTDSYN    = record['tip_ftdsyn']
    d1.LBDSYN    = record['tip_lbdsyn']
    d1.NAMNDEM   = record['tip_namndem']
    d1.MCIAMEM   = record['tip_mciamem']
    d1.MCIAPLUS  = record['tip_mciaplus']
    d1.MCIAPLAN  = record['tip_mciaplan']
    d1.MCIAPATT  = record['tip_mciapatt']
    d1.MCIAPEX   = record['tip_mciapex']
    d1.MCIAPVIS  = record['tip_mciapvis']
    d1.MCINON1   = record['tip_mcinon1']
    d1.MCIN1LAN  = record['tip_mcin1lan']
    d1.MCIN1ATT  = record['tip_mcin1att']
    d1.MCIN1EX   = record['tip_mcin1ex']
    d1.MCIN1VIS  = record['tip_mcin1vis']
    d1.MCINON2   = record['tip_mcinon2']
    d1.MCIN2LAN  = record['tip_mcin2lan']
    d1.MCIN2ATT  = record['tip_mcin2att']
    d1.MCIN2EX   = record['tip_mcin2ex']
    d1.MCIN2VIS  = record['tip_mcin2vis']
    d1.IMPNOMCI  = record['tip_impnomci']
    d1.AMYLPET   = record['tip_amylpet']
    d1.AMYLCSF   = record['tip_amylcsf']
    d1.FDGAD     = record['tip_fdgad']
    d1.HIPPATR   = record['tip_hippatr']
    d1.TAUPETAD  = record['tip_taupetad']
    d1.CSFTAU    = record['tip_csftau']
    d1.FDGFTLD   = record['tip_fdgftld']
    d1.TPETFTLD  = record['tip_tpetftld']
    d1.MRFTLD    = record['tip_mrftld']
    d1.DATSCAN   = record['tip_datscan']
    d1.OTHBIOM   = record['tip_othbiom']
    d1.OTHBIOMX  = record['tip_othbiomx']
    d1.IMAGLINF  = record['tip_imaglinf']
    d1.IMAGLAC   = record['tip_imaglac']
    d1.IMAGMACH  = record['tip_imagmach']
    d1.IMAGMICH  = record['tip_imagmich']
    d1.IMAGMWMH  = record['tip_imagmwmh']
    d1.IMAGEWMH  = record['tip_imagewmh']
    d1.ADMUT     = record['tip_admut']
    d1.FTLDMUT   = record['tip_ftldmut']
    d1.OTHMUT    = record['tip_othmut']
    d1.OTHMUTX   = record['tip_othmutx']
    d1.ALZDIS    = record['tip_alzdis']
    d1.ALZDISIF  = record['tip_alzdisif']
    d1.LBDIS     = record['tip_lbdis']
    d1.LBDIF     = record['tip_lbdif']
    d1.PARK      = record['tip_park']
    d1.MSA       = record['tip_msa']
    d1.MSAIF     = record['tip_msaif']
    d1.PSP       = record['tip_psp']
    d1.PSPIF     = record['tip_pspif']
    d1.CORT      = record['tip_cort']
    d1.CORTIF    = record['tip_cortif']
    d1.FTLDMO    = record['tip_ftldmo']
    d1.FTLDMOIF  = record['tip_ftldmoif']
    d1.FTLDNOS   = record['tip_ftldnos']
    d1.FTLDNOIF  = record['tip_ftldnoif']
    d1.FTLDSUBT  = record['tip_ftldsubt']
    d1.FTLDSUBX  = record['tip_ftldsubx']
    d1.CVD       = record['tip_cvd']
    d1.CVDIF     = record['tip_cvdif']
    d1.PREVSTK   = record['tip_prevstk']
    d1.STROKDEC  = record['tip_strokdec']
    d1.STKIMAG   = record['tip_stkimag']
    d1.INFNETW   = record['tip_infnetw']
    d1.INFWMH    = record['tip_infwmh']
    d1.ESSTREM   = record['tip_esstrem']
    d1.ESSTREIF  = record['tip_esstreif']
    d1.DOWNS     = record['tip_downs']
    d1.DOWNSIF   = record['tip_downsif']
    d1.HUNT      = record['tip_hunt']
    d1.HUNTIF    = record['tip_huntif']
    d1.PRION     = record['tip_prion']
    d1.PRIONIF   = record['tip_prionif']
    d1.BRNINJ    = record['tip_brninj']
    d1.BRNINJIF  = record['tip_brninjif']
    d1.BRNINCTE  = record['tip_brnincte']
    d1.HYCEPH    = record['tip_hyceph']
    d1.HYCEPHIF  = record['tip_hycephif']
    d1.EPILEP    = record['tip_epilep']
    d1.EPILEPIF  = record['tip_epilepif']
    d1.NEOP      = record['tip_neop']
    d1.NEOPIF    = record['tip_neopif']
    d1.NEOPSTAT  = record['tip_neopstat']
    d1.HIV       = record['tip_hiv']
    d1.HIVIF     = record['tip_hivif']
    d1.OTHCOG    = record['tip_othcog']
    d1.OTHCOGIF  = record['tip_othcogif']
    d1.OTHCOGX   = record['tip_othcogx']
    d1.DEP       = record['tip_dep']
    d1.DEPIF     = record['tip_depif']
    d1.DEPTREAT  = record['tip_deptreat']
    d1.BIPOLDX   = record['tip_bipoldx']
    d1.BIPOLDIF  = record['tip_bipoldif']
    d1.SCHIZOP   = record['tip_schizop']
    d1.SCHIZOIF  = record['tip_schizoif']
    d1.ANXIET    = record['tip_anxiet']
    d1.ANXIETIF  = record['tip_anxietif']
    d1.DELIR     = record['tip_delir']
    d1.DELIRIF   = record['tip_delirif']
    d1.PTSDDX    = record['tip_ptsddx']
    d1.PTSDDXIF  = record['tip_ptsddxif']
    d1.OTHPSY    = record['tip_othpsy']
    d1.OTHPSYIF  = record['tip_othpsyif']
    d1.OTHPSYX   = record['tip_othpsyx']
    d1.ALCDEM    = record['tip_alcdem']
    d1.ALCDEMIF  = record['tip_alcdemif']
    d1.ALCABUSE  = record['tip_alcabuse']
    d1.IMPSUB    = record['tip_impsub']
    d1.IMPSUBIF  = record['tip_impsubif']
    d1.DYSILL    = record['tip_dysill']
    d1.DYSILLIF  = record['tip_dysillif']
    d1.MEDS      = record['tip_meds']
    d1.MEDSIF    = record['tip_medsif']
    d1.COGOTH    = record['tip_cogoth']
    d1.COGOTHIF  = record['tip_cogothif']
    d1.COGOTHX   = record['tip_cogothx']
    d1.COGOTH2   = record['tip_cogoth2']
    d1.COGOTH2F  = record['tip_cogoth2f']
    d1.COGOTH2X  = record['tip_cogoth2x']
    d1.COGOTH3   = record['tip_cogoth3']
    d1.COGOTH3F  = record['tip_cogoth3f']
    d1.COGOTH3X  = record['tip_cogoth3x']
    packet.append(d1)


def add_d2(record, packet):
    d2 = tip_forms.FormD2()
    d2.CANCER    = record['tip_cancer']
    d2.CANCSITE  = record['tip_cancsite']
    d2.DIABET    = record['tip_diabet']
    d2.MYOINF    = record['tip_myoinf']
    d2.CONGHRT   = record['tip_conghrt']
    d2.AFIBRILL  = record['tip_afibrill']
    d2.HYPERT    = record['tip_hypert']
    d2.ANGINA    = record['tip_angina']
    d2.HYPCHOL   = record['tip_hypchol']
    d2.VB12DEF   = record['tip_vb12def']
    d2.THYDIS    = record['tip_thydis']
    d2.ARTH      = record['tip_arth']
    d2.ARTYPE    = record['tip_artype']
    d2.ARTYPEX   = record['tip_artypex']
    d2.ARTUPEX   = record['tip_artupex']
    d2.ARTLOEX   = record['tip_artloex']
    d2.ARTSPIN   = record['tip_artspin']
    d2.ARTUNKN   = record['tip_artunkn']
    d2.URINEINC  = record['tip_urineinc']
    d2.BOWLINC   = record['tip_bowlinc']
    d2.SLEEPAP   = record['tip_sleepap']
    d2.REMDIS    = record['tip_remdis']
    d2.HYPOSOM   = record['tip_hyposom']
    d2.SLEEPOTH  = record['tip_sleepoth']
    d2.SLEEPOTX  = record['tip_sleepotx']
    d2.ANGIOCP   = record['tip_angiocp']
    d2.ANGIOPCI  = record['tip_angiopci']
    d2.PACEMAKE  = record['tip_pacemake']
    d2.HVALVE    = record['tip_hvalve']
    d2.ANTIENC   = record['tip_antienc']
    d2.ANTIENCX  = record['tip_antiencx']
    d2.OTHCOND   = record['tip_othcond']
    d2.OTHCONDX  = record['tip_othcondx']
    packet.append(d2)


def update_header(record, packet):
    for header in packet:
        header.PACKET = "IT"
        header.FORMID = header.form_name
        header.FORMVER = "3"
        header.ADCID = record['adcid']
        header.PTID = record['ptid']
        header.VISITMO = record['visitmo']
        header.VISITDAY = record['visitday']
        header.VISITYR = record['visityr']
        header.VISITNUM = record['visitnum']
        header.INITIALS = record['initials']
