# script to compute national/sub-national nitrogen response curves
# from the GGCMI Phase 2 emulators
# (Franke et al. 2020, doi: 10.5194/gmd-13-3995-2020)
# and to compute optimized N distribution patterns.
#
# written by Christoph Müller, PIK
# Licensed under GNU Affero General Public License (AGPLv3)
# see https://www.gnu.org/licenses/agpl-3.0.html

# clean start
rm(list = ls(all = TRUE))

require(ncdf4)
require(fields)
require(reshape)
require(nloptr)
require(stringi)

local <- F
if (local) {
  do.ggcm <- 7
  do.crop <- 1
  do.obj <- 5
} else {
  shellarg <- commandArgs(TRUE)
  do.ggcm <- as.numeric(strsplit(shellarg, "_")[[1]][1])
  do.crop <- as.numeric(strsplit(shellarg, "_")[[1]][2])
  do.obj <- as.numeric(strsplit(shellarg, "_")[[1]][3])
}

path.rclim <- "/p/projects/macmit/users/cmueller/GGCMI/phase2/CMIP/CMIP5/"
path.gs <- "/p/projects/macmit/data/GGCMI/AgMIP.input/other.inputs/phase2.masks/"
path.lu <- "/p/projects/macmit/data/GGCMI/AgMIP.output/processed/masks/"
path.fert <- "/p/projects/lpjml/input/Fertilizer/histsoc/"
path.coeff <- "/p/projects/macmit/data/GGCMI/AgMIP.output/Jim_Emulator/EMULATOR_PARAMS/"
path.figs <- "/p/projects/macmit/users/cmueller/GGCMI/phase2/planetary_n/figures/"

# exclude soybean, which is an N-fixer
crops <- c("mai", "ric", "soy", "wwh", "swh")[-3]
crops.gs <- c("Maize", "Rice", "Soybeans", "wwh", "swh")[-3]
crops.nice <- c("Maize", "Rice", "Soybeans", "Winter Wheat", "Spring Wheat", "all combined")[-3]
crops.param <- c("maize", "rice", "soy", "winter_wheat", "spring_wheat")[-3]
crops.param2 <- c("maize", "rice", "soy", "winter_wheat", "spring_wheat", "all")[-3]
crops.fert <- c("mai", "ric", "soy", "whe", "whe")[-3]
bnf <- c(0, 33, 0, 0, 0)[-3]
allcropfertilizer <- 142454050385 # taken from PB_downscaling_cropwise 05022021.xlsx compiled by Helena from total N fertilizer and crop BNF

ndaymonth <- c(31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31)
ggcms <- c("CARAIB", "EPIC-TAMU", "GEPIC", "JULES", "LPJ-GUESS", "LPJmL", "PEPIC", "PROMET", "pDSSAT")[-c(1, 4, 8)] # pDSSAT produces segmentation fault in optimization, no idea why
ggcms <- c(ggcms, "ensemble.mean")

ENERG <- c(3560, 2800, 3350, 3340, 3340) # kcal/kg fresh matter
ENERG <- ENERG * 1e3 # kcal/tFM
FRESHMATTER <- 100 / c(88, 87, 91, 88, 88)
ENERG_DM <- ENERG * FRESHMATTER * 1e-6 # Gcal/tDM

# planetary boundary in kg from Tg to kg multiply with 1e9
PBN <- c(62, 82) * 1e9 # https://www.stockholmresilience.org/research/planetary-boundaries/planetary-boundaries/about-the-research/quantitative-evolution-of-boundaries.html
# plot?
pl <- F

readmap.nc <- function(filename, var = "", lo = "lon", la = "lat", starttime = 1) {
  nc <- nc_open(filename)
  if (var == "") var <- names(nc$var)[1]
  lon <- ncvar_get(nc, lo)
  if (min(lon) >= 0) {
    cat("WARNING! Longitude does not contain negative values, shifting >180 by 360\n")
    lon[lon > 180] <- lon[lon > 180] - 360
  }
  lat <- ncvar_get(nc, la)
  if (lat[1] > lat[length(lat)]) {
    cat("WARNING, inverting latitudes\n")
  }
  if (starttime == 1) {
    buf <- ncvar_get(nc, var)
  } else {
    buf <- ncvar_get(nc, var, start = c(1, 1, starttime))
  }
  nc_close(nc)
  if (length(dim(buf)) == 2) {
    buf <- buf[order(lon), order(lat)]
  } else if (length(dim(buf)) == 3) {
    buf <- buf[order(lon), order(lat), ]
  } else if (length(dim(buf)) > 3) {
    cat("WARNING, cannot adjust lon/lat setting for 4-dim array\n")
  }
  buf
}

# emulator

emulator.national <- function(cf, C, T, W, N, irrig = F, country = "", warn = F) {
  if (all(cf == 0)) {
    return(0)
  }
  if (length(cf) == 34 | (irrig & length(cf) == 19)) {
    # cat("34 coefficients found\n")
    if (irrig) {
      Y <- (cf[1] + cf[2] * C + cf[3] * T + cf[4] * N + cf[5] * C^2 + cf[6] * C * T + cf[7] * C * N + cf[8] * T^2 +
        cf[9] * T * N + cf[10] * N^2 + cf[11] * C^3 + cf[12] * C^2 * T + cf[13] * C^2 * N + cf[14] * C * T^2 +
        cf[15] * C * T * N + cf[16] * C * N^2 + cf[17] * T^3 + cf[18] * T^2 * N + cf[19] * T * N^2)
    } else {
      Y <- (cf[1] + cf[2] * C + cf[3] * T + cf[4] * W + cf[5] * N + cf[6] * C^2 + cf[7] * C * T + cf[8] * C * W +
        cf[9] * C * N + cf[10] * T^2 + cf[11] * T * W + cf[12] * T * N + cf[13] * W^2 + cf[14] * W * N + cf[15] * N^2 +
        cf[16] * C^3 + cf[17] * C^2 * T + cf[18] * C^2 * W + cf[19] * C^2 * N + cf[20] * C * T^2 + cf[21] * C * T * W +
        cf[22] * C * T * N + cf[23] * C * W^2 + cf[24] * C * W * N + cf[25] * C * N^2 + cf[26] * T^3 + cf[27] * T^2 * W +
        cf[28] * T^2 * N + cf[29] * T * W^2 + cf[30] * T * W * N + cf[31] * T * N^2 + cf[32] * W^3 +
        cf[33] * W^2 * N + cf[34] * W * N^2)
    }
  } else if (length(cf) == 20 | (irrig & length(cf) == 10)) {
    # cat("20 coefficients found\n")
    if (irrig) {
      Y <- (cf[1] + cf[2] * C + cf[3] * T + cf[4] * C^2 + cf[5] * C * T + cf[6] * T^2 + cf[7] * C^3 +
        cf[8] * C^2 * T + cf[9] * C * T^2 + cf[10] * T^3)
    } else {
      Y <- (cf[1] + cf[2] * C + cf[3] * T + cf[4] * W + cf[5] * C^2 + cf[6] * C * T + cf[7] * C * W + cf[8] * T^2 +
        cf[9] * T * W + cf[10] * W^2 + cf[11] * C^3 + cf[12] * C^2 * T + cf[13] * C^2 * W + cf[14] * C * T^2 +
        cf[15] * C * T * W + cf[16] * C * W^2 + cf[17] * T^3 + cf[18] * T^2 * W + cf[19] * T * W^2 + cf[20] * W^3)
    }
  }
  if (all(!is.finite(Y))) {
    Y[] <- 0
    # if(warn) cat("WARNING! no valid values for",country,irrig,"\n")
  }
  Y
}

gcm <- 3
ggcm <- 7
cr <- 1
nlevels <- seq(10, 200, by = 10)
nlevels2 <- c(10:200)
upbound <- 200
objectives <- c("yieldmax", "Nminimization", "yieldmaxPB62", "yieldmaxPB82", "scale62", "scale82")
firstwheat <- which(crops == "wwh")

# load country metadata
cntry <- read.csv(paste0(path.lu, "aggr/gadm0.meta.csv"), header = F)
cntry1 <- read.csv(paste0(path.lu, "aggr/gadm1.meta.csv"), header = F)

# countries to include
c2i <- c(1:dim(cntry)[1])

# loop through states for Australia, Brazil, Canada, China, India, Russia, USA
subnational <- which(cntry[, 1] %in% c(14, 32, 41, 48, 105, 186, 240))

europe <- c(
  "ALB", "AUT", "BLR", "BEL", "BIH", "BGR", "HVR", "CZE", "DNK", "EST",
  "FIN", "FRA", "GEO", "DEU", "GRC", "HUN",
  "IRL", "ITA", "KO-", "LVA", "LTU", "LUX", "MLT", "MDA", "MNE", "MSR",
  "NLD", "NOR", "POL", "PRT", "SRB", "SVK", "SVN", "ESP",
  "SWE", "CHE", "UKR", "GBR"
)

# for testing
# c2i <- which(cntry[,3] %in% c(europe[-c(5,7,19,23:26)]))
if (local) c2i <- which(cntry[, 3] %in% c(europe[-c(1, 2, 3, 5, 7, 9:11, 13, 19, 23:26, 31)]))

# load country pixel map
gadm0 <- readmap.nc(paste0(path.lu, "aggr/gadm0.mask.nc4"))
gadm1 <- readmap.nc(paste0(path.lu, "aggr/gadm.mask.01.nc4"), var = "gadm1")

for (ggcm in do.ggcm) {
  sets <- list()
  ii <- 0
  nsets <- frac_totalN <- frac_totalN_norice <- numeric(length(crops))
  for (cr in 1:length(crops)) {
    cat("processing", ggcms[ggcm], "for", crops.nice[cr], "\n")
    # load crop area
    if (cr < firstwheat) {
      fn <- paste0(path.lu, "weight/", crops.param[cr], ".nc4")
      mrf <- readmap.nc(fn, "rainfed")
      mir <- readmap.nc(fn, "irrigated")
    } else {
      fn <- "/p/projects/macmit/data/GGCMI/AgMIP.input/phase3/landuse/winter_spring_wheat_separation/winter_and_spring_wheat_areas_phase3.nc4"
      mrf <- readmap.nc(fn, if (cr == firstwheat) "wwh_area_rf" else if (cr == firstwheat + 1) "swh_area_rf" else "rainfed")
      mir <- readmap.nc(fn, if (cr == firstwheat) "wwh_area_ir" else if (cr == firstwheat + 1) "swh_area_ir" else "irrigated")
    }
    mrf[!is.finite(mrf)] <- 0
    mir[!is.finite(mir)] <- 0
    mt <- mrf + mir
    # using 3-year average for fertilizers from histsoc data for 2013-2015
    nfert <- apply(readmap.nc(paste0(path.fert, "fertilizer_application_histsoc_", crops.fert[cr], "_1850-2015.nc"), "fertrate", lo = "longitude", la = "latitude")[, , 164:166], c(1, 2), mean, na.rm = T)
    nfert[!is.finite(nfert)] <- NA
    assign(paste0("nfert_", crops[cr]), nfert)
    assign(paste0("mt_", crops[cr]), mt)
    assign(paste0("mrf_", crops[cr]), mrf)
    assign(paste0("mir_", crops[cr]), mir)

    go <- F
    if (ggcm == length(ggcms)) { # ensemble mean
      gcounter <- 0
      for (ggcm2 in 1:(length(ggcms) - 1)) {
        fn <- paste0(path.coeff, ggcms[ggcm2], "_",
          crops.param[cr], "_ggcmi_phase2_emulator_A0.nc4")
        if (file.exists(fn)) {
          go <- T
          # load emulator coefficients
          if (gcounter == 0) {
            coeff_rf <- readmap.nc(fn, "K_rf")
            coeff_ir <- readmap.nc(fn, "K_ir")
          } else {
            coeff_rf <- coeff_rf + readmap.nc(fn, "K_rf")
            coeff_ir <- coeff_ir + readmap.nc(fn, "K_ir")
          }
          gcounter <- gcounter + 1
        }
      }
      coeff_rf <- coeff_rf / gcounter
      coeff_ir <- coeff_ir / gcounter
    } else {
      fn <- paste0(path.coeff, ggcms[ggcm], "_",
        crops.param[cr], "_ggcmi_phase2_emulator_A0.nc4")
      if (file.exists(fn)) {
        go <- T
        # load emulator coefficients
        coeff_rf <- readmap.nc(fn, "K_rf")
        coeff_ir <- readmap.nc(fn, "K_ir")
      }
    }

    lcounter <- 0
    ra <- NULL # range(f_n1(nlevels),f_n2(nlevels))
    for (cc in c2i) {
      if (cc %in% subnational) {
        states <- which(cntry1[, 1] == cntry[cc, 1])
        gadmx <- gadm1
        cntrx <- cntry1
      } else {
        states <- cc
        gadmx <- gadm0
        cntrx <- cntry
      }
      for (st in states) {
        # checking state and country, in countries without subnational resolution, this is the same
        if (cc %in% subnational) {
          l <- which(gadm0 == cntry[cc, 1] & gadmx == paste0(cntry[cc, 1],
            stri_pad_left(cntrx[st, 2], 2, 0)) & mt > 0, arr.ind = T)
        } else {
          l <- which(gadm0 == cntry[cc, 1] & mt > 0, arr.ind = T)
        }
        cat("processing", as.character(cntry[cc, 2]),
          as.character(cntrx[st, dim(cntrx)[2]]), "\n")
        if (length(l) > 0) {
          total_rf <- total_ir <- 0
          wcoeff_rf <- vector("numeric", dim(coeff_rf)[3])
          wcoeff_ir <- vector("numeric", if (dim(coeff_ir)[3] == 34) 19 else 10)
          for (i in 1:dim(l)[1]) {
            wght_rf <- mrf[l[i, 1], l[i, 2]]
            wght_ir <- mir[l[i, 1], l[i, 2]]
            if (all(is.finite(coeff_rf[l[i, 1], l[i, 2], ]))) {
              wcoeff_rf <- wcoeff_rf + coeff_rf[l[i, 1], l[i, 2], ] * wght_rf
              total_rf <- total_rf + wght_rf
            }
            if (all(is.finite(coeff_ir[l[i, 1], l[i, 2],
              c(1:if (dim(coeff_ir)[3] == 34) 19 else 10)]))) {
              wcoeff_ir <- wcoeff_ir + coeff_ir[l[i, 1], l[i, 2],
                c(1:if (dim(coeff_ir)[3] == 34) 19 else 10)] * wght_ir
              total_ir <- total_ir + wght_ir
            }
          }
          # at least 1000ha for that crop in this country
          if ((total_rf + total_ir) > 1000) {
            if (total_rf > 0) {
              wcoeff_rf <- wcoeff_rf / total_rf 
            } else {
              wcoeff_rf[] <- 0
            }
            if (total_ir > 0) {
              wcoeff_ir <- wcoeff_ir / total_ir 
            } else {
              wcoeff_ir[] <- 0
            }
            lcounter <- lcounter + 1
            eval(parse(text = paste("f", lcounter + ii,
              "<- function(nl,warn=FALSE) { return((emulator.national(sets[[",
              lcounter + ii,
              "]]$wcoeff_rf, 400, 0, 1, nl,irrig = F, as.character(cntry[sets[[",
              lcounter + ii,
              "]]$cid, 2]),warn = warn) * sets[[",
              lcounter + ii,
              "]]$total_rf + emulator.national(sets[[", lcounter + ii,
              "]]$wcoeff_ir, 400, 0, 1, nl, irrig = T,as.character(cntry[sets[[",
              lcounter + ii,
              "]]$cid, 2]), warn = warn) * sets[[", lcounter + ii,
              "]]$total_ir)/sets[[", lcounter + ii, "]]$total)}",
              sep = ""
            )))
            f_nresponse <- get(paste0("f", lcounter + ii))
            sets[[lcounter + ii]] <- list(
              total_rf = total_rf, total_ir = total_ir,
              total = total_rf + total_ir,
              wcoeff_rf = wcoeff_rf, wcoeff_ir = wcoeff_ir,
              l = l, cid = cntry[cc, 1], sid =
                if (cc %in% subnational) cntrx[st, 2] else NA
            )
            if (max(f_nresponse(nlevels2)) == 0) {
              cat(
                "no N sensitivity in", as.character(cntry[cc, 2]), if (cc %in% subnational) as.character(cntry1[st, 4]),
                "for", crops[cr], " removing again\n"
              )
              sets[[lcounter + ii]] <- NULL
              lcounter <- lcounter - 1
            } else {
              ra <- range(ra, f_nresponse(nlevels, warn = T))
              cat(
                "adding", as.character(cntry[cc, 2]),
                if (cc %in% subnational) as.character(cntry1[st, 4]),
                "with cropland areas of ", total_rf + total_ir, total_rf,
                total_ir, "for", crops[cr], "\n"
              )
            }
          } else {
            cat(
              "skipping", as.character(cntry[cc, 2]),
              if (cc %in% subnational) as.character(cntry1[st, 4]),
              "for too little cropland", total_rf + total_ir, total_rf,
              total_ir, "for", crops[cr], "\n"
            )
          }
        }
      }
    }
    # assign(paste0("sets",crops[cr]),sets)
    ii <- ii + lcounter
    nsets[cr] <- lcounter
  } # cr
  for (objective in objectives[do.obj]) {
    cat("processing data for", objective, "\n")
    if (pl) {
      png(paste0(
        path.figs, ggcms[ggcm], "_",
        crops.param[cr], "_yield_curves_ub", upbound,
        "_", objective, "_", length(sets), "_countries.png"
      ),
      width = 8 * 300, height = 8 * 300, res = 300, pointsize = 12
      )
      plot(nlevels, nlevels,
        ylim = ra, xlim = c(0, 250), xlab = "N levels (kg/ha)",
        ylab = "national yield response (tDM/ha)",
        type = "n", main = paste(ggcms[ggcm], crops.nice[cr])
      )
    }
    total_prod <- total_fert <- total_area <- total_area_ir <- total_bnf <- 0
    min_total_prod <- 0
    x0 <- xorig <- NULL
    fert <- prods <- yields <- list()
    cnts <- NULL
    if (objective != objectives[2]) {
      ub <- NULL
      lb <- NULL
      mains <- "total_prod"
      hs <- ""
    } else {
      ub <- NULL
      lb <- NULL
      # hs needs to be >=0, so here it's all coutries production at new N input minus the original production
      hs <- ""
      # main minimization function
      mains <- "total_fert"
    }
    ii <- 0
    for (cr in 1:length(crops)) {
      total_prod_cr <- total_area_cr <- total_fert_cr <-
        total_area_ir_cr <- total_bnf_cr <- 0
      nfert <- get(paste0("nfert_", crops[cr]))
      mt <- get(paste0("mt_", crops[cr]))
      mrf <- get(paste0("mrf_", crops[cr]))
      mir <- get(paste0("mir_", crops[cr]))
      for (i in 1:nsets[cr]) {
        f_nresponse <- get(paste0("f", i + ii))
        if (pl) lines(nlevels, f_nresponse(nlevels), col = 1 + i)
        if (pl) {
          text(202, f_nresponse(200),
            as.character(cntry[which(cntry[, 1] == sets[[i + ii]]$cid), 2]),
            adj = c(0, 0.5), cex = 0.5, col = 1 + i
          )
        }
        if (sets[[i + ii]]$cid %in% cntry[subnational, 1]) {
          fer <- sum(nfert[gadm1 == paste0(sets[[i + ii]]$cid,
            stri_pad_left(sets[[i + ii]]$sid, 2, 0))] *
            mt[gadm1 == paste0(sets[[i + ii]]$cid,
            stri_pad_left(sets[[i + ii]]$sid, 2, 0))], na.rm = T)
        } else {
          fer <- sum(nfert[gadm0 == cntry[which(cntry[, 1] ==
            sets[[i + ii]]$cid), 1]] *
            mt[gadm0 == cntry[which(cntry[, 1] ==
            sets[[i + ii]]$cid), 1]], na.rm = T)
        }
        fert[[i + ii]] <- fer
        total_fert_cr <- total_fert_cr + fer
        total_bnf_cr <- total_bnf_cr + bnf[cr] * sets[[i + ii]]$total
        # add BNF Nr to xx
        xx <- fer / sets[[i + ii]]$total + bnf[cr]
        xorig <- c(xorig, xx) # keep original N input for scaling activity
        if (pl) points(xx, f_nresponse(xx), col = i + 1)
        # don't force lower input level to 10, as that can artificially increase the N requirement
        # if(xx<10){
        #   #cat("correcting N inputs to at least 10kg/ha",as.character(cntry[sets[[i+ii]]$cid,2]),"\n")
        #   xx <- 10
        # }
        # correcting high input levels is necessary though, otherwise yields can become negative as fertilizer inputs and harvested areas are not necessarily consistent.
        if (xx > 200) {
          # cat("correcting N inputs to at most 200kg/ha",as.character(cntry[sets[[i+ii]]$cid,2]),"\n")
          xx <- 200
        }
        x0 <- c(x0, xx)
        yield <- f_nresponse(xx)
        pro <- yield * sets[[i + ii]]$total * ENERG_DM[cr]
        prods[[i + ii]] <- pro
        yields[[i + ii]] <- yield
        cat(
          as.character(cntry[which(cntry[, 1] == sets[[i + ii]]$cid), 2]),
          crops.nice[cr], xorig[i + ii], xx, pro, yield, total_prod_cr, "\n"
        )
        total_prod_cr <- total_prod_cr + pro
        total_area_cr <- total_area_cr + sets[[i + ii]]$total
        total_area_ir_cr <- total_area_ir_cr + sets[[i + ii]]$total_ir
        buf <- as.character(cntry[which(cntry[, 1] == sets[[i + ii]]$cid), 2])
        if (sets[[i + ii]]$cid %in% cntry[subnational, 1]) {
          buf <- paste(buf, as.character(cntry1[which(cntry1[, 1] ==
          sets[[i + ii]]$cid & cntry1[, 2] == sets[[i + ii]]$sid), 4]), sep = "_")
        }
        cnts <- c(cnts, buf)
      }
      # compute fraction of crop fertilizer use from total
      assign(paste0("total_prod_", crops[cr]), total_prod_cr)
      assign(paste0("total_fert_", crops[cr]), total_fert_cr)
      assign(paste0("total_area_", crops[cr]), total_area_cr)
      assign(paste0("total_area_ir_", crops[cr]), total_area_ir_cr)
      assign(paste0("total_bnf_", crops[cr]), total_bnf_cr)
      assign(paste0("uniform_fert_", crops[cr]), total_fert_cr / total_area_cr)
      total_prod <- total_prod + total_prod_cr
      total_area <- total_area + total_area_cr
      total_area_ir <- total_area + total_area_cr
      total_fert <- total_fert + total_fert_cr
      total_bnf <- total_bnf + total_bnf_cr
      if (pl & (length(cnts) < 10)) legend("topleft", bty = "n",
      legend = cnts, col = c(1:length(cnts)) + 1, lty = 1)

      if (objective %in% objectives[c(1, 3, 4)]) {
        for (i in 1:nsets[cr]) {
          # skip countries with no N response
          f_nresponse <- get(paste0("f", i + ii))
          if (max(f_nresponse(nlevels2)) == 0) {
            lb <- c(lb, 0) # allow zero input if yield is zero anyway
            ub <- c(ub, max(lb[length(lb)] + 0.1, x0[i + ii]))
          } else {
            # lower bound cannot be less than bnf, or 10 unless the baseline is less than 10
            lb <- c(lb, min(x0[i + ii], max(bnf[cr], 10)))
            # upper bound cannot exceed fertilizer rate that results in highest yields
            ub <- c(ub, max(x0[i + ii], nlevels2[which.max(f_nresponse(nlevels2))[1]]))
          }
          # maximize production, by minimizing negative production
          mains <- paste0(mains, "-f", i + ii, "(x[", i + ii,
            "])*sets[[", i + ii, "]]$total*ENERG_DM[", cr, "]")
          # total fertilizer + BNF must not be exceeded: current total Nr - optimized total Nr must not be negative
          hs <- paste0(hs, "-x[", i + ii, "]*sets[[", i + ii, "]]$total")
        }
      } else if (objective == objectives[2]) {
        for (i in 1:nsets[cr]) {
          # skip countries with no N response
          f_nresponse <- get(paste0("f", i + ii))
          if (max(f_nresponse(nlevels2)) > 0) {
            # upper bound is current rate or N input rate that results in highest yields
            ub <- c(ub, max(x0[i + ii],
              nlevels2[which.max(f_nresponse(nlevels2))[1]]))
            # lower bound cannot be less than bnf of 10 unless the baseline is less than 10
            lb <- c(lb, min(x0[i + ii], max(bnf[cr], 10)))
          } else {
            # no responsiveness to N input
            # lower bound is bnf (rice) or 10 (otherwise) or smaller if original rate is lower than that
            lb <- c(lb, min(x0[i + ii], max(bnf[cr], 10)))
            # upper bound at current levels or just a little over the lower bound
            ub <- c(ub, max(x0[i + ii], lb[length(lb)] + 0.1)) # apparently needs to be higher than lower bound
            cat("skipping", as.character(cntry[which(cntry[, 1] ==
              sets[[i + ii]]$cid), 2]), "\n")
          }
          hs <- paste0(hs, "+f", i + ii, "(x[", i + ii,
            "])*sets[[", i + ii, "]]$total*ENERG_DM[", cr, "]")
          mains <- paste0(mains, "+x[", i + ii, "]*sets[[", i + ii, "]]$total")
        }
      } # else do nothing (scaling cases)
      ii <- ii + nsets[cr]
    } # cr
    for (cr in 1:length(crops)) {
      frac_totalN[cr] <- (get(paste0("total_fert_", crops[cr])) +
        get(paste0("total_bnf_", crops[cr]))) / (allcropfertilizer + total_bnf)
      if (cr != which(crops == "ric")) frac_totalN_norice[cr] <-
        get(paste0("total_fert_", crops[cr]))
    }
    frac_totalN_norice <- frac_totalN_norice / sum(frac_totalN_norice)
    cereal_frac <- sum(frac_totalN)
    cat("\n\ncereal frac is", cereal_frac, "\n\n", frac_totalN, "\n")
    if (objective == objectives[1]) {
      max_total_n <- (total_fert + total_bnf)
    } else if (objective == objectives[3]) {
      max_total_n <- PBN[1] * cereal_frac
    } #+total_bnf
    else if (objective == objectives[4]) {
      max_total_n <- PBN[2] * cereal_frac
    } #+total_bnf
    else if (objective == objectives[5]) {
      max_total_n <- PBN[1] * cereal_frac
    } #+total_bnf
    else if (objective == objectives[6]) {
      max_total_n <- PBN[2] * cereal_frac
    } #+total_bnf
    else {
      max_total_n <- Inf
    }
    if (objective %in% objectives[c(1, 3, 4)]) {
      # generate fnc function from string 'main'
      eval(parse(text = paste0("fnc <- function(x) { return(", mains, ")}")))
      # generate hin function from string 'hs'
      # hin needs to be >=0
      eval(parse(text = paste0("hin <- function(x){ h <- numeric(1); h[1] <- max_total_n",
        hs, "; return(h)}")))
      #x0 <- (lb+x0)/2
      # change start value to lb if optimization doesn't find solution if starting with x0
      #x0 <- lb
      x0 <- x0
      #x0 <- (2 * lb + x0) / 3
      cat("x0", x0, "\n")
      cat("lb", lb, "\n")
      x0[!is.finite(x0)] <- lb[!is.finite(x0)]
      x0[x0 < lb] <- lb[x0 < lb]
      cat("x0corr", x0, "\n")
      cat("lbcorr", lb, "\n")
    } else if (objective == objectives[2]) {
      # generate fnc function from string 'main'
      eval(parse(text = paste0("fnc <- function(x) { return(", mains, ")}")))
      # generate hin function from string 'hs'
      # hin needs to be >=0
      eval(parse(text = paste0("hin <- function(x){ h <- numeric(1); h[1] <- ",
        hs, "-total_prod; return(h)}")))
      min_total_prod <- total_prod
      x0 <- lb
    } else { # scaling objectives that don't need optimization
      scale_n <- max_total_n / (total_fert + total_bnf)
      cat("scaling by", scale_n, "allowing for", max_total_n,
        "while applying", total_fert, "and", total_bnf, "\n")
      coln <- c(
        "crop", "orig. N-fert rate (kgN/ha)", "BNF rate (kgN/ha)",
        "orig. BNF total (kgN)",
        "orig. N total (kgN) incl. BNF",
        "orig. production (kcal)", "orig. yield (tDM/ha)",
        "total cropland (ha)", "frac. irrig (-)",
        "mod. N-fert rate (kgN/ha)", "mod. N total(kgN) incl. BNF",
        "mod. production (kcal)", "mod. yield (tDM/ha)",
        "uniform N-fert rate (kgN/ha)", "uniform N yield (tDM/ha)",
        "uniform N production (kcal)"
      )
      tab <- array(NA, dim = c((length(sets) + length(crops)), length(coln)))
      rownames(tab) <- c(
        "global", cnts[1:nsets[1]], "global", cnts[c(1:nsets[2]) + nsets[1]],
        "global", cnts[c(1:nsets[3]) + sum(nsets[1:2])],
        "global", cnts[c(1:nsets[4]) + sum(nsets[1:3])]
      )
      colnames(tab) <- coln
      ii <- 0
      total_prod_scale <- total_n_scale <- total_fert_scale <-
        total_bnf_scale <- 0
      for (cr in 1:length(crops)) {
        total_prod_scale_cr <- 0
        total_n_scale_cr <- 0
        total_fert_scale_cr <- 0
        total_bnf_scale_cr <- 0
        mod_bnf <- 0
        total_fert_cr <- get(paste0("total_fert_", crops[cr]))
        total_prod_cr <- get(paste0("total_prod_", crops[cr]))
        total_area_cr <- get(paste0("total_area_", crops[cr]))
        total_area_ir_cr <- get(paste0("total_area_ir_", crops[cr]))
        total_bnf_cr <- get(paste0("total_bnf_", crops[cr]))
        nuniform <- ((total_fert_cr + total_bnf_cr) / total_area_cr)
        uniformprod <- uniformfert <- 0
        for (i in 1:nsets[cr]) {
          f_nresponse <- get(paste0("f", i + ii))
          mod_nfert <- xorig[i + ii] * scale_n
          if (bnf[cr] > 0) { # reduce mineral fertilizer more to keep BNF constant
            # mod_nfert <- bnf[cr] * scale_n + mod_nfert - bnf[cr]
            mod_bnf <- bnf[cr] * scale_n
            mod_nfert <- (xorig[i + ii] - bnf[cr]) * scale_n
          }
          if ((mod_nfert + mod_bnf) > 200) {
            cat(
              "too high nfert in", as.character(cntry[which(cntry[, 1] ==
                sets[[i + ii]]$cid), 2]),
              "for", crops.nice[cr], "setting to 200 starting at",
              xorig[i + ii],
              "scaling to", mod_nfert,
              "with bnf of", bnf[cr], "\n"
            )
            mod_nfert <- 200 - mod_bnf
          }
          yield <- f_nresponse(mod_nfert + mod_bnf)
          prod_scale <- yield * sets[[i + ii]]$total * ENERG_DM[cr]
          total_prod_scale_cr <- total_prod_scale_cr + prod_scale
          total_n_scale_cr <- total_n_scale_cr + (mod_nfert + mod_bnf) *
            sets[[i + ii]]$total
          total_fert_scale_cr <- total_fert_scale_cr + mod_nfert *
            sets[[i + ii]]$total
          total_bnf_scale_cr <- total_bnf_scale_cr + mod_bnf *
            sets[[i + ii]]$total
          yielduniform <- f_nresponse(nuniform)
          uniformprod <- uniformprod + yielduniform *
            sets[[i + ii]]$total * ENERG_DM[cr]
          uniformfert <- uniformfert + nuniform * sets[[i + ii]]$total
          tab[i + ii + cr, ] <- c(
            cr, fert[[i + ii]] / sets[[i + ii]]$total, bnf[cr], bnf[cr] * sets[[i + ii]]$total,
            fert[[i + ii]] + bnf[cr] * sets[[i + ii]]$total,
            prods[[i + ii]], yields[[i + ii]], sets[[i + ii]]$total,
            sets[[i + ii]]$total_ir / sets[[i + ii]]$total,
            mod_nfert, (mod_nfert + mod_bnf) * sets[[i + ii]]$total,
            prod_scale, yield,
            nuniform, yielduniform, yielduniform * sets[[i + ii]]$total * ENERG_DM[cr]
          )
        }
        tab[cr + ii, ] <- c(
          cr, total_fert_cr / total_area_cr, bnf[cr], total_bnf_cr,
          total_fert_cr + total_bnf_cr,
          total_prod_cr, total_prod_cr / total_area_cr / ENERG_DM[cr],
          total_area_cr, total_area_ir_cr / total_area_cr,
          total_n_scale_cr / total_area_cr, total_n_scale_cr, total_prod_scale_cr,
          total_prod_scale_cr / total_area_cr / ENERG_DM[cr],
          nuniform, uniformprod / total_area_cr / ENERG_DM[cr], uniformprod
        )
        dp <- total_prod_scale_cr - total_prod_cr
        df <- total_fert_scale_cr - total_fert_cr
        dn <- total_n_scale_cr - (total_fert_cr + total_bnf_cr)
        cat(
          ggcms[ggcm], crops.nice[cr], "total production ",
            if (dp > 0) "increased" else "decreased", "by", dp, "from",
          total_prod_cr, "to", total_prod_scale_cr,
          "(", (total_prod_scale_cr / total_prod_cr - 1) * 100, "%) for",
          objective, "\n"
        )
        cat(
          ggcms[ggcm], crops.nice[cr], "total fertilizer use ",
          if (df > 0) "increased" else "decreased", "by", df, "from",
          total_fert_cr, "to", total_fert_scale_cr,
          "(", (total_fert_scale_cr / total_fert_cr - 1) * 100, "%) for",
          objective, "\n"
        )
        cat(
          ggcms[ggcm], crops.nice[cr], "total N use ",
          if (dn > 0) "increased" else "decreased", "by", dn, "from",
          total_fert_cr + total_bnf_cr, "to", total_n_scale_cr,
          "(", (total_n_scale_cr / total_fert_cr - 1) * 100, "%) split in fert of",
          total_fert_scale_cr, "and BNF of",
          total_bnf_scale_cr, "for", objective, "\n"
        )
        total_prod_scale <- total_prod_scale + total_prod_scale_cr
        total_n_scale <- total_n_scale + total_n_scale_cr
        total_fert_scale <- total_fert_scale + total_fert_scale_cr
        ii <- ii + nsets[cr]
      }
      dp <- total_prod_scale - total_prod
      df <- total_fert_scale - total_fert
      dn <- total_n_scale - (total_fert + total_bnf)
      cat(
        ggcms[ggcm], "all crops total production ",
        if (dp > 0) "increased" else "decreased", "by", dp, "from",
        total_prod, "to", total_prod_scale,
        "(", (total_prod_scale / total_prod - 1) * 100, "%)",
        "with upper bound", upbound, "for",
        objective, "requesting at least ", min_total_prod,
        " kcal production\n"
      )
      cat(
        ggcms[ggcm], "all crops total fertilizer use ",
        if (df > 0) "increased" else "decreased", "by", df, "from",
        total_fert, "to", total_fert_scale,
        "(", (total_fert_scale / total_fert - 1) * 100, "%) with upper bound",
        upbound, "for",
        objective, " allowing at most ", max_total_n, " kgN\n"
      )
      cat(
        ggcms[ggcm], "all crops total N use ",
        if (dn > 0) "increased" else "decreased", "by", dn, "from",
        total_fert + total_bnf, "(with BNF of ", total_bnf, ")to",
        total_n_scale,
        "(", (total_n_scale / (total_fert + total_bnf) - 1) * 100, "%), of which",
        total_bnf_scale,
        "is BNF; with upper bound", upbound, "for", objective,
        " allowing at most ", max_total_n, " kgN\n"
      )
      fn <- paste0(
        path.figs, ggcms[ggcm],
        "_all_crops_combined_C400_T0_W0_Nact_A0_Noptimization_BNF_",
        objective, "_", length(sets),
        "subnational_", length(subnational), "_countries.csv"
      )
      write.csv(tab, file = fn)
      cat("\nwriting data to", fn, "\n\n")
      fn <- paste0(
        path.figs, ggcms[ggcm],
        "_all_crops_combined_C400_T0_W0_Nact_A0_Noptimization_BNF_",
        objective, "_", length(sets), "subnational_",
        length(subnational), "_countries.Rdata"
      )
      save.image(file = fn)
      next
    }
    # lb <- rep(10,length(ub))
    maxe <- maxr <- 600000 # 1e6 #2500000
    cat("start optimization with", maxe, "iterations\n")
    cat("hin:\n")
    str(hin)
    cat("fnc:\n")
    str(fnc)
    cat(ub - lb, "\n")
    cat(
      "starting optimization for", ggcms[ggcm],
      "all crops combined with at most ", maxe,
      "iterations, requesting at most ", max_total_n,
      " kgN and at least ", min_total_prod, " kcal production\n"
    )
    runtime <- system.time({
      results <- cobyla(x0, fnc, lower = lb, upper = ub, hin = hin,
      nl.info = T, control = list(xtol_rel = 1e-05, maxeval = maxe))
    })
    cat("runtime:\n", runtime)
    cat("\n\n")
    maxr <- results$iter
    # if done 1e6 iterations, stop, throw warning
    if (maxr > maxe) {
      maxr <- 0
      cat("WARNING! stopped at ", maxr, " iteration for",
      ggcms[ggcm], "all crop combined\n")
    } else {
      cat("optimization finished at", results$iter, "iterations\n")
    }
    # }

    # save intermediate results and everything
    fn <- paste0(
      path.figs, ggcms[ggcm],
      "_all_crops_combined_C400_T0_W0_Nact_A0_Noptimization_BNF_ub",
      upbound, "_", objective, "_", length(sets), "_maxe_", maxe, "_it_", maxr,
      "subnational_", length(subnational),
      #"_countries_x0ismeanlblbx0_fixPB_intermediate.Rdata"
      "_countries_x0isx0_fixPB_intermediate.Rdata"
    )
    save.image(file = fn)
    coln <- c(
      "crop", "orig. N-fert rate (kgN/ha)", "BNF rate (kgN/ha)",
      "orig. BNF total (kgN)", "orig. N total (kgN) incl. BNF",
      "orig. production (kcal)", "orig. yield (tDM/ha)",
      "total cropland (ha)", "frac. irrig (-)",
      "mod. N-fert rate (kgN/ha)", "mod. N total(kgN) incl. BNF",
      "mod. production (kcal)", "mod. yield (tDM/ha)",
      "max yield (tDM/ha)", "max yield N rate", "max yield N total (kgN)",
      "uniform currentN production (kcal)", "uniform currentN yield (tDM/ha)",
      "uniform currentN rate (kgN/ha)", "uniform currentN N total (kgN)",
      "uniform Nboundary62 yield (tDM/ha)", "uniform N62 rate (kgN/ha)",
      "uniform Nboundary62 N total (kgN)",
      "uniform Nboundary82 yield (tDM/ha)", "uniform N82 rate (kgN/ha)",
      "uniform Nboundary82 N total (kgN)"
    )
    tab <- array(NA, dim = c((length(sets) + length(crops)), length(coln)))
    rownames(tab) <- c(
      "global", cnts[1:nsets[1]], "global", cnts[c(1:nsets[2]) + nsets[1]],
      "global", cnts[c(1:nsets[3]) + sum(nsets[1:2])], "global", cnts[c(1:nsets[4]) + sum(nsets[1:3])]
    ) # rep(c("global",cnts),length(crops))
    colnames(tab) <- coln
    total_prod_opt <- 0
    total_n_opt <- 0
    ii <- 0
    for (cr in 1:length(crops)) {
      total_prod_opt_cr <- 0
      total_n_opt_cr <- 0
      total_fert_cr <- get(paste0("total_fert_", crops[cr]))
      total_prod_cr <- get(paste0("total_prod_", crops[cr]))
      total_area_cr <- get(paste0("total_area_", crops[cr]))
      total_area_ir_cr <- get(paste0("total_area_ir_", crops[cr]))
      total_bnf_cr <- get(paste0("total_bnf_", crops[cr]))
      nboundary <- if (total_bnf_cr > 0) rep(bnf[cr], 
        length(PBN)) else (PBN * cereal_frac - total_bnf_ric) *
        frac_totalN_norice[cr] / total_area_cr
      cat(crops[cr], nboundary, "\n")
      nuniform <- ((total_fert_cr + total_bnf_cr) / total_area_cr)
      maxprod <- uboundaryprod <- uboundaryprod2 <- maxfert <-
        uboundaryfert <- uboundaryfert2 <- uniformprod <- uniformfert <- 0
      for (i in 1:nsets[cr]) {
        f_nresponse <- get(paste0("f", i + ii))
        total_prod_opt_cr <- total_prod_opt_cr +
          f_nresponse(results$par[i + ii]) * sets[[i + ii]]$total * ENERG_DM[cr]
        total_n_opt_cr <- total_n_opt_cr + results$par[i + ii] *
          sets[[i + ii]]$total
        if (pl) points(results$par[i + ii],
          f_nresponse(results$par[i + ii]), col = i + 1, pch = 5)
        yield <- f_nresponse(results$par[i + ii])
        nmax <- nlevels2[which.max(f_nresponse(nlevels2))[1]]
        yieldmax <- f_nresponse(nmax)
        yielduniform <- f_nresponse(nuniform)
        yieldboundary <- f_nresponse(nboundary[1])
        yieldboundary2 <- f_nresponse(nboundary[2])
        maxprod <- maxprod + yieldmax * sets[[i + ii]]$total * ENERG_DM[cr]
        maxfert <- maxfert + nmax * sets[[i + ii]]$total
        uniformprod <- uniformprod + yielduniform *
          sets[[i + ii]]$total * ENERG_DM[cr]
        uniformfert <- uniformfert + nuniform *
          sets[[i + ii]]$total
        uboundaryprod <- uboundaryprod + yieldboundary *
          sets[[i + ii]]$total * ENERG_DM[cr]
        uboundaryfert <- uboundaryfert + nboundary[1] *
          sets[[i + ii]]$total
        uboundaryprod2 <- uboundaryprod2 + yieldboundary2 *
          sets[[i + ii]]$total * ENERG_DM[cr]
        uboundaryfert2 <- uboundaryfert2 + nboundary[2] *
          sets[[i + ii]]$total
        tab[i + ii + cr, ] <- c(
          cr, fert[[i + ii]] / sets[[i + ii]]$total, bnf[cr],
          bnf[cr] * sets[[i + ii]]$total, fert[[i + ii]] +
            bnf[cr] * sets[[i + ii]]$total,
          prods[[i + ii]], yields[[i + ii]], sets[[i + ii]]$total,
          sets[[i + ii]]$total_ir / sets[[i + ii]]$total,
          (results$par[i + ii] - bnf[cr]), (results$par[i + ii]) *
            sets[[i + ii]]$total,
          yield * sets[[i + ii]]$total * ENERG_DM[cr], yield,
          yieldmax, nmax, nmax * sets[[i + ii]]$total,
          yielduniform * sets[[i + ii]]$total * ENERG_DM[cr],
          yielduniform, nuniform,
          nuniform * sets[[i + ii]]$total,
          yieldboundary, nboundary[1], nboundary[1] * sets[[i + ii]]$total,
          yieldboundary2, nboundary[2], nboundary[2] * sets[[i + ii]]$total
        )
      }
      if (pl) legend("bottomright", pch = c(1, 5), col = 1,
        legend = c("original", "optimized"), bty = "n", ncol = 2)
      if (pl) dev.off()
      cat(
        "total N change in ", as.character(cntry[which(cntry[, 1] ==
        sets[[1 + ii]]$cid), 2]),
        results$par[1] * sets[[1 + ii]]$total - fert[[1 + ii]],
        "kgN, changing from", fert[[1 + ii]] / sets[[1 + ii]]$total,
        "kgN/ha to", results$par[1 + ii], "kgN/ha\n"
      )
      cat(
        "total N change in ", as.character(cntry[which(cntry[, 1] ==
        sets[[2 + ii]]$cid), 2]),
        results$par[2] * sets[[2 + ii]]$total - fert[[2 + ii]],
        "kgN, changing from", fert[[2 + ii]] / sets[[2 + ii]]$total,
        "kgN/ha to", results$par[2 + ii], "kgN/ha\n"
      )
      tab[cr + ii, ] <- c(
        cr, total_fert_cr / total_area_cr, bnf[cr], total_bnf_cr,
        total_fert_cr + total_bnf_cr,
        total_prod_cr, total_prod_cr / total_area_cr / ENERG_DM[cr],
        total_area_cr, total_area_ir_cr / total_area_cr,
        (total_n_opt_cr - total_bnf_cr) / total_area_cr, total_n_opt_cr,
        total_prod_opt_cr, total_prod_opt_cr / total_area_cr / ENERG_DM[cr],
        maxprod / total_area_cr / ENERG_DM[cr], maxfert / total_area_cr, maxfert,
        uniformprod, uniformprod / total_area_cr / ENERG_DM[cr],
        nuniform, nuniform * total_area_cr,
        uboundaryprod / total_area_cr / ENERG_DM[cr], nboundary[1],
        nboundary[1] * total_area_cr,
        uboundaryprod2 / total_area_cr / ENERG_DM[cr], nboundary[2],
        nboundary[2] * total_area_cr
      )
      dp <- total_prod_opt_cr - total_prod_cr
      df <- total_n_opt_cr - total_fert_cr
      cat(
        ggcms[ggcm], crops.nice[cr], "total production ",
        if (dp > 0) "increased" else "decreased", "by", dp, "from",
        total_prod_cr, "to", total_prod_opt_cr, "(",
        (total_prod_opt_cr / total_prod_cr - 1) * 100, "%)",
        "with upper bound", upbound, "for", objective, "\n"
      )
      cat(
        ggcms[ggcm], crops.nice[cr], "total fertilizer use ",
        if (df > 0) "increased" else "decreased", "by", df, "from",
        total_fert_cr, "to", total_n_opt_cr, "(",
        (total_n_opt_cr / total_fert_cr - 1) * 100,
        "%) with upper bound", upbound, "for", objective, "\n"
      )
      total_n_opt <- total_n_opt + total_n_opt_cr
      total_prod_opt <- total_prod_opt + total_prod_opt_cr
      ii <- ii + nsets[cr]
    } # cr
    dp <- total_prod_opt - total_prod
    df <- total_n_opt - total_fert
    cat(
      ggcms[ggcm], "all crops total production ",
      if (dp > 0) "increased" else "decreased", "by", dp,
      "from", total_prod, "to", total_prod_opt,
      "(", (total_prod_opt / total_prod - 1) * 100, "%)",
      "with upper bound", upbound, "for", objective, "requesting at least ",
      min_total_prod, " kcal production\n"
    )
    cat(
      ggcms[ggcm], "all crops total fertilizer use ",
      if (df > 0) "increased" else "decreased", "by", df,
      "from", total_fert, "to", total_n_opt, "(",
      (total_n_opt / total_fert - 1) * 100,
      "%) with upper bound", upbound, "for", objective,
      " allowing at most ", max_total_n, " kgN\n"
    )
    fn <- paste0(
      path.figs, ggcms[ggcm],
      "_all_crops_combined_C400_T0_W0_Nact_A0_Noptimization_BNF_ub",
      upbound, "_", objective, "_", length(sets), "_maxe_", maxe,
      "_it_", maxr, "subnational_",
      length(subnational), #"_countries_x0ismeanlblbx0_fixPB.csv"
      "_countries_x0isx0_fixPB.csv"
    )
    write.csv(tab, file = fn)
    cat("\nwriting data to", fn, "\n\n")
    fn <- paste0(
      path.figs, ggcms[ggcm],
      "_all_crops_combined_C400_T0_W0_Nact_A0_Noptimization_BNF_ub",
      upbound, "_", objective, "_", length(sets), "_maxe_", maxe, "_it_", maxr,
      "subnational_", length(subnational),
      #"_countries_x0ismeanlblbx0_fixPB.Rdata"
      "_countries_x0isx0_fixPB.Rdata"
    )
    save.image(file = fn)
  } # objectives
} # ggcm
