function bibStruct = readBibTeX(bibFile)
% Import BibTeX file as MATLAB structure.
%     A = readBibTeX(FILE) reads content from the
%     BibTeX file provided by string FILE and results
%     in a structure A with the labels at the first
%     levels and the found BibTeX keys at the second
%     level.
%
%     Example:
%        bibdata = readBibTeX('mybibs.bib')

% (c) Norbert Marwan, 2024-02-14

% open BibTeX file
fid = fopen(bibFile, 'r');
if fid == -1
    error('Cannot open BibTeX file.');
end

% initialise structure
bibStruct = struct();

% read line by line
tline = fgetl(fid);
while ischar(tline)

    % check if new paper starts
    if startsWith(tline, '@')
    
        % extract type of paper and label
        [type, label] = strtok(tline, '{');

        % remove unnecessary characters
        type = strrep(type, '@', '');
        label = strtrim(label(2:end-1));
        label = strrep(label, '-', '');

        % initialise struktur for this paper
        refStruct = struct('type', type);

        % read attributes of the paper
        tline = fgetl(fid);
        while ~startsWith(tline, '}')
            parts = strsplit(tline, '=', 'CollapseDelimiters', 1);
            if numel(parts) == 2
                % extract key and value
                key = strtrim(parts{1});
                value = strtrim(parts{2}(1:end-1));

                % remove '{' and '}' from value
                value = regexprep(value, '[{}]', '');

                % add entry to the structure
                refStruct.(key) = value;
            end

            % read next line
            tline = fgetl(fid);
        end

        % add paper to the bibStruct
        bibStruct.(label) = refStruct;
    end

    
    % read next line
    tline = fgetl(fid);
end

% cloe file
fclose(fid);
