%% Figures and tables related to journals

% import paper from BibTeX file
bibdata = readBibTeX('../rp.bib');

% all labels
labels = fieldnames(bibdata);

years = 1987:2024;
topN = 10;


% list of journals
journalPublications = dictionary('Chaos', 0);
journalPublicationsYear = zeros(1, length(years));

for i = 1:length(labels)
    paper = bibdata.(labels{i});

    % skip if paper is software or related
    if strcmpi(paper.annote, 'software') | strcmpi(paper.annote, 'related')
       continue
    end

    if isfield(paper,'journal')
        journal = paper.journal;
        year_ = 0;
        if isfield(paper,'year')
            year_ = str2num(paper.year) - 1986;
        end

       if isKey(journalPublications,journal)
            journalPublications(journal) = journalPublications(journal) + 1; % increment value of key year_
            idx = find(strcmp(keys(journalPublications), journal)); % position of journal in list

            if year_ > 0
                journalPublicationsYear(idx,year_) = journalPublicationsYear(idx,year_) + 1;
            end
       else % journal appears for first time
            journalPublications = insert(journalPublications, journal, 1); % add value 1 for key year_
            idx = find(strcmp(keys(journalPublications), journal)); % position of journal in list
            if year_ > 0
                journalPublicationsYear(idx,year_) = 1;
            end
       end
    end
end

length(values(journalPublications)) % number of journals


%% Table: Top journals
% sort journals by number of publications
[j idx] = sort(values(journalPublications),'desc');
k = keys(journalPublications);
journalPublications(k(idx));

% show list for manuscript
rank = 1;
for i = 1:length(journalPublications.values)
    if i > 1 & journalPublications(k(idx(i-1))) ~= journalPublications(k(idx(i)))
        rank = rank + 1;
    end
    
    if rank <= topN || journalPublications(k(idx(i))) >= 25
        disp(sprintf('%d\t&%s\t&%d\t&%1.2f\\\\',rank,k{idx(i)}, journalPublications(k(idx(i))), 100*journalPublications(k(idx(i)))/sum(values(journalPublications)) ))
    end
end

% export list as file
f = fopen('../Data/journals.txt', 'w');
fprintf(f,'Rank\tJournal\tPublications\tFraction\n');
rank = 1;
for i = 1:length(idx)
    if i > 1 & journalPublications(k(idx(i-1))) ~= journalPublications(k(idx(i)))
        rank = rank + 1;
    end
    fprintf(f,'%d\t%s\t%d\t%1.2f\n',rank,k{idx(i)}, journalPublications(k(idx(i))), 100*journalPublications(k(idx(i)))/sum(values(journalPublications)) );
end
fclose(f);



%% Figure: Top journals over the years

% journal ranks by decade
%d = [5 10 15 20 25 30 40]; d(d > size(journalPublicationsYear,2)) = size(journalPublicationsYear,2);
d = [1:38]; d(d > size(journalPublicationsYear,2)) = size(journalPublicationsYear,2);
pubYear = zeros(length(journalPublicationsYear), length(d));
for i = 1:length(d)
    x = sum(journalPublicationsYear(:,1:d(i)),2);
    pubYear(:, i) = x;
    
    disp([repmat(num2str(1986+i),topN,1), k(idx(1:topN)), x(idx(1:topN))])
        
end

topN = 16;

% prepare colormap
c = [
    0 0.4470 0.7410;
    0.8500 0.3250 0.0980;
    0.9290 0.6940 0.1250;
    0.6 0.5 0.5;
    0.4940 0.1840 0.5560;
    0.4660 0.6740 0.1880;
    0.3010 0.7450 0.9330;
    0.6350 0.0780 0.1840;
    0.7 0.7 0.7
];
cm = [c; .9*c; 1.1*c]; cm(cm>1) = 1;

pubYear100 = 1*pubYear(idx(1:topN),:) ./ sum(pubYear(idx(1:topN),:));

clf
set(gcf, 'Pos', [731 100 528 670])
h = bar(1986+d, pubYear100, 1, 'stacked', 'EdgeColor', 'none');
for i = 1:length(h)
   set(h(i), 'FaceColor', cm(i,:))
end
xlabel('Year'), ylabel('Fraction')

% add text (cluster ID)
pubYearCumsum = cumsum(pubYear100(:,end));
for i = 1:topN
   text(2025, pubYearCumsum(i)-pubYear100(i,end)/2, k{idx(i)}, 'color', cm(i,:))
end
axis([1987 2024 0 1])
pos = get(gca, 'Pos');
set(gca, 'Pos', [pos(1) pos(2) .9*pos(3) pos(4)]) % make axes object a bit smaller to show text at the right border









%% Journals citations
% (not included in the paper)

% import paper from BibTeX file
bibdata = readBibTeX('../rp.bib');
% all labels
labels = fieldnames(bibdata);

% import citation file
citationData = jsondecode(fileread('../Data/citations.json'));


% find cited journals
journalCitations = dictionary('Chaos', 0);
for i = 1:length(labels)

    paper = bibdata.(labels{i});
    % skip if paper is software or related
    if strcmpi(paper.annote, 'software') | strcmpi(paper.annote, 'related')
       continue
    end
    
    citedWorks = citationData.(labels{i});
    if length(citedWorks)
        for j = 1:length(citedWorks)
            if ~isfield(bibdata, citedWorks{j}) | strcmpi(bibdata.(citedWorks{j}).annote, 'software') | strcmpi(bibdata.(citedWorks{j}).annote, 'related') 
                continue
            end
            citedPaper = bibdata.(citedWorks{j});
            if isfield(citedPaper,'journal') & isKey(journalCitations,citedPaper.journal)
                journalCitations(citedPaper.journal) = journalCitations(citedPaper.journal) + 1;
            end
            if isfield(citedPaper,'journal') & ~isKey(journalCitations,citedPaper.journal)
                journalCitations = insert(journalCitations, citedPaper.journal, 1); % add value 1 for key year_
            end
       end
    end
end


% sort journals by number of citations
[j idx] = sort(values(journalCitations),'desc');
k = keys(journalCitations);
journalCitations(k(idx));


% show list for manuscript
rank = 1;
for i = 1:topN
    if i > 1 & journalCitations(k(idx(i-1))) ~= journalCitations(k(idx(i)))
        rank = rank + 1;
    end
    disp(sprintf('%d\t&%s\t&%d\t&%1.2f\\\\',rank,k{idx(i)}, journalCitations(k(idx(i))), 100*journalCitations(k(idx(i)))/sum(values(journalCitations)) ))
end



% average number of citations
journalAvgCitations = dictionary('Chaos', 0);
for i = 1:length(idx)
    if isKey(journalCitations, k{idx(i)}) & isKey(journalPublications, k{idx(i)})
       journalAvgCitations = insert(journalAvgCitations, k{idx(i)},  journalCitations(k{idx(i)}) / journalPublications(k{idx(i)}));
       disp(sprintf('%s:  citations: %i - publications: %i - avg: %f', k{idx(i)}, journalCitations(k{idx(i)}), journalPublications(k{idx(i)}), journalAvgCitations(k{idx(i)})))
    end
end

% sort journal by number of average citations
[j idx] = sort(values(journalAvgCitations),'desc');
k = keys(journalAvgCitations);
journalAvgCitations(k(idx));

% show list for manuscript
rank = 1;
oldNum = 0;
for i = 1:length(k)
    if isKey(journalAvgCitations, k(idx(i)))
        if i > 1 & oldNum ~= journalAvgCitations(k(idx(i)))
            rank = rank + 1;
        end
        if i <= topN | journalAvgCitations(k{idx(i)}) >= 30
            disp(sprintf('%d\t&%s\t&%.0f\t&%.0f\t&%1.2f\\\\',rank, k{idx(i)}, journalPublications(k{idx(i)}), journalCitations(k{idx(i)}),  journalAvgCitations(k{idx(i)}) ))
        end
        oldNum = journalAvgCitations(k(idx(i)));
    end
end
