#!/usr/bin/env bash
# Thomas Reerink
#
# This scripts needs four or five arguments
#
# first   argument is the MIP name
# second  argument is the experiment name or MIP name in the latter case all MIP experiments are taken.
# third   argument is the ec-earth model configuration
# fourth  OPTIONAL argument is the meta data template json file which is used as input, the file, e.g. (default): resources/metadata-templates/cmip6-CMIP-piControl-metadata-template.json
#
# For examples how to call this script, run it without arguments.
#
# With this script it is possible to generate a dedicated metadata template json file for each ec-earth3 cmip6 MIP experiment for all active components.
#
# This script is part of the subpackage genecec (GENerate EC-Eearth Control output files)
# which is part of ece2cmor3.
#
# Note that this script is called in a loop over the MIP experiments by the script:
#  genecec.py

if [ "$#" -eq 3 ] || [ "$#" -eq 4 ]; then
 mip=$1
 experiment=$2
 ececonf=$3
 input_template=../resources/metadata-templates/cmip6-CMIP-piControl-metadata-template.json
 if [ "$#" -eq 4 ]; then input_template=$4; fi


 if [ "${ececonf}" != 'EC-EARTH-AOGCM' ] && [ "${ececonf}" != 'EC-EARTH-HR' ] && [ "${ececonf}" != 'EC-EARTH-LR' ] && [ "${ececonf}" != 'EC-EARTH-CC' ] && [ "${ececonf}" != 'EC-EARTH-GrisIS' ] && [ "${ececonf}" != 'EC-EARTH-AerChem' ] && [ "${ececonf}" != 'EC-EARTH-Veg' ] && [ "${ececonf}" != 'EC-EARTH-Veg-LR' ]; then
  echo ' Error in ' $0 ': unkown ec-earth configuration: '  ${ececonf}
  echo ' Valid options: ' 'EC-EARTH-AOGCM' 'EC-EARTH-HR' 'EC-EARTH-LR' 'EC-EARTH-CC' 'EC-EARTH-GrisIS' 'EC-EARTH-AerChem' 'EC-EARTH-Veg' 'EC-EARTH-Veg-LR'
  exit
 fi

 # Some of the experiments do not use nemo as they are atmosphere-only experiments. In that case the source_type has to become AGCM instead of AOGCM:
 declare -a agcm_exceptions=('amip' 'SST' 'piClim')
 adjust_to_agcm=false
 for j in "${agcm_exceptions[@]}"
 do
  if [[ "${experiment}" == *"${j}"* ]]; then
   echo
   echo ' The ' ${experiment} ' experiment has been identified as an AGCM experiment because it carries the substring: ' ${j}
   adjust_to_agcm=true
  fi
 done

 if [ "${adjust_to_agcm}" = true ]; then
  source_type_index=8  # The AGCM case
 else
  source_type_index=7  # The usual AOGCM case
 fi

 # Add your NEWCOMPONENT to this if-blocks if you want to extend ece2cmor3 for more model components.
 if [ "${ececonf}" = 'EC-EARTH-AOGCM'   ]; then declare -a model_components=('ifs' 'nemo'             ); fi
 if [ "${ececonf}" = 'EC-EARTH-HR'      ]; then declare -a model_components=('ifs' 'nemo'             ); fi
 if [ "${ececonf}" = 'EC-EARTH-LR'      ]; then declare -a model_components=('ifs' 'nemo'             ); fi
 if [ "${ececonf}" = 'EC-EARTH-CC'      ]; then declare -a model_components=('ifs' 'nemo' 'tm5' 'lpjg'); fi
 if [ "${ececonf}" = 'EC-EARTH-GrisIS'  ]; then declare -a model_components=('ifs' 'nemo'             ); fi
 if [ "${ececonf}" = 'EC-EARTH-AerChem' ]; then declare -a model_components=('ifs' 'nemo' 'tm5'       ); fi
 if [ "${ececonf}" = 'EC-EARTH-Veg'     ]; then declare -a model_components=('ifs' 'nemo'       'lpjg'); fi
 if [ "${ececonf}" = 'EC-EARTH-Veg-LR'  ]; then declare -a model_components=('ifs' 'nemo'       'lpjg'); fi

 #                    NAME IN SCRIPT                                 ECE CONF NAME       IFS RES     NEMO RES      TM5 RES                                      LPJG RES   PISCES RES  PISM RES    source_type AOGCM  source_type AGCM
 if [ "${ececonf}" = 'EC-EARTH-AOGCM'   ]; then declare -a ece_res=('EC-Earth3'          'T255L91'  'ORCA1L75'    'none'                                        'none'     'none'      'none'      'AOGCM'            'AGCM'          ); fi
 if [ "${ececonf}" = 'EC-EARTH-HR'      ]; then declare -a ece_res=('EC-Earth3-HR'       'T511L91'  'ORCA025L75'  'none'                                        'none'     'none'      'none'      'AOGCM'            'AGCM'          ); fi
 if [ "${ececonf}" = 'EC-EARTH-LR'      ]; then declare -a ece_res=('EC-Earth3-LR'       'T159L62'  'ORCA1L75'    'none'                                        'none'     'none'      'none'      'AOGCM'            'AGCM'          ); fi
 if [ "${ececonf}" = 'EC-EARTH-CC'      ]; then declare -a ece_res=('EC-Earth3-CC'       'T255L91'  'ORCA1L75'    'native regular 2x3 degree latxlon grid L10'  'T255L91'  'ORCA1L75'  'none'      'AOGCM BGC'        'AGCM BGC'      ); fi
 if [ "${ececonf}" = 'EC-EARTH-GrisIS'  ]; then declare -a ece_res=('EC-Earth3-GrIS'     'T255L91'  'ORCA1L75'    'none'                                        'none'     'none'      '5 x 5 km'  'AOGCM ISM'        'AGCM ISM'      ); fi
 if [ "${ececonf}" = 'EC-EARTH-AerChem' ]; then declare -a ece_res=('EC-Earth3-AerChem'  'T255L91'  'ORCA1L75'    'native regular 2x3 degree latxlon grid L34'  'none'     'none'      'none'      'AOGCM AER CHEM'   'AGCM AER CHEM' ); fi
 if [ "${ececonf}" = 'EC-EARTH-Veg'     ]; then declare -a ece_res=('EC-Earth3-Veg'      'T255L91'  'ORCA1L75'    'none'                                        'T255L91'  'none'      'none'      'AOGCM'            'AGCM'          ); fi
 if [ "${ececonf}" = 'EC-EARTH-Veg-LR'  ]; then declare -a ece_res=('EC-Earth3-Veg-LR'   'T159L62'  'ORCA1L75'    'none'                                        'T159L62'  'none'      'none'      'AOGCM'            'AGCM'          ); fi

 if [ "${ececonf}" = 'EC-EARTH-AOGCM'   ]; then declare -a nom_res=('EC-Earth3'          '100 km'   '100 km'      'none'                                        'none'     'none'      'none'      'AOGCM'            'AGCM'          ); fi
 if [ "${ececonf}" = 'EC-EARTH-HR'      ]; then declare -a nom_res=('EC-Earth3-HR'        '50 km'    '25 km'      'none'                                        'none'     'none'      'none'      'AOGCM'            'AGCM'          ); fi
 if [ "${ececonf}" = 'EC-EARTH-LR'      ]; then declare -a nom_res=('EC-Earth3-LR'       '100 km'   '100 km'      'none'                                        'none'     'none'      'none'      'AOGCM'            'AGCM'          ); fi
 if [ "${ececonf}" = 'EC-EARTH-CC'      ]; then declare -a nom_res=('EC-Earth3-CC'       '100 km'   '100 km'      '250 km'                                      '100 km'   '100 km'    'none'      'AOGCM BGC'        'AGCM BGC'      ); fi
 if [ "${ececonf}" = 'EC-EARTH-GrisIS'  ]; then declare -a nom_res=('EC-Earth3-GrIS'     '100 km'   '100 km'      'none'                                        'none'     'none'      '5 km'      'AOGCM ISM'        'AGCM ISM'      ); fi
 if [ "${ececonf}" = 'EC-EARTH-AerChem' ]; then declare -a nom_res=('EC-Earth3-AerChem'  '100 km'   '100 km'      '250 km'                                      'none'     'none'      'none'      'AOGCM AER CHEM'   'AGCM AER CHEM' ); fi
 if [ "${ececonf}" = 'EC-EARTH-Veg'     ]; then declare -a nom_res=('EC-Earth3-Veg'      '100 km'   '100 km'      'none'                                        '100 km'   'none'      'none'      'AOGCM'            'AGCM'          ); fi
 if [ "${ececonf}" = 'EC-EARTH-Veg-LR'  ]; then declare -a nom_res=('EC-Earth3-Veg-LR'   '100 km'   '100 km'      'none'                                        '100 km'   'none'      'none'      'AOGCM'            'AGCM'          ); fi
 # https://www.earthsystemcog.org/site_media/projects/wip/CMIP6_global_attributes_filenames_CVs_v6.2.6.pdf
 # https://github.com/WCRP-CMIP/CMIP6_CVs/blob/master/CMIP6_nominal_resolution.json
 # IFS  T511   T255   T159    ORCA1   ORCA0.25  TM5                                               LPJG=IFS
 #      40 km  80 km  125 km  100 km  25 km     native regular 2x3 degree latxlon grid => 348 km



 # Note that the code block below is produced by the create-parent-experiment-check-code.py script which
 # collects the content from the cmor tables from the CMIP6_CV.json file (the sorted-code-parent-check.sh is pasted):
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'hist-1950HC'                  ]; then declare -a parent_info=('AerChemMIP'                   'hist-1950HC'                  'CMIP'                         'historical'                   'AOGCM'                        'BGC'                          'historical forcing, but with1950s halocarbon concentrations; initialized in 1950'                                                                                                                       'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'hist-piAer'                   ]; then declare -a parent_info=('AerChemMIP'                   'hist-piAer'                   'CMIP'                         'piControl'                    'AOGCM'                        'CHEM BGC'                     'historical forcing, but with pre-industrial aerosol emissions'                                                                                                                                          'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'hist-piNTCF'                  ]; then declare -a parent_info=('AerChemMIP'                   'hist-piNTCF'                  'CMIP'                         'piControl'                    'AOGCM'                        'CHEM BGC'                     'historical forcing, but with pre-industrial NTCF emissions'                                                                                                                                             'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'histSST-1950HC'               ]; then declare -a parent_info=('AerChemMIP'                   'histSST-1950HC'               'AerChemMIP'                   'histSST'                      'AGCM'                         'BGC'                          'historical SSTs and historical forcing, but with 1950 halocarbon concentrations. Experiment is initialized from histSST (AerChemMIP) simulation from January 1950'                                      'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'histSST-noLu'                 ]; then declare -a parent_info=('AerChemMIP'                   'histSST-noLu'                 'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'historical SSTs and historical forcing, but with pre-industrial LULCC'                                                                                                                                  'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'histSST-piAer'                ]; then declare -a parent_info=('AerChemMIP'                   'histSST-piAer'                'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'historical SSTs and historical forcing, but with pre-industrial aerosol emissions'                                                                                                                      'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'histSST-piCH4'                ]; then declare -a parent_info=('AerChemMIP'                   'histSST-piCH4'                'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'historical SSTs and historical forcing, but with pre-industrial methane concentrations'                                                                                                                 'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'histSST-piN2O'                ]; then declare -a parent_info=('AerChemMIP'                   'histSST-piN2O'                'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'historical SSTs and historical forcings, but with pre-industrial N2O concentrations'                                                                                                                    'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'histSST-piNTCF'               ]; then declare -a parent_info=('AerChemMIP'                   'histSST-piNTCF'               'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'historical SSTs and historical forcing, but with pre-industrial NTCF emissions'                                                                                                                         'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'histSST-piO3'                 ]; then declare -a parent_info=('AerChemMIP'                   'histSST-piO3'                 'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'historical SSTs and historical forcing, but with pre-industrial ozone precursor emissions'                                                                                                              'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'histSST'                      ]; then declare -a parent_info=('AerChemMIP'                   'histSST'                      'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'historical prescribed SSTs and historical forcing'                                                                                                                                                      'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-2xDMS'                 ]; then declare -a parent_info=('AerChemMIP'                   'piClim-2xDMS'                 'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'pre-industrial climatological SSTs and forcing, but with doubled emissions of DMS'                                                                                                                      'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-2xdust'                ]; then declare -a parent_info=('AerChemMIP'                   'piClim-2xdust'                'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'pre-industrial climatological SSTs and forcing, but with doubled emissions of dust'                                                                                                                     'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-2xfire'                ]; then declare -a parent_info=('AerChemMIP'                   'piClim-2xfire'                'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'pre-industrial climatological SSTs and forcing, but with doubled emissions from fires'                                                                                                                  'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-2xNOx'                 ]; then declare -a parent_info=('AerChemMIP'                   'piClim-2xNOx'                 'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'pre-industrial climatological SSTs and forcing, but with doubled production of NOx due to lightning'                                                                                                    'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-2xss'                  ]; then declare -a parent_info=('AerChemMIP'                   'piClim-2xss'                  'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'pre-industrial climatological SSTs and forcing, but with doubled emissions of sea salt'                                                                                                                 'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-2xVOC'                 ]; then declare -a parent_info=('AerChemMIP'                   'piClim-2xVOC'                 'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'pre-industrial climatological SSTs and forcing, but with doubled emissions of biogenic VOCs'                                                                                                            'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-aer'                   ]; then declare -a parent_info=('AerChemMIP'                   'piClim-aer'                   'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'effective radiative forcing by present-day aerosols'                                                                                                                                                    'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-BC'                    ]; then declare -a parent_info=('AerChemMIP'                   'piClim-BC'                    'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'pre-industrial climatological SSTs and forcing, but with 2014 black carbon emissions'                                                                                                                   'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-CH4'                   ]; then declare -a parent_info=('AerChemMIP'                   'piClim-CH4'                   'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'pre-industrial climatological SSTs and forcing, but with 2014 methane concentrations (including chemistry)'                                                                                             'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-control'               ]; then declare -a parent_info=('AerChemMIP'                   'piClim-control'               'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'Control simulation providing baseline for evaluating effective radiative forcing (ERF)'                                                                                                                 'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-HC'                    ]; then declare -a parent_info=('AerChemMIP'                   'piClim-HC'                    'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'pre-industrial climatological SSTs and forcing, but with 2014 halocarbon concentrations (including chemistry)'                                                                                          'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-N2O'                   ]; then declare -a parent_info=('AerChemMIP'                   'piClim-N2O'                   'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'pre-industrial climatological SSTs and forcing, but with 2014 N2O concentrations (including chemistry)'                                                                                                 'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-NH3'                   ]; then declare -a parent_info=('AerChemMIP'                   'piClim-NH3'                   'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'pre-industrial climatological SSTs and forcing, but with 2014 ammonia emissions'                                                                                                                        'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-NOx'                   ]; then declare -a parent_info=('AerChemMIP'                   'piClim-NOx'                   'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'pre-industrial climatological SSTs and forcing, but with 2014 NOx emissions'                                                                                                                            'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-NTCF'                  ]; then declare -a parent_info=('AerChemMIP'                   'piClim-NTCF'                  'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'pre-industrial climatological SSTs and forcing, but with 2014 NTCF emissions'                                                                                                                           'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-O3'                    ]; then declare -a parent_info=('AerChemMIP'                   'piClim-O3'                    'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'pre-industrial climatological SSTs and forcing, but with 2014 ozone precursor emissions'                                                                                                                'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-OC'                    ]; then declare -a parent_info=('AerChemMIP'                   'piClim-OC'                    'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'pre-industrial climatological SSTs and forcing, but with 2014 organic carbon emissions'                                                                                                                 'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-SO2'                   ]; then declare -a parent_info=('AerChemMIP'                   'piClim-SO2'                   'CMIP'                         'piControl'                    'AGCM'                         'CHEM BGC'                     'pre-industrial climatological SSTs and forcing, but with 2014 SO2 emissions'                                                                                                                            'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'piClim-VOC'                   ]; then declare -a parent_info=('AerChemMIP'                   'piClim-VOC'                   'CMIP'                         'piControl'                    'AGCM'                         'BGC'                          'pre-industrial climatological SSTs and forcing, but with 2014 VOC emissions'                                                                                                                            'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370-lowNTCFCH4'            ]; then declare -a parent_info=('AerChemMIP'                   'ssp370-lowNTCFCH4'            'CMIP'                         'historical'                   'AOGCM'                        'CHEM BGC'                     'SSP3-7.0, with low NTCF emissions and methane concentrations'                                                                                                                                           'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370-lowNTCF'               ]; then declare -a parent_info=('AerChemMIP'                   'ssp370-lowNTCF'               'CMIP'                         'historical'                   'AOGCM'                        'CHEM BGC'                     'SSP3-7.0, with low NTCF emissions'                                                                                                                                                                      'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370pdSST'                  ]; then declare -a parent_info=('AerChemMIP'                   'ssp370pdSST'                  'CMIP'                         'historical'                   'AGCM'                         'CHEM BGC'                     'SSP3-7.0, with SSTs prescribed as present day'                                                                                                                                                          'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370SST-lowAer'             ]; then declare -a parent_info=('AerChemMIP'                   'ssp370SST-lowAer'             'CMIP'                         'historical'                   'AGCM'                         'CHEM BGC'                     'SSP3-7.0, prescribed SSTs, with low aerosol emissions'                                                                                                                                                  'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370SST-lowBC'              ]; then declare -a parent_info=('AerChemMIP'                   'ssp370SST-lowBC'              'CMIP'                         'historical'                   'AGCM'                         'CHEM BGC'                     'SSP3-7.0, prescribed SSTs, with low black carbon emissions'                                                                                                                                             'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370SST-lowCH4'             ]; then declare -a parent_info=('AerChemMIP'                   'ssp370SST-lowCH4'             'CMIP'                         'historical'                   'AGCM'                         'BGC'                          'SSP3-7.0, prescribed SSTs, with low methane concentrations'                                                                                                                                             'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370SST-lowNTCFCH4'         ]; then declare -a parent_info=('AerChemMIP'                   'ssp370SST-lowNTCFCH4'         'CMIP'                         'historical'                   'AGCM'                         'CHEM BGC'                     'SSP3-7.0, prescribed SSTs, with low NTCF emissions and methane concentrations'                                                                                                                          'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370SST-lowNTCF'            ]; then declare -a parent_info=('AerChemMIP'                   'ssp370SST-lowNTCF'            'CMIP'                         'historical'                   'AGCM'                         'CHEM BGC'                     'SSP3-7.0, prescribed SSTs, with low NTCF emissions'                                                                                                                                                     'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370SST-lowO3'              ]; then declare -a parent_info=('AerChemMIP'                   'ssp370SST-lowO3'              'CMIP'                         'historical'                   'AGCM'                         'BGC'                          'SSP3-7.0, prescribed SSTs, with low ozone precursor emissions'                                                                                                                                          'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370SST-ssp126Lu'           ]; then declare -a parent_info=('AerChemMIP'                   'ssp370SST-ssp126Lu'           'CMIP'                         'historical'                   'AGCM'                         'CHEM BGC'                     'SSP3-7.0, prescribed SSTs, with SSP1-2.6 land use'                                                                                                                                                      'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370SST'                    ]; then declare -a parent_info=('AerChemMIP'                   'ssp370SST'                    'CMIP'                         'historical'                   'AGCM'                         'CHEM BGC'                     'SSP3-7.0, with SSTs prescribed from ssp370'                                                                                                                                                             'none'); fi
 if [ "$mip" = 'AerChemMIP'                   ] && [ "$experiment" = 'ssp370'                       ]; then declare -a parent_info=('AerChemMIP'                   'ssp370'                       'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'gap-filling scenario reaching 7.0 based on SSP3'                                                                                                                                                        'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = '1pctCO2-bgc'                  ]; then declare -a parent_info=('C4MIP'                        '1pctCO2-bgc'                  'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM'                     'biogeochemically-coupled version of 1 percent per year increasing CO2 experiment'                                                                                                                       'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = '1pctCO2Ndep-bgc'              ]; then declare -a parent_info=('C4MIP'                        '1pctCO2Ndep-bgc'              'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM'                     'biogeochemically-coupled version of 1 percent per year increasing CO2 experiment with increasing N-deposition'                                                                                          'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = '1pctCO2Ndep'                  ]; then declare -a parent_info=('C4MIP'                        '1pctCO2Ndep'                  'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM'                     '1 percent per year increasing CO2 experiment with increasing N-deposition'                                                                                                                              'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = '1pctCO2-rad'                  ]; then declare -a parent_info=('C4MIP'                        '1pctCO2-rad'                  'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM'                     'radiatively-coupled version of 1 percent per year increasing CO2 experiment'                                                                                                                            'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'esm-1pct-brch-1000PgC'        ]; then declare -a parent_info=('C4MIP'                        'esm-1pct-brch-1000PgC'        'CMIP'                         '1pctCO2'                      'AOGCM'                        'AER CHEM'                     'zero emissions simulation branched from 1% run after 1000 PgC cumulative emission'                                                                                                                      'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'esm-1pct-brch-2000PgC'        ]; then declare -a parent_info=('C4MIP'                        'esm-1pct-brch-2000PgC'        'CMIP'                         '1pctCO2'                      'AOGCM'                        'AER CHEM'                     'zero emissions simulation branched from 1% run after 2000 PgC cumulative emission'                                                                                                                      'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'esm-1pct-brch-750PgC'         ]; then declare -a parent_info=('C4MIP'                        'esm-1pct-brch-750PgC'         'CMIP'                         '1pctCO2'                      'AOGCM'                        'AER CHEM'                     'zero emissions simulation branched from 1% run after 750 PgC cumulative emission'                                                                                                                       'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'esm-1pctCO2'                  ]; then declare -a parent_info=('C4MIP'                        'esm-1pctCO2'                  'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'emissions driven 1% run'                                                                                                                                                                                'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'esm-bell-1000PgC'             ]; then declare -a parent_info=('C4MIP'                        'esm-bell-1000PgC'             'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'emissions driven 1000PgC bell-curve'                                                                                                                                                                    'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'esm-bell-2000PgC'             ]; then declare -a parent_info=('C4MIP'                        'esm-bell-2000PgC'             'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'emissions driven 2000PgC bell-curve'                                                                                                                                                                    'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'esm-bell-750PgC'              ]; then declare -a parent_info=('C4MIP'                        'esm-bell-750PgC'              'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'emissions driven 750PgC bell-curve'                                                                                                                                                                     'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'esm-ssp585'                   ]; then declare -a parent_info=('C4MIP'                        'esm-ssp585'                   'CMIP'                         'esm-hist'                     'AOGCM'                        'AER CHEM'                     'emission-driven RCP8.5 based on SSP5'                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'hist-bgc'                     ]; then declare -a parent_info=('C4MIP'                        'hist-bgc'                     'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM'                     'biogeochemically-coupled version of the simulation of the recent past with CO2 concentration prescribed'                                                                                                'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'ssp534-over-bgc'              ]; then declare -a parent_info=('C4MIP'                        'ssp534-over-bgc'              'C4MIP'                        'ssp585-bgc'                   'AOGCM'                        'AER CHEM'                     'biogeochemically-coupled version of the RCP3.4-overshoot based on SSP5'                                                                                                                                 'none'); fi
 if [ "$mip" = 'C4MIP'                        ] && [ "$experiment" = 'ssp585-bgc'                   ]; then declare -a parent_info=('C4MIP'                        'ssp585-bgc'                   'C4MIP'                        'hist-bgc'                     'AOGCM'                        'AER CHEM'                     'biogeochemically-coupled version of the RCP8.5 based on SSP5'                                                                                                                                           'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = '1pctCO2-cdr'                  ]; then declare -a parent_info=('CDRMIP'                       '1pctCO2-cdr'                  'CMIP'                         '1pctCO2'                      'AOGCM'                        'AER CHEM'                     '1 percent per year decrease in CO2 from 4xCO2'                                                                                                                                                          'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-1pct-brch-1000PgC'        ]; then declare -a parent_info=('CDRMIP'                       'esm-1pct-brch-1000PgC'        'CMIP'                         '1pctCO2'                      'AOGCM'                        'AER CHEM'                     'zero emissions simulation branched from 1% run after 1000 PgC cumulative emission'                                                                                                                      'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-1pct-brch-2000PgC'        ]; then declare -a parent_info=('CDRMIP'                       'esm-1pct-brch-2000PgC'        'CMIP'                         '1pctCO2'                      'AOGCM'                        'AER CHEM'                     'zero emissions simulation branched from 1% run after 2000 PgC cumulative emission'                                                                                                                      'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-1pct-brch-750PgC'         ]; then declare -a parent_info=('CDRMIP'                       'esm-1pct-brch-750PgC'         'CMIP'                         '1pctCO2'                      'AOGCM'                        'AER CHEM'                     'zero emissions simulation branched from 1% run after 750 PgC cumulative emission'                                                                                                                       'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-1pctCO2'                  ]; then declare -a parent_info=('CDRMIP'                       'esm-1pctCO2'                  'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'emissions driven 1% run'                                                                                                                                                                                'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-bell-1000PgC'             ]; then declare -a parent_info=('CDRMIP'                       'esm-bell-1000PgC'             'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'emissions driven 1000PgC bell-curve'                                                                                                                                                                    'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-bell-2000PgC'             ]; then declare -a parent_info=('CDRMIP'                       'esm-bell-2000PgC'             'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'emissions driven 2000PgC bell-curve'                                                                                                                                                                    'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-bell-750PgC'              ]; then declare -a parent_info=('CDRMIP'                       'esm-bell-750PgC'              'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'emissions driven 750PgC bell-curve'                                                                                                                                                                     'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-pi-cdr-pulse'             ]; then declare -a parent_info=('CDRMIP'                       'esm-pi-cdr-pulse'             'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'pulse removal of 100 Gt carbon from pre-industrial atmosphere'                                                                                                                                          'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-pi-CO2pulse'              ]; then declare -a parent_info=('CDRMIP'                       'esm-pi-CO2pulse'              'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'pulse addition of 100 Gt carbon to pre-industrial atmosphere'                                                                                                                                           'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-ssp534-over'              ]; then declare -a parent_info=('CDRMIP'                       'esm-ssp534-over'              'C4MIP'                        'esm-ssp585'                   'AOGCM'                        'AER CHEM'                     'emission-driven SSP5-3.4-OS scenario'                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-ssp585ext'                ]; then declare -a parent_info=('CDRMIP'                       'esm-ssp585ext'                'C4MIP'                        'esm-ssp585'                   'AOGCM'                        'AER CHEM'                     'emission-driven long-term extension of the SSP5-8.5 scenario'                                                                                                                                           'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-ssp585-ocn-alk-stop'      ]; then declare -a parent_info=('CDRMIP'                       'esm-ssp585-ocn-alk-stop'      'CDRMIP'                       'esm-ssp585-ocn-alk'           'AOGCM'                        'AER CHEM'                     'emission-driven SSP5-8.5 scenario with alkalinization terminated in year 2070'                                                                                                                          'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-ssp585-ocn-alk'           ]; then declare -a parent_info=('CDRMIP'                       'esm-ssp585-ocn-alk'           'C4MIP'                        'esm-ssp585'                   'AOGCM'                        'AER CHEM'                     'emission-driven SSP5-8.5 scenario but with ocean alkalinization from year 2020 onward'                                                                                                                  'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-ssp585-ssp126Lu-ext'      ]; then declare -a parent_info=('CDRMIP'                       'esm-ssp585-ssp126Lu-ext'      'LUMIP'                        'esm-ssp585-ssp126Lu'          'AOGCM'                        'AER CHEM'                     'extension of the LUMIP emissions-driven simulation following SSP5-8.5 with SSP1-2.6 land use'                                                                                                           'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-yr2010CO2-cdr-pulse'      ]; then declare -a parent_info=('CDRMIP'                       'esm-yr2010CO2-cdr-pulse'      'CDRMIP'                       'esm-yr2010CO2-control'        'AOGCM'                        'AER CHEM'                     'instantaneous 100 Gt C removal from industrial era atmosphere'                                                                                                                                          'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-yr2010CO2-CO2pulse'       ]; then declare -a parent_info=('CDRMIP'                       'esm-yr2010CO2-CO2pulse'       'CDRMIP'                       'esm-yr2010CO2-control'        'AOGCM'                        'AER CHEM'                     'instantaneous 100 Gt C addition to an industrial era atmosphere'                                                                                                                                        'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-yr2010CO2-control'        ]; then declare -a parent_info=('CDRMIP'                       'esm-yr2010CO2-control'        'CDRMIP'                       'esm-piControl'                'AOGCM'                        'AER CHEM'                     'historical emissions followed by fixed 2010 emissions (both model-diagnosed)'                                                                                                                           'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'esm-yr2010CO2-noemit'         ]; then declare -a parent_info=('CDRMIP'                       'esm-yr2010CO2-noemit'         'CDRMIP'                       'esm-yr2010CO2-control'        'AOGCM'                        'AER CHEM'                     'branches from esm-yr2010CO2-control with zero emissions'                                                                                                                                                'none'); fi
 if [ "$mip" = 'CDRMIP'                       ] && [ "$experiment" = 'yr2010CO2'                    ]; then declare -a parent_info=('CDRMIP'                       'yr2010CO2'                    'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM'                     'concentration-driven fixed 2010 forcing'                                                                                                                                                                'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'a4SSTice-4xCO2'               ]; then declare -a parent_info=('CFMIP'                        'a4SSTice-4xCO2'               'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'as piSST but with SSTs and sea ice from abrupt-4xCO2, and 4xCO2 seen by radiation and vegetation'                                                                                                       'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'a4SSTice'                     ]; then declare -a parent_info=('CFMIP'                        'a4SSTice'                     'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'as piSST but with SSTs and sea ice from abrupt-4xCO2'                                                                                                                                                   'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'a4SST'                        ]; then declare -a parent_info=('CFMIP'                        'a4SST'                        'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'as piSST but with SSTs from abrupt-4xCO2'                                                                                                                                                               'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'abrupt-0p5xCO2'               ]; then declare -a parent_info=('CFMIP'                        'abrupt-0p5xCO2'               'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'abrupt halving of CO2'                                                                                                                                                                                  'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'abrupt-2xCO2'                 ]; then declare -a parent_info=('CFMIP'                        'abrupt-2xCO2'                 'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'abrupt doubling of CO2'                                                                                                                                                                                 'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'abrupt-solm4p'                ]; then declare -a parent_info=('CFMIP'                        'abrupt-solm4p'                'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'abrupt 4% decrease in solar constant'                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'abrupt-solp4p'                ]; then declare -a parent_info=('CFMIP'                        'abrupt-solp4p'                'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'abrupt 4% increase in solar constant'                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'amip-4xCO2'                   ]; then declare -a parent_info=('CFMIP'                        'amip-4xCO2'                   'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'AMIP SSTs with 4xCO2'                                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'amip-a4SST-4xCO2'             ]; then declare -a parent_info=('CFMIP'                        'amip-a4SST-4xCO2'             'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'as AMIP but with warming pattern from abrupt-4xCO2 added to SSTs and 4xCO2 seen by radiation and vegetation'                                                                                            'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'amip-future4K'                ]; then declare -a parent_info=('CFMIP'                        'amip-future4K'                'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'AMIP with patterned 4K SST increase'                                                                                                                                                                    'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'amip-lwoff'                   ]; then declare -a parent_info=('CFMIP'                        'amip-lwoff'                   'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'AMIP experiment with longwave cloud-radiative effects off'                                                                                                                                              'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'amip-m4K'                     ]; then declare -a parent_info=('CFMIP'                        'amip-m4K'                     'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'AMIP with uniform 4K SST decrease'                                                                                                                                                                      'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'amip-p4K-lwoff'               ]; then declare -a parent_info=('CFMIP'                        'amip-p4K-lwoff'               'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'AMIP experiment with uniform 4K SST increase and with longwave cloud radiative effects off'                                                                                                             'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'amip-p4K'                     ]; then declare -a parent_info=('CFMIP'                        'amip-p4K'                     'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'AMIP with uniform 4K SST increase'                                                                                                                                                                      'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'amip-piForcing'               ]; then declare -a parent_info=('CFMIP'                        'amip-piForcing'               'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'AMIP SSTs with pre-industrial anthropogenic and natural forcing'                                                                                                                                        'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'aqua-4xCO2'                   ]; then declare -a parent_info=('CFMIP'                        'aqua-4xCO2'                   'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'aquaplanet with control SST and 4xCO2'                                                                                                                                                                  'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'aqua-control-lwoff'           ]; then declare -a parent_info=('CFMIP'                        'aqua-control-lwoff'           'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'aquaplanet control with longwave cloud radiative effects off'                                                                                                                                           'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'aqua-control'                 ]; then declare -a parent_info=('CFMIP'                        'aqua-control'                 'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'aquaplanet control'                                                                                                                                                                                     'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'aqua-p4K-lwoff'               ]; then declare -a parent_info=('CFMIP'                        'aqua-p4K-lwoff'               'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'aquaplanet with uniform 4K SST increase and with longwave cloud radiative effects off'                                                                                                                  'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'aqua-p4K'                     ]; then declare -a parent_info=('CFMIP'                        'aqua-p4K'                     'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'aquaplanet with uniform 4K SST increase'                                                                                                                                                                'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'piSST-4xCO2-rad'              ]; then declare -a parent_info=('CFMIP'                        'piSST-4xCO2-rad'              'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'as piSST with radiation-only seeing 4xCO2'                                                                                                                                                              'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'piSST-4xCO2'                  ]; then declare -a parent_info=('CFMIP'                        'piSST-4xCO2'                  'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'as piSST with radiation and vegetation seeing 4xCO2'                                                                                                                                                    'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'piSST-pxK'                    ]; then declare -a parent_info=('CFMIP'                        'piSST-pxK'                    'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'as piSST with uniform SST increase with magnitude based on abrupt-4xCO2 response'                                                                                                                       'none'); fi
 if [ "$mip" = 'CFMIP'                        ] && [ "$experiment" = 'piSST'                        ]; then declare -a parent_info=('CFMIP'                        'piSST'                        'no parent'                    'no parent'                    'AGCM'                         'AER CHEM'                     'experiment forced with pre-industrial SSTs, sea ice and atmospheric constituents'                                                                                                                       'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = '1pctCO2'                      ]; then declare -a parent_info=('CMIP'                         '1pctCO2'                      'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 '1 percent per year increase in CO2'                                                                                                                                                                     'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'abrupt-4xCO2'                 ]; then declare -a parent_info=('CMIP'                         'abrupt-4xCO2'                 'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'abrupt quadrupling of CO2'                                                                                                                                                                              'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'amip'                         ]; then declare -a parent_info=('CMIP'                         'amip'                         'no parent'                    'no parent'                    'AGCM'                         'AER CHEM BGC'                 'AMIP'                                                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'esm-hist-ext'                 ]; then declare -a parent_info=('CMIP'                         'esm-hist-ext'                 'CMIP'                         'esm-hist'                     'AOGCM'                        'AER CHEM'                     'post-2014 all-forcing simulation with atmospheric CO2 concentration calculated'                                                                                                                         'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'esm-hist'                     ]; then declare -a parent_info=('CMIP'                         'esm-hist'                     'CMIP'                         'esm-piControl'                'AOGCM'                        'AER CHEM'                     'all-forcing simulation of the recent past with atmospheric CO2 concentration calculated'                                                                                                                'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'esm-piControl-spinup'         ]; then declare -a parent_info=('CMIP'                         'esm-piControl-spinup'         'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM'                     'pre-industrial control simulation with CO2 concentration calculated (spin-up)'                                                                                                                          'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'esm-piControl'                ]; then declare -a parent_info=('CMIP'                         'esm-piControl'                'CMIP'                         'esm-piControl-spinup'         'AOGCM'                        'AER CHEM'                     'pre-industrial control simulation with CO2 concentration calculated'                                                                                                                                    'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'historical-cmip5'             ]; then declare -a parent_info=('CMIP'                         'historical-cmip5'             'CMIP'                         'piControl-cmip5'              'AOGCM'                        'AER CHEM BGC'                 'all-forcing simulation of the recent past (CMIP5-era [1850-2005] forcing)'                                                                                                                              'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'historical-ext'               ]; then declare -a parent_info=('CMIP'                         'historical-ext'               'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'post-2014 all-forcing simulation'                                                                                                                                                                       'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'historical'                   ]; then declare -a parent_info=('CMIP'                         'historical'                   'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'all-forcing simulation of the recent past'                                                                                                                                                              'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'piControl-cmip5'              ]; then declare -a parent_info=('CMIP'                         'piControl-cmip5'              'CMIP'                         'piControl-spinup-cmip5'       'AOGCM'                        'AER CHEM BGC'                 'pre-industrial control (CMIP5-era [1850-2005] forcing)'                                                                                                                                                 'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'piControl-spinup-cmip5'       ]; then declare -a parent_info=('CMIP'                         'piControl-spinup-cmip5'       'CMIP'                         'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'pre-industrial control (spin-up; CMIP5-era [1850-2005] forcing)'                                                                                                                                        'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'piControl-spinup'             ]; then declare -a parent_info=('CMIP'                         'piControl-spinup'             'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'pre-industrial control (spin-up)'                                                                                                                                                                       'none'); fi
 if [ "$mip" = 'CMIP'                         ] && [ "$experiment" = 'piControl'                    ]; then declare -a parent_info=('CMIP'                         'piControl'                    'CMIP'                         'piControl-spinup'             'AOGCM'                        'AER CHEM BGC'                 'pre-industrial control'                                                                                                                                                                                 'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-aer-cmip5'               ]; then declare -a parent_info=('DAMIP'                        'hist-aer-cmip5'               'CMIP'                         'piControl-cmip5'              'AOGCM'                        'AER CHEM BGC'                 'historical anthropogenic aerosols-only run (CMIP5-era historical [1850-2005] and RCP4.5 [2006-2020] forcing)'                                                                                           'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-aer'                     ]; then declare -a parent_info=('DAMIP'                        'hist-aer'                     'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'historical anthropogenic aerosols-only run'                                                                                                                                                             'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-all-aer2'                ]; then declare -a parent_info=('DAMIP'                        'hist-all-aer2'                'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'historical ALL-forcing run with alternate estimates of aerosol forcing'                                                                                                                                 'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-all-nat2'                ]; then declare -a parent_info=('DAMIP'                        'hist-all-nat2'                'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'historical ALL-forcing run with alternate estimates of natural forcing'                                                                                                                                 'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-CO2'                     ]; then declare -a parent_info=('DAMIP'                        'hist-CO2'                     'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'historical CO2-only run'                                                                                                                                                                                'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-GHG-cmip5'               ]; then declare -a parent_info=('DAMIP'                        'hist-GHG-cmip5'               'CMIP'                         'piControl-cmip5'              'AOGCM'                        'AER CHEM BGC'                 'historical well-mixed GHG-only run (CMIP5-era historical [1850-2005] and RCP4.5 [2006-2020] forcing)'                                                                                                   'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-GHG'                     ]; then declare -a parent_info=('DAMIP'                        'hist-GHG'                     'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'historical well-mixed GHG-only run'                                                                                                                                                                     'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-nat-cmip5'               ]; then declare -a parent_info=('DAMIP'                        'hist-nat-cmip5'               'CMIP'                         'piControl-cmip5'              'AOGCM'                        'AER CHEM BGC'                 'historical natural-only run (CMIP5-era historical [1850-2005] and RCP4.5 [2006-2020] forcing)'                                                                                                          'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-nat'                     ]; then declare -a parent_info=('DAMIP'                        'hist-nat'                     'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'historical natural-only run'                                                                                                                                                                            'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-sol'                     ]; then declare -a parent_info=('DAMIP'                        'hist-sol'                     'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'historical solar-only run'                                                                                                                                                                              'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-stratO3'                 ]; then declare -a parent_info=('DAMIP'                        'hist-stratO3'                 'CMIP'                         'piControl'                    'AOGCM'                        'AER BGC'                      'historical stratospheric ozone-only run'                                                                                                                                                                'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-totalO3'                 ]; then declare -a parent_info=('DAMIP'                        'hist-totalO3'                 'CMIP'                         'piControl'                    'AOGCM'                        'AER BGC'                      'historical total ozone-only run'                                                                                                                                                                        'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'hist-volc'                    ]; then declare -a parent_info=('DAMIP'                        'hist-volc'                    'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'historical volcanic-only run'                                                                                                                                                                           'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'ssp245-aer'                   ]; then declare -a parent_info=('DAMIP'                        'ssp245-aer'                   'DAMIP'                        'hist-aer'                     'AOGCM'                        'AER CHEM BGC'                 'aerosol-only SSP2-4.5 run'                                                                                                                                                                              'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'ssp245-cov-aer'               ]; then declare -a parent_info=('DAMIP'                        'ssp245-cov-aer'               'ScenarioMIP'                  'ssp245'                       'AOGCM'                        'AER CHEM BGC'                 '2-year Covid-19 emissions blip including anthropogenic aerosols only, based upon ssp245'                                                                                                                'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'ssp245-cov-fossil'            ]; then declare -a parent_info=('DAMIP'                        'ssp245-cov-fossil'            'ScenarioMIP'                  'ssp245'                       'AOGCM'                        'AER CHEM BGC'                 '2-year Covid-19 emissions blip followed by increased emissions due to a fossil-fuel based recovery, based upon ssp245'                                                                                  'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'ssp245-cov-GHG'               ]; then declare -a parent_info=('DAMIP'                        'ssp245-cov-GHG'               'ScenarioMIP'                  'ssp245'                       'AOGCM'                        'AER CHEM BGC'                 '2-year Covid-19 emissions blip including well mixed GHG only, based upon ssp245'                                                                                                                        'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'ssp245-covid'                 ]; then declare -a parent_info=('DAMIP'                        'ssp245-covid'                 'ScenarioMIP'                  'ssp245'                       'AOGCM'                        'AER CHEM BGC'                 '2-year Covid-19 emissions blip based upon ssp245'                                                                                                                                                       'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'ssp245-cov-modgreen'          ]; then declare -a parent_info=('DAMIP'                        'ssp245-cov-modgreen'          'ScenarioMIP'                  'ssp245'                       'AOGCM'                        'AER CHEM BGC'                 '2-year Covid-19 emissions blip followed by moderate-green stimulus recovery, based upon ssp245'                                                                                                         'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'ssp245-cov-strgreen'          ]; then declare -a parent_info=('DAMIP'                        'ssp245-cov-strgreen'          'ScenarioMIP'                  'ssp245'                       'AOGCM'                        'AER CHEM BGC'                 '2-year Covid-19 emissions blip followed by strong-green stimulus recovery, based upon ssp245'                                                                                                           'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'ssp245-GHG'                   ]; then declare -a parent_info=('DAMIP'                        'ssp245-GHG'                   'DAMIP'                        'hist-GHG'                     'AOGCM'                        'AER CHEM BGC'                 'well-mixed GHG-only SSP2-4.5 run'                                                                                                                                                                       'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'ssp245-nat'                   ]; then declare -a parent_info=('DAMIP'                        'ssp245-nat'                   'DAMIP'                        'hist-nat'                     'AOGCM'                        'AER CHEM BGC'                 'natural-only SSP2-4.5 run'                                                                                                                                                                              'none'); fi
 if [ "$mip" = 'DAMIP'                        ] && [ "$experiment" = 'ssp245-stratO3'               ]; then declare -a parent_info=('DAMIP'                        'ssp245-stratO3'               'DAMIP'                        'hist-stratO3'                 'AOGCM'                        'AER BGC'                      'stratospheric ozone-only SSP2-4.5 (ssp245) run'                                                                                                                                                         'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppA-assim'                  ]; then declare -a parent_info=('DCPP'                         'dcppA-assim'                  'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'Assimilation run paralleling the historical simulation, which may be used to generate hindcast initial conditions'                                                                                      'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppA-hindcast-niff'          ]; then declare -a parent_info=('DCPP'                         'dcppA-hindcast-niff'          'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'hindcast initialized based on observations but without using knowledge of subsequent historical forcing'                                                                                                's1960 s1961 s1962 s1963 s1964 s1965 s1966 s1967 s1968 s1969 s1970 s1971 s1972 s1973 s1974 s1975 s1976 s1977 s1978 s1979 s1980 s1981 s1982 s1983 s1984 s1985 s1986 s1987 s1988 s1989 s1990 s1991 s1992 s1993 s1994 s1995 s1996 s1997 s1998 s1999 s2000 s2001 s2002 s2003 s2004 s2005 s2006 s2007 s2008 s2009 s2010 s2011 s2012 s2013 s2014 s2015 s2016 s2017 s2018 s2019'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppA-hindcast'               ]; then declare -a parent_info=('DCPP'                         'dcppA-hindcast'               'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'hindcast initialized based on observations and using historical forcing'                                                                                                                                's1960 s1961 s1962 s1963 s1964 s1965 s1966 s1967 s1968 s1969 s1970 s1971 s1972 s1973 s1974 s1975 s1976 s1977 s1978 s1979 s1980 s1981 s1982 s1983 s1984 s1985 s1986 s1987 s1988 s1989 s1990 s1991 s1992 s1993 s1994 s1995 s1996 s1997 s1998 s1999 s2000 s2001 s2002 s2003 s2004 s2005 s2006 s2007 s2008 s2009 s2010 s2011 s2012 s2013 s2014 s2015 s2016 s2017 s2018 s2019'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppA-historical-niff'        ]; then declare -a parent_info=('DCPP'                         'dcppA-historical-niff'        'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'hindcast initialized from historical climate simulation but without using knowledge of subsequent historical forcing'                                                                                   's1960 s1961 s1962 s1963 s1964 s1965 s1966 s1967 s1968 s1969 s1970 s1971 s1972 s1973 s1974 s1975 s1976 s1977 s1978 s1979 s1980 s1981 s1982 s1983 s1984 s1985 s1986 s1987 s1988 s1989 s1990 s1991 s1992 s1993 s1994 s1995 s1996 s1997 s1998 s1999 s2000 s2001 s2002 s2003 s2004 s2005 s2006 s2007 s2008 s2009 s2010 s2011 s2012 s2013 s2014 s2015 s2016 s2017 s2018 s2019'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppB-forecast'               ]; then declare -a parent_info=('DCPP'                         'dcppB-forecast'               'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'forecast initialized from observations with forcing from ssp245'                                                                                                                                        's2017 s2018 s2019 s2020 s2021 s2022 s2023 s2024 s2025 s2026 s2027 s2028 s2029'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-amv-ExTrop-neg'         ]; then declare -a parent_info=('DCPP'                         'dcppC-amv-ExTrop-neg'         'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'Idealized climate impact of negative extratropical AMV anomaly pattern'                                                                                                                                 'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-amv-ExTrop-pos'         ]; then declare -a parent_info=('DCPP'                         'dcppC-amv-ExTrop-pos'         'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'Idealized climate impact of positive extratropical AMV anomaly pattern'                                                                                                                                 'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-amv-neg'                ]; then declare -a parent_info=('DCPP'                         'dcppC-amv-neg'                'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'Idealized climate impact of negative AMV anomaly pattern'                                                                                                                                               'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-amv-pos'                ]; then declare -a parent_info=('DCPP'                         'dcppC-amv-pos'                'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'Idealized climate impact of positive AMV anomaly pattern'                                                                                                                                               'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-amv-Trop-neg'           ]; then declare -a parent_info=('DCPP'                         'dcppC-amv-Trop-neg'           'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'Idealized climate impact of negative tropical AMV anomaly pattern'                                                                                                                                      'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-amv-Trop-pos'           ]; then declare -a parent_info=('DCPP'                         'dcppC-amv-Trop-pos'           'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'idealized positive tropical AMV anomaly pattern'                                                                                                                                                        'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-atl-control'            ]; then declare -a parent_info=('DCPP'                         'dcppC-atl-control'            'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'Idealized Atlantic control'                                                                                                                                                                             'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-atl-pacemaker'          ]; then declare -a parent_info=('DCPP'                         'dcppC-atl-pacemaker'          'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'pacemaker Atlantic experiment'                                                                                                                                                                          's1910 s1920 s1950'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-atl-spg'                ]; then declare -a parent_info=('DCPP'                         'dcppC-atl-spg'                'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'predictability of 1990s warming of Atlantic sub-polar gyre'                                                                                                                                             's1992 s1993 s1994 s1995 s1996 s1997 s1998 s1999'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-forecast-addAgung'      ]; then declare -a parent_info=('DCPP'                         'dcppC-forecast-addAgung'      'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 '2015 forecast with added Agung forcing'                                                                                                                                                                 's2014'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-forecast-addElChichon'  ]; then declare -a parent_info=('DCPP'                         'dcppC-forecast-addElChichon'  'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 '2015 forecast with added El Chichon forcing'                                                                                                                                                            's2014'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-forecast-addPinatubo'   ]; then declare -a parent_info=('DCPP'                         'dcppC-forecast-addPinatubo'   'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 '2015 forecast with added Pinatubo forcing'                                                                                                                                                              's2014'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-hindcast-noAgung'       ]; then declare -a parent_info=('DCPP'                         'dcppC-hindcast-noAgung'       'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'hindcast but with only background volcanic forcing to be the same as that used in the 2015 forecast'                                                                                                    's1962'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-hindcast-noElChichon'   ]; then declare -a parent_info=('DCPP'                         'dcppC-hindcast-noElChichon'   'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'hindcast but with only background volcanic forcing to be the same as that used in the 2015 forecast'                                                                                                    's1981'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-hindcast-noPinatubo'    ]; then declare -a parent_info=('DCPP'                         'dcppC-hindcast-noPinatubo'    'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'hindcast but with only background volcanic forcing to be the same as that used in the 2015 forecast'                                                                                                    's1990'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-ipv-neg'                ]; then declare -a parent_info=('DCPP'                         'dcppC-ipv-neg'                'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'idealized negative IPV anomaly pattern'                                                                                                                                                                 'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-ipv-NexTrop-neg'        ]; then declare -a parent_info=('DCPP'                         'dcppC-ipv-NexTrop-neg'        'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'idealized negative northern extratropical IPV anomaly pattern'                                                                                                                                          'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-ipv-NexTrop-pos'        ]; then declare -a parent_info=('DCPP'                         'dcppC-ipv-NexTrop-pos'        'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'idealized positive northern extratropical IPV anomaly pattern'                                                                                                                                          'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-ipv-pos'                ]; then declare -a parent_info=('DCPP'                         'dcppC-ipv-pos'                'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'idealized positive IPV anomaly pattern'                                                                                                                                                                 'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-pac-control'            ]; then declare -a parent_info=('DCPP'                         'dcppC-pac-control'            'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'idealized Pacific control'                                                                                                                                                                              'none'); fi
 if [ "$mip" = 'DCPP'                         ] && [ "$experiment" = 'dcppC-pac-pacemaker'          ]; then declare -a parent_info=('DCPP'                         'dcppC-pac-pacemaker'          'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'pacemaker Pacific experiment'                                                                                                                                                                           's1910 s1920 s1950'); fi
 if [ "$mip" = 'FAFMIP'                       ] && [ "$experiment" = 'faf-all'                      ]; then declare -a parent_info=('FAFMIP'                       'faf-all'                      'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'control plus perturbative surface fluxes of momentum, heat and water into ocean'                                                                                                                        'none'); fi
 if [ "$mip" = 'FAFMIP'                       ] && [ "$experiment" = 'faf-antwater-stress'          ]; then declare -a parent_info=('FAFMIP'                       'faf-antwater-stress'          'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'control plus perturbative surface fluxes of momentum and freshwater into ocean, the latter around the coast of Antarctica only'                                                                         'none'); fi
 if [ "$mip" = 'FAFMIP'                       ] && [ "$experiment" = 'faf-heat-NA0pct'              ]; then declare -a parent_info=('FAFMIP'                       'faf-heat-NA0pct'              'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'control plus perturbative surface flux of heat into ocean'                                                                                                                                              'none'); fi
 if [ "$mip" = 'FAFMIP'                       ] && [ "$experiment" = 'faf-heat-NA50pct'             ]; then declare -a parent_info=('FAFMIP'                       'faf-heat-NA50pct'             'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'control plus perturbative surface flux of heat into ocean'                                                                                                                                              'none'); fi
 if [ "$mip" = 'FAFMIP'                       ] && [ "$experiment" = 'faf-heat'                     ]; then declare -a parent_info=('FAFMIP'                       'faf-heat'                     'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'control plus perturbative surface flux of heat into ocean'                                                                                                                                              'none'); fi
 if [ "$mip" = 'FAFMIP'                       ] && [ "$experiment" = 'faf-passiveheat'              ]; then declare -a parent_info=('FAFMIP'                       'faf-passiveheat'              'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'control plus surface flux of passive heat tracer into ocean'                                                                                                                                            'none'); fi
 if [ "$mip" = 'FAFMIP'                       ] && [ "$experiment" = 'faf-stress'                   ]; then declare -a parent_info=('FAFMIP'                       'faf-stress'                   'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'control plus perturbative surface flux of momentum into ocean'                                                                                                                                          'none'); fi
 if [ "$mip" = 'FAFMIP'                       ] && [ "$experiment" = 'faf-water'                    ]; then declare -a parent_info=('FAFMIP'                       'faf-water'                    'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'control plus perturbative surface flux of water into ocean'                                                                                                                                             'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'futureSST-4xCO2-solar'        ]; then declare -a parent_info=('GeoMIP'                       'futureSST-4xCO2-solar'        'GeoMIP'                       'G1'                           'AGCM'                         'AER CHEM'                     'year 100 SSTs from abrupt-4xCO2 with quadrupled CO2 and solar reduction'                                                                                                                                'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'G1'                           ]; then declare -a parent_info=('GeoMIP'                       'G1'                           'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'abrupt quadrupling of CO2 plus reduction in total solar irradiance'                                                                                                                                     'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'G6solar'                      ]; then declare -a parent_info=('GeoMIP'                       'G6solar'                      'ScenarioMIP'                  'ssp585'                       'AOGCM'                        'AER CHEM BGC'                 'total solar irradiance reduction to reduce net forcing from SSP585 to SSP245'                                                                                                                           'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'G6SST1'                       ]; then declare -a parent_info=('GeoMIP'                       'G6SST1'                       'ScenarioMIP'                  'ssp585'                       'AGCM'                         'AER CHEM'                     'SSTs, forcings, and other prescribed conditions from year 2020 of SSP5-8.5'                                                                                                                             'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'G6SST2-solar'                 ]; then declare -a parent_info=('GeoMIP'                       'G6SST2-solar'                 'GeoMIP'                       'G6solar'                      'AGCM'                         'AER CHEM'                     'SSTs from year 2020 of SSP5-8.5; forcings and other prescribed conditions from year 2100 of G6solar'                                                                                                    'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'G6SST2-sulfur'                ]; then declare -a parent_info=('GeoMIP'                       'G6SST2-sulfur'                'GeoMIP'                       'G6sulfur'                     'AGCM'                         'AER CHEM'                     'SSTs from year 2020 of SSP5-8.5; forcings and other prescribed conditions from year 2100 of G6sulfur'                                                                                                   'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'G6sulfur'                     ]; then declare -a parent_info=('GeoMIP'                       'G6sulfur'                     'ScenarioMIP'                  'ssp585'                       'AOGCM'                        'AER CHEM BGC'                 'stratospheric sulfate aerosol injection to reduce net forcing from SSP585 to SSP245'                                                                                                                    'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'G7cirrus'                     ]; then declare -a parent_info=('GeoMIP'                       'G7cirrus'                     'ScenarioMIP'                  'ssp585'                       'AOGCM'                        'AER CHEM BGC'                 'increase cirrus ice crystal fall speed to reduce net forcing in SSP585 by 1 W m-2'                                                                                                                      'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'G7SST1-cirrus'                ]; then declare -a parent_info=('GeoMIP'                       'G7SST1-cirrus'                'ScenarioMIP'                  'ssp585'                       'AGCM'                         'AER CHEM'                     'SSTs from year 2020 of SSP5-8.5; forcings and other prescribed conditions from year 2020 of SSP5-8.5 and cirrus thinning'                                                                               'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'G7SST2-cirrus'                ]; then declare -a parent_info=('GeoMIP'                       'G7SST2-cirrus'                'GeoMIP'                       'G7cirrus'                     'AGCM'                         'AER CHEM'                     'SSTs from year 2100 of SSP5-8.5; forcings and other prescribed conditions from year 2100 of G7cirrus'                                                                                                   'none'); fi
 if [ "$mip" = 'GeoMIP'                       ] && [ "$experiment" = 'piSST-4xCO2-solar'            ]; then declare -a parent_info=('GeoMIP'                       'piSST-4xCO2-solar'            'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM'                     'preindustrial control SSTs with quadrupled CO2 and solar reduction'                                                                                                                                     'none'); fi
 if [ "$mip" = 'GMMIP'                        ] && [ "$experiment" = 'amip-hist'                    ]; then declare -a parent_info=('GMMIP'                        'amip-hist'                    'no parent'                    'no parent'                    'AGCM'                         'AER CHEM BGC'                 'AMIP-style simulation covering the period 1870-2014'                                                                                                                                                    'none'); fi
 if [ "$mip" = 'GMMIP'                        ] && [ "$experiment" = 'amip-hld'                     ]; then declare -a parent_info=('GMMIP'                        'amip-hld'                     'no parent'                    'no parent'                    'AGCM'                         'AER CHEM BGC'                 'same as "amip" run, but surface elevations of the East African Highlands in Africa, Sierra Madre in N. America and Andes in S. America reduced to 500m'                                                 'none'); fi
 if [ "$mip" = 'GMMIP'                        ] && [ "$experiment" = 'amip-TIP-nosh'                ]; then declare -a parent_info=('GMMIP'                        'amip-TIP-nosh'                'no parent'                    'no parent'                    'AGCM'                         'AER CHEM BGC'                 'same as "amip" run, but sensible heat not allowed for elevations of the Tibetan-Iranian Plateau and Himalayas above 500m'                                                                               'none'); fi
 if [ "$mip" = 'GMMIP'                        ] && [ "$experiment" = 'amip-TIP'                     ]; then declare -a parent_info=('GMMIP'                        'amip-TIP'                     'no parent'                    'no parent'                    'AGCM'                         'AER CHEM BGC'                 'same as "amip" run, but surface elevations of the Tibetan-Iranian Plateau and Himalayas reduced to 500m'                                                                                                'none'); fi
 if [ "$mip" = 'GMMIP'                        ] && [ "$experiment" = 'hist-resAMO'                  ]; then declare -a parent_info=('GMMIP'                        'hist-resAMO'                  'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'initialized from "historical" run year 1870 and SSTs in the AMO domain (0deg-70degN, 70degW-0deg) restored to AMIP SSTs with historical forcings'                                                       'none'); fi
 if [ "$mip" = 'GMMIP'                        ] && [ "$experiment" = 'hist-resIPO'                  ]; then declare -a parent_info=('GMMIP'                        'hist-resIPO'                  'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'initialized from "historical" run year 1870 and SSTs in tropical lobe of the IPO domain (20degS-20degN, 175degE-75degW) restored to AMIP SSTs with historical forcings'                                 'none'); fi
 if [ "$mip" = 'HighResMIP'                   ] && [ "$experiment" = 'control-1950'                 ]; then declare -a parent_info=('HighResMIP'                   'control-1950'                 'HighResMIP'                   'spinup-1950'                  'AOGCM'                        'AER'                          'coupled control with fixed 1950s forcing (HighResMIP equivalent of pre-industrial control)'                                                                                                             'none'); fi
 if [ "$mip" = 'HighResMIP'                   ] && [ "$experiment" = 'highres-future'               ]; then declare -a parent_info=('HighResMIP'                   'highres-future'               'HighResMIP'                   'hist-1950'                    'AOGCM'                        'AER'                          'coupled future 2015-2050 using a scenario as close to CMIP5 RCP8.5 as possible within CMIP6'                                                                                                            'none'); fi
 if [ "$mip" = 'HighResMIP'                   ] && [ "$experiment" = 'highresSST-4xCO2'             ]; then declare -a parent_info=('HighResMIP'                   'highresSST-4xCO2'             'HighResMIP'                   'highresSST-present'           'AGCM'                         'AER'                          'highresSST-present SST with 4xCO2 concentrations'                                                                                                                                                       'none'); fi
 if [ "$mip" = 'HighResMIP'                   ] && [ "$experiment" = 'highresSST-future'            ]; then declare -a parent_info=('HighResMIP'                   'highresSST-future'            'HighResMIP'                   'highresSST-present'           'AGCM'                         'AER'                          'forced atmosphere experiment for 2015-2050 using SST/sea-ice derived from CMIP5 RCP8.5 simulations and a scenario as close to RCP8.5 as possible within CMIP6'                                          'none'); fi
 if [ "$mip" = 'HighResMIP'                   ] && [ "$experiment" = 'highresSST-LAI'               ]; then declare -a parent_info=('HighResMIP'                   'highresSST-LAI'               'HighResMIP'                   'highresSST-present'           'AGCM'                         'AER'                          'common LAI dataset within the highresSST-present experiment'                                                                                                                                            'none'); fi
 if [ "$mip" = 'HighResMIP'                   ] && [ "$experiment" = 'highresSST-p4K'               ]; then declare -a parent_info=('HighResMIP'                   'highresSST-p4K'               'HighResMIP'                   'highresSST-present'           'AGCM'                         'AER'                          'uniform 4K warming of highresSST-present SST'                                                                                                                                                           'none'); fi
 if [ "$mip" = 'HighResMIP'                   ] && [ "$experiment" = 'highresSST-present'           ]; then declare -a parent_info=('HighResMIP'                   'highresSST-present'           'no parent'                    'no parent'                    'AGCM'                         'AER'                          'forced atmosphere experiment for 1950-2014'                                                                                                                                                             'none'); fi
 if [ "$mip" = 'HighResMIP'                   ] && [ "$experiment" = 'highresSST-smoothed'          ]; then declare -a parent_info=('HighResMIP'                   'highresSST-smoothed'          'HighResMIP'                   'highresSST-present'           'AGCM'                         'AER'                          'smoothed SST version of highresSST-present'                                                                                                                                                             'none'); fi
 if [ "$mip" = 'HighResMIP'                   ] && [ "$experiment" = 'hist-1950'                    ]; then declare -a parent_info=('HighResMIP'                   'hist-1950'                    'HighResMIP'                   'spinup-1950'                  'AOGCM'                        'AER'                          'coupled historical 1950-2014'                                                                                                                                                                           'none'); fi
 if [ "$mip" = 'HighResMIP'                   ] && [ "$experiment" = 'spinup-1950'                  ]; then declare -a parent_info=('HighResMIP'                   'spinup-1950'                  'no parent'                    'no parent'                    'AOGCM'                        'AER'                          'coupled spinup with fixed 1950s forcings from 1950 initial conditions (with ocean at rest) to provide initial condition for control-1950 and hist-1950'                                                 'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = '1pctCO2-4xext'                ]; then declare -a parent_info=('ISMIP6'                       '1pctCO2-4xext'                'CMIP'                         '1pctCO2'                      'AOGCM'                        'AER CHEM BGC'                 'extension from year 140 of 1pctCO2 with 4xCO2'                                                                                                                                                          'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = '1pctCO2to4x-withism'          ]; then declare -a parent_info=('ISMIP6'                       '1pctCO2to4x-withism'          'ISMIP6'                       'piControl-withism'            'AOGCM'                        'AER CHEM BGC'                 'simulation with interactive ice sheet forced by 1 percent per year increase in CO2 to 4xCO2 (subsequently held fixed)'                                                                                  'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'historical-withism'           ]; then declare -a parent_info=('ISMIP6'                       'historical-withism'           'ISMIP6'                       'piControl-withism'            'AOGCM'                        'AER CHEM BGC'                 'historical with interactive ice sheet'                                                                                                                                                                  'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-1pctCO2to4x-self'         ]; then declare -a parent_info=('ISMIP6'                       'ism-1pctCO2to4x-self'         'ISMIP6'                       'ism-piControl-self'           'ISM'                          ''                             'offline ice sheet model forced by ISMs own AOGCM 1pctCO2to4x output'                                                                                                                                    'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-1pctCO2to4x-std'          ]; then declare -a parent_info=('ISMIP6'                       'ism-1pctCO2to4x-std'          'ISMIP6'                       'ism-pdControl-std'            'ISM'                          ''                             'offline ice sheet model forced by ISMIP6-specified AOGCM 1pctCO2to4x output'                                                                                                                            'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-amip-std'                 ]; then declare -a parent_info=('ISMIP6'                       'ism-amip-std'                 'ISMIP6'                       'ism-ctrl-std'                 'ISM'                          ''                             'offline ice sheet forced by ISMIP6-specified AGCM AMIP output'                                                                                                                                          'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-asmb-std'                 ]; then declare -a parent_info=('ISMIP6'                       'ism-asmb-std'                 'ISMIP6'                       'ism-ctrl-std'                 'ISM'                          ''                             'offline ice sheet forced by initMIP synthetic atmospheric experiment'                                                                                                                                   'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-bsmb-std'                 ]; then declare -a parent_info=('ISMIP6'                       'ism-bsmb-std'                 'ISMIP6'                       'ism-ctrl-std'                 'ISM'                          ''                             'offline ice sheet forced by initMIP synthetic oceanic experiment'                                                                                                                                       'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-ctrl-std'                 ]; then declare -a parent_info=('ISMIP6'                       'ism-ctrl-std'                 'no parent'                    'no parent'                    'ISM'                          ''                             'offline ice sheet model initMIP control'                                                                                                                                                                'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-historical-self'          ]; then declare -a parent_info=('ISMIP6'                       'ism-historical-self'          'ISMIP6'                       'ism-piControl-self'           'ISM'                          ''                             'offline ice sheet forced by ISMs own AOGCM historical output'                                                                                                                                           'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-historical-std'           ]; then declare -a parent_info=('ISMIP6'                       'ism-historical-std'           'ISMIP6'                       'ism-pdControl-std'            'ISM'                          ''                             'offline ice sheet forced by ISMIP6-specified AOGCM historical output'                                                                                                                                   'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-lig127k-std'              ]; then declare -a parent_info=('ISMIP6'                       'ism-lig127k-std'              'no parent'                    'no parent'                    'ISM'                          ''                             'offline ice sheet forced by ISMIP6-specified AGCM last interglacial output'                                                                                                                             'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-pdControl-std'            ]; then declare -a parent_info=('ISMIP6'                       'ism-pdControl-std'            'no parent'                    'no parent'                    'ISM'                          ''                             'offline ice sheet forced by ISMIP6-specified AOGCM pdControl output'                                                                                                                                    'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-piControl-self'           ]; then declare -a parent_info=('ISMIP6'                       'ism-piControl-self'           'no parent'                    'no parent'                    'ISM'                          ''                             'offline ice sheet forced by ISMs own AOGCM piControl output'                                                                                                                                            'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-ssp585-self'              ]; then declare -a parent_info=('ISMIP6'                       'ism-ssp585-self'              'ISMIP6'                       'ism-historical-self'          'ISM'                          ''                             'offline ice sheet forced by ISMs own AOGCM ssp585 output'                                                                                                                                               'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ism-ssp585-std'               ]; then declare -a parent_info=('ISMIP6'                       'ism-ssp585-std'               'ISMIP6'                       'ism-historical-std'           'ISM'                          ''                             'offline ice sheet forced by ISMIP6-specified AOGCM ssp585 output'                                                                                                                                       'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'piControl-withism'            ]; then declare -a parent_info=('ISMIP6'                       'piControl-withism'            'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'preindustrial control with interactive ice sheet'                                                                                                                                                       'none'); fi
 if [ "$mip" = 'ISMIP6'                       ] && [ "$experiment" = 'ssp585-withism'               ]; then declare -a parent_info=('ISMIP6'                       'ssp585-withism'               'ISMIP6'                       'historical-withism'           'AOGCM'                        'AER CHEM BGC'                 'ssp585 with interactive ice sheet'                                                                                                                                                                      'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'amip-lfmip-pdLC'              ]; then declare -a parent_info=('LS3MIP'                       'amip-lfmip-pdLC'              'no parent'                    'no parent'                    'AGCM'                         'AER CHEM BGC'                 'prescribed modern land surface climatology from historical, prescribed SST and sea-ice from historical plus scenario runs'                                                                              'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'amip-lfmip-pObs'              ]; then declare -a parent_info=('LS3MIP'                       'amip-lfmip-pObs'              'no parent'                    'no parent'                    'AGCM'                         'AER CHEM BGC'                 'prescribed land (from pseudo-observations) and AMIP SSTs'                                                                                                                                               'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'amip-lfmip-rmLC'              ]; then declare -a parent_info=('LS3MIP'                       'amip-lfmip-rmLC'              'no parent'                    'no parent'                    'AGCM'                         'AER CHEM BGC'                 'prescribed land surface climatology from historical plus scenario 30yr running mean, prescribed SST and sea-ice from historical plus scenario runs'                                                     'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'land-hist-cruNcep'            ]; then declare -a parent_info=('LS3MIP'                       'land-hist-cruNcep'            'no parent'                    'no parent'                    'LAND'                         'BGC'                          'as land-hist with CRU-NCEP forcings'                                                                                                                                                                    'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'land-hist-princeton'          ]; then declare -a parent_info=('LS3MIP'                       'land-hist-princeton'          'no parent'                    'no parent'                    'LAND'                         'BGC'                          'as land-hist with Princeton forcings'                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'land-hist'                    ]; then declare -a parent_info=('LS3MIP'                       'land-hist'                    'no parent'                    'no parent'                    'LAND'                         'BGC'                          'historical land-only'                                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'land-hist-wfdei'              ]; then declare -a parent_info=('LS3MIP'                       'land-hist-wfdei'              'no parent'                    'no parent'                    'LAND'                         'BGC'                          'as land-hist with WFDEI forcings'                                                                                                                                                                       'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'land-ssp126'                  ]; then declare -a parent_info=('LS3MIP'                       'land-ssp126'                  'no parent'                    'no parent'                    'LAND'                         'BGC'                          'future ssp1-2.6 land only'                                                                                                                                                                              'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'land-ssp434'                  ]; then declare -a parent_info=('LS3MIP'                       'land-ssp434'                  'no parent'                    'no parent'                    'LAND'                         'BGC'                          'future ssp4-3.4 land only'                                                                                                                                                                              'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'land-ssp585'                  ]; then declare -a parent_info=('LS3MIP'                       'land-ssp585'                  'no parent'                    'no parent'                    'LAND'                         'BGC'                          'future ssp5-8.5 land only'                                                                                                                                                                              'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'lfmip-initLC'                 ]; then declare -a parent_info=('LS3MIP'                       'lfmip-initLC'                 'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'initialized from "historical" run year 1980, but with land conditions initialized from pseudo-observations'                                                                                             'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'lfmip-pdLC-cruNcep'           ]; then declare -a parent_info=('LS3MIP'                       'lfmip-pdLC-cruNcep'           'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'as LFMIP-pdLC with Land-Hist-cruNcep'                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'lfmip-pdLC-princeton'         ]; then declare -a parent_info=('LS3MIP'                       'lfmip-pdLC-princeton'         'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'as LFMIP-pdLC with Land-Hist-princeton'                                                                                                                                                                 'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'lfmip-pdLC'                   ]; then declare -a parent_info=('LS3MIP'                       'lfmip-pdLC'                   'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'prescribed land conditions (from current climate climatology) and initialized from "historical" run year 1980'                                                                                          'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'lfmip-pdLC-wfdei'             ]; then declare -a parent_info=('LS3MIP'                       'lfmip-pdLC-wfdei'             'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'as LFMIP-pdLC with Land-Hist-wfdei'                                                                                                                                                                     'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'lfmip-rmLC-cruNcep'           ]; then declare -a parent_info=('LS3MIP'                       'lfmip-rmLC-cruNcep'           'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'as LFMIP-rmLC with Land-Hist-cruNcep'                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'lfmip-rmLC-princeton'         ]; then declare -a parent_info=('LS3MIP'                       'lfmip-rmLC-princeton'         'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'as LFMIP-rmLC with Land-Hist-princeton'                                                                                                                                                                 'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'lfmip-rmLC'                   ]; then declare -a parent_info=('LS3MIP'                       'lfmip-rmLC'                   'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'prescribed land conditions (from running mean climatology) and initialized from "historical" run year 1980'                                                                                             'none'); fi
 if [ "$mip" = 'LS3MIP'                       ] && [ "$experiment" = 'lfmip-rmLC-wfdei'             ]; then declare -a parent_info=('LS3MIP'                       'lfmip-rmLC-wfdei'             'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'as LFMIP-rmLC with Land-Hist-wfdei'                                                                                                                                                                     'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'deforest-globe'               ]; then declare -a parent_info=('LUMIP'                        'deforest-globe'               'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'idealized transient global deforestation'                                                                                                                                                               'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'esm-ssp585-ssp126Lu'          ]; then declare -a parent_info=('LUMIP'                        'esm-ssp585-ssp126Lu'          'CMIP'                         'esm-hist'                     'AOGCM'                        'AER CHEM'                     'emissions-driven SSP5-8.5 with SSP1-2.6 land use'                                                                                                                                                       'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'hist-noLu'                    ]; then declare -a parent_info=('LUMIP'                        'hist-noLu'                    'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'historical with no land-use change'                                                                                                                                                                     'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-cClim'                   ]; then declare -a parent_info=('LUMIP'                        'land-cClim'                   'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only constant climate'                                                                                                                                                                  'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-cCO2'                    ]; then declare -a parent_info=('LUMIP'                        'land-cCO2'                    'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only constant CO2'                                                                                                                                                                      'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-crop-grass'              ]; then declare -a parent_info=('LUMIP'                        'land-crop-grass'              'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only with cropland as natural grassland'                                                                                                                                                'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-crop-noFert'             ]; then declare -a parent_info=('LUMIP'                        'land-crop-noFert'             'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only with no fertilizer'                                                                                                                                                                'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-crop-noIrrigFert'        ]; then declare -a parent_info=('LUMIP'                        'land-crop-noIrrigFert'        'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only with managed crops but with irrigation and fertilization held constant'                                                                                                            'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-crop-noIrrig'            ]; then declare -a parent_info=('LUMIP'                        'land-crop-noIrrig'            'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only with no irrigation'                                                                                                                                                                'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-hist-altLu1'             ]; then declare -a parent_info=('LUMIP'                        'land-hist-altLu1'             'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only alternate land-use history'                                                                                                                                                        'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-hist-altLu2'             ]; then declare -a parent_info=('LUMIP'                        'land-hist-altLu2'             'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only alternate land use history'                                                                                                                                                        'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-hist-altStartYear'       ]; then declare -a parent_info=('LUMIP'                        'land-hist-altStartYear'       'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only alternate start year'                                                                                                                                                              'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-hist'                    ]; then declare -a parent_info=('LUMIP'                        'land-hist'                    'no parent'                    'no parent'                    'LAND'                         'BGC'                          'historical land-only'                                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-noFire'                  ]; then declare -a parent_info=('LUMIP'                        'land-noFire'                  'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only with no human fire management'                                                                                                                                                     'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-noLu'                    ]; then declare -a parent_info=('LUMIP'                        'land-noLu'                    'no parent'                    'no parent'                    'LAND'                         'BGC'                          'historical land-only with no land-use change'                                                                                                                                                           'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-noPasture'               ]; then declare -a parent_info=('LUMIP'                        'land-noPasture'               'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only with constant pastureland'                                                                                                                                                         'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-noShiftCultivate'        ]; then declare -a parent_info=('LUMIP'                        'land-noShiftCultivate'        'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only with shifting cultivation turned off'                                                                                                                                              'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'land-noWoodHarv'              ]; then declare -a parent_info=('LUMIP'                        'land-noWoodHarv'              'no parent'                    'no parent'                    'LAND'                         ''                             'historical land-only with no wood harvest'                                                                                                                                                              'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'ssp126-ssp370Lu'              ]; then declare -a parent_info=('LUMIP'                        'ssp126-ssp370Lu'              'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'SSP1-2.6 with SSP3-7.0 land use'                                                                                                                                                                        'none'); fi
 if [ "$mip" = 'LUMIP'                        ] && [ "$experiment" = 'ssp370-ssp126Lu'              ]; then declare -a parent_info=('LUMIP'                        'ssp370-ssp126Lu'              'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'SSP3-7.0 with SSP1-2.6 land use'                                                                                                                                                                        'none'); fi
 if [ "$mip" = 'OMIP'                         ] && [ "$experiment" = 'omip1-spunup'                 ]; then declare -a parent_info=('OMIP'                         'omip1-spunup'                 'no parent'                    'no parent'                    'OGCM'                         ''                             'OMIP experiment forced by Large and Yeager (CORE-2, NCEP) atmospheric data set and initialized from at least a 2000-year spin up of the coupled physical-biogeochemical model'                          'none'); fi
 if [ "$mip" = 'OMIP'                         ] && [ "$experiment" = 'omip1'                        ]; then declare -a parent_info=('OMIP'                         'omip1'                        'no parent'                    'no parent'                    'OGCM'                         'BGC'                          'OMIP experiment forced by Large and Yeager (CORE-2, NCEP) atmospheric data set and initialized with observed physical and biogeochemical ocean data'                                                    'none'); fi
 if [ "$mip" = 'OMIP'                         ] && [ "$experiment" = 'omip2-spunup'                 ]; then declare -a parent_info=('OMIP'                         'omip2-spunup'                 'no parent'                    'no parent'                    'OGCM'                         ''                             'OMIP experiment forced by JRA55-do atmospheric data set and initialized from at least a 2000-year spin up of the coupled physical-biogeochemical model'                                                 'none'); fi
 if [ "$mip" = 'OMIP'                         ] && [ "$experiment" = 'omip2'                        ]; then declare -a parent_info=('OMIP'                         'omip2'                        'no parent'                    'no parent'                    'OGCM'                         'BGC'                          'OMIP experiment forced by JRA55-do atmospheric data set and initialized with observed physical and biogeochemical ocean data'                                                                           'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'amip-climSIC'                 ]; then declare -a parent_info=('PAMIP'                        'amip-climSIC'                 'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'AMIP with climatological SIC'                                                                                                                                                                           'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'amip-climSST'                 ]; then declare -a parent_info=('PAMIP'                        'amip-climSST'                 'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'AMIP with climatological SST'                                                                                                                                                                           'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'futSST-pdSIC'                 ]; then declare -a parent_info=('PAMIP'                        'futSST-pdSIC'                 'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with future SST and present day SIC'                                                                                                                                              'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'modelSST-futArcSIC'           ]; then declare -a parent_info=('PAMIP'                        'modelSST-futArcSIC'           'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with present day coupled model SST and future Arctic SIC'                                                                                                                         'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'modelSST-pdSIC'               ]; then declare -a parent_info=('PAMIP'                        'modelSST-pdSIC'               'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice present day control with coupled model SST'                                                                                                                                       'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pa-futAntSIC-ext'             ]; then declare -a parent_info=('PAMIP'                        'pa-futAntSIC-ext'             'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'Partially-coupled extended simulation with future Antarctic SIC'                                                                                                                                        'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pa-futAntSIC'                 ]; then declare -a parent_info=('PAMIP'                        'pa-futAntSIC'                 'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'Partially-coupled time slice constrained by future Antarctic SIC'                                                                                                                                       'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pa-futArcSIC-ext'             ]; then declare -a parent_info=('PAMIP'                        'pa-futArcSIC-ext'             'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'Partially-coupled extended simulation with future Arctic SIC'                                                                                                                                           'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pa-futArcSIC'                 ]; then declare -a parent_info=('PAMIP'                        'pa-futArcSIC'                 'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'Partially-coupled time slice constrained by future Arctic SIC'                                                                                                                                          'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pa-pdSIC-ext'                 ]; then declare -a parent_info=('PAMIP'                        'pa-pdSIC-ext'                 'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'Partially-coupled extended simulation constrained by present day SIC'                                                                                                                                   'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pa-pdSIC'                     ]; then declare -a parent_info=('PAMIP'                        'pa-pdSIC'                     'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'Partially-coupled time slice contrained by present day SIC'                                                                                                                                             'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pa-piAntSIC'                  ]; then declare -a parent_info=('PAMIP'                        'pa-piAntSIC'                  'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'Partially-coupled time slice with pre-industrial Antarctic SIC'                                                                                                                                         'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pa-piArcSIC'                  ]; then declare -a parent_info=('PAMIP'                        'pa-piArcSIC'                  'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'Partially-coupled time slice constrained by pre-industrial Arctic SIC'                                                                                                                                  'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pdSST-futAntSIC'              ]; then declare -a parent_info=('PAMIP'                        'pdSST-futAntSIC'              'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with present day SST and future Antarctic SIC'                                                                                                                                    'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pdSST-futArcSICSIT'           ]; then declare -a parent_info=('PAMIP'                        'pdSST-futArcSICSIT'           'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with present day SST and future Arctic SIC and sea ice thickness'                                                                                                                 'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pdSST-futArcSIC'              ]; then declare -a parent_info=('PAMIP'                        'pdSST-futArcSIC'              'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with present day SST and future Arctic SIC'                                                                                                                                       'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pdSST-futBKSeasSIC'           ]; then declare -a parent_info=('PAMIP'                        'pdSST-futBKSeasSIC'           'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with present day SST and future Barents and Kara Seas SIC'                                                                                                                        'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pdSST-futOkhotskSIC'          ]; then declare -a parent_info=('PAMIP'                        'pdSST-futOkhotskSIC'          'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with present day SST and future Sea of Okhotsk SIC'                                                                                                                               'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pdSST-pdSICSIT'               ]; then declare -a parent_info=('PAMIP'                        'pdSST-pdSICSIT'               'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice constrained by present day conditions including sea ice thickness'                                                                                                                'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pdSST-pdSIC'                  ]; then declare -a parent_info=('PAMIP'                        'pdSST-pdSIC'                  'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with present day SST and SIC'                                                                                                                                                     'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pdSST-piAntSIC'               ]; then declare -a parent_info=('PAMIP'                        'pdSST-piAntSIC'               'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with present day SST and pre-industrial Antarctic SIC'                                                                                                                            'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'pdSST-piArcSIC'               ]; then declare -a parent_info=('PAMIP'                        'pdSST-piArcSIC'               'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with present day SST and pre-industrial Arctic SIC'                                                                                                                               'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'piSST-pdSIC'                  ]; then declare -a parent_info=('PAMIP'                        'piSST-pdSIC'                  'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with pre-industrial SST and present day SIC'                                                                                                                                      'none'); fi
 if [ "$mip" = 'PAMIP'                        ] && [ "$experiment" = 'piSST-piSIC'                  ]; then declare -a parent_info=('PAMIP'                        'piSST-piSIC'                  'CMIP'                         'amip'                         'AGCM'                         'AER CHEM BGC'                 'Atmosphere time slice with pre-industrial SST and SIC'                                                                                                                                                  'none'); fi
 if [ "$mip" = 'PMIP'                         ] && [ "$experiment" = 'esm-past1000'                 ]; then declare -a parent_info=('PMIP'                         'esm-past1000'                 'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM'                     'last millennium experiment with interactive carbon cycle'                                                                                                                                               'none'); fi
 if [ "$mip" = 'PMIP'                         ] && [ "$experiment" = 'lgm'                          ]; then declare -a parent_info=('PMIP'                         'lgm'                          'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'last glacial maximum'                                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'PMIP'                         ] && [ "$experiment" = 'lig127k'                      ]; then declare -a parent_info=('PMIP'                         'lig127k'                      'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'last interglacial (127k)'                                                                                                                                                                               'none'); fi
 if [ "$mip" = 'PMIP'                         ] && [ "$experiment" = 'midHolocene'                  ]; then declare -a parent_info=('PMIP'                         'midHolocene'                  'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'mid-Holocene'                                                                                                                                                                                           'none'); fi
 if [ "$mip" = 'PMIP'                         ] && [ "$experiment" = 'midPliocene-eoi400'           ]; then declare -a parent_info=('PMIP'                         'midPliocene-eoi400'           'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'mid-Pliocene warm period'                                                                                                                                                                               'none'); fi
 if [ "$mip" = 'PMIP'                         ] && [ "$experiment" = 'past1000-solaronly'           ]; then declare -a parent_info=('PMIP'                         'past1000-solaronly'           'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'last millennium experiment using only solar forcing'                                                                                                                                                    'none'); fi
 if [ "$mip" = 'PMIP'                         ] && [ "$experiment" = 'past1000'                     ]; then declare -a parent_info=('PMIP'                         'past1000'                     'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'last millennium'                                                                                                                                                                                        'none'); fi
 if [ "$mip" = 'PMIP'                         ] && [ "$experiment" = 'past1000-volconly'            ]; then declare -a parent_info=('PMIP'                         'past1000-volconly'            'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'last millennium experiment using only volcanic forcing'                                                                                                                                                 'none'); fi
 if [ "$mip" = 'PMIP'                         ] && [ "$experiment" = 'past2k'                       ]; then declare -a parent_info=('PMIP'                         'past2k'                       'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 'last two millennia experiment'                                                                                                                                                                          'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'hist-spAer-aer'               ]; then declare -a parent_info=('RFMIP'                        'hist-spAer-aer'               'CMIP'                         'piControl'                    'AOGCM'                        'CHEM BGC'                     'historical simulation with specified anthropogenic aerosols, no other forcings'                                                                                                                         'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'hist-spAer-all'               ]; then declare -a parent_info=('RFMIP'                        'hist-spAer-all'               'CMIP'                         'piControl'                    'AOGCM'                        'CHEM BGC'                     'historical simulation with specified anthropogenic aerosols'                                                                                                                                            'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-4xCO2'                 ]; then declare -a parent_info=('RFMIP'                        'piClim-4xCO2'                 'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM'                     'effective radiative forcing by 4xCO2'                                                                                                                                                                   'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-aer'                   ]; then declare -a parent_info=('RFMIP'                        'piClim-aer'                   'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'effective radiative forcing by present-day aerosols'                                                                                                                                                    'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-anthro'                ]; then declare -a parent_info=('RFMIP'                        'piClim-anthro'                'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM'                     'effective radiative forcing by present day anthropogenic agents'                                                                                                                                        'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-control'               ]; then declare -a parent_info=('RFMIP'                        'piClim-control'               'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'Control simulation providing baseline for evaluating effective radiative forcing (ERF)'                                                                                                                 'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-ghg'                   ]; then declare -a parent_info=('RFMIP'                        'piClim-ghg'                   'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'effective radiative forcing by present-day greenhouse gases'                                                                                                                                            'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-histaer'               ]; then declare -a parent_info=('RFMIP'                        'piClim-histaer'               'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'transient effective radiative forcing by aerosols'                                                                                                                                                      'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-histall'               ]; then declare -a parent_info=('RFMIP'                        'piClim-histall'               'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'transient effective radiative forcing'                                                                                                                                                                  'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-histghg'               ]; then declare -a parent_info=('RFMIP'                        'piClim-histghg'               'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'transient effective radiative forcing by greenhouse gases'                                                                                                                                              'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-histnat'               ]; then declare -a parent_info=('RFMIP'                        'piClim-histnat'               'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'transient effective radiative forcing by natural perturbations'                                                                                                                                         'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-lu'                    ]; then declare -a parent_info=('RFMIP'                        'piClim-lu'                    'CMIP'                         'piControl'                    'AGCM'                         'AER CHEM BGC'                 'effective radiative forcing by present-day land use'                                                                                                                                                    'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-spAer-aer'             ]; then declare -a parent_info=('RFMIP'                        'piClim-spAer-aer'             'CMIP'                         'piControl'                    'AGCM'                         ''                             'effective radiative forcing at present day with specified anthropogenic aerosol optical properties, all forcings'                                                                                       'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-spAer-anthro'          ]; then declare -a parent_info=('RFMIP'                        'piClim-spAer-anthro'          'CMIP'                         'piControl'                    'AGCM'                         ''                             'effective radiative forcing at present day with specified anthropogenic aerosol optical properties, anthropogenic forcings'                                                                             'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-spAer-histaer'         ]; then declare -a parent_info=('RFMIP'                        'piClim-spAer-histaer'         'CMIP'                         'piControl'                    'AGCM'                         ''                             'transient effective radiative forcing with specified anthropogenic aerosol optical properties, aerosol forcing'                                                                                         'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'piClim-spAer-histall'         ]; then declare -a parent_info=('RFMIP'                        'piClim-spAer-histall'         'CMIP'                         'piControl'                    'AGCM'                         ''                             'transient effective radiative forcing with specified anthropogenic aerosol optical properties, all forcings'                                                                                            'none'); fi
 if [ "$mip" = 'RFMIP'                        ] && [ "$experiment" = 'rad-irf'                      ]; then declare -a parent_info=('RFMIP'                        'rad-irf'                      'no parent'                    'no parent'                    'RAD'                          ''                             'offline assessment of radiative transfer parmeterizations in clear skies'                                                                                                                               'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'rcp26-cmip5'                  ]; then declare -a parent_info=('ScenarioMIP'                  'rcp26-cmip5'                  'CMIP'                         'historical-cmip5'             'AOGCM'                        'AER CHEM BGC'                 'future projection based on CMIP5-era RCP2.6 scenario (CMIP5-era [2006-2100] forcing)'                                                                                                                   'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'rcp45-cmip5'                  ]; then declare -a parent_info=('ScenarioMIP'                  'rcp45-cmip5'                  'CMIP'                         'historical-cmip5'             'AOGCM'                        'AER CHEM BGC'                 'future projection based on CMIP5-era RCP4.5 scenario (CMIP5-era [2006-2100] forcing)'                                                                                                                   'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'rcp60-cmip5'                  ]; then declare -a parent_info=('ScenarioMIP'                  'rcp60-cmip5'                  'CMIP'                         'historical-cmip5'             'AOGCM'                        'AER CHEM BGC'                 'future projection based on CMIP5-era RCP6.0 scenario (CMIP5-era [2006-2100] forcing)'                                                                                                                   'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'rcp85-cmip5'                  ]; then declare -a parent_info=('ScenarioMIP'                  'rcp85-cmip5'                  'CMIP'                         'historical-cmip5'             'AOGCM'                        'AER CHEM BGC'                 'future projection based on CMIP5-era RCP8.5 scenario (CMIP5-era [2006-2100] forcing)'                                                                                                                   'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'ssp119'                       ]; then declare -a parent_info=('ScenarioMIP'                  'ssp119'                       'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'low-end scenario reaching 1.9 W m-2, based on SSP1'                                                                                                                                                     'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'ssp126'                       ]; then declare -a parent_info=('ScenarioMIP'                  'ssp126'                       'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'update of RCP2.6 based on SSP1'                                                                                                                                                                         'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'ssp245'                       ]; then declare -a parent_info=('ScenarioMIP'                  'ssp245'                       'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'update of RCP4.5 based on SSP2'                                                                                                                                                                         'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'ssp370'                       ]; then declare -a parent_info=('ScenarioMIP'                  'ssp370'                       'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'gap-filling scenario reaching 7.0 based on SSP3'                                                                                                                                                        'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'ssp434'                       ]; then declare -a parent_info=('ScenarioMIP'                  'ssp434'                       'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'gap-filling scenario reaching 3.4 based on SSP4'                                                                                                                                                        'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'ssp460'                       ]; then declare -a parent_info=('ScenarioMIP'                  'ssp460'                       'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'update of RCP6.0 based on SSP4'                                                                                                                                                                         'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'ssp534-over'                  ]; then declare -a parent_info=('ScenarioMIP'                  'ssp534-over'                  'ScenarioMIP'                  'ssp585'                       'AOGCM'                        'AER CHEM BGC'                 'overshoot of 3.4 W/m**2 branching from ssp585 in 2040'                                                                                                                                                  'none'); fi
 if [ "$mip" = 'ScenarioMIP'                  ] && [ "$experiment" = 'ssp585'                       ]; then declare -a parent_info=('ScenarioMIP'                  'ssp585'                       'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'update of RCP8.5 based on SSP5'                                                                                                                                                                         'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'control-slab'                 ]; then declare -a parent_info=('VolMIP'                       'control-slab'                 'no parent'                    'no parent'                    'AGCM'                         'AER CHEM BGC'                 'control with slab ocean'                                                                                                                                                                                'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'dcppC-forecast-addPinatubo'   ]; then declare -a parent_info=('VolMIP'                       'dcppC-forecast-addPinatubo'   'no parent'                    'no parent'                    'AOGCM'                        'AER CHEM BGC'                 '2015 forecast with added Pinatubo forcing'                                                                                                                                                              's2014'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'volc-cluster-21C'             ]; then declare -a parent_info=('VolMIP'                       'volc-cluster-21C'             'CMIP'                         'historical'                   'AOGCM'                        'AER CHEM BGC'                 'volcanic cluster experiment under 21st century SSP2-4.5 scenario'                                                                                                                                       'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'volc-cluster-ctrl'            ]; then declare -a parent_info=('VolMIP'                       'volc-cluster-ctrl'            'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 '19th century volcanic cluster initialized from PiControl'                                                                                                                                               'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'volc-cluster-mill'            ]; then declare -a parent_info=('VolMIP'                       'volc-cluster-mill'            'PMIP'                         'past1000'                     'AOGCM'                        'AER CHEM BGC'                 '19th century volcanic cluster initialized from past1000'                                                                                                                                                'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'volc-long-eq'                 ]; then declare -a parent_info=('VolMIP'                       'volc-long-eq'                 'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'idealized equatorial volcanic eruption emitting 56.2 Tg SO2'                                                                                                                                            'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'volc-long-hlN'                ]; then declare -a parent_info=('VolMIP'                       'volc-long-hlN'                'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'idealized Northern Hemisphere high-latitude eruption emitting 28.1 Tg of SO2'                                                                                                                           'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'volc-long-hlS'                ]; then declare -a parent_info=('VolMIP'                       'volc-long-hlS'                'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'Idealized Southern Hemisphere high-latitude eruption emitting 28.1 Tg of SO2'                                                                                                                           'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'volc-pinatubo-full'           ]; then declare -a parent_info=('VolMIP'                       'volc-pinatubo-full'           'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'Pinatubo experiment'                                                                                                                                                                                    'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'volc-pinatubo-slab'           ]; then declare -a parent_info=('VolMIP'                       'volc-pinatubo-slab'           'VolMIP'                       'control-slab'                 'AGCM'                         'AER CHEM BGC'                 'Pinatubo experiment with slab ocean'                                                                                                                                                                    'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'volc-pinatubo-strat'          ]; then declare -a parent_info=('VolMIP'                       'volc-pinatubo-strat'          'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'Pinatubo experiment with partial radiative forcing, includes only stratospheric warming'                                                                                                                'none'); fi
 if [ "$mip" = 'VolMIP'                       ] && [ "$experiment" = 'volc-pinatubo-surf'           ]; then declare -a parent_info=('VolMIP'                       'volc-pinatubo-surf'           'CMIP'                         'piControl'                    'AOGCM'                        'AER CHEM BGC'                 'Pinatubo experiment with partial radiative forcing, solar radiation scattering only'                                                                                                                    'none'); fi

#echo ' Parent info: ' ${parent_info[2]} ${parent_info[3]}

 echo
 for i in "${model_components[@]}"
 do
    component="$i"

    if [ "$#" -eq 4 ]; then 
     echo ' Running ' $0 ${mip} ${experiment} ${ececonf} ${input_template} '  for ' ${component}
    else
     echo ' Running ' $0 ${mip} ${experiment} ${ececonf}                   '  for ' ${component}
    fi

    # Add your NEWCOMPONENT to this check if you want to extend ece2cmor3 for more model components.
    if [ "${component}" != 'ifs' ] && [ "${component}" != 'nemo' ] && [ "${component}" != 'tm5' ] && [ "${component}" != 'lpjg' ]; then
     echo ' Error in ' $0 ': unkown ec-earth component: '  ${component} '  Valid options: ifs, nemo, tm5, or lpjg'
     exit
    fi

    # Add your NEWCOMPONENT to this if-block if you want to extend ece2cmor3 for more model components.
    if [ "${component}" = 'ifs' ]; then
     grid_label='gr'
     res_index=1
    elif [ "${component}" = 'nemo' ]; then
     grid_label='gn'
     res_index=2
    elif [ "${component}" = 'tm5' ]; then
     grid_label='gn'
     res_index=3
    elif [ "${component}" = 'lpjg' ]; then
     grid_label='gr'
     res_index=4
    fi

    output_template=metadata-cmip6-${mip}-${experiment}-${ececonf}-${component}-template.json

    # Creating and adjusting with sed the output meta data template json file:
    sed    's/"activity_id":                  "CMIP"/"activity_id":                  "'${mip}'"/' ${input_template} >            ${output_template}
    sed -i 's/"experiment_id":                "piControl"/"experiment_id":                "'${experiment}'"/'                    ${output_template}
    sed -i 's/"source_id":                    "EC-Earth3"/"source_id":                    "'${ece_res[0]}'"/'                    ${output_template}
    sed -i 's/"source":                       "EC-Earth3 (2019)"/"source":                       "'${ece_res[0]}' (2019)"/'      ${output_template}  # The 2019 is correct as long no P verison from 2017 is taken.
    sed -i 's/"source_type":                  "AOGCM"/"source_type":                  "'"${ece_res[${source_type_index}]}"'"/'   ${output_template}  # Note the double quote for the spaces in the variable
    sed -i 's/"grid_label":                   "gr"/"grid_label":                   "'${grid_label}'"/'                           ${output_template}
    sed -i 's/"grid":                         "T255L91"/"grid":                         "'"${ece_res[${res_index}]}"'"/'         ${output_template}
    sed -i 's/"nominal_resolution":           "100 km"/"nominal_resolution":           "'"${nom_res[${res_index}]}"'"/'          ${output_template}
    if [ "${ececonf}" = 'EC-EARTH-AOGCM' ] && [ "${mip}" = 'CMIP' ] && [ "${experiment}" = 'piControl' ]; then
     sed -i 's/"parent_source_id":             "EC-Earth3"/"parent_source_id":             "'${ece_res[0]}'-Veg"/'               ${output_template}
    else
     sed -i 's/"parent_source_id":             "EC-Earth3-Veg"/"parent_source_id":             "'${ece_res[0]}'"/'               ${output_template}
    fi
    sed -i 's/"parent_activity_id":           "CMIP"/"parent_activity_id":           "'"${parent_info[2]}"'"/'                   ${output_template}
    sed -i 's/"parent_experiment_id":         "piControl-spinup"/"parent_experiment_id":         "'"${parent_info[3]}"'"/'       ${output_template}

 done
 echo

else
 echo
 echo "  This scripts requires three or four arguments, e.g.:"
 echo "   $0 CMIP historical EC-EARTH-Veg"
 echo "   $0 CMIP historical EC-EARTH-CC  ../resources/metadata-templates/cmip6-CMIP-piControl-metadata-template.json"
 echo
fi



#       "source_type":{
#           "AER":"aerosol treatment in an atmospheric model where concentrations are calculated based on emissions, transformation, and removal processes (rather than being prescribed or omitted entirely)",
#           "AGCM":"atmospheric general circulation model run with prescribed ocean surface conditions and usually a model of the land surface",
#           "AOGCM":"coupled atmosphere-ocean global climate model, additionally including explicit representation of at least the land and sea ice",
#           "BGC":"biogeochemistry model component that at the very least accounts for carbon reservoirs and fluxes in the atmosphere, terrestrial biosphere, and ocean",
#           "CHEM":"chemistry treatment in an atmospheric model that calculates atmospheric oxidant concentrations (including at least ozone), rather than prescribing them",
#           "ISM":"ice-sheet model that includes ice-flow",
#           "LAND":"land model run uncoupled from the atmosphere",
#           "OGCM":"ocean general circulation model run uncoupled from an AGCM but, usually including a sea-ice model",
#           "RAD":"radiation component of an atmospheric model run 'offline'",
#           "SLAB":"slab-ocean used with an AGCM in representing the atmosphere-ocean coupled system"
#       },
#       "grid_label":{
#           "gm":"global mean data",
#           "gn":"data reported on a model's native grid",
#           "gna":"data reported on a native grid in the region of Antarctica",
#           "gng":"data reported on a native grid in the region of Greenland",
#           "gnz":"zonal mean data reported on a model's native latitude grid",
#           "gr":"regridded data reported on the data provider's preferred target grid",
#           "gr1":"regridded data reported on a grid other than the native grid and other than the preferred target grid",
#           "gr1a":"regridded data reported in the region of Antarctica on a grid other than the native grid and other than the preferred target grid",
#           "gr1g":"regridded data reported in the region of Greenland on a grid other than the native grid and other than the preferred target grid",
#           "gr1z":"regridded zonal mean data reported on a grid other than the native latitude grid and other than the preferred latitude target grid",
#           "gr2":"regridded data reported on a grid other than the native grid and other than the preferred target grid",
#           "gr2a":"regridded data reported in the region of Antarctica on a grid other than the native grid and other than the preferred target grid",
#           "gr2g":"regridded data reported in the region of Greenland on a grid other than the native grid and other than the preferred target grid",
#           "gr2z":"regridded zonal mean data reported on a grid other than the native latitude grid and other than the preferred latitude target grid",
#           "gr3":"regridded data reported on a grid other than the native grid and other than the preferred target grid",
#           "gr3a":"regridded data reported in the region of Antarctica on a grid other than the native grid and other than the preferred target grid",
#           "gr3g":"regridded data reported in the region of Greenland on a grid other than the native grid and other than the preferred target grid",
#           "gr3z":"regridded zonal mean data reported on a grid other than the native latitude grid and other than the preferred latitude target grid",
#           "gr4":"regridded data reported on a grid other than the native grid and other than the preferred target grid",
#           "gr4a":"regridded data reported in the region of Antarctica on a grid other than the native grid and other than the preferred target grid",
#           "gr4g":"regridded data reported in the region of Greenland on a grid other than the native grid and other than the preferred target grid",
#           "gr4z":"regridded zonal mean data reported on a grid other than the native latitude grid and other than the preferred latitude target grid",
#           "gr5":"regridded data reported on a grid other than the native grid and other than the preferred target grid",
#           "gr5a":"regridded data reported in the region of Antarctica on a grid other than the native grid and other than the preferred target grid",
#           "gr5g":"regridded data reported in the region of Greenland on a grid other than the native grid and other than the preferred target grid",
#           "gr5z":"regridded zonal mean data reported on a grid other than the native latitude grid and other than the preferred latitude target grid",
#           "gr6":"regridded data reported on a grid other than the native grid and other than the preferred target grid",
#           "gr6a":"regridded data reported in the region of Antarctica on a grid other than the native grid and other than the preferred target grid",
#           "gr6g":"regridded data reported in the region of Greenland on a grid other than the native grid and other than the preferred target grid",
#           "gr6z":"regridded zonal mean data reported on a grid other than the native latitude grid and other than the preferred latitude target grid",
#           "gr7":"regridded data reported on a grid other than the native grid and other than the preferred target grid",
#           "gr7a":"regridded data reported in the region of Antarctica on a grid other than the native grid and other than the preferred target grid",
#           "gr7g":"regridded data reported in the region of Greenland on a grid other than the native grid and other than the preferred target grid",
#           "gr7z":"regridded zonal mean data reported on a grid other than the native latitude grid and other than the preferred latitude target grid",
#           "gr8":"regridded data reported on a grid other than the native grid and other than the preferred target grid",
#           "gr8a":"regridded data reported in the region of Antarctica on a grid other than the native grid and other than the preferred target grid",
#           "gr8g":"regridded data reported in the region of Greenland on a grid other than the native grid and other than the preferred target grid",
#           "gr8z":"regridded zonal mean data reported on a grid other than the native latitude grid and other than the preferred latitude target grid",
#           "gr9":"regridded data reported on a grid other than the native grid and other than the preferred target grid",
#           "gr9a":"regridded data reported in the region of Antarctica on a grid other than the native grid and other than the preferred target grid",
#           "gr9g":"regridded data reported in the region of Greenland on a grid other than the native grid and other than the preferred target grid",
#           "gr9z":"regridded zonal mean data reported on a grid other than the native latitude grid and other than the preferred latitude target grid",
#           "gra":"regridded data in the region of Antarctica reported on the data provider's preferred target grid",
#           "grg":"regridded data in the region of Greenland reported on the data provider's preferred target grid",
#           "grz":"regridded zonal mean data reported on the data provider's preferred latitude target grid"
#       },
#       "nominal_resolution":[
#           "0.5 km",
#           "1 km",
#           "10 km",
#           "100 km",
#           "1000 km",
#           "10000 km",
#           "1x1 degree",
#           "2.5 km",
#           "25 km",
#           "250 km",
#           "2500 km",
#           "5 km",
#           "50 km",
#           "500 km",
#           "5000 km"
