####I. Packages and functions ####
library(tidyverse)
library(padr)
# library(chron) ## installed, not loaded but used

Sys.getlocale()
Sys.setlocale("LC_TIME", "en_US.UTF-8")
Sys.timezone()
Timezone <- Sys.timezone()

# II. Importing Data -------------------------------------------------------
## datasets
#patient information
load(file = "./Unit1_Mothers.RData")
load(file = "./Unit2_Mothers.RData")
load(file = "./Unit2_Mothers_outpatient.RData")
load(file = "./Unit3_Mothers.RData")

#nurse information
load(file = "./Unit1_Nurses.RData")
load(file = "./Unit2_Nurses.RData")
load(file = "./Unit3_Nurses.RData")



# 1. Unit 1 (prenatal) ---------------------------------------------------------------
# 1.1 Patients ------------------------------------------------------------
unit1_NP_CF<- unit1_NP_CF %>%
  rename(shift_type_pat =shift_type) %>%
  mutate(cf_weight_u1 = round(cf_weight_u1, 4)) %>%
  mutate(weight_u1 = round(weight_u1, 4))


# 1.1.1 Hours per patient -------------------------------------------------
unit1_NP_CF_hours <- unit1_NP_CF %>%
  mutate(demand_hours = NumWomen_cat*15) %>% ## counting patients in total full
  mutate(demand_hours_weighted = NumWomen_cat*cf_weight_u1*15) %>% ## weighted data + patient to nurse ratio
  mutate(demand_hours_ptn = NumWomen_cat*weight_u1*15) ## only patient to nurse ratio

## summarise demand_hours and weighted demand_hours over all categories per time frame
## currently this is not in hours but in minutes.
unit1_NP_CF_hours_adj <- unit1_NP_CF_hours %>%
  group_by(date_worked, Time, Days, Weekday, Year, Month, shift_type_pat) %>%
  summarise(demand_hours_total = sum(demand_hours),
         demand_hours_total_weighted = sum(demand_hours_weighted),
         demand_hours_total_w_ptn = sum(demand_hours_ptn))
## 140,239 observations

## For the calculation of patient to nurse ratio, the number of patients is needed.
## The easiest way seems to calculate the total number of women from the summarised data

unit1_NP_CF_hours_adj <- unit1_NP_CF_hours_adj %>%
  mutate(NumPatient = demand_hours_total/15)

  
# 1.2 Nurses --------------------------------------------------------------
unit1_NN_RM <- unit1_NN %>%
  rename(shift_type_nurse = shift_type) %>%
  filter(nurse_group %in% "Hebammen") %>% ## Hebammen = Midwives
  select(!unit_id)
## There are 0 midwives for some time.
## This was confirmed with looking at the original PEP plan.

# 1.2.1. Hours per nurse --------------------------------------------------
unit1_NN_RM_hours <- unit1_NN_RM %>%
  mutate(supply_hours = NumNurse*15)
## 140,228 observations

# 1.3 Demand-supply match ----------------------------------------------
Pat_nurse_u1 <- left_join(unit1_NN_RM_hours, unit1_NP_CF_hours_adj, by = c("date_worked", "Time", "Days", "Weekday", "Year", "Month"), relationship = "many-to-many")
## 140,228 observations

Pat_nurse_u1 <- Pat_nurse_u1 %>%
  filter(!(is.na(demand_hours_total)))
## 140,212 observations

## Creating supply demand match
Pat_nurse_u1$sdm <- (Pat_nurse_u1$supply_hours)-(Pat_nurse_u1$demand_hours_total_weighted)
Pat_nurse_u1$sdm_unweighted <-  (Pat_nurse_u1$supply_hours)-(Pat_nurse_u1$demand_hours_total_w_ptn)
## positive number = oversupply
## negative number = undersupply

## Creating patient to nurse ratio
Pat_nurse_u1$PNR <- (Pat_nurse_u1$NumPatient)/(Pat_nurse_u1$NumNurse)

Pat_nurse_u1$PNR[Pat_nurse_u1$PNR %in% "NaN"] <- NA
Pat_nurse_u1$PNR[Pat_nurse_u1$PNR %in% "Inf"] <- NA


# save(Pat_nurse_u1, file= "./Pat_nurse_u1.RData")


# 1.4 Mean calculations ---------------------------------------------------

## patients
Patient_Weekdays_unit1 <- Pat_nurse_u1%>%
  # group_by(nurse_group, Days) %>%
  group_by(nurse_group, Days, Year) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_Pat = round(mean(NumPatient),1),
                STD_DEV_Pat = round(sd(NumPatient),1),
                Median_Pat = round(median(NumPatient),1),
                IQR_Pat = round(IQR(NumPatient),1),
                Minimum_Pat = round(min(NumPatient),1),
                Maximum_Pat = round(max(NumPatient),1),
                CI_Pat = qnorm(0.975)*(STD_DEV_Pat/sqrt(n())))
  ))

## nurses
Nurse_Weekdays_unit1 <- Pat_nurse_u1 %>%
  # group_by(nurse_group, Days) %>%
  group_by(nurse_group, Days, Year) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_Nur = round(mean(NumNurse),1),
                STD_DEV_Nur = round(sd(NumNurse),1),
                Median_Nur = round(median(NumNurse),1),
                IQR_Nur = round(IQR(NumNurse),1),
                Minimum_Nur = round(min(NumNurse),1),
                Maximum_Nur = round(max(NumNurse),1),
                CI_Nur = qnorm(0.975)*(STD_DEV_Nur/sqrt(n())))
  ))

## supply-demand match
SDM_Weekdays_unit1 <- Pat_nurse_u1 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_SDM = round(mean(sdm, na.rm = T),1),
                STD_DEV_SDM = round(sd(sdm, na.rm = T),1),
                Median_SDM = round(median(sdm, na.rm = T),1),
                IQR_SDM = round(IQR(sdm, na.rm = T),1),
                Minimum_SDM = round(min(sdm, na.rm = T),1),
                Maximum_SDM = round(max(sdm, na.rm = T),1),
                CI_SDM = qnorm(0.975)*(STD_DEV_SDM/sqrt(n())))
  ))

## supply-demand match unweighted
SDM_uw_Weekdays_unit1 <- Pat_nurse_u1 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_SDM_uw = round(mean(sdm_unweighted, na.rm = T),1),
                STD_DEV_SDM_uw = round(sd(sdm_unweighted, na.rm = T),1),
                Median_SDM_uw = round(median(sdm_unweighted, na.rm = T),1),
                IQR_SDM_uw = round(IQR(sdm_unweighted, na.rm = T),1),
                Minimum_SDM_uw = round(min(sdm_unweighted, na.rm = T),1),
                Maximum_SDM_uw = round(max(sdm_unweighted, na.rm = T),1),
                CI_SDM_uw = qnorm(0.975)*(STD_DEV_SDM_uw/sqrt(n())))
  ))

## nurse-to-patient ratio
PNR_Weekdays_unit1 <- Pat_nurse_u1 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  # group_by(date_worked, shift_type_nurse) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_PNR = round(mean(PNR, na.rm = T),1),
                STD_DEV_PNR = round(sd(PNR, na.rm = T),1),
                Median_PNR = round(median(PNR, na.rm = T),1),
                IQR_PNR = round(IQR(PNR, na.rm = T),1),
                Minimum_PNR = round(min(PNR, na.rm = T),1),
                Maximum_PNR = round(max(PNR, na.rm = T),1),
                CI_PNR = qnorm(0.975)*(STD_DEV_PNR/sqrt(n())))
  ))
# tapply(PNR_Weekdays_unit1$Median_PNR, PNR_Weekdays_unit1$shift_type_nurse, summary)

### creating data frame with all information together
Pat_Nurse_Weekdays_u1 <- left_join(Nurse_Weekdays_unit1, Patient_Weekdays_unit1, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_Weekdays_u1 <- left_join(Pat_Nurse_Weekdays_u1, SDM_Weekdays_unit1, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_PNR_Weekdays_u1 <- left_join(Pat_Nur_SDM_Weekdays_u1, PNR_Weekdays_unit1, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_PNR_Weekdays_u1_ext <- left_join(Pat_Nur_SDM_PNR_Weekdays_u1, SDM_uw_Weekdays_unit1, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_PNR_Weekdays_u1_ext$Time <- chron::chron(times = Pat_Nur_SDM_PNR_Weekdays_u1_ext$Time)


rm(Pat_Nurse_Weekdays_u1, Patient_Weekdays_unit1, Nurse_Weekdays_unit1, PNR_Weekdays_unit1, Pat_Nur_SDM_Weekdays_u1, SDM_Weekdays_unit1,
   unit1_NP_CF_hours, unit1_NN_RM)

# save(Pat_Nur_PNR_Weekdays_u1, file = "./Unit1_Weekdays.RData")

# 1.4.1.1 Mean calculations per shift -------------------------------------

## care demand unweighted
Demand_Shifts_unit1 <- Pat_nurse_u1 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_Demand = sum(demand_hours_total_w_ptn),
                Sum_Demand_hour = Sum_Demand/60,
                Sum_Demand_RM = Sum_Demand_hour/8)
  ))

tapply(Demand_Shifts_unit1$Sum_Demand_hour, Demand_Shifts_unit1$shift_type_nurse, summary)

## care demand weighted
Demand_Shifts_unit1_weighted <- Pat_nurse_u1 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_Demand = sum(demand_hours_total_weighted),
                Sum_Demand_hour = Sum_Demand/60,
                Sum_Demand_RM = Sum_Demand_hour/8)
  ))

tapply(Demand_Shifts_unit1_weighted$Sum_Demand_hour, Demand_Shifts_unit1_weighted$shift_type_nurse, summary)

## care supply
Supply_Shifts_unit1 <- Pat_nurse_u1 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_Supply = sum(supply_hours),
                Sum_Supply_hour = Sum_Supply/60,
                Sum_Suppply_RM = Sum_Supply_hour/8)
  ))

tapply(Supply_Shifts_unit1$Sum_Supply_hour, Supply_Shifts_unit1$shift_type_nurse, summary)


## supply-demand match per shift weighted (and unweighted)
SDM_Shifts_unit1 <- Pat_nurse_u1 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_HSD = sum(sdm_unweighted),
                Sum_HSD_hour = Sum_HSD/60,
                Sum_HSD_numMw = Sum_HSD_hour/8)
  ))

tapply(SDM_Shifts_unit1$Sum_HSD_hour, SDM_Shifts_unit1$shift_type_nurse, summary)

rm(Demand_Shifts_unit1_weighted, Demand_Shifts_unit1, Supply_Shifts_unit1)


# 1.4.1.2 Number of shifts with certain threshold --------------------------
SDM_Shifts_unit1_undersupply <- SDM_Shifts_unit1 %>%
  filter(!shift_type_nurse %in% "overlap") %>%
  filter(Sum_HSD_hour < (-8))
  # filter(Sum_HSD_hour > 8)

rm(SDM_Shifts_unit1_undersupply)

# 1.5 Plots ---------------------------------------------------------------
Pat_Nur_SDM_PNR_Weekdays_u1_ext$Days <- factor(Pat_Nur_SDM_PNR_Weekdays_u1_ext$Days, levels = c("Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday","Sunday"))

Pat_Nur_SDM_PNR_Weekdays_u1_ext <- Pat_Nur_SDM_PNR_Weekdays_u1_ext %>%
  mutate(change = if_else(Mean_SDM > 0, "Positive demand-supply match", "Negative demand-supply match"))


# 1.5.1 Supply-demand match plot-----------------------------------------------
ggplot(Pat_Nur_SDM_PNR_Weekdays_u1_ext, aes(Time, Mean_SDM))+
  geom_line()+
  facet_grid(rows = vars(Year), cols = vars(Days))+
  geom_hline(yintercept = 0, linetype = "dashed", color = "red") +
  chron::scale_x_chron(format = "%H:%M", expand = c(0,0), n = 6) +
  geom_area(aes(y=if_else(Mean_SDM>0, Mean_SDM, 0), fill = "Positive demand-supply match"))+
  geom_area(aes(y=if_else(Mean_SDM<0, Mean_SDM, 0), fill = "Negative demand-supply match"))+
  scale_fill_manual(name = "Legend",
                    breaks = c("Positive demand-supply match", "Negative demand-supply match"),
                    values = c("Positive demand-supply match" = "#4393C3", "Negative demand-supply match" = "#D6604D"))

## only weighted by target patient to nurse ratio
## day shift 1:4
## late shift 1:4
## night shift 1:6
ggplot(Pat_Nur_SDM_PNR_Weekdays_u1_ext, aes(Time, Mean_SDM_uw))+
  geom_line()+
  facet_grid(rows = vars(Year), cols = vars(Days))+
  geom_hline(yintercept = 0, linetype = "dashed", color = "red") +
  chron::scale_x_chron(format = "%H:%M", expand = c(0,0), n = 6) +
  geom_area(aes(y=if_else(Mean_SDM_uw>0, Mean_SDM_uw, 0), fill = "Positive demand-supply match"))+
  geom_area(aes(y=if_else(Mean_SDM_uw<0, Mean_SDM_uw, 0), fill = "Negative demand-supply match"))+
  scale_fill_manual(name = "Legend",
                    breaks = c("Positive demand-supply match", "Negative demand-supply match"),
                    values = c("Positive demand-supply match" = "#4393C3", "Negative demand-supply match" = "#D6604D"))


# 1.5.2 Supply-demand shifts ----------------------------------------------
SDM_Shifts_unit1 <- SDM_Shifts_unit1 %>%
  filter(!shift_type_nurse %in% "overlap") %>%
  mutate(shift_type_nurse = str_replace(shift_type_nurse, "dayshift", "Day shift"),
         shift_type_nurse = str_replace(shift_type_nurse, "lateshift", "Late shift"),
         shift_type_nurse = str_replace(shift_type_nurse, "nightshift", "Night shift"))


Test <- ggplot(SDM_Shifts_unit1, aes(date_worked))+
  geom_area(aes(y=if_else(Sum_HSD_numMw>0, Sum_HSD_numMw, 0), fill = "Positive match (met care demand + excess staff)"))+
  geom_area(aes(y=if_else(Sum_HSD_numMw<0, Sum_HSD_numMw, 0), fill = "Negative match (unmet care demand)"))+
  facet_grid(rows = vars(shift_type_nurse))+
  geom_hline(yintercept = 0, color = "black")+
  scale_y_continuous(limits = c(-8,8), breaks = seq(from = -8, to = 8, by = 2)) +
  labs(x = " ", y = "Number of midwives")+
  chron::scale_x_chron(format = "%b %Y")+
  scale_fill_manual(name = "Legend",
                    breaks = c("Positive match (met care demand + excess staff)", "Negative match (unmet care demand)"),
                    values = c("Positive match (met care demand + excess staff)" = "#4393C3", "Negative match (unmet care demand)" = "#D6604D"))+
  theme(panel.grid.major = element_line(linewidth=0.2, linetype = "solid", colour = "grey70"),
        panel.grid.minor = element_line(linewidth=0.2, linetype = "solid", colour = "grey85"),
        panel.border = element_rect(linetype = "solid", colour = "white", fill = NA),
        panel.background = element_rect(linetype = "solid", colour = "grey", fill = "grey95"),
        axis.text.x =element_text(size=18, color = "black"),
        axis.text.y =element_text(size=18, color = "black"),
        axis.title.x =element_text(size=24, color = "black"),
        axis.title.y = element_text(size=24, color = "black"),
        plot.title = element_text(size=28, color = "black"),
        legend.position = "bottom",
        legend.title = element_text(size=24),
        strip.text.x = element_text(size=24, color = "black"),
        strip.text.y = element_text(size=18, color = "black"),
        legend.text = element_text(size=24, color = "black"))
Test


# 2. Unit 2 (labour) ---------------------------------------------------------------


# 2.1 Patients ------------------------------------------------------------
unit2_NP_CF<- unit2_NP_CF %>%
  rename(shift_type_pat =shift_type)

# 2.1.1 Hours per patient -----------------------------------------------
unit2_NP_CF_hours <- unit2_NP_CF %>%
  mutate(demand_hours = NumWomen_cat*15) %>%
  mutate(demand_hours_weighted = NumWomen_cat*cf_weight_u2*15)

## summarise demand_hours and weighted demand_hours over all categories per time frame
## currently this is not in hours but in minutes.
unit2_NP_CF_hours_adj <- unit2_NP_CF_hours %>%
  group_by(date_worked, Time, Days, Weekday, Year, Month, shift_type_pat) %>%
  summarise(demand_hours_total = sum(demand_hours),
            demand_hours_total_weighted = sum(demand_hours_weighted))
## 140,239 observations

## For the calculation of patient to nurse ratio, the number of patients is needed.
## The easiest way seems to calculate the total number of women from the summarised data
unit2_NP_CF_hours_adj <- unit2_NP_CF_hours_adj %>%
  mutate(NumPatient = demand_hours_total/15)

### For the calculation of outpatient time, the outpatient hours per shift are added
unit2_NP_out_hours <- unit2_NP_outpatient %>%
  ungroup()%>%
  select(date, shift_level)



# 2.2 Nurses --------------------------------------------------------------
unit2_NN_RM <- unit2_NN %>%
  rename(shift_type_nurse = shift_type_NN) %>%
  filter(nurse_group %in% "Hebammen") %>% ## Hebammen = Midwives
  select(!unit_id)
## 140,228 observations

# 2.2.1 Hours per nurse -------------------------------------------------
unit2_NN_RM_hours <- unit2_NN_RM %>%
  mutate(supply_hours = NumNurse*15)
## 140,228 observations

# 2.3 Demand-supply match ----------------------------------------------
Pat_nurse_u2 <- left_join(unit2_NN_RM_hours, unit2_NP_CF_hours_adj, by = c("date_worked", "Time", "Days", "Weekday", "Year", "Month"), relationship = "many-to-many")
## 140,228 observations

Pat_nurse_u2 <- Pat_nurse_u2 %>%
  filter(!(is.na(demand_hours_total)))
## 140,212 observations

## Creating supply demand match
Pat_nurse_u2$sdm <- (Pat_nurse_u2$supply_hours)-(Pat_nurse_u2$demand_hours_total_weighted)
Pat_nurse_u2$sdm_unweighted <-  (Pat_nurse_u2$supply_hours)-(Pat_nurse_u2$demand_hours_total)
## positive number = oversupply
## negative number = undersupply

## Creating patient to nurse ratio
Pat_nurse_u2$PNR <- (Pat_nurse_u2$NumPatient)/(Pat_nurse_u2$NumNurse)

Pat_nurse_u2$PNR[Pat_nurse_u2$PNR %in% "NaN"] <- NA
Pat_nurse_u2$PNR[Pat_nurse_u2$PNR %in% "Inf"] <- NA
# 
# save(Pat_nurse_u2, file= "./Pat_nurse_u2.RData")

# 2.4 Mean calculations ---------------------------------------------------

## patients
Patient_Weekdays_unit2 <- Pat_nurse_u2%>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_Pat = round(mean(NumPatient),1),
                STD_DEV_Pat = round(sd(NumPatient),1),
                Median_Pat = round(median(NumPatient),1),
                IQR_Pat = round(IQR(NumPatient),1),
                Minimum_Pat = round(min(NumPatient),1),
                Maximum_Pat = round(max(NumPatient),1),
                CI_Pat = qnorm(0.975)*(STD_DEV_Pat/sqrt(n())))
  ))


## nurses
Nurse_Weekdays_unit2 <- Pat_nurse_u2 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_Nur = round(mean(NumNurse),1),
                STD_DEV_Nur = round(sd(NumNurse),1),
                Median_Nur = round(median(NumNurse),1),
                IQR_Nur = round(IQR(NumNurse),1),
                Minimum_Nur = round(min(NumNurse),1),
                Maximum_Nur = round(max(NumNurse),1),
                CI_Nur = qnorm(0.975)*(STD_DEV_Nur/sqrt(n())))
  ))

## supply-demand match
SDM_Weekdays_unit2 <- Pat_nurse_u2 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_SDM = round(mean(sdm, na.rm = T),1),
                STD_DEV_SDM = round(sd(sdm, na.rm = T),1),
                Median_SDM = round(median(sdm, na.rm = T),1),
                IQR_SDM = round(IQR(sdm, na.rm = T),1),
                Minimum_SDM = round(min(sdm, na.rm = T),1),
                Maximum_SDM = round(max(sdm, na.rm = T),1),
                CI_SDM = qnorm(0.975)*(STD_DEV_SDM/sqrt(n())))
  ))

## supply-demand match unweighted
SDM_uw_Weekdays_unit2 <- Pat_nurse_u2 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_SDM_uw = round(mean(sdm_unweighted, na.rm = T),1),
                STD_DEV_SDM_uw = round(sd(sdm_unweighted, na.rm = T),1),
                Median_SDM_uw = round(median(sdm_unweighted, na.rm = T),1),
                IQR_SDM_uw = round(IQR(sdm_unweighted, na.rm = T),1),
                Minimum_SDM_uw = round(min(sdm_unweighted, na.rm = T),1),
                Maximum_SDM_uw = round(max(sdm_unweighted, na.rm = T),1),
                CI_SDM_uw = qnorm(0.975)*(STD_DEV_SDM_uw/sqrt(n())))
  ))


## nurse-to-patient ratio
PNR_Weekdays_unit2 <- Pat_nurse_u2 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  # group_by(date_worked, shift_type_nurse) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_PNR = round(mean(PNR, na.rm = T),1),
                STD_DEV_PNR = round(sd(PNR, na.rm = T),1),
                Median_PNR = round(median(PNR, na.rm = T),1),
                IQR_PNR = round(IQR(PNR, na.rm = T),1),
                Minimum_PNR = round(min(PNR, na.rm = T),1),
                Maximum_PNR = round(max(PNR, na.rm = T),1),
                CI_PNR = qnorm(0.975)*(STD_DEV_PNR/sqrt(n())))
  ))

# tapply(PNR_Weekdays_unit2$Median_PNR, PNR_Weekdays_unit2$shift_type_nurse, summary)

# ### creating data frame with all information together
Pat_Nurse_Weekdays_u2 <- left_join(Nurse_Weekdays_unit2, Patient_Weekdays_unit2, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_Weekdays_u2 <- left_join(Pat_Nurse_Weekdays_u2, SDM_Weekdays_unit2, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_PNR_Weekdays_u2 <- left_join(Pat_Nur_SDM_Weekdays_u2, PNR_Weekdays_unit2, by = c("nurse_group", "Days", "Time", "Year"))
 
Pat_Nur_SDM_PNR_Weekdays_u2_ext <- left_join(Pat_Nur_SDM_PNR_Weekdays_u2, SDM_uw_Weekdays_unit2, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_PNR_Weekdays_u2_ext$Time <- chron::chron(times = Pat_Nur_SDM_PNR_Weekdays_u2_ext$Time)
 
rm(Pat_Nurse_Weekdays_u2, Patient_Weekdays_unit2, Nurse_Weekdays_unit2, PNR_Weekdays_unit2, SDM_Weekdays_unit2, Pat_Nur_SDM_Weekdays_u2,
   unit2_NN_RM, unit2_NP_CF_hours, SDM_uw_Weekdays_unit2)
# 

# save(Pat_Nur_PNR_Weekdays_u2, file = ./Unit2_Weekdays.RData")



# 2.4.1.1 Mean calculations per shift + outpatient-------------------------------------

## care demand unweighted
Demand_Shifts_unit2 <- Pat_nurse_u2 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_Demand = sum(demand_hours_total),
                Sum_Demand_hour = Sum_Demand/60,
                Sum_Demand_RM = Sum_Demand_hour/8)
  ))

tapply(Demand_Shifts_unit2$Sum_Demand_hour, Demand_Shifts_unit2$shift_type_nurse, summary)

## care demand unweighted + outpatient per shift (dailycount/3)
unit2_NP_out_hours2 <- left_join(Demand_Shifts_unit2, unit2_NP_out_hours, by = c("date_worked" = "date"))

Demand_Shifts_unit2_full <- unit2_NP_out_hours2 %>%
  mutate(shift_level = if_else(is.na(shift_level), 0, shift_level)) %>%
  mutate(Sum_Demand_hour_full = Sum_Demand_hour+shift_level,
         Sum_Demand_RM_full = Sum_Demand_hour_full/8)

tapply(Demand_Shifts_unit2_full$Sum_Demand_hour_full, Demand_Shifts_unit2_full$shift_type_nurse, summary)


## care demand weighted
Demand_Shifts_unit2_weighted <- Pat_nurse_u2 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_Demand_w = sum(demand_hours_total_weighted),
                Sum_Demand_hour_w = Sum_Demand_w/60,
                Sum_Demand_RM_w = Sum_Demand_hour_w/8)
  ))

tapply(Demand_Shifts_unit2_weighted$Sum_Demand_hour_w, Demand_Shifts_unit2_weighted$shift_type_nurse, summary)

## care demand unweighted + outpatient per shift (dailycount/3)
unit2_NP_out_hours2 <- left_join(Demand_Shifts_unit2_weighted, unit2_NP_out_hours, by = c("date_worked" = "date"))

Demand_Shifts_unit2_weighted_full <- unit2_NP_out_hours2 %>%
  mutate(shift_level = if_else(is.na(shift_level), 0, shift_level)) %>%
  mutate(Sum_Demand_hour_w_full = Sum_Demand_hour_w+shift_level,
         Sum_Demand_RM_w_full = Sum_Demand_hour_w_full/8)

tapply(Demand_Shifts_unit2_weighted_full$Sum_Demand_hour_w_full, Demand_Shifts_unit2_weighted_full$shift_type_nurse, summary)

## care supply
Supply_Shifts_unit2 <- Pat_nurse_u2 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_Supply = sum(supply_hours),
                Sum_Supply_hour = Sum_Supply/60,
                Sum_Suppply_RM = Sum_Supply_hour/8)
  ))

tapply(Supply_Shifts_unit2$Sum_Supply_hour, Supply_Shifts_unit2$shift_type_nurse, summary)


## supply-demand match per shift weighted (and unweighted)
SDM_Shifts_unit2 <- Pat_nurse_u2 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_HSD = sum(sdm_unweighted),
      # summarise(Sum_HSD = sum(sdm),
                Sum_HSD_hour = Sum_HSD/60,
                Sum_HSD_numMw = Sum_HSD_hour/8)
  ))

tapply(SDM_Shifts_unit2$Sum_HSD_hour, SDM_Shifts_unit2$shift_type_nurse, summary)

## supply-demand match per  shift weighted (and unweighted) + outpatient per shift (dailycount/3)
Demand_all <- left_join(Demand_Shifts_unit2_full, Demand_Shifts_unit2_weighted_full, by = c("date_worked", "shift_type_nurse", "shift_level")) 
Demand_Supply_all_unit2 <- left_join(Demand_all, Supply_Shifts_unit2, by = c("date_worked", "shift_type_nurse"))

SDM_Shifts_unit2_full <- Demand_Supply_all_unit2 %>%
  mutate(Sum_HSD_hour_full = Sum_Supply_hour-Sum_Demand_hour_full,
         Sum_HSD_hour_w_full = Sum_Supply_hour-Sum_Demand_hour_w_full,
         Sum_HSD_numMw_full = Sum_HSD_hour_full/8,
         Sum_HSD_numMw_w_full = Sum_HSD_hour_w_full/8)

tapply(SDM_Shifts_unit2_full$Sum_HSD_hour_full, SDM_Shifts_unit2_full$shift_type_nurse, summary)
tapply(SDM_Shifts_unit2_full$Sum_HSD_hour_w_full, SDM_Shifts_unit2_full$shift_type_nurse, summary)

rm(Demand_Shifts_unit2, Demand_Shifts_unit2_weighted, Supply_Shifts_unit2, Demand_all,
   Demand_Shifts_unit2_full, Demand_Shifts_unit2_weighted_full, unit2_NP_out_hours, unit2_NP_out_hours2)

# 2.4.1.2 Number of shifts with certain threshold --------------------------
SDM_Shifts_unit2_undersupply <- SDM_Shifts_unit2_full %>%
  filter(!shift_type_nurse %in% "overlap") %>%
  filter(Sum_HSD_hour_full < (-8))
  # filter(Sum_HSD_hour_w_full < (-8))
  # filter(Sum_HSD_hour_full > 8)
  # filter(Sum_HSD_hour_w_full > 8)

rm(SDM_Shifts_unit2_undersupply)


# 2.5 Plots ---------------------------------------------------------------
Pat_Nur_SDM_PNR_Weekdays_u2_ext$Days <- factor(Pat_Nur_SDM_PNR_Weekdays_u2_ext$Days, levels = c("Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday","Sunday"))

# 2.5.1. Supply-demand match plots ----------------------------------------------
ggplot(Pat_Nur_SDM_PNR_Weekdays_u2_ext, aes(Time, Mean_SDM))+
  geom_line()+
  facet_grid(rows = vars(Year), cols = vars(Days))+
  geom_hline(yintercept = 0, linetype = "dashed", color = "red") +
  chron::scale_x_chron(format = "%H:%M", expand = c(0,0), n = 6) +
  geom_area(aes(y=if_else(Mean_SDM>0, Mean_SDM, 0), fill = "Positive demand-supply match"))+
  geom_area(aes(y=if_else(Mean_SDM<(-10), Mean_SDM, 0), fill = "Negative demand-supply match"))+
  scale_fill_manual(name = "Legend",
                    breaks = c("Positive demand-supply match", "Negative demand-supply match"),
                    values = c("Positive demand-supply match" = "#4393C3", "Negative demand-supply match" = "#D6604D"))

## unweighted demand, based on 1:1 care.
ggplot(Pat_Nur_SDM_PNR_Weekdays_u2_ext, aes(Time, Mean_SDM_uw))+
  geom_line()+
  facet_grid(rows = vars(Year), cols = vars(Days))+
  geom_hline(yintercept = 0, linetype = "dashed", color = "red") +
  chron::scale_x_chron(format = "%H:%M", expand = c(0,0), n = 6) +
  geom_area(aes(y=if_else(Mean_SDM_uw>0, Mean_SDM_uw, 0), fill = "Positive demand-supply match"))+
  geom_area(aes(y=if_else(Mean_SDM_uw<0, Mean_SDM_uw, 0), fill = "Negative demand-supply match"))+
  scale_fill_manual(name = "Legend",
                    breaks = c("Positive demand-supply match", "Negative demand-supply match"),
                    values = c("Positive demand-supply match" = "#4393C3", "Negative demand-supply match" = "#D6604D"))

## unweighted demand, based on 1:1 care, divided by 15 to get number of midwives.
SDM_undweighted_midwife <- Pat_Nur_SDM_PNR_Weekdays_u2_ext %>%
  mutate(Mean_Number_Mw = (Mean_SDM_uw/15))

ggplot(SDM_undweighted_midwife, aes(Time, Mean_Number_Mw))+
  geom_line()+
  facet_grid(rows = vars(Year), cols = vars(Days))+
  geom_hline(yintercept = 0, linetype = "dashed", color = "red") +
  chron::scale_x_chron(format = "%H:%M", expand = c(0,0), n = 6) +
  labs(title = "Supply - Demand match [Unit 2]", x = "Time", y = "Number of midwives")+
  scale_y_continuous(limits = c(-7,7), breaks = seq(from = -7, to = 7, by = 1)) +
  geom_area(aes(y=if_else(Mean_Number_Mw>0, Mean_Number_Mw, 0), fill = "Positive demand-supply match"))+
  geom_area(aes(y=if_else(Mean_Number_Mw<0, Mean_Number_Mw, 0), fill = "Negative demand-supply match"))+
  scale_fill_manual(name = "Legend",
                    breaks = c("Positive demand-supply match", "Negative demand-supply match"),
                    values = c("Positive demand-supply match" = "#4393C3", "Negative demand-supply match" = "#D6604D"))
  


# 2.5.2 Supply-demand shifts ----------------------------------------------
SDM_Shifts_unit2 <- SDM_Shifts_unit2_full %>%
  filter(!shift_type_nurse %in% "overlap") %>%
  mutate(shift_type_nurse = str_replace(shift_type_nurse, "dayshift", "Day shift"),
         shift_type_nurse = str_replace(shift_type_nurse, "lateshift", "Late shift"),
         shift_type_nurse = str_replace(shift_type_nurse, "nightshift", "Night shift"))

SDM_Shifts_unit2 <- SDM_Shifts_unit2_full %>%
  filter(shift_type_nurse %in% "dayshift") %>%
  mutate(shift_type_nurse = str_replace(shift_type_nurse, "dayshift", "Unweighted")) ## change manually from weighted to unweighted

Test <- ggplot(SDM_Shifts_unit2, aes(date_worked))+
  # ggplot(HSD_sub_Mw_u2_short, aes(date_worked))+
  geom_area(aes(y=if_else(Sum_HSD_numMw_full>0, Sum_HSD_numMw_full, 0), fill = "Positive match (met care demand + excess staff)"))+ # un-weighted
  geom_area(aes(y=if_else(Sum_HSD_numMw_full<0, Sum_HSD_numMw_full, 0), fill = "Negative match (unmet care demand)"))+ # un-weighted
  # geom_area(aes(y=if_else(Sum_HSD_numMw_w_full>0, Sum_HSD_numMw_w_full, 0), fill = "Positive match (met care demand + excess staff)"))+ # weighted
  # geom_area(aes(y=if_else(Sum_HSD_numMw_w_full<0, Sum_HSD_numMw_w_full, 0), fill = "Negative match (unmet care demand)"))+ # weighted
  facet_grid(rows = vars(shift_type_nurse))+
  geom_hline(yintercept = 0, color = "black")+
  scale_y_continuous(limits = c(-8,8), breaks = seq(from = -8, to = 8, by = 2)) +
  # labs(title = "Demand-supply match weighted [Unit 2]", x = " ", y = "Number of midwives")+ ## change manually from weighted to unweighted
  labs(x = " ", y = "Number of midwives")+ 
    # labs(title = "Demand-supply match of all day shifts [Unit 2]", x = " ", y = "Number of midwives")+
  chron::scale_x_chron(format = "%b %Y")+
  scale_fill_manual(name = "Legend",
                    breaks = c("Positive match (met care demand + excess staff)", "Negative match (unmet care demand)"),
                    values = c("Positive match (met care demand + excess staff)" = "#4393C3", "Negative match (unmet care demand)" = "#D6604D"))+
  theme(panel.grid.major = element_line(linewidth=0.2, linetype = "solid", colour = "grey70"),
        panel.grid.minor = element_line(linewidth=0.2, linetype = "solid", colour = "grey85"),
        panel.border = element_rect(linetype = "solid", colour = "white", fill = NA),
        panel.background = element_rect(linetype = "solid", colour = "grey", fill = "grey95"),
        axis.text.x =element_text(size=18, color = "black"),
        axis.text.y =element_text(size=18, color = "black"),
        axis.title.x =element_text(size=24, color = "black"),
        axis.title.y = element_text(size=24, color = "black"),
        plot.title = element_text(size=28, color = "black"),
        # panel.spacing = unit(0, "lines"),
        legend.position = "bottom",
        legend.title = element_text(size=24),
        strip.text.x = element_text(size=24, color = "black"),
        strip.text.y = element_text(size=18, color = "black"),
        legend.text = element_text(size=24, color = "black"))
Test
# rm(SDM_Shifts_unit2)

# 3. Unit 3 (postnatal) ---------------------------------------------------------------

# 3.1 Patients ------------------------------------------------------------
unit3_NP_CF <- unit3_NP_CF %>%
  rename(shift_type_pat = shift_type) %>%
  mutate(cf_weight_u3 = round(cf_weight_u3, 4)) %>%
  mutate(weight_u3 = round(weight_u3, 4))


# 3.1.1 Hours per patient -------------------------------------------------
unit3_NP_CF_hours <- unit3_NP_CF %>%
  mutate(demand_hours = NumWomen_cat*15) %>% ## counting patients in total full
  mutate(demand_hours_weighted = NumWomen_cat*cf_weight_u3*15) %>% ## weights based on Nurse to patient ratio & CF from birth
  mutate(demand_hours_ptn = NumWomen_cat*weight_u3*15)

## summarise demand_hours and weighted demand_hours over all categories per time frame
## currently this is not in hours but in minutes.
unit3_NP_CF_hours_adj <- unit3_NP_CF_hours %>%
  group_by(date_worked, Time, Days, Weekday, Year, Month, shift_type_pat) %>%
  summarise(demand_hours_total = sum(demand_hours),
            demand_hours_total_weighted = sum(demand_hours_weighted),
            demand_hours_total_w_ptn = sum(demand_hours_ptn))
## 140,239 observations

## For the calculation of patient to nurse ratio, the number of patients is needed.
## The easiest way seems to calculate the total number of women from the summarised data
unit3_NP_CF_hours_adj <- unit3_NP_CF_hours_adj %>%
  mutate(NumPatient = demand_hours_total/15)

# 3.2 Nurses --------------------------------------------------------------
unit3_NN_RM <- unit3_NN %>%
  rename(shift_type_nurse = shift_type) %>%
  filter(nurse_group %in% "Pflege_") %>% ## PFlege = Nurses (& Midwives)
  select(!unit_id)


# 3.2.1 Hours per nurse ---------------------------------------------------
unit3_NN_RM_hours <- unit3_NN_RM %>%
  mutate(supply_hours = NumNurse*15)
## 140,228 observations

# 3.3 Demand-supply match ----------------------------------------------
Pat_nurse_u3 <- left_join(unit3_NN_RM_hours, unit3_NP_CF_hours_adj, by = c("date_worked", "Time", "Days", "Weekday", "Year", "Month"), relationship = "many-to-many")
## 140,228 observations

Pat_nurse_u3 <- Pat_nurse_u3 %>%
  filter(!(is.na(demand_hours_total)))
## 140,212 observations

## Creating supply demand match
Pat_nurse_u3$sdm <- (Pat_nurse_u3$supply_hours)-(Pat_nurse_u3$demand_hours_total_weighted)
Pat_nurse_u3$sdm_unweighted <-  (Pat_nurse_u3$supply_hours)-(Pat_nurse_u3$demand_hours_total_w_ptn)
## positive number = oversupply
## negative number = undersupply

## Creating patient to nurse ratio
Pat_nurse_u3$PNR <- (Pat_nurse_u3$NumPatient)/(Pat_nurse_u3$NumNurse)

Pat_nurse_u3$PNR[Pat_nurse_u3$PNR %in% "NaN"] <- NA
Pat_nurse_u3$PNR[Pat_nurse_u3$PNR %in% "Inf"] <- NA


# save(Pat_nurse_u3, file= "./Pat_nurse_u3.RData")


# 3.4 Mean calculations ---------------------------------------------------

## patients
Patient_Weekdays_unit3 <- Pat_nurse_u3%>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_Pat = round(mean(NumPatient),1),
                STD_DEV_Pat = round(sd(NumPatient),1),
                Median_Pat = round(median(NumPatient),1),
                IQR_Pat = round(IQR(NumPatient),1),
                Minimum_Pat = round(min(NumPatient),1),
                Maximum_Pat = round(max(NumPatient),1),
                CI_Pat = qnorm(0.975)*(STD_DEV_Pat/sqrt(n())))
  ))


## nurses
Nurse_Weekdays_unit3 <- Pat_nurse_u3 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_Nur = round(mean(NumNurse),1),
                STD_DEV_Nur = round(sd(NumNurse),1),
                Median_Nur = round(median(NumNurse),1),
                IQR_Nur = round(IQR(NumNurse),1),
                Minimum_Nur = round(min(NumNurse),1),
                Maximum_Nur = round(max(NumNurse),1),
                CI_Nur = qnorm(0.975)*(STD_DEV_Nur/sqrt(n())))
  ))


## supply-demand match
SDM_Weekdays_unit3 <- Pat_nurse_u3 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_SDM = round(mean(sdm, na.rm = T),1),
                STD_DEV_SDM = round(sd(sdm, na.rm = T),1),
                Median_SDM = round(median(sdm, na.rm = T),1),
                IQR_SDM = round(IQR(sdm, na.rm = T),1),
                Minimum_SDM = round(min(sdm, na.rm = T),1),
                Maximum_SDM = round(max(sdm, na.rm = T),1),
                CI_SDM = qnorm(0.975)*(STD_DEV_SDM/sqrt(n())))
  ))

## supply-demand match unweighted
SDM_uw_Weekdays_unit3 <- Pat_nurse_u3 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_SDM_uw = round(mean(sdm_unweighted, na.rm = T),1),
                STD_DEV_SDM_uw = round(sd(sdm_unweighted, na.rm = T),1),
                Median_SDM_uw = round(median(sdm_unweighted, na.rm = T),1),
                IQR_SDM_uw = round(IQR(sdm_unweighted, na.rm = T),1),
                Minimum_SDM_uw = round(min(sdm_unweighted, na.rm = T),1),
                Maximum_SDM_uw = round(max(sdm_unweighted, na.rm = T),1),
                CI_SDM_uw = qnorm(0.975)*(STD_DEV_SDM_uw/sqrt(n())))
  ))

## supply-demand match per shift
SDM_Shifts_unit3 <- Pat_nurse_u3 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_HSD = sum(sdm_unweighted),
    # summarise(Sum_HSD = sum(sdm),
                Sum_HSD_hour = Sum_HSD/60,
                Sum_HSD_numMw = Sum_HSD_hour/8)
  ))

## nurse-to-patient ratio
PNR_Weekdays_unit3 <- Pat_nurse_u3 %>%
  group_by(nurse_group, Days, Year) %>%
  # group_by(nurse_group, Days) %>%
  # group_by(date_worked, shift_type_nurse) %>%
  group_modify(~ (
    .x %>%
      group_by(Time) %>%
      summarise(Mean_PNR = round(mean(PNR, na.rm = T),1),
                STD_DEV_PNR = round(sd(PNR, na.rm = T),1),
                Median_PNR = round(median(PNR, na.rm = T),1),
                IQR_PNR = round(IQR(PNR, na.rm = T),1),
                Minimum_PNR = round(min(PNR, na.rm = T),1),
                Maximum_PNR = round(max(PNR, na.rm = T),1),
                CI_PNR = qnorm(0.975)*(STD_DEV_PNR/sqrt(n())))
  ))

# tapply(PNR_Weekdays_unit3$Median_PNR, PNR_Weekdays_unit3$shift_type_nurse, summary)

### creating data frame with all information together
Pat_Nurse_Weekdays_u3 <- left_join(Nurse_Weekdays_unit3, Patient_Weekdays_unit3, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_Weekdays_u3 <- left_join(Pat_Nurse_Weekdays_u3, SDM_Weekdays_unit3, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_PNR_Weekdays_u3 <- left_join(Pat_Nur_SDM_Weekdays_u3, PNR_Weekdays_unit3, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_PNR_Weekdays_u3_ext <- left_join(Pat_Nur_SDM_PNR_Weekdays_u3, SDM_uw_Weekdays_unit3, by = c("nurse_group", "Days", "Time", "Year"))

Pat_Nur_SDM_PNR_Weekdays_u3_ext$Time <- chron::chron(times = Pat_Nur_SDM_PNR_Weekdays_u3_ext$Time)

rm(Pat_Nurse_Weekdays_u3, Patient_Weekdays_unit3, Nurse_Weekdays_unit3, PNR_Weekdays_unit3, SDM_Weekdays_unit3, Pat_Nur_SDM_Weekdays_u3,
   unit3_NN_RM, unit3_NP_CF_hours, SDM_uw_Weekdays_unit3)

# save(Pat_Nur_PNR_Weekdays_u3, file = "./Unit3_Weekdays.RData")


# 3.4.1.1 Mean calculations per shift -------------------------------------


## care demand unweighted (only by patient to nurse ratio)
Demand_Shifts_unit3 <- Pat_nurse_u3 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_Demand = sum(demand_hours_total_w_ptn),
                Sum_Demand_hour = Sum_Demand/60,
                Sum_Demand_RM = Sum_Demand_hour/8)
  ))

tapply(Demand_Shifts_unit3$Sum_Demand_hour, Demand_Shifts_unit3$shift_type_nurse, summary)

## care demand weighted (+ patient to nurse ratio)
Demand_Shifts_unit3_weighted <- Pat_nurse_u3 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_Demand = sum(demand_hours_total_weighted),
                Sum_Demand_hour = Sum_Demand/60,
                Sum_Demand_RM = Sum_Demand_hour/8)
  ))

tapply(Demand_Shifts_unit3_weighted$Sum_Demand_hour, Demand_Shifts_unit3_weighted$shift_type_nurse, summary)

## care supply
Supply_Shifts_unit3 <- Pat_nurse_u3 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_Supply = sum(supply_hours),
                Sum_Supply_hour = Sum_Supply/60,
                Sum_Suppply_RM = Sum_Supply_hour/8)
  ))

tapply(Supply_Shifts_unit3$Sum_Supply_hour, Supply_Shifts_unit3$shift_type_nurse, summary)


## supply-demand match per shift weighted (and unweighted)
SDM_Shifts_unit3 <- Pat_nurse_u3 %>%
  group_by(date_worked) %>%
  group_modify(~ (
    .x %>%
      group_by(shift_type_nurse) %>%
      summarise(Sum_HSD = sum(sdm_unweighted),
      # summarise(Sum_HSD = sum(sdm),
                Sum_HSD_hour = Sum_HSD/60,
                Sum_HSD_numMw = Sum_HSD_hour/8)
  ))

tapply(SDM_Shifts_unit3$Sum_HSD_hour, SDM_Shifts_unit3$shift_type_nurse, summary)

rm(Demand_Shifts_unit3, Demand_Shifts_unit3_weighted, Supply_Shifts_unit3)

# 3.4.1.2 Number of shifts with certain threshold --------------------------
SDM_Shifts_unit3_undersupply <- SDM_Shifts_unit3 %>%
  filter(!shift_type_nurse %in% "overlap") %>%
  filter(Sum_HSD_hour < (-8))
  # filter(Sum_HSD_hour > 8)


rm(SDM_Shifts_unit3_undersupply)

# 3.5 Plots ---------------------------------------------------------------
Pat_Nur_SDM_PNR_Weekdays_u3_ext$Days <- factor(Pat_Nur_SDM_PNR_Weekdays_u3_ext$Days, levels = c("Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday","Sunday"))


# 3.5.1 Supply-demand match plot-----------------------------------------------
ggplot(Pat_Nur_SDM_PNR_Weekdays_u3_ext, aes(Time, Mean_SDM))+
  geom_line()+
  facet_grid(rows = vars(Year), cols = vars(Days))+
  geom_hline(yintercept = 0, linetype = "dashed", color = "red") +
  chron::scale_x_chron(format = "%H:%M", expand = c(0,0), n = 6) +
geom_area(aes(y=if_else(Mean_SDM>0, Mean_SDM, 0), fill = "Positive demand-supply match"))+
geom_area(aes(y=if_else(Mean_SDM<0, Mean_SDM, 0), fill = "Negative demand-supply match"))+
scale_fill_manual(name = "Legend",
                  breaks = c("Positive demand-supply match", "Negative demand-supply match"),
                  values = c("Positive demand-supply match" = "#4393C3", "Negative demand-supply match" = "#D6604D"))

## only weighted by target patient to nurse ratio
## day shift 1:4
## late shift 1:4
## night shift 1:6
ggplot(Pat_Nur_SDM_PNR_Weekdays_u3_ext, aes(Time, Mean_SDM_uw))+
  geom_line()+
  facet_grid(rows = vars(Year), cols = vars(Days))+
  geom_hline(yintercept = 0, linetype = "dashed", color = "red") +
  chron::scale_x_chron(format = "%H:%M", expand = c(0,0), n = 6) +
  geom_area(aes(y=if_else(Mean_SDM_uw>0, Mean_SDM_uw, 0), fill = "Positive demand-supply match"))+
  geom_area(aes(y=if_else(Mean_SDM_uw<0, Mean_SDM_uw, 0), fill = "Negative demand-supply match"))+
  scale_fill_manual(name = "Legend",
                    breaks = c("Positive demand-supply match", "Negative demand-supply match"),
                    values = c("Positive demand-supply match" = "#4393C3", "Negative demand-supply match" = "#D6604D"))


# 3.5.2 Supply-demand shifts ----------------------------------------------
SDM_Shifts_unit3 <- SDM_Shifts_unit3 %>%
  filter(!shift_type_nurse %in% "overlap") %>%
  mutate(shift_type_nurse = str_replace(shift_type_nurse, "dayshift", "Day shift"),
         shift_type_nurse = str_replace(shift_type_nurse, "lateshift", "Late shift"),
         shift_type_nurse = str_replace(shift_type_nurse, "nightshift", "Night shift"))


Test <- ggplot(SDM_Shifts_unit3, aes(date_worked))+
  geom_area(aes(y=if_else(Sum_HSD_numMw>0, Sum_HSD_numMw, 0), fill = "Positive match (met care demand + excess staff)"))+
  geom_area(aes(y=if_else(Sum_HSD_numMw<0, Sum_HSD_numMw, 0), fill = "Negative match (unmet care demand)"))+
  facet_grid(rows = vars(shift_type_nurse))+
  geom_hline(yintercept = 0, color = "black")+
  scale_y_continuous(limits = c(-8,8), breaks = seq(from = -8, to = 8, by = 2)) +
  # labs(title = "Demand-supply match unweighted [Unit 3]", x = " ", y = "Number of nurses and midwives")+ ## change manually from weighted to unweighted
  labs(x = " ", y = "Number of nurses and midwives")+
  chron::scale_x_chron(format = "%b %Y")+
  scale_fill_manual(name = "Legend",
                    breaks = c("Positive match (met care demand + excess staff)", "Negative match (unmet care demand)"),
                    values = c("Positive match (met care demand + excess staff)" = "#4393C3", "Negative match (unmet care demand)" = "#D6604D"))+
  theme(panel.grid.major = element_line(linewidth=0.2, linetype = "solid", colour = "grey70"),
        panel.grid.minor = element_line(linewidth=0.2, linetype = "solid", colour = "grey85"),
        panel.border = element_rect(linetype = "solid", colour = "white", fill = NA),
        panel.background = element_rect(linetype = "solid", colour = "grey", fill = "grey95"),
        axis.text.x =element_text(size=18, color = "black"),
        axis.text.y =element_text(size=18, color = "black"),
        axis.title.x =element_text(size=24, color = "black"),
        axis.title.y = element_text(size=24, color = "black"),
        plot.title = element_text(size=28, color = "black"),
        # panel.spacing = unit(0, "lines"),
        legend.position = "bottom",
        legend.title = element_text(size=24),
        strip.text.x = element_text(size=24, color = "black"),
        strip.text.y = element_text(size=18, color = "black"),
        legend.text = element_text(size=24, color = "black"))


Test

# ggsave(Test, file = "test.eps", width = 44.0, height = 28, units = "cm", dpi = 300)
# ggsave(Test, file = "test.pdf", width = 44.0, height = 28, units = "cm", dpi = 300)

# rm(SDM_Shifts_unit3)


# 3.5.4 Poster - continuous supply-demand ---------------------------------
U3_SDM <- Pat_Nur_SDM_PNR_Weekdays_u3_ext %>%
  filter(Year == 2022) %>%
  mutate(Mean_SDM_numMw = Mean_SDM/15)%>%
  mutate(shift = if_else(Time >= "07:30:00" & Time < "14:45:00", 1,
                             if_else( Time >= "14:45:00" & Time < "23:00:00", 0.5, 
                                      if_else(Time >= "23:00:00"| 
                                                Time < "07:30:00", 0.5, NA))))

 Test <-ggplot(U3_SDM, aes(Time, Mean_SDM_numMw))+
  facet_grid(rows = vars(Year), cols = vars(Days))+
  labs(title = "Yearly average demand-supply match, postnatal unit", x = "Time", y = "Number of nurses & midwives")+
  chron::scale_x_chron(format = "%H:%M", expand = c(0,0), n = 3) +
  scale_y_continuous(limits = c(-7,7), breaks = seq(from = -7, to = 7, by = 1)) +
  # geom_area(aes(y=if_else(Mean_SDM_numMw>0, Mean_SDM_numMw, 0), fill = "Surplus supply"))+
  # geom_area(aes(y=if_else(Mean_SDM_numMw<0, Mean_SDM_numMw, 0), fill = "Unmet demand"))+
    geom_col(aes(y=if_else(Mean_SDM_numMw>0, Mean_SDM_numMw, 0), fill = "Surplus supply", alpha = shift), width=0.0105)+
    geom_col(aes(y=if_else(Mean_SDM_numMw<0, Mean_SDM_numMw, 0), fill = "Unmet demand", alpha = shift), width=0.0105)+
  scale_fill_manual(name = "Legend",
                    breaks = c("Surplus supply", "Unmet demand"),
                    values = c("Surplus supply" = "#192E4E", "Unmet demand" = "#DB64A2"))+
  geom_line(aes(alpha = shift))+
    scale_alpha(guide = "none", range=c(0.5,1))+
  # geom_line()+
  theme(panel.grid.major = element_line(linewidth=0.2, linetype = "solid", colour = "grey70"),
        panel.grid.minor = element_line(linewidth=0.2, linetype = "solid", colour = "grey85"),
        panel.border = element_rect(linetype = "solid", colour = "white", fill = NA),
        panel.background = element_rect(linetype = "solid", colour = "grey", fill = "grey95"),
        axis.text.x =element_text(size=18, color = "black"),
        axis.text.y =element_text(size=18, color = "black"),
        axis.title.x =element_text(size=24, color = "black"),
        axis.title.y = element_text(size=24, color = "black"),
        plot.title = element_text(size=28, color = "black"),
        legend.position = "bottom",
        legend.title = element_text("Legend", size=24),
        strip.text.x = element_text(size=18, color = "black"),
        strip.text.y = element_text(size=18, color = "black"),
        legend.text = element_text(size=24, color = "black"))
Test


# 3.5.5 Manuscript Appendix Weekdays --------------------------------------
## This is weighted data. The difference is visible as well with unweighted data.
## To be able to show the difference better, we use weighted data. 
U3_SDM <- Pat_Nur_SDM_PNR_Weekdays_u3_ext %>%
  mutate(Mean_SDM_numMw = Mean_SDM/15)%>%
  # mutate(Mean_SDM_numMw = Mean_SDM_uw/15)%>%
  mutate(shift = if_else(Time >= "07:30:00" & Time < "14:45:00", 1,
                         if_else( Time >= "14:45:00" & Time < "23:00:00", 0.5, 
                                  if_else(Time >= "23:00:00"| 
                                            Time < "07:30:00", 0.5, NA))))

Test <-ggplot(U3_SDM, aes(Time, Mean_SDM_numMw))+
  facet_grid(rows = vars(Year), cols = vars(Days))+
  labs(x = "Time", y = "Number of nurses & midwives")+
  geom_hline(yintercept = 0, color = "black")+
  chron::scale_x_chron(format = "%H:%M", expand = c(0,0), n = 3) +
  scale_y_continuous(limits = c(-8,8), breaks = seq(from = -8, to = 8, by = 2)) +
  # geom_area(aes(y=if_else(Mean_SDM_numMw>0, Mean_SDM_numMw, 0), fill = "Positive match (met care demand + excess staff)"))+
  # geom_area(aes(y=if_else(Mean_SDM_numMw<0, Mean_SDM_numMw, 0), fill = "Negative match (unmet care demand)"))+
  geom_col(aes(y=if_else(Mean_SDM_numMw>0, Mean_SDM_numMw, 0), fill = "Positive match (met care demand + excess staff)", alpha = shift), width=0.0105)+
  geom_col(aes(y=if_else(Mean_SDM_numMw<0, Mean_SDM_numMw, 0), fill = "Negative match (unmet care demand)", alpha = shift), width=0.0105)+
  scale_fill_manual(name = "Legend",
                    breaks = c("Positive match (met care demand + excess staff)", "Negative match (unmet care demand)"),
                    values = c("Positive match (met care demand + excess staff)" = "#4393C3", "Negative match (unmet care demand)" = "#D6604D"))+
  geom_line(aes(alpha = shift))+
  scale_alpha(guide = "none", range=c(0.5,1))+
  theme(panel.grid.major = element_line(linewidth=0.2, linetype = "solid", colour = "grey70"),
        panel.grid.minor = element_line(linewidth=0.2, linetype = "solid", colour = "grey85"),
        panel.border = element_rect(linetype = "solid", colour = "white", fill = NA),
        panel.background = element_rect(linetype = "solid", colour = "grey", fill = "grey95"),
        axis.text.x =element_text(size=14, color = "black"),
        axis.text.y =element_text(size=14, color = "black"),
        axis.title.x =element_text(size=24, color = "black"),
        axis.title.y = element_text(size=24, color = "black"),
        plot.title = element_text(size=28, color = "black"),
        legend.position = "bottom",
        legend.title = element_text("Legend", size=24),
        strip.text.x = element_text(size=18, color = "black"),
        strip.text.y = element_text(size=18, color = "black"),
        legend.text = element_text(size=24, color = "black"))
Test

rm(U3_SDM)
rm(Test)

Sys.setenv(TZ=Timezone)
