package org.jlibsedml;

import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.jdom.Comment;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.Namespace;
import org.jdom.contrib.input.LineNumberSAXBuilder;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;
import org.jlibsedml.extensions.XMLUtils;
import org.jlibsedml.validation.ValidatorController;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Encapsulates a {@link SedML} model and provides additional validation
 * services.<br/>
 * E.g., typical usage might be:
 * 
 * <pre>
 * SEDMLDocument doc = new SEDMLDocument();
 * SedML sedml = doc.getSedMLModel();
 * //  create sedml object....
 * doc.validate();
 * if(doc.hasErrors(){
 *    List<SedMLError> errs = doc.getErrors();
 *    // show errors?
 * }
 * // now write model to file or to String
 *  doc.writeDocument (myFile);
 *  String doc = doc.writeDocumentToString();
 * </pre>
 * 
 * @author Richard Adams
 * 
 */
public class SEDMLDocument {
    Logger log = LoggerFactory.getLogger(SEDMLDocument.class);
    private List<SedMLError> errors = new ArrayList<SedMLError>();

    private SedML sedml;

    private boolean isValidationPerformed;

    private static final String jlibsedmlVersion = "2.2.3";
    static final String PROVENANCE = "This file was generated by jlibsedml, version "
            + jlibsedmlVersion + ".";

    /**
     * No parameter can be null, errors can be an empty list.
     * 
     * @param model
     *            A SedML element
     * @param errors
     *            A non-null <code>List</code> of {@link SedMLError}
     * @throws IllegalArgumentException
     *             if any arg is null.
     */
    public SEDMLDocument(SedML model, List<SedMLError> errors) {
        Assert.checkNoNullArgs(model, errors);
        this.sedml = model;
        this.errors = errors;
    }

    /**
     * Alternative constructor for creating a {@link SEDMLDocument}
     * 
     * @param sedML
     *            An already created SED-ML model object.
     * @throws IllegalArgumentException
     *             if any arg is null.
     */
    public SEDMLDocument(SedML sedML) {
        this(sedML, new ArrayList<SedMLError>());
    }

    /**
     * Default constructor creates empty SED-ML element with default version,
     * currently level 1, version 1.
     * @since 1.1
     */
    public SEDMLDocument() {
        this(1, 1);
    }
    
   /**
    * 
    * @param level The SED-ML level (1)
    * @param version The SED-ML version ( 1 or 2)
    * @throws IllegalArgumentException if values are invalid
    * @since 2.2.3
    */
    public SEDMLDocument(int level, int version) {
        if(version == 1) {
            this.sedml = new SedML(level, version, Namespace.getNamespace(SEDMLTags.SEDML_L1V1_NS));
        } else if (version == 2) {
            this.sedml = new SedML(level, version, Namespace.getNamespace(SEDMLTags.SEDML_L1V2_NS));
        } else {
            throw new IllegalArgumentException("Invalid version must be 1 or 2");
        }     
        sedml.setAdditionalNamespaces(Arrays.asList(new Namespace[] { Namespace
                .getNamespace(SEDMLTags.MATHML_NS_PREFIX, SEDMLTags.MATHML_NS) }));
    }

    /**
     * Gets a read-only list of this document's errors.
     * 
     * @return An unmodifiable (read-only), non-null list of this document's
     *         errors.
     */
    public List<SedMLError> getErrors() {
        return Collections.unmodifiableList(errors);
    }

    /**
     * Returns a {@link SedMLValidationReport} if validate() has previously been
     * called on this object, otherwise <code>null</code>.
     * 
     * @return A SedMLValidationReport or <code>null</code>.
     */
    public SedMLValidationReport getValidationReport() {
        if (!isValidationPerformed) {
            return null;

        } else {
            return new SedMLValidationReport(errors,
                    getSedMLDocumentAsString(sedml));
        }
    }

    /**
     * A boolean test as to whether the SEDML referenced by this document has
     * errors or not.
     * 
     * @return <code>true</code> if this document has at least one validation
     *         error.
     */
    public boolean hasErrors() {
        return errors.size() > 0;
    }

    /**
     * Gets the SED-ML model contained in this document.
     * 
     * @return A non-null {@link SedML} object
     */
    public SedML getSedMLModel() {
        return sedml;
    }

    /**
     * Validates this document. Validation may terminate prematurely if errors
     * are serious ( for example, if validation against the XML schema fails),
     * so the list of errors may not be complete, and new errors may be revealed
     * on subsequent invocations of this method. <br/>
     * This validate method validates an existing object model of SED-ML, and
     * not the original input file. To validate the raw input from an external
     * file, use
     * 
     * <pre>
     * Libsedml.validate(InputStream)
     * </pre>
     * 
     * @return An unmodifiable, non-null <code>List</code> of {@link SedMLError}
     *         .
     * @throws XMLException
     *             if the XML generated from the underlying SED-ML object is
     *             unable to be parsed, is unavailable or unreadable.
     */
    public List<SedMLError> validate() throws XMLException {
        Document doc = createDocument(writeDocumentToString());
        List<SedMLError> errs = new ValidatorController().validate(sedml, doc);
        errs.addAll(errors);
        errors = errs;
        isValidationPerformed = true;
        return getErrors();
    }

    /**
     * @see Object#toString()
     */
    public String toString() {
        return "SEDML Document for " + sedml.getNotes();
    }

    /**
     * Writes out a document to file. This operation will write valid and
     * invalid documents. To check a document is valid, call validate() and
     * hasErrors() before writing the document.
     * 
     * @param file
     *            A {@link File} that can be written to.
     * 
     * @throws IllegalArgumentException
     *             if <code>file</code> argument is null
     */
    public void writeDocument(File file) {
        Assert.checkNoNullArgs(file);

        String xmlString = getSedMLDocumentAsString(sedml);
        try {
            Libsedml.writeXMLStringToFile(xmlString, file.getAbsolutePath(),
                    true);
        } catch (IOException e) {
            throw new RuntimeException("Unable to write SEDML to file : "
                    + e.getMessage(), e);
        }
    }

    static String getSedMLDocumentAsString(SedML sedRoot) {
        SEDMLWriter producer = new SEDMLWriter();
        Element root = producer.getXML(sedRoot);

        root.addContent(0, new Comment(PROVENANCE));
        Document sedmlDoc = new Document();
        sedmlDoc.setRootElement(root);
        String xmlString = SEDMLUtils.xmlToString(sedmlDoc, true);
        return xmlString;
    }

    /**
     * Writes the document contents to formatted XML format, and returns it as a
     * <code>String</code>.
     * 
     * @return A <code>String</code> of XML
     */
    public String writeDocumentToString() {
        return getSedMLDocumentAsString(sedml);
    }

    /**
     * Getter for the SED-ML version of this document
     * @return A <code>Version</code> object.
     */
    public Version getVersion() {
        return new Version(sedml.getLevel(), sedml.getVersion());
    }

    /**
     * Gets a model variant, by applying whatever changes are defined in a
     * model's List of changes. This operation only performs one set of changes.
     * For example, if 3 models are defined in a SED-ML document: <font
     * style=bold>model1</font>, model2 and model3. If Model 3 is derived from
     * model2, and model2 from model1, then this method will only perform one
     * set of changes. E.g., if model3 is the desired output, then model2 must
     * be supplied as input.
     * <p/>
     * 
     * In order for an XPath expression to be applied, the namespace of the XML
     * to which the XPath is evaluated must be known. Since this library is
     * model-agnostic, this method tries to resolve the XPath prefix with a
     * Namespace defined in the model, by looking for the Namespace URI
     * containing the XPath prefix in a case-insensitive match.<br/>
     * For example, an XPath prefix of 'math' will match the namespace
     * 'http://www.w3.org/1998/Math/MathML'. The method <code>
     *  canResolveXPathExpressions (String model_ID, final String originalModel);
     * </code> in this class provides a test as to whether prefixes can be
     * resolved.
     * <p/>
     * @param model_ID
     *            The id of the SEDML Model element containing the description
     *            whose changes are to be applied.
     * @param originalModel
     *            A String representation of the XML of the model to be altered.
     * 
     * @return A String representation of the changed model. The original model
     *         will be unchanged by this operation. IF there are no changes, or
     *         the model_ID cannot be resolved, the original model will be
     *         returned unchanged.
     * @throws XMLException
     *             if XML cannot be parsed or the XPath expression applied.
     * @throws IllegalArgumentException
     *             if <code>modelID</code> is not defined in the ListOfModels for
     *             this document.
     */
    public String getChangedModel(String modelID, final String originalModel)
            throws XPathExpressionException, XMLException {

        String xmlString = "";
        Model m = sedml.getModelWithId(modelID);

        if (m == null || !m.hasChanges()) {
            return originalModel;
        }
        List<Change> changes = m.getListOfChanges();
        try {
            org.w3c.dom.Document doc = ModelTransformationUtils
                    .getXMLDocumentFromModelString(originalModel);

            XPathFactory xpf = XPathFactory.newInstance();
            XPath xpath = xpf.newXPath();

            for (Change change : changes) {
                org.jdom.Document docj = createDocument(originalModel);
                NamespaceContextHelper nc = new NamespaceContextHelper(docj);
                nc.process(change.getTargetXPath());
                xpath.setNamespaceContext(nc);
                if (change.getChangeKind().equals(SEDMLTags.CHANGE_ATTRIBUTE_KIND)) {
                    ModelTransformationUtils.applyAttributeChange(doc, xpath, change);
                } else if (change.getChangeKind().equals(SEDMLTags.REMOVE_XML_KIND)) {
                    ModelTransformationUtils.deleteXMLElement(doc, change.getTargetXPath().getTargetAsString(), xpath);
                } else if (change.getChangeKind().equals(SEDMLTags.ADD_XML_KIND)) {
                    AddXML addXML = (AddXML) change;
                    for (Element el : addXML.getNewXML().getXml()) {
                        el.setNamespace(Namespace.NO_NAMESPACE);
                        String elAsString = new XMLOutputter().outputString(el);
                        log.debug(elAsString);
                        ModelTransformationUtils.addXMLelement(doc, elAsString,
                                addXML.getTargetXPath().getTargetAsString(),
                                xpath);
                    }
                } else if (change.getChangeKind().equals(
                        SEDMLTags.CHANGE_XML_KIND)) {
                    ChangeXML changeXML = (ChangeXML) change;
                    ModelTransformationUtils.changeXMLElement(doc, changeXML
                            .getNewXML(), changeXML.getTargetXPath()
                            .getTargetAsString(), xpath);

                }
            }
            xmlString = ModelTransformationUtils.exportChangedXMLAsString(doc);
        } catch (Exception e) {
            throw new XMLException("Error generating new model"
                    + e.getMessage(), e);
        }
        return xmlString;
    }

    private org.jdom.Document createDocument(String sedml) throws XMLException {
        return new XMLUtils().readDoc(sedml);
    }

    /**
     * Boolean test for whether XPath expressions can be mapped to XML
     * namespaces in the model XML.
     * 
     * @param modelID
     * @param originalModel
     * @return <code>true</code> if can be mapped, false otherwise.
     * @throws XMLException
     */
    public boolean canResolveXPathExpressions(String modelID,
            final String originalModel) throws XMLException {
        org.jdom.Document doc = createDocument(originalModel);
        Model model = sedml.getModelWithId(modelID);
        List<Change> changes = model.getListOfChanges();
        for (Change change : changes) {
            XPathTarget target = change.getTargetXPath();
            NamespaceContextHelper nc = new NamespaceContextHelper(doc);
            nc.process(target);
            if (!nc.isAllXPathPrefixesMapped(target)) {
                return false;
            }
        }
        return true;
    }
}
