/*
 * Copyright (C) 1999-2011 University of Connecticut Health Center
 *
 * Licensed under the MIT License (the "License").
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *  http://www.opensource.org/licenses/mit-license.php
 */

package cbit.vcell.math;

import cbit.vcell.geometry.Geometry;
import cbit.vcell.geometry.GeometryOwner;
import cbit.vcell.geometry.SubVolume;
import cbit.vcell.geometry.surface.GeometricRegion;
import cbit.vcell.geometry.surface.SurfaceGeometricRegion;
import cbit.vcell.geometry.surface.VolumeGeometricRegion;
import cbit.vcell.mapping.AbstractMathMapping;
import cbit.vcell.math.MathCompareResults.Decision;
import cbit.vcell.math.ParticleObservable.ObservableType;
import cbit.vcell.math.PdeEquation.BoundaryConditionValue;
import cbit.vcell.math.SubDomain.BoundaryConditionSpec;
import cbit.vcell.math.Variable.Domain;
import cbit.vcell.model.Model.ReservedSymbol;
import cbit.vcell.model.common.VCellErrorMessages;
import cbit.vcell.parser.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.vcell.util.*;
import org.vcell.util.Issue.IssueCategory;
import org.vcell.util.Issue.IssueSource;
import org.vcell.util.Token;
import org.vcell.util.IssueContext.ContextType;
import org.vcell.util.document.BioModelChildSummary.MathType;
import org.vcell.util.document.KeyValue;
import org.vcell.util.document.Version;
import org.vcell.util.document.Versionable;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import java.io.IOException;
import java.io.Serializable;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * This class was generated by a SmartGuide.
 */
@SuppressWarnings("serial")
public class MathDescription implements Versionable, Matchable, SymbolTable, Serializable, ProblemRequirements, IssueSource {

    public SourceSymbolMapping getSourceSymbolMapping(){
        return sourceSymbolMapping;
    }

    public void setSourceSymbolMapping(SourceSymbolMapping sourceSymbolMapping){
        this.sourceSymbolMapping = sourceSymbolMapping;
    }

    private transient SourceSymbolMapping sourceSymbolMapping;
    private final static Logger lg = LogManager.getLogger(MathDescription.class);

    public static final TreeMap<Long, TreeSet<String>> originalHasLowPrecisionConstants = new TreeMap<>();
    public final static String MATH_FUNC_INIT_SUFFIX_PREFIX = "_init";
    public final static String MATH_FUNC_STOCH_INIT_SUFFIX_PREFIX = "_initCount";

    protected transient java.util.ArrayList<ChangeListener> aChangeListener = null;
    private Version version = null;
    private ArrayList<SubDomain> subDomainList = new ArrayList<SubDomain>();
    private ArrayList<Variable> variableList = new ArrayList<Variable>();
    private HashMap<String, Variable> variableHashTable = new HashMap<String, Variable>();
    private Geometry geometry = null;
    private java.lang.String fieldName = new String("NoName");
    protected transient java.beans.VetoableChangeSupport vetoPropertyChange;
    protected transient java.beans.PropertyChangeSupport propertyChange;
    private java.lang.String fieldDescription = new String();
    private transient java.lang.String fieldWarning = null;
    private final PostProcessingBlock postProcessingBlock = new PostProcessingBlock(this);

    private ArrayList<Event> eventList = new ArrayList<Event>();

    private boolean bRegionSizeFunctionsUsed = false;

    /**
     * {@link #getVCML()} will include comments
     */
    private boolean commenting = false;

    private static final char NEWLINE_CHAR = '\n';

    private ArrayList<ParticleMolecularType> particleMolecularTypes = new ArrayList<ParticleMolecularType>();

    /**
     * MathDescription constructor comment.
     */
    public MathDescription(Version argVersion, SourceSymbolMapping mathSymbolMapping){
        super();
        this.version = argVersion;
        if(argVersion != null){
            this.fieldName = argVersion.getName();
            this.fieldDescription = argVersion.getAnnot();
        }
        this.sourceSymbolMapping = mathSymbolMapping;
    }

    public MathDescription(Version argVersion){
        this(argVersion, null);
    }


    /**
     * Insert the method's description here.
     * Creation date: (9/2/2004 3:05:54 PM)
     *
     * @param mathDescription cbit.vcell.math.MathDescription
     */
    public MathDescription(MathDescription mathDescription){
        this.fieldName = mathDescription.getName();
        this.fieldDescription = mathDescription.getDescription();
        this.version = mathDescription.getVersion();
        this.geometry = mathDescription.getGeometry();
        try {
            read_database(new CommentStringTokenizer(mathDescription.getVCML_database()));
        } catch(MathException e){
            throw new RuntimeException(e.getMessage(), e);
        }
    }


    public MathDescription(String name, SourceSymbolMapping mathSymbolMapping){
        this.fieldName = name;
        this.version = null;
        this.sourceSymbolMapping = mathSymbolMapping;
    }

    public MathDescription(String name){
        this(name, null);
    }

    /**
     * Add a javax.swing.event.ChangeListener.
     */
    public void addChangeListener(ChangeListener newListener){
        if(aChangeListener == null){
            aChangeListener = new java.util.ArrayList<ChangeListener>();
        }
        ;
        if(!aChangeListener.contains(newListener)){
            aChangeListener.add(newListener);
        }
    }


    /**
     * The addPropertyChangeListener method was generated to support the propertyChange field.
     */
    public synchronized void addPropertyChangeListener(java.beans.PropertyChangeListener listener){
        getPropertyChange().addPropertyChangeListener(listener);
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @param subDomain cbit.vcell.math.SubDomain
     * @throws java.lang.Exception The exception description.
     */
    public void addSubDomain(SubDomain subDomain) throws MathException{
        addSubDomain0(subDomain);
        fireStateChanged();
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @param subDomain cbit.vcell.math.SubDomain
     * @throws java.lang.Exception The exception description.
     */
    private void addSubDomain0(SubDomain subDomain) throws MathException{
        if(subDomain == null){
            throw new MathException("subdomain is null");
        }
        if(subDomain instanceof CompartmentSubDomain){
            if(getCompartmentSubDomain(subDomain.getName()) != null){
                throw new MathException("subDomain " + subDomain.getName() + " already exists");
            }
        } else if(subDomain instanceof MembraneSubDomain){
            if(getMembraneSubDomain(subDomain.getName()) != null){
                throw new MathException("subDomain " + subDomain.getName() + " already exists");
            }
        } else if(subDomain instanceof FilamentSubDomain){
            if(getSubDomain(subDomain.getName()) != null){
                throw new MathException("subDomain " + subDomain.getName() + " already exists");
            }
        }

        subDomainList.add(subDomain);
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @param var cbit.vcell.math.Variable
     */
    public void addVariable(Variable var) throws MathException, ExpressionBindingException{
        addVariable0(var);
        fireStateChanged();
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @param var cbit.vcell.math.Variable
     */
    private void addVariable0(Variable var) throws MathException, ExpressionBindingException{
        if(getVariable(var.getName()) != null){
            throw new MathException("variable " + var.getName() + " already exists");
        }
        variableList.add(var);
        variableHashTable.put(var.getName(), var);
        //var.bind(this);
        if(var instanceof VolVariable){
            //
            // for Volume Variables, also create an InsideVariable and an OutsideVariable for use in JumpConditions
            //
            InsideVariable inVar = new InsideVariable(var.getName() + InsideVariable.INSIDE_VARIABLE_SUFFIX, var.getName());
            variableList.add(inVar);
            variableHashTable.put(inVar.getName(), inVar);
            inVar.bind(this);
            OutsideVariable outVar = new OutsideVariable(var.getName() + OutsideVariable.OUTSIDE_VARIABLE_SUFFIX, var.getName());
            variableList.add(outVar);
            variableHashTable.put(outVar.getName(), outVar);
            outVar.bind(this);
        }
    }


    /**
     * The addVetoableChangeListener method was generated to support the vetoPropertyChange field.
     */
    public synchronized void addVetoableChangeListener(java.beans.VetoableChangeListener listener){
        getVetoPropertyChange().addVetoableChangeListener(listener);
    }


    /**
     * This method was created by a SmartGuide.
     */
    private void clearAll(){
        subDomainList.clear();
        variableList.clear();
        variableHashTable.clear();
        eventList.clear();
    }


    /**
     * Insert the method's description here.
     * Creation date: (4/24/2003 3:32:01 PM)
     */
    public void clearVersion(){
        version = null;
    }


    public static void updateReservedSymbols(MathDescription updateThis, ReservedSymbol[] reservedSymbols){
        //Code to make ServerDocumentManager.saveBioModel(...) ignore old LowPrecisionMathConstants that were converted to HighPrecisionMathConstants
        //
        for(int i = 0; i < updateThis.variableList.size(); i++){
            //Deal with Model.reservedConstantsMap
            for(int j = 0; j < reservedSymbols.length; j++){
                if(reservedSymbols[j].getName().equals(updateThis.variableList.get(i).getName()) && reservedSymbols[j].getExpression() != null){
                    //System.out.println("--Found "+updateThis.variableList.get(i)+" "+reservedSymbols[j].getName()+" "+reservedSymbols[j].getExpression().infix());
                    updateThis.variableList.get(i).getExpression().substituteInPlace(updateThis.variableList.get(i).getExpression(), reservedSymbols[j].getExpression());
                    break;
                }
            }
            //Deal with VCUnitDefinition.getDimensionlessScale().molecules_per_uM_um3
            if(updateThis.variableList.get(i) instanceof Function){
                Function f = (Function) updateThis.variableList.get(i);
                String expr = (f.getExpression() != null ? f.getExpression().infix() : null);
                try {
                    if(expr != null && (expr.equals("(602.0 / 1.0)"))){
                        f.getExpression().substituteInPlace(f.getExpression(), new Expression("(" + ExpressionUtils.value_molecules_per_uM_um3_NUMERATOR + " / 1000000.0)"));
                    } else if(expr != null && (expr.equals("(1.0 / 602.0)"))){
                        f.getExpression().substituteInPlace(f.getExpression(), new Expression("(1000000.0 / " + ExpressionUtils.value_molecules_per_uM_um3_NUMERATOR + ")"));
                    }
                } catch(ExpressionException e){
                    lg.error(e);
                }
            }
        }
    }

    public boolean compareEqual(Matchable object){

        MathDescription mathDesc = null;
        if(object == null){
            return false;
        }
        if(!(object instanceof MathDescription)){
            return false;
        } else {
            mathDesc = (MathDescription) object;
        }

        if(!Compare.isEqual(getName(), mathDesc.getName())){
            return false;
        }
        if(!Compare.isEqual(getDescription(), mathDesc.getDescription())){
            return false;
        }
        //
        // compare subdomains
        //
        if(!Compare.isEqual(subDomainList, mathDesc.subDomainList)){
            return false;
        }
        //
        // compare variables
        //
        if(!Compare.isEqual(variableList, mathDesc.variableList)){
            return false;
        }
        //
        // compare geometry
        //
        if(!Compare.isEqualOrNull(geometry, mathDesc.geometry)){
            return false;
        }

        //
        // compare events
        //
        if(!Compare.isEqual(eventList, mathDesc.eventList)){
            return false;
        }
        //
        // compare particleMolecularTypes
        //
        if(!Compare.isEqual(particleMolecularTypes, mathDesc.particleMolecularTypes)){
            return false;
        }

        return Compare.isEqualOrNull(postProcessingBlock, mathDesc.postProcessingBlock);
    }


    private MathCompareResults compareEquivalentCanonicalMath(MathDescription newMathDesc){
        try {
            MathDescription oldMathDesc = this;
            if(oldMathDesc.compareEqual(newMathDesc)){
                return new MathCompareResults(Decision.MathEquivalent_FLATTENED);
            } else {
                //if (!bSilent) System.out.println("------NATIVE MATHS ARE DIFFERENT----------------------");
                if(!oldMathDesc.postProcessingBlock.compareEqual(newMathDesc.postProcessingBlock)){
                    logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_PostProcessingBlock, "");
                    return new MathCompareResults(Decision.MathDifferent_DIFFERENT_PostProcessingBlock, "Post processing block does not match");
                }
                List<Variable> oldVarList = Collections.list(oldMathDesc.getVariables());
                List<Variable> newVarList = Collections.list(newMathDesc.getVariables());
                oldVarList.sort(Comparator.comparing(Variable::getName));
                newVarList.sort(Comparator.comparing(Variable::getName));
                Variable[] oldVars = oldVarList.toArray(Variable[]::new);
                Variable[] newVars = newVarList.toArray(Variable[]::new);


                if(oldVars.length != newVars.length){
                    //
                    // number of state variables are not equal (canonical maths only have state variables)
                    //
                    String msg = oldVars.length + " vs " + newVars.length;
                    logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_NUMBER_OF_VARIABLES, msg);
                    return new MathCompareResults(Decision.MathDifferent_DIFFERENT_NUMBER_OF_VARIABLES, msg);
                }
                if(!Compare.isEqual(oldVars, newVars)){
                    //
                    // variables are not strictly equal - must try to ignore domains and try again
                    //
                    boolean bIgnoreMissingDomains = true;
                    for(Variable oldVar : oldVars){
                        boolean bFound = false;
                        for(Variable newVar : newVars){
                            if(oldVar.compareEqual(newVar, bIgnoreMissingDomains)){
                                bFound = true;
                                break;
                            }
                        }
                        if(!bFound){
                            //
                            // variable names are not equivalent (nothing much we can do) ... what about change of variables???
                            //
                            String msg = "variable '" + oldVar.getQualifiedName() + "' not matched";
                            logMathTexts(this, newMathDesc, Decision.MathDifferent_VARIABLES_DONT_MATCH, msg);
                            return new MathCompareResults(Decision.MathDifferent_VARIABLES_DONT_MATCH, msg);
                        }
                    }
                    for(Variable newVar : newVars){
                        boolean bFound = false;
                        for(Variable oldVar : oldVars){
                            if(newVar.compareEqual(oldVar, bIgnoreMissingDomains)){
                                bFound = true;
                                break;
                            }
                        }
                        if(!bFound){
                            //
                            // variable names are not equivalent (nothing much we can do) ... what about change of variables???
                            //
                            String msg = "variable '" + newVar.getQualifiedName() + "' not matched";
                            logMathTexts(this, newMathDesc, Decision.MathDifferent_VARIABLES_DONT_MATCH, msg);
                            return new MathCompareResults(Decision.MathDifferent_VARIABLES_DONT_MATCH, msg);
                        }
                    }
                }
                //
                // go through the list of SubDomains, and compare equations one by one and "correct" new one if possible
                //
                List<SubDomain> subDomainsOld = Collections.list(oldMathDesc.getSubDomains());
                List<SubDomain> subDomainsNew = Collections.list(newMathDesc.getSubDomains());
                Collections.sort(subDomainsOld);
                Collections.sort(subDomainsNew);

                if(subDomainsOld.size() != subDomainsNew.size()){
                    String msg = subDomainsOld.size() + " vs " + subDomainsNew.size();
                    logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_NUMBER_OF_SUBDOMAINS, msg);
                    return new MathCompareResults(Decision.MathDifferent_DIFFERENT_NUMBER_OF_SUBDOMAINS, msg);
                }
                for(int i = 0; i < subDomainsOld.size(); i++){
                    // compare boundary type
                    if(getGeometry().getDimension() > 0){
                        if(subDomainsOld.get(i) instanceof CompartmentSubDomain csdOld && subDomainsNew.get(i) instanceof CompartmentSubDomain csdNew){
                            if(!Compare.isEqualOrNull(csdOld.getBoundaryConditionXm(), csdNew.getBoundaryConditionXm())
                                    || !Compare.isEqualOrNull(csdOld.getBoundaryConditionXp(), csdNew.getBoundaryConditionXp())
                                    || !Compare.isEqualOrNull(csdOld.getBoundaryConditionYm(), csdNew.getBoundaryConditionYm())
                                    || !Compare.isEqualOrNull(csdOld.getBoundaryConditionYp(), csdNew.getBoundaryConditionYp())
                                    || !Compare.isEqualOrNull(csdOld.getBoundaryConditionZm(), csdNew.getBoundaryConditionZm())
                                    || !Compare.isEqualOrNull(csdOld.getBoundaryConditionZp(), csdNew.getBoundaryConditionZp())
                            ){
                                logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_BC_TYPE, "");
                                return new MathCompareResults(Decision.MathDifferent_DIFFERENT_BC_TYPE);
                            }
                        } else if(subDomainsOld.get(i) instanceof MembraneSubDomain msdOld && subDomainsNew.get(i) instanceof MembraneSubDomain msdNew){
                            if(!Compare.isEqualOrNull(msdOld.getBoundaryConditionXm(), msdNew.getBoundaryConditionXm())
                                    || !Compare.isEqualOrNull(msdOld.getBoundaryConditionXp(), msdNew.getBoundaryConditionXp())
                                    || !Compare.isEqualOrNull(msdOld.getBoundaryConditionYm(), msdNew.getBoundaryConditionYm())
                                    || !Compare.isEqualOrNull(msdOld.getBoundaryConditionYp(), msdNew.getBoundaryConditionYp())
                                    || !Compare.isEqualOrNull(msdOld.getBoundaryConditionZm(), msdNew.getBoundaryConditionZm())
                                    || !Compare.isEqualOrNull(msdOld.getBoundaryConditionZp(), msdNew.getBoundaryConditionZp())
                            ){
                                logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_BC_TYPE, "");
                                return new MathCompareResults(Decision.MathDifferent_DIFFERENT_BC_TYPE);
                            }
                            if(!compareUpdate(msdNew.getVelocityX(), msdOld.getVelocityX(), msdNew::setVelocityX)){
                                logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_VELOCITY, "x");
                                return new MathCompareResults(Decision.MathDifferent_DIFFERENT_VELOCITY, "x");
                            }
                            if(!compareUpdate(msdNew.getVelocityY(), msdOld.getVelocityY(), msdNew::setVelocityY)){
                                logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_VELOCITY, "y");
                                return new MathCompareResults(Decision.MathDifferent_DIFFERENT_VELOCITY, "y");
                            }
                        }
                        // apply standard, implicit VCell defaults for boundary conditions ...
                        //    1) for Neumann, 'null' boundary condition expression is same as zero flux (set expression to 0.0)
                        //    2) for Dirichlet, 'null' boundary condition expression is same as setting value to initial condition (set expression to 'init expression')
                        if(subDomainsOld.get(i) instanceof SubDomain.DomainWithBoundaryConditions){
                            SubDomain.DomainWithBoundaryConditions oldSubdomainWithBC = (SubDomain.DomainWithBoundaryConditions) subDomainsOld.get(i);
                            setMissingEquationBoundaryConditionsToDefault(oldSubdomainWithBC, geometry.getDimension());
                        }
                        if(subDomainsNew.get(i) instanceof SubDomain.DomainWithBoundaryConditions){
                            SubDomain.DomainWithBoundaryConditions newSubdomainWithBCs = (SubDomain.DomainWithBoundaryConditions) subDomainsNew.get(i);
                            setMissingEquationBoundaryConditionsToDefault(newSubdomainWithBCs, geometry.getDimension());
                        }
                    }

                    for(int j = 0; j < oldVars.length; j++){
                        //
                        // test equation for this subdomain and variable
                        //
                        {
                            Equation oldEqu = subDomainsOld.get(i).getEquation(oldVars[j]);
                            Equation newEqu = subDomainsNew.get(i).getEquation(newVars[j]);
                            if(!Compare.isEqualOrNull(oldEqu, newEqu)){
                                boolean bFoundDifference = false;
                                //
                                // equation didn't compare exactly, lets try to evaluate some instead
                                //
                                if(oldEqu == null){
                                    //
                                    // only one MathDescription had Equation for this Variable.
                                    //
                                    String msg = "only one mathDescription had equation for '" + oldVars[j].getQualifiedName() + "' in SubDomain '" + subDomainsOld.get(i).getName() + "'";
                                    logMathTexts(this, newMathDesc, Decision.MathDifferent_EQUATION_ADDED, msg);
                                    return new MathCompareResults(Decision.MathDifferent_EQUATION_ADDED, msg);
                                }
                                if(newEqu == null){
                                    //
                                    // only one MathDescription had Equation for this Variable.
                                    //
                                    String msg = "only one mathDescription had equation for '" + oldVars[j].getQualifiedName() + "' in SubDomain '" + subDomainsOld.get(i).getName() + "'";
                                    logMathTexts(this, newMathDesc, Decision.MathDifferent_EQUATION_REMOVED, msg);
                                    return new MathCompareResults(Decision.MathDifferent_EQUATION_REMOVED, msg);

                                }
                                ArrayList<Expression> oldExps = new ArrayList<>();
                                ArrayList<Expression> newExps = new ArrayList<>();
                                boolean bOdePdeMismatch = false;
                                if(oldEqu instanceof PdeEquation && newEqu instanceof OdeEquation && oldEqu.getExpressions(newMathDesc).size() == 3 && ((PdeEquation) oldEqu).getDiffusionExpression().isZero()){
                                    oldExps.add(oldEqu.getRateExpression());
                                    oldExps.add(oldEqu.getInitialExpression());
                                    newExps.add(newEqu.getRateExpression());
                                    newExps.add(newEqu.getInitialExpression());
                                    bOdePdeMismatch = true;
                                } else if(oldEqu instanceof OdeEquation && newEqu instanceof PdeEquation && newEqu.getExpressions(newMathDesc).size() == 3 && ((PdeEquation) newEqu).getDiffusionExpression().isZero()){
                                    oldExps.add(oldEqu.getRateExpression());
                                    oldExps.add(oldEqu.getInitialExpression());
                                    newExps.add(newEqu.getRateExpression());
                                    newExps.add(newEqu.getInitialExpression());
                                    bOdePdeMismatch = true;
                                } else {
                                    oldExps.addAll(oldEqu.getExpressions(oldMathDesc));
                                    newExps.addAll(newEqu.getExpressions(newMathDesc));
                                }
                                if(oldExps.size() != newExps.size()){
                                    String msg = "equations have different number of expressions";
                                    logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_NUMBER_OF_EXPRESSIONS, msg);
                                    return new MathCompareResults(Decision.MathDifferent_DIFFERENT_NUMBER_OF_EXPRESSIONS, msg);
                                }
                                for(int k = 0; k < oldExps.size(); k++){
                                    if(!oldExps.get(k).compareEqual(newExps.get(k))){
                                        bFoundDifference = true;
                                        if(!ExpressionUtils.functionallyEquivalent(oldExps.get(k), newExps.get(k))){
                                            Expression oe = new Expression(oldExps.get(k));
                                            Expression ne = new Expression(newExps.get(k));
                                            MathUtilities.substituteCommonDiscontinuitiesInPlace(oe, ne, "BOOLEAN_");
                                            if(!ExpressionUtils.functionallyEquivalent(oe, ne, false)){
                                                //
                                                // difference couldn't be reconciled
                                                //
                                                String msg = "expressions are different: '" + oldExps.get(k).infix() + "' vs '" + newExps.get(k).infix() + "'";
                                                logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_EXPRESSION, msg);
                                                return new MathCompareResults(Decision.MathDifferent_DIFFERENT_EXPRESSION, msg);
                                            }
                                        } else {
                                            //if (!bSilent) System.out.println("expressions are equivalent Old: '"+oldExps[k]+"'\n"+
                                            //"expressions are equivalent New: '"+newExps[k]+"'");
                                        }
                                    }
                                }
                                if(!oldEqu.getVariable().compareEqual(newEqu.getVariable())){
                                    bFoundDifference = true;
                                    boolean bIgnoreMissingDomains = true;
                                    if(!oldEqu.getVariable().compareEqual(newEqu.getVariable(), bIgnoreMissingDomains)){
                                        String msg = "var1='" + oldEqu.getVariable().getQualifiedName() + "', var2='" + newEqu.getVariable().getQualifiedName() + "'";
                                        logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_VARIABLE_IN_EQUATION, msg);
                                        return new MathCompareResults(Decision.MathDifferent_DIFFERENT_VARIABLE_IN_EQUATION, msg);
                                    }
                                }
                                boolean bPdeDimensionFilteringDifferent = false;
                                if(oldEqu instanceof PdeEquation && newEqu instanceof PdeEquation){
                                    PdeEquation oldPde = (PdeEquation) oldEqu;
                                    PdeEquation newPde = (PdeEquation) newEqu;
                                    bPdeDimensionFilteringDifferent |= oldPde.getBoundaryYm() == null ^ newPde.getBoundaryYm() == null;
                                    bPdeDimensionFilteringDifferent |= oldPde.getBoundaryYp() == null ^ newPde.getBoundaryYp() == null;
                                    bPdeDimensionFilteringDifferent |= oldPde.getBoundaryZm() == null ^ newPde.getBoundaryZm() == null;
                                    bPdeDimensionFilteringDifferent |= oldPde.getBoundaryZp() == null ^ newPde.getBoundaryZp() == null;
                                    bPdeDimensionFilteringDifferent |= oldPde.getGradientY() == null ^ newPde.getGradientY() == null;
                                    bPdeDimensionFilteringDifferent |= oldPde.getGradientZ() == null ^ newPde.getGradientZ() == null;
                                    bPdeDimensionFilteringDifferent |= oldPde.getVelocityY() == null ^ newPde.getVelocityY() == null;
                                    bPdeDimensionFilteringDifferent |= oldPde.getVelocityZ() == null ^ newPde.getVelocityZ() == null;
                                }
                                //
                                // equation was not strictly "equal" but passed all tests, replace with old equation and move on
                                //
                                if(bFoundDifference || bOdePdeMismatch || bPdeDimensionFilteringDifferent){
                                    subDomainsNew.get(i).replaceEquation(oldEqu);
                                } else {
                                    //
                                    // couldn't find the smoking gun, just plain bad
                                    //
                                    String msg = "couldn't find problem with equation for " + oldVars[j].getName() + " in compartment " + subDomainsOld.get(i).getName();
                                    logMathTexts(this, newMathDesc, Decision.MathDifferent_UNKNOWN_DIFFERENCE_IN_EQUATION, msg);
                                    return new MathCompareResults(Decision.MathDifferent_UNKNOWN_DIFFERENCE_IN_EQUATION, msg);
                                }
                            }
                        }

                        {
                            ParticleProperties oldPP = subDomainsOld.get(i).getParticleProperties(oldVars[j]);
                            ParticleProperties newPP = subDomainsNew.get(i).getParticleProperties(newVars[j]);
                            if(!Compare.isEqualOrNull(oldPP, newPP)){
                                logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_PARTICLE_PROPERTIES, "");
                                return new MathCompareResults(Decision.MathDifferent_DIFFERENT_PARTICLE_PROPERTIES);
                            }
                        }
                        //
                        // if a membrane, test jumpCondition for this subdomain and variable
                        //
                        if(subDomainsOld.get(i) instanceof MembraneSubDomain && oldVars[j] instanceof VolVariable){
                            JumpCondition oldJumpCondition = ((MembraneSubDomain) subDomainsOld.get(i)).getJumpCondition((VolVariable) oldVars[j]);
                            JumpCondition newJumpCondition = ((MembraneSubDomain) subDomainsNew.get(i)).getJumpCondition((VolVariable) newVars[j]);
                            if(!Compare.isEqualOrNull(oldJumpCondition, newJumpCondition)){
                                boolean bFoundDifference = false;
                                //
                                // equation didn't compare exactly, lets try to evaluate some instead
                                //
                                if(oldJumpCondition == null){
                                    //
                                    // only one MathDescription had Equation for this Variable.
                                    //
                                    logMathTexts(this, newMathDesc, Decision.MathDifferent_EQUATION_ADDED, "oldJumpCondition is null");
                                    return new MathCompareResults(Decision.MathDifferent_EQUATION_ADDED);
                                }
                                if(newJumpCondition == null){
                                    //
                                    // only one MathDescription had Equation for this Variable.
                                    //
                                    logMathTexts(this, newMathDesc, Decision.MathDifferent_EQUATION_REMOVED, "newJumpCondition is null");
                                    return new MathCompareResults(Decision.MathDifferent_EQUATION_REMOVED);
                                }
                                final Vector<Expression> oldJC = oldJumpCondition.getExpressions(oldMathDesc);
                                Expression oldExps[] = oldJC.toArray(new Expression[oldJC.size()]);
                                final Vector<Expression> newJC = newJumpCondition.getExpressions(newMathDesc);
                                Expression newExps[] = newJC.toArray(new Expression[newJC.size()]);
                                if(oldExps.length != newExps.length){
                                    String msg = "jump condition has different number of expressions";
                                    logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_NUMBER_OF_EXPRESSIONS, msg);
                                    return new MathCompareResults(Decision.MathDifferent_DIFFERENT_NUMBER_OF_EXPRESSIONS, msg);
                                }
                                for(int k = 0; k < oldExps.length; k++){
                                    if(!oldExps[k].compareEqual(newExps[k])){
                                        bFoundDifference = true;
                                        if(!ExpressionUtils.functionallyEquivalent(oldExps[k], newExps[k])){
                                            Expression oe = new Expression(oldExps[k]);
                                            Expression ne = new Expression(newExps[k]);
                                            MathUtilities.substituteCommonDiscontinuitiesInPlace(oe, ne, "BOOLEAN_");
                                            if(!ExpressionUtils.functionallyEquivalent(oe, ne, false)){
                                                //
                                                // difference couldn't be reconciled
                                                //
                                                String msg = "expressions are different: '" + oldExps[k].infix() + "' vs '"
                                                        + newExps[k].infix() + "'";
                                                logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_EXPRESSION,
                                                        msg);
                                                return new MathCompareResults(Decision.MathDifferent_DIFFERENT_EXPRESSION,
                                                        msg);
                                            }
                                        } else {
                                            //if (!bSilent) System.out.println("expressions are equivalent Old: '"+oldExps[k]+"'\n"+
                                            //"expressions are equivalent New: '"+newExps[k]+"'");
                                        }
                                    }
                                }
                                //
                                // equation was not strictly "equal" but passed all tests, replace with old equation and move on
                                //
                                if(bFoundDifference){
                                    ((MembraneSubDomain) subDomainsNew.get(i)).replaceJumpCondition(oldJumpCondition);
                                } else {
                                    //
                                    // couldn't find the smoking gun, just plain bad
                                    //
                                    String msg = "couldn't find problem with jumpCondition for " + oldVars[j].getName() + " in compartment " + subDomainsOld.get(i).getName();
                                    logMathTexts(this, newMathDesc, Decision.MathDifferent_UNKNOWN_DIFFERENCE_IN_EQUATION, msg);
                                    return new MathCompareResults(Decision.MathDifferent_UNKNOWN_DIFFERENCE_IN_EQUATION, msg);
                                }
                            }
                        }
                    }
                    //
                    // test fast system for subdomain
                    //
                    FastSystem oldFastSystem = subDomainsOld.get(i).getFastSystem();
                    FastSystem newFastSystem = subDomainsNew.get(i).getFastSystem();
                    if(!Compare.isEqualOrNull(oldFastSystem, newFastSystem)){
                        boolean bFoundDifference = false;
                        //
                        // fastSystems didn't compare exactly, lets try to evaluate some expressions instead
                        //
                        if(oldFastSystem == null){
                            //
                            // only one MathDescription had Equation for this Variable.
                            //
                            logMathTexts(this, newMathDesc, Decision.MathDifferent_EQUATION_ADDED, "oldFastSystem is null");
                            return new MathCompareResults(Decision.MathDifferent_EQUATION_ADDED);
                        }
                        if(newFastSystem == null){
                            //
                            // only one MathDescription had Equation for this Variable.
                            //
                            logMathTexts(this, newMathDesc, Decision.MathDifferent_EQUATION_REMOVED, "newFastSystem is null");
                            return new MathCompareResults(Decision.MathDifferent_EQUATION_REMOVED);
                        }
                        Enumeration<Expression> oldFastInvExpEnum = oldFastSystem.getFastInvariantExpressions();
                        Enumeration<Expression> newFastInvExpEnum = newFastSystem.getFastInvariantExpressions();
                        if(oldFastInvExpEnum == null || newFastInvExpEnum == null) throw new NullPointerException();
                        List<Expression> oldFastInvariantExpsList = Collections.list(oldFastInvExpEnum);
                        List<Expression> newFastInvariantExpsList = Collections.list(newFastInvExpEnum);

                        // remove the trivial fast invariant expressions (those with less than 2 variables in them)
                        Expression[] oldFastInvariantExps = oldFastInvariantExpsList.stream().filter(fiExp -> fiExp.getSymbols() != null && fiExp.getSymbols().length > 1).toArray(Expression[]::new);
                        if(oldFastInvariantExps.length != oldFastSystem.getNumFastInvariants()){
                            bFoundDifference = true;
                        }
                        Expression[] newFastInvariantExps = newFastInvariantExpsList.stream().filter(fiExp -> fiExp.getSymbols() != null && fiExp.getSymbols().length > 1).toArray(Expression[]::new);
                        if(newFastInvariantExps.length != newFastSystem.getNumFastInvariants()){
                            bFoundDifference = true;
                        }
                        if(oldFastInvariantExps.length != newFastInvariantExps.length){
                            String msg = "fast invariants have different number of expressions";
                            logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_NUMBER_OF_EXPRESSIONS, msg);
                            return new MathCompareResults(Decision.MathDifferent_DIFFERENT_NUMBER_OF_EXPRESSIONS, msg);
                        }
                        for(int kk = 0; kk < oldFastInvariantExps.length; kk++){
                            // check same index first using simple compare
                            if(oldFastInvariantExps[kk].compareEqual(newFastInvariantExps[kk])){
                                continue;
                            } else {
                                bFoundDifference = true;
                            }
                            //
                            // check all fast invariant expressions because they may be out of order
                            //
                            boolean bFoundMatch = false;
                            for(Expression newFastInvariantExp : newFastInvariantExps){
                                if(oldFastInvariantExps[kk].compareEqual(newFastInvariantExp)){
                                    bFoundMatch = true;
                                    break;
                                } else {
                                    bFoundDifference = true;
                                    if(ExpressionUtils.functionallyEquivalent(oldFastInvariantExps[kk], newFastInvariantExp)){
                                        bFoundMatch = true;
                                        break;
                                    } else {
                                        if(MathUtilities.compareEquivalent(new FastInvariant(oldFastInvariantExps[kk]), new FastInvariant(newFastInvariantExp))){
                                            bFoundMatch = true;
                                            break;
                                        }
                                    }
                                }
                            }
                            if(!bFoundMatch){
                                String msg = "could not find a match for fast invariant expression'" + oldFastInvariantExps[kk] + "'";
                                logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_FASTINV_EXPRESSION, msg);
                                return new MathCompareResults(Decision.MathDifferent_DIFFERENT_FASTINV_EXPRESSION, msg);
                            }
                        }
                        Enumeration<Expression> oldFastRateExpEnum = oldFastSystem.getFastRateExpressions();
                        Enumeration<Expression> newFastRateExpEnum = newFastSystem.getFastRateExpressions();
                        if(oldFastRateExpEnum == null || newFastRateExpEnum == null) throw new NullPointerException();
                        Expression[] oldFastRateExps = Collections.list(oldFastRateExpEnum).toArray(Expression[]::new);
                        Expression[] newFastRateExps = Collections.list(newFastRateExpEnum).toArray(Expression[]::new);
                        if(oldFastRateExps.length != newFastRateExps.length){
                            String msg = "fast rates have different number of expressions";
                            logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_NUMBER_OF_EXPRESSIONS, msg);
                            return new MathCompareResults(Decision.MathDifferent_DIFFERENT_NUMBER_OF_EXPRESSIONS, msg);
                        }
                        for(int k = 0; k < oldFastRateExps.length; k++){
                            if(!oldFastRateExps[k].compareEqual(newFastRateExps[k])){
                                bFoundDifference = true;
                                if(!ExpressionUtils.functionallyEquivalent(oldFastRateExps[k], newFastRateExps[k])){
                                    //
                                    // difference couldn't be reconciled
                                    //
                                    String msg = "fast rate expressions are different Old: '" + oldFastRateExps[k] + "'\n" +
                                            "fast rate expressions are different New: '" + newFastRateExps[k] + "'";
                                    logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_FASTRATE_EXPRESSION, msg);
                                    return new MathCompareResults(Decision.MathDifferent_DIFFERENT_FASTRATE_EXPRESSION, msg);
                                } else {
                                    //if (!bSilent) System.out.println("expressions are equivalent Old: '"+oldExps[k]+"'\n"+
                                    //"expressions are equivalent New: '"+newExps[k]+"'");
                                }
                            }
                        }
                        //
                        // equation was not strictly "equal" but passed all tests, replace with old equation and move on
                        //
                        if(bFoundDifference){
                            subDomainsNew.get(i).setFastSystem(oldFastSystem);
                        } else {
                            //
                            // couldn't find the smoking gun, just plain bad
                            //
                            String msg = "couldn't find problem with FastSystem for compartment " + subDomainsOld.get(i).getName();
                            logMathTexts(this, newMathDesc, Decision.MathDifferent_UNKNOWN_DIFFERENCE_IN_EQUATION, msg);
                            return new MathCompareResults(Decision.MathDifferent_UNKNOWN_DIFFERENCE_IN_EQUATION, msg);
                        }
                    }
                    //
                    // 1) get list of old and new ParticleJumpProcesses
                    // 2) filter out 'no-op' PJPs (e.g. those only with trivial Actions such as open->open)
                    // 3) add trivial Symmetry factor (1.0) if missing
                    //
                    List<ParticleJumpProcess> oldPjpList = subDomainsOld.get(i).getParticleJumpProcesses().stream().filter(pjp -> !pjp.actionsNoop()).toList();
                    List<ParticleJumpProcess> newPjpList = subDomainsNew.get(i).getParticleJumpProcesses().stream().filter(pjp -> !pjp.actionsNoop()).toList();
                    ;
                    for(ParticleJumpProcess oldPjp : oldPjpList)
                        if(oldPjp.getProcessSymmetryFactor() == null)
                            oldPjp.setProcessSymmetryFactor(new ParticleJumpProcess.ProcessSymmetryFactor(1.0));
                    for(ParticleJumpProcess newPjp : newPjpList)
                        if(newPjp.getProcessSymmetryFactor() == null)
                            newPjp.setProcessSymmetryFactor(new ParticleJumpProcess.ProcessSymmetryFactor(1.0));

                    if(oldPjpList.size() != newPjpList.size()){
                        Set<String> oldPjpNameSet = oldPjpList.stream().map(ParticleJumpProcess::getName).collect(Collectors.toSet());
                        Set<String> newPjpNameSet = newPjpList.stream().map(ParticleJumpProcess::getName).collect(Collectors.toSet());
                        Set<String> removedPjpNames = new LinkedHashSet<>(oldPjpNameSet);
                        removedPjpNames.removeAll(newPjpNameSet);
                        Set<String> addedPjpNames = new LinkedHashSet<>(newPjpNameSet);
                        addedPjpNames.removeAll(oldPjpNameSet);
                        String msg = "removed PJPs=" + removedPjpNames + ", added PJPs=" + addedPjpNames;
                        logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_NUMBER_OF_PARTICLE_JUMP_PROCESS, msg);
                        return new MathCompareResults(Decision.MathDifferent_DIFFERENT_NUMBER_OF_PARTICLE_JUMP_PROCESS, msg);
                    }
                    for(ParticleJumpProcess oldPjp : oldPjpList){
                        boolean bEqual = false;
                        boolean bEqualAfterKMOLE = false;
                        boolean bSymmetryFactorDifferent = false;
                        ParticleJumpProcess matchingPjp = null;
                        for(ParticleJumpProcess newPjp : newPjpList){
                            if(Compare.isEqualOrNull(oldPjp.getName(), newPjp.getName())){
                                matchingPjp = newPjp;
                                bSymmetryFactorDifferent = oldPjp.getProcessSymmetryFactor().factor != matchingPjp.getProcessSymmetryFactor().factor;
                                double savedSymmetryFactor = matchingPjp.getProcessSymmetryFactor().factor;
                                try {
                                    if(bSymmetryFactorDifferent){
                                        matchingPjp.setProcessSymmetryFactor(new ParticleJumpProcess.ProcessSymmetryFactor(oldPjp.getProcessSymmetryFactor().factor));
                                    }
                                    if(oldPjp.compareEqual(newPjp)){
                                        bEqual = true;
                                    } else if(newPjp.getParticleRateDefinition() instanceof MacroscopicRateConstant){
                                        final double KMOLE_new = 1.0 / 602.214179;
                                        final double KMOLE_value_old = 1.0 / 602.0;
                                        MacroscopicRateConstant macroscopicRateConstant = (MacroscopicRateConstant) newPjp.getParticleRateDefinition();
                                        final int MAX_POWER_KMOLE = 2;
                                        Expression savedExp = new Expression(macroscopicRateConstant.getExpression());
                                        for(int pow = 1; pow <= MAX_POWER_KMOLE; pow++){
                                            Expression scaledRate = Expression.mult(macroscopicRateConstant.getExpression(), new Expression(KMOLE_value_old)).flattenSafe();
                                            macroscopicRateConstant.getExpression().substituteInPlace(macroscopicRateConstant.getExpression(), scaledRate);
                                            if(oldPjp.compareEqual(newPjp)){
                                                bEqualAfterKMOLE = true;
                                                break;
                                            }
                                        }
                                        macroscopicRateConstant.setExpression(savedExp);
                                    }
                                } finally {
                                    if(bSymmetryFactorDifferent){
                                        matchingPjp.setProcessSymmetryFactor(new ParticleJumpProcess.ProcessSymmetryFactor(savedSymmetryFactor));
                                    }
                                }
                                break;
                            }
                        }
                        if(bEqualAfterKMOLE && !bSymmetryFactorDifferent){
                            String msg = "PJP='" + oldPjp.getName() + "', " +
                                    "old='" + Arrays.asList(oldPjp.getParticleRateDefinition().getExpressions()).stream().map(e -> e.infix()).collect(Collectors.toList()) + "', " +
                                    "new='" + ((matchingPjp == null) ? "null" : Arrays.asList(matchingPjp.getParticleRateDefinition().getExpressions()).stream().map(e -> e.infix()).collect(Collectors.toList())) + "'";
                            logMathTexts(this, newMathDesc, Decision.MathDifferent_LEGACY_RATE_PARTICLE_JUMP_PROCESS, msg);
                            return new MathCompareResults(Decision.MathDifferent_LEGACY_RATE_PARTICLE_JUMP_PROCESS, msg);
                        }
                        if((bEqual || bEqualAfterKMOLE) && bSymmetryFactorDifferent){
                            String msg = "PJP='" + oldPjp.getName() + "', " +
                                    "ProcessSymmetryFactor: old='" + oldPjp.getProcessSymmetryFactor().getFactor() + "', " +
                                    "new='" + ((matchingPjp == null) ? "null" : matchingPjp.getProcessSymmetryFactor().getFactor()) + "'";
                            logMathTexts(this, newMathDesc, Decision.MathDifferent_LEGACY_SYMMETRY_PARTICLE_JUMP_PROCESS, msg);
                            return new MathCompareResults(Decision.MathDifferent_LEGACY_SYMMETRY_PARTICLE_JUMP_PROCESS, msg);
                        }
                        if(!bEqual){
                            String msg = "PJP='" + oldPjp.getName() + "', SymmetryFactor: old='" + oldPjp.getProcessSymmetryFactor().getFactor() + "' " +
                                    "new='" + ((matchingPjp == null) ? "null" : matchingPjp.getProcessSymmetryFactor().getFactor()) + "', rate: " +
                                    "old='" + Arrays.asList(oldPjp.getParticleRateDefinition().getExpressions()).stream().map(e -> e.infix()).collect(Collectors.toList()) + "', " +
                                    "new='" + ((matchingPjp == null) ? "null" : Arrays.asList(matchingPjp.getParticleRateDefinition().getExpressions()).stream().map(e -> e.infix()).collect(Collectors.toList())) + "'";
                            logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_PARTICLE_JUMP_PROCESS, msg);
                            return new MathCompareResults(Decision.MathDifferent_DIFFERENT_PARTICLE_JUMP_PROCESS, msg);
                        }
                    }
                }

                //
                // after repairing aspects of MathDescription, now see if same
                //
                return oldMathDesc.compareInvariantAttributes(newMathDesc, true);
            }
        } catch(Exception e){
            lg.error("unexpected failure comparing math descriptions: " + e.getMessage(), e);
            logMathTexts(this, newMathDesc, Decision.MathDifferent_FAILURE_UNKNOWN, "Failure comparing math: " + e.getMessage());
            return new MathCompareResults(Decision.MathDifferent_FAILURE_UNKNOWN, "failed to compare math: " + e.getMessage());
        }
    }

    private void setMissingEquationBoundaryConditionsToDefault(SubDomain.DomainWithBoundaryConditions subdomain, int geometryDim){
        List<PdeEquation> pdeEquations = subdomain.getEquationCollection().stream().filter(e -> e instanceof PdeEquation).map(e -> (PdeEquation) e).collect(Collectors.toList());
        //
        // CompartmentSubDomains have correct boundary condition types for external boundaries.
        //
        SubDomain.DomainWithBoundaryConditions subdomainForBoundaryTypes = subdomain;
        if(subdomain instanceof MembraneSubDomain){
            //
            // MembraneSubDomain boundary condition types are ignored for most VCell solvers,
            // instead they defer to the inside CompartmentSubDomain's definition.
            //
            subdomainForBoundaryTypes = ((MembraneSubDomain) subdomain).getInsideCompartment();
        }
        for(PdeEquation equ : pdeEquations){
            Expression initExp = equ.getInitialExpression();
            if(geometryDim >= 1){
                BoundaryConditionType xmType = subdomainForBoundaryTypes.getBoundaryConditionXm();
                BoundaryConditionType xpType = subdomainForBoundaryTypes.getBoundaryConditionXp();
                equ.setBoundaryXm(replaceDefaultBC(equ.getBoundaryXm(), xmType, initExp));
                equ.setBoundaryXp(replaceDefaultBC(equ.getBoundaryXp(), xpType, initExp));
            }
            if(geometryDim >= 2){
                BoundaryConditionType ymType = subdomainForBoundaryTypes.getBoundaryConditionYm();
                BoundaryConditionType ypType = subdomainForBoundaryTypes.getBoundaryConditionYp();
                equ.setBoundaryYm(replaceDefaultBC(equ.getBoundaryYm(), ymType, initExp));
                equ.setBoundaryYp(replaceDefaultBC(equ.getBoundaryYp(), ypType, initExp));
            }
            if(geometryDim == 3){
                BoundaryConditionType zmType = subdomainForBoundaryTypes.getBoundaryConditionZm();
                BoundaryConditionType zpType = subdomainForBoundaryTypes.getBoundaryConditionZp();
                equ.setBoundaryZm(replaceDefaultBC(equ.getBoundaryZm(), zmType, initExp));
                equ.setBoundaryZp(replaceDefaultBC(equ.getBoundaryZp(), zpType, initExp));
            }
        }
    }


    private Expression replaceDefaultBC(Expression bcExp, BoundaryConditionType bcType, Expression initExp){
        if(bcExp == null){
            if(bcType.isNEUMANN()){
                return new Expression(0.0);
            } else if(bcType.isDIRICHLET()){
                return new Expression(initExp);
            }
        }
        return bcExp;
    }

    /**
     * compare two expressions; if different but functionally equivalent, set the new to be the same as the old
     *
     * @param nExp
     * @param oExp
     * @param newSet nExp setter, non null
     * @return true if both null, or equal / equivalent
     */
    private static boolean compareUpdate(Expression nExp, Expression oExp, Consumer<Expression> newSet){

        Objects.requireNonNull(newSet);
        switch(Compare.nullState(nExp, oExp)){
            case BOTH_NULL:
                return true;
            case FIRST_NULL:
            case SECOND_NULL:
                return false;
            case NEITHER_NULL:
                //fall-through
        }

        if(Compare.isEqual(nExp, oExp)){
            return true;
        }
        if(ExpressionUtils.functionallyEquivalent(nExp, oExp)){
            newSet.accept(oExp);
        }
        return false;
    }

    private static void logMathTexts(MathDescription math1, MathDescription math2, Decision decision, String msg){
        if(lg.isTraceEnabled()){
            try {
                lg.trace("maths different: " + decision.name() + " details: " + msg
                        + "\n===================MATH 1==================\n"
                        + math1.getVCML_database() + "\n"
                        + "==================MATH 2====================\n"
                        + math2.getVCML_database() + "\n"
                        + "==================END MATHS=================");
            } catch(Exception e){
                lg.error("error logging math description text: " + e.getMessage(), e);
            }
        }
    }

    private MathCompareResults compareInvariantAttributes(MathDescription newMathDesc, boolean bAlreadyFlattened){
        //
        // making cannonical math descriptions is expensive, so the idea is to quickly identify those differences that are invariant of any cannonical form.
        //


        //
        // compare subdomain names and types only SubDomain.compareEqual() would recursively compare all of the equations.
        //
        if(subDomainList.size() != newMathDesc.subDomainList.size()){
            String msg = subDomainList.size() + " vs " + newMathDesc.subDomainList.size();
            logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_NUMBER_OF_SUBDOMAINS, msg);
            return new MathCompareResults(Decision.MathDifferent_DIFFERENT_NUMBER_OF_SUBDOMAINS, msg);
        }
        for(int i = 0; i < subDomainList.size(); i++){
            SubDomain subDomain = subDomainList.get(i);
            SubDomain otherSubDomain = newMathDesc.getSubDomain(subDomain.getName());
            if(otherSubDomain == null || !otherSubDomain.getClass().equals(subDomain.getClass())){
                String msg = "subdomain " + subDomain.getName();
                logMathTexts(this, newMathDesc, Decision.MathDifferent_SUBDOMAINS_DONT_MATCH, msg);
                return new MathCompareResults(Decision.MathDifferent_SUBDOMAINS_DONT_MATCH, msg);
            }
        }

        //
        // compare state variables (make sure number of state variables are equal).
        //
        HashSet<String> thisStateVarHash = getStateVariableNames();
        HashSet<String> otherStateVarHash = newMathDesc.getStateVariableNames();
        if(thisStateVarHash.size() != otherStateVarHash.size()){
            String msg = thisStateVarHash.size() + " vs " + otherStateVarHash.size();
            logMathTexts(this, newMathDesc, Decision.MathDifferent_DIFFERENT_NUMBER_OF_VARIABLES, msg);
            return new MathCompareResults(Decision.MathDifferent_DIFFERENT_NUMBER_OF_VARIABLES, msg);
        }
        //
        // disregard vars present in both sets (for efficiency).
        //
        HashSet<String> intersectionStateVarHash = new HashSet<String>(thisStateVarHash);
        intersectionStateVarHash.retainAll(otherStateVarHash);
        //
        // if intersection of state variables is smaller than entire set,
        // then must make sure that each missing state variable exists as a Function (dependent variable).
        //
        if(intersectionStateVarHash.size() != thisStateVarHash.size()){
            // trim both sets to remove intersection
            thisStateVarHash.removeAll(intersectionStateVarHash);
            otherStateVarHash.removeAll(intersectionStateVarHash);
            //
            // check this MathDescription's (extra) state variables are present in other MathDescription (as Function).
            //
            Iterator<String> iterThis = thisStateVarHash.iterator();
            ArrayList<String> varsNotFoundMath1 = new ArrayList<>();
            while (iterThis.hasNext()) {
                String varName = iterThis.next();
                Variable var = newMathDesc.getVariable(varName);
                if(var == null || (var instanceof Constant)){
                    varsNotFoundMath1.add(varName);
                }
            }
            if(varsNotFoundMath1.size() > 0){
                logMathTexts(this, newMathDesc, Decision.MathDifferent_VARIABLE_NOT_FOUND_AS_FUNCTION, "vars = " + varsNotFoundMath1.toString());
            }
            //
            // check other MathDescription's (extra) state variables are present in this MathDescription (as Function).
            //
            Iterator<String> iterOther = otherStateVarHash.iterator();
            ArrayList<String> varsNotFoundMath2 = new ArrayList<>();
            while (iterOther.hasNext()) {
                String varName = iterOther.next();
                Variable var = getVariable(varName);
                if(var == null || (var instanceof Constant)){
                    varsNotFoundMath2.add(varName);
                }
            }
            if(varsNotFoundMath2.size() > 0){
                logMathTexts(this, newMathDesc, Decision.MathDifferent_VARIABLE_NOT_FOUND_AS_FUNCTION, "vars = " + varsNotFoundMath2.toString());
            }
            if(varsNotFoundMath1.size() > 0 || varsNotFoundMath2.size() > 0)
                return new MathCompareResults(Decision.MathDifferent_VARIABLE_NOT_FOUND_AS_FUNCTION, "number of bvariables not found " + varsNotFoundMath1.size() + "," + varsNotFoundMath2.size(), varsNotFoundMath1, varsNotFoundMath2);
        }

        //
        // compare geometry
        //
        if(!Compare.isEqualOrNull(
                (geometry != null ? geometry.getGeometrySpec() : null),
                (newMathDesc.geometry != null ? newMathDesc.geometry.getGeometrySpec() : null))){
            logMathTexts(this, newMathDesc, Decision.MathDifferent_GEOMETRYSPEC_DIFFERENT, "");
            return new MathCompareResults(Decision.MathDifferent_GEOMETRYSPEC_DIFFERENT);
        }

        if(bAlreadyFlattened){
            return new MathCompareResults(Decision.MathEquivalent_FLATTENED);
        } else {
            return new MathCompareResults(Decision.MathEquivalent_NATIVE);
        }
    }


    /**
     * The firePropertyChange method was generated to support the propertyChange field.
     */
    public void firePropertyChange(java.lang.String propertyName, java.lang.Object oldValue, java.lang.Object newValue){
        getPropertyChange().firePropertyChange(propertyName, oldValue, newValue);
    }

    /**
     * Method to support listener events.
     */
    protected void fireStateChanged(){
        if(aChangeListener != null){
            ChangeEvent e = new ChangeEvent(this);
            for(ChangeListener listener : aChangeListener){
                listener.stateChanged(e);
            }
        }
    }

    /**
     * The fireVetoableChange method was generated to support the vetoPropertyChange field.
     */
    public void fireVetoableChange(java.lang.String propertyName, java.lang.Object oldValue, java.lang.Object newValue) throws java.beans.PropertyVetoException{
        getVetoPropertyChange().fireVetoableChange(propertyName, oldValue, newValue);
    }

    public static MathDescription fromEditor(MathDescription oldMathDesc, String vcml) throws MathException, java.beans.PropertyVetoException{

        CommentStringTokenizer tokens = new CommentStringTokenizer(vcml);
        MathDescription mathDesc = new MathDescription(oldMathDesc.getVersion());
        mathDesc.clearAll();
        mathDesc.setGeometry0(oldMathDesc.getGeometry());
        mathDesc.read_database(tokens);

        mathDesc.refreshDependencies();

        //
        // compute warning string (if necessary)
        //
        if(!mathDesc.isValid()){
            lg.warn("Math is invalid after parsing: '" + mathDesc.getWarning() + "'");
        }

        return mathDesc;
    }


    public CompartmentSubDomain getCompartmentSubDomain(String name){
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain instanceof CompartmentSubDomain){
                CompartmentSubDomain compartment = (CompartmentSubDomain) subDomain;
                if(compartment.getName().equals(name)){
                    return compartment;
                }
            }
        }
        return null;
    }


    /**
     * Insert the method's description here.
     * Creation date: (8/21/00 1:14:56 PM)
     *
     * @return java.util.Enumeration
     */
    public Enumeration<Constant> getConstants(){
        return new Enumeration<Constant>() {
            int count = 0;

            public boolean hasMoreElements(){
                for(int i = count; i < variableList.size(); i++){
                    if(variableList.get(i) instanceof Constant){
                        return true;
                    }
                }
                return false;
            }

            public Constant nextElement(){
                synchronized (variableList) {
                    for(int i = count; i < variableList.size(); i++){
                        if(variableList.get(i) instanceof Constant){
                            count = i + 1;
                            return (Constant) variableList.get(i);
                        }
                    }
                }
                throw new NoSuchElementException("ArrayList Enumeration (Constant)");
            }
        };
    }

    public java.lang.String getDescription(){
        return fieldDescription;
    }


    public SymbolTableEntry getEntry(String id){
        SymbolTableEntry entry = null;

        entry = ReservedMathSymbolEntries.getEntry(id, false);
        if(entry != null){
            if(entry instanceof SymbolTableFunctionEntry){
                if(entry.equals(MathFunctionDefinitions.Function_regionArea_current)
                        || entry.equals(MathFunctionDefinitions.Function_regionArea_indexed)
                        || entry.equals(MathFunctionDefinitions.Function_regionVolume_current)
                        || entry.equals(MathFunctionDefinitions.Function_regionVolume_indexed)){
                    bRegionSizeFunctionsUsed = true;
                }
            }
            return entry;
        }

        entry = getVariable(id);
        if(entry != null){
            return entry;
        }

        return null;
    }


    /**
     * Insert the method's description here.
     * Creation date: (8/21/00 1:14:56 PM)
     *
     * @return java.util.Enumeration
     */
    public Enumeration<FilamentVariable> getFilamentVariables(){
        return new Enumeration<FilamentVariable>() {
            int count = 0;

            public boolean hasMoreElements(){
                for(int i = count; i < variableList.size(); i++){
                    if(variableList.get(i) instanceof FilamentVariable){
                        return true;
                    }
                }
                return false;
            }

            public FilamentVariable nextElement(){
                synchronized (variableList) {
                    for(int i = count; i < variableList.size(); i++){
                        if(variableList.get(i) instanceof FilamentVariable){
                            count = i + 1;
                            return (FilamentVariable) variableList.get(i);
                        }
                    }
                }
                throw new NoSuchElementException("ArrayList Enumeration (FilamentVariable)");
            }
        };
    }


    /**
     * Insert the method's description here.
     * Creation date: (10/9/2002 10:54:06 PM)
     *
     * @return cbit.vcell.math.MathDescription
     */
    public static Function[] getFlattenedFunctions(MathSymbolTableFactory mathSymbolTableFactory, MathDescription originalMathDescription, String functionNames[]) throws MathException, ExpressionException{
        //
        // clone current mathdescription
        //
        MathDescription newMath = new MathDescription(originalMathDescription);

        //
        // make a "identity" simulation (no overrides), this will help to substitute/flatten expressions.
        //
        MathSymbolTable simSymbolTable = mathSymbolTableFactory.createMathSymbolTable(newMath);
        Function functions[] = new Function[functionNames.length];
        for(int i = 0; i < functionNames.length; i++){
            for(int j = 0; j < originalMathDescription.variableList.size(); j++){
                Variable var = originalMathDescription.variableList.get(j);
                if(var.getName().equals(functionNames[i])){
                    if(var instanceof Function){
                        Function function = (Function) var;
                        Expression exp1 = new Expression(function.getExpression());
                        try {
                            exp1 = simSymbolTable.substituteFunctions(exp1);
                            functions[i] = new Function(function.getName(), exp1.flatten(), function.getDomain());
                        } catch(MathException e){
                            throw new RuntimeException("Substitute function failed on function " + function.getName() + " " + e.getMessage(), e);
                        }
                    }
                }
            }
        }
        return functions;
    }


    /**
     * Insert the method's description here.
     * Creation date: (8/21/00 1:14:56 PM)
     *
     * @return java.util.Enumeration
     */
    public Enumeration<Function> getFunctions(){
        return new Enumeration<Function>() {
            int count = 0;

            public boolean hasMoreElements(){
                for(int i = count; i < variableList.size(); i++){
                    if(variableList.get(i) instanceof Function){
                        return true;
                    }
                }
                return false;
            }

            public Function nextElement(){
                synchronized (variableList) {
                    for(int i = count; i < variableList.size(); i++){
                        if(variableList.get(i) instanceof Function){
                            count = i + 1;
                            return (Function) variableList.get(i);
                        }
                    }
                }
                throw new NoSuchElementException("ArrayList Enumeration (Function)");
            }
        };
    }


    /**
     * This method was created in VisualAge.
     *
     * @return cbit.vcell.geometry.Geometry
     */
    public Geometry getGeometry(){
        return geometry;
    }


    /**
     * This method was created in VisualAge.
     *
     * @param compartmentSubDomain cbit.vcell.math.CompartmentSubDomain
     * @return int
     * @throws MathException
     */
    public int getHandle(CompartmentSubDomain compartmentSubDomain) throws MathException{
        SubVolume subVolume = geometry.getGeometrySpec().getSubVolume(compartmentSubDomain.getName());
        if(subVolume == null){
            throw new MathException("couldn't find a subVolume named " + compartmentSubDomain + " in Geometry");
        }
        return subVolume.getHandle();
    }

    public KeyValue getKey(){
        return (getVersion() != null) ? getVersion().getVersionKey() : null;
    }

    public MembraneSubDomain getMembraneSubDomain(CompartmentSubDomain compartment1, CompartmentSubDomain compartment2){
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain instanceof MembraneSubDomain){
                MembraneSubDomain membraneSubDomain = (MembraneSubDomain) subDomain;
                if((membraneSubDomain.getInsideCompartment() == compartment1 && membraneSubDomain.getOutsideCompartment() == compartment2) ||
                        (membraneSubDomain.getInsideCompartment() == compartment2 && membraneSubDomain.getOutsideCompartment() == compartment1)){
                    return membraneSubDomain;
                }
            }
        }
        return null;
    }


    private MembraneSubDomain getMembraneSubDomain(String membraneName){
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain instanceof MembraneSubDomain){
                MembraneSubDomain membrane = (MembraneSubDomain) subDomain;
                if(membrane.getName().equalsIgnoreCase(membraneName)){
                    return membrane;
                }
            }
        }
        return null;
    }


    public MembraneSubDomain[] getMembraneSubDomains(CompartmentSubDomain compartment){
        ArrayList<MembraneSubDomain> membraneSubDomainList = new ArrayList<MembraneSubDomain>();
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain instanceof MembraneSubDomain){
                MembraneSubDomain membraneSubDomain = (MembraneSubDomain) subDomain;
                if((membraneSubDomain.getInsideCompartment() == compartment) || (membraneSubDomain.getOutsideCompartment() == compartment)){
                    membraneSubDomainList.add(membraneSubDomain);
                }
            }
        }
        Collections.sort(membraneSubDomainList);
        return membraneSubDomainList.toArray(new MembraneSubDomain[membraneSubDomainList.size()]);
    }


    public java.lang.String getName(){
        return fieldName;
    }


    public int getNumVariables(){
        return variableList.size();
    }


    /**
     * Accessor for the propertyChange field.
     */
    protected java.beans.PropertyChangeSupport getPropertyChange(){
        if(propertyChange == null){
            propertyChange = new java.beans.PropertyChangeSupport(this);
        }
        ;
        return propertyChange;
    }

    public List<Variable> getStateVariables(){
        List<Variable> stateVars = new ArrayList<>();
        for(int i = 0; i < variableList.size(); i++){
            Variable var = variableList.get(i);
            if(var instanceof VolVariable || var instanceof MemVariable || var instanceof FilamentVariable ||
                    var instanceof VolumeRegionVariable || var instanceof MembraneRegionVariable || var instanceof FilamentRegionVariable ||
                    var instanceof ParticleVariable || var instanceof StochVolVariable || var instanceof PointVariable){
                stateVars.add(var);
            }
        }
        return stateVars;
    }

    public HashSet<String> getStateVariableNames(){
        List<Variable> stateVars = getStateVariables();
        HashSet<String> stateVarNameSet = new HashSet<String>();
        for(Variable var : stateVars){
            if(var instanceof VolVariable || var instanceof MemVariable || var instanceof FilamentVariable ||
                    var instanceof VolumeRegionVariable || var instanceof MembraneRegionVariable || var instanceof FilamentRegionVariable ||
                    var instanceof ParticleVariable || var instanceof StochVolVariable || var instanceof PointVariable){
                stateVarNameSet.add(var.getName());
            }
        }
        return stateVarNameSet;
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @param name java.lang.String
     * @return cbit.vcell.math.SubDomain
     * @throws java.lang.Exception The exception description.
     */
    public SubDomain getSubDomain(String name){
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain.getName().equals(name)){
                return subDomain;
            }
        }
        return null;
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @return java.util.Enumeration
     */
    public Enumeration<SubDomain> getSubDomains(){
        return Collections.enumeration(subDomainList);
    }

    /**
     * Collection equivalent of {@link #getSubDomains()}
     *
     * @return unmodifiable collection
     */
    public Collection<SubDomain> getSubDomainCollection(){
        return Collections.unmodifiableCollection(subDomainList);
    }

    /**
     * This method was created by a SmartGuide.
     *
     * @param name java.lang.String
     * @return cbit.vcell.math.Variable
     */
    public Variable getVariable(String name){
        return variableHashTable.get(name);
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @return java.util.Enumeration
     */
    public Enumeration<Variable> getVariables(){
        return Collections.enumeration(variableList);
    }

    public List<Variable> getVariableList(){
        return variableList;
    }

    public Map<String, Variable> getVariableMap(){
        return variableHashTable;
    }

    public Iterator<Event> getEvents(){
        return eventList.iterator();
    }

    public int getNumEvents(){
        return eventList.size();
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/29/00 12:39:15 PM)
     *
     * @return java.lang.String
     */
    public String getVCML() throws MathException{
        return getVCML_database();
    }

    private String getVCML(VCMLProvider obj) throws MathException{
        final char SPACE = ' ';
        if(commenting){
            String before = obj.getBeforeComment();
            String after = obj.getAfterComment();
            StringBuilder sb = new StringBuilder();
            if(before != null){
                sb.append(SPACE);
                sb.append(Commented.BEFORE_COMMENT);
                sb.append(SPACE);
                sb.append(before);
                sb.append(SPACE);
                sb.append(Commented.END_BEFORE_COMMENT);
                sb.append(NEWLINE_CHAR);
            }
            if(after == null){
                sb.append(obj.getVCML());
            } else {
                String baseVCML = obj.getVCML().trim();

                //look for newline at very end of String, but don't flag newlines in the midst of the text
                int newlinePos = Commented.CHAR_NOT_FOUND;
                int scan = baseVCML.length() - 1;
                while (baseVCML.charAt(scan) == NEWLINE_CHAR) {
                    newlinePos = scan;
                    --scan;
                    if(scan == 0){
                        throw new MathException(obj.getClass().getSimpleName() + "#getVCML( ) returned empty String");
                    }
                }
                boolean hasNewline = newlinePos != Commented.CHAR_NOT_FOUND;
                if(hasNewline){ //if new line present, remove for now
                    baseVCML = baseVCML.substring(0, newlinePos);
                }
                sb.append(baseVCML);
                sb.append(SPACE);
                sb.append(Commented.AFTER_COMMENT);
                sb.append(SPACE);
                sb.append(after);
                if(hasNewline){ //now add newline back if it was present
                    sb.append(NEWLINE_CHAR);
                }
            }
            return sb.toString();
        }
        return obj.getVCML();

    }

    /**
     * add variables of specific type to output. Add newline if any variables added.
     *
     * @param clzz         class or superclass of desired variable
     * @param buffer       where to add output
     * @param needsNewline TODO
     * @throws MathException from {@link Variable#getVCML()}
     */
    private <T extends Variable> void addVariablesOfType(Class<T> clzz, StringBuilder buffer, boolean needsNewline) throws MathException{
        int start = buffer.length();
        for(Variable var : variableList){
            if(clzz.isAssignableFrom(var.getClass())){
                buffer.append(getVCML(var));
                if(needsNewline){
                    buffer.append(NEWLINE_CHAR);
                }
            }
        }
        if(buffer.length() != start){
            buffer.append(NEWLINE_CHAR);
        }
    }

    /**
     * @return {@link #getVCML_database(boolean)} with comments on
     * @throws MathException
     */
    public String getVCML_database() throws MathException{
        return getVCML_database(true);
    }

    /**
     * get VCML representation
     *
     * @param includeComments if true, includes comments for elements
     * @return VCML
     * @throws MathException
     */
    public String getVCML_database(boolean includeComments) throws MathException{
        commenting = includeComments;

        //
        // regular VCML exception, no name, and no geometry
        //
        StringBuilder buffer = new StringBuilder();
        buffer.append(VCML.MathDescription + " {\n");
        buffer.append("\n");

        addVariablesOfType(ParameterVariable.class, buffer, true);
        addVariablesOfType(Constant.class, buffer, true);
        addVariablesOfType(VolVariable.class, buffer, true);
        addVariablesOfType(MemVariable.class, buffer, true);
        addVariablesOfType(VolumeRegionVariable.class, buffer, true);
        addVariablesOfType(MembraneRegionVariable.class, buffer, true);
        addVariablesOfType(FilamentVariable.class, buffer, true);
        addVariablesOfType(FilamentRegionVariable.class, buffer, true);
        addVariablesOfType(PointVariable.class, buffer, true);
        addVariablesOfType(StochVolVariable.class, buffer, false);
        addVariablesOfType(VolumeParticleVariable.class, buffer, false);
        addVariablesOfType(MembraneParticleVariable.class, buffer, false);
        // ParticleMolecularType
        boolean bSpaceNeeded = false;
        for(ParticleMolecularType particleMolecularType : particleMolecularTypes){
            buffer.append(particleMolecularType.getVCML());
            bSpaceNeeded = true;
        }
        if(bSpaceNeeded){
            buffer.append("\n");
            bSpaceNeeded = false;
        }

        addVariablesOfType(ParticleSpeciesPattern.class, buffer, false);
        addVariablesOfType(ParticleObservable.class, buffer, false);
        addVariablesOfType(Function.class, buffer, true);
        addVariablesOfType(RandomVariable.class, buffer, true);

        // Event
        for(Event event : eventList){
            buffer.append(getVCML(event) + "\n");
        }
//	if (eventList.size() > 0) {
//		buffer.append("\n");
//	}

        // Post Processing block
        if(postProcessingBlock.getNumDataGenerators() > 0){
            buffer.append(postProcessingBlock.getVCML() + "\n");
        }

        final int dimension = getGeometry().getDimension();
        Enumeration<SubDomain> enum2 = getSubDomains();
        while (enum2.hasMoreElements()) {
            SubDomain subDomain = enum2.nextElement();
            buffer.append(getVCML(subDomain.getVCMLProvider(dimension)) + "\n");
        }
        buffer.append("}\n");
        final String rval = buffer.toString();
        lg.debug(rval);
        return rval;
    }

    /**
     * This method was created in VisualAge.
     *
     * @return cbit.sql.Version
     */
    public Version getVersion(){
        return version;
    }


    /**
     * Accessor for the vetoPropertyChange field.
     */
    protected java.beans.VetoableChangeSupport getVetoPropertyChange(){
        if(vetoPropertyChange == null){
            vetoPropertyChange = new java.beans.VetoableChangeSupport(this);
        }
        ;
        return vetoPropertyChange;
    }


    /**
     * Gets the warning property (java.lang.String) value.
     *
     * @return The warning property value.
     * @see #setWarning
     */
    public java.lang.String getWarning(){
        return fieldWarning;
    }


    /**
     * This method was created in VisualAge.
     *
     * @return boolean
     */
    public boolean hasFastSystems(){

        //
        // Check each subDomain for FastSystems
        //
        for(int i = 0; i < subDomainList.size(); i++){
            SubDomain subDomain = subDomainList.get(i);
            if(subDomain.getFastSystem() != null){
                return true;
            }
        }
        return false;
    }


    public boolean hasPeriodicBoundaryCondition(){

        //
        // Check each subDomain for Periodic Boundary Condition
        //
        for(int i = 0; i < subDomainList.size(); i++){
            SubDomain subDomain = subDomainList.get(i);
            BoundaryConditionType[] bctypes = new BoundaryConditionType[0];
            if(subDomain instanceof CompartmentSubDomain){
                CompartmentSubDomain dom = (CompartmentSubDomain) subDomain;
                bctypes = new BoundaryConditionType[]{
                        dom.getBoundaryConditionXm(),
                        dom.getBoundaryConditionXp(),
                        dom.getBoundaryConditionYm(),
                        dom.getBoundaryConditionYp(),
                        dom.getBoundaryConditionZm(),
                        dom.getBoundaryConditionZp()
                };
            } else if(subDomain instanceof MembraneSubDomain){
                MembraneSubDomain dom = (MembraneSubDomain) subDomain;
                bctypes = new BoundaryConditionType[]{
                        dom.getBoundaryConditionXm(),
                        dom.getBoundaryConditionXp(),
                        dom.getBoundaryConditionYm(),
                        dom.getBoundaryConditionYp(),
                        dom.getBoundaryConditionZm(),
                        dom.getBoundaryConditionZp()
                };
            }
            for(BoundaryConditionType bct : bctypes){
                if(bct.isPERIODIC()){
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * The hasListeners method was generated to support the propertyChange field.
     */
    public synchronized boolean hasListeners(java.lang.String propertyName){
        return getPropertyChange().hasListeners(propertyName);
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @param volVariable cbit.vcell.math.VolVariable
     * @return boolean
     */
    public boolean hasVelocity(VolVariable volVariable){
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain instanceof CompartmentSubDomain){
                Equation equation = subDomain.getEquation(volVariable);
                if(equation instanceof PdeEquation){
                    PdeEquation pdeEqn = (PdeEquation) equation;
                    if((pdeEqn.getVelocityX() != null && !pdeEqn.getVelocityX().isZero())
                            || (pdeEqn.getVelocityY() != null && !pdeEqn.getVelocityY().isZero())
                            || (pdeEqn.getVelocityZ() != null && !pdeEqn.getVelocityZ().isZero())){
                        return true;
                    }
                }
            }
        }
        return false;
    }

    public boolean hasVelocity(){
        for(Variable var : variableList){
            if(var instanceof VolVariable){
                if(hasVelocity((VolVariable) var)){
                    return true;
                }
            }
        }
        return false;
    }

    public boolean hasGradient(){
        for(Variable var : variableList){
            if(var instanceof VolVariable){
                if(hasGradient((VolVariable) var)){
                    return true;
                }
            }
        }
        return false;
    }

    public boolean hasGradient(VolVariable volVariable){
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain instanceof CompartmentSubDomain){
                Equation equation = subDomain.getEquation(volVariable);
                if(equation instanceof PdeEquation){
                    PdeEquation pdeEqn = (PdeEquation) equation;
                    if((pdeEqn.getGradientX() != null && !pdeEqn.getGradientX().isZero())
                            || (pdeEqn.getGradientY() != null && !pdeEqn.getGradientY().isZero())
                            || (pdeEqn.getGradientZ() != null && !pdeEqn.getGradientZ().isZero())){
                        return true;
                    }
                }
            }
        }
        return false;
    }

    public boolean isPDE(MemVariable memVariable){
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain instanceof MembraneSubDomain){
                Equation equation = subDomain.getEquation(memVariable);
                if(equation instanceof PdeEquation){
                    return true;
                }
            }
        }
        return false;
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @param volVariable cbit.vcell.math.VolVariable
     * @return boolean
     */
    public boolean isPDE(VolVariable volVariable){
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain instanceof CompartmentSubDomain){
                Equation equation = subDomain.getEquation(volVariable);
                if(equation instanceof PdeEquation){
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * This method was created by a SmartGuide.
     *
     * @param volVariable cbit.vcell.math.VolVariable
     * @return boolean
     */
    public boolean isPdeSteady(VolVariable volVariable){
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain instanceof CompartmentSubDomain){
                Equation equation = subDomain.getEquation(volVariable);
                if(equation instanceof PdeEquation){
                    if(((PdeEquation) equation).isSteady()){
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * This method was created in VisualAge.
     *
     * @return boolean
     */
    public boolean isSpatial(){
        if(getGeometry() == null){
            return false;
        }
        return (getGeometry().getDimension() > 0);
    }

    public boolean isSpatial3D(){
        if(getGeometry() == null){
            return false;
        }
        return (getGeometry().getDimension() == 3);
    }

    public boolean isSpatial2D(){
        if(getGeometry() == null){
            return false;
        }
        return (getGeometry().getDimension() == 2);
    }


    public boolean isNonSpatialStoch(){
        if(getGeometry().getDimension() != 0){
            return false;
        }
        Enumeration<Variable> enum1 = getVariables();
        while (enum1.hasMoreElements()) {
            Variable var = enum1.nextElement();
            if(var instanceof StochVolVariable){
                return true;
            }
            if(!(var instanceof Constant || var instanceof Function)){
                return false;
            }
        }
        return false;
    }


    @Override
    public boolean isRuleBased(){
        if(particleMolecularTypes.size() > 0){
            if(!isLangevin()){    // in math description, isRuleBased() and isLangevin() are mutually exclusive
                return true;
            } else {
                return false;
            }
        }
        return false;
    }

    @Override
    public boolean isLangevin(){
        if(!isSpatial3D()){
            return false;
        }
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            List<ParticleJumpProcess> particleJumpProcesses = subDomain.getParticleJumpProcesses();
            for(ParticleJumpProcess pjp : particleJumpProcesses){
                if(pjp instanceof LangevinParticleJumpProcess){
                    return true;    // the first jump process instance is enough to decide one way or another
                } else {
                    return false;
                }
            }
        }
        return false;
    }

    public boolean isSpatialHybrid(){
        if(getGeometry().getDimension() == 0){
            return false;
        }
        boolean bHasParticleVariable = false;
        boolean bHasPdeVariable = false;
        Enumeration<Variable> enum1 = getVariables();
        while (enum1.hasMoreElements()) {
            Variable var = enum1.nextElement();
            if(var instanceof ParticleVariable){
                bHasParticleVariable = true;
            } else if(var instanceof VolVariable || var instanceof MemVariable){
                bHasPdeVariable = true;
            } else if(var instanceof InsideVariable || var instanceof OutsideVariable){
            } else if(!(var instanceof Constant) && !(var instanceof Function)){
                return false;
            }
        }
        return bHasParticleVariable && bHasPdeVariable;
    }

    public boolean isSpatialStoch(){
        if(getGeometry().getDimension() == 0){
            return false;
        }
        boolean hasParticle = false;
        Enumeration<Variable> enum1 = getVariables();
        while (enum1.hasMoreElements()) {
            Variable var = enum1.nextElement();
            if(var instanceof ParticleVariable){
                hasParticle = true;
            }
            if(!(var instanceof ParticleVariable || var instanceof Constant || var instanceof Function)){
                return false;
            }
        }
        return hasParticle;
    }

    public boolean isValid(){
        IssueContext issueContext = new IssueContext(ContextType.MathDescription, this, null);
        ArrayList<Issue> issueList = new ArrayList<Issue>();
        gatherIssues(issueContext, issueList);
        boolean hasError = false;
        if(issueList.size() > 0){
            setWarning(issueList.get(0).getMessage());
            for(Issue issue : issueList){
                if(issue.getSeverity() == Issue.Severity.ERROR){
                    hasError = true;
                    setWarning(issue.getMessage());
                    break;
                }
            }
        }
        return !hasError;
    }

    /**
     * This method was created in VisualAge.
     *
     * @return boolean
     */
    public void gatherIssues(IssueContext issueContext, List<Issue> issueList){
        issueContext = issueContext.newChildContext(ContextType.MathDescription, this);
        setWarning(null);
        if(geometry == null){
            Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_NoGeometry, VCellErrorMessages.MATH_DESCRIPTION_GEOMETRY_1, Issue.SEVERITY_ERROR);
            issueList.add(issue);
        }

        if(isSpatialStoch() && geometry.getDimension() != 3){
            Issue issue = new Issue(geometry, issueContext, IssueCategory.Smoldyn_Geometry_3DWarning, "VCell spatial stochastic models only support 3D geometry.", Issue.SEVERITY_ERROR);
            issueList.add(issue);
        }

        // check Constant are really constants
        for(Variable var : variableList){
            if(var instanceof Constant){
                try {
                    var.getExpression().evaluateConstant();
                } catch(Exception ex){
                    String msg = "Constant cannot be evaluated to a number, " + var.getName() + "=" + var.getExpression().infix();
                    lg.error(msg, ex);
                    Issue issue = new Issue(var, issueContext, IssueCategory.MathDescription_Constant_NotANumber, VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_CONSTANT, var.getExpression().infix()), Issue.SEVERITY_ERROR);
                    issueList.add(issue);
                }
            }
        }

        //
        // count number of variables of each type
        //
        int volVarCount = 0;
        int memVarCount = 0;
        int filVarCount = 0;
        int volRegionVarCount = 0;
        int memRegionVarCount = 0;
        int filRegionVarCount = 0;
        int stochVarCount = 0;
        int pointVarCount = 0;
        for(int i = 0; i < variableList.size(); i++){
            Variable var = variableList.get(i);
            if(var instanceof VolVariable){
                volVarCount++;
            } else if(var instanceof MemVariable){
                memVarCount++;
            } else if(var instanceof FilamentVariable){
                filVarCount++;
            } else if(var instanceof VolumeRegionVariable){
                volRegionVarCount++;
            } else if(var instanceof MembraneRegionVariable){
                memRegionVarCount++;
            } else if(var instanceof FilamentRegionVariable){
                filRegionVarCount++;
            } else if(var instanceof StochVolVariable){
                stochVarCount++;
            } else if(var instanceof PointVariable){
                pointVarCount++;
            }
        }
        //
        // check that all equation rates and initial conditions ... etc can be bound to this MathDescription (e.g. no unresolved identifiers).
        //
        for(int i = 0; i < subDomainList.size(); i++){
            try {
                SubDomain subDomain = subDomainList.get(i);
                Enumeration<Equation> equEnum = subDomain.getEquations();
                while (equEnum.hasMoreElements()) {
                    Equation equ = equEnum.nextElement();
                    equ.checkValid(this, subDomain);
                    equ.bind(this);
                }
                FastSystem fastSystem = subDomain.getFastSystem();
                if(fastSystem != null){
                    Enumeration<FastRate> frEnum = fastSystem.getFastRates();
                    while (frEnum.hasMoreElements()) {
                        FastRate fr = frEnum.nextElement();
                        fr.bind(this);
                    }
                    Enumeration<FastInvariant> fiEnum = fastSystem.getFastInvariants();
                    while (fiEnum.hasMoreElements()) {
                        FastInvariant fi = fiEnum.nextElement();
                        fi.bind(this);
                    }
                }
                for(ParticleProperties pp : subDomain.getParticleProperties()){
                    pp.bind(this);
                }
                for(ParticleJumpProcess pjp : subDomain.getParticleJumpProcesses()){
                    pjp.bind(this);

                    Expression rateDefinition = null;
                    JumpProcessRateDefinition jprd = pjp.getParticleRateDefinition();
                    if(jprd instanceof MacroscopicRateConstant){
                        rateDefinition = MathUtilities.substituteFunctions(((MacroscopicRateConstant) jprd).getExpression(), this);
                    } else if(jprd instanceof InteractionRadius){
                        rateDefinition = MathUtilities.substituteFunctions(((InteractionRadius) jprd).getExpression(), this);
                    } else {
                        new RuntimeException("The jump process rate definition is not supported");
                    }

                    String symbols[] = rateDefinition.getSymbols();
                    if(symbols != null){
                        for(String symbol : symbols){ //throw exception for particle variables, particle variable cannot be referenced
                            Variable var = getVariable(symbol);
                            if(var instanceof ParticleVariable){
                                throw new MathException("Stochastic variables can not be referenced in functions and equations.");
                            }
                            if(subDomain instanceof CompartmentSubDomain){
                                if((var instanceof MembraneRegionVariable || var instanceof MemVariable)){
                                    throw new MathException("Volume reaction: " + pjp.getName() + " cannot reference membrane variable: " + var.getName() + ".");
                                }
                            }
                        }
                    }
                }
            } catch(ExpressionBindingException e){
                Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_ExpressionBindingException, e.getMessage(), Issue.SEVERITY_ERROR);
                issueList.add(issue);
            } catch(ExpressionException e){
                Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_ExpressionException, e.getMessage(), Issue.SEVERITY_ERROR);
                issueList.add(issue);
            } catch(MathException e){
                Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_MathException, e.getMessage(), Issue.SEVERITY_ERROR);
                issueList.add(issue);
            }
        }
        //
        // ODE only
        //
        if(geometry.getDimension() == 0){
            //
            // Check that only 1 subdomain is defined and that it is a volumeSubdomain
            //
            if(subDomainList.size() != 1){
                Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_CompartmentalModel, VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_MODEL_1, Issue.SEVERITY_ERROR);
                issueList.add(issue);
            } else if(subDomainList.size() == 1){
                if(!(subDomainList.get(0) instanceof CompartmentSubDomain)){
                    Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_CompartmentalModel, VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_MODEL_2, Issue.SEVERITY_ERROR);
                    issueList.add(issue);
                }
                CompartmentSubDomain subDomain = (CompartmentSubDomain) subDomainList.get(0);
                //distinguish ODE model and stochastic model
                if(isNonSpatialStoch()){
                    if(stochVarCount == 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_StochasticModel, VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_STOCHASTIC_MODEL_1, Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    if(subDomain.getJumpProcesses().size() == 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_StochasticModel, VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_STOCHASTIC_MODEL_2, Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    //check variable initial condition
                    for(VarIniCondition varIniCondition : subDomain.getVarIniConditions()){
                        Expression iniExp = varIniCondition.getIniVal();
                        try {
                            iniExp.bindExpression(this);
                        } catch(Exception ex){
                            lg.error(ex.getMessage(), ex);
                            setWarning(ex.getMessage());
                        }
                    }
                    //check probability rate
                    for(JumpProcess jumpProcess : subDomain.getJumpProcesses()){
                        Expression probExp = jumpProcess.getProbabilityRate();
                        try {
                            probExp.bindExpression(this);
                        } catch(Exception ex){
                            lg.error(ex.getMessage(), ex);
                            setWarning(ex.getMessage());
                        }
                    }
                } else if(isRuleBased()){
                } else {
                    // ODE model
                    //
                    // Check that all equations are ODEs
                    //
                    int odeCount = 0;
                    Enumeration<Equation> enum_equ = subDomain.getEquations();
                    while (enum_equ.hasMoreElements()) {
                        Equation equ = enum_equ.nextElement();
                        if(equ instanceof OdeEquation){
                            odeCount++;
                        } else {
                            Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_CompartmentalModel,
                                    VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_MODEL_3, Issue.SEVERITY_ERROR);
                            issueList.add(issue);
                        }
                    }
                    if(odeCount == 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_CompartmentalModel,
                                VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_MODEL_4, Issue.Severity.WARNING);
                        issueList.add(issue);
                    }

                    if(volVarCount != odeCount){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_CompartmentalModel,
                                VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_MODEL_5, Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    if(memVarCount > 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_CompartmentalModel,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_MODEL_6, VCML.MembraneVariable), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    if(filVarCount > 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_CompartmentalModel,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_MODEL_6, VCML.FilamentVariable), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    if(volRegionVarCount > 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_CompartmentalModel,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_MODEL_6, VCML.VolumeRegionVariable), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    if(memRegionVarCount > 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_CompartmentalModel,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_MODEL_6, VCML.MembraneRegionVariable), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    if(filRegionVarCount > 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_CompartmentalModel,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_COMPARTMENT_MODEL_6, VCML.FilamentRegionVariable), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                }
            }
            //
            // spatial (PDE and ODE)
            //
        } else {
            //
            // Check that the number of CompartmentSubdomains equals the number of VolumeSubVolumes in the Geometry
            // Check that the number of FilamentSubdomains equals the number of Filaments in the Geometry
            //
            int compartmentCount = 0;
            int membraneCount = 0;
            int filamentCount = 0;
            int pointCount = 0;
            for(int i = 0; i < subDomainList.size(); i++){
                SubDomain subDomain = (SubDomain) subDomainList.get(i);
                if(subDomain instanceof CompartmentSubDomain){
                    if(geometry.getGeometrySpec().getSubVolume(subDomain.getName()) == null){
                        Issue issue = new Issue(subDomain, issueContext, IssueCategory.MathDescription_SpatialModel_Subdomain,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_1, subDomain.getName()), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    compartmentCount++;
                } else if(subDomain instanceof MembraneSubDomain){
                    membraneCount++;
                } else if(subDomain instanceof FilamentSubDomain){
                    filamentCount++;
                } else if(subDomain instanceof PointSubDomain){
                    pointCount++;
                } else {
                    Issue issue = new Issue(subDomain, issueContext, IssueCategory.MathDescription_SpatialModel_Subdomain,
                            VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_2, subDomain.getName()), Issue.SEVERITY_ERROR);
                    issueList.add(issue);
                }
            }
            if(geometry.getGeometrySpec().getNumSubVolumes() != compartmentCount){
                Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_SpatialModel,
                        VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_3, geometry.getGeometrySpec().getNumSubVolumes(), compartmentCount), Issue.SEVERITY_ERROR);
                issueList.add(issue);
            }
            if(geometry.getGeometrySpec().getFilamentGroup().getFilamentCount() != filamentCount){
//			setWarning("Spatial model, there are "+geometry.getGeometrySpec().getFilamentGroup().getFilamentCount()+" filaments in geometry, but "+filamentCount+" "+VCML.FilamentSubDomain+"'s, must be equal");
//			return false;
            }
            if(filamentCount == 0 && (filVarCount > 0 || filRegionVarCount > 0)){
                Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_SpatialModel,
                        VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_4, Issue.SEVERITY_ERROR);
                issueList.add(issue);
            }
            if(membraneCount == 0 && (memVarCount > 0 || memRegionVarCount > 0)){
                Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_SpatialModel,
                        VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_5, Issue.SEVERITY_ERROR);
                issueList.add(issue);
            }
            //
            // Check that there are no duplicate Subdomains and that priorities are unique
            //
            for(int i = 0; i < subDomainList.size(); i++){
                SubDomain subDomain1 = (SubDomain) subDomainList.get(i);
                for(int j = 0; j < subDomainList.size(); j++){
                    if(i != j){
                        SubDomain subDomain2 = subDomainList.get(j);
                        if(subDomain1.getName().equals(subDomain2.getName())){
                            Issue issue = new Issue(subDomain1, issueContext, IssueCategory.MathDescription_SpatialModel_Subdomain,
                                    VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_6, subDomain1.getName(), subDomain2.getName()), Issue.SEVERITY_ERROR);
                            issueList.add(issue);
                        }
                        if(subDomain1 instanceof MembraneSubDomain && subDomain2 instanceof MembraneSubDomain){
                            MembraneSubDomain memSubDomain1 = (MembraneSubDomain) subDomain1;
                            MembraneSubDomain memSubDomain2 = (MembraneSubDomain) subDomain2;
                            if((memSubDomain1.getInsideCompartment() == memSubDomain2.getInsideCompartment() && memSubDomain1.getOutsideCompartment() == memSubDomain2.getOutsideCompartment()) ||
                                    (memSubDomain1.getInsideCompartment() == memSubDomain2.getOutsideCompartment() && memSubDomain1.getOutsideCompartment() == memSubDomain2.getInsideCompartment())){
                                Issue issue = new Issue(subDomain1, issueContext, IssueCategory.MathDescription_SpatialModel_Subdomain,
                                        VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_7, memSubDomain1.getInsideCompartment().getName(), memSubDomain1.getOutsideCompartment().getName()), Issue.SEVERITY_ERROR);
                                issueList.add(issue);
                            }
                        }
                    }
                }

            }
            // check periodic boundary conditons
            for(int i = 0; i < subDomainList.size(); i++){
                SubDomain subDomain = (SubDomain) subDomainList.get(i);
                if(subDomain instanceof CompartmentSubDomain){
                    CompartmentSubDomain compartmentSubDomain = (CompartmentSubDomain) subDomain;
                    BoundaryConditionType bctM = compartmentSubDomain.getBoundaryConditionXm();
                    BoundaryConditionType bctP = compartmentSubDomain.getBoundaryConditionXp();
                    if(bctM.isPERIODIC() && !bctP.isPERIODIC() || !bctM.isPERIODIC() && bctP.isPERIODIC()){
                        Issue issue = new Issue(subDomain, issueContext, IssueCategory.MathDescription_SpatialModel_Subdomain,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_9, "Xm", "Xp", subDomain.getName()), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    bctM = compartmentSubDomain.getBoundaryConditionYm();
                    bctP = compartmentSubDomain.getBoundaryConditionYp();
                    if(bctM.isPERIODIC() && !bctP.isPERIODIC() || !bctM.isPERIODIC() && bctP.isPERIODIC()){
                        Issue issue = new Issue(subDomain, issueContext, IssueCategory.MathDescription_SpatialModel_Subdomain,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_9, "Ym", "Yp", subDomain.getName()), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    bctM = compartmentSubDomain.getBoundaryConditionZm();
                    bctP = compartmentSubDomain.getBoundaryConditionZp();
                    if(bctM.isPERIODIC() && !bctP.isPERIODIC() || !bctM.isPERIODIC() && bctP.isPERIODIC()){
                        Issue issue = new Issue(subDomain, issueContext, IssueCategory.MathDescription_SpatialModel_Subdomain,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_9, "Zm", "Zp", subDomain.getName()), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                } else if(subDomain instanceof MembraneSubDomain){
                    MembraneSubDomain membraneSubDomain = (MembraneSubDomain) subDomain;
                    BoundaryConditionType bctM = membraneSubDomain.getBoundaryConditionXm();
                    BoundaryConditionType bctP = membraneSubDomain.getBoundaryConditionXp();
                    if(bctM.isPERIODIC() && !bctP.isPERIODIC() || !bctM.isPERIODIC() && bctP.isPERIODIC()){
                        Issue issue = new Issue(subDomain, issueContext, IssueCategory.MathDescription_SpatialModel_Subdomain,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_9, "Xm", "Xp", subDomain.getName()), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    bctM = membraneSubDomain.getBoundaryConditionYm();
                    bctP = membraneSubDomain.getBoundaryConditionYp();
                    if(bctM.isPERIODIC() && !bctP.isPERIODIC() || !bctM.isPERIODIC() && bctP.isPERIODIC()){
                        Issue issue = new Issue(subDomain, issueContext, IssueCategory.MathDescription_SpatialModel_Subdomain,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_9, "Ym", "Yp", subDomain.getName()), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    bctM = membraneSubDomain.getBoundaryConditionZm();
                    bctP = membraneSubDomain.getBoundaryConditionZp();
                    if(bctM.isPERIODIC() && !bctP.isPERIODIC() || !bctM.isPERIODIC() && bctP.isPERIODIC()){
                        Issue issue = new Issue(subDomain, issueContext, IssueCategory.MathDescription_SpatialModel_Subdomain,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_9, "Zm", "Zp", subDomain.getName()), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                }
            }

            try {
                if(geometry.getGeometrySpec().getDimension() > 0){
                    //
                    // Check that there is a MembraneSubdomain for each unique subVolume-subVolume interface in Geometry
                    // each ResolvedSurfaceLocation is an instance of a subVolume-subVolume interface (one-to-one with region boundaries).
                    //
                    GeometricRegion regions[] = geometry.getGeometrySurfaceDescription().getGeometricRegions();
                    //if (regions==null){
                    //try {
                    //geometry.getGeometrySurfaceDescription().updateAll();
                    //regions = geometry.getGeometrySurfaceDescription().getGeometricRegions();
                    //}catch (Exception e){
                    //lg.error(e);
                    //}
                    //}
                    if(regions == null){
                        Issue issue = new Issue(geometry, issueContext, IssueCategory.MathDescription_SpatialModel_Geometry,
                                VCellErrorMessages.MATH_DESCRIPTION_GEOMETRY_2, Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    } else {
                        for(int i = 0; i < regions.length; i++){
                            if(regions[i] instanceof SurfaceGeometricRegion){
                                SurfaceGeometricRegion surfaceRegion = (SurfaceGeometricRegion) regions[i];
                                SubVolume subVolume1 = ((VolumeGeometricRegion) surfaceRegion.getAdjacentGeometricRegions()[0]).getSubVolume();
                                CompartmentSubDomain compartment1 = getCompartmentSubDomain(subVolume1.getName());
                                if(compartment1 == null){
                                    Issue issue = new Issue(geometry, issueContext, IssueCategory.MathDescription_SpatialModel_Geometry,
                                            VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_GEOMETRY_3, getGeometry().getName(), subVolume1.getName()), Issue.SEVERITY_ERROR);
                                    issueList.add(issue);
                                }
                                SubVolume subVolume2 = ((VolumeGeometricRegion) surfaceRegion.getAdjacentGeometricRegions()[1]).getSubVolume();
                                CompartmentSubDomain compartment2 = getCompartmentSubDomain(subVolume2.getName());
                                if(compartment2 == null){
                                    Issue issue = new Issue(geometry, issueContext, IssueCategory.MathDescription_SpatialModel_Geometry,
                                            VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_GEOMETRY_3, getGeometry().getName(), subVolume2.getName()), Issue.SEVERITY_ERROR);
                                    issueList.add(issue);
                                }
                                MembraneSubDomain membraneSubDomain = getMembraneSubDomain(compartment1, compartment2);
                                if(compartment2 != null && compartment1 != null && membraneSubDomain == null){
                                    Issue issue = new Issue(geometry, issueContext, IssueCategory.MathDescription_SpatialModel_Geometry,
                                            VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_GEOMETRY_4, compartment1.getName(), compartment2.getName()), Issue.SEVERITY_ERROR);
                                    issueList.add(issue);
                                }
                            }
                        }
                        //
                        // Check that for each MembraneSubdomain there exists an actual surface in the geometry
                        //
                        for(int i = 0; i < subDomainList.size(); i++){
                            if(subDomainList.get(i) instanceof MembraneSubDomain){
                                MembraneSubDomain membraneSubDomain = (MembraneSubDomain) subDomainList.get(i);
                                boolean bFoundSurfaceInGeometry = false;
                                for(int j = 0; j < regions.length; j++){
                                    if(regions[j] instanceof SurfaceGeometricRegion){
                                        SurfaceGeometricRegion surfaceRegion = (SurfaceGeometricRegion) regions[j];
                                        VolumeGeometricRegion volumeRegion1 = (VolumeGeometricRegion) surfaceRegion.getAdjacentGeometricRegions()[0];
                                        VolumeGeometricRegion volumeRegion2 = (VolumeGeometricRegion) surfaceRegion.getAdjacentGeometricRegions()[1];
                                        String memInsideName = membraneSubDomain.getInsideCompartment().getName();
                                        String memOutsideName = membraneSubDomain.getOutsideCompartment().getName();
                                        if((memInsideName.equals(volumeRegion1.getSubVolume().getName()) && memOutsideName.equals(volumeRegion2.getSubVolume().getName())) ||
                                                (memInsideName.equals(volumeRegion2.getSubVolume().getName()) && memOutsideName.equals(volumeRegion1.getSubVolume().getName()))){
                                            bFoundSurfaceInGeometry = true;
                                            break;
                                        }
                                    }
                                }
                                if(!bFoundSurfaceInGeometry){
                                    Issue issue = new Issue(geometry, issueContext, IssueCategory.MathDescription_SpatialModel_Geometry,
                                            VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_GEOMETRY_5, membraneSubDomain.getInsideCompartment().getName(), membraneSubDomain.getOutsideCompartment().getName()), Issue.SEVERITY_ERROR);
                                    issueList.add(issue);
                                }
                            }
                        }
                    }
                }
                //}catch (GeometryException e){
                //lg.error(e);
                //setWarning("error validating MathDescription: "+e.getMessage());
                //return false;
                //}catch (ImageException e){
                //lg.error(e);
                //setWarning("error validating MathDescription: "+e.getMessage());
                //return false;
                //}catch (ExpressionException e){
                //lg.error(e);
                //setWarning("error validating MathDescription: "+e.getMessage());
                //return false;
            } catch(Exception e){
                lg.error(e.getMessage(), e);
                Issue issue = new Issue(geometry, issueContext, IssueCategory.MathDescription_SpatialModel_Geometry, e.getMessage(), Issue.SEVERITY_ERROR);
                issueList.add(issue);
            }
            //
            // Check that all equations for the same VolVariable are the same type (ODE or PDE)
            // ... and that all PDE equations have jump conditions defined on the appropriate membranes
            //
            for(int i = 0; i < variableList.size(); i++){
                Variable var = variableList.get(i);
                String varName = var.getName();
                if(var instanceof VolVariable){
                    VolVariable volVar = (VolVariable) var;

                    int pdeRefCount = 0;
                    int odeRefCount = 0;
                    int steadyPdeCount = 0;
                    int measureCount = 0;
                    for(int j = 0; j < subDomainList.size(); j++){
                        SubDomain subDomain = subDomainList.get(j);
                        Equation equ = subDomain.getEquation(volVar);
                        if(equ instanceof PdeEquation){
                            if(((PdeEquation) equ).isSteady()){
                                steadyPdeCount++;
                            } else {
                                pdeRefCount++;
                            }
                            //
                            // for each PDE, make sure that a jump condition all membranes that border this compartment
                            //
                            for(int k = 0; k < subDomainList.size(); k++){
                                SubDomain subDomain2 = subDomainList.get(k);
                                if(subDomain2 instanceof MembraneSubDomain){
                                    MembraneSubDomain membraneSubDomain = (MembraneSubDomain) subDomain2;
                                    if(membraneSubDomain.getInsideCompartment() == subDomain || membraneSubDomain.getOutsideCompartment() == subDomain){
                                        JumpCondition jumpCondition = membraneSubDomain.getJumpCondition(volVar);
                                        BoundaryConditionValue boundaryValue = ((PdeEquation) equ).getBoundaryConditionValue(membraneSubDomain.getName());
                                        // if PDE variable does not have jump condition OR boundaryValue (neither or both are not allowed), its an error.
                                        if((jumpCondition == null && boundaryValue == null) || (jumpCondition != null && boundaryValue != null)){
                                            Issue issue = new Issue(equ, issueContext, IssueCategory.MathDescription_SpatialModel_Equation,
                                                    VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_10, varName, subDomain.getName(), membraneSubDomain.getName()), Issue.SEVERITY_ERROR);
                                            issueList.add(issue);
                                        }
                                        if(boundaryValue != null && (subDomain.getBoundaryConditionSpec(membraneSubDomain.getName()) == null)){
                                            Issue issue = new Issue(equ, issueContext, IssueCategory.MathDescription_SpatialModel_Equation,
                                                    VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_10A, varName, subDomain.getName(), membraneSubDomain.getName(), membraneSubDomain.getName(), subDomain.getName()), Issue.SEVERITY_ERROR);
                                            issueList.add(issue);
                                        }
                                    }
                                }
                            }
                        } else if(equ instanceof OdeEquation){
                            odeRefCount++;
                        } else if(equ instanceof MeasureEquation){
                            measureCount++;
                        }
                        //
                        // for each JumpCondition, make sure that there is at least one PDE defined in inside or outside compartment
                        // and for each side, if the PDE is missing, then the corresponding flux term should be identically zero.
                        //
                        if(subDomain instanceof MembraneSubDomain){
                            MembraneSubDomain memSubDomain = (MembraneSubDomain) subDomain;
                            JumpCondition jumpCondition = memSubDomain.getJumpCondition(volVar);
                            if(jumpCondition != null){
                                boolean bInsidePresent = (memSubDomain.getInsideCompartment().getEquation(volVar) instanceof PdeEquation);
                                boolean bOutsidePresent = (memSubDomain.getOutsideCompartment().getEquation(volVar) instanceof PdeEquation);
                                if(!bInsidePresent && !bOutsidePresent){
                                    Issue issue = new Issue(equ, issueContext, IssueCategory.MathDescription_SpatialModel_Equation,
                                            VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_11, varName, memSubDomain.getName(), memSubDomain.getInsideCompartment().getName(), memSubDomain.getOutsideCompartment().getName()), Issue.SEVERITY_ERROR);
                                    issueList.add(issue);
                                }
                                //
                                // if either side is missing, then flux term should be identically zero.
                                //
                                if(!bInsidePresent && !jumpCondition.getInFluxExpression().isZero()){
                                    Issue issue = new Issue(equ, issueContext, IssueCategory.MathDescription_SpatialModel_Equation,
                                            VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_12, varName, memSubDomain.getName(), memSubDomain.getInsideCompartment().getName()), Issue.SEVERITY_ERROR);
                                    issueList.add(issue);
                                }
                                if(!bOutsidePresent && !jumpCondition.getOutFluxExpression().isZero()){
                                    Issue issue = new Issue(equ, issueContext, IssueCategory.MathDescription_SpatialModel_Equation,
                                            VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_13, varName, memSubDomain.getName(), memSubDomain.getOutsideCompartment().getName()), Issue.SEVERITY_ERROR);
                                    issueList.add(issue);
                                }

                            }
                        }
                    }
                    if(odeRefCount > 0 && pdeRefCount > 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_SpatialModel,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_14, varName), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    if(steadyPdeCount > 0 && pdeRefCount > 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_SpatialModel,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_15, varName), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    if(odeRefCount == 0 && pdeRefCount == 0 && steadyPdeCount == 0 && measureCount == 0){
                        Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_SpatialModel,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_16, varName), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                }

                //
                // Check that there is an equation (OdeEquation) for a given MembraneVariable on each membrane.
                //
                else if(var instanceof MemVariable){
                    int pdeRefCount = 0;
                    int odeRefCount = 0;
                    int steadyPdeCount = 0;
                    for(int j = 0; j < subDomainList.size(); j++){
                        SubDomain subDomain = subDomainList.get(j);
                        Equation equ = subDomain.getEquation(var);
                        if(equ instanceof PdeEquation){
                            if(((PdeEquation) equ).isSteady()){
                                steadyPdeCount++;
                            } else {
                                pdeRefCount++;
                            }
                        } else if(equ instanceof OdeEquation){
                            odeRefCount++;
                        }
                    }
                    if(odeRefCount > 0 && pdeRefCount > 0){
                        Issue issue = new Issue(var, issueContext, IssueCategory.MathDescription_SpatialModel_Variable,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_14, varName), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    if(steadyPdeCount > 0 && pdeRefCount > 0){
                        Issue issue = new Issue(var, issueContext, IssueCategory.MathDescription_SpatialModel_Variable,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_15, varName), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                    if(odeRefCount == 0 && pdeRefCount == 0 && steadyPdeCount == 0){
                        Issue issue = new Issue(var, issueContext, IssueCategory.MathDescription_SpatialModel_Variable,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_16, varName), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                }
                //
                // Check that there is an equation (OdeEquation) for a given FilamentVariable on each FilamentSubDomain.
                //
                else if(var instanceof FilamentVariable){
                    for(int j = 0; j < subDomainList.size(); j++){
                        SubDomain subDomain = subDomainList.get(j);
                        if(subDomain instanceof FilamentSubDomain){
                            Equation equ = subDomain.getEquation(var);
                            if(!(equ instanceof OdeEquation)){
                                Issue issue = new Issue(var, issueContext, IssueCategory.MathDescription_SpatialModel_Variable,
                                        VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_21, varName, subDomain.getName()), Issue.SEVERITY_ERROR);
                                issueList.add(issue);
                            }
                        }
                    }
                }
                //
                // Check that there is a region equation (VolumeRegionEquation) for a given VolumeRegionVariable on each CompartmentSubDomain.
                //
                else if(var instanceof VolumeRegionVariable){
                    VolumeRegionVariable volRegionVar = (VolumeRegionVariable) var;
                    int count = 0;
                    for(int j = 0; j < subDomainList.size(); j++){
                        SubDomain subDomain = subDomainList.get(j);
                        if(subDomain instanceof CompartmentSubDomain){
                            Equation equ = subDomain.getEquation(volRegionVar);
                            if(equ instanceof VolumeRegionEquation){
                                count++;

                                //
                                // for each VolumeRegionEquation, make sure that a jump condition all membranes that border this compartment
                                //
                                for(int k = 0; k < subDomainList.size(); k++){
                                    SubDomain subDomain2 = subDomainList.get(k);
                                    if(subDomain2 instanceof MembraneSubDomain){
                                        MembraneSubDomain membraneSubDomain = (MembraneSubDomain) subDomain2;
                                        if(membraneSubDomain.getInsideCompartment() == subDomain || membraneSubDomain.getOutsideCompartment() == subDomain){
                                            if(membraneSubDomain.getJumpCondition(volRegionVar) == null){
                                                Issue issue = new Issue(var, issueContext, IssueCategory.MathDescription_SpatialModel_Variable,
                                                        VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_17, varName, subDomain.getName(), membraneSubDomain.getName()), Issue.SEVERITY_ERROR);
                                                issueList.add(issue);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                    if(count == 0){
                        Issue issue = new Issue(var, issueContext, IssueCategory.MathDescription_SpatialModel_Variable,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_18, varName), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                }
                //
                // Check that there is a region equation (MembraneRegionEquation) for a given MembraneRegionVariable on each MembraneSubDomain.
                //
                else if(var instanceof MembraneRegionVariable){
                    int count = 0;
                    for(int j = 0; j < subDomainList.size(); j++){
                        SubDomain subDomain = subDomainList.get(j);
                        if(subDomain instanceof MembraneSubDomain){
                            Equation equ = subDomain.getEquation(var);
                            if(equ instanceof MembraneRegionEquation){
                                count++;
                            }
                        }
                    }
                    if(count == 0){
                        Issue issue = new Issue(var, issueContext, IssueCategory.MathDescription_SpatialModel_Variable,
                                VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_19, varName), Issue.SEVERITY_ERROR);
                        issueList.add(issue);
                    }
                }
                //
                // Check that there is a region equation (FilamentRegionEquation) for a given FilamentRegionVariable on each FilamentSubDomain.
                //
                else if(var instanceof FilamentRegionVariable){
                    for(int j = 0; j < subDomainList.size(); j++){
                        SubDomain subDomain = subDomainList.get(j);
                        if(subDomain instanceof FilamentSubDomain){
                            Equation equ = subDomain.getEquation(var);
                            if(!(equ instanceof FilamentRegionEquation)){
                                Issue issue = new Issue(var, issueContext, IssueCategory.MathDescription_SpatialModel_Variable,
                                        VCellErrorMessages.getErrorMessage(VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_20, varName, subDomain.getName()), Issue.SEVERITY_ERROR);
                                issueList.add(issue);
                            }
                        }
                    }
                }
            }
        }
        if(eventList.size() > 0 && isSpatial()){
            Issue issue = new Issue(this, issueContext, IssueCategory.MathDescription_SpatialModel, VCellErrorMessages.MATH_DESCRIPTION_SPATIAL_MODEL_22, Issue.SEVERITY_ERROR);
            issueList.add(issue);
        }
        for(Event event : eventList){
            try {
                event.bind(this);
            } catch(ExpressionBindingException e){
                Issue issue = new Issue(event, issueContext, IssueCategory.MathDescription_SpatialModel_Event, e.getMessage(), Issue.SEVERITY_ERROR);
                issueList.add(issue);
            }
        }

        for(DataGenerator dataGenerator : postProcessingBlock.getDataGeneratorList()){
            try {
                dataGenerator.bind(this);
            } catch(ExpressionBindingException e){
                Issue issue = new Issue(dataGenerator, issueContext, IssueCategory.MathDescription_SpatialModel_PostProcessingBlock, e.getMessage(), Issue.SEVERITY_ERROR);
                issueList.add(issue);
            }
        }
    }


    /**
     * Insert the method's description here.
     * Creation date: (10/9/2002 10:54:06 PM)
     *
     * @return cbit.vcell.math.MathDescription
     */
    void makeCanonical(MathSymbolTableFactory mathSymbolTableFactory) throws MathException, ExpressionException{

        // this method operates on argument; call createCanonicalMathDescription to make new one
        MathDescription newMath = this;

        boolean bRoundCoefficients = false;

        //
        // make a "identity" simulation (no overrides), this will help to substitute/flatten expressions.
        //
        MathSymbolTable mathSymbolTable = mathSymbolTableFactory.createMathSymbolTable(newMath);

        //
        // for top-level objects, substitute all events,
        //
        for(Event event : eventList){
            event.flatten(mathSymbolTable, bRoundCoefficients);
        }

        for(Variable var : variableList){
            if(var instanceof RandomVariable){
                ((RandomVariable) var).flatten(mathSymbolTable, bRoundCoefficients);
            }
        }

        if(postProcessingBlock != null){
            postProcessingBlock.flatten(mathSymbolTable, bRoundCoefficients);
        }

        //
        // for each subdomain, substitute all rates, initial conditions, boundary conditions, jump conditions
        //
        for(int i = 0; i < newMath.subDomainList.size(); i++){
            SubDomain subDomain = newMath.subDomainList.get(i);
            Enumeration<Equation> equEnum = subDomain.getEquations();
            while (equEnum.hasMoreElements()) {
                Equation equ = equEnum.nextElement();
                equ.flatten(mathSymbolTable, bRoundCoefficients);
            }
            for(VarIniCondition varIniCondition : subDomain.getVarIniConditions()){
                varIniCondition.flatten(mathSymbolTable, bRoundCoefficients);
            }
            for(JumpProcess jumpProcess : subDomain.getJumpProcesses()){
                jumpProcess.flatten(mathSymbolTable, bRoundCoefficients);
            }
            for(ParticleJumpProcess jumpProcess : subDomain.getParticleJumpProcesses()){
                jumpProcess.flatten(mathSymbolTable, bRoundCoefficients);
            }
            for(ParticleProperties particleProperty : subDomain.getParticleProperties()){
                particleProperty.flatten(mathSymbolTable, bRoundCoefficients);
            }
            if(subDomain instanceof MembraneSubDomain){
                Enumeration<JumpCondition> jcEnum = ((MembraneSubDomain) subDomain).getJumpConditions();
                while (jcEnum.hasMoreElements()) {
                    JumpCondition jc = jcEnum.nextElement();
                    jc.flatten(mathSymbolTable, bRoundCoefficients);
                }
            }
        }

        //
        // remove all "dummy" equations (rate=0,initial=0)
        // remove all Priorities (shouldn't be used in BioModels)
        //
        for(int i = 0; i < newMath.subDomainList.size(); i++){
            SubDomain subDomain = newMath.subDomainList.get(i);
            subDomain.trimTrivialEquations(newMath);
        }

        //
        // substitute all fast rates, fast invariants
        //
        for(int i = 0; i < newMath.subDomainList.size(); i++){
            SubDomain subDomain = newMath.subDomainList.get(i);
            FastSystem fastSystem = subDomain.getFastSystem();
            if(fastSystem != null){
                fastSystem.flatten(mathSymbolTable, bRoundCoefficients);
            }
        }

        //
        // get rid of all non-variables (functions, constants).
        //
        ArrayList<Variable> newVarList = new ArrayList<Variable>(newMath.variableList);
        Iterator<Variable> newVarListIter = newVarList.iterator();
        while (newVarListIter.hasNext()) {
            Variable var = newVarListIter.next();
            if(var instanceof Constant || var instanceof Function || var instanceof InsideVariable || var instanceof OutsideVariable){
                newVarListIter.remove();
            }
        }
        Variable newVariables[] = newVarList.toArray(new Variable[newVarList.size()]);
        newMath.setAllVariables(newVariables);

        //
        // rebind all mathDescription
        //
        for(int i = 0; i < newMath.subDomainList.size(); i++){
            SubDomain subDomain = newMath.subDomainList.get(i);
            FastSystem fastSystem = subDomain.getFastSystem();
            if(fastSystem != null){
                fastSystem.rebind();
            }
            Enumeration<Equation> equEnum = subDomain.getEquations();
            while (equEnum.hasMoreElements()) {
                Equation equ = equEnum.nextElement();
                equ.bind(mathSymbolTable);
            }
            if(subDomain instanceof MembraneSubDomain){
                Enumeration<JumpCondition> jcEnum = ((MembraneSubDomain) subDomain).getJumpConditions();
                while (jcEnum.hasMoreElements()) {
                    JumpCondition jc = jcEnum.nextElement();
                    jc.bind(mathSymbolTable);
                }
            }
        }
    }

    /**
     * conditionally transcribe comments from tokens to provider, if provider implements {@link Commented}
     *
     * @param provider
     * @param token
     */
    private void transcribeComments(Object provider, Token token){
        if(provider instanceof Commented){
            Commented c = (Commented) provider;
            c.setBeforeComment(token.getBeforeComment());
            c.setAfterComment(token.getAfterComment());
        }
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @param tokens java.util.StringTokenizer
     * @throws java.lang.Exception The exception description.
     */
    public void read_database(CommentStringTokenizer tokens) throws MathException{

        clearAll();

        VariableHash varHash = new VariableHash();
        try {
            Token token = tokens.next();
            if(token.getValue().equalsIgnoreCase(VCML.MathDescription)){
                //token = tokens.nextToken();
                //setName(token);
                //CHECK THIS!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                //if(!token.equals(argName)){
                //	throw new DataAccessException("MathDescription Version Name and Token Name Don't match");
                //}
                String tokenStr = tokens.nextToken();
                if(!tokenStr.equalsIgnoreCase(VCML.BeginBlock)){
                    throw new MathException("unexpected token " + tokenStr + " expecting " + VCML.BeginBlock);
                }
            }
            while (tokens.hasMoreTokens()) {
                token = tokens.next();
                String tokenStr = token.getValue();
                if(tokenStr.equalsIgnoreCase(VCML.EndBlock)){
                    break;
                }
                if(tokenStr.equalsIgnoreCase(VCML.VolumeVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    VolVariable var = new VolVariable(name, domain);
                    transcribeComments(var, token);
                    varHash.addVariable(var);
//
// done in addVariable0()
//
//			InsideVariable inside = new InsideVariable(token+"_INSIDE", token);
//			addVariable0(inside);
//			OutsideVariable outside = new OutsideVariable(token+"_OUTSIDE", token);
//			addVariable0(outside);
                    continue;
                }
                //
                // this is still here to gracefully read old mathDescriptions
                //
                if(tokenStr.equalsIgnoreCase(VCML.Task)){
                    while (tokens.hasMoreTokens()) {
                        tokenStr = tokens.nextToken(); // toss away until end of block
                        if(tokenStr.equalsIgnoreCase(VCML.EndBlock)){
                            break;
                        }
                    }
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.MembraneVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    MemVariable var = new MemVariable(name, domain);
                    transcribeComments(var, token);
                    varHash.addVariable(var);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.FilamentVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    FilamentVariable var = new FilamentVariable(name, domain);
                    transcribeComments(var, token);
                    varHash.addVariable(var);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.PointVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    PointVariable var = new PointVariable(name, domain);
                    transcribeComments(var, token);
                    varHash.addVariable(var);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.VolumeRegionVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    VolumeRegionVariable var = new VolumeRegionVariable(name, domain);
                    transcribeComments(var, token);
                    varHash.addVariable(var);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.MembraneRegionVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    MembraneRegionVariable var = new MembraneRegionVariable(name, domain);
                    transcribeComments(var, token);
                    varHash.addVariable(var);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.FilamentRegionVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    FilamentRegionVariable var = new FilamentRegionVariable(name, domain);
                    transcribeComments(var, token);
                    varHash.addVariable(var);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.Constant)){
                    tokenStr = tokens.nextToken();
                    Expression exp = MathFunctionDefinitions.fixFunctionSyntax(tokens);
                    Constant constant = new Constant(tokenStr, exp);
                    transcribeComments(constant, token);
                    varHash.addVariable(constant);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase("Parameter")){
                    tokenStr = tokens.nextToken();
                    ParameterVariable pv = new ParameterVariable(tokenStr);
                    transcribeComments(pv, token);
                    varHash.addVariable(pv);
                    continue;
                }
                //stochastic variable
                if(tokenStr.equalsIgnoreCase(VCML.StochVolVariable)){
                    tokenStr = tokens.nextToken();
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    StochVolVariable var = new StochVolVariable(name);
                    transcribeComments(var, token);
                    varHash.addVariable(var);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.VolumeParticleVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    VolumeParticleVariable var = new VolumeParticleVariable(name, domain);
                    transcribeComments(var, token);
                    varHash.addVariable(var);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.MembraneParticleVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    MembraneParticleVariable var = new MembraneParticleVariable(name, domain);
                    transcribeComments(var, token);
                    varHash.addVariable(var);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.ParticleMolecularType)){
                    tokenStr = tokens.nextToken();
                    String name = tokenStr;
                    tokenStr = tokens.nextToken();
                    ParticleMolecularType particleMolecularType = new ParticleMolecularType(name);
                    if(!tokenStr.equalsIgnoreCase(VCML.BeginBlock)){
                        throw new MathException("unexpected token " + tokenStr + " expecting " + VCML.BeginBlock);
                    }
                    particleMolecularType.read(tokens);
                    addParticleMolecularType(particleMolecularType);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.LangevinParticleMolecularType)){
                    tokenStr = tokens.nextToken();
                    String name = tokenStr;
                    tokenStr = tokens.nextToken();
                    ParticleMolecularType particleMolecularType = new LangevinParticleMolecularType(name);
                    if(!tokenStr.equalsIgnoreCase(VCML.BeginBlock)){
                        throw new MathException("unexpected token " + tokenStr + " expecting " + VCML.BeginBlock);
                    }
                    particleMolecularType.read(tokens);
                    addParticleMolecularType(particleMolecularType);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.VolumeParticleObservable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    VolumeParticleObservable particle = new VolumeParticleObservable(name, domain, ObservableType.Molecules);
                    // TODO: here
                    particle.read(this, tokens, varHash);
                    varHash.addVariable(particle);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.VolumeParticleSpeciesPattern)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    VolumeParticleSpeciesPattern var = new VolumeParticleSpeciesPattern(name, domain, null); // will set the locationName inside .read()
                    var.read(this, tokens);
                    // this is not really a variable and shouldn't be in the hash
                    // sadly the way we check consistency for compartments makes in necessary
                    varHash.addVariable(var);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.Function)){
                    tokenStr = tokens.nextToken();
                    Expression exp = MathFunctionDefinitions.fixFunctionSyntax(tokens);
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);

/** ---------------------------------------------------------------
 * ATTENTATION: this is a quick fix for a specific user to load his model
 * with a function name as "ATP/ADP".  This syntax is not allowed.
 -----------------------------------------------------------------------*/
                    if(name.equals("ATP/ADP")){
                        name = "ATP_ADP_renamed";
                        System.err.print("Applying species function name change ATP/ADP to ATP_ADP for a specific user (key=2288008)");
                        Thread.dumpStack();
                    }

                    Function function = new Function(name, exp, domain);
                    transcribeComments(function, token);
                    varHash.addVariable(function);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.CompartmentSubDomain)){
                    if(variableList.size() == 0){
                        setAllVariables(varHash.getAlphabeticallyOrderedVariables());
                    }
                    CompartmentSubDomain subDomain = new CompartmentSubDomain(token, this, tokens);
                    addSubDomain0(subDomain);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.MembraneSubDomain)){
                    if(variableList.size() == 0){
                        setAllVariables(varHash.getAlphabeticallyOrderedVariables());
                    }
                    MembraneSubDomain subDomain = MembraneSubDomain.create(this, token, tokens);
                    addSubDomain0(subDomain);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.FilamentSubDomain)){
                    if(variableList.size() == 0){
                        setAllVariables(varHash.getAlphabeticallyOrderedVariables());
                    }
                    tokenStr = tokens.nextToken();
                    String subDomainName = tokenStr;
                    tokenStr = tokens.nextToken();
                    CompartmentSubDomain outsideCompartment = getCompartmentSubDomain(tokenStr);
                    if(outsideCompartment == null){
                        throw new MathFormatException("defined membrane subdomain without a corresponding outside volume subdomain first");
                    }
                    FilamentSubDomain subDomain = new FilamentSubDomain(subDomainName, outsideCompartment);
                    subDomain.read(this, tokens);
                    addSubDomain0(subDomain);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.PointSubDomain)){
                    if(variableList.size() == 0){
                        setAllVariables(varHash.getAlphabeticallyOrderedVariables());
                    }
                    tokenStr = tokens.nextToken();
                    String subDomainName = tokenStr;
                    PointSubDomain subDomain = new PointSubDomain(subDomainName);
                    subDomain.parseBlock(this, tokens);
                    addSubDomain0(subDomain);
                    continue;
                }
                //
                // this is here so that old mathDescriptions are read gracefully.
                //
                if(tokenStr.equalsIgnoreCase(VCML.Mesh)){
                    while (tokens.hasMoreTokens()) {
                        tokenStr = tokens.nextToken(); // toss away until end of block
                        if(tokenStr.equalsIgnoreCase(VCML.EndBlock)){
                            break;
                        }
                    }
                    continue;
                }

                if(tokenStr.equalsIgnoreCase(VCML.Event)){
                    if(variableList.size() == 0){
                        setAllVariables(varHash.getAlphabeticallyOrderedVariables());
                    }
                    //Event event = new Event(tokenStr, this, tokens);
                    Event event = new Event(token, this, tokens);
                    eventList.add(event);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.VolumeRandomVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    RandomVariable randomVariable = new VolumeRandomVariable(name, this, tokens, domain);
                    transcribeComments(randomVariable, token);
                    varHash.addVariable(randomVariable);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.MembraneRandomVariable)){
                    tokenStr = tokens.nextToken();
                    Domain domain = Variable.getDomainFromCombinedIdentifier(tokenStr);
                    String name = Variable.getNameFromCombinedIdentifier(tokenStr);
                    RandomVariable randomVariable = new MembraneRandomVariable(name, this, tokens, domain);
                    transcribeComments(randomVariable, token);
                    varHash.addVariable(randomVariable);
                    continue;
                }
                if(tokenStr.equalsIgnoreCase(VCML.PostProcessingBlock)){
                    postProcessingBlock.read(tokens);
                    continue;
                }
                throw new MathFormatException("unexpected identifier " + tokenStr);
            }
        } catch(Exception e){
            throw new MathException("line #" + tokens.lineIndex() + " Exception: " + e.getMessage(), e);
        }
        refreshDependencies();
        fireStateChanged();
    }


    public void addParticleMolecularType(ParticleMolecularType particleMolecularType){
        if(!particleMolecularTypes.contains(particleMolecularType)){
            particleMolecularTypes.add(particleMolecularType);
        }
    }


    public void refreshDependencies(){
        for(Variable var : Collections.list(getVariables())){
            try {
                var.bind(this);
            } catch(ExpressionBindingException e){
                lg.warn("unable to bind expression for math variable " + var.getName() + " when reading from VCML: " + e.getMessage());
            }
        }

        for(SubDomain subDomain : this.subDomainList){
            subDomain.refreshDependencies(this);
        }
    }


    /**
     * Remove a javax.swing.event.ChangeListener.
     */
    public void removeChangeListener(javax.swing.event.ChangeListener newListener){
        if(aChangeListener != null){
            aChangeListener.remove(newListener);
        }
        ;
    }


    public synchronized void removePropertyChangeListener(java.beans.PropertyChangeListener listener){
        getPropertyChange().removePropertyChangeListener(listener);
    }

    public synchronized void removeVetoableChangeListener(java.beans.VetoableChangeListener listener){
        getVetoPropertyChange().removeVetoableChangeListener(listener);
    }


    public void setAllVariables(Variable vars[]) throws MathException, ExpressionBindingException{
        // make sure it's OK
        VariableHash hash = new VariableHash();
        for(int i = 0; i < vars.length; i++){
            hash.addVariable(vars[i]);
        }
        hash.getTopologicallyReorderedVariables();

        variableList.clear();
        variableHashTable.clear();
        // adding without binding
        for(Variable var : vars){
            if(var instanceof InsideVariable || var instanceof OutsideVariable){ // if vars has inside and outside already, ignore
                continue;
            }
            if(getVariable(var.getName()) != null){
                throw new MathException("variable " + var.getName() + " already exists");
            }
            variableList.add(var);
            variableHashTable.put(var.getName(), var);
            if(var instanceof VolVariable || var instanceof VolumeRegionVariable){
                //
                // for Volume Variables, also create an InsideVariable and an OutsideVariable for use in JumpConditions
                //
                InsideVariable inVar = new InsideVariable(var.getName() + InsideVariable.INSIDE_VARIABLE_SUFFIX, var.getName());
                variableList.add(inVar);
                variableHashTable.put(inVar.getName(), inVar);
                OutsideVariable outVar = new OutsideVariable(var.getName() + OutsideVariable.OUTSIDE_VARIABLE_SUFFIX, var.getName());
                variableList.add(outVar);
                variableHashTable.put(outVar.getName(), outVar);
            }
        }
        // bind each variable
        Iterator<Variable> iter = variableList.iterator();
        while (iter.hasNext()) {
            iter.next().bind(this);
        }

        for(int i = 0; i < subDomainList.size(); i++){
            SubDomain subDomain = subDomainList.get(i);
            Enumeration<Equation> equEnum = subDomain.getEquations();
            while (equEnum.hasMoreElements()) {
                Equation equ = equEnum.nextElement();
                equ.bind(this);
            }
            FastSystem fastSystem = subDomain.getFastSystem();
            if(fastSystem != null){
                Enumeration<FastRate> frEnum = fastSystem.getFastRates();
                while (frEnum.hasMoreElements()) {
                    FastRate fr = frEnum.nextElement();
                    fr.bind(this);
                }
                Enumeration<FastInvariant> fiEnum = fastSystem.getFastInvariants();
                while (fiEnum.hasMoreElements()) {
                    FastInvariant fi = fiEnum.nextElement();
                    fi.bind(this);
                }
            }
        }
        for(Event event : eventList){
            event.bind(this);
        }
        fireStateChanged();
    }


    /**
     * Sets the description property (java.lang.String) value.
     *
     * @param description The new value for the property.
     * @throws java.beans.PropertyVetoException The exception description.
     * @see #getDescription
     */
    public void setDescription(java.lang.String description) throws java.beans.PropertyVetoException{
        String oldValue = fieldDescription;
        fireVetoableChange("description", oldValue, description);
        fieldDescription = description;
        firePropertyChange("description", oldValue, description);
    }

    public void setGeometry(Geometry argGeometry) throws java.beans.PropertyVetoException{
        Geometry oldValue = this.geometry;
        fireVetoableChange(GeometryOwner.PROPERTY_NAME_GEOMETRY, oldValue, argGeometry);
//	this.geometry = argGeometry;
        setGeometry0(argGeometry);
        firePropertyChange(GeometryOwner.PROPERTY_NAME_GEOMETRY, oldValue, argGeometry);
        fireStateChanged();
        //
        // compute warning string (if necessary)
        //
        isValid();

    }

    private void setGeometry0(Geometry geometry){
        if(this.geometry != geometry){
            this.geometry = geometry;
        }
    }

    public void setName(java.lang.String name) throws java.beans.PropertyVetoException{
        String oldValue = fieldName;
        fireVetoableChange("name", oldValue, name);
        fieldName = name;
        firePropertyChange("name", oldValue, name);
    }


    /**
     * Sets the warning property (java.lang.String) value.
     *
     * @param warning The new value for the property.
     * @see #getWarning
     */
    private void setWarning(java.lang.String warning){
        String oldValue = fieldWarning;
        fieldWarning = warning;
        firePropertyChange("warning", oldValue, warning);
    }


    /**
     * Insert the method's description here.
     * Creation date: (10/9/2002 10:54:06 PM)
     *
     * @return cbit.vcell.math.MathDescription
     */
    void substituteInPlace(MathSymbolTableFactory mathSymbolTableFactory, Function functionsToSubstitute[]) throws MathException, ExpressionException{
        //
        // make a "identity" simulation (no overrides), this will help to substitute/flatten expressions.
        //
        for(int i = 0; i < functionsToSubstitute.length; i++){
            if(functionsToSubstitute[i] != null){
                //variableList.insertElementAt(functionsToSubstitute[i], 0);
                variableList.add(0, functionsToSubstitute[i]);
                variableHashTable.put(functionsToSubstitute[i].getName(), functionsToSubstitute[i]);
            }
        }
        MathSymbolTable simSymbolTable = mathSymbolTableFactory.createMathSymbolTable(this);
        MathDescription newMath = this;
        boolean bRoundCoefficients = false;
        //
        // substitute all rates, initial conditions, boundary conditions, jump conditions, fast rates, fast invariants
        //
        for(int i = 0; i < newMath.subDomainList.size(); i++){
            SubDomain subDomain = newMath.subDomainList.get(i);
            FastSystem fastSystem = subDomain.getFastSystem();
            if(fastSystem != null){
                fastSystem.flatten(simSymbolTable, bRoundCoefficients);
            }
            Enumeration<Equation> equEnum = subDomain.getEquations();
            while (equEnum.hasMoreElements()) {
                Equation equ = equEnum.nextElement();
                equ.flatten(simSymbolTable, bRoundCoefficients);
            }
            if(subDomain instanceof MembraneSubDomain){
                Enumeration<JumpCondition> jcEnum = ((MembraneSubDomain) subDomain).getJumpConditions();
                while (jcEnum.hasMoreElements()) {
                    JumpCondition jc = jcEnum.nextElement();
                    jc.flatten(simSymbolTable, bRoundCoefficients);
                }
            }
        }

        //
        // rebind all mathDescription
        //
        for(int i = 0; i < newMath.subDomainList.size(); i++){
            SubDomain subDomain = newMath.subDomainList.get(i);
            FastSystem fastSystem = subDomain.getFastSystem();
            if(fastSystem != null){
                fastSystem.rebind();
            }
            Enumeration<Equation> equEnum = subDomain.getEquations();
            while (equEnum.hasMoreElements()) {
                Equation equ = equEnum.nextElement();
                equ.bind(simSymbolTable);
            }
            if(subDomain instanceof MembraneSubDomain){
                Enumeration<JumpCondition> jcEnum = ((MembraneSubDomain) subDomain).getJumpConditions();
                while (jcEnum.hasMoreElements()) {
                    JumpCondition jc = jcEnum.nextElement();
                    jc.bind(simSymbolTable);
                }
            }
        }
    }

    public static MathCompareResults testEquivalencyWithRename(MathSymbolTableFactory mathSymbolTableFactory, MathDescription mathDescription1, MathDescription mathDescription2) throws ExpressionBindingException{
        MathCompareResults invariantResults = mathDescription1.compareInvariantAttributes(mathDescription2, false);
        NameScope identityNameScope = new AbstractNameScope() {
            @Override
            public NameScope[] getChildren(){
                return new NameScope[0];
            }

            @Override
            public String getName(){
                return "default";
            }

            @Override
            public NameScope getParent(){
                return null;
            }

            @Override
            public ScopedSymbolTable getScopedSymbolTable(){
                return null;
            }

            @Override
            public String getSymbolName(SymbolTableEntry symbolTableEntry){
                return symbolTableEntry.getName();
            }
        };
        if(!invariantResults.isEquivalent()){
            if(invariantResults.decision == Decision.MathDifferent_VARIABLE_NOT_FOUND_AS_FUNCTION){
                List<String> varsNotFoundMath1 = invariantResults.varsNotFoundMath1;
                List<String> varsNotFoundMath2 = invariantResults.varsNotFoundMath2;
                if(varsNotFoundMath1 != null && varsNotFoundMath1.size() > 0 && varsNotFoundMath2 != null && varsNotFoundMath2.size() == varsNotFoundMath1.size()){
                    for(int i = 0; i < varsNotFoundMath1.size(); i++){
                        String oldName = varsNotFoundMath2.get(i);
                        String newName = varsNotFoundMath1.get(i);
                        mathDescription2.getVariable(oldName).rename(newName);
                    }
                    // find replacement name for each symbol in varsNotFound
                    List<Expression> allExpressions = new ArrayList<>();
                    mathDescription2.getAllExpressions(allExpressions);
                    for(Expression exp : allExpressions){
                        exp.substituteInPlace(exp, exp.renameBoundSymbols(identityNameScope));
                    }
                }
                return testEquivalency(mathSymbolTableFactory, mathDescription1, mathDescription2);
            } else {
                return invariantResults;
            }
        }
        return testEquivalency(mathSymbolTableFactory, mathDescription1, mathDescription2);
    }

    public static MathCompareResults testEquivalency(MathSymbolTableFactory mathSymbolTableFactory, MathDescription oldMath, MathDescription newMath){

        try {

            MathCompareResults invariantResults = newMath.compareInvariantAttributes(oldMath, false);
            if(!invariantResults.isEquivalent()){
                if(invariantResults.decision.equals(Decision.MathDifferent_VARIABLE_NOT_FOUND_AS_FUNCTION)
                        || invariantResults.decision.equals(Decision.MathDifferent_DIFFERENT_NUMBER_OF_VARIABLES)){
                    // this may be due to legacy math format:
                    // variable naming identical across structures
                    // no subdomains associated with variables
                    // equations are present for state variables in multiple subdomains
                    MathDescription copyMath1 = new MathDescription(oldMath); //clone before surgery
                    boolean bRenameSuccess = tryLegacyVarNameDomainExtensive(copyMath1, newMath);
                    if(bRenameSuccess){
                        //try again to see whether invariants are equivalent after rename
                        MathCompareResults invariantResults2 = newMath.compareInvariantAttributes(copyMath1, false);
                        if(!invariantResults2.isEquivalent()){
                            lg.error("Could not fix invariants by renaming");
                            lg.error("Initial invariantResults: " + invariantResults);
                            lg.error("After rename: " + invariantResults2);
                            if(!invariantResults2.decision.equals(Decision.MathDifferent_DIFFERENT_NUMBER_OF_VARIABLES)){
                                // can't be equivalent or equal
                                return invariantResults2;
                            }
                        }
                        // either now it works for the quick test or fails again with different number of vars
                        // number of vars can be also due to mass conservation choice differences which is not fixed by renaming, but tested later with expansion
                        // now proceed with the patched copy
                        oldMath = copyMath1;
                    } else {
                        // could not rename variables, may not have legacy naming issues
                        lg.error("Attempt to rename legacy variables failed");
                        if(!invariantResults.decision.equals(Decision.MathDifferent_DIFFERENT_NUMBER_OF_VARIABLES)){
                            // can't be equivalent or equal
                            return invariantResults;
                        }
                        // number of vars can be also due to mass conservation choice differences which is not fixed by renaming, but tested later with expansion
                        // we proceed with the original math
                    }
                } else if(!invariantResults.decision.equals(Decision.MathDifferent_DIFFERENT_NUMBER_OF_VARIABLES)){
                    // can't be equivalent or equal
                    return invariantResults;
                }
            }

            // invariants are equivalent, now do a full compare
            if(newMath.compareEqual(oldMath)){
                return new MathCompareResults(Decision.MathEquivalent_NATIVE);
            } else {
                MathDescription[] canonicalMaths = MathUtilities.getCanonicalMathDescriptions(oldMath, newMath);
                MathDescription canonicalMath1 = canonicalMaths[0];
                MathDescription canonicalMath2 = canonicalMaths[1];
                // now compare
                return canonicalMath1.compareEquivalentCanonicalMath(canonicalMath2);
            }
        } catch(Exception e){
            String msg = "failure while testing for math equivalency: " + e.getMessage();
            lg.error(msg, e);
            return new MathCompareResults(Decision.MathDifferent_FAILURE_UNKNOWN, e.getMessage());
        }
    }

    public void getAllExpressions(List<Expression> expressionList){
        for(Variable var : variableList){
            if(var.getExpression() != null){
                expressionList.add(var.getExpression());
            }
        }
        for(SubDomain subDomain : subDomainList){
            subDomain.getAllExpressions(expressionList, this);
        }
    }

    public boolean hasDiscontinuities() throws ExpressionException{
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain sd = enum1.nextElement();
            Enumeration<Equation> enum_equ = sd.getEquations();
            while (enum_equ.hasMoreElements()) {
                Equation equation = enum_equ.nextElement();
                if(equation.hasDiscontinuities(this)){
                    return true;
                }
            }
            if(sd.getFastSystem() != null){
                Expression[] exps = sd.getFastSystem().getExpressions();
                for(int i = 0; i < exps.length; i++){
                    if(exps[i].getDiscontinuities().size() > 0){
                        return true;
                    }
                }
            }
        }
        for(Variable var : variableList){
            Expression expression = var.getExpression();
            if(expression != null && expression.getDiscontinuities().size() > 0){
                return true;
            }
        }
        return false;
    }


    /**
     * This method was created by a SmartGuide.
     *
     * @return java.lang.String
     */
    public String toString(){
        return "Math@" + Integer.toHexString(hashCode()) + "(" + version + ")";
    }


    public MathType getMathType(){
        if(isNonSpatialStoch() || isSpatialStoch() || isSpatialHybrid()){
            return MathType.Stochastic;
        } else if(isRuleBased() && !isLangevin()){
            return MathType.RuleBased;
        } else if(isLangevin()){
            return MathType.SpringSaLaD;
        } else {
            return MathType.Deterministic;
        }
    }


    public void getEntries(Map<String, SymbolTableEntry> entryMap){
        ReservedMathSymbolEntries.getAll(entryMap, false);
        for(Variable v : variableList){
            entryMap.put(v.getName(), v);
        }
    }

    public void addEvent(Event event) throws MathException{
        if(getEvent(event.getName()) != null){
            throw new MathException("Event '" + event.getName() + "' already exists.");
        }
        eventList.add(event);
    }

    public Event getEvent(String name){
        for(Event e : eventList){
            if(e.getName().equals(name)){
                return e;
            }
        }
        return null;
    }

    public boolean hasEvents(){
        return eventList.size() > 0;
    }

    public boolean hasRandomVariables(){
        for(Variable var : variableList){
            if(var instanceof RandomVariable){
                return true;
            }
        }
        return false;
    }

    public boolean hasVolumeRegionEquations(){
        Enumeration<SubDomain> enum1 = getSubDomains();
        while (enum1.hasMoreElements()) {
            SubDomain subDomain = enum1.nextElement();
            if(subDomain instanceof CompartmentSubDomain){
                Enumeration<Equation> equations = subDomain.getEquations();
                while (equations.hasMoreElements()) {
                    if(equations.nextElement() instanceof VolumeRegionEquation){
                        return true;
                    }
                }
            }
        }
        return false;
    }


    public boolean hasRegionSizeFunctions(){
        return bRegionSizeFunctionsUsed;
    }


    public final PostProcessingBlock getPostProcessingBlock(){
        return postProcessingBlock;
    }


    public boolean hasDirichletAtMembrane(){
        for(SubDomain subDomain : subDomainList){
            if(subDomain instanceof CompartmentSubDomain){
                List<BoundaryConditionSpec> bcsList = ((CompartmentSubDomain) subDomain).getBoundaryconditionSpecs();
                for(BoundaryConditionSpec boundaryConditionSpec : bcsList){
                    if(boundaryConditionSpec.getBoundaryConditionType().isDIRICHLET()){
                        return true;
                    }
                }
            }
        }
        return false;
    }


    public ParticleMolecularType getParticleMolecularType(String molecularTypeName){
        for(ParticleMolecularType particleMolecularType : particleMolecularTypes){
            if(particleMolecularType.getName().equals(molecularTypeName)){
                return particleMolecularType;
            }
        }
        return null;
    }

    public List<ParticleMolecularType> getParticleMolecularTypes(){
        return Collections.unmodifiableList(particleMolecularTypes);
    }


    // if at least one species observable is present
// the solver will need a -cb flag in its command line
    public boolean hasSpeciesObservable(){
        Enumeration<Variable> enum1 = getVariables();
        while (enum1.hasMoreElements()) {
            Variable var = enum1.nextElement();
            if(var instanceof ParticleObservable){
                if(((ParticleObservable) var).getType() == ObservableType.Species){
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public boolean isMovingMembrane(){
        Predicate<SubDomain> movingMembrane = s -> {
            MembraneSubDomain m = CastingUtils.downcast(MembraneSubDomain.class, s);
            return m != null && m.isMoving();
        };
        return subDomainList.stream().filter(movingMembrane).findAny().isPresent();
    }

    private static boolean tryLegacyVarNameDomainExtensive(MathDescription mathDescription1, MathDescription mathDescription2)
            throws ExpressionException{
        // will return true if rename compatible
        HashSet<String> stateVarNames1 = mathDescription1.getStateVariableNames();
        HashSet<String> varNames2 = new HashSet<String>();
        for(Variable var : Collections.list(mathDescription2.getVariables())){
            varNames2.add(var.getName());
        }
        ArrayList<String> sv1 = new ArrayList<String>(stateVarNames1);
        ArrayList<String> v2 = new ArrayList<String>(varNames2);
        List<Expression> exps = new ArrayList<Expression>();
        mathDescription1.getAllExpressions(exps);
        boolean bVarsInMultipleSubDomains = false;
        HashMap<String, List<String>> multiDomainVarMap = new HashMap<String, List<String>>(); // old name to new name(s)

        for(int i = 0; i < sv1.size(); i++){
            String oldName = sv1.get(i);
            String newName = null;
            Variable var = mathDescription1.getVariable(oldName);
            int nameMatches = 0;
            ArrayList<String> newNames = new ArrayList<String>();
            for(int j = 0; j < v2.size(); j++){
            }
            for(int j = 0; j < v2.size(); j++){
                boolean bMatch = false;
                if(v2.get(j).contains("_init")
                        || v2.get(j).endsWith(InsideVariable.INSIDE_VARIABLE_SUFFIX)
                        || v2.get(j).endsWith(OutsideVariable.OUTSIDE_VARIABLE_SUFFIX)
                        || v2.get(j).endsWith(AbstractMathMapping.PARAMETER_DIFFUSION_RATE_SUFFIX)
                        || v2.get(j).endsWith(AbstractMathMapping.PARAMETER_VELOCITY_X_SUFFIX)
                        || v2.get(j).endsWith(AbstractMathMapping.PARAMETER_VELOCITY_Y_SUFFIX)
                        || v2.get(j).endsWith(AbstractMathMapping.PARAMETER_VELOCITY_Z_SUFFIX)
                )
                    continue;
                if(v2.get(j).equals(oldName)) bMatch = true;
                int underscorePos = v2.get(j).lastIndexOf("_");
                if(underscorePos >= 1){
                    if(v2.get(j).substring(0, underscorePos).equals(oldName)) bMatch = true;
                    if(v2.get(j).substring(0, underscorePos + 1).equals(oldName))
                        bMatch = true; //oldName may end in underscore
                }
                if(bMatch){
                    newName = v2.get(j);
                    nameMatches++;
                    newNames.add(newName);
                }
            }
            if(nameMatches == 0){
                lg.error("did not find any match for variable " + var.getName());
                return false;
            }
            if(nameMatches > 1){
                bVarsInMultipleSubDomains = true;
                multiDomainVarMap.put(oldName, newNames);
                continue;
            }
            // variable is present in single domain, just substitute and match domain stuff
            try {
                // check if var in new math is constant
                if(mathDescription2.getVariable(newName) instanceof Constant){
                    for(SubDomain sd : mathDescription1.getSubDomainCollection()){
                        Equation eq = sd.getEquation(var);
                        if(eq != null){
                            if(!eq.getRateExpression().isZero()){
                                lg.error("variable " + var.getName() + " mapped to constant but has non-zero rate equation: " + eq.getRateExpression());
                                return false;
                            }
                            if(eq instanceof PdeEquation && !((PdeEquation) eq).getDiffusionExpression().isZero()){
                                lg.error("variable " + var.getName() + " mapped to constant but has non-zero diffusion: " + ((PdeEquation) eq).getDiffusionExpression());
                                return false;
                            }
                            sd.removeEquation(var);
                        }
                    }
                    mathDescription1.variableHashTable.remove(oldName);
                    mathDescription1.variableList.remove(var);
                    if(var instanceof VolVariable){
                        String oldNameIN = oldName + InsideVariable.INSIDE_VARIABLE_SUFFIX;
                        String oldNameOUT = oldName + OutsideVariable.OUTSIDE_VARIABLE_SUFFIX;
                        mathDescription1.variableList.remove(mathDescription1.getVariable(oldNameIN));
                        mathDescription1.variableList.remove(mathDescription1.getVariable(oldNameOUT));
                        mathDescription1.variableHashTable.remove(oldNameIN);
                        mathDescription1.variableHashTable.remove(oldNameOUT);
                    }
                    continue;
                }
                // now fix with new name, domain, etc and substitute
                if(var.getDomain() == null){
                    SubDomain sd = mathDescription1.getSubDomain(mathDescription2.getVariable(newName).getDomain().getName());
                    var.setDomain(new Domain(sd));
                    for(SubDomain sd1 : mathDescription1.getSubDomainCollection()){
                        if(sd1 != sd){
                            sd1.removeEquation(oldName);
                        }
                    }
                }
                var.rename(newName);
                if(var instanceof VolVariable){
                    mathDescription1.getVariable(oldName + InsideVariable.INSIDE_VARIABLE_SUFFIX).rename(newName + InsideVariable.INSIDE_VARIABLE_SUFFIX);
                    mathDescription1.getVariable(oldName + OutsideVariable.OUTSIDE_VARIABLE_SUFFIX).rename(newName + OutsideVariable.OUTSIDE_VARIABLE_SUFFIX);
                }
                for(Expression exp : exps){
                    if(!exp.isNumeric()){
                        exp.substituteInPlace(new Expression(oldName), new Expression(newName));
                        exp.substituteInPlace(new Expression(oldName + InsideVariable.INSIDE_VARIABLE_SUFFIX), new Expression(newName));
                        exp.substituteInPlace(new Expression(oldName + OutsideVariable.OUTSIDE_VARIABLE_SUFFIX), new Expression(newName));
                    }
                }
            } catch(Exception e){
                lg.error("failed to rename legacy variable " + oldName, e);
                return false;
            }
        }

        if(bVarsInMultipleSubDomains){
            for(String oldName : multiDomainVarMap.keySet()){
                String oldNameIN = oldName + InsideVariable.INSIDE_VARIABLE_SUFFIX;
                String oldNameOUT = oldName + OutsideVariable.OUTSIDE_VARIABLE_SUFFIX;
                String oldNameSTOCHINIT = oldName + AbstractMathMapping.MATH_FUNC_SUFFIX_SPECIES_INIT_COUNT;
                List<String> newNames = multiDomainVarMap.get(oldName);
                Variable oldVar = mathDescription1.getVariable(oldName);
                HashMap<SubDomain, String> subDomainMap = new HashMap<SubDomain, String>();
                // check if vars in new math are constants
                int nConst = 0;
                int nNames = newNames.size();
                for(String newName : newNames){
                    if(mathDescription2.getVariable(newName) instanceof Constant) nConst++;
                }
                if(nConst != nNames){
                    // remap only to variables
                    List<String> varNames = new ArrayList<String>();
                    for(String newName : newNames){
                        if(!(mathDescription2.getVariable(newName) instanceof Constant)
                                && !(mathDescription2.getVariable(newName) instanceof Function)) varNames.add(newName);
                    }
                    newNames = varNames;
                } else {
                    // all constants in new math, check that old math is equivalent constant and remove equations
                    for(SubDomain sd : mathDescription1.getSubDomainCollection()){
                        Equation eq = sd.getEquation(oldVar);
                        if(eq != null){
                            if(!eq.getRateExpression().isZero()){
                                lg.error("variable " + oldVar.getName() + " mapped to constant but has non-zero rate equation: " + eq.getRateExpression());
                                return false;
                            }
                            if(eq instanceof PdeEquation && !((PdeEquation) eq).getDiffusionExpression().isZero()){
                                lg.error("variable " + oldVar.getName() + " mapped to constant but has non-zero diffusion: " + ((PdeEquation) eq).getDiffusionExpression());
                                return false;
                            }
                            sd.removeEquation(oldVar);
                        }
                    }
                    mathDescription1.variableHashTable.remove(oldName);
                    mathDescription1.variableList.remove(oldVar);
                    if(oldVar instanceof VolVariable){
                        mathDescription1.variableList.remove(mathDescription1.getVariable(oldNameIN));
                        mathDescription1.variableList.remove(mathDescription1.getVariable(oldNameOUT));
                        mathDescription1.variableHashTable.remove(oldNameIN);
                        mathDescription1.variableHashTable.remove(oldNameOUT);
                    }
                    continue;
                }
                // collect subdomains
                for(String newName : newNames){
                    Variable v = mathDescription2.getVariable(newName);
                    Domain d = v.getDomain();
                    subDomainMap.put(mathDescription1.getSubDomain(d.getName()), newName);
                }
                // first remove equations from where they shouldn't be
                for(SubDomain sd1 : mathDescription1.getSubDomainCollection()){
                    if(!subDomainMap.containsKey(sd1)){
                        sd1.removeEquation(oldName);
                    }
                }
                // remove old variable
                mathDescription1.variableHashTable.remove(oldName);
                mathDescription1.variableList.remove(oldVar);
                if(oldVar instanceof VolVariable){
                    mathDescription1.variableList.remove(mathDescription1.getVariable(oldNameIN));
                    mathDescription1.variableList.remove(mathDescription1.getVariable(oldNameOUT));
                    mathDescription1.variableHashTable.remove(oldNameIN);
                    mathDescription1.variableHashTable.remove(oldNameOUT);
                }
                // create new Variables of same type
                HashSet<String> existingVars = new HashSet<String>(); // can happen, need to avoid naming conflicts
                for(String newName : newNames){
                    if(mathDescription1.getVariable(newName) != null){
                        existingVars.add(newName);
                        SubDomain sd = mathDescription1.getSubDomain(mathDescription2.getVariable(newName).getDomain().getName());
                        mathDescription1.getVariable(newName).setDomain(new Domain(sd));
                        continue;
                    }
                    try {
                        Variable clonedVar = (Variable) BeanUtils.cloneSerializable(oldVar);
                        clonedVar.rename(newName);
                        SubDomain sd = mathDescription1.getSubDomain(mathDescription2.getVariable(newName).getDomain().getName());
                        clonedVar.setDomain(new Domain(sd));
                        mathDescription1.addVariable0(clonedVar);
                    } catch(ClassNotFoundException | ExpressionBindingException | IOException | MathException e){
                        lg.error("could not add new variable " + newName);
                        return false;
                    }
                }
                mathDescription1.getAllExpressions(exps);
                // replace variable for equations which should stay
                boolean bStochastic = false;
                for(String newName : newNames){
                    String newNameSTOCHINIT = newName + AbstractMathMapping.MATH_FUNC_SUFFIX_SPECIES_INIT_COUNT;
                    Variable newVar = mathDescription1.getVariable(newName);
                    SubDomain sd = mathDescription1.getSubDomain(newVar.getDomain().getName());
                    if(!sd.getEquationCollection().isEmpty() && !existingVars.contains(newName)){
                        sd.getEquation(oldVar).setVar(newVar);
                    }
                    if(!sd.getJumpProcesses().isEmpty()){
                        bStochastic = true;
                        for(JumpProcess jp : sd.getJumpProcesses()){
                            for(Action a : jp.getActions()){
                                if(a.getVar() == oldVar) a.setVar(newVar);
                            }
                        }
                    }
                    if(bStochastic){
                        // need to rename init vars and substitute
                        for(Variable var : mathDescription1.variableList){
                            if(var.getName().equals(oldNameSTOCHINIT)){
                                var.rename(newNameSTOCHINIT);
                                for(Expression exp : exps){
                                    if(!exp.isNumeric()){
                                        exp.substituteInPlace(new Expression(oldNameSTOCHINIT), new Expression(newNameSTOCHINIT));
                                    }
                                }
                            }
                        }

                        // need to rename init conditions
                        for(VarIniCondition vic : sd.getVarIniConditions()){
                            if(vic.getVar().getName().equals(oldName)){
                                vic.getVar().rename(newName);
                            }
//							vic.getIniVal().substituteInPlace(
//									new Expression(oldName + AbstractMathMapping.MATH_FUNC_SUFFIX_SPECIES_INIT_COUNT),
//									new Expression(
//											newName + AbstractMathMapping.MATH_FUNC_SUFFIX_SPECIES_INIT_COUNT));
//						}
                        }
                    }
                }
                // split jump conditions
                for(SubDomain sd : mathDescription1.getSubDomainCollection()){
                    if(sd instanceof MembraneSubDomain){
                        MembraneSubDomain msd = (MembraneSubDomain) sd;
                        JumpCondition jc = msd.getJumpCondition(oldVar);
                        if(jc != null){
                            try {
                                CompartmentSubDomain isd = msd.getInsideCompartment();
                                CompartmentSubDomain osd = msd.getOutsideCompartment();
                                if(isd != null && subDomainMap.get(isd) != null){
                                    JumpCondition ijc = (JumpCondition) BeanUtils.cloneSerializable(jc);
                                    ijc.setVar(mathDescription1.getVariable(subDomainMap.get(isd)));
                                    ijc.setOutFlux(new Expression(0.0));
                                    msd.addJumpCondition(ijc);
                                }
                                if(osd != null && subDomainMap.get(osd) != null){
                                    JumpCondition ojc = (JumpCondition) BeanUtils.cloneSerializable(jc);
                                    ojc.setVar(mathDescription1.getVariable(subDomainMap.get(osd)));
                                    ojc.setInFlux(new Expression(0.0));
                                    msd.addJumpCondition(ojc);
                                }
                                msd.removeJumpCondition(oldVar);
                            } catch(ClassNotFoundException | IOException | MathException e){
                                lg.error("could not split jump condition for " + oldName);
                                return false;
                            }
                        }
                    }
                }
                if(subDomainMap.size() == 1 && subDomainMap.containsValue(oldName)){
                    // don't need to substitute
                    continue;
                }
                // replace symbols in all functions with corresponding new domain variable
                for(Variable func : mathDescription1.variableList){
                    if(func instanceof Function){
                        Expression exp = func.getExpression();
                        if(exp.getSymbols() == null) continue;
                        for(String symbol : exp.getSymbols()){
                            if(symbol.equals(oldName) || symbol.equals(oldNameIN) || symbol.equals(oldNameOUT)){
                                String newName = null;
                                SubDomain sd = null;
                                if(newNames.size() > 1){
                                    // try to figure out domain and which new name to use, hope for matching function name...
                                    sd = mathDescription1.getSubDomain(mathDescription2.getVariable(func.getName()).getDomain().getName());
                                    func.setDomain(new Domain(sd));
                                    newName = subDomainMap.get(sd);
                                } else {
                                    newName = newNames.get(0);
                                    sd = mathDescription1.getSubDomain(mathDescription2.getVariable(newName).getDomain().getName());
                                    func.setDomain(new Domain(sd));
                                }
                                if(newName != null){
                                    exp.substituteInPlace(new Expression(oldName), new Expression(newName));
                                    exp.substituteInPlace(new Expression(oldNameIN), new Expression(newName));
                                    exp.substituteInPlace(new Expression(oldNameOUT), new Expression(newName));
                                } else {
                                    if(sd instanceof MembraneSubDomain){
                                        MembraneSubDomain msd = (MembraneSubDomain) sd;
                                        String newNameIN = subDomainMap.get(msd.getInsideCompartment());
                                        String newNameOUT = subDomainMap.get(msd.getOutsideCompartment());
                                        if((newNameIN) != null)
                                            exp.substituteInPlace(new Expression(oldNameIN), new Expression(newNameIN));
                                        if((newNameOUT) != null)
                                            exp.substituteInPlace(new Expression(oldNameOUT), new Expression(newNameOUT));
                                    }
                                }
                            }
                        }
                    }
                }
                // substitute in fast systems
                for(SubDomain sd : mathDescription1.getSubDomainCollection()){
                    FastSystem fastSystem = sd.getFastSystem();
                    if(fastSystem != null){
                        for(Expression exp : fastSystem.getExpressions()){
                            exp.substituteInPlace(new Expression(oldName), new Expression(subDomainMap.get(sd)));
                        }
                    }
                }
            }
        }
        // finally, rename functions in old math if they map to a state variable in the new math

        return true;
    }

}
