/*
 * Copyright (C) 1999-2011 University of Connecticut Health Center
 *
 * Licensed under the MIT License (the "License").
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *  http://www.opensource.org/licenses/mit-license.php
 */

package cbit.vcell.bionetgen;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.vcell.model.rbm.RbmUtils;
import org.vcell.util.Matchable;
import org.vcell.util.Pair;

import cbit.vcell.parser.Expression;
/**
 * Insert the type's description here.
 * Creation date: (1/13/2006 5:28:44 PM)
 * @author: Jim Schaff
 */
public abstract class BNGSpecies implements Matchable, Serializable {

	//
	// Storing the index of a species in the network file generated by BioNetGen. 
	// This index is required for filling out reactions listed in the network file.
	//
	private int networkFileIndex = 0;			
	private String name = null;
	private Expression concentration = null;

public BNGSpecies(String argName, Expression argConc, int argNetwkFileIndx) {
	super();
	name = argName;
	concentration = argConc;
	networkFileIndex = argNetwkFileIndx;
}

public boolean compareEqual(org.vcell.util.Matchable object) {
	if (this == object) {
		return (true);
	}
	if (object != null && object instanceof BNGSpecies) {
		BNGSpecies species = (BNGSpecies) object;
		//
		// check for true equality
		//
		if (!org.vcell.util.Compare.isEqual(getName(),species.getName())){
			return false;
		}
		if (!org.vcell.util.Compare.isEqualOrNull(getConcentration(),species.getConcentration())){
			return false;
		}
		return true;
	}
	return false;
}

public Expression getConcentration() {
	return concentration;
}
public String getName() {
	return name;			// may contain compartment information
}
public String extractName() {
	if(!(name.contains(":") && name.contains("@"))) {
		return name;
	} else {
		String shortName = name.substring(name.indexOf(":")+1);
		return shortName;
	}
}
public boolean hasCompartment() {
	if(!(name.contains(":") && name.contains("@"))) {
		return false;
	}
	return true;
}
public java.lang.String extractCompartment() {
	if(name.contains(":") && name.contains("@")) {
		// ex: @cyt:Grb2(sh2!1...   will return cyt
		String compartmentName = name.substring(1, name.indexOf(":"));
		return compartmentName;
	} else if(name.contains(RbmUtils.SiteStruct) && name.contains(RbmUtils.SiteProduct)) {
		//  ex: Grb2(AAA~cyt, AAB...   will return cyt
		String compartment = name.substring(name.indexOf(RbmUtils.SiteStruct) + RbmUtils.SiteStruct.length() +1, name.indexOf(RbmUtils.SiteProduct) -1);
		return compartment;
	}
	return null;
}

public int getNetworkFileIndex() {
	return networkFileIndex;
}

public abstract boolean isWellDefined();
public abstract BNGSpecies[] parseBNGSpeciesName();

public void setConcentration(Expression newConc) {
	concentration = newConc;
}
public void setName(java.lang.String newName) {
	name = newName;
}

public String toBnglString() {
	return new String(getNetworkFileIndex() + " " + getName() + " " + getConcentration().infix());
}
public String toString() {
	return new String(getNetworkFileIndex() + ";\t" + getName() + ";\t" + getConcentration().infix());
}
public String toStringMedium() {
	return new String(extractName() + " " + getConcentration().infix());
}
public String toStringShort() {
	return new String(getName());
}

public enum SignatureDetailLevel {
	MoleculesOnly,
	ComponentsAndStates		// molecules and components with states, no bond
}
public static String getShortSignature(BNGSpecies s, SignatureDetailLevel signatureDetaillevel) {
	
	List<BNGSpecies> list = new ArrayList<>(); 
	if(s instanceof BNGComplexSpecies) {
		list.addAll(Arrays.asList(s.parseBNGSpeciesName()));
	} else {
		list.add(s);
	}
	List<String> entities = new ArrayList<>();
	for(BNGSpecies ss : list) {
		String name = ss.getName();
		name = name.substring(0, name.indexOf("("));
		
		String body = "";
		if(signatureDetaillevel == SignatureDetailLevel.ComponentsAndStates) {
			body = getComponentsAndStates(ss);
		}
		name += "(" + body + ")";
		entities.add(name);
	}
	Collections.sort(entities);
	
	String name = "";
	int i = 0;
	for(String str : entities) {
		if(i>0) {
			name += ".";
		}
		i++;
		name += str;
	}
	return name;
}
private static String getComponentsAndStates(BNGSpecies s) {
	
	if(!(s instanceof BNGMultiStateSpecies)) {
		return "";
	}
	BNGMultiStateSpecies mss = (BNGMultiStateSpecies)s;		//  A(s~Y!1,t~X,v!1,u)
	List<String> entities = new ArrayList<>();

	for(BNGSpeciesComponent c : mss.getComponents()) {
		String name = c.getComponentName();
		if(c.getCurrentState() != null) {
			String state = c.getCurrentState();
			if(state.contains("!")) {		// get rid of an explicit bond (if present) ex: from s~Y!1 we only keep  s~Y
				state = state.substring(0, state.indexOf("!"));
			}
			name += "~" + state;
//			System.out.println(name);
			entities.add(name);
		} else {
			continue;					// we don't care about the sites without state, be they with no bond or explicit bond
		}
	}
	Collections.sort(entities);

	String name = "";
	int i = 0;
	for(String str : entities) {
		if(i>0) {
			name += ",";
		}
		i++;
		name += str;
	}
	return name;
}

public static Pair<List<BNGSpecies>, List<BNGSpecies>> diff(List<BNGSpecies> older, BNGSpecies[] newer) {
	List<BNGSpecies> removed = new ArrayList<>();
	List<BNGSpecies> added = new ArrayList<>();
	
	// whatever is present in 'older' list and is missing in 'newer' list - means that it was removed
	boolean found;
	for(BNGSpecies o : older) {
		found = false;
		for(BNGSpecies n : newer) {
			if(o.getNetworkFileIndex() == n.getNetworkFileIndex()) {
				found = true;
				break;
			}
		}
		if(!found) {
//			System.out.println(o);
//			System.out.println(n);
			removed.add(o);
		}
	}
	// whatever is present in 'newer' list and is missing in 'older' list - means that it was added
	for(BNGSpecies n : newer) {
		found = false;
		for(BNGSpecies o : older) {
			if(n.getNetworkFileIndex() == o.getNetworkFileIndex()) {
				found = true;
				break;
			}
		}
		if(!found) {
			added.add(n);
		}
	}
	Pair<List<BNGSpecies>, List<BNGSpecies>> p = new Pair<>(removed, added);
	return p;
}


// =================================================================================================================
public static void main(String[] argv)
{
//	permutingArray(java.util.Arrays.asList(9, 8, 7, 6, 4), 0);
	
	try {
		String a = "EGF(rb!1).EGF(rb!2).EGFR(ecd!1,tmd!3,y1068~p,y1173~u).EGFR(ecd!2,tmd!3,y1068~u,y1173~p)";
		String b = "EGF(rb!1).EGF(rb!2).EGFR(ecd!1,tmd!3,y1068~u,y1173~p).EGFR(ecd!2,tmd!3,y1068~p,y1173~u)";
		String c = "EGF(rb!1).EGF(rb!2).EGFR(ecd!1,tmd!3,y1068~u,y1173~p).EGFR(ecd!2,tmd!3,y1068~p,y1173~v)";	// slightly different
//		String c = "EGF(rb~Y!1).EGF(rb~pY!1).EGFR(ecd!2,tmd!3,y1068~u,y1173~p).EGFR(ecd!2,tmd!3,y1068~p,y1173~u)";
		String d = "A(s,t!+,v!1).B(s~Y,t~Y!+,u~Y!?,v~!1)";
//		String e = "A(s~Y!1).A(s~X).B(s!1).B(s)";
		String e = "B(t~Y!1,s~X,v!1,u,x!3).A(s~Y!2,t~X,v!2,x!3,u)";

		List<BNGSpecies> list = new ArrayList<>();
		
		BNGSpecies aa = new  BNGComplexSpecies(a, new Expression("0.0"), 1);
		BNGSpecies bb = new  BNGComplexSpecies(b, new Expression("0.0"), 2);
		BNGSpecies cc = new  BNGComplexSpecies(c, new Expression("0.0"), 3);
		BNGSpecies dd = new  BNGComplexSpecies(d, new Expression("0.0"), 4);
		BNGSpecies ee = new  BNGComplexSpecies(e, new Expression("0.0"), 5);
//		list.add(aa);
//		list.add(bb);
//		list.add(cc);
//		list.add(dd);
		list.add(ee);
		
		for(BNGSpecies ours : list) {
			
			String sig = getShortSignature(ours, SignatureDetailLevel.ComponentsAndStates);
		}

		
		System.out.println("done");
		
	} catch (Throwable e) {
		System.out.println("Uncaught exception in BNGSpecies.main()");
		e.printStackTrace();
	}
}


//@Override
//public boolean equals(Object thatObject) {
//	if(!(thatObject instanceof BNGSpecies)) {
//		return false;
//	}
//	BNGSpecies that = (BNGSpecies)thatObject;
//
//	
//	
//	return true;
//}

//private List<String> extractComponentPatterns(String mtp) {
//	String input = mtp.substring(mtp.indexOf("(")+1);
//	input = input.substring(0, input.indexOf(")"));
//	System.out.println(input);
//	
//	List<String> mcpList = new ArrayList<> ();
//	String delimiters = "'";
//	StringTokenizer tokenizer = new StringTokenizer(input, delimiters);
//	String token = new String("");
//
//	while (tokenizer.hasMoreTokens()) {
//		token = tokenizer.nextToken();
//		token = token.trim();
//		mcpList.add(token);
//	}
//	Collections.sort(mcpList);
//	return mcpList;
//}
}
