
<!-- README.md is generated from README.Rmd. Please edit that file -->

# Workshop

## Introducing spatial availability, a singly-constrained competitive-access accessibility measure

Antonio Paez (School of Earth, Environment and Society, McMaster
University, Canada)  
email: <paezha@mcmaster.ca>

Anastasia Soukhov (School of Earth, Environment and Society, McMaster
University, Canada)  
email: <soukhoa@mcmaster.ca>

<!-- badges: start -->
<!-- badges: end -->
<!--
ALL IMAGES WERE CREATED BY ANTONIO USING TAYASUI SKETCHES ON AN IPAD
-->

## Summary

Accessibility measures are widely used to summarize the ease of reaching
potential destinations. As such, they combine, into a single summary
measure, properties of the land use system, on the one hand, and the
transportation system and travel behavior on the other. Defined as the
weighted sum of the opportunities that can be reached given the cost of
movement, accessibility is used in transportation planning, health
planning, economic analysis, etc.

This workshop introduces *spatial availability*. Much like
accessibility, spatial availability measures the ease of reaching
potential destinations. However, unlike accessibility, it makes
opportunities available uniquely to members of the population. For
example, a job, once it is available to someone, it is no longer
available to somebody else. In effect, spatial availability is a
singly-constrained accessibility measure that preserves the number of
opportunities.

In this workshop, we explain the intuitions behind spatial availability
and describe the mechanisms to implement it. A key to this is the idea
of *proportional allocation*, and the use of proportional allocation
factors.

The use of proportional allocation factors as a mechanism for
constraining the spatial availability means that the results are easier
to interpret than those obtained from accessibility analysis, and they
are more intuitive as well.

One exercise is provided, meant to be solved by hand. The workshop
finishes with a practical example of implementation in `R`. Data from a
real survey in the Greater Toronto and Hamilton Area and use of package
{accessibility} give hands-on practice that can serve as a launching pad
for your own experiments and applications.

## Materials

- Packages

- [{accessibility}](https://github.com/ipeaGIT/accessibility) (Install
  from CRAN).

- [{dplyr}](https://dplyr.tidyverse.org/) (Install from CRAN).

- [{ggplot}](https://ggplot2.tidyverse.org/) (Install from CRAN).

- [{leaflet}](https://rstudio.github.io/leaflet/) (Install from CRAN).

- [{patchwork}](https://patchwork.data-imaginist.com/articles/patchwork.html)
  (Install from CRAN).

- [{sf}](https://r-spatial.github.io/sf/) (Install from CRAN).

- Data (sourced from the Transportation Tomorrow Survey of the Greater
  Toronto and Hamilton Area)

- [{TTS2016R}](https://soukhova.github.io/TTS2016R/) (Install from
  GitHub):

<!-- -->

    remotes::install_github("soukhova/TTS2016R")

## Reproducible environment

A reproducible environment was created using [{renv}](). The repository
provides all the infrastructure to replicate the environment used to
create the workshop.

The workshop was developed using `R` 4.3.2 (“Eye Holes”).

How to use this repository. Install the appropriate version of `R` (make
sure to pick the correct operating system) - this is the programming
language, it comes as a ‘core package’ here: <https://cran.rstudio.com/>

Install RStudio (make sure to pick the correct operating system) - this
is the an IDE here:
<https://www.rstudio.com/products/rstudio/download/>.

Download the Rtools43 installer (Windows) and run it to install:
<https://cran.r-project.org/bin/windows/Rtools/rtools43/rtools.html>. If
using a Mac this may be achieved through installing Xcode (we don’t have
Macs so not sure if this works, but it should!):
<https://mac.r-project.org/tools/>

Download the code as a .zip file from this repository:
<https://github.com/paezha/Workshop-Spatial-Availability>.
Alternatively, if you work with GitHub, you can fork the repository. The
repository contains a `renv.lock` file that specifies all the versions
of the packages used in the workshop.

If you downloaded the repository as a .zip file, unzip the file and
store it in an apprioriate directory (i.e., NOT the Downloads folder).
Use this folder to work on the workshop.

Double click the file called `Workshop-Spatial-Availability.Rproj`. This
is be the `R` project for the workshop. This will launch RStudio.

You will see a message in your console saying that your library is out
of synch with the lock file. To ‘restore’ the state of the project to
match the versions specified in the `renv.lock` file run the following:

    renv::restore()

Run `renv::status()` in the `R` Console. The output should show a table
where the second and third columns are populated with “y” characters.
This means the versions of packages on your system matches that
specified by the renv.lock file. Congrats!

## Key concepts

- Accessibility.  
- Impedance function.  
- Gravity model.  
- Spatial availability.  
- Proportional allocation.  
- Congestion.  
- Transportation Tomorrow Survey.

## From accessibility to spatial availability

### What does a transportation system do?

The main role of land use systems is to organize activities in space.
Transportation provides the means to connect spatially disparate
activities. Often, the output of these systems is examined in isolation.
For example, the operation of transportation systems is
[measured](https://en.wikipedia.org/wiki/Units_of_measurement_in_transportation)
in units of length, units of mobility (the potential for movement), or
in terms of realized movement (such as VKT, PKT, or t-km).

A more holistic approach to understand the efficiency of transportation
systems is the concept of *accessibility* (Handy and Niemeier 1997).
Accessibility is a summary measure of the joint performance of a land
use-transportation system, and it quantifies the potential to reach
spatially disperse opportunities.

Consider the following situation, with a single population center and a
single center where opportunities (e.g., jobs) are located.

![](README_files/figure-gfm/fig-basic-system-1.png)<!-- -->

How accessible are the opportunities? It depends on the travel behavior
under a given transportation system.

People may be unwilling to travel too far for some opportunities, even
if those opportunities seem essential (i.e., no one travels on a daily
basis from Toronto for a job in Vancouver.) And for some people, the
cost of reaching opportunities may exceed their willingness to pay.

The travel behavior of a population is modelled using an *impedance
function* (also sometimes called a *distance-decay function*, or a
*deterrence function*.)

![](README_files/figure-gfm/fig-basic-system-w-impedance-1.png)<!-- -->

An impedance function captures the effect of the friction of movement:
it is a well-known regularity that most people prefer to spend less
time/effort/money travelling than more. The impedance function models
the propensity to travel by the cost/length/duration of trips. At some
point, that propensity becomes zero, when the cost of reaching the
opportunity outweighs the benefits that the opportunity brings.

### Impedance functions

A number of different impedance functions have been used in the
literature. The simplest is a binary function as follows:

![f(c\_{ij}) = \begin{cases}
1 & \text{if }c\_{ij}\le\delta\\
0 & \text{otherwise}
\end{cases}](https://latex.codecogs.com/png.latex?f%28c_%7Bij%7D%29%20%3D%20%5Cbegin%7Bcases%7D%0A1%20%26%20%5Ctext%7Bif%20%7Dc_%7Bij%7D%5Cle%5Cdelta%5C%5C%0A0%20%26%20%5Ctext%7Botherwise%7D%0A%5Cend%7Bcases%7D "f(c_{ij}) = \begin{cases}
1 & \text{if }c_{ij}\le\delta\\
0 & \text{otherwise}
\end{cases}")

In the above, ![c_ij](https://latex.codecogs.com/png.latex?c_ij "c_ij")
is the cost of reaching destination
![j](https://latex.codecogs.com/png.latex?j "j") from origin
![i](https://latex.codecogs.com/png.latex?i "i"), and
![\delta](https://latex.codecogs.com/png.latex?%5Cdelta "\delta") is a
*threshold*, that is, the tolerance for long trips. What kind of
behavior does this function represent?

![](README_files/figure-gfm/fig-binary-impedance-1.png)<!-- -->

This function has been used in numerous studies, including:

- Kawabata and Shen (2006) chose these values because they approximate
  the mean travel time in the region.
  - ![\delta = 30](https://latex.codecogs.com/png.latex?%5Cdelta%20%3D%2030 "\delta = 30")
    min (Boston and LA)
  - ![\delta = 45](https://latex.codecogs.com/png.latex?%5Cdelta%20%3D%2045 "\delta = 45")
    min (Tokyo)
- Korsu and Wenglenski (2010)
  - ![\delta = 60](https://latex.codecogs.com/png.latex?%5Cdelta%20%3D%2060 "\delta = 60")
    min (Paris)
- Lin, Chen, and Hsieh (2016)
  - ![\delta = 30](https://latex.codecogs.com/png.latex?%5Cdelta%20%3D%2030 "\delta = 30")
    min (Taipei)
- Boisjoly, Moreno-Monroy, and El-Geneidy (2017)
  - ![\delta = 60](https://latex.codecogs.com/png.latex?%5Cdelta%20%3D%2060 "\delta = 60")
    min (Sao Paulo)

An alternative is the inverse power function:

![f(c\_{ij}) = c\_{ij}^{-\beta}](https://latex.codecogs.com/png.latex?f%28c_%7Bij%7D%29%20%3D%20c_%7Bij%7D%5E%7B-%5Cbeta%7D "f(c_{ij}) = c_{ij}^{-\beta}")

This function was used, for example, by Matas, Raymond, and Roig (2010)
for Barcelona and Madrid, with
![c\_{ij](https://latex.codecogs.com/png.latex?c_%7Bij "c_{ij") the
travel time in minutes and
![\beta = -1](https://latex.codecogs.com/png.latex?%5Cbeta%20%3D%20-1 "\beta = -1"):

![f(c\_{ij}) = \frac{1}{c\_{ij}}](https://latex.codecogs.com/png.latex?f%28c_%7Bij%7D%29%20%3D%20%5Cfrac%7B1%7D%7Bc_%7Bij%7D%7D "f(c_{ij}) = \frac{1}{c_{ij}}")

![](README_files/figure-gfm/fig-inverse-power-impedance-1.png)<!-- -->

The negative exponential function is as follows:

![f(c\_{ij}) = \exp(-\beta c\_{ij}^\gamma) = e^{-\beta c\_{ij}^\gamma}](https://latex.codecogs.com/png.latex?f%28c_%7Bij%7D%29%20%3D%20%5Cexp%28-%5Cbeta%20c_%7Bij%7D%5E%5Cgamma%29%20%3D%20e%5E%7B-%5Cbeta%20c_%7Bij%7D%5E%5Cgamma%7D "f(c_{ij}) = \exp(-\beta c_{ij}^\gamma) = e^{-\beta c_{ij}^\gamma}")

Cheng and Bertolini (2013) used this measure for the case of Amsterdam,
with
![\beta = 0.15](https://latex.codecogs.com/png.latex?%5Cbeta%20%3D%200.15 "\beta = 0.15")
and
![\gamma = 1](https://latex.codecogs.com/png.latex?%5Cgamma%20%3D%201 "\gamma = 1"):
![](README_files/figure-gfm/fig-negative-exponential-impedance-1.png)<!-- -->

### What is accessibility?

Accessibility is defined as a weighted sum of opportunities:

![S_i =\sum_j \underbrace{O_j}\_{\text{Opportunities }} \underbrace{f(c\_{ij})}\_{\text{ Weights (impedance)}}](https://latex.codecogs.com/png.latex?S_i%20%3D%5Csum_j%20%5Cunderbrace%7BO_j%7D_%7B%5Ctext%7BOpportunities%20%7D%7D%20%5Cunderbrace%7Bf%28c_%7Bij%7D%29%7D_%7B%5Ctext%7B%20Weights%20%28impedance%29%7D%7D "S_i =\sum_j \underbrace{O_j}_{\text{Opportunities }} \underbrace{f(c_{ij})}_{\text{ Weights (impedance)}}")

Given the spatial distribution of opportunities
![O\_{j}](https://latex.codecogs.com/png.latex?O_%7Bj%7D "O_{j}"), the
propensity to travel
![f(\cdot)](https://latex.codecogs.com/png.latex?f%28%5Ccdot%29 "f(\cdot)"),
and the cost of travel in a given transportation system
![c\_{ij}](https://latex.codecogs.com/png.latex?c_%7Bij%7D "c_{ij}"),
accessibility summarizes the opportunity landscape.

Back to our toy system:

![](README_files/figure-gfm/fig-basic-system-revisited-1.png)<!-- -->

What is the accessibility in these cases?

![](README_files/figure-gfm/fig-accessibility-boston-LA-kawabata-1.png)<!-- -->

![](README_files/figure-gfm/fig-accessibility-tokyo-kawabata-1.png)<!-- -->

![](README_files/figure-gfm/fig-accessibility-barcelona-matas-1.png)<!-- -->

Modify the set up as follows. How does accessibility change?

![](README_files/figure-gfm/fig-basic-system-2-1.png)<!-- -->

What happens if we add a population center? How does accessibility
change?

![](README_files/figure-gfm/fig-basic-system-3-1.png)<!-- -->

Back to our initial system. Let us ask a slightly different question.
How many jobs are taken?

![](README_files/figure-gfm/fig-basic-system-revisited-2-1.png)<!-- -->

The impedance function models the propensity to travel, and at some
point it drops to zero. But that is a summary for the population, and
depends on what opportunities are available.

![](README_files/figure-gfm/fig-basic-system-w-impedance-revisited-1.png)<!-- -->

How many jobs are taken in each of these cases?

<img src="README_files/figure-gfm/jobs-taken-1.png" height="15" />

Could it be that all those jobs remain vacant? What if instead those
**available** jobs were allocated to people who are willing to reach
them? ![](README_files/figure-gfm/jobs-taken-matas-1.png)<!-- -->

Accessibility is low because it is expensive to reach the opportunities.
But, the jobs are there and people are willing to reach them.

![](README_files/figure-gfm/spatially-available-jobs-lu2-1.png)<!-- -->

The number of jobs available equals the total number of jobs in the
region. That is because jobs are allocated **proportionally**.

Proportional allocationo of opportunities means that, unlike
accessibility, the number of opportunities available can change when
there are new population centers.

![](README_files/figure-gfm/spatially-available-jobs-lu3-1.png)<!-- -->

Let us see how this works.

- **O2** has opportunities that are available to both **P1** and **P2**.
  But **P1** can reach those opportunities with less “friction” (it is
  closer).

The total “friction” (impedance) of reaching **O2** is:

![\underbrace{0.1}\_{\text{from P1}} + \overbrace{0.018}^ {\text{from P2}}](https://latex.codecogs.com/png.latex?%5Cunderbrace%7B0.1%7D_%7B%5Ctext%7Bfrom%20P1%7D%7D%20%2B%20%5Coverbrace%7B0.018%7D%5E%20%7B%5Ctext%7Bfrom%20P2%7D%7D "\underbrace{0.1}_{\text{from P1}} + \overbrace{0.018}^ {\text{from P2}}")

Jobs are allocated preferentially to those who can reach them with lower
friction. Then the jobs available to **P1** from **O2** are:

![80\cdot\frac{0.1}{0.118} = 80(0.847) = 67.8](https://latex.codecogs.com/png.latex?80%5Ccdot%5Cfrac%7B0.1%7D%7B0.118%7D%20%3D%2080%280.847%29%20%3D%2067.8 "80\cdot\frac{0.1}{0.118} = 80(0.847) = 67.8")

And, the jobs available to **P2** from **O2** are:

![80\cdot\frac{0.018}{0.118} = 80(0.153) = 12.2](https://latex.codecogs.com/png.latex?80%5Ccdot%5Cfrac%7B0.018%7D%7B0.118%7D%20%3D%2080%280.153%29%20%3D%2012.2 "80\cdot\frac{0.018}{0.118} = 80(0.153) = 12.2")

The total “friction” (impedance) of reaching **O1** is:

![\underbrace{0.028}\_{\text{from P1}} + \overbrace{0.1}^ {\text{from P2}}](https://latex.codecogs.com/png.latex?%5Cunderbrace%7B0.028%7D_%7B%5Ctext%7Bfrom%20P1%7D%7D%20%2B%20%5Coverbrace%7B0.1%7D%5E%20%7B%5Ctext%7Bfrom%20P2%7D%7D "\underbrace{0.028}_{\text{from P1}} + \overbrace{0.1}^ {\text{from P2}}")

Again, opportunities are allocated preferentially to those who can reach
them with lower friction. Then the jobs available to **P1** from **O1**
are:

![200\cdot\frac{0.028}{0.128} = 200(0.219) = 43.8](https://latex.codecogs.com/png.latex?200%5Ccdot%5Cfrac%7B0.028%7D%7B0.128%7D%20%3D%20200%280.219%29%20%3D%2043.8 "200\cdot\frac{0.028}{0.128} = 200(0.219) = 43.8")

And, the jobs available to **P2** from **O1** are:

![200\cdot\frac{0.1}{0.128} = 200(0.781) = 156.2](https://latex.codecogs.com/png.latex?200%5Ccdot%5Cfrac%7B0.1%7D%7B0.128%7D%20%3D%20200%280.781%29%20%3D%20156.2 "200\cdot\frac{0.1}{0.128} = 200(0.781) = 156.2")

The opportunities available to **P1** are as follows:

![\underbrace{200\cdot\frac{0.028}{0.128} = 200(0.219) = 43.8}\_{\text{opportunities available from O1}}](https://latex.codecogs.com/png.latex?%5Cunderbrace%7B200%5Ccdot%5Cfrac%7B0.028%7D%7B0.128%7D%20%3D%20200%280.219%29%20%3D%2043.8%7D_%7B%5Ctext%7Bopportunities%20available%20from%20O1%7D%7D "\underbrace{200\cdot\frac{0.028}{0.128} = 200(0.219) = 43.8}_{\text{opportunities available from O1}}")

![\underbrace{80\cdot\frac{0.1}{0.118} = 80(0.847) = 67.8}\_{\text{opportunities available from O2}}](https://latex.codecogs.com/png.latex?%5Cunderbrace%7B80%5Ccdot%5Cfrac%7B0.1%7D%7B0.118%7D%20%3D%2080%280.847%29%20%3D%2067.8%7D_%7B%5Ctext%7Bopportunities%20available%20from%20O2%7D%7D "\underbrace{80\cdot\frac{0.1}{0.118} = 80(0.847) = 67.8}_{\text{opportunities available from O2}}")

![43.8 + 67.8 = 111.6](https://latex.codecogs.com/png.latex?43.8%20%2B%2067.8%20%3D%20111.6 "43.8 + 67.8 = 111.6")

The opportunities available to **P2** are as follows:

![\underbrace{200\cdot\frac{0.1}{0.128} = 200(0.781) = 156.2}\_{\text{opportunities available from O1}}](https://latex.codecogs.com/png.latex?%5Cunderbrace%7B200%5Ccdot%5Cfrac%7B0.1%7D%7B0.128%7D%20%3D%20200%280.781%29%20%3D%20156.2%7D_%7B%5Ctext%7Bopportunities%20available%20from%20O1%7D%7D "\underbrace{200\cdot\frac{0.1}{0.128} = 200(0.781) = 156.2}_{\text{opportunities available from O1}}")

![\underbrace{80\cdot\frac{0.018}{0.118} = 80(0.153) = 12.2}\_{\text{opportunities available from O2}}](https://latex.codecogs.com/png.latex?%5Cunderbrace%7B80%5Ccdot%5Cfrac%7B0.018%7D%7B0.118%7D%20%3D%2080%280.153%29%20%3D%2012.2%7D_%7B%5Ctext%7Bopportunities%20available%20from%20O2%7D%7D "\underbrace{80\cdot\frac{0.018}{0.118} = 80(0.153) = 12.2}_{\text{opportunities available from O2}}")

![156.2 + 12.2 = 168.4](https://latex.codecogs.com/png.latex?156.2%20%2B%2012.2%20%3D%20168.4 "156.2 + 12.2 = 168.4")

And so, the total jobs available are:

![\underbrace{111.6}\_{\text{to P1}} + \overbrace{168.4}^{\text{to P2}} = 280](https://latex.codecogs.com/png.latex?%5Cunderbrace%7B111.6%7D_%7B%5Ctext%7Bto%20P1%7D%7D%20%2B%20%5Coverbrace%7B168.4%7D%5E%7B%5Ctext%7Bto%20P2%7D%7D%20%3D%20280 "\underbrace{111.6}_{\text{to P1}} + \overbrace{168.4}^{\text{to P2}} = 280")

The proportional allocation mechanism *constrains* the available
opportunities to match the total number of opportunities in the system.
Compare to the accessibility:

![\underbrace{13.6}\_{\text{to P1}} + \overbrace{21.44}^{\text{to P2}} = 35.04](https://latex.codecogs.com/png.latex?%5Cunderbrace%7B13.6%7D_%7B%5Ctext%7Bto%20P1%7D%7D%20%2B%20%5Coverbrace%7B21.44%7D%5E%7B%5Ctext%7Bto%20P2%7D%7D%20%3D%2035.04 "\underbrace{13.6}_{\text{to P1}} + \overbrace{21.44}^{\text{to P2}} = 35.04")

Formalizing this, we see that spatial availability, call it
![V_i](https://latex.codecogs.com/png.latex?V_i "V_i"), is the weighted
and *constrained* sum of opportunities:

![V_i = \sum_j^J O_j\frac{f(c\_{ij})}{\sum_n^N f(c\_{nj})}](https://latex.codecogs.com/png.latex?V_i%20%3D%20%5Csum_j%5EJ%20O_j%5Cfrac%7Bf%28c_%7Bij%7D%29%7D%7B%5Csum_n%5EN%20f%28c_%7Bnj%7D%29%7D "V_i = \sum_j^J O_j\frac{f(c_{ij})}{\sum_n^N f(c_{nj})}")

where:

- ![N](https://latex.codecogs.com/png.latex?N "N") is the number of
  locations with population
- ![J](https://latex.codecogs.com/png.latex?J "J") is the number of
  locations with opportunities

Compare to accessibility:

![S_i = \sum_j^JO_jf(c\_{ij})](https://latex.codecogs.com/png.latex?S_i%20%3D%20%5Csum_j%5EJO_jf%28c_%7Bij%7D%29 "S_i = \sum_j^JO_jf(c_{ij})")

Let us change the set up to illustrate a different point.

![](README_files/figure-gfm/fig-basic-system-4-1.png)<!-- -->

Since the impedance is the same, the previous proportional allocation
mechanism would have us give 100 opportunities to **P1** and 100
opportunities to **P2**, which seems a bit unfair, not to mention
wasteful.

To improve on this situation, opportunities will be allocated
proportionally based on *demand*.

The proportion of the total population at **P1** is:

![\frac{80}{270} = 0.296](https://latex.codecogs.com/png.latex?%5Cfrac%7B80%7D%7B270%7D%20%3D%200.296 "\frac{80}{270} = 0.296")

The proportion of the total population at **P2** is:

![\frac{190}{270} = 0.704](https://latex.codecogs.com/png.latex?%5Cfrac%7B190%7D%7B270%7D%20%3D%200.704 "\frac{190}{270} = 0.704")

In this way, the jobs *available* to **P1** from **O1** is:

![200\cdot \frac{80}{270} = 200(0.296) = 59.26](https://latex.codecogs.com/png.latex?200%5Ccdot%20%5Cfrac%7B80%7D%7B270%7D%20%3D%20200%280.296%29%20%3D%2059.26 "200\cdot \frac{80}{270} = 200(0.296) = 59.26")

And the jobs *available* to **P2** from **O1** is:

![200\cdot \frac{190}{270} = 200(0.704) = 140.74](https://latex.codecogs.com/png.latex?200%5Ccdot%20%5Cfrac%7B190%7D%7B270%7D%20%3D%20200%280.704%29%20%3D%20140.74 "200\cdot \frac{190}{270} = 200(0.704) = 140.74")

This mechanism for allocating opportunities is impartial with respect to
the population: the ratio of opportunities to population is identical,
as expected:

![\frac{59.26}{80} = \frac{140.74}{190} = 0.741](https://latex.codecogs.com/png.latex?%5Cfrac%7B59.26%7D%7B80%7D%20%3D%20%5Cfrac%7B140.74%7D%7B190%7D%20%3D%200.741 "\frac{59.26}{80} = \frac{140.74}{190} = 0.741")

It also satisfies the following:

![\sum_i V_i = \sum_j O_j](https://latex.codecogs.com/png.latex?%5Csum_i%20V_i%20%3D%20%5Csum_j%20O_j "\sum_i V_i = \sum_j O_j")

As we can verify, the sum of available opportunities is preserved:

![59.26 + 140.74 = 200](https://latex.codecogs.com/png.latex?59.26%20%2B%20140.74%20%3D%20200 "59.26 + 140.74 = 200")

When we formalize this we see again that *spatial availability* is a
weighted sum of the opportunities, constrained to match the total:

![V_i = \sum_j O_j\frac{P_i}{\sum_n^N P_n}](https://latex.codecogs.com/png.latex?V_i%20%3D%20%5Csum_j%20O_j%5Cfrac%7BP_i%7D%7B%5Csum_n%5EN%20P_n%7D "V_i = \sum_j O_j\frac{P_i}{\sum_n^N P_n}")

## Spatial availability

We have described two proportional allocation mechanisms:

1.  By the cost of interaction (through the impedance function)

2.  By the size of the population

These are consistent with the principles of spatial interaction (e.g.,
the gravity model):

1.  The friction of space

2.  The importance of size (mass)

We call these quantities *proportional allocation factors*:

![F^C\_{ij} = \frac{f(c\_{ij})}{\sum_n^N f(c\_{nj})}](https://latex.codecogs.com/png.latex?F%5EC_%7Bij%7D%20%3D%20%5Cfrac%7Bf%28c_%7Bij%7D%29%7D%7B%5Csum_n%5EN%20f%28c_%7Bnj%7D%29%7D "F^C_{ij} = \frac{f(c_{ij})}{\sum_n^N f(c_{nj})}")

and:

![F^P_i = \frac{P_i}{\sum_n^N P_m}](https://latex.codecogs.com/png.latex?F%5EP_i%20%3D%20%5Cfrac%7BP_i%7D%7B%5Csum_n%5EN%20P_m%7D "F^P_i = \frac{P_i}{\sum_n^N P_m}")

The two factors are combined as follows:

![F\_{ij}^T = \frac{F^C\_{ij}F^P_i}{\sum_n^NF^C\_{ij}F^P_i}](https://latex.codecogs.com/png.latex?F_%7Bij%7D%5ET%20%3D%20%5Cfrac%7BF%5EC_%7Bij%7DF%5EP_i%7D%7B%5Csum_n%5ENF%5EC_%7Bij%7DF%5EP_i%7D "F_{ij}^T = \frac{F^C_{ij}F^P_i}{\sum_n^NF^C_{ij}F^P_i}")

to give the following expression for *spatial availability*:

![V_i = \sum_j^J O_jF^T\_{ij}](https://latex.codecogs.com/png.latex?V_i%20%3D%20%5Csum_j%5EJ%20O_jF%5ET_%7Bij%7D "V_i = \sum_j^J O_jF^T_{ij}")

Compare again to accessibility:

![S_i = \sum_j^JO_jf(c\_{ij})](https://latex.codecogs.com/png.latex?S_i%20%3D%20%5Csum_j%5EJO_jf%28c_%7Bij%7D%29 "S_i = \sum_j^JO_jf(c_{ij})")

Spatial availability is a measure of the number of opportunities that
are available to a location from all reachable destinations. It is a
singly constrained measure of accessibility.

- It constrains the sum of available opportunities to match the total
  number of opportunities in the region
- It is easier to interpret than accessibility
- It provides intuitive results
- It gives a natural baseline for equity analysis

## Exercise

This exercise is meant to be solved by hand. This is a great way to
develop a good feeling for the mechanics of the method. See the figure
below. It represents a tiny system with only two population centers
(![a](https://latex.codecogs.com/png.latex?a "a") and
![b](https://latex.codecogs.com/png.latex?b "b")) and two employment
centers (![A](https://latex.codecogs.com/png.latex?A "A") and
![B](https://latex.codecogs.com/png.latex?B "B")).

![](images/Exercise.png)

For the exercise use the following impedance function for
![c\_{ij}](https://latex.codecogs.com/png.latex?c_%7Bij%7D "c_{ij}"),
where ![c](https://latex.codecogs.com/png.latex?c "c") is trip duration
in minutes:

![f(c\_{ij}) = \begin{cases}
\frac{1}{c\_{ij}^\beta} &\text{for } c\_{ij}\le\delta\\
0 & \text{for } c\_{ij}\>\delta\\
\end{cases}](https://latex.codecogs.com/png.latex?f%28c_%7Bij%7D%29%20%3D%20%5Cbegin%7Bcases%7D%0A%5Cfrac%7B1%7D%7Bc_%7Bij%7D%5E%5Cbeta%7D%20%26%5Ctext%7Bfor%20%7D%20c_%7Bij%7D%5Cle%5Cdelta%5C%5C%0A0%20%26%20%5Ctext%7Bfor%20%7D%20c_%7Bij%7D%3E%5Cdelta%5C%5C%0A%5Cend%7Bcases%7D "f(c_{ij}) = \begin{cases}
\frac{1}{c_{ij}^\beta} &\text{for } c_{ij}\le\delta\\
0 & \text{for } c_{ij}>\delta\\
\end{cases}")

Assume that the number of people and jobs at each location is as shown
below:

![\begin{array}{l}
P\_{a} = 80 \\
P\_{b} = 120 \\
O\_{A} = 100 \\
O\_{B} = 50 
\end{array}](https://latex.codecogs.com/png.latex?%5Cbegin%7Barray%7D%7Bl%7D%0AP_%7Ba%7D%20%3D%2080%20%5C%5C%0AP_%7Bb%7D%20%3D%20120%20%5C%5C%0AO_%7BA%7D%20%3D%20100%20%5C%5C%0AO_%7BB%7D%20%3D%2050%20%0A%5Cend%7Barray%7D "\begin{array}{l}
P_{a} = 80 \\
P_{b} = 120 \\
O_{A} = 100 \\
O_{B} = 50 
\end{array}")

Further, the travel times in minutes between these locations are:

![\begin{array}{l}
c\_{aA} = 15 \\
c\_{Ab} = 20 \\
c\_{bB} = 15 
\end{array}](https://latex.codecogs.com/png.latex?%5Cbegin%7Barray%7D%7Bl%7D%0Ac_%7BaA%7D%20%3D%2015%20%5C%5C%0Ac_%7BAb%7D%20%3D%2020%20%5C%5C%0Ac_%7BbB%7D%20%3D%2015%20%0A%5Cend%7Barray%7D "\begin{array}{l}
c_{aA} = 15 \\
c_{Ab} = 20 \\
c_{bB} = 15 
\end{array}")

Use
![\beta=1](https://latex.codecogs.com/png.latex?%5Cbeta%3D1 "\beta=1")
and
![\delta=45](https://latex.codecogs.com/png.latex?%5Cdelta%3D45 "\delta=45")
min.

As a reminder, accessibility is:

![S_i = \sum_j O_jf(c\_{ij})](https://latex.codecogs.com/png.latex?S_i%20%3D%20%5Csum_j%20O_jf%28c_%7Bij%7D%29 "S_i = \sum_j O_jf(c_{ij})")

### Questions: Part 1

1.  What is the *global* jobs/population ratio in this system?
2.  What does
    ![\delta=45](https://latex.codecogs.com/png.latex?%5Cdelta%3D45 "\delta=45")
    min represent?
3.  Calculate the impedance for each population center-employment center
    pair.
4.  Calculate the accessibility.
5.  Tabulate
    ![F^c\_{ij}](https://latex.codecogs.com/png.latex?F%5Ec_%7Bij%7D "F^c_{ij}")
    for each population center-employment center pair. What do these
    values represent?
6.  Tabulate
    ![F^P\_{i}](https://latex.codecogs.com/png.latex?F%5EP_%7Bi%7D "F^P_{i}")
    for each population center. What do these values represent?
7.  Tabulate
    ![F^T\_{ij}](https://latex.codecogs.com/png.latex?F%5ET_%7Bij%7D "F^T_{ij}")
    for each population center-employment center pair. What do these
    values represent?
8.  Calculate the jobs available from each employment center to each
    population center (i.e.,
    ![V_ij](https://latex.codecogs.com/png.latex?V_ij "V_ij")).
9.  Calculate the jobs available at each population center (i.e.,
    ![V_i](https://latex.codecogs.com/png.latex?V_i "V_i")).
10. Verify that
    ![\sum_iV_i = \sum_jO_j](https://latex.codecogs.com/png.latex?%5Csum_iV_i%20%3D%20%5Csum_jO_j "\sum_iV_i = \sum_jO_j")
11. Calculate the jobs per capita at each population center (i.e.,
    ![v_i = \frac{V_i}{P_i}](https://latex.codecogs.com/png.latex?v_i%20%3D%20%5Cfrac%7BV_i%7D%7BP_i%7D "v_i = \frac{V_i}{P_i}"))
12. Compare to the global jobs/population ratio.
13. In which of the two population centers is the risk of unemployment
    higher? Discuss.
14. What do you think would happen if
    ![\beta](https://latex.codecogs.com/png.latex?%5Cbeta "\beta") was
    less than one or greater than one? Explain.

### Solution: Part 1

The spatial availability is as follows:

![\begin{array}{l}
V\_{a} = 47.1 \text{ jobs}\\ 
V\_{b} = 102.9 \text{ jobs}
\end{array}](https://latex.codecogs.com/png.latex?%5Cbegin%7Barray%7D%7Bl%7D%0AV_%7Ba%7D%20%3D%2047.1%20%5Ctext%7B%20jobs%7D%5C%5C%20%0AV_%7Bb%7D%20%3D%20102.9%20%5Ctext%7B%20jobs%7D%0A%5Cend%7Barray%7D "\begin{array}{l}
V_{a} = 47.1 \text{ jobs}\\ 
V_{b} = 102.9 \text{ jobs}
\end{array}")

Next, suppose that there is an upgrade in the transportation system that
shortens the duration of trips between
![a-A](https://latex.codecogs.com/png.latex?a-A "a-A"), so the new
travel times are:

![\begin{array}{l}
c\_{aA} = 5 \\
c\_{Ab} = 20 \\
c\_{bB} = 15 
\end{array}](https://latex.codecogs.com/png.latex?%5Cbegin%7Barray%7D%7Bl%7D%0Ac_%7BaA%7D%20%3D%205%20%5C%5C%0Ac_%7BAb%7D%20%3D%2020%20%5C%5C%0Ac_%7BbB%7D%20%3D%2015%20%0A%5Cend%7Barray%7D "\begin{array}{l}
c_{aA} = 5 \\
c_{Ab} = 20 \\
c_{bB} = 15 
\end{array}")

Continue to use
![\beta =1](https://latex.codecogs.com/png.latex?%5Cbeta%20%3D1 "\beta =1")
and
![\delta=45](https://latex.codecogs.com/png.latex?%5Cdelta%3D45 "\delta=45")
min.

### Questions: Part 2

1.  Does the accessibility of
    ![b](https://latex.codecogs.com/png.latex?b "b") change? Discuss.
2.  Explain the changes to spatial availability
    ![V_i](https://latex.codecogs.com/png.latex?V_i "V_i") and spatial
    availability per capita
    ![v_i](https://latex.codecogs.com/png.latex?v_i "v_i") compared to
    those calculated using the earlier travel times.

### Solution: Part 2

The spatial availability is now as follows:

![\begin{array}{l}
V\_{a} = 82.7 \text{ jobs}\\ 
V\_{b} = 67.3 \text{ jobs}
\end{array}](https://latex.codecogs.com/png.latex?%5Cbegin%7Barray%7D%7Bl%7D%0AV_%7Ba%7D%20%3D%2082.7%20%5Ctext%7B%20jobs%7D%5C%5C%20%0AV_%7Bb%7D%20%3D%2067.3%20%5Ctext%7B%20jobs%7D%0A%5Cend%7Barray%7D "\begin{array}{l}
V_{a} = 82.7 \text{ jobs}\\ 
V_{b} = 67.3 \text{ jobs}
\end{array}")

## Practical example

For this practical example we will use data from the 2016 edition of the
[Transportation Tomorrow
Survey](http://dmg.utoronto.ca/tts-introduction/) (TTS). This is a
comprehensive travel survey conducted in the Greater Toronto and
Hamilton Area every five years, and it includes rich information about
travel patterns and the socio-demographic composition of the population
in the region. For convenience, parts of the survey have been sourced in
[{TTS2016R}](https://soukhova.github.io/TTS2016R/), an [open data
product](https://rdcu.be/dn6yP). The data are augmented with other
information, such as travel time tables, and distributed as
analysis-ready data objects.

We will begin by loading the packages used in this example.

    library(accessibility) # Transport Accessibility Measures # Transport Accessibility Measures
    library(dplyr) # A Grammar of Data Manipulation # A Grammar of Data Manipulation # A Grammar of Data Manipulation
    library(ggplot2) # Create Elegant Data Visualisations Using the Grammar of Graphics # Create Elegant Data Visualisations Using the Grammar of Graphics # Create Elegant Data Visualisations Using the Grammar of Graphics
    library(leaflet) # Create Interactive Web Maps with the JavaScript 'Leaflet' Library # Create Interactive Web Maps with the JavaScript 'Leaflet' Library
    library(patchwork) # The Composer of Plots # The Composer of Plots # The Composer of Plots
    library(sf) # Simple Features for R # Simple Features for R
    library(tidyr) # Tidy Messy Data # Tidy Messy Data
    library(TTS2016R) # An augmented 2016 Transportation Tomorrow Survey (TTS) data package: worker and place of employment counts, trips and estimated travel time to work in the Greater Golden Horsehoe area, Canada # An augmented 2016 Transportation Tomorrow Survey (TTS) data package: worker and place of employment counts, trips and estimated travel time to work in the Greater Golden Horsehoe area, Canada

We will use two objects from the data package, information aggregated at
the level of traffic analysis zones (TAZs) and an origin-destination
table:

``` r
data("ggh_taz") # Traffic analysis zones.
data("od") # Origin-destination table.
```

The data sets are documented, and you can check the help files like so:

    ?ggh_ttz
    ?od

For the exercise we will use a slice of the data, so we will extract the
parts of the data corresponding to the City of Toronto. According to the
TTS [Data
Guide](http://dmg.utoronto.ca/wp-content/uploads/2022/06/2016TTS_DataGuide.pdf)
of TTS (p. 29), the identifiers of the TAZs in Toronto are 1-625. We
will create a vector with those identifiers:

``` r
TO_taz <- c(1:625) |> 
  as.character()
```

Using the vector we just created we will filter the zones corresponding
to the City of Toronto, and then select the three variables that will be
used in the example, namely the zone identifier (using the GTA06 zoning
system), the number of workers in the zone (work age population), and
the number of jobs in the zone:

``` r
# Filter the traffic analysis zones in the City of Toronto.
lu <- ggh_taz |>
  filter(GTA06 %in% TO_taz) |>
  # Rename and select only the three variables needed for the example.
  transmute(id = GTA06,
            P = workers,
            O = jobs)
```

This selection somewhat simplifies a characteristic of the data set,
because not all the workers living in Toronto work there, and not all
jobs in Toronto are taken by people who live in Toronto. Given this
caveat, we can calculate the jobs/workers ratio in the region:

``` r
sum(lu$O)/sum(lu$P)
#> [1] 1.11989
```

A summary of the table shows that workers in Toronto tend to be more
dispersed than jobs. Some TAZs have zero workers (there is no housing in
the zone) and some have zero jobs (they are purely residential):

``` r
lu |> 
  # Drop the geometry before reporting the summary.
  st_drop_geometry() |>
  summary()
#>       id                  P              O        
#>  Length:625         Min.   :   0   Min.   :    0  
#>  Class :character   1st Qu.: 697   1st Qu.:  305  
#>  Mode  :character   Median :1471   Median :  643  
#>                     Mean   :1719   Mean   : 1925  
#>                     3rd Qu.:2442   3rd Qu.: 1704  
#>                     Max.   :8491   Max.   :41821
```

The overall jobs/workers ratio is probably a little bit optimistic
because Toronto tends to attract many commuting trips from beyond the
city boundaries. Since table `lu` is a simple features object with
geometry, we can map the variables in the form of choroplet maps. The
next two chunks of code are for workers and jobs:

``` r
Pi_plot <- ggplot() +
  geom_sf(data = lu,
          aes(fill = P)) +
  geom_sf(data = lu,
          fill = NA) +
  scale_fill_viridis_c(direction = 1) +
  theme_void() +
  ggtitle("Workers per TAZ")
```

``` r
Oi_plot <- ggplot() +
  geom_sf(data = lu,
          aes(fill = O)) +
  geom_sf(data = lu,
          fill = NA) +
  scale_fill_viridis_c(direction = 1) +
  theme_void() +
  ggtitle("Jobs per TAZ")
```

The two maps can be plotted side-by-side using the syntax of package
{patchwork}:

``` r
Pi_plot + Oi_plot
```

![](README_files/figure-gfm/unnamed-chunk-7-1.png)<!-- -->

The maps show that jobs are highly concentrated in downtown Toronto. The
number of inhabitants downtown is not as high, and there are places in
the periphery of the core of the city and towards the edges with
relatively high populations.

Next, we will prepare the travel time matrix by filtering TAZ that are
in Toronto.

``` r
# Rename the table.
od_ggh <- od

# Filter the origin and destinations in the City of Toronto.
od <- od |>
  filter(Origin %in% lu$id, 
         Destination %in% lu$id) |>
  # Rename and select only the three variables needed for the example.
  transmute(from_id = Origin,
            to_id = Destination,
            travel_time)
```

Travel times in the table are in minutes. The longest travel time
between any origin and any destination is:

``` r
max(od$travel_time)
#> [1] 146
```

We will use an inverse power impedance function of the following form
for the example:

![f(c\_{ij}) = c\_{ij}^{\beta}](https://latex.codecogs.com/png.latex?f%28c_%7Bij%7D%29%20%3D%20c_%7Bij%7D%5E%7B%5Cbeta%7D "f(c_{ij}) = c_{ij}^{\beta}")

We present the example with
![\beta = 1](https://latex.codecogs.com/png.latex?%5Cbeta%20%3D%201 "\beta = 1")
(try this value first; you can experiment with other values later if you
wish). This is the shape of the curve with the initial value of
![\beta](https://latex.codecogs.com/png.latex?%5Cbeta "\beta"):

``` r
# Parameter for inverse power impedance function.
beta <- 1

# Plot impedance function with {ggplot}
ggplot() +
  geom_function(fun = function(x) 1/(abs(x)^beta),
                xlim = c(1, max(od$travel_time))) +
  xlab("t (travel time in min)") +
  ylab("f(t)") +
  theme_minimal()
```

![](README_files/figure-gfm/unnamed-chunk-10-1.png)<!-- -->

Package [{accessibility}](https://ipeagit.github.io/accessibility/)
includes functions to calculate both accessibility metrics and spatial
availability. The documentation can be consulted like so:

    ?accessibility

The inputs are a travel matrix (which in this example is our `od` table)
and a land use table (which in our example is the `lu` table containing
the opportunities and demand).

We calculate the accessibility first. The function to do this is
`accessibiliy::gravity()`. We need to specify the name of the columns
with the opportunities, the cost, as well as the impedance function
(here `accessibility::decay_power()`):

``` r
# Use table `od` with the travel time information for origin-destination pairs.
Si <- gravity(travel_matrix = od, 
              # Use table `lu` with the land use information.
              land_use_data = lu, 
              # Use column `O` for the opportunities; this column contains the number of jobs per zone.
              opportunity = "O", 
              # Use column `travel_time` as the cost variable.
              travel_cost = "travel_time", 
              # Use the power impedance function with parameter beta.
              decay_function = decay_power(beta)) |>
  # Rename the opportunities $S_i$ for accessibility.
  rename(Si = O)
```

Next, we calculate the spatial availability. The inputs are very
similar, with the addition of a variable for the demand (in our case the
population). You will recall that the population is needed to calculate
the proportional allocation factor. The spatial availability can be
reported in a detailed table that gives
![V\_{ij}](https://latex.codecogs.com/png.latex?V_%7Bij%7D "V_{ij}"),
that is, the number of jobs available to each origin from each
destination. The alternative (`detailed_results = FALSE`) returns the
spatial availability aggregated by zone.

``` r
# Use table `od` with the travel time information for origin-destination pairs.
Vij <- spatial_availability(travel_matrix = od,
                            # Use table `lu` with the land use information.
                            land_use_data = lu, 
                            # Use column `O` for the opportunities; this column contains the number of jobs per zone.
                            opportunity = "O", 
                            # Use column `travel_time` as the cost variable.
                            travel_cost = "travel_time", 
                            # Use column `P` for the demand; this column contains the number of workers per zone.
                            demand = "P", 
                            # Use the power impedance function with parameter beta.
                            decay_function = decay_power(beta),
                            # Return the detailed results.
                            detailed_results = TRUE) |>
  # Rename the opportunities $V_i$ for spatial availability.
  rename(Vij = O)
```

If the detailed results are requested, the spatial availability by
origin can be computed as follows:

``` r
Vi <- Vij |>
  # Group by origin; this means that the operations in summarize() will be over the destinations.
  group_by(from_id) |>
  # Sum the opportunities available from every destination
  summarize(Vi = sum(Vij)) |>
  # Rename the column to `id`
  rename(id = from_id)
```

<!--
Compare results to our home-brewed function `spav()` above. The differences are rounding errors.
&#10;```r
junkij <- spav() |> select(-starts_with("geometry"))
junki <-  junkij |>
st_drop_geometry() |>
group_by(from_id) |>
summarize(Vi = sum(Vij)) |>
rename(id = from_id)
&#10;junk_compare <- junki |> left_join(Vi, by = "id")  
&#10;junk_compare |> mutate(diff = abs(Vi.x - Vi.y)/Vi.y) |> summarize(mean_diff = mean(diff))  
#> # A tibble: 1 × 1
#>   mean_diff
#>       <dbl>
#> 1   0.00186
```
-->

Join the results to the zoning system:

``` r
results <- lu |>
  left_join(Si,
            by = "id") |>
  left_join(Vi,
            by = "id")
```

Verify the total of jobs in the region, the jobs accessible, and the
jobs available:

``` r
# Sum of jobs in land use table,
sum(lu$O)
#> [1] 1202986
# Sum of jobs accessible.
sum(results$Si, na.rm = TRUE)
#> [1] 15421360
# Sum of jobs available.
sum(results$Vi, na.rm = TRUE)
#> [1] 1202843
```

We see that the jobs in the City of Toronto is preserved (with a small
difference caused by zones that cannot be reached). The sum of
accessibility in contrast is two orders of magnitude larger than the
number of jobs in the table. This complicates the interpretation of the
results. When we obtain a summary of these results we find that the
maximum accessibility for any one zone is several hundred thousands of
jobs, compared to less than sixteen thousand jobs according to spatial
availability (notice the 75 NA values; these are zones with a population
of zero). Recall that the maximum number of jobs in any one zone (check
the summary of the `lu` table above) is 41,821. Why are the values of
the accessibility so high?

``` r
results |> 
  st_drop_geometry() |>
  select(Si, Vi) |>
  summary()
#>        Si                  Vi           
#>  Min.   :    42.88   Min.   :    0.037  
#>  1st Qu.: 12475.44   1st Qu.:  579.082  
#>  Median : 21153.41   Median : 1370.369  
#>  Mean   : 28038.84   Mean   : 2186.987  
#>  3rd Qu.: 34883.20   3rd Qu.: 2575.955  
#>  Max.   :134653.00   Max.   :21152.968  
#>  NA's   :75          NA's   :75
```

Accessibility, in addition, has much greater range of variation, which
indicates that the high sum of accessibility is not the result of a few
extraordinary values. The [interquartile
range](https://en.wikipedia.org/wiki/Interquartile_range) (a measure of
dispersion) of ![S_i](https://latex.codecogs.com/png.latex?S_i "S_i") is
22,407.76 but only 1,996.87 for
![V_i](https://latex.codecogs.com/png.latex?V_i "V_i"). The latter value
is more in line with the interquartile ranges of the population
(1,996.87) and jobs (1,399).

Maps of accessibility and spatial availability can be plotted using
package {ggplot2}. This code is for the map of accessibility:

``` r
Si_plot <- ggplot() +
  geom_sf(data = results,
          aes(fill = Si)) +
  geom_sf(data = lu,
          fill = NA) +
  scale_fill_viridis_c(direction = 1) +
  theme_void() +
  ggtitle("Accessibility")
```

This code is for the map of spatial availability:

``` r
Vi_plot <- ggplot() +
  geom_sf(data = results,
          aes(fill = Vi)) +
  geom_sf(data = lu,
          fill = NA) +
  scale_fill_viridis_c(direction = 1) +
  theme_void() +
  ggtitle("Spatial Availability")
```

The two maps can be plotted side-by-side using the syntax of package
{patchwork}:

``` r
Si_plot / Vi_plot
```

![](README_files/figure-gfm/applied-example-display-results-1.png)<!-- -->

As expected, the map of spatial availability in noticeably flatter with
its smaller interquartile range. High levels of accessibility do not
necessarily translate into high spatial availability in the same places.
Why is that? High accessibility creates a somewhat paradoxical effect:
the ease to reach destinations also means that there is more competition
for the same opportunities.

But, could it be that accessibility is just some sort of scaled-up
version of spatial availability? That somehow the two give similar
results but for some multiplicative constant? To answer this, we can
plot the ratio of accessibility to spatial availability. If the two
measures were similar but for the scale, we would expect that ratio to
be more or less constant. However, what we see is that the ratio of
![S_i](https://latex.codecogs.com/png.latex?S_i "S_i") to
![V_i](https://latex.codecogs.com/png.latex?V_i "V_i") indicates that in
some cases accessibility is tens, hundreds, or even thousands of times
greater than spatial availability. This suggests that
![S_i](https://latex.codecogs.com/png.latex?S_i "S_i") is not simply a
scaled-up version of
![V_i](https://latex.codecogs.com/png.latex?V_i "V_i") but something
different.

``` r
ggplot() +
  geom_sf(data = results,
          aes(fill = Si/Vi)) +
  geom_sf(data = lu,
          fill = NA) +
  scale_fill_viridis_c(direction = 1, 
                       trans = "log10") +
  theme_void()
```

![](README_files/figure-gfm/applied-example-ratio-accessibility-to-availability-1.png)<!-- -->

We can further explore the results if we calculate the number of
accessible jobs per capita
(![s_i](https://latex.codecogs.com/png.latex?s_i "s_i")) and spatially
available jobs per capita
(![v_i](https://latex.codecogs.com/png.latex?v_i "v_i"):

``` r
results <- results |>
  mutate(si = Si/P,
         vi = Vi/P)
```

This is the summary of these two indicators.

``` r
results |> 
  st_drop_geometry() |>
  select(si, vi) |>
  summary()
#>        si                 vi         
#>  Min.   :  0.1992   Min.   :0.00283  
#>  1st Qu.:  8.5324   1st Qu.:0.51484  
#>  Median : 12.4421   Median :0.79628  
#>  Mean   : 19.2023   Mean   :0.85798  
#>  3rd Qu.: 18.2990   3rd Qu.:1.11207  
#>  Max.   :735.8087   Max.   :2.66881  
#>  NA's   :75         NA's   :75
```

Accessibility per capita is not particularly meaningful: as seen from
the summary above, accessibility per capita does not particularly
resemble the ratio of jobs-to-workers in the city. This is a result of
accessibility being a sum of unconstrained opportunities. The summary
reveals that 75/% of zones have more than 8.53 jobs accessible per
person, which clearly is not a realistic estimate of the ease of
reaching jobs while keeping in mind that many others will be doing the
same. Spatially available jobs per capita do show some variability but
they are more in line with the global jobs-to-workers ratio. It is not
implausible to think that some zones, being very well served by the
transportation system and/or facing little competition, can have around
two jobs available per worker.

Returning to the map of spatial availability, notice its resemblance to
the distribution of the population. The correlation between these two
variables is quite high:

``` r
cor(results$P, results$Vi, use = "pairwise.complete.obs")
#> [1] 0.933404
```

This should not be surprising since the spatial availability depends
directly on the population: part of the proportional allocation
mechanism works to ensure that opportunities are proportionally
available by population. That said, the global jobs-to-workers ratio
provides a useful reference; in the next figure, we plot the population
vs the spatial availability. The black line passes through the origin
and has a slope of 1.1198902, that is, equal to the global
jobs-to-workers ratio. If a zone had as many jobs available to its
population as the global ratio the point would be on the line; points
below the line are zones with *fewer* jobs available than what they
would have, given their population, if the jobs were equally
distributed. In contrast, points above the line are zones with
populations with *more* jobs available to them than their equal share.

``` r
ggplot(data = results |>
         drop_na(Vi), 
       # Plot P in the x axis and Vi in the y axis.
       aes(x = P, y = Vi)) + 
  # Plot the data as points, use alpha < 1 to control the transparency of the points.
  geom_point(aes(color = Vi/P - sum(lu$O)/sum(lu$P),
                 size = Vi/P)) + 
  geom_point(aes(size = Vi/P),
             shape = 1,
             alpha = 0.2) +
  # Plot a line with a given intercept and slope.
  geom_abline(intercept = 0, 
              slope = sum(lu$O)/sum(lu$P),
              #color = "orange"
  ) +
  scale_color_gradient2("Dev. from global jobs-to-workers ratio") +
  scale_size("Spatial availability per capita (v_i)") +
  theme_minimal()
```

![](README_files/figure-gfm/applied-example-scatterplot-population-to-availability-1.png)<!-- -->

Many zones with small populations have lower spatial availability to
employment. Not always, but frequently. Zones with large populations
tend to enjoy higher employment availability. Not a single zone with
population greater than 5,000 has fewer jobs available than workers. The
map below shows zones with populations greater than 5,000.

``` r
ggplot() +
  geom_sf(data = lu,
          aes(fill = P>5000)) +
  theme_minimal()
```

![](README_files/figure-gfm/applied-example-large-population-map-1.png)<!-- -->

What could be the reason for this? Zones with large populations are not
necessarily centrally located, and some are quite peripheral. But how
well connected are they? To answer this question we need calculate the
travel time to the opportunities available. To do this we join the
travel times to the detailed spatial availability results:

``` r
results_2 <- Vij |>
  # Join o-d travel times to detailed spatial availability results.  
  left_join(od |>
              select(from_id, to_id, travel_time),
            by = c("from_id", "to_id"))
```

Next, we calculate the total travel time to available opportunities for
every origin-destination pair:

``` r
results_2 <- results_2 |>
  # Calculate the total travel time to opportunities available.
  mutate(total_time_opp_av = Vij * travel_time)
```

We can now calculate the average time to available opportunity by
origin. This quantity represents the time typically needed to reach an
available opportunity:

``` r
results_2 <- results_2 |>
  # Group by origin.
  group_by(from_id) |>
  # Compute the spatial availability by origin. The mean time is the total travel time by origin to spatially available opportunities, divided by the opportunities available to that origin.
  summarize(Vi = sum(Vij),
            mean_time = sum(total_time_opp_av)/Vi) |>
  # Rename the id.
  rename(id = from_id)
```

The scatterplot below is of zonal population vs mean travel time to
available opportunity by origin. The horizontal line is the system-wide
mean travel time to available opportunity:

``` r
results_2 |>
  # Join the population values to table `results_2`
  left_join(results |>
              select(id, P),
            by = "id") |>
  mutate(vi = Vi/P) |>
  # Pass to ggplot, where the population will be plotted on the x axis and the mean travel time to available opportunity in the y axis.
  ggplot(aes(x = P,
             y = mean_time)) +
  # The color of the points will depend on the deviation from the global mean of the travel time to available opportunity.
  geom_point(aes(color = mean_time - mean(results_2$mean_time),
                 size = vi)) +
  # For ease of visualization add silhouettes for the points
  geom_point(aes(size = vi),
             shape = 1,
             alpha = 0.3) +
  # Plot a horizontal line to reres
  geom_hline(yintercept = mean(results_2$mean_time)) +
  # Change the title of the legends to something informative
  scale_color_gradient2("Dev. from global mean travel time") +
  scale_size("Spatial availability per capita (v_i)") +
  theme_minimal()
```

![](README_files/figure-gfm/unnamed-chunk-12-1.png)<!-- -->

The plot shows that zones with larger populations tend to be
consistently are not necessarily the best connected, but nonetheless do
not deviate much from the mean travel time to available opportunities:
notice how zones with populations greater than 5,000 are much closer to
the horizontal line (system-wise mean travel time to available
opportunity). Of the smaller zones, quite a few tend to be have, on
average, short travel times to available opportunties (which is
reflected in the large values of
![v_i](https://latex.codecogs.com/png.latex?v_i "v_i")), but many are
poorly connected to available opportunities, requiring on average longer
travel times, which is associated with lower values of
![v_i](https://latex.codecogs.com/png.latex?v_i "v_i").

<!--
&#10;```r
si_plot <- ggplot() +
geom_sf(data = results,
aes(fill = si)) +
geom_sf(data = lu,
fill = NA) +
scale_fill_viridis_c(direction = 1) +
theme_void() +
ggtitle("Accessibility per capita")
```
&#10;
```r
vi_plot <- ggplot() +
geom_sf(data = results,
aes(fill = vi)) +
geom_sf(data = lu,
fill = NA) +
scale_fill_viridis_c(direction = 1) +
theme_void() +
ggtitle("Spatial Availability per capita")
```
&#10;Render:
&#10;```r
si_plot + vi_plot
```
&#10;![](README_files/figure-gfm/applied-example-display-per-capita-results-1.png)<!-- -->

The same is true of
![s_i](https://latex.codecogs.com/png.latex?s_i "s_i"): the ratio of
![s_i](https://latex.codecogs.com/png.latex?s_i "s_i") to
![v_i](https://latex.codecogs.com/png.latex?v_i "v_i") indicates it is
not simply a scaled-up version of
![v_i](https://latex.codecogs.com/png.latex?v_i "v_i") but rather
something different:

``` r
ggplot() +
geom_sf(data = results,
aes(fill = si/vi)) +
geom_sf(data = lu,
fill = NA) +
scale_fill_viridis_c(direction = 1, trans = "log10") +
theme_void()
```

![](README_files/figure-gfm/applied-example-ratio-accessibility-to-availability-per-capita-1.png)<!-- -->
–\>

The values of spatial availability per capita can be compared to the
overall jobs/workers ratio. In this plot we use a diverging gradient
with the mid-point set at that overall value. Shades of red indicate
that the spatial availability per capita is below the regional level,
and shades of blue indicate a value greater than the overall
jobs/workers ratio. This way we can explore deviations from equality:

``` r
ggplot() +
  geom_sf(data = results,
          aes(fill = vi)) +
  geom_sf(data = lu,
          fill = NA) +
  scale_fill_gradient2(midpoint = sum(lu$O)/sum(lu$P)) +
  theme_void() +
  ggtitle("Spatial Availability per capita")
```

![](README_files/figure-gfm/unnamed-chunk-13-1.png)<!-- -->

The code in the next two chunks uses package {leaflet} to create
interactive maps to further explore the results. Some zones had high
accessibility, and yet relatively low availability. This could be caused
by the competition for opportunities in well-connected central parts of
Toronto.

Map to explore accessibility:

``` r

bins <- quantile(results$si, na.rm = TRUE) |> as.numeric()
pal <- colorBin("RdBu", domain = results$si, bins = bins)

labels <- sprintf(
  "<strong>%s</strong><br/>%g workers<br/>%g jobs accessible<br/>%g jobs / person",
  results$id, 
  results$P,
  results$Si,
  results$si
) %>% lapply(htmltools::HTML)

leaflet(data = results |> st_transform(crs = 4326 )) |> 
  addPolygons(
    fillColor = ~pal(si),
    weight = 2,
    opacity = 1,
    color = "white",
    dashArray = "3",
    fillOpacity = 0.7,
    highlightOptions = highlightOptions(
      weight = 5,
      color = "#666",
      dashArray = "",
      fillOpacity = 0.7,
      bringToFront = TRUE),
    label = labels,
    labelOptions = labelOptions(
      style = list("font-weight" = "normal", padding = "3px 8px"),
      textsize = "15px",
      direction = "auto"))
```

<div class="leaflet html-widget html-fill-item" id="htmlwidget-de898e7f4341ccb18f82" style="width:672px;height:480px;"></div>
<script type="application/json" data-for="htmlwidget-de898e7f4341ccb18f82">{"x":{"options":{"crs":{"crsClass":"L.CRS.EPSG3857","code":null,"proj4def":null,"projectedBounds":null,"options":{}}},"calls":[{"method":"addPolygons","args":[[[[{"lng":[-79.32001254169437,-79.3199653524059,-79.31984435444851,-79.31973710824104,-79.31938109528923,-79.31906498257649,-79.31862663408982,-79.31857401140016,-79.31847082507515,-79.31837676906608,-79.31827452563785,-79.3181614454177,-79.31805252191,-79.31791013902796,-79.31776685926327,-79.31763747436437,-79.31620233935811,-79.31562078001946,-79.31552098547022,-79.31537723867929,-79.31527374766245,-79.31517588696516,-79.31510324371638,-79.3150319471619,-79.31497692909228,-79.31493980165554,-79.31489922944101,-79.31487440609828,-79.3148379147252,-79.31354375320987,-79.31258112160381,-79.31032313725834,-79.309182454487,-79.30803514361105,-79.30690527909496,-79.30576221662591,-79.30549403348779,-79.30472654860321,-79.30488430470433,-79.30524019047131,-79.30544763753963,-79.3056745921652,-79.30595648405709,-79.30629764030942,-79.30677872935991,-79.30723174250446,-79.30761974715807,-79.31128306812478,-79.31485497236476,-79.31723658943426,-79.31890009774799,-79.31903001515812,-79.31991220990322,-79.32001254169437],"lat":[43.66368584787615,43.66369469629969,43.66372442452242,43.66376344685224,43.66396525834684,43.66423976181083,43.66458448219322,43.66463772008238,43.66472172103148,43.66481935926915,43.66489887290203,43.66497822695921,43.66504422907052,43.66509614554177,43.66514354762231,43.66517315003561,43.66548517852723,43.6656251416663,43.66564617578874,43.66568456595986,43.66572805042189,43.66578962183452,43.66583815195442,43.66592261993662,43.66600282688693,43.66608329760942,43.66617722055083,43.66628946976017,43.66673462154655,43.6670215991114,43.66723703063231,43.66772587439186,43.66798365455931,43.66823673414918,43.66849465169251,43.66875236236771,43.66809105915248,43.6662158994276,43.66565102252117,43.66512509183869,43.66491662067357,43.66473085337096,43.66451898396092,43.66429439915091,43.66411248581235,43.66396148159412,43.66386370503121,43.66304452071481,43.66226888078766,43.66172775503645,43.66134597643577,43.6616267702363,43.66363099090071,43.66368584787615]}]],[[{"lng":[-79.34745199658671,-79.34368150642779,-79.34936678447178,-79.34948024898068,-79.34950048043166,-79.34960877293558,-79.3496659118333,-79.34975758687497,-79.35060690114361,-79.35246899171084,-79.35372185945599,-79.35390211246794,-79.35420743099725,-79.35433241561689,-79.34764063035739,-79.34745199658671],"lat":[43.6505085145752,43.64500165861386,43.641770886028,43.64170050826433,43.6417233055998,43.64184180627262,43.64190564582589,43.64201499434957,43.64308526713675,43.64530457739347,43.64687584655626,43.64708549022371,43.64748156684223,43.64764099261651,43.65041212952899,43.6505085145752]}]],[[{"lng":[-79.42581148444972,-79.42678298037218,-79.42811209084883,-79.42981645609377,-79.43090031309173,-79.43099802680635,-79.43144146077299,-79.43247644877205,-79.43367724147058,-79.43559074199699,-79.43583990274092,-79.4360818720487,-79.4361112079259,-79.43634530973327,-79.43457204993308,-79.43254976383592,-79.43166866397301,-79.43106826572438,-79.43080130116411,-79.43070793254228,-79.43057653931808,-79.43047103682555,-79.43036975520104,-79.43029404273932,-79.4301430718067,-79.43002178592764,-79.42986424032503,-79.42966454827507,-79.42943858391584,-79.42817382103583,-79.42639144515587,-79.42605320483788,-79.42587774832852,-79.42586903448345,-79.4256521074706,-79.42554706038568,-79.42505698820023,-79.42473868906492,-79.42581148444972],"lat":[43.6494756499815,43.64949352829628,43.64955241578647,43.64957584472393,43.64961773686164,43.64961907837856,43.64964316939151,43.64968887688497,43.64976376368639,43.64988449423022,43.65049572415033,43.65112026814288,43.65118368403629,43.65180820977093,43.65213951605266,43.65256208131451,43.652734535825,43.65284332189987,43.65289366905021,43.652905890115,43.65291758894122,43.65292064112259,43.65291474911476,43.65290470717787,43.65287112629663,43.65285595713458,43.65286279484599,43.65290956251864,43.65295605882181,43.65322232595943,43.65359397895993,43.65272044231058,43.65225432959382,43.65223170441813,43.65169750464651,43.65144841077287,43.65023511166856,43.64942936099894,43.6494756499815]}]],[[{"lng":[-79.43670948411248,-79.43698105562063,-79.43728832647878,-79.43760423374796,-79.43762259106066,-79.43856902282241,-79.43892901360998,-79.43894139900958,-79.43916325717504,-79.43865868277398,-79.43759590211708,-79.43642983213302,-79.43535399240638,-79.43410159664087,-79.43382379514229,-79.43290515195339,-79.43242279641871,-79.43166038323965,-79.4311198397818,-79.43040900576108,-79.42942593470875,-79.42854897129993,-79.4274249971967,-79.42639144515587,-79.42817382103583,-79.42943858391584,-79.42966454827507,-79.42986424032503,-79.43002178592764,-79.4301430718067,-79.43029404273932,-79.43036975520104,-79.43047103682555,-79.43057653931808,-79.43070793254228,-79.43080130116411,-79.43106826572438,-79.43166866397301,-79.43254976383592,-79.43457204993308,-79.43634530973327,-79.43670948411248],"lat":[43.65273608340292,43.65347364600806,43.65427011941372,43.65508480411021,43.65513006537147,43.65757850685662,43.65849281239561,43.65851998773869,43.65909032746455,43.65920496294169,43.65941999092158,43.65964268798502,43.65987561052905,43.66013752110485,43.66017872468138,43.6603867720894,43.66048377054865,43.6606534414971,43.66078555409514,43.66093783060777,43.66114496510408,43.65895846448468,43.65616524002327,43.65359397895993,43.65322232595943,43.65295605882181,43.65290956251864,43.65286279484599,43.65285595713458,43.65287112629663,43.65290470717787,43.65291474911476,43.65292064112259,43.65291758894122,43.652905890115,43.65289366905021,43.65284332189987,43.652734535825,43.65256208131451,43.65213951605266,43.65180820977093,43.65273608340292]}]],[[{"lng":[-79.4274249971967,-79.42854897129993,-79.42942593470875,-79.42819314312591,-79.42804240274685,-79.4268464932985,-79.42564784206459,-79.42477796266402,-79.42430300360022,-79.42374469271266,-79.4236904219246,-79.42269221412843,-79.42394746215649,-79.42517398319626,-79.42639144515587,-79.4274249971967],"lat":[43.65616524002327,43.65895846448468,43.66114496510408,43.6614295853049,43.66147261227406,43.66173071906081,43.66197986350402,43.65976192481158,43.65853983029216,43.65709603366854,43.65697375740058,43.65439385881424,43.65411401818952,43.65386086409492,43.65359397895993,43.65616524002327]}]],[[{"lng":[-79.42804240274685,-79.42819314312591,-79.42942593470875,-79.4294816519641,-79.42956255683515,-79.42960743907176,-79.42962982111581,-79.42965195024621,-79.4296586726218,-79.42965894613202,-79.43050501570048,-79.43170713260841,-79.43274936792619,-79.43299809831832,-79.43293496791368,-79.42938567233485,-79.4290084672577,-79.42870969780662,-79.42842143200193,-79.42814401891575,-79.42784927219331,-79.42755984442169,-79.42728526674597,-79.42652954494611,-79.42608916521927,-79.42564784206459,-79.4268464932985,-79.42804240274685],"lat":[43.66147261227406,43.6614295853049,43.66114496510408,43.66126725846312,43.66143940918154,43.66153013643301,43.6615934583281,43.66170178698822,43.66180090189631,43.66223759502966,43.66430204418361,43.66717282476959,43.66969473059135,43.67026535972635,43.6702779971491,43.67096762069997,43.67002604429051,43.66929709284092,43.6686087941498,43.66794774007119,43.66721434046176,43.66651252034372,43.66586500605702,43.66407643004396,43.66300785458517,43.66197986350402,43.66173071906081,43.66147261227406]}]],[[{"lng":[-79.43666701026184,-79.4370477260271,-79.43742703516801,-79.43779585317513,-79.43791311264486,-79.43817413176581,-79.43902452131663,-79.43926763695283,-79.4392787097228,-79.4366233863104,-79.43660986082352,-79.43627547591433,-79.43594203734493,-79.43519037155419,-79.43502664657177,-79.43486211812544,-79.43469675682032,-79.4345322280047,-79.43436686630463,-79.43420172364493,-79.43403635920936,-79.43387051523547,-79.43370455407097,-79.43353850528344,-79.43336947639089,-79.43299809831832,-79.43274936792619,-79.43170713260841,-79.43050501570048,-79.42965894613202,-79.4296586726218,-79.42965195024621,-79.42962982111581,-79.42960743907176,-79.42956255683515,-79.4294816519641,-79.42942593470875,-79.43040900576108,-79.4311198397818,-79.43166038323965,-79.43242279641871,-79.43290515195339,-79.43382379514229,-79.43410159664087,-79.43535399240638,-79.43666701026184],"lat":[43.66295940759039,43.66385149664577,43.66470755690124,43.66555447034023,43.66584422860851,43.66641951586393,43.66845695390747,43.66897798166701,43.66900063798407,43.66951355902423,43.66952237603657,43.6695943185214,43.6696573609585,43.66982261111535,43.66986087839093,43.66989463340864,43.66993287780056,43.66996663233711,43.67000487624502,43.67003862189518,43.67007695530649,43.67010618984835,43.67013992356168,43.6701691548137,43.6702028459905,43.67026535972635,43.66969473059135,43.66717282476959,43.66430204418361,43.66223759502966,43.66180090189631,43.66170178698822,43.6615934583281,43.66153013643301,43.66143940918154,43.66126725846312,43.66114496510408,43.66093783060777,43.66078555409514,43.6606534414971,43.66048377054865,43.6603867720894,43.66017872468138,43.66013752110485,43.65987561052905,43.66295940759039]}]],[[{"lng":[-79.44680402223901,-79.44716627593425,-79.44786971378468,-79.44860393712193,-79.44900398986871,-79.44889137838926,-79.4487811344042,-79.44867149018101,-79.44856113022404,-79.44845148591908,-79.44834277543946,-79.44823289918244,-79.44812407268221,-79.44801442820589,-79.44790548566034,-79.44779665903002,-79.44768843207271,-79.44757937337602,-79.44747135078116,-79.44736383934011,-79.44725549613507,-79.44714747336246,-79.44704056140988,-79.44693325421153,-79.44682551038173,-79.44679134767574,-79.44671962063487,-79.44561821179634,-79.44443351661712,-79.44326109411325,-79.4392787097228,-79.43926763695283,-79.43902452131663,-79.43817413176581,-79.43791311264486,-79.43779585317513,-79.43742703516801,-79.4370477260271,-79.43666701026184,-79.43535399240638,-79.43642983213302,-79.43759590211708,-79.43865868277398,-79.43916325717504,-79.43973169547871,-79.44037700200448,-79.44073556437571,-79.44156668285623,-79.4417233052942,-79.44272197508872,-79.44398478793212,-79.44520683050061,-79.44571781841771,-79.44610277431178,-79.44680402223901],"lat":[43.66184147854295,43.66273328332098,43.66446259064568,43.66626882851254,43.66725566147511,43.66726772621613,43.667275231938,43.66728724671711,43.66729925167463,43.66731126624127,43.667318792359,43.66733980561099,43.66735183096736,43.66736384511179,43.66738036970607,43.66739239474843,43.66740892885451,43.66742995258185,43.667446489258,43.66746303277625,43.66748406591435,43.66750060228139,43.66752165466264,43.66753820055044,43.66756374244724,43.66756777917791,43.66758480841978,43.66778597252384,43.66800390644372,43.66821758423094,43.66900063798407,43.66897798166701,43.66845695390747,43.66641951586393,43.66584422860851,43.66555447034023,43.66470755690124,43.66385149664577,43.66295940759039,43.65987561052905,43.65964268798502,43.65941999092158,43.65920496294169,43.65909032746455,43.6589854736994,43.65885024895206,43.65877412308723,43.65858732423867,43.6585534516563,43.65833301060601,43.65808456022846,43.65782653915941,43.65917065525482,43.66011678482502,43.66184147854295]}]],[[{"lng":[-79.46003491659883,-79.45946050677183,-79.45854839112233,-79.45806631931644,-79.45641807460156,-79.45495487645576,-79.45308652190592,-79.45219811643877,-79.45219279986803,-79.45210212485109,-79.45200530193542,-79.45190837678459,-79.45181166955321,-79.45177516926594,-79.45171556235624,-79.45161863723224,-79.45152181435645,-79.45142570717894,-79.45132876860896,-79.45123254572277,-79.4511357228586,-79.45103938423486,-79.4509424591276,-79.45084612047354,-79.45075001329825,-79.45065295660702,-79.45055733359195,-79.45047900293088,-79.45046039494316,-79.45036477189069,-79.45026853536575,-79.45017313030426,-79.45007679150022,-79.44997820335587,-79.44977928888943,-79.4496635104728,-79.44955316442591,-79.44944302283803,-79.44933197466477,-79.44922151268742,-79.44911126878421,-79.44900398986871,-79.44860393712193,-79.44786971378468,-79.44716627593425,-79.44680402223901,-79.44610277431178,-79.44571781841771,-79.44520683050061,-79.44523884101751,-79.44776062504295,-79.44808609609559,-79.44878470713039,-79.44892248528589,-79.4498588334838,-79.45133093109389,-79.45201993301831,-79.45338869245904,-79.45352628943529,-79.45453549891477,-79.45600794078679,-79.45726739482471,-79.4578060577511,-79.45930741812759,-79.46003491659883],"lat":[43.66722020158632,43.66721696111229,43.66720916333836,43.66720266023016,43.66718041017835,43.66716513921067,43.66713536413032,43.66713233760297,43.66713226559843,43.66712653648763,43.66712522505406,43.66712391215355,43.6671181011001,43.66711760663293,43.66711679911666,43.66711548597068,43.66711417412787,43.66711287190081,43.66711605934946,43.66711925641467,43.66711794424475,43.66712564060141,43.6671243268819,43.66713202307507,43.66713072027952,43.66713849659642,43.66714620224779,43.66714964125519,43.66714938895826,43.66715709444787,43.66716479153852,43.66716799880101,43.667175694343,43.66718335929632,43.66719866536989,43.66721059789397,43.66721810297677,43.66722561072692,43.66722860504735,43.66724060925908,43.66724811530085,43.66725566147511,43.66626882851254,43.66446259064568,43.66273328332098,43.66184147854295,43.66011678482502,43.65917065525482,43.65782653915941,43.65782247361015,43.65729403375606,43.65723543943874,43.65707388474296,43.65704874855341,43.65686781839477,43.65836888478994,43.65906696670214,43.66045858177615,43.66060006565389,43.66162662867841,43.66312322760346,43.66440520413597,43.66495268730319,43.66647654946119,43.66722020158632]}]],[[{"lng":[-79.4370005208459,-79.43830835578167,-79.43971898820837,-79.44100799720991,-79.44143390607006,-79.44204680112124,-79.4421375978639,-79.44301181864398,-79.4440337050213,-79.44403747296023,-79.44641268593845,-79.4475310795553,-79.4477773919123,-79.4498588334838,-79.44892248528589,-79.44878470713039,-79.44808609609559,-79.44776062504295,-79.44523884101751,-79.44520683050061,-79.44398478793212,-79.44272197508872,-79.4417233052942,-79.44156668285623,-79.44073556437571,-79.44037700200448,-79.43973169547871,-79.43916325717504,-79.43894139900958,-79.43892901360998,-79.43856902282241,-79.43762259106066,-79.43760423374796,-79.43728832647878,-79.43698105562063,-79.43670948411248,-79.43634530973327,-79.4361112079259,-79.4360818720487,-79.43583990274092,-79.43559074199699,-79.4370005208459],"lat":[43.64997579972731,43.65007919794987,43.6501615695213,43.65027368189259,43.6503154991985,43.65044538369762,43.65046912656565,43.65067917535212,43.65090464102416,43.65090919335672,43.65335000889301,43.65449523569863,43.65474613807454,43.65686781839477,43.65704874855341,43.65707388474296,43.65723543943874,43.65729403375606,43.65782247361015,43.65782653915941,43.65808456022846,43.65833301060601,43.6585534516563,43.65858732423867,43.65877412308723,43.65885024895206,43.6589854736994,43.65909032746455,43.65851998773869,43.65849281239561,43.65757850685662,43.65513006537147,43.65508480411021,43.65427011941372,43.65347364600806,43.65273608340292,43.65180820977093,43.65118368403629,43.65112026814288,43.65049572415033,43.64988449423022,43.64997579972731]}]],[[{"lng":[-79.43364615607692,-79.43441950414582,-79.43557154378222,-79.43581056959688,-79.43603266168071,-79.43604867875808,-79.43628677423212,-79.43652367317316,-79.43676105261146,-79.43691801301179,-79.4369976176225,-79.43723358474912,-79.43746933385506,-79.4376047570093,-79.43770425175249,-79.43793858615648,-79.43817290699785,-79.43840619225787,-79.43863979707814,-79.43887248523335,-79.43910269260618,-79.43920750230467,-79.43930996140148,-79.43941168857691,-79.43951343257356,-79.43961424257525,-79.43971433730572,-79.43981444650164,-79.43991382613589,-79.4400129872709,-79.44011153535192,-79.44021010980306,-79.44030702292152,-79.44040383402417,-79.44050096617696,-79.44059706219467,-79.44069234072983,-79.44078690398923,-79.44088146510866,-79.44097521106816,-79.44106884081604,-79.44114738626283,-79.44116187168692,-79.44125408506373,-79.44134536456565,-79.44143674647249,-79.44152742493047,-79.44161717185467,-79.44170702118075,-79.44179583445742,-79.44188426735158,-79.44197176640169,-79.44205904710023,-79.44214561019385,-79.44223169296252,-79.44231754328473,-79.44240175842019,-79.44248597374713,-79.44256969228648,-79.44265308797011,-79.44273565440734,-79.44281608865352,-79.44290807304991,-79.4429982919527,-79.44308862734113,-79.44317826149596,-79.4432668433476,-79.44335563217737,-79.44344198214137,-79.44352925223167,-79.44361558861213,-79.44370120739705,-79.44378672651263,-79.44387093105583,-79.44395491739709,-79.4440337050213,-79.44301181864398,-79.4421375978639,-79.44204680112124,-79.44143390607006,-79.44100799720991,-79.43971898820837,-79.43830835578167,-79.4370005208459,-79.43559074199699,-79.43367724147058,-79.43247644877205,-79.43144146077299,-79.43070684664879,-79.43061604679848,-79.43036825239632,-79.43002458735569,-79.42983357105291,-79.42950664910846,-79.42919151594018,-79.42888038612818,-79.42864781152848,-79.42879604986385,-79.42894522346005,-79.42909449727975,-79.42924426975372,-79.42939426441185,-79.429544154897,-79.42964129865393,-79.42969569597459,-79.42984732459186,-79.42999886404363,-79.43015192420391,-79.43030428432209,-79.43045673202732,-79.43092899638596,-79.43364615607692],"lat":[43.6448742229321,43.64523599023685,43.64575597189928,43.64586285742251,43.64596041862283,43.64596963988038,43.64608092269059,43.64618318656177,43.64629454860716,43.64636871200444,43.64640580899955,43.64651255968941,43.64662380794669,43.64668876388944,43.64673954539041,43.64684627230253,43.64695749958408,43.64707321327036,43.64718451982968,43.64729572336839,43.64741589462263,43.64746683742192,43.64751774802554,43.6475732395993,43.64762414025537,43.64767952910446,43.64773490809925,43.64778578616781,43.64784115522528,43.64790102224926,43.64796088081938,43.64801182761358,43.64807166370063,43.64813149831298,43.64818683618833,43.64824666087869,43.64830647433439,43.64836627795187,43.64842617148225,43.64848596379351,43.64855025547764,43.64860083823363,43.64861003788093,43.64866980905844,43.64873406846021,43.64879832918512,43.64885816922958,43.64892240751407,43.64898664712386,43.64905537357201,43.64911509269479,43.64917930005304,43.64924800540223,43.64931679092327,43.64937647774088,43.64944516339472,43.64951382669795,43.64958248994119,43.6496466453173,43.64971538728751,43.649788528911,43.64985714038825,43.64992590917245,43.65000365589869,43.65007690310544,43.6501456396536,43.65022345388665,43.65030118084874,43.65037437349035,43.65044757859724,43.65052527195559,43.65060304549687,43.65068072759544,43.65075389069617,43.65083155179732,43.65090464102416,43.65067917535212,43.65046912656565,43.65044538369762,43.6503154991985,43.65027368189259,43.6501615695213,43.65007919794987,43.64997579972731,43.64988449423022,43.64976376368639,43.64968887688497,43.64964316939151,43.64770619191886,43.64747089128137,43.64683716334574,43.6459590619302,43.64546573479712,43.64462837100987,43.64381367340283,43.64302612611417,43.64249621717176,43.64257477286542,43.64264884018439,43.64272299870652,43.64280157490415,43.64287556289932,43.64294963928644,43.64299598450032,43.64301923711729,43.64309333698761,43.64316302441406,43.64323714354481,43.64330675181727,43.64338086212708,43.64360348671057,43.6448742229321]}]],[[{"lng":[-79.44138187993416,-79.44165972209922,-79.44185480688631,-79.44217954163518,-79.44252961970909,-79.44286355716211,-79.443195854664,-79.44352051004202,-79.44386706093356,-79.4439797358355,-79.44421130312706,-79.4445587050615,-79.4448988178568,-79.44508612009358,-79.44531948134916,-79.44532008815428,-79.44531114506937,-79.4440337050213,-79.44395491739709,-79.44387093105583,-79.44378672651263,-79.44370120739705,-79.44361558861213,-79.44352925223167,-79.44344198214137,-79.44335563217737,-79.4432668433476,-79.44317826149596,-79.44308862734113,-79.4429982919527,-79.44290807304991,-79.44281608865352,-79.44273565440734,-79.44265308797011,-79.44256969228648,-79.44248597374713,-79.44240175842019,-79.44231754328473,-79.44223169296252,-79.44214561019385,-79.44205904710023,-79.44197176640169,-79.44188426735158,-79.44179583445742,-79.44170702118075,-79.44161717185467,-79.44152742493047,-79.44143674647249,-79.44134536456565,-79.44125408506373,-79.44116187168692,-79.44114738626283,-79.44106884081604,-79.44097521106816,-79.44088146510866,-79.44078690398923,-79.44069234072983,-79.44059706219467,-79.44050096617696,-79.44040383402417,-79.44030702292152,-79.44021010980306,-79.44011153535192,-79.4400129872709,-79.43991382613589,-79.43981444650164,-79.43971433730572,-79.43961424257525,-79.43951343257356,-79.43941168857691,-79.43930996140148,-79.43920750230467,-79.43910269260618,-79.43887248523335,-79.43863979707814,-79.43840619225787,-79.43817290699785,-79.43793858615648,-79.43770425175249,-79.4376047570093,-79.43746933385506,-79.43723358474912,-79.4369976176225,-79.43691801301179,-79.43676105261146,-79.43652367317316,-79.43628677423212,-79.43604867875808,-79.43603266168071,-79.43581056959688,-79.43557154378222,-79.43441950414582,-79.43364615607692,-79.43092899638596,-79.43045673202732,-79.43030428432209,-79.43015192420391,-79.42999886404363,-79.42984732459186,-79.42969569597459,-79.42964129865393,-79.429544154897,-79.42939426441185,-79.42924426975372,-79.42909449727975,-79.42894522346005,-79.42879604986385,-79.42864781152848,-79.42850017166153,-79.42835254728374,-79.42819783758823,-79.42863213714163,-79.42971240940624,-79.43010951417877,-79.43157650289676,-79.43235690792088,-79.43314013996333,-79.4341652826127,-79.43450491477489,-79.43540171910306,-79.43596103834014,-79.43665655532664,-79.43726517159442,-79.43791063468001,-79.43910302540813,-79.43985079062598,-79.44095183611753,-79.44138187993416],"lat":[43.64090573087844,43.64163436722002,43.64209622318938,43.64296502966558,43.64383859149741,43.64471202225793,43.64558993063029,43.6464632328096,43.64733233104837,43.64764902773359,43.64822390187807,43.64908850834732,43.64996651757358,43.65048227398528,43.65111127248286,43.65114719898767,43.65119658870422,43.65090464102416,43.65083155179732,43.65075389069617,43.65068072759544,43.65060304549687,43.65052527195559,43.65044757859724,43.65037437349035,43.65030118084874,43.65022345388665,43.6501456396536,43.65007690310544,43.65000365589869,43.64992590917245,43.64985714038825,43.649788528911,43.64971538728751,43.6496466453173,43.64958248994119,43.64951382669795,43.64944516339472,43.64937647774088,43.64931679092327,43.64924800540223,43.64917930005304,43.64911509269479,43.64905537357201,43.64898664712386,43.64892240751407,43.64885816922958,43.64879832918512,43.64873406846021,43.64866980905844,43.64861003788093,43.64860083823363,43.64855025547764,43.64848596379351,43.64842617148225,43.64836627795187,43.64830647433439,43.64824666087869,43.64818683618833,43.64813149831298,43.64807166370063,43.64801182761358,43.64796088081938,43.64790102224926,43.64784115522528,43.64778578616781,43.64773490809925,43.64767952910446,43.64762414025537,43.6475732395993,43.64751774802554,43.64746683742192,43.64741589462263,43.64729572336839,43.64718451982968,43.64707321327036,43.64695749958408,43.64684627230253,43.64673954539041,43.64668876388944,43.64662380794669,43.64651255968941,43.64640580899955,43.64636871200444,43.64629454860716,43.64618318656177,43.64608092269059,43.64596963988038,43.64596041862283,43.64586285742251,43.64575597189928,43.64523599023685,43.6448742229321,43.64360348671057,43.64338086212708,43.64330675181727,43.64323714354481,43.64316302441406,43.64309333698761,43.64301923711729,43.64299598450032,43.64294963928644,43.64287556289932,43.64280157490415,43.64272299870652,43.64264884018439,43.64257477286542,43.64249621717176,43.64242217055291,43.64234362291732,43.64230548699341,43.64220334413609,43.64199313994636,43.64191757666302,43.64161796432663,43.64146654402293,43.64129274195768,43.64108624079626,43.64101878683404,43.64083302173842,43.6407146482554,43.64057553992082,43.64044433123621,43.64032712605049,43.64009577003768,43.63994394268278,43.63973382556108,43.64090573087844]}]],[[{"lng":[-79.34948024898068,-79.34961963115225,-79.3540759297083,-79.3555785862916,-79.35622238255408,-79.35222669928265,-79.35227344893275,-79.35229306705511,-79.35231303340828,-79.35250853526773,-79.35659352345118,-79.35815640583803,-79.35401819111479,-79.35439160929758,-79.35634788577451,-79.35669615700048,-79.35474326277854,-79.35507041423848,-79.35420743099725,-79.35390211246794,-79.35372185945599,-79.35246899171084,-79.35060690114361,-79.34975758687497,-79.3496659118333,-79.34960877293558,-79.34950048043166,-79.34948024898068],"lat":[43.64170050826433,43.64161700157216,43.63909245402804,43.64093295231567,43.6417075692922,43.64341929409717,43.64360901136705,43.64365430450752,43.64369060060618,43.64393656466068,43.64216760100179,43.64399083657411,43.64578159405401,43.64621465926067,43.64540544005546,43.64582013199376,43.64666991503583,43.64709781087755,43.64748156684223,43.64708549022371,43.64687584655626,43.64530457739347,43.64308526713675,43.64201499434957,43.64190564582589,43.64184180627262,43.6417233055998,43.64170050826433]}]],[[{"lng":[-79.42902456219936,-79.42860710829163,-79.42736231389426,-79.4255088797443,-79.42561934617861,-79.4257327042681,-79.42584603163216,-79.42595986980804,-79.42607392751579,-79.42618848075206,-79.42630323824582,-79.42641871091742,-79.4265098331839,-79.42653336600468,-79.42664943632725,-79.4267649086726,-79.42688109619397,-79.42699715101092,-79.42711405595995,-79.42723104544561,-79.42734783046824,-79.4274648197631,-79.42758312219102,-79.42770062222186,-79.42780431221856,-79.42781832652422,-79.42793579609626,-79.42805469617849,-79.42817381568251,-79.42829281781707,-79.42841242023955,-79.42853143734926,-79.42865113950459,-79.42877165908435,-79.42887434809916,-79.42887445028263,-79.42889137621438,-79.42892787077534,-79.42901109328292,-79.42913152557399,-79.42925182555007,-79.42937224270972,-79.42949276200905,-79.42961297471442,-79.42973349392621,-79.4298545090212,-79.42997481115017,-79.43009582619342,-79.4301439695204,-79.4302170605402,-79.43033807552276,-79.43045907553493,-79.43058020759374,-79.4307019228556,-79.4308049385625,-79.4309089760947,-79.43101269216587,-79.43111651042797,-79.43122034358537,-79.43132416185495,-79.43142798013216,-79.43156604470315,-79.43163479924247,-79.43173850051006,-79.43184209961731,-79.43194602013138,-79.43204961927702,-79.43215240097825,-79.43225632153062,-79.4323598037514,-79.43246267290969,-79.43256627217804,-79.43266905400128,-79.43277180630325,-79.43287457342301,-79.43297745752119,-79.43308010543815,-79.43318193827808,-79.43328492465902,-79.43338757505931,-79.4334894080432,-79.43359064268857,-79.43369257795413,-79.43379511167944,-79.43389541211712,-79.43465987769326,-79.43556677590271,-79.43567225973544,-79.43577751018734,-79.4358820454637,-79.43598752958616,-79.43609206508053,-79.43619588539734,-79.43629958913446,-79.43634574770822,-79.43640342419279,-79.43650620851083,-79.4366093140013,-79.43671139775481,-79.43681418244611,-79.43691553667269,-79.43701690315146,-79.43711744014252,-79.43721879474417,-79.43731921536822,-79.43741893526168,-79.43751875746631,-79.4376175435256,-79.43771654850124,-79.43779472207223,-79.43781461952948,-79.43791269069543,-79.43998498812134,-79.44162134282507,-79.44257640209109,-79.44266742884389,-79.44275772636129,-79.44284978930096,-79.44294091862588,-79.44303346472371,-79.44312614116959,-79.44321880140718,-79.44331309914163,-79.44340657948607,-79.44350170898676,-79.44359591889311,-79.44369104863188,-79.4437868798651,-79.44388294351725,-79.44397950428466,-79.44407626955481,-79.4441729466859,-79.44427054364191,-79.4443687376138,-79.44446632088535,-79.44456628247657,-79.44464763148665,-79.44472866012319,-79.4448108268474,-79.44489372287788,-79.4449758897767,-79.44505878597039,-79.44514238372435,-79.4452262136581,-79.44531062911597,-79.44539505850659,-79.44548040770255,-79.44556587067842,-79.44565111783891,-79.44573718261505,-79.44582324747311,-79.44590954440092,-79.44599632475398,-79.44608322117082,-79.44617103738604,-79.44625876525599,-79.4463466975678,-79.44643501117993,-79.44652377491464,-79.44661230681027,-79.44670095471287,-79.44679042021234,-79.44684588318013,-79.4468806148472,-79.44696997826566,-79.44707168210627,-79.44718050476338,-79.44728763853126,-79.44739536721494,-79.44750391586204,-79.44761176294111,-79.44771949424796,-79.44782826124708,-79.44793753929967,-79.44804620428457,-79.44806704494472,-79.44841069065797,-79.44756161043006,-79.4473964188868,-79.44725479295018,-79.4470776343823,-79.4469013955608,-79.44670184597655,-79.44620720498406,-79.44381320899127,-79.44248383094242,-79.44220966268982,-79.44116937424387,-79.44095183611753,-79.43985079062598,-79.43910302540813,-79.43791063468001,-79.43726517159442,-79.43665655532664,-79.43596103834014,-79.43540171910306,-79.43450491477489,-79.4341652826127,-79.43314013996333,-79.43235690792088,-79.43157650289676,-79.43010951417877,-79.42971240940624,-79.42902456219936],"lat":[43.64212700012245,43.64202223829202,43.63895350138281,43.63415138637722,43.63412140083874,43.63408245293535,43.63405250658448,43.63402256715599,43.63398812959262,43.63395819978303,43.63392827267074,43.63389835528972,43.63387710463706,43.63386842653567,43.63384301819343,43.63381310045705,43.63378319244757,43.63376228458245,43.63373229618873,43.63370689990401,43.63368150068577,43.63365610415848,43.63363522662232,43.63360983687988,43.63358875808784,43.63358444982633,43.63356806151231,43.63355169275268,43.6335308258419,43.63351445823763,43.6335025097879,43.63348164109568,43.63346978378795,43.63345793759249,43.63344386569674,43.63344386710143,43.63344157919361,43.63343757982343,43.63342522066827,43.63340887184782,43.6334015231721,43.63338967493483,43.63337782797409,43.63336597667339,43.63335412945729,43.63334678997178,43.63333935054294,43.63333201080097,43.63332817105097,43.63332017289758,43.63331283289802,43.63330999360998,43.63329815391622,43.63329532419264,43.63329223759649,43.63328916493614,43.63329058881336,43.63329201399923,43.63328893824975,43.63329036324699,43.63329178814998,43.63328918189077,43.63329462645537,43.63330055051601,43.63330647308075,43.63330789891575,43.63331382129284,43.63331973236217,43.63332115791545,43.63333157945295,43.63334199248833,43.63334791439866,43.63335382500488,43.63336873720499,43.63337914847028,43.63338506020151,43.63340006071407,43.63341495993949,43.63342087279609,43.63343578304558,43.63345068199898,43.63346107161549,43.63347597178965,43.63349538111965,43.63351025870952,43.63366926606877,43.63386631644754,43.63389026552053,43.63392321339293,43.63395615138075,43.63398010016662,43.63401303796523,43.63404596588256,43.63408339315587,43.6340930267738,43.63411182004079,43.6341492345479,43.6341821523105,43.63421505600441,43.6342524702394,43.63428986481811,43.63432284844999,43.63436023167326,43.63439762598791,43.63443950848757,43.63447688027314,43.63451425337036,43.63455611326103,43.63459347501343,43.63462605089617,43.63463532495938,43.63467717482289,43.63555626726286,43.63625392795964,43.63665862740932,43.63670037719176,43.63674661800956,43.63678388072081,43.63682563168248,43.63687190291615,43.6369091737622,43.63695103537713,43.63699282919954,43.63703461181137,43.63707191576232,43.6371092071109,43.63714651090731,43.63718832522018,43.63722114052926,43.63725846356954,43.63729578931218,43.63732861272911,43.637361448585,43.63739888354838,43.63743622009447,43.63746458683755,43.63749720124205,43.63753431227263,43.63756693768839,43.63759507192474,43.63762769722408,43.63765583134279,43.63768847598945,43.63771212164607,43.63774477729342,43.63777293202421,43.63780109920341,43.63782485684355,43.63785302250741,43.63788119922668,43.6379093758822,43.63792855354131,43.63795673979561,43.63798042651842,43.6380041256803,43.6380233225301,43.63804252209169,43.63807072885585,43.63808543853933,43.63810914708991,43.63812835610499,43.63814757616252,43.63815733198545,43.63816230501495,43.6381815235447,43.63821378266732,43.63824748873406,43.6382713594774,43.63829991928738,43.63832840011319,43.63835237026304,43.63838083978212,43.63840482222196,43.63842881149844,43.63845279234545,43.63845730632283,43.63877290423507,43.63861274954812,43.63859250139355,43.63858607629662,43.63858591979022,43.63858577550766,43.63860556883517,43.63870696019242,43.63917409387861,43.6394305486148,43.63948541552621,43.63968278035119,43.63973382556108,43.63994394268278,43.64009577003768,43.64032712605049,43.64044433123621,43.64057553992082,43.6407146482554,43.64083302173842,43.64101878683404,43.64108624079626,43.64129274195768,43.64146654402293,43.64161796432663,43.64191757666302,43.64199313994636,43.64212700012245]}]],[[{"lng":[-79.43465987769326,-79.43389541211712,-79.43379511167944,-79.43369257795413,-79.43359064268857,-79.4334894080432,-79.43338757505931,-79.43328492465902,-79.43318193827808,-79.43308010543815,-79.43297745752119,-79.43287457342301,-79.43277180630325,-79.43266905400128,-79.43256627217804,-79.43246267290969,-79.4323598037514,-79.43225632153062,-79.43215240097825,-79.43204961927702,-79.43194602013138,-79.43184209961731,-79.43173850051006,-79.43163479924247,-79.43156604470315,-79.43142798013216,-79.43132416185495,-79.43122034358537,-79.43111651042797,-79.43101269216587,-79.4309089760947,-79.4308049385625,-79.4307019228556,-79.43058020759374,-79.43045907553493,-79.43033807552276,-79.4302170605402,-79.4301439695204,-79.43009582619342,-79.42997481115017,-79.4298545090212,-79.42973349392621,-79.42961297471442,-79.42949276200905,-79.42937224270972,-79.42925182555007,-79.42913152557399,-79.42901109328292,-79.42892787077534,-79.42889137621438,-79.42887445028263,-79.42805500583616,-79.42714462241734,-79.42791819942505,-79.42816845238016,-79.42850672785042,-79.42884253653887,-79.42916577562094,-79.42947406964184,-79.42979258206131,-79.43011534426675,-79.43041737974191,-79.43071658220008,-79.43104605771204,-79.43116553182612,-79.43148866047238,-79.4318210707884,-79.43213890069147,-79.4324422539983,-79.43274842360738,-79.43305821325112,-79.43336309905381,-79.43363808810228,-79.43380901816764,-79.43389943171664,-79.43418207363767,-79.43426730978939,-79.43465987769326],"lat":[43.63366926606877,43.63351025870952,43.63349538111965,43.63347597178965,43.63346107161549,43.63345068199898,43.63343578304558,43.63342087279609,43.63341495993949,43.63340006071407,43.63338506020151,43.63337914847028,43.63336873720499,43.63335382500488,43.63334791439866,43.63334199248833,43.63333157945295,43.63332115791545,43.63331973236217,43.63331382129284,43.63330789891575,43.63330647308075,43.63330055051601,43.63329462645537,43.63328918189077,43.63329178814998,43.63329036324699,43.63328893824975,43.63329201399923,43.63329058881336,43.63328916493614,43.63329223759649,43.63329532419264,43.63329815391622,43.63330999360998,43.63331283289802,43.63332017289758,43.63332817105097,43.63333201080097,43.63333935054294,43.63334678997178,43.63335412945729,43.63336597667339,43.63337782797409,43.63338967493483,43.6334015231721,43.63340887184782,43.63342522066827,43.63343757982343,43.63344157919361,43.63344386710143,43.631551972448,43.62967646405432,43.62977748839165,43.62980757659128,43.62985273781474,43.62990236517349,43.62995190880724,43.63001015823367,43.63006854721699,43.63010898949675,43.63014464517275,43.63020276638952,43.63028380764738,43.63030795293952,43.63036189951335,43.63042056359634,43.63050144191281,43.63057302867497,43.63062673908219,43.63069850249069,43.63077019783984,43.63085957685166,43.63092943505281,43.63096668238988,43.63106057598394,43.6324663404104,43.63366926606877]}]],[[{"lng":[-79.43465987769326,-79.43426730978939,-79.43418207363767,-79.43446118377648,-79.43474887824962,-79.43504858464537,-79.43528930720393,-79.43552699454035,-79.43570677339137,-79.43587157757926,-79.43596267856954,-79.43607192510916,-79.43609665400334,-79.43615291170619,-79.43629195186472,-79.436384953336,-79.43661191427333,-79.43686302040726,-79.43709657960515,-79.43734755619957,-79.43756100519747,-79.43782717986946,-79.4380940380904,-79.43834633200288,-79.43857530036306,-79.43880241592375,-79.43898229064909,-79.43915193135742,-79.4393084971578,-79.43932948217943,-79.43930527341323,-79.43927574863355,-79.43940136599569,-79.43956442818705,-79.43969382930985,-79.43975447234205,-79.4399006725654,-79.44001654113781,-79.4400409451377,-79.43996469415397,-79.44010586511165,-79.44032814129024,-79.44055350564842,-79.44081969782212,-79.4410970523857,-79.44139973502371,-79.44170145292665,-79.44197092261027,-79.44223306998677,-79.44246765092456,-79.442681503759,-79.44289291925824,-79.4430959773948,-79.44326489556349,-79.44345656390691,-79.44366919265782,-79.44384331381031,-79.44392091060732,-79.44390152276549,-79.44396452098557,-79.44414300944543,-79.44434551008824,-79.44454011735799,-79.44473722436011,-79.44496669277564,-79.44522064542447,-79.44545716926793,-79.44570253980693,-79.44592777999571,-79.44613912324631,-79.44635937347553,-79.44661544088092,-79.44687987704314,-79.44716609394428,-79.44742839926015,-79.44767480467758,-79.44791329477282,-79.44813529815991,-79.44836283305678,-79.44860599552986,-79.44884530132242,-79.44909731320257,-79.44934835248321,-79.44951498546988,-79.44974609934526,-79.44995505603296,-79.45020673566805,-79.45039426166555,-79.45063007525552,-79.45087526063577,-79.45116323963028,-79.4514759244756,-79.45178144083803,-79.45208527153031,-79.45238373772121,-79.45274869957198,-79.45294473244181,-79.4531869387303,-79.45349299447682,-79.4535078352327,-79.45351586646505,-79.45352398896813,-79.4535304763868,-79.45353706600231,-79.45354191833837,-79.45354709048959,-79.45355040993314,-79.45355393376497,-79.45356044400395,-79.45356870477063,-79.45357685012321,-79.45358582626567,-79.45359643751674,-79.45360693335863,-79.45362791317855,-79.45363100182023,-79.45364478113892,-79.45365834286194,-79.45367355293347,-79.45368958057753,-79.45370572365913,-79.45372328425739,-79.45374072946441,-79.45375900547049,-79.45377891661163,-79.45375041758267,-79.453741935445,-79.45360112475871,-79.45345245838055,-79.45330236134112,-79.45315389947147,-79.45300460683202,-79.45285544299693,-79.45270709679808,-79.45255771543047,-79.4524093693843,-79.45227562385492,-79.45226019253612,-79.45211174444842,-79.45196328308995,-79.45181505290647,-79.45166682282668,-79.45151907704928,-79.45137074496067,-79.45122343491498,-79.45121190041115,-79.45107500083964,-79.45092818853992,-79.45078067440478,-79.45063337821178,-79.45048646399366,-79.45033918150159,-79.45019247186521,-79.450046709042,-79.4499338121489,-79.44982124890268,-79.44970918317958,-79.44959733543077,-79.44948525861149,-79.44937351319044,-79.44926144771119,-79.44915031559246,-79.44903928574185,-79.44892815377275,-79.4488171240717,-79.44870588774333,-79.44859589372582,-79.44848536161115,-79.44837516337213,-79.44826588496058,-79.44815558470921,-79.44806704494472,-79.44804620428457,-79.44793753929967,-79.44782826124708,-79.44771949424796,-79.44761176294111,-79.44750391586204,-79.44739536721494,-79.44728763853126,-79.44718050476338,-79.44707168210627,-79.44696997826566,-79.4468806148472,-79.44684588318013,-79.44679042021234,-79.44670095471287,-79.44661230681027,-79.44652377491464,-79.44643501117993,-79.4463466975678,-79.44625876525599,-79.44617103738604,-79.44608322117082,-79.44599632475398,-79.44590954440092,-79.44582324747311,-79.44573718261505,-79.44565111783891,-79.44556587067842,-79.44548040770255,-79.44539505850659,-79.44531062911597,-79.4452262136581,-79.44514238372435,-79.44505878597039,-79.4449758897767,-79.44489372287788,-79.4448108268474,-79.44472866012319,-79.44464763148665,-79.44456628247657,-79.44446632088535,-79.4443687376138,-79.44427054364191,-79.4441729466859,-79.44407626955481,-79.44397950428466,-79.44388294351725,-79.4437868798651,-79.44369104863188,-79.44359591889311,-79.44350170898676,-79.44340657948607,-79.44331309914163,-79.44321880140718,-79.44312614116959,-79.44303346472371,-79.44294091862588,-79.44284978930096,-79.44275772636129,-79.44266742884389,-79.44257640209109,-79.44162134282507,-79.43998498812134,-79.43791269069543,-79.43781461952948,-79.43779472207223,-79.43771654850124,-79.4376175435256,-79.43751875746631,-79.43741893526168,-79.43731921536822,-79.43721879474417,-79.43711744014252,-79.43701690315146,-79.43691553667269,-79.43681418244611,-79.43671139775481,-79.4366093140013,-79.43650620851083,-79.43640342419279,-79.43634574770822,-79.43629958913446,-79.43619588539734,-79.43609206508053,-79.43598752958616,-79.4358820454637,-79.43577751018734,-79.43567225973544,-79.43556677590271,-79.43465987769326],"lat":[43.63366926606877,43.6324663404104,43.63106057598394,43.63117242468983,43.6312528829483,43.63134709812228,43.63148092440041,43.63160570652771,43.6317522908351,43.63182206210146,43.63177379731822,43.63180229856417,43.63186565160601,43.63184841721297,43.63189082905546,43.63198212239499,43.63204374085442,43.63212819433183,43.63223500256151,43.63232845529539,43.63246190387169,43.63256456541312,43.63264482046312,43.63274279005018,43.6328494425028,43.63296057023311,43.63304863707612,43.6330779602465,43.63317011927082,43.63328743303558,43.63341313174916,43.6335388478814,43.63359457577418,43.63366881896447,43.63375610551807,43.63380644491676,43.63384895010953,43.63394964472985,43.63408500920136,43.63418749246819,43.63430194547937,43.63442659726845,43.63457811699222,43.63471227910237,43.63480617345807,43.63487331625542,43.63492244091571,43.63495762203173,43.63491618493806,43.63483386216296,43.63474666539764,43.63468653131814,43.63468479659318,43.63471860485986,43.63479773317593,43.63487273557341,43.63494262220052,43.63499769126864,43.6350874482079,43.63513781743414,43.63521676515376,43.6352961294589,43.63538429804773,43.63549050450215,43.63557464570506,43.63567262266419,43.63578845619865,43.63588631535327,43.63598390025685,43.63608129577666,43.63610679526629,43.63618238266622,43.63626249424212,43.63633840009373,43.63641398040347,43.63647584101928,43.63645207375939,43.63640557691581,43.63635465369263,43.63627243483272,43.63618116101954,43.63618008002864,43.63626450514476,43.63641530034005,43.63649945426623,43.63647528214091,43.63648319628833,43.63651724633942,43.63653394652285,43.63649225966157,43.63652766998111,43.63657232638877,43.6366214758999,43.63665268760412,43.63670174001896,43.63679670234004,43.63685786964182,43.63696017093861,43.63831939721008,43.63834660429085,43.63836030611702,43.63837842020076,43.63839651215933,43.63841460550034,43.63843267533371,43.63844624845024,43.63846429754162,43.63848234939839,43.63852744792758,43.63856806909821,43.63861318975024,43.63865382059868,43.6386944735701,43.63873962602257,43.63879851347772,43.63882556153043,43.6388662573611,43.63891145128945,43.63895216647422,43.63899289271831,43.63902911947812,43.63906986645729,43.6391151129159,43.63915586956806,43.63919664833879,43.6391917617289,43.63919164697178,43.63918524077076,43.63917422705106,43.63916319377658,43.63915218243604,43.63914566070098,43.63913013842757,43.63911462702892,43.63910360246527,43.63908809068217,43.63907727747309,43.63907706849897,43.6390615549476,43.63905054206706,43.63903053003833,43.63901051781873,43.63899951405536,43.6389795000696,43.63895949974139,43.63895484238932,43.63893948398968,43.63892399107002,43.63890398740478,43.63887948546029,43.63886399059145,43.63883498741008,43.63881949493721,43.63879501302956,43.6387844798194,43.63876494893641,43.63874992573461,43.63873040433646,43.63871979178708,43.63870027155811,43.63868524792054,43.63866573579488,43.63864622494899,43.63862671260932,43.63860720154959,43.63858777759823,43.63856377933783,43.63854877575356,43.63852477450834,43.63850078564555,43.63847678280349,43.63845730632283,43.63845279234545,43.63842881149844,43.63840482222196,43.63838083978212,43.63835237026304,43.63832840011319,43.63829991928738,43.6382713594774,43.63824748873406,43.63821378266732,43.6381815235447,43.63816230501495,43.63815733198545,43.63814757616252,43.63812835610499,43.63810914708991,43.63808543853933,43.63807072885585,43.63804252209169,43.6380233225301,43.6380041256803,43.63798042651842,43.63795673979561,43.63792855354131,43.6379093758822,43.63788119922668,43.63785302250741,43.63782485684355,43.63780109920341,43.63777293202421,43.63774477729342,43.63771212164607,43.63768847598945,43.63765583134279,43.63762769722408,43.63759507192474,43.63756693768839,43.63753431227263,43.63749720124205,43.63746458683755,43.63743622009447,43.63739888354838,43.637361448585,43.63732861272911,43.63729578931218,43.63725846356954,43.63722114052926,43.63718832522018,43.63714651090731,43.6371092071109,43.63707191576232,43.63703461181137,43.63699282919954,43.63695103537713,43.6369091737622,43.63687190291615,43.63682563168248,43.63678388072081,43.63674661800956,43.63670037719176,43.63665862740932,43.63625392795964,43.63555626726286,43.63467717482289,43.63463532495938,43.63462605089617,43.63459347501343,43.63455611326103,43.63451425337036,43.63447688027314,43.63443950848757,43.63439762598791,43.63436023167326,43.63432284844999,43.63428986481811,43.6342524702394,43.63421505600441,43.6341821523105,43.6341492345479,43.63411182004079,43.6340930267738,43.63408339315587,43.63404596588256,43.63401303796523,43.63398010016662,43.63395615138075,43.63392321339293,43.63389026552053,43.63386631644754,43.63366926606877]}]],[[{"lng":[-79.44620720498406,-79.44670184597655,-79.4469013955608,-79.4470776343823,-79.44725479295018,-79.4473964188868,-79.44756161043006,-79.44841069065797,-79.44806704494472,-79.44815558470921,-79.44826588496058,-79.44837516337213,-79.44848536161115,-79.44859589372582,-79.44870588774333,-79.4488171240717,-79.44892815377275,-79.44903928574185,-79.44915031559246,-79.44926144771119,-79.44937351319044,-79.44948525861149,-79.44959733543077,-79.44970918317958,-79.44982124890268,-79.4499338121489,-79.450046709042,-79.45019247186521,-79.45033918150159,-79.45048646399366,-79.45063337821178,-79.45078067440478,-79.45092818853992,-79.45107500083964,-79.45121190041115,-79.45122343491498,-79.45137074496067,-79.45151907704928,-79.45166682282668,-79.45181505290647,-79.45196328308995,-79.45211174444842,-79.45226019253612,-79.45227562385492,-79.4524093693843,-79.45255771543047,-79.45270709679808,-79.45285544299693,-79.45300460683202,-79.45315389947147,-79.45330236134112,-79.45345245838055,-79.45360112475871,-79.453741935445,-79.45375041758267,-79.45377891661163,-79.45379964302688,-79.45382026958318,-79.45384253128113,-79.45386469081383,-79.45388838329775,-79.453912178008,-79.45393762107489,-79.45398252717459,-79.45402660033629,-79.45404357455125,-79.45438216038809,-79.4546247374806,-79.45464218386785,-79.45475607069349,-79.45517346560656,-79.45551816112527,-79.45587689119181,-79.45624247680929,-79.45315643884007,-79.45134048808239,-79.44904774725806,-79.44870027476028,-79.44610647065004,-79.44352051004202,-79.443195854664,-79.44286355716211,-79.44252961970909,-79.44217954163518,-79.44185480688631,-79.44165972209922,-79.44138187993416,-79.44095183611753,-79.44116937424387,-79.44220966268982,-79.44248383094242,-79.44381320899127,-79.44620720498406],"lat":[43.63870696019242,43.63860556883517,43.63858577550766,43.63858591979022,43.63858607629662,43.63859250139355,43.63861274954812,43.63877290423507,43.63845730632283,43.63847678280349,43.63850078564555,43.63852477450834,43.63854877575356,43.63856377933783,43.63858777759823,43.63860720154959,43.63862671260932,43.63864622494899,43.63866573579488,43.63868524792054,43.63870027155811,43.63871979178708,43.63873040433646,43.63874992573461,43.63876494893641,43.6387844798194,43.63879501302956,43.63881949493721,43.63883498741008,43.63886399059145,43.63887948546029,43.63890398740478,43.63892399107002,43.63893948398968,43.63895484238932,43.63895949974139,43.6389795000696,43.63899951405536,43.63901051781873,43.63903053003833,43.63905054206706,43.6390615549476,43.63907706849897,43.63907727747309,43.63908809068217,43.63910360246527,43.63911462702892,43.63913013842757,43.63914566070098,43.63915218243604,43.63916319377658,43.63917422705106,43.63918524077076,43.63919164697178,43.6391917617289,43.63919664833879,43.63923752815636,43.63927831659812,43.6393191271568,43.63935993632883,43.63940076623413,43.63944159751721,43.63947795005011,43.63956857837837,43.63966378648487,43.63969552339294,43.64039335346371,43.64093243785155,43.64097768417642,43.64128079386161,43.64229493988542,43.64311446703897,43.6439971072157,43.6448799287761,43.64549094301312,43.64586712294921,43.64633123642865,43.64543963596859,43.64595370600923,43.6464632328096,43.64558993063029,43.64471202225793,43.64383859149741,43.64296502966558,43.64209622318938,43.64163436722002,43.64090573087844,43.63973382556108,43.63968278035119,43.63948541552621,43.6394305486148,43.63917409387861,43.63870696019242]}]],[[{"lng":[-79.44938201049821,-79.44971512656257,-79.45001594917329,-79.45005873057657,-79.45033869508482,-79.45064707008548,-79.45105756270837,-79.45119936865808,-79.45129085305976,-79.45137655441218,-79.45143211865616,-79.45176628594629,-79.45183380560856,-79.4521770705627,-79.45244175173248,-79.44988920762498,-79.4498588334838,-79.4477773919123,-79.4475310795553,-79.44641268593845,-79.44403747296023,-79.4440337050213,-79.44531114506937,-79.44532008815428,-79.44531948134916,-79.44508612009358,-79.4448988178568,-79.4445587050615,-79.44421130312706,-79.4439797358355,-79.44386706093356,-79.44352051004202,-79.44610647065004,-79.44870027476028,-79.44904774725806,-79.44938201049821],"lat":[43.64720015131312,43.64804645434512,43.64882489281922,43.64894249995393,43.64965764010599,43.65044518121102,43.65150866853632,43.65187526413265,43.65211514869148,43.65234136168538,43.65252674697913,43.65355318895342,43.65387826798653,43.655183985779,43.65631300798366,43.65686372934526,43.65686781839477,43.65474613807454,43.65449523569863,43.65335000889301,43.65090919335672,43.65090464102416,43.65119658870422,43.65114719898767,43.65111127248286,43.65048227398528,43.64996651757358,43.64908850834732,43.64822390187807,43.64764902773359,43.64733233104837,43.6464632328096,43.64595370600923,43.64543963596859,43.64633123642865,43.64720015131312]}]],[[{"lng":[-79.4566301082278,-79.45697481645922,-79.45731898391813,-79.4576489937234,-79.45779272680798,-79.45790003570585,-79.45796587475371,-79.45802278421775,-79.4581165473348,-79.45828647813866,-79.45839099412272,-79.45849745643123,-79.45906187387702,-79.46003622945791,-79.4588026837495,-79.45864231028533,-79.45703793685989,-79.45688193113422,-79.45426128165352,-79.45380772076793,-79.45244175173248,-79.4521770705627,-79.45183380560856,-79.45176628594629,-79.45143211865616,-79.45137655441218,-79.45129085305976,-79.45119936865808,-79.45105756270837,-79.45064707008548,-79.45033869508482,-79.45005873057657,-79.45001594917329,-79.44971512656257,-79.44938201049821,-79.44904774725806,-79.45134048808239,-79.45315643884007,-79.45624247680929,-79.4566301082278],"lat":[43.64581255819483,43.64664108278675,43.64748310205361,43.6482754177111,43.64862393719867,43.64891354156094,43.64915757591817,43.64945091113049,43.64990236979401,43.65056199352076,43.65086506276883,43.65113205985722,43.65243164831159,43.65470033935445,43.65498528043133,43.65501462485648,43.65534864738915,43.65538705037022,43.65593695003398,43.65603433839185,43.65631300798366,43.655183985779,43.65387826798653,43.65355318895342,43.65252674697913,43.65234136168538,43.65211514869148,43.65187526413265,43.65150866853632,43.65044518121102,43.64965764010599,43.64894249995393,43.64882489281922,43.64804645434512,43.64720015131312,43.64633123642865,43.64586712294921,43.64549094301312,43.6448799287761,43.64581255819483]}]],[[{"lng":[-79.4588026837495,-79.46003622945791,-79.4609044320629,-79.4609609513539,-79.46147888218013,-79.4618745350365,-79.46237385181074,-79.46294755841612,-79.46238681436782,-79.46171537608259,-79.46049341882494,-79.45881929222114,-79.45641632959448,-79.45600794078679,-79.45453549891477,-79.45352628943529,-79.45338869245904,-79.45201993301831,-79.45133093109389,-79.4498588334838,-79.44988920762498,-79.45244175173248,-79.45380772076793,-79.45426128165352,-79.45688193113422,-79.45703793685989,-79.45864231028533,-79.4588026837495],"lat":[43.65498528043133,43.65470033935445,43.65679196159833,43.65694125675973,43.65830331061107,43.65916842294143,43.66029166724778,43.66165446418377,43.66175494592784,43.66188094119336,43.66217514806018,43.66252625944001,43.66303854362499,43.66312322760346,43.66162662867841,43.66060006565389,43.66045858177615,43.65906696670214,43.65836888478994,43.65686781839477,43.65686372934526,43.65631300798366,43.65603433839185,43.65593695003398,43.65538705037022,43.65534864738915,43.65501462485648,43.65498528043133]}]],[[{"lng":[-79.46379281961092,-79.46396820348612,-79.46454749832257,-79.4647997248221,-79.46504687124192,-79.46506019023983,-79.46532786574156,-79.46535357167703,-79.46377494375268,-79.46299074370783,-79.46296661448244,-79.46115619529932,-79.46003491659883,-79.45930741812759,-79.4578060577511,-79.45726739482471,-79.45600794078679,-79.45641632959448,-79.45881929222114,-79.46049341882494,-79.46171537608259,-79.46238681436782,-79.46294755841612,-79.46379281961092],"lat":[43.66364673124664,43.6640361768662,43.66539913015984,43.66599224292683,43.66657619478669,43.66660337981939,43.66723279702219,43.66729165552348,43.66727053645231,43.66725999250242,43.66725966798791,43.66723980610503,43.66722020158632,43.66647654946119,43.66495268730319,43.66440520413597,43.66312322760346,43.66303854362499,43.66252625944001,43.66217514806018,43.66188094119336,43.66175494592784,43.66165446418377,43.66364673124664]}]],[[{"lng":[-79.47287416775356,-79.47181895288178,-79.46933361992103,-79.46614115599833,-79.46535357167703,-79.46532786574156,-79.46506019023983,-79.46504687124192,-79.4647997248221,-79.46454749832257,-79.46396820348612,-79.46379281961092,-79.46294755841612,-79.46401203289453,-79.46505541175605,-79.46593197377292,-79.46661893305094,-79.46823422418016,-79.46926957521107,-79.4694261707248,-79.47001005587367,-79.47008310932181,-79.47013933074432,-79.47018472153188,-79.47022166918387,-79.47081254510695,-79.47152942596635,-79.47212444646387,-79.47285991454545,-79.47292068836583,-79.47287416775356],"lat":[43.66734738603417,43.66731977788611,43.66729101891109,43.66726171945703,43.66729165552348,43.66723279702219,43.66660337981939,43.66657619478669,43.66599224292683,43.66539913015984,43.6640361768662,43.66364673124664,43.66165446418377,43.66140312485303,43.66116058433043,43.66097871957128,43.6608529077226,43.66048739017177,43.66025361551763,43.66037733064043,43.66082633970162,43.6608813300207,43.66092259188935,43.6609997169873,43.66109923419675,43.66253423797176,43.66415546577402,43.66551409013784,43.66720766380648,43.66734800765162,43.66734738603417]}]],[[{"lng":[-79.46723321814275,-79.46821207553589,-79.46846521566657,-79.46921963853367,-79.46948359195218,-79.46952059242977,-79.46956182062384,-79.46959540351762,-79.46963959643593,-79.46941913634888,-79.46936457154443,-79.46933729896826,-79.46931273813294,-79.46929843661451,-79.46928047896661,-79.46926957521107,-79.46823422418016,-79.46661893305094,-79.46593197377292,-79.46505541175605,-79.46401203289453,-79.46294755841612,-79.46237385181074,-79.4618745350365,-79.46147888218013,-79.4609609513539,-79.4609044320629,-79.46003622945791,-79.46102350506087,-79.46232922575203,-79.46369332437844,-79.46528584146431,-79.46689115822331,-79.46723321814275],"lat":[43.65397321943868,43.6563272405777,43.65688444013583,43.65867731603184,43.65936509784025,43.65940610261422,43.65943366089964,43.65945220475271,43.65947980271037,43.65980092458175,43.65990380751913,43.6599709577031,43.66005614831838,43.66012797324715,43.6601907471448,43.66025361551763,43.66048739017177,43.6608529077226,43.66097871957128,43.66116058433043,43.66140312485303,43.66165446418377,43.66029166724778,43.65916842294143,43.65830331061107,43.65694125675973,43.65679196159833,43.65470033935445,43.65448849649099,43.6541954103896,43.6538941820446,43.65353290043417,43.65319427344895,43.65397321943868]}]],[[{"lng":[-79.35443070959595,-79.35608944929439,-79.35673288639025,-79.35683270022189,-79.34964727071362,-79.34951027998723,-79.3495052113867,-79.34863753616769,-79.34816360963904,-79.3481273917169,-79.34798395741714,-79.34784452879958,-79.34771766914741,-79.34760214035427,-79.34748247898548,-79.34740725048421,-79.34730916246103,-79.34728368143995,-79.34745199658671,-79.34764063035739,-79.35433241561689,-79.3543594571878,-79.35443070959595],"lat":[43.64775943467788,43.64908878434539,43.64987239444795,43.65010788292163,43.65372300062611,43.65359499366154,43.6535859184164,43.65252446581466,43.65194589681746,43.65190936513937,43.65174066395306,43.65156760948938,43.65141724168786,43.65124894345966,43.65108967742221,43.65098056504385,43.65079901563933,43.65074013383955,43.6505085145752,43.65041212952899,43.64764099261651,43.64767288919245,43.64775943467788]}]],[[{"lng":[-79.46990322952067,-79.4702342229118,-79.47056833399478,-79.47063337764043,-79.47104206540013,-79.47172473947252,-79.47211894150807,-79.47238249964488,-79.47295992102957,-79.47317150878786,-79.47337068984832,-79.47369385036204,-79.47413240364901,-79.47435905576991,-79.4745386487493,-79.47466615852801,-79.47475343742039,-79.47487763186659,-79.47505507536698,-79.47600816183635,-79.47625309393496,-79.47744525095665,-79.47761584776185,-79.47772398841254,-79.47781685533414,-79.47785706559957,-79.47790222451253,-79.47801297603785,-79.47838427141325,-79.47867516347164,-79.47897621724164,-79.47928761273575,-79.4795560050766,-79.479571814159,-79.47988066674344,-79.48018294004399,-79.48041406779841,-79.48045860231151,-79.47907250588817,-79.47815119322519,-79.47766853077884,-79.47722211977161,-79.47629350277784,-79.47550929426087,-79.4751702387189,-79.47436428222301,-79.47407819546382,-79.472859937065,-79.4717787128216,-79.47116841014132,-79.47001933974914,-79.46870203766387,-79.46689115822331,-79.46528584146431,-79.46369332437844,-79.46232922575203,-79.46102350506087,-79.46003622945791,-79.45906187387702,-79.45849745643123,-79.45839099412272,-79.45828647813866,-79.4581165473348,-79.45802278421775,-79.45796587475371,-79.45790003570585,-79.45779272680798,-79.4576489937234,-79.45731898391813,-79.45697481645922,-79.4566301082278,-79.45624247680929,-79.45587689119181,-79.45551816112527,-79.45517346560656,-79.45475607069349,-79.45464218386785,-79.4546247374806,-79.45438216038809,-79.45404357455125,-79.45402660033629,-79.45398252717459,-79.45393762107489,-79.453912178008,-79.45388838329775,-79.45386469081383,-79.45384253128113,-79.45382026958318,-79.45379964302688,-79.45377891661163,-79.45386321381582,-79.45398399446272,-79.45410549049917,-79.45422686887807,-79.45434836498058,-79.45446904355515,-79.45459053971433,-79.45471213809651,-79.45483364745121,-79.45495514368658,-79.45507687058067,-79.45519836685475,-79.4553198631572,-79.4554423054265,-79.45556309948951,-79.45568563087176,-79.45580725557852,-79.45592886722046,-79.45605058106736,-79.45617170783412,-79.4562934347131,-79.45641494417426,-79.45653690376307,-79.45665861760672,-79.45677964208468,-79.45690148409389,-79.45702239334996,-79.45714363511438,-79.45726525969907,-79.45738638625063,-79.4575075257151,-79.45762865220557,-79.45774979159083,-79.45787021556981,-79.45891012428879,-79.45903432900076,-79.45915721795579,-79.45922489626709,-79.45928000465339,-79.45940371103714,-79.45952659980284,-79.45964878592378,-79.45977147016573,-79.45989375832775,-79.4600159442144,-79.46013801508406,-79.46026021584881,-79.46038157121555,-79.4605029392342,-79.46062442207155,-79.46074589211388,-79.46086654452341,-79.4609874265966,-79.46110807882755,-79.4612280282808,-79.4613480798301,-79.46146792689385,-79.46158705850333,-79.46170661141448,-79.46182656037007,-79.46194497632281,-79.46206740292399,-79.46218980650927,-79.46231305040405,-79.46243534928213,-79.46255683051305,-79.46267946316894,-79.46280035610619,-79.46292123637606,-79.46304223124591,-79.46316311127329,-79.46328388899083,-79.46340428281161,-79.46352364206236,-79.46364380624566,-79.46376268150513,-79.46388202788651,-79.46400078589787,-79.4641187262165,-79.46423736930113,-79.46435459398042,-79.46447263605171,-79.46458904289268,-79.46470626712568,-79.46482289044459,-79.46493858377781,-79.46505438923644,-79.46516997778311,-79.46528495303001,-79.46539995282203,-79.46551422475218,-79.46562839432788,-79.46574183607827,-79.46585551900162,-79.46596804071196,-79.46598216777546,-79.46884518069666,-79.46886319026889,-79.46889405712288,-79.46918349518117,-79.46918584556396,-79.46921201206746,-79.4692412322659,-79.46927188317407,-79.46930193303062,-79.46933361797726,-79.46936600625079,-79.46939841865853,-79.46943233988063,-79.46946617099766,-79.46950081968497,-79.46953721541635,-79.46957349924679,-79.46960989734232,-79.46964771412922,-79.46968634849381,-79.46972499492665,-79.46976445893347,-79.46990322952067],"lat":[43.63720310230798,43.63747768591229,43.63781082387175,43.63787020786693,43.63825385384868,43.63895173453934,43.6393306818726,43.63956834930155,43.64011628115451,43.64029914979193,43.64041892770853,43.64057618936384,43.64084769690425,43.64101726131678,43.64120429137169,43.64137703288705,43.6415357342334,43.64168601606234,43.64183691823571,43.64245736199742,43.64263625848248,43.64342200488876,43.64353680350421,43.64361476165683,43.64371052045268,43.6437740705888,43.64386028184961,43.64407780681315,43.6449336277303,43.64565784690438,43.64640911672367,43.64715602223314,43.64785293331177,43.6479071561041,43.64865411602621,43.64941899169585,43.65006580353431,43.65020142680432,43.65049365097062,43.65070202747944,43.65081712740322,43.65090570207921,43.65112746936171,43.65127904882457,43.65135554432057,43.65153392198624,43.65160211923057,43.65187400345223,43.65212061087725,43.65225207060484,43.65252034894052,43.65281336795241,43.65319427344895,43.65353290043417,43.6538941820446,43.6541954103896,43.65448849649099,43.65470033935445,43.65243164831159,43.65113205985722,43.65086506276883,43.65056199352076,43.64990236979401,43.64945091113049,43.64915757591817,43.64891354156094,43.64862393719867,43.6482754177111,43.64748310205361,43.64664108278675,43.64581255819483,43.6448799287761,43.6439971072157,43.64311446703897,43.64229493988542,43.64128079386161,43.64097768417642,43.64093243785155,43.64039335346371,43.63969552339294,43.63966378648487,43.63956857837837,43.63947795005011,43.63944159751721,43.63940076623413,43.63935993632883,43.6393191271568,43.63927831659812,43.63923752815636,43.63919664833879,43.63920228980651,43.6392084247362,43.63921456921465,43.63922530303699,43.63923144725808,43.63923758029459,43.63924372425893,43.63924986947621,43.6392515123169,43.63925765589454,43.63925480037379,43.6392609436933,43.63926708688401,43.6392642406425,43.63926587304054,43.63926752878604,43.63926467110618,43.63926631416395,43.63926795847296,43.6392605926346,43.63925773581716,43.63925937697729,43.63924743093502,43.63924907459609,43.63924170673163,43.63923434977693,43.63923148114331,43.63921961478275,43.63921675554859,43.63920938841893,43.63919752029116,43.63919015290364,43.63917828451773,43.63916640635016,43.63904540195541,43.63903357353925,43.63901722620485,43.63900913652841,43.63900087735963,43.6389845407794,43.63896819304469,43.63894733466338,43.63893098390847,43.63891012663996,43.63888926786222,43.63887290844836,43.63884745854416,43.63883108923027,43.63881021891329,43.63878484896912,43.63876397976777,43.63874309942604,43.63871321996014,43.63869233936045,43.63866694812648,43.6386415581408,43.63861616527598,43.63859076265474,43.63856086453476,43.63853547266435,43.63851006003738,43.63847569916667,43.63845024992359,43.63841589978587,43.63839053889127,43.6383651668664,43.63833071808898,43.63829633580124,43.63826645425993,43.63823207308636,43.6382021912852,43.63817230798013,43.63813341729517,43.63809901361979,43.63806912168501,43.63802561914136,43.63799571596273,43.63795680266082,43.63791787824646,43.63788346419878,43.6378445299237,43.63780560650981,43.63776666100526,43.63772772636193,43.63768428248046,43.63764523701338,43.63760178190962,43.63756282482085,43.63752385937912,43.63747589205889,43.63743241589583,43.6373889382438,43.63734995175271,43.63729746524746,43.6372584661808,43.63724965370437,43.63603198336656,43.63602322254512,43.63605964443909,43.63646870583022,43.63646873731141,43.63650509616056,43.63654599695052,43.6365869168947,43.63662332773357,43.63665976046276,43.63670070364921,43.63673264505566,43.63677360875248,43.63681007018549,43.63684654255669,43.63687862728598,43.63691512153099,43.6369471162476,43.63698363099558,43.63702015667796,43.637052181462,43.6370842171794,43.63720310230798]}]],[[{"lng":[-79.4531869387303,-79.45342866858608,-79.45366266168058,-79.45390000846328,-79.45414595292459,-79.45437034859054,-79.45462193051861,-79.45483267249563,-79.45498478303118,-79.45518690464183,-79.45542527008431,-79.45566291979857,-79.45572837217372,-79.4558071486577,-79.45611731468144,-79.45642667583938,-79.45675238719706,-79.45707800737705,-79.45738159301114,-79.45768811961617,-79.45798023549192,-79.458229865757,-79.45846203321226,-79.4586780167355,-79.45892491850879,-79.45917046964405,-79.45944239894472,-79.4597259905068,-79.46001204970098,-79.46027934129994,-79.46055760491687,-79.46083042575302,-79.46106650797708,-79.4612839940458,-79.46154708147,-79.46173291070427,-79.46191767969991,-79.46215245517897,-79.46234623990733,-79.46254795616993,-79.46283552869643,-79.4631360792489,-79.46344328230965,-79.46373212836154,-79.46401511181003,-79.46426237572773,-79.46450680469799,-79.46475227719809,-79.46499420930562,-79.46515868088349,-79.46527348947235,-79.46551087676137,-79.46574304824517,-79.46598273003336,-79.46625216519,-79.46655241130371,-79.46680604320552,-79.46706672539709,-79.467200532232,-79.46735491257381,-79.4676061628435,-79.46787462100548,-79.46812078465609,-79.46835186940112,-79.46844411138888,-79.46857249223345,-79.46876952605638,-79.46896739605785,-79.46915619239942,-79.46931300989915,-79.46956249167432,-79.46981099045647,-79.4700641765011,-79.47031542862439,-79.4705544461177,-79.47061512966926,-79.47068886861835,-79.47073911206441,-79.47079432663813,-79.4706926768384,-79.47201110865161,-79.47241826260159,-79.47278593508995,-79.47324538601171,-79.47331427859305,-79.47346146272641,-79.47380672478531,-79.47394943945501,-79.47423315289215,-79.47420162131712,-79.47412126533465,-79.47404008944694,-79.47395820039286,-79.47387642508536,-79.4737931168556,-79.47370947850079,-79.47362616999182,-79.47354204385896,-79.47345649871764,-79.47337073505875,-79.47328496179249,-79.47319870095396,-79.47311150854824,-79.47302432777218,-79.47293631759993,-79.47284770594777,-79.4727596955011,-79.47266944859351,-79.47258011896055,-79.47248998578981,-79.4723989210011,-79.47230784425821,-79.47222252535606,-79.4722153604061,-79.47212379612232,-79.47203141422007,-79.47193810065922,-79.47190751099848,-79.47184490100989,-79.46886319026889,-79.46884518069666,-79.46598216777546,-79.46596804071196,-79.46585551900162,-79.46574183607827,-79.46562839432788,-79.46551422475218,-79.46539995282203,-79.46528495303001,-79.46516997778311,-79.46505438923644,-79.46493858377781,-79.46482289044459,-79.46470626712568,-79.46458904289268,-79.46447263605171,-79.46435459398042,-79.46423736930113,-79.4641187262165,-79.46400078589787,-79.46388202788651,-79.46376268150513,-79.46364380624566,-79.46352364206236,-79.46340428281161,-79.46328388899083,-79.46316311127329,-79.46304223124591,-79.46292123637606,-79.46280035610619,-79.46267946316894,-79.46255683051305,-79.46243534928213,-79.46231305040405,-79.46218980650927,-79.46206740292399,-79.46194497632281,-79.46182656037007,-79.46170661141448,-79.46158705850333,-79.46146792689385,-79.4613480798301,-79.4612280282808,-79.46110807882755,-79.4609874265966,-79.46086654452341,-79.46074589211388,-79.46062442207155,-79.4605029392342,-79.46038157121555,-79.46026021584881,-79.46013801508406,-79.4600159442144,-79.45989375832775,-79.45977147016573,-79.45964878592378,-79.45952659980284,-79.45940371103714,-79.45928000465339,-79.45922489626709,-79.45915721795579,-79.45903432900076,-79.45891012428879,-79.45787021556981,-79.45774979159083,-79.45762865220557,-79.4575075257151,-79.45738638625063,-79.45726525969907,-79.45714363511438,-79.45702239334996,-79.45690148409389,-79.45677964208468,-79.45665861760672,-79.45653690376307,-79.45641494417426,-79.4562934347131,-79.45617170783412,-79.45605058106736,-79.45592886722046,-79.45580725557852,-79.45568563087176,-79.45556309948951,-79.4554423054265,-79.4553198631572,-79.45519836685475,-79.45507687058067,-79.45495514368658,-79.45483364745121,-79.45471213809651,-79.45459053971433,-79.45446904355515,-79.45434836498058,-79.45422686887807,-79.45410549049917,-79.45398399446272,-79.45386321381582,-79.45377891661163,-79.45375900547049,-79.45374072946441,-79.45372328425739,-79.45370572365913,-79.45368958057753,-79.45367355293347,-79.45365834286194,-79.45364478113892,-79.45363100182023,-79.45362791317855,-79.45360693335863,-79.45359643751674,-79.45358582626567,-79.45357685012321,-79.45356870477063,-79.45356044400395,-79.45355393376497,-79.45355040993314,-79.45354709048959,-79.45354191833837,-79.45353706600231,-79.4535304763868,-79.45352398896813,-79.45351586646505,-79.4535078352327,-79.45349299447682,-79.4531869387303],"lat":[43.63696017093861,43.63708506052016,43.63720075280045,43.63725347534019,43.63730181263217,43.63734535693106,43.63735326003401,43.63738311567025,43.63740317624045,43.63741040946497,43.63735961857126,43.63730881750795,43.6373277060914,43.63733327152747,43.63729695241177,43.63725612053765,43.63721550860129,43.63720640182157,43.63718349357049,43.63713352790649,43.63704744864489,43.63691578505608,43.6367793843357,43.63669218636644,43.63676312153594,43.63677093330067,43.6367385906885,43.63673791192124,43.63673267463287,43.63667776230709,43.63660949400553,43.63651414544856,43.63640029736791,43.63632211678355,43.6363076542996,43.63622013465152,43.63614610357054,43.63602323355479,43.63586821422539,43.63576290279561,43.6357622698545,43.63576181070567,43.63572093078137,43.63566630013938,43.63562509311819,43.63552939286564,43.63542456187214,43.63531083229623,43.6351835514967,43.63501913156529,43.63489014291802,43.63481681255776,43.63470740330077,43.63458000015925,43.63452060154024,43.63447961987431,43.63440650476709,43.63433348365766,43.63428126568996,43.63417972190296,43.63407956678277,43.63400214703238,43.63389292048878,43.63372938872674,43.63351457449518,43.63331365755091,43.63311825133287,43.63295427329491,43.63278126129539,43.63264382909308,43.63255255802536,43.63240735066924,43.6322667065628,43.63216195438861,43.63208413425959,43.63201292950208,43.6319509015049,43.63192456748809,43.63191180311403,43.63124860849271,43.63236017788363,43.63261767931483,43.63281172704531,43.63300240857262,43.63303483619998,43.63309990688913,43.63323054727925,43.63328646553491,43.63342978595289,43.63345187025717,43.63350481006721,43.63355782889408,43.63361074811929,43.63365916775432,43.63370756686016,43.63376496359512,43.63381336257426,43.633861750572,43.63390561850466,43.63395407452431,43.63400694137898,43.63405079955663,43.6340991462725,43.63414299202741,43.63419132768275,43.63423515418367,43.63428348969815,43.63432729420828,43.63437120092633,43.63441050576479,43.63445429912868,43.63450259330858,43.63453746123653,43.63454186650812,43.63458115192545,43.63462042633815,43.63466418926819,43.63467728342871,43.63470345259594,43.63602322254512,43.63603198336656,43.63724965370437,43.6372584661808,43.63729746524746,43.63734995175271,43.6373889382438,43.63743241589583,43.63747589205889,43.63752385937912,43.63756282482085,43.63760178190962,43.63764523701338,43.63768428248046,43.63772772636193,43.63776666100526,43.63780560650981,43.6378445299237,43.63788346419878,43.63791787824646,43.63795680266082,43.63799571596273,43.63802561914136,43.63806912168501,43.63809901361979,43.63813341729517,43.63817230798013,43.6382021912852,43.63823207308636,43.63826645425993,43.63829633580124,43.63833071808898,43.6383651668664,43.63839053889127,43.63841589978587,43.63845024992359,43.63847569916667,43.63851006003738,43.63853547266435,43.63856086453476,43.63859076265474,43.63861616527598,43.6386415581408,43.63866694812648,43.63869233936045,43.63871321996014,43.63874309942604,43.63876397976777,43.63878484896912,43.63881021891329,43.63883108923027,43.63884745854416,43.63887290844836,43.63888926786222,43.63891012663996,43.63893098390847,43.63894733466338,43.63896819304469,43.6389845407794,43.63900087735963,43.63900913652841,43.63901722620485,43.63903357353925,43.63904540195541,43.63916640635016,43.63917828451773,43.63919015290364,43.63919752029116,43.63920938841893,43.63921675554859,43.63921961478275,43.63923148114331,43.63923434977693,43.63924170673163,43.63924907459609,43.63924743093502,43.63925937697729,43.63925773581716,43.6392605926346,43.63926795847296,43.63926631416395,43.63926467110618,43.63926752878604,43.63926587304054,43.6392642406425,43.63926708688401,43.6392609436933,43.63925480037379,43.63925765589454,43.6392515123169,43.63924986947621,43.63924372425893,43.63923758029459,43.63923144725808,43.63922530303699,43.63921456921465,43.6392084247362,43.63920228980651,43.63919664833879,43.63915586956806,43.6391151129159,43.63906986645729,43.63902911947812,43.63899289271831,43.63895216647422,43.63891145128945,43.6388662573611,43.63882556153043,43.63879851347772,43.63873962602257,43.6386944735701,43.63865382059868,43.63861318975024,43.63856806909821,43.63852744792758,43.63848234939839,43.63846429754162,43.63844624845024,43.63843267533371,43.63841460550034,43.63839651215933,43.63837842020076,43.63836030611702,43.63834660429085,43.63831939721008,43.63696017093861]}]],[[{"lng":[-79.47511682919856,-79.47548784175009,-79.4756909598234,-79.4758261620393,-79.47599802768437,-79.47616359218165,-79.47623928637428,-79.47632831008029,-79.47651540541982,-79.47670081280292,-79.47672320273536,-79.47681872304454,-79.4768605531889,-79.47696376888642,-79.47707439138748,-79.47720347585624,-79.47728092352607,-79.47748818013592,-79.47782793393783,-79.47810693201205,-79.47844368223605,-79.47883599266262,-79.47935022067281,-79.47984055810807,-79.48038676087432,-79.48084822219177,-79.48138910842187,-79.48187209634891,-79.48232037348801,-79.4824676457077,-79.48299990667208,-79.48323289753543,-79.48336521653033,-79.48360491936411,-79.48384168069563,-79.48404138219371,-79.4842390985078,-79.48438526560115,-79.48452235959614,-79.48466349576853,-79.48472390767667,-79.48482277075284,-79.48490949882833,-79.48503418894708,-79.48508888255171,-79.48511531353093,-79.48509231621672,-79.48510231802139,-79.48506040506035,-79.4850144083656,-79.48499474118087,-79.4849937336399,-79.48498543947235,-79.48497879133936,-79.4850532685889,-79.48520537767118,-79.4854286473109,-79.48569719864908,-79.48596386244182,-79.48633270045548,-79.48665240101491,-79.48708096080233,-79.48749384465742,-79.48793352082401,-79.4882009704381,-79.48898511612039,-79.48909445214507,-79.48927762540661,-79.48942542362207,-79.48964701424676,-79.4897871005892,-79.4898908169264,-79.48995755812153,-79.49008738821988,-79.49018340626371,-79.49021158987941,-79.49034909958525,-79.49044691929763,-79.4905804117373,-79.49077522789065,-79.49097142740914,-79.49119022693853,-79.49134791637896,-79.49142469772288,-79.49153908910432,-79.49172630830441,-79.49187469606196,-79.49199067625005,-79.49219584841696,-79.49234797981605,-79.4924223437982,-79.49250697915751,-79.49254097162454,-79.49266040511559,-79.49269335513287,-79.49277482827982,-79.49280042600552,-79.49280706139572,-79.49278292570038,-79.49275914480889,-79.49261102232035,-79.49032937738625,-79.48889018572706,-79.48746355405565,-79.48609651736402,-79.48506717034131,-79.48489678181481,-79.48472958439635,-79.48465792605936,-79.48457073124247,-79.48447197527496,-79.48399720344084,-79.48356261299342,-79.4829570645708,-79.48185974233327,-79.48174886437613,-79.48045860231151,-79.48041406779841,-79.48018294004399,-79.47988066674344,-79.479571814159,-79.4795560050766,-79.47928761273575,-79.47897621724164,-79.47867516347164,-79.47838427141325,-79.47801297603785,-79.47790222451253,-79.47785706559957,-79.47781685533414,-79.47772398841254,-79.47761584776185,-79.47744525095665,-79.47625309393496,-79.47600816183635,-79.47505507536698,-79.47487763186659,-79.47475343742039,-79.47466615852801,-79.4745386487493,-79.47435905576991,-79.47413240364901,-79.47369385036204,-79.47337068984832,-79.47317150878786,-79.47295992102957,-79.47238249964488,-79.47211894150807,-79.47172473947252,-79.47104206540013,-79.47063337764043,-79.47056833399478,-79.4702342229118,-79.46990322952067,-79.46976445893347,-79.46972499492665,-79.46968634849381,-79.46964771412922,-79.46960989734232,-79.46957349924679,-79.46953721541635,-79.46950081968497,-79.46946617099766,-79.46943233988063,-79.46939841865853,-79.46936600625079,-79.46933361797726,-79.46930193303062,-79.46927188317407,-79.4692412322659,-79.46921201206746,-79.46918584556396,-79.46918349518117,-79.46889405712288,-79.46886319026889,-79.47184490100989,-79.47190751099848,-79.47193810065922,-79.47203141422007,-79.47212379612232,-79.4722153604061,-79.47222252535606,-79.47230784425821,-79.4723989210011,-79.47248998578981,-79.47258011896055,-79.47266944859351,-79.4727596955011,-79.47284770594777,-79.47293631759993,-79.47302432777218,-79.47311150854824,-79.47319870095396,-79.47328496179249,-79.47337073505875,-79.47345649871764,-79.47354204385896,-79.47362616999182,-79.47370947850079,-79.4737931168556,-79.47387642508536,-79.47395820039286,-79.47404008944694,-79.47412126533465,-79.47420162131712,-79.47423315289215,-79.47434453242019,-79.47511682919856],"lat":[43.63384676394473,43.634099361169,43.63443073697277,43.63474761353417,43.63503356149076,43.63534643145677,43.63547797106752,43.63563219357128,43.63595885318289,43.63622697635179,43.63626328315272,43.63643559603644,43.63674231477933,43.63698233581902,43.63720436124286,43.637399716277,43.63752677747608,43.63772749466292,43.63786714158586,43.63798788428219,43.63806438469946,43.63811461734141,43.6381259614067,43.63812798330176,43.63810373961119,43.6380873690508,43.63802254047419,43.63797944552032,43.63796289388412,43.6379603483856,43.6379944206948,43.63800201402972,43.63801277206933,43.63803395685527,43.63804610002224,43.63808475742589,43.63814139234199,43.63821534752083,43.6382801801101,43.63836316033836,43.63839996967819,43.63846879607836,43.63856896885616,43.63881826934711,43.63910706131825,43.6396296230798,43.64005250662249,43.64026877942376,43.64061039336906,43.64086634326278,43.6412217551638,43.64138386945434,43.64155930026436,43.64173025184355,43.64191128087428,43.64202582313636,43.64208729570017,43.64210435735636,43.64209888815326,43.64206326426279,43.64201798645928,43.64192463734411,43.64180407289513,43.64163425989796,43.64151176868827,43.64123398151099,43.64119941860704,43.64118833685102,43.64122629875781,43.64135075553367,43.64145171971799,43.6415476121152,43.64163401380461,43.64183836615037,43.64199267016097,43.64204705501499,43.6423775377587,43.64268039981089,43.64306934276289,43.64345909534542,43.64382636062011,43.64418942285367,43.64441664669646,43.64453918837445,43.64470723683607,43.64493935093029,43.64508084143096,43.64518139474917,43.64534172825509,43.6454562612666,43.64552025681304,43.64560689300524,43.64567035596211,43.64582505667428,43.6458975079204,43.64620465328341,43.64645254818048,43.64672278822871,43.64690701265413,43.64711383697698,43.64743154742332,43.64792823336275,43.64823787970537,43.64854758408273,43.64884914797597,43.64907415344094,43.64907189476454,43.64918670516609,43.64923076547267,43.64927461972677,43.6493228216469,43.64943814302268,43.64954940451344,43.64968090041106,43.64989597252283,43.64992600690396,43.65020142680432,43.65006580353431,43.64941899169585,43.64865411602621,43.6479071561041,43.64785293331177,43.64715602223314,43.64640911672367,43.64565784690438,43.6449336277303,43.64407780681315,43.64386028184961,43.6437740705888,43.64371052045268,43.64361476165683,43.64353680350421,43.64342200488876,43.64263625848248,43.64245736199742,43.64183691823571,43.64168601606234,43.6415357342334,43.64137703288705,43.64120429137169,43.64101726131678,43.64084769690425,43.64057618936384,43.64041892770853,43.64029914979193,43.64011628115451,43.63956834930155,43.6393306818726,43.63895173453934,43.63825385384868,43.63787020786693,43.63781082387175,43.63747768591229,43.63720310230798,43.6370842171794,43.637052181462,43.63702015667796,43.63698363099558,43.6369471162476,43.63691512153099,43.63687862728598,43.63684654255669,43.63681007018549,43.63677360875248,43.63673264505566,43.63670070364921,43.63665976046276,43.63662332773357,43.6365869168947,43.63654599695052,43.63650509616056,43.63646873731141,43.63646870583022,43.63605964443909,43.63602322254512,43.63470345259594,43.63467728342871,43.63466418926819,43.63462042633815,43.63458115192545,43.63454186650812,43.63453746123653,43.63450259330858,43.63445429912868,43.63441050576479,43.63437120092633,43.63432729420828,43.63428348969815,43.63423515418367,43.63419132768275,43.63414299202741,43.6340991462725,43.63405079955663,43.63400694137898,43.63395407452431,43.63390561850466,43.633861750572,43.63381336257426,43.63376496359512,43.63370756686016,43.63365916775432,43.63361074811929,43.63355782889408,43.63350481006721,43.63345187025717,43.63342978595289,43.63347178224245,43.63384676394473]}]],[[{"lng":[-79.49955049178516,-79.49792552794734,-79.49805848692486,-79.49619567653971,-79.49510309646428,-79.49506341081648,-79.49503132115075,-79.495002840052,-79.49495645762372,-79.49431589075299,-79.4938002792175,-79.49280217707603,-79.49262287306624,-79.49148460791248,-79.49026126589406,-79.48901852357776,-79.48884095484354,-79.48877512853282,-79.48848706827557,-79.48828662252606,-79.48840753209873,-79.4883098797241,-79.48828945996306,-79.48870741096769,-79.48818436483107,-79.4879201100248,-79.48777428730475,-79.48762457106922,-79.48749387570524,-79.48733533385705,-79.48720982028358,-79.48706722310739,-79.48692260464249,-79.48596847268112,-79.48565200529863,-79.48533614174576,-79.48499074917133,-79.48489678181481,-79.48506717034131,-79.48609651736402,-79.48746355405565,-79.48889018572706,-79.49032937738625,-79.49261102232035,-79.49240817143513,-79.49226032582268,-79.49202621880279,-79.49195990078825,-79.49176930403247,-79.49152573687866,-79.49127589002022,-79.49101270273471,-79.49086721950236,-79.49092136874768,-79.49095407273485,-79.49107444279196,-79.49122747934841,-79.49134718723886,-79.4914848234651,-79.49195110543288,-79.49247072142633,-79.49304693300851,-79.4936386957859,-79.494255339559,-79.49482633166717,-79.49543499155443,-79.49601181825291,-79.49637102170044,-79.49663605161861,-79.49724408524196,-79.49775679165658,-79.4978512412787,-79.49846000887786,-79.49906237553742,-79.49970601606631,-79.50033052274078,-79.50093483743824,-79.50151222449466,-79.50205871101448,-79.50247781631424,-79.50277005721783,-79.50286045621895,-79.50284852264214,-79.50278368544471,-79.50268111111482,-79.50262343116428,-79.5024991483245,-79.50224581901537,-79.50193109795114,-79.50160139485683,-79.50122399017798,-79.50080569907861,-79.50039502014012,-79.50001572348734,-79.49972530613859,-79.49955049178516],"lat":[43.65785031503451,43.65808102775963,43.65846995409255,43.65885064134429,43.65908390672053,43.65909238621353,43.65913247295025,43.65915910409816,43.65937013186957,43.6594472141175,43.65951243653348,43.65959830084893,43.65963644537648,43.65989607970207,43.6601590798554,43.66040821727842,43.66001418974443,43.65979276877811,43.65913171866668,43.65868787627569,43.65865796829004,43.6584090298491,43.65835474735818,43.65825675118494,43.65698027192856,43.65633303811298,43.65596643490187,43.65560428099698,43.65528288792007,43.65489361058793,43.65458587886413,43.65424182253179,43.65388873717199,43.65154852511661,43.65078365687653,43.65002761771348,43.64918566641543,43.64907189476454,43.64907415344094,43.64884914797597,43.64854758408273,43.64823787970537,43.64792823336275,43.64743154742332,43.64785656073656,43.64809766662153,43.64837823355965,43.64845837724795,43.6487080104639,43.64905587708105,43.64937674403922,43.64967492909628,43.65003768236766,43.65044358091689,43.65053403351786,43.65085978761314,43.65124898735318,43.65142160706271,43.65162146951259,43.65190677747867,43.65211626940665,43.65225439859716,43.65229820805653,43.65227932772734,43.65219682831557,43.65209222672161,43.65205030753224,43.6520055224981,43.65197300072187,43.65196299307043,43.65197873544746,43.65197997687937,43.65205540211572,43.65211282591294,43.65213478004372,43.65216548155799,43.65226802145369,43.65239262027007,43.65257982571161,43.65286897344291,43.65321047018212,43.65362583992928,43.65404437013205,43.65443970207671,43.65479403022895,43.65490129942375,43.65512922391448,43.65546807310752,43.65578361184077,43.6560764480828,43.65635056364263,43.65661973059674,43.6568664006869,43.65715858154107,43.65752394443836,43.65785031503451]}]],[[{"lng":[-79.47929263481451,-79.47912271232738,-79.47897589276104,-79.47869019088817,-79.47866611162203,-79.47824920816345,-79.47778911840436,-79.47710981605591,-79.47629350277784,-79.47722211977161,-79.47766853077884,-79.47815119322519,-79.47907250588817,-79.48045860231151,-79.48174886437613,-79.48185974233327,-79.4829570645708,-79.48356261299342,-79.48399720344084,-79.48447197527496,-79.48457073124247,-79.48465792605936,-79.48472958439635,-79.48489678181481,-79.48499074917133,-79.48533614174576,-79.48565200529863,-79.48596847268112,-79.48692260464249,-79.48706722310739,-79.48720982028358,-79.48733533385705,-79.48749387570524,-79.48762457106922,-79.48777428730475,-79.4879201100248,-79.48818436483107,-79.48870741096769,-79.48828945996306,-79.4883098797241,-79.48840753209873,-79.48828662252606,-79.48553213475644,-79.48554524619776,-79.4854556490501,-79.48408364285805,-79.48286558293773,-79.48272745780478,-79.48255610760152,-79.48147923598137,-79.48104227942595,-79.48023631295821,-79.4801245833155,-79.4800049245237,-79.47972755986885,-79.47970219438683,-79.47942264476657,-79.47929263481451],"lat":[43.65820388242512,43.65784594973661,43.65748841427185,43.65683178176515,43.6567729478065,43.65581308833195,43.65475795024233,43.65308765848955,43.65112746936171,43.65090570207921,43.65081712740322,43.65070202747944,43.65049365097062,43.65020142680432,43.64992600690396,43.64989597252283,43.64968090041106,43.64954940451344,43.64943814302268,43.6493228216469,43.64927461972677,43.64923076547267,43.64918670516609,43.64907189476454,43.64918566641543,43.65002761771348,43.65078365687653,43.65154852511661,43.65388873717199,43.65424182253179,43.65458587886413,43.65489361058793,43.65528288792007,43.65560428099698,43.65596643490187,43.65633303811298,43.65698027192856,43.65825675118494,43.65835474735818,43.6584090298491,43.65865796829004,43.65868787627569,43.65929523627512,43.6593269172434,43.65934373405623,43.65964970816039,43.65991711015675,43.65991977752348,43.65992200358245,43.66014184398766,43.66022605796154,43.66040888753999,43.66044349999136,43.66005472994976,43.65936670394518,43.65920883042591,43.65857928794516,43.65820388242512]}]],[[{"lng":[-79.47778911840436,-79.47824920816345,-79.47866611162203,-79.47869019088817,-79.47897589276104,-79.47912271232738,-79.47929263481451,-79.47942264476657,-79.47970219438683,-79.47972755986885,-79.4800049245237,-79.47878865683388,-79.47765626471127,-79.47654761362899,-79.47520871513797,-79.47501832004843,-79.47494579031212,-79.47477482844758,-79.47462732360745,-79.47445992660155,-79.47426865005575,-79.47411882432694,-79.47380782120088,-79.47371506786685,-79.47361594052116,-79.47346335639041,-79.4733832009294,-79.47318771346508,-79.47224089741145,-79.47081254510695,-79.47022166918387,-79.47018472153188,-79.47013933074432,-79.47008310932181,-79.47001005587367,-79.4694261707248,-79.46926957521107,-79.46928047896661,-79.46929843661451,-79.46931273813294,-79.46933729896826,-79.46936457154443,-79.46941913634888,-79.46963959643593,-79.46959540351762,-79.46956182062384,-79.46952059242977,-79.46948359195218,-79.46921963853367,-79.46846521566657,-79.46821207553589,-79.46723321814275,-79.46689115822331,-79.46870203766387,-79.47001933974914,-79.47116841014132,-79.4717787128216,-79.472859937065,-79.47407819546382,-79.47436428222301,-79.4751702387189,-79.47550929426087,-79.47629350277784,-79.47710981605591,-79.47778911840436],"lat":[43.65475795024233,43.65581308833195,43.6567729478065,43.65683178176515,43.65748841427185,43.65784594973661,43.65820388242512,43.65857928794516,43.65920883042591,43.65936670394518,43.66005472994976,43.660331114317,43.66054567859041,43.66082806354214,43.66111177702503,43.66116324900131,43.6611802852957,43.66122760517411,43.66129765300396,43.66138994026197,43.66150441362765,43.66160143598677,43.66179541853859,43.66184369118076,43.66190538170379,43.66196185709433,43.66200129567822,43.66204369450929,43.66223367785876,43.66253423797176,43.66109923419675,43.6609997169873,43.66092259188935,43.6608813300207,43.66082633970162,43.66037733064043,43.66025361551763,43.6601907471448,43.66012797324715,43.66005614831838,43.6599709577031,43.65990380751913,43.65980092458175,43.65947980271037,43.65945220475271,43.65943366089964,43.65940610261422,43.65936509784025,43.65867731603184,43.65688444013583,43.6563272405777,43.65397321943868,43.65319427344895,43.65281336795241,43.65252034894052,43.65225207060484,43.65212061087725,43.65187400345223,43.65160211923057,43.65153392198624,43.65135554432057,43.65127904882457,43.65112746936171,43.65308765848955,43.65475795024233]}]],[[{"lng":[-79.47292068836583,-79.47285991454545,-79.47212444646387,-79.47152942596635,-79.47081254510695,-79.47224089741145,-79.47318771346508,-79.4733832009294,-79.47346335639041,-79.47361594052116,-79.47371506786685,-79.47380782120088,-79.47411882432694,-79.47426865005575,-79.47445992660155,-79.47462732360745,-79.47477482844758,-79.47494579031212,-79.47501832004843,-79.47520871513797,-79.47654761362899,-79.47765626471127,-79.47878865683388,-79.4800049245237,-79.4801245833155,-79.48023631295821,-79.48024788834726,-79.48079588127128,-79.48121097645452,-79.48119082516725,-79.48120950155625,-79.48122985732843,-79.48144623172998,-79.48184534554072,-79.48182897653064,-79.48185257246257,-79.48186894148471,-79.48204485035113,-79.48223359016629,-79.48225515124953,-79.48266395797562,-79.48229434064683,-79.48241322617865,-79.48221536875347,-79.48225797331895,-79.48253400682852,-79.48255901360321,-79.48276715772928,-79.4826312549837,-79.48261806567926,-79.48051793555369,-79.47831944052767,-79.47496660384597,-79.47395844821156,-79.47292068836583],"lat":[43.66734800765162,43.66720766380648,43.66551409013784,43.66415546577402,43.66253423797176,43.66223367785876,43.66204369450929,43.66200129567822,43.66196185709433,43.66190538170379,43.66184369118076,43.66179541853859,43.66160143598677,43.66150441362765,43.66138994026197,43.66129765300396,43.66122760517411,43.6611802852957,43.66116324900131,43.66111177702503,43.66082806354214,43.66054567859041,43.660331114317,43.66005472994976,43.66044349999136,43.66040888753999,43.66044054868097,43.66031280349535,43.66165539662229,43.66165962985379,43.66171389040905,43.66170965989176,43.66222583253093,43.66318103213986,43.66318531572512,43.66323514048515,43.66323085689671,43.66366088119414,43.66411349071461,43.66411827808045,43.66509160740132,43.66517681009085,43.66547545661228,43.66548183117538,43.66557691854926,43.666246916008,43.66630576138553,43.66681272982096,43.66681992757723,43.66681975246245,43.666980982655,43.66710026530732,43.66728530633286,43.66732586127547,43.66734800765162]}]],[[{"lng":[-79.48104227942595,-79.48147923598137,-79.48255610760152,-79.48272745780478,-79.48286558293773,-79.48408364285805,-79.4854556490501,-79.48554524619776,-79.48553213475644,-79.48828662252606,-79.48848706827557,-79.48877512853282,-79.48884095484354,-79.48901852357776,-79.49026126589406,-79.49148460791248,-79.49262287306624,-79.49280217707603,-79.4938002792175,-79.49431589075299,-79.49495645762372,-79.49507344042527,-79.49526375445699,-79.49551351770545,-79.49537063030785,-79.49460878904345,-79.49451656548874,-79.49444083174876,-79.49437405333349,-79.49428578292674,-79.49426885306417,-79.49423741594053,-79.49421682566221,-79.49420454695219,-79.49421159833534,-79.4942178646227,-79.49425578268223,-79.49431384096798,-79.49435205589575,-79.49439771127473,-79.49447097222777,-79.49452189516329,-79.49452775318619,-79.49455344255304,-79.49458069851453,-79.49457920366605,-79.49459051097688,-79.49462513835067,-79.49468596903121,-79.49473175864013,-79.49480159664341,-79.49498253585429,-79.49507161576879,-79.49530695566206,-79.49550412360067,-79.49562375123681,-79.4959524130187,-79.49603109822161,-79.49616272438736,-79.4962778391193,-79.49643556446884,-79.49662552831899,-79.49683865929347,-79.49701420666804,-79.49724026082917,-79.49752006356887,-79.49762704886066,-79.49771671570286,-79.49842283280798,-79.49854258273412,-79.49860409811353,-79.49919842040707,-79.49921690700911,-79.49922415746016,-79.49143344147042,-79.48816600330204,-79.48632307547572,-79.48525936390801,-79.48276715772928,-79.48255901360321,-79.48253400682852,-79.48225797331895,-79.48221536875347,-79.48241322617865,-79.48229434064683,-79.48266395797562,-79.48225515124953,-79.48223359016629,-79.48204485035113,-79.48186894148471,-79.48185257246257,-79.48182897653064,-79.48184534554072,-79.48144623172998,-79.48122985732843,-79.48120950155625,-79.48119082516725,-79.48121097645452,-79.48079588127128,-79.48024788834726,-79.48023631295821,-79.48104227942595],"lat":[43.66022605796154,43.66014184398766,43.65992200358245,43.65991977752348,43.65991711015675,43.65964970816039,43.65934373405623,43.6593269172434,43.65929523627512,43.65868787627569,43.65913171866668,43.65979276877811,43.66001418974443,43.66040821727842,43.6601590798554,43.65989607970207,43.65963644537648,43.65959830084893,43.65951243653348,43.6594472141175,43.65937013186957,43.65940317957773,43.65951371024389,43.65969253869072,43.65976717517822,43.66025693739419,43.66035024426403,43.66043026519068,43.66052840817184,43.66068037036374,43.6607071535054,43.66081926511183,43.66092701858815,43.66102137847382,43.66111158203073,43.66123319244129,43.66132821368078,43.66142350024741,43.66147801608238,43.66154172200692,43.66165071186124,43.66178191257061,43.6618585072641,43.66198046351634,43.66222387820614,43.66234097533379,43.66244914901067,43.66255312882581,43.66268896097788,43.66278417573831,43.66287061512635,43.66301703098368,43.66313973183581,43.66337247310005,43.66357761517604,43.66366479950428,43.66391209084638,43.66402115105005,43.66427503079395,43.66441607737835,43.66452176622791,43.66461428583395,43.66468010365796,43.66472742271123,43.6647754056218,43.66483759779582,43.66486601041255,43.66491219939775,43.66540777020271,43.66549036201599,43.66559919485552,43.66551698041866,43.6658052890152,43.6658549855605,43.66628371587959,43.66649259678859,43.66659882252308,43.66666124653612,43.66681272982096,43.66630576138553,43.666246916008,43.66557691854926,43.66548183117538,43.66547545661228,43.66517681009085,43.66509160740132,43.66411827808045,43.66411349071461,43.66366088119414,43.66323085689671,43.66323514048515,43.66318531572512,43.66318103213986,43.66222583253093,43.66170965989176,43.66171389040905,43.66165962985379,43.66165539662229,43.66031280349535,43.66044054868097,43.66040888753999,43.66022605796154]}]],[[{"lng":[-79.50140480706405,-79.50147066903212,-79.50190395173702,-79.50228163115484,-79.50244842942725,-79.50268515024408,-79.50307296016476,-79.50343702433103,-79.50374537392406,-79.50408417821522,-79.5041928331417,-79.50445765257072,-79.5048488089714,-79.50513504283948,-79.50539452313728,-79.50551320004266,-79.50575866885474,-79.50615687880072,-79.50659240977588,-79.50707878284166,-79.50749045378774,-79.50793257963228,-79.5083652259555,-79.50883311070596,-79.50928120469905,-79.50977413639471,-79.5102384316192,-79.510567883048,-79.51097278814454,-79.51107539702843,-79.51129428770795,-79.51148448772933,-79.51152667767417,-79.51167966653489,-79.51183102003569,-79.51195040958004,-79.51197481736587,-79.51200452362937,-79.51211651686428,-79.51220258451986,-79.51170932899684,-79.51152772142095,-79.51134212553589,-79.51113663820769,-79.51091662166237,-79.51067333039798,-79.51036917133504,-79.51011417595005,-79.50997227910082,-79.50967321623769,-79.5093693465696,-79.50906509160849,-79.50877233426138,-79.50854653311161,-79.50840065508255,-79.50813506070435,-79.50777762766553,-79.50761067772075,-79.50738424171702,-79.50715614880887,-79.50684565134603,-79.50653113806376,-79.5062212906131,-79.50595538799894,-79.50561456092487,-79.5002083846297,-79.50013640642712,-79.50006491979698,-79.49999283934886,-79.49991994097014,-79.49984786052907,-79.49977565816667,-79.49970346339136,-79.49963208878464,-79.49955979399469,-79.49948841937348,-79.49941682052462,-79.49934462794384,-79.49927385686189,-79.49922415746016,-79.49921690700911,-79.49919842040707,-79.49860409811353,-79.49854258273412,-79.49842283280798,-79.49771671570286,-79.49762704886066,-79.49752006356887,-79.49724026082917,-79.49701420666804,-79.49683865929347,-79.49662552831899,-79.49643556446884,-79.4962778391193,-79.49616272438736,-79.49603109822161,-79.4959524130187,-79.49562375123681,-79.49550412360067,-79.49530695566206,-79.49507161576879,-79.49498253585429,-79.49480159664341,-79.49473175864013,-79.49468596903121,-79.49462513835067,-79.49459051097688,-79.49457920366605,-79.49458069851453,-79.49455344255304,-79.49452775318619,-79.49452189516329,-79.49447097222777,-79.49439771127473,-79.49435205589575,-79.49431384096798,-79.49425578268223,-79.4942178646227,-79.49421159833534,-79.49420454695219,-79.49421682566221,-79.49423741594053,-79.49426885306417,-79.49428578292674,-79.49437405333349,-79.49444083174876,-79.49451656548874,-79.49460878904345,-79.49537063030785,-79.49551351770545,-79.49526375445699,-79.49507344042527,-79.49495645762372,-79.495002840052,-79.49503132115075,-79.49506341081648,-79.49510309646428,-79.49619567653971,-79.49805848692486,-79.49792552794734,-79.49955049178516,-79.49951887752411,-79.49943010193428,-79.49950961484748,-79.49976666933043,-79.50013507384992,-79.50060191944577,-79.5010438261163,-79.50140480706405],"lat":[43.65995898344737,43.65999585530732,43.66027609866374,43.66055120069919,43.66067041328989,43.66084014311202,43.66111528521686,43.66140370817293,43.66169140055603,43.66193889141773,43.66199882697279,43.66213291259907,43.66233157489218,43.66254245663124,43.66273939466369,43.66279946002184,43.66290169234148,43.66300601072789,43.66301170270239,43.66290544136606,43.66278028831797,43.66261052106906,43.66247654661295,43.6623071120562,43.66211491227285,43.66199522187957,43.66192925848645,43.6619605586724,43.66204235235963,43.66206169331161,43.66211405614019,43.66219764212486,43.66221610572052,43.66229911671734,43.66238210621422,43.66244667520746,43.66246049609656,43.66248338802989,43.66256586467942,43.66262558858147,43.66301084610809,43.66317952021348,43.66334814207421,43.66354360070491,43.66369376932033,43.66387523144464,43.66407381380588,43.66422802596527,43.66432078772826,43.66447442435252,43.66462799749355,43.66476815172378,43.66488585998971,43.66495943100378,43.6649980364623,43.66508017937871,43.66517453566169,43.66521736600462,43.6652504171559,43.66529253829496,43.66533349237401,43.66538789621945,43.66541085297565,43.66544338543893,43.66546593555498,43.66580030044423,43.66579935584581,43.66580741971085,43.66580647368006,43.66580551686793,43.66580457074575,43.66581262503482,43.66581626836981,43.66581983240589,43.66582338431649,43.66582694826256,43.66583951127699,43.66584306439335,43.66585113715868,43.6658549855605,43.6658052890152,43.66551698041866,43.66559919485552,43.66549036201599,43.66540777020271,43.66491219939775,43.66486601041255,43.66483759779582,43.6647754056218,43.66472742271123,43.66468010365796,43.66461428583395,43.66452176622791,43.66441607737835,43.66427503079395,43.66402115105005,43.66391209084638,43.66366479950428,43.66357761517604,43.66337247310005,43.66313973183581,43.66301703098368,43.66287061512635,43.66278417573831,43.66268896097788,43.66255312882581,43.66244914901067,43.66234097533379,43.66222387820614,43.66198046351634,43.6618585072641,43.66178191257061,43.66165071186124,43.66154172200692,43.66147801608238,43.66142350024741,43.66132821368078,43.66123319244129,43.66111158203073,43.66102137847382,43.66092701858815,43.66081926511183,43.6607071535054,43.66068037036374,43.66052840817184,43.66043026519068,43.66035024426403,43.66025693739419,43.65976717517822,43.65969253869072,43.65951371024389,43.65940317957773,43.65937013186957,43.65915910409816,43.65913247295025,43.65909238621353,43.65908390672053,43.65885064134429,43.65846995409255,43.65808102775963,43.65785031503451,43.65790841342391,43.65829892785753,43.65868264972121,43.65903719496487,43.65933909855941,43.65955236185285,43.65976520612723,43.65995898344737]}]],[[{"lng":[-79.50813506070435,-79.50471516645423,-79.5013109717854,-79.49339577959064,-79.48744425035059,-79.48761831356533,-79.48748429751711,-79.48738100041034,-79.48729002759548,-79.48716450986944,-79.48160976091238,-79.48159240098279,-79.48156349523235,-79.48152291004087,-79.48147680315627,-79.48142165300762,-79.48130350283496,-79.48114045175582,-79.48062965578102,-79.47857379873079,-79.47882981002581,-79.47855892119014,-79.48076788612263,-79.48038771654352,-79.48025040414127,-79.48006577340381,-79.48118934059235,-79.48170620446763,-79.48237356552073,-79.48337218933054,-79.48393047764499,-79.48404199410034,-79.48366030396653,-79.48321683489833,-79.482902451981,-79.48276715772928,-79.48525936390801,-79.48632307547572,-79.48816600330204,-79.49143344147042,-79.49922415746016,-79.49927385686189,-79.49934462794384,-79.49941682052462,-79.49948841937348,-79.49955979399469,-79.49963208878464,-79.49970346339136,-79.49977565816667,-79.49984786052907,-79.49991994097014,-79.49999283934886,-79.50006491979698,-79.50013640642712,-79.5002083846297,-79.50561456092487,-79.50595538799894,-79.5062212906131,-79.50653113806376,-79.50684565134603,-79.50715614880887,-79.50738424171702,-79.50761067772075,-79.50777762766553,-79.50813506070435],"lat":[43.66508017937871,43.66762411761994,43.66897070496156,43.67080243577781,43.6723355491958,43.67275203686511,43.67277726942771,43.67278490432893,43.67277019715437,43.67270552138281,43.67413564706239,43.67408581516808,43.67403141886791,43.67398586941952,43.67394024658866,43.67391250765836,43.67388843254889,43.67386826160649,43.67383446577794,43.67369909091558,43.67363498350107,43.67299655354762,43.6724992300304,43.67155778301564,43.67122729246388,43.67076564276491,43.67051492974297,43.67040026989193,43.67025150808406,43.67003071150416,43.66990758923792,43.66988206237868,43.66896302074026,43.66789021295686,43.66713418874865,43.66681272982096,43.66666124653612,43.66659882252308,43.66649259678859,43.66628371587959,43.6658549855605,43.66585113715868,43.66584306439335,43.66583951127699,43.66582694826256,43.66582338431649,43.66581983240589,43.66581626836981,43.66581262503482,43.66580457074575,43.66580551686793,43.66580647368006,43.66580741971085,43.66579935584581,43.66580030044423,43.66546593555498,43.66544338543893,43.66541085297565,43.66538789621945,43.66533349237401,43.66529253829496,43.6652504171559,43.66521736600462,43.66517453566169,43.66508017937871]}]],[[{"lng":[-79.36134490312664,-79.35683270022189,-79.35673288639025,-79.35608944929439,-79.35443070959595,-79.35941493983509,-79.36006550179343,-79.36073858361942,-79.36102966565109,-79.36107065515341,-79.3611226022395,-79.36115995294841,-79.36134490312664],"lat":[43.64836739829941,43.65010788292163,43.64987239444795,43.64908878434539,43.64775943467788,43.64558003929622,43.64639515053254,43.64720616999458,43.64773264404664,43.64782325242209,43.6479320220265,43.6480135761315,43.64836739829941]}]],[[{"lng":[-79.51220258451986,-79.51239544775189,-79.51257470431779,-79.51268517421857,-79.51279491207416,-79.51285994578311,-79.51294643317596,-79.51303959394866,-79.51307190405899,-79.51306649572277,-79.5129773094211,-79.51278896697283,-79.51251990482774,-79.51215497248666,-79.5117779211842,-79.51144869229859,-79.51126479484714,-79.51133438359541,-79.51147724891477,-79.51150124481293,-79.51162272969675,-79.51172875318355,-79.51179441995417,-79.51183366712964,-79.51185949201164,-79.5119562629073,-79.51208094329895,-79.51223978467466,-79.51234542066324,-79.5124220844705,-79.51243017163199,-79.51240835976863,-79.51229179707792,-79.51203814634599,-79.51164706379608,-79.5112112060467,-79.51071778977717,-79.51020683541306,-79.50985049939797,-79.50968804795393,-79.5092347883446,-79.50882532888535,-79.5085107349457,-79.50826640064059,-79.50806638920673,-79.50788417778662,-79.50767530069243,-79.50745027214826,-79.50743076754915,-79.50723253849812,-79.50715781959993,-79.50692668674411,-79.50654708283044,-79.50610624552262,-79.50575253750812,-79.50551866873158,-79.50545717332911,-79.50553308793263,-79.50562297086724,-79.50569266923547,-79.50595976010214,-79.50639570676938,-79.50690755900796,-79.50743880658024,-79.50793520801324,-79.50841054126847,-79.50843764288882,-79.50888226918786,-79.5092398374674,-79.50952746920348,-79.50976811523243,-79.51000110013513,-79.51010746670538,-79.51022334350091,-79.50788707822251,-79.50768400642863,-79.507476111217,-79.50721789693173,-79.50695787883198,-79.50671955922992,-79.50649394257744,-79.50623060786188,-79.50598834890542,-79.50578874777824,-79.50550832203551,-79.50528737435896,-79.50508882490954,-79.50483957643284,-79.50463686286719,-79.50455519387178,-79.50420886971128,-79.50407333155479,-79.50397337513071,-79.50383170088999,-79.50368615962901,-79.50349522117668,-79.50334626695718,-79.50320048334881,-79.50303269192656,-79.50288505803657,-79.50272208334306,-79.50257892069692,-79.50242387509442,-79.50232231818813,-79.50220318976699,-79.50207229031292,-79.50178455314898,-79.50161964083578,-79.50147247037974,-79.50129798335621,-79.50109042310672,-79.50089146344177,-79.50068756514996,-79.5004993242446,-79.50034432187539,-79.49881317534388,-79.4977305634352,-79.49755334561938,-79.497216147598,-79.49685618384052,-79.49649263816873,-79.49636638965268,-79.4961409782062,-79.49569869121891,-79.49542099540207,-79.49541072847232,-79.49496882393991,-79.49473663189544,-79.4941513452988,-79.49374544455205,-79.49339577959064,-79.5013109717854,-79.50471516645423,-79.50813506070435,-79.50840065508255,-79.50854653311161,-79.50877233426138,-79.50906509160849,-79.5093693465696,-79.50967321623769,-79.50997227910082,-79.51011417595005,-79.51036917133504,-79.51067333039798,-79.51091662166237,-79.51113663820769,-79.51134212553589,-79.51152772142095,-79.51170932899684,-79.51220258451986],"lat":[43.66262558858147,43.66291616516938,43.66331467911566,43.66370329553843,43.66410090436777,43.66451143446502,43.66493124570766,43.6653376405927,43.66575224575651,43.66616185912325,43.66653887522573,43.66685158655338,43.66713615079295,43.66734753989519,43.66758568616073,43.66788755846811,43.66827234153954,43.66865142435903,43.66904946563719,43.66911729357923,43.66947004591039,43.6698856111268,43.67033665892227,43.67060273053669,43.67077869685186,43.67122114744787,43.67166837230118,43.67211163069219,43.67257220037225,43.67305039691294,43.67350969667994,43.67357692813138,43.67393108175524,43.67431495764106,43.67464303094789,43.6749615175715,43.67522064914772,43.67550214508817,43.6756820413444,43.67576103146717,43.67602968227822,43.67638001119148,43.67674507976785,43.67715607476583,43.6775946539561,43.67802896414113,43.67844942274471,43.67886516897801,43.67890092240514,43.67926300594312,43.67934304830002,43.67959217598818,43.67983027103418,43.68000464015354,43.68022956798112,43.68055967533013,43.68091454159995,43.68128920884808,43.68163696254933,43.68195303514159,43.68221767616459,43.68233589991534,43.68239660049413,43.68242604496727,43.68239201779237,43.68235771348777,43.68235806720415,43.68243588559577,43.68262518212503,43.68290799678896,43.68321270313131,43.6835128980752,43.68381585290339,43.68411902161269,43.68455223735258,43.68457668193394,43.68460997517176,43.68464261076962,43.68465721815593,43.68466319646415,43.68466024823231,43.68464771450745,43.6846310447028,43.68461043111115,43.68460226329755,43.68456786663213,43.68453376252671,43.68448549175096,43.68443782933535,43.68441425561677,43.68437371518107,43.68433593299687,43.68430311745487,43.68426525465943,43.68421833902663,43.68416623772044,43.68413728102464,43.684108365661,43.68408816387026,43.68406372488328,43.68403908472008,43.68402820696353,43.68402617534433,43.684024844493,43.68401428123005,43.68400806457456,43.68397728671378,43.68399762973063,43.68399570007162,43.68399791301488,43.68401769597859,43.68403309036778,43.68406192266502,43.68408645902269,43.68411143123907,43.68446950416077,43.68108788817015,43.68057685351481,43.67976205676564,43.67893345644022,43.67801928839235,43.67777007135129,43.67723580504506,43.67619456949444,43.67554258688223,43.67551544558972,43.67449671709711,43.67398504362269,43.67258174618181,43.6716669204374,43.67080243577781,43.66897070496156,43.66762411761994,43.66508017937871,43.6649980364623,43.66495943100378,43.66488585998971,43.66476815172378,43.66462799749355,43.66447442435252,43.66432078772826,43.66422802596527,43.66407381380588,43.66387523144464,43.66369376932033,43.66354360070491,43.66334814207421,43.66317952021348,43.66301084610809,43.66262558858147]}]],[[{"lng":[-79.51327893565173,-79.51154116633218,-79.51111565680537,-79.50948133867057,-79.50917931709742,-79.5091607649172,-79.50912704640515,-79.50560512706244,-79.50636972670773,-79.50364346029056,-79.50333926128469,-79.50312269096293,-79.50294649984897,-79.5017730923565,-79.50171588793602,-79.50165848158197,-79.50165148210395,-79.50151195407206,-79.50143905086306,-79.50109090882245,-79.50093401820392,-79.50088133675091,-79.5005913238402,-79.5003716622014,-79.50011426114686,-79.49881317534388,-79.50034432187539,-79.5004993242446,-79.50068756514996,-79.50089146344177,-79.50109042310672,-79.50129798335621,-79.50147247037974,-79.50161964083578,-79.50178455314898,-79.50207229031292,-79.50220318976699,-79.50232231818813,-79.50242387509442,-79.50257892069692,-79.50272208334306,-79.50288505803657,-79.50303269192656,-79.50320048334881,-79.50334626695718,-79.50349522117668,-79.50368615962901,-79.50383170088999,-79.50397337513071,-79.50407333155479,-79.50420886971128,-79.50455519387178,-79.50463686286719,-79.50483957643284,-79.50508882490954,-79.50528737435896,-79.50550832203551,-79.50578874777824,-79.50598834890542,-79.50623060786188,-79.50649394257744,-79.50671955922992,-79.50695787883198,-79.50721789693173,-79.507476111217,-79.50768400642863,-79.50788707822251,-79.51022334350091,-79.51027480270795,-79.51035421931689,-79.51046455501823,-79.51063727268465,-79.51075499776194,-79.51085549386043,-79.51086921055843,-79.51096459674402,-79.51110170416788,-79.51125442494994,-79.51143199891128,-79.51161002902312,-79.51184581308951,-79.51214070744055,-79.51237321507695,-79.51254220860278,-79.51262621170264,-79.51272571409665,-79.51273293154678,-79.51286347584782,-79.51295099582372,-79.51299984280796,-79.51310252262745,-79.51319168167436,-79.5132638000653,-79.51330329208706,-79.51328950832038,-79.51327893565173],"lat":[43.69337877777022,43.69354970097059,43.69363867935967,43.69398654829348,43.69331195845067,43.6932802093696,43.69326176546119,43.69404863246577,43.69470236230048,43.69533752841463,43.69546857516779,43.69556926186472,43.69572908022054,43.69510588692906,43.69488008660287,43.69465419357257,43.69462709575728,43.69406255032998,43.6937194277499,43.69209863985661,43.69147985366018,43.69129453118863,43.69013828783618,43.68927562248783,43.68846188241579,43.68446950416077,43.68411143123907,43.68408645902269,43.68406192266502,43.68403309036778,43.68401769597859,43.68399791301488,43.68399570007162,43.68399762973063,43.68397728671378,43.68400806457456,43.68401428123005,43.684024844493,43.68402617534433,43.68402820696353,43.68403908472008,43.68406372488328,43.68408816387026,43.684108365661,43.68413728102464,43.68416623772044,43.68421833902663,43.68426525465943,43.68430311745487,43.68433593299687,43.68437371518107,43.68441425561677,43.68443782933535,43.68448549175096,43.68453376252671,43.68456786663213,43.68460226329755,43.68461043111115,43.6846310447028,43.68464771450745,43.68466024823231,43.68466319646415,43.68465721815593,43.68464261076962,43.68460997517176,43.68457668193394,43.68455223735258,43.68411902161269,43.68426372492583,43.68444038976215,43.68465337563791,43.68497078776793,43.68532349114236,43.68564896378975,43.68569415267154,43.68606906978425,43.6864490313674,43.68679759903771,43.68714658022798,43.68747306199674,43.68781829959291,43.68811920610837,43.68846890094358,43.68881776863493,43.68921504109327,43.68958550909979,43.68960810808888,43.69000598599798,43.6903852998622,43.69051196377759,43.69077885944255,43.69115819440314,43.69156881466298,43.69197892040027,43.69237942103971,43.69337877777022]}]],[[{"lng":[-79.50294649984897,-79.50312269096293,-79.50333926128469,-79.50364346029056,-79.50636972670773,-79.50560512706244,-79.50912704640515,-79.5091607649172,-79.50917931709742,-79.50948133867057,-79.51111565680537,-79.51154116633218,-79.51327893565173,-79.5129177069078,-79.51280638865033,-79.51275601922129,-79.51273230049019,-79.51263523297787,-79.5124366681359,-79.51222616203752,-79.51201850682452,-79.51189003424686,-79.5119122206588,-79.51195458157945,-79.51223528719827,-79.51264395713311,-79.51309392800104,-79.51352559835553,-79.513892844259,-79.51428874702823,-79.51464503699731,-79.51503893611294,-79.51546707149153,-79.51568606212241,-79.51586750141531,-79.5163287197812,-79.51677123145747,-79.51718466572133,-79.51759615616832,-79.51799343972282,-79.51880597441372,-79.51913558854628,-79.51947598255013,-79.51974353408444,-79.52003073953992,-79.52030540616416,-79.52068083083311,-79.5207867937889,-79.51987667254963,-79.51619179870897,-79.51531426860912,-79.51379127201479,-79.51211508408414,-79.50832101146774,-79.50702671490963,-79.50603025587236,-79.50497922984557,-79.50487237594004,-79.50444249771247,-79.50382763245243,-79.50363156026462,-79.50344524937702,-79.50323748609753,-79.50288334279945,-79.50480995690695,-79.50823655116915,-79.50534477762662,-79.50236890365773,-79.50218247425192,-79.50414101615925,-79.50294649984897],"lat":[43.69572908022054,43.69556926186472,43.69546857516779,43.69533752841463,43.69470236230048,43.69404863246577,43.69326176546119,43.6932802093696,43.69331195845067,43.69398654829348,43.69363867935967,43.69354970097059,43.69337877777022,43.69396380014702,43.69423250213946,43.69451090919947,43.69481225812217,43.69516666452251,43.69547924064052,43.69577806789523,43.6960815228917,43.696381507583,43.69648982065223,43.69668850656772,43.69694421763948,43.69710266139089,43.69722554297439,43.69724016058831,43.69719092472492,43.69708345712451,43.69703857741415,43.69701218931397,43.69695023653568,43.69691257197351,43.69687892034002,43.69679488892257,43.69680963501674,43.69683750514132,43.69691936064469,43.69694242731578,43.69711499763847,43.69727239352317,43.69746584614621,43.69771695738056,43.69796373147727,43.69824193956246,43.69874650160563,43.69897751453379,43.69927630009482,43.70015149195532,43.70034714110265,43.70071002031153,43.70108438780331,43.70197140644885,43.70227416616806,43.70251320010562,43.70276510452231,43.70278621161497,43.70282559673935,43.70287156052576,43.70227477108762,43.70156549400428,43.70084252258576,43.69954582564965,43.69908476015407,43.69833727349777,43.69691281947667,43.69758979004861,43.69675898168463,43.69634794669961,43.69572908022054]}]],[[{"lng":[-79.53080000371189,-79.5307892190315,-79.53077476838625,-79.53075959934512,-79.53074300551035,-79.53072619944804,-79.53070867719381,-79.53068972253347,-79.53067056326788,-79.53065079013199,-79.53063019849452,-79.53060797202644,-79.53058584570832,-79.53056290089411,-79.530539027636,-79.53051465046545,-79.53048853396142,-79.53046230754107,-79.53043557721833,-79.5304153722075,-79.53040791845149,-79.53038650861541,-79.53036345631621,-79.53033886711729,-79.53031417563585,-79.53028815408882,-79.53026192794616,-79.53023498563263,-79.53020732714977,-79.53017863789461,-79.53014934244572,-79.53011933083218,-79.53008849527434,-79.53005694901904,-79.53002519050635,-79.52999210961499,-79.52995881646387,-79.52992480715156,-79.52988997936501,-79.52985453773441,-79.5298181753181,-79.529780994432,-79.5297439159049,-79.52970601670978,-79.52966719343809,-79.52962776182311,-79.52958761405856,-79.52954715169581,-79.52950607549343,-79.52946418853632,-79.52942157771996,-79.52937896694898,-79.52933553771507,-79.52929129001954,-79.52924621613646,-79.52920125233889,-79.52915547008273,-79.52910886383293,-79.52906215311192,-79.52901493861241,-79.52896669328439,-79.52891844800092,-79.52886948657084,-79.52882052518598,-79.52877074534295,-79.5287202415898,-79.52866964333282,-79.52861750265731,-79.52856597589889,-79.52851394539825,-79.5284609863468,-79.52836875487671,-79.5282764211959,-79.52818347594452,-79.52808950546938,-79.52799573972283,-79.52790176162867,-79.52780727989678,-79.52771248349217,-79.52761768717829,-79.52752217477055,-79.52742584395801,-79.52733043404565,-79.52639711137964,-79.52605909149214,-79.52545194959393,-79.5252932431122,-79.52424284235649,-79.52375001567935,-79.52340566625365,-79.5232995422113,-79.52316060963959,-79.52306368562432,-79.52296472373426,-79.52214235968279,-79.52113535906994,-79.51882478433636,-79.51691818611214,-79.51379127201479,-79.51531426860912,-79.51619179870897,-79.51987667254963,-79.5207867937889,-79.52092925797427,-79.52093069016865,-79.52103242736698,-79.5211698845734,-79.52142372265334,-79.52179557811434,-79.52208887333171,-79.52212647892857,-79.52260521066543,-79.52280370274394,-79.52331843034806,-79.52384711876276,-79.52443228366698,-79.52496116429508,-79.52556700093619,-79.52608201395026,-79.52663926103141,-79.52716695226309,-79.52720397079848,-79.52772547398337,-79.52824534194237,-79.52856018181407,-79.52876914802327,-79.52902147276741,-79.5291609195847,-79.52939582417466,-79.52970484029767,-79.53008611123828,-79.5303727273743,-79.53073115183801,-79.53114732618293,-79.53170002548778,-79.53192813088489,-79.53218087816953,-79.53269668687463,-79.53316034779591,-79.53364788749818,-79.53411890118052,-79.53457012379234,-79.53500548127778,-79.53542670212956,-79.53577277769384,-79.53595148416457,-79.53601062784327,-79.53594030143192,-79.5357133443228,-79.53559028199722,-79.5355311194546,-79.53549540748851,-79.53548686327342,-79.53543280038336,-79.53543800445105,-79.53545787821402,-79.53554604573807,-79.53564852314149,-79.53582701285499,-79.53400976108838,-79.53228817247997,-79.5322893723049,-79.53253945537942,-79.53233676896171,-79.53085465060042,-79.53153192503514,-79.53080000371189],"lat":[43.70928993229683,43.70925378565104,43.70920849985409,43.70916329483899,43.70911357050392,43.70906834443716,43.70902310915964,43.70897785546256,43.70893259913137,43.70888733490391,43.70884206014875,43.7087966743493,43.70875137985296,43.70870607482764,43.70866525885911,43.70861993540576,43.70857458958113,43.70853374333689,43.70848838960648,43.70845662278202,43.70844752493026,43.70840223960668,43.70836134413701,43.70832051891772,43.70827969237752,43.70823884872634,43.70819800243775,43.70815714693222,43.70811628220922,43.70807990522236,43.708039019431,43.70799812442058,43.70796162978463,43.70792071501831,43.70788429851331,43.70784336398886,43.70780692772473,43.70777048223823,43.70773402621253,43.70769756227933,43.70766108648937,43.70762460015806,43.70758811513136,43.70755170955404,43.70751961303552,43.70748309769579,43.70744657312785,43.70741454549845,43.70738250995608,43.70734596296538,43.70731390764532,43.70728185230982,43.7072497864256,43.70721770999173,43.70719012390878,43.70715803822385,43.70712594198715,43.70709824610745,43.707070638884,43.70703852415586,43.7070108971402,43.70698327010467,43.70695563382974,43.70692799753434,43.7069003506812,43.70687719548758,43.70684953805377,43.70682636173972,43.70680319330638,43.70677551736237,43.70675233043927,43.70671963577121,43.70668693971236,43.70665414568241,43.70662142839213,43.70658871366152,43.70656049711981,43.7065277730115,43.70649954577232,43.70647131845563,43.70644308183354,43.70641483458594,43.70638659912419,43.70610441800874,43.70603254385571,43.70586717654282,43.70582011864811,43.70550940305214,43.70533199983259,43.70520143302066,43.70515955275037,43.70509924439674,43.70506198379976,43.70502019577976,43.70467639990405,43.7042086818739,43.70312534358489,43.70220923791739,43.70071002031153,43.70034714110265,43.70015149195532,43.69927630009482,43.69897751453379,43.6992764248428,43.69927644337874,43.69952540539336,43.69971172544014,43.69986363308992,43.70003948176926,43.70015580023531,43.70017879157948,43.70037402370134,43.70045049607957,43.70068030810362,43.70085430563178,43.70104145270739,43.70123291145798,43.70141617860357,43.70164346128153,43.70188469913388,43.70206262931561,43.70207210840551,43.70222286021066,43.70231507551826,43.70240923981049,43.7025559623226,43.70289237474433,43.70324974887001,43.70354542719553,43.70381955352282,43.70402250231601,43.70429633870513,43.7045800996077,43.70477899257595,43.70489411914701,43.70492405566971,43.70496331024445,43.70499243911352,43.70500289231793,43.70492804095766,43.70482155853465,43.70467881235974,43.70454027200239,43.70447365488918,43.70450059452173,43.7045839021204,43.70476929095041,43.70498443817429,43.70531919564046,43.70568679101923,43.7060822111258,43.70627529658098,43.70636979815195,43.7067742855173,43.70684636823841,43.70714819003815,43.70754549803967,43.70788897729956,43.70811190332955,43.70854329716249,43.70896238767151,43.70910193407599,43.70995151358299,43.71043510863763,43.71067271158458,43.70915071660963,43.70928993229683]}]],[[{"lng":[-79.51689025463885,-79.51573538908843,-79.51459486826978,-79.51345801898442,-79.51231807269791,-79.51151993175189,-79.51146066390146,-79.51142035159413,-79.50928327552933,-79.50891610970827,-79.50776551339631,-79.50629454708188,-79.50618651718975,-79.50577842605301,-79.50555616596407,-79.50533466914142,-79.50510485522292,-79.50504722536529,-79.50482172699155,-79.50458753157046,-79.50439548405512,-79.50423887968356,-79.50382763245243,-79.50444249771247,-79.50487237594004,-79.50497922984557,-79.50603025587236,-79.50702671490963,-79.50832101146774,-79.51211508408414,-79.51379127201479,-79.51691818611214,-79.51882478433636,-79.52113535906994,-79.52214235968279,-79.52296472373426,-79.52306368562432,-79.52316060963959,-79.5232995422113,-79.52340566625365,-79.52375001567935,-79.52424284235649,-79.5252932431122,-79.52545194959393,-79.52435007028043,-79.52310231131013,-79.52371029359691,-79.52319192303312,-79.52232699675102,-79.52222462210605,-79.52083419642767,-79.52050106599339,-79.52027852637337,-79.51915759996206,-79.5173301462895,-79.51719019754222,-79.51689025463885],"lat":[43.70762916138774,43.70790241186241,43.70816215405216,43.7084220226094,43.70869084150836,43.70887399151655,43.70888672253776,43.70888619744625,43.70941214314984,43.70948387002588,43.70973891142285,43.71008886615477,43.71012796305082,43.71019464385878,43.70923734618816,43.70840608586518,43.70754320922301,43.70736241541361,43.70663912548923,43.70582129026528,43.70515253906898,43.70446615736472,43.70287156052576,43.70282559673935,43.70278621161497,43.70276510452231,43.70251320010562,43.70227416616806,43.70197140644885,43.70108438780331,43.70071002031153,43.70220923791739,43.70312534358489,43.7042086818739,43.70467639990405,43.70502019577976,43.70506198379976,43.70509924439674,43.70515955275037,43.70520143302066,43.70533199983259,43.70550940305214,43.70582011864811,43.70586717654282,43.70613201590611,43.70643555846944,43.70754642927189,43.70767926270894,43.70787071727379,43.70768026148418,43.70803594619911,43.70726178289644,43.70688072759558,43.70713185716662,43.70750434040499,43.70754753461929,43.70762916138774]}]],[[{"lng":[-79.52332917561074,-79.52320451897559,-79.52252220446626,-79.52009800995906,-79.51939386543773,-79.51779886905091,-79.51544184160645,-79.51476316341504,-79.51222101897667,-79.50857013034069,-79.50665040159184,-79.50606460227287,-79.50596072785851,-79.50577842605301,-79.50618651718975,-79.50629454708188,-79.50776551339631,-79.50891610970827,-79.50928327552933,-79.51142035159413,-79.51146066390146,-79.51151993175189,-79.51231807269791,-79.51345801898442,-79.51459486826978,-79.51573538908843,-79.51689025463885,-79.51719019754222,-79.5173301462895,-79.51915759996206,-79.52027852637337,-79.52050106599339,-79.52083419642767,-79.52222462210605,-79.52232699675102,-79.52319192303312,-79.52371029359691,-79.52310231131013,-79.52435007028043,-79.52545194959393,-79.52605909149214,-79.52639711137964,-79.52733043404565,-79.52742584395801,-79.52752217477055,-79.52761768717829,-79.52771248349217,-79.52780727989678,-79.52790176162867,-79.52799573972283,-79.52808950546938,-79.52818347594452,-79.5282764211959,-79.52836875487671,-79.5284609863468,-79.52851394539825,-79.52856597589889,-79.52861750265731,-79.52866964333282,-79.5287202415898,-79.52877074534295,-79.52882052518598,-79.52886948657084,-79.52891844800092,-79.52896669328439,-79.52901493861241,-79.52906215311192,-79.52910886383293,-79.52915547008273,-79.52920125233889,-79.52924621613646,-79.52929129001954,-79.52933553771507,-79.52937896694898,-79.52942157771996,-79.52946418853632,-79.52950607549343,-79.52954715169581,-79.52958761405856,-79.52962776182311,-79.52966719343809,-79.52970601670978,-79.5297439159049,-79.529780994432,-79.5298181753181,-79.52985453773441,-79.52988997936501,-79.52992480715156,-79.52995881646387,-79.52999210961499,-79.53002519050635,-79.53005694901904,-79.53008849527434,-79.53011933083218,-79.53014934244572,-79.53017863789461,-79.53020732714977,-79.53023498563263,-79.53026192794616,-79.53028815408882,-79.53031417563585,-79.53033886711729,-79.53036345631621,-79.53038650861541,-79.53040791845149,-79.5304153722075,-79.53043557721833,-79.53046230754107,-79.53048853396142,-79.53051465046545,-79.530539027636,-79.53056290089411,-79.53058584570832,-79.53060797202644,-79.53063019849452,-79.53065079013199,-79.53067056326788,-79.53068972253347,-79.53070867719381,-79.53072619944804,-79.53074300551035,-79.53075959934512,-79.53077476838625,-79.5307892190315,-79.53080000371189,-79.53153192503514,-79.53085465060042,-79.53233676896171,-79.53253945537942,-79.5322893723049,-79.53228817247997,-79.53400976108838,-79.53582701285499,-79.53589542671081,-79.53628402382145,-79.53674875613156,-79.53716928564705,-79.5374029887802,-79.53763587624663,-79.53811803354725,-79.538516888651,-79.53889229545491,-79.53911977515172,-79.53931620378779,-79.53944981096355,-79.53956254867445,-79.53958837411997,-79.53938501800157,-79.5391176811991,-79.53875366491522,-79.53839243894495,-79.53816534542071,-79.53811051821803,-79.53793158419839,-79.53787281039996,-79.53784717586807,-79.53784148502363,-79.53783179449746,-79.53617061995688,-79.5322001045803,-79.52993532492658,-79.52964971838739,-79.52824337580326,-79.52683282567578,-79.52669247910828,-79.52655745344158,-79.52655213240635,-79.52641086241549,-79.52626909082475,-79.52612813774144,-79.52598615327727,-79.52584438133316,-79.52570250694691,-79.52555981382331,-79.52541743560087,-79.52527484458612,-79.52513154521461,-79.52498834807724,-79.52484493811463,-79.52470174077888,-79.52332917561074],"lat":[43.71606045221133,43.71563880861881,43.71540484948953,43.71491879165808,43.71471171714737,43.71407422483569,43.71341277039065,43.71322445222586,43.71249364048381,43.71182932548955,43.7114990879213,43.71141392430039,43.71103439267247,43.71019464385878,43.71012796305082,43.71008886615477,43.70973891142285,43.70948387002588,43.70941214314984,43.70888619744625,43.70888672253776,43.70887399151655,43.70869084150836,43.7084220226094,43.70816215405216,43.70790241186241,43.70762916138774,43.70754753461929,43.70750434040499,43.70713185716662,43.70688072759558,43.70726178289644,43.70803594619911,43.70768026148418,43.70787071727379,43.70767926270894,43.70754642927189,43.70643555846944,43.70613201590611,43.70586717654282,43.70603254385571,43.70610441800874,43.70638659912419,43.70641483458594,43.70644308183354,43.70647131845563,43.70649954577232,43.7065277730115,43.70656049711981,43.70658871366152,43.70662142839213,43.70665414568241,43.70668693971236,43.70671963577121,43.70675233043927,43.70677551736237,43.70680319330638,43.70682636173972,43.70684953805377,43.70687719548758,43.7069003506812,43.70692799753434,43.70695563382974,43.70698327010467,43.7070108971402,43.70703852415586,43.707070638884,43.70709824610745,43.70712594198715,43.70715803822385,43.70719012390878,43.70721770999173,43.7072497864256,43.70728185230982,43.70731390764532,43.70734596296538,43.70738250995608,43.70741454549845,43.70744657312785,43.70748309769579,43.70751961303552,43.70755170955404,43.70758811513136,43.70762460015806,43.70766108648937,43.70769756227933,43.70773402621253,43.70777048223823,43.70780692772473,43.70784336398886,43.70788429851331,43.70792071501831,43.70796162978463,43.70799812442058,43.708039019431,43.70807990522236,43.70811628220922,43.70815714693222,43.70819800243775,43.70823884872634,43.70827969237752,43.70832051891772,43.70836134413701,43.70840223960668,43.70844752493026,43.70845662278202,43.70848838960648,43.70853374333689,43.70857458958113,43.70861993540576,43.70866525885911,43.70870607482764,43.70875137985296,43.7087966743493,43.70884206014875,43.70888733490391,43.70893259913137,43.70897785546256,43.70902310915964,43.70906834443716,43.70911357050392,43.70916329483899,43.70920849985409,43.70925378565104,43.70928993229683,43.70915071660963,43.71067271158458,43.71043510863763,43.70995151358299,43.70910193407599,43.70896238767151,43.70854329716249,43.70811190332955,43.70819829884894,43.70832930471733,43.70832625410339,43.70825962178461,43.70825361102293,43.70831060335811,43.70848330917967,43.70873155471222,43.70899300205712,43.70930647924652,43.70965115611303,43.7099950297585,43.71033854651233,43.71070804851612,43.71106552953592,43.71137277133462,43.71160225935966,43.71181368788648,43.71203142213783,43.71208473276992,43.71238860155933,43.71260839849962,43.71270709247081,43.71281063262408,43.71298604754035,43.71337896311126,43.71432302531619,43.71483421364588,43.71489805423582,43.71521311481779,43.71552801379104,43.71555771146402,43.71558747756876,43.71558740896255,43.71561718440268,43.71564236218492,43.71566755034737,43.71569722602681,43.71572240327657,43.71574757902853,43.71577274404058,43.71579341194319,43.71581857791634,43.71583923357512,43.71585989037348,43.71588504523898,43.71590570167508,43.71606045221133]}]],[[{"lng":[-79.50606460227287,-79.50665040159184,-79.50857013034069,-79.51222101897667,-79.51476316341504,-79.51544184160645,-79.51779886905091,-79.51939386543773,-79.52009800995906,-79.52252220446626,-79.52320451897559,-79.52332917561074,-79.52308888459206,-79.52044754593325,-79.5204164296259,-79.51904501191781,-79.51818607153271,-79.51562131474125,-79.51264156829711,-79.50852642962631,-79.50781985661014,-79.50778154063526,-79.50773682712529,-79.50748264994938,-79.50721425019474,-79.5069548038728,-79.50683762759806,-79.5066848995824,-79.50642453340467,-79.50621883861125,-79.50613426111087,-79.50610735463302,-79.50606460227287],"lat":[43.71141392430039,43.7114990879213,43.71182932548955,43.71249364048381,43.71322445222586,43.71341277039065,43.71407422483569,43.71471171714737,43.71491879165808,43.71540484948953,43.71563880861881,43.71606045221133,43.71608750366173,43.71635040269327,43.71635450082334,43.71650336340662,43.71659575187095,43.71686871423478,43.71719914240474,43.71766771382695,43.71774850979237,43.71757247082152,43.71737375331642,43.7164835585443,43.71544914553575,43.71449145561809,43.71405773925146,43.71364606314602,43.71269718169126,43.7119787449477,43.71166698067712,43.71156760704982,43.71141392430039]}]],[[{"lng":[-79.49014492561886,-79.49002467457812,-79.48997966269756,-79.48993566794846,-79.48989383894535,-79.48986270696021,-79.48983300529689,-79.48980715367254,-79.48980410393808,-79.48977675813019,-79.48976050939049,-79.48973669985241,-79.48970886184179,-79.48968679877477,-79.48965808428467,-79.48961664306536,-79.48959383671352,-79.48959488469593,-79.48957746817902,-79.48956607766922,-79.48951229718142,-79.48943140096105,-79.48938250364884,-79.48933149227426,-79.4892804573639,-79.48923522240094,-79.48919043973937,-79.48908360232637,-79.48879419244643,-79.48853900888848,-79.4884720715707,-79.48844123242831,-79.48837824550886,-79.48832617942018,-79.48826675488397,-79.48810469465327,-79.48765684504126,-79.49022093956316,-79.49171891566328,-79.49185997084045,-79.49306400714451,-79.4934371920875,-79.4940859233576,-79.49552716865645,-79.49823350073818,-79.50142101073952,-79.50250153085047,-79.50422849585844,-79.50596072785851,-79.50606460227287,-79.50610735463302,-79.50613426111087,-79.50621883861125,-79.50642453340467,-79.5066848995824,-79.50683762759806,-79.5069548038728,-79.50721425019474,-79.50748264994938,-79.50773682712529,-79.50778154063526,-79.50781985661014,-79.5064690317962,-79.50463474065452,-79.50384023596592,-79.50372928981601,-79.50361803664801,-79.5036080082061,-79.50350688578604,-79.50339573489912,-79.50328529060975,-79.50317402538957,-79.50306287441087,-79.50295243000505,-79.50284198555777,-79.50273154106901,-79.50262088213172,-79.50251125620117,-79.50240070924868,-79.50229076647234,-79.50218102830067,-79.50207129007126,-79.50196226810311,-79.5018524274271,-79.5017434031011,-79.50163509727989,-79.50152596292341,-79.5014176569797,-79.50130923880401,-79.50120102521782,-79.49976750840526,-79.49786341978201,-79.49525720341218,-79.49355649617092,-79.49247946582823,-79.49225259053227,-79.49177481676568,-79.49167316982727,-79.49163652127159,-79.49155585519161,-79.49149794710164,-79.49144901246478,-79.49141622704023,-79.49137081425272,-79.49084534751405,-79.49052112968239,-79.49014492561886],"lat":[43.71235281533151,43.7121035818402,43.71201296721142,43.71188185706072,43.71178678345966,43.7116692562095,43.71155183787013,43.71140746443507,43.71131290325683,43.71119542600916,43.71110069039718,43.71100135386663,43.71087495812333,43.71073054469493,43.71067165226944,43.71050006676528,43.71032413652307,43.71018461953276,43.71010787242544,43.71003570595614,43.7098908732581,43.70974577213133,43.70965510587695,43.70955081864835,43.70945562309508,43.70937400717965,43.70927439324276,43.70911085489912,43.70862083058361,43.70827997110364,43.70816196984359,43.70809404699579,43.70803470090257,43.70794399225951,43.70780808626569,43.70746386659972,43.70636842936565,43.7057991094319,43.70548572590769,43.7054560805012,43.70517930880549,43.70511212315533,43.70494513465173,43.70616156323626,43.70797910475536,43.71005656882432,43.71037689068368,43.71090370912938,43.71103439267247,43.71141392430039,43.71156760704982,43.71166698067712,43.7119787449477,43.71269718169126,43.71364606314602,43.71405773925146,43.71449145561809,43.71544914553575,43.7164835585443,43.71737375331642,43.71757247082152,43.71774850979237,43.71790649213987,43.7181300590284,43.71821868168709,43.71822623107349,43.71823377633191,43.71823364501657,43.71824132282164,43.71824886920284,43.71826092572056,43.71827306139854,43.71828060745448,43.71829266364859,43.71830471973529,43.71831677571456,43.71833332976726,43.71834539625889,43.71835745057537,43.71837401369314,43.71839057938666,43.71840714497388,43.71842371984486,43.71844028387862,43.71845694852932,43.71847353247681,43.71849460644721,43.71851119018663,43.7185322733413,43.71855335907615,43.71879119534653,43.71913085281391,43.71959626506576,43.71989801957112,43.72009086213529,43.7193900358133,43.71811418226483,43.71786528627418,43.71778378473974,43.71753507554001,43.71739928145542,43.7172771087653,43.71716406119985,43.71702393100377,43.71520282356641,43.71383888186945,43.71235281533151]}]],[[{"lng":[-79.50596072785851,-79.50422849585844,-79.50250153085047,-79.50142101073952,-79.49823350073818,-79.49552716865645,-79.4940859233576,-79.49559267824073,-79.49613018292106,-79.49667214344974,-79.4977242028648,-79.49879836043949,-79.50004184529266,-79.5019878760633,-79.50269739665058,-79.50290147062731,-79.50305795318693,-79.50321277940884,-79.50340567267212,-79.50357408824537,-79.50367705524346,-79.50382763245243,-79.50423887968356,-79.50439548405512,-79.50458753157046,-79.50482172699155,-79.50504722536529,-79.50510485522292,-79.50533466914142,-79.50555616596407,-79.50577842605301,-79.50596072785851],"lat":[43.71103439267247,43.71090370912938,43.71037689068368,43.71005656882432,43.70797910475536,43.70616156323626,43.70494513465173,43.70462281391024,43.7045037704778,43.7043578671474,43.70412855789927,43.70389511824904,43.70360978904165,43.70318062352403,43.70303238683158,43.70299905241811,43.7029740962593,43.70295812019507,43.70293814144455,43.70293125483629,43.70291009806662,43.70287156052576,43.70446615736472,43.70515253906898,43.70582129026528,43.70663912548923,43.70736241541361,43.70754320922301,43.70840608586518,43.70923734618816,43.71019464385878,43.71103439267247]}]],[[{"lng":[-79.50382763245243,-79.50367705524346,-79.50357408824537,-79.50340567267212,-79.50321277940884,-79.50305795318693,-79.50290147062731,-79.50269739665058,-79.5019878760633,-79.50004184529266,-79.49879836043949,-79.4977242028648,-79.49667214344974,-79.49613018292106,-79.49559267824073,-79.4940859233576,-79.49356963500428,-79.49353022759965,-79.49349000178645,-79.49344895756623,-79.49340729955429,-79.49336542657267,-79.49332295020453,-79.49327965543416,-79.49323532723383,-79.49319120369402,-79.49314627217146,-79.49310061413232,-79.49305413994531,-79.49300745894062,-79.49296026645433,-79.49291286940603,-79.49286484812812,-79.4928160188921,-79.49276626895359,-79.49271661092041,-79.49266613448526,-79.49261483964931,-79.49256365763206,-79.49251164675401,-79.49245942083964,-79.49240669391408,-79.49235386473174,-79.4923001066187,-79.49224564064137,-79.49219126652743,-79.49010827173994,-79.49006228832884,-79.49001702107861,-79.48997185616696,-79.4899267935938,-79.48988317396714,-79.48983954371451,-79.48979674259199,-79.48975404381213,-79.48971206345409,-79.48967078631928,-79.48962963288253,-79.48958928723016,-79.48954965773645,-79.48951074440035,-79.48947204639892,-79.4894338492574,-79.48939740199847,-79.48936023865838,-79.4893243945899,-79.48930786436013,-79.48928876586602,-79.48925395560119,-79.48921822464969,-79.4891833037027,-79.48914899434236,-79.48911490033731,-79.48908162478752,-79.48904895236305,-79.48901721141526,-79.48898618661663,-79.4889551618507,-79.48892495553451,-79.4888955676667,-79.48886679364195,-79.48883905340615,-79.48881191627379,-79.48878477916992,-79.48875857581643,-79.4887337917107,-79.48870912068098,-79.48868526809343,-79.4886619162901,-79.48863949597796,-79.4886179964052,-79.48859639455654,-79.48857642955414,-79.48855718068397,-79.4885378295353,-79.48852011522717,-79.48849873112695,-79.48847806090058,-79.48845799374071,-79.48843885807969,-79.48842043854928,-79.4884028374487,-79.48838627014875,-79.48837010130265,-79.48835567158581,-79.4883411395874,-79.48832742601063,-79.48831464619175,-79.48830328555532,-79.48829182263339,-79.48828129120503,-79.4882722835121,-79.48826327583062,-79.48825581343803,-79.48824905637358,-79.48824251469765,-79.48823729214438,-79.4879397487037,-79.48792847733681,-79.48791719293317,-79.48790416951618,-79.48789064544951,-79.48787618994635,-79.48786091610771,-79.48784492849445,-79.48782823339172,-79.4878098907917,-79.48779154823343,-79.4877723873541,-79.48775179438529,-79.48773099913564,-79.48770958790166,-79.48768664229402,-79.48766287838141,-79.4876390122291,-79.48761371627143,-79.48758841811011,-79.48756158559385,-79.48753383249293,-79.48750526110469,-79.48747596289549,-79.4874308548162,-79.4873857446007,-79.48733899781664,-79.48729225116358,-79.48724478858965,-79.4871966123608,-79.48714679731906,-79.48709708470993,-79.48704645160764,-79.48699498944403,-79.48694292679578,-79.48688911219318,-79.48683541090236,-79.48654118608194,-79.48732404968463,-79.48772534029298,-79.48980077632646,-79.49383050321515,-79.49763411018127,-79.50005524924065,-79.5017730923565,-79.50294649984897,-79.50414101615925,-79.50218247425192,-79.50236890365773,-79.50534477762662,-79.50823655116915,-79.50480995690695,-79.50288334279945,-79.50323748609753,-79.50344524937702,-79.50363156026462,-79.50382763245243],"lat":[43.70287156052576,43.70291009806662,43.70293125483629,43.70293814144455,43.70295812019507,43.7029740962593,43.70299905241811,43.70303238683158,43.70318062352403,43.70360978904165,43.70389511824904,43.70412855789927,43.7043578671474,43.7045037704778,43.70462281391024,43.70494513465173,43.70447013617235,43.70443360870272,43.70439707043052,43.70436052135497,43.70432396417267,43.70429190514056,43.70425533713822,43.7042187583297,43.70418666687869,43.70415457810888,43.70411797766839,43.70408586862996,43.70405365879108,43.70402153622592,43.70398940689653,43.70395727484981,43.70392963555079,43.70389748457583,43.70386532143679,43.70383766048808,43.7038099887231,43.7037823061408,43.70375012402361,43.70372243194971,43.70369923801533,43.70367153644769,43.70364383350658,43.70362052926558,43.70359280467996,43.70356958228044,43.70252917515413,43.70250606259891,43.70248295948776,43.70245985771086,43.70243675726837,43.70240917487473,43.70238609332466,43.70235852171406,43.70233095143985,43.70230330062403,43.70228025010729,43.70224819920891,43.70222065997266,43.70219313018764,43.70216560985457,43.70213359135357,43.70210608046062,43.70207409168465,43.70204209343125,43.70201461360611,43.70199639108694,43.70198263561544,43.70195066843331,43.70191868906895,43.70189113138281,43.70186367179146,43.7018317140359,43.70179976709099,43.70177232911154,43.70174040243735,43.70170848522315,43.70167656800082,43.70164466159196,43.70161276599718,43.70158087851161,43.70154450368754,43.70151263783301,43.70148077197225,43.70144432743155,43.70141249267445,43.70137615840589,43.70133983495544,43.70130801912403,43.70127171460531,43.70123542225902,43.70119912855584,43.70116285649669,43.70112659390531,43.70109032995775,43.70105408765572,43.70101320579156,43.70097242338517,43.70093614995299,43.70089538783622,43.70085463518822,43.70081389336249,43.70076866420642,43.7007279413199,43.70068724143527,43.70064654019563,43.70060584977978,43.70056058069176,43.70051992139688,43.70047926074758,43.70043411141686,43.70039348324325,43.70035285506881,43.70030774633305,43.70026714792898,43.70022205137138,43.70018147326535,43.69849839272139,43.6984216365101,43.69834947114787,43.69827728277405,43.69820058677001,43.69812837944263,43.69805616128453,43.69798384365893,43.69790710568496,43.69783484691144,43.69776258813473,43.69769031852577,43.69761802996234,43.69754564869756,43.69747334929687,43.69740102958657,43.69732869904178,43.69725636713827,43.69718392628693,43.69711157542014,43.69703920424072,43.69696682087059,43.696894426663,43.69682652383249,43.69674031770409,43.69665420154865,43.69656806371218,43.69648192585726,43.69639577850543,43.69630953166565,43.69622335313001,43.69613717592765,43.69605098651633,43.69596928711044,43.69588298870486,43.69580125810357,43.69571502797103,43.69526094127298,43.69513177511424,43.69505147694078,43.69462423054617,43.69379053289442,43.69298063646738,43.69422798172262,43.69510588692906,43.69572908022054,43.69634794669961,43.69675898168463,43.69758979004861,43.69691281947667,43.69833727349777,43.69908476015407,43.69954582564965,43.70084252258576,43.70156549400428,43.70227477108762,43.70287156052576]}]],[[{"lng":[-79.35851975139897,-79.35782035762831,-79.35749502685303,-79.35736241263416,-79.35725608825483,-79.35711869033116,-79.35696416097777,-79.35651579935043,-79.35521924580006,-79.35315822727759,-79.35286800570809,-79.35242476349815,-79.35218166681285,-79.35172834670693,-79.35082553543702,-79.34964727071362,-79.35683270022189,-79.35851975139897],"lat":[43.65390029403001,43.65428192529838,43.65447529501149,43.65439237057512,43.65437283819598,43.65437536413449,43.654391146679,43.65448381226681,43.65476672789054,43.65525475426087,43.65532267809469,43.65543331543737,43.65547932177113,43.65555830394888,43.65476192169896,43.65372300062611,43.65010788292163,43.65390029403001]}]],[[{"lng":[-79.49763411018127,-79.49797398484498,-79.4930437639637,-79.49001316843541,-79.48897282322606,-79.48706536712399,-79.48827471942658,-79.48952225792776,-79.49065021575581,-79.49120794052595,-79.49216494082668,-79.49292008710817,-79.49374339397022,-79.49433957198642,-79.49479726797891,-79.49594118341078,-79.49685804678086,-79.49791965097978,-79.49881317534388,-79.50011426114686,-79.5003716622014,-79.5005913238402,-79.50088133675091,-79.50093401820392,-79.50109090882245,-79.50143905086306,-79.50151195407206,-79.50165148210395,-79.50165848158197,-79.50171588793602,-79.5017730923565,-79.50005524924065,-79.49763411018127],"lat":[43.69298063646738,43.69281406568302,43.69039468276621,43.68890508495866,43.68839622318321,43.68745710180712,43.68725246590454,43.68698539774817,43.68672114767321,43.68659348190348,43.68634487370886,43.68616120096289,43.68595600654935,43.68577932111842,43.68564572884049,43.685269111225,43.68497510526109,43.68466940133914,43.68446950416077,43.68846188241579,43.68927562248783,43.69013828783618,43.69129453118863,43.69147985366018,43.69209863985661,43.6937194277499,43.69406255032998,43.69462709575728,43.69465419357257,43.69488008660287,43.69510588692906,43.69422798172262,43.69298063646738]}]],[[{"lng":[-79.48654118608194,-79.48609534298666,-79.48525963458506,-79.4843409562711,-79.4839740889859,-79.48329163033262,-79.48286816568844,-79.48371594878157,-79.48658485253432,-79.48676167591331,-79.48706536712399,-79.48897282322606,-79.49001316843541,-79.4930437639637,-79.49797398484498,-79.49763411018127,-79.49383050321515,-79.48980077632646,-79.48772534029298,-79.48732404968463,-79.48654118608194],"lat":[43.69526094127298,43.69460671108889,43.69329907731888,43.69192741275452,43.69128781443143,43.68963138986724,43.68836035509668,43.68816897105517,43.68754075970016,43.68750709310434,43.68745710180712,43.68839622318321,43.68890508495866,43.69039468276621,43.69281406568302,43.69298063646738,43.69379053289442,43.69462423054617,43.69505147694078,43.69513177511424,43.69526094127298]}]],[[{"lng":[-79.48398965766269,-79.48612792746465,-79.4870599824342,-79.48828986786445,-79.48907436504302,-79.48932805374538,-79.49024287116175,-79.49086124019806,-79.49172184439119,-79.4937253010997,-79.49461852251234,-79.49506278782555,-79.49544869887174,-79.49557409960258,-79.49564956280568,-79.49577168042656,-79.49589981132563,-79.49640201539232,-79.49649930230349,-79.49679828115264,-79.49710010542755,-79.49714838581968,-79.49739435957721,-79.4977305634352,-79.49881317534388,-79.49791965097978,-79.49685804678086,-79.49594118341078,-79.49479726797891,-79.49433957198642,-79.49374339397022,-79.49292008710817,-79.49216494082668,-79.49120794052595,-79.49065021575581,-79.48952225792776,-79.48827471942658,-79.48706536712399,-79.48287585826444,-79.48300061551862,-79.48339896534294,-79.48338939608513,-79.48340316735884,-79.48398965766269],"lat":[43.68422440453474,43.68372153992473,43.6834997400115,43.68322795033937,43.68307619823467,43.6830120367203,43.68280357598685,43.68266311984974,43.68245393133439,43.68199415505102,43.68178078352162,43.68169211332252,43.68144954858553,43.68139718722411,43.68135767144301,43.68130976766363,43.68127094492827,43.6811560259271,43.68114380280989,43.68112523065524,43.68111560715058,43.68111174103909,43.68110597333674,43.68108788817015,43.68446950416077,43.68466940133914,43.68497510526109,43.685269111225,43.68564572884049,43.68577932111842,43.68595600654935,43.68616120096289,43.68634487370886,43.68659348190348,43.68672114767321,43.68698539774817,43.68725246590454,43.68745710180712,43.68534414941019,43.68522877919165,43.68486489316269,43.68484676212056,43.68481993879621,43.68422440453474]}]],[[{"lng":[-79.48872530947224,-79.48761831356533,-79.48744425035059,-79.49339577959064,-79.49374544455205,-79.4941513452988,-79.49473663189544,-79.49496882393991,-79.49541072847232,-79.49542099540207,-79.49569869121891,-79.4961409782062,-79.49636638965268,-79.49649263816873,-79.49685618384052,-79.497216147598,-79.49755334561938,-79.4977305634352,-79.49739435957721,-79.49714838581968,-79.49710010542755,-79.49679828115264,-79.49649930230349,-79.49640201539232,-79.49589981132563,-79.49577168042656,-79.49564956280568,-79.49557409960258,-79.49544869887174,-79.49506278782555,-79.49461852251234,-79.4937253010997,-79.49172184439119,-79.49086124019806,-79.49024287116175,-79.48932805374538,-79.48907436504302,-79.48828986786445,-79.4870599824342,-79.48612792746465,-79.48398965766269,-79.48340316735884,-79.48338939608513,-79.48339896534294,-79.48300061551862,-79.48287585826444,-79.48069510579327,-79.47969438882107,-79.47979494444193,-79.47983225073646,-79.4799328196124,-79.47998265948912,-79.47999522737257,-79.48010229279274,-79.48022067786175,-79.48035533231464,-79.48046967976461,-79.48061465258613,-79.48194941450626,-79.48272010196371,-79.48281126905111,-79.48289816515707,-79.48296787828976,-79.48300426438763,-79.48302692912824,-79.48305052757132,-79.48307228969142,-79.48309381381112,-79.48314275283643,-79.4831901065233,-79.48326245623657,-79.48382007923801,-79.4839285712147,-79.48413391673967,-79.48424327042157,-79.48432977809436,-79.48445692740275,-79.48473613257336,-79.4852129414195,-79.48537239438451,-79.48555587562296,-79.48572074547458,-79.4859112928566,-79.48644722304131,-79.4866256112079,-79.48679097115883,-79.4869908112444,-79.4871694727792,-79.48737002803644,-79.48753795191804,-79.48772804395942,-79.48789798023613,-79.4880859351638,-79.48825671054776,-79.48844313336592,-79.48862697583766,-79.48872530947224],"lat":[43.67538681544384,43.67275203686511,43.6723355491958,43.67080243577781,43.6716669204374,43.67258174618181,43.67398504362269,43.67449671709711,43.67551544558972,43.67554258688223,43.67619456949444,43.67723580504506,43.67777007135129,43.67801928839235,43.67893345644022,43.67976205676564,43.68057685351481,43.68108788817015,43.68110597333674,43.68111174103909,43.68111560715058,43.68112523065524,43.68114380280989,43.6811560259271,43.68127094492827,43.68130976766363,43.68135767144301,43.68139718722411,43.68144954858553,43.68169211332252,43.68178078352162,43.68199415505102,43.68245393133439,43.68266311984974,43.68280357598685,43.6830120367203,43.68307619823467,43.68322795033937,43.6834997400115,43.68372153992473,43.68422440453474,43.68481993879621,43.68484676212056,43.68486489316269,43.68522877919165,43.68534414941019,43.68424375759714,43.68373974895134,43.68365106633119,43.68362005548398,43.68344576363738,43.68331139613169,43.68327555517977,43.68311944365498,43.68297689558069,43.68283015272441,43.68266513564352,43.68248243153143,43.68059587977717,43.67948959224904,43.67934677004675,43.67919479901662,43.67904268984626,43.67888563723537,43.6788184228236,43.67874662976228,43.67858938300879,43.67844563614088,43.67800959695771,43.67788410693502,43.67778154374262,43.67712720323551,43.67700702499569,43.67679820100003,43.67670062899856,43.67662066795815,43.67656384093054,43.67646852075742,43.676317305968,43.67626531708184,43.67621373640723,43.6761619088356,43.67610592019571,43.67594648144802,43.6758858295587,43.6758429189466,43.6757915525622,43.67574440649989,43.67569304891737,43.67564576015672,43.67560776653183,43.67557400693035,43.67553148341229,43.67548873233635,43.67544609794225,43.67541252118036,43.67538681544384]}]],[[{"lng":[-79.48307228969142,-79.48305052757132,-79.48302692912824,-79.48300426438763,-79.48296787828976,-79.48289816515707,-79.48281126905111,-79.48272010196371,-79.48194941450626,-79.48061465258613,-79.48046967976461,-79.48035533231464,-79.48022067786175,-79.48010229279274,-79.47999522737257,-79.47998265948912,-79.4799328196124,-79.47983225073646,-79.47979494444193,-79.47969438882107,-79.47794775477203,-79.47749229016682,-79.47717651568284,-79.4770206849508,-79.47656569785792,-79.47610235015321,-79.47605283187227,-79.47557403745833,-79.47514308501941,-79.47467147940982,-79.47424855919326,-79.47382587908236,-79.47337111519379,-79.4727613142269,-79.47240910539077,-79.47262562045937,-79.47344089874197,-79.47381557508309,-79.47440602431568,-79.47599171612144,-79.47582797586428,-79.47578479210239,-79.47366639498148,-79.47364654131658,-79.47361194520363,-79.47364647833265,-79.47386721562582,-79.47349032428656,-79.4741819741095,-79.47503254699465,-79.47835575965397,-79.47823215421809,-79.47857379873079,-79.48062965578102,-79.48114045175582,-79.48130350283496,-79.48142165300762,-79.48147680315627,-79.48152291004087,-79.48156349523235,-79.48159240098279,-79.48160976091238,-79.48716450986944,-79.48729002759548,-79.48738100041034,-79.48748429751711,-79.48761831356533,-79.48872530947224,-79.48862697583766,-79.48844313336592,-79.48825671054776,-79.4880859351638,-79.48789798023613,-79.48772804395942,-79.48753795191804,-79.48737002803644,-79.4871694727792,-79.4869908112444,-79.48679097115883,-79.4866256112079,-79.48644722304131,-79.4859112928566,-79.48572074547458,-79.48555587562296,-79.48537239438451,-79.4852129414195,-79.48473613257336,-79.48445692740275,-79.48432977809436,-79.48424327042157,-79.48413391673967,-79.4839285712147,-79.48382007923801,-79.48326245623657,-79.4831901065233,-79.48314275283643,-79.48309381381112,-79.48307228969142],"lat":[43.67858938300879,43.67874662976228,43.6788184228236,43.67888563723537,43.67904268984626,43.67919479901662,43.67934677004675,43.67948959224904,43.68059587977717,43.68248243153143,43.68266513564352,43.68283015272441,43.68297689558069,43.68311944365498,43.68327555517977,43.68331139613169,43.68344576363738,43.68362005548398,43.68365106633119,43.68373974895134,43.68285662645934,43.68259400174593,43.6824186659202,43.68232656908085,43.68204135193213,43.68174710947116,43.68171494206091,43.6814249024616,43.68114450123393,43.68084564174506,43.68054734027141,43.68027604652174,43.67995499149033,43.67948305930243,43.679212792032,43.6792066838881,43.67901044025358,43.67937111474005,43.67923046477455,43.67885985305033,43.67847049193965,43.6783619816137,43.67671748329489,43.67664520184652,43.67654112634592,43.67628052851374,43.67567565888336,43.67519342727896,43.67497311168248,43.67478632972412,43.67406986275845,43.67378006160992,43.67369909091558,43.67383446577794,43.67386826160649,43.67388843254889,43.67391250765836,43.67394024658866,43.67398586941952,43.67403141886791,43.67408581516808,43.67413564706239,43.67270552138281,43.67277019715437,43.67278490432893,43.67277726942771,43.67275203686511,43.67538681544384,43.67541252118036,43.67544609794225,43.67548873233635,43.67553148341229,43.67557400693035,43.67560776653183,43.67564576015672,43.67569304891737,43.67574440649989,43.6757915525622,43.6758429189466,43.6758858295587,43.67594648144802,43.67610592019571,43.6761619088356,43.67621373640723,43.67626531708184,43.676317305968,43.67646852075742,43.67656384093054,43.67662066795815,43.67670062899856,43.67679820100003,43.67700702499569,43.67712720323551,43.67778154374262,43.67788410693502,43.67800959695771,43.67844563614088,43.67858938300879]}]],[[{"lng":[-79.46564251111076,-79.46594803682676,-79.4676135632419,-79.48366030396653,-79.48404199410034,-79.48393047764499,-79.48337218933054,-79.48237356552073,-79.48170620446763,-79.48118934059235,-79.48006577340381,-79.48025040414127,-79.48038771654352,-79.48076788612263,-79.47855892119014,-79.47882981002581,-79.47857379873079,-79.47823215421809,-79.47835575965397,-79.47503254699465,-79.4741819741095,-79.47349032428656,-79.47386721562582,-79.47364647833265,-79.47361194520363,-79.47364654131658,-79.47366639498148,-79.47578479210239,-79.47582797586428,-79.47599171612144,-79.47440602431568,-79.47381557508309,-79.47344089874197,-79.47262562045937,-79.47240910539077,-79.47214376048504,-79.47172082101743,-79.4713309068823,-79.47095677517343,-79.47057495959338,-79.47020907419905,-79.46979427306306,-79.46951289853638,-79.46942872893497,-79.46907087022153,-79.4686973019159,-79.46828198115837,-79.46613330707855,-79.46566464468556,-79.46564251111076],"lat":[43.67292325296386,43.67279502449871,43.67252624295574,43.66896302074026,43.66988206237868,43.66990758923792,43.67003071150416,43.67025150808406,43.67040026989193,43.67051492974297,43.67076564276491,43.67122729246388,43.67155778301564,43.6724992300304,43.67299655354762,43.67363498350107,43.67369909091558,43.67378006160992,43.67406986275845,43.67478632972412,43.67497311168248,43.67519342727896,43.67567565888336,43.67628052851374,43.67654112634592,43.67664520184652,43.67671748329489,43.6783619816137,43.67847049193965,43.67885985305033,43.67923046477455,43.67937111474005,43.67901044025358,43.6792066838881,43.679212792032,43.67900660919285,43.67868588298584,43.67837000795416,43.67808134884137,43.67776107854518,43.67745011225739,43.67711589473051,43.67686898194611,43.67679583843509,43.67647138289374,43.67613330269197,43.67569554924486,43.67342027236676,43.67294578516162,43.67292325296386]}]],[[{"lng":[-79.48366030396653,-79.4676135632419,-79.46594803682676,-79.46564251111076,-79.46448039060201,-79.46174274679139,-79.46090620104448,-79.46003491659883,-79.46115619529932,-79.46296661448244,-79.46299074370783,-79.46377494375268,-79.46535357167703,-79.46614115599833,-79.46933361992103,-79.47181895288178,-79.47287416775356,-79.47292068836583,-79.47395844821156,-79.47496660384597,-79.47831944052767,-79.48051793555369,-79.48261806567926,-79.4826312549837,-79.48276715772928,-79.482902451981,-79.48321683489833,-79.48366030396653],"lat":[43.66896302074026,43.67252624295574,43.67279502449871,43.67292325296386,43.67174143027935,43.66895395034078,43.66810531720957,43.66722020158632,43.66723980610503,43.66725966798791,43.66725999250242,43.66727053645231,43.66729165552348,43.66726171945703,43.66729101891109,43.66731977788611,43.66734738603417,43.66734800765162,43.66732586127547,43.66728530633286,43.66710026530732,43.666980982655,43.66681975246245,43.66681992757723,43.66681272982096,43.66713418874865,43.66789021295686,43.66896302074026]}]],[[{"lng":[-79.46566464468556,-79.46304004292399,-79.4625067456835,-79.46152648301752,-79.46121872572694,-79.46000725937589,-79.45895842985324,-79.45870896541253,-79.45763692710423,-79.45711504883376,-79.45599070646225,-79.45595560657415,-79.45588003765525,-79.45584315271589,-79.45580566745011,-79.45576818224581,-79.45572904786968,-79.45568981129732,-79.45564913005664,-79.45560847519759,-79.45556617116544,-79.45552385633594,-79.45548095207357,-79.45543791927638,-79.45539326364747,-79.45534789230622,-79.45530250784789,-79.45525610091049,-79.45520919595843,-79.45516145985009,-79.45511290577595,-79.45506353604689,-79.45501416407613,-79.45496417865091,-79.45491315523519,-79.45486224966191,-79.45480981035678,-79.45475737112334,-79.45470399845304,-79.45461561917031,-79.45452735555817,-79.45443816091927,-79.45434826163276,-79.45425823376216,-79.45416690330948,-79.4540753552718,-79.45398379413453,-79.4538915306881,-79.45379845171846,-79.45370443705205,-79.45361062706311,-79.45351589697987,-79.4534539499471,-79.45339384353166,-79.45333363494709,-79.45327402662134,-79.45321533863294,-79.45315688421606,-79.45309974349811,-79.45304200270114,-79.452985782406,-79.45293006008501,-79.45287385330251,-79.45281916700397,-79.45276520990807,-79.45271135514477,-79.45265831846955,-79.45260588430916,-79.45255438155012,-79.45250289225893,-79.45245282116626,-79.45240295465005,-79.45235359946082,-79.45230456450123,-79.45225672980897,-79.45220933100485,-79.45216316159234,-79.45211707877483,-79.45207182744662,-79.45202739419102,-79.45198317893562,-79.45193934589621,-79.45189704666505,-79.45185508111929,-79.45181453369912,-79.45177378416102,-79.45136835702425,-79.450505771506,-79.45044956162005,-79.45039263846895,-79.45033644241435,-79.45028175309884,-79.45022799767924,-79.45017425595286,-79.45012111659783,-79.45006982904216,-79.45001832356432,-79.44996681821644,-79.44991766463222,-79.44986770679256,-79.44982020527439,-79.44977270156602,-79.44972611818336,-79.44968004614564,-79.4496347785874,-79.44958962930251,-79.44954601147802,-79.4495030072312,-79.44946020758536,-79.44941905755813,-79.44937779178854,-79.44933795753167,-79.44929894132378,-79.44926002746165,-79.44922266096712,-79.44918610120686,-79.4491500504375,-79.44911421787951,-79.44900398986871,-79.44911126878421,-79.44922151268742,-79.44933197466477,-79.44944302283803,-79.44955316442591,-79.4496635104728,-79.44977928888943,-79.44997820335587,-79.45007679150022,-79.45017313030426,-79.45026853536575,-79.45036477189069,-79.45046039494316,-79.45047900293088,-79.45055733359195,-79.45065295660702,-79.45075001329825,-79.45084612047354,-79.4509424591276,-79.45103938423486,-79.4511357228586,-79.45123254572277,-79.45132876860896,-79.45142570717894,-79.45152181435645,-79.45161863723224,-79.45171556235624,-79.45177516926594,-79.45181166955321,-79.45190837678459,-79.45200530193542,-79.45210212485109,-79.45219279986803,-79.45219811643877,-79.45308652190592,-79.45495487645576,-79.45641807460156,-79.45806631931644,-79.45854839112233,-79.45946050677183,-79.46003491659883,-79.46090620104448,-79.46174274679139,-79.46448039060201,-79.46564251111076,-79.46566464468556],"lat":[43.67294578516162,43.67352733780321,43.67364619298689,43.6738716450533,43.67393051591334,43.67418875975419,43.6744267688206,43.67448191791502,43.67471510079524,43.67483858588116,43.67508455018905,43.67509307800643,43.67498844357405,43.674942935029,43.67489291734266,43.67484289964452,43.67479736066669,43.6747518202942,43.67471076140336,43.67466070081902,43.67461511895054,43.6745739479176,43.67452385688757,43.67448276613953,43.67443715242376,43.67439152901675,43.67435040643204,43.67430927000849,43.67426362581624,43.6742224713872,43.6741813058807,43.67414003930668,43.67409886270116,43.67405767778065,43.67402106984311,43.67397987243745,43.67393865427184,43.67389743608296,43.67386070626642,43.67380099796689,43.67373679014458,43.67367698066046,43.67361275057824,43.67355752072239,43.67349327113946,43.67343351955823,43.67337826874453,43.67331850733735,43.6732586448013,43.67320336056772,43.67314807902602,43.67309278495632,43.67305143750578,43.67301011493069,43.67296879094109,43.67293197606225,43.67289517360858,43.67284928222819,43.67280799963854,43.67276220787862,43.67272093769056,43.67268417523464,43.6726384041552,43.67259715465121,43.67255141397431,43.67250567465715,43.67245994639086,43.67241863725634,43.67237292971048,43.67232272130444,43.67227703309824,43.67223134764015,43.67218566908451,43.67213549382487,43.67209433683751,43.67204418369485,43.67199395716248,43.6719483228237,43.67189819870894,43.67184808565672,43.67179347452041,43.67174787058288,43.67170228740546,43.67164770669341,43.6715976461962,43.67154749292578,43.67103329455463,43.66994117773149,43.66986389825801,43.66978651906432,43.66970473870037,43.66963198078127,43.66955473447872,43.66947298731486,43.66939565929451,43.66931394536233,43.66923672947171,43.66915951355883,43.66908232951988,43.66900063352895,43.66891888081302,43.66883721806717,43.66875556778493,43.66867392441939,43.66859679296936,43.66851507206312,43.66843346193529,43.66835186011432,43.66827026105214,43.66818418333281,43.66810260504907,43.66802104617292,43.66793949838188,43.66785795196531,43.66777192550943,43.66769032098914,43.66760881338373,43.66752280770495,43.66725566147511,43.66724811530085,43.66724060925908,43.66722860504735,43.66722561072692,43.66721810297677,43.66721059789397,43.66719866536989,43.66718335929632,43.667175694343,43.66716799880101,43.66716479153852,43.66715709444787,43.66714938895826,43.66714964125519,43.66714620224779,43.66713849659642,43.66713072027952,43.66713202307507,43.6671243268819,43.66712564060141,43.66711794424475,43.66711925641467,43.66711605934946,43.66711287190081,43.66711417412787,43.66711548597068,43.66711679911666,43.66711760663293,43.6671181011001,43.66712391215355,43.66712522505406,43.66712653648763,43.66713226559843,43.66713233760297,43.66713536413032,43.66716513921067,43.66718041017835,43.66720266023016,43.66720916333836,43.66721696111229,43.66722020158632,43.66810531720957,43.66895395034078,43.67174143027935,43.67292325296386,43.67294578516162]}]],[[{"lng":[-79.47240910539077,-79.47176908243543,-79.47055601703505,-79.47046702522023,-79.47037432646707,-79.46832178040417,-79.46607722130317,-79.46599877197021,-79.4653186587398,-79.46546614979388,-79.46485493136382,-79.46377422574437,-79.46269595162472,-79.46164570320484,-79.4610975033025,-79.46113019065517,-79.46042909825437,-79.46019985752376,-79.45951514671103,-79.45884307493247,-79.45795607838791,-79.45743911913382,-79.45721934594579,-79.45718776314359,-79.45715605201899,-79.45712610543643,-79.45709604357805,-79.45706669756568,-79.45703828497653,-79.45701058823447,-79.45698439927425,-79.4569575206677,-79.4569327787576,-79.45690825215756,-79.45689563750211,-79.45688444139692,-79.45686225372248,-79.45683997689194,-79.45681851815417,-79.45679787750748,-79.45677806803013,-79.45675906356189,-79.45673101052398,-79.45670082552461,-79.45666970487657,-79.45663808609147,-79.45660543170783,-79.45657207468173,-79.45653810418271,-79.45650391614087,-79.45648763595587,-79.45646831197922,-79.45643187442671,-79.45639473425558,-79.4563573634293,-79.45631928998765,-79.45627978503229,-79.45624017789707,-79.45619915468249,-79.45615871653348,-79.45611593969977,-79.45607336746028,-79.4560299772583,-79.45598555144915,-79.45595560657415,-79.45599070646225,-79.45711504883376,-79.45763692710423,-79.45870896541253,-79.45895842985324,-79.46000725937589,-79.46121872572694,-79.46152648301752,-79.4625067456835,-79.46304004292399,-79.46566464468556,-79.46613330707855,-79.46828198115837,-79.4686973019159,-79.46907087022153,-79.46942872893497,-79.46951289853638,-79.46979427306306,-79.47020907419905,-79.47057495959338,-79.47095677517343,-79.4713309068823,-79.47172082101743,-79.47214376048504,-79.47240910539077],"lat":[43.679212792032,43.67922674031978,43.67948507435236,43.67926333357637,43.67928009694869,43.67971179975491,43.68019039970216,43.68019384773056,43.68035134420979,43.68071349591157,43.68084481856759,43.68107335122539,43.68130649738063,43.68153550958046,43.68165415732493,43.68174011672932,43.68189271048481,43.68155204603474,43.68052090835561,43.67951244194652,43.67814540139957,43.67740908022461,43.67709545345543,43.67705451794999,43.67702258273542,43.67698166930182,43.67694525532001,43.67690885099474,43.67686795824791,43.67682707515877,43.67679521445283,43.67675434239666,43.67671791017992,43.67667706986882,43.67665439445889,43.67663623921791,43.67659993149434,43.67655912154601,43.67651832263989,43.6764775347765,43.67643225711635,43.67639149133951,43.67633259930246,43.67626908743076,43.67621015395208,43.67614671272054,43.67608775851068,43.67602429378483,43.67596082076339,43.67590184581003,43.67587452980914,43.67583826068235,43.6757792553241,43.67571573944662,43.67566122247506,43.67560219498326,43.67554314814165,43.67548409990566,43.67542044148879,43.67536588301501,43.6753067919154,43.6752477035633,43.67518860414422,43.67513399173407,43.67509307800643,43.67508455018905,43.67483858588116,43.67471510079524,43.67448191791502,43.6744267688206,43.67418875975419,43.67393051591334,43.6738716450533,43.67364619298689,43.67352733780321,43.67294578516162,43.67342027236676,43.67569554924486,43.67613330269197,43.67647138289374,43.67679583843509,43.67686898194611,43.67711589473051,43.67745011225739,43.67776107854518,43.67808134884137,43.67837000795416,43.67868588298584,43.67900660919285,43.679212792032]}]],[[{"lng":[-79.47240910539077,-79.4727613142269,-79.47337111519379,-79.47382587908236,-79.47424855919326,-79.47467147940982,-79.47514308501941,-79.47557403745833,-79.47605283187227,-79.47610235015321,-79.47656569785792,-79.4770206849508,-79.47717651568284,-79.47749229016682,-79.47794775477203,-79.47969438882107,-79.48069510579327,-79.48287585826444,-79.48706536712399,-79.48676167591331,-79.48658485253432,-79.48371594878157,-79.48286816568844,-79.48102831566352,-79.47888731290745,-79.47717094336414,-79.47608811801626,-79.47505204076248,-79.47464628487874,-79.47434844578251,-79.47426676632355,-79.47392370828004,-79.47357218968229,-79.47354655082859,-79.47320220576823,-79.47280125594737,-79.47240890881592,-79.47229125163919,-79.47205735772548,-79.47193915500154,-79.47158769736463,-79.47088933500955,-79.47054780238706,-79.47037432646707,-79.47046702522023,-79.47055601703505,-79.47176908243543,-79.47240910539077],"lat":[43.679212792032,43.67948305930243,43.67995499149033,43.68027604652174,43.68054734027141,43.68084564174506,43.68114450123393,43.6814249024616,43.68171494206091,43.68174710947116,43.68204135193213,43.68232656908085,43.6824186659202,43.68259400174593,43.68285662645934,43.68373974895134,43.68424375759714,43.68534414941019,43.68745710180712,43.68750709310434,43.68754075970016,43.68816897105517,43.68836035509668,43.68880411073246,43.68928883875414,43.68966215945731,43.68991337496055,43.69016061295424,43.68921889811577,43.68851267493044,43.68834045610878,43.68752551375601,43.68674196434659,43.68665160168118,43.68582763774948,43.68490839501535,43.68398926574291,43.6837355462565,43.68323271691692,43.68291597556131,43.68212800934683,43.68046652502107,43.67968760013649,43.67928009694869,43.67926333357637,43.67948507435236,43.67922674031978,43.679212792032]}]],[[{"lng":[-79.35851975139897,-79.35683270022189,-79.36134490312664,-79.36143692030814,-79.36152889686386,-79.36216215021196,-79.36272749505048,-79.36125247055946,-79.36034208119173,-79.36069362537397,-79.36057936008517,-79.3604995769884,-79.36035261822374,-79.36013311232914,-79.36005827398014,-79.35926969036778,-79.35851975139897],"lat":[43.65390029403001,43.65010788292163,43.64836739829941,43.64854425768912,43.64875271362544,43.6502068901671,43.65156106749578,43.6519091002835,43.65209867844435,43.65248189763354,43.65263779436866,43.65271766826495,43.65282367450539,43.65296905878258,43.65301749615345,43.65348337593034,43.65390029403001]}]],[[{"lng":[-79.47357218968229,-79.47392370828004,-79.47426676632355,-79.47434844578251,-79.47464628487874,-79.47505204076248,-79.47390096395647,-79.4727324657954,-79.47150434484328,-79.47039059605436,-79.46971084031554,-79.46942477251375,-79.46877223227841,-79.46836034321403,-79.46785397124633,-79.46768539453461,-79.46747447214746,-79.46663909888878,-79.46574177728924,-79.46536866125608,-79.46506270797008,-79.46485244020339,-79.46460928848053,-79.46440109458649,-79.46433857047144,-79.46427582714165,-79.46421460570457,-79.46415431756456,-79.46409382734248,-79.46403426813245,-79.46397633306951,-79.46391830848354,-79.46386097489764,-79.46380530547587,-79.46374962134581,-79.46369476824206,-79.46364073357269,-79.46358741734942,-79.46353468747785,-79.46348290351727,-79.46343195058948,-79.46338168856687,-79.46333144161891,-79.46328282902925,-79.46323491993351,-79.46318795677288,-79.46314159711622,-79.46311244484188,-79.46308257667478,-79.46305414049696,-79.4630258193076,-79.46299902203793,-79.46297293848917,-79.46294685500088,-79.46292148752845,-79.4628978588615,-79.4628748439236,-79.46285204854959,-79.46283056790166,-79.46280990553278,-79.46278946042679,-79.46277033233059,-79.46275213744387,-79.46273454590691,-79.46271788527687,-79.46270192796817,-79.46268679928421,-79.46246688659103,-79.46222025864607,-79.46195676431799,-79.46182294810102,-79.46180901268403,-79.46179425912021,-79.46177877695921,-79.46176332257816,-79.46174631913135,-79.46172776892018,-79.46170912920178,-79.46169069406305,-79.46167050762186,-79.46164950535118,-79.46162779765891,-79.46160525910842,-79.46158201744407,-79.46155805993331,-79.46153327156672,-79.46150766508465,-79.461481357805,-79.46145421737124,-79.46142627157792,-79.46139841536097,-79.46136892288052,-79.46133851002924,-79.4613073685919,-79.46128825673527,-79.46127563911149,-79.46124368193502,-79.46121113442672,-79.46117704018157,-79.46114272869504,-79.46110699827494,-79.4610712679236,-79.46073754295364,-79.46042909825437,-79.46113019065517,-79.4610975033025,-79.46164570320484,-79.46269595162472,-79.46377422574437,-79.46485493136382,-79.46546614979388,-79.4653186587398,-79.46599877197021,-79.46607722130317,-79.46832178040417,-79.47037432646707,-79.47054780238706,-79.47088933500955,-79.47158769736463,-79.47193915500154,-79.47205735772548,-79.47229125163919,-79.47240890881592,-79.47280125594737,-79.47320220576823,-79.47354655082859,-79.47357218968229],"lat":[43.68674196434659,43.68752551375601,43.68834045610878,43.68851267493044,43.68921889811577,43.69016061295424,43.6904153979339,43.69067894048602,43.69097318067977,43.69121942744684,43.6913543593751,43.69141813352795,43.69154892334974,43.69161541989635,43.69170765431684,43.69170989511903,43.6917115678182,43.69178138039101,43.69187295024143,43.69193545551719,43.69201236469122,43.69207255441536,43.69215030597493,43.69179643019208,43.69169647788205,43.69160111361685,43.69151027077439,43.69141493943457,43.69131951532564,43.69121969271026,43.69112439289867,43.69102459084667,43.69093380006522,43.69083393959065,43.69073866991666,43.6906389103792,43.69053916181679,43.6904393328662,43.69034860380669,43.6902488854255,43.690144677186,43.69004948022646,43.68994969241886,43.68985001658482,43.6897548511961,43.68965069649173,43.68955096086984,43.68947855270454,43.68940613490332,43.68933373635208,43.6892568383318,43.68918437178685,43.68911200485421,43.68903963791563,43.68896728060076,43.68889494666669,43.68882262098185,43.68874570721661,43.68867340215828,43.68860110810173,43.68852431595488,43.68845204252721,43.68837528064023,43.68830293785368,43.68822619659766,43.68815396580781,43.6880773351732,43.68721000295899,43.68624328851293,43.68520424021138,43.68467573091104,43.68461252919293,43.68454931646045,43.68449059492973,43.68442278172781,43.68435953870387,43.68430077586806,43.68423751081266,43.6841742485074,43.68411546363595,43.68405657775833,43.68399327138933,43.68393445484259,43.68387112781435,43.68380779114422,43.68374894429509,43.68369008642642,43.68362662808516,43.6835677495527,43.68350435915903,43.68344547097577,43.68338656075546,43.68332763813546,43.68327320670915,43.68323244027847,43.68320976533245,43.68315523288433,43.68309178047429,43.68303280823952,43.68297833408098,43.68291933979263,43.68286034549353,43.6823606496737,43.68189271048481,43.68174011672932,43.68165415732493,43.68153550958046,43.68130649738063,43.68107335122539,43.68084481856759,43.68071349591157,43.68035134420979,43.68019384773056,43.68019039970216,43.67971179975491,43.67928009694869,43.67968760013649,43.68046652502107,43.68212800934683,43.68291597556131,43.68323271691692,43.6837355462565,43.68398926574291,43.68490839501535,43.68582763774948,43.68665160168118,43.68674196434659]}]],[[{"lng":[-79.48286816568844,-79.48329163033262,-79.4839740889859,-79.4843409562711,-79.48525963458506,-79.48609534298666,-79.48654118608194,-79.48545187009206,-79.48350432141798,-79.48042529734492,-79.47554334419412,-79.47544938553618,-79.47523001308571,-79.47515704155118,-79.47513682997084,-79.47492522080059,-79.4748087801687,-79.47473678286967,-79.4744972868999,-79.47443063337926,-79.47427053509692,-79.47423612661105,-79.47410427431507,-79.47390096395647,-79.47505204076248,-79.47608811801626,-79.47717094336414,-79.47888731290745,-79.48102831566352,-79.48286816568844],"lat":[43.68836035509668,43.68963138986724,43.69128781443143,43.69192741275452,43.69329907731888,43.69460671108889,43.69526094127298,43.69546264821612,43.69590051458926,43.69660237046933,43.69768096632441,43.69722952259977,43.69630821112835,43.69596957218395,43.69587478141513,43.6949220657206,43.69434870409607,43.69406868087302,43.69298947373421,43.69274093861635,43.69192853010585,43.69178844949018,43.6912059691148,43.6904153979339,43.69016061295424,43.68991337496055,43.68966215945731,43.68928883875414,43.68880411073246,43.68836035509668]}]],[[{"lng":[-79.46971084031554,-79.47039059605436,-79.47150434484328,-79.4727324657954,-79.47390096395647,-79.47410427431507,-79.47423612661105,-79.47427053509692,-79.47443063337926,-79.4744972868999,-79.47473678286967,-79.4748087801687,-79.47492522080059,-79.47513682997084,-79.47515704155118,-79.47523001308571,-79.47544938553618,-79.47554334419412,-79.47459273534771,-79.46879185965052,-79.46709132422171,-79.46653780441432,-79.46653425120095,-79.4660918489766,-79.46602723570945,-79.4659648585903,-79.46590259665717,-79.46583890312414,-79.46577417701165,-79.46570874780068,-79.4656432020337,-79.46558980903932,-79.46557634175275,-79.46550854661383,-79.46548869765077,-79.46546709736886,-79.46544488570228,-79.46542268426214,-79.46539883379781,-79.46537488109786,-79.46534961119379,-79.46532351057073,-79.46529638942955,-79.46526875461767,-79.46524019928492,-79.46521092800786,-79.46518083850238,-79.46514993077125,-79.46511841168559,-79.46508607208388,-79.46505281197838,-79.46501893823023,-79.46497483596974,-79.46494851902185,-79.46491229522447,-79.4648751361299,-79.46483737591406,-79.46479786441458,-79.46475836551703,-79.46471813818874,-79.46460928848053,-79.46485244020339,-79.46506270797008,-79.46536866125608,-79.46574177728924,-79.46663909888878,-79.46747447214746,-79.46768539453461,-79.46785397124633,-79.46836034321403,-79.46877223227841,-79.46942477251375,-79.46971084031554],"lat":[43.6913543593751,43.69121942744684,43.69097318067977,43.69067894048602,43.6904153979339,43.6912059691148,43.69178844949018,43.69192853010585,43.69274093861635,43.69298947373421,43.69406868087302,43.69434870409607,43.6949220657206,43.69587478141513,43.69596957218395,43.69630821112835,43.69722952259977,43.69768096632441,43.69785732370497,43.69926093612879,43.69966582346565,43.69776311584292,43.69775397614048,43.69614991300578,43.69591490354788,43.69568451508527,43.69544962712338,43.69521471990063,43.69498420975525,43.69474918912164,43.69451875790155,43.69433340983065,43.69428371796173,43.69405325644624,43.69398097385293,43.69391316874361,43.69384526539977,43.69377295120372,43.69370511586354,43.69363727914486,43.69356492372722,43.69349705815231,43.69342908884588,43.69336120265407,43.69329330409258,43.69322539590814,43.69315747672651,43.69308954654708,43.69302151812756,43.69295356869836,43.69288560689569,43.69281763684045,43.69274952938502,43.69268616181611,43.69261807014068,43.69255455691492,43.69248653459599,43.69242298974346,43.6923549440404,43.69229138954273,43.69215030597493,43.69207255441536,43.69201236469122,43.69193545551719,43.69187295024143,43.69178138039101,43.6917115678182,43.69170989511903,43.69170765431684,43.69161541989635,43.69154892334974,43.69141813352795,43.6913543593751]}]],[[{"lng":[-79.47805131916368,-79.47749639621897,-79.47737248782995,-79.47710690452034,-79.47709414631009,-79.47680067740356,-79.47676104240129,-79.47595700367296,-79.47593051756894,-79.47554334419412,-79.48042529734492,-79.48350432141798,-79.48545187009206,-79.48654118608194,-79.48683541090236,-79.48688911219318,-79.48694292679578,-79.48699498944403,-79.48704645160764,-79.48709708470993,-79.48714679731906,-79.4871966123608,-79.48724478858965,-79.48729225116358,-79.48733899781664,-79.4873857446007,-79.4874308548162,-79.48747596289549,-79.48750526110469,-79.48753383249293,-79.48756158559385,-79.48758841811011,-79.48761371627143,-79.4876390122291,-79.48766287838141,-79.48768664229402,-79.48770958790166,-79.48773099913564,-79.48775179438529,-79.4877723873541,-79.48779154823343,-79.4878098907917,-79.48782823339172,-79.48784492849445,-79.48786091610771,-79.48787618994635,-79.48789064544951,-79.48790416951618,-79.48791719293317,-79.48792847733681,-79.4879397487037,-79.48823729214438,-79.48824251469765,-79.48824905637358,-79.48825581343803,-79.48826327583062,-79.4882722835121,-79.48828129120503,-79.48829182263339,-79.48830328555532,-79.48831464619175,-79.48832742601063,-79.4883411395874,-79.48835567158581,-79.48837010130265,-79.48838627014875,-79.4884028374487,-79.48842043854928,-79.48843885807969,-79.48845799374071,-79.48847806090058,-79.48849873112695,-79.48852011522717,-79.4885378295353,-79.48855718068397,-79.48857642955414,-79.48859639455654,-79.4886179964052,-79.48863949597796,-79.4886619162901,-79.48868526809343,-79.48870912068098,-79.4887337917107,-79.48875857581643,-79.48878477916992,-79.48881191627379,-79.48883905340615,-79.48886679364195,-79.4888955676667,-79.48892495553451,-79.4889551618507,-79.48898618661663,-79.48901721141526,-79.48904895236305,-79.48908162478752,-79.48911490033731,-79.48914899434236,-79.4891833037027,-79.48921822464969,-79.48925395560119,-79.48928876586602,-79.48930786436013,-79.4893243945899,-79.48936023865838,-79.48939740199847,-79.4894338492574,-79.48947204639892,-79.48951074440035,-79.48954965773645,-79.48958928723016,-79.48962963288253,-79.48967078631928,-79.48971206345409,-79.48975404381213,-79.48979674259199,-79.48983954371451,-79.48988317396714,-79.4899267935938,-79.48997185616696,-79.49001702107861,-79.49006228832884,-79.49010827173994,-79.49219126652743,-79.49224564064137,-79.4923001066187,-79.49235386473174,-79.49240669391408,-79.49245942083964,-79.49251164675401,-79.49256365763206,-79.49261483964931,-79.49266613448526,-79.49271661092041,-79.49276626895359,-79.4928160188921,-79.49286484812812,-79.49291286940603,-79.49296026645433,-79.49300745894062,-79.49305413994531,-79.49310061413232,-79.49314627217146,-79.49319120369402,-79.49323532723383,-79.49327965543416,-79.49332295020453,-79.49336542657267,-79.49340729955429,-79.49344895756623,-79.49349000178645,-79.49353022759965,-79.49356963500428,-79.4940859233576,-79.4934371920875,-79.49306400714451,-79.49185997084045,-79.49171891566328,-79.49022093956316,-79.48765684504126,-79.48242613321007,-79.47903269999237,-79.47892039365574,-79.47878960183324,-79.47868134172087,-79.47853818110721,-79.47841293664017,-79.47830889395537,-79.47805131916368],"lat":[43.70865398221823,43.7061795946686,43.70558822319141,43.70455368582653,43.70426536191891,43.70301899509334,43.70283833682124,43.69942414053503,43.69930226026983,43.69768096632441,43.69660237046933,43.69590051458926,43.69546264821612,43.69526094127298,43.69571502797103,43.69580125810357,43.69588298870486,43.69596928711044,43.69605098651633,43.69613717592765,43.69622335313001,43.69630953166565,43.69639577850543,43.69648192585726,43.69656806371218,43.69665420154865,43.69674031770409,43.69682652383249,43.696894426663,43.69696682087059,43.69703920424072,43.69711157542014,43.69718392628693,43.69725636713827,43.69732869904178,43.69740102958657,43.69747334929687,43.69754564869756,43.69761802996234,43.69769031852577,43.69776258813473,43.69783484691144,43.69790710568496,43.69798384365893,43.69805616128453,43.69812837944263,43.69820058677001,43.69827728277405,43.69834947114787,43.6984216365101,43.69849839272139,43.70018147326535,43.70022205137138,43.70026714792898,43.70030774633305,43.70035285506881,43.70039348324325,43.70043411141686,43.70047926074758,43.70051992139688,43.70056058069176,43.70060584977978,43.70064654019563,43.70068724143527,43.7007279413199,43.70076866420642,43.70081389336249,43.70085463518822,43.70089538783622,43.70093614995299,43.70097242338517,43.70101320579156,43.70105408765572,43.70109032995775,43.70112659390531,43.70116285649669,43.70119912855584,43.70123542225902,43.70127171460531,43.70130801912403,43.70133983495544,43.70137615840589,43.70141249267445,43.70144432743155,43.70148077197225,43.70151263783301,43.70154450368754,43.70158087851161,43.70161276599718,43.70164466159196,43.70167656800082,43.70170848522315,43.70174040243735,43.70177232911154,43.70179976709099,43.7018317140359,43.70186367179146,43.70189113138281,43.70191868906895,43.70195066843331,43.70198263561544,43.70199639108694,43.70201461360611,43.70204209343125,43.70207409168465,43.70210608046062,43.70213359135357,43.70216560985457,43.70219313018764,43.70222065997266,43.70224819920891,43.70228025010729,43.70230330062403,43.70233095143985,43.70235852171406,43.70238609332466,43.70240917487473,43.70243675726837,43.70245985771086,43.70248295948776,43.70250606259891,43.70252917515413,43.70356958228044,43.70359280467996,43.70362052926558,43.70364383350658,43.70367153644769,43.70369923801533,43.70372243194971,43.70375012402361,43.7037823061408,43.7038099887231,43.70383766048808,43.70386532143679,43.70389748457583,43.70392963555079,43.70395727484981,43.70398940689653,43.70402153622592,43.70405365879108,43.70408586862996,43.70411797766839,43.70415457810888,43.70418666687869,43.7042187583297,43.70425533713822,43.70429190514056,43.70432396417267,43.70436052135497,43.70439707043052,43.70443360870272,43.70447013617235,43.70494513465173,43.70511212315533,43.70517930880549,43.7054560805012,43.70548572590769,43.7057991094319,43.70636842936565,43.70753263079176,43.70831137928809,43.70833689048658,43.70838916147913,43.70843273028358,43.70848033533311,43.70852367775741,43.70855830037092,43.70865398221823]}]],[[{"lng":[-79.4809633558813,-79.48095611283179,-79.48095231507614,-79.48094698229286,-79.48094941562968,-79.48094490380983,-79.48094406241535,-79.4808928155516,-79.48071470333193,-79.48011437210423,-79.4794594574423,-79.47945437840524,-79.47925947466982,-79.47884640675632,-79.4786274380737,-79.47826975000001,-79.47805131916368,-79.47830889395537,-79.47841293664017,-79.47853818110721,-79.47868134172087,-79.47878960183324,-79.47892039365574,-79.47903269999237,-79.48242613321007,-79.48765684504126,-79.48810469465327,-79.48826675488397,-79.48832617942018,-79.48837824550886,-79.48844123242831,-79.4884720715707,-79.48853900888848,-79.48879419244643,-79.48908360232637,-79.48919043973937,-79.48923522240094,-79.4892804573639,-79.48933149227426,-79.48938250364884,-79.48943140096105,-79.48951229718142,-79.48956607766922,-79.48957746817902,-79.48959488469593,-79.48959383671352,-79.48961664306536,-79.48965808428467,-79.48968679877477,-79.48970886184179,-79.48973669985241,-79.48976050939049,-79.48977675813019,-79.48980410393808,-79.48980715367254,-79.48983300529689,-79.48986270696021,-79.48989383894535,-79.48993566794846,-79.48997966269756,-79.49002467457812,-79.49014492561886,-79.49052112968239,-79.49084534751405,-79.49137081425272,-79.49141622704023,-79.49144901246478,-79.49149794710164,-79.49155585519161,-79.49163652127159,-79.49167316982727,-79.49177481676568,-79.49225259053227,-79.49247946582823,-79.4918349904404,-79.49169002232483,-79.4915449389224,-79.49140007289709,-79.49125582075953,-79.49111155789512,-79.49096719255917,-79.49082374809271,-79.4906802011553,-79.490536654081,-79.49039392554384,-79.49025119687053,-79.49010845737436,-79.4899663317269,-79.48982442129933,-79.48968229308747,-79.48954098565575,-79.48939966732765,-79.48925825723809,-79.48911766566985,-79.48897706317408,-79.48883727918017,-79.48869750580357,-79.48855761913367,-79.48841865330282,-79.48827936946725,-79.48801324657146,-79.48774722776274,-79.48748089055192,-79.48721547380813,-79.48695005651417,-79.48668555969073,-79.48642085538141,-79.48615615278894,-79.48589225733025,-79.48562755362103,-79.48536437336021,-79.48510027148777,-79.48130337037644,-79.48126288039543,-79.48121753296937,-79.48119871408817,-79.48116450139393,-79.48114503463522,-79.4811280239722,-79.48111101332678,-79.48109543540809,-79.48107995984176,-79.48106613303845,-79.48105229488337,-79.48103907075932,-79.48102759773188,-79.48101591094941,-79.48100596161625,-79.48099592132893,-79.48098740471858,-79.48097982051173,-79.48097223631341,-79.4809659711137,-79.4809633558813],"lat":[43.72156817058833,43.72151856346558,43.72148250509682,43.72144642632296,43.72141495176794,43.72137879388673,43.72134727580109,43.72113504823439,43.72050686327515,43.71786076167469,43.71484934936441,43.714714252067,43.71398231800379,43.71221657253462,43.71117824925764,43.70961578293905,43.70865398221823,43.70855830037092,43.70852367775741,43.70848033533311,43.70843273028358,43.70838916147913,43.70833689048658,43.70831137928809,43.70753263079176,43.70636842936565,43.70746386659972,43.70780808626569,43.70794399225951,43.70803470090257,43.70809404699579,43.70816196984359,43.70827997110364,43.70862083058361,43.70911085489912,43.70927439324276,43.70937400717965,43.70945562309508,43.70955081864835,43.70965510587695,43.70974577213133,43.7098908732581,43.71003570595614,43.71010787242544,43.71018461953276,43.71032413652307,43.71050006676528,43.71067165226944,43.71073054469493,43.71087495812333,43.71100135386663,43.71110069039718,43.71119542600916,43.71131290325683,43.71140746443507,43.71155183787013,43.7116692562095,43.71178678345966,43.71188185706072,43.71201296721142,43.7121035818402,43.71235281533151,43.71383888186945,43.71520282356641,43.71702393100377,43.71716406119985,43.7172771087653,43.71739928145542,43.71753507554001,43.71778378473974,43.71786528627418,43.71811418226483,43.7193900358133,43.72009086213529,43.72020388519995,43.72022897782632,43.72025865975409,43.72028375335947,43.72030885488571,43.72033845707553,43.72036805772963,43.72039767036224,43.72042728146099,43.7204568923772,43.72048651392561,43.72051613529356,43.72055025732719,43.72058438729331,43.72061401893908,43.72064823853687,43.72068237878637,43.72072101970376,43.72075515824564,43.72078930743391,43.7208279572913,43.72086661779877,43.72090077728812,43.72093943609455,43.72097810690946,43.72102127433489,43.72108085743022,43.72114035126191,43.72120434124519,43.72126834279108,43.72133234371156,43.72139635620324,43.72146045535266,43.72152446389025,43.7215929834933,43.7216569907882,43.72172551864629,43.72179403367916,43.7227252120798,43.72254904553962,43.72235039951667,43.72226463068501,43.72211555343831,43.72208378780849,43.72205205481973,43.72202032182855,43.72198860787778,43.72195689528517,43.72192070361979,43.72188901278717,43.72185733011442,43.72182116973018,43.72178941746922,43.72175777832125,43.72172163697768,43.72169001687206,43.72165390817397,43.72161779947528,43.72158620929456,43.72156817058833]}]],[[{"lng":[-79.48121753296937,-79.48126288039543,-79.48130337037644,-79.48123876126344,-79.47816248597842,-79.47553040656157,-79.47329121699202,-79.47255630242334,-79.47251963323048,-79.472481160453,-79.47229260987525,-79.47205568596223,-79.47173816350944,-79.4708889552108,-79.47009879737357,-79.4698107253468,-79.46976672929061,-79.47093693341705,-79.47189808099843,-79.47337137436423,-79.4745495818707,-79.47573908671328,-79.47604263757752,-79.47635397765684,-79.47659848002321,-79.47687752009099,-79.4771282415967,-79.47744873333924,-79.47789089754799,-79.47805131916368,-79.47826975000001,-79.4786274380737,-79.47884640675632,-79.47925947466982,-79.47945437840524,-79.4794594574423,-79.48011437210423,-79.48071470333193,-79.4808928155516,-79.48094406241535,-79.48094490380983,-79.48094941562968,-79.48094698229286,-79.48095231507614,-79.48095611283179,-79.4809633558813,-79.4809659711137,-79.48097223631341,-79.48097982051173,-79.48098740471858,-79.48099592132893,-79.48100596161625,-79.48101591094941,-79.48102759773188,-79.48103907075932,-79.48105229488337,-79.48106613303845,-79.48107995984176,-79.48109543540809,-79.48111101332678,-79.4811280239722,-79.48114503463522,-79.48116450139393,-79.48119871408817,-79.48121753296937],"lat":[43.72235039951667,43.72254904553962,43.7227252120798,43.72273785638106,43.72349378473395,43.724133940025,43.72468026253927,43.72486407407894,43.72468354463866,43.72448939810518,43.72354599209437,43.72236329667957,43.72074742812116,43.71642772345413,43.71241045659625,43.71098851713433,43.71079888633047,43.71050839238199,43.7102961990437,43.70999163855289,43.70971921840196,43.70946503075411,43.70939706325087,43.70932460777715,43.70926035249283,43.70917405204145,43.7090783715479,43.70894752210661,43.708768871495,43.70865398221823,43.70961578293905,43.71117824925764,43.71221657253462,43.71398231800379,43.714714252067,43.71484934936441,43.71786076167469,43.72050686327515,43.72113504823439,43.72134727580109,43.72137879388673,43.72141495176794,43.72144642632296,43.72148250509682,43.72151856346558,43.72156817058833,43.72158620929456,43.72161779947528,43.72165390817397,43.72169001687206,43.72172163697768,43.72175777832125,43.72178941746922,43.72182116973018,43.72185733011442,43.72188901278717,43.72192070361979,43.72195689528517,43.72198860787778,43.72202032182855,43.72205205481973,43.72208378780849,43.72211555343831,43.72226463068501,43.72235039951667]}]],[[{"lng":[-79.46976672929061,-79.46934111530972,-79.4688659300184,-79.46882545521818,-79.46878077377917,-79.46873549331353,-79.4686887785364,-79.46864196404027,-79.46859421452548,-79.46854698142459,-79.46854596833192,-79.46850731657491,-79.46849679939771,-79.46844681462113,-79.46839609945788,-79.4683438646033,-79.46829151320469,-79.46823856059248,-79.46818468758165,-79.46812998648463,-79.46807456731491,-79.46801844696377,-79.46796067553197,-79.46790312366974,-79.4678445345527,-79.46709132422171,-79.46879185965052,-79.47459273534771,-79.47554334419412,-79.47593051756894,-79.47595700367296,-79.47676104240129,-79.47680067740356,-79.47709414631009,-79.47710690452034,-79.47737248782995,-79.47749639621897,-79.47805131916368,-79.47789089754799,-79.47744873333924,-79.4771282415967,-79.47687752009099,-79.47659848002321,-79.47635397765684,-79.47604263757752,-79.47573908671328,-79.4745495818707,-79.47337137436423,-79.47189808099843,-79.47093693341705,-79.46976672929061],"lat":[43.71079888633047,43.70867726941925,43.70628483630685,43.70608174919413,43.70588310669212,43.70567986512386,43.70547669433176,43.70527343215804,43.70507474844558,43.70487589158774,43.70487146703439,43.70471341416467,43.70466826325038,43.70446495848513,43.70426623491999,43.70406289994094,43.70386415438709,43.70366089974712,43.70345763273915,43.70325876557733,43.70305997877741,43.7028565915267,43.70265777312593,43.70245436661521,43.70225553718126,43.69966582346565,43.69926093612879,43.69785732370497,43.69768096632441,43.69930226026983,43.69942414053503,43.70283833682124,43.70301899509334,43.70426536191891,43.70455368582653,43.70558822319141,43.7061795946686,43.70865398221823,43.708768871495,43.70894752210661,43.7090783715479,43.70917405204145,43.70926035249283,43.70932460777715,43.70939706325087,43.70946503075411,43.70971921840196,43.70999163855289,43.7102961990437,43.71050839238199,43.71079888633047]}]],[[{"lng":[-79.45473985655367,-79.45552787739848,-79.45681041442705,-79.45858003125193,-79.46086368318353,-79.46207440774293,-79.46322275681302,-79.46659271493084,-79.46874582338711,-79.46976672929061,-79.4698107253468,-79.47009879737357,-79.4708889552108,-79.47173816350944,-79.47205568596223,-79.47229260987525,-79.472481160453,-79.47251963323048,-79.47255630242334,-79.47178001580548,-79.47156229122064,-79.47134353304864,-79.47112499128387,-79.47090656363478,-79.47068720243203,-79.4704679432085,-79.47024787707437,-79.47002769606659,-79.4698076269641,-79.46958694577091,-79.4693661619047,-79.46914537770732,-79.46613405969399,-79.46267354142167,-79.45787826023626,-79.45777654708442,-79.45764200677043,-79.45758148069999,-79.45707022099133,-79.45689230822576,-79.45686135177864,-79.45676432621451,-79.45665348217256,-79.45662218024754,-79.45646099405799,-79.45640016465312,-79.45619576558235,-79.45616133261036,-79.45566936793028,-79.4555052002427,-79.45528163542781,-79.45518878323175,-79.45473985655367],"lat":[43.71408116752363,43.71393878687064,43.71366345908267,43.71327316113649,43.71274572844305,43.71248287717473,43.71223726823589,43.71147667947517,43.71100585001324,43.71079888633047,43.71098851713433,43.71241045659625,43.71642772345413,43.72074742812116,43.72236329667957,43.72354599209437,43.72448939810518,43.72468354463866,43.72486407407894,43.72505623955799,43.72510742950179,43.72516301616596,43.72521410432331,43.72526069260648,43.72531176895968,43.72536284625565,43.7254094113462,43.72546047545577,43.72550712967632,43.72555368525511,43.72560023903149,43.72564679237585,43.72627264771562,43.7270185888452,43.72806148878693,43.72766844095363,43.72698679724418,43.72668441467861,43.72474164116915,43.72384786451503,43.72368991213014,43.72323841366544,43.7227191237051,43.72257466951,43.72184324320467,43.72161287141042,43.72071873546908,43.72045712319823,43.71824904409294,43.71763451128278,43.71655107267809,43.71614913965276,43.71408116752363]}]],[[{"lng":[-79.46976672929061,-79.46874582338711,-79.46659271493084,-79.46322275681302,-79.46207440774293,-79.46086368318353,-79.45858003125193,-79.45681041442705,-79.45552787739848,-79.45473985655367,-79.4544944228337,-79.45433754451534,-79.45401325801959,-79.45397381164356,-79.45383819269011,-79.45377936188522,-79.45365954083742,-79.45344190619083,-79.45338673281827,-79.45316911098482,-79.4530974013596,-79.45366568554873,-79.46433308256847,-79.46475232480302,-79.4651551430842,-79.4684238105885,-79.46854698142459,-79.46859421452548,-79.46864196404027,-79.4686887785364,-79.46873549331353,-79.46878077377917,-79.46882545521818,-79.4688659300184,-79.46934111530972,-79.46976672929061],"lat":[43.71079888633047,43.71100585001324,43.71147667947517,43.71223726823589,43.71248287717473,43.71274572844305,43.71327316113649,43.71366345908267,43.71393878687064,43.71408116752363,43.71306503593265,43.71234707677499,43.71080758235204,43.71065392498363,43.70998135242032,43.7097059059172,43.70905596173587,43.70816614104242,43.70789983576354,43.70697400661719,43.7066543655905,43.70656816591655,43.70429011611505,43.70419573693243,43.70559105547841,43.70485389640038,43.70487589158774,43.70507474844558,43.70527343215804,43.70547669433176,43.70567986512386,43.70588310669212,43.70608174919413,43.70628483630685,43.70867726941925,43.71079888633047]}]],[[{"lng":[-79.46854698142459,-79.4684238105885,-79.4651551430842,-79.46475232480302,-79.46433308256847,-79.45366568554873,-79.4530974013596,-79.45298474349843,-79.45291254658794,-79.4526102643936,-79.452465241469,-79.45240916316179,-79.45225365815998,-79.45637616452561,-79.46036199195792,-79.46394638903381,-79.46709132422171,-79.4678445345527,-79.46790312366974,-79.46796067553197,-79.46801844696377,-79.46807456731491,-79.46812998648463,-79.46818468758165,-79.46823856059248,-79.46829151320469,-79.4683438646033,-79.46839609945788,-79.46844681462113,-79.46849679939771,-79.46850731657491,-79.46854596833192,-79.46854698142459],"lat":[43.70487589158774,43.70485389640038,43.70559105547841,43.70419573693243,43.70429011611505,43.70656816591655,43.7066543655905,43.70615214997434,43.70576399705747,43.70451744971818,43.70381765155995,43.7035828403499,43.70296841880666,43.70204278067,43.70114656906383,43.70035293708817,43.69966582346565,43.70225553718126,43.70245436661521,43.70265777312593,43.7028565915267,43.70305997877741,43.70325876557733,43.70345763273915,43.70366089974712,43.70386415438709,43.70406289994094,43.70426623491999,43.70446495848513,43.70466826325038,43.70471341416467,43.70487146703439,43.70487589158774]}]],[[{"lng":[-79.36401044653344,-79.36429391986236,-79.36444990467039,-79.36342946625452,-79.3632111100456,-79.36225117645441,-79.36145271651047,-79.36100126291419,-79.36050068207473,-79.36017677389826,-79.35957135260308,-79.35891130447253,-79.35774698080155,-79.35653495348453,-79.35476986896008,-79.35464656864448,-79.35454138203646,-79.35431156344174,-79.3542695166567,-79.35419159057675,-79.35409665346901,-79.35397266670385,-79.35386137749113,-79.35374176632941,-79.35363374643997,-79.35352096406697,-79.35340542304597,-79.35336376057985,-79.35329470496147,-79.3531887100561,-79.35308845883357,-79.35298861845577,-79.35292626839781,-79.35288391418347,-79.35278061799475,-79.35267556538552,-79.35259391923746,-79.35250205146369,-79.352407465686,-79.3523211206064,-79.35223561186105,-79.35213798113142,-79.35205232892339,-79.35196457207027,-79.35172834670693,-79.35218166681285,-79.35242476349815,-79.35286800570809,-79.35315822727759,-79.35521924580006,-79.35651579935043,-79.35696416097777,-79.35711869033116,-79.35725608825483,-79.35736241263416,-79.35749502685303,-79.35782035762831,-79.35851975139897,-79.35926969036778,-79.36005827398014,-79.36013311232914,-79.36035261822374,-79.3604995769884,-79.36057936008517,-79.36069362537397,-79.36034208119173,-79.36125247055946,-79.36272749505048,-79.36294164699616,-79.36299932929141,-79.36323299284675,-79.36358552213281,-79.36401044653344],"lat":[43.65455065615483,43.65521654413004,43.65561045553927,43.65582539923416,43.65587187335879,43.65608767305169,43.65626536074995,43.65636241057666,43.65647225691166,43.65655312907798,43.65669756513609,43.65682762047388,43.65709004827623,43.65736519122122,43.65777207236302,43.65779730336225,43.65781829401421,43.65786458605156,43.65778287219486,43.65765572144817,43.65755983279038,43.65744552157216,43.65734930717354,43.65723955974829,43.65714348224121,43.65703833397801,43.65695556093434,43.65692345351142,43.65686844626767,43.65677689853398,43.65670343755258,43.6566074773034,43.65656156851378,43.65652485990441,43.65644685374477,43.6563508180925,43.65627762485447,43.65620428420325,43.65612190225204,43.65604855105319,43.65597080084699,43.65588387374699,43.65581512337467,43.6557373405461,43.65555830394888,43.65547932177113,43.65543331543737,43.65532267809469,43.65525475426087,43.65476672789054,43.65448381226681,43.654391146679,43.65437536413449,43.65437283819598,43.65439237057512,43.65447529501149,43.65428192529838,43.65390029403001,43.65348337593034,43.65301749615345,43.65296905878258,43.65282367450539,43.65271766826495,43.65263779436866,43.65248189763354,43.65209867844435,43.6519091002835,43.65156106749578,43.65206392908505,43.65219519502687,43.65272525163108,43.65355866840074,43.65455065615483]}]],[[{"lng":[-79.45129413820973,-79.45370076819394,-79.45438930920493,-79.4628229970005,-79.46247386468978,-79.46294935215241,-79.46296318596654,-79.46532757178736,-79.46562891406008,-79.4656432020337,-79.46570874780068,-79.46577417701165,-79.46583890312414,-79.46590259665717,-79.4659648585903,-79.46602723570945,-79.4660918489766,-79.46653425120095,-79.46653780441432,-79.46709132422171,-79.46394638903381,-79.46036199195792,-79.45637616452561,-79.45225365815998,-79.45210994550904,-79.45191840136103,-79.45175687449262,-79.45158131531753,-79.4515450486785,-79.45142202303228,-79.45129413820973],"lat":[43.6986875730291,43.69851203596319,43.69869688944955,43.69679167327475,43.69584635707301,43.69567100280078,43.69500936134619,43.69448112228679,43.69446914504156,43.69451875790155,43.69474918912164,43.69498420975525,43.69521471990063,43.69544962712338,43.69568451508527,43.69591490354788,43.69614991300578,43.69775397614048,43.69776311584292,43.69966582346565,43.70035293708817,43.70114656906383,43.70204278067,43.70296841880666,43.70234524474687,43.70146927662248,43.70075124967297,43.70000602629477,43.69978498595614,43.69926111301325,43.6986875730291]}]],[[{"lng":[-79.46562891406008,-79.46532757178736,-79.46296318596654,-79.46294935215241,-79.46247386468978,-79.4628229970005,-79.45438930920493,-79.45370076819394,-79.45129413820973,-79.45113024418171,-79.45096365884369,-79.45087330724076,-79.45079139601671,-79.45077249831284,-79.45073203221513,-79.45066219296362,-79.45055927343688,-79.45045092879198,-79.45035169305135,-79.45026776020714,-79.45119080145336,-79.4515613844777,-79.45231898283616,-79.45278757288413,-79.45348683759046,-79.45394551512946,-79.45508815105366,-79.45588330922124,-79.45622334790754,-79.4571766149226,-79.45737373905138,-79.45787444775401,-79.45851762418552,-79.45966765089717,-79.46031592734542,-79.46081507349224,-79.46097364361624,-79.46220385326451,-79.46280354323729,-79.46289177708164,-79.46304234095385,-79.46325252347582,-79.46333721487242,-79.46341945131492,-79.46346279387829,-79.46362969685403,-79.46398718903949,-79.4642114826012,-79.46444565942798,-79.46460928848053,-79.46471813818874,-79.46475836551703,-79.46479786441458,-79.46483737591406,-79.4648751361299,-79.46491229522447,-79.46494851902185,-79.46497483596974,-79.46501893823023,-79.46505281197838,-79.46508607208388,-79.46511841168559,-79.46514993077125,-79.46518083850238,-79.46521092800786,-79.46524019928492,-79.46526875461767,-79.46529638942955,-79.46532351057073,-79.46534961119379,-79.46537488109786,-79.46539883379781,-79.46542268426214,-79.46544488570228,-79.46546709736886,-79.46548869765077,-79.46550854661383,-79.46557634175275,-79.46558980903932,-79.46562891406008],"lat":[43.69446914504156,43.69448112228679,43.69500936134619,43.69567100280078,43.69584635707301,43.69679167327475,43.69869688944955,43.69851203596319,43.6986875730291,43.69791099980631,43.69717948980277,43.6967461689867,43.69639839149531,43.69632161828898,43.69617244663961,43.69602746777349,43.69589554335818,43.69579946328073,43.69573510364981,43.69568445445915,43.69546741687844,43.69540933466192,43.6952350561707,43.69513787822684,43.69497162415507,43.69486980630541,43.69461960973184,43.69443231229196,43.6943558882229,43.69414362145705,43.69410577350096,43.69398839360394,43.69383764541737,43.69360100564187,43.69345454699288,43.69334181417039,43.69329893971487,43.69302293513602,43.69289138200985,43.69287006379027,43.69283608077937,43.69277139224938,43.69273652303345,43.69270162074591,43.69267519747629,43.69257832927744,43.6923985935601,43.69229808487417,43.69221121156096,43.69215030597493,43.69229138954273,43.6923549440404,43.69242298974346,43.69248653459599,43.69255455691492,43.69261807014068,43.69268616181611,43.69274952938502,43.69281763684045,43.69288560689569,43.69295356869836,43.69302151812756,43.69308954654708,43.69315747672651,43.69322539590814,43.69329330409258,43.69336120265407,43.69342908884588,43.69349705815231,43.69356492372722,43.69363727914486,43.69370511586354,43.69377295120372,43.69384526539977,43.69391316874361,43.69398097385293,43.69405325644624,43.69428371796173,43.69433340983065,43.69446914504156]}]],[[{"lng":[-79.45787444775401,-79.45781273678449,-79.45763552552243,-79.45331282475955,-79.45568044514376,-79.45633442916481,-79.45761875604242,-79.45891566081708,-79.46019985752376,-79.46042909825437,-79.46073754295364,-79.4610712679236,-79.46110699827494,-79.46114272869504,-79.46117704018157,-79.46121113442672,-79.46124368193502,-79.46127563911149,-79.46128825673527,-79.4613073685919,-79.46133851002924,-79.46136892288052,-79.46139841536097,-79.46142627157792,-79.46145421737124,-79.461481357805,-79.46150766508465,-79.46153327156672,-79.46155805993331,-79.46158201744407,-79.46160525910842,-79.46162779765891,-79.46164950535118,-79.46167050762186,-79.46169069406305,-79.46170912920178,-79.46172776892018,-79.46174631913135,-79.46176332257816,-79.46177877695921,-79.46179425912021,-79.46180901268403,-79.46182294810102,-79.46195676431799,-79.46222025864607,-79.46246688659103,-79.46268679928421,-79.46270192796817,-79.46271788527687,-79.46273454590691,-79.46275213744387,-79.46277033233059,-79.46278946042679,-79.46280990553278,-79.46283056790166,-79.46285204854959,-79.4628748439236,-79.4628978588615,-79.46292148752845,-79.46294685500088,-79.46297293848917,-79.46299902203793,-79.4630258193076,-79.46305414049696,-79.46308257667478,-79.46311244484188,-79.46314159711622,-79.46318795677288,-79.46323491993351,-79.46328282902925,-79.46333144161891,-79.46338168856687,-79.46343195058948,-79.46348290351727,-79.46353468747785,-79.46358741734942,-79.46364073357269,-79.46369476824206,-79.46374962134581,-79.46380530547587,-79.46386097489764,-79.46391830848354,-79.46397633306951,-79.46403426813245,-79.46409382734248,-79.46415431756456,-79.46421460570457,-79.46427582714165,-79.46433857047144,-79.46440109458649,-79.46460928848053,-79.46444565942798,-79.4642114826012,-79.46398718903949,-79.46362969685403,-79.46346279387829,-79.46341945131492,-79.46333721487242,-79.46325252347582,-79.46304234095385,-79.46289177708164,-79.46280354323729,-79.46220385326451,-79.46097364361624,-79.46081507349224,-79.46031592734542,-79.45966765089717,-79.45851762418552,-79.45787444775401],"lat":[43.69398839360394,43.69387701619195,43.69344945979456,43.68305177309612,43.68253781912203,43.68239587077943,43.68211596474617,43.68183180286282,43.68155204603474,43.68189271048481,43.6823606496737,43.68286034549353,43.68291933979263,43.68297833408098,43.68303280823952,43.68309178047429,43.68315523288433,43.68320976533245,43.68323244027847,43.68327320670915,43.68332763813546,43.68338656075546,43.68344547097577,43.68350435915903,43.6835677495527,43.68362662808516,43.68369008642642,43.68374894429509,43.68380779114422,43.68387112781435,43.68393445484259,43.68399327138933,43.68405657775833,43.68411546363595,43.6841742485074,43.68423751081266,43.68430077586806,43.68435953870387,43.68442278172781,43.68449059492973,43.68454931646045,43.68461252919293,43.68467573091104,43.68520424021138,43.68624328851293,43.68721000295899,43.6880773351732,43.68815396580781,43.68822619659766,43.68830293785368,43.68837528064023,43.68845204252721,43.68852431595488,43.68860110810173,43.68867340215828,43.68874570721661,43.68882262098185,43.68889494666669,43.68896728060076,43.68903963791563,43.68911200485421,43.68918437178685,43.6892568383318,43.68933373635208,43.68940613490332,43.68947855270454,43.68955096086984,43.68965069649173,43.6897548511961,43.68985001658482,43.68994969241886,43.69004948022646,43.690144677186,43.6902488854255,43.69034860380669,43.6904393328662,43.69053916181679,43.6906389103792,43.69073866991666,43.69083393959065,43.69093380006522,43.69102459084667,43.69112439289867,43.69121969271026,43.69131951532564,43.69141493943457,43.69151027077439,43.69160111361685,43.69169647788205,43.69179643019208,43.69215030597493,43.69221121156096,43.69229808487417,43.6923985935601,43.69257832927744,43.69267519747629,43.69270162074591,43.69273652303345,43.69277139224938,43.69283608077937,43.69287006379027,43.69289138200985,43.69302293513602,43.69329893971487,43.69334181417039,43.69345454699288,43.69360100564187,43.69383764541737,43.69398839360394]}]],[[{"lng":[-79.45258619235183,-79.45331282475955,-79.45763552552243,-79.45781273678449,-79.45787444775401,-79.45737373905138,-79.4571766149226,-79.45622334790754,-79.45588330922124,-79.45508815105366,-79.45394551512946,-79.45348683759046,-79.45278757288413,-79.45231898283616,-79.4515613844777,-79.45119080145336,-79.45026776020714,-79.44974146706326,-79.44894415624137,-79.44879460400534,-79.44829406288787,-79.44807669767155,-79.44740894134485,-79.4473649393598,-79.44681009993502,-79.44641221846344,-79.44610646136481,-79.44575944116467,-79.44678350401931,-79.44782136599355,-79.44899103802456,-79.45018660425355,-79.4513921827629,-79.45258619235183],"lat":[43.68320955422592,43.68305177309612,43.69344945979456,43.69387701619195,43.69398839360394,43.69410577350096,43.69414362145705,43.6943558882229,43.69443231229196,43.69461960973184,43.69486980630541,43.69497162415507,43.69513787822684,43.6952350561707,43.69540933466192,43.69546741687844,43.69568445445915,43.69443476960333,43.69253767051178,43.69217096928581,43.69094863270472,43.69038296543232,43.68873075815799,43.68860413216486,43.68726393451825,43.6862411184058,43.68556172010896,43.68470162936161,43.68447690862663,43.68425236611309,43.68399818607754,43.68373525316219,43.6834724433821,43.68320955422592]}]],[[{"lng":[-79.45258619235183,-79.4513921827629,-79.45018660425355,-79.44899103802456,-79.44782136599355,-79.44678350401931,-79.44575944116467,-79.44493843392839,-79.4444175379007,-79.44354526057661,-79.44243941449943,-79.4421727380592,-79.44196415992981,-79.44307095227235,-79.44505904830255,-79.44565330008837,-79.44621198006588,-79.44742061427991,-79.4481426841225,-79.44862537753515,-79.44941620132475,-79.44981713225874,-79.45059030448981,-79.45349464436063,-79.45476445781951,-79.45531475892534,-79.45595560657415,-79.45598555144915,-79.4560299772583,-79.45607336746028,-79.45611593969977,-79.45615871653348,-79.45619915468249,-79.45624017789707,-79.45627978503229,-79.45631928998765,-79.4563573634293,-79.45639473425558,-79.45643187442671,-79.45646831197922,-79.45648763595587,-79.45650391614087,-79.45653810418271,-79.45657207468173,-79.45660543170783,-79.45663808609147,-79.45666970487657,-79.45670082552461,-79.45673101052398,-79.45675906356189,-79.45677806803013,-79.45679787750748,-79.45681851815417,-79.45683997689194,-79.45686225372248,-79.45688444139692,-79.45689563750211,-79.45690825215756,-79.4569327787576,-79.4569575206677,-79.45698439927425,-79.45701058823447,-79.45703828497653,-79.45706669756568,-79.45709604357805,-79.45712610543643,-79.45715605201899,-79.45718776314359,-79.45721934594579,-79.45743911913382,-79.45795607838791,-79.45884307493247,-79.45951514671103,-79.46019985752376,-79.45891566081708,-79.45761875604242,-79.45633442916481,-79.45568044514376,-79.45331282475955,-79.45258619235183],"lat":[43.68320955422592,43.6834724433821,43.68373525316219,43.68399818607754,43.68425236611309,43.68447690862663,43.68470162936161,43.68487950299846,43.68360285906925,43.68144364025039,43.67874993141657,43.67880480972727,43.67817614540946,43.67795267544879,43.67749804446057,43.67736650689433,43.67726157871029,43.6769988519654,43.6768331197118,43.67672714784764,43.6765487477648,43.67645966553873,43.67629911195922,43.67564052347077,43.67534703961631,43.67522845048442,43.67509307800643,43.67513399173407,43.67518860414422,43.6752477035633,43.6753067919154,43.67536588301501,43.67542044148879,43.67548409990566,43.67554314814165,43.67560219498326,43.67566122247506,43.67571573944662,43.6757792553241,43.67583826068235,43.67587452980914,43.67590184581003,43.67596082076339,43.67602429378483,43.67608775851068,43.67614671272054,43.67621015395208,43.67626908743076,43.67633259930246,43.67639149133951,43.67643225711635,43.6764775347765,43.67651832263989,43.67655912154601,43.67659993149434,43.67663623921791,43.67665439445889,43.67667706986882,43.67671791017992,43.67675434239666,43.67679521445283,43.67682707515877,43.67686795824791,43.67690885099474,43.67694525532001,43.67698166930182,43.67702258273542,43.67705451794999,43.67709545345543,43.67740908022461,43.67814540139957,43.67951244194652,43.68052090835561,43.68155204603474,43.68183180286282,43.68211596474617,43.68239587077943,43.68253781912203,43.68305177309612,43.68320955422592]}]],[[{"lng":[-79.44098508304808,-79.44210048519331,-79.44375557645458,-79.44474023157576,-79.44509098137469,-79.44560394012547,-79.44603783085564,-79.44722235480917,-79.44747994244486,-79.44829276516222,-79.45108933696622,-79.45136835702425,-79.45177378416102,-79.45181453369912,-79.45185508111929,-79.45189704666505,-79.45193934589621,-79.45198317893562,-79.45202739419102,-79.45207182744662,-79.45211707877483,-79.45216316159234,-79.45220933100485,-79.45225672980897,-79.45230456450123,-79.45235359946082,-79.45240295465005,-79.45245282116626,-79.45250289225893,-79.45255438155012,-79.45260588430916,-79.45265831846955,-79.45271135514477,-79.45276520990807,-79.45281916700397,-79.45287385330251,-79.45293006008501,-79.452985782406,-79.45304200270114,-79.45309974349811,-79.45315688421606,-79.45321533863294,-79.45327402662134,-79.45333363494709,-79.45339384353166,-79.4534539499471,-79.45351589697987,-79.45361062706311,-79.45370443705205,-79.45379845171846,-79.4538915306881,-79.45398379413453,-79.4540753552718,-79.45416690330948,-79.45425823376216,-79.45434826163276,-79.45443816091927,-79.45452735555817,-79.45461561917031,-79.45470399845304,-79.45475737112334,-79.45480981035678,-79.45486224966191,-79.45491315523519,-79.45496417865091,-79.45501416407613,-79.45506353604689,-79.45511290577595,-79.45516145985009,-79.45520919595843,-79.45525610091049,-79.45530250784789,-79.45534789230622,-79.45539326364747,-79.45543791927638,-79.45548095207357,-79.45552385633594,-79.45556617116544,-79.45560847519759,-79.45564913005664,-79.45568981129732,-79.45572904786968,-79.45576818224581,-79.45580566745011,-79.45584315271589,-79.45588003765525,-79.45595560657415,-79.45531475892534,-79.45476445781951,-79.45349464436063,-79.45059030448981,-79.44981713225874,-79.44941620132475,-79.44862537753515,-79.4481426841225,-79.44742061427991,-79.44621198006588,-79.44565330008837,-79.44505904830255,-79.44307095227235,-79.44196415992981,-79.4421727380592,-79.44104224743559,-79.43994828186069,-79.43884652175043,-79.43777125606091,-79.43680058006395,-79.43590542868095,-79.4356787939342,-79.43590106833334,-79.43554420560477,-79.43525461924087,-79.43481598767957,-79.43450380851353,-79.43408724629691,-79.43403174163628,-79.43377131395067,-79.4346395518008,-79.43591733651529,-79.43713024303877,-79.43731659091004,-79.43746075195403,-79.43764165288935,-79.43808029802037,-79.43833671477772,-79.43838430540882,-79.43951430556535,-79.43979664731071,-79.44098508304808],"lat":[43.6729856351758,43.67272628361513,43.67237056812703,43.67215883140281,43.67208708683047,43.6720040444789,43.67192892676401,43.67172438715149,43.67168737754614,43.67154988305189,43.67109252756343,43.67103329455463,43.67154749292578,43.6715976461962,43.67164770669341,43.67170228740546,43.67174787058288,43.67179347452041,43.67184808565672,43.67189819870894,43.6719483228237,43.67199395716248,43.67204418369485,43.67209433683751,43.67213549382487,43.67218566908451,43.67223134764015,43.67227703309824,43.67232272130444,43.67237292971048,43.67241863725634,43.67245994639086,43.67250567465715,43.67255141397431,43.67259715465121,43.6726384041552,43.67268417523464,43.67272093769056,43.67276220787862,43.67280799963854,43.67284928222819,43.67289517360858,43.67293197606225,43.67296879094109,43.67301011493069,43.67305143750578,43.67309278495632,43.67314807902602,43.67320336056772,43.6732586448013,43.67331850733735,43.67337826874453,43.67343351955823,43.67349327113946,43.67355752072239,43.67361275057824,43.67367698066046,43.67373679014458,43.67380099796689,43.67386070626642,43.67389743608296,43.67393865427184,43.67397987243745,43.67402106984311,43.67405767778065,43.67409886270116,43.67414003930668,43.6741813058807,43.6742224713872,43.67426362581624,43.67430927000849,43.67435040643204,43.67439152901675,43.67443715242376,43.67448276613953,43.67452385688757,43.6745739479176,43.67461511895054,43.67466070081902,43.67471076140336,43.6747518202942,43.67479736066669,43.67484289964452,43.67489291734266,43.674942935029,43.67498844357405,43.67509307800643,43.67522845048442,43.67534703961631,43.67564052347077,43.67629911195922,43.67645966553873,43.6765487477648,43.67672714784764,43.6768331197118,43.6769988519654,43.67726157871029,43.67736650689433,43.67749804446057,43.67795267544879,43.67817614540946,43.67880480972727,43.67904595170656,43.67927866907515,43.679515770481,43.6797486323867,43.67995599877234,43.68014629650328,43.67958497899963,43.67953851020246,43.67867825311365,43.67798104246441,43.67692611971048,43.67616999434693,43.675160290965,43.67502900098071,43.67439970098565,43.67428547830498,43.67413643571545,43.6739684875341,43.67395753272378,43.67393249776637,43.67389887284944,43.67378334126624,43.67370132614361,43.67368847340409,43.67331673118881,43.67324856981993,43.6729856351758]}]],[[{"lng":[-79.43316565772852,-79.43299809831832,-79.43336947639089,-79.43353850528344,-79.43370455407097,-79.43387051523547,-79.43403635920936,-79.43420172364493,-79.43436686630463,-79.4345322280047,-79.43469675682032,-79.43486211812544,-79.43502664657177,-79.43519037155419,-79.43594203734493,-79.43627547591433,-79.43660986082352,-79.4366233863104,-79.4392787097228,-79.44326109411325,-79.44443351661712,-79.44561821179634,-79.44671962063487,-79.44679134767574,-79.44682551038173,-79.44693325421153,-79.44704056140988,-79.44714747336246,-79.44725549613507,-79.44736383934011,-79.44747135078116,-79.44757937337602,-79.44768843207271,-79.44779665903002,-79.44790548566034,-79.44801442820589,-79.44812407268221,-79.44823289918244,-79.44834277543946,-79.44845148591908,-79.44856113022404,-79.44867149018101,-79.4487811344042,-79.44889137838926,-79.44900398986871,-79.44911421787951,-79.4491500504375,-79.44918610120686,-79.44922266096712,-79.44926002746165,-79.44929894132378,-79.44933795753167,-79.44937779178854,-79.44941905755813,-79.44946020758536,-79.4495030072312,-79.44954601147802,-79.44958962930251,-79.4496347785874,-79.44968004614564,-79.44972611818336,-79.44977270156602,-79.44982020527439,-79.44986770679256,-79.44991766463222,-79.44996681821644,-79.45001832356432,-79.45006982904216,-79.45012111659783,-79.45017425595286,-79.45022799767924,-79.45028175309884,-79.45033644241435,-79.45039263846895,-79.45044956162005,-79.450505771506,-79.45136835702425,-79.45108933696622,-79.44829276516222,-79.44747994244486,-79.44722235480917,-79.44603783085564,-79.44560394012547,-79.44509098137469,-79.44474023157576,-79.44375557645458,-79.44210048519331,-79.44098508304808,-79.43979664731071,-79.43951430556535,-79.43838430540882,-79.43833671477772,-79.43808029802037,-79.43764165288935,-79.43746075195403,-79.43731659091004,-79.43713024303877,-79.43591733651529,-79.4346395518008,-79.43316565772852],"lat":[43.67070875690955,43.67026535972635,43.6702028459905,43.6701691548137,43.67013992356168,43.67010618984835,43.67007695530649,43.67003862189518,43.67000487624502,43.66996663233711,43.66993287780056,43.66989463340864,43.66986087839093,43.66982261111535,43.6696573609585,43.6695943185214,43.66952237603657,43.66951355902423,43.66900063798407,43.66821758423094,43.66800390644372,43.66778597252384,43.66758480841978,43.66756777917791,43.66756374244724,43.66753820055044,43.66752165466264,43.66750060228139,43.66748406591435,43.66746303277625,43.667446489258,43.66742995258185,43.66740892885451,43.66739239474843,43.66738036970607,43.66736384511179,43.66735183096736,43.66733980561099,43.667318792359,43.66731126624127,43.66729925167463,43.66728724671711,43.667275231938,43.66726772621613,43.66725566147511,43.66752280770495,43.66760881338373,43.66769032098914,43.66777192550943,43.66785795196531,43.66793949838188,43.66802104617292,43.66810260504907,43.66818418333281,43.66827026105214,43.66835186011432,43.66843346193529,43.66851507206312,43.66859679296936,43.66867392441939,43.66875556778493,43.66883721806717,43.66891888081302,43.66900063352895,43.66908232951988,43.66915951355883,43.66923672947171,43.66931394536233,43.66939565929451,43.66947298731486,43.66955473447872,43.66963198078127,43.66970473870037,43.66978651906432,43.66986389825801,43.66994117773149,43.67103329455463,43.67109252756343,43.67154988305189,43.67168737754614,43.67172438715149,43.67192892676401,43.6720040444789,43.67208708683047,43.67215883140281,43.67237056812703,43.67272628361513,43.6729856351758,43.67324856981993,43.67331673118881,43.67368847340409,43.67370132614361,43.67378334126624,43.67389887284944,43.67393249776637,43.67395753272378,43.6739684875341,43.67413643571545,43.67428547830498,43.67070875690955]}]],[[{"lng":[-79.43316565772852,-79.4346395518008,-79.43377131395067,-79.43342961049703,-79.43260953194411,-79.43216122107962,-79.43148003117653,-79.43097268630068,-79.43021562787625,-79.42868158628205,-79.42831712851368,-79.42762898430234,-79.42715167551805,-79.42672032722351,-79.42625184692604,-79.42535307754392,-79.42511042314729,-79.42406358938395,-79.42308270906518,-79.42297950898538,-79.42271318721414,-79.42270163970075,-79.4225773881786,-79.42241318053246,-79.42237197808123,-79.42215054374074,-79.42184150710601,-79.42291844511576,-79.42696153477263,-79.42938567233485,-79.43293496791368,-79.43299809831832,-79.43316565772852],"lat":[43.67070875690955,43.67428547830498,43.67439970098565,43.67444902966261,43.67456381363309,43.67459817329119,43.67467434533059,43.67471698211887,43.67472909074974,43.67474851798973,43.67472100124793,43.67476554756706,43.67478598553609,43.67485215440529,43.67492222130992,43.67507646863781,43.67510463204785,43.67531525185111,43.67555386822041,43.67541282272516,43.67480592144936,43.67477425497414,43.67439436478785,43.67397791476798,43.67382881250045,43.67322694041562,43.67241231041669,43.67220662727424,43.6714339825421,43.67096762069997,43.6702779971491,43.67026535972635,43.67070875690955]}]],[[{"lng":[-79.42297401049193,-79.42284201876933,-79.4208949569955,-79.41920074994938,-79.41811838688221,-79.41719534219682,-79.41591566412801,-79.41569766598845,-79.41483743916822,-79.41468130745143,-79.41673801731623,-79.41709352524065,-79.41961442438605,-79.42180815817346,-79.42184150710601,-79.42215054374074,-79.42237197808123,-79.42241318053246,-79.4225773881786,-79.42270163970075,-79.42271318721414,-79.42297401049193],"lat":[43.67540014401069,43.67551084787019,43.67605856007666,43.67651197488697,43.67659367975482,43.67665012486106,43.67659521575629,43.67652566886569,43.67433227925183,43.67390251718238,43.67346724478075,43.67339115115848,43.67287232506234,43.6724163509035,43.67241231041669,43.67322694041562,43.67382881250045,43.67397791476798,43.67439436478785,43.67477425497414,43.67480592144936,43.67540014401069]}]],[[{"lng":[-79.41569766598845,-79.41591566412801,-79.41719534219682,-79.41811838688221,-79.41920074994938,-79.4208949569955,-79.42284201876933,-79.42297401049193,-79.42297950898538,-79.42308270906518,-79.42316621667926,-79.42330039126179,-79.42339140673199,-79.42347575830613,-79.4235004930016,-79.42355103844548,-79.42362122672598,-79.42368000931998,-79.42375259479897,-79.42381160006667,-79.42392327346072,-79.42404295474427,-79.42426618817048,-79.42499599030744,-79.42504876468745,-79.42535707498108,-79.42570872075457,-79.42617382376118,-79.42631002597257,-79.42638920399919,-79.42627704359407,-79.42618209164723,-79.42469770756945,-79.42456823239242,-79.42464422538698,-79.42384841024707,-79.42353656752216,-79.42353397090777,-79.42341616831018,-79.42218823065591,-79.42082044111989,-79.41976400787753,-79.41852956548682,-79.41842014396309,-79.41829000051877,-79.41826160531426,-79.41780013635801,-79.41768602680546,-79.41765516259404,-79.41746780307541,-79.41707867609085,-79.41697045862047,-79.41674670971246,-79.41656677095865,-79.41642653551609,-79.41577483624475,-79.41569766598845],"lat":[43.67652566886569,43.67659521575629,43.67665012486106,43.67659367975482,43.67651197488697,43.67605856007666,43.67551084787019,43.67540014401069,43.67541282272516,43.67555386822041,43.67571255594429,43.67603406914865,43.67626937750465,43.67650918488489,43.67663105344935,43.67688839845746,43.67711441727521,43.67732677568376,43.67748081127392,43.67759874143758,43.67777582107301,43.67794400900091,43.67821273692186,43.67998305271634,43.68010080630641,43.68085240257623,43.68170361712217,43.68162000305664,43.68195504340886,43.68215867910759,43.6821796396889,43.68195328163814,43.68228849959506,43.68231822183848,43.6824948087671,43.68265487406421,43.68271817788155,43.6827001380281,43.68272551897175,43.68297863262062,43.6832703996871,43.68348993897409,43.68375192528437,43.68348935257809,43.68315888782104,43.6830909796964,43.68191856050002,43.68162441505528,43.68156097358817,43.68104517419808,43.68009889400042,43.679818242385,43.679243424474,43.67877273634655,43.67832501310575,43.67672235265001,43.67652566886569]}]],[[{"lng":[-79.35464656864448,-79.35476986896008,-79.35653495348453,-79.35774698080155,-79.35891130447253,-79.35957135260308,-79.36017677389826,-79.36050068207473,-79.36100126291419,-79.36145271651047,-79.36225117645441,-79.3632111100456,-79.36342946625452,-79.36444990467039,-79.36498970823509,-79.36548816272061,-79.36600332544009,-79.36522828279942,-79.36515666588737,-79.36451742819638,-79.36398323524594,-79.36374462654271,-79.3630210331726,-79.36100762327487,-79.36047413335649,-79.36042972669462,-79.36014668528571,-79.36001039819939,-79.35963357448483,-79.35949941338727,-79.35814922644948,-79.35662225325376,-79.35637721237316,-79.35607698922136,-79.35587652009283,-79.35566580747171,-79.35543705891679,-79.35543143725914,-79.35540179895301,-79.35536473697189,-79.3553264071826,-79.35528100665637,-79.35524288167403,-79.35520604452823,-79.35519680724347,-79.35515714923018,-79.35510773963449,-79.35506278936516,-79.35501867528706,-79.35501233833023,-79.3549729051613,-79.35492866861102,-79.3548837191432,-79.35483535269864,-79.3547911374053,-79.35475606292404,-79.35471947344152,-79.35471736807864,-79.35469560003376,-79.35466383844346,-79.35463231918686,-79.35458959690413,-79.35454722412065,-79.35450593330778,-79.35446815593725,-79.35443758052804,-79.35441158207179,-79.3543823941651,-79.35433212810963,-79.35431156344174,-79.35454138203646,-79.35464656864448],"lat":[43.65779730336225,43.65777207236302,43.65736519122122,43.65709004827623,43.65682762047388,43.65669756513609,43.65655312907798,43.65647225691166,43.65636241057666,43.65626536074995,43.65608767305169,43.65587187335879,43.65582539923416,43.65561045553927,43.65693724897775,43.65816442578931,43.65943234827975,43.65960130498387,43.65961378356546,43.65974875994709,43.65987164348724,43.65991773863331,43.66010092000402,43.66052234410784,43.66066772637234,43.66067609132925,43.66073504452794,43.66076009495026,43.66084920815287,43.66087878949761,43.66119256514156,43.66155329392071,43.66161728521725,43.66165347611385,43.66167309731385,43.66169707179139,43.66172987839052,43.6617072924094,43.66157174537656,43.66142717948586,43.66129159736841,43.66113781938689,43.66100224018314,43.66087568153344,43.66083945042826,43.6607038491258,43.66055460439796,43.66041433584628,43.66026966831296,43.66024707201026,43.66012947793939,43.65998930962016,43.65984904099799,43.65971781522216,43.65957314612917,43.65944202164459,43.65930646433788,43.65929743199113,43.65918459314382,43.6590445142815,43.65891803197243,43.6587823863171,43.65866015871109,43.65852453362984,43.65840255219755,43.65827149237362,43.65814508948344,43.65802314167752,43.65790539135792,43.65786458605156,43.65781829401421,43.65779730336225]}]],[[{"lng":[-79.42308270906518,-79.42406358938395,-79.42511042314729,-79.42535307754392,-79.42625184692604,-79.42672032722351,-79.42715167551805,-79.42762898430234,-79.42831712851368,-79.42868158628205,-79.43021562787625,-79.43097268630068,-79.43148003117653,-79.43216122107962,-79.43260953194411,-79.43342961049703,-79.43377131395067,-79.43403174163628,-79.43408724629691,-79.43450380851353,-79.43481598767957,-79.43525461924087,-79.43554420560477,-79.43590106833334,-79.4356787939342,-79.43590542868095,-79.43490423632693,-79.43393661144331,-79.43387122055037,-79.43280465253027,-79.43268636285698,-79.43252038634985,-79.4324491062208,-79.43240060109412,-79.43093650964673,-79.42908156776099,-79.42847108470787,-79.4279096300334,-79.42638920399919,-79.42631002597257,-79.42617382376118,-79.42570872075457,-79.42535707498108,-79.42504876468745,-79.42499599030744,-79.42426618817048,-79.42404295474427,-79.42392327346072,-79.42381160006667,-79.42375259479897,-79.42368000931998,-79.42362122672598,-79.42355103844548,-79.4235004930016,-79.42347575830613,-79.42339140673199,-79.42330039126179,-79.42316621667926,-79.42308270906518],"lat":[43.67555386822041,43.67531525185111,43.67510463204785,43.67507646863781,43.67492222130992,43.67485215440529,43.67478598553609,43.67476554756706,43.67472100124793,43.67474851798973,43.67472909074974,43.67471698211887,43.67467434533059,43.67459817329119,43.67456381363309,43.67444902966261,43.67439970098565,43.67502900098071,43.675160290965,43.67616999434693,43.67692611971048,43.67798104246441,43.67867825311365,43.67953851020246,43.67958497899963,43.68014629650328,43.68036223110506,43.68056952486923,43.68058213179705,43.68080165303613,43.68079102879281,43.68082025946448,43.68085078884342,43.68089063260603,43.68119929409307,43.68158799395844,43.68171913188938,43.68183743798339,43.68215867910759,43.68195504340886,43.68162000305664,43.68170361712217,43.68085240257623,43.68010080630641,43.67998305271634,43.67821273692186,43.67794400900091,43.67777582107301,43.67759874143758,43.67748081127392,43.67732677568376,43.67711441727521,43.67688839845746,43.67663105344935,43.67650918488489,43.67626937750465,43.67603406914865,43.67571255594429,43.67555386822041]}]],[[{"lng":[-79.43590542868095,-79.43680058006395,-79.43777125606091,-79.43884652175043,-79.43994828186069,-79.44104224743559,-79.4421727380592,-79.44243941449943,-79.44354526057661,-79.4444175379007,-79.44493843392839,-79.44575944116467,-79.44610646136481,-79.44502016991071,-79.44396107272627,-79.44283553439624,-79.44160851834644,-79.44061526135003,-79.43965685685428,-79.43868266901926,-79.43840538147673,-79.43829633616672,-79.43821427812377,-79.43807991432021,-79.43805289763888,-79.4373382094744,-79.43691140921524,-79.43658832089032,-79.43604438485735,-79.43590542868095],"lat":[43.68014629650328,43.67995599877234,43.6797486323867,43.679515770481,43.67927866907515,43.67904595170656,43.67880480972727,43.67874993141657,43.68144364025039,43.68360285906925,43.68487950299846,43.68470162936161,43.68556172010896,43.68579449999887,43.68602772983515,43.68624645149615,43.68649987637254,43.68670237586322,43.68690993359521,43.68712617963005,43.68636613174665,43.68611708623177,43.68590441748201,43.68554691152433,43.68545652211505,43.6837134134086,43.68267666539602,43.68186188401341,43.68049936713467,43.68014629650328]}]],[[{"lng":[-79.43183099233703,-79.43126258899204,-79.43124171837127,-79.43093930783748,-79.43073200892655,-79.43057375246916,-79.43032075692381,-79.42994304706193,-79.42992750907955,-79.42992572048934,-79.42993972661495,-79.42987669188273,-79.42984452964167,-79.42954464780362,-79.42908156776099,-79.43093650964673,-79.43240060109412,-79.4324491062208,-79.43252038634985,-79.43268636285698,-79.43280465253027,-79.43387122055037,-79.43393661144331,-79.43490423632693,-79.43590542868095,-79.43604438485735,-79.43658832089032,-79.43691140921524,-79.4373382094744,-79.43805289763888,-79.43807991432021,-79.43821427812377,-79.43829633616672,-79.43840538147673,-79.43868266901926,-79.43910870880985,-79.43925937915236,-79.43936551451954,-79.43953792095289,-79.43959900427012,-79.43964676057026,-79.43994657506835,-79.44001176706705,-79.44031262708265,-79.44057921480216,-79.44062825969003,-79.4408335235005,-79.44029199664662,-79.43976710690312,-79.4388918825681,-79.43868024253793,-79.43780066960127,-79.43750915074015,-79.43673859563407,-79.43575513511298,-79.43489436572891,-79.43477015701399,-79.43464124369612,-79.434466874595,-79.43420410648901,-79.43333549862852,-79.43298827055777,-79.43251531456364,-79.43216666733794,-79.43196189265848,-79.43183099233703],"lat":[43.6888063019808,43.68731280899259,43.68722250237838,43.6864800056421,43.68595495246398,43.68550708950707,43.68488679676469,43.68394521849817,43.68388199091132,43.6838054491966,43.68371562145396,43.68354362708479,43.68346666805267,43.68274229601926,43.68158799395844,43.68119929409307,43.68089063260603,43.68085078884342,43.68082025946448,43.68079102879281,43.68080165303613,43.68058213179705,43.68056952486923,43.68036223110506,43.68014629650328,43.68049936713467,43.68186188401341,43.68267666539602,43.6837134134086,43.68545652211505,43.68554691152433,43.68590441748201,43.68611708623177,43.68636613174665,43.68712617963005,43.68816299999132,43.68860624641497,43.68899937338108,43.68955544134208,43.68977241366247,43.68990359486777,43.69060102282276,43.69078195292275,43.69147489309694,43.69214026870808,43.6922534630284,43.69264794100272,43.69232998223622,43.69202566102674,43.6915320110826,43.69143909972993,43.69096779698856,43.69081077744109,43.69041306307609,43.68990890638695,43.68945593262566,43.68940472042552,43.68935344364871,43.68931504704744,43.68925302402211,43.68904298536363,43.68894370359139,43.68885169833904,43.68881090766844,43.68881259914384,43.6888063019808]}]],[[{"lng":[-79.43600679130638,-79.43536568641906,-79.4347854561985,-79.43473546957064,-79.43455058643156,-79.43437327021937,-79.43405387527834,-79.43359229489066,-79.43283202548278,-79.43229546633073,-79.43191731919053,-79.43069868945456,-79.43024076790449,-79.42949044999706,-79.42948567787731,-79.42925674471081,-79.42925690650529,-79.43004282880983,-79.42895169715577,-79.42826464150095,-79.42776165191982,-79.42608283942266,-79.42402233790104,-79.42137904371037,-79.42125301387443,-79.42071872387712,-79.4202904007266,-79.4199519836835,-79.41960637640808,-79.41940271606029,-79.41758652577205,-79.41462120790673,-79.41286379794913,-79.41280995031417,-79.41259009205149,-79.41188401905632,-79.41245226541362,-79.41345713767724,-79.41339694524143,-79.41350841449751,-79.41828371863102,-79.41839773212341,-79.41852956548682,-79.41976400787753,-79.42082044111989,-79.42218823065591,-79.42341616831018,-79.42353397090777,-79.42353656752216,-79.42384841024707,-79.42464422538698,-79.42456823239242,-79.42469770756945,-79.42618209164723,-79.42627704359407,-79.42638920399919,-79.4279096300334,-79.42847108470787,-79.42908156776099,-79.42954464780362,-79.42984452964167,-79.42987669188273,-79.42993972661495,-79.42992572048934,-79.42992750907955,-79.42994304706193,-79.43032075692381,-79.43057375246916,-79.43073200892655,-79.43093930783748,-79.43124171837127,-79.43126258899204,-79.43183099233703,-79.43196189265848,-79.43216666733794,-79.43251531456364,-79.43298827055777,-79.43333549862852,-79.4333144371234,-79.43332151284902,-79.43334426931702,-79.43340318806176,-79.4341428395292,-79.43422275831608,-79.43425966486024,-79.43431363790437,-79.43440778986717,-79.43452267723535,-79.43461614474469,-79.43467411900352,-79.43471551105634,-79.43478388933541,-79.4357295077421,-79.43596424697728,-79.43656712203314,-79.43715344354145,-79.43717761439241,-79.43644950896811,-79.43600679130638],"lat":[43.6987631196495,43.69716972227113,43.69573919071499,43.69563498306244,43.69553198827038,43.69543314774579,43.69524981180779,43.69499404047919,43.69457753677302,43.69427410157469,43.69406447798975,43.69338826337151,43.6929462772553,43.69272271084941,43.69272129496687,43.69266017566138,43.69229730710739,43.69174431115353,43.69116435027241,43.69133629338825,43.69150437343482,43.69122929817031,43.69138229168369,43.69077382123167,43.69046088091189,43.6891350647462,43.68808014079471,43.6872020873761,43.68636444245797,43.68587992821386,43.68626897676472,43.68685819882045,43.68721188950817,43.68722473468294,43.68668138038389,43.68493391244413,43.68482028062264,43.68461809729936,43.68446422742814,43.68443876891355,43.68348746444565,43.68378160788377,43.68375192528437,43.68348993897409,43.6832703996871,43.68297863262062,43.68272551897175,43.6827001380281,43.68271817788155,43.68265487406421,43.6824948087671,43.68231822183848,43.68228849959506,43.68195328163814,43.6821796396889,43.68215867910759,43.68183743798339,43.68171913188938,43.68158799395844,43.68274229601926,43.68346666805267,43.68354362708479,43.68371562145396,43.6838054491966,43.68388199091132,43.68394521849817,43.68488679676469,43.68550708950707,43.68595495246398,43.6864800056421,43.68722250237838,43.68731280899259,43.6888063019808,43.68881259914384,43.68881090766844,43.68885169833904,43.68894370359139,43.68904298536363,43.68916422376062,43.68931294393918,43.68945278705148,43.68960662890219,43.69145803660727,43.69163467074576,43.69170269145188,43.69177544696878,43.691884850947,43.69203495790038,43.69216676732086,43.69226667350702,43.69237076352029,43.69254273817084,43.69486974079343,43.69550318380641,43.69692960828711,43.69845464483774,43.69850898736988,43.69867915776899,43.6987631196495]}]],[[{"lng":[-79.42137904371037,-79.42402233790104,-79.42608283942266,-79.42776165191982,-79.42826464150095,-79.42895169715577,-79.43004282880983,-79.42925690650529,-79.42925674471081,-79.42948567787731,-79.42949044999706,-79.43024076790449,-79.43069868945456,-79.43191731919053,-79.43229546633073,-79.43283202548278,-79.43359229489066,-79.43405387527834,-79.43437327021937,-79.43455058643156,-79.43473546957064,-79.4347854561985,-79.43536568641906,-79.43600679130638,-79.43502133543768,-79.43479226621031,-79.43381512772916,-79.43285673737452,-79.43277318712188,-79.43188386984262,-79.43102761892919,-79.43089505252505,-79.42989128052452,-79.42881939577838,-79.42775374547892,-79.42760077548009,-79.42667439233998,-79.42554968469251,-79.42522878414262,-79.42475412758377,-79.42433014422323,-79.42403377318591,-79.42356540789795,-79.42332303240309,-79.423228523329,-79.42273532655746,-79.42257097389378,-79.42231447029317,-79.42186157664634,-79.42167557511874,-79.42158826649055,-79.42137904371037],"lat":[43.69077382123167,43.69138229168369,43.69122929817031,43.69150437343482,43.69133629338825,43.69116435027241,43.69174431115353,43.69229730710739,43.69266017566138,43.69272129496687,43.69272271084941,43.6929462772553,43.69338826337151,43.69406447798975,43.69427410157469,43.69457753677302,43.69499404047919,43.69524981180779,43.69543314774579,43.69553198827038,43.69563498306244,43.69573919071499,43.69716972227113,43.6987631196495,43.69898368054826,43.69902105232823,43.69922839376477,43.69942680189309,43.69944825111227,43.6996385963221,43.69982038658669,43.69984566245287,43.70005692551678,43.70028983829914,43.70052273675648,43.70056114104393,43.70075102562203,43.70098759138143,43.70020890849509,43.69901833457403,43.69794098210639,43.69726615668368,43.69610276008944,43.69549619246902,43.69528775252635,43.69409691355914,43.69370746902406,43.69305119302986,43.69192842132992,43.69150266840831,43.69128541445136,43.69077382123167]}]],[[{"lng":[-79.44935582184597,-79.45026776020714,-79.44767418232749,-79.44732188090097,-79.44530253779401,-79.44436506168007,-79.44278190589527,-79.44024473011275,-79.4393786649678,-79.43923750097893,-79.43904342209751,-79.43840546130755,-79.43831319290788,-79.4372821444324,-79.43717761439241,-79.43715344354145,-79.43656712203314,-79.43596424697728,-79.4357295077421,-79.43478388933541,-79.43471551105634,-79.43467411900352,-79.43461614474469,-79.43452267723535,-79.43440778986717,-79.43431363790437,-79.43425966486024,-79.43422275831608,-79.4341428395292,-79.43340318806176,-79.43334426931702,-79.43332151284902,-79.4333144371234,-79.43333549862852,-79.43420410648901,-79.434466874595,-79.43464124369612,-79.43477015701399,-79.43489436572891,-79.43575513511298,-79.43673859563407,-79.43750915074015,-79.43780066960127,-79.43868024253793,-79.4388918825681,-79.43976710690312,-79.44029199664662,-79.4408335235005,-79.44122866026642,-79.44156555389637,-79.44182068940576,-79.44204458321353,-79.44219446310798,-79.44238891609494,-79.44258454905176,-79.44309654818093,-79.44327310063584,-79.44459294996689,-79.44569161197062,-79.44608613560619,-79.44715937659917,-79.44935582184597],"lat":[43.69534342098972,43.69568445445915,43.69625256804471,43.69635130500559,43.69675603707155,43.69696382847187,43.69729343468718,43.69783964278649,43.69803486482279,43.69806444395676,43.69810230215135,43.69823770832208,43.69826795444829,43.69848341083574,43.69850898736988,43.69845464483774,43.69692960828711,43.69550318380641,43.69486974079343,43.69254273817084,43.69237076352029,43.69226667350702,43.69216676732086,43.69203495790038,43.691884850947,43.69177544696878,43.69170269145188,43.69163467074576,43.69145803660727,43.68960662890219,43.68945278705148,43.68931294393918,43.68916422376062,43.68904298536363,43.68925302402211,43.68931504704744,43.68935344364871,43.68940472042552,43.68945593262566,43.68990890638695,43.69041306307609,43.69081077744109,43.69096779698856,43.69143909972993,43.6915320110826,43.69202566102674,43.69232998223622,43.69264794100272,43.69284246347573,43.69298658928095,43.6930755873338,43.6931461548974,43.69320230013693,43.69326787467611,43.69331555097414,43.69346655948299,43.69353197866526,43.69395963136959,43.6942717356266,43.69437612287556,43.69471037236035,43.69534342098972]}]],[[{"lng":[-79.44740894134485,-79.44807669767155,-79.44829406288787,-79.44879460400534,-79.44894415624137,-79.44974146706326,-79.45026776020714,-79.44935582184597,-79.44715937659917,-79.44608613560619,-79.44569161197062,-79.44459294996689,-79.44327310063584,-79.44309654818093,-79.44258454905176,-79.44238891609494,-79.44219446310798,-79.44204458321353,-79.44182068940576,-79.44156555389637,-79.44122866026642,-79.4408335235005,-79.44062825969003,-79.44057921480216,-79.44031262708265,-79.44001176706705,-79.43994657506835,-79.43964676057026,-79.43959900427012,-79.43953792095289,-79.43936551451954,-79.43925937915236,-79.43910870880985,-79.43868266901926,-79.43965685685428,-79.44061526135003,-79.44160851834644,-79.44283553439624,-79.44396107272627,-79.44502016991071,-79.44610646136481,-79.44641221846344,-79.44681009993502,-79.4473649393598,-79.44740894134485],"lat":[43.68873075815799,43.69038296543232,43.69094863270472,43.69217096928581,43.69253767051178,43.69443476960333,43.69568445445915,43.69534342098972,43.69471037236035,43.69437612287556,43.6942717356266,43.69395963136959,43.69353197866526,43.69346655948299,43.69331555097414,43.69326787467611,43.69320230013693,43.6931461548974,43.6930755873338,43.69298658928095,43.69284246347573,43.69264794100272,43.6922534630284,43.69214026870808,43.69147489309694,43.69078195292275,43.69060102282276,43.68990359486777,43.68977241366247,43.68955544134208,43.68899937338108,43.68860624641497,43.68816299999132,43.68712617963005,43.68690993359521,43.68670237586322,43.68649987637254,43.68624645149615,43.68602772983515,43.68579449999887,43.68556172010896,43.6862411184058,43.68726393451825,43.68860413216486,43.68873075815799]}]],[[{"lng":[-79.43888659592514,-79.43882195129036,-79.44067371086309,-79.43999638493645,-79.43923750097893,-79.4393786649678,-79.44024473011275,-79.44278190589527,-79.44436506168007,-79.44530253779401,-79.44732188090097,-79.44767418232749,-79.45026776020714,-79.45035169305135,-79.45045092879198,-79.45055927343688,-79.45066219296362,-79.45073203221513,-79.45077249831284,-79.45079139601671,-79.45087330724076,-79.45096365884369,-79.45113024418171,-79.45129413820973,-79.45142202303228,-79.4515450486785,-79.45158131531753,-79.45175687449262,-79.45191840136103,-79.45210994550904,-79.45225365815998,-79.44916683404374,-79.44726251136711,-79.44484000494316,-79.44226778332413,-79.44100433650192,-79.44009853873821,-79.43967580975909,-79.43959203562916,-79.43938321566951,-79.43898319200193,-79.43888659592514],"lat":[43.70229598958992,43.70203845970182,43.70160905131961,43.6999385792459,43.69806444395676,43.69803486482279,43.69783964278649,43.69729343468718,43.69696382847187,43.69675603707155,43.69635130500559,43.69625256804471,43.69568445445915,43.69573510364981,43.69579946328073,43.69589554335818,43.69602746777349,43.69617244663961,43.69632161828898,43.69639839149531,43.6967461689867,43.69717948980277,43.69791099980631,43.6986875730291,43.69926111301325,43.69978498595614,43.70000602629477,43.70075124967297,43.70146927662248,43.70234524474687,43.70296841880666,43.70364690969478,43.70406673945497,43.70460551468916,43.70517379473112,43.7054041202297,43.70557639034848,43.70560212622701,43.70524540387238,43.70435567649329,43.7026709817699,43.70229598958992]}]],[[{"lng":[-79.45240916316179,-79.452465241469,-79.4526102643936,-79.45291254658794,-79.45298474349843,-79.45316911098482,-79.45338673281827,-79.45344190619083,-79.45365954083742,-79.45377936188522,-79.45383819269011,-79.45397381164356,-79.45401325801959,-79.45433754451534,-79.4544944228337,-79.45473985655367,-79.45465083671917,-79.45447888300608,-79.45433896693419,-79.45420507674575,-79.45409688260966,-79.45393587704149,-79.45378616696267,-79.45363578933485,-79.45348963363111,-79.45334866712345,-79.45322419008667,-79.45311692993947,-79.45300072741642,-79.45285290735266,-79.45271878485778,-79.4526029160052,-79.45250556738317,-79.45241869604816,-79.44945940528275,-79.44839711915431,-79.44651007848671,-79.44606089037229,-79.44497014630295,-79.44445437227623,-79.44424744224914,-79.44388952347133,-79.44387194898428,-79.44326678648318,-79.44304195146485,-79.44299157078058,-79.44294005059454,-79.44288731698985,-79.44283376737819,-79.44277929478648,-79.44272492471922,-79.44266890590555,-79.44261228528997,-79.44255544603232,-79.44251287431968,-79.44244978226405,-79.44240013683147,-79.44235037498969,-79.44229989707947,-79.44224788458143,-79.44219566993074,-79.44214294152067,-79.44208847390634,-79.44203410876025,-79.44197902756541,-79.44192231182691,-79.44146378094328,-79.44143099345101,-79.44033762999629,-79.44030933160381,-79.43957492762391,-79.43952989512144,-79.43948590023075,-79.43944158406072,-79.43939912396756,-79.43935646168261,-79.43931553638016,-79.43927428978111,-79.4392348992567,-79.43919622496486,-79.43915755075321,-79.43912042569751,-79.43908410704215,-79.43904778612639,-79.43901523615622,-79.43967580975909,-79.44009853873821,-79.44100433650192,-79.44226778332413,-79.44484000494316,-79.44726251136711,-79.44916683404374,-79.45225365815998,-79.45240916316179],"lat":[43.7035828403499,43.70381765155995,43.70451744971818,43.70576399705747,43.70615214997434,43.70697400661719,43.70789983576354,43.70816614104242,43.70905596173587,43.7097059059172,43.70998135242032,43.71065392498363,43.71080758235204,43.71234707677499,43.71306503593265,43.71408116752363,43.71407996375522,43.71409564227368,43.71410734289696,43.71412353582843,43.714135575226,43.71415140102931,43.71415387647338,43.71417434664213,43.71418587177124,43.714206558942,43.71422287812248,43.71423042824453,43.71425136017562,43.71427186393513,43.71429705395443,43.71430898802142,43.71432117276677,43.71431999635247,43.71496412224606,43.71520635015251,43.71560839678699,43.71569681028598,43.71593411829134,43.7160441242949,43.71549659796471,43.71457784432419,43.71453709612412,43.71291714016066,43.71231085462976,43.71219314249833,43.71207991580556,43.71196217156487,43.71184432615892,43.71172655816849,43.71160879154718,43.71149541339898,43.71137761605037,43.71126431668144,43.7111692156952,43.71106024202025,43.71096954556663,43.7108833484934,43.71079714163679,43.71071091383831,43.71062459323959,43.7105383556321,43.71045209428907,43.71036583431604,43.71027956455372,43.71019318245832,43.70949818826194,43.70944823019583,43.70778559022924,43.707740194089,43.70662265421289,43.70656352639383,43.70649991173293,43.70644079365996,43.70637719992359,43.70631351338979,43.7062499405813,43.70619086436248,43.70612731248502,43.70606377037574,43.70600022825391,43.70593220628474,43.70586860629079,43.70580509627371,43.70574613874738,43.70560212622701,43.70557639034848,43.7054041202297,43.70517379473112,43.70460551468916,43.70406673945497,43.70364690969478,43.70296841880666,43.7035828403499]}]],[[{"lng":[-79.45640016465312,-79.45213565018025,-79.45099789303232,-79.44867802779282,-79.44729835016108,-79.44689277378521,-79.44590589132973,-79.44559372719246,-79.4455278900431,-79.4454114365266,-79.44501313160514,-79.44460053415058,-79.44445437227623,-79.44497014630295,-79.44606089037229,-79.44651007848671,-79.44839711915431,-79.44945940528275,-79.45241869604816,-79.45250556738317,-79.4526029160052,-79.45271878485778,-79.45285290735266,-79.45300072741642,-79.45311692993947,-79.45322419008667,-79.45334866712345,-79.45348963363111,-79.45363578933485,-79.45378616696267,-79.45393587704149,-79.45409688260966,-79.45420507674575,-79.45433896693419,-79.45447888300608,-79.45465083671917,-79.45473985655367,-79.45518878323175,-79.45528163542781,-79.4555052002427,-79.45566936793028,-79.45616133261036,-79.45619576558235,-79.45640016465312],"lat":[43.72161287141042,43.72251857835202,43.72276880966852,43.72327304712384,43.72354246201495,43.72245194398707,43.71989502177465,43.71912092778906,43.71893549188599,43.7186727606199,43.71760042802067,43.7164242866346,43.7160441242949,43.71593411829134,43.71569681028598,43.71560839678699,43.71520635015251,43.71496412224606,43.71431999635247,43.71432117276677,43.71430898802142,43.71429705395443,43.71427186393513,43.71425136017562,43.71423042824453,43.71422287812248,43.714206558942,43.71418587177124,43.71417434664213,43.71415387647338,43.71415140102931,43.714135575226,43.71412353582843,43.71410734289696,43.71409564227368,43.71407996375522,43.71408116752363,43.71614913965276,43.71655107267809,43.71763451128278,43.71824904409294,43.72045712319823,43.72071873546908,43.72161287141042]}]],[[{"lng":[-79.35543705891679,-79.35566580747171,-79.35587652009283,-79.35607698922136,-79.35637721237316,-79.35662225325376,-79.35814922644948,-79.35949941338727,-79.35963357448483,-79.36001039819939,-79.36014668528571,-79.36042972669462,-79.36047413335649,-79.36100762327487,-79.3630210331726,-79.36374462654271,-79.36398323524594,-79.36451742819638,-79.36515666588737,-79.36522828279942,-79.36600332544009,-79.36651260221024,-79.36702277427345,-79.3660324763046,-79.36596535455757,-79.36402507041474,-79.3631714087045,-79.36232103361399,-79.36148551394174,-79.36031390359655,-79.35962217133688,-79.35923197877999,-79.35891003372882,-79.35685721338926,-79.35669852747021,-79.35647739731876,-79.35629090356707,-79.35625381667964,-79.35621697552935,-79.35618661811722,-79.35614366390909,-79.35611436775164,-79.35607364483329,-79.35603349185016,-79.35598442503054,-79.35594615615526,-79.35590134294094,-79.35586769289054,-79.35584224050481,-79.35582464101718,-79.35579180911219,-79.3557469966604,-79.35570416719987,-79.35565109767528,-79.35560049923801,-79.35559827081522,-79.35555638509184,-79.35550909961708,-79.35546649622799,-79.35543705891679],"lat":[43.66172987839052,43.66169707179139,43.66167309731385,43.66165347611385,43.66161728521725,43.66155329392071,43.66119256514156,43.66087878949761,43.66084920815287,43.66076009495026,43.66073504452794,43.66067609132925,43.66066772637234,43.66052234410784,43.66010092000402,43.65991773863331,43.65987164348724,43.65974875994709,43.65961378356546,43.65960130498387,43.65943234827975,43.6606955930494,43.66194543508158,43.66216533336033,43.66217787670985,43.66258230525238,43.66278622052294,43.66296308037683,43.66315382996661,43.6634205900304,43.66357729074529,43.6637022187864,43.66379211742312,43.66437043747015,43.66441766693281,43.66448200221777,43.66452883119474,43.66441127138607,43.66428471306828,43.66417166102229,43.66404051373163,43.66391856491786,43.66379186068691,43.66367425668859,43.66353852038178,43.66341185140136,43.66326267319928,43.66313165964013,43.66304127314908,43.66297791570555,43.66284691388916,43.66269773562507,43.66256208891969,43.6624082007716,43.66227244223715,43.66226790915615,43.66212768492968,43.66198297198775,43.66184282741573,43.66172987839052]}]],[[{"lng":[-79.45668617642528,-79.45152558338101,-79.44926384106542,-79.44909832212511,-79.44909136243348,-79.44905084093645,-79.44902130200161,-79.44899907148397,-79.44858368279483,-79.44857858757864,-79.44857278977652,-79.44856615942219,-79.44855872414254,-79.44855045631337,-79.44854066948422,-79.44853096887383,-79.44852046334714,-79.44850830653144,-79.44849625207436,-79.44848257396583,-79.44846888205782,-79.44845365502742,-79.44843760927604,-79.44842166588825,-79.44840420120789,-79.44838590631235,-79.44812734679857,-79.44807425391085,-79.44780859295201,-79.44777306524723,-79.44729835016108,-79.44867802779282,-79.45099789303232,-79.45213565018025,-79.45640016465312,-79.45646099405799,-79.45662218024754,-79.45665348217256,-79.45676432621451,-79.45686135177864,-79.45689230822576,-79.45707022099133,-79.45758148069999,-79.45764200677043,-79.45777654708442,-79.45787826023626,-79.45784089097273,-79.45668617642528],"lat":[43.72831104329099,43.72942966935394,43.72992129250363,43.72995955502446,43.72995946056879,43.7297562766453,43.72957583666531,43.72944941757034,43.72716592412134,43.72712534614086,43.72708025764278,43.72703965881965,43.72699454808922,43.72695392703351,43.72690869435585,43.72686805384587,43.72682290142791,43.72678222756807,43.72674155509643,43.72669635959993,43.72665566489303,43.72661494934128,43.72657422267112,43.72653349738829,43.72648825046824,43.72644740323202,43.72575965336841,43.72561931195904,43.72491346092771,43.72481845776811,43.72354246201495,43.72327304712384,43.72276880966852,43.72251857835202,43.72161287141042,43.72184324320467,43.72257466951,43.7227191237051,43.72323841366544,43.72368991213014,43.72384786451503,43.72474164116915,43.72668441467861,43.72698679724418,43.72766844095363,43.72806148878693,43.72806548546581,43.72831104329099]}]],[[{"lng":[-79.44807425391085,-79.44812734679857,-79.44838590631235,-79.44840420120789,-79.44842166588825,-79.44843760927604,-79.44845365502742,-79.44846888205782,-79.44848257396583,-79.44849625207436,-79.44850830653144,-79.44852046334714,-79.44853096887383,-79.44854066948422,-79.44855045631337,-79.44855872414254,-79.44856615942219,-79.44857278977652,-79.44857858757864,-79.44858368279483,-79.44899907148397,-79.44902130200161,-79.44905084093645,-79.44909136243348,-79.44890815079462,-79.44889176124673,-79.44873495530524,-79.44853430254227,-79.4484883203788,-79.44847418244626,-79.44647185470231,-79.44645546484652,-79.44395359252979,-79.44386786886621,-79.44378264272602,-79.44369763308337,-79.44361250918176,-79.44352750177127,-79.44344319655987,-79.44335878893983,-79.44327520006344,-79.44319161112908,-79.44310780320461,-79.44302513530128,-79.44294204369496,-79.4428600921236,-79.44277720508461,-79.44269585323332,-79.44261358015825,-79.44253304697376,-79.44245169491603,-79.44237105924421,-79.44229102331512,-79.44221118967468,-79.44213127024206,-79.44205215523714,-79.44197375660956,-79.44188297343794,-79.44179290663348,-79.44170364421036,-79.4416142936788,-79.44152574751658,-79.44143720125315,-79.44134855253725,-79.44126082488195,-79.44117370890002,-79.44108659515288,-79.44099958365757,-79.44091327412572,-79.4408277976992,-79.44074199970899,-79.44065712277312,-79.44057234807892,-79.44048828973771,-79.4404042312874,-79.44032088919187,-79.4402384658205,-79.44015582551832,-79.44007412072752,-79.43999229901041,-79.43991202693331,-79.43819430695643,-79.43701885913757,-79.4369903456524,-79.43625205843708,-79.43369461170577,-79.43365320039705,-79.43360199286225,-79.4334173875537,-79.43338305046672,-79.43334104163965,-79.43318946062722,-79.43316421151683,-79.43300341515562,-79.43295646291578,-79.43275062488121,-79.43259325450184,-79.43252413293776,-79.43233763312807,-79.43214995786971,-79.43201561859051,-79.43194273935907,-79.43181983310338,-79.43172653548321,-79.43161153360005,-79.43151020520915,-79.43310332707939,-79.43635478191293,-79.43959707547262,-79.44283211622813,-79.44566322402156,-79.44689814192104,-79.44729835016108,-79.44777306524723,-79.44780859295201,-79.44807425391085],"lat":[43.72561931195904,43.72575965336841,43.72644740323202,43.72648825046824,43.72653349738829,43.72657422267112,43.72661494934128,43.72665566489303,43.72669635959993,43.72674155509643,43.72678222756807,43.72682290142791,43.72686805384587,43.72690869435585,43.72695392703351,43.72699454808922,43.72703965881965,43.72708025764278,43.72712534614086,43.72716592412134,43.72944941757034,43.72957583666531,43.7297562766453,43.72995946056879,43.72999298171308,43.72999726022559,43.73003113950877,43.7300779262174,43.73008630387353,43.73009061289645,43.73051809782834,43.73052237598629,43.73105563872911,43.73107247531124,43.73109381957951,43.7311107557769,43.73113210131034,43.73114894739178,43.73117030394977,43.73119165905035,43.73121302524198,43.73123439137152,43.73126025543109,43.7312816339868,43.73130750768419,43.73132889588064,43.73135477224523,43.73138066947084,43.73140655408333,43.73143246234983,43.73145835939809,43.73148426615307,43.73151468200097,43.73154519056961,43.73157110691849,43.73160153515303,43.73163197309976,43.73166224212991,43.73169252085587,43.731727311454,43.73175759980605,43.73179240003243,43.73182720018844,43.73186199887805,43.73189681006222,43.7319317195421,43.73196653896622,43.73200135971856,43.7320406909557,43.73207553252015,43.73211487060668,43.73215422119681,43.73219357311873,43.73223293475355,43.73227229632461,43.73231166761069,43.73235114139513,43.73239502311287,43.73243441656708,43.73247830933999,43.73251772223621,43.73351615950586,43.73419341803002,43.73421103191842,43.73463761563016,43.73613778119752,43.73597058764609,43.73576284107632,43.73500846824263,43.73486846739372,43.73469676453373,43.73407796314048,43.73388398512835,43.73317503760558,43.7330437755259,43.73215408093588,43.73149469034731,43.73125960157403,43.73037476225108,43.7296293465855,43.72906479142132,43.72873062918493,43.7282067393809,43.72782728699556,43.72738902393044,43.72693294449191,43.72659913289754,43.72590992036741,43.72521580898465,43.72452609755214,43.72390740752678,43.72365855000136,43.72354246201495,43.72481845776811,43.72491346092771,43.72561931195904]}]],[[{"lng":[-79.4455278900431,-79.44559372719246,-79.44590589132973,-79.44689277378521,-79.44729835016108,-79.44689814192104,-79.44566322402156,-79.44283211622813,-79.43959707547262,-79.43635478191293,-79.43310332707939,-79.43151020520915,-79.43141751868789,-79.43124651171256,-79.43108243210919,-79.43105096174777,-79.43089749708683,-79.43072157224205,-79.43065103328722,-79.43052248977726,-79.43029936972265,-79.4301612649691,-79.43012554031121,-79.42996753824151,-79.42980479121373,-79.43296250765476,-79.43508912791549,-79.43767468443647,-79.43836117904164,-79.44132812382836,-79.44320777823282,-79.44332707181007,-79.44400976857311,-79.44445437227623,-79.44460053415058,-79.44501313160514,-79.4454114365266,-79.4455278900431],"lat":[43.71893549188599,43.71912092778906,43.71989502177465,43.72245194398707,43.72354246201495,43.72365855000136,43.72390740752678,43.72452609755214,43.72521580898465,43.72590992036741,43.72659913289754,43.72693294449191,43.72661651384972,43.72579030747107,43.72500479466552,43.72483773637062,43.72421440416674,43.72342413700648,43.72306750089735,43.72255253378793,43.72161767621307,43.72093595092523,43.72078692777301,43.72002841212741,43.71925182704731,43.71848481500648,43.718036762911,43.71749133377863,43.71731608677722,43.71668585634444,43.71631080077752,43.71628541994701,43.71613709310256,43.7160441242949,43.7164242866346,43.71760042802067,43.7186727606199,43.71893549188599]}]],[[{"lng":[-79.42771267523683,-79.42768214941763,-79.42747678470603,-79.42745156473582,-79.42726943063109,-79.42714083801597,-79.42727013814844,-79.43427763629099,-79.43614642344258,-79.43795802670495,-79.43853218799455,-79.43901523615622,-79.43904778612639,-79.43908410704215,-79.43912042569751,-79.43915755075321,-79.43919622496486,-79.4392348992567,-79.43927428978111,-79.43931553638016,-79.43935646168261,-79.43939912396756,-79.43944158406072,-79.43948590023075,-79.43952989512144,-79.43957492762391,-79.44030933160381,-79.44033762999629,-79.44143099345101,-79.44146378094328,-79.44192231182691,-79.44197902756541,-79.44203410876025,-79.44208847390634,-79.44214294152067,-79.44219566993074,-79.44224788458143,-79.44229989707947,-79.44235037498969,-79.44240013683147,-79.44244978226405,-79.44251287431968,-79.44255544603232,-79.44261228528997,-79.44266890590555,-79.44272492471922,-79.44277929478648,-79.44283376737819,-79.44288731698985,-79.44294005059454,-79.44299157078058,-79.44304195146485,-79.44326678648318,-79.44387194898428,-79.44388952347133,-79.44424744224914,-79.44445437227623,-79.44400976857311,-79.44332707181007,-79.44320777823282,-79.44132812382836,-79.43836117904164,-79.43767468443647,-79.43508912791549,-79.43296250765476,-79.42980479121373,-79.4295806444957,-79.42940200422434,-79.42921382601517,-79.42899417187863,-79.42877851877883,-79.42858742399548,-79.42835198557412,-79.42819964427866,-79.4281361048784,-79.42793417151783,-79.4279215604267,-79.42771267523683],"lat":[43.71061996772321,43.71048001707927,43.70973434809516,43.70959897137207,43.70876360179142,43.70828004599431,43.70822781367377,43.70676192509192,43.70637333290411,43.70597492647319,43.70585215433549,43.70574613874738,43.70580509627371,43.70586860629079,43.70593220628474,43.70600022825391,43.70606377037574,43.70612731248502,43.70619086436248,43.7062499405813,43.70631351338979,43.70637719992359,43.70644079365996,43.70649991173293,43.70656352639383,43.70662265421289,43.707740194089,43.70778559022924,43.70944823019583,43.70949818826194,43.71019318245832,43.71027956455372,43.71036583431604,43.71045209428907,43.7105383556321,43.71062459323959,43.71071091383831,43.71079714163679,43.7108833484934,43.71096954556663,43.71106024202025,43.7111692156952,43.71126431668144,43.71137761605037,43.71149541339898,43.71160879154718,43.71172655816849,43.71184432615892,43.71196217156487,43.71207991580556,43.71219314249833,43.71231085462976,43.71291714016066,43.71453709612412,43.71457784432419,43.71549659796471,43.7160441242949,43.71613709310256,43.71628541994701,43.71631080077752,43.71668585634444,43.71731608677722,43.71749133377863,43.718036762911,43.71848481500648,43.71925182704731,43.71841588509874,43.71762107690652,43.71680363231325,43.71594524573356,43.71505540673424,43.71419741181508,43.71323087295021,43.71259404851047,43.7123230252917,43.71152789366872,43.71141519548243,43.71061996772321]}]],[[{"lng":[-79.42714083801597,-79.42704702194396,-79.42684427234467,-79.4266257276519,-79.4263606627915,-79.42634854041714,-79.42627746119078,-79.42625024015095,-79.42613791189922,-79.42589937241507,-79.42587107376956,-79.42582182350729,-79.42574544121624,-79.42571102658094,-79.42570714157357,-79.4256977097491,-79.42568216500312,-79.42567273569711,-79.42566977389185,-79.42566340882161,-79.42564880396931,-79.42562499465546,-79.42561287003979,-79.42559990212952,-79.42554968469251,-79.42667439233998,-79.42760077548009,-79.42775374547892,-79.42881939577838,-79.42989128052452,-79.43089505252505,-79.43102761892919,-79.43188386984262,-79.43277318712188,-79.43285673737452,-79.43381512772916,-79.43479226621031,-79.43502133543768,-79.43600679130638,-79.43644950896811,-79.43717761439241,-79.4372821444324,-79.43831319290788,-79.43840546130755,-79.43904342209751,-79.43923750097893,-79.43999638493645,-79.44067371086309,-79.43882195129036,-79.43888659592514,-79.43898319200193,-79.43938321566951,-79.43959203562916,-79.43967580975909,-79.43901523615622,-79.43853218799455,-79.43795802670495,-79.43614642344258,-79.43427763629099,-79.42727013814844,-79.42714083801597],"lat":[43.70828004599431,43.70791867548394,43.70710102521404,43.70620664015855,43.70518558618804,43.70514040933393,43.70486477995495,43.70474737927969,43.70426863705385,43.70333346590635,43.7032025472888,43.70297672916236,43.70263801239398,43.70247100151642,43.70238092810999,43.70226818329665,43.70208792936927,43.70188516468964,43.70170508409551,43.70159247153571,43.7014662424732,43.70133979657939,43.70123610665279,43.70110990012884,43.70098759138143,43.70075102562203,43.70056114104393,43.70052273675648,43.70028983829914,43.70005692551678,43.69984566245287,43.69982038658669,43.6996385963221,43.69944825111227,43.69942680189309,43.69922839376477,43.69902105232823,43.69898368054826,43.6987631196495,43.69867915776899,43.69850898736988,43.69848341083574,43.69826795444829,43.69823770832208,43.69810230215135,43.69806444395676,43.6999385792459,43.70160905131961,43.70203845970182,43.70229598958992,43.7026709817699,43.70435567649329,43.70524540387238,43.70560212622701,43.70574613874738,43.70585215433549,43.70597492647319,43.70637333290411,43.70676192509192,43.70822781367377,43.70828004599431]}]],[[{"lng":[-79.41280995031417,-79.41286379794913,-79.41462120790673,-79.41758652577205,-79.41940271606029,-79.41960637640808,-79.4199519836835,-79.4202904007266,-79.42071872387712,-79.42125301387443,-79.42137904371037,-79.42158826649055,-79.42167557511874,-79.42186157664634,-79.42231447029317,-79.42257097389378,-79.42273532655746,-79.423228523329,-79.42332303240309,-79.42356540789795,-79.42403377318591,-79.42433014422323,-79.42475412758377,-79.42522878414262,-79.42554968469251,-79.42312770922652,-79.42208075323803,-79.42087867700195,-79.41987227188538,-79.41909621553008,-79.41808601370424,-79.41802097431835,-79.41796100947668,-79.41790797898355,-79.41767117204299,-79.417330597693,-79.41702501186849,-79.41666701056677,-79.41633021446785,-79.41602947871905,-79.41578057959552,-79.41572250753707,-79.41541755203066,-79.41508267141405,-79.41501483152915,-79.4144659730954,-79.41441295561567,-79.41410838729732,-79.41364117211891,-79.41333919755061,-79.41299636673654,-79.41265731632254,-79.41263107786665,-79.41261930705708,-79.41261402651334,-79.41259038180182,-79.41256294037562,-79.41243282006386,-79.41229751801043,-79.41226123253085,-79.41280995031417],"lat":[43.68722473468294,43.68721188950817,43.68685819882045,43.68626897676472,43.68587992821386,43.68636444245797,43.6872020873761,43.68808014079471,43.6891350647462,43.69046088091189,43.69077382123167,43.69128541445136,43.69150266840831,43.69192842132992,43.69305119302986,43.69370746902406,43.69409691355914,43.69528775252635,43.69549619246902,43.69610276008944,43.69726615668368,43.69794098210639,43.69901833457403,43.70020890849509,43.70098759138143,43.70150349864078,43.70173210102303,43.70199464836978,43.7021787808647,43.70234367638597,43.70137989712811,43.70131589284591,43.70123404481972,43.70112978779014,43.7005502917828,43.69975322037835,43.69901514548057,43.69820432753399,43.6974163072212,43.69670089163273,43.69608062472999,43.69595829272733,43.69519771687288,43.69445472909302,43.69431875823558,43.69292465661278,43.69282039801868,43.69204632069584,43.69096832770568,43.69021687916053,43.68942426332748,43.68866320590719,43.68850980734634,43.68842853576902,43.6883339413521,43.68819858281567,43.6880946785564,43.68776420772426,43.68742466272981,43.68733413852621,43.68722473468294]}]],[[{"lng":[-79.41188401905632,-79.41133404109995,-79.41115718652863,-79.41104858345302,-79.41022602174222,-79.41007896098274,-79.40999595481524,-79.40992072159939,-79.40956709578209,-79.40956730268594,-79.40393882551417,-79.40361978352919,-79.40368095366485,-79.4037176002073,-79.4037441204202,-79.40374790426242,-79.40374722267404,-79.40374406974263,-79.40373713312587,-79.4037263472209,-79.40373532947152,-79.40376222447145,-79.40381693442947,-79.40389143400323,-79.40396437989938,-79.4040192252794,-79.40405592159992,-79.40407964545729,-79.40410810935165,-79.4041254586936,-79.40412221982722,-79.40411741316262,-79.40409428745478,-79.40407075052553,-79.40402601215202,-79.40395560712645,-79.40390049228834,-79.4038716555177,-79.40380781249574,-79.40432733305711,-79.40486767928688,-79.40523667408874,-79.40539178333714,-79.40555551638526,-79.40577731442897,-79.40585920608127,-79.40600349193943,-79.40607795255094,-79.40616359299965,-79.4069658812664,-79.40811874320049,-79.40924381759091,-79.41012476825676,-79.41035458470286,-79.41221945923343,-79.4133559676753,-79.41471383073922,-79.41577483624475,-79.41642653551609,-79.41656677095865,-79.41674670971246,-79.41697045862047,-79.41707867609085,-79.41746780307541,-79.41765516259404,-79.41768602680546,-79.41780013635801,-79.41826160531426,-79.41829000051877,-79.41842014396309,-79.41852956548682,-79.41839773212341,-79.41828371863102,-79.41350841449751,-79.41339694524143,-79.41345713767724,-79.41245226541362,-79.41188401905632],"lat":[43.68493391244413,43.68504140396553,43.68461765114891,43.68453062207118,43.68249778016045,43.68213565368601,43.68193186347434,43.68175527741521,43.68092550184088,43.68092541470017,43.68076709720445,43.68075813995583,43.6807274873676,43.68069190117274,43.68062925749526,43.6805752982513,43.68051227461822,43.68045371747322,43.68039510747743,43.68027333957843,43.68016994185174,43.68006229331553,43.67992802722102,43.67980745049742,43.67966443700053,43.67949416461996,43.67933705176239,43.6792023527958,43.67900461586542,43.67885182382638,43.67870774629704,43.67859965494684,43.67846871270676,43.67836486079497,43.67826971481245,43.6781742103543,43.67809233241167,43.67806492360148,43.67801902180821,43.67795876175666,43.67790779204638,43.67781842065907,43.67787909819666,43.67793089382239,43.67796099418516,43.67796663755593,43.67796865015163,43.67796518769656,43.67795287909424,43.67772542192953,43.67740841176381,43.67710891716327,43.67686902963852,43.67683171749547,43.67679453834558,43.67680132085395,43.67673464564269,43.67672235265001,43.67832501310575,43.67877273634655,43.679243424474,43.679818242385,43.68009889400042,43.68104517419808,43.68156097358817,43.68162441505528,43.68191856050002,43.6830909796964,43.68315888782104,43.68348935257809,43.68375192528437,43.68378160788377,43.68348746444565,43.68443876891355,43.68446422742814,43.68461809729936,43.68482028062264,43.68493391244413]}]],[[{"lng":[-79.41483743916822,-79.41577483624475,-79.41471383073922,-79.4133559676753,-79.41221945923343,-79.41035458470286,-79.41012476825676,-79.40924381759091,-79.40811874320049,-79.4069658812664,-79.40616359299965,-79.40607795255094,-79.40600349193943,-79.40585920608127,-79.40577731442897,-79.40555551638526,-79.40539178333714,-79.40523667408874,-79.40486767928688,-79.40432733305711,-79.40380781249574,-79.4031913507319,-79.40306143696574,-79.40299870842452,-79.40299432680978,-79.40299805793745,-79.4029997745279,-79.40303644953202,-79.40303890266095,-79.40302001747726,-79.40296334420592,-79.40254266033796,-79.40243165192,-79.40231793035593,-79.39933320346751,-79.39905330493507,-79.39892727733334,-79.39872119666948,-79.398550561497,-79.39833934144951,-79.39812870351679,-79.39800019849307,-79.39793218295796,-79.39791901186771,-79.3979855841245,-79.39809551604205,-79.39820532804956,-79.39831606019904,-79.39842742316299,-79.39853875105959,-79.39865081202583,-79.39876297746613,-79.39887583872208,-79.39898931335824,-79.3991024135959,-79.39919006705568,-79.39927795970513,-79.39936655066202,-79.39945515888608,-79.39954434581159,-79.39963365226002,-79.39972389865351,-79.39981390334405,-79.3999046237393,-79.39999534404507,-79.40008710414293,-79.40017876188041,-79.40027019769526,-79.40036314996011,-79.40045622169582,-79.40054929333988,-79.40064224535813,-79.40073603261661,-79.40082980251861,-79.40092358959961,-79.40101887831364,-79.40112752677429,-79.40123614062949,-79.40134547016778,-79.40145573711609,-79.40156588447663,-79.40167591227444,-79.40178583772658,-79.40189692226433,-79.40200756330584,-79.40211934623102,-79.40223090737888,-79.40234236618015,-79.40245464293092,-79.40256632558024,-79.40267930079911,-79.4027915772737,-79.4028713023593,-79.40290487620589,-79.40301785118137,-79.40313166121121,-79.40495516829996,-79.40732217604464,-79.4078359077135,-79.41134952067497,-79.41217411318017,-79.41370453363464,-79.41466339723326,-79.41468130745143,-79.41483743916822],"lat":[43.67433227925183,43.67672235265001,43.67673464564269,43.67680132085395,43.67679453834558,43.67683171749547,43.67686902963852,43.67710891716327,43.67740841176381,43.67772542192953,43.67795287909424,43.67796518769656,43.67796865015163,43.67796663755593,43.67796099418516,43.67793089382239,43.67787909819666,43.67781842065907,43.67790779204638,43.67795876175666,43.67801902180821,43.67805091895055,43.67862541290411,43.67914674341603,43.67925020539167,43.67934477871849,43.67951143005219,43.67968298088577,43.67976403332032,43.67983578555298,43.67992060275636,43.68052695018378,43.68064692562379,43.68078045606562,43.68143481527793,43.68076177480916,43.68045385069537,43.67995576243499,43.67954819064725,43.67903202557925,43.67852487025996,43.67819890626217,43.67801782240344,43.677986130696,43.67796005788492,43.6779075867709,43.6778596148791,43.67781165577721,43.67775920439826,43.67771575443776,43.67766781363068,43.67761978415668,43.67757635535045,43.67753293502533,43.67748050732257,43.6774502281089,43.67741095015828,43.67737618293008,43.67733691486472,43.67730665685464,43.67727189943842,43.67723256408381,43.67720231731349,43.67717208049286,43.67714184359864,43.67710712017882,43.67707239525197,43.67704216815498,43.67701646322113,43.67698625887331,43.67695605444808,43.67693034928295,43.67690015472,43.67687446084734,43.67684426612739,43.67682300434125,43.67678401536661,43.67675402782514,43.67672405019167,43.67668958455495,43.67665961814942,43.67663415097585,43.67660868226475,43.67657422763475,43.67654876871651,43.67651882464586,43.67649337837571,43.67646793056553,43.67644249408053,43.67641245814936,43.67639153221725,43.67636609539647,43.67634920557229,43.67634067275407,43.6763197464831,43.67629433075744,43.67591461528419,43.67542091907403,43.67531555220272,43.67459009534616,43.67442592562477,43.67410510190521,43.67390676974484,43.67390251718238,43.67433227925183]}]],[[{"lng":[-79.40380781249574,-79.4038716555177,-79.40390049228834,-79.40395560712645,-79.40402601215202,-79.40407075052553,-79.40409428745478,-79.40411741316262,-79.40412221982722,-79.4041254586936,-79.40410810935165,-79.40407964545729,-79.40405592159992,-79.4040192252794,-79.40396437989938,-79.40389143400323,-79.40381693442947,-79.40376222447145,-79.40373532947152,-79.4037263472209,-79.40373713312587,-79.40374406974263,-79.40374722267404,-79.40374790426242,-79.4037441204202,-79.4037176002073,-79.40368095366485,-79.40361978352919,-79.40393882551417,-79.40956730268594,-79.40956709578209,-79.40992072159939,-79.40999595481524,-79.41007896098274,-79.41022602174222,-79.41104858345302,-79.41115718652863,-79.41133404109995,-79.41188401905632,-79.41259009205149,-79.41280995031417,-79.41226123253085,-79.41229751801043,-79.41243282006386,-79.41256294037562,-79.41259038180182,-79.41261402651334,-79.41261930705708,-79.41263107786665,-79.41265731632254,-79.40947857773844,-79.40950328611946,-79.40794228008035,-79.40639034049839,-79.40488609937312,-79.403005146678,-79.40304969016685,-79.40307209036868,-79.40307198909697,-79.40232629039119,-79.40148256503196,-79.40111392297075,-79.40082837097997,-79.40051775385426,-79.40019738402502,-79.40015689315855,-79.3997916782411,-79.39933320346751,-79.40231793035593,-79.40243165192,-79.40254266033796,-79.40296334420592,-79.40302001747726,-79.40303890266095,-79.40303644953202,-79.4029997745279,-79.40299805793745,-79.40299432680978,-79.40299870842452,-79.40306143696574,-79.4031913507319,-79.40380781249574],"lat":[43.67801902180821,43.67806492360148,43.67809233241167,43.6781742103543,43.67826971481245,43.67836486079497,43.67846871270676,43.67859965494684,43.67870774629704,43.67885182382638,43.67900461586542,43.6792023527958,43.67933705176239,43.67949416461996,43.67966443700053,43.67980745049742,43.67992802722102,43.68006229331553,43.68016994185174,43.68027333957843,43.68039510747743,43.68045371747322,43.68051227461822,43.6805752982513,43.68062925749526,43.68069190117274,43.6807274873676,43.68075813995583,43.68076709720445,43.68092541470017,43.68092550184088,43.68175527741521,43.68193186347434,43.68213565368601,43.68249778016045,43.68453062207118,43.68461765114891,43.68504140396553,43.68493391244413,43.68668138038389,43.68722473468294,43.68733413852621,43.68742466272981,43.68776420772426,43.6880946785564,43.68819858281567,43.6883339413521,43.68842853576902,43.68850980734634,43.68866320590719,43.6893348484181,43.68942971334916,43.68977714934594,43.6900751793801,43.69038735708293,43.69077526919279,43.69073538265468,43.69068168366972,43.6906276702472,43.6887128738142,43.68663466472935,43.68571112285325,43.68501838366817,43.68430278786192,43.68351944950817,43.68341085853671,43.68253687101222,43.68143481527793,43.68078045606562,43.68064692562379,43.68052695018378,43.67992060275636,43.67983578555298,43.67976403332032,43.67968298088577,43.67951143005219,43.67934477871849,43.67925020539167,43.67914674341603,43.67862541290411,43.67805091895055,43.67801902180821]}]],[[{"lng":[-79.40912499815217,-79.40781705919872,-79.40587112292431,-79.40559971318669,-79.40540029039576,-79.40526973062325,-79.40504061770011,-79.40466208200078,-79.40349452425282,-79.40308266327379,-79.40239162275932,-79.40236076390505,-79.40231077007211,-79.40217961833812,-79.40188314716701,-79.40157721863714,-79.40127259685514,-79.40111649870308,-79.40103406830448,-79.4008567917314,-79.40146212606504,-79.4015734571316,-79.40168223352421,-79.40174561367766,-79.40184566163471,-79.403005146678,-79.40488609937312,-79.40639034049839,-79.40794228008035,-79.40950328611946,-79.40947857773844,-79.41265731632254,-79.41299636673654,-79.41333919755061,-79.41364117211891,-79.41410838729732,-79.41441295561567,-79.4144659730954,-79.41501483152915,-79.41508267141405,-79.41541755203066,-79.41572250753707,-79.41578057959552,-79.41602947871905,-79.41633021446785,-79.41666701056677,-79.41702501186849,-79.417330597693,-79.41767117204299,-79.41790797898355,-79.41796100947668,-79.41802097431835,-79.41808601370424,-79.41909621553008,-79.41809222875656,-79.41469999717489,-79.41457024212146,-79.41061922181392,-79.40912499815217],"lat":[43.69925210811293,43.69882430206867,43.69817585618414,43.69747432545629,43.69694483638018,43.69654683680962,43.69597183272026,43.69501215747446,43.69525249911531,43.69534126546772,43.69548914169585,43.69510603550646,43.69488478769656,43.69456779417747,43.69372178191902,43.69291623551531,43.69209261246191,43.69174835262027,43.69163908528638,43.69119091572245,43.69105535278348,43.69107041303633,43.69103142544863,43.69101430787401,43.6909842000579,43.69077526919279,43.69038735708293,43.6900751793801,43.68977714934594,43.68942971334916,43.6893348484181,43.68866320590719,43.68942426332748,43.69021687916053,43.69096832770568,43.69204632069584,43.69282039801868,43.69292465661278,43.69431875823558,43.69445472909302,43.69519771687288,43.69595829272733,43.69608062472999,43.69670089163273,43.6974163072212,43.69820432753399,43.69901514548057,43.69975322037835,43.7005502917828,43.70112978779014,43.70123404481972,43.70131589284591,43.70137989712811,43.70234367638597,43.7025638345732,43.70112133846362,43.70107452843732,43.69974109314045,43.69925210811293]}]],[[{"lng":[-79.35629090356707,-79.35647739731876,-79.35669852747021,-79.35685721338926,-79.35891003372882,-79.35923197877999,-79.35962217133688,-79.36031390359655,-79.36148551394174,-79.36232103361399,-79.3631714087045,-79.36402507041474,-79.36596535455757,-79.3660324763046,-79.36702277427345,-79.36717834123654,-79.36753069800899,-79.36796897395776,-79.36804383639215,-79.36828590132312,-79.36861963084363,-79.36725150055283,-79.36561665709522,-79.36517760274778,-79.36377317062824,-79.36363899535161,-79.36307447416533,-79.361269150224,-79.36087543697138,-79.35834740296079,-79.358257371394,-79.35824675896988,-79.3580381281235,-79.35773220535779,-79.35767280420464,-79.3575951227283,-79.35754338984674,-79.35749435627665,-79.35745195055354,-79.3574140000562,-79.35737157163402,-79.35733235373222,-79.35730746963991,-79.35656400221063,-79.35651877661375,-79.35648475581753,-79.35645016285002,-79.35641696263478,-79.35638623423965,-79.35636151714962,-79.35633457404334,-79.35629090356707],"lat":[43.66452883119474,43.66448200221777,43.66441766693281,43.66437043747015,43.66379211742312,43.6637022187864,43.66357729074529,43.6634205900304,43.66315382996661,43.66296308037683,43.66278622052294,43.66258230525238,43.66217787670985,43.66216533336033,43.66194543508158,43.66232583336494,43.66319974351389,43.66427742521685,43.66445862494565,43.66505180448728,43.66586693178186,43.66615355245028,43.66652177189433,43.66661910531261,43.66692307895551,43.66695275493446,43.66707519972146,43.66746350986302,43.6675478838074,43.66868212971812,43.66882936976267,43.66884722135255,43.66888023207417,43.66892534736064,43.66885697854833,43.66878384597573,43.66871558730048,43.6686383653736,43.66854314461875,43.66845707992717,43.66836644984605,43.66826686387184,43.66818548796868,43.66550965899521,43.66538298022147,43.66526546459603,43.66513893870119,43.66502134483231,43.66489937550775,43.6647819936442,43.66465998872243,43.66452883119474]}]],[[{"lng":[-79.41061922181392,-79.41457024212146,-79.41469999717489,-79.41809222875656,-79.41736686598777,-79.41708130827887,-79.41594993169277,-79.41549230091437,-79.41493447654442,-79.41438142101279,-79.41387180663453,-79.41321982529141,-79.41206319653254,-79.41163336870564,-79.41094615244589,-79.41024584046627,-79.40985035009832,-79.40938670164525,-79.40936021714258,-79.40934926712964,-79.4093584526488,-79.40941587167323,-79.40912499815217,-79.41061922181392],"lat":[43.69974109314045,43.70107452843732,43.70112133846362,43.7025638345732,43.7027114160727,43.70277497590826,43.70302485755983,43.70311321409601,43.70324050801185,43.7033498613962,43.70346890705975,43.70360388766192,43.70383996714107,43.70393301547271,43.70407649530966,43.70233801403283,43.70131060603771,43.70015612999692,43.7000702424666,43.70001607811884,43.69996669503468,43.69990439029972,43.69925210811293,43.69974109314045]}]],[[{"lng":[-79.41735744113032,-79.41701761269283,-79.41666112241842,-79.41656780797085,-79.41647967725439,-79.41640532319306,-79.41633438612681,-79.41626167769149,-79.41621220381261,-79.41708730729694,-79.41702110240372,-79.41693874620964,-79.41688124875276,-79.41676181988558,-79.41657441089386,-79.41649231695116,-79.41639480827128,-79.41617853761647,-79.41594993169277,-79.41708130827887,-79.41736686598777,-79.41809222875656,-79.41909621553008,-79.41987227188538,-79.42087867700195,-79.42208075323803,-79.42312770922652,-79.42554968469251,-79.42559990212952,-79.42561287003979,-79.42562499465546,-79.42564880396931,-79.42566340882161,-79.42566977389185,-79.42567273569711,-79.42568216500312,-79.4256977097491,-79.42570714157357,-79.42571102658094,-79.42574544121624,-79.42582182350729,-79.42587107376956,-79.42589937241507,-79.42613791189922,-79.42625024015095,-79.42627746119078,-79.42634854041714,-79.4263606627915,-79.4266257276519,-79.42684427234467,-79.42704702194396,-79.42714083801597,-79.42699670028226,-79.42670653662434,-79.42599995026411,-79.42423780774837,-79.42127580242813,-79.41931689845968,-79.41769196717661,-79.41735744113032],"lat":[43.70994140683683,43.70909483510379,43.70818951854252,43.70796308611666,43.70774581743671,43.70759616441851,43.70749165861891,43.70739613020422,43.70735493563565,43.70715092303679,43.70701047528264,43.70683820679503,43.70666637264419,43.70613351118601,43.70541507698656,43.70499975823401,43.70466074264589,43.70383388386541,43.70302485755983,43.70277497590826,43.7027114160727,43.7025638345732,43.70234367638597,43.7021787808647,43.70199464836978,43.70173210102303,43.70150349864078,43.70098759138143,43.70110990012884,43.70123610665279,43.70133979657939,43.7014662424732,43.70159247153571,43.70170508409551,43.70188516468964,43.70208792936927,43.70226818329665,43.70238092810999,43.70247100151642,43.70263801239398,43.70297672916236,43.7032025472888,43.70333346590635,43.70426863705385,43.70474737927969,43.70486477995495,43.70514040933393,43.70518558618804,43.70620664015855,43.70710102521404,43.70791867548394,43.70828004599431,43.70833216395094,43.70839118338062,43.70853898798286,43.70891088045805,43.70953632527577,43.70995052316702,43.71049975081613,43.70994140683683]}]],[[{"lng":[-79.41419266130457,-79.41348752982238,-79.41466060744042,-79.41484569315946,-79.41516668353667,-79.41725013001857,-79.4184495818034,-79.42046497676006,-79.42220602805547,-79.42195064842689,-79.42153356008292,-79.42110491433657,-79.42060217597475,-79.42023369299233,-79.41988088160366,-79.41952405127675,-79.41931689845968,-79.42127580242813,-79.42423780774837,-79.42599995026411,-79.42670653662434,-79.42699670028226,-79.42714083801597,-79.42726943063109,-79.42745156473582,-79.42747678470603,-79.42768214941763,-79.42771267523683,-79.4279215604267,-79.42793417151783,-79.4281361048784,-79.42819964427866,-79.42835198557412,-79.42858742399548,-79.42877851877883,-79.42899417187863,-79.42921382601517,-79.42940200422434,-79.4295806444957,-79.42980479121373,-79.42828684065179,-79.42662590269309,-79.4264027396474,-79.42554314289887,-79.42437150291779,-79.42246505246113,-79.4214327412941,-79.41996998856952,-79.41848861669416,-79.41764616442686,-79.41674821047104,-79.41559758291535,-79.41419266130457],"lat":[43.72256131899529,43.71875639540384,43.71852062066851,43.7184852899621,43.71842411780249,43.71800730613936,43.71777176768901,43.71737195936279,43.71702233968331,43.71637058173019,43.71529331703424,43.71418897507188,43.71309251927041,43.71220055155193,43.71133580514924,43.71045299801448,43.70995052316702,43.70953632527577,43.70891088045805,43.70853898798286,43.70839118338062,43.70833216395094,43.70828004599431,43.70876360179142,43.70959897137207,43.70973434809516,43.71048001707927,43.71061996772321,43.71141519548243,43.71152789366872,43.7123230252917,43.71259404851047,43.71323087295021,43.71419741181508,43.71505540673424,43.71594524573356,43.71680363231325,43.71762107690652,43.71841588509874,43.71925182704731,43.71959112816067,43.71993294143014,43.71998388040612,43.72015658050793,43.7203925768117,43.72081196170581,43.72102734622325,43.72133120120237,43.72165737464972,43.72183034249885,43.72201153712587,43.72224773438002,43.72256131899529]}]],[[{"lng":[-79.41419266130457,-79.41559758291535,-79.41674821047104,-79.41764616442686,-79.41848861669416,-79.41996998856952,-79.4214327412941,-79.42246505246113,-79.42437150291779,-79.42554314289887,-79.4264027396474,-79.42662590269309,-79.42828684065179,-79.42980479121373,-79.42996753824151,-79.43012554031121,-79.4301612649691,-79.43029936972265,-79.43052248977726,-79.43065103328722,-79.43072157224205,-79.43089749708683,-79.43105096174777,-79.43108243210919,-79.43124651171256,-79.43141751868789,-79.43151020520915,-79.43161153360005,-79.43172653548321,-79.43181983310338,-79.43025754353923,-79.42890562416468,-79.4275615554871,-79.4263863333501,-79.42488520648314,-79.42337239953063,-79.42197182731768,-79.42063891014128,-79.41909939977401,-79.41660823378056,-79.41641894080367,-79.41622963999028,-79.41614270062141,-79.41600178023052,-79.4158636929355,-79.41572251716021,-79.41541789287425,-79.41508177907919,-79.41472956146148,-79.41455163281027,-79.4144847502865,-79.41433836106683,-79.41419266130457],"lat":[43.72256131899529,43.72224773438002,43.72201153712587,43.72183034249885,43.72165737464972,43.72133120120237,43.72102734622325,43.72081196170581,43.7203925768117,43.72015658050793,43.71998388040612,43.71993294143014,43.71959112816067,43.71925182704731,43.72002841212741,43.72078692777301,43.72093595092523,43.72161767621307,43.72255253378793,43.72306750089735,43.72342413700648,43.72421440416674,43.72483773637062,43.72500479466552,43.72579030747107,43.72661651384972,43.72693294449191,43.72738902393044,43.72782728699556,43.7282067393809,43.7285184515797,43.72878352412254,43.72908460644485,43.72936108233602,43.72966906217032,43.72997677106964,43.73026359498409,43.73054683579721,43.73086771186344,43.73137351750607,43.73033549071158,43.7293018742828,43.72883247800421,43.72829031778806,43.72773010270933,43.72737256851922,43.7266210028711,43.72580598592241,43.72491872886007,43.7244840765843,43.72410497652086,43.72332868714569,43.72256131899529]}]],[[{"lng":[-79.42281640259775,-79.42279529129176,-79.42276877027614,-79.42275673819368,-79.4227423516525,-79.42271676778194,-79.42269210524927,-79.4226679387926,-79.42264470945862,-79.42262219672307,-79.42259966821709,-79.42257807683562,-79.42255791862277,-79.42253776043086,-79.42251842119967,-79.4224998009238,-79.42248211541238,-79.42246523305664,-79.42244886255455,-79.42243341337399,-79.4224189013202,-79.42240416874152,-79.42223887801779,-79.421855327777,-79.42163603034859,-79.42142145072376,-79.42109936455216,-79.42082972905111,-79.42057283001168,-79.42050244325118,-79.42042027206769,-79.42030226477583,-79.42027266586237,-79.42025171034066,-79.42009201481484,-79.42004274419813,-79.41995894364058,-79.4198805594197,-79.41981091537681,-79.41969364059626,-79.41959594417375,-79.4194887128904,-79.41947658771554,-79.4194273151036,-79.41940155516679,-79.41689593917873,-79.41675822159841,-79.41660823378056,-79.41909939977401,-79.42063891014128,-79.42197182731768,-79.42337239953063,-79.42488520648314,-79.4263863333501,-79.4275615554871,-79.42890562416468,-79.43025754353923,-79.43181983310338,-79.43194273935907,-79.43201561859051,-79.43214995786971,-79.43233763312807,-79.43252413293776,-79.43259325450184,-79.43275062488121,-79.43295646291578,-79.43300341515562,-79.43316421151683,-79.43318946062722,-79.43334104163965,-79.43338305046672,-79.4334173875537,-79.43360199286225,-79.43365320039705,-79.43369461170577,-79.43367337871305,-79.43119765638001,-79.4310353014606,-79.42746632303927,-79.42665112734964,-79.42370557920137,-79.42310308573592,-79.4230984791496,-79.42306509918427,-79.42303315240821,-79.42300118989637,-79.42297016449176,-79.42296475473417,-79.42293893437868,-79.42290944455517,-79.42287995475843,-79.42285107919771,-79.42282312733954,-79.42281640259775],"lat":[43.74229977432957,43.74227697812204,43.74224960627794,43.74223593730911,43.74222223584091,43.74219037595479,43.74215852878027,43.74213118941285,43.74209936201089,43.74206754449587,43.74204022772366,43.74200842291813,43.74197663789239,43.74194485286326,43.74191307913595,43.74188122531105,43.74184497344159,43.7418132336221,43.74178150086632,43.74174978082734,43.74171357275844,43.74168186260834,43.74129240748307,43.7403822369171,43.73986600791173,43.73943536199083,43.73869266352228,43.73808571795222,43.7375553741724,43.73742837417807,43.73724719962255,43.73701142796759,43.73687148876186,43.73677667872308,43.73591460579078,43.7357517996556,43.73536796854408,43.73501571903372,43.73462317160048,43.73412626304954,43.73376483427877,43.73323655755719,43.73310586168642,43.73277201827386,43.73238898935912,43.73293961091275,43.73218595067511,43.73137351750607,43.73086771186344,43.73054683579721,43.73026359498409,43.72997677106964,43.72966906217032,43.72936108233602,43.72908460644485,43.72878352412254,43.7285184515797,43.7282067393809,43.72873062918493,43.72906479142132,43.7296293465855,43.73037476225108,43.73125960157403,43.73149469034731,43.73215408093588,43.7330437755259,43.73317503760558,43.73388398512835,43.73407796314048,43.73469676453373,43.73486846739372,43.73500846824263,43.73576284107632,43.73597058764609,43.73613778119752,43.73615099307116,43.73760262167492,43.73769491299478,43.73981577150821,43.74029983678444,43.74216351155886,43.74255137389396,43.74255131032467,43.74252384389883,43.74249639724151,43.74247345132279,43.74244601736311,43.74244144173923,43.74241858056968,43.74239116778598,43.74236375499495,43.7423363506738,43.74230886907495,43.74229977432957]}]],[[{"lng":[-79.40837774950745,-79.40832368104647,-79.40827185093583,-79.40826488767689,-79.40825323156987,-79.40824251384397,-79.40823249593296,-79.40822319473193,-79.40821471262539,-79.40820716889689,-79.40820032497579,-79.40819419775973,-79.4081889065334,-79.40818431510958,-79.40818163520923,-79.40817963820575,-79.40817740403722,-79.40817516748484,-79.40817376691652,-79.408174005489,-79.40816059012059,-79.4081496329908,-79.40813879516745,-79.40813266802182,-79.40811996944313,-79.40811220559524,-79.40810432008543,-79.40809819202711,-79.40809278066845,-79.40808714764437,-79.40808337444744,-79.40808021555959,-79.4080770421556,-79.4080831992454,-79.4080884179573,-79.40809303925636,-79.40809590311952,-79.4080988693549,-79.4081003191239,-79.40810093053378,-79.40810094454582,-79.40809991782609,-79.40809817442921,-79.40809583602886,-79.40809245452201,-79.40810153078736,-79.40811040227427,-79.4081192568464,-79.40812669496025,-79.40813321162636,-79.40813899471138,-79.40814408040049,-79.40814834464329,-79.40815189220865,-79.40815462071681,-79.4081565132727,-79.40815688700366,-79.40815726073306,-79.40815702017456,-79.40815514151963,-79.4081530412081,-79.40814952684904,-79.40814611248952,-79.40814116242383,-79.40813517166096,-79.40812858352004,-79.40812728636905,-79.40812281443178,-79.40809858056022,-79.40803885050171,-79.40796709230106,-79.40786531815047,-79.40782891621288,-79.40776142342307,-79.4076410165323,-79.40743958993819,-79.40725161469665,-79.40721910792981,-79.4071060129917,-79.40702417710361,-79.40698428201502,-79.40692403119031,-79.40684914193277,-79.40671415975115,-79.40648670151339,-79.40644222115591,-79.40624014239597,-79.40590228291785,-79.40560137446872,-79.4054120296167,-79.40512850517914,-79.40492121264577,-79.40502061816022,-79.40822619277316,-79.4088281713228,-79.4101163087045,-79.41122156305613,-79.41242455623357,-79.41631291970462,-79.41696852834154,-79.41689593917873,-79.41940155516679,-79.4194273151036,-79.41947658771554,-79.4194887128904,-79.41959594417375,-79.41969364059626,-79.41981091537681,-79.4198805594197,-79.41995894364058,-79.42004274419813,-79.42009201481484,-79.42025171034066,-79.42027266586237,-79.42030226477583,-79.42042027206769,-79.42050244325118,-79.42057283001168,-79.42082972905111,-79.42109936455216,-79.42142145072376,-79.42163603034859,-79.421855327777,-79.42223887801779,-79.42240416874152,-79.4224189013202,-79.42243341337399,-79.42244886255455,-79.42246523305664,-79.42248211541238,-79.4224998009238,-79.42251842119967,-79.42253776043086,-79.42255791862277,-79.42257807683562,-79.42259966821709,-79.42262219672307,-79.42264470945862,-79.4226679387926,-79.42269210524927,-79.42271676778194,-79.4227423516525,-79.42275673819368,-79.42276877027614,-79.42279529129176,-79.42281640259775,-79.42282312733954,-79.42285107919771,-79.42287995475843,-79.42290944455517,-79.42293893437868,-79.42296475473417,-79.42297016449176,-79.42300118989637,-79.42303315240821,-79.42306509918427,-79.4230984791496,-79.42310308573592,-79.42141981536861,-79.41984960493028,-79.41934220807079,-79.41926402363133,-79.41918747702942,-79.4191109463672,-79.41903521850763,-79.41896020713666,-79.41888519327581,-79.41881077981583,-79.41873730367469,-79.41866464640115,-79.41859187053451,-79.41851981115339,-79.41844868912021,-79.41837826746837,-79.41830774094839,-79.41823885465388,-79.41816986586363,-79.41810169594287,-79.41803434489367,-79.41796687520048,-79.41790105952064,-79.41783502281481,-79.41777002352077,-79.41770572693282,-79.41764122547177,-79.41678520730689,-79.4166816432294,-79.41657737595061,-79.4164722753258,-79.41636657625645,-79.41625993669237,-79.4161534177067,-79.41604526023183,-79.41593709993009,-79.41582810620741,-79.41571851406673,-79.41560820481757,-79.41549625467283,-79.41545369008452,-79.41541051115294,-79.4153665131187,-79.41532179835889,-79.41527638323137,-79.41523084929403,-79.41518377721795,-79.41513672380989,-79.41508893730195,-79.41504045043696,-79.41499184473888,-79.41494171965172,-79.41489159449029,-79.41484063382291,-79.41478907519011,-79.4147365926858,-79.4146843336447,-79.41463113670417,-79.41457722302383,-79.41452249022191,-79.41446787614137,-79.41441160747642,-79.414355355159,-79.41429836966874,-79.41424140053374,-79.41418371228245,-79.41412602633133,-79.41143805563222,-79.40837774950745],"lat":[43.75361096118208,43.75339416220802,43.75318180531723,43.7531547025878,43.75312303353165,43.75308687659022,43.75305523035162,43.75302359409586,43.75299196924935,43.75295585651822,43.75292425449064,43.75289266244649,43.75285658109176,43.75282501044095,43.7528024683298,43.75278893764323,43.75275730981038,43.75272577196326,43.75268974480577,43.75268074621746,43.75259954212713,43.75254537801521,43.75248671460783,43.7524551225592,43.75237392844676,43.75231521783972,43.75226109651199,43.75220249871047,43.75214391089261,43.75208982094217,43.75203125594451,43.75197269950432,43.75191855379857,43.75183312140263,43.75175217688889,43.75166672309451,43.75158574577224,43.75150476987502,43.75141918187452,43.75133817316955,43.75125265518412,43.7511716236559,43.75109058214245,43.75100494136258,43.75092387702603,43.75087899389752,43.75083410791517,43.75079372265338,43.75074881670081,43.75070389791086,43.75066346985814,43.75061844111308,43.75057349094327,43.75052853078916,43.75048355922458,43.75044307697176,43.75039807260289,43.75035306823367,43.75030805530668,43.75026301955962,43.75022248168207,43.75017733312865,43.75013227598704,43.75008719745144,43.75004660537532,43.7500015040188,43.74999248403267,43.74995641407175,43.74978495004873,43.74955006810595,43.74927441990064,43.74892183712982,43.74879989406587,43.74853339706908,43.74809953716351,43.74729537926414,43.74653641800702,43.74641443897862,43.74605269566017,43.7456823860588,43.74537567453007,43.74501016669108,43.74464445471172,43.74414287593608,43.74329793426466,43.74301366330526,43.74184932648079,43.74037252845067,43.73900876880658,43.73819127189721,43.7369718741027,43.73600550367949,43.73597538443332,43.73526373958079,43.73512809240275,43.73484436682836,43.73460318384103,43.73433625222302,43.73347623989386,43.73333229102172,43.73293961091275,43.73238898935912,43.73277201827386,43.73310586168642,43.73323655755719,43.73376483427877,43.73412626304954,43.73462317160048,43.73501571903372,43.73536796854408,43.7357517996556,43.73591460579078,43.73677667872308,43.73687148876186,43.73701142796759,43.73724719962255,43.73742837417807,43.7375553741724,43.73808571795222,43.73869266352228,43.73943536199083,43.73986600791173,43.7403822369171,43.74129240748307,43.74168186260834,43.74171357275844,43.74174978082734,43.74178150086632,43.7418132336221,43.74184497344159,43.74188122531105,43.74191307913595,43.74194485286326,43.74197663789239,43.74200842291813,43.74204022772366,43.74206754449587,43.74209936201089,43.74213118941285,43.74215852878027,43.74219037595479,43.74222223584091,43.74223593730911,43.74224960627794,43.74227697812204,43.74229977432957,43.74230886907495,43.7423363506738,43.74236375499495,43.74239116778598,43.74241858056968,43.74244144173923,43.74244601736311,43.74247345132279,43.74249639724151,43.74252384389883,43.74255131032467,43.74255137389396,43.74363564022747,43.74463583663466,43.74498448521779,43.74503291428461,43.74508136595514,43.74512531683188,43.74517377972634,43.74522225248332,43.7452708151753,43.74532379706842,43.74537229092073,43.74542079605717,43.74547380046778,43.7455268147474,43.74557534099219,43.74562837784823,43.74568150322786,43.7457345612486,43.74578761780833,43.74584068566178,43.74589376481043,43.74595134323793,43.74600444356339,43.74606204175167,43.74611524332312,43.74617286552964,43.74623048486228,43.74720001747492,43.74732911040285,43.74745378254813,43.74758285398798,43.74770741607385,43.7478365560033,43.74796110652184,43.74808563422597,43.7482102518046,43.7483392686642,43.74846377616007,43.7485882736047,43.74871283820563,43.74876175853616,43.7488106703312,43.74885957075054,43.74890846121298,43.74895284098522,43.74900172005055,43.7490506677838,43.74909502477774,43.74914387254299,43.74918820961447,43.74923704597568,43.74928136028094,43.74932567456275,43.74937447819084,43.74941877254154,43.74946314407079,43.74950292769772,43.74954719924685,43.74959146082642,43.74963571101487,43.7496754618653,43.74971969068914,43.74975941875304,43.74980363757335,43.74984335563322,43.74988315370366,43.74992286175738,43.75164575995702,43.75361096118208]}],[{"lng":[-79.40648670151339,-79.40525258991767,-79.40468155433838,-79.40525248754824,-79.40648670151339],"lat":[43.74329793426466,43.74356427837492,43.74366883166084,43.74356427694625,43.74329793426466]}]],[[{"lng":[-79.41572251716021,-79.4158636929355,-79.41600178023052,-79.41614270062141,-79.41622963999028,-79.41641894080367,-79.41660823378056,-79.41675822159841,-79.41689593917873,-79.41696852834154,-79.41631291970462,-79.41242455623357,-79.41122156305613,-79.4101163087045,-79.4088281713228,-79.40822619277316,-79.40502061816022,-79.40492121264577,-79.40482232706044,-79.40456226621298,-79.40454012521897,-79.40451929775787,-79.40444264203157,-79.40438447561232,-79.40420952230716,-79.40419587748488,-79.40399234458927,-79.40396760985001,-79.40375082469062,-79.40370435131302,-79.40354329353734,-79.40345511767237,-79.40333738744658,-79.4032824395962,-79.40311630641165,-79.40305414452374,-79.40293205310925,-79.40282694898322,-79.40250341804757,-79.40235204005178,-79.40218355351945,-79.40316614243737,-79.40356004193004,-79.40454240149279,-79.4055981949749,-79.40710823435724,-79.40878306916871,-79.41001603979713,-79.41133397348312,-79.41209687902555,-79.4131917620668,-79.41419266130457,-79.41433836106683,-79.4144847502865,-79.41455163281027,-79.41472956146148,-79.41508177907919,-79.41541789287425,-79.41572251716021],"lat":[43.72737256851922,43.72773010270933,43.72829031778806,43.72883247800421,43.7293018742828,43.73033549071158,43.73137351750607,43.73218595067511,43.73293961091275,43.73333229102172,43.73347623989386,43.73433625222302,43.73460318384103,43.73484436682836,43.73512809240275,43.73526373958079,43.73597538443332,43.73600550367949,43.73552242974009,43.73435727919071,43.73425794874415,43.7341631376006,43.73389191912792,43.73365705654143,43.73299279090335,43.73289357901709,43.73196785756569,43.73187758270659,43.73112721346489,43.73086091695279,43.73025994774761,43.72992555390105,43.72942421105202,43.72915329491714,43.72857025792891,43.72831274375684,43.72782934331447,43.72742719766478,43.72628374876085,43.72567382081263,43.72509525023714,43.72491085206358,43.72483533811022,43.72460141413708,43.72437759828016,43.72405649294041,43.72372415950149,43.72345775994665,43.72318343629289,43.72302750370043,43.72279056772682,43.72256131899529,43.72332868714569,43.72410497652086,43.7244840765843,43.72491872886007,43.72580598592241,43.7266210028711,43.72737256851922]}]],[[{"lng":[-79.41484569315946,-79.41466060744042,-79.41348752982238,-79.41419266130457,-79.4131917620668,-79.41209687902555,-79.41133397348312,-79.41001603979713,-79.40878306916871,-79.40710823435724,-79.4055981949749,-79.40454240149279,-79.40356004193004,-79.40316614243737,-79.40218355351945,-79.40217888971405,-79.40205584692515,-79.40192824752953,-79.40183147330791,-79.40179518364664,-79.40158320345598,-79.40139052578066,-79.40117732052595,-79.4011298703641,-79.40093709162787,-79.40083943460988,-79.40061450496347,-79.40047026100886,-79.40043144896519,-79.40021941039898,-79.40009694045273,-79.40002565374697,-79.3998280364227,-79.40165007520855,-79.40373866723939,-79.405709665651,-79.40592797395797,-79.40756925028863,-79.40927725034396,-79.41147943416414,-79.41165461477389,-79.41159948348398,-79.41215357065629,-79.41227694181011,-79.41441966820413,-79.41475478916402,-79.41484569315946],"lat":[43.7184852899621,43.71852062066851,43.71875639540384,43.72256131899529,43.72279056772682,43.72302750370043,43.72318343629289,43.72345775994665,43.72372415950149,43.72405649294041,43.72437759828016,43.72460141413708,43.72483533811022,43.72491085206358,43.72509525023714,43.72495106379162,43.72431911666058,43.72378162611699,43.72345170144143,43.72309552672869,43.72196254647896,43.72097395699665,43.71997607787876,43.71954322996499,43.71874367909572,43.71832813233256,43.71724447903146,43.7165987297686,43.71636854642281,43.71544719784512,43.71494128321808,43.7145622025966,43.71364555593574,43.71328388049061,43.71283578990085,43.71241311644658,43.71237115207614,43.71200686035377,43.71163447184345,43.71117890861868,43.71114191498901,43.7115047383739,43.71576217848995,43.71593943100478,43.71763085450009,43.71839616979626,43.7184852899621]}]],[[{"lng":[-79.41484569315946,-79.41475478916402,-79.41441966820413,-79.41227694181011,-79.41215357065629,-79.41159948348398,-79.41165461477389,-79.41339767235034,-79.41553121308756,-79.41735744113032,-79.41769196717661,-79.41931689845968,-79.41952405127675,-79.41988088160366,-79.42023369299233,-79.42060217597475,-79.42110491433657,-79.42153356008292,-79.42195064842689,-79.42220602805547,-79.42046497676006,-79.4184495818034,-79.41725013001857,-79.41516668353667,-79.41484569315946],"lat":[43.7184852899621,43.71839616979626,43.71763085450009,43.71593943100478,43.71576217848995,43.7115047383739,43.71114191498901,43.71077337275738,43.71032578017644,43.70994140683683,43.71049975081613,43.70995052316702,43.71045299801448,43.71133580514924,43.71220055155193,43.71309251927041,43.71418897507188,43.71529331703424,43.71637058173019,43.71702233968331,43.71737195936279,43.71777176768901,43.71800730613936,43.71842411780249,43.7184852899621]}]],[[{"lng":[-79.41735744113032,-79.41553121308756,-79.41339767235034,-79.41165461477389,-79.41147943416414,-79.41112037848028,-79.41077244875856,-79.410423436649,-79.41039561982531,-79.41034639844216,-79.4102593486501,-79.41020248050344,-79.41014832033497,-79.40995548552522,-79.40973447156786,-79.40974345093902,-79.4097267372954,-79.40968364090935,-79.40960939384601,-79.40952459069899,-79.40942042102111,-79.40928878621055,-79.40914618599422,-79.40901952885936,-79.4089609565671,-79.40889861582194,-79.40880307459413,-79.40870529840116,-79.40862803129518,-79.40855795756249,-79.4085046025814,-79.4096101049808,-79.41094615244589,-79.41163336870564,-79.41206319653254,-79.41321982529141,-79.41387180663453,-79.41438142101279,-79.41493447654442,-79.41549230091437,-79.41594993169277,-79.41617853761647,-79.41639480827128,-79.41649231695116,-79.41657441089386,-79.41676181988558,-79.41688124875276,-79.41693874620964,-79.41702110240372,-79.41708730729694,-79.41621220381261,-79.41626167769149,-79.41633438612681,-79.41640532319306,-79.41647967725439,-79.41656780797085,-79.41666112241842,-79.41701761269283,-79.41735744113032],"lat":[43.70994140683683,43.71032578017644,43.71077337275738,43.71114191498901,43.71117890861868,43.71032305036223,43.70944934164983,43.70856661473486,43.70850771495224,43.70837191059067,43.70817265614554,43.70794672556413,43.70770741967425,43.70689896964305,43.70607653406125,43.70600014218621,43.70592780373833,43.70582368122601,43.7057326281458,43.7056189231478,43.70554095645338,43.70547602030786,43.70541552237078,43.70535974720974,43.70531842282529,43.70527254495104,43.70518569584569,43.70504021263927,43.70488160199728,43.70468708354887,43.70459172967294,43.70435506480411,43.70407649530966,43.70393301547271,43.70383996714107,43.70360388766192,43.70346890705975,43.7033498613962,43.70324050801185,43.70311321409601,43.70302485755983,43.70383388386541,43.70466074264589,43.70499975823401,43.70541507698656,43.70613351118601,43.70666637264419,43.70683820679503,43.70701047528264,43.70715092303679,43.70735493563565,43.70739613020422,43.70749165861891,43.70759616441851,43.70774581743671,43.70796308611666,43.70818951854252,43.70909483510379,43.70994140683683]}]],[[{"lng":[-79.31888151367416,-79.31900931787014,-79.31912977351543,-79.31923550770105,-79.3193183165593,-79.31941346621082,-79.3194367035358,-79.32147223245312,-79.32151366516551,-79.32155688174306,-79.32160482307037,-79.3221036585644,-79.32225430836662,-79.32278599860115,-79.32294055950395,-79.32295446744492,-79.32298074287131,-79.32305348825632,-79.32409204876423,-79.3263228944669,-79.32707441245719,-79.3271416472319,-79.32772877496147,-79.32784782918401,-79.32820274399978,-79.32828749865166,-79.3283340445376,-79.32841089322322,-79.32842333075148,-79.32422631156331,-79.32194283841298,-79.31952249152323,-79.31723658943426,-79.31485497236476,-79.31128306812478,-79.30761974715807,-79.30723174250446,-79.30677872935991,-79.30629764030942,-79.30595648405709,-79.3056745921652,-79.30544763753963,-79.30524019047131,-79.30488430470433,-79.30472654860321,-79.30451659249077,-79.30448678069368,-79.30444726139976,-79.30405589501295,-79.30444949545675,-79.30464497913714,-79.30483328419311,-79.30501092055181,-79.30520029266732,-79.30536186051219,-79.30557747942829,-79.30579578468705,-79.30600688465694,-79.30623245170315,-79.30643074515075,-79.30659661907751,-79.30675602247778,-79.30689278699704,-79.30702571218889,-79.30715132739131,-79.30728987361418,-79.30741737874925,-79.30753312440119,-79.30762433829726,-79.30768861273748,-79.30774341214004,-79.30775128228846,-79.30774380263348,-79.30770350117096,-79.30768638044745,-79.30765653828928,-79.30761125808661,-79.30749644795434,-79.30733815328655,-79.30715558152218,-79.30694440396927,-79.30674231929237,-79.30655587135423,-79.30639690701908,-79.30628254067024,-79.30632874629858,-79.30639912264139,-79.30637187896376,-79.30625914749992,-79.30608609418695,-79.30587766164422,-79.30566125086565,-79.30545456612739,-79.30527519009189,-79.30512006827867,-79.305005853051,-79.30495675367904,-79.3049988559802,-79.30509160896295,-79.30520953445442,-79.30536921391693,-79.30552575828182,-79.30562488596999,-79.30572116309439,-79.30581745911618,-79.30588541286967,-79.30597227931059,-79.30600512227799,-79.30597860245679,-79.30596784878831,-79.30591861486467,-79.3057888647736,-79.30563677319834,-79.30547809430543,-79.30532245582643,-79.30517912812287,-79.30504858561024,-79.30495680577452,-79.30495118809723,-79.30508098949818,-79.30527770747513,-79.30549406500432,-79.30571506119261,-79.30594474592775,-79.30616506643898,-79.30637102349944,-79.30657584622664,-79.30675193579742,-79.30691743796598,-79.30706885295056,-79.3072223549883,-79.30737499046083,-79.30749389133111,-79.30761290801891,-79.30770002768126,-79.30777117930859,-79.3078744770614,-79.30795675817731,-79.30802593682019,-79.30809211796959,-79.30818508197989,-79.30828326092065,-79.3083732413338,-79.30837400733967,-79.30833690301414,-79.30829169169789,-79.30830650923293,-79.30848469328765,-79.30864340506686,-79.30872811774805,-79.30879088744651,-79.30886682466762,-79.30893654537464,-79.30903249047057,-79.3091158602806,-79.30919164852735,-79.3092824721131,-79.30941357097159,-79.3095422110509,-79.30969964586964,-79.30988019323671,-79.31006349437038,-79.31021318465727,-79.31022807653304,-79.31027211082126,-79.31045344391677,-79.31065831163788,-79.31083564891757,-79.31102839703372,-79.31116821154986,-79.31124337893007,-79.31129330354409,-79.31127742812451,-79.31120888774005,-79.31107294738375,-79.31090670984986,-79.31076327725464,-79.31062515987094,-79.31047567893539,-79.3103098193809,-79.310172706309,-79.31004216517098,-79.30989654936729,-79.3097429368614,-79.30958613177039,-79.30946435061645,-79.3093682738677,-79.3092671972821,-79.30910612979385,-79.30893088438619,-79.30875942092088,-79.30861441040503,-79.30848706145359,-79.30836980758409,-79.30828488925472,-79.30829912008076,-79.30828269185594,-79.30820868923348,-79.30812950001412,-79.30804877731228,-79.30793966791427,-79.30783410600303,-79.30771128524205,-79.3075651557924,-79.30742674077935,-79.30728099017166,-79.30716390861467,-79.30710112311904,-79.30704249896704,-79.30697942573984,-79.30689907259865,-79.30680743773772,-79.30674894735164,-79.30674098159061,-79.30679030879858,-79.30690729630008,-79.30706200251917,-79.30719499055384,-79.30734009457333,-79.30747858202217,-79.30767396209869,-79.30791801715465,-79.30814544062987,-79.30836083247826,-79.30857641085059,-79.30878585665985,-79.30898339491769,-79.30917535640269,-79.30937556799678,-79.30955144561162,-79.30958754097021,-79.30952862643362,-79.30952841593577,-79.30955038095803,-79.3095919598826,-79.30970795458818,-79.3099776877406,-79.31018310265452,-79.31033063074716,-79.31039194996291,-79.31055392998768,-79.31091005105664,-79.31107733819589,-79.31115692889942,-79.31119016706569,-79.31117185947325,-79.31113753166143,-79.31023785580579,-79.31021492887339,-79.31025236949402,-79.31032210431486,-79.31040486462389,-79.3105515139108,-79.31073635210581,-79.31084303791171,-79.31093567885294,-79.31106789298859,-79.31124351521794,-79.31143223874778,-79.3116257971795,-79.31180135205489,-79.31194237451403,-79.31204755516242,-79.31215869231336,-79.31228155031914,-79.31242773455777,-79.31255346317238,-79.3126311972659,-79.31260826408561,-79.31250785266128,-79.3123743532793,-79.31223341557984,-79.31217298204841,-79.31210956270385,-79.31201028841303,-79.3119516593498,-79.31186787769261,-79.31172826451292,-79.311660223926,-79.31168271205365,-79.31179297184296,-79.31183038345674,-79.31175474487948,-79.31170783588334,-79.3116870844943,-79.31170396416361,-79.3117604469784,-79.31179370633369,-79.31191564139864,-79.31202594010516,-79.31192695654786,-79.31196470655445,-79.31203385534094,-79.3121371424125,-79.31185670730686,-79.31196396411219,-79.31217414295998,-79.31238899215933,-79.3121317395502,-79.31221804170687,-79.31245415153454,-79.31264477130995,-79.31242747774427,-79.31240903711441,-79.31244893429066,-79.31247864700089,-79.31255541847477,-79.31263600976419,-79.31269064787138,-79.31274043406772,-79.3128322626338,-79.31295869180256,-79.31310719175993,-79.31323156530738,-79.31325189096501,-79.31361941135759,-79.31374783740206,-79.31384848534537,-79.31388909735961,-79.31383821940877,-79.31388160058459,-79.31397382997562,-79.31402071794071,-79.31403573587848,-79.3140747264972,-79.31414387575192,-79.31424693979667,-79.31434574178239,-79.31439427334604,-79.31439506745619,-79.31411502826268,-79.31402330583693,-79.31397757144357,-79.31406196477586,-79.31402399938716,-79.31388779772702,-79.3137745930927,-79.3137347162827,-79.31366111441541,-79.31354062234553,-79.31339394410033,-79.31330404577302,-79.31326782254568,-79.31332042334876,-79.31347514500833,-79.31365771225808,-79.31384523369309,-79.31403863662858,-79.31422550449113,-79.31433621321807,-79.31433231112537,-79.31438167734039,-79.31445188267008,-79.31452430578196,-79.31462151379452,-79.31471452217677,-79.31475935870711,-79.31482253946989,-79.31491302805033,-79.31545323024993,-79.31608202813362,-79.31705843745384,-79.31709283211647,-79.31704696336462,-79.31718223659402,-79.31766394781353,-79.31788489306378,-79.31791080274567,-79.31786411157633,-79.31797557667687,-79.31804344511993,-79.31804163474386,-79.31765108903987,-79.31719176392829,-79.31602426962826,-79.31560067751587,-79.31530656977391,-79.31513539083711,-79.31497035913438,-79.31475932088232,-79.31452315227742,-79.31428322845063,-79.31408033934517,-79.31389842756153,-79.31368892958609,-79.31349358382133,-79.31330521364188,-79.31312694281475,-79.31312192452872,-79.31330393358063,-79.31350341415249,-79.31370287270443,-79.3139387398337,-79.31418365652529,-79.31441454520981,-79.31465387791891,-79.31486679087872,-79.31508462567493,-79.31533063348709,-79.31558786467401,-79.31584905677481,-79.31609380770404,-79.31624601937349,-79.31625910336774,-79.31679396088528,-79.31680811399364,-79.316769163123,-79.31665594415436,-79.31649951055378,-79.3163262807722,-79.31619843062508,-79.31603278221159,-79.31597971441535,-79.31603315258245,-79.31616501271921,-79.31635772353908,-79.31641235314579,-79.31629379548254,-79.31621996155005,-79.31615853520833,-79.31607653457159,-79.31603471841227,-79.31614024528203,-79.31634930604137,-79.31658965180857,-79.31681053936283,-79.31692953720079,-79.31712719270044,-79.31730605718924,-79.31752109532655,-79.31769626295396,-79.31784175315464,-79.31799531119809,-79.31811858728743,-79.31821525772473,-79.31825566149162,-79.3183157772167,-79.31840728865319,-79.31849996798425,-79.31859264915954,-79.318749278745,-79.31888151367416],"lat":[43.65247348019872,43.65225030918957,43.65204494400342,43.65183945219308,43.6516200300499,43.65140538035762,43.65132011267,43.65058476418717,43.65065738905832,43.65072103807127,43.65078025543607,43.65137280313317,43.65155514571222,43.65220209491957,43.65237999295498,43.65243871030636,43.65249760914111,43.65258419598417,43.65384198220807,43.65650831075416,43.65744670393295,43.65753329747332,43.65827564604795,43.65842592162556,43.65887680473431,43.65897706690472,43.6590317599577,43.65919951164174,43.65922669965719,43.66016472866803,43.66068943759277,43.66120317613041,43.66172775503645,43.66226888078766,43.66304452071481,43.66386370503121,43.66396148159412,43.66411248581235,43.66429439915091,43.66451898396092,43.66473085337096,43.66491662067357,43.66512509183869,43.66565102252117,43.6662158994276,43.66571767376927,43.66567222138678,43.66564462899997,43.66423414688279,43.66410495708323,43.66399083103875,43.66388100916184,43.66373961168753,43.66362089302623,43.66350167172497,43.66338784264871,43.66328755601737,43.66319616418097,43.66311398839332,43.66300881290883,43.66286723862503,43.66271206519304,43.66256546813187,43.6624099020243,43.6622451354312,43.66208965227128,43.66193400541118,43.66176009018321,43.66160390560959,43.66143832000704,43.66128600707257,43.66111508516702,43.66096635099737,43.66079921676535,43.66063692680022,43.66048336044544,43.66032965547702,43.66019742582527,43.66009146804849,43.65999424248069,43.65995060498209,43.65988909776965,43.65990433912477,43.65996049659055,43.66003081785468,43.66012602390657,43.66021267613975,43.66029779162619,43.66036813709098,43.66040158024433,43.66039849057203,43.66037727818453,43.66031570049613,43.66021401830337,43.66010810451034,43.65998038219427,43.65985362560058,43.65970571665697,43.65954946686728,43.65940718336995,43.65935103815577,43.65932635332227,43.65921970874923,43.65907260272942,43.65890299183852,43.65871936765357,43.65855411784835,43.65837897514091,43.65821654554039,43.65804984862557,43.65789149329331,43.65776354143092,43.65760825209483,43.65746177688695,43.65731993757741,43.65715127454094,43.65700071498264,43.65684631922299,43.65665710326489,43.65651049495798,43.65642789306331,43.65639959421568,43.65642087489359,43.65644228393993,43.65651306489603,43.65657463053362,43.65660017087183,43.65658927700095,43.65655121991566,43.65649044883136,43.65640270236878,43.65632394489366,43.65622659352579,43.65618334595186,43.65612162185778,43.65598764500145,43.65584505202833,43.65567523171384,43.65552772242801,43.65534857158684,43.65518791128199,43.65502723807352,43.65488903852108,43.65472701326844,43.65455533538429,43.65437012444627,43.65419930515119,43.65412083432392,43.65412768437339,43.65422805047575,43.65434150482009,43.65447765906526,43.65459121618643,43.65473225741606,43.65485501841562,43.65500017226424,43.65510512937245,43.65521059219434,43.65529801442941,43.65537235936856,43.65540653676175,43.65542275052639,43.65541596194397,43.65532616181286,43.65519628342571,43.65514045120318,43.65515698285343,43.65521811738319,43.65528398041273,43.6553850690153,43.65549420419352,43.65562106040239,43.6557423532493,43.65586736886311,43.65595997127829,43.65600252454238,43.65600490551291,43.65600736487612,43.6560321611539,43.6560612160886,43.65608619483209,43.65612477364181,43.65609561412902,43.65607083717261,43.65605051380164,43.65599920121245,43.65593926672592,43.65589717223094,43.65589028818836,43.65589669716964,43.65590316185082,43.65593261288553,43.65596673614716,43.65600550971343,43.65607626893846,43.65614849584562,43.65623827296598,43.65633178881406,43.65642063679216,43.65650946201028,43.65661136992105,43.65669982720831,43.65676561377169,43.65681296093616,43.65685142013928,43.65688526950536,43.65691054133201,43.65691411229173,43.65690424181887,43.65688530337543,43.65684810483293,43.65685574923166,43.65695390494366,43.6570663122844,43.65718866055131,43.65727591307937,43.65736372425227,43.65746021554134,43.65756588815885,43.65768505566142,43.65773746026961,43.65776808018198,43.65781195639693,43.65786915710456,43.65794886523767,43.65802848220952,43.65811701464972,43.65817836813292,43.65822634030042,43.65825594811358,43.65821597290604,43.65818359434303,43.6581655871728,43.65812990396997,43.65805850277386,43.65806471948893,43.65814972696639,43.6583148010352,43.65838899859493,43.65835389708561,43.65824826745887,43.65792936849176,43.65751765753507,43.65727577895085,43.6572537650644,43.65721739637871,43.65719438400708,43.65694253111857,43.65692868907273,43.65686622846036,43.6567862412577,43.65667935047595,43.65659149821987,43.65660773362999,43.65667232478823,43.65676371435735,43.65683327364393,43.65687187710937,43.65688816880849,43.65691803467202,43.65691162613223,43.65688220216929,43.65680714847344,43.65674577571904,43.65672958610615,43.65676775300409,43.65684621683342,43.65697339303996,43.65707657777943,43.65712010528522,43.65717664713186,43.65722857795345,43.65730879371277,43.65739337633414,43.6574729283134,43.65752157348029,43.65755184301167,43.65752727551855,43.65776941515168,43.6577697473799,43.65781638638467,43.65795196947673,43.65810397662315,43.65819780493969,43.6583325288027,43.65836878628497,43.65845513997403,43.65856374566018,43.65852953876072,43.65874271563206,43.6587862637009,43.65880032435241,43.6587923436968,43.65884338038912,43.65900136469391,43.65907496525067,43.65890244004142,43.65906323847786,43.6592259767845,43.65929926760948,43.65911371183515,43.65925605792161,43.65940147284383,43.6594282066125,43.65944229881069,43.65948774770142,43.65943036814296,43.65940455202728,43.65947287395399,43.65955012624663,43.6595648950029,43.6596477796961,43.65977149915759,43.65989495234427,43.6599177574132,43.66018874109445,43.66028074642265,43.66036775072399,43.66044036613441,43.66051163166989,43.66054377882323,43.66054063854445,43.66058634042061,43.66062707109492,43.66067724747283,43.66072777871841,43.66078331121716,43.66085228379497,43.66091601381035,43.66092052653595,43.66111903127999,43.66104557191728,43.66107190326496,43.66115875757826,43.661185203557,43.66115168700928,43.66123553594933,43.66127545666508,43.66127437060845,43.66128609558258,43.66131993902552,43.66140413145902,43.66150712010444,43.66159350570949,43.66160479118783,43.66158948128217,43.66156524215706,43.66155009151719,43.66155284819838,43.66161299439172,43.66169395505348,43.66173519231751,43.66174072882635,43.66173279496495,43.66177473771506,43.66185712755929,43.66194330798049,43.66196233372956,43.66194107295799,43.66180500468558,43.66165673630306,43.66144597883983,43.66143748331935,43.66094160641533,43.66089858830649,43.66086067050914,43.66105296539958,43.66104434473959,43.66088162101617,43.66091476874661,43.66089776358682,43.66085272681087,43.66079746702961,43.66077720124661,43.65990013008034,43.65957872650523,43.65942135676403,43.65933322388252,43.65922276633336,43.65909362606275,43.65900012288007,43.65890197278236,43.65875944841664,43.65861282219629,43.65848361264672,43.65834569955584,43.65820338806882,43.65806563633808,43.65791252776584,43.65780718986505,43.65768410544832,43.6575654313493,43.65750139626497,43.65744649621735,43.65739588975384,43.65737241344174,43.65735754908884,43.65737876502715,43.6574274019566,43.65748061477544,43.65753847642245,43.65757359014022,43.65759383690392,43.65759402967541,43.65731834467828,43.65730955110226,43.65729088332476,43.65726220958998,43.65722839813176,43.65729795238507,43.65740859411861,43.65745566452884,43.65736036122318,43.65724853325563,43.65713344977242,43.65703276567688,43.65690754205026,43.65682477711644,43.65669315983617,43.65653471922598,43.65636238235222,43.65616822249,43.65596714164909,43.6558081852321,43.65566769333815,43.65555383042664,43.65537554234199,43.65520741447521,43.65501641447435,43.65486654592479,43.65469799596075,43.6545155958593,43.65431522021731,43.6541189897448,43.6539087745283,43.65370682328621,43.65352766718281,43.65332637790263,43.65311619370851,43.65290591943783,43.65268317283622,43.65247348019872]}]],[[{"lng":[-79.36472656955553,-79.36430890403285,-79.36423880212871,-79.36337778243815,-79.36308938318966,-79.36301310066492,-79.36297217499042,-79.3629363199376,-79.36289764216127,-79.36287301713736,-79.36286683488123,-79.36286633665834,-79.36288882446894,-79.36291611792704,-79.36292883229652,-79.36295142214701,-79.36297528090049,-79.36299539655761,-79.36301434542207,-79.36302973989153,-79.36303120785787,-79.3630318195093,-79.36304226338829,-79.36304362724302,-79.36302910250025,-79.36299701124649,-79.36293807154865,-79.36285037968831,-79.36277681824069,-79.36267806387343,-79.36256254111998,-79.36243301232199,-79.36231417933743,-79.36222526212019,-79.36215638630443,-79.36208301318545,-79.36203438388249,-79.36194268799622,-79.3618391709188,-79.36174581676133,-79.3616534031522,-79.36155309651862,-79.36145176631665,-79.36135254311228,-79.36124589655047,-79.36115596039873,-79.36105753553736,-79.36095939628419,-79.36086344595735,-79.360776718187,-79.36068796594225,-79.36060052283202,-79.36051261166369,-79.36044344968278,-79.36038911342784,-79.36033134239747,-79.36026547304256,-79.36023852955583,-79.36021197793151,-79.36016471743687,-79.36011686390223,-79.36007739453449,-79.36003778243867,-79.35999242620454,-79.35995011467185,-79.35990995038408,-79.359863774169,-79.35982440777096,-79.35978187440283,-79.35973956351404,-79.35970056579791,-79.3596450455282,-79.35958026447116,-79.35949930711124,-79.35940704174995,-79.35931643335279,-79.35845889827084,-79.35773220535779,-79.3580381281235,-79.35824675896988,-79.358257371394,-79.35834740296079,-79.36087543697138,-79.361269150224,-79.36307447416533,-79.36363899535161,-79.36377317062824,-79.36517760274778,-79.36561665709522,-79.36725150055283,-79.36861963084363,-79.36885591850874,-79.36910808713215,-79.36917884537964,-79.36954973105055,-79.36968685829299,-79.36981883148481,-79.37039495019492,-79.37085787599484,-79.37088598346551,-79.37090785402934,-79.37093007033621,-79.37095971160858,-79.37100865782256,-79.37105715132844,-79.37111400712574,-79.37115979855096,-79.3712070019911,-79.37125586087531,-79.37131414587033,-79.37136572159932,-79.37140492561414,-79.37143177735494,-79.37144627680946,-79.3714076278962,-79.37136969470464,-79.37133235553192,-79.37129419838818,-79.37125698091472,-79.37121976344228,-79.37118230254286,-79.37114518731271,-79.37110784811817,-79.37107120516326,-79.37103396820267,-79.37099722298457,-79.37096129572333,-79.3709247744646,-79.37088884718632,-79.3708530221441,-79.37081781056749,-79.37078270122713,-79.37074757237622,-79.37071317873558,-79.37068490033745,-79.36990428003057,-79.36882098662447,-79.36819088482859,-79.36735694929686,-79.36725912623389,-79.36717320352115,-79.36707881368358,-79.36698536373626,-79.36686491960003,-79.36676795002202,-79.3667231651783,-79.36629949497582,-79.36472656955553],"lat":[43.67506276031906,43.67514230217935,43.67515480181159,43.67531801354768,43.67491320326123,43.67486710050991,43.67480350015794,43.67472646940377,43.67464039618493,43.67456793727071,43.67448683055999,43.67439230225079,43.67430260423557,43.67421297505646,43.67411404498673,43.67402434841664,43.67394817304293,43.67386294202812,43.67376419126895,43.6736833036061,43.67362931252826,43.67360681624568,43.67353494970369,43.67345845208855,43.67339072890032,43.67331825307102,43.67323630059347,43.67318103226697,43.6731214653024,43.67307053936592,43.67301937308331,43.67299051103242,43.67297080412922,43.67296061787412,43.67292812371507,43.67291807010316,43.67290837109704,43.67285745567233,43.67281996377046,43.67277361544622,43.67272277952014,43.67267633136568,43.67260736340035,43.67254742757537,43.67247838319603,43.67242299136279,43.67236756773881,43.67229414408791,43.67223425477695,43.67216999666306,43.67210120843291,43.67203693992307,43.67196357259142,43.67188156218433,43.67179976444415,43.67173142042557,43.67164495614023,43.6716014498674,43.67155857935212,43.6714723820202,43.67138167514771,43.67129108856702,43.67120950194379,43.67110973881214,43.67101461038651,43.67091951276,43.67081982782554,43.6707292426281,43.67063852194163,43.67054339344097,43.67046181551045,43.67037099838477,43.6702709562572,43.67019777796732,43.67011543526245,43.67005112031469,43.67037386103582,43.66892534736064,43.66888023207417,43.66884722135255,43.66882936976267,43.66868212971812,43.6675478838074,43.66746350986302,43.66707519972146,43.66695275493446,43.66692307895551,43.66661910531261,43.66652177189433,43.66615355245028,43.66586693178186,43.66645102549646,43.6670984495284,43.66726149579392,43.66816708058514,43.66851570494248,43.66885075255833,43.6702500521671,43.67139567838987,43.6714455901988,43.67149541309254,43.67153623888801,43.67158617254213,43.67164088237906,43.67168208270866,43.67172799323983,43.67175565197866,43.67178783183487,43.67181553425412,43.67183886997909,43.67185310803774,43.67186716976669,43.67187655444477,43.67188126208051,43.67188071130929,43.6718801707248,43.67188413960433,43.67188809681408,43.67188756639244,43.67188703595868,43.67189550406513,43.67189497506429,43.67189894387006,43.67190742359853,43.67191139383733,43.67191987208436,43.67192836197927,43.67193234238454,43.67194083225638,43.67194932357434,43.67195782362658,43.67196632512544,43.67197932734613,43.67198783902699,43.6720009388942,43.67257061722024,43.67332492273048,43.67377061922141,43.67436202365042,43.67442364019924,43.67447642467808,43.67452458709869,43.67456826186282,43.67461155078932,43.67464167205139,43.67466362706347,43.674761094431,43.67506276031906]}]],[[{"lng":[-79.41147943416414,-79.40927725034396,-79.40756925028863,-79.40592797395797,-79.405709665651,-79.40373866723939,-79.40165007520855,-79.3998280364227,-79.39980957819941,-79.39970572414673,-79.3995514122764,-79.39946470639697,-79.4011709838246,-79.40104696800768,-79.4023676595442,-79.40420596143974,-79.40412899616128,-79.40395896864956,-79.40374196988046,-79.40250626404799,-79.40104981284222,-79.40103275401378,-79.40085985906259,-79.40084524483488,-79.40062534652597,-79.40063722945686,-79.40027697757797,-79.40009721675167,-79.40027048365238,-79.40143983914588,-79.40172329697668,-79.40275283964856,-79.40373841616336,-79.40488039434514,-79.40612145934321,-79.40651688104737,-79.40739448571948,-79.4085046025814,-79.40855795756249,-79.40862803129518,-79.40870529840116,-79.40880307459413,-79.40889861582194,-79.4089609565671,-79.40901952885936,-79.40914618599422,-79.40928878621055,-79.40942042102111,-79.40952459069899,-79.40960939384601,-79.40968364090935,-79.4097267372954,-79.40974345093902,-79.40973447156786,-79.40995548552522,-79.41014832033497,-79.41020248050344,-79.4102593486501,-79.41034639844216,-79.41039561982531,-79.410423436649,-79.41077244875856,-79.41112037848028,-79.41147943416414],"lat":[43.71117890861868,43.71163447184345,43.71200686035377,43.71237115207614,43.71241311644658,43.71283578990085,43.71328388049061,43.71364555593574,43.71352377091075,43.71299552145369,43.71228265497392,43.71183323255521,43.71148497645646,43.71055603825585,43.71024962869494,43.70982602586467,43.70963698985349,43.70942198865276,43.70861301093139,43.70886958340017,43.70916347286994,43.70871187519757,43.70860521302275,43.70828111746074,43.70789842673292,43.70770946153311,43.70678198648484,43.70637969209764,43.70634511963223,43.70609583486359,43.70602778385475,43.70581703899974,43.70559675875725,43.70535605825177,43.7050988144272,43.70501872007386,43.70482840953125,43.70459172967294,43.70468708354887,43.70488160199728,43.70504021263927,43.70518569584569,43.70527254495104,43.70531842282529,43.70535974720974,43.70541552237078,43.70547602030786,43.70554095645338,43.7056189231478,43.7057326281458,43.70582368122601,43.70592780373833,43.70600014218621,43.70607653406125,43.70689896964305,43.70770741967425,43.70794672556413,43.70817265614554,43.70837191059067,43.70850771495224,43.70856661473486,43.70944934164983,43.71032305036223,43.71117890861868]}]],[[{"lng":[-79.40009721675167,-79.40004360909138,-79.39994460270674,-79.40001292265167,-79.40002052437748,-79.39983365350234,-79.39953392683663,-79.39940349129647,-79.39796091237328,-79.39782464942931,-79.39781338932413,-79.39767196221902,-79.39765384984683,-79.39749311262291,-79.39733223535826,-79.39720703669599,-79.39714695677642,-79.39702700069658,-79.39694204464537,-79.39684951096248,-79.39663094493483,-79.39645944037431,-79.39619253638269,-79.39609321294949,-79.39704295533215,-79.40007828896606,-79.40013855806662,-79.403001478893,-79.40295098489131,-79.40416760528761,-79.40587112292431,-79.40781705919872,-79.40912499815217,-79.40941587167323,-79.4093584526488,-79.40934926712964,-79.40936021714258,-79.40938670164525,-79.40985035009832,-79.41024584046627,-79.41094615244589,-79.4096101049808,-79.4085046025814,-79.40739448571948,-79.40651688104737,-79.40612145934321,-79.40488039434514,-79.40373841616336,-79.40275283964856,-79.40172329697668,-79.40143983914588,-79.40027048365238,-79.40009721675167],"lat":[43.70637969209764,43.70625966535469,43.70566108790566,43.70511652454405,43.70473845780552,43.70428826281609,43.70425642939,43.70390082468786,43.70419657982329,43.70422527620596,43.70414085980001,43.70339153240253,43.70328775564987,43.7025066502319,43.70178855641758,43.70112956563618,43.70084957141497,43.70028967571998,43.69977969173775,43.69932370329258,43.69829423023077,43.69736902987193,43.69612273955688,43.69565315196622,43.69563947183515,43.69638416312213,43.6965381307778,43.69728942252399,43.69714468500732,43.69759836760607,43.69817585618414,43.69882430206867,43.69925210811293,43.69990439029972,43.69996669503468,43.70001607811884,43.7000702424666,43.70015612999692,43.70131060603771,43.70233801403283,43.70407649530966,43.70435506480411,43.70459172967294,43.70482840953125,43.70501872007386,43.7050988144272,43.70535605825177,43.70559675875725,43.70581703899974,43.70602778385475,43.70609583486359,43.70634511963223,43.70637969209764]}]],[[{"lng":[-79.39201828674122,-79.39256150841811,-79.39256207643112,-79.39256166490037,-79.39201828674122],"lat":[43.69198194634033,43.6923684826423,43.69237020101131,43.69237028524269,43.69198194634033]}],[{"lng":[-79.39212572139331,-79.39215197038629,-79.3921256094941,-79.39212581645997,-79.39212572139331],"lat":[43.6917560675293,43.69181909074629,43.69175642603479,43.69175633892679,43.6917560675293]}],[{"lng":[-79.39609321294949,-79.39527200102582,-79.3951580058711,-79.39505055299654,-79.39492417672767,-79.39487913679186,-79.39480930776534,-79.39455546825879,-79.39412993887616,-79.39796331566119,-79.39985632519952,-79.40075664336713,-79.40148256503196,-79.40232629039119,-79.40307198909697,-79.40307209036868,-79.40304969016685,-79.403005146678,-79.40184566163471,-79.40174561367766,-79.40168223352421,-79.4015734571316,-79.40146212606504,-79.4008567917314,-79.40103406830448,-79.40111649870308,-79.40127259685514,-79.40157721863714,-79.40188314716701,-79.40217961833812,-79.40231077007211,-79.40236076390505,-79.40239162275932,-79.40308266327379,-79.40349452425282,-79.40466208200078,-79.40504061770011,-79.40526973062325,-79.40540029039576,-79.40559971318669,-79.40587112292431,-79.40416760528761,-79.40295098489131,-79.403001478893,-79.40013855806662,-79.40007828896606,-79.39704295533215,-79.39609321294949],"lat":[43.69565315196622,43.69177643650618,43.69125515033402,43.69077635530531,43.69019386137622,43.6899951847608,43.68972855490351,43.68912626601573,43.68807587486012,43.68731022545266,43.68693156687859,43.68675512749142,43.68663466472935,43.6887128738142,43.6906276702472,43.69068168366972,43.69073538265468,43.69077526919279,43.6909842000579,43.69101430787401,43.69103142544863,43.69107041303633,43.69105535278348,43.69119091572245,43.69163908528638,43.69174835262027,43.69209261246191,43.69291623551531,43.69372178191902,43.69456779417747,43.69488478769656,43.69510603550646,43.69548914169585,43.69534126546772,43.69525249911531,43.69501215747446,43.69597183272026,43.69654683680962,43.69694483638018,43.69747432545629,43.69817585618414,43.69759836760607,43.69714468500732,43.69728942252399,43.6965381307778,43.69638416312213,43.69563947183515,43.69565315196622]}]],[[{"lng":[-79.3997916782411,-79.40015689315855,-79.40019738402502,-79.40051775385426,-79.40082837097997,-79.40111392297075,-79.40148256503196,-79.40075664336713,-79.39985632519952,-79.39796331566119,-79.39412993887616,-79.39369298630356,-79.39331995700053,-79.39317357046622,-79.39284201875316,-79.39237230808048,-79.39202015052945,-79.39192696317451,-79.3916182397882,-79.39154234475868,-79.39132821734373,-79.39118244660715,-79.39139314764495,-79.39321130493595,-79.39579685674016,-79.39733706515308,-79.39744381067001,-79.39755151136362,-79.39765909209261,-79.39776737333783,-79.39787658982276,-79.39791901186771,-79.39793218295796,-79.39800019849307,-79.39812870351679,-79.39833934144951,-79.398550561497,-79.39872119666948,-79.39892727733334,-79.39905330493507,-79.39933320346751,-79.3997916782411],"lat":[43.68253687101222,43.68341085853671,43.68351944950817,43.68430278786192,43.68501838366817,43.68571112285325,43.68663466472935,43.68675512749142,43.68693156687859,43.68731022545266,43.68807587486012,43.68712434344676,43.68622772099985,43.6858744945712,43.68501896262504,43.68393184490229,43.68309402447795,43.68286766298854,43.68214297750928,43.68196637011076,43.68142314539424,43.68107451586637,43.68096495673139,43.68015757832443,43.67897385891423,43.67826612561503,43.67821811145568,43.67816110857117,43.67810860491149,43.67806052196016,43.67800804102774,43.677986130696,43.67801782240344,43.67819890626217,43.67852487025996,43.67903202557925,43.67954819064725,43.67995576243499,43.68045385069537,43.68076177480916,43.68143481527793,43.68253687101222]}]],[[{"lng":[-79.39527200102582,-79.39267570214365,-79.39267580682916,-79.39256207643112,-79.39256150841811,-79.39201828674122,-79.39196562655498,-79.39215197038629,-79.39212572139331,-79.39228638647455,-79.39236117493448,-79.39235215346052,-79.39226100394745,-79.39205971880841,-79.39198431493256,-79.39196980166639,-79.39192892615606,-79.39189763978453,-79.39187917728884,-79.39069150819977,-79.39068822563389,-79.39056538750111,-79.39045764592372,-79.39038782596599,-79.39030915597137,-79.39021529704723,-79.39010755043257,-79.39006078560332,-79.39002912819547,-79.39000542606452,-79.38999713320452,-79.38996870200238,-79.38986924250621,-79.3897640998869,-79.38959581417804,-79.38943597519824,-79.38896739326032,-79.38891638252737,-79.38870265841874,-79.38864447051496,-79.38847053484976,-79.38820341623693,-79.3879520555259,-79.3877282453354,-79.38748675299048,-79.3872626230807,-79.38718758785055,-79.38702358636834,-79.3867803449904,-79.3865233210505,-79.38637792564934,-79.38626873571833,-79.38607497537924,-79.38592542047658,-79.38584639896099,-79.38564206656507,-79.38543414646958,-79.38528162944505,-79.38530612287187,-79.3854290115885,-79.38552724158066,-79.38542534878547,-79.38544490994701,-79.38547850515727,-79.3855251614265,-79.38562561281582,-79.38576966956097,-79.38575787397903,-79.38557946679164,-79.3853559171659,-79.38524287064752,-79.38529985753571,-79.38517613721207,-79.3849583806748,-79.38485114511319,-79.38479324953418,-79.38799870486943,-79.38806140026918,-79.39078842440153,-79.39079701446738,-79.39118244660715,-79.39132821734373,-79.39154234475868,-79.3916182397882,-79.39192696317451,-79.39202015052945,-79.39237230808048,-79.39284201875316,-79.39317357046622,-79.39331995700053,-79.39369298630356,-79.39412993887616,-79.39455546825879,-79.39480930776534,-79.39487913679186,-79.39492417672767,-79.39505055299654,-79.3951580058711,-79.39527200102582],"lat":[43.69177643650618,43.69234533334611,43.69234524479854,43.69237020101131,43.6923684826423,43.69198194634033,43.69185814805878,43.69181909074629,43.6917560675293,43.69172069959675,43.69166350880381,43.69160288846781,43.69131507256444,43.69121645932139,43.69123160193099,43.69121186337728,43.69122524130039,43.6911133563116,43.69108825099425,43.69133934141219,43.69133200356772,43.69105103167251,43.69093248827076,43.69084598589298,43.6907728618105,43.69071293669269,43.69062139888086,43.69057573008007,43.69051227009919,43.69032739520094,43.69022816635821,43.69007473181666,43.68990679353097,43.68977919405616,43.6896597967118,43.68957652616093,43.6891782442934,43.68914601816052,43.68895837387426,43.68893054742316,43.68879306474313,43.68863176256988,43.68852010304323,43.6884494306815,43.688401013384,43.68825831958753,43.68802311857973,43.68785876787893,43.6877383083844,43.68766257376948,43.68747597982372,43.68724029591736,43.68702601259505,43.68690687433843,43.68684724519007,43.68671824080824,43.68666579242046,43.68659612248622,43.68643443250282,43.68624703668248,43.68608638839307,43.68591832175927,43.68571164203893,43.68550047953791,43.68532109857885,43.68519198858052,43.68505440241325,43.68488769868693,43.6847726533929,43.68469288801769,43.6846012706707,43.68447154678421,43.68433926952323,43.68421907619594,43.68408253054413,43.68396018517508,43.68251975410951,43.68249363242894,43.68124900844729,43.68124912940638,43.68107451586637,43.68142314539424,43.68196637011076,43.68214297750928,43.68286766298854,43.68309402447795,43.68393184490229,43.68501896262504,43.6858744945712,43.68622772099985,43.68712434344676,43.68807587486012,43.68912626601573,43.68972855490351,43.6899951847608,43.69019386137622,43.69077635530531,43.69125515033402,43.69177643650618]}]],[[{"lng":[-79.36675317757052,-79.36653556145096,-79.36443177832872,-79.36424750583684,-79.36425126937121,-79.36417183725356,-79.36403543454188,-79.36371837176634,-79.36358573491998,-79.36358163282968,-79.36382481482792,-79.3640937184214,-79.36443002936534,-79.36481838980434,-79.36494304976164,-79.36467274428398,-79.36431831421774,-79.36401680459875,-79.36374734517432,-79.36345547925156,-79.36337778243815,-79.36423880212871,-79.36430890403285,-79.36472656955553,-79.36629949497582,-79.3667231651783,-79.36676795002202,-79.36686491960003,-79.36698536373626,-79.36707881368358,-79.36717320352115,-79.36725912623389,-79.36735694929686,-79.36819088482859,-79.36882098662447,-79.36990428003057,-79.37068490033745,-79.37071317873558,-79.37074757237622,-79.37078270122713,-79.37081781056749,-79.3708530221441,-79.37088884718632,-79.3709247744646,-79.37096129572333,-79.37099722298457,-79.37103396820267,-79.37107120516326,-79.37110784811817,-79.37114518731271,-79.37118230254286,-79.37121976344228,-79.37125698091472,-79.37129419838818,-79.37133235553192,-79.37136969470464,-79.3714076278962,-79.37144627680946,-79.37316513058396,-79.37359230573503,-79.37489678515858,-79.37588013861573,-79.37593972950822,-79.37603795068392,-79.37615021794055,-79.37630427835406,-79.37644351530244,-79.37684208424197,-79.37713166822429,-79.37840146317906,-79.37973681858989,-79.3800352898014,-79.38069640765619,-79.38120216515512,-79.38220486168991,-79.38405747471265,-79.38419104039379,-79.38427772923926,-79.38439918284698,-79.38449362420079,-79.38458266484737,-79.38467843550332,-79.38480070722538,-79.38490516914499,-79.38501400938816,-79.3851014141769,-79.38520115431345,-79.38528361434449,-79.38531328510182,-79.38540610919247,-79.38549408835958,-79.38559820701407,-79.38565367333655,-79.38570302309833,-79.38582561232464,-79.38593644295783,-79.38602205474116,-79.38612087515574,-79.38620604135134,-79.38632992364241,-79.38642517610828,-79.3865017169424,-79.38658732148491,-79.38667599366491,-79.38676974187831,-79.38682528282413,-79.38690030120182,-79.38696773489205,-79.38702083533914,-79.38707368401191,-79.38711616887451,-79.38717160555487,-79.38721197007456,-79.38725313684813,-79.38728358560488,-79.38731606120336,-79.38734025449997,-79.38737177105654,-79.38739738869448,-79.38740747873226,-79.38742192911182,-79.38744155787313,-79.3874637588458,-79.38748928184427,-79.38751843853089,-79.38754257390944,-79.38758024322844,-79.38761944639441,-79.38766654135105,-79.38770268423353,-79.38776641725605,-79.38783906458347,-79.38794743478273,-79.38806995238726,-79.38815475674669,-79.38822166677164,-79.38834900843037,-79.38836763670932,-79.38897426083743,-79.38922513295003,-79.38926442445536,-79.38951909245803,-79.38955978902527,-79.3895821521099,-79.38969021848034,-79.38988926850209,-79.3900292594561,-79.39041994413944,-79.39071542312776,-79.39118244660715,-79.39079701446738,-79.39078842440153,-79.38806140026918,-79.38799870486943,-79.38479324953418,-79.38369339738928,-79.38349233987066,-79.38346237534182,-79.38343053523374,-79.38337814647126,-79.38335386507785,-79.38333774405162,-79.3833325190578,-79.38332732434698,-79.3833172836041,-79.3832794640763,-79.3832682730861,-79.38321020557332,-79.38293007214131,-79.38263751732082,-79.38250822360344,-79.38240395837575,-79.38235452051669,-79.38225542192419,-79.38217197856117,-79.38205118517971,-79.37921085479206,-79.37920719233277,-79.37903865293309,-79.37882212038841,-79.37855387453762,-79.37832453671393,-79.37808495425126,-79.3777974006893,-79.37747862003226,-79.37719270977094,-79.37691161251017,-79.37660457912527,-79.37630345065934,-79.37602063903992,-79.37574293359118,-79.3754471356248,-79.37516266029522,-79.37487719957323,-79.37457500119541,-79.37455260660958,-79.37424635607756,-79.37391229164645,-79.37356974588594,-79.3732137601876,-79.37287194449247,-79.37255473464931,-79.37225717804905,-79.37199662213955,-79.37171691232761,-79.37146314726314,-79.37124404350817,-79.37117337886754,-79.37007640416221,-79.36883919119879,-79.36818667841598,-79.36724241979742,-79.36717334271309,-79.36725054075887,-79.36675317757052],"lat":[43.67850422518018,43.67855062593285,43.67901573652897,43.67917513524635,43.67917969012104,43.67919655715278,43.67907757843423,43.67878938574773,43.67842281509709,43.678058174912,43.67778250450762,43.67747119372747,43.67722395115749,43.67699986736297,43.67672249857164,43.67645307145443,43.67622285869209,43.67593497935147,43.6756384660681,43.67539114235202,43.67531801354768,43.67515480181159,43.67514230217935,43.67506276031906,43.674761094431,43.67466362706347,43.67464167205139,43.67461155078932,43.67456826186282,43.67452458709869,43.67447642467808,43.67442364019924,43.67436202365042,43.67377061922141,43.67332492273048,43.67257061722024,43.6720009388942,43.67198783902699,43.67197932734613,43.67196632512544,43.67195782362658,43.67194932357434,43.67194083225638,43.67193234238454,43.67192836197927,43.67191987208436,43.67191139383733,43.67190742359853,43.67189894387006,43.67189497506429,43.67189550406513,43.67188703595868,43.67188756639244,43.67188809681408,43.67188413960433,43.6718801707248,43.67188071130929,43.67188126208051,43.67214429718537,43.67219997854281,43.67235806736391,43.67245756588089,43.67246291376134,43.67247331157498,43.67247490686778,43.67245909180827,43.67244297592875,43.6723496153445,43.67314608596909,43.6731461068558,43.67314254214298,43.67316927850273,43.67323266025686,43.67328033456734,43.6733575475914,43.67349178297518,43.67351167557103,43.67351740223515,43.67352362031129,43.67353395742036,43.67353971709873,43.67355007284536,43.67355630205842,43.67356678045328,43.67358182163706,43.67358755779367,43.67360247020576,43.67361713837547,43.67361305658887,43.67361886908208,43.67363370509585,43.67365318002995,43.67366746660024,43.67367716573203,43.67370590338907,43.6737344748411,43.67375368800316,43.67376858662326,43.67379679537714,43.67383455277499,43.6738719058073,43.67390899457789,43.6739552132805,43.67400147522518,43.67405681077803,43.67410269492986,43.6741622669967,43.67422623297009,43.67429449761338,43.67434925564551,43.67441286940707,43.6744855779412,43.6745672558049,43.67465335596533,43.67473480384864,43.67482078133533,43.67490664193744,43.67500169791376,43.6751145846623,43.67517324017473,43.67524095923355,43.67531775337977,43.67539017282657,43.67547605215227,43.675548479712,43.67559833136514,43.67563937194798,43.67568043415916,43.67571710668863,43.6757311196355,43.67575002281632,43.67576455070635,43.6757705804454,43.67576780761189,43.67576000172597,43.67575194340266,43.67574473729655,43.67574049899142,43.6755825148594,43.67623428659513,43.67632945162077,43.67695427008717,43.67703136074535,43.67709018900179,43.67737536538601,43.67784636492543,43.67820400687641,43.67913689782719,43.67989290740061,43.68107451586637,43.68124912940638,43.68124900844729,43.68249363242894,43.68251975410951,43.68396018517508,43.68445783912011,43.68400930594443,43.6839098600089,43.6837697884804,43.68355299921274,43.68339052956323,43.68323276637015,43.68311116534603,43.6829579676986,43.68281379352024,43.68266022434351,43.68249793984348,43.68229457313207,43.68175489995802,43.68112944113916,43.68088905792307,43.68072995690917,43.68066624304008,43.68054331308697,43.68044310970338,43.68032428342611,43.67980692869117,43.67980237574174,43.67975497521707,43.67978791172917,43.67978860707129,43.67975384586308,43.67972343980392,43.67971935833747,43.67972383480679,43.67974228017856,43.67978329822139,43.67981494512834,43.67982867108438,43.6798156505067,43.67970818068353,43.67955094192651,43.67942528040939,43.67929069217825,43.67919637422428,43.67919605569894,43.67916469332361,43.67913293436622,43.67912355875553,43.67910048778141,43.67906411443559,43.67903709249117,43.67900584859303,43.67897954201577,43.67891254290392,43.67880540378147,43.67867625307587,43.67865274088517,43.67853357730991,43.67823677235463,43.67865064147225,43.67924497013433,43.67916296508681,43.67911905801586,43.67850422518018]}]],[[{"lng":[-79.36724241979742,-79.36818667841598,-79.36883919119879,-79.37007640416221,-79.37117337886754,-79.37124404350817,-79.37146314726314,-79.37171691232761,-79.37199662213955,-79.37225717804905,-79.37255473464931,-79.37287194449247,-79.3732137601876,-79.37356974588594,-79.37391229164645,-79.37424635607756,-79.37455260660958,-79.37457500119541,-79.37487719957323,-79.37516266029522,-79.3754471356248,-79.37574293359118,-79.37602063903992,-79.37630345065934,-79.37660457912527,-79.37691161251017,-79.37719270977094,-79.37747862003226,-79.3777974006893,-79.37808495425126,-79.37832453671393,-79.37855387453762,-79.37882212038841,-79.37903865293309,-79.37920719233277,-79.37921085479206,-79.38205118517971,-79.38217197856117,-79.38225542192419,-79.38235452051669,-79.38240395837575,-79.38250822360344,-79.38263751732082,-79.38293007214131,-79.38321020557332,-79.3832682730861,-79.3832794640763,-79.3833172836041,-79.38332732434698,-79.3833325190578,-79.38333774405162,-79.38335386507785,-79.38337814647126,-79.38343053523374,-79.38346237534182,-79.38349233987066,-79.38369339738928,-79.38357648697502,-79.38351685561148,-79.38345734504708,-79.3833975927306,-79.38333665032049,-79.38327621920754,-79.38321517444328,-79.38315331149086,-79.38309132519881,-79.38302874627675,-79.38296628819838,-79.38290360692444,-79.38284043289399,-79.38277715656014,-79.38271282023031,-79.38264894884983,-79.38258413837963,-79.38251922560512,-79.38245523321072,-79.3823897067538,-79.38232397572364,-79.38225846786727,-79.38219224410481,-79.38212591804721,-79.38205969423153,-79.38199334942763,-79.38192642837151,-79.38185785229753,-79.37898701386938,-79.37886057858353,-79.37873496133292,-79.37860934396188,-79.37848372647038,-79.37835812782521,-79.3782331047444,-79.37821058642605,-79.37810900195282,-79.37798491801836,-79.3778606104207,-79.37773652381289,-79.37761325767114,-79.37637032392134,-79.37628336226049,-79.37619566548801,-79.37610856310364,-79.37602313522794,-79.37593685080034,-79.37585152501316,-79.37576665300814,-79.37568192149021,-79.37559788659803,-79.37551386835011,-79.37543063222869,-79.37491532239932,-79.37484284318457,-79.37476962630333,-79.37469712764923,-79.37462546631541,-79.37455439926538,-79.3744834343839,-79.37441246939899,-79.37434303839501,-79.37428221744425,-79.37427360728974,-79.37420487270835,-79.37413697548338,-79.37406895416309,-79.37400257152703,-79.37393620808623,-79.37386972296807,-79.37380489342148,-79.37373992287212,-79.37367497153028,-79.37361165645372,-79.37354905475514,-79.37348655766056,-79.37342395819144,-79.37336275175652,-79.373301583906,-79.37324121478963,-79.3731818876638,-79.37312366608867,-79.37306493060778,-79.37300762929108,-79.37295043014659,-79.37289384454533,-79.37283817931163,-79.37278312762487,-79.37272828038891,-79.37267462155029,-79.37262130579717,-79.37256860603068,-79.37251670495672,-79.37246482319796,-79.37241435377646,-79.37237848500766,-79.37236386483336,-79.37184266100438,-79.37158250743451,-79.37107282782202,-79.37066655909373,-79.37058144600417,-79.36997449186198,-79.36948909693776,-79.36914188219367,-79.36889200087877,-79.3685638938437,-79.36833849826516,-79.36820143403672,-79.3680082162705,-79.36785917990842,-79.36771157265936,-79.36761474081285,-79.36751790601159,-79.36746155918449,-79.3673968821206,-79.36735718571741,-79.36731582755988,-79.36728428457427,-79.3672693355162,-79.36727146368109,-79.36727862953319,-79.36753435341596,-79.36646146345639,-79.36784084817225,-79.36791350333375,-79.36793608878784,-79.36791512770988,-79.36787070719954,-79.36769425174806,-79.367586071728,-79.36750715500469,-79.36724241979742],"lat":[43.67924497013433,43.67865064147225,43.67823677235463,43.67853357730991,43.67865274088517,43.67867625307587,43.67880540378147,43.67891254290392,43.67897954201577,43.67900584859303,43.67903709249117,43.67906411443559,43.67910048778141,43.67912355875553,43.67913293436622,43.67916469332361,43.67919605569894,43.67919637422428,43.67929069217825,43.67942528040939,43.67955094192651,43.67970818068353,43.6798156505067,43.67982867108438,43.67981494512834,43.67978329822139,43.67974228017856,43.67972383480679,43.67971935833747,43.67972343980392,43.67975384586308,43.67978860707129,43.67978791172917,43.67975497521707,43.67980237574174,43.67980692869117,43.68032428342611,43.68044310970338,43.68054331308697,43.68066624304008,43.68072995690917,43.68088905792307,43.68112944113916,43.68175489995802,43.68229457313207,43.68249793984348,43.68266022434351,43.68281379352024,43.6829579676986,43.68311116534603,43.68323276637015,43.68339052956323,43.68355299921274,43.6837697884804,43.6839098600089,43.68400930594443,43.68445783912011,43.68451019757489,43.68453636012032,43.68455802334023,43.68458868511511,43.68461032801678,43.68463197811917,43.68465361950774,43.68467524928761,43.68470146831166,43.68472308789365,43.68474020814836,43.68476182621328,43.6847789362679,43.68479604483971,43.68482214038072,43.68483473945739,43.68485182620989,43.68486891147749,43.68488600973549,43.68490308624299,43.68492015981698,43.6849327355097,43.68494530102996,43.68495786506354,43.68497043050602,43.68498749519964,43.68499554969039,43.68500808171006,43.68558871408101,43.68561842757325,43.68564815253058,43.68567787734777,43.68570760202476,43.68573282582909,43.68576705966624,43.68577124110064,43.68580130642727,43.68583105231922,43.6858652959027,43.68589513150621,43.68592488860251,43.68637543033996,43.68640570179208,43.68644046373327,43.68647973505385,43.68651002810192,43.68654931091454,43.68657960528448,43.6866234090421,43.68665821273168,43.68669752726083,43.68673243098379,43.68677625774431,43.68704799301885,43.68708747131436,43.68713153009291,43.68717550901889,43.68721499880741,43.68725899800354,43.68730299860825,43.68734699916693,43.68739102150261,43.68743066528224,43.68743504379417,43.68748357695217,43.68752762098169,43.68757625422459,43.68762482072233,43.68766888645435,43.68771745141565,43.68776153889709,43.68781462633395,43.68786321300701,43.68791182292967,43.68796053298977,43.68800915448386,43.68805777448613,43.68811541628075,43.68816405659246,43.68821720924021,43.68826587568961,43.68831905884563,43.68837232467504,43.68842552087207,43.68847871849444,43.68853192482329,43.68858514422828,43.68863837234188,43.68869160333968,43.68875385323104,43.68880719600457,43.6888604575049,43.68891823135353,43.68897150445436,43.68902929864463,43.68906929677077,43.68909159353244,43.68960637477036,43.68987731886904,43.69039676172255,43.69079614977132,43.69070491628084,43.69049371558138,43.69032016408381,43.69017117758546,43.69005508631442,43.68988827677319,43.68976353185137,43.68967155480653,43.68953367568775,43.68939651727803,43.68923687409482,43.68911396400808,43.68896854880456,43.68887322290289,43.6887417699784,43.68862858788018,43.68849746805106,43.68834398344395,43.68820864988407,43.68807365032896,43.68797923173535,43.68655570817277,43.68405537440415,43.68148646685052,43.68126686530453,43.68117716778251,43.6809202211837,43.68067203155506,43.68014721474457,43.67981709616546,43.67957732558059,43.67924497013433]}]],[[{"lng":[-79.39069150819977,-79.38953387839307,-79.38951401647702,-79.3887271005982,-79.38905503640164,-79.3888313369288,-79.38910329973326,-79.38188222517633,-79.38134216907856,-79.38088301960518,-79.38061394497116,-79.38058418631813,-79.38030576669011,-79.38013441213158,-79.3800677469125,-79.38006860933145,-79.37995026655695,-79.37752932932358,-79.37728161249993,-79.37672161718838,-79.37638152193581,-79.37479289018133,-79.3744375979351,-79.3740858810816,-79.37376466728151,-79.37337599428938,-79.37080394319386,-79.37066655909373,-79.37107282782202,-79.37158250743451,-79.37184266100438,-79.37236386483336,-79.37237848500766,-79.37241435377646,-79.37246482319796,-79.37251670495672,-79.37256860603068,-79.37262130579717,-79.37267462155029,-79.37272828038891,-79.37278312762487,-79.37283817931163,-79.37289384454533,-79.37295043014659,-79.37300762929108,-79.37306493060778,-79.37312366608867,-79.3731818876638,-79.37324121478963,-79.373301583906,-79.37336275175652,-79.37342395819144,-79.37348655766056,-79.37354905475514,-79.37361165645372,-79.37367497153028,-79.37373992287212,-79.37380489342148,-79.37386972296807,-79.37393620808623,-79.37400257152703,-79.37406895416309,-79.37413697548338,-79.37420487270835,-79.37427360728974,-79.37428221744425,-79.37434303839501,-79.37441246939899,-79.3744834343839,-79.37455439926538,-79.37462546631541,-79.37469712764923,-79.37476962630333,-79.37484284318457,-79.37491532239932,-79.37543063222869,-79.37551386835011,-79.37559788659803,-79.37568192149021,-79.37576665300814,-79.37585152501316,-79.37593685080034,-79.37602313522794,-79.37610856310364,-79.37619566548801,-79.37628336226049,-79.37637032392134,-79.37761325767114,-79.37773652381289,-79.3778606104207,-79.37798491801836,-79.37810900195282,-79.37821058642605,-79.3782331047444,-79.37835812782521,-79.37848372647038,-79.37860934396188,-79.37873496133292,-79.37886057858353,-79.37898701386938,-79.38185785229753,-79.38192642837151,-79.38199334942763,-79.38205969423153,-79.38212591804721,-79.38219224410481,-79.38225846786727,-79.38232397572364,-79.3823897067538,-79.38245523321072,-79.38251922560512,-79.38258413837963,-79.38264894884983,-79.38271282023031,-79.38277715656014,-79.38284043289399,-79.38290360692444,-79.38296628819838,-79.38302874627675,-79.38309132519881,-79.38315331149086,-79.38321517444328,-79.38327621920754,-79.38333665032049,-79.3833975927306,-79.38345734504708,-79.38351685561148,-79.38357648697502,-79.38369339738928,-79.38479324953418,-79.38485114511319,-79.3849583806748,-79.38517613721207,-79.38529985753571,-79.38524287064752,-79.3853559171659,-79.38557946679164,-79.38575787397903,-79.38576966956097,-79.38562561281582,-79.3855251614265,-79.38547850515727,-79.38544490994701,-79.38542534878547,-79.38552724158066,-79.3854290115885,-79.38530612287187,-79.38528162944505,-79.38543414646958,-79.38564206656507,-79.38584639896099,-79.38592542047658,-79.38607497537924,-79.38626873571833,-79.38637792564934,-79.3865233210505,-79.3867803449904,-79.38702358636834,-79.38718758785055,-79.3872626230807,-79.38748675299048,-79.3877282453354,-79.3879520555259,-79.38820341623693,-79.38847053484976,-79.38864447051496,-79.38870265841874,-79.38891638252737,-79.38896739326032,-79.38943597519824,-79.38959581417804,-79.3897640998869,-79.38986924250621,-79.38996870200238,-79.38999713320452,-79.39000542606452,-79.39002912819547,-79.39006078560332,-79.39010755043257,-79.39021529704723,-79.39030915597137,-79.39038782596599,-79.39045764592372,-79.39056538750111,-79.39068822563389,-79.39069150819977],"lat":[43.69133934141219,43.69158400127117,43.6915424021985,43.69172134188197,43.69251228953465,43.69253623189216,43.69310584128228,43.69465244733161,43.69508184573536,43.69477836561532,43.69499744224474,43.69505760393439,43.69511820239492,43.694998927762,43.69501985760703,43.69502203030984,43.69472769798193,43.69364443997458,43.69354181077945,43.69334031474908,43.69338499423761,43.69290331080218,43.69279473451561,43.69266370301302,43.69252851349134,43.69233393993964,43.69093322840079,43.69079614977132,43.69039676172255,43.68987731886904,43.68960637477036,43.68909159353244,43.68906929677077,43.68902929864463,43.68897150445436,43.68891823135353,43.6888604575049,43.68880719600457,43.68875385323104,43.68869160333968,43.68863837234188,43.68858514422828,43.68853192482329,43.68847871849444,43.68842552087207,43.68837232467504,43.68831905884563,43.68826587568961,43.68821720924021,43.68816405659246,43.68811541628075,43.68805777448613,43.68800915448386,43.68796053298977,43.68791182292967,43.68786321300701,43.68781462633395,43.68776153889709,43.68771745141565,43.68766888645435,43.68762482072233,43.68757625422459,43.68752762098169,43.68748357695217,43.68743504379417,43.68743066528224,43.68739102150261,43.68734699916693,43.68730299860825,43.68725899800354,43.68721499880741,43.68717550901889,43.68713153009291,43.68708747131436,43.68704799301885,43.68677625774431,43.68673243098379,43.68669752726083,43.68665821273168,43.6866234090421,43.68657960528448,43.68654931091454,43.68651002810192,43.68647973505385,43.68644046373327,43.68640570179208,43.68637543033996,43.68592488860251,43.68589513150621,43.6858652959027,43.68583105231922,43.68580130642727,43.68577124110064,43.68576705966624,43.68573282582909,43.68570760202476,43.68567787734777,43.68564815253058,43.68561842757325,43.68558871408101,43.68500808171006,43.68499554969039,43.68498749519964,43.68497043050602,43.68495786506354,43.68494530102996,43.6849327355097,43.68492015981698,43.68490308624299,43.68488600973549,43.68486891147749,43.68485182620989,43.68483473945739,43.68482214038072,43.68479604483971,43.6847789362679,43.68476182621328,43.68474020814836,43.68472308789365,43.68470146831166,43.68467524928761,43.68465361950774,43.68463197811917,43.68461032801678,43.68458868511511,43.68455802334023,43.68453636012032,43.68451019757489,43.68445783912011,43.68396018517508,43.68408253054413,43.68421907619594,43.68433926952323,43.68447154678421,43.6846012706707,43.68469288801769,43.6847726533929,43.68488769868693,43.68505440241325,43.68519198858052,43.68532109857885,43.68550047953791,43.68571164203893,43.68591832175927,43.68608638839307,43.68624703668248,43.68643443250282,43.68659612248622,43.68666579242046,43.68671824080824,43.68684724519007,43.68690687433843,43.68702601259505,43.68724029591736,43.68747597982372,43.68766257376948,43.6877383083844,43.68785876787893,43.68802311857973,43.68825831958753,43.688401013384,43.6884494306815,43.68852010304323,43.68863176256988,43.68879306474313,43.68893054742316,43.68895837387426,43.68914601816052,43.6891782442934,43.68957652616093,43.6896597967118,43.68977919405616,43.68990679353097,43.69007473181666,43.69022816635821,43.69032739520094,43.69051227009919,43.69057573008007,43.69062139888086,43.69071293669269,43.6907728618105,43.69084598589298,43.69093248827076,43.69105103167251,43.69133200356772,43.69133934141219]}]],[[{"lng":[-79.39069150819977,-79.39187917728884,-79.39189763978453,-79.39192892615606,-79.39196980166639,-79.39198431493256,-79.39205971880841,-79.39226100394745,-79.39235215346052,-79.39236117493448,-79.39228638647455,-79.39212572139331,-79.39212581645997,-79.3921256094941,-79.39215197038629,-79.39196562655498,-79.39201828674122,-79.39256166490037,-79.39256207643112,-79.39267580682916,-79.39267570214365,-79.39527200102582,-79.39609321294949,-79.38621714646899,-79.37312813431124,-79.37283400186757,-79.37249109825385,-79.37172560939625,-79.37202730541564,-79.37260913774887,-79.37320923376437,-79.37380537406389,-79.37443755303239,-79.37633441390129,-79.37852308913595,-79.37883653546575,-79.37935983231648,-79.3797117286755,-79.37993730941297,-79.38006860933145,-79.3800677469125,-79.38013441213158,-79.38030576669011,-79.38058418631813,-79.38061394497116,-79.38088301960518,-79.38134216907856,-79.38188222517633,-79.38910329973326,-79.3888313369288,-79.38905503640164,-79.3887271005982,-79.38951401647702,-79.38953387839307,-79.39069150819977],"lat":[43.69133934141219,43.69108825099425,43.6911133563116,43.69122524130039,43.69121186337728,43.69123160193099,43.69121645932139,43.69131507256444,43.69160288846781,43.69166350880381,43.69172069959675,43.6917560675293,43.69175633892679,43.69175642603479,43.69181909074629,43.69185814805878,43.69198194634033,43.69237028524269,43.69237020101131,43.69234524479854,43.69234533334611,43.69177643650618,43.69565315196622,43.69749941143102,43.7003573031403,43.69955626000053,43.69872751547862,43.69674472623215,43.69669051165256,43.69659977704589,43.69649579648655,43.69636015950475,43.69621612008524,43.69582440431555,43.69536028100035,43.6953062151278,43.69519202066935,43.69511599269545,43.69506067795322,43.69502203030984,43.69501985760703,43.694998927762,43.69511820239492,43.69505760393439,43.69499744224474,43.69477836561532,43.69508184573536,43.69465244733161,43.69310584128228,43.69253623189216,43.69251228953465,43.69172134188197,43.6915424021985,43.69158400127117,43.69133934141219]}]],[[{"lng":[-79.39703264757955,-79.39699999855985,-79.39008798352425,-79.39020397357419,-79.38782489703539,-79.38781914744938,-79.38748275390003,-79.38701855698309,-79.38696093831875,-79.38682170263515,-79.38673213392967,-79.38651769905233,-79.3864659511336,-79.38621714646899,-79.39609321294949,-79.39619253638269,-79.39645944037431,-79.39663094493483,-79.39684951096248,-79.39694204464537,-79.39702700069658,-79.39703264757955],"lat":[43.7003158606883,43.70031603285351,43.7016960020966,43.7019849793643,43.70247976714129,43.70246519283853,43.7016410862402,43.70050010529944,43.70030565934453,43.6998309997299,43.6992940271854,43.69832301621796,43.69816925179995,43.69749941143102,43.69565315196622,43.69612273955688,43.69736902987193,43.69829423023077,43.69932370329258,43.69977969173775,43.70028967571998,43.7003158606883]}]],[[{"lng":[-79.37467143770039,-79.37484502509189,-79.3749889246197,-79.37515543724234,-79.37533557488031,-79.37551912821084,-79.37570233612502,-79.37583981226481,-79.37602202976547,-79.37614652830915,-79.37629789598932,-79.37648797323983,-79.37664969280488,-79.376787246451,-79.37697342688317,-79.37717609383911,-79.37725477107969,-79.37729965310652,-79.37740259844463,-79.37833435769508,-79.37838900325755,-79.37868940146679,-79.37903342392705,-79.37943950477037,-79.37989347472269,-79.38055411461119,-79.37945595409327,-79.37940293405153,-79.37905577298925,-79.37684208424197,-79.37644351530244,-79.37630427835406,-79.37615021794055,-79.37603795068392,-79.37593972950822,-79.37588013861573,-79.37489678515858,-79.37359230573503,-79.37316513058396,-79.37144627680946,-79.37143177735494,-79.37140492561414,-79.37136572159932,-79.37131414587033,-79.37125586087531,-79.3712070019911,-79.37115979855096,-79.37111400712574,-79.37105715132844,-79.37100865782256,-79.37095971160858,-79.37093007033621,-79.37090785402934,-79.37088598346551,-79.37085787599484,-79.37039495019492,-79.36981883148481,-79.36968685829299,-79.36954973105055,-79.36973901533074,-79.37090113247426,-79.37219551450923,-79.37376509271596,-79.37467143770039],"lat":[43.66708750731662,43.66697294957907,43.66686697146682,43.66675681366803,43.66666476336208,43.66658635438897,43.66651694224041,43.66647838697904,43.66644946937853,43.66643323440821,43.66642188207368,43.66642008149185,43.66642687981338,43.66644233669691,43.6664674860514,43.66651087327723,43.66649848743445,43.66649912472745,43.66647808133417,43.66627966907457,43.66643807005416,43.66711757451324,43.66792831633591,43.6688885605431,43.66996632838907,43.67160541870059,43.67181940331444,43.67183215449595,43.67190833918069,43.6723496153445,43.67244297592875,43.67245909180827,43.67247490686778,43.67247331157498,43.67246291376134,43.67245756588089,43.67235806736391,43.67219997854281,43.67214429718537,43.67188126208051,43.67187655444477,43.67186716976669,43.67185310803774,43.67183886997909,43.67181553425412,43.67178783183487,43.67175565197866,43.67172799323983,43.67168208270866,43.67164088237906,43.67158617254213,43.67153623888801,43.67149541309254,43.6714455901988,43.67139567838987,43.6702500521671,43.66885075255833,43.66851570494248,43.66816708058514,43.66816077891681,43.66790278914846,43.66762857748618,43.66729075229741,43.66708750731662]}]],[[{"lng":[-79.38782489703539,-79.39020397357419,-79.39008798352425,-79.39699999855985,-79.39703264757955,-79.39714695677642,-79.39720703669599,-79.39733223535826,-79.39749311262291,-79.39765384984683,-79.39767196221902,-79.39781338932413,-79.39782464942931,-79.39796091237328,-79.39940349129647,-79.39953392683663,-79.39983365350234,-79.40002052437748,-79.40001292265167,-79.39994460270674,-79.40004360909138,-79.40009721675167,-79.3983543369483,-79.39681735617505,-79.39548060558543,-79.3941211109054,-79.39280708288449,-79.39148869521263,-79.39018524875107,-79.38928006135832,-79.3889218618416,-79.38855848224784,-79.38814460381401,-79.38782489703539],"lat":[43.70247976714129,43.7019849793643,43.7016960020966,43.70031603285351,43.7003158606883,43.70084957141497,43.70112956563618,43.70178855641758,43.7025066502319,43.70328775564987,43.70339153240253,43.70414085980001,43.70422527620596,43.70419657982329,43.70390082468786,43.70425642939,43.70428826281609,43.70473845780552,43.70511652454405,43.70566108790566,43.70625966535469,43.70637969209764,43.70672795975194,43.70705316382807,43.70732246586411,43.70759602362034,43.70785670166484,43.70813071613963,43.70840051488058,43.70607830329136,43.70517278618696,43.70428528877204,43.70328914408132,43.70247976714129]}]],[[{"lng":[-79.3983543369483,-79.40009721675167,-79.40027697757797,-79.40063722945686,-79.40062534652597,-79.40084524483488,-79.40085985906259,-79.40103275401378,-79.40104981284222,-79.40250626404799,-79.40374196988046,-79.40395896864956,-79.40412899616128,-79.40420596143974,-79.4023676595442,-79.40104696800768,-79.4011709838246,-79.39946470639697,-79.3995514122764,-79.39893784271946,-79.39857630195817,-79.39785865308569,-79.39550260097553,-79.39310832936367,-79.39218882370785,-79.39176738277088,-79.39144590428677,-79.39121473793242,-79.39064540089342,-79.39018524875107,-79.39148869521263,-79.39280708288449,-79.3941211109054,-79.39548060558543,-79.39681735617505,-79.3983543369483],"lat":[43.70672795975194,43.70637969209764,43.70678198648484,43.70770946153311,43.70789842673292,43.70828111746074,43.70860521302275,43.70871187519757,43.70916347286994,43.70886958340017,43.70861301093139,43.70942198865276,43.70963698985349,43.70982602586467,43.71024962869494,43.71055603825585,43.71148497645646,43.71183323255521,43.71228265497392,43.71240404856988,43.7124664973559,43.71261406259586,43.71310321042617,43.71359627312162,43.71378147289223,43.71260051592795,43.71170461636155,43.71112514618149,43.70960902949856,43.70840051488058,43.70813071613963,43.70785670166484,43.70759602362034,43.70732246586411,43.70705316382807,43.70672795975194]}]],[[{"lng":[-79.39018524875107,-79.39064540089342,-79.39121473793242,-79.39144590428677,-79.39176738277088,-79.39218882370785,-79.39310832936367,-79.39550260097553,-79.39785865308569,-79.39857630195817,-79.39893784271946,-79.3995514122764,-79.39970572414673,-79.39980957819941,-79.3998280364227,-79.40002565374697,-79.40009694045273,-79.40021941039898,-79.40043144896519,-79.40047026100886,-79.39911267762086,-79.398760691094,-79.39767952692883,-79.39414533205853,-79.38842706527684,-79.3882819400104,-79.38825301917475,-79.38737270862637,-79.38733625629949,-79.38708218650874,-79.38662713819146,-79.38640944377528,-79.38636205277537,-79.38619873430787,-79.38601081392919,-79.38569676468298,-79.38460918984322,-79.38446737360854,-79.38395425031619,-79.38381248022516,-79.38293965396662,-79.38196911826023,-79.38148122142437,-79.38149813880639,-79.38103474979737,-79.38004580314477,-79.37897177144852,-79.37890949251934,-79.37885527255224,-79.37747763058449,-79.37737304154105,-79.37729920270006,-79.37714605519656,-79.37833457612136,-79.37963138088372,-79.38019530371758,-79.38091828399104,-79.38123182768513,-79.38223616047937,-79.38319581697414,-79.3835787950806,-79.38489657239397,-79.38618838196,-79.38753061912861,-79.3878768208006,-79.38886684273177,-79.39018524875107],"lat":[43.70840051488058,43.70960902949856,43.71112514618149,43.71170461636155,43.71260051592795,43.71378147289223,43.71359627312162,43.71310321042617,43.71261406259586,43.7124664973559,43.71240404856988,43.71228265497392,43.71299552145369,43.71352377091075,43.71364555593574,43.7145622025966,43.71494128321808,43.71544719784512,43.71636854642281,43.7165987297686,43.71685878006554,43.71693045518219,43.71715385250538,43.71788318195216,43.71904522206446,43.71907468215274,43.71893924489564,43.71559538028001,43.71545083447365,43.71479892737269,43.71362656939227,43.71306528385151,43.71306911564847,43.71264812794584,43.71216377875047,43.71134898569514,43.71157676000333,43.7116017610953,43.7116980283884,43.71168702145216,43.71186821599106,43.71207061533801,43.71217623065029,43.71221697910166,43.7123094366627,43.7125070578142,43.71271687637761,43.71275659176536,43.71277832749952,43.71308734701247,43.71239262072741,43.71202249169033,43.7110299206441,43.71076764210991,43.7105202995869,43.7104112688194,43.71026398009626,43.7102053182606,43.71001249553166,43.7098099398784,43.70973434003331,43.70945140643877,43.70918600481327,43.70892130011465,43.708854168866,43.70864758287835,43.70840051488058]}]],[[{"lng":[-79.40061450496347,-79.40083943460988,-79.40093709162787,-79.4011298703641,-79.40117732052595,-79.40139052578066,-79.40158320345598,-79.40179518364664,-79.40183147330791,-79.40192824752953,-79.40205584692515,-79.40217888971405,-79.40218355351945,-79.40071956786213,-79.40000141148415,-79.39929585654291,-79.39919281623818,-79.39884533010554,-79.39755723887444,-79.39723523707809,-79.39638889213956,-79.3960841164932,-79.39514122474768,-79.39395859750778,-79.39198232605149,-79.39006509165802,-79.38986417839745,-79.38964880330056,-79.38944357510375,-79.3892387078679,-79.38903429276044,-79.38882164994935,-79.38876806514861,-79.3882819400104,-79.38842706527684,-79.39414533205853,-79.39767952692883,-79.398760691094,-79.39911267762086,-79.40047026100886,-79.40061450496347],"lat":[43.71724447903146,43.71832813233256,43.71874367909572,43.71954322996499,43.71997607787876,43.72097395699665,43.72196254647896,43.72309552672869,43.72345170144143,43.72378162611699,43.72431911666058,43.72495106379162,43.72509525023714,43.72540343195865,43.72556441595408,43.72572107096803,43.72574213243264,43.72582737360485,43.72608396745987,43.72614246522188,43.72633772625003,43.72638746054996,43.72659026908227,43.72685280435107,43.72725269420354,43.72767588642339,43.726691663586,43.72566663766622,43.72466434908256,43.72367556799023,43.72267779080359,43.72166639416443,43.72140458199502,43.71907468215274,43.71904522206446,43.71788318195216,43.71715385250538,43.71693045518219,43.71685878006554,43.7165987297686,43.71724447903146]}]],[[{"lng":[-79.40235204005178,-79.40250341804757,-79.40282694898322,-79.40293205310925,-79.40305414452374,-79.40311630641165,-79.4032824395962,-79.40333738744658,-79.40345511767237,-79.40354329353734,-79.40370435131302,-79.40375082469062,-79.40396760985001,-79.40399234458927,-79.40419587748488,-79.40420952230716,-79.40438447561232,-79.40444264203157,-79.40451929775787,-79.40454012521897,-79.40341047248397,-79.39997719331399,-79.39239638916499,-79.3912556737379,-79.39063658828762,-79.39041847960584,-79.39026980657793,-79.3902862630068,-79.39020864823539,-79.39015253711618,-79.39006509165802,-79.39198232605149,-79.39395859750778,-79.39514122474768,-79.3960841164932,-79.39638889213956,-79.39723523707809,-79.39755723887444,-79.39884533010554,-79.39919281623818,-79.39929585654291,-79.40000141148415,-79.40071956786213,-79.40218355351945,-79.40235204005178],"lat":[43.72567382081263,43.72628374876085,43.72742719766478,43.72782934331447,43.72831274375684,43.72857025792891,43.72915329491714,43.72942421105202,43.72992555390105,43.73025994774761,43.73086091695279,43.73112721346489,43.73187758270659,43.73196785756569,43.73289357901709,43.73299279090335,43.73365705654143,43.73389191912792,43.7341631376006,43.73425794874415,43.73450331715708,43.73525214753368,43.73690607775287,43.73243307617469,43.72998447314131,43.72914854150188,43.72855663996518,43.72843534554642,43.72831722691861,43.72809579880385,43.72767588642339,43.72725269420354,43.72685280435107,43.72659026908227,43.72638746054996,43.72633772625003,43.72614246522188,43.72608396745987,43.72582737360485,43.72574213243264,43.72572107096803,43.72556441595408,43.72540343195865,43.72509525023714,43.72567382081263]}]],[[{"lng":[-79.39006509165802,-79.38951975090619,-79.38669949269733,-79.38653065586779,-79.38638738614993,-79.38623588203529,-79.38604478977467,-79.38588844720728,-79.38569334418868,-79.38552525987355,-79.38536695398081,-79.38495910049357,-79.38446367405814,-79.38397377390341,-79.38358101585089,-79.38331412895567,-79.38307610915585,-79.38295233476207,-79.38287608183026,-79.38217752335126,-79.3820851010364,-79.38187421613338,-79.38185261839338,-79.3814492429341,-79.38035233414723,-79.37978050349416,-79.37906731102495,-79.37870407052367,-79.37808049713723,-79.37796914012002,-79.37760998701586,-79.37747763058449,-79.37885527255224,-79.37890949251934,-79.37897177144852,-79.38004580314477,-79.38103474979737,-79.38149813880639,-79.38148122142437,-79.38196911826023,-79.38293965396662,-79.38381248022516,-79.38395425031619,-79.38446737360854,-79.38460918984322,-79.38569676468298,-79.38601081392919,-79.38619873430787,-79.38636205277537,-79.38640944377528,-79.38662713819146,-79.38708218650874,-79.38733625629949,-79.38737270862637,-79.38825301917475,-79.3882819400104,-79.38876806514861,-79.38882164994935,-79.38903429276044,-79.3892387078679,-79.38944357510375,-79.38964880330056,-79.38986417839745,-79.39006509165802],"lat":[43.72767588642339,43.72778522730935,43.728375761592,43.72839138170012,43.72841186356128,43.72841872604137,43.72842052822013,43.72840931801638,43.72836605324102,43.72832316986575,43.72826692149604,43.72816663750066,43.72803351718675,43.72790056289571,43.72781399037185,43.72775620318544,43.72772141871997,43.72770607433622,43.72768249040682,43.72756908024129,43.72754526686211,43.72748376822906,43.72736193616171,43.72663588790448,43.72572897342458,43.72500953332848,43.7217580873958,43.71971363203925,43.71671100264626,43.71606560199088,43.71384557110965,43.71308734701247,43.71277832749952,43.71275659176536,43.71271687637761,43.7125070578142,43.7123094366627,43.71221697910166,43.71217623065029,43.71207061533801,43.71186821599106,43.71168702145216,43.7116980283884,43.7116017610953,43.71157676000333,43.71134898569514,43.71216377875047,43.71264812794584,43.71306911564847,43.71306528385151,43.71362656939227,43.71479892737269,43.71545083447365,43.71559538028001,43.71893924489564,43.71907468215274,43.72140458199502,43.72166639416443,43.72267779080359,43.72367556799023,43.72466434908256,43.72566663766622,43.726691663586,43.72767588642339]}]],[[{"lng":[-79.37492364358467,-79.37460261264278,-79.37412097483453,-79.37377860188603,-79.37336297418558,-79.37312813431124,-79.38621714646899,-79.3864659511336,-79.38651769905233,-79.38673213392967,-79.38682170263515,-79.38696093831875,-79.38701855698309,-79.38748275390003,-79.38781914744938,-79.38814460381401,-79.38855848224784,-79.3889218618416,-79.38928006135832,-79.39018524875107,-79.38886684273177,-79.3878768208006,-79.38753061912861,-79.38618838196,-79.38489657239397,-79.3835787950806,-79.38319581697414,-79.38223616047937,-79.38123182768513,-79.38091828399104,-79.38019530371758,-79.37963138088372,-79.37833457612136,-79.37714605519656,-79.37708877982956,-79.37704040258336,-79.37699445862286,-79.37694146548731,-79.37632609021958,-79.37599318004906,-79.37563667221599,-79.37492364358467],"lat":[43.70510078705213,43.704263358595,43.70299605027606,43.70205929383686,43.70093686292952,43.7003573031403,43.69749941143102,43.69816925179995,43.69832301621796,43.6992940271854,43.6998309997299,43.70030565934453,43.70050010529944,43.7016410862402,43.70246519283853,43.70328914408132,43.70428528877204,43.70517278618696,43.70607830329136,43.70840051488058,43.70864758287835,43.708854168866,43.70892130011465,43.70918600481327,43.70945140643877,43.70973434003331,43.7098099398784,43.71001249553166,43.7102053182606,43.71026398009626,43.7104112688194,43.7105202995869,43.71076764210991,43.7110299206441,43.71088048477734,43.71065924953921,43.71049197061458,43.7102976756651,43.70869972626345,43.70780812153824,43.70693418424479,43.70510078705213]}]],[[{"lng":[-79.37249109825385,-79.37283400186757,-79.37312813431124,-79.37336297418558,-79.37377860188603,-79.37412097483453,-79.37460261264278,-79.37492364358467,-79.37563667221599,-79.37599318004906,-79.37632609021958,-79.37694146548731,-79.37699445862286,-79.37704040258336,-79.37708877982956,-79.37714605519656,-79.37388901291646,-79.37356375456464,-79.37205847797308,-79.36964745111467,-79.36721713257899,-79.36479033599213,-79.36443514613259,-79.36408019048697,-79.36371906489478,-79.36351440782676,-79.36278237886361,-79.36245747072203,-79.36227395388751,-79.36180600257603,-79.36146984570156,-79.36112144220357,-79.360848957893,-79.36080985515395,-79.36073967582085,-79.36067252330854,-79.36063771896099,-79.36056676209559,-79.36054652677778,-79.36068739280078,-79.36080826669523,-79.36098689668012,-79.36114234429124,-79.36131313748773,-79.36147191806197,-79.36156263170665,-79.3624083664622,-79.36352425059961,-79.36416990417391,-79.36461453861941,-79.36527400850247,-79.36558971544983,-79.36681764849399,-79.366904688104,-79.36698806494353,-79.36708629340471,-79.36718943463632,-79.36726062735953,-79.3681851468687,-79.36931791491106,-79.37038439306315,-79.37077663409754,-79.37136781522534,-79.37172560939625,-79.37249109825385],"lat":[43.69872751547862,43.69955626000053,43.7003573031403,43.70093686292952,43.70205929383686,43.70299605027606,43.704263358595,43.70510078705213,43.70693418424479,43.70780812153824,43.70869972626345,43.7102976756651,43.71049197061458,43.71065924953921,43.71088048477734,43.7110299206441,43.71172637185356,43.71179375870874,43.71208748330513,43.71256631240153,43.71307191076719,43.71355941377632,43.71267195842467,43.71180259915311,43.71089705244986,43.71035382391897,43.70854258521322,43.70771406971117,43.70727475430385,43.70611111379974,43.70526443,43.70441306848734,43.70376542923559,43.70363433972761,43.70347570847738,43.70334871753015,43.70326269948799,43.70316265982106,43.70313536359324,43.70309228455702,43.70303550563409,43.70294354707669,43.70281974886239,43.70267357552998,43.70253632169099,43.70245210358104,43.70170337926928,43.70052183158308,43.69996386108513,43.69958755088832,43.69909278562623,43.69883615392351,43.69794432933438,43.69787805835745,43.6978072339805,43.69773212076833,43.69767958262437,43.69765800484463,43.69745516270135,43.69721018493521,43.6969868456006,43.69690682951621,43.69678463755631,43.69674472623215,43.69872751547862]}]],[[{"lng":[-79.38144327843139,-79.38125706093891,-79.38111532522471,-79.3810780161311,-79.38094569775133,-79.38089996047707,-79.38076731196638,-79.3806541119822,-79.38051473423147,-79.38037678460063,-79.38021410110433,-79.38009514335876,-79.37994694504756,-79.37975331224935,-79.37954943440427,-79.37934071645374,-79.37915709020635,-79.37904094367848,-79.3790950861042,-79.37913561321776,-79.37904211904132,-79.37886859827194,-79.37860699934767,-79.37851199390717,-79.37842622650017,-79.3781795563329,-79.37789111353477,-79.37764720400762,-79.37741404797781,-79.37724780524195,-79.37702110843425,-79.37679665987389,-79.37659221466653,-79.37641135241741,-79.37623497077793,-79.37609298463516,-79.37600632179424,-79.37593130415722,-79.37569950379749,-79.37545833522674,-79.3752343136089,-79.37501916376452,-79.37481179328057,-79.37460669174858,-79.37438568157962,-79.37422247457904,-79.37406137674395,-79.37391830007981,-79.37379782537687,-79.37380740872663,-79.37394995592932,-79.37418686222144,-79.37424546576321,-79.37412504479073,-79.37394044555542,-79.37378934371668,-79.37366372791038,-79.37351437812326,-79.37330790027585,-79.37307643837889,-79.37281504923472,-79.37257775852645,-79.37238732520834,-79.37217637465585,-79.37202849240816,-79.37196892387054,-79.37208858440027,-79.37233103729193,-79.37257112207115,-79.37252379875954,-79.37239089157529,-79.37208642306601,-79.37182842349631,-79.37157395039243,-79.37128432272416,-79.37098684172692,-79.3708100277784,-79.37089512716757,-79.3707887045768,-79.37053974756259,-79.3703139188878,-79.3700261614285,-79.36976355086802,-79.36948949997505,-79.36924819719967,-79.36901179913738,-79.36873873916839,-79.36853275081249,-79.36835339201447,-79.36815216895066,-79.36813349734784,-79.36795233495599,-79.36771274702836,-79.36747022645801,-79.36743570459116,-79.36733930427179,-79.36705984273262,-79.36677865961481,-79.36664825550308,-79.36653027025666,-79.36607842431432,-79.36585271304156,-79.3657777567397,-79.3657346102094,-79.36519507498072,-79.3650486269366,-79.36493115986113,-79.36475039821988,-79.36469676517255,-79.36469657924924,-79.36471682440231,-79.36481999873739,-79.36490818887407,-79.36498172901,-79.36499730359242,-79.36489617937356,-79.36483289627611,-79.36480025340278,-79.36473011742102,-79.36470781199328,-79.36480079539959,-79.3649985056858,-79.36513638793113,-79.36527476119666,-79.3654061726463,-79.36552403347378,-79.36557458355409,-79.36553302487818,-79.36544196613411,-79.36516285310597,-79.36484165347998,-79.36455585477624,-79.36431463060985,-79.36416994816939,-79.36401812117602,-79.36381511316293,-79.36370055408919,-79.36348348413063,-79.36324000176573,-79.36311484802195,-79.36242876431575,-79.36216917605658,-79.36211465397548,-79.36574081825894,-79.36690841106206,-79.36939136784162,-79.37182151619821,-79.37808049713723,-79.37870407052367,-79.37906731102495,-79.37978050349416,-79.38035233414723,-79.3814492429341,-79.38185261839338,-79.38187421613338,-79.38189445787661,-79.38189617833444,-79.38188796775565,-79.38144327843139],"lat":[43.73023270928385,43.73014455309852,43.73009267447124,43.73007900307537,43.73000061191515,43.72997295805946,43.7298270474678,43.72968132230768,43.72955340995622,43.72940292280461,43.72922048807599,43.72905676687349,43.72891054485957,43.72878627331995,43.72868886199219,43.72862288844877,43.72867429566364,43.72883918407152,43.72898407314565,43.729299715911,43.7294560136876,43.72956157541289,43.72953535904558,43.72937197606333,43.72915462243184,43.72901159171055,43.72894448398355,43.72892301749298,43.72899622354714,43.72914689585852,43.72926520253061,43.72938363070461,43.72951575550488,43.72966171791717,43.7298123346955,43.7299858548223,43.73014665811365,43.73031221782973,43.73035393284216,43.73026498610255,43.73014018488748,43.72997509068983,43.72981010672395,43.72961805881705,43.72941237133477,43.72918941192275,43.7289755742408,43.72878440759839,43.7286206582632,43.72846326069352,43.72830316424459,43.72797787425371,43.7278211740601,43.72762141804486,43.72744766447827,43.72724297054243,43.72703854910694,43.72687888944406,43.72679493287318,43.7268321464483,43.72685543049145,43.72682054451673,43.72671881089825,43.72676981771144,43.72695675178568,43.72712243905019,43.72725926293954,43.72730772637203,43.72738316138831,43.72758062013893,43.72773626114035,43.72770941922921,43.72761122309111,43.72752648951592,43.72749985695593,43.72739659520868,43.72722303759254,43.72685967220344,43.72669161902062,43.72663396760694,43.72671635563912,43.72670775044689,43.72663639967378,43.72658297896175,43.72666964505955,43.7268283061633,43.72696843953495,43.72712312278917,43.72727359491524,43.72725271765326,43.72705431829385,43.72689419706978,43.72680075556302,43.72681529495213,43.72703084833881,43.72723660570968,43.72729562612334,43.72716558049675,43.72693857808609,43.72668033626872,43.72626419804041,43.72608984354876,43.72584571916909,43.72560646061815,43.7249595151579,43.7248673104793,43.7247936142,43.72459298484611,43.72447969303467,43.72443008966201,43.72422783563525,43.7239726666209,43.72372637506152,43.72350678962709,43.72327296188099,43.72302387141531,43.72277991333512,43.72252730170144,43.72228315546359,43.72207579147324,43.72187898920615,43.72174228801445,43.72155071885694,43.72134106263317,43.7211313966686,43.72090803380199,43.7207017119854,43.72054349342687,43.72047017535433,43.72043467574165,43.72052009954901,43.7205610193589,43.72061157844757,43.72076713151765,43.72087298137489,43.72075304948549,43.72053527233108,43.72035662581668,43.72029462592495,43.7201352090952,43.72014338318429,43.72004514240491,43.71999485021841,43.71915089289482,43.71892444241035,43.71846461232136,43.71795005404406,43.71671100264626,43.71971363203925,43.7217580873958,43.72500953332848,43.72572897342458,43.72663588790448,43.72736193616171,43.72748376822906,43.72770919345244,43.72793426638243,43.7280376724578,43.73023270928385]}]],[[{"lng":[-79.37388901291646,-79.37714605519656,-79.37729920270006,-79.37737304154105,-79.37747763058449,-79.37760998701586,-79.37796914012002,-79.37808049713723,-79.37182151619821,-79.36939136784162,-79.36690841106206,-79.36574081825894,-79.36211465397548,-79.35322178409335,-79.35311640850098,-79.35292905172834,-79.3526993846237,-79.35265408419538,-79.35262381129414,-79.35268489183376,-79.35261592793478,-79.35330831235153,-79.3541606777341,-79.35494380773456,-79.35591188332974,-79.35684383593426,-79.35742955356037,-79.3579280114668,-79.35870359036822,-79.35981302484076,-79.36230583738967,-79.36479033599213,-79.36721713257899,-79.36964745111467,-79.37205847797308,-79.37356375456464,-79.37388901291646],"lat":[43.71172637185356,43.7110299206441,43.71202249169033,43.71239262072741,43.71308734701247,43.71384557110965,43.71606560199088,43.71671100264626,43.71795005404406,43.71846461232136,43.71892444241035,43.71915089289482,43.71999485021841,43.72170379824374,43.72092802222198,43.71997993736276,43.71777069902423,43.71769803037309,43.71740944148492,43.71727979371607,43.71622539101975,43.71601923360618,43.71580196334403,43.71561960782176,43.71542649503787,43.71525526948845,43.71514216374568,43.71504580504157,43.71485431322164,43.71460018385011,43.71408204602525,43.71355941377632,43.71307191076719,43.71256631240153,43.71208748330513,43.71179375870874,43.71172637185356]}]],[[{"lng":[-79.37788316653594,-79.37833435769508,-79.37740259844463,-79.37729965310652,-79.37725477107969,-79.37717609383911,-79.37697342688317,-79.376787246451,-79.37664969280488,-79.37648797323983,-79.37629789598932,-79.37614652830915,-79.37602202976547,-79.37583981226481,-79.37570233612502,-79.37551912821084,-79.37533557488031,-79.37515543724234,-79.3749889246197,-79.37484502509189,-79.37467143770039,-79.37376509271596,-79.37219551450923,-79.37090113247426,-79.36973901533074,-79.36954973105055,-79.36917884537964,-79.36910808713215,-79.36885591850874,-79.36861963084363,-79.36828590132312,-79.36804383639215,-79.36796897395776,-79.36929475361349,-79.37050735247765,-79.37150506798167,-79.37215205974096,-79.3722533850442,-79.37304872154483,-79.37433665171579,-79.37670442353101,-79.37788316653594],"lat":[43.66521084592367,43.66627966907457,43.66647808133417,43.66649912472745,43.66649848743445,43.66651087327723,43.6664674860514,43.66644233669691,43.66642687981338,43.66642008149185,43.66642188207368,43.66643323440821,43.66644946937853,43.66647838697904,43.66651694224041,43.66658635438897,43.66666476336208,43.66675681366803,43.66686697146682,43.66697294957907,43.66708750731662,43.66729075229741,43.66762857748618,43.66790278914846,43.66816077891681,43.66816708058514,43.66726149579392,43.6670984495284,43.66645102549646,43.66586693178186,43.66505180448728,43.66445862494565,43.66427742521685,43.66398569048605,43.66371492170568,43.66350850342528,43.66338269055721,43.66335712769619,43.66318382133743,43.66290958321159,43.66241184991879,43.66521084592367]}]],[[{"lng":[-79.35882925552387,-79.35889216779633,-79.35901504634633,-79.35910338971354,-79.35915402660169,-79.3592284782175,-79.35929114288791,-79.35936874541585,-79.35947934907642,-79.35959547716253,-79.35971172820148,-79.35982491055815,-79.35990899992086,-79.35998633784256,-79.36006146369752,-79.36015318518686,-79.36025810333483,-79.3603601782804,-79.36044093428437,-79.36054652677778,-79.36056676209559,-79.36063771896099,-79.36067252330854,-79.36073967582085,-79.36080985515395,-79.360848957893,-79.36112144220357,-79.36146984570156,-79.36180600257603,-79.36227395388751,-79.36245747072203,-79.36278237886361,-79.36351440782676,-79.36371906489478,-79.36408019048697,-79.36443514613259,-79.36479033599213,-79.36230583738967,-79.35981302484076,-79.35870359036822,-79.3579280114668,-79.35742955356037,-79.35684383593426,-79.35591188332974,-79.35494380773456,-79.3541606777341,-79.35330831235153,-79.35261592793478,-79.35247846177306,-79.35240362898388,-79.35230911187077,-79.35207390891597,-79.35175313252516,-79.35140236108015,-79.35134107185503,-79.35106867698457,-79.35069909349723,-79.35033728335648,-79.35000855971123,-79.34987492323505,-79.34960600871807,-79.34926467317983,-79.3489076417453,-79.34855038238585,-79.34821728749721,-79.34921622508304,-79.34924863514779,-79.34925863078892,-79.34927046806322,-79.34928281935353,-79.34929596544855,-79.34931005355257,-79.34932475551004,-79.34934037826012,-79.34935661485953,-79.34937377224679,-79.34939175057627,-79.34941093295704,-79.34943044344094,-79.34945067007686,-79.34947161286273,-79.34949406910535,-79.34951664881156,-79.34953982115822,-79.34956393793394,-79.34958874719241,-79.34961429377651,-79.34964145612432,-79.34966863960241,-79.34969641572135,-79.34972501028578,-79.34975454924829,-79.3497846783828,-79.34981572826821,-79.34984749427876,-79.34987997641257,-79.34991317466776,-79.34994719135386,-79.34998047064389,-79.35001156495917,-79.35004345407734,-79.35007605930961,-79.35010878793346,-79.35014292769654,-79.35017788587807,-79.3502129463014,-79.35024933916141,-79.35028583179135,-79.35032396131486,-79.35036198845332,-79.3504015501692,-79.35044098837872,-79.35048147073233,-79.3505226716441,-79.3505644627562,-79.35060709337627,-79.35065041897769,-79.35069468638558,-79.35073883030222,-79.35078463218044,-79.35083031057181,-79.35087752598599,-79.35088792245918,-79.35092484115665,-79.35097277009625,-79.35102151742086,-79.35107118543718,-79.35228424248848,-79.35369825499866,-79.35505368203255,-79.35586920396533,-79.35627894123346,-79.35882925552387],"lat":[43.70205271168277,43.70207612012263,43.70211839360638,43.70215116909132,43.70217440112194,43.70220247605372,43.70223497271359,43.70226759378075,43.7023096905353,43.70237437142299,43.70243455296957,43.70249469035326,43.70256341205696,43.7026320368055,43.70269171780129,43.70276955082523,43.70284757313542,43.70294355850279,43.70302573494543,43.70313536359324,43.70316265982106,43.70326269948799,43.70334871753015,43.70347570847738,43.70363433972761,43.70376542923559,43.70441306848734,43.70526443,43.70611111379974,43.70727475430385,43.70771406971117,43.70854258521322,43.71035382391897,43.71089705244986,43.71180259915311,43.71267195842467,43.71355941377632,43.71408204602525,43.71460018385011,43.71485431322164,43.71504580504157,43.71514216374568,43.71525526948845,43.71542649503787,43.71561960782176,43.71580196334403,43.71601923360618,43.71622539101975,43.71614680256367,43.71593417772649,43.71573018094298,43.71565477368988,43.71569515709457,43.71569459807318,43.7156937138445,43.71569428454275,43.71573396069827,43.71577383789424,43.71585010948129,43.71602371790559,43.71613235880767,43.7162130370916,43.7163023995014,43.71628823504428,43.71618439923827,43.71297100321066,43.7128679488879,43.7128095805559,43.71275123882575,43.71269281449934,43.7126389926508,43.7125806834286,43.71252238307124,43.71246409601211,43.71240581781712,43.71234755291963,43.71228920985741,43.71223547518231,43.71217724426324,43.71211902368425,43.71206081344498,43.71200712604037,43.71194893943451,43.71189526236275,43.71183700792682,43.71178335448577,43.71172521070789,43.71167159123864,43.71161347108775,43.71155986046841,43.71150626166133,43.7114480854853,43.7113945088232,43.71134094544892,43.71128739240679,43.71123384969618,43.71118031731645,43.71112679674429,43.71107776649422,43.71101511172487,43.71095705942214,43.71089901744944,43.71083647626789,43.7107784564305,43.71072044839816,43.71066244183056,43.7106043644682,43.71054637855352,43.71048841625711,43.7104304524693,43.71037251082062,43.71031906835558,43.71026113996755,43.7102031319169,43.71014972337032,43.71009182594412,43.71003843951331,43.70998056567745,43.70992719102263,43.70986933929697,43.70981598675137,43.70976256634866,43.70974921343947,43.70970923738501,43.7096559172582,43.70960260892011,43.70954931384585,43.7083647791388,43.70701208914198,43.70572605271852,43.70492293067291,43.7045236463357,43.70205271168277]}]],[[{"lng":[-79.34821728749721,-79.3481995051345,-79.3478538626469,-79.34747042781298,-79.34709043828322,-79.3467919405893,-79.3468815072976,-79.34691230175034,-79.34672306234252,-79.34636377622404,-79.34599503242234,-79.34563496059673,-79.34528496396636,-79.34493263969627,-79.34479097277632,-79.34471411412879,-79.34445146213879,-79.34412224620216,-79.34376981288041,-79.34343087047219,-79.34333621348588,-79.34327765389327,-79.3430866974844,-79.34284664342196,-79.34248529275636,-79.342122663106,-79.34181342157315,-79.34146580514528,-79.34109205842576,-79.34074457965815,-79.34044069497619,-79.34020045655774,-79.34002667624371,-79.33984282507119,-79.33951593987652,-79.3394707397926,-79.33948172716909,-79.33971924001969,-79.33999317648127,-79.34027773624254,-79.34029756107651,-79.34027217188815,-79.34018106496124,-79.34004172386362,-79.34004021110157,-79.33994588035081,-79.33974572662756,-79.3395214259877,-79.33943073552237,-79.33939909789331,-79.3394191465855,-79.3394094571201,-79.3393758199344,-79.33927009856708,-79.33900906067258,-79.33858074766304,-79.33832814040687,-79.33824631739617,-79.33799729424388,-79.33790070196711,-79.34361370642478,-79.34642369352753,-79.34813812114128,-79.35023064195966,-79.35165579449549,-79.35275672347527,-79.35275158807127,-79.35276666010454,-79.35283014167707,-79.35313974299604,-79.35335736422563,-79.35362848865275,-79.35396731437463,-79.35439272430952,-79.35504292190006,-79.35552439238911,-79.35571049090277,-79.35582399931431,-79.35595540948898,-79.35611263955803,-79.35627752106561,-79.35645891343768,-79.35665276427785,-79.35684393562467,-79.35702270847202,-79.35720195156095,-79.35731967361757,-79.35744197844829,-79.35769980499491,-79.35792145997146,-79.35809867747288,-79.35816514941425,-79.35826719990628,-79.35839502906624,-79.35845266260444,-79.35850282854788,-79.35853050983648,-79.35864172607705,-79.3587502214243,-79.35882925552387,-79.35627894123346,-79.35586920396533,-79.35505368203255,-79.35369825499866,-79.35228424248848,-79.35107118543718,-79.35102151742086,-79.35097277009625,-79.35092484115665,-79.35088792245918,-79.35087752598599,-79.35083031057181,-79.35078463218044,-79.35073883030222,-79.35069468638558,-79.35065041897769,-79.35060709337627,-79.3505644627562,-79.3505226716441,-79.35048147073233,-79.35044098837872,-79.3504015501692,-79.35036198845332,-79.35032396131486,-79.35028583179135,-79.35024933916141,-79.3502129463014,-79.35017788587807,-79.35014292769654,-79.35010878793346,-79.35007605930961,-79.35004345407734,-79.35001156495917,-79.34998047064389,-79.34994719135386,-79.34991317466776,-79.34987997641257,-79.34984749427876,-79.34981572826821,-79.3497846783828,-79.34975454924829,-79.34972501028578,-79.34969641572135,-79.34966863960241,-79.34964145612432,-79.34961429377651,-79.34958874719241,-79.34956393793394,-79.34953982115822,-79.34951664881156,-79.34949406910535,-79.34947161286273,-79.34945067007686,-79.34943044344094,-79.34941093295704,-79.34939175057627,-79.34937377224679,-79.34935661485953,-79.34934037826012,-79.34932475551004,-79.34931005355257,-79.34929596544855,-79.34928281935353,-79.34927046806322,-79.34925863078892,-79.34924863514779,-79.34921622508304,-79.34821728749721],"lat":[43.71618439923827,43.71617964122046,43.71611154077039,43.71611499839479,43.71608249675666,43.71593414746222,43.7156697955796,43.71539118050822,43.71516330391718,43.71508167867033,43.71503133222048,43.71509363455067,43.71521009335864,43.71521399296108,43.71503190196796,43.71477864393759,43.71454078820421,43.71437848394212,43.71427876588868,43.71407580986241,43.71390781143073,43.71380794094474,43.7135576185344,43.71334700258706,43.71325174224454,43.71312495535791,43.71294033930166,43.71277776035546,43.7126868166672,43.71268627273569,43.71278988335767,43.7129979401318,43.71302692270153,43.71300174696415,43.71278545153924,43.71265426637263,43.71247879786957,43.71222119222922,43.71196402518238,43.71168900784494,43.71150475565094,43.7113603555686,43.71120140798688,43.71105085165549,43.71104632870152,43.71098194467986,43.71083941647605,43.71075964077893,43.71070431118523,43.7105824150106,43.71041608009148,43.71026290597867,43.71013188875305,43.71005383569754,43.70995543114199,43.70979167039613,43.70958086235504,43.70951215781727,43.70917546372925,43.70887834442428,43.70746508061307,43.70687014776043,43.70634663336892,43.70462678942528,43.70327302439352,43.70201485005013,43.70187353504794,43.70187996366924,43.70191238562611,43.70199336460912,43.70203250852101,43.70204991792116,43.70205029792766,43.70204292183541,43.70200727322277,43.70195569000234,43.70193136197642,43.70191949221524,43.70190787989027,43.70188763686996,43.70187200467838,43.70184760773902,43.70185039559822,43.70183514064954,43.70182420811113,43.7018222840283,43.70182847730501,43.70183923731968,43.70185644591501,43.70186413219101,43.70188018154527,43.70188563757971,43.70189610570743,43.70190694406683,43.70192127494231,43.7019354985213,43.70194489811911,43.7019870044846,43.70202006968812,43.70205271168277,43.7045236463357,43.70492293067291,43.70572605271852,43.70701208914198,43.7083647791388,43.70954931384585,43.70960260892011,43.7096559172582,43.70970923738501,43.70974921343947,43.70976256634866,43.70981598675137,43.70986933929697,43.70992719102263,43.70998056567745,43.71003843951331,43.71009182594412,43.71014972337032,43.7102031319169,43.71026113996755,43.71031906835558,43.71037251082062,43.7104304524693,43.71048841625711,43.71054637855352,43.7106043644682,43.71066244183056,43.71072044839816,43.7107784564305,43.71083647626789,43.71089901744944,43.71095705942214,43.71101511172487,43.71107776649422,43.71112679674429,43.71118031731645,43.71123384969618,43.71128739240679,43.71134094544892,43.7113945088232,43.7114480854853,43.71150626166133,43.71155986046841,43.71161347108775,43.71167159123864,43.71172521070789,43.71178335448577,43.71183700792682,43.71189526236275,43.71194893943451,43.71200712604037,43.71206081344498,43.71211902368425,43.71217724426324,43.71223547518231,43.71228920985741,43.71234755291963,43.71240581781712,43.71246409601211,43.71252238307124,43.7125806834286,43.7126389926508,43.71269281449934,43.71275123882575,43.7128095805559,43.7128679488879,43.71297100321066,43.71618439923827]}]],[[{"lng":[-79.35275672347527,-79.35165579449549,-79.35023064195966,-79.34813812114128,-79.34642369352753,-79.34361370642478,-79.33790070196711,-79.33789871767745,-79.33788066939712,-79.33781077168661,-79.33780160624499,-79.33786278258934,-79.33789314693443,-79.33801606050548,-79.33779885349074,-79.33739233669436,-79.33718728595116,-79.33726881415453,-79.33752755694859,-79.33786931197722,-79.33816707075731,-79.33815703294819,-79.33781929893809,-79.33734963928661,-79.33690218823982,-79.33642642937744,-79.33599702961023,-79.33563626461289,-79.33542683108857,-79.33528060138252,-79.33504856906872,-79.33503807793242,-79.33528174196698,-79.33540474312925,-79.33571839660233,-79.3361586710475,-79.33657648696153,-79.33690941434645,-79.33723936898616,-79.33769104708482,-79.33819030550866,-79.33863326213796,-79.33907467854338,-79.33942663096823,-79.33971265181792,-79.34008465119292,-79.34047256290364,-79.34091562906531,-79.34136192534703,-79.3418304261987,-79.34235837977475,-79.3428594508471,-79.34328338167502,-79.34342766121696,-79.3435449083849,-79.34363571807297,-79.3437366517793,-79.34386624793176,-79.34399463235998,-79.34414423362023,-79.34431777031287,-79.34444021595107,-79.34456748528282,-79.34470782611295,-79.34480962016492,-79.34488606803816,-79.34504341049337,-79.34545220798321,-79.34580321951184,-79.34601911936538,-79.34629604236811,-79.3465947023672,-79.34695532710381,-79.34733211608932,-79.34761622517748,-79.34785916567867,-79.34808562418425,-79.34835713926051,-79.34869902857494,-79.34903217373791,-79.34936945664376,-79.34974709071315,-79.35011189620666,-79.35039840641095,-79.35055083705214,-79.35090421363206,-79.35141725875987,-79.35147148445428,-79.35152021019564,-79.35158411487002,-79.35166119469143,-79.35173218149157,-79.35181928411501,-79.35193760775067,-79.35205087993229,-79.3521676514316,-79.35224316128462,-79.35235772467288,-79.35247492702787,-79.35259681671469,-79.35270070257519,-79.35275158807127,-79.35275672347527],"lat":[43.70201485005013,43.70327302439352,43.70462678942528,43.70634663336892,43.70687014776043,43.70746508061307,43.70887834442428,43.70887237426855,43.70881809998856,43.70842540795333,43.70808311006008,43.70775083727146,43.70769276616678,43.70745591183857,43.70720519860067,43.70698314761547,43.70673711104878,43.70642763956168,43.70615675414391,43.70591867248025,43.70563935110575,43.70532854728093,43.70514359591219,43.70507374970583,43.70498171987015,43.70482167141732,43.70461737623646,43.70439598514918,43.70425340413253,43.7041567533247,43.70384721597324,43.70352749325458,43.70324298016474,43.70319526112242,43.70307371199893,43.70289558291055,43.70270794319338,43.70245173180802,43.70215487729463,43.70192289635936,43.70170951948311,43.70148181893086,43.70125418428075,43.7009531421047,43.7006196340223,43.70029646612116,43.69998244002457,43.69972772619203,43.69945964456481,43.69923230224629,43.69912743711146,43.6990176787146,43.69880318703892,43.69868825295602,43.69859993277996,43.69854264618713,43.69848559627535,43.69842446067887,43.69838131133005,43.69832946729387,43.69827796985295,43.69825723890149,43.69825458165223,43.69825661459313,43.6982760930034,43.69829070323375,43.69828848101633,43.69834841243529,43.69847051976853,43.69863117944272,43.69880181379995,43.69894116504931,43.69900039408662,43.69914546456063,43.69933411357167,43.69955376362115,43.69975508092023,43.69992554206996,43.69998908582205,43.69999840034453,43.69997626674498,43.6999231179311,43.69985636970551,43.69964445848858,43.69940351536587,43.69998492203153,43.70087919928203,43.70096550031997,43.70102471606548,43.70109765370952,43.7011752823479,43.70124373105503,43.70132159417049,43.70141782141921,43.7015004727358,43.70157867341185,43.70162027111609,43.70168943778314,43.70174072845212,43.70179649754144,43.70185200697278,43.70187353504794,43.70201485005013]}]],[[{"lng":[-79.32275952981237,-79.32247448754319,-79.32206177810747,-79.32156798096054,-79.3210885194288,-79.32100568849091,-79.32083499869238,-79.32066181539828,-79.32052412750015,-79.32037155666323,-79.32032052332359,-79.32028002000342,-79.32019604345251,-79.32006051702908,-79.31948964977546,-79.31952055877953,-79.31932756768488,-79.3192801967726,-79.31942976478636,-79.31987086947056,-79.32042918740808,-79.32190037528443,-79.32351536656469,-79.32709256693113,-79.3290035320923,-79.32937530647762,-79.33145033868887,-79.33385346714471,-79.33789314693443,-79.33786278258934,-79.33780160624499,-79.33781077168661,-79.33788066939712,-79.33789871767745,-79.33790070196711,-79.33799729424388,-79.33824631739617,-79.33832814040687,-79.33858074766304,-79.33900906067258,-79.33927009856708,-79.3393758199344,-79.3394094571201,-79.3394191465855,-79.33939909789331,-79.33943073552237,-79.3395214259877,-79.33974572662756,-79.33994588035081,-79.34004021110157,-79.34004172386362,-79.34018106496124,-79.34027217188815,-79.34029756107651,-79.34027773624254,-79.33999317648127,-79.33971924001969,-79.33948172716909,-79.3394707397926,-79.33951593987652,-79.33984282507119,-79.34002667624371,-79.34020045655774,-79.34044069497619,-79.34074457965815,-79.34109205842576,-79.34146580514528,-79.34181342157315,-79.342122663106,-79.34248529275636,-79.34284664342196,-79.3430866974844,-79.34327765389327,-79.34333621348588,-79.34343087047219,-79.34376981288041,-79.34412224620216,-79.34445146213879,-79.34471411412879,-79.34479097277632,-79.34493263969627,-79.34528496396636,-79.34563496059673,-79.34599503242234,-79.34636377622404,-79.34672306234252,-79.34691230175034,-79.3468815072976,-79.3467919405893,-79.34709043828322,-79.34747042781298,-79.3478538626469,-79.3481995051345,-79.34821728749721,-79.34855038238585,-79.3489076417453,-79.34926467317983,-79.34960600871807,-79.34987492323505,-79.35000855971123,-79.35033728335648,-79.35069909349723,-79.35106867698457,-79.35134107185503,-79.35140236108015,-79.35175313252516,-79.35207390891597,-79.35230911187077,-79.35240362898388,-79.35247846177306,-79.35261592793478,-79.35177989866335,-79.35111623513276,-79.35043852699992,-79.34996833005572,-79.34882677186212,-79.34811724754711,-79.34773519670443,-79.34724677706113,-79.34502611949827,-79.34352452469204,-79.34298982724134,-79.34129568211006,-79.33963064799372,-79.33905432594243,-79.3388625645922,-79.3388570169345,-79.3381502659587,-79.33665719007584,-79.33549279380244,-79.33520007599755,-79.33511304304348,-79.33470072793347,-79.33343714204135,-79.3320127944791,-79.33034625702989,-79.33005872898953,-79.32848324781486,-79.32689652329447,-79.32618369093687,-79.32528711592356,-79.32364018621816,-79.32264225631988,-79.32232887520487,-79.32232966395425,-79.32251256169765,-79.32263513053998,-79.32275208134936,-79.32292162273006,-79.32309134805368,-79.32322087866126,-79.32324984739364,-79.32325764557217,-79.32327275692968,-79.32330837777455,-79.32333283517602,-79.32335080260528,-79.32338199357169,-79.32348708882294,-79.32364795292102,-79.32381314751115,-79.32402985938316,-79.32435199227096,-79.32447840393966,-79.32464258769245,-79.3250244899529,-79.32518618377172,-79.32540683342273,-79.32556823211375,-79.32568074880818,-79.32584158946354,-79.3261151588532,-79.32638801155602,-79.32654944003097,-79.32662292337042,-79.32667519137124,-79.32670719853202,-79.32672120837194,-79.32671791429073,-79.32671085955145,-79.32669549260395,-79.32659660616933,-79.32644652187814,-79.32606673754184,-79.32562611426532,-79.32524485337011,-79.32483014833981,-79.32464993595114,-79.32443781053711,-79.3242153103598,-79.32402878646094,-79.32362400571027,-79.32325993154743,-79.32323933138713,-79.32284143977218,-79.32255628031672,-79.32247216931928,-79.32244079742138,-79.32246243386801,-79.32253990189159,-79.32311740703491,-79.32343803438845,-79.32359469509557,-79.32374756047277,-79.32392152314884,-79.32385701882596,-79.32365940603195,-79.32360134289165,-79.32356855800759,-79.32355598607984,-79.32334895664695,-79.32293001224431,-79.3228776945099,-79.32275952981237],"lat":[43.71638546948572,43.71631827173899,43.71614558630207,43.71602130910524,43.71591524429261,43.71590502533733,43.71590701823202,43.7159359800688,43.71594745943865,43.71596772183224,43.71596247072912,43.71593937049435,43.71583461366658,43.71554896996683,43.71562618456721,43.71340720858689,43.7132918459158,43.71268333744566,43.71255050792912,43.71166562095495,43.71155995376257,43.71122725088313,43.71086828267542,43.71008775656662,43.70966102748873,43.70956744175316,43.70910256131151,43.70857482260396,43.70769276616678,43.70775083727146,43.70808311006008,43.70842540795333,43.70881809998856,43.70887237426855,43.70887834442428,43.70917546372925,43.70951215781727,43.70958086235504,43.70979167039613,43.70995543114199,43.71005383569754,43.71013188875305,43.71026290597867,43.71041608009148,43.7105824150106,43.71070431118523,43.71075964077893,43.71083941647605,43.71098194467986,43.71104632870152,43.71105085165549,43.71120140798688,43.7113603555686,43.71150475565094,43.71168900784494,43.71196402518238,43.71222119222922,43.71247879786957,43.71265426637263,43.71278545153924,43.71300174696415,43.71302692270153,43.7129979401318,43.71278988335767,43.71268627273569,43.7126868166672,43.71277776035546,43.71294033930166,43.71312495535791,43.71325174224454,43.71334700258706,43.7135576185344,43.71380794094474,43.71390781143073,43.71407580986241,43.71427876588868,43.71437848394212,43.71454078820421,43.71477864393759,43.71503190196796,43.71521399296108,43.71521009335864,43.71509363455067,43.71503133222048,43.71508167867033,43.71516330391718,43.71539118050822,43.7156697955796,43.71593414746222,43.71608249675666,43.71611499839479,43.71611154077039,43.71617964122046,43.71618439923827,43.71628823504428,43.7163023995014,43.7162130370916,43.71613235880767,43.71602371790559,43.71585010948129,43.71577383789424,43.71573396069827,43.71569428454275,43.7156937138445,43.71569459807318,43.71569515709457,43.71565477368988,43.71573018094298,43.71593417772649,43.71614680256367,43.71622539101975,43.71646989120735,43.71668095473033,43.71691872748193,43.71709206811642,43.71741315209551,43.71762353545647,43.71772153444145,43.71787209546967,43.71855570079466,43.71904264375095,43.71921502053033,43.71970814376743,43.72026926938976,43.72043202091847,43.72049224709938,43.72049666742642,43.72071152994771,43.72117150034823,43.72152821597692,43.7216184721776,43.72164871096981,43.72177323110272,43.7221509849989,43.7225848909869,43.72308725618638,43.72315966113631,43.72355980742641,43.72397327021147,43.72416087356818,43.72439538039664,43.72482591932283,43.72510383804894,43.72518484592344,43.72507224330568,43.72483187616409,43.72458153121459,43.72430868884162,43.72401852429813,43.72369235444048,43.7233610935403,43.72304194984507,43.72260096931867,43.72238955548436,43.72216953065902,43.72207077817333,43.72203953501792,43.72200398491125,43.72196501813225,43.72192686940026,43.72190228687231,43.72186945776501,43.72183367320407,43.72180852099871,43.72176141759883,43.72164530992359,43.72158466633231,43.72146187292014,43.72137871960898,43.7213488614919,43.72131512026922,43.72126961728049,43.72122410313889,43.72119496051663,43.72117353177536,43.72113828941026,43.72110275033876,43.7210444428701,43.72096787813452,43.7208866673399,43.72076941707924,43.7206149360036,43.72048671085855,43.72039112867974,43.72032616059813,43.72041959483016,43.72054863529721,43.72060900689446,43.72064640541431,43.72061163586446,43.72054588674458,43.72033731554441,43.72009342246977,43.72007961728114,43.7198261346777,43.71956080268035,43.71946054657209,43.71939257145069,43.71928486577124,43.71920498551734,43.71899732498839,43.71885349755268,43.71871617579279,43.71852037552279,43.7181222507153,43.71771162605793,43.71731705262904,43.71708665121444,43.71695555200417,43.71693286269958,43.71661916806307,43.71642397943479,43.71640970867689,43.71638546948572]}]],[[{"lng":[-79.35385495150413,-79.35255736363662,-79.35018806866671,-79.35001739866418,-79.34956411462316,-79.34264059302487,-79.34239014338009,-79.34092124740545,-79.34084638883603,-79.34059216755361,-79.34017820771771,-79.3399136002614,-79.33960125288324,-79.33953381392557,-79.33938394472173,-79.33914634550209,-79.3388625645922,-79.33905432594243,-79.33963064799372,-79.34129568211006,-79.34298982724134,-79.34352452469204,-79.34502611949827,-79.34724677706113,-79.34773519670443,-79.34811724754711,-79.34882677186212,-79.34996833005572,-79.35043852699992,-79.35111623513276,-79.35177989866335,-79.35261592793478,-79.35268489183376,-79.35262381129414,-79.35265408419538,-79.3526993846237,-79.35292905172834,-79.35311640850098,-79.35288540678502,-79.35274716378426,-79.35269231118967,-79.35266085223967,-79.35247346941267,-79.35223046316658,-79.35199574216735,-79.35196069880276,-79.35187463197873,-79.35166540311697,-79.35144443207014,-79.35124617192056,-79.35099948433141,-79.35075719985431,-79.35071217413415,-79.35066985129401,-79.35055631613588,-79.35051163977072,-79.35033349919919,-79.35031675576212,-79.35031388702153,-79.35036535999528,-79.35036144679293,-79.350250336293,-79.34996322305473,-79.34985901367196,-79.34984799692971,-79.34985894279043,-79.35001405999391,-79.35016198986135,-79.35018012029009,-79.35017841592898,-79.35010729240764,-79.34990857313915,-79.34988556587507,-79.34989388992241,-79.34997493688516,-79.35016908094671,-79.35019957267426,-79.35040377690963,-79.35055235461544,-79.35065843206129,-79.35077724571097,-79.35066719334939,-79.35045871873977,-79.35030879258903,-79.35020850595329,-79.35010623423864,-79.35006964102392,-79.35021263543094,-79.3504531581161,-79.35069556896997,-79.35095879620408,-79.35123128246454,-79.35149849794553,-79.3517465979489,-79.35189636195481,-79.35206341651624,-79.35229480790903,-79.3525765532077,-79.35276537124636,-79.3527289440953,-79.35289738786607,-79.35302708838408,-79.35323973591335,-79.35340804816171,-79.35343896597439,-79.3535175461913,-79.35357788480005,-79.35385495150413],"lat":[43.72799384559902,43.7287972924231,43.73019081633491,43.73007132697916,43.72984873458186,43.72742548747745,43.72742653572726,43.72764358965746,43.72765168474203,43.72670297012652,43.72516626860475,43.72421254021816,43.72301947596473,43.72278885651432,43.7222329692429,43.72136965058393,43.72049224709938,43.72043202091847,43.72026926938976,43.71970814376743,43.71921502053033,43.71904264375095,43.71855570079466,43.71787209546967,43.71772153444145,43.71762353545647,43.71741315209551,43.71709206811642,43.71691872748193,43.71668095473033,43.71646989120735,43.71622539101975,43.71727979371607,43.71740944148492,43.71769803037309,43.71777069902423,43.71997993736276,43.72092802222198,43.72097870447674,43.72095420689127,43.72117396385637,43.72135814026662,43.72151297286045,43.72163549626968,43.72175822866606,43.72196476801216,43.72216165952125,43.72234768248372,43.72248402496267,43.72264328957299,43.72269824263662,43.72273075385579,43.72273010400536,43.72272049120535,43.72264233592431,43.72263268909384,43.72264812141491,43.72266138261464,43.7226838460592,43.72281511732231,43.72284206666129,43.72296648980733,43.72302094700902,43.72311396261821,43.72314981130169,43.72317247422368,43.7232872383813,43.72329837621615,43.72331664186551,43.72335262502924,43.72348221633004,43.72354236056957,43.72356003222426,43.72359616019511,43.7237278586668,43.72385677906175,43.72387972414744,43.72395918868632,43.72402884799065,43.72403938105757,43.72414911920135,43.72428715092777,43.72441917083987,43.7245925443595,43.72482973796222,43.7250758147261,43.72530492586146,43.72549603116642,43.72566612935984,43.72582716233778,43.72584896514486,43.72581688930469,43.72586125326109,43.72590083990132,43.72603199780021,43.72617105650274,43.72628241650252,43.72619645933072,43.72624005013294,43.7264372976212,43.72683581098953,43.72702681114104,43.72714240003208,43.72728435549597,43.7273479045865,43.72750657075029,43.72771448468551,43.72799384559902]}]],[[{"lng":[-79.38111532522471,-79.38052518022955,-79.36190222725826,-79.35901585255002,-79.3583450386964,-79.35721870579913,-79.35596805108175,-79.35584393634372,-79.35582200556701,-79.35569943423747,-79.35553205968756,-79.3553947218786,-79.35533816392724,-79.3552608483476,-79.35515866565488,-79.35505866681994,-79.35457649198892,-79.35453884598532,-79.35470113321468,-79.35468877657442,-79.3544793061085,-79.35425870619326,-79.35414365337812,-79.35409232681381,-79.35401286482481,-79.35385495150413,-79.35357788480005,-79.3535175461913,-79.35343896597439,-79.35340804816171,-79.35323973591335,-79.35302708838408,-79.35289738786607,-79.3527289440953,-79.35276537124636,-79.3525765532077,-79.35229480790903,-79.35206341651624,-79.35189636195481,-79.3517465979489,-79.35149849794553,-79.35123128246454,-79.35095879620408,-79.35069556896997,-79.3504531581161,-79.35021263543094,-79.35006964102392,-79.35010623423864,-79.35020850595329,-79.35030879258903,-79.35045871873977,-79.35066719334939,-79.35077724571097,-79.35065843206129,-79.35055235461544,-79.35040377690963,-79.35019957267426,-79.35016908094671,-79.34997493688516,-79.34989388992241,-79.34988556587507,-79.34990857313915,-79.35010729240764,-79.35017841592898,-79.35018012029009,-79.35016198986135,-79.35001405999391,-79.34985894279043,-79.34984799692971,-79.34985901367196,-79.34996322305473,-79.350250336293,-79.35036144679293,-79.35036535999528,-79.35031388702153,-79.35031675576212,-79.35033349919919,-79.35051163977072,-79.35055631613588,-79.35066985129401,-79.35071217413415,-79.35075719985431,-79.35099948433141,-79.35124617192056,-79.35144443207014,-79.35166540311697,-79.35187463197873,-79.35196069880276,-79.35199574216735,-79.35223046316658,-79.35247346941267,-79.35266085223967,-79.35269231118967,-79.35274716378426,-79.35288540678502,-79.35311640850098,-79.35322178409335,-79.36211465397548,-79.36216917605658,-79.36242876431575,-79.36311484802195,-79.36324000176573,-79.36348348413063,-79.36370055408919,-79.36381511316293,-79.36401812117602,-79.36416994816939,-79.36431463060985,-79.36455585477624,-79.36484165347998,-79.36516285310597,-79.36544196613411,-79.36553302487818,-79.36557458355409,-79.36552403347378,-79.3654061726463,-79.36527476119666,-79.36513638793113,-79.3649985056858,-79.36480079539959,-79.36470781199328,-79.36473011742102,-79.36480025340278,-79.36483289627611,-79.36489617937356,-79.36499730359242,-79.36498172901,-79.36490818887407,-79.36481999873739,-79.36471682440231,-79.36469657924924,-79.36469676517255,-79.36475039821988,-79.36493115986113,-79.3650486269366,-79.36519507498072,-79.3657346102094,-79.3657777567397,-79.36585271304156,-79.36607842431432,-79.36653027025666,-79.36664825550308,-79.36677865961481,-79.36705984273262,-79.36733930427179,-79.36743570459116,-79.36747022645801,-79.36771274702836,-79.36795233495599,-79.36813349734784,-79.36815216895066,-79.36835339201447,-79.36853275081249,-79.36873873916839,-79.36901179913738,-79.36924819719967,-79.36948949997505,-79.36976355086802,-79.3700261614285,-79.3703139188878,-79.37053974756259,-79.3707887045768,-79.37089512716757,-79.3708100277784,-79.37098684172692,-79.37128432272416,-79.37157395039243,-79.37182842349631,-79.37208642306601,-79.37239089157529,-79.37252379875954,-79.37257112207115,-79.37233103729193,-79.37208858440027,-79.37196892387054,-79.37202849240816,-79.37217637465585,-79.37238732520834,-79.37257775852645,-79.37281504923472,-79.37307643837889,-79.37330790027585,-79.37351437812326,-79.37366372791038,-79.37378934371668,-79.37394044555542,-79.37412504479073,-79.37424546576321,-79.37418686222144,-79.37394995592932,-79.37380740872663,-79.37379782537687,-79.37391830007981,-79.37406137674395,-79.37422247457904,-79.37438568157962,-79.37460669174858,-79.37481179328057,-79.37501916376452,-79.3752343136089,-79.37545833522674,-79.37569950379749,-79.37593130415722,-79.37600632179424,-79.37609298463516,-79.37623497077793,-79.37641135241741,-79.37659221466653,-79.37679665987389,-79.37702110843425,-79.37724780524195,-79.37741404797781,-79.37764720400762,-79.37789111353477,-79.3781795563329,-79.37842622650017,-79.37851199390717,-79.37860699934767,-79.37886859827194,-79.37904211904132,-79.37913561321776,-79.3790950861042,-79.37904094367848,-79.37915709020635,-79.37934071645374,-79.37954943440427,-79.37975331224935,-79.37994694504756,-79.38009514335876,-79.38021410110433,-79.38037678460063,-79.38051473423147,-79.3806541119822,-79.38076731196638,-79.38089996047707,-79.38094569775133,-79.3810780161311,-79.38111532522471],"lat":[43.73009267447124,43.73023329444548,43.73416986031199,43.73477964622213,43.73470249649694,43.73454218863731,43.73452870291459,43.73400921629143,43.73381535927769,43.7332507949824,43.73261366032657,43.73218850304639,43.73195804977287,43.73172288674537,43.73148727569576,43.73131029856255,43.72956562333752,43.72940745737586,43.72926126222141,43.72909931953137,43.72895335257019,43.7288137963484,43.72859609282556,43.72839920136381,43.72815302409099,43.72799384559902,43.72771448468551,43.72750657075029,43.7273479045865,43.72728435549597,43.72714240003208,43.72702681114104,43.72683581098953,43.7264372976212,43.72624005013294,43.72619645933072,43.72628241650252,43.72617105650274,43.72603199780021,43.72590083990132,43.72586125326109,43.72581688930469,43.72584896514486,43.72582716233778,43.72566612935984,43.72549603116642,43.72530492586146,43.7250758147261,43.72482973796222,43.7245925443595,43.72441917083987,43.72428715092777,43.72414911920135,43.72403938105757,43.72402884799065,43.72395918868632,43.72387972414744,43.72385677906175,43.7237278586668,43.72359616019511,43.72356003222426,43.72354236056957,43.72348221633004,43.72335262502924,43.72331664186551,43.72329837621615,43.7232872383813,43.72317247422368,43.72314981130169,43.72311396261821,43.72302094700902,43.72296648980733,43.72284206666129,43.72281511732231,43.7226838460592,43.72266138261464,43.72264812141491,43.72263268909384,43.72264233592431,43.72272049120535,43.72273010400536,43.72273075385579,43.72269824263662,43.72264328957299,43.72248402496267,43.72234768248372,43.72216165952125,43.72196476801216,43.72175822866606,43.72163549626968,43.72151297286045,43.72135814026662,43.72117396385637,43.72095420689127,43.72097870447674,43.72092802222198,43.72170379824374,43.71999485021841,43.72004514240491,43.72014338318429,43.7201352090952,43.72029462592495,43.72035662581668,43.72053527233108,43.72075304948549,43.72087298137489,43.72076713151765,43.72061157844757,43.7205610193589,43.72052009954901,43.72043467574165,43.72047017535433,43.72054349342687,43.7207017119854,43.72090803380199,43.7211313966686,43.72134106263317,43.72155071885694,43.72174228801445,43.72187898920615,43.72207579147324,43.72228315546359,43.72252730170144,43.72277991333512,43.72302387141531,43.72327296188099,43.72350678962709,43.72372637506152,43.7239726666209,43.72422783563525,43.72443008966201,43.72447969303467,43.72459298484611,43.7247936142,43.7248673104793,43.7249595151579,43.72560646061815,43.72584571916909,43.72608984354876,43.72626419804041,43.72668033626872,43.72693857808609,43.72716558049675,43.72729562612334,43.72723660570968,43.72703084833881,43.72681529495213,43.72680075556302,43.72689419706978,43.72705431829385,43.72725271765326,43.72727359491524,43.72712312278917,43.72696843953495,43.7268283061633,43.72666964505955,43.72658297896175,43.72663639967378,43.72670775044689,43.72671635563912,43.72663396760694,43.72669161902062,43.72685967220344,43.72722303759254,43.72739659520868,43.72749985695593,43.72752648951592,43.72761122309111,43.72770941922921,43.72773626114035,43.72758062013893,43.72738316138831,43.72730772637203,43.72725926293954,43.72712243905019,43.72695675178568,43.72676981771144,43.72671881089825,43.72682054451673,43.72685543049145,43.7268321464483,43.72679493287318,43.72687888944406,43.72703854910694,43.72724297054243,43.72744766447827,43.72762141804486,43.7278211740601,43.72797787425371,43.72830316424459,43.72846326069352,43.7286206582632,43.72878440759839,43.7289755742408,43.72918941192275,43.72941237133477,43.72961805881705,43.72981010672395,43.72997509068983,43.73014018488748,43.73026498610255,43.73035393284216,43.73031221782973,43.73014665811365,43.7299858548223,43.7298123346955,43.72966171791717,43.72951575550488,43.72938363070461,43.72926520253061,43.72914689585852,43.72899622354714,43.72892301749298,43.72894448398355,43.72901159171055,43.72915462243184,43.72937197606333,43.72953535904558,43.72956157541289,43.7294560136876,43.729299715911,43.72898407314565,43.72883918407152,43.72867429566364,43.72862288844877,43.72868886199219,43.72878627331995,43.72891054485957,43.72905676687349,43.72922048807599,43.72940292280461,43.72955340995622,43.72968132230768,43.7298270474678,43.72997295805946,43.73000061191515,43.73007900307537,43.73009267447124]}]],[[{"lng":[-79.36190222725826,-79.38052518022955,-79.38111532522471,-79.38125706093891,-79.38144327843139,-79.38133817222077,-79.38113145067743,-79.3811003578359,-79.38109212629421,-79.38109544674452,-79.3811258513383,-79.38120249015451,-79.38154245296751,-79.38177355767071,-79.38183435315862,-79.38189645295937,-79.38228578969651,-79.38257381914377,-79.38264153161799,-79.3828875822887,-79.38303848680114,-79.3832732852967,-79.38339855400021,-79.38361393474359,-79.38378117987899,-79.38404983284947,-79.38414255587453,-79.38414455370379,-79.38398353395961,-79.38383408249669,-79.38370361420615,-79.38347615015896,-79.38324193555739,-79.38293078430982,-79.38265944149121,-79.38230286778796,-79.38203935286809,-79.38179518238002,-79.38152486129921,-79.38141461709894,-79.38134958645247,-79.38123700004881,-79.38102871175498,-79.38070750148314,-79.38041886085387,-79.38016975823093,-79.37996248663143,-79.37990583353491,-79.37972175330378,-79.37951180966347,-79.37927683554813,-79.37901038362206,-79.37868578997856,-79.37839201951398,-79.37839953943792,-79.37845102398795,-79.37849810512932,-79.37869774761157,-79.37882296976811,-79.37877927372786,-79.37866459495881,-79.37846555455315,-79.3783644286092,-79.37822901786092,-79.37820199728175,-79.37807555609385,-79.377784537895,-79.37742012180314,-79.37708730660614,-79.37681734302734,-79.37656877288686,-79.3763003222524,-79.37610956856453,-79.37583914628905,-79.37559141550116,-79.37530490204685,-79.3752169990816,-79.37501537817357,-79.37485210334489,-79.3745840354907,-79.37428975160762,-79.37409247814371,-79.37380929893627,-79.37362528831643,-79.37328350082122,-79.37310707983498,-79.37310660463547,-79.37299894827692,-79.37287020576657,-79.37262361445013,-79.37252066565226,-79.37247648764709,-79.37242099878263,-79.37235000253212,-79.37222773738655,-79.37210510649372,-79.37209034664434,-79.37188109068398,-79.37166467975486,-79.37156304909367,-79.37134770138776,-79.3712551402766,-79.371108958132,-79.37085122521843,-79.37062619651724,-79.37063664138377,-79.37064857287695,-79.37089332786037,-79.37102288113508,-79.37103451179965,-79.37074215248491,-79.37061585871692,-79.37088271505509,-79.37099727866645,-79.3707907451243,-79.37050563372297,-79.3703015252941,-79.37025039708826,-79.37015698951274,-79.36995181990105,-79.36965479241168,-79.36960148948715,-79.36936338794249,-79.36910505272169,-79.36897841307982,-79.36885745877143,-79.36873044219824,-79.36869454440577,-79.36869089916364,-79.3686535025257,-79.36854651234496,-79.3683744411528,-79.36818526015722,-79.36795337848712,-79.36770365132749,-79.36740648286319,-79.36716000161169,-79.36693912724004,-79.36676065633888,-79.3666910570296,-79.3666633885491,-79.36656436041021,-79.36626448814664,-79.36602880937264,-79.36581706464924,-79.36558313239068,-79.36527659290024,-79.36503374992074,-79.36479259344905,-79.36446790091523,-79.36434756256023,-79.3643155729481,-79.36411761773154,-79.36387298761262,-79.36364962847733,-79.36351422869309,-79.36351518365778,-79.36364166432161,-79.36353186728942,-79.36344125403338,-79.36345608584563,-79.36329444338534,-79.36306466238513,-79.36290642259452,-79.36275503853572,-79.36247739740567,-79.36221873230015,-79.36201668525992,-79.36190222725826],"lat":[43.73416986031199,43.73023329444548,43.73009267447124,43.73014455309852,43.73023270928385,43.73064539970013,43.73163745546535,43.73175404015181,43.73186194677489,43.7319971128846,43.73219108526082,43.7326603616596,43.73450643280361,43.73563521736571,43.73602775231654,43.73640230182018,43.73851912665558,43.74004479948825,43.74035191336503,43.74153941870591,43.74224838511793,43.74341313524081,43.7439867094751,43.74509717237043,43.74609001895674,43.74741728009758,43.74786877705338,43.7480533445385,43.74809607703707,43.7481389729254,43.74819573001999,43.74828703246924,43.74845025425553,43.74864848448289,43.74880217772306,43.74852698287542,43.74842432097378,43.74829933760869,43.74816048046226,43.74796078682434,43.74772131495573,43.74749458221414,43.74738360838207,43.74732054454324,43.74723084566875,43.74708778508116,43.74688230361149,43.74662485581117,43.74645571035663,43.74632661642608,43.74623326477359,43.74617547376644,43.74606734666727,43.74591005551135,43.74565810858316,43.74542019839471,43.74525433104097,43.74525266288197,43.74512841269079,43.74490715576239,43.7446939836112,43.74450652996968,43.74428454790831,43.7440124785959,43.74374644848323,43.74348359820891,43.74330834067042,43.74320414556326,43.7430598894728,43.7428219150527,43.74265634930845,43.74256710687276,43.74238426778938,43.7422543975878,43.74222837140704,43.74217478751707,43.7419529005921,43.74177449612937,43.74158304377374,43.74159723481748,43.74156604265332,43.74140120128869,43.74131165340285,43.7411109022669,43.7410340217825,43.7409009822843,43.74077485859148,43.7406247940831,43.74047442921776,43.74047091763494,43.74060448027021,43.74071187412294,43.74078318918401,43.7407956808199,43.74078043632952,43.74077868938061,43.74078298006999,43.74085651510033,43.74088043721888,43.74073486817169,43.74074980292054,43.74061345480445,43.74056636148492,43.74065270647135,43.74056397973577,43.74035258350106,43.74019963093258,43.74013110521815,43.73993040888747,43.73970543659289,43.73961124926539,43.73948792252933,43.73937461233249,43.73916019941264,43.73896362353933,43.73885153508935,43.73870909447407,43.73864985276901,43.73854490836928,43.73841145391359,43.73826318565231,43.73802378410166,43.73793045738861,43.73789976485808,43.73801948333703,43.73810327512762,43.73802944655717,43.73795241772385,43.73794336375927,43.73783480677561,43.73763064595808,43.73742114483582,43.73721581005963,43.73703696090369,43.73688936310303,43.73679950946644,43.73672847322734,43.73663079663346,43.73659673917388,43.73661824919845,43.73669437009482,43.73679647673082,43.73669316860388,43.73654126702827,43.73662375734698,43.73674643859082,43.7366925431048,43.73651803217241,43.73642456219577,43.73640191151042,43.73623356339062,43.73603956405671,43.73601422549044,43.73596571347655,43.73589941166878,43.73580295241888,43.73568144007447,43.73550762375405,43.73529450608742,43.73511767077159,43.73491524977936,43.73478240669829,43.73477911528013,43.73466423423089,43.73454954120858,43.73451405611493,43.7344698403941,43.73433191542765,43.73416986031199]}]],[[{"lng":[-79.40648670151339,-79.40525248754824,-79.40468155433838,-79.40440641508087,-79.4043011164689,-79.40425331956349,-79.40420674301409,-79.40259204529215,-79.40248839467601,-79.4023685522001,-79.40224896240677,-79.40213503992854,-79.40199787929336,-79.40191048794691,-79.40166975932564,-79.40145568133204,-79.40099075780689,-79.40058763007022,-79.39999969389442,-79.39962947668181,-79.39907398085217,-79.39642934376944,-79.39419898132383,-79.39160026148903,-79.38899969838862,-79.38638403204681,-79.38526830734963,-79.38414455370379,-79.38414255587453,-79.38404983284947,-79.38378117987899,-79.38361393474359,-79.38339855400021,-79.3832732852967,-79.38303848680114,-79.3828875822887,-79.38264153161799,-79.38257381914377,-79.38228578969651,-79.38189645295937,-79.38183435315862,-79.38177355767071,-79.38154245296751,-79.38120249015451,-79.3811258513383,-79.38109544674452,-79.38109212629421,-79.3811003578359,-79.38113145067743,-79.38133817222077,-79.38144327843139,-79.38188796775565,-79.38189617833444,-79.38189445787661,-79.38187421613338,-79.3820851010364,-79.38217752335126,-79.38287608183026,-79.38295233476207,-79.38307610915585,-79.38331412895567,-79.38358101585089,-79.38397377390341,-79.38446367405814,-79.38495910049357,-79.38536695398081,-79.38552525987355,-79.38569334418868,-79.38588844720728,-79.38604478977467,-79.38623588203529,-79.38638738614993,-79.38653065586779,-79.38669949269733,-79.38951975090619,-79.39006509165802,-79.39015253711618,-79.39020864823539,-79.3902862630068,-79.39026980657793,-79.39041847960584,-79.39063658828762,-79.3912556737379,-79.39239638916499,-79.39997719331399,-79.40341047248397,-79.40454012521897,-79.40456226621298,-79.40482232706044,-79.40492121264577,-79.40512850517914,-79.4054120296167,-79.40560137446872,-79.40590228291785,-79.40624014239597,-79.40644222115591,-79.40648670151339],"lat":[43.74329793426466,43.74356427694625,43.74366883166084,43.74373709553743,43.7437842355565,43.74380247213749,43.74382873745302,43.74413033587485,43.74414238975461,43.74415871812116,43.74418864279859,43.744209554656,43.74423014142901,43.74424692312098,43.7443110706451,43.74436658856209,43.74445460389098,43.74455248420673,43.74470646809593,43.744791302946,43.744896045781,43.74544876944143,43.74592614691777,43.74647031185903,43.7470279844626,43.74756278965206,43.74780367337669,43.7480533445385,43.74786877705338,43.74741728009758,43.74609001895674,43.74509717237043,43.7439867094751,43.74341313524081,43.74224838511793,43.74153941870591,43.74035191336503,43.74004479948825,43.73851912665558,43.73640230182018,43.73602775231654,43.73563521736571,43.73450643280361,43.7326603616596,43.73219108526082,43.7319971128846,43.73186194677489,43.73175404015181,43.73163745546535,43.73064539970013,43.73023270928385,43.7280376724578,43.72793426638243,43.72770919345244,43.72748376822906,43.72754526686211,43.72756908024129,43.72768249040682,43.72770607433622,43.72772141871997,43.72775620318544,43.72781399037185,43.72790056289571,43.72803351718675,43.72816663750066,43.72826692149604,43.72832316986575,43.72836605324102,43.72840931801638,43.72842052822013,43.72841872604137,43.72841186356128,43.72839138170012,43.728375761592,43.72778522730935,43.72767588642339,43.72809579880385,43.72831722691861,43.72843534554642,43.72855663996518,43.72914854150188,43.72998447314131,43.73243307617469,43.73690607775287,43.73525214753368,43.73450331715708,43.73425794874415,43.73435727919071,43.73552242974009,43.73600550367949,43.7369718741027,43.73819127189721,43.73900876880658,43.74037252845067,43.74184932648079,43.74301366330526,43.74329793426466]}]],[[{"lng":[-79.40837774950745,-79.40837303971898,-79.4067339369156,-79.40566330382724,-79.40327524712794,-79.40249373450509,-79.40037547393391,-79.39936440080524,-79.39926503533862,-79.39916464570716,-79.3990637590901,-79.39896193550622,-79.39886011171417,-79.39875769095907,-79.39865433080844,-79.39855098808677,-79.39844703078884,-79.39834307089338,-79.39823747485039,-79.39813187860797,-79.39802638456328,-79.39791937204861,-79.3968156894763,-79.39600743312423,-79.39578335747701,-79.39539645832035,-79.39526673892509,-79.39517901695804,-79.395107217573,-79.39506570415809,-79.39500333818785,-79.39492339635466,-79.39486266892163,-79.39477851642667,-79.39468405308668,-79.3946037533415,-79.3945133143509,-79.3944063183643,-79.39430643375519,-79.39421537540252,-79.3941166555729,-79.39404482932932,-79.39394279708617,-79.39382392397019,-79.39370717082983,-79.39361164925926,-79.39352309058981,-79.39344812242545,-79.39338446930253,-79.39333462133197,-79.39329454908349,-79.39328892905849,-79.39329213280641,-79.39330798727278,-79.39332194791899,-79.39335206841263,-79.39341100265626,-79.39395926489105,-79.3950845615246,-79.3951207810436,-79.39516681188533,-79.3952116676433,-79.39525333382488,-79.39529440107691,-79.39535530056217,-79.3953999024063,-79.39544148610398,-79.39546922953637,-79.3954965097802,-79.39551730627177,-79.39553902180369,-79.39555853833318,-79.39557177352027,-79.39558500868003,-79.39558916709441,-79.39559875198375,-79.39560386994393,-79.39560449833336,-79.39560015864512,-79.39559064612229,-79.39558748154532,-79.39559460909889,-79.39560217958588,-79.39562827946413,-79.39570260292003,-79.39581229816226,-79.39712781252003,-79.39828934002747,-79.39823125830243,-79.39816706670322,-79.3981268932793,-79.39810002278828,-79.39807079750325,-79.39801927024149,-79.39800041994681,-79.39793472850997,-79.39790111494432,-79.39783438211877,-79.39777329814194,-79.39774726105735,-79.39765727342045,-79.39772700396637,-79.39778530281947,-79.39785692903193,-79.39792171549045,-79.39797801927523,-79.39813883327629,-79.39828465514935,-79.39829908020226,-79.39831195187264,-79.39832071406107,-79.39833761724617,-79.39835286124082,-79.39835984760762,-79.3983609311278,-79.39835802419226,-79.39834683965725,-79.398327432855,-79.39830896516504,-79.39828882417443,-79.39825100916353,-79.39820942145349,-79.398032483416,-79.39761121592957,-79.39754417987329,-79.39744863920522,-79.39735601738597,-79.39727087491747,-79.39714062515064,-79.39674781279949,-79.39667134901052,-79.39655932311346,-79.39650904996078,-79.39646629692629,-79.39642934376944,-79.39907398085217,-79.39962947668181,-79.39999969389442,-79.40058763007022,-79.40099075780689,-79.40145568133204,-79.40166975932564,-79.40191048794691,-79.40199787929336,-79.40213503992854,-79.40224896240677,-79.4023685522001,-79.40248839467601,-79.40259204529215,-79.40420674301409,-79.40425331956349,-79.4043011164689,-79.40440641508087,-79.40468155433838,-79.40525258991767,-79.40648670151339,-79.40671415975115,-79.40684914193277,-79.40692403119031,-79.40698428201502,-79.40702417710361,-79.4071060129917,-79.40721910792981,-79.40725161469665,-79.40743958993819,-79.4076410165323,-79.40776142342307,-79.40782891621288,-79.40786531815047,-79.40796709230106,-79.40803885050171,-79.40809858056022,-79.40812281443178,-79.40812728636905,-79.40812858352004,-79.40813517166096,-79.40814116242383,-79.40814611248952,-79.40814952684904,-79.4081530412081,-79.40815514151963,-79.40815702017456,-79.40815726073306,-79.40815688700366,-79.4081565132727,-79.40815462071681,-79.40815189220865,-79.40814834464329,-79.40814408040049,-79.40813899471138,-79.40813321162636,-79.40812669496025,-79.4081192568464,-79.40811040227427,-79.40810153078736,-79.40809245452201,-79.40809583602886,-79.40809817442921,-79.40809991782609,-79.40810094454582,-79.40810093053378,-79.4081003191239,-79.4080988693549,-79.40809590311952,-79.40809303925636,-79.4080884179573,-79.4080831992454,-79.4080770421556,-79.40808021555959,-79.40808337444744,-79.40808714764437,-79.40809278066845,-79.40809819202711,-79.40810432008543,-79.40811220559524,-79.40811996944313,-79.40813266802182,-79.40813879516745,-79.4081496329908,-79.40816059012059,-79.408174005489,-79.40817376691652,-79.40817516748484,-79.40817740403722,-79.40817963820575,-79.40818163520923,-79.40818431510958,-79.4081889065334,-79.40819419775973,-79.40820032497579,-79.40820716889689,-79.40821471262539,-79.40822319473193,-79.40823249593296,-79.40824251384397,-79.40825323156987,-79.40826488767689,-79.40827185093583,-79.40832368104647,-79.40837774950745],"lat":[43.75361096118208,43.75361089558039,43.75464154650629,43.75531543944013,43.75683519374341,43.75732297608306,43.75864497803327,43.75929254933432,43.75935417070078,43.7594157776318,43.75947296657969,43.75953455324053,43.75959613980672,43.75965321696608,43.75971478181791,43.75977184586981,43.75982890121357,43.75988604644454,43.75994307862608,43.76000011070641,43.76005714412108,43.76010965519837,43.75943235608194,43.75899333302129,43.7588956674559,43.75869669420615,43.75861385543827,43.7585540210931,43.75848099748303,43.75843990586344,43.75838951943516,43.75832988419055,43.75827952070083,43.75819732145342,43.75813298109131,43.75808675342593,43.7580404732643,43.75801196418904,43.75799714781427,43.75797786441705,43.75795397232452,43.75793946002329,43.7579245231279,43.75789584657043,43.75784919588774,43.75779384166275,43.75773858524869,43.75766542602115,43.75757901300186,43.75749729500616,43.7574292172967,43.75737512686231,43.757316659547,43.75724477723872,43.75715945546459,43.75707886185393,43.75696266574575,43.75629054920861,43.7549107944801,43.75486629365488,43.75482643156611,43.75477305009628,43.75472403474885,43.75467060004068,43.75459944000659,43.754532552035,43.75445211880422,43.75437599216881,43.75431336188204,43.75425955243521,43.75420584590758,43.75413860563838,43.75408027907665,43.75402195251242,43.75395449659464,43.75388711686745,43.7538105824936,43.75372957414752,43.75366199890715,43.75360785389746,43.75355379801672,43.75348638378252,43.75343688956346,43.75338774552425,43.75333927853598,43.75330481094405,43.75302161763615,43.75275884559467,43.75260940991249,43.75245097665753,43.75235589343632,43.75231500815278,43.75227408985089,43.75222835792689,43.75220549885947,43.75216857023032,43.75215009513593,43.75211765280339,43.75208078868252,43.75206241977685,43.75200714648673,43.75195411280567,43.75189191688182,43.75182540683374,43.75175420987006,43.75167848300703,43.7512710604803,43.75092644093902,43.75087713261735,43.75083230347614,43.75076491195007,43.75068413166314,43.7506348348114,43.75057642030216,43.75050892114228,43.75044127603061,43.75036910395765,43.75027881280732,43.75018403385918,43.75009823336551,43.75000309319235,43.74990348918296,43.74946432592129,43.74846352762352,43.74832746880323,43.74812358583299,43.74793315656997,43.74768891064987,43.74730441220691,43.74628609573283,43.7461273992847,43.74587377364251,43.74571994554294,43.74557981584607,43.74544876944143,43.744896045781,43.744791302946,43.74470646809593,43.74455248420673,43.74445460389098,43.74436658856209,43.7443110706451,43.74424692312098,43.74423014142901,43.744209554656,43.74418864279859,43.74415871812116,43.74414238975461,43.74413033587485,43.74382873745302,43.74380247213749,43.7437842355565,43.74373709553743,43.74366883166084,43.74356427837492,43.74329793426466,43.74414287593608,43.74464445471172,43.74501016669108,43.74537567453007,43.7456823860588,43.74605269566017,43.74641443897862,43.74653641800702,43.74729537926414,43.74809953716351,43.74853339706908,43.74879989406587,43.74892183712982,43.74927441990064,43.74955006810595,43.74978495004873,43.74995641407175,43.74999248403267,43.7500015040188,43.75004660537532,43.75008719745144,43.75013227598704,43.75017733312865,43.75022248168207,43.75026301955962,43.75030805530668,43.75035306823367,43.75039807260289,43.75044307697176,43.75048355922458,43.75052853078916,43.75057349094327,43.75061844111308,43.75066346985814,43.75070389791086,43.75074881670081,43.75079372265338,43.75083410791517,43.75087899389752,43.75092387702603,43.75100494136258,43.75109058214245,43.7511716236559,43.75125265518412,43.75133817316955,43.75141918187452,43.75150476987502,43.75158574577224,43.75166672309451,43.75175217688889,43.75183312140263,43.75191855379857,43.75197269950432,43.75203125594451,43.75208982094217,43.75214391089261,43.75220249871047,43.75226109651199,43.75231521783972,43.75237392844676,43.7524551225592,43.75248671460783,43.75254537801521,43.75259954212713,43.75268074621746,43.75268974480577,43.75272577196326,43.75275730981038,43.75278893764323,43.7528024683298,43.75282501044095,43.75285658109176,43.75289266244649,43.75292425449064,43.75295585651822,43.75299196924935,43.75302359409586,43.75305523035162,43.75308687659022,43.75312303353165,43.7531547025878,43.75318180531723,43.75339416220802,43.75361096118208]}]],[[{"lng":[-79.39642934376944,-79.39646629692629,-79.39650904996078,-79.39655932311346,-79.39667134901052,-79.39674781279949,-79.39714062515064,-79.39727087491747,-79.39735601738597,-79.39744863920522,-79.39754417987329,-79.39761121592957,-79.398032483416,-79.39820942145349,-79.39825100916353,-79.39828882417443,-79.39830896516504,-79.398327432855,-79.39834683965725,-79.39835802419226,-79.3983609311278,-79.39835984760762,-79.39835286124082,-79.39833761724617,-79.39832071406107,-79.39831195187264,-79.39829908020226,-79.39828465514935,-79.39813883327629,-79.39797801927523,-79.39792171549045,-79.39785692903193,-79.39778530281947,-79.39772700396637,-79.39765727342045,-79.39774726105735,-79.39777329814194,-79.39783438211877,-79.39790111494432,-79.39793472850997,-79.39800041994681,-79.39801927024149,-79.39807079750325,-79.39810002278828,-79.3981268932793,-79.39816706670322,-79.39823125830243,-79.39828934002747,-79.39712781252003,-79.39581229816226,-79.39570260292003,-79.39562827946413,-79.39560217958588,-79.39559460909889,-79.39558748154532,-79.39559064612229,-79.39560015864512,-79.39560449833336,-79.39560386994393,-79.39559875198375,-79.39558916709441,-79.39558500868003,-79.39557177352027,-79.39555853833318,-79.39553902180369,-79.39551730627177,-79.3954965097802,-79.39546922953637,-79.39544148610398,-79.3953999024063,-79.39535530056217,-79.39529440107691,-79.39525333382488,-79.3952116676433,-79.39516681188533,-79.3951207810436,-79.3950845615246,-79.39395926489105,-79.39341100265626,-79.39335206841263,-79.39332194791899,-79.39330798727278,-79.39329213280641,-79.39328892905849,-79.39329454908349,-79.39333462133197,-79.39338446930253,-79.39344812242545,-79.39352309058981,-79.39361164925926,-79.39370717082983,-79.39382392397019,-79.39394279708617,-79.39404482932932,-79.3941166555729,-79.39421537540252,-79.39430643375519,-79.3944063183643,-79.3945133143509,-79.3946037533415,-79.39468405308668,-79.39477851642667,-79.39486266892163,-79.39492339635466,-79.39500333818785,-79.39506570415809,-79.395107217573,-79.39517901695804,-79.39526673892509,-79.39539645832035,-79.39578335747701,-79.39600743312423,-79.3968156894763,-79.39791937204861,-79.39781305843327,-79.39770532875757,-79.39759747878463,-79.39748882474599,-79.39738037771662,-79.39727100891911,-79.39716092314698,-79.39705083718412,-79.39694005200722,-79.39682914648581,-79.39671764416302,-79.39660532007426,-79.39649287801645,-79.39638454950662,-79.3962754194509,-79.39616616901883,-79.39605620162348,-79.39594635429157,-79.39583556736669,-79.39572500291986,-79.39565434053712,-79.39561341192058,-79.3955013111721,-79.39538982465804,-79.39527690439695,-79.39516400187274,-79.39505108129811,-79.39493746168219,-79.39482382399777,-79.39470928248861,-79.39459494563262,-79.39448050382444,-79.39436452828279,-79.39424937179116,-79.39413351633074,-79.3940168235571,-79.39390096782772,-79.39378355796713,-79.39366688276861,-79.39354959303922,-79.39343138157776,-79.39331316998397,-79.39319516064933,-79.39307625004537,-79.39295732126121,-79.39283851281853,-79.39271888505789,-79.39259925718159,-79.39247891238635,-79.39235918187789,-79.39223893925092,-79.39211695569757,-79.38714082954894,-79.38708656383218,-79.38704612021282,-79.38691450269242,-79.38683381434696,-79.38667911197115,-79.38662372877423,-79.38631283107797,-79.38615384369962,-79.38603481406113,-79.38592500925638,-79.38581680142788,-79.38568536922271,-79.38551789894078,-79.38529560517982,-79.38524340745414,-79.38510551084391,-79.3849598617436,-79.38485463661094,-79.38470322656444,-79.38460861827784,-79.38449356211808,-79.38439648781342,-79.3842544768176,-79.38414455370379,-79.38526830734963,-79.38638403204681,-79.38899969838862,-79.39160026148903,-79.39419898132383,-79.39642934376944],"lat":[43.74544876944143,43.74557981584607,43.74571994554294,43.74587377364251,43.7461273992847,43.74628609573283,43.74730441220691,43.74768891064987,43.74793315656997,43.74812358583299,43.74832746880323,43.74846352762352,43.74946432592129,43.74990348918296,43.75000309319235,43.75009823336551,43.75018403385918,43.75027881280732,43.75036910395765,43.75044127603061,43.75050892114228,43.75057642030216,43.7506348348114,43.75068413166314,43.75076491195007,43.75083230347614,43.75087713261735,43.75092644093902,43.7512710604803,43.75167848300703,43.75175420987006,43.75182540683374,43.75189191688182,43.75195411280567,43.75200714648673,43.75206241977685,43.75208078868252,43.75211765280339,43.75215009513593,43.75216857023032,43.75220549885947,43.75222835792689,43.75227408985089,43.75231500815278,43.75235589343632,43.75245097665753,43.75260940991249,43.75275884559467,43.75302161763615,43.75330481094405,43.75333927853598,43.75338774552425,43.75343688956346,43.75348638378252,43.75355379801672,43.75360785389746,43.75366199890715,43.75372957414752,43.7538105824936,43.75388711686745,43.75395449659464,43.75402195251242,43.75408027907665,43.75413860563838,43.75420584590758,43.75425955243521,43.75431336188204,43.75437599216881,43.75445211880422,43.754532552035,43.75459944000659,43.75467060004068,43.75472403474885,43.75477305009628,43.75482643156611,43.75486629365488,43.7549107944801,43.75629054920861,43.75696266574575,43.75707886185393,43.75715945546459,43.75724477723872,43.757316659547,43.75737512686231,43.7574292172967,43.75749729500616,43.75757901300186,43.75766542602115,43.75773858524869,43.75779384166275,43.75784919588774,43.75789584657043,43.7579245231279,43.75793946002329,43.75795397232452,43.75797786441705,43.75799714781427,43.75801196418904,43.7580404732643,43.75808675342593,43.75813298109131,43.75819732145342,43.75827952070083,43.75832988419055,43.75838951943516,43.75843990586344,43.75848099748303,43.7585540210931,43.75861385543827,43.75869669420615,43.7588956674559,43.75899333302129,43.75943235608194,43.76010965519837,43.76016667692368,43.76021917774338,43.76027617772309,43.7603287553935,43.76038124584419,43.76043372326235,43.76048619051796,43.76053865766411,43.76058661394511,43.76063906937855,43.76068701538097,43.76073503976186,43.7607874733061,43.76083546351944,43.76087894143109,43.76092691849707,43.76097488539404,43.76101835292138,43.76106630810037,43.76110976534633,43.76113586895492,43.76115329809017,43.76119673354777,43.76124017752032,43.76128360124848,43.76132252416528,43.76136594766477,43.76140486028471,43.76144827348551,43.76148717292713,43.76152607512872,43.7615650657597,43.76160394470517,43.76164283504222,43.76167721449174,43.76171608300309,43.76175046221213,43.76178932040511,43.76182368785688,43.7618580465495,43.76189239216194,43.76192673764979,43.761961175878,43.7619910103429,43.76202534537552,43.76205518102903,43.76208500503406,43.76211482891187,43.76214464257831,43.76217446476019,43.7622042796106,43.76223406983697,43.76330295025861,43.76312655737216,43.7629504495842,43.76235896763723,43.76183553866211,43.76083396416387,43.76058112911779,43.75911807162116,43.75821545637199,43.7575069460257,43.75682116055539,43.75622082551834,43.75556624082379,43.75481212579256,43.7537239851979,43.75347119422147,43.75284793388921,43.75214363660778,43.75164245340051,43.75074894410854,43.75025682253843,43.74968348458295,43.74921842364808,43.74855459510378,43.7480533445385,43.74780367337669,43.74756278965206,43.7470279844626,43.74647031185903,43.74592614691777,43.74544876944143]}]],[[{"lng":[-79.37215205974096,-79.37150506798167,-79.37050735247765,-79.36929475361349,-79.36796897395776,-79.36753069800899,-79.36717834123654,-79.36702277427345,-79.36651260221024,-79.36600332544009,-79.36665621181169,-79.36733892113361,-79.36793091643327,-79.36797049669437,-79.36855482902861,-79.36949851312409,-79.37101912415312,-79.37133920292918,-79.3714196328043,-79.37153915060388,-79.37166339004001,-79.3717782249589,-79.37192119048125,-79.37205004922079,-79.37213094610958,-79.37223708653346,-79.37229279240925,-79.37237726147985,-79.37254924357684,-79.37262878077144,-79.3727422499983,-79.37285547021689,-79.3729435184934,-79.37300765231507,-79.37309627724336,-79.37319894237318,-79.37330420123132,-79.3733969175517,-79.37344558819258,-79.37455423129187,-79.37588716402503,-79.3765545609014,-79.37656691863995,-79.37670442353101,-79.37433665171579,-79.37304872154483,-79.3722533850442,-79.37215205974096],"lat":[43.66338269055721,43.66350850342528,43.66371492170568,43.66398569048605,43.66427742521685,43.66319974351389,43.66232583336494,43.66194543508158,43.6606955930494,43.65943234827975,43.65928405557604,43.65911818111177,43.65898260427006,43.65897866848724,43.65886539405275,43.65865381069511,43.6582883176773,43.65828837858037,43.65829402578726,43.65829122788013,43.65828399610229,43.65827663020104,43.65826516391561,43.65825349650262,43.65824564679841,43.65821565148454,43.65819393979815,43.65816363569178,43.65808497653349,43.65803659792311,43.65796619730759,43.65788228996902,43.65778002005982,43.65769991473263,43.65760656493401,43.65750450290244,43.65739347565185,43.65732277897954,43.65729646557103,43.65706009185757,43.66043302307911,43.66204522865871,43.66207241031807,43.66241184991879,43.66290958321159,43.66318382133743,43.66335712769619,43.66338269055721]}]],[[{"lng":[-79.38714082954894,-79.38348132575379,-79.38242485571513,-79.38056313032935,-79.37973070290914,-79.3795323785688,-79.37933415639333,-79.37913511474122,-79.3789368920861,-79.37873703071558,-79.37853809074689,-79.37833845287322,-79.37813869076753,-79.37793884764663,-79.37773838988466,-79.37753793191099,-79.37733747372559,-79.37713711773361,-79.37693596156377,-79.37673478592164,-79.3765336293726,-79.37633245332094,-79.37613057955539,-79.37592942244491,-79.37572766765069,-79.37552507651465,-79.37545720149596,-79.37532309962839,-79.37512124437691,-79.37491324464084,-79.37470534715453,-79.37449675213216,-79.37428887382437,-79.3740801566568,-79.37387156124463,-79.37366286330203,-79.37345426764035,-79.37324497456686,-79.37303555942525,-79.37282686099898,-79.37261767001921,-79.3724082741343,-79.37219908296747,-79.37198968445438,-79.37177967385337,-79.37157048240555,-79.36906760198991,-79.36575081638631,-79.36524704252503,-79.36291698145654,-79.36279622175728,-79.36266263850541,-79.36260561702778,-79.36256825813702,-79.36253952232863,-79.36251111412433,-79.36248239873862,-79.3624553039618,-79.36242925117247,-79.36240381284402,-79.3623791937851,-79.36235469752968,-79.36233161460255,-79.36230924853577,-79.36228770173263,-79.36226709697169,-79.36224708628814,-79.36222812005087,-79.36220975034098,-79.36218822404463,-79.3621676194149,-79.36214783404574,-79.36212866312727,-79.36211041387027,-79.36209288146496,-79.36207534908272,-79.36205935278061,-79.36204417572935,-79.36202971552358,-79.36201536019618,-79.36200253848443,-79.36199045400808,-79.36197916838057,-79.36196788276757,-79.36195825601754,-79.36194932570585,-79.36194029300235,-79.36193302155986,-79.36190853749353,-79.3618957568812,-79.36189678878657,-79.36190515493948,-79.3619050100349,-79.36190463807323,-79.36186570253902,-79.36179565909954,-79.36164211659914,-79.36146470103017,-79.36125814779373,-79.36120346991541,-79.36095471656067,-79.36075696184061,-79.36058016127677,-79.36039720140229,-79.36017886027419,-79.35995888979716,-79.36036547631332,-79.36062049207035,-79.3608370527259,-79.36191443080125,-79.3628216510599,-79.36302946881682,-79.3632008574348,-79.36336825299952,-79.36348042340518,-79.36413775741131,-79.36548284297372,-79.36900361949525,-79.37044282450597,-79.37164852365525,-79.37364188474146,-79.37486374942821,-79.37543304977802,-79.37606768516649,-79.37626374915425,-79.37734778335718,-79.37874514500224,-79.37930251602299,-79.37998452895447,-79.38064769205042,-79.38078369927845,-79.3818893145413,-79.38210529623395,-79.38234808652948,-79.38251149369029,-79.38265944149121,-79.38293078430982,-79.38324193555739,-79.38347615015896,-79.38370361420615,-79.38383408249669,-79.38398353395961,-79.38414455370379,-79.3842544768176,-79.38439648781342,-79.38449356211808,-79.38460861827784,-79.38470322656444,-79.38485463661094,-79.3849598617436,-79.38510551084391,-79.38524340745414,-79.38529560517982,-79.38551789894078,-79.38568536922271,-79.38581680142788,-79.38592500925638,-79.38603481406113,-79.38615384369962,-79.38631283107797,-79.38662372877423,-79.38667911197115,-79.38683381434696,-79.38691450269242,-79.38704612021282,-79.38708656383218,-79.38714082954894],"lat":[43.76330295025861,43.76407518110707,43.76429878044731,43.76469108812113,43.76486841715406,43.76490611327332,43.76494381049665,43.76498149575102,43.76501919227688,43.76505686520715,43.76509455086203,43.76512772532127,43.76516548863038,43.76519865947682,43.76523182124684,43.76526498266152,43.76529814372086,43.76533130587927,43.76535995537617,43.76539310518525,43.76542175396725,43.76545490306085,43.76548354094076,43.7655121886493,43.76553641658273,43.76556504319378,43.76557470044282,43.76559367817872,43.76561781359254,43.76564186126239,43.76566591000747,43.76568544750776,43.76570499482385,43.76572903076791,43.76574856711994,43.76576810163138,43.76578763721717,43.76580266155001,43.76582218470389,43.76584171767973,43.76585674231618,43.76587176365059,43.76588678751727,43.76590189806522,43.76591690948824,43.7659319321989,43.76605385845743,43.76620451051097,43.76622881088205,43.76633507087413,43.76618021864592,43.76600726878728,43.76593443662779,43.76588439091707,43.76585247253329,43.76581605789513,43.76577963884755,43.76574765394074,43.76571127303745,43.76567490093422,43.76563854056677,43.7655976810119,43.76556134264963,43.76552501455685,43.76548869820168,43.76544789440229,43.76541160005588,43.7653708197339,43.76533445888195,43.76529364185885,43.76525283804229,43.76521204596493,43.7651712626914,43.76513049262565,43.7650897328323,43.76504897303632,43.76500823525652,43.76496750921791,43.76492679345294,43.76488598917047,43.76484529688931,43.76480011423237,43.76475944396805,43.7647187737026,43.76467362627378,43.76463298977026,43.76459235179793,43.76454723813084,43.7637365373658,43.76355172544976,43.76260186108202,43.76206636865238,43.76198534957447,43.76168369093395,43.76134997281571,43.76101130773803,43.76029786692717,43.7595434849683,43.75871216867946,43.75844123767211,43.75734375954475,43.75642705026073,43.75567717625852,43.75484169547778,43.75388427091815,43.75294923715237,43.75290547157255,43.75289112674602,43.75288973256973,43.75292768726551,43.75294059918825,43.75294807749138,43.75292802789181,43.75290792084293,43.75289602453591,43.75278391121953,43.75248800308032,43.751763967364,43.751451430275,43.75122097581056,43.75079917908513,43.75054191497333,43.75041498161067,43.75028447298505,43.75023774870395,43.75002800982404,43.74973718846163,43.74961906842661,43.74946670517735,43.74934098643234,43.74930690613238,43.749070424141,43.74903297376775,43.74895080292797,43.7488765998475,43.74880217772306,43.74864848448289,43.74845025425553,43.74828703246924,43.74819573001999,43.7481389729254,43.74809607703707,43.7480533445385,43.74855459510378,43.74921842364808,43.74968348458295,43.75025682253843,43.75074894410854,43.75164245340051,43.75214363660778,43.75284793388921,43.75347119422147,43.7537239851979,43.75481212579256,43.75556624082379,43.75622082551834,43.75682116055539,43.7575069460257,43.75821545637199,43.75911807162116,43.76058112911779,43.76083396416387,43.76183553866211,43.76235896763723,43.7629504495842,43.76312655737216,43.76330295025861]}]],[[{"lng":[-79.36039720140229,-79.36058016127677,-79.36075696184061,-79.36095471656067,-79.36120346991541,-79.36125814779373,-79.36146470103017,-79.36164211659914,-79.36179565909954,-79.36186570253902,-79.36190463807323,-79.3619050100349,-79.36190515493948,-79.36189678878657,-79.3618957568812,-79.36190853749353,-79.36193302155986,-79.36194029300235,-79.36194932570585,-79.36195825601754,-79.36196788276757,-79.36197916838057,-79.36199045400808,-79.36200253848443,-79.36201536019618,-79.36202971552358,-79.36204417572935,-79.36205935278061,-79.36207534908272,-79.36209288146496,-79.36211041387027,-79.36212866312727,-79.36214783404574,-79.3621676194149,-79.36218822404463,-79.36220975034098,-79.36222812005087,-79.36224708628814,-79.36226709697169,-79.36228770173263,-79.36230924853577,-79.36233161460255,-79.36235469752968,-79.3623791937851,-79.36240381284402,-79.36242925117247,-79.3624553039618,-79.36248239873862,-79.36251111412433,-79.36253952232863,-79.36256825813702,-79.36260561702778,-79.36266263850541,-79.36279622175728,-79.36291698145654,-79.36276845192161,-79.36012752170177,-79.35965254089474,-79.35965686391766,-79.35965565700673,-79.35965459411068,-79.35965891710443,-79.35965433418409,-79.35965234779461,-79.3595992630315,-79.35950146860623,-79.3593794888735,-79.35923602214017,-79.35906613548316,-79.35897567247029,-79.35892676512364,-79.35876498919416,-79.35863141526336,-79.3584714040597,-79.35832816438172,-79.3581954116008,-79.35804041902756,-79.35791041129275,-79.35775349256478,-79.35760111902751,-79.35743942578371,-79.35729030617381,-79.35717620840623,-79.3570299145362,-79.35688845285607,-79.35673504990859,-79.3566168918002,-79.35654695038453,-79.35644774245768,-79.35626811056757,-79.35608728918163,-79.35589717015509,-79.35572955914749,-79.35561069210962,-79.35548260892921,-79.35536572751396,-79.35520280458614,-79.35506017618495,-79.3549275622714,-79.35481584072534,-79.35473335275231,-79.35470354641076,-79.35466583673862,-79.35461493815774,-79.35457745211725,-79.35451639538248,-79.35441732456418,-79.35431507971094,-79.35422897267441,-79.35417532946956,-79.35412062053095,-79.35400457402199,-79.35388057501564,-79.35374446679455,-79.35362543578286,-79.35351131722325,-79.35344794977229,-79.35333878553543,-79.3532367652863,-79.35316894371401,-79.35313343844719,-79.35312916286563,-79.35314969185424,-79.35318645774254,-79.3532119178949,-79.35321264434732,-79.35319215543169,-79.35312460371306,-79.35300626635208,-79.35288361294749,-79.35274599094694,-79.35261943777002,-79.35256180674395,-79.35251994015735,-79.35246057801105,-79.35234820824756,-79.35222907695503,-79.35209569414098,-79.35198434547667,-79.35188845938625,-79.35179435646995,-79.35168585404497,-79.3515746045545,-79.35146378232044,-79.351352508014,-79.35126349337834,-79.35119227155685,-79.35116162160134,-79.35115362344766,-79.35115639861935,-79.35113588378277,-79.35108342207529,-79.35099192766569,-79.35088833591597,-79.35075480413909,-79.35061839153605,-79.35042431362007,-79.35039224461981,-79.35022176069491,-79.35007828731594,-79.34993375172833,-79.34979030445017,-79.34965273290307,-79.34952854199651,-79.34941374500335,-79.34940694821256,-79.34942850584689,-79.34947387729201,-79.34949580330931,-79.34952286026622,-79.3495503155391,-79.34957669981914,-79.34960916219755,-79.34965817616862,-79.34969232091041,-79.34973637508408,-79.34977540502639,-79.34980890539536,-79.34983011391226,-79.34984410951817,-79.34983833677656,-79.34984279470279,-79.34983443828138,-79.34982452216158,-79.34979351239939,-79.34974427206816,-79.34963584683983,-79.3495006136752,-79.34935261716998,-79.34923914533226,-79.34912960731575,-79.34902116182272,-79.34890475549378,-79.34880344887034,-79.34875195196956,-79.34869217569084,-79.34862238632419,-79.34853344487077,-79.34840795134375,-79.34827509544846,-79.34813029780423,-79.34796795286402,-79.34785520023121,-79.347693303374,-79.34757113562414,-79.34746299105682,-79.34732394508083,-79.34720306086014,-79.34717441206509,-79.34707484222336,-79.34692975010013,-79.34679409974852,-79.3466310399579,-79.3464693194055,-79.34637024541837,-79.34629764310782,-79.34626845282557,-79.34624870187197,-79.34622848215572,-79.34615282518087,-79.34606617191643,-79.34596913178666,-79.34584176250252,-79.34567264204594,-79.34547095365164,-79.345268921217,-79.34504709061096,-79.3448561636996,-79.34465960955662,-79.3444845426194,-79.34436853023787,-79.34425640667264,-79.34416623835139,-79.34408927864345,-79.34401459603897,-79.3440091229229,-79.34389066602247,-79.3470335880067,-79.34866885723075,-79.35087218509895,-79.35202370249405,-79.35263471904855,-79.35408871385768,-79.35521045018685,-79.35554644244142,-79.35748186482101,-79.35872661897722,-79.35995888979716,-79.36017886027419,-79.36039720140229],"lat":[43.75484169547778,43.75567717625852,43.75642705026073,43.75734375954475,43.75844123767211,43.75871216867946,43.7595434849683,43.76029786692717,43.76101130773803,43.76134997281571,43.76168369093395,43.76198534957447,43.76206636865238,43.76260186108202,43.76355172544976,43.7637365373658,43.76454723813084,43.76459235179793,43.76463298977026,43.76467362627378,43.7647187737026,43.76475944396805,43.76480011423237,43.76484529688931,43.76488598917047,43.76492679345294,43.76496750921791,43.76500823525652,43.76504897303632,43.7650897328323,43.76513049262565,43.7651712626914,43.76521204596493,43.76525283804229,43.76529364185885,43.76533445888195,43.7653708197339,43.76541160005588,43.76544789440229,43.76548869820168,43.76552501455685,43.76556134264963,43.7655976810119,43.76563854056677,43.76567490093422,43.76571127303745,43.76574765394074,43.76577963884755,43.76581605789513,43.76585247253329,43.76588439091707,43.76593443662779,43.76600726878728,43.76618021864592,43.76633507087413,43.76634194466477,43.76645710585505,43.76647729381905,43.76640534082216,43.76633330843862,43.76622977153335,43.76615781853308,43.76604063824989,43.76602710690993,43.76592282317868,43.76580889565106,43.76569903170951,43.76560695288528,43.76551449447207,43.76547718778254,43.76544497879341,43.76537064018402,43.76531020928415,43.76523130464713,43.76515723170804,43.76507430738556,43.76495055485224,43.76487217058444,43.76478889845718,43.76474169898128,43.76471687004191,43.76471922728938,43.76472208786787,43.76473348728589,43.76476295977064,43.76480576317665,43.76484466235755,43.76484365642303,43.76483772854667,43.76481705084693,43.7647874438867,43.76475320191942,43.76472378447003,43.76469506826112,43.76466621928875,43.76465103422804,43.76464868913347,43.76466463973281,43.76468523527486,43.76468362676363,43.76460592306114,43.76447496656839,43.76434830718635,43.76419454223926,43.7640634751196,43.76393647947895,43.76381352731254,43.76369052934474,43.76357676560455,43.76346337937883,43.76333656496585,43.7632088669691,43.76313956654538,43.76311960189737,43.76312688880058,43.76315675107931,43.76317384174699,43.76321727807248,43.76328782290719,43.76338136525237,43.76348437521043,43.76358792528401,43.7636872418945,43.76381829910727,43.7639357805599,43.7640213089311,43.76408852777459,43.76414606643291,43.76415786362618,43.76412909000325,43.76408209669246,43.76396774866625,43.76383179948893,43.76371867230425,43.76364580132368,43.76360817351551,43.76359295280334,43.76355952266595,43.76354441393773,43.76352952812805,43.76350566607439,43.76349509905295,43.76350249606786,43.76352790290789,43.76356230500211,43.76361053100295,43.76368601933346,43.7637575921343,43.76382499085263,43.76388804408739,43.76395985314717,43.76403111117352,43.76408380211492,43.764095809913,43.76409388265482,43.76405140517988,43.76396758659802,43.76394911986394,43.76387905436157,43.7638139695861,43.76373536646495,43.76364327605152,43.76356477314791,43.76345045581488,43.76327767177467,43.76315604814941,43.76299423560697,43.76284185894679,43.76266663889482,43.76255441612778,43.76237927592295,43.76223562683882,43.76212798306102,43.76198466080228,43.76184562470994,43.76173364736254,43.76164419217229,43.76155465711057,43.76148744924275,43.76142922911172,43.76141564291365,43.76133919124928,43.76124446073141,43.76115429864971,43.76106383193562,43.76099560664158,43.76092652651736,43.7608749728242,43.76083682744655,43.76081718442778,43.76078859624288,43.76072401618552,43.76063231532866,43.76050932594264,43.76037796436862,43.76026457691619,43.76014654377619,43.76005073851272,43.75995431484195,43.75989838312912,43.75985578146105,43.75979942308037,43.75977978908458,43.75973693962631,43.75969466448111,43.75967509673684,43.75965058100735,43.75963983064586,43.75963491530559,43.7596424768936,43.75964037794608,43.75963391447474,43.75957754385144,43.75949859763267,43.7594026440523,43.75925306215987,43.7591265232229,43.75899570996466,43.75885588802551,43.75872426559999,43.7586013958287,43.75848746754664,43.75837760107171,43.75826253902626,43.7581695999072,43.75806315261216,43.75794291537861,43.75782753628611,43.75768966064245,43.75753409219015,43.75740629480028,43.75727864365969,43.75714230867219,43.75700616502789,43.75686546336465,43.75681137268865,43.75649008868957,43.7558062638701,43.7554337344742,43.75495235677507,43.75468982120807,43.75455010048659,43.75423789304794,43.75398839915406,43.75391221794577,43.75348536896519,43.75321960490621,43.75294923715237,43.75388427091815,43.75484169547778]}]],[[{"lng":[-79.35965254089474,-79.35697006156346,-79.35337324134723,-79.350733916594,-79.34878179793132,-79.34676558272204,-79.3436244438809,-79.34061420217522,-79.3389502435375,-79.33877669961306,-79.33877130249316,-79.33875607128726,-79.33874073521575,-79.33874014292049,-79.33872337319013,-79.33870588664978,-79.33869967033327,-79.33868760304418,-79.33866858048626,-79.33864792196168,-79.33862797783277,-79.33862726099046,-79.33860600778227,-79.33859238237866,-79.33858381081511,-79.33856007781066,-79.33853624245215,-79.33851179520397,-79.33848570703233,-79.33846351031583,-79.33843977757083,-79.33841604487732,-79.33839077616234,-79.33836540509763,-79.33833860291486,-79.33831179829913,-79.33828335527157,-79.3382549123063,-79.33822575257419,-79.33819505684534,-79.33817882997249,-79.33816413668366,-79.33813190503061,-79.33809957104484,-79.33807603529054,-79.33806649811424,-79.33803252581393,-79.33799699534245,-79.33796169444138,-79.33792544729738,-79.33789509519782,-79.33786627919626,-79.33783910169132,-79.33781192423437,-79.33778618046264,-79.33776053913887,-79.33773571957235,-79.33771222878812,-79.33768965966568,-79.33766721520377,-79.33764618219317,-79.33762596843526,-79.33760659615068,-79.33758802089264,-79.33756996016788,-79.33755294322452,-79.33752058831575,-79.3374904863042,-79.33746061138996,-79.33743132871329,-79.33740288754203,-79.33737596270097,-79.33734916006362,-79.33732294964666,-79.33729829753619,-79.33727372563413,-79.33725081443525,-79.33722780337874,-79.33720620123071,-79.3371848261781,-79.33716498717015,-79.33714514820481,-79.33712674287747,-79.33710854238858,-79.33709165336667,-79.33685344020637,-79.33587142693519,-79.33598904319395,-79.34388300935721,-79.34389066602247,-79.3440091229229,-79.34401459603897,-79.34408927864345,-79.34416623835139,-79.34425640667264,-79.34436853023787,-79.3444845426194,-79.34465960955662,-79.3448561636996,-79.34504709061096,-79.345268921217,-79.34547095365164,-79.34567264204594,-79.34584176250252,-79.34596913178666,-79.34606617191643,-79.34615282518087,-79.34622848215572,-79.34624870187197,-79.34626845282557,-79.34629764310782,-79.34637024541837,-79.3464693194055,-79.3466310399579,-79.34679409974852,-79.34692975010013,-79.34707484222336,-79.34717441206509,-79.34720306086014,-79.34732394508083,-79.34746299105682,-79.34757113562414,-79.347693303374,-79.34785520023121,-79.34796795286402,-79.34813029780423,-79.34827509544846,-79.34840795134375,-79.34853344487077,-79.34862238632419,-79.34869217569084,-79.34875195196956,-79.34880344887034,-79.34890475549378,-79.34902116182272,-79.34912960731575,-79.34923914533226,-79.34935261716998,-79.3495006136752,-79.34963584683983,-79.34974427206816,-79.34979351239939,-79.34982452216158,-79.34983443828138,-79.34984279470279,-79.34983833677656,-79.34984410951817,-79.34983011391226,-79.34980890539536,-79.34977540502639,-79.34973637508408,-79.34969232091041,-79.34965817616862,-79.34960916219755,-79.34957669981914,-79.3495503155391,-79.34952286026622,-79.34949580330931,-79.34947387729201,-79.34942850584689,-79.34940694821256,-79.34941374500335,-79.34952854199651,-79.34965273290307,-79.34979030445017,-79.34993375172833,-79.35007828731594,-79.35022176069491,-79.35039224461981,-79.35042431362007,-79.35061839153605,-79.35075480413909,-79.35088833591597,-79.35099192766569,-79.35108342207529,-79.35113588378277,-79.35115639861935,-79.35115362344766,-79.35116162160134,-79.35119227155685,-79.35126349337834,-79.351352508014,-79.35146378232044,-79.3515746045545,-79.35168585404497,-79.35179435646995,-79.35188845938625,-79.35198434547667,-79.35209569414098,-79.35222907695503,-79.35234820824756,-79.35246057801105,-79.35251994015735,-79.35256180674395,-79.35261943777002,-79.35274599094694,-79.35288361294749,-79.35300626635208,-79.35312460371306,-79.35319215543169,-79.35321264434732,-79.3532119178949,-79.35318645774254,-79.35314969185424,-79.35312916286563,-79.35313343844719,-79.35316894371401,-79.3532367652863,-79.35333878553543,-79.35344794977229,-79.35351131722325,-79.35362543578286,-79.35374446679455,-79.35388057501564,-79.35400457402199,-79.35412062053095,-79.35417532946956,-79.35422897267441,-79.35431507971094,-79.35441732456418,-79.35451639538248,-79.35457745211725,-79.35461493815774,-79.35466583673862,-79.35470354641076,-79.35473335275231,-79.35481584072534,-79.3549275622714,-79.35506017618495,-79.35520280458614,-79.35536572751396,-79.35548260892921,-79.35561069210962,-79.35572955914749,-79.35589717015509,-79.35608728918163,-79.35626811056757,-79.35644774245768,-79.35654695038453,-79.3566168918002,-79.35673504990859,-79.35688845285607,-79.3570299145362,-79.35717620840623,-79.35729030617381,-79.35743942578371,-79.35760111902751,-79.35775349256478,-79.35791041129275,-79.35804041902756,-79.3581954116008,-79.35832816438172,-79.3584714040597,-79.35863141526336,-79.35876498919416,-79.35892676512364,-79.35897567247029,-79.35906613548316,-79.35923602214017,-79.3593794888735,-79.35950146860623,-79.3595992630315,-79.35965234779461,-79.35965433418409,-79.35965891710443,-79.35965459411068,-79.35965565700673,-79.35965686391766,-79.35965254089474],"lat":[43.76647729381905,43.76660537833934,43.7667742385137,43.76688929556457,43.76700063155901,43.76712901056351,43.76729965478168,43.76748561640034,43.76759655776156,43.76760753763989,43.76758045354511,43.76751262801366,43.76744489097362,43.76744038142413,43.76737262353732,43.7673048552256,43.76727775921579,43.76723257438339,43.76716478373195,43.76709687927155,43.76702907521268,43.76702906478995,43.76695674075427,43.76691603419328,43.76688890393005,43.76682104476628,43.76675318410863,43.76668522452918,43.76661733110194,43.76654949425463,43.76648163506631,43.76641377587313,43.76634589433773,43.76627801130774,43.7662100174409,43.76614211355053,43.76607418582612,43.76600625809487,43.76593831993177,43.76587035942168,43.76583852683017,43.76580680655919,43.76573882369269,43.76567083932805,43.76562548761803,43.7656073451454,43.76553933693179,43.76547580698298,43.7654076894034,43.76534414900676,43.7652896962279,43.76523526578534,43.76518085916975,43.76512645254797,43.76507206677535,43.76501768248682,43.76496322012791,43.7649088671128,43.76485452750073,43.76479568875735,43.76474137148342,43.7646870661241,43.76462827206295,43.76457399053402,43.76451962646879,43.76446086666494,43.7643973827601,43.76433393162453,43.76426598284435,43.76420254361332,43.764134615678,43.76407112072034,43.76400321661217,43.76393982205516,43.76387194922641,43.76380857849952,43.76374073099224,43.76367279200774,43.76360946448028,43.76354163931203,43.76347383649245,43.76340603366937,43.7633382517057,43.7632704727194,43.76320712373781,43.76221777051889,43.75824336304536,43.75821662994846,43.75644460817487,43.75649008868957,43.75681137268865,43.75686546336465,43.75700616502789,43.75714230867219,43.75727864365969,43.75740629480028,43.75753409219015,43.75768966064245,43.75782753628611,43.75794291537861,43.75806315261216,43.7581695999072,43.75826253902626,43.75837760107171,43.75848746754664,43.7586013958287,43.75872426559999,43.75885588802551,43.75899570996466,43.7591265232229,43.75925306215987,43.7594026440523,43.75949859763267,43.75957754385144,43.75963391447474,43.75964037794608,43.7596424768936,43.75963491530559,43.75963983064586,43.75965058100735,43.75967509673684,43.75969466448111,43.75973693962631,43.75977978908458,43.75979942308037,43.75985578146105,43.75989838312912,43.75995431484195,43.76005073851272,43.76014654377619,43.76026457691619,43.76037796436862,43.76050932594264,43.76063231532866,43.76072401618552,43.76078859624288,43.76081718442778,43.76083682744655,43.7608749728242,43.76092652651736,43.76099560664158,43.76106383193562,43.76115429864971,43.76124446073141,43.76133919124928,43.76141564291365,43.76142922911172,43.76148744924275,43.76155465711057,43.76164419217229,43.76173364736254,43.76184562470994,43.76198466080228,43.76212798306102,43.76223562683882,43.76237927592295,43.76255441612778,43.76266663889482,43.76284185894679,43.76299423560697,43.76315604814941,43.76327767177467,43.76345045581488,43.76356477314791,43.76364327605152,43.76373536646495,43.7638139695861,43.76387905436157,43.76394911986394,43.76396758659802,43.76405140517988,43.76409388265482,43.764095809913,43.76408380211492,43.76403111117352,43.76395985314717,43.76388804408739,43.76382499085263,43.7637575921343,43.76368601933346,43.76361053100295,43.76356230500211,43.76352790290789,43.76350249606786,43.76349509905295,43.76350566607439,43.76352952812805,43.76354441393773,43.76355952266595,43.76359295280334,43.76360817351551,43.76364580132368,43.76371867230425,43.76383179948893,43.76396774866625,43.76408209669246,43.76412909000325,43.76415786362618,43.76414606643291,43.76408852777459,43.7640213089311,43.7639357805599,43.76381829910727,43.7636872418945,43.76358792528401,43.76348437521043,43.76338136525237,43.76328782290719,43.76321727807248,43.76317384174699,43.76315675107931,43.76312688880058,43.76311960189737,43.76313956654538,43.7632088669691,43.76333656496585,43.76346337937883,43.76357676560455,43.76369052934474,43.76381352731254,43.76393647947895,43.7640634751196,43.76419454223926,43.76434830718635,43.76447496656839,43.76460592306114,43.76468362676363,43.76468523527486,43.76466463973281,43.76464868913347,43.76465103422804,43.76466621928875,43.76469506826112,43.76472378447003,43.76475320191942,43.7647874438867,43.76481705084693,43.76483772854667,43.76484365642303,43.76484466235755,43.76480576317665,43.76476295977064,43.76473348728589,43.76472208786787,43.76471922728938,43.76471687004191,43.76474169898128,43.76478889845718,43.76487217058444,43.76495055485224,43.76507430738556,43.76515723170804,43.76523130464713,43.76531020928415,43.76537064018402,43.76544497879341,43.76547718778254,43.76551449447207,43.76560695288528,43.76569903170951,43.76580889565106,43.76592282317868,43.76602710690993,43.76604063824989,43.76615781853308,43.76622977153335,43.76633330843862,43.76640534082216,43.76647729381905]}]],[[{"lng":[-79.33877669961306,-79.33860633025151,-79.33799645701376,-79.33753595935123,-79.33746658525511,-79.33735450736826,-79.33724652574261,-79.33708926398849,-79.33562000437189,-79.33553600806883,-79.33395144389149,-79.33205352262377,-79.32992933115671,-79.32886072975144,-79.32599505778229,-79.32446360520558,-79.32381744638717,-79.32317128759361,-79.32252515218899,-79.32187840246191,-79.3212322438408,-79.3205862111618,-79.3197822193865,-79.31969947205775,-79.31961910665758,-79.31961252096615,-79.31958468929737,-79.31952838810545,-79.31944589676664,-79.31936330099725,-79.31929743191947,-79.31923225391367,-79.31916791910923,-79.31910509963195,-79.31904230159218,-79.31881612528737,-79.31858418970982,-79.31815300255789,-79.31810623820257,-79.31802777294492,-79.3179438005976,-79.31751989076866,-79.31722825888824,-79.31707521152801,-79.317031270443,-79.31688561246179,-79.31660703371563,-79.31631321585591,-79.31616470080256,-79.31607200495634,-79.31588148831607,-79.31561109693585,-79.31523339913841,-79.31551034361692,-79.31644567529281,-79.31737144806682,-79.31771123687879,-79.31819286624221,-79.31854373759762,-79.31918489098085,-79.31940923646002,-79.31989406602693,-79.32019937489332,-79.32056089507392,-79.32091283822565,-79.32136135811892,-79.32165915083083,-79.32198354790837,-79.32233781629607,-79.32303335308197,-79.32348186839069,-79.32371640631821,-79.32404270324649,-79.3242179749239,-79.32435922057459,-79.32452882034326,-79.32475472809128,-79.32498334444976,-79.32598936571075,-79.32767411844334,-79.32968774086262,-79.32975182827116,-79.33063258832381,-79.33268911435356,-79.33318401736474,-79.33329546357146,-79.33334963945791,-79.33340391769815,-79.33345727436797,-79.3335108583,-79.33356329335849,-79.33361613794867,-79.33366808347431,-79.33371931220938,-79.3337704385078,-79.33382072313285,-79.33387113260206,-79.33392095015944,-79.33397064529088,-79.33401869961018,-79.33405175971913,-79.33406675388765,-79.33411493298459,-79.33416167854676,-79.33421587619836,-79.33427253120465,-79.33432918616204,-79.33438666020369,-79.33444403180417,-79.3345021200961,-79.33456041312103,-79.33461919562208,-79.33464033312853,-79.33467892203103,-79.33473852357523,-79.33479812507714,-79.33485854816368,-79.33491958305912,-79.33498082269392,-79.33504195989414,-79.33510391618077,-79.33516574764292,-79.33524646381161,-79.33532777188165,-79.33540907989544,-79.3354912069814,-79.33558656481001,-79.33565617772,-79.33573902137218,-79.33582184262728,-79.33587105728418,-79.33587142693519,-79.33685344020637,-79.33709165336667,-79.33710854238858,-79.33712674287747,-79.33714514820481,-79.33716498717015,-79.3371848261781,-79.33720620123071,-79.33722780337874,-79.33725081443525,-79.33727372563413,-79.33729829753619,-79.33732294964666,-79.33734916006362,-79.33737596270097,-79.33740288754203,-79.33743132871329,-79.33746061138996,-79.3374904863042,-79.33752058831575,-79.33755294322452,-79.33756996016788,-79.33758802089264,-79.33760659615068,-79.33762596843526,-79.33764618219317,-79.33766721520377,-79.33768965966568,-79.33771222878812,-79.33773571957235,-79.33776053913887,-79.33778618046264,-79.33781192423437,-79.33783910169132,-79.33786627919626,-79.33789509519782,-79.33792544729738,-79.33796169444138,-79.33799699534245,-79.33803252581393,-79.33806649811424,-79.33807603529054,-79.33809957104484,-79.33813190503061,-79.33816413668366,-79.33817882997249,-79.33819505684534,-79.33822575257419,-79.3382549123063,-79.33828335527157,-79.33831179829913,-79.33833860291486,-79.33836540509763,-79.33839077616234,-79.33841604487732,-79.33843977757083,-79.33846351031583,-79.33848570703233,-79.33851179520397,-79.33853624245215,-79.33856007781066,-79.33858381081511,-79.33859238237866,-79.33860600778227,-79.33862726099046,-79.33862797783277,-79.33864792196168,-79.33866858048626,-79.33868760304418,-79.33869967033327,-79.33870588664978,-79.33872337319013,-79.33874014292049,-79.33874073521575,-79.33875607128726,-79.33877130249316,-79.33877669961306],"lat":[43.76760753763989,43.76761856341255,43.76765920455163,43.76768851303585,43.76769650554163,43.76770387663237,43.76771130722151,43.76772252146967,43.76780915198191,43.76781242938645,43.76791545250565,43.76803179359607,43.7681537980674,43.76821469033275,43.76831245201439,43.76833951559493,43.76834354140075,43.76834756355054,43.7683470814514,43.76834658667404,43.7683505978541,43.76835010629312,43.76835629029436,43.76817953698647,43.76799822768898,43.76798462803026,43.76792120562291,43.76779885234116,43.76761301070967,43.76742725749645,43.76725516304423,43.76708766967041,43.76691568772245,43.76674813894356,43.76657617952335,43.7660011429349,43.76541242827054,43.76450580481824,43.76440159490242,43.76421131048063,43.76395802180933,43.76301540537754,43.76237179741845,43.7620138791114,43.76191871218115,43.76159690995534,43.76098508915786,43.76033244459204,43.75995208703921,43.7597436776993,43.75936720169671,43.75875999989272,43.75786315494278,43.75781313667134,43.75770980699681,43.75759291608304,43.75762492487553,43.7576725232608,43.75775870401343,43.75804278463598,43.75818111279621,43.75855740992448,43.75880053865463,43.75908491122369,43.75936023004256,43.75967297413951,43.75990689661761,43.76010529149676,43.76029053119041,43.76066090650189,43.76091954085078,43.76103109443069,43.76116190717595,43.76124990568387,43.76128348367314,43.76129947348664,43.76129828483733,43.76128813356452,43.76104632288984,43.7606613169333,43.76020007697259,43.76018301001005,43.75997974574385,43.75951007765864,43.75930566367109,43.75925777893695,43.75923606443319,43.75921435139647,43.7591926248928,43.7591664007349,43.75914466074023,43.75912292669378,43.7590966785679,43.75907041996554,43.75904415984627,43.75902238837472,43.75899611775746,43.75896533754332,43.75893896644723,43.75891266142522,43.75889513966065,43.75888635638222,43.75885555219459,43.75882472708734,43.75879851154345,43.75877233179643,43.75874615202039,43.75871998415567,43.75869381476866,43.75866765579893,43.75864149978351,43.75861985181633,43.75861115799927,43.75859371663125,43.75857208053927,43.75855044441539,43.75852873021407,43.75850711491483,43.75848550256622,43.75846388869206,43.7584422867195,43.75842518383892,43.75839935417301,43.7583780340174,43.75835671380295,43.75833540546192,43.75830978884017,43.75829279903949,43.75827150095704,43.7582547034327,43.75824191735961,43.75824336304536,43.76221777051889,43.76320712373781,43.7632704727194,43.7633382517057,43.76340603366937,43.76347383649245,43.76354163931203,43.76360946448028,43.76367279200774,43.76374073099224,43.76380857849952,43.76387194922641,43.76393982205516,43.76400321661217,43.76407112072034,43.764134615678,43.76420254361332,43.76426598284435,43.76433393162453,43.7643973827601,43.76446086666494,43.76451962646879,43.76457399053402,43.76462827206295,43.7646870661241,43.76474137148342,43.76479568875735,43.76485452750073,43.7649088671128,43.76496322012791,43.76501768248682,43.76507206677535,43.76512645254797,43.76518085916975,43.76523526578534,43.7652896962279,43.76534414900676,43.7654076894034,43.76547580698298,43.76553933693179,43.7656073451454,43.76562548761803,43.76567083932805,43.76573882369269,43.76580680655919,43.76583852683017,43.76587035942168,43.76593831993177,43.76600625809487,43.76607418582612,43.76614211355053,43.7662100174409,43.76627801130774,43.76634589433773,43.76641377587313,43.76648163506631,43.76654949425463,43.76661733110194,43.76668522452918,43.76675318410863,43.76682104476628,43.76688890393005,43.76691603419328,43.76695674075427,43.76702906478995,43.76702907521268,43.76709687927155,43.76716478373195,43.76723257438339,43.76727775921579,43.7673048552256,43.76737262353732,43.76744038142413,43.76744489097362,43.76751262801366,43.76758045354511,43.76760753763989]}]],[[{"lng":[-79.32498334444976,-79.32475472809128,-79.32452882034326,-79.32435922057459,-79.3242179749239,-79.32404270324649,-79.32371640631821,-79.32348186839069,-79.32303335308197,-79.32233781629607,-79.32198354790837,-79.32165915083083,-79.32136135811892,-79.32091283822565,-79.32056089507392,-79.32019937489332,-79.31989406602693,-79.31940923646002,-79.31918489098085,-79.31854373759762,-79.31819286624221,-79.31771123687879,-79.31737144806682,-79.31644567529281,-79.31551034361692,-79.31523339913841,-79.31482390661469,-79.31460394814916,-79.31448348754184,-79.3140071692717,-79.31490604240219,-79.31591621204895,-79.31620386152846,-79.31644591013946,-79.31657954296398,-79.31690784331207,-79.31731236293219,-79.31831235498549,-79.31896778915272,-79.31918172326934,-79.31943227089749,-79.31952239183336,-79.31964532253046,-79.31971314739985,-79.31978972190015,-79.3198521204083,-79.31993248302254,-79.31999751161715,-79.32006310744406,-79.32050688807448,-79.32159619420703,-79.3217636658093,-79.32386616507328,-79.32555450114373,-79.32642989005754,-79.32651082541567,-79.32655825890248,-79.32663334819671,-79.32682548051734,-79.32706065962316,-79.32745560968019,-79.32783202758047,-79.32800044249407,-79.32812181371973,-79.32826468474852,-79.3284091705838,-79.3285604807371,-79.32869372826876,-79.3287987496623,-79.32927930680241,-79.33014878508314,-79.33087688422172,-79.33136746710444,-79.33157131745953,-79.3317928342538,-79.3320319152035,-79.33250215916604,-79.33273948745403,-79.33287752916129,-79.33325177284225,-79.33369316192922,-79.33392831398075,-79.33393638211835,-79.3339595278037,-79.33398400202566,-79.33400858121728,-79.33403387716798,-79.33406080881581,-79.3340877430625,-79.33411621313492,-79.33414668604856,-79.33417869232048,-79.33421092843216,-79.33424457558122,-79.33427834519787,-79.33431342598277,-79.33434965556521,-79.33438637477356,-79.33442403552314,-79.33446251799056,-79.33450159498463,-79.3345415086481,-79.33458296073529,-79.33462461775905,-79.33466665957724,-79.33471056941343,-79.33475509373933,-79.33479982301667,-79.33484598338016,-79.33489439891856,-79.33515241504053,-79.33541653770165,-79.33555765259371,-79.33567547860588,-79.33585416833373,-79.33587105728418,-79.33582184262728,-79.33573902137218,-79.33565617772,-79.33558656481001,-79.3354912069814,-79.33540907989544,-79.33532777188165,-79.33524646381161,-79.33516574764292,-79.33510391618077,-79.33504195989414,-79.33498082269392,-79.33491958305912,-79.33485854816368,-79.33479812507714,-79.33473852357523,-79.33467892203103,-79.33464033312853,-79.33461919562208,-79.33456041312103,-79.3345021200961,-79.33444403180417,-79.33438666020369,-79.33432918616204,-79.33427253120465,-79.33421587619836,-79.33416167854676,-79.33411493298459,-79.33406675388765,-79.33405175971913,-79.33401869961018,-79.33397064529088,-79.33392095015944,-79.33387113260206,-79.33382072313285,-79.3337704385078,-79.33371931220938,-79.33366808347431,-79.33361613794867,-79.33356329335849,-79.3335108583,-79.33345727436797,-79.33340391769815,-79.33334963945791,-79.33329546357146,-79.33318401736474,-79.33268911435356,-79.33063258832381,-79.32975182827116,-79.32968774086262,-79.32767411844334,-79.32598936571075,-79.32498334444976],"lat":[43.76128813356452,43.76129828483733,43.76129947348664,43.76128348367314,43.76124990568387,43.76116190717595,43.76103109443069,43.76091954085078,43.76066090650189,43.76029053119041,43.76010529149676,43.75990689661761,43.75967297413951,43.75936023004256,43.75908491122369,43.75880053865463,43.75855740992448,43.75818111279621,43.75804278463598,43.75775870401343,43.7576725232608,43.75762492487553,43.75759291608304,43.75770980699681,43.75781313667134,43.75786315494278,43.75670910591119,43.75605754581004,43.75570019437683,43.75433352185479,43.75414423876284,43.75391599043892,43.75387071916667,43.75385178098047,43.75384924887037,43.7538810907874,43.75396356460827,43.75420792454136,43.75433909449668,43.75435124351926,43.7543324240246,43.75432924850767,43.75431755352504,43.75430954902912,43.7542926712605,43.75428458688994,43.75426776472558,43.7542462161087,43.75423367768592,43.754132178625,43.75388253843083,43.75383098688589,43.75335306032937,43.75296813745381,43.75275582463682,43.75294613982769,43.75300084587933,43.75303345218371,43.75310377992165,43.75316573568878,43.7532795401256,43.75339316201646,43.7532560070975,43.75316776405715,43.75307983543559,43.7529964312041,43.75294013229822,43.75289248087612,43.75285800934027,43.75274801210738,43.75252667245363,43.75236168338398,43.75225182379349,43.75223229570902,43.7522220270772,43.75222101630231,43.75220537485787,43.75218633323011,43.752161341628,43.75207669217625,43.75197510689952,43.75192002347159,43.75196515059935,43.75208701369424,43.75220898617142,43.75233087015256,43.75245276457634,43.75257477285885,43.75269669115169,43.75281863182676,43.7529496035879,43.75308068771134,43.75320718420685,43.75333820220882,43.75346481105465,43.75359584993024,43.75372240459082,43.75385346732227,43.75398013283503,43.7541067202914,43.75423772732645,43.754364425631,43.75449105632275,43.75461768998398,43.75474892020168,43.75487558665978,43.75500226205389,43.75512894041511,43.75525572963014,43.75538246166826,43.75604353945495,43.7567136174269,43.75714335290105,43.75754124234778,43.75817865816736,43.75824191735961,43.7582547034327,43.75827150095704,43.75829279903949,43.75830978884017,43.75833540546192,43.75835671380295,43.7583780340174,43.75839935417301,43.75842518383892,43.7584422867195,43.75846388869206,43.75848550256622,43.75850711491483,43.75852873021407,43.75855044441539,43.75857208053927,43.75859371663125,43.75861115799927,43.75861985181633,43.75864149978351,43.75866765579893,43.75869381476866,43.75871998415567,43.75874615202039,43.75877233179643,43.75879851154345,43.75882472708734,43.75885555219459,43.75888635638222,43.75889513966065,43.75891266142522,43.75893896644723,43.75896533754332,43.75899611775746,43.75902238837472,43.75904415984627,43.75907041996554,43.7590966785679,43.75912292669378,43.75914466074023,43.7591664007349,43.7591926248928,43.75921435139647,43.75923606443319,43.75925777893695,43.75930566367109,43.75951007765864,43.75997974574385,43.76018301001005,43.76020007697259,43.7606613169333,43.76104632288984,43.76128813356452]}]],[[{"lng":[-79.34639409704555,-79.34638151528704,-79.34634385037054,-79.34627273951706,-79.3462161449585,-79.34615623097207,-79.34607188305031,-79.34598706853464,-79.34586835468122,-79.34576064375422,-79.345586796381,-79.34549600465988,-79.34542194414756,-79.34535684984952,-79.34528617533337,-79.34527581763069,-79.34529338297892,-79.34531377186144,-79.34535078756393,-79.3453851279592,-79.34544199270724,-79.34549439594402,-79.34552720089428,-79.34556905326022,-79.34559595526385,-79.34560807337516,-79.34562018593715,-79.34565078689488,-79.3456512524447,-79.3456380657245,-79.34560313023071,-79.34555195864593,-79.34549184186322,-79.34542498666542,-79.34532745581168,-79.34523643985726,-79.3451402453282,-79.34498245208195,-79.34478206021546,-79.34465637542303,-79.34452316571809,-79.34440164958235,-79.34436108316314,-79.344095423425,-79.34395854299895,-79.34382374103838,-79.34369954340939,-79.34361675294525,-79.34359726529672,-79.3436220173914,-79.34366269838702,-79.34370655350561,-79.34376138343256,-79.34381304200976,-79.3438463951436,-79.34384438122343,-79.3438508437368,-79.34385457650362,-79.34385065775643,-79.34388300935721,-79.33598904319395,-79.33587142693519,-79.33587105728418,-79.33585416833373,-79.33567547860588,-79.33555765259371,-79.33541653770165,-79.33515241504053,-79.33489439891856,-79.33484598338016,-79.33479982301667,-79.33475509373933,-79.33471056941343,-79.33466665957724,-79.33462461775905,-79.33458296073529,-79.3345415086481,-79.33450159498463,-79.33446251799056,-79.33442403552314,-79.33438637477356,-79.33434965556521,-79.33431342598277,-79.33427834519787,-79.33424457558122,-79.33421092843216,-79.33417869232048,-79.33414668604856,-79.33411621313492,-79.3340877430625,-79.33406080881581,-79.33403387716798,-79.33400858121728,-79.33398400202566,-79.3339595278037,-79.33393638211835,-79.33392831398075,-79.33391415793874,-79.33389275287951,-79.33387206705312,-79.33385219784059,-79.33383386438727,-79.33381555596966,-79.33379793930187,-79.33378196075307,-79.33376669890924,-79.33375156447013,-79.33373783840287,-79.33372493140457,-79.33371286844444,-79.33367047474681,-79.33362909813231,-79.33350555867945,-79.3335006175469,-79.33349590365576,-79.3334903483066,-79.33348399647534,-79.33347680568924,-79.33346889581574,-79.33346018947051,-79.33345064168152,-79.33344037731115,-79.33342929399045,-79.33341762145314,-79.33340428601313,-79.33339084823298,-79.33337659151488,-79.3333617206041,-79.33334523678587,-79.33332872801608,-79.33331150269116,-79.33329263947938,-79.33327377631295,-79.33325419660024,-79.33323308388057,-79.33321186634471,-79.3331899322714,-79.33316646270573,-79.33314299319731,-79.33305002262991,-79.3328497356906,-79.33255755042374,-79.33253539077241,-79.33251509629375,-79.33249611006076,-79.33247804765384,-79.33246049461368,-79.33244386289557,-79.33242827741428,-79.33241307887319,-79.33239890400932,-79.33238547078099,-79.33237272907714,-79.3323609312345,-79.33235054395415,-79.33234005433246,-79.33233051106329,-79.33232247820879,-79.33231446793113,-79.33230797055816,-79.33230159812193,-79.33229663621903,-79.33229549810767,-79.33228745403954,-79.33228261708773,-79.3322827420116,-79.33227558374109,-79.33226948378861,-79.33245552160164,-79.33264660739361,-79.33274734195986,-79.33283319710786,-79.33287688105703,-79.33293699806144,-79.33297471005814,-79.3330422868806,-79.33311575797489,-79.33320310254138,-79.33330389114592,-79.33341694043854,-79.33354670937703,-79.33369119570976,-79.33383394176887,-79.33399718231792,-79.33415994599305,-79.3343198882722,-79.33444383455206,-79.33462252144523,-79.33477745106582,-79.33493275491355,-79.33509960319087,-79.33525735438498,-79.33537229646838,-79.33549437237352,-79.33562045370201,-79.33566226566288,-79.33578957979132,-79.33585330549889,-79.33591023756821,-79.33599518817796,-79.33605704982098,-79.33614163367706,-79.33623520600466,-79.33630096720049,-79.33638456202182,-79.3364744234962,-79.33655019223492,-79.33660625661415,-79.33665009086086,-79.33663825020699,-79.33658172334785,-79.33649007195766,-79.33637176103365,-79.33629865929679,-79.3362849334887,-79.33623934716974,-79.33622816246464,-79.33622215381031,-79.33624218145725,-79.33626293468814,-79.33630966652508,-79.33638358320303,-79.33645491868408,-79.33653794632926,-79.33665139915517,-79.33678700533795,-79.33693764867941,-79.33708948493474,-79.3372228152657,-79.33736549184742,-79.33754660353873,-79.33767569064649,-79.33780021781473,-79.33796990774633,-79.33811573773789,-79.33827157721561,-79.33840367336377,-79.33860072760069,-79.33879342429076,-79.33896426493557,-79.33906966719572,-79.33908733703109,-79.33919708069145,-79.33922143490557,-79.33928963128176,-79.33939232917432,-79.33950620987534,-79.3396005388993,-79.33970477579767,-79.33978826219459,-79.33988128082439,-79.33996748759608,-79.34004861263797,-79.34010694637072,-79.34017013306809,-79.34023087312585,-79.34026069816782,-79.34024703875269,-79.34019918827194,-79.34015783084872,-79.34009637707472,-79.34003728008361,-79.3399961922474,-79.33997949090893,-79.33995748073794,-79.33993858986564,-79.33989712850328,-79.33986315524623,-79.33982340517456,-79.33975401586696,-79.33966174188004,-79.33955958167816,-79.33941817696791,-79.33932986596361,-79.33924729917408,-79.3391939985473,-79.3391810319628,-79.33918791119963,-79.33922246688518,-79.33926636049715,-79.33932913216343,-79.33941267365994,-79.33950887308177,-79.33959920112079,-79.33969057494264,-79.3398116509107,-79.3399338167205,-79.34005174896043,-79.34013004135743,-79.34031080041932,-79.34043224987526,-79.3405340000827,-79.34065009702471,-79.3407545017786,-79.340845598318,-79.34096496191925,-79.34105476456892,-79.34116918061059,-79.34127164830313,-79.34137045256338,-79.34146948352276,-79.34159165074193,-79.34170708330308,-79.34180895899429,-79.34192109387857,-79.34206055549639,-79.34213705799949,-79.34220523976805,-79.34225889727884,-79.34228591854954,-79.34231264784472,-79.34233932524729,-79.34237407676393,-79.34241024406396,-79.34243728924368,-79.34243232660377,-79.34241453021764,-79.34239814271984,-79.34232494797858,-79.34227811374753,-79.34223876348013,-79.34219772497416,-79.34217375688293,-79.34218294340685,-79.34222329587763,-79.34225324977601,-79.34228701100609,-79.34233760137927,-79.34240493796071,-79.34246922526657,-79.34253502659077,-79.34257642300042,-79.3425921035779,-79.34258912308023,-79.34258797690141,-79.34258574151048,-79.34255245473393,-79.34251911997606,-79.34246859227312,-79.34245513200428,-79.34247021093137,-79.3425146817266,-79.3425970393934,-79.34269971179373,-79.34280062872664,-79.34290238657708,-79.34300508770225,-79.34311426053567,-79.34324126910876,-79.34337570787974,-79.34349644971596,-79.34362522064549,-79.3436624865728,-79.34384757217455,-79.34398828421563,-79.34412037470351,-79.34425518603835,-79.34436237677383,-79.34446438355285,-79.34455246684259,-79.34466604846331,-79.3448040553408,-79.34484132129533,-79.34500993373409,-79.34510369132097,-79.34518800843469,-79.34528540601353,-79.3453902149901,-79.34547757640658,-79.34554722631304,-79.34562887072784,-79.34574544828079,-79.34586334086474,-79.34599503327236,-79.34612989962093,-79.34624508768735,-79.34634313541581,-79.34638753373376,-79.34639409704555],"lat":[43.75153830333009,43.75157862980738,43.75161859328449,43.75169408030946,43.75177427833507,43.75186343018805,43.75196582121608,43.7520591135066,43.75219692450765,43.75230338802257,43.75247199720827,43.75255169973393,43.75262264255598,43.75270271706442,43.75281430739181,43.75288167164122,43.75297644594612,43.75308026303306,43.75317540900346,43.75325242240558,43.75332976194787,43.75339803496307,43.75347502608782,43.75356565104472,43.75365614954622,43.75375534587275,43.75387704685986,43.75400801741772,43.75410263409792,43.75418346025341,43.75427297341669,43.75435324959146,43.75441989337607,43.75448202862665,43.75455713245687,43.75461882770491,43.75467144598461,43.754736674854,43.75482379118323,43.75487607188203,43.75489664676756,43.75491288997819,43.7549168031526,43.75494896119167,43.754987486014,43.75506204837892,43.75514576619037,43.75522108221019,43.75533341342524,43.75542379117888,43.75552340171865,43.75562305825176,43.75571846291442,43.75581373156459,43.75590423392782,43.75600322557359,43.75610684102099,43.7562240097615,43.756250958645,43.75644460817487,43.75821662994846,43.75824336304536,43.75824191735961,43.75817865816736,43.75754124234778,43.75714335290105,43.7567136174269,43.75604353945495,43.75538246166826,43.75525572963014,43.75512894041511,43.75500226205389,43.75487558665978,43.75474892020168,43.75461768998398,43.75449105632275,43.754364425631,43.75423772732645,43.7541067202914,43.75398013283503,43.75385346732227,43.75372240459082,43.75359584993024,43.75346481105465,43.75333820220882,43.75320718420685,43.75308068771134,43.7529496035879,43.75281863182676,43.75269669115169,43.75257477285885,43.75245276457634,43.75233087015256,43.75220898617142,43.75208701369424,43.75196515059935,43.75192002347159,43.75184330093378,43.75172146320518,43.7515995459385,43.75147773059211,43.75135593763357,43.75122955406641,43.75110777154933,43.75098601291415,43.75086426472544,43.75073792742141,43.75061620161907,43.75049448775678,43.75036819522981,43.75002100385181,43.74947110445518,43.74782609549,43.74774950724466,43.74766842136034,43.74759182415468,43.74751071437949,43.74743400330032,43.74735737175124,43.74727622763217,43.74719957219151,43.74712290629797,43.74704622845827,43.74696495105132,43.74688824036014,43.74681152817385,43.74673480404059,43.74665807094635,43.74657672335218,43.74649996636258,43.74642319891809,43.74634640757976,43.74626961623811,43.74619281444061,43.74611590025846,43.74603907456166,43.74596223840793,43.74588537985063,43.74580852128846,43.74544258782514,43.74471483222403,43.74340952297568,43.74333718432951,43.74326037194135,43.74318807959508,43.74311571070732,43.74304343926838,43.74297118127119,43.74289443758381,43.74282220049573,43.74274997834421,43.74267317603892,43.74260097479799,43.74252428637406,43.74245211948983,43.74237995111023,43.74230320556545,43.74223107303767,43.74215443988211,43.74208232976131,43.74200572050673,43.74193363279326,43.74191561235871,43.74178487721185,43.74170829036269,43.74170379122977,43.74160016476253,43.74151005660825,43.74147055269768,43.7414299518932,43.74140891694918,43.74133365334835,43.74129828303678,43.74123164575668,43.74118268541148,43.74109815307688,43.74102261857826,43.74095637833381,43.74090383698608,43.74085597532702,43.74082186026262,43.74079246062443,43.74076303542779,43.74072940790274,43.74070927606318,43.74068010094977,43.74066390381554,43.74065300532435,43.74066426512807,43.7406620273031,43.74066445842874,43.7406847606067,43.74070893989929,43.7407602287359,43.740829579612,43.74086169532033,43.74097157257903,43.74108502459579,43.74118496478759,43.74131222869004,43.74143015439957,43.74154850095412,43.74167138925624,43.7417668668235,43.74185810303491,43.74194492945868,43.7420451435454,43.74217648742131,43.74227614654518,43.74239308906481,43.74250028915216,43.74258897404757,43.7426862726453,43.74277072649161,43.74279303142642,43.74292298542544,43.74299033690862,43.74306676566643,43.7431570763862,43.74322489288824,43.7432975885825,43.74337968195395,43.74346623865422,43.74354846456426,43.74362663234975,43.7436962105845,43.74374791367155,43.74378613105451,43.74381957799079,43.74382615503219,43.74376127580243,43.7437001404159,43.7436298466213,43.74355579865798,43.743503908207,43.74345666408724,43.74344958309855,43.74345244834781,43.74344624799654,43.74347123630406,43.74350364497643,43.7435090329033,43.74356220895708,43.74357399535371,43.74363349892926,43.74373410248471,43.74383027747705,43.74392616825498,43.74402670386998,43.74409993217184,43.74419139273991,43.7442736623422,43.74436936092223,43.74447823123562,43.74457826012027,43.74465565863705,43.74473260808319,43.7448044249556,43.74487574518552,43.74493815780313,43.74502737422416,43.74511653482853,43.74522846179999,43.74527322871652,43.74533592232266,43.74543025793983,43.74555568230547,43.74563620590367,43.74572564746815,43.74584625507494,43.74595743819086,43.74605947566404,43.74614743987066,43.74622267267426,43.74631608272795,43.74639182425734,43.74646815200411,43.74652676438497,43.7465992818825,43.74666743413128,43.74672694883589,43.74676867151304,43.74679707521067,43.74681189065136,43.74682222026266,43.74685098516061,43.74686626290367,43.74686797615572,43.74687361442882,43.74690324575228,43.74691851255753,43.74694249505579,43.74698468961589,43.74703121523222,43.74707754751893,43.74714238397524,43.74720220016821,43.74725337170262,43.74727736395936,43.74729680199509,43.7473117422827,43.74732701827345,43.74733769544444,43.74735717767477,43.7473723077282,43.74741033893284,43.74744745653196,43.74751145908036,43.74759334470731,43.74767925482347,43.74778316450775,43.74785556676205,43.74797309566463,43.74807273130094,43.74821715410513,43.74831160211893,43.74839236110504,43.74847773149654,43.74860269634474,43.74869653690502,43.74883090457016,43.74891141633974,43.74901909145112,43.74910924376988,43.74919084634114,43.749263296135,43.74933139022209,43.74941314127523,43.74950413713597,43.7495905877723,43.74968156128412,43.74975876793442,43.74984001250299,43.74994799210337,43.75001548974718,43.75009647444499,43.75020410446866,43.75034315042261,43.75045944221252,43.7505537669276,43.75065309659715,43.75073025783595,43.75078546384746,43.75083196254074,43.75086043187964,43.75088441237375,43.75090390550511,43.75091899143464,43.75092983491214,43.75094978787213,43.75096504116193,43.75097140881049,43.75097194900027,43.75095662793336,43.75094966539143,43.75094707868899,43.75095353316082,43.75095508632441,43.7509700671327,43.75098484613302,43.75097748969847,43.75097948886423,43.75098002866972,43.75100497565972,43.75101083451752,43.75102105753237,43.75103606087596,43.75106458428893,43.75110185687682,43.75113887295637,43.75119406645521,43.75124526468217,43.75127847802393,43.75131639194737,43.75135435165788,43.75139652736214,43.75144745679909,43.75149769979436,43.75153830333009]}]],[[{"lng":[-79.35995888979716,-79.35872661897722,-79.35748186482101,-79.35554644244142,-79.35521045018685,-79.35408871385768,-79.35263471904855,-79.35202370249405,-79.35087218509895,-79.34866885723075,-79.3470335880067,-79.34389066602247,-79.34388300935721,-79.34385065775643,-79.34385457650362,-79.3438508437368,-79.34384438122343,-79.3438463951436,-79.34381304200976,-79.34376138343256,-79.34370655350561,-79.34366269838702,-79.3436220173914,-79.34359726529672,-79.34361675294525,-79.34369954340939,-79.34382374103838,-79.34395854299895,-79.344095423425,-79.34436108316314,-79.34440164958235,-79.34452316571809,-79.34465637542303,-79.34478206021546,-79.34498245208195,-79.3451402453282,-79.34523643985726,-79.34532745581168,-79.34542498666542,-79.34549184186322,-79.34555195864593,-79.34560313023071,-79.3456380657245,-79.3456512524447,-79.34565078689488,-79.34562018593715,-79.34560807337516,-79.34559595526385,-79.34556905326022,-79.34552720089428,-79.34549439594402,-79.34544199270724,-79.3453851279592,-79.34535078756393,-79.34531377186144,-79.34529338297892,-79.34527581763069,-79.34528617533337,-79.34535684984952,-79.34542194414756,-79.34549600465988,-79.345586796381,-79.34576064375422,-79.34586835468122,-79.34598706853464,-79.34607188305031,-79.34615623097207,-79.3462161449585,-79.34627273951706,-79.34634385037054,-79.34638151528704,-79.34639409704555,-79.34638753373376,-79.34634313541581,-79.34624508768735,-79.34612989962093,-79.34599503327236,-79.34586334086474,-79.34574544828079,-79.34562887072784,-79.34554722631304,-79.34547757640658,-79.3453902149901,-79.34528540601353,-79.34518800843469,-79.34510369132097,-79.34500993373409,-79.34484132129533,-79.3448040553408,-79.34466604846331,-79.34455246684259,-79.34446438355285,-79.34436237677383,-79.34425518603835,-79.34412037470351,-79.34398828421563,-79.34384757217455,-79.3436624865728,-79.34362522064549,-79.34349644971596,-79.34337570787974,-79.34324126910876,-79.34311426053567,-79.34300508770225,-79.34290238657708,-79.34280062872664,-79.34269971179373,-79.3425970393934,-79.3425146817266,-79.34247021093137,-79.34245513200428,-79.34246859227312,-79.34251911997606,-79.34255245473393,-79.34258574151048,-79.34258797690141,-79.34258912308023,-79.3425921035779,-79.34257642300042,-79.34253502659077,-79.34246922526657,-79.34240493796071,-79.34233760137927,-79.34228701100609,-79.34225324977601,-79.34222329587763,-79.34218294340685,-79.34217375688293,-79.34219772497416,-79.34223876348013,-79.34227811374753,-79.34232494797858,-79.34239814271984,-79.34241453021764,-79.34243232660377,-79.34243728924368,-79.34241024406396,-79.34237407676393,-79.34233932524729,-79.34231264784472,-79.34228591854954,-79.34225889727884,-79.34220523976805,-79.34213705799949,-79.34206055549639,-79.34192109387857,-79.34180895899429,-79.34170708330308,-79.34159165074193,-79.34146948352276,-79.34137045256338,-79.34127164830313,-79.34116918061059,-79.34105476456892,-79.34096496191925,-79.340845598318,-79.3407545017786,-79.34065009702471,-79.3405340000827,-79.34043224987526,-79.34031080041932,-79.34013004135743,-79.34005174896043,-79.3399338167205,-79.3398116509107,-79.33969057494264,-79.33959920112079,-79.33950887308177,-79.33941267365994,-79.33932913216343,-79.33926636049715,-79.33922246688518,-79.33918791119963,-79.3391810319628,-79.3391939985473,-79.33924729917408,-79.33932986596361,-79.33941817696791,-79.33955958167816,-79.33966174188004,-79.33975401586696,-79.33982340517456,-79.33986315524623,-79.33989712850328,-79.33993858986564,-79.33995748073794,-79.33997949090893,-79.3399961922474,-79.34003728008361,-79.34009637707472,-79.34015783084872,-79.34019918827194,-79.34024703875269,-79.34026069816782,-79.34023087312585,-79.34017013306809,-79.34010694637072,-79.34004861263797,-79.33996748759608,-79.33988128082439,-79.33978826219459,-79.33970477579767,-79.3396005388993,-79.33950620987534,-79.33939232917432,-79.33928963128176,-79.33922143490557,-79.33919708069145,-79.33908733703109,-79.33906966719572,-79.33914917850315,-79.33920105464028,-79.3392460490824,-79.33947672786032,-79.34007866115087,-79.34156213690179,-79.34331482211245,-79.34429905353703,-79.34502751902427,-79.34559683735912,-79.34660115654141,-79.34733197529467,-79.34815344893465,-79.34882636409647,-79.35264419844864,-79.35313859080004,-79.35777832006637,-79.35798410989074,-79.35801529203563,-79.35831449861928,-79.35850825523356,-79.35871351664663,-79.35903623834317,-79.35922264227895,-79.35950448069235,-79.3595869492723,-79.35972037938306,-79.35990706868408,-79.35995888979716],"lat":[43.75294923715237,43.75321960490621,43.75348536896519,43.75391221794577,43.75398839915406,43.75423789304794,43.75455010048659,43.75468982120807,43.75495235677507,43.7554337344742,43.7558062638701,43.75649008868957,43.75644460817487,43.756250958645,43.7562240097615,43.75610684102099,43.75600322557359,43.75590423392782,43.75581373156459,43.75571846291442,43.75562305825176,43.75552340171865,43.75542379117888,43.75533341342524,43.75522108221019,43.75514576619037,43.75506204837892,43.754987486014,43.75494896119167,43.7549168031526,43.75491288997819,43.75489664676756,43.75487607188203,43.75482379118323,43.754736674854,43.75467144598461,43.75461882770491,43.75455713245687,43.75448202862665,43.75441989337607,43.75435324959146,43.75427297341669,43.75418346025341,43.75410263409792,43.75400801741772,43.75387704685986,43.75375534587275,43.75365614954622,43.75356565104472,43.75347502608782,43.75339803496307,43.75332976194787,43.75325242240558,43.75317540900346,43.75308026303306,43.75297644594612,43.75288167164122,43.75281430739181,43.75270271706442,43.75262264255598,43.75255169973393,43.75247199720827,43.75230338802257,43.75219692450765,43.7520591135066,43.75196582121608,43.75186343018805,43.75177427833507,43.75169408030946,43.75161859328449,43.75157862980738,43.75153830333009,43.75149769979436,43.75144745679909,43.75139652736214,43.75135435165788,43.75131639194737,43.75127847802393,43.75124526468217,43.75119406645521,43.75113887295637,43.75110185687682,43.75106458428893,43.75103606087596,43.75102105753237,43.75101083451752,43.75100497565972,43.75098002866972,43.75097948886423,43.75097748969847,43.75098484613302,43.7509700671327,43.75095508632441,43.75095353316082,43.75094707868899,43.75094966539143,43.75095662793336,43.75097194900027,43.75097140881049,43.75096504116193,43.75094978787213,43.75092983491214,43.75091899143464,43.75090390550511,43.75088441237375,43.75086043187964,43.75083196254074,43.75078546384746,43.75073025783595,43.75065309659715,43.7505537669276,43.75045944221252,43.75034315042261,43.75020410446866,43.75009647444499,43.75001548974718,43.74994799210337,43.74984001250299,43.74975876793442,43.74968156128412,43.7495905877723,43.74950413713597,43.74941314127523,43.74933139022209,43.749263296135,43.74919084634114,43.74910924376988,43.74901909145112,43.74891141633974,43.74883090457016,43.74869653690502,43.74860269634474,43.74847773149654,43.74839236110504,43.74831160211893,43.74821715410513,43.74807273130094,43.74797309566463,43.74785556676205,43.74778316450775,43.74767925482347,43.74759334470731,43.74751145908036,43.74744745653196,43.74741033893284,43.7473723077282,43.74735717767477,43.74733769544444,43.74732701827345,43.7473117422827,43.74729680199509,43.74727736395936,43.74725337170262,43.74720220016821,43.74714238397524,43.74707754751893,43.74703121523222,43.74698468961589,43.74694249505579,43.74691851255753,43.74690324575228,43.74687361442882,43.74686797615572,43.74686626290367,43.74685098516061,43.74682222026266,43.74681189065136,43.74679707521067,43.74676867151304,43.74672694883589,43.74666743413128,43.7465992818825,43.74652676438497,43.74646815200411,43.74639182425734,43.74631608272795,43.74622267267426,43.74614743987066,43.74605947566404,43.74595743819086,43.74584625507494,43.74572564746815,43.74563620590367,43.74555568230547,43.74543025793983,43.74533592232266,43.74527322871652,43.74522846179999,43.74511653482853,43.74502737422416,43.74493815780313,43.74487574518552,43.7448044249556,43.74473260808319,43.74465565863705,43.74457826012027,43.74447823123562,43.74436936092223,43.7442736623422,43.74419139273991,43.74409993217184,43.74402670386998,43.74392616825498,43.74383027747705,43.74373410248471,43.74363349892926,43.74357399535371,43.74356220895708,43.7435090329033,43.74350364497643,43.74340208887314,43.74334072967902,43.74327269904758,43.74301499603344,43.74278195029849,43.74272544392644,43.74282531407212,43.74301448664134,43.74322452191285,43.74340524273239,43.74383251682698,43.74425168586068,43.74484166672788,43.74546433246446,43.74464400063529,43.74464914660254,43.74361984835132,43.74353548712541,43.74368347658842,43.74514185479771,43.74597749524198,43.74707435589811,43.74859158162408,43.74949462912991,43.75077721639914,43.75119257796943,43.75179330004792,43.7526603424854,43.75294923715237]}]],[[{"lng":[-79.37606768516649,-79.37543304977802,-79.37486374942821,-79.37364188474146,-79.37164852365525,-79.37044282450597,-79.36900361949525,-79.36548284297372,-79.36413775741131,-79.36348042340518,-79.36336825299952,-79.3632008574348,-79.36302946881682,-79.3628216510599,-79.36191443080125,-79.3608370527259,-79.36062049207035,-79.36036547631332,-79.35995888979716,-79.35990706868408,-79.35972037938306,-79.3595869492723,-79.35950448069235,-79.35922264227895,-79.35903623834317,-79.35871351664663,-79.35850825523356,-79.35831449861928,-79.35801529203563,-79.35798410989074,-79.35778636790864,-79.3574219541885,-79.35734667593341,-79.35710372796876,-79.35687209146752,-79.35667526339822,-79.35635485489763,-79.35622768686501,-79.35596805108175,-79.35721870579913,-79.3583450386964,-79.35901585255002,-79.36190222725826,-79.36201668525992,-79.36221873230015,-79.36247739740567,-79.36275503853572,-79.36290642259452,-79.36306466238513,-79.36329444338534,-79.36345608584563,-79.36344125403338,-79.36353186728942,-79.36364166432161,-79.36351518365778,-79.36351422869309,-79.36364962847733,-79.36387298761262,-79.36411761773154,-79.3643155729481,-79.36434756256023,-79.36446790091523,-79.36479259344905,-79.36503374992074,-79.36527659290024,-79.36558313239068,-79.36581706464924,-79.36602880937264,-79.36626448814664,-79.36656436041021,-79.3666633885491,-79.3666910570296,-79.36676065633888,-79.36693912724004,-79.36716000161169,-79.36740648286319,-79.36770365132749,-79.36795337848712,-79.36818526015722,-79.3683744411528,-79.36854651234496,-79.3686535025257,-79.36869089916364,-79.36869454440577,-79.36873044219824,-79.36885745877143,-79.36897841307982,-79.36910505272169,-79.36936338794249,-79.36960148948715,-79.36965479241168,-79.36995181990105,-79.37015698951274,-79.37025039708826,-79.3703015252941,-79.37050563372297,-79.3707907451243,-79.37099727866645,-79.37088271505509,-79.37061585871692,-79.37074215248491,-79.37103451179965,-79.37102288113508,-79.37089332786037,-79.37064857287695,-79.37063664138377,-79.37062619651724,-79.37085122521843,-79.371108958132,-79.3712551402766,-79.37134770138776,-79.37156304909367,-79.37166467975486,-79.37188109068398,-79.37209034664434,-79.37210510649372,-79.37222773738655,-79.37235000253212,-79.37242099878263,-79.37247648764709,-79.37252066565226,-79.37262361445013,-79.37287020576657,-79.37299894827692,-79.37310660463547,-79.37310707983498,-79.37328350082122,-79.37362528831643,-79.37380929893627,-79.37409247814371,-79.37428975160762,-79.3745840354907,-79.37485210334489,-79.37501537817357,-79.3752169990816,-79.37530490204685,-79.37559141550116,-79.37583914628905,-79.37610956856453,-79.3763003222524,-79.37656877288686,-79.37681734302734,-79.37708730660614,-79.37742012180314,-79.377784537895,-79.37807555609385,-79.37820199728175,-79.37822901786092,-79.3783644286092,-79.37846555455315,-79.37866459495881,-79.37877927372786,-79.37882296976811,-79.37869774761157,-79.37849810512932,-79.37845102398795,-79.37839953943792,-79.37839201951398,-79.37868578997856,-79.37901038362206,-79.37927683554813,-79.37951180966347,-79.37972175330378,-79.37990583353491,-79.37996248663143,-79.38016975823093,-79.38041886085387,-79.38070750148314,-79.38102871175498,-79.38123700004881,-79.38134958645247,-79.38141461709894,-79.38152486129921,-79.38179518238002,-79.38203935286809,-79.38230286778796,-79.38265944149121,-79.38251149369029,-79.38234808652948,-79.38210529623395,-79.3818893145413,-79.38078369927845,-79.38064769205042,-79.37998452895447,-79.37930251602299,-79.37874514500224,-79.37734778335718,-79.37626374915425,-79.37606768516649],"lat":[43.75028447298505,43.75041498161067,43.75054191497333,43.75079917908513,43.75122097581056,43.751451430275,43.751763967364,43.75248800308032,43.75278391121953,43.75289602453591,43.75290792084293,43.75292802789181,43.75294807749138,43.75294059918825,43.75292768726551,43.75288973256973,43.75289112674602,43.75290547157255,43.75294923715237,43.7526603424854,43.75179330004792,43.75119257796943,43.75077721639914,43.74949462912991,43.74859158162408,43.74707435589811,43.74597749524198,43.74514185479771,43.74368347658842,43.74353548712541,43.74259518594062,43.74085221801455,43.74048196718805,43.73939806402472,43.73830523087867,43.73750113926976,43.73612796840205,43.73555892780831,43.73452870291459,43.73454218863731,43.73470249649694,43.73477964622213,43.73416986031199,43.73433191542765,43.7344698403941,43.73451405611493,43.73454954120858,43.73466423423089,43.73477911528013,43.73478240669829,43.73491524977936,43.73511767077159,43.73529450608742,43.73550762375405,43.73568144007447,43.73580295241888,43.73589941166878,43.73596571347655,43.73601422549044,43.73603956405671,43.73623356339062,43.73640191151042,43.73642456219577,43.73651803217241,43.7366925431048,43.73674643859082,43.73662375734698,43.73654126702827,43.73669316860388,43.73679647673082,43.73669437009482,43.73661824919845,43.73659673917388,43.73663079663346,43.73672847322734,43.73679950946644,43.73688936310303,43.73703696090369,43.73721581005963,43.73742114483582,43.73763064595808,43.73783480677561,43.73794336375927,43.73795241772385,43.73802944655717,43.73810327512762,43.73801948333703,43.73789976485808,43.73793045738861,43.73802378410166,43.73826318565231,43.73841145391359,43.73854490836928,43.73864985276901,43.73870909447407,43.73885153508935,43.73896362353933,43.73916019941264,43.73937461233249,43.73948792252933,43.73961124926539,43.73970543659289,43.73993040888747,43.74013110521815,43.74019963093258,43.74035258350106,43.74056397973577,43.74065270647135,43.74056636148492,43.74061345480445,43.74074980292054,43.74073486817169,43.74088043721888,43.74085651510033,43.74078298006999,43.74077868938061,43.74078043632952,43.7407956808199,43.74078318918401,43.74071187412294,43.74060448027021,43.74047091763494,43.74047442921776,43.7406247940831,43.74077485859148,43.7409009822843,43.7410340217825,43.7411109022669,43.74131165340285,43.74140120128869,43.74156604265332,43.74159723481748,43.74158304377374,43.74177449612937,43.7419529005921,43.74217478751707,43.74222837140704,43.7422543975878,43.74238426778938,43.74256710687276,43.74265634930845,43.7428219150527,43.7430598894728,43.74320414556326,43.74330834067042,43.74348359820891,43.74374644848323,43.7440124785959,43.74428454790831,43.74450652996968,43.7446939836112,43.74490715576239,43.74512841269079,43.74525266288197,43.74525433104097,43.74542019839471,43.74565810858316,43.74591005551135,43.74606734666727,43.74617547376644,43.74623326477359,43.74632661642608,43.74645571035663,43.74662485581117,43.74688230361149,43.74708778508116,43.74723084566875,43.74732054454324,43.74738360838207,43.74749458221414,43.74772131495573,43.74796078682434,43.74816048046226,43.74829933760869,43.74842432097378,43.74852698287542,43.74880217772306,43.7488765998475,43.74895080292797,43.74903297376775,43.749070424141,43.74930690613238,43.74934098643234,43.74946670517735,43.74961906842661,43.74973718846163,43.75002800982404,43.75023774870395,43.75028447298505]}]],[[{"lng":[-79.35798410989074,-79.35777832006637,-79.35313859080004,-79.35264419844864,-79.34882636409647,-79.34815344893465,-79.34733197529467,-79.34660115654141,-79.34559683735912,-79.34502751902427,-79.34429905353703,-79.34331482211245,-79.34156213690179,-79.34007866115087,-79.33947672786032,-79.3392460490824,-79.33920105464028,-79.33914917850315,-79.33906966719572,-79.33896426493557,-79.33879342429076,-79.33860072760069,-79.33840367336377,-79.33827157721561,-79.33811573773789,-79.33796990774633,-79.33780021781473,-79.33767569064649,-79.33754660353873,-79.33736549184742,-79.3372228152657,-79.33708948493474,-79.33693764867941,-79.33678700533795,-79.33665139915517,-79.33653794632926,-79.33645491868408,-79.33638358320303,-79.33630966652508,-79.33626293468814,-79.33624218145725,-79.33622215381031,-79.33622816246464,-79.33623934716974,-79.3362849334887,-79.33629865929679,-79.33637176103365,-79.33649007195766,-79.33658172334785,-79.33663825020699,-79.33665009086086,-79.33660625661415,-79.33655019223492,-79.3364744234962,-79.33638456202182,-79.33630096720049,-79.33623520600466,-79.33614163367706,-79.33605704982098,-79.33599518817796,-79.33591023756821,-79.33585330549889,-79.33578957979132,-79.33566226566288,-79.33562045370201,-79.33549437237352,-79.33537229646838,-79.33525735438498,-79.33509960319087,-79.33493275491355,-79.33477745106582,-79.33462252144523,-79.33444383455206,-79.3343198882722,-79.33415994599305,-79.33399718231792,-79.33383394176887,-79.33369119570976,-79.33354670937703,-79.33341694043854,-79.33330389114592,-79.33320310254138,-79.33311575797489,-79.3330422868806,-79.33297471005814,-79.33293699806144,-79.33287688105703,-79.33283319710786,-79.33274734195986,-79.33264660739361,-79.33245552160164,-79.33226948378861,-79.33225543760361,-79.33208501659011,-79.33191070573078,-79.33158886816148,-79.33124883160355,-79.33089424752013,-79.33064502993223,-79.33052244553214,-79.33049712705333,-79.33052254538136,-79.33067130707444,-79.33078733701866,-79.33079817341451,-79.33077171964698,-79.33071272413856,-79.33063942186797,-79.33054313439344,-79.33207631809047,-79.33227223105025,-79.33470590696102,-79.33578867384193,-79.33584329733985,-79.33606397616427,-79.33707115295671,-79.33820560344198,-79.34096568556355,-79.34343659904651,-79.34489901953653,-79.34653451438436,-79.34822559008536,-79.34994131099444,-79.35105397071382,-79.35213870434626,-79.35331419247464,-79.35420087206747,-79.35436339046467,-79.35506668498175,-79.35596805108175,-79.35622768686501,-79.35635485489763,-79.35667526339822,-79.35687209146752,-79.35710372796876,-79.35734667593341,-79.3574219541885,-79.35778636790864,-79.35798410989074],"lat":[43.74353548712541,43.74361984835132,43.74464914660254,43.74464400063529,43.74546433246446,43.74484166672788,43.74425168586068,43.74383251682698,43.74340524273239,43.74322452191285,43.74301448664134,43.74282531407212,43.74272544392644,43.74278195029849,43.74301499603344,43.74327269904758,43.74334072967902,43.74340208887314,43.74350364497643,43.74347123630406,43.74344624799654,43.74345244834781,43.74344958309855,43.74345666408724,43.743503908207,43.74355579865798,43.7436298466213,43.7437001404159,43.74376127580243,43.74382615503219,43.74381957799079,43.74378613105451,43.74374791367155,43.7436962105845,43.74362663234975,43.74354846456426,43.74346623865422,43.74337968195395,43.7432975885825,43.74322489288824,43.7431570763862,43.74306676566643,43.74299033690862,43.74292298542544,43.74279303142642,43.74277072649161,43.7426862726453,43.74258897404757,43.74250028915216,43.74239308906481,43.74227614654518,43.74217648742131,43.7420451435454,43.74194492945868,43.74185810303491,43.7417668668235,43.74167138925624,43.74154850095412,43.74143015439957,43.74131222869004,43.74118496478759,43.74108502459579,43.74097157257903,43.74086169532033,43.740829579612,43.7407602287359,43.74070893989929,43.7406847606067,43.74066445842874,43.7406620273031,43.74066426512807,43.74065300532435,43.74066390381554,43.74068010094977,43.74070927606318,43.74072940790274,43.74076303542779,43.74079246062443,43.74082186026262,43.74085597532702,43.74090383698608,43.74095637833381,43.74102261857826,43.74109815307688,43.74118268541148,43.74123164575668,43.74129828303678,43.74133365334835,43.74140891694918,43.7414299518932,43.74147055269768,43.74151005660825,43.74151435256041,43.74152986895217,43.74154532830521,43.74156313441996,43.74153566429098,43.74144946822452,43.74125669745208,43.74100276295175,43.74090787295258,43.74080022136403,43.74061335452596,43.74038090976086,43.74024153838894,43.74012853800697,43.73999264746694,43.73989255554336,43.73978312584457,43.7394903583515,43.73945270922524,43.73893440459644,43.73873863417387,43.73872592682525,43.73868863186009,43.73850066041472,43.73828311437608,43.73768840525643,43.73716604391456,43.73685398422385,43.73649949067995,43.73614127614645,43.73578348242419,43.73556540383979,43.73531990590473,43.73508029648435,43.7348949381693,43.73485677009845,43.73470486163475,43.73452870291459,43.73555892780831,43.73612796840205,43.73750113926976,43.73830523087867,43.73939806402472,43.74048196718805,43.74085221801455,43.74259518594062,43.74353548712541]}]],[[{"lng":[-79.30978135790338,-79.31143106237415,-79.31280520100456,-79.31371193597877,-79.31418672341137,-79.31422838309322,-79.31606194683742,-79.31615757589692,-79.31619270952804,-79.31798921948389,-79.31926763907326,-79.31969303457912,-79.32213118333115,-79.32415450298033,-79.32469408587808,-79.32565913987516,-79.32743053912611,-79.3284316249289,-79.33054313439344,-79.33063942186797,-79.33071272413856,-79.33077171964698,-79.33079817341451,-79.33078733701866,-79.33067130707444,-79.33052254538136,-79.33049712705333,-79.33052244553214,-79.33064502993223,-79.33089424752013,-79.33124883160355,-79.33158886816148,-79.33191070573078,-79.33208501659011,-79.33225543760361,-79.33226948378861,-79.33227558374109,-79.3322827420116,-79.33228261708773,-79.33228745403954,-79.33229549810767,-79.33229663621903,-79.33230159812193,-79.33230797055816,-79.33231446793113,-79.33232247820879,-79.33233051106329,-79.33234005433246,-79.33235054395415,-79.3323609312345,-79.33237272907714,-79.33238547078099,-79.33239890400932,-79.33241307887319,-79.33242827741428,-79.33244386289557,-79.33246049461368,-79.33247804765384,-79.33249611006076,-79.33251509629375,-79.33253539077241,-79.33255755042374,-79.3328497356906,-79.33305002262991,-79.33314299319731,-79.33316646270573,-79.3331899322714,-79.33321186634471,-79.33323308388057,-79.33325419660024,-79.33327377631295,-79.33329263947938,-79.33331150269116,-79.33332872801608,-79.33334523678587,-79.3333617206041,-79.33337659151488,-79.33339084823298,-79.33340428601313,-79.33341762145314,-79.33342929399045,-79.33344037731115,-79.33345064168152,-79.33346018947051,-79.33346889581574,-79.33347680568924,-79.33348399647534,-79.3334903483066,-79.33349590365576,-79.3335006175469,-79.33350555867945,-79.33362909813231,-79.33367047474681,-79.33371286844444,-79.33372493140457,-79.33373783840287,-79.33375156447013,-79.33376669890924,-79.33378196075307,-79.33379793930187,-79.33381555596966,-79.33383386438727,-79.33385219784059,-79.33387206705312,-79.33389275287951,-79.33391415793874,-79.33392831398075,-79.33369316192922,-79.33325177284225,-79.33287752916129,-79.33273948745403,-79.33250215916604,-79.3320319152035,-79.3317928342538,-79.33157131745953,-79.33136746710444,-79.33087688422172,-79.33014878508314,-79.32927930680241,-79.3287987496623,-79.32869372826876,-79.3285604807371,-79.3284091705838,-79.32826468474852,-79.32812181371973,-79.32800044249407,-79.32783202758047,-79.32745560968019,-79.32706065962316,-79.32682548051734,-79.32663334819671,-79.32655825890248,-79.32651082541567,-79.32642989005754,-79.32555450114373,-79.32386616507328,-79.3217636658093,-79.32159619420703,-79.32050688807448,-79.32006310744406,-79.31999751161715,-79.31993248302254,-79.3198521204083,-79.31978972190015,-79.31971314739985,-79.31964532253046,-79.31952239183336,-79.31943227089749,-79.31918172326934,-79.31896778915272,-79.31831235498549,-79.31731236293219,-79.31690784331207,-79.31657954296398,-79.31644591013946,-79.31620386152846,-79.31591621204895,-79.31490604240219,-79.3140071692717,-79.31379297142637,-79.31352435367943,-79.31325090502736,-79.31301319168239,-79.3127409999295,-79.31242221840938,-79.312099727399,-79.31195888073846,-79.31180599150883,-79.31169918372653,-79.31140205096777,-79.31116290625614,-79.31111719694574,-79.31092698140831,-79.31077854199425,-79.3105653887675,-79.31029527980378,-79.31001739582902,-79.30978135790338],"lat":[43.74213869284663,43.74179391492066,43.74161608087756,43.74153035455436,43.74149595143527,43.74149296528126,43.74137146947747,43.74136837775494,43.74136439452483,43.74126032556281,43.74117552473478,43.74114577335246,43.74100156223199,43.74084212397982,43.74078252188134,43.74065713707189,43.74037233327141,43.74018443901329,43.73978312584457,43.73989255554336,43.73999264746694,43.74012853800697,43.74024153838894,43.74038090976086,43.74061335452596,43.74080022136403,43.74090787295258,43.74100276295175,43.74125669745208,43.74144946822452,43.74153566429098,43.74156313441996,43.74154532830521,43.74152986895217,43.74151435256041,43.74151005660825,43.74160016476253,43.74170379122977,43.74170829036269,43.74178487721185,43.74191561235871,43.74193363279326,43.74200572050673,43.74208232976131,43.74215443988211,43.74223107303767,43.74230320556545,43.74237995111023,43.74245211948983,43.74252428637406,43.74260097479799,43.74267317603892,43.74274997834421,43.74282220049573,43.74289443758381,43.74297118127119,43.74304343926838,43.74311571070732,43.74318807959508,43.74326037194135,43.74333718432951,43.74340952297568,43.74471483222403,43.74544258782514,43.74580852128846,43.74588537985063,43.74596223840793,43.74603907456166,43.74611590025846,43.74619281444061,43.74626961623811,43.74634640757976,43.74642319891809,43.74649996636258,43.74657672335218,43.74665807094635,43.74673480404059,43.74681152817385,43.74688824036014,43.74696495105132,43.74704622845827,43.74712290629797,43.74719957219151,43.74727622763217,43.74735737175124,43.74743400330032,43.74751071437949,43.74759182415468,43.74766842136034,43.74774950724466,43.74782609549,43.74947110445518,43.75002100385181,43.75036819522981,43.75049448775678,43.75061620161907,43.75073792742141,43.75086426472544,43.75098601291415,43.75110777154933,43.75122955406641,43.75135593763357,43.75147773059211,43.7515995459385,43.75172146320518,43.75184330093378,43.75192002347159,43.75197510689952,43.75207669217625,43.752161341628,43.75218633323011,43.75220537485787,43.75222101630231,43.7522220270772,43.75223229570902,43.75225182379349,43.75236168338398,43.75252667245363,43.75274801210738,43.75285800934027,43.75289248087612,43.75294013229822,43.7529964312041,43.75307983543559,43.75316776405715,43.7532560070975,43.75339316201646,43.7532795401256,43.75316573568878,43.75310377992165,43.75303345218371,43.75300084587933,43.75294613982769,43.75275582463682,43.75296813745381,43.75335306032937,43.75383098688589,43.75388253843083,43.754132178625,43.75423367768592,43.7542462161087,43.75426776472558,43.75428458688994,43.7542926712605,43.75430954902912,43.75431755352504,43.75432924850767,43.7543324240246,43.75435124351926,43.75433909449668,43.75420792454136,43.75396356460827,43.7538810907874,43.75384924887037,43.75385178098047,43.75387071916667,43.75391599043892,43.75414423876284,43.75433352185479,43.75371814254956,43.75299834782186,43.75220646593025,43.75149171700691,43.75067743755884,43.74973185177628,43.74883112964944,43.74842396359958,43.7479534261013,43.74761427682056,43.74672301903793,43.74603516022863,43.7459489665755,43.7453968589105,43.74495798228705,43.74436961724931,43.74354176904974,43.74271389514005,43.74213869284663]}]],[[{"lng":[-79.37390410701661,-79.37455423129187,-79.37344558819258,-79.3733969175517,-79.37330420123132,-79.37319894237318,-79.37309627724336,-79.37300765231507,-79.3729435184934,-79.37285547021689,-79.3727422499983,-79.37262878077144,-79.37254924357684,-79.37237726147985,-79.37229279240925,-79.37223708653346,-79.37213094610958,-79.37205004922079,-79.37192119048125,-79.3717782249589,-79.37166339004001,-79.37153915060388,-79.3714196328043,-79.37133920292918,-79.37101912415312,-79.36949851312409,-79.36855482902861,-79.36797049669437,-79.36793091643327,-79.36733892113361,-79.36665621181169,-79.36600332544009,-79.36548816272061,-79.36498970823509,-79.36444990467039,-79.36574325014494,-79.36719133343394,-79.36788292620639,-79.36792732760628,-79.36897520933034,-79.36944667223771,-79.37008120270082,-79.37206561735788,-79.37324203575994,-79.37390410701661],"lat":[43.65535359772634,43.65706009185757,43.65729646557103,43.65732277897954,43.65739347565185,43.65750450290244,43.65760656493401,43.65769991473263,43.65778002005982,43.65788228996902,43.65796619730759,43.65803659792311,43.65808497653349,43.65816363569178,43.65819393979815,43.65821565148454,43.65824564679841,43.65825349650262,43.65826516391561,43.65827663020104,43.65828399610229,43.65829122788013,43.65829402578726,43.65828837858037,43.6582883176773,43.65865381069511,43.65886539405275,43.65897866848724,43.65898260427006,43.65911818111177,43.65928405557604,43.65943234827975,43.65816442578931,43.65693724897775,43.65561045553927,43.65533180025951,43.65501033097463,43.65485817308188,43.65484980513649,43.65461711872868,43.65452032280552,43.65439434355908,43.65394993057746,43.6537235368555,43.65535359772634]}]],[[{"lng":[-79.33054313439344,-79.3284316249289,-79.32743053912611,-79.32565913987516,-79.32469408587808,-79.32415450298033,-79.32213118333115,-79.31969303457912,-79.31926763907326,-79.31798921948389,-79.31619270952804,-79.31615757589692,-79.31606194683742,-79.31422838309322,-79.31418672341137,-79.31433569708653,-79.314421351524,-79.32029907175195,-79.32160253679068,-79.32256359357638,-79.32289212907106,-79.32312708711615,-79.324950549143,-79.3264450712722,-79.32669719627287,-79.32680811446149,-79.32681564224636,-79.3269220368681,-79.32710883455523,-79.32741658229924,-79.32764615436244,-79.32767516398459,-79.32771758401611,-79.32791818899254,-79.32818453038286,-79.32850332214726,-79.32879930965224,-79.32906226959534,-79.32938553404048,-79.32973088303369,-79.33004865589562,-79.3303729785804,-79.33067582379711,-79.33096565905339,-79.33110284937094,-79.33117951522522,-79.33111544678626,-79.3309854192088,-79.33083863676914,-79.3306705092791,-79.33064768524883,-79.33064782263043,-79.33066297158747,-79.33078753527272,-79.33101234153339,-79.33128697183128,-79.33146740051082,-79.33150353530031,-79.33153304914499,-79.33154544978088,-79.3315165763467,-79.33141990929863,-79.33122439291466,-79.33107660505473,-79.3308722387017,-79.33065094873045,-79.3304112760046,-79.33023266583002,-79.33012662342411,-79.33009970505509,-79.33014313810065,-79.33026655886637,-79.33042934591623,-79.33054313439344],"lat":[43.73978312584457,43.74018443901329,43.74037233327141,43.74065713707189,43.74078252188134,43.74084212397982,43.74100156223199,43.74114577335246,43.74117552473478,43.74126032556281,43.74136439452483,43.74136837775494,43.74137146947747,43.74149296528126,43.74149595143527,43.74107352913003,43.74092959186672,43.73524573656267,43.73592734079938,43.73509176181761,43.73484020949385,43.73474598687491,43.73362363605774,43.73278432026503,43.73267980942108,43.73275957030615,43.73276868245631,43.73283073322231,43.73293960076914,43.73292610191956,43.73306899181708,43.7333620689163,43.73365525223285,43.73391933362241,43.73413927679901,43.7342295487392,43.73413476682607,43.73397207678774,43.73392729280829,43.73400885764825,43.73413952901925,43.73431089387889,43.7344998578841,43.73471122588231,43.73496978441099,43.73525455464457,43.73549676084247,43.73570640686765,43.73592940089332,43.73614299095719,43.73618316616271,43.7362371796897,43.73627790964002,43.73640134524796,43.73658016638851,43.73672370701949,43.73685245864776,43.73689340489489,43.73694784735406,43.73708755816349,43.73715015000527,43.73727935632593,43.73744303666174,43.73762100665307,43.737793559035,43.73799286956041,43.73818299915904,43.73838293245031,43.73862452460012,43.73886718295907,43.73912446229767,43.73937390853798,43.73963284172051,43.73978312584457]}]],[[{"lng":[-79.34343659904651,-79.34096568556355,-79.33820560344198,-79.33707115295671,-79.33606397616427,-79.33584329733985,-79.33578867384193,-79.33470590696102,-79.33227223105025,-79.33207631809047,-79.33054313439344,-79.33042934591623,-79.33026655886637,-79.33014313810065,-79.33009970505509,-79.33012662342411,-79.33023266583002,-79.3304112760046,-79.33065094873045,-79.3308722387017,-79.33107660505473,-79.33122439291466,-79.33141990929863,-79.3315165763467,-79.33154544978088,-79.33153304914499,-79.33150353530031,-79.33146740051082,-79.33128697183128,-79.33101234153339,-79.33078753527272,-79.33066297158747,-79.33064782263043,-79.33064768524883,-79.3306705092791,-79.33083863676914,-79.3309854192088,-79.33111544678626,-79.33117951522522,-79.33110284937094,-79.33096565905339,-79.33067582379711,-79.3303729785804,-79.33004865589562,-79.32973088303369,-79.32938553404048,-79.32906226959534,-79.32879930965224,-79.32850332214726,-79.32818453038286,-79.32791818899254,-79.32771758401611,-79.32767516398459,-79.32764615436244,-79.32741658229924,-79.32710883455523,-79.3269220368681,-79.33145894746725,-79.33479430623903,-79.33670212242734,-79.33821511316749,-79.33950466671098,-79.34014693315244,-79.34084770930595,-79.34142107346001,-79.3418334551255,-79.34209928515224,-79.34253688661104,-79.34286457327261,-79.34317129036482,-79.34336451937043,-79.34343659904651],"lat":[43.73716604391456,43.73768840525643,43.73828311437608,43.73850066041472,43.73868863186009,43.73872592682525,43.73873863417387,43.73893440459644,43.73945270922524,43.7394903583515,43.73978312584457,43.73963284172051,43.73937390853798,43.73912446229767,43.73886718295907,43.73862452460012,43.73838293245031,43.73818299915904,43.73799286956041,43.737793559035,43.73762100665307,43.73744303666174,43.73727935632593,43.73715015000527,43.73708755816349,43.73694784735406,43.73689340489489,43.73685245864776,43.73672370701949,43.73658016638851,43.73640134524796,43.73627790964002,43.7362371796897,43.73618316616271,43.73614299095719,43.73592940089332,43.73570640686765,43.73549676084247,43.73525455464457,43.73496978441099,43.73471122588231,43.7344998578841,43.73431089387889,43.73413952901925,43.73400885764825,43.73392729280829,43.73397207678774,43.73413476682607,43.7342295487392,43.73413927679901,43.73391933362241,43.73365525223285,43.7333620689163,43.73306899181708,43.73292610191956,43.73293960076914,43.73283073322231,43.73026247674951,43.72836797353217,43.73025645855521,43.72941410607822,43.72858649062937,43.72780347161165,43.72765206399202,43.72965008483035,43.73118675792421,43.73221701500454,43.73379455954266,43.73505535074017,43.73631583671671,43.73702096820578,43.73716604391456]}]],[[{"lng":[-79.34242967286394,-79.34264059302487,-79.34281596230001,-79.34956411462316,-79.34988309872317,-79.35018806866671,-79.35255736363662,-79.35385495150413,-79.35401286482481,-79.35409232681381,-79.35414365337812,-79.35425870619326,-79.3544793061085,-79.35468877657442,-79.35470113321468,-79.35453884598532,-79.35457649198892,-79.35505866681994,-79.35515866565488,-79.3552608483476,-79.35533816392724,-79.3553947218786,-79.35553205968756,-79.35569943423747,-79.35582200556701,-79.35584393634372,-79.35596805108175,-79.35506668498175,-79.35436339046467,-79.35420087206747,-79.35331419247464,-79.35213870434626,-79.35105397071382,-79.34994131099444,-79.34822559008536,-79.34653451438436,-79.34489901953653,-79.34343659904651,-79.34336451937043,-79.34317129036482,-79.34286457327261,-79.34253688661104,-79.34209928515224,-79.3418334551255,-79.34142107346001,-79.34084770930595,-79.34092124740545,-79.34242967286394],"lat":[43.72742242813018,43.72742548747745,43.7274777215533,43.72984873458186,43.72998828028665,43.73019081633491,43.7287972924231,43.72799384559902,43.72815302409099,43.72839920136381,43.72859609282556,43.7288137963484,43.72895335257019,43.72909931953137,43.72926126222141,43.72940745737586,43.72956562333752,43.73131029856255,43.73148727569576,43.73172288674537,43.73195804977287,43.73218850304639,43.73261366032657,43.7332507949824,43.73381535927769,43.73400921629143,43.73452870291459,43.73470486163475,43.73485677009845,43.7348949381693,43.73508029648435,43.73531990590473,43.73556540383979,43.73578348242419,43.73614127614645,43.73649949067995,43.73685398422385,43.73716604391456,43.73702096820578,43.73631583671671,43.73505535074017,43.73379455954266,43.73221701500454,43.73118675792421,43.72965008483035,43.72765206399202,43.72764358965746,43.72742242813018]}]],[[{"lng":[-79.34084770930595,-79.34026028117376,-79.33950466671098,-79.33898517562695,-79.33670212242734,-79.33479430623903,-79.33145894746725,-79.3269220368681,-79.32681564224636,-79.32680811446149,-79.32669719627287,-79.32653337925643,-79.32628878633,-79.32605607534043,-79.32572419849038,-79.32533404958608,-79.32501135323021,-79.32473952037532,-79.32467202186535,-79.32470776504489,-79.32471459224294,-79.32477927817015,-79.32485682545011,-79.32499893410375,-79.32518553641955,-79.3251107853787,-79.32469303828789,-79.32426784677422,-79.32385022183182,-79.32340478709001,-79.32300166393604,-79.32264621008375,-79.32232758966553,-79.32203354967987,-79.32187010927147,-79.32174173269723,-79.32146374369769,-79.32113300671732,-79.32075812822298,-79.32048145630459,-79.32010531330262,-79.3197106716877,-79.31960542951866,-79.3197475696329,-79.32004871056111,-79.32029686493428,-79.32040550495832,-79.32021550252827,-79.31977260650663,-79.31927923585722,-79.31889611066269,-79.31875894869924,-79.31865721327239,-79.31885709134882,-79.31930384285209,-79.31968566749435,-79.31995766526616,-79.32025051236708,-79.32040677604512,-79.32066643551455,-79.32073851978565,-79.32111252227114,-79.32148609637208,-79.32189084302117,-79.32225323584744,-79.32266166529857,-79.32303630573898,-79.32319553485893,-79.32323253718214,-79.32318387982957,-79.32298968574679,-79.32274912432615,-79.32249264007177,-79.32232780051676,-79.32232887520487,-79.32264225631988,-79.32364018621816,-79.32528711592356,-79.32618369093687,-79.32689652329447,-79.32848324781486,-79.33005872898953,-79.33034625702989,-79.3320127944791,-79.33343714204135,-79.33470072793347,-79.33511304304348,-79.33520007599755,-79.33549279380244,-79.33665719007584,-79.3381502659587,-79.3388570169345,-79.3388625645922,-79.33914634550209,-79.33938394472173,-79.33953381392557,-79.33960125288324,-79.3399136002614,-79.34017820771771,-79.34059216755361,-79.34084770930595],"lat":[43.72765206399202,43.72775110640766,43.72858649062937,43.72895711173951,43.73025645855521,43.72836797353217,43.73026247674951,43.73283073322231,43.73276868245631,43.73275957030615,43.73267980942108,43.7325620057703,43.73231528146643,43.73208682456192,43.73199635405735,43.7319321243932,43.73178786482978,43.73154532887805,43.73121576899524,43.73093264233549,43.73087873086983,43.73056002078637,43.73022799630104,43.72991951296578,43.729656601199,43.72944396024975,43.72945584026607,43.72942259994129,43.72937137514758,43.72937384238624,43.72946703815498,43.72966436439444,43.72990282899607,43.73017757133456,43.73033279494125,43.7304569365316,43.73071399927633,43.73093868959773,43.73110880874306,43.73134779454578,43.73150439005821,43.73138147261217,43.73109186341274,43.73078329713916,43.73050407414625,43.73017906184419,43.72981608071457,43.72952963706123,43.7294376072122,43.72940784674675,43.72925809007142,43.72914354799538,43.72906553473182,43.72891085177998,43.72883190525211,43.72866198296602,43.728454346986,43.7282516078177,43.72801976458582,43.72792005871204,43.72789411226881,43.72780958855755,43.72768904960472,43.72753737043843,43.72742116572825,43.72732814046257,43.72719852012252,43.72694430172552,43.72661618415471,43.72632731829781,43.72605891147283,43.72581223855323,43.72559692815743,43.7253333620578,43.72518484592344,43.72510383804894,43.72482591932283,43.72439538039664,43.72416087356818,43.72397327021147,43.72355980742641,43.72315966113631,43.72308725618638,43.7225848909869,43.7221509849989,43.72177323110272,43.72164871096981,43.7216184721776,43.72152821597692,43.72117150034823,43.72071152994771,43.72049666742642,43.72049224709938,43.72136965058393,43.7222329692429,43.72278885651432,43.72301947596473,43.72421254021816,43.72516626860475,43.72670297012652,43.72765206399202]}]],[[{"lng":[-79.32669719627287,-79.3264450712722,-79.324950549143,-79.32312708711615,-79.32289212907106,-79.32256359357638,-79.32160253679068,-79.32029907175195,-79.314421351524,-79.31433569708653,-79.31418672341137,-79.31371193597877,-79.31280520100456,-79.31143106237415,-79.30978135790338,-79.30933611933561,-79.30921761842284,-79.30888556285312,-79.30845269360657,-79.30881679307568,-79.30920712909014,-79.30940310332618,-79.30951767753692,-79.30962190829251,-79.30970738349794,-79.30977210998888,-79.30981813496525,-79.30988647340206,-79.31001038533206,-79.310186546734,-79.31033236478608,-79.31053465599047,-79.31070616786182,-79.31086278976325,-79.31099422642976,-79.31105368896957,-79.31113827673973,-79.31124599667268,-79.31135451123772,-79.31147659071929,-79.31159311293868,-79.3116767791407,-79.31176193241811,-79.3118462908933,-79.31193184744069,-79.31200599082914,-79.31206263502851,-79.31212316865511,-79.31219815214986,-79.31227666367505,-79.31234931062819,-79.31242820101644,-79.31250476110482,-79.31256942019168,-79.31262264321646,-79.31266829193089,-79.31270111602556,-79.31274033395519,-79.31274708550198,-79.31275892695113,-79.31275388641519,-79.31273171002681,-79.31270583130605,-79.31238639749937,-79.31205331134981,-79.31199011877918,-79.31194275719983,-79.31190401463861,-79.31184708414595,-79.31178588524089,-79.31171280794683,-79.31163712646898,-79.31156169552015,-79.31154298382059,-79.31147463927169,-79.31117285440072,-79.31104726246987,-79.31097546924279,-79.31088015458394,-79.3108287771844,-79.310789172391,-79.31074915004029,-79.31073147508489,-79.31067335595982,-79.31034040569358,-79.31021860022724,-79.31001538991305,-79.30989830315728,-79.30980247536344,-79.30974909647519,-79.30969462450693,-79.30967486716408,-79.30966256986687,-79.30965013514351,-79.30963849495417,-79.30963362808029,-79.30963271202124,-79.30963339577693,-79.3096379012237,-79.3096526891304,-79.30966604422986,-79.30967494236617,-79.30969353019425,-79.30973047726779,-79.30976056748241,-79.30979091072169,-79.30982783330882,-79.30986544797517,-79.30990628600951,-79.30996162381824,-79.30998082271823,-79.31003788142063,-79.31006046297547,-79.31008266492307,-79.31011798481561,-79.31017082567882,-79.31024110980731,-79.31032413207255,-79.31040927905589,-79.31048605825411,-79.31055856893168,-79.31060532592741,-79.31068160123729,-79.3107736638085,-79.31084673967693,-79.31091782274179,-79.31101978164475,-79.31113038490032,-79.31123487175978,-79.31132386575159,-79.31140856490241,-79.31145588683775,-79.31148325901296,-79.31154945906198,-79.3116051484445,-79.31167746451256,-79.3117527482791,-79.31181160995756,-79.31185020889654,-79.31188821783704,-79.31194222147812,-79.31198918790237,-79.3120439318923,-79.31213251906389,-79.31223387049248,-79.31235371815052,-79.31243667649755,-79.31253648994773,-79.31262320419823,-79.31274652434273,-79.31284730631553,-79.31290522216617,-79.31295185058036,-79.3130222144385,-79.31310109204964,-79.31317927454917,-79.31325121437449,-79.31331751899127,-79.31337487234737,-79.31342101905327,-79.3134519217729,-79.31351057851478,-79.31368091820841,-79.3138291603648,-79.31402432665421,-79.31422079284503,-79.31434679805011,-79.31460257122244,-79.31479644870986,-79.31506096849481,-79.31528705804091,-79.31554247041993,-79.31578060480182,-79.3160163617742,-79.31628584207637,-79.3166298863021,-79.31686793458408,-79.3171358173568,-79.31734743922775,-79.31761529823082,-79.31778567877834,-79.31801361890214,-79.31808624542344,-79.31828204396142,-79.31848193564294,-79.31905872611929,-79.31950120948001,-79.31996720643353,-79.32040253325211,-79.32188768976268,-79.3219243719316,-79.3222843134334,-79.32232887520487,-79.32232780051676,-79.32249264007177,-79.32274912432615,-79.32298968574679,-79.32318387982957,-79.32323253718214,-79.32319553485893,-79.32303630573898,-79.32266166529857,-79.32225323584744,-79.32189084302117,-79.32148609637208,-79.32111252227114,-79.32073851978565,-79.32066643551455,-79.32040677604512,-79.32025051236708,-79.31995766526616,-79.31968566749435,-79.31930384285209,-79.31885709134882,-79.31865721327239,-79.31875894869924,-79.31889611066269,-79.31927923585722,-79.31977260650663,-79.32021550252827,-79.32040550495832,-79.32029686493428,-79.32004871056111,-79.3197475696329,-79.31960542951866,-79.3197106716877,-79.32010531330262,-79.32048145630459,-79.32075812822298,-79.32113300671732,-79.32146374369769,-79.32174173269723,-79.32187010927147,-79.32203354967987,-79.32232758966553,-79.32264621008375,-79.32300166393604,-79.32340478709001,-79.32385022183182,-79.32426784677422,-79.32469303828789,-79.3251107853787,-79.32518553641955,-79.32499893410375,-79.32485682545011,-79.32477927817015,-79.32471459224294,-79.32470776504489,-79.32467202186535,-79.32473952037532,-79.32501135323021,-79.32533404958608,-79.32572419849038,-79.32605607534043,-79.32628878633,-79.32653337925643,-79.32669719627287],"lat":[43.73267980942108,43.73278432026503,43.73362363605774,43.73474598687491,43.73484020949385,43.73509176181761,43.73592734079938,43.73524573656267,43.74092959186672,43.74107352913003,43.74149595143527,43.74153035455436,43.74161608087756,43.74179391492066,43.74213869284663,43.74091207749395,43.74065818065468,43.73982041014097,43.73879210607614,43.73868047056695,43.73860072880588,43.73855402789911,43.73851071341878,43.73844924198603,43.7383919939746,43.73834794185674,43.73831711594924,43.73828662006068,43.73823894223924,43.73818303505968,43.73813559081926,43.73809357234801,43.7380646013595,43.73804441191889,43.73800584617234,43.737988721256,43.7379629657465,43.73793755204583,43.73791665094315,43.73790495209947,43.73787966814128,43.73785389861491,43.73783715291143,43.7378158944453,43.73778115074507,43.73773714644936,43.73771097745878,43.73768486589179,43.73763646294181,43.73757911013669,43.73750816779,43.73743731760302,43.73736193198954,43.73727277768666,43.73719704736246,43.73712120516664,43.73703167070873,43.73693322873538,43.73691523457796,43.73682539027282,43.73675330052243,43.7366314472098,43.73653204401091,43.7357214752105,43.73491970567196,43.73479274533207,43.7347109383962,43.73462484783002,43.73455199142708,43.73449257481079,43.73441497882389,43.73434625621105,43.73426862534979,43.73424584406491,43.73416381689619,43.73388470850779,43.7337703282591,43.73370175269275,43.7336103245886,43.73353304879549,43.73345585699279,43.73336074518203,43.73329747044811,43.73317958627553,43.732377903494,43.73207444793078,43.73154924121297,43.73118734256161,43.73081225533895,43.73044229688525,43.73000930874479,43.72999101263584,43.72993681917909,43.72985111695103,43.72976992743635,43.72966633330416,43.72958971337042,43.72948170038243,43.72940525066538,43.72931995112781,43.72923463039216,43.72921675816712,43.72916743251588,43.72909146267508,43.72901539139636,43.72893032192513,43.72885885265052,43.72879189456265,43.72871589219106,43.72862669138593,43.7285774647397,43.72851079416406,43.72847962138427,43.72846194587613,43.72842195954363,43.72835963746758,43.72828866140299,43.72820887172417,43.72813361436368,43.72805373225388,43.72798728987365,43.72795197344475,43.72788999758721,43.72784184797138,43.72778441568828,43.72773595582893,43.72767444940398,43.72760406872049,43.72752909657144,43.7274808111385,43.72745055604602,43.7274242495415,43.72741565208549,43.72738062256712,43.72735894060244,43.72732850234144,43.7272981078801,43.72727647268266,43.72725453810605,43.72722809383566,43.72721538873112,43.72719807868127,43.72718088350054,43.72715068530134,43.72710708264979,43.72706834406467,43.72703806249611,43.72699452688055,43.72695079774646,43.72689410616496,43.72684158264708,43.7268244338832,43.72679361555028,43.72674514375109,43.72667420274791,43.72659884048226,43.72652338601297,43.72642984445741,43.72634517267137,43.72625124349916,43.72617518319679,43.72601851512708,43.7260300309543,43.7260592243187,43.72609811176811,43.72612351516444,43.72613887690402,43.72617415643006,43.72618601775804,43.72621242331805,43.72622925975116,43.72625552992238,43.72625904003522,43.72626701558144,43.7262844894766,43.7262715544443,43.72625255619284,43.72624299875712,43.72622811128291,43.726223053338,43.72620755785034,43.72618390743651,43.72617147353831,43.72612934563364,43.7260872776247,43.72599224162128,43.72590404949641,43.72578037366866,43.72566974803224,43.72528188998392,43.7252734267405,43.72519319360875,43.72518484592344,43.7253333620578,43.72559692815743,43.72581223855323,43.72605891147283,43.72632731829781,43.72661618415471,43.72694430172552,43.72719852012252,43.72732814046257,43.72742116572825,43.72753737043843,43.72768904960472,43.72780958855755,43.72789411226881,43.72792005871204,43.72801976458582,43.7282516078177,43.728454346986,43.72866198296602,43.72883190525211,43.72891085177998,43.72906553473182,43.72914354799538,43.72925809007142,43.72940784674675,43.7294376072122,43.72952963706123,43.72981608071457,43.73017906184419,43.73050407414625,43.73078329713916,43.73109186341274,43.73138147261217,43.73150439005821,43.73134779454578,43.73110880874306,43.73093868959773,43.73071399927633,43.7304569365316,43.73033279494125,43.73017757133456,43.72990282899607,43.72966436439444,43.72946703815498,43.72937384238624,43.72937137514758,43.72942259994129,43.72945584026607,43.72944396024975,43.729656601199,43.72991951296578,43.73022799630104,43.73056002078637,43.73087873086983,43.73093264233549,43.73121576899524,43.73154532887805,43.73178786482978,43.7319321243932,43.73199635405735,43.73208682456192,43.73231528146643,43.7325620057703,43.73267980942108]}]],[[{"lng":[-79.30449599129854,-79.30437481093878,-79.30412377579958,-79.30397372223175,-79.30393794094432,-79.30345852319476,-79.30263306569702,-79.30273224677396,-79.30291089031147,-79.30313241090317,-79.30344350024095,-79.30371038220549,-79.30382616905624,-79.30402610125662,-79.3042903757926,-79.30458338517138,-79.30479483208305,-79.30492650909319,-79.30504401517213,-79.3052135061255,-79.30540622661913,-79.30561811217522,-79.30576082120727,-79.30598721136624,-79.30620042790171,-79.30641126655735,-79.3067217821668,-79.30673710765333,-79.30700159380392,-79.30748329167214,-79.30809836246377,-79.30879052182172,-79.30894737993748,-79.3092705572151,-79.30968315836009,-79.30983070485264,-79.31121909658792,-79.31297429420987,-79.31351057851478,-79.3134519217729,-79.31342101905327,-79.31337487234737,-79.31331751899127,-79.31325121437449,-79.31317927454917,-79.31310109204964,-79.3130222144385,-79.31295185058036,-79.31290522216617,-79.31284730631553,-79.31274652434273,-79.31262320419823,-79.31253648994773,-79.31243667649755,-79.31235371815052,-79.31223387049248,-79.31213251906389,-79.3120439318923,-79.31198918790237,-79.31194222147812,-79.31188821783704,-79.31185020889654,-79.31181160995756,-79.3117527482791,-79.31167746451256,-79.3116051484445,-79.31154945906198,-79.31148325901296,-79.31145588683775,-79.31140856490241,-79.31132386575159,-79.31123487175978,-79.31113038490032,-79.31101978164475,-79.31091782274179,-79.31084673967693,-79.3107736638085,-79.31068160123729,-79.31060532592741,-79.31055856893168,-79.31048605825411,-79.31040927905589,-79.31032413207255,-79.31024110980731,-79.31017082567882,-79.31011798481561,-79.31008266492307,-79.31006046297547,-79.31003788142063,-79.30998082271823,-79.30996162381824,-79.30990628600951,-79.30986544797517,-79.30982783330882,-79.30979091072169,-79.30976056748241,-79.30973047726779,-79.30969353019425,-79.30967494236617,-79.30966604422986,-79.3096526891304,-79.3096379012237,-79.30963339577693,-79.30963271202124,-79.30963362808029,-79.30963849495417,-79.30965013514351,-79.30966256986687,-79.30967486716408,-79.30969462450693,-79.30974909647519,-79.30980247536344,-79.30989830315728,-79.31001538991305,-79.31021860022724,-79.31034040569358,-79.31067335595982,-79.31073147508489,-79.31074915004029,-79.310789172391,-79.3108287771844,-79.31088015458394,-79.31097546924279,-79.31104726246987,-79.31117285440072,-79.31147463927169,-79.31154298382059,-79.31156169552015,-79.31163712646898,-79.31171280794683,-79.31178588524089,-79.31184708414595,-79.31190401463861,-79.31194275719983,-79.31199011877918,-79.31205331134981,-79.31238639749937,-79.31270583130605,-79.31273171002681,-79.31275388641519,-79.31275892695113,-79.31274708550198,-79.31274033395519,-79.31270111602556,-79.31266829193089,-79.31262264321646,-79.31256942019168,-79.31250476110482,-79.31242820101644,-79.31234931062819,-79.31227666367505,-79.31219815214986,-79.31212316865511,-79.31206263502851,-79.31200599082914,-79.31193184744069,-79.3118462908933,-79.31176193241811,-79.3116767791407,-79.31159311293868,-79.31147659071929,-79.31135451123772,-79.31124599667268,-79.31113827673973,-79.31105368896957,-79.31099422642976,-79.31086278976325,-79.31070616786182,-79.31053465599047,-79.31033236478608,-79.310186546734,-79.31001038533206,-79.30988647340206,-79.30981813496525,-79.30977210998888,-79.30970738349794,-79.30962190829251,-79.30951767753692,-79.30940310332618,-79.30920712909014,-79.30881679307568,-79.30845269360657,-79.30832100591424,-79.30772461263533,-79.30763896615731,-79.30741543983187,-79.30674842703037,-79.30621380820195,-79.30588035443692,-79.30559780132876,-79.30510103961983,-79.30498781561562,-79.30449599129854],"lat":[43.72918504055463,43.72888608889163,43.72826555158947,43.7278942455563,43.72781260711223,43.72668015931063,43.7247185330446,43.72469750142034,43.72468665181816,43.72467643865699,43.72468105782487,43.72468501991025,43.72467773673765,43.72468520525212,43.72469812921399,43.72473398373528,43.72475062367878,43.72475707801941,43.72477232394301,43.72478834077153,43.72480470184084,43.72483484966224,43.72486397145096,43.7248989246274,43.72492909110017,43.72496372289803,43.72501783525589,43.72502256331815,43.72505348800603,43.72511454532354,43.7252091722196,43.72531843965278,43.72534326590372,43.72538414583367,43.72544876248495,43.72547344978164,43.72566051289917,43.72594308442919,43.72601851512708,43.72617518319679,43.72625124349916,43.72634517267137,43.72642984445741,43.72652338601297,43.72659884048226,43.72667420274791,43.72674514375109,43.72679361555028,43.7268244338832,43.72684158264708,43.72689410616496,43.72695079774646,43.72699452688055,43.72703806249611,43.72706834406467,43.72710708264979,43.72715068530134,43.72718088350054,43.72719807868127,43.72721538873112,43.72722809383566,43.72725453810605,43.72727647268266,43.7272981078801,43.72732850234144,43.72735894060244,43.72738062256712,43.72741565208549,43.7274242495415,43.72745055604602,43.7274808111385,43.72752909657144,43.72760406872049,43.72767444940398,43.72773595582893,43.72778441568828,43.72784184797138,43.72788999758721,43.72795197344475,43.72798728987365,43.72805373225388,43.72813361436368,43.72820887172417,43.72828866140299,43.72835963746758,43.72842195954363,43.72846194587613,43.72847962138427,43.72851079416406,43.7285774647397,43.72862669138593,43.72871589219106,43.72879189456265,43.72885885265052,43.72893032192513,43.72901539139636,43.72909146267508,43.72916743251588,43.72921675816712,43.72923463039216,43.72931995112781,43.72940525066538,43.72948170038243,43.72958971337042,43.72966633330416,43.72976992743635,43.72985111695103,43.72993681917909,43.72999101263584,43.73000930874479,43.73044229688525,43.73081225533895,43.73118734256161,43.73154924121297,43.73207444793078,43.732377903494,43.73317958627553,43.73329747044811,43.73336074518203,43.73345585699279,43.73353304879549,43.7336103245886,43.73370175269275,43.7337703282591,43.73388470850779,43.73416381689619,43.73424584406491,43.73426862534979,43.73434625621105,43.73441497882389,43.73449257481079,43.73455199142708,43.73462484783002,43.7347109383962,43.73479274533207,43.73491970567196,43.7357214752105,43.73653204401091,43.7366314472098,43.73675330052243,43.73682539027282,43.73691523457796,43.73693322873538,43.73703167070873,43.73712120516664,43.73719704736246,43.73727277768666,43.73736193198954,43.73743731760302,43.73750816779,43.73757911013669,43.73763646294181,43.73768486589179,43.73771097745878,43.73773714644936,43.73778115074507,43.7378158944453,43.73783715291143,43.73785389861491,43.73787966814128,43.73790495209947,43.73791665094315,43.73793755204583,43.7379629657465,43.737988721256,43.73800584617234,43.73804441191889,43.7380646013595,43.73809357234801,43.73813559081926,43.73818303505968,43.73823894223924,43.73828662006068,43.73831711594924,43.73834794185674,43.7383919939746,43.73844924198603,43.73851071341878,43.73855402789911,43.73860072880588,43.73868047056695,43.73879210607614,43.7384795005929,43.73703009354902,43.73681719011022,43.73626467253722,43.73465217936495,43.73335671164511,43.73254591656278,43.73186199210976,43.73065259828591,43.73037627015609,43.72918504055463]}]],[[{"lng":[-79.30759195292612,-79.30785359577678,-79.31214505144551,-79.30779945284132,-79.30698168195029,-79.30690665858644,-79.30467251289264,-79.30497075151708,-79.3051120381314,-79.30900361310822,-79.31004970532929,-79.31118232212893,-79.31197686805577,-79.3133325960508,-79.31948964977546,-79.32006051702908,-79.32019604345251,-79.32028002000342,-79.32032052332359,-79.32037155666323,-79.32052412750015,-79.32066181539828,-79.32083499869238,-79.32100568849091,-79.3210885194288,-79.32156798096054,-79.32206177810747,-79.32247448754319,-79.32275952981237,-79.3228776945099,-79.32293001224431,-79.32334895664695,-79.32355598607984,-79.32356855800759,-79.32360134289165,-79.32365940603195,-79.32385701882596,-79.32392152314884,-79.32374756047277,-79.32359469509557,-79.32343803438845,-79.32311740703491,-79.32253990189159,-79.32246243386801,-79.32244079742138,-79.32247216931928,-79.32255628031672,-79.32284143977218,-79.32323933138713,-79.32325993154743,-79.32362400571027,-79.32402878646094,-79.3242153103598,-79.32443781053711,-79.32464993595114,-79.32483014833981,-79.32524485337011,-79.32562611426532,-79.32606673754184,-79.32644652187814,-79.32659660616933,-79.32669549260395,-79.32671085955145,-79.32671791429073,-79.32672120837194,-79.32670719853202,-79.32667519137124,-79.32662292337042,-79.32654944003097,-79.32638801155602,-79.3261151588532,-79.32584158946354,-79.32568074880818,-79.32556823211375,-79.32540683342273,-79.32518618377172,-79.3250244899529,-79.32464258769245,-79.32447840393966,-79.32435199227096,-79.32402985938316,-79.32381314751115,-79.32364795292102,-79.32348708882294,-79.32338199357169,-79.32335080260528,-79.32333283517602,-79.32330837777455,-79.32327275692968,-79.32325764557217,-79.32324984739364,-79.32322087866126,-79.32309134805368,-79.32292162273006,-79.32275208134936,-79.32263513053998,-79.32251256169765,-79.32232966395425,-79.32232887520487,-79.3222843134334,-79.3219243719316,-79.32188768976268,-79.32040253325211,-79.31996720643353,-79.31950120948001,-79.31905872611929,-79.31848193564294,-79.31828204396142,-79.31808624542344,-79.31801361890214,-79.31778567877834,-79.31761529823082,-79.31734743922775,-79.3171358173568,-79.31686793458408,-79.3166298863021,-79.31628584207637,-79.3160163617742,-79.31578060480182,-79.31554247041993,-79.31528705804091,-79.31506096849481,-79.31479644870986,-79.31460257122244,-79.31434679805011,-79.31422079284503,-79.31402432665421,-79.3138291603648,-79.31368091820841,-79.31351057851478,-79.31297429420987,-79.31121909658792,-79.30983070485264,-79.30968315836009,-79.3092705572151,-79.30894737993748,-79.30879052182172,-79.30809836246377,-79.30759195292612],"lat":[43.72513127796736,43.72496870706777,43.7226911330536,43.72038691071489,43.71847727883064,43.71830612060768,43.71874343783174,43.71693019019364,43.71677673230844,43.71592167960328,43.71568960152899,43.71544519969692,43.71726669053358,43.71698064337452,43.71562618456721,43.71554896996683,43.71583461366658,43.71593937049435,43.71596247072912,43.71596772183224,43.71594745943865,43.7159359800688,43.71590701823202,43.71590502533733,43.71591524429261,43.71602130910524,43.71614558630207,43.71631827173899,43.71638546948572,43.71640970867689,43.71642397943479,43.71661916806307,43.71693286269958,43.71695555200417,43.71708665121444,43.71731705262904,43.71771162605793,43.7181222507153,43.71852037552279,43.71871617579279,43.71885349755268,43.71899732498839,43.71920498551734,43.71928486577124,43.71939257145069,43.71946054657209,43.71956080268035,43.7198261346777,43.72007961728114,43.72009342246977,43.72033731554441,43.72054588674458,43.72061163586446,43.72064640541431,43.72060900689446,43.72054863529721,43.72041959483016,43.72032616059813,43.72039112867974,43.72048671085855,43.7206149360036,43.72076941707924,43.7208866673399,43.72096787813452,43.7210444428701,43.72110275033876,43.72113828941026,43.72117353177536,43.72119496051663,43.72122410313889,43.72126961728049,43.72131512026922,43.7213488614919,43.72137871960898,43.72146187292014,43.72158466633231,43.72164530992359,43.72176141759883,43.72180852099871,43.72183367320407,43.72186945776501,43.72190228687231,43.72192686940026,43.72196501813225,43.72200398491125,43.72203953501792,43.72207077817333,43.72216953065902,43.72238955548436,43.72260096931867,43.72304194984507,43.7233610935403,43.72369235444048,43.72401852429813,43.72430868884162,43.72458153121459,43.72483187616409,43.72507224330568,43.72518484592344,43.72519319360875,43.7252734267405,43.72528188998392,43.72566974803224,43.72578037366866,43.72590404949641,43.72599224162128,43.7260872776247,43.72612934563364,43.72617147353831,43.72618390743651,43.72620755785034,43.726223053338,43.72622811128291,43.72624299875712,43.72625255619284,43.7262715544443,43.7262844894766,43.72626701558144,43.72625904003522,43.72625552992238,43.72622925975116,43.72621242331805,43.72618601775804,43.72617415643006,43.72613887690402,43.72612351516444,43.72609811176811,43.7260592243187,43.7260300309543,43.72601851512708,43.72594308442919,43.72566051289917,43.72547344978164,43.72544876248495,43.72538414583367,43.72534326590372,43.72531843965278,43.7252091722196,43.72513127796736]}]],[[{"lng":[-79.30759195292612,-79.30748329167214,-79.30700159380392,-79.30673710765333,-79.3067217821668,-79.30641126655735,-79.30620042790171,-79.30598721136624,-79.30576082120727,-79.30561811217522,-79.30540622661913,-79.3052135061255,-79.30504401517213,-79.30492650909319,-79.30479483208305,-79.30458338517138,-79.3042903757926,-79.30402610125662,-79.30382616905624,-79.30371038220549,-79.30344350024095,-79.30313241090317,-79.30291089031147,-79.30273224677396,-79.30263306569702,-79.3020882481737,-79.30164303441843,-79.30101103299764,-79.30057960197917,-79.30054628304266,-79.30016701740576,-79.29977758367545,-79.2992742714531,-79.29914853371717,-79.30176960922961,-79.30374229312632,-79.30532604405174,-79.3052555985237,-79.30518901624554,-79.30513731864296,-79.30508414104683,-79.30504339095786,-79.30500392178068,-79.30497555177425,-79.30494853217374,-79.30492716195791,-79.30490811502364,-79.30489584136049,-79.30487441383633,-79.30485511286106,-79.30482669718259,-79.30479943300564,-79.3051120381314,-79.30497075151708,-79.30467251289264,-79.30690665858644,-79.30698168195029,-79.30779945284132,-79.31214505144551,-79.30785359577678,-79.30759195292612],"lat":[43.72513127796736,43.72511454532354,43.72505348800603,43.72502256331815,43.72501783525589,43.72496372289803,43.72492909110017,43.7248989246274,43.72486397145096,43.72483484966224,43.72480470184084,43.72478834077153,43.72477232394301,43.72475707801941,43.72475062367878,43.72473398373528,43.72469812921399,43.72468520525212,43.72467773673765,43.72468501991025,43.72468105782487,43.72467643865699,43.72468665181816,43.72469750142034,43.7247185330446,43.72337338207786,43.72226825993815,43.72080028141227,43.71978105809458,43.71969954513447,43.71876193331575,43.71785126466283,43.71666425049808,43.71636531495291,43.71580998982515,43.71540710643598,43.71509303201621,43.71519550963356,43.71530254547579,43.71540080010241,43.71552612860486,43.71562454556278,43.71573198344849,43.71583058401348,43.71590669974147,43.71597848831168,43.71613123881379,43.71626158492606,43.71641889109244,43.71658063974892,43.71670624541331,43.71684546108361,43.71677673230844,43.71693019019364,43.71874343783174,43.71830612060768,43.71847727883064,43.72038691071489,43.7226911330536,43.72496870706777,43.72513127796736]}]],[[{"lng":[-79.2959731622243,-79.29634379147136,-79.29682720812664,-79.29707488981089,-79.29721647774991,-79.29737267011274,-79.29759044584665,-79.2983073499995,-79.29951406996268,-79.30076430536938,-79.30193238698622,-79.30263244044387,-79.30323106352935,-79.30417867669712,-79.3044975332476,-79.30583718334124,-79.30711907121203,-79.30795807041636,-79.31011485687242,-79.31052717098451,-79.31107814526109,-79.31213093675612,-79.31282352236229,-79.31266779619665,-79.31247636163184,-79.31223978447156,-79.31172509494461,-79.31124128838005,-79.31100478188404,-79.31064555841964,-79.30956623902351,-79.30894206876049,-79.30801967919137,-79.30741459526311,-79.30650822640133,-79.30595426728692,-79.30570194112991,-79.30545180286053,-79.30532604405174,-79.30374229312632,-79.30176960922961,-79.29914853371717,-79.29894913513293,-79.29865015896475,-79.29862610797278,-79.29835553901431,-79.29804639038319,-79.29776337429811,-79.29769570284152,-79.29739176239734,-79.29706915554129,-79.29680846453672,-79.29674890975376,-79.2964350626691,-79.29617691793341,-79.2959731622243],"lat":[43.70876834446705,43.70883238340581,43.70885309248734,43.70887478771898,43.70885889373051,43.70884771814676,43.70880145201943,43.70863659206411,43.70837090416588,43.70807884471449,43.70783965252855,43.70766542298747,43.70753478289894,43.70732812163605,43.70724283352094,43.70694304309999,43.70665147320783,43.70647026730455,43.70599348194973,43.70589605604887,43.70577358220309,43.70555508637555,43.70538968587463,43.70560352363044,43.705875346587,43.70620942617698,43.70691315805632,43.70754082728593,43.70789741014519,43.70833778696185,43.7097309916567,43.7105186094468,43.71167538942841,43.71244527727085,43.71357986705039,43.71427398797584,43.71461232096855,43.71492377000919,43.71509303201621,43.71540710643598,43.71580998982515,43.71636531495291,43.7158986567967,43.71519637516901,43.71514200541511,43.71450315950413,43.71377380885989,43.71311227116936,43.71295372901854,43.71222895500453,43.71145880217027,43.7108336924144,43.71068868353412,43.70990974737796,43.70927108105755,43.70876834446705]}]],[[{"lng":[-79.29346317602048,-79.29312585773337,-79.29284821565417,-79.29277246216023,-79.29242835876374,-79.29231054035017,-79.29221005973501,-79.29206424103815,-79.29189952543126,-79.29169686165912,-79.29147171983232,-79.29128174498952,-79.29114378562976,-79.29094627065183,-79.29075599491199,-79.29047187373537,-79.29021982898286,-79.28997609701335,-79.28980782518957,-79.28967080224344,-79.28966364286237,-79.29239462251493,-79.29522643347421,-79.29732260450338,-79.29898863658401,-79.30059090207919,-79.30319354897618,-79.30629944258928,-79.3074444145127,-79.3121354950129,-79.31294768619328,-79.31529191187776,-79.31718621904805,-79.31809549323485,-79.31935925438223,-79.3195476109295,-79.3195628675048,-79.32003221463896,-79.32052423473627,-79.32048405503438,-79.32042454105419,-79.32029452941217,-79.32005155171285,-79.31934664479749,-79.31845288705757,-79.31640924964206,-79.31580725334321,-79.31565292230962,-79.31549628568683,-79.31536235943615,-79.31528716936501,-79.31513788347199,-79.31495371358683,-79.31480114708634,-79.31470549008576,-79.31462265149115,-79.31453768842289,-79.31438798417743,-79.31419242592354,-79.3140411869263,-79.31377216047233,-79.31355178039553,-79.31330559651798,-79.31306296203013,-79.31290556008885,-79.31282352236229,-79.31213093675612,-79.31107814526109,-79.31052717098451,-79.31011485687242,-79.30795807041636,-79.30711907121203,-79.30583718334124,-79.3044975332476,-79.30417867669712,-79.30323106352935,-79.30263244044387,-79.30193238698622,-79.30076430536938,-79.29951406996268,-79.2983073499995,-79.29759044584665,-79.29737267011274,-79.29721647774991,-79.29707488981089,-79.29682720812664,-79.29634379147136,-79.2959731622243,-79.29570889058581,-79.29527089119229,-79.29496769786931,-79.29473615972472,-79.29448132327251,-79.29420554277173,-79.2938118865932,-79.29365522822289,-79.29346317602048],"lat":[43.70300906458313,43.70221617501422,43.70158171023217,43.70140954122554,43.70060754608731,43.70033113556961,43.700100083871,43.69980074970604,43.69947862797,43.69910192689858,43.69870247448048,43.69834396628066,43.69806725330559,43.69765011897427,43.69721958960699,43.69654469287973,43.69596911694476,43.69535324595633,43.69494104808425,43.69452031567354,43.69450220457639,43.69452504257428,43.69457173445505,43.69490905360265,43.69551017152886,43.69608772191418,43.6969772635064,43.69804065213542,43.69842228630922,43.69982423302114,43.69985423042612,43.70028940237258,43.70067298171207,43.70086199661065,43.70112814403724,43.70163511429618,43.70167143654092,43.70283527027868,43.70404903594676,43.70404394443651,43.70403856874101,43.70405466170135,43.70411410366092,43.70427486588557,43.7044732637543,43.70493387003091,43.70507362159636,43.70510285377797,43.70512304980217,43.70514358041485,43.70514247192158,43.70512676797788,43.70511054945217,43.70507679276992,43.70505737814013,43.70505165542826,43.70504140034069,43.70506619807392,43.70509482014106,43.70511959477768,43.70516963706987,43.70522939862569,43.70529327986837,43.70534371003503,43.70537289316584,43.70538968587463,43.70555508637555,43.70577358220309,43.70589605604887,43.70599348194973,43.70647026730455,43.70665147320783,43.70694304309999,43.70724283352094,43.70732812163605,43.70753478289894,43.70766542298747,43.70783965252855,43.70807884471449,43.70837090416588,43.70863659206411,43.70880145201943,43.70884771814676,43.70885889373051,43.70887478771898,43.70885309248734,43.70883238340581,43.70876834446705,43.70818368735929,43.70721817590992,43.70649790698616,43.70593615071984,43.705351631341,43.7047082859814,43.70381553608433,43.70345761954135,43.70300906458313]}]],[[{"lng":[-79.36272749505048,-79.36401734266575,-79.36560797801316,-79.36668004256072,-79.36787988956891,-79.36906114542219,-79.37020656350883,-79.3702643696614,-79.37030253231958,-79.37034316744104,-79.3704001753382,-79.37145179774353,-79.37190094994519,-79.37212762807732,-79.37226099100323,-79.37235276855148,-79.37270343196198,-79.37281865132583,-79.37296988985581,-79.37324203575994,-79.37206561735788,-79.37008120270082,-79.36944667223771,-79.36897520933034,-79.36792732760628,-79.36788292620639,-79.36719133343394,-79.36574325014494,-79.36444990467039,-79.36429391986236,-79.36401044653344,-79.36358552213281,-79.36323299284675,-79.36299932929141,-79.36294164699616,-79.36272749505048],"lat":[43.65156106749578,43.65126896829615,43.65091346107162,43.65067664029939,43.65038771211029,43.6500939149292,43.64983560264483,43.64982292391363,43.64981446606554,43.64980154243425,43.64978435122441,43.64944367044865,43.65045857004824,43.65096600406276,43.65130557059187,43.6515229269765,43.65234278628876,43.65269559682723,43.65303991798187,43.6537235368555,43.65394993057746,43.65439434355908,43.65452032280552,43.65461711872868,43.65484980513649,43.65485817308188,43.65501033097463,43.65533180025951,43.65561045553927,43.65521654413004,43.65455065615483,43.65355866840074,43.65272525163108,43.65219519502687,43.65206392908505,43.65156106749578]}]],[[{"lng":[-79.30629944258928,-79.30319354897618,-79.30059090207919,-79.29898863658401,-79.29732260450338,-79.29522643347421,-79.29239462251493,-79.28966364286237,-79.28942030898888,-79.28916718366297,-79.28900914943705,-79.28868385788435,-79.28834597137691,-79.28791137744338,-79.28774356733877,-79.29267761646013,-79.29326606406431,-79.29369241042164,-79.29425111030952,-79.29480362694783,-79.29583403905865,-79.29701910780491,-79.29992754623188,-79.30153011997217,-79.30282451208613,-79.30340033729561,-79.30350580172113,-79.3039247443068,-79.30404749735752,-79.30413670460845,-79.30626997866878,-79.30830187036139,-79.30841236204535,-79.30851906950025,-79.308866436214,-79.3084229640768,-79.30812765892315,-79.30885514832822,-79.31124678537687,-79.31307900221809,-79.31313672920989,-79.31346427358783,-79.31379982425892,-79.31417160452615,-79.31423599812062,-79.31450472560516,-79.31481784410643,-79.31511994047628,-79.31542511453465,-79.31569814737369,-79.31572653240008,-79.31577273670877,-79.31584274511019,-79.31607796222792,-79.31647199785237,-79.31651824430243,-79.31681140973616,-79.31710449525579,-79.3174193374681,-79.31774111798991,-79.31803696206268,-79.31821676072563,-79.31835386474013,-79.31866904702791,-79.31882542696212,-79.31886823515296,-79.31901959684758,-79.31935925438223,-79.31809549323485,-79.31718621904805,-79.31529191187776,-79.31294768619328,-79.3121354950129,-79.3074444145127,-79.30629944258928],"lat":[43.69804065213542,43.6969772635064,43.69608772191418,43.69551017152886,43.69490905360265,43.69457173445505,43.69452504257428,43.69450220457639,43.69383673736515,43.69322072385433,43.69286268979376,43.69207896803278,43.69122754178828,43.69017212110156,43.68977342977266,43.68869923111813,43.69007209126652,43.69106444475982,43.69236934037733,43.6922469676741,43.6920102834517,43.6917352958767,43.6910716766258,43.69069932986231,43.69039899081292,43.69027242160633,43.69049462588964,43.69040182283964,43.6903991436899,43.69039596658708,43.68990981089087,43.68944020845414,43.68939683417192,43.68935340376896,43.68927302605208,43.68824446713018,43.68754685259332,43.68737749115451,43.68682313933843,43.68639100947492,43.68651788944975,43.68727907096235,43.68809438150812,43.68888772048773,43.68905070614932,43.68964430000212,43.69037375758657,43.69108054677314,43.69178729044037,43.69242163367635,43.69248956689056,43.69261177460488,43.69279743699701,43.69335011408069,43.69420228579528,43.69429748786656,43.6950176435864,43.69571529236455,43.69644017657471,43.69711115014149,43.69781333791816,43.69821207096344,43.69857875854047,43.69926322517086,43.69966170263144,43.69977035605145,43.70015075545265,43.70112814403724,43.70086199661065,43.70067298171207,43.70028940237258,43.69985423042612,43.69982423302114,43.69842228630922,43.69804065213542]}]],[[{"lng":[-79.31307900221809,-79.31124678537687,-79.30885514832822,-79.30812765892315,-79.3084229640768,-79.308866436214,-79.30851906950025,-79.30841236204535,-79.30830187036139,-79.30626997866878,-79.30413670460845,-79.30404749735752,-79.3039247443068,-79.30350580172113,-79.30340033729561,-79.30282451208613,-79.30153011997217,-79.29992754623188,-79.29701910780491,-79.29583403905865,-79.29480362694783,-79.29425111030952,-79.29369241042164,-79.29326606406431,-79.29267761646013,-79.28774356733877,-79.28754109093352,-79.29031742534598,-79.29245758938728,-79.29298907978549,-79.29413217643486,-79.29526905349944,-79.29695781896086,-79.29775497424311,-79.29800639091742,-79.29873900979848,-79.29920627474452,-79.29965173012772,-79.30047682823248,-79.30092312138866,-79.30160419768781,-79.30219532921217,-79.30242346797507,-79.30326207672047,-79.30395245325752,-79.30501733213374,-79.30612831241464,-79.30733713184088,-79.30824422062565,-79.30910582503144,-79.31129542432654,-79.31149248748973,-79.31183018899874,-79.31220072205009,-79.31276623775973,-79.31307900221809],"lat":[43.68639100947492,43.68682313933843,43.68737749115451,43.68754685259332,43.68824446713018,43.68927302605208,43.68935340376896,43.68939683417192,43.68944020845414,43.68990981089087,43.69039596658708,43.6903991436899,43.69040182283964,43.69049462588964,43.69027242160633,43.69039899081292,43.69069932986231,43.6910716766258,43.6917352958767,43.6920102834517,43.6922469676741,43.69236934037733,43.69106444475982,43.69007209126652,43.68869923111813,43.68977342977266,43.68928428892468,43.68862793640231,43.6881422317536,43.68802405450094,43.68778006659274,43.68753138342626,43.68713338032142,43.68696521784684,43.68690594878184,43.68671872422485,43.68659515083548,43.68647575201107,43.68622687837296,43.68610298604109,43.6858744677542,43.68567170508864,43.68558498444746,43.68527336742295,43.68498646023784,43.68457007438617,43.6841363579284,43.68366807188299,43.68331683342632,43.68297832769689,43.68213283075708,43.68259493435524,43.68341928499587,43.6843026323934,43.68567955592359,43.68639100947492]}]],[[{"lng":[-79.2983579985399,-79.29859916503945,-79.29883821542178,-79.29947096975727,-79.30014215950375,-79.30014774384144,-79.30014343003576,-79.30012871325668,-79.30013306996152,-79.30020102756836,-79.30038102666464,-79.30057364095677,-79.30092312138866,-79.30047682823248,-79.29965173012772,-79.29920627474452,-79.29873900979848,-79.29800639091742,-79.29775497424311,-79.29695781896086,-79.29526905349944,-79.29413217643486,-79.29298907978549,-79.29245758938728,-79.29031742534598,-79.28754109093352,-79.28698936999641,-79.28656361587723,-79.28623675113977,-79.28584637310017,-79.28526844450445,-79.28509624838104,-79.28496770904644,-79.28466358015721,-79.28436674042463,-79.28404708538649,-79.28495058031636,-79.28505472909288,-79.28596129425306,-79.28607546387833,-79.28707415405501,-79.28817397397252,-79.28839656045584,-79.28919060398347,-79.28993730501266,-79.29068188036261,-79.29124461841528,-79.29184774661148,-79.29257925626237,-79.29281584881203,-79.29303150409382,-79.29323821130028,-79.29451037735903,-79.29585883503408,-79.29629946573354,-79.29801138065443,-79.29805371863101,-79.2980871879596,-79.29810151332427,-79.29810217139763,-79.29805106942948,-79.29804719828327,-79.29804470150884,-79.29806173801374,-79.29812474219318,-79.29818692866111,-79.2983579985399],"lat":[43.6804194006563,43.68058061641571,43.68073711912582,43.68219171664757,43.68382233087456,43.68390352186331,43.68394396664598,43.68399775966635,43.68406083839482,43.68423738803068,43.68481187157781,43.68526942654636,43.68610298604109,43.68622687837296,43.68647575201107,43.68659515083548,43.68671872422485,43.68690594878184,43.68696521784684,43.68713338032142,43.68753138342626,43.68778006659274,43.68802405450094,43.6881422317536,43.68862793640231,43.68928428892468,43.68793436478838,43.68689707496122,43.68609531802279,43.68515307661111,43.68376683233448,43.68333656399339,43.68303297838859,43.68230357363237,43.68158787042839,43.68080430886771,43.68071885017548,43.68071141119935,43.6806258105408,43.68061852093671,43.68053447314288,43.68041592358668,43.68038775068693,43.68035913198043,43.68027569734267,43.6802418270004,43.68019172997307,43.68014223391251,43.68010365489754,43.68007568264422,43.68004289616159,43.68000097363551,43.67970930804339,43.67937825699124,43.67929030606944,43.67887912863684,43.67891576717571,43.67896577661273,43.67903800592335,43.6791775466541,43.67978009929399,43.67988806557327,43.67995104234827,43.68003231402472,43.68015937027528,43.68026832030388,43.6804194006563]}]],[[{"lng":[-79.29801138065443,-79.29629946573354,-79.29585883503408,-79.29451037735903,-79.29323821130028,-79.29303150409382,-79.29281584881203,-79.29257925626237,-79.29184774661148,-79.29124461841528,-79.29068188036261,-79.28993730501266,-79.28919060398347,-79.28839656045584,-79.28817397397252,-79.28707415405501,-79.28607546387833,-79.28596129425306,-79.28505472909288,-79.28495058031636,-79.28404708538649,-79.28402496742331,-79.28314216870316,-79.28120400137765,-79.28184813780639,-79.28237747044096,-79.28306840322685,-79.28348372989727,-79.28411039732174,-79.28522298267413,-79.28631549539509,-79.28756885521354,-79.28875116896221,-79.28991778177031,-79.29099788705284,-79.29190604463996,-79.29255969270213,-79.29272631875536,-79.29351466542143,-79.29445723424422,-79.29546500070154,-79.29574106927176,-79.29658194779466,-79.29686082702926,-79.29800360557007,-79.29874868486563,-79.29926717023213,-79.29977074257449,-79.29998667797703,-79.30051996898625,-79.30116840352672,-79.30099001383864,-79.30071026941889,-79.30046270333273,-79.29990375590066,-79.29891624811646,-79.29801138065443],"lat":[43.67887912863684,43.67929030606944,43.67937825699124,43.67970930804339,43.68000097363551,43.68004289616159,43.68007568264422,43.68010365489754,43.68014223391251,43.68019172997307,43.6802418270004,43.68027569734267,43.68035913198043,43.68038775068693,43.68041592358668,43.68053447314288,43.68061852093671,43.6806258105408,43.68071141119935,43.68071885017548,43.68080430886771,43.68075122508366,43.67862978914899,43.67394563209395,43.6738156974795,43.6737156307618,43.6735729796278,43.67347119485722,43.6733544880621,43.67311013025933,43.67288787557493,43.6726275117931,43.67237966339604,43.67214047976593,43.67191808455247,43.67174253016048,43.67160376519833,43.671574747642,43.67139288943124,43.67120441557545,43.67098530936894,43.67093091374426,43.67074990774493,43.6707045533111,43.67045592969949,43.67226769549052,43.67353146270385,43.67475899745521,43.67532042408404,43.67661591231916,43.67812024556821,43.67820311248026,43.67830697701272,43.67839790644496,43.67850661831723,43.67868546582692,43.67887912863684]}]],[[{"lng":[-79.29546500070154,-79.29445723424422,-79.29351466542143,-79.29272631875536,-79.29255969270213,-79.29190604463996,-79.29099788705284,-79.28991778177031,-79.28875116896221,-79.28756885521354,-79.28631549539509,-79.28522298267413,-79.28411039732174,-79.28348372989727,-79.28306840322685,-79.28237747044096,-79.28184813780639,-79.28120400137765,-79.28118169866936,-79.28104747272306,-79.28000520018149,-79.28011526600359,-79.28003245949613,-79.28006676238,-79.28002163937958,-79.28000280967602,-79.28001459350146,-79.28004894310277,-79.28010047809092,-79.28013374430323,-79.28017149425456,-79.28029784214438,-79.28046549480676,-79.28061515311116,-79.28073322453385,-79.2808665529857,-79.2810211234099,-79.28122143428349,-79.28121648955015,-79.28122975951895,-79.28132391944729,-79.28138530800108,-79.28156241047154,-79.2817693860632,-79.28191969866107,-79.28196609594509,-79.28207142367354,-79.28224594348325,-79.28247123265379,-79.28267743934211,-79.2826856835734,-79.2828289479946,-79.28292793700354,-79.28302702836181,-79.28321821669641,-79.28345416501782,-79.28352530804061,-79.28351908246982,-79.28361919973051,-79.28376317368347,-79.28395556362062,-79.2841174604177,-79.28431928190611,-79.28446586863615,-79.28458641602568,-79.28475795574431,-79.28492888135983,-79.2850711975499,-79.2851258129686,-79.28518876421262,-79.28528762946748,-79.28535219326524,-79.28551265729723,-79.28567685465001,-79.28583364006909,-79.28594779471497,-79.28607756834829,-79.28620102031051,-79.28638177475712,-79.28665332128311,-79.28693404056702,-79.28713670191831,-79.28727883867479,-79.28734733887222,-79.28739145514271,-79.28748137352473,-79.28754397142437,-79.28759501395385,-79.28768296452151,-79.28778354148676,-79.28785613165883,-79.28792059360593,-79.28801621137775,-79.28819737606634,-79.28830186277894,-79.28841225627249,-79.28854199459383,-79.28865051780832,-79.28887175710207,-79.28910664489773,-79.28927630589678,-79.28940397337306,-79.2896488029894,-79.28986160674349,-79.2900735154662,-79.2903439301117,-79.29057781776868,-79.29072247946974,-79.29090560631906,-79.2910581673557,-79.29113560967342,-79.29124840078606,-79.29140039936905,-79.2915452664453,-79.29175995812699,-79.29198799075134,-79.29215560216512,-79.29231692457586,-79.29254978356423,-79.29273510715835,-79.2929052707224,-79.29308731818635,-79.29326282096442,-79.2933896635375,-79.29352910311238,-79.29368587012416,-79.29375019633832,-79.29390448620525,-79.2939766056281,-79.29409260656919,-79.29421660646418,-79.29425200272186,-79.29432092690291,-79.29440122497226,-79.29449849075507,-79.29459304631365,-79.29470426841746,-79.29480780724218,-79.29493751759314,-79.29512135075709,-79.29516724694506,-79.29525812750292,-79.29536488073084,-79.29559034442364,-79.29578697641102,-79.29590800875854,-79.29617559200565,-79.29625182935885,-79.29640611750288,-79.29671203200894,-79.2969645833679,-79.297191583348,-79.29732182531603,-79.29744414084172,-79.29768946717745,-79.29786663802352,-79.29806030088344,-79.29821997117058,-79.29830477845115,-79.29847760428723,-79.29875030196628,-79.29896797423442,-79.29917655069961,-79.29923345584302,-79.29930011949324,-79.29943739451299,-79.29962214560958,-79.29979490754033,-79.2999494466477,-79.30011494638313,-79.30027691715613,-79.30037731455721,-79.30050565252135,-79.30067262420575,-79.30084932587445,-79.30107396345019,-79.30124372708983,-79.30139046721743,-79.30150526532293,-79.30164357891965,-79.30194013556792,-79.30209870489419,-79.30211775009992,-79.30213524853637,-79.30219254293614,-79.30225976127107,-79.3023750071557,-79.30241385470104,-79.3024719461461,-79.30252576849082,-79.30259339192014,-79.30269119390178,-79.30286829791565,-79.30307617144074,-79.30324508824923,-79.3034223874746,-79.30350985917227,-79.30358378739976,-79.30369415403719,-79.30384975486331,-79.30405589501295,-79.30444726139976,-79.30448678069368,-79.30451659249077,-79.30472654860321,-79.30549403348779,-79.30576221662591,-79.30463212688655,-79.30386431512129,-79.30350527717214,-79.30222238537529,-79.30189283323011,-79.30085380431731,-79.29981330928614,-79.29889362314157,-79.29800360557007,-79.29686082702926,-79.29658194779466,-79.29574106927176,-79.29546500070154],"lat":[43.67098530936894,43.67120441557545,43.67139288943124,43.671574747642,43.67160376519833,43.67174253016048,43.67191808455247,43.67214047976593,43.67237966339604,43.6726275117931,43.67288787557493,43.67311013025933,43.6733544880621,43.67347119485722,43.6735729796278,43.6737156307618,43.6738156974795,43.67394563209395,43.67385977765258,43.67388926570333,43.67137057677103,43.67134522731742,43.67115034796163,43.67114186223997,43.67104216107356,43.67099236665075,43.67098804300779,43.67100656400372,43.67112886666941,43.67121038535466,43.67122445648688,43.67119485089221,43.67116586667887,43.67113661148245,43.67107087286675,43.67101886665316,43.67095367680502,43.6709701927419,43.67087109633231,43.67083978890508,43.6708592090696,43.67090514232448,43.67092130863058,43.67081180575448,43.67065202962473,43.6706032161425,43.6706363065922,43.67065693401511,43.67061531018906,43.67047887810827,43.67037988992915,43.67035953774345,43.6703925321568,43.67042552802209,43.67041039637719,43.67034642572688,43.67024397127579,43.67013585370513,43.67013285644647,43.67016652577469,43.67013790793118,43.67011333291462,43.67008035506949,43.67005104870059,43.66998075222649,43.66986179998837,43.66974283827292,43.66972697004099,43.66979530466877,43.66981875432476,43.6698022338062,43.66979420054699,43.66976960205145,43.66971805331647,43.66965739115945,43.66959599935561,43.66944941264568,43.669442262086,43.66946297668886,43.66943095020889,43.66940365160943,43.669366180362,43.66932780166849,43.66929282025396,43.66926647538212,43.66924081687216,43.66921915988057,43.66919741972308,43.66915822855574,43.66911022449344,43.66903929602054,43.66897724773393,43.66893817117526,43.66886886897086,43.66881642200954,43.66875047038783,43.66871640542234,43.66874503666698,43.66873034556158,43.6687068563557,43.66868239025136,43.66863479029215,43.66853493171094,43.66849310640785,43.66842876229261,43.66832469401195,43.66822466973261,43.66815481747437,43.66806303528079,43.66798870970783,43.66796286173999,43.66797805132432,43.66800282901313,43.66800499485931,43.66795419222484,43.66789458657087,43.66783407750592,43.66774646818463,43.66762842045608,43.6675455801273,43.66744910003951,43.66734829619379,43.66722939028832,43.66712767144315,43.66702172957504,43.66690704424232,43.66684949161108,43.66673917997458,43.66665923852895,43.66655744705916,43.66645127385967,43.66642929716477,43.66635380881013,43.66625589511429,43.66615832460833,43.66605171155879,43.66598135518672,43.66602340923206,43.66606585367078,43.66601008326077,43.66592065779969,43.66588600548055,43.66586509287473,43.66578743764323,43.66569134797977,43.66560313269662,43.66554410839906,43.66553174200057,43.66547543901243,43.66538097680539,43.66530372257829,43.66528009915849,43.66528203974717,43.6652568560517,43.66524700769346,43.66528565476917,43.66537405826848,43.66545295323042,43.66549922617765,43.66552880559021,43.66550585946594,43.6654865946637,43.66548519810402,43.66558515720162,43.66564016140003,43.66566470928084,43.66562694930458,43.66557550763576,43.66551020157456,43.66544064735361,43.66536203838798,43.66539953966076,43.66546896341981,43.66548494942902,43.66546057079319,43.66538289243513,43.66529989693943,43.66523906401851,43.66520476221771,43.66517071975789,43.66510311050553,43.6650289494137,43.66493021019127,43.66488996109356,43.66486380625979,43.66487380689598,43.66490702599673,43.66490760307234,43.66490396498063,43.66489576243889,43.66488776486197,43.66485771042665,43.66481533060206,43.66475531334518,43.66467680329365,43.66459841743524,43.66455020489827,43.66451079326679,43.66444491641245,43.66437070881284,43.66423414688279,43.66564462899997,43.66567222138678,43.66571767376927,43.6662158994276,43.66809105915248,43.66875236236771,43.66899215999006,43.66916989888746,43.66925458955389,43.66955519888879,43.66961331649577,43.66983642570338,43.67006409465089,43.67026204545228,43.67045592969949,43.6707045533111,43.67074990774493,43.67093091374426,43.67098530936894]}]],[[{"lng":[-79.31163025007605,-79.31130034560707,-79.31116964169696,-79.31081668808277,-79.31035935151276,-79.31018940082485,-79.31011548854194,-79.31004808691389,-79.30996959478023,-79.30985980151239,-79.30907430486782,-79.30791417613713,-79.307269266264,-79.30654214061714,-79.30589035932404,-79.30559538613828,-79.30516296091582,-79.30487313653931,-79.30470889158801,-79.30450293864497,-79.30435050042783,-79.30417518646397,-79.30407530020284,-79.3038785372723,-79.30336004390632,-79.30247444185002,-79.30116840352672,-79.30051996898625,-79.29998667797703,-79.29977074257449,-79.29926717023213,-79.29874868486563,-79.29800360557007,-79.29889362314157,-79.29981330928614,-79.30085380431731,-79.30189283323011,-79.30222238537529,-79.30350527717214,-79.30386431512129,-79.30463212688655,-79.30576221662591,-79.30690527909496,-79.30803514361105,-79.309182454487,-79.31032313725834,-79.31258112160381,-79.31246369946615,-79.31186457895406,-79.31163025007605],"lat":[43.66910476741898,43.66973021380731,43.66996683563106,43.67063694994794,43.67149456200242,43.67180270843459,43.67189604655631,43.67197606791581,43.67206492722879,43.6721939226134,43.672862044251,43.67392088220165,43.67446954566728,43.67512960217676,43.67569615988971,43.6759529347506,43.67630669239472,43.67651844169385,43.67659702326458,43.67672908781924,43.67679884204229,43.67688175959704,43.67693428928567,43.67701238706824,43.67722532934093,43.67758594063516,43.67812024556821,43.67661591231916,43.67532042408404,43.67475899745521,43.67353146270385,43.67226769549052,43.67045592969949,43.67026204545228,43.67006409465089,43.66983642570338,43.66961331649577,43.66955519888879,43.66925458955389,43.66916989888746,43.66899215999006,43.66875236236771,43.66849465169251,43.66823673414918,43.66798365455931,43.66772587439186,43.66723703063231,43.66749644522046,43.66868504453424,43.66910476741898]}]],[[{"lng":[-79.30848653947299,-79.3089487967702,-79.30928734579211,-79.30965863598641,-79.31001536881946,-79.31074227816681,-79.31107677476258,-79.31129542432654,-79.30910582503144,-79.30824422062565,-79.30733713184088,-79.30612831241464,-79.30501733213374,-79.30395245325752,-79.30326207672047,-79.30242346797507,-79.30219532921217,-79.30160419768781,-79.30092312138866,-79.30057364095677,-79.30038102666464,-79.30020102756836,-79.30013306996152,-79.30012871325668,-79.30014343003576,-79.30014774384144,-79.30014215950375,-79.29947096975727,-79.29883821542178,-79.29859916503945,-79.2983579985399,-79.29818692866111,-79.29812474219318,-79.29806173801374,-79.29804470150884,-79.29804719828327,-79.29805106942948,-79.29810217139763,-79.29810151332427,-79.2980871879596,-79.29805371863101,-79.29801138065443,-79.29891624811646,-79.29990375590066,-79.30046270333273,-79.30071026941889,-79.30099001383864,-79.30116840352672,-79.30247444185002,-79.30336004390632,-79.3038785372723,-79.30407530020284,-79.30417518646397,-79.30435050042783,-79.30450293864497,-79.30470889158801,-79.30487313653931,-79.30516296091582,-79.30559538613828,-79.30589035932404,-79.30654214061714,-79.307269266264,-79.30791417613713,-79.30848653947299],"lat":[43.67532043532843,43.67645279699747,43.67725916440752,43.67817854001451,43.67903018402896,43.68078767911311,43.6816343997819,43.68213283075708,43.68297832769689,43.68331683342632,43.68366807188299,43.6841363579284,43.68457007438617,43.68498646023784,43.68527336742295,43.68558498444746,43.68567170508864,43.6858744677542,43.68610298604109,43.68526942654636,43.68481187157781,43.68423738803068,43.68406083839482,43.68399775966635,43.68394396664598,43.68390352186331,43.68382233087456,43.68219171664757,43.68073711912582,43.68058061641571,43.6804194006563,43.68026832030388,43.68015937027528,43.68003231402472,43.67995104234827,43.67988806557327,43.67978009929399,43.6791775466541,43.67903800592335,43.67896577661273,43.67891576717571,43.67887912863684,43.67868546582692,43.67850661831723,43.67839790644496,43.67830697701272,43.67820311248026,43.67812024556821,43.67758594063516,43.67722532934093,43.67701238706824,43.67693428928567,43.67688175959704,43.67679884204229,43.67672908781924,43.67659702326458,43.67651844169385,43.67630669239472,43.6759529347506,43.67569615988971,43.67512960217676,43.67446954566728,43.67392088220165,43.67532043532843]}]],[[{"lng":[-79.3089487967702,-79.30848653947299,-79.30791417613713,-79.30907430486782,-79.30985980151239,-79.30996959478023,-79.31004808691389,-79.31011548854194,-79.31018940082485,-79.31035935151276,-79.31081668808277,-79.31116964169696,-79.31130034560707,-79.31163025007605,-79.31186457895406,-79.31246369946615,-79.31258112160381,-79.31354375320987,-79.3148379147252,-79.31658956940325,-79.31690447636197,-79.31723100630819,-79.31756449397672,-79.3180420650217,-79.31837383753965,-79.31929700049034,-79.32019138727199,-79.32028832768808,-79.32078923872204,-79.32144155770062,-79.3194269522065,-79.31486238937751,-79.31363402802022,-79.31129542432654,-79.31107677476258,-79.31074227816681,-79.31001536881946,-79.30965863598641,-79.30928734579211,-79.3089487967702],"lat":[43.67645279699747,43.67532043532843,43.67392088220165,43.672862044251,43.6721939226134,43.67206492722879,43.67197606791581,43.67189604655631,43.67180270843459,43.67149456200242,43.67063694994794,43.66996683563106,43.66973021380731,43.66910476741898,43.66868504453424,43.66749644522046,43.66723703063231,43.6670215991114,43.66673462154655,43.66635525714543,43.66715666405571,43.66792673410111,43.66869699558304,43.66989706237728,43.67072580911889,43.67290032249227,43.67505631072451,43.67533238684216,43.67652819306508,43.67813149107594,43.67888973710527,43.68074489588412,43.6812174745959,43.68213283075708,43.6816343997819,43.68078767911311,43.67903018402896,43.67817854001451,43.67725916440752,43.67645279699747]}]],[[{"lng":[-79.32362678963605,-79.32396201646065,-79.32429004731348,-79.32464649394076,-79.32495288433381,-79.32511084527056,-79.32237875899192,-79.31915894131808,-79.31674181002941,-79.31483122056859,-79.3146507537723,-79.31430992936005,-79.31396008554729,-79.31360058298162,-79.31356035684176,-79.31307900221809,-79.31276623775973,-79.31220072205009,-79.31183018899874,-79.31149248748973,-79.31129542432654,-79.31363402802022,-79.31486238937751,-79.3194269522065,-79.32144155770062,-79.32145305222028,-79.32182863438575,-79.32213416840005,-79.32244610916833,-79.32272340332757,-79.32301061063413,-79.32362678963605],"lat":[43.68335871122294,43.68416048485152,43.68497115387988,43.68578673946242,43.6865925881435,43.686964075025,43.68755422886602,43.6882452582705,43.68876789320586,43.68917642094223,43.68877767220653,43.68798938241343,43.68718736544279,43.68637620278635,43.68628558926781,43.68639100947492,43.68567955592359,43.6843026323934,43.68341928499587,43.68259493435524,43.68213283075708,43.6812174745959,43.68074489588412,43.67888973710527,43.67813149107594,43.67816316691859,43.6791005701514,43.67982089558242,43.68050980717678,43.68121612331745,43.68190467021373,43.68335871122294]}]],[[{"lng":[-79.32703056512422,-79.3201420264121,-79.31935925438223,-79.31901959684758,-79.31886823515296,-79.31882542696212,-79.31866904702791,-79.31835386474013,-79.31821676072563,-79.31803696206268,-79.31774111798991,-79.3174193374681,-79.31710449525579,-79.31681140973616,-79.31651824430243,-79.31647199785237,-79.31607796222792,-79.31584274511019,-79.31577273670877,-79.31572653240008,-79.31569814737369,-79.31542511453465,-79.31511994047628,-79.31481784410643,-79.31450472560516,-79.31423599812062,-79.31417160452615,-79.31379982425892,-79.31346427358783,-79.31313672920989,-79.31307900221809,-79.31356035684176,-79.31360058298162,-79.31396008554729,-79.31430992936005,-79.3146507537723,-79.31483122056859,-79.31674181002941,-79.31915894131808,-79.32237875899192,-79.32511084527056,-79.32527788715396,-79.32558257303715,-79.32584598960494,-79.32618572571947,-79.32656197429461,-79.32710261370661,-79.32727618044861,-79.32800606204255,-79.32876952603272,-79.32947740163222,-79.32954395096985,-79.32957746828509,-79.32958359686552,-79.3289127672322,-79.32843742078452,-79.32721981842947,-79.32703056512422],"lat":[43.70124647067802,43.70128890632542,43.70112814403724,43.70015075545265,43.69977035605145,43.69966170263144,43.69926322517086,43.69857875854047,43.69821207096344,43.69781333791816,43.69711115014149,43.69644017657471,43.69571529236455,43.6950176435864,43.69429748786656,43.69420228579528,43.69335011408069,43.69279743699701,43.69261177460488,43.69248956689056,43.69242163367635,43.69178729044037,43.69108054677314,43.69037375758657,43.68964430000212,43.68905070614932,43.68888772048773,43.68809438150812,43.68727907096235,43.68651788944975,43.68639100947492,43.68628558926781,43.68637620278635,43.68718736544279,43.68798938241343,43.68877767220653,43.68917642094223,43.68876789320586,43.6882452582705,43.68755422886602,43.686964075025,43.6873401957452,43.68804240469179,43.68877560514225,43.68966745755849,43.69058225848246,43.69191823698481,43.69234396094429,43.6941284828246,43.69593590565075,43.69776051426108,43.69793243460187,43.69812205604484,43.69826617722649,43.69907122805367,43.69971259794563,43.7010648808382,43.70124647067802]}]],[[{"lng":[-79.36272749505048,-79.36216215021196,-79.36152889686386,-79.36143692030814,-79.36134490312664,-79.36646947934049,-79.370211350007,-79.37077618487262,-79.37091913175271,-79.37145179774353,-79.3704001753382,-79.37034316744104,-79.37030253231958,-79.3702643696614,-79.37020656350883,-79.36906114542219,-79.36787988956891,-79.36668004256072,-79.36560797801316,-79.36401734266575,-79.36272749505048],"lat":[43.65156106749578,43.6502068901671,43.64875271362544,43.64854425768912,43.64836739829941,43.64723421595726,43.64641876043878,43.64783131504439,43.64822053122698,43.64944367044865,43.64978435122441,43.64980154243425,43.64981446606554,43.64982292391363,43.64983560264483,43.6500939149292,43.65038771211029,43.65067664029939,43.65091346107162,43.65126896829615,43.65156106749578]}]],[[{"lng":[-79.31948964977546,-79.3133325960508,-79.31197686805577,-79.31118232212893,-79.31004970532929,-79.30900361310822,-79.3051120381314,-79.30479943300564,-79.30482669718259,-79.30485511286106,-79.30487441383633,-79.30489584136049,-79.30490811502364,-79.30492716195791,-79.30494853217374,-79.30497555177425,-79.30500392178068,-79.30504339095786,-79.30508414104683,-79.30513731864296,-79.30518901624554,-79.3052555985237,-79.30532604405174,-79.30545180286053,-79.30570194112991,-79.30595426728692,-79.30650822640133,-79.30741459526311,-79.30801967919137,-79.30894206876049,-79.30956623902351,-79.31064555841964,-79.31100478188404,-79.31124128838005,-79.31172509494461,-79.31223978447156,-79.31247636163184,-79.31266779619665,-79.31282352236229,-79.31290556008885,-79.31306296203013,-79.31330559651798,-79.31355178039553,-79.31377216047233,-79.3140411869263,-79.31419242592354,-79.31438798417743,-79.31453768842289,-79.31462265149115,-79.31470549008576,-79.31480114708634,-79.31495371358683,-79.31513788347199,-79.31528716936501,-79.31536235943615,-79.31628020589558,-79.31643587297343,-79.31647397669163,-79.31653174648108,-79.31661545681166,-79.31674829439966,-79.31687407322657,-79.31755643432635,-79.3169945860126,-79.31731222283101,-79.31643584760634,-79.31777949844965,-79.31894594857896,-79.31906500389751,-79.31987086947056,-79.31942976478636,-79.3192801967726,-79.31932756768488,-79.31952055877953,-79.31948964977546],"lat":[43.71562618456721,43.71698064337452,43.71726669053358,43.71544519969692,43.71568960152899,43.71592167960328,43.71677673230844,43.71684546108361,43.71670624541331,43.71658063974892,43.71641889109244,43.71626158492606,43.71613123881379,43.71597848831168,43.71590669974147,43.71583058401348,43.71573198344849,43.71562454556278,43.71552612860486,43.71540080010241,43.71530254547579,43.71519550963356,43.71509303201621,43.71492377000919,43.71461232096855,43.71427398797584,43.71357986705039,43.71244527727085,43.71167538942841,43.7105186094468,43.7097309916567,43.70833778696185,43.70789741014519,43.70754082728593,43.70691315805632,43.70620942617698,43.705875346587,43.70560352363044,43.70538968587463,43.70537289316584,43.70534371003503,43.70529327986837,43.70522939862569,43.70516963706987,43.70511959477768,43.70509482014106,43.70506619807392,43.70504140034069,43.70505165542826,43.70505737814013,43.70507679276992,43.70511054945217,43.70512676797788,43.70514247192158,43.70514358041485,43.70727301859741,43.70767148832143,43.70772156046264,43.70776742133476,43.70782257625688,43.70787404382003,43.70792540725999,43.70821010548352,43.70894017248739,43.7090663766167,43.71016547998556,43.71070702127776,43.71118823756824,43.7112497618981,43.71166562095495,43.71255050792912,43.71268333744566,43.7132918459158,43.71340720858689,43.71562618456721]}]],[[{"lng":[-79.31935925438223,-79.3201420264121,-79.32703056512422,-79.32665313670275,-79.32701191880604,-79.32734945644305,-79.32774195088089,-79.32815619002878,-79.32821595243715,-79.32855027172666,-79.32894125067043,-79.32927575300981,-79.32963994362497,-79.32997810487826,-79.33027399457147,-79.33059324527551,-79.33092109836558,-79.33129084713904,-79.33160130424235,-79.33180601624817,-79.33185181254963,-79.33187929797023,-79.33197363819609,-79.33210655764205,-79.3322173273511,-79.33234186007242,-79.33247294012027,-79.3325815473499,-79.33288716940397,-79.3330785228477,-79.33308243254562,-79.33319399457491,-79.33342286750742,-79.33377176327822,-79.33414275831628,-79.33450332298224,-79.33468144813926,-79.33488821668806,-79.33540474312925,-79.33528174196698,-79.33503807793242,-79.33504856906872,-79.33528060138252,-79.33542683108857,-79.33563626461289,-79.33599702961023,-79.33642642937744,-79.33690218823982,-79.33734963928661,-79.33781929893809,-79.33815703294819,-79.33816707075731,-79.33786931197722,-79.33752755694859,-79.33726881415453,-79.33718728595116,-79.33739233669436,-79.33779885349074,-79.33801606050548,-79.33789314693443,-79.33385346714471,-79.33145033868887,-79.32937530647762,-79.3290035320923,-79.32709256693113,-79.32351536656469,-79.32190037528443,-79.32042918740808,-79.31987086947056,-79.31906500389751,-79.31894594857896,-79.31777949844965,-79.31643584760634,-79.31731222283101,-79.3169945860126,-79.31755643432635,-79.31687407322657,-79.31674829439966,-79.31661545681166,-79.31653174648108,-79.31647397669163,-79.31643587297343,-79.31628020589558,-79.31536235943615,-79.31549628568683,-79.31565292230962,-79.31580725334321,-79.31640924964206,-79.31845288705757,-79.31934664479749,-79.32005155171285,-79.32029452941217,-79.32042454105419,-79.32048405503438,-79.32052423473627,-79.32003221463896,-79.3195628675048,-79.3195476109295,-79.31935925438223],"lat":[43.70112814403724,43.70128890632542,43.70124647067802,43.70162226794112,43.70151949749285,43.70142991791133,43.7014579871338,43.70151373912542,43.70151011245609,43.7015105019379,43.7015342239277,43.70162013267613,43.70166146455749,43.70163039912457,43.70156270716316,43.70163038526371,43.70163517432777,43.7016495762176,43.70175763195653,43.7019812588434,43.70223398252565,43.70252703776782,43.70278956198034,43.70303005425649,43.70311718964775,43.70321361777005,43.70326954242095,43.70329813310924,43.7033521029812,43.7033638963935,43.70335945244042,43.7032260502213,43.70305826092269,43.70300933688677,43.7030012428029,43.70303350435582,43.7030721081911,43.70311112912105,43.70319526112242,43.70324298016474,43.70352749325458,43.70384721597324,43.7041567533247,43.70425340413253,43.70439598514918,43.70461737623646,43.70482167141732,43.70498171987015,43.70507374970583,43.70514359591219,43.70532854728093,43.70563935110575,43.70591867248025,43.70615675414391,43.70642763956168,43.70673711104878,43.70698314761547,43.70720519860067,43.70745591183857,43.70769276616678,43.70857482260396,43.70910256131151,43.70956744175316,43.70966102748873,43.71008775656662,43.71086828267542,43.71122725088313,43.71155995376257,43.71166562095495,43.7112497618981,43.71118823756824,43.71070702127776,43.71016547998556,43.7090663766167,43.70894017248739,43.70821010548352,43.70792540725999,43.70787404382003,43.70782257625688,43.70776742133476,43.70772156046264,43.70767148832143,43.70727301859741,43.70514358041485,43.70512304980217,43.70510285377797,43.70507362159636,43.70493387003091,43.7044732637543,43.70427486588557,43.70411410366092,43.70405466170135,43.70403856874101,43.70404394443651,43.70404903594676,43.70283527027868,43.70167143654092,43.70163511429618,43.70112814403724]}]],[[{"lng":[-79.33540474312925,-79.33488821668806,-79.33468144813926,-79.33450332298224,-79.33414275831628,-79.33377176327822,-79.33342286750742,-79.33319399457491,-79.33308243254562,-79.3330785228477,-79.33288716940397,-79.3325815473499,-79.33247294012027,-79.33234186007242,-79.3322173273511,-79.33210655764205,-79.33197363819609,-79.33187929797023,-79.33185181254963,-79.33180601624817,-79.33160130424235,-79.33129084713904,-79.33092109836558,-79.33059324527551,-79.33027399457147,-79.32997810487826,-79.32963994362497,-79.32927575300981,-79.32894125067043,-79.32855027172666,-79.32821595243715,-79.32815619002878,-79.32774195088089,-79.32734945644305,-79.32701191880604,-79.32665313670275,-79.32703056512422,-79.32721981842947,-79.32843742078452,-79.3289127672322,-79.32958359686552,-79.32957746828509,-79.32954395096985,-79.32947740163222,-79.32876952603272,-79.32800606204255,-79.3290527525747,-79.32996267373278,-79.33011516495098,-79.33112732423518,-79.33203930149514,-79.33235588793204,-79.33306661894898,-79.33338744142779,-79.33428342935836,-79.33454238341169,-79.33568437178145,-79.33675835567786,-79.33775111930366,-79.33870287181719,-79.33966709836939,-79.3406417919396,-79.34167791264491,-79.34193010888885,-79.34233210715649,-79.34244427848066,-79.34271879118127,-79.34310417332586,-79.34325405100178,-79.34330612532784,-79.34335773987041,-79.34338009113192,-79.34340786348362,-79.34347091251497,-79.34453135399198,-79.34463527351771,-79.34504341049337,-79.34488606803816,-79.34480962016492,-79.34470782611295,-79.34456748528282,-79.34444021595107,-79.34431777031287,-79.34414423362023,-79.34399463235998,-79.34386624793176,-79.3437366517793,-79.34363571807297,-79.3435449083849,-79.34342766121696,-79.34328338167502,-79.3428594508471,-79.34235837977475,-79.3418304261987,-79.34136192534703,-79.34091562906531,-79.34047256290364,-79.34008465119292,-79.33971265181792,-79.33942663096823,-79.33907467854338,-79.33863326213796,-79.33819030550866,-79.33769104708482,-79.33723936898616,-79.33690941434645,-79.33657648696153,-79.3361586710475,-79.33571839660233,-79.33540474312925],"lat":[43.70319526112242,43.70311112912105,43.7030721081911,43.70303350435582,43.7030012428029,43.70300933688677,43.70305826092269,43.7032260502213,43.70335945244042,43.7033638963935,43.7033521029812,43.70329813310924,43.70326954242095,43.70321361777005,43.70311718964775,43.70303005425649,43.70278956198034,43.70252703776782,43.70223398252565,43.7019812588434,43.70175763195653,43.7016495762176,43.70163517432777,43.70163038526371,43.70156270716316,43.70163039912457,43.70166146455749,43.70162013267613,43.7015342239277,43.7015105019379,43.70151011245609,43.70151373912542,43.7014579871338,43.70142991791133,43.70151949749285,43.70162226794112,43.70124647067802,43.7010648808382,43.69971259794563,43.69907122805367,43.69826617722649,43.69812205604484,43.69793243460187,43.69776051426108,43.69593590565075,43.6941284828246,43.69389164696505,43.69371590669746,43.69370004114013,43.6934717736268,43.69325094687935,43.69318805244831,43.69303638636226,43.692964548874,43.69278848089218,43.69273374229191,43.69249373336872,43.69224831227372,43.69202402458755,43.69181731607023,43.69161069082281,43.6913952974388,43.69116719413435,43.69181017472992,43.69276139531756,43.69302407996992,43.69373039623795,43.6947173808051,43.69513823600062,43.69524260376505,43.69531536788242,43.69534719884051,43.69537910838077,43.6954565391955,43.69734017737221,43.69753522532169,43.69828848101633,43.69829070323375,43.6982760930034,43.69825661459313,43.69825458165223,43.69825723890149,43.69827796985295,43.69832946729387,43.69838131133005,43.69842446067887,43.69848559627535,43.69854264618713,43.69859993277996,43.69868825295602,43.69880318703892,43.6990176787146,43.69912743711146,43.69923230224629,43.69945964456481,43.69972772619203,43.69998244002457,43.70029646612116,43.7006196340223,43.7009531421047,43.70125418428075,43.70148181893086,43.70170951948311,43.70192289635936,43.70215487729463,43.70245173180802,43.70270794319338,43.70289558291055,43.70307371199893,43.70319526112242]}]],[[{"lng":[-79.33900168244035,-79.33920510338676,-79.3395153752485,-79.33981900201415,-79.34024706389592,-79.34052793669443,-79.34081666652257,-79.34097399018543,-79.34167791264491,-79.3406417919396,-79.33966709836939,-79.33870287181719,-79.33775111930366,-79.33675835567786,-79.33568437178145,-79.33454238341169,-79.33428342935836,-79.33338744142779,-79.33306661894898,-79.33235588793204,-79.33203930149514,-79.33112732423518,-79.33011516495098,-79.32996267373278,-79.3290527525747,-79.32800606204255,-79.32727618044861,-79.32710261370661,-79.32656197429461,-79.32618572571947,-79.32584598960494,-79.32558257303715,-79.32527788715396,-79.32511084527056,-79.32869151340694,-79.331349476513,-79.33389187792663,-79.33491600303863,-79.33585689290004,-79.33673851835806,-79.33764323070227,-79.33878987435027,-79.33883871778798,-79.33900168244035],"lat":[43.68452860636894,43.68505376867259,43.68583713008001,43.68657979502778,43.68762601323956,43.68831883468177,43.68901636045685,43.68941473210095,43.69116719413435,43.6913952974388,43.69161069082281,43.69181731607023,43.69202402458755,43.69224831227372,43.69249373336872,43.69273374229191,43.69278848089218,43.692964548874,43.69303638636226,43.69318805244831,43.69325094687935,43.6934717736268,43.69370004114013,43.69371590669746,43.69389164696505,43.6941284828246,43.69234396094429,43.69191823698481,43.69058225848246,43.68966745755849,43.68877560514225,43.68804240469179,43.6873401957452,43.686964075025,43.68620163859143,43.68562366408899,43.6850754498031,43.68485632461706,43.68465389213173,43.68445968134071,43.68427021054801,43.68402573671646,43.6841390617105,43.68452860636894]}]],[[{"lng":[-79.32362678963605,-79.32301061063413,-79.32272340332757,-79.32244610916833,-79.32213416840005,-79.32182863438575,-79.32145305222028,-79.32144155770062,-79.32249074422177,-79.32323593151237,-79.32326729005888,-79.32385912472878,-79.32623351484671,-79.32627430207309,-79.32965532516366,-79.3296640056632,-79.33025900556883,-79.33094783457064,-79.33125586468468,-79.33155767982394,-79.33162235341648,-79.33183342592139,-79.33196729324783,-79.33236985887879,-79.33270309051935,-79.33340649339345,-79.3337461165866,-79.33389187792663,-79.331349476513,-79.32869151340694,-79.32511084527056,-79.32495288433381,-79.32464649394076,-79.32429004731348,-79.32396201646065,-79.32362678963605],"lat":[43.68335871122294,43.68190467021373,43.68121612331745,43.68050980717678,43.67982089558242,43.6791005701514,43.67816316691859,43.67813149107594,43.67773721599563,43.67745558721995,43.67744254432433,43.67721258330862,43.67628398973812,43.67626658310554,43.67493388346568,43.67495651536428,43.67636988379268,43.67798257334714,43.67875683311158,43.67944557222102,43.6796625644811,43.68013392992781,43.6804824610182,43.68147882636914,43.68227154362287,43.68392043762715,43.68470865376094,43.6850754498031,43.68562366408899,43.68620163859143,43.686964075025,43.6865925881435,43.68578673946242,43.68497115387988,43.68416048485152,43.68335871122294]}]],[[{"lng":[-79.32840305565266,-79.32874152538807,-79.32905867897217,-79.32924370831658,-79.32930531050063,-79.32965532516366,-79.32627430207309,-79.32623351484671,-79.32385912472878,-79.32326729005888,-79.32323593151237,-79.32249074422177,-79.32144155770062,-79.32078923872204,-79.32028832768808,-79.32019138727199,-79.31929700049034,-79.32038764624963,-79.32122261999022,-79.32184880250918,-79.32290185844141,-79.32397834573554,-79.32492607306969,-79.3250617053744,-79.32592500729203,-79.32691025457684,-79.32800984926074,-79.32840305565266],"lat":[43.67185876022457,43.67268757477493,43.67346656549361,43.6739239623041,43.67407789707696,43.67493388346568,43.67626658310554,43.67628398973812,43.67721258330862,43.67744254432433,43.67745558721995,43.67773721599563,43.67813149107594,43.67652819306508,43.67533238684216,43.67505631072451,43.67290032249227,43.67265971019108,43.67246493294338,43.67230759277741,43.6720754064222,43.6718389629588,43.67160530311503,43.67157578405597,43.67139480054026,43.67117967784338,43.67091661857675,43.67185876022457]}]],[[{"lng":[-79.3148379147252,-79.31487440609828,-79.31489922944101,-79.31493980165554,-79.31497692909228,-79.3150319471619,-79.31510324371638,-79.31517588696516,-79.31527374766245,-79.31537723867929,-79.31552098547022,-79.31562078001946,-79.31620233935811,-79.31763747436437,-79.31776685926327,-79.31791013902796,-79.31805252191,-79.3181614454177,-79.31827452563785,-79.31837676906608,-79.31847082507515,-79.31857401140016,-79.31862663408982,-79.31906498257649,-79.31938109528923,-79.31973710824104,-79.31984435444851,-79.3199653524059,-79.32001254169437,-79.32018824031584,-79.32056228297756,-79.32165372298368,-79.32354355561013,-79.32525703116198,-79.32662220503474,-79.32946368069595,-79.32972163937671,-79.32998515698664,-79.3302113180209,-79.32898464432608,-79.32786682488511,-79.32733693096526,-79.32620329732451,-79.32597415241978,-79.32529552644301,-79.32557802374399,-79.326008091853,-79.32609430384296,-79.32662608734871,-79.32739561557871,-79.32769172212001,-79.32800984926074,-79.32691025457684,-79.32592500729203,-79.3250617053744,-79.32492607306969,-79.32397834573554,-79.32290185844141,-79.32184880250918,-79.32122261999022,-79.32038764624963,-79.31929700049034,-79.31837383753965,-79.3180420650217,-79.31756449397672,-79.31723100630819,-79.31690447636197,-79.31658956940325,-79.3148379147252],"lat":[43.66673462154655,43.66628946976017,43.66617722055083,43.66608329760942,43.66600282688693,43.66592261993662,43.66583815195442,43.66578962183452,43.66572805042189,43.66568456595986,43.66564617578874,43.6656251416663,43.66548517852723,43.66517315003561,43.66514354762231,43.66509614554177,43.66504422907052,43.66497822695921,43.66489887290203,43.66481935926915,43.66472172103148,43.66463772008238,43.66458448219322,43.66423976181083,43.66396525834684,43.66376344685224,43.66372442452242,43.66369469629969,43.66368584787615,43.6636529626461,43.66358194291393,43.66333223753402,43.6629142861774,43.66251172321857,43.66220314725984,43.66153787958673,43.66210436682015,43.66272503683847,43.66323254503474,43.66352527504566,43.66380149057062,43.66390185033148,43.66416881604815,43.66421497079711,43.66438965947848,43.66505103659705,43.66608833859105,43.66630564989305,43.66759199471252,43.66944453275944,43.67017361847904,43.67091661857675,43.67117967784338,43.67139480054026,43.67157578405597,43.67160530311503,43.6718389629588,43.6720754064222,43.67230759277741,43.67246493294338,43.67265971019108,43.67290032249227,43.67072580911889,43.66989706237728,43.66869699558304,43.66792673410111,43.66715666405571,43.66635525714543,43.66673462154655]}]],[[{"lng":[-79.34007962442661,-79.33890064344692,-79.33880722345877,-79.33769139776379,-79.33630285356134,-79.33588227268378,-79.33532212795494,-79.3348778630513,-79.33430299140431,-79.33405804066525,-79.33329173485384,-79.33280154698662,-79.33225043529637,-79.33141174508016,-79.3302113180209,-79.32998515698664,-79.32972163937671,-79.32946368069595,-79.32977378472847,-79.32989639524304,-79.3302787929654,-79.33089488502948,-79.33113883337971,-79.33215079675116,-79.33317281860137,-79.33416868425174,-79.33513227905198,-79.33659049964071,-79.33779145975878,-79.33894753448091,-79.34007962442661],"lat":[43.66105324379152,43.66133318019615,43.66134982634269,43.66159024912901,43.66190329836655,43.66200519914936,43.66210956651946,43.66222462164132,43.66232885944592,43.66241080817906,43.66256617221844,43.66266263544718,43.6627986265967,43.66299343075633,43.66323254503474,43.66272503683847,43.66210436682015,43.66153787958673,43.66142988685497,43.66134165847063,43.66088346167975,43.66015870609066,43.66011275758784,43.65987088301562,43.65966065327158,43.65944562210437,43.65924352514312,43.65892718313089,43.65866100312659,43.65839415808025,43.66105324379152]}]],[[{"lng":[-79.35091165186124,-79.35119633675332,-79.35133343330924,-79.35172834670693,-79.35196457207027,-79.35205232892339,-79.35213798113142,-79.35223561186105,-79.3523211206064,-79.352407465686,-79.35250205146369,-79.35259391923746,-79.35267556538552,-79.35278061799475,-79.35288391418347,-79.35292626839781,-79.35298861845577,-79.35308845883357,-79.3531887100561,-79.35329470496147,-79.35336376057985,-79.35340542304597,-79.35352096406697,-79.35363374643997,-79.35374176632941,-79.35386137749113,-79.35397266670385,-79.35409665346901,-79.35419159057675,-79.3542695166567,-79.35431156344174,-79.3541609282956,-79.35390755924684,-79.35374230601761,-79.35360988641591,-79.3533342720492,-79.35222561426546,-79.35155455177248,-79.35091660833449,-79.3498387817659,-79.34863894905524,-79.34848603767962,-79.34743424388323,-79.34673074982821,-79.34639809722597,-79.34610230824869,-79.34578981785214,-79.34566142005532,-79.34545785911178,-79.34444562516774,-79.34357815846563,-79.34249324479146,-79.3412550565607,-79.34078597764056,-79.34007962442661,-79.33894753448091,-79.34017093859296,-79.34138482842538,-79.34248489642364,-79.34335126017132,-79.34433047646905,-79.34531899021087,-79.34631892622086,-79.34635543963407,-79.34754699045922,-79.34776035507399,-79.34869674764846,-79.35091165186124],"lat":[43.65571765085597,43.65565415314445,43.65562912488499,43.65555830394888,43.6557373405461,43.65581512337467,43.65588387374699,43.65597080084699,43.65604855105319,43.65612190225204,43.65620428420325,43.65627762485447,43.6563508180925,43.65644685374477,43.65652485990441,43.65656156851378,43.6566074773034,43.65670343755258,43.65677689853398,43.65686844626767,43.65692345351142,43.65695556093434,43.65703833397801,43.65714348224121,43.65723955974829,43.65734930717354,43.65744552157216,43.65755983279038,43.65765572144817,43.65778287219486,43.65786458605156,43.65789842493458,43.65795328901144,43.65798691678564,43.65801651631757,43.65808456170199,43.65832523006273,43.65848659112199,43.65862592104703,43.65887151209535,43.65914683716726,43.65918513667593,43.65941298709367,43.65957394078783,43.65965914830154,43.6597313852922,43.65978987671519,43.65981052319257,43.65985717748016,43.66008107203065,43.66028013988974,43.66052096770768,43.66078665670509,43.66087886961884,43.66105324379152,43.65839415808025,43.65813287025387,43.65787134119233,43.6576170604367,43.65742707681348,43.6572161265028,43.65698288732218,43.65675421584705,43.65675024317557,43.65648841525987,43.65641939384339,43.65622137996591,43.65571765085597]}]],[[{"lng":[-79.33676282368644,-79.33669362999994,-79.33658744030737,-79.33648063447869,-79.33637291075102,-79.336265164758,-79.33615672493745,-79.33604828492976,-79.33586760735889,-79.33568715342166,-79.33550575916384,-79.33532448880796,-79.33514240002258,-79.33495957287174,-79.33477689185349,-79.33459314103392,-79.33440974338109,-79.33429365428198,-79.3342253005903,-79.33404190203494,-79.33385684483385,-79.33367180694299,-79.33348582858623,-79.33329938312764,-79.32965532516366,-79.32930531050063,-79.32924370831658,-79.32905867897217,-79.32874152538807,-79.32840305565266,-79.32800984926074,-79.32769172212001,-79.32739561557871,-79.32662608734871,-79.32609430384296,-79.326008091853,-79.32557802374399,-79.32529552644301,-79.32597415241978,-79.32620329732451,-79.32733693096526,-79.32786682488511,-79.32898464432608,-79.3302113180209,-79.33141174508016,-79.33225043529637,-79.33280154698662,-79.33358182946627,-79.33415973769425,-79.33430541922672,-79.33451620963896,-79.33500726774687,-79.33531101524095,-79.33559640477056,-79.33657421182831,-79.33757095954498,-79.33890640364763,-79.34038759599871,-79.34042812572415,-79.34112839723501,-79.34229617015004,-79.34226974383465,-79.34220971535345,-79.34214978402119,-79.3420890371308,-79.34202675653795,-79.34196519147132,-79.34190207104982,-79.34183897213001,-79.3417744633914,-79.34170901023371,-79.34164378555236,-79.3415774950054,-79.34151063422389,-79.34144293371091,-79.3413752547354,-79.34130695970725,-79.3412202794229,-79.34113349666846,-79.34104540628474,-79.34095719172025,-79.34086815900406,-79.34080861850127,-79.34071478658001,-79.34060605796095,-79.34049651108428,-79.34038696386824,-79.34027612825288,-79.34016492274313,-79.34009112856936,-79.34005304476686,-79.33994137094311,-79.33982877662206,-79.33971474811311,-79.33960061950863,-79.33948577487884,-79.3393710321531,-79.33929511123473,-79.33925467259235,-79.33913888492017,-79.33902181148231,-79.33890402202383,-79.33880993773715,-79.33871996482894,-79.33871431960905,-79.33861810723361,-79.33852167075216,-79.33842441611806,-79.33832748992108,-79.33822894957063,-79.33813038709708,-79.33803087980672,-79.33793162317309,-79.33783142420326,-79.33773050932599,-79.33762890051472,-79.33755404301152,-79.33752645158066,-79.33742484237804,-79.33732179908738,-79.33721865584361,-79.33711397874745,-79.33700989289554,-79.33690511315407,-79.33679984403337,-79.33676282368644],"lat":[43.67198064522312,43.67201573622594,43.67206820269935,43.67212075012462,43.67217319406684,43.67223013858946,43.67228257190105,43.67233500510623,43.67241789335657,43.67249637358966,43.67257925082647,43.67265762857171,43.67273599410444,43.67281884959358,43.67289270490505,43.67297563636495,43.67304948064232,43.67309729968089,43.67312781039542,43.67320165406581,43.67327997426369,43.67335388345593,43.67343218961778,43.67350148665133,43.67493388346568,43.67407789707696,43.6739239623041,43.67346656549361,43.67268757477493,43.67185876022457,43.67091661857675,43.67017361847904,43.66944453275944,43.66759199471252,43.66630564989305,43.66608833859105,43.66505103659705,43.66438965947848,43.66421497079711,43.66416881604815,43.66390185033148,43.66380149057062,43.66352527504566,43.66323254503474,43.66299343075633,43.6627986265967,43.66266263544718,43.66449719628759,43.66585169435844,43.6661644776263,43.66673026619334,43.66789445137663,43.66854261080156,43.66924460412015,43.66901119546039,43.66879164698909,43.66849590515167,43.668166256277,43.66815784283413,43.66801047462845,43.66774826714828,43.6677793908917,43.66784594525504,43.66791268103565,43.66797931493211,43.66804592654444,43.66811254850312,43.66818364887144,43.66825024850546,43.66831232663664,43.66837898205686,43.66844104972368,43.66851210390508,43.66857414775121,43.66864068038038,43.66870271227103,43.66876482519941,43.66884458524458,43.66892434373513,43.66899958216639,43.66907931973537,43.66915904535684,43.66920778205127,43.66926493282526,43.66935787529191,43.66945080577096,43.66954373613815,43.66962773568825,43.66972514276406,43.66978258405108,43.66981353794364,43.66990193597746,43.66999032052806,43.6700787741499,43.67016713618164,43.67025548769361,43.67034384056912,43.67039674942104,43.67042775884295,43.67052059730482,43.67060441494291,43.67068822205179,43.6707543695895,43.67081607580751,43.67082049475125,43.67088220020486,43.67094831330623,43.67101441442961,43.6710760192368,43.67113309948108,43.67119468032568,43.67126083837183,43.67131790794011,43.67137946472299,43.67144101100373,43.67149804609248,43.67154196722676,43.67155956987594,43.67161660477577,43.67167370873972,43.6717307211354,43.67178771111799,43.67184921061485,43.67190619891065,43.67195867898,43.67198064522312]}]],[[{"lng":[-79.36646947934049,-79.36134490312664,-79.36115995294841,-79.3611226022395,-79.36107065515341,-79.36102966565109,-79.36167994434753,-79.36182587613639,-79.3653694258756,-79.36617680141491,-79.36638629793592,-79.36646947934049],"lat":[43.64723421595726,43.64836739829941,43.6480135761315,43.6479320220265,43.64782325242209,43.64773264404664,43.64670196043981,43.64652401120835,43.64518807288613,43.64656883109848,43.64704398378139,43.64723421595726]}]],[[{"lng":[-79.34229617015004,-79.34112839723501,-79.34042812572415,-79.34038759599871,-79.33890640364763,-79.33757095954498,-79.33657421182831,-79.33559640477056,-79.33531101524095,-79.33500726774687,-79.33451620963896,-79.33430541922672,-79.33415973769425,-79.33358182946627,-79.33280154698662,-79.33329173485384,-79.33405804066525,-79.33430299140431,-79.3348778630513,-79.33532212795494,-79.33588227268378,-79.33630285356134,-79.33769139776379,-79.33880722345877,-79.33890064344692,-79.34007962442661,-79.34078597764056,-79.3412550565607,-79.34249324479146,-79.34357815846563,-79.34444562516774,-79.34545785911178,-79.34566142005532,-79.34578981785214,-79.34572328025308,-79.34565379836947,-79.34560224765661,-79.34551638352274,-79.34542183112396,-79.34530466334367,-79.34518113881205,-79.34511990840019,-79.34503722584367,-79.34489298738785,-79.34489051243899,-79.34485731091713,-79.34482353875347,-79.34478880358643,-79.34475337412856,-79.3447172264862,-79.34468026337143,-79.34464248230815,-79.3446032913385,-79.34456397660321,-79.3445246592928,-79.34448403678199,-79.34444237040236,-79.3443998646049,-79.34435689043896,-79.34431371170392,-79.34429023099338,-79.34426920136542,-79.34422446746201,-79.34417750578785,-79.34410857508639,-79.34403776096433,-79.34396635216324,-79.3438949455559,-79.34382212889066,-79.3437483678663,-79.34367460902715,-79.34360013425723,-79.34352424943056,-79.34344825960824,-79.34337153479703,-79.34329388956806,-79.34326652266712,-79.3432155499321,-79.34313730976153,-79.34304422602416,-79.34295030247098,-79.3428555797454,-79.34275932558191,-79.34266317328103,-79.34263311001345,-79.34256607887995,-79.34246851363172,-79.34236919085399,-79.34229617015004],"lat":[43.66774826714828,43.66801047462845,43.66815784283413,43.668166256277,43.66849590515167,43.66879164698909,43.66901119546039,43.66924460412015,43.66854261080156,43.66789445137663,43.66673026619334,43.6661644776263,43.66585169435844,43.66449719628759,43.66266263544718,43.66256617221844,43.66241080817906,43.66232885944592,43.66222462164132,43.66210956651946,43.66200519914936,43.66190329836655,43.66159024912901,43.66134982634269,43.66133318019615,43.66105324379152,43.66087886961884,43.66078665670509,43.66052096770768,43.66028013988974,43.66008107203065,43.65985717748016,43.65981052319257,43.65978987671519,43.66023910527581,43.66068379012589,43.66111973245028,43.661568771028,43.66201759369735,43.66247058982013,43.66290548968159,43.66307564158915,43.66334909615192,43.66377469363626,43.66377915880613,43.66385969620185,43.66393122329421,43.66401173845543,43.66408774253506,43.6641638262205,43.66423980806135,43.66431577804124,43.66438722657114,43.66446317430723,43.66453921201168,43.66461063976035,43.66468655338464,43.66476695584513,43.6648383494768,43.66490974012758,43.66494540804461,43.66498120148874,43.66505707058513,43.66512840637172,43.66523993293202,43.66536043418255,43.66547651577658,43.66559250733492,43.66570397739882,43.66582002476007,43.66593598208252,43.66605192897797,43.66616335437172,43.66627486821091,43.6663907823343,43.66650668306043,43.66654229441855,43.66661807265229,43.66672955365028,43.66684523036745,43.66696539584035,43.66708113865345,43.66719676915694,43.66731240105387,43.66733447011737,43.66743252021475,43.66754372045926,43.66765930611574,43.66774826714828]}]],[[{"lng":[-79.3559510599049,-79.35576141680215,-79.35528026027235,-79.35516600316492,-79.35513960640995,-79.35471925882032,-79.3543265747284,-79.35398044452529,-79.353964782067,-79.35351251227262,-79.35276324373197,-79.3526105875702,-79.35245795225421,-79.3494449548419,-79.34852852596406,-79.34702116916931,-79.34518271302272,-79.34276533050642,-79.34229617015004,-79.34236919085399,-79.34246851363172,-79.34256607887995,-79.34263311001345,-79.34266317328103,-79.34275932558191,-79.3428555797454,-79.34295030247098,-79.34304422602416,-79.34313730976153,-79.3432155499321,-79.34326652266712,-79.34329388956806,-79.34337153479703,-79.34344825960824,-79.34352424943056,-79.34360013425723,-79.34367460902715,-79.3437483678663,-79.34382212889066,-79.3438949455559,-79.34396635216324,-79.34403776096433,-79.34410857508639,-79.34417750578785,-79.34422446746201,-79.34426920136542,-79.34429023099338,-79.34431371170392,-79.34435689043896,-79.3443998646049,-79.34444237040236,-79.34448403678199,-79.3445246592928,-79.34456397660321,-79.3446032913385,-79.34464248230815,-79.34468026337143,-79.3447172264862,-79.34475337412856,-79.34478880358643,-79.34482353875347,-79.34485731091713,-79.34489051243899,-79.34489298738785,-79.34503722584367,-79.34511990840019,-79.34518113881205,-79.34530466334367,-79.34542183112396,-79.34551638352274,-79.34560224765661,-79.34565379836947,-79.34572328025308,-79.34578981785214,-79.34610230824869,-79.34639809722597,-79.34673074982821,-79.34743424388323,-79.34848603767962,-79.34863894905524,-79.3498387817659,-79.35091660833449,-79.35155455177248,-79.35222561426546,-79.3533342720492,-79.35360988641591,-79.35374230601761,-79.35390755924684,-79.3541609282956,-79.35431156344174,-79.35433212810963,-79.3543823941651,-79.35441158207179,-79.35443758052804,-79.35446815593725,-79.35450593330778,-79.35454722412065,-79.35458959690413,-79.35463231918686,-79.35466383844346,-79.35469560003376,-79.35471736807864,-79.35471947344152,-79.35475606292404,-79.3547911374053,-79.35483535269864,-79.3548837191432,-79.35492866861102,-79.3549729051613,-79.35501233833023,-79.35501867528706,-79.35506278936516,-79.35510773963449,-79.35515714923018,-79.35519680724347,-79.35520604452823,-79.35524288167403,-79.35528100665637,-79.3553264071826,-79.35536473697189,-79.35540179895301,-79.35543143725914,-79.35543705891679,-79.35546649622799,-79.35550909961708,-79.35555638509184,-79.35559827081522,-79.35560049923801,-79.35565109767528,-79.35570416719987,-79.3557469966604,-79.35579180911219,-79.35582464101718,-79.35584224050481,-79.35586769289054,-79.35590134294094,-79.35594615615526,-79.35598442503054,-79.35603349185016,-79.35607364483329,-79.35611436775164,-79.35614366390909,-79.35618661811722,-79.35621697552935,-79.35625381667964,-79.35629090356707,-79.3559510599049],"lat":[43.66459595895621,43.66463824074562,43.66477543951495,43.66480980320268,43.66481392428967,43.66493390182205,43.66504077312353,43.66513940159915,43.66514367702208,43.66526769132221,43.6654774435777,43.66547974411984,43.66547754374485,43.66614540702916,43.6663483060325,43.66667309123497,43.66708767383143,43.66762895330388,43.66774826714828,43.66765930611574,43.66754372045926,43.66743252021475,43.66733447011737,43.66731240105387,43.66719676915694,43.66708113865345,43.66696539584035,43.66684523036745,43.66672955365028,43.66661807265229,43.66654229441855,43.66650668306043,43.6663907823343,43.66627486821091,43.66616335437172,43.66605192897797,43.66593598208252,43.66582002476007,43.66570397739882,43.66559250733492,43.66547651577658,43.66536043418255,43.66523993293202,43.66512840637172,43.66505707058513,43.66498120148874,43.66494540804461,43.66490974012758,43.6648383494768,43.66476695584513,43.66468655338464,43.66461063976035,43.66453921201168,43.66446317430723,43.66438722657114,43.66431577804124,43.66423980806135,43.6641638262205,43.66408774253506,43.66401173845543,43.66393122329421,43.66385969620185,43.66377915880613,43.66377469363626,43.66334909615192,43.66307564158915,43.66290548968159,43.66247058982013,43.66201759369735,43.661568771028,43.66111973245028,43.66068379012589,43.66023910527581,43.65978987671519,43.6597313852922,43.65965914830154,43.65957394078783,43.65941298709367,43.65918513667593,43.65914683716726,43.65887151209535,43.65862592104703,43.65848659112199,43.65832523006273,43.65808456170199,43.65801651631757,43.65798691678564,43.65795328901144,43.65789842493458,43.65786458605156,43.65790539135792,43.65802314167752,43.65814508948344,43.65827149237362,43.65840255219755,43.65852453362984,43.65866015871109,43.6587823863171,43.65891803197243,43.6590445142815,43.65918459314382,43.65929743199113,43.65930646433788,43.65944202164459,43.65957314612917,43.65971781522216,43.65984904099799,43.65998930962016,43.66012947793939,43.66024707201026,43.66026966831296,43.66041433584628,43.66055460439796,43.6607038491258,43.66083945042826,43.66087568153344,43.66100224018314,43.66113781938689,43.66129159736841,43.66142717948586,43.66157174537656,43.6617072924094,43.66172987839052,43.66184282741573,43.66198297198775,43.66212768492968,43.66226790915615,43.66227244223715,43.6624082007716,43.66256208891969,43.66269773562507,43.66284691388916,43.66297791570555,43.66304127314908,43.66313165964013,43.66326267319928,43.66341185140136,43.66353852038178,43.66367425668859,43.66379186068691,43.66391856491786,43.66404051373163,43.66417166102229,43.66428471306828,43.66441127138607,43.66452883119474,43.66459595895621]}]],[[{"lng":[-79.34229617015004,-79.34276533050642,-79.34518271302272,-79.34702116916931,-79.34852852596406,-79.3494449548419,-79.35245795225421,-79.3526105875702,-79.35276324373197,-79.35351251227262,-79.353964782067,-79.35398044452529,-79.3543265747284,-79.35471925882032,-79.35513960640995,-79.35516600316492,-79.35528026027235,-79.35576141680215,-79.3559510599049,-79.35629090356707,-79.35633457404334,-79.35636151714962,-79.35638623423965,-79.35641696263478,-79.35645016285002,-79.35648475581753,-79.35651877661375,-79.35656400221063,-79.35730746963991,-79.35733235373222,-79.35737157163402,-79.3574140000562,-79.35745195055354,-79.35749435627665,-79.35754338984674,-79.3575951227283,-79.35767280420464,-79.35773220535779,-79.35845889827084,-79.35931643335279,-79.35940704174995,-79.35949930711124,-79.35958026447116,-79.3596450455282,-79.35970056579791,-79.35973956351404,-79.35978187440283,-79.35982440777096,-79.359863774169,-79.35990995038408,-79.35995011467185,-79.35999242620454,-79.36003778243867,-79.36007739453449,-79.36011686390223,-79.36016471743687,-79.36021197793151,-79.36023852955583,-79.36006850004372,-79.35968762251183,-79.35944393766424,-79.35610415258961,-79.35536424796553,-79.35491874915415,-79.35451550144228,-79.35437246043884,-79.35424319321029,-79.35408378718253,-79.35392052008575,-79.35103778781044,-79.35024857407764,-79.34734688826978,-79.34718894147791,-79.34702770162502,-79.3448383671862,-79.34472743096876,-79.3445902934515,-79.34237978935217,-79.34203651252471,-79.34168150872502,-79.34125823302219,-79.34110003870495,-79.34100671279649,-79.34080861850127,-79.34086815900406,-79.34095719172025,-79.34104540628474,-79.34113349666846,-79.3412202794229,-79.34130695970725,-79.3413752547354,-79.34144293371091,-79.34151063422389,-79.3415774950054,-79.34164378555236,-79.34170901023371,-79.3417744633914,-79.34183897213001,-79.34190207104982,-79.34196519147132,-79.34202675653795,-79.3420890371308,-79.34214978402119,-79.34220971535345,-79.34226974383465,-79.34229617015004],"lat":[43.66774826714828,43.66762895330388,43.66708767383143,43.66667309123497,43.6663483060325,43.66614540702916,43.66547754374485,43.66547974411984,43.6654774435777,43.66526769132221,43.66514367702208,43.66513940159915,43.66504077312353,43.66493390182205,43.66481392428967,43.66480980320268,43.66477543951495,43.66463824074562,43.66459595895621,43.66452883119474,43.66465998872243,43.6647819936442,43.66489937550775,43.66502134483231,43.66513893870119,43.66526546459603,43.66538298022147,43.66550965899521,43.66818548796868,43.66826686387184,43.66836644984605,43.66845707992717,43.66854314461875,43.6686383653736,43.66871558730048,43.66878384597573,43.66885697854833,43.66892534736064,43.67037386103582,43.67005112031469,43.67011543526245,43.67019777796732,43.6702709562572,43.67037099838477,43.67046181551045,43.67054339344097,43.67063852194163,43.6707292426281,43.67081982782554,43.67091951276,43.67101461038651,43.67110973881214,43.67120950194379,43.67129108856702,43.67138167514771,43.6714723820202,43.67155857935212,43.6716014498674,43.67175285457731,43.6718906104535,43.67216617507907,43.67325090567075,43.67249291243201,43.67200030128468,43.67126065119606,43.6709929419143,43.67084254707918,43.67066471196683,43.67048222998371,43.67114309866096,43.6713182300927,43.67195865729317,43.67195187203528,43.67194953993569,43.67242655578612,43.67242494899555,43.6724499686047,43.67296263863246,43.67211579048933,43.67126427003485,43.67028122819119,43.66994576725021,43.66972836404366,43.66920778205127,43.66915904535684,43.66907931973537,43.66899958216639,43.66892434373513,43.66884458524458,43.66876482519941,43.66870271227103,43.66864068038038,43.66857414775121,43.66851210390508,43.66844104972368,43.66837898205686,43.66831232663664,43.66825024850546,43.66818364887144,43.66811254850312,43.66804592654444,43.66797931493211,43.66791268103565,43.66784594525504,43.6677793908917,43.66774826714828]}]],[[{"lng":[-79.3426797787574,-79.34298114171017,-79.3432953489139,-79.34359780779874,-79.34389564554093,-79.34421843765455,-79.34450629834946,-79.34482977129537,-79.343071600025,-79.34176563783426,-79.34059558079151,-79.34009984296128,-79.33984957134342,-79.33980283097415,-79.33965981565476,-79.33948159556529,-79.33885941365733,-79.33880483488372,-79.33813641769417,-79.3377925388239,-79.33717834757141,-79.3370666495068,-79.33676282368644,-79.33679984403337,-79.33690511315407,-79.33700989289554,-79.33711397874745,-79.33721865584361,-79.33732179908738,-79.33742484237804,-79.33752645158066,-79.33755404301152,-79.33762890051472,-79.33773050932599,-79.33783142420326,-79.33793162317309,-79.33803087980672,-79.33813038709708,-79.33822894957063,-79.33832748992108,-79.33842441611806,-79.33852167075216,-79.33861810723361,-79.33871431960905,-79.33871996482894,-79.33880993773715,-79.33890402202383,-79.33902181148231,-79.33913888492017,-79.33925467259235,-79.33929511123473,-79.3393710321531,-79.33948577487884,-79.33960061950863,-79.33971474811311,-79.33982877662206,-79.33994137094311,-79.34005304476686,-79.34009112856936,-79.34016492274313,-79.34027612825288,-79.34038696386824,-79.34049651108428,-79.34060605796095,-79.34071478658001,-79.34080861850127,-79.34100671279649,-79.34110003870495,-79.34125823302219,-79.34168150872502,-79.34203651252471,-79.34237978935217,-79.3426797787574],"lat":[43.67366932667891,43.67438953680154,43.67514134927861,43.67587057552263,43.67662673994953,43.67740568008431,43.67813460215526,43.67894514730003,43.67929784328477,43.67956255377818,43.67978412279493,43.67988053630346,43.67917015347753,43.6790659514057,43.67873971144177,43.67831393768579,43.67686880528623,43.67673739279057,43.67513404885941,43.67433669063134,43.67290507749176,43.6726829028509,43.67198064522312,43.67195867898,43.67190619891065,43.67184921061485,43.67178771111799,43.6717307211354,43.67167370873972,43.67161660477577,43.67155956987594,43.67154196722676,43.67149804609248,43.67144101100373,43.67137946472299,43.67131790794011,43.67126083837183,43.67119468032568,43.67113309948108,43.6710760192368,43.67101441442961,43.67094831330623,43.67088220020486,43.67082049475125,43.67081607580751,43.6707543695895,43.67068822205179,43.67060441494291,43.67052059730482,43.67042775884295,43.67039674942104,43.67034384056912,43.67025548769361,43.67016713618164,43.6700787741499,43.66999032052806,43.66990193597746,43.66981353794364,43.66978258405108,43.66972514276406,43.66962773568825,43.66954373613815,43.66945080577096,43.66935787529191,43.66926493282526,43.66920778205127,43.66972836404366,43.66994576725021,43.67028122819119,43.67126427003485,43.67211579048933,43.67296263863246,43.67366932667891]}]],[[{"lng":[-79.33676282368644,-79.3370666495068,-79.33717834757141,-79.3377925388239,-79.33813641769417,-79.33880483488372,-79.33885941365733,-79.33948159556529,-79.33965981565476,-79.33980283097415,-79.33984957134342,-79.34009984296128,-79.33942264728459,-79.33780620522525,-79.33730293050941,-79.3353544256495,-79.33484830126579,-79.33236985887879,-79.33196729324783,-79.33183342592139,-79.33162235341648,-79.33155767982394,-79.33125586468468,-79.33094783457064,-79.33025900556883,-79.3296640056632,-79.32965532516366,-79.33329938312764,-79.33348582858623,-79.33367180694299,-79.33385684483385,-79.33404190203494,-79.3342253005903,-79.33429365428198,-79.33440974338109,-79.33459314103392,-79.33477689185349,-79.33495957287174,-79.33514240002258,-79.33532448880796,-79.33550575916384,-79.33568715342166,-79.33586760735889,-79.33604828492976,-79.33615672493745,-79.336265164758,-79.33637291075102,-79.33648063447869,-79.33658744030737,-79.33669362999994,-79.33676282368644],"lat":[43.67198064522312,43.6726829028509,43.67290507749176,43.67433669063134,43.67513404885941,43.67673739279057,43.67686880528623,43.67831393768579,43.67873971144177,43.6790659514057,43.67917015347753,43.67988053630346,43.68000581804647,43.68030648019069,43.6804161848959,43.68083800844744,43.68094315980314,43.68147882636914,43.6804824610182,43.68013392992781,43.6796625644811,43.67944557222102,43.67875683311158,43.67798257334714,43.67636988379268,43.67495651536428,43.67493388346568,43.67350148665133,43.67343218961778,43.67335388345593,43.67327997426369,43.67320165406581,43.67312781039542,43.67309729968089,43.67304948064232,43.67297563636495,43.67289270490505,43.67281884959358,43.67273599410444,43.67265762857171,43.67257925082647,43.67249637358966,43.67241789335657,43.67233500510623,43.67228257190105,43.67223013858946,43.67217319406684,43.67212075012462,43.67206820269935,43.67201573622594,43.67198064522312]}]],[[{"lng":[-79.33484830126579,-79.3353544256495,-79.33730293050941,-79.33763923816331,-79.33798624307119,-79.33834744066027,-79.33878987435027,-79.33764323070227,-79.33673851835806,-79.33585689290004,-79.33491600303863,-79.33389187792663,-79.3337461165866,-79.33340649339345,-79.33270309051935,-79.33236985887879,-79.33484830126579],"lat":[43.68094315980314,43.68083800844744,43.6804161848959,43.68124043972681,43.68209626602859,43.68291637956673,43.68402573671646,43.68427021054801,43.68445968134071,43.68465389213173,43.68485632461706,43.6850754498031,43.68470865376094,43.68392043762715,43.68227154362287,43.68147882636914,43.68094315980314]}]],[[{"lng":[-79.33730293050941,-79.33780620522525,-79.33942264728459,-79.34009984296128,-79.34059558079151,-79.34176563783426,-79.343071600025,-79.34482977129537,-79.34514331044595,-79.34543011972049,-79.34606161463569,-79.34611299895323,-79.34626083339015,-79.34543455092476,-79.34444602883057,-79.34315273548157,-79.34198260974131,-79.34079535154453,-79.33883871778798,-79.33878987435027,-79.33834744066027,-79.33798624307119,-79.33763923816331,-79.33730293050941],"lat":[43.6804161848959,43.68030648019069,43.68000581804647,43.67988053630346,43.67978412279493,43.67956255377818,43.67929784328477,43.67894514730003,43.67967452954122,43.68043503085752,43.68217741777766,43.68234019747334,43.68271600997318,43.68287517771837,43.68306340618964,43.68331481280983,43.68353639480883,43.68376680807102,43.6841390617105,43.68402573671646,43.68291637956673,43.68209626602859,43.68124043972681,43.6804161848959]}]],[[{"lng":[-79.34610373714598,-79.34482977129537,-79.34450629834946,-79.34421843765455,-79.34389564554093,-79.34359780779874,-79.3432953489139,-79.34298114171017,-79.3426797787574,-79.34237978935217,-79.3445902934515,-79.34472743096876,-79.3448383671862,-79.34702770162502,-79.34718894147791,-79.34734688826978,-79.35024857407764,-79.35064269405321,-79.35082577952637,-79.3511249835649,-79.35149681323549,-79.3518408093237,-79.35189353547662,-79.35216412248069,-79.35277595720959,-79.35195258551131,-79.35073732155037,-79.34971848430177,-79.34961019297367,-79.34849624749488,-79.34847043676957,-79.34829735331496,-79.34733222987262,-79.34625338497901,-79.34610373714598],"lat":[43.67869353167186,43.67894514730003,43.67813460215526,43.67740568008431,43.67662673994953,43.67587057552263,43.67514134927861,43.67438953680154,43.67366932667891,43.67296263863246,43.6724499686047,43.67242494899555,43.67242655578612,43.67194953993569,43.67195187203528,43.67195865729317,43.6713182300927,43.67215021414378,43.67259854624746,43.6733412090221,43.67425595686662,43.67508928384757,43.67522957546225,43.67588630341363,43.67738072591926,43.67754898505299,43.67778801022016,43.67798493914349,43.67800588029739,43.67822392974842,43.67823255874891,43.67826606528918,43.67845465718766,43.67866419032658,43.67869353167186]}]],[[{"lng":[-79.36068796594225,-79.360776718187,-79.36086344595735,-79.36095939628419,-79.36105753553736,-79.36115596039873,-79.36124589655047,-79.36135254311228,-79.36145176631665,-79.36155309651862,-79.3616534031522,-79.36174581676133,-79.3618391709188,-79.36194268799622,-79.36203438388249,-79.36208301318545,-79.36215638630443,-79.36222526212019,-79.36231417933743,-79.36243301232199,-79.36256254111998,-79.36267806387343,-79.36277681824069,-79.36285037968831,-79.36293807154865,-79.36299701124649,-79.36302910250025,-79.36304362724302,-79.36304226338829,-79.3630318195093,-79.36303120785787,-79.36302973989153,-79.36301434542207,-79.36299539655761,-79.36297528090049,-79.36295142214701,-79.36292883229652,-79.36291611792704,-79.36288882446894,-79.36286633665834,-79.36286683488123,-79.36287301713736,-79.36289764216127,-79.3629363199376,-79.36297217499042,-79.36301310066492,-79.36308938318966,-79.36337778243815,-79.36335570988037,-79.36043505464673,-79.35940851000056,-79.35885912993025,-79.35635778819892,-79.35504758394408,-79.35404419370954,-79.35324500469504,-79.35277595720959,-79.35216412248069,-79.35189353547662,-79.3518408093237,-79.35149681323549,-79.3511249835649,-79.35082577952637,-79.35064269405321,-79.35024857407764,-79.35103778781044,-79.35392052008575,-79.35408378718253,-79.35424319321029,-79.35437246043884,-79.35451550144228,-79.35491874915415,-79.35536424796553,-79.35610415258961,-79.35944393766424,-79.35968762251183,-79.36006850004372,-79.36023852955583,-79.36026547304256,-79.36033134239747,-79.36038911342784,-79.36044344968278,-79.36051261166369,-79.36060052283202,-79.36068796594225],"lat":[43.67210120843291,43.67216999666306,43.67223425477695,43.67229414408791,43.67236756773881,43.67242299136279,43.67247838319603,43.67254742757537,43.67260736340035,43.67267633136568,43.67272277952014,43.67277361544622,43.67281996377046,43.67285745567233,43.67290837109704,43.67291807010316,43.67292812371507,43.67296061787412,43.67297080412922,43.67299051103242,43.67301937308331,43.67307053936592,43.6731214653024,43.67318103226697,43.67323630059347,43.67331825307102,43.67339072890032,43.67345845208855,43.67353494970369,43.67360681624568,43.67362931252826,43.6736833036061,43.67376419126895,43.67386294202812,43.67394817304293,43.67402434841664,43.67411404498673,43.67421297505646,43.67430260423557,43.67439230225079,43.67448683055999,43.67456793727071,43.67464039618493,43.67472646940377,43.67480350015794,43.67486710050991,43.67491320326123,43.67531801354768,43.67530590480072,43.67589715134759,43.67609396626533,43.67619410223277,43.67668054045525,43.67693625005737,43.67713794098763,43.67729746572389,43.67738072591926,43.67588630341363,43.67522957546225,43.67508928384757,43.67425595686662,43.6733412090221,43.67259854624746,43.67215021414378,43.6713182300927,43.67114309866096,43.67048222998371,43.67066471196683,43.67084254707918,43.6709929419143,43.67126065119606,43.67200030128468,43.67249291243201,43.67325090567075,43.67216617507907,43.6718906104535,43.67175285457731,43.6716014498674,43.67164495614023,43.67173142042557,43.67179976444415,43.67188156218433,43.67196357259142,43.67203693992307,43.67210120843291]}]],[[{"lng":[-79.36335570988037,-79.36337778243815,-79.36345547925156,-79.36374734517432,-79.36401680459875,-79.36431831421774,-79.36467274428398,-79.36494304976164,-79.36481838980434,-79.36443002936534,-79.3640937184214,-79.36382481482792,-79.36358163282968,-79.36358573491998,-79.36371837176634,-79.36403543454188,-79.36417183725356,-79.36267958963852,-79.36242544063192,-79.36226636991839,-79.36168018834385,-79.36232942201133,-79.36237033248423,-79.36202853533827,-79.36195358286714,-79.36175428872279,-79.36140561455241,-79.36127782649501,-79.36061940462162,-79.36031278140753,-79.3575625537791,-79.35742670912568,-79.35734904477806,-79.3572026548741,-79.35517815120492,-79.35500886933669,-79.35495083314031,-79.35483473181846,-79.35473451184356,-79.35437449287132,-79.35365528785262,-79.35346114446081,-79.35343019324756,-79.35343015883259,-79.35344284389707,-79.35345968335483,-79.35344971538775,-79.35324500469504,-79.35404419370954,-79.35504758394408,-79.35635778819892,-79.35885912993025,-79.35940851000056,-79.36043505464673,-79.36335570988037],"lat":[43.67530590480072,43.67531801354768,43.67539114235202,43.6756384660681,43.67593497935147,43.67622285869209,43.67645307145443,43.67672249857164,43.67699986736297,43.67722395115749,43.67747119372747,43.67778250450762,43.678058174912,43.67842281509709,43.67878938574773,43.67907757843423,43.67919655715278,43.67949484826756,43.67954521875017,43.67957894725479,43.67969216232169,43.68125458432502,43.68168735689472,43.68173647041988,43.68187051619454,43.6817280382775,43.68169153229924,43.68165819290972,43.68168475746555,43.68164435038191,43.68169487666056,43.68170192619478,43.68195745701375,43.68243704992545,43.68283111310772,43.68241449433823,43.68226962686464,43.68183126832064,43.68158668136893,43.68068561809438,43.6787980707379,43.67830007318995,43.67818251110402,43.67810149252742,43.67803416022917,43.67793087974858,43.67780911898245,43.67729746572389,43.67713794098763,43.67693625005737,43.67668054045525,43.67619410223277,43.67609396626533,43.67589715134759,43.67530590480072]}]],[[{"lng":[-79.370211350007,-79.36646947934049,-79.36638629793592,-79.36617680141491,-79.3653694258756,-79.36898666219723,-79.36937947968696,-79.36943957155121,-79.36949126371005,-79.36952779142362,-79.36981041679637,-79.36984213739673,-79.369888533645,-79.36994069947228,-79.36999922622907,-79.370211350007],"lat":[43.64641876043878,43.64723421595726,43.64704398378139,43.64656883109848,43.64518807288613,43.64393418432234,43.6445116006543,43.64460697966937,43.64470673984398,43.6448152857096,43.64553966235385,43.64562563438684,43.64572090781763,43.64582967658902,43.6459565401769,43.64641876043878]}]],[[{"lng":[-79.35344284389707,-79.35343015883259,-79.35343019324756,-79.35346114446081,-79.35365528785262,-79.35437449287132,-79.35473451184356,-79.35483473181846,-79.35495083314031,-79.35500886933669,-79.35517815120492,-79.35279293018228,-79.35142781135731,-79.34938533884227,-79.34873898026426,-79.34857454278057,-79.34821641448579,-79.34811010302842,-79.34802472970593,-79.3474978721956,-79.34626083339015,-79.34611299895323,-79.34606161463569,-79.34543011972049,-79.34514331044595,-79.34482977129537,-79.34610373714598,-79.34625338497901,-79.34733222987262,-79.34829735331496,-79.34847043676957,-79.34849624749488,-79.34961019297367,-79.34971848430177,-79.35073732155037,-79.35195258551131,-79.35277595720959,-79.35324500469504,-79.35344971538775,-79.35345968335483,-79.35344284389707],"lat":[43.67803416022917,43.67810149252742,43.67818251110402,43.67830007318995,43.6787980707379,43.68068561809438,43.68158668136893,43.68183126832064,43.68226962686464,43.68241449433823,43.68283111310772,43.68328745653525,43.68355142457964,43.68394512567987,43.68407531892575,43.68368126576041,43.6828298118405,43.68258072012384,43.68238135196702,43.68248184867164,43.68271600997318,43.68234019747334,43.68217741777766,43.68043503085752,43.67967452954122,43.67894514730003,43.67869353167186,43.67866419032658,43.67845465718766,43.67826606528918,43.67823255874891,43.67822392974842,43.67800588029739,43.67798493914349,43.67778801022016,43.67754898505299,43.67738072591926,43.67729746572389,43.67780911898245,43.67793087974858,43.67803416022917]}]],[[{"lng":[-79.34167791264491,-79.34097399018543,-79.34081666652257,-79.34052793669443,-79.34024706389592,-79.33981900201415,-79.3395153752485,-79.33920510338676,-79.33900168244035,-79.33883871778798,-79.34079535154453,-79.34198260974131,-79.34315273548157,-79.34444602883057,-79.34543455092476,-79.34626083339015,-79.34643705963597,-79.34659831157981,-79.34677150053692,-79.34691199679888,-79.34719088342173,-79.34742616284012,-79.34775561162826,-79.34809370297059,-79.34845787328928,-79.34880255381121,-79.34915598088106,-79.34953887489088,-79.34990479994865,-79.35027860530538,-79.35035077585781,-79.35101557935661,-79.35102144873275,-79.35102774249856,-79.3510435219137,-79.35103833055531,-79.35099605722905,-79.35092561336452,-79.35031107050739,-79.34952148134519,-79.3494411545163,-79.34937330651891,-79.3493164242562,-79.34928580852761,-79.34926439612185,-79.34925477722986,-79.35055083705214,-79.35039840641095,-79.35011189620666,-79.34974709071315,-79.34936945664376,-79.34903217373791,-79.34869902857494,-79.34835713926051,-79.34808562418425,-79.34785916567867,-79.34761622517748,-79.34733211608932,-79.34695532710381,-79.3465947023672,-79.34629604236811,-79.34601911936538,-79.34580321951184,-79.34545220798321,-79.34504341049337,-79.34463527351771,-79.34453135399198,-79.34347091251497,-79.34340786348362,-79.34338009113192,-79.34335773987041,-79.34330612532784,-79.34325405100178,-79.34310417332586,-79.34271879118127,-79.34244427848066,-79.34233210715649,-79.34193010888885,-79.34167791264491],"lat":[43.69116719413435,43.68941473210095,43.68901636045685,43.68831883468177,43.68762601323956,43.68657979502778,43.68583713008001,43.68505376867259,43.68452860636894,43.6841390617105,43.68376680807102,43.68353639480883,43.68331481280983,43.68306340618964,43.68287517771837,43.68271600997318,43.68318675408729,43.68358076425037,43.68402445784896,43.68441816725294,43.68513344901812,43.68577158246051,43.68665422081772,43.68752798104936,43.68844721705457,43.68942459309611,43.69039309232229,43.69136201590532,43.69233078306893,43.69323646874372,43.69337704138043,43.69509737354068,43.6951109612104,43.69516506393066,43.69526881443463,43.69537235235583,43.69544375822643,43.69552375954858,43.69604159566489,43.69669202010603,43.69676287577093,43.69683391162627,43.69690060484761,43.69697217845363,43.69704397501459,43.69719686973662,43.69940351536587,43.69964445848858,43.69985636970551,43.6999231179311,43.69997626674498,43.69999840034453,43.69998908582205,43.69992554206996,43.69975508092023,43.69955376362115,43.69933411357167,43.69914546456063,43.69900039408662,43.69894116504931,43.69880181379995,43.69863117944272,43.69847051976853,43.69834841243529,43.69828848101633,43.69753522532169,43.69734017737221,43.6954565391955,43.69537910838077,43.69534719884051,43.69531536788242,43.69524260376505,43.69513823600062,43.6947173808051,43.69373039623795,43.69302407996992,43.69276139531756,43.69181017472992,43.69116719413435]}]],[[{"lng":[-79.3575625537791,-79.36031278140753,-79.36061940462162,-79.36127782649501,-79.36140561455241,-79.36175428872279,-79.36195358286714,-79.36202853533827,-79.36237033248423,-79.36232942201133,-79.36168018834385,-79.36226636991839,-79.36242544063192,-79.36267958963852,-79.36417183725356,-79.36441819464505,-79.3648559965048,-79.36526549468546,-79.36561500495883,-79.36569092981189,-79.36609551078142,-79.36633114396112,-79.36635530322829,-79.36615272148327,-79.36581751705381,-79.36539577490933,-79.36501957646948,-79.36464024217135,-79.36424020560459,-79.36377685867727,-79.36337744468409,-79.36326555437661,-79.36306169256764,-79.36277166915845,-79.36242064879501,-79.36208034678664,-79.36183717487438,-79.36169516422801,-79.36164004180952,-79.36172111968601,-79.36192461695978,-79.36216346105903,-79.36236865829338,-79.36251407223656,-79.36254985020291,-79.36235287165863,-79.36215514375191,-79.36196038100495,-79.36185696458919,-79.36166214221382,-79.36138061721206,-79.36117368889273,-79.36092351448403,-79.36068524248424,-79.3604827039153,-79.3602575770486,-79.36004161280533,-79.35992461127411,-79.36005040211745,-79.3603468719821,-79.36042599715329,-79.36085343109922,-79.36124137144846,-79.36153699771741,-79.36175037137161,-79.36183833235808,-79.36165546505356,-79.36131125326351,-79.36096809846885,-79.36063768827809,-79.36031627218415,-79.3599835436756,-79.35983551770576,-79.35961638079993,-79.35922386995709,-79.35882217509003,-79.35844717694459,-79.35815930875381,-79.35804951641344,-79.35764849939393,-79.35729003117613,-79.35691517121867,-79.35659760928931,-79.35641243991682,-79.35618944522294,-79.35607812247903,-79.35603146276917,-79.3559894895501,-79.35580656346774,-79.35553572975779,-79.35530130993406,-79.35500782456783,-79.35464597287026,-79.3542813050162,-79.35388186346573,-79.35351328421491,-79.3531368865531,-79.35273714432529,-79.35234714632561,-79.35190193446688,-79.35146148267332,-79.35108542473907,-79.35082555546835,-79.35058826397386,-79.35055083705214,-79.34925477722986,-79.34926439612185,-79.34928580852761,-79.3493164242562,-79.34937330651891,-79.3494411545163,-79.34952148134519,-79.35031107050739,-79.35092561336452,-79.35099605722905,-79.35103833055531,-79.3510435219137,-79.35102774249856,-79.35102144873275,-79.35101557935661,-79.35035077585781,-79.35027860530538,-79.34990479994865,-79.34953887489088,-79.34915598088106,-79.34880255381121,-79.34845787328928,-79.34809370297059,-79.34775561162826,-79.34742616284012,-79.34719088342173,-79.34691199679888,-79.34677150053692,-79.34659831157981,-79.34643705963597,-79.34626083339015,-79.3474978721956,-79.34802472970593,-79.34811010302842,-79.34821641448579,-79.34857454278057,-79.34873898026426,-79.34938533884227,-79.35142781135731,-79.35279293018228,-79.35517815120492,-79.3572026548741,-79.35734904477806,-79.35742670912568,-79.3575625537791],"lat":[43.68169487666056,43.68164435038191,43.68168475746555,43.68165819290972,43.68169153229924,43.6817280382775,43.68187051619454,43.68173647041988,43.68168735689472,43.68125458432502,43.67969216232169,43.67957894725479,43.67954521875017,43.67949484826756,43.67919655715278,43.67940271855844,43.67972405393263,43.68004057176091,43.68030221814769,43.68036181703096,43.68068267252573,43.68102811754212,43.68139313422157,43.68171440035118,43.6819842586473,43.68223478413976,43.68249955268814,43.68273276809005,43.68292958785591,43.68305798473027,43.68320538937036,43.68329830793436,43.68346642604227,43.68378193247259,43.68406055608276,43.68436624820607,43.68472293271001,43.68509907097698,43.68553046675743,43.68594581111693,43.68638541523102,43.68681202255613,43.68719313720525,43.68760940283583,43.68800159237711,43.68834093564237,43.68845962889176,43.6885783642977,43.68867149286716,43.68884873987285,43.68922287725163,43.68961158669847,43.68997266949275,43.69034742544444,43.69066859156646,43.69096251724081,43.69128808091459,43.69162406648174,43.69199045234874,43.69218383831638,43.69223898552257,43.69249276202092,43.69277288652146,43.69313279408634,43.69347351776508,43.69383935924663,43.69417440224754,43.6944485284863,43.69469125189828,43.69494306906151,43.69512758943289,43.69530285453862,43.69535024099493,43.69542370249207,43.69551717936325,43.69553841708802,43.6954700170467,43.69538477363943,43.69534718822847,43.69526490873946,43.69529576429759,43.69542099390694,43.69563247512782,43.69588195763588,43.6961488096845,43.69642185880991,43.69673634699226,43.69705531353164,43.69730482726305,43.69753957273354,43.69779725656997,43.69804067673121,43.69822450890965,43.698286772649,43.69828101947082,43.69834781542236,43.69846841936603,43.69859318654869,43.69864607699098,43.69867116387454,43.69872791463597,43.69886201967054,43.69908782008732,43.69934104173668,43.69940351536587,43.69719686973662,43.69704397501459,43.69697217845363,43.69690060484761,43.69683391162627,43.69676287577093,43.69669202010603,43.69604159566489,43.69552375954858,43.69544375822643,43.69537235235583,43.69526881443463,43.69516506393066,43.6951109612104,43.69509737354068,43.69337704138043,43.69323646874372,43.69233078306893,43.69136201590532,43.69039309232229,43.68942459309611,43.68844721705457,43.68752798104936,43.68665422081772,43.68577158246051,43.68513344901812,43.68441816725294,43.68402445784896,43.68358076425037,43.68318675408729,43.68271600997318,43.68248184867164,43.68238135196702,43.68258072012384,43.6828298118405,43.68368126576041,43.68407531892575,43.68394512567987,43.68355142457964,43.68328745653525,43.68283111310772,43.68243704992545,43.68195745701375,43.68170192619478,43.68169487666056]}]],[[{"lng":[-79.37080394319386,-79.37006156010996,-79.36930286455539,-79.36686565007028,-79.36351138403131,-79.35889216779633,-79.35882925552387,-79.3587502214243,-79.35864172607705,-79.35853050983648,-79.35850282854788,-79.35845266260444,-79.35839502906624,-79.35826719990628,-79.35816514941425,-79.35809867747288,-79.35792145997146,-79.35769980499491,-79.35744197844829,-79.35731967361757,-79.35720195156095,-79.35702270847202,-79.35684393562467,-79.35665276427785,-79.35645891343768,-79.35627752106561,-79.35611263955803,-79.35595540948898,-79.35582399931431,-79.35571049090277,-79.35552439238911,-79.35504292190006,-79.35439272430952,-79.35396731437463,-79.35362848865275,-79.35335736422563,-79.35313974299604,-79.35283014167707,-79.35276666010454,-79.35270070257519,-79.35259681671469,-79.35247492702787,-79.35235772467288,-79.35224316128462,-79.3521676514316,-79.35205087993229,-79.35193760775067,-79.35181928411501,-79.35173218149157,-79.35166119469143,-79.35158411487002,-79.35152021019564,-79.35147148445428,-79.35141725875987,-79.35090421363206,-79.35055083705214,-79.35058826397386,-79.35082555546835,-79.35108542473907,-79.35146148267332,-79.35190193446688,-79.35234714632561,-79.35273714432529,-79.3531368865531,-79.35351328421491,-79.35388186346573,-79.3542813050162,-79.35464597287026,-79.35500782456783,-79.35530130993406,-79.35553572975779,-79.35580656346774,-79.3559894895501,-79.35603146276917,-79.35607812247903,-79.35618944522294,-79.35641243991682,-79.35659760928931,-79.35691517121867,-79.35729003117613,-79.35764849939393,-79.35804951641344,-79.35815930875381,-79.35844717694459,-79.35882217509003,-79.35922386995709,-79.35961638079993,-79.35983551770576,-79.3599835436756,-79.36031627218415,-79.36063768827809,-79.36096809846885,-79.36131125326351,-79.36165546505356,-79.36183833235808,-79.36175037137161,-79.36153699771741,-79.36124137144846,-79.36085343109922,-79.36042599715329,-79.3603468719821,-79.36005040211745,-79.35992461127411,-79.36004161280533,-79.3602575770486,-79.3604827039153,-79.36068524248424,-79.36092351448403,-79.36117368889273,-79.36138061721206,-79.36166214221382,-79.36185696458919,-79.36196038100495,-79.36215514375191,-79.36235287165863,-79.36254985020291,-79.36251407223656,-79.36236865829338,-79.36216346105903,-79.36192461695978,-79.36172111968601,-79.36164004180952,-79.36169516422801,-79.36183717487438,-79.36208034678664,-79.36242064879501,-79.36277166915845,-79.36306169256764,-79.36326555437661,-79.36337744468409,-79.36377685867727,-79.36424020560459,-79.36464024217135,-79.36501957646948,-79.36539577490933,-79.36581751705381,-79.36615272148327,-79.36635530322829,-79.36633114396112,-79.36609551078142,-79.36569092981189,-79.36561500495883,-79.36526549468546,-79.3648559965048,-79.36441819464505,-79.36417183725356,-79.36425126937121,-79.36424750583684,-79.36443177832872,-79.36653556145096,-79.36675317757052,-79.36725054075887,-79.36717334271309,-79.36724241979742,-79.36750715500469,-79.367586071728,-79.36769425174806,-79.36787070719954,-79.36791512770988,-79.36793608878784,-79.36791350333375,-79.36784084817225,-79.36646146345639,-79.36753435341596,-79.36727862953319,-79.36727146368109,-79.3672693355162,-79.36728428457427,-79.36731582755988,-79.36735718571741,-79.3673968821206,-79.36746155918449,-79.36751790601159,-79.36761474081285,-79.36771157265936,-79.36785917990842,-79.3680082162705,-79.36820143403672,-79.36833849826516,-79.3685638938437,-79.36889200087877,-79.36914188219367,-79.36948909693776,-79.36997449186198,-79.37058144600417,-79.37066655909373,-79.37080394319386],"lat":[43.69093322840079,43.69137769299127,43.69213293876918,43.69434800190218,43.69756431660585,43.70207612012263,43.70205271168277,43.70202006968812,43.7019870044846,43.70194489811911,43.7019354985213,43.70192127494231,43.70190694406683,43.70189610570743,43.70188563757971,43.70188018154527,43.70186413219101,43.70185644591501,43.70183923731968,43.70182847730501,43.7018222840283,43.70182420811113,43.70183514064954,43.70185039559822,43.70184760773902,43.70187200467838,43.70188763686996,43.70190787989027,43.70191949221524,43.70193136197642,43.70195569000234,43.70200727322277,43.70204292183541,43.70205029792766,43.70204991792116,43.70203250852101,43.70199336460912,43.70191238562611,43.70187996366924,43.70185200697278,43.70179649754144,43.70174072845212,43.70168943778314,43.70162027111609,43.70157867341185,43.7015004727358,43.70141782141921,43.70132159417049,43.70124373105503,43.7011752823479,43.70109765370952,43.70102471606548,43.70096550031997,43.70087919928203,43.69998492203153,43.69940351536587,43.69934104173668,43.69908782008732,43.69886201967054,43.69872791463597,43.69867116387454,43.69864607699098,43.69859318654869,43.69846841936603,43.69834781542236,43.69828101947082,43.698286772649,43.69822450890965,43.69804067673121,43.69779725656997,43.69753957273354,43.69730482726305,43.69705531353164,43.69673634699226,43.69642185880991,43.6961488096845,43.69588195763588,43.69563247512782,43.69542099390694,43.69529576429759,43.69526490873946,43.69534718822847,43.69538477363943,43.6954700170467,43.69553841708802,43.69551717936325,43.69542370249207,43.69535024099493,43.69530285453862,43.69512758943289,43.69494306906151,43.69469125189828,43.6944485284863,43.69417440224754,43.69383935924663,43.69347351776508,43.69313279408634,43.69277288652146,43.69249276202092,43.69223898552257,43.69218383831638,43.69199045234874,43.69162406648174,43.69128808091459,43.69096251724081,43.69066859156646,43.69034742544444,43.68997266949275,43.68961158669847,43.68922287725163,43.68884873987285,43.68867149286716,43.6885783642977,43.68845962889176,43.68834093564237,43.68800159237711,43.68760940283583,43.68719313720525,43.68681202255613,43.68638541523102,43.68594581111693,43.68553046675743,43.68509907097698,43.68472293271001,43.68436624820607,43.68406055608276,43.68378193247259,43.68346642604227,43.68329830793436,43.68320538937036,43.68305798473027,43.68292958785591,43.68273276809005,43.68249955268814,43.68223478413976,43.6819842586473,43.68171440035118,43.68139313422157,43.68102811754212,43.68068267252573,43.68036181703096,43.68030221814769,43.68004057176091,43.67972405393263,43.67940271855844,43.67919655715278,43.67917969012104,43.67917513524635,43.67901573652897,43.67855062593285,43.67850422518018,43.67911905801586,43.67916296508681,43.67924497013433,43.67957732558059,43.67981709616546,43.68014721474457,43.68067203155506,43.6809202211837,43.68117716778251,43.68126686530453,43.68148646685052,43.68405537440415,43.68655570817277,43.68797923173535,43.68807365032896,43.68820864988407,43.68834398344395,43.68849746805106,43.68862858788018,43.6887417699784,43.68887322290289,43.68896854880456,43.68911396400808,43.68923687409482,43.68939651727803,43.68953367568775,43.68967155480653,43.68976353185137,43.68988827677319,43.69005508631442,43.69017117758546,43.69032016408381,43.69049371558138,43.69070491628084,43.69079614977132,43.69093322840079]}]],[[{"lng":[-79.38006860933145,-79.37993730941297,-79.3797117286755,-79.37935983231648,-79.37883653546575,-79.37852308913595,-79.37633441390129,-79.37443755303239,-79.37380537406389,-79.37320923376437,-79.37260913774887,-79.37202730541564,-79.37172560939625,-79.37136781522534,-79.37077663409754,-79.37038439306315,-79.36931791491106,-79.3681851468687,-79.36726062735953,-79.36718943463632,-79.36708629340471,-79.36698806494353,-79.366904688104,-79.36681764849399,-79.36558971544983,-79.36527400850247,-79.36461453861941,-79.36416990417391,-79.36352425059961,-79.3624083664622,-79.36156263170665,-79.36147191806197,-79.36131313748773,-79.36114234429124,-79.36098689668012,-79.36080826669523,-79.36068739280078,-79.36054652677778,-79.36044093428437,-79.3603601782804,-79.36025810333483,-79.36015318518686,-79.36006146369752,-79.35998633784256,-79.35990899992086,-79.35982491055815,-79.35971172820148,-79.35959547716253,-79.35947934907642,-79.35936874541585,-79.35929114288791,-79.3592284782175,-79.35915402660169,-79.35910338971354,-79.35901504634633,-79.35889216779633,-79.36351138403131,-79.36686565007028,-79.36930286455539,-79.37006156010996,-79.37080394319386,-79.37337599428938,-79.37376466728151,-79.3740858810816,-79.3744375979351,-79.37479289018133,-79.37638152193581,-79.37672161718838,-79.37728161249993,-79.37752932932358,-79.37995026655695,-79.38006860933145],"lat":[43.69502203030984,43.69506067795322,43.69511599269545,43.69519202066935,43.6953062151278,43.69536028100035,43.69582440431555,43.69621612008524,43.69636015950475,43.69649579648655,43.69659977704589,43.69669051165256,43.69674472623215,43.69678463755631,43.69690682951621,43.6969868456006,43.69721018493521,43.69745516270135,43.69765800484463,43.69767958262437,43.69773212076833,43.6978072339805,43.69787805835745,43.69794432933438,43.69883615392351,43.69909278562623,43.69958755088832,43.69996386108513,43.70052183158308,43.70170337926928,43.70245210358104,43.70253632169099,43.70267357552998,43.70281974886239,43.70294354707669,43.70303550563409,43.70309228455702,43.70313536359324,43.70302573494543,43.70294355850279,43.70284757313542,43.70276955082523,43.70269171780129,43.7026320368055,43.70256341205696,43.70249469035326,43.70243455296957,43.70237437142299,43.7023096905353,43.70226759378075,43.70223497271359,43.70220247605372,43.70217440112194,43.70215116909132,43.70211839360638,43.70207612012263,43.69756431660585,43.69434800190218,43.69213293876918,43.69137769299127,43.69093322840079,43.69233393993964,43.69252851349134,43.69266370301302,43.69279473451561,43.69290331080218,43.69338499423761,43.69334031474908,43.69354181077945,43.69364443997458,43.69472769798193,43.69502203030984]}]],[[{"lng":[-79.49286649301884,-79.49286526013421,-79.49180510165861,-79.49035322889472,-79.48672479576177,-79.48524023368216,-79.48425196425443,-79.48306405263598,-79.4815061441155,-79.47923922410104,-79.47764834083407,-79.47731401631329,-79.47718844064988,-79.47701262599733,-79.47684035300003,-79.47671989847858,-79.47660730037356,-79.47652672024016,-79.47645285882845,-79.47635546248739,-79.47625560191234,-79.47617808655534,-79.47552899078342,-79.47543805054778,-79.47534678261324,-79.47525644569902,-79.47516681224086,-79.47511682919856,-79.47434453242019,-79.47423315289215,-79.47394943945501,-79.47380672478531,-79.47346146272641,-79.47331427859305,-79.47324538601171,-79.47278593508995,-79.47241826260159,-79.47201110865161,-79.4706926768384,-79.47075956988861,-79.47077619322819,-79.47082534061462,-79.47098267337948,-79.47123774019137,-79.47152875803482,-79.47180867412482,-79.47202336040621,-79.47204370621509,-79.472248754785,-79.47234642453544,-79.47241899509831,-79.47246826594079,-79.47255991722187,-79.47274486364225,-79.47295297884276,-79.47312975914586,-79.47329838324858,-79.47350578400665,-79.47372715467765,-79.47396790923828,-79.47417119904019,-79.47435259834536,-79.47455825716759,-79.4747271399512,-79.47493339206019,-79.47512305393633,-79.47524190543315,-79.47536525437576,-79.47550364620692,-79.47566418084088,-79.47585609241482,-79.47605185355869,-79.47625305613489,-79.47643818045059,-79.47663314930884,-79.47677320696266,-79.47686294676521,-79.47691582538272,-79.476980131321,-79.47699630505289,-79.47683846946697,-79.47657429055664,-79.47631282490936,-79.47607717543761,-79.47596246876984,-79.47594493642252,-79.47598222520988,-79.4760619146478,-79.47611866290164,-79.47622248785626,-79.47627720125149,-79.47631886144255,-79.47637720790863,-79.47652177542935,-79.47668469910535,-79.47687454804043,-79.47707765969611,-79.47719444572857,-79.47707685886574,-79.4769302097245,-79.47676611050848,-79.47673839182201,-79.47686128499014,-79.47698862835392,-79.47718706881903,-79.47739174172969,-79.47751308059597,-79.47747926408077,-79.4773703857031,-79.47717292725426,-79.47704767979526,-79.47698224835219,-79.47695127372583,-79.47705744242229,-79.47707624495857,-79.47718730858912,-79.47735845557864,-79.47756810768101,-79.47778702212062,-79.47794422700798,-79.47804749993551,-79.47821893877514,-79.47834573141209,-79.47844721740435,-79.4786426875721,-79.47883133220313,-79.47884709861651,-79.47877231450742,-79.47872117396986,-79.47850095962376,-79.47828376513233,-79.47805490947947,-79.4778477376456,-79.47764054254985,-79.47747598429704,-79.4773335592342,-79.47722376090165,-79.47707418523223,-79.47697246935788,-79.4768463890294,-79.47665969133124,-79.4764256890987,-79.47617805171602,-79.47594697403467,-79.47571836314886,-79.47552375650918,-79.47542183484194,-79.47535567024646,-79.47523880744419,-79.47501756222707,-79.47478964430871,-79.47467382823164,-79.47450622552442,-79.47423829025665,-79.47398726572928,-79.47380069099403,-79.47369476642298,-79.47360309572886,-79.47346152519971,-79.4732909734027,-79.47316041565171,-79.47314521515536,-79.4731991449997,-79.47329738934422,-79.47340557298088,-79.47351856858452,-79.47361456637834,-79.47369431089236,-79.47377632060315,-79.47377911456019,-79.47367241388341,-79.47353963781896,-79.47346562991497,-79.47337240784259,-79.47332999077497,-79.47318229009119,-79.47301173720591,-79.47281029130664,-79.47259220467656,-79.47238677107609,-79.47222941500696,-79.47205548076661,-79.47190661484808,-79.47176822200272,-79.47164233987687,-79.47154734380746,-79.47161419247919,-79.47166836556588,-79.47179536986501,-79.4719675581333,-79.47198643623457,-79.47201353777045,-79.47183072120144,-79.47161110097421,-79.47133973089049,-79.47109520611883,-79.47089309599518,-79.47065691254024,-79.47044287334936,-79.47028806258739,-79.47019962379251,-79.4701532112902,-79.47010478191443,-79.4700611645471,-79.4699987266754,-79.46989898331363,-79.46976334580833,-79.46960617344011,-79.46943654379173,-79.46921586136203,-79.4691159142382,-79.46919085740599,-79.46937585761685,-79.46960452691347,-79.4698664448325,-79.47016003112172,-79.47045582317344,-79.47076546311149,-79.47108499886821,-79.47138048237031,-79.47167208314853,-79.47194804247998,-79.47223884476219,-79.4725228686227,-79.47276620220138,-79.47299783660607,-79.47322708843579,-79.47341947439533,-79.47362667084566,-79.4738285906643,-79.47401940580738,-79.47419682659017,-79.47437974903086,-79.47453906095159,-79.47467751649459,-79.47480996893263,-79.47480331647485,-79.4747296128546,-79.47461599679222,-79.47444613425817,-79.47433452810228,-79.47430704792446,-79.47439349128338,-79.47462805994577,-79.4748864825123,-79.4751206424647,-79.47529694328989,-79.47533744743748,-79.47542693350839,-79.47550310434383,-79.47560660666744,-79.47575901649937,-79.4759678238126,-79.4761839310233,-79.47636292012777,-79.4765383853458,-79.4767988655142,-79.47704729017926,-79.47720817241076,-79.47719406890246,-79.47713113319483,-79.47715023503494,-79.47703886072324,-79.47704732659126,-79.47713809160786,-79.47728006171523,-79.47740802683809,-79.47753221305406,-79.47776949850515,-79.47803225765571,-79.47828786494141,-79.47854675259444,-79.47882111297781,-79.47907668697839,-79.47933898193099,-79.47957636832456,-79.479806209324,-79.48004879514853,-79.48029224835041,-79.48054839170976,-79.48081031426162,-79.48106447267601,-79.48101512440371,-79.4808084958579,-79.48058496564994,-79.48047838093274,-79.48060083242555,-79.48082221492163,-79.48105529087059,-79.48128979717889,-79.48156537716999,-79.48196158818448,-79.48208166973546,-79.48223053763054,-79.48240953364656,-79.48260410605806,-79.48282026092809,-79.48304559798429,-79.48327009848387,-79.48349415796832,-79.4837289237758,-79.48392707772926,-79.48413594881171,-79.48432371525442,-79.48453428722598,-79.48473372320663,-79.48494184376163,-79.48513548927353,-79.48530911003255,-79.4855055085247,-79.4855805530788,-79.48561709107916,-79.48564295815243,-79.48568114179125,-79.48578930537798,-79.48592664654586,-79.48606975281591,-79.48621088380624,-79.48635164116138,-79.48652416306147,-79.4866969971568,-79.48687653692753,-79.48694590546455,-79.48700559779078,-79.48708009162561,-79.48709537551078,-79.48702464933487,-79.48676470091411,-79.48676344551022,-79.48675405089658,-79.48665487089957,-79.48661542248503,-79.48670865582493,-79.48686752304681,-79.48708875855337,-79.48749674866882,-79.48791597188352,-79.48823258426253,-79.48843864924198,-79.48874566774727,-79.48887571783389,-79.48912055610198,-79.48947787336819,-79.48973788548214,-79.49013227258105,-79.49030453605481,-79.49063172266315,-79.49095543266601,-79.49105003976355,-79.49118296308619,-79.49146343274971,-79.49205873985409,-79.49248888367923,-79.49286649301884],"lat":[43.6284678586652,43.62846811247095,43.62870177571531,43.62904269212029,43.62983217406657,43.63014567622515,43.6303757209259,43.63063452484054,43.63096500907638,43.63167322063286,43.63217425331393,43.63228232593588,43.63231675127425,43.63237742331223,43.63245164543058,43.63252205688637,43.63259707398318,43.63265901461073,43.63273013686595,43.63281435845605,43.63290304815864,43.63296502939596,43.63347858514063,43.63355397997044,43.63363828243369,43.6337180961934,43.63380242031522,43.63384676394473,43.63347178224245,43.63342978595289,43.63328646553491,43.63323054727925,43.63309990688913,43.63303483619998,43.63300240857262,43.63281172704531,43.63261767931483,43.63236017788363,43.63124860849271,43.63083981786977,43.63073651610889,43.63066065581568,43.6305952449806,43.63058965492834,43.63062955561678,43.63065580396852,43.63061816486659,43.63061393581814,43.63047255323414,43.63028481457641,43.63008314718778,43.62988125836414,43.62968893801298,43.62957429217182,43.62958607508007,43.62970105322705,43.62977082160452,43.62978259396289,43.62972703671495,43.62964464151678,43.62950782303943,43.62937971409455,43.62927443391381,43.62924509045492,43.62926584699701,43.62918276758199,43.6290179042842,43.62885301083147,43.62871973524511,43.62867686590328,43.62875135213629,43.62875846359568,43.62867562619506,43.62864658675273,43.62856366558171,43.6284169074881,43.62820655392563,43.62798661694125,43.62777151329738,43.62756008924586,43.62742745531936,43.62736542062519,43.6273214257102,43.62729127758449,43.6272357354265,43.62710037997988,43.62694334029299,43.62682737544745,43.62669301042646,43.6265683651534,43.62642956188379,43.62628149243601,43.62612023226811,43.62600513194756,43.62590377926383,43.62590180839358,43.62586841652018,43.62574844410042,43.62562084778168,43.62551086828245,43.62547267292527,43.62540469764133,43.6252938090589,43.62516047436192,43.62515411639883,43.62514784111966,43.62500533370701,43.62484284520371,43.62470636306399,43.62460912010983,43.6244814216713,43.62433651596361,43.62418297725117,43.62405836239203,43.62389657487758,43.62373592663528,43.62361217722127,43.6235339510324,43.62350986050787,43.62336341923246,43.62317115898109,43.6229708941638,43.62276994466757,43.6225552449724,43.62238671680161,43.62224969487807,43.62211487296001,43.62207327807113,43.62201408362473,43.62198864700338,43.62206227345457,43.62214033529006,43.6222095933129,43.6222878527655,43.62239368617182,43.62251340740446,43.62266047948746,43.62278001509809,43.62288218404931,43.62300212246446,43.62309415645659,43.62318555956271,43.62329478462627,43.62339981893175,43.62345978507149,43.62340767788429,43.62327119663446,43.62313528230909,43.62300769371518,43.62290571864776,43.62281706725802,43.62268048993472,43.6225387202395,43.62247662998969,43.62247327865295,43.62255630750292,43.62270342790862,43.62286883282601,43.62301097588789,43.62313922847471,43.62325001086139,43.62335792319015,43.62340365420393,43.62349948877208,43.62366297186698,43.62382210808826,43.6239179123741,43.62397749107398,43.62410911687303,43.62421276850532,43.62410772929557,43.62397092435747,43.62386641160231,43.62384266117574,43.62377007768799,43.62363748400817,43.62349117167567,43.62339845898119,43.62341804997259,43.62348282023514,43.6236427549902,43.62381156003077,43.62396710677538,43.62414088760573,43.62422022335676,43.6243269783525,43.62447190610725,43.62465726384709,43.62481199799082,43.6249583339731,43.62516122389162,43.62534612949716,43.62531667893008,43.6251787105821,43.62509856339334,43.62514480423356,43.62526362890585,43.62532348356862,43.62523960032368,43.62505298513863,43.62484466284372,43.62462348987159,43.6243931977425,43.62415855896394,43.62393258030938,43.62371069326781,43.62350173870967,43.62328358354026,43.62310126978191,43.62294527834536,43.62276380735473,43.62255326145797,43.62241170531341,43.62236966685047,43.62246328482683,43.62252122761494,43.62255669388061,43.62253374089723,43.62251551053102,43.62247445229713,43.62243334137843,43.62239643163254,43.62238681599419,43.62239061219839,43.62232634811584,43.62220341345552,43.62205344014591,43.62185787336735,43.62166259407118,43.6214536507349,43.62123114561481,43.62101287239153,43.62079926340117,43.62058975002281,43.62038454915426,43.62017017586109,43.6199765415199,43.61979542554071,43.61960036367908,43.61942705625551,43.61924552411278,43.61904711063204,43.61884562677649,43.6187272289449,43.61870817226846,43.61878781464023,43.61892069742203,43.61910128016978,43.61926451204024,43.61944106941453,43.61963608553219,43.61976414787871,43.61981194297756,43.61984183078052,43.61984421700352,43.61994566931661,43.62000315377057,43.62010098653243,43.62023366089511,43.62039560115946,43.62055680075824,43.62071909341613,43.6208032195599,43.62092936202918,43.62107460532573,43.62125212819319,43.62143387534775,43.62157506248723,43.6215962272808,43.62156371809301,43.62153111308005,43.6214940501032,43.62152020729992,43.62158212231846,43.62165312830492,43.62167438071059,43.62165034163088,43.62167616327047,43.62176492073383,43.62180433473623,43.62186182921773,43.62191921991549,43.6218780545465,43.62181229339814,43.62176431142336,43.62169087749486,43.62162940056002,43.62163684442449,43.62166694856077,43.62173767078171,43.62180884818704,43.62197614944336,43.62200475064803,43.62200222658692,43.622004603436,43.62202969209797,43.62208657433845,43.6221481691205,43.62221866445341,43.62230265667034,43.62239579260087,43.62246143605489,43.62253172236463,43.6225837241849,43.62262702596909,43.62267017972851,43.62271344826157,43.62274302138561,43.62277232880258,43.62284694837175,43.62291095757876,43.62296995547424,43.62304690622708,43.6231014248466,43.62320188118496,43.62332072817974,43.62342164713888,43.62353613295188,43.62363702044484,43.62374282945266,43.62380363175079,43.62383751621444,43.62388344514069,43.62391583277996,43.62399783793678,43.62410156458618,43.62421567540178,43.62442918498776,43.62443033864133,43.62448296665332,43.62458643808725,43.62467593689982,43.62491131639597,43.6250034409025,43.62507838655065,43.62511979488774,43.62520645097531,43.62543569213702,43.62561854989806,43.6258841213065,43.62598441397977,43.6261047683392,43.62619051029984,43.62620294849278,43.62629818035304,43.62634546647455,43.62664694756487,43.62708341349313,43.62726470482596,43.62740158136796,43.62758523580867,43.62807029356675,43.62822000170701,43.6284678586652]}]],[[{"lng":[-79.4925243343052,-79.49277040338879,-79.49291063421458,-79.49315485733212,-79.49297952032531,-79.49277693281336,-79.49255883775311,-79.49232943747714,-79.49209596035313,-79.49184736247355,-79.49162287989193,-79.49144064769276,-79.49128981426449,-79.49113644337245,-79.49099719604119,-79.49084504889706,-79.49072368380581,-79.49061939589126,-79.49054825504922,-79.4905147306868,-79.49037067772534,-79.49033331205516,-79.49026383609709,-79.49012235021809,-79.49004922678867,-79.49001851027036,-79.48987755766544,-79.48987206000835,-79.48977599290963,-79.48970333495245,-79.48962284695668,-79.48945424818469,-79.48930444887259,-79.48915222256787,-79.48865321077183,-79.4882214738259,-79.48733841489015,-79.48690727294894,-79.48645905798554,-79.48618233221937,-79.48609639207298,-79.48602787313239,-79.48595179398767,-79.48586642636324,-79.48576798395332,-79.48566024449059,-79.48553884733708,-79.48540837907335,-79.48526696833407,-79.4842514443958,-79.4841477850285,-79.48398833326857,-79.48392268954326,-79.48387268775872,-79.48382750950664,-79.48342070738538,-79.4832566035611,-79.48322377592777,-79.48317168650318,-79.48306678133214,-79.48291968512524,-79.4827387285449,-79.48256341432021,-79.48246760303375,-79.48221680907473,-79.48201836560409,-79.48196158818448,-79.48156537716999,-79.48141389843197,-79.48142021301616,-79.48154703826086,-79.48160592075617,-79.48161457115104,-79.48144415623493,-79.48120508942601,-79.48098290610614,-79.48079256615583,-79.48066736145739,-79.48052837733221,-79.48031912104743,-79.48010652437159,-79.47992822083268,-79.47978450798898,-79.47967033020974,-79.47951315497582,-79.47943575884595,-79.47954983049502,-79.47945184250277,-79.47932769117114,-79.47921393863936,-79.47910758497906,-79.47897178542853,-79.47876105593687,-79.4785264204413,-79.47832293623833,-79.47806581521547,-79.47782261162263,-79.47756793174798,-79.47733476315589,-79.47711874737475,-79.47695917448553,-79.47676817898775,-79.4766235436897,-79.47650322457345,-79.47646895104302,-79.47652172792516,-79.47665433895371,-79.47685148854221,-79.47705537926261,-79.4772599618105,-79.47746722202406,-79.47766632026233,-79.47780459922144,-79.47784744694141,-79.47794267181636,-79.47809056689485,-79.47810692395058,-79.47815651538066,-79.4783134128415,-79.47836273606457,-79.47842877727382,-79.47859627551594,-79.47875906249095,-79.47882958876545,-79.47876922180888,-79.47872400874768,-79.47864584571354,-79.47850132252249,-79.47832642459345,-79.47811815533012,-79.47791682447547,-79.47774208746596,-79.47758055198568,-79.47748895007103,-79.47738659706671,-79.4772569948376,-79.47707637557227,-79.4768963378006,-79.47671985167763,-79.47657343325636,-79.47645290541702,-79.47630575952374,-79.47615379792046,-79.47604219110389,-79.47601651170865,-79.47593657385383,-79.47581254031532,-79.47565007493813,-79.47544924234893,-79.47528612816404,-79.47513805467003,-79.47502110936193,-79.47496087435103,-79.47511035317416,-79.47526889237272,-79.47550055790138,-79.47572241730694,-79.47588313279824,-79.47600999624717,-79.47610612100306,-79.47612698687358,-79.47623059116923,-79.47637086861501,-79.47650675378087,-79.47663443353804,-79.47676221253215,-79.47689101550996,-79.47701358386382,-79.47714321158992,-79.47724942431978,-79.47734011793121,-79.4774346723093,-79.47749618236377,-79.47740908003038,-79.47728414863298,-79.47719570710859,-79.4770340444133,-79.47691274437581,-79.47699652611713,-79.4771371925566,-79.47727401052565,-79.47744715982002,-79.47758311416108,-79.47779240352779,-79.4779954165097,-79.47817479820561,-79.47826777854422,-79.47829179598617,-79.47830366847025,-79.47847789899343,-79.47860745113893,-79.47872993108994,-79.47882346953682,-79.47891367934633,-79.47900191392982,-79.4790688456338,-79.47916319717881,-79.479213418798,-79.47926743413996,-79.4792176727056,-79.47912882144769,-79.47906738302834,-79.47899335557877,-79.47889464257375,-79.47887296400125,-79.47891858864232,-79.47905250473875,-79.47922485266301,-79.47942559883832,-79.47959219557245,-79.4797852844497,-79.47994596023189,-79.480030158961,-79.48011824221425,-79.48020950162231,-79.48028052669622,-79.48041540434562,-79.48056500596836,-79.48079052593896,-79.48103084525199,-79.48127737161731,-79.48153315812182,-79.48178012605095,-79.48202613857846,-79.48225128680106,-79.48243210284878,-79.48260895423996,-79.48269590871341,-79.48267731872146,-79.48265753825565,-79.4826874008136,-79.48277822502823,-79.48291056367172,-79.48306144085083,-79.48318686662118,-79.48327994742026,-79.48332938821667,-79.48341574731276,-79.48356281968795,-79.48371740646965,-79.48387196560012,-79.48401908968562,-79.48411846553537,-79.48416726741215,-79.48413405451545,-79.48406354507883,-79.48398516969156,-79.48390665917655,-79.48381332306221,-79.48372247330263,-79.48366315573982,-79.48360656328933,-79.4835247692519,-79.48347288629135,-79.48344969739837,-79.48341640383683,-79.48336440721681,-79.48331330430779,-79.48323332089423,-79.48313621587177,-79.48309050215747,-79.48308462529198,-79.48308298291097,-79.48307213345667,-79.4830444979291,-79.48293769691581,-79.4828208443586,-79.48281713840763,-79.48283067340141,-79.48269000539011,-79.48251666266241,-79.4823241622211,-79.48215928492284,-79.4820157851033,-79.481920423529,-79.48176686504029,-79.48160338416456,-79.48145651004207,-79.48128962506513,-79.48114640635436,-79.48100505979629,-79.48082793745782,-79.48066886280296,-79.4805121153107,-79.48041298371659,-79.48036695087453,-79.48034201894129,-79.48031673344981,-79.48027543427112,-79.48024168313175,-79.48018753222624,-79.48016237291075,-79.48010823505678,-79.48009483448411,-79.48009412124222,-79.48011713136046,-79.48023496512094,-79.48036006093645,-79.48045672406283,-79.48051867834688,-79.48053713836212,-79.48051533550823,-79.48047353456566,-79.48044834029042,-79.48043856038943,-79.48041556485617,-79.48034850531589,-79.48026953814374,-79.48016664250441,-79.48004978645652,-79.4799498224323,-79.47985470564248,-79.4797604058423,-79.47966079360363,-79.47952706951421,-79.47943052140282,-79.47932976204237,-79.47920122672632,-79.47906077239749,-79.47894058845426,-79.47883604789021,-79.47886377895647,-79.47898400916709,-79.47911884840485,-79.4792409842797,-79.47937547280624,-79.47951550088902,-79.47965080761828,-79.47977939474731,-79.47992531350977,-79.48006757972914,-79.48020607769988,-79.48033430268164,-79.48044644184787,-79.48055018251397,-79.48062037624084,-79.48071848194542,-79.48082528451363,-79.48093115403132,-79.48104031475482,-79.48116451501362,-79.4812275212718,-79.48122029920977,-79.48122047939614,-79.48130060584367,-79.48136151724076,-79.4814364269423,-79.48154793227296,-79.48163566782368,-79.4816094839224,-79.48153467172344,-79.48145363871986,-79.48135166045094,-79.48124827836713,-79.48119037065021,-79.4812298998545,-79.48134692274108,-79.48147436183591,-79.48156393476195,-79.48163546092579,-79.4817897017992,-79.48193228029902,-79.48209533753335,-79.48224369443363,-79.48239967719681,-79.48252241625043,-79.48263886534444,-79.48280775880595,-79.48298592687013,-79.48314525013193,-79.4832633431288,-79.48341845941155,-79.48357101024885,-79.48371142545116,-79.48388617361,-79.48407479359291,-79.48425309536367,-79.48438435383677,-79.48450392661118,-79.48461705206766,-79.48473565674317,-79.48484617092271,-79.484946068732,-79.48501453377234,-79.48507278265811,-79.4851480723074,-79.48518290480632,-79.48525726140603,-79.4853116295068,-79.48536996842829,-79.48541750004054,-79.48543219811192,-79.48545312541852,-79.48542103833678,-79.48539626338713,-79.48538156464171,-79.48540369843302,-79.48544409914992,-79.48549250033237,-79.48552090044591,-79.48556526912364,-79.48562458696151,-79.48573756614068,-79.48583155478889,-79.48580522294921,-79.4858164120042,-79.4858922970849,-79.48600565630885,-79.48607019630329,-79.48608820987675,-79.4861384533254,-79.48615900227202,-79.4861911589948,-79.48616997754908,-79.48624740401294,-79.48637189051431,-79.48650758122272,-79.4866537133892,-79.48681198026135,-79.48697666736784,-79.48714362792795,-79.48727696991608,-79.4873952332333,-79.48749523210545,-79.48756615974395,-79.48763428866974,-79.48767107573735,-79.48758234621957,-79.48748115680242,-79.4873665623813,-79.48724254911652,-79.48718685998084,-79.48721980412631,-79.48727075021715,-79.48731213432757,-79.4873630884088,-79.48737918786294,-79.48730626382107,-79.48721875788301,-79.48710490819795,-79.48699045653665,-79.48691024652076,-79.48688106481707,-79.48689164894127,-79.48692094760129,-79.48689490299716,-79.48683785639591,-79.48675782440807,-79.48665611110458,-79.48652547447745,-79.48655976321065,-79.48669987216324,-79.48679622966866,-79.48678680473951,-79.48675541438243,-79.48672958423703,-79.48671164313545,-79.4866998309105,-79.48669092161148,-79.48667391062085,-79.48665435185679,-79.48661608949325,-79.48658207451732,-79.48646750338283,-79.48644032153781,-79.48641877921149,-79.48640676871722,-79.48640507508168,-79.48641386473528,-79.48643441423887,-79.48649857237,-79.48650813049012,-79.48649486131175,-79.48651925204184,-79.48652427283561,-79.4865102909059,-79.48652482197845,-79.48655794358275,-79.48659543599759,-79.48664199793754,-79.48670841886361,-79.48678544144563,-79.48689834062243,-79.48693713918436,-79.48696276093871,-79.48699734854932,-79.48703649440698,-79.4870895340226,-79.48713305036937,-79.48714519311956,-79.48714882146042,-79.48712398172889,-79.48719292317541,-79.48728772573949,-79.48742048915061,-79.48755247347474,-79.48764364897583,-79.48774084832982,-79.4877918998973,-79.48781839456508,-79.48790079137127,-79.48793313968119,-79.48794339333629,-79.48798340890994,-79.48810415895461,-79.4881792534524,-79.4882540707483,-79.48833148357357,-79.48838809802457,-79.48840789092095,-79.48845592554875,-79.48853232338621,-79.48867044982805,-79.48879394161062,-79.48890777190969,-79.48898883880217,-79.489108579803,-79.4891984223696,-79.4892643741928,-79.48934596577114,-79.48934093492797,-79.48930428006253,-79.48939004339694,-79.48945345066011,-79.48956367970813,-79.48966102325178,-79.48974827809396,-79.48987036399114,-79.49000521211627,-79.49017936215813,-79.49030816439459,-79.49043625334433,-79.49049447740342,-79.49056824506303,-79.49068897482641,-79.49079414176543,-79.49094306207046,-79.49105989011656,-79.49119317423171,-79.49126160938992,-79.49136546474826,-79.49153147907668,-79.49167525012315,-79.49179840761325,-79.49192723272166,-79.49207241153103,-79.49219605765568,-79.49223546272385,-79.49236726812778,-79.4924293458039,-79.49248810644593,-79.49251346680838,-79.4925243343052],"lat":[43.60253226122765,43.60279215764202,43.60306407121831,43.60364360870901,43.60371781526076,43.6038276707845,43.6039734200301,43.60410542648891,43.60427788835587,43.60449525097222,43.60470834019506,43.60493557985964,43.60514513937759,43.6053907637892,43.60566808195425,43.60595864280288,43.60623169564625,43.60649597172358,43.60673809007552,43.60696729197851,43.60830238738973,43.60855845483992,43.60966507028449,43.61046007050609,43.61093630775272,43.61114295108047,43.61210008628294,43.61213602218238,43.61272448264056,43.61299817764538,43.61321316523691,43.61344958366352,43.61377177056479,43.61403982235991,43.61494262014975,43.61572927779957,43.61727532227414,43.61809799098464,43.6189204318531,43.6193309546673,43.61942433865392,43.61949544803505,43.61956645722471,43.61963293227216,43.61969914401238,43.61976523248585,43.6198176367024,43.61986091845999,43.61990855610948,43.62015624445837,43.62018187606131,43.62021126860596,43.62023290318218,43.62025024415382,43.62025864696941,43.62041528796584,43.620507722623,43.6205207901582,43.62055610735776,43.62063123310496,43.62074630827999,43.62091953756832,43.62108374933456,43.62119950479574,43.62156085045219,43.62190488659704,43.62197614944336,43.62180884818704,43.62154568417785,43.62138823099983,43.6212097836102,43.62103052349463,43.62083709284437,43.62071321012855,43.62066952399658,43.62069807871364,43.62078106898661,43.62092352861418,43.62106121383059,43.6210674338117,43.62097908673783,43.62086418900345,43.62072715576406,43.62058610418075,43.62052099849822,43.62046145506796,43.6203593582199,43.62028153662378,43.62028888698531,43.62034588723787,43.62026345297357,43.62014911932538,43.62009230201089,43.61999465653346,43.61986132681476,43.6197678820818,43.61970612954648,43.61969823634965,43.61966812386716,43.61960223097655,43.61947848602423,43.6193769173958,43.61924445894926,43.61907181490072,43.61889122559333,43.61870738559985,43.61854702494169,43.61843712591439,43.61836782591912,43.61830753689824,43.61831930016031,43.61840747237511,43.6185488469871,43.61869354161119,43.6188118373495,43.61890832905713,43.6190255744447,43.61908933964599,43.6191004307757,43.61898847090225,43.61889032677337,43.6187890321146,43.61869217556258,43.6185985019315,43.61847166876079,43.61833153407385,43.61817286651099,43.61803591084485,43.61791205378397,43.61781475847382,43.61772196622453,43.61761611465589,43.61754194609954,43.61748671316676,43.6173999195239,43.61734409014159,43.61738228318622,43.61739338730124,43.61745405014812,43.61754211986792,43.61765303983216,43.61774560056914,43.61784268799109,43.61794022327119,43.61803440315691,43.61813236059571,43.61813520765457,43.61806552503653,43.6180178357259,43.6180021566712,43.61797767592846,43.61788609440902,43.61779076841358,43.61766664308645,43.61767325923651,43.61760883359224,43.61750376714335,43.61733928109632,43.61718793664937,43.61703168115774,43.61694193811975,43.61685770921441,43.61673354956046,43.61656882162186,43.61641739731777,43.61626606421013,43.61607414501383,43.61588223260958,43.61568591306259,43.61549819351011,43.61530585606171,43.61512248207131,43.61495676209861,43.61482056987372,43.61473779634613,43.61464659675271,43.61454541932075,43.61444027847317,43.6143152750977,43.61423613026676,43.61418394037455,43.61422675665354,43.61427357830047,43.61432587756786,43.61435558755155,43.61439848572133,43.61443132117906,43.61442713988285,43.61441829582268,43.61435751043587,43.61422870412746,43.61406829608605,43.61392091589569,43.61375557708786,43.61360371511277,43.61343347530698,43.61328619566186,43.61313832875963,43.61302643080973,43.61290874100721,43.61277702784152,43.6126320861738,43.61249606909239,43.61238672982856,43.61223331505609,43.61208538707768,43.61196564041777,43.61191842202418,43.61191209077686,43.6119998273002,43.61211492234531,43.61228359878591,43.61247835439929,43.61267307157977,43.61286341987626,43.61305791009587,43.61320832858911,43.61335435157472,43.61341136218954,43.61343256052002,43.61342233336613,43.61338522231409,43.61336149674509,43.61331075156046,43.61322822132713,43.61314501200977,43.61305283765936,43.61291896006761,43.61279268343512,43.6126527977736,43.61251366116711,43.61238433593795,43.61232298765159,43.61236558982434,43.61245727574193,43.61257553869854,43.61270222462583,43.61281148616187,43.61290795995472,43.61298193789595,43.61302449782143,43.61305795679004,43.61310878656452,43.6132040462086,43.61329362703388,43.61338721424304,43.613480697073,43.61358768125126,43.61369905979137,43.61379687802053,43.61390861770841,43.61403389671343,43.61414542814019,43.61426626849433,43.61439199062455,43.6145131676253,43.61463850747862,43.61473235171332,43.61477179989098,43.61475250694629,43.61468888534517,43.61458528286301,43.61446364229876,43.61434647062187,43.61424257934811,43.61417814689728,43.61417209476383,43.61410002853167,43.61399659368241,43.61395421676308,43.6139474143767,43.61396286266666,43.61396076323283,43.61403096452204,43.61409271286664,43.61409967531152,43.61407049721406,43.61405954378343,43.61404382320405,43.61404192007619,43.61402653845558,43.61401968327521,43.61402657092508,43.61404249142122,43.61409968732012,43.61420710084114,43.61433279915782,43.61447658704428,43.61461106984643,43.61475015405983,43.61487555382893,43.61501025119618,43.6151310600102,43.61526141265998,43.61538302191762,43.61545534492569,43.61547932710687,43.61549899467926,43.61553178729778,43.61559562585117,43.61569039359951,43.61578921716262,43.61587868270264,43.61598637326662,43.61609876982336,43.61618407431414,43.61620118696816,43.61620013699976,43.61616726134834,43.61616120633096,43.61622289179565,43.6163071469565,43.61639141290348,43.61647560810543,43.61655493838861,43.61663917414911,43.6167323559218,43.61682066695842,43.61689531605306,43.61699283007222,43.61708596108088,43.61714934502739,43.61723646528311,43.61732377981048,43.61742451835994,43.61752983216464,43.61762621738003,43.61773154174313,43.61782777444431,43.61793783071261,43.61803874607561,43.61813511009159,43.61821333310213,43.61830943637433,43.6183738304324,43.61832975296446,43.61827704435929,43.61821986018343,43.61816725458885,43.61810569051172,43.6180353240642,43.61796414433495,43.61791003562277,43.61783793100408,43.61782549253579,43.61789840890042,43.61797142119904,43.61803591751968,43.61808659477383,43.618140259668,43.61819327851099,43.61824171359137,43.61828996033331,43.61836960573985,43.61848586384338,43.61857190929286,43.61863647912319,43.61862016811885,43.61863036028282,43.61870791857496,43.61867837006417,43.61861274797364,43.6185699029852,43.61852236149144,43.61844791469081,43.61843154010855,43.61839707764251,43.61839031760958,43.61839268255152,43.6183722918096,43.61829274988303,43.6181957848775,43.61810328668353,43.61800162516249,43.61798143801781,43.61797043671755,43.61795929826837,43.61788443083356,43.61778249194667,43.61768496851272,43.61756051123037,43.6174448586738,43.61732465423911,43.61719052988548,43.61705627007704,43.61693114819872,43.61680558009275,43.61668503685487,43.6165506354589,43.61642087773981,43.61628197464754,43.6161605506793,43.61603929927313,43.6159578549268,43.61589451172462,43.61580429566175,43.6156919723525,43.61562949282509,43.61555361610375,43.61546847220742,43.61537003669827,43.61525820602885,43.61513367311111,43.6150043874308,43.61490942620815,43.61480154888923,43.61468102544269,43.61461051005919,43.61453484685779,43.61443597200961,43.61431510874834,43.61426586920486,43.61418977700854,43.61407687981138,43.61395187546254,43.61384099748724,43.61374377093655,43.61372320059066,43.61371179284444,43.6137499815911,43.61373418740522,43.613636838827,43.61352137629612,43.61339666981072,43.6132535742839,43.61312835591978,43.6129938106974,43.61288911209139,43.61278415850747,43.61268811958026,43.61260095800736,43.61256421238838,43.61245662283454,43.61234017937602,43.61220569509526,43.61208484065044,43.61197252704333,43.61188145056107,43.61182177852153,43.61175725664668,43.61168822562214,43.61161064565738,43.6115111458349,43.6113987592714,43.61127761831985,43.6111467425742,43.61103787170521,43.61092878650997,43.61087342699833,43.61086719614931,43.61079563310307,43.61077048185435,43.61075825440392,43.61072662214877,43.61065409944661,43.61060433573437,43.61054558431071,43.61048691403148,43.61041027791394,43.61034703772945,43.61032427342091,43.61030576252546,43.61030531213037,43.61033080141859,43.61033494253506,43.61033015619127,43.61031199286742,43.61029396616663,43.61024907188457,43.61019974224261,43.61011057047581,43.61004318101959,43.60995298394352,43.60987678876003,43.60982734349131,43.60973704695445,43.60968322654927,43.60962965230844,43.60958513806918,43.60954974604169,43.60949661265694,43.6094526217309,43.60940001354939,43.60936451862479,43.6093288492545,43.60921678039305,43.60907776544325,43.60893884436444,43.60880888936425,43.60872803084224,43.60870557352121,43.60858371586048,43.60845850841284,43.60832473107791,43.60824996994167,43.60820220475895,43.60809079445338,43.60797955379596,43.60785870036016,43.60775552628544,43.60768450925103,43.60756790937699,43.60746452040875,43.60738853158166,43.60729560642921,43.60720198724051,43.60709495114376,43.60696994504092,43.60687167024373,43.6067953237841,43.60682746653652,43.60674295650662,43.60673127980615,43.60668790194614,43.60662639193577,43.6065104358271,43.60636789445022,43.60621154455009,43.60605037772583,43.60588482647655,43.60580824176558,43.60575824550389,43.60564235116559,43.60552157021129,43.60551402480104,43.60542528971791,43.60528240837907,43.60513989715325,43.60504265514233,43.6049549342855,43.60482160346852,43.60468817306052,43.60455390985187,43.60445586046616,43.60436743332838,43.6042517041077,43.60411863819746,43.60399415056134,43.60386528894111,43.6037311601135,43.60360650091312,43.60350066628613,43.6033854457149,43.60326104083087,43.60315921596084,43.60305301560363,43.60293761887684,43.60281210862266,43.60274633115563,43.60271114137593,43.60265331241941,43.60259513293701,43.60253226122765]}]],[[{"lng":[-79.49684685602216,-79.49818675404019,-79.49834332751216,-79.49812865900168,-79.49673382832636,-79.49548168022565,-79.49410063401712,-79.49232674894277,-79.49184368829629,-79.49085090559777,-79.48969605939226,-79.4887334253124,-79.48694590546455,-79.48687653692753,-79.4866969971568,-79.48652416306147,-79.48635164116138,-79.48621088380624,-79.48606975281591,-79.48592664654586,-79.48578930537798,-79.48568114179125,-79.48564295815243,-79.48561709107916,-79.4855805530788,-79.4855055085247,-79.48530911003255,-79.48513548927353,-79.48494184376163,-79.48473372320663,-79.48453428722598,-79.48432371525442,-79.48413594881171,-79.48392707772926,-79.4837289237758,-79.48349415796832,-79.48327009848387,-79.48304559798429,-79.48282026092809,-79.48260410605806,-79.48240953364656,-79.48223053763054,-79.48208166973546,-79.48196158818448,-79.48201836560409,-79.48221680907473,-79.48246760303375,-79.48256341432021,-79.4827387285449,-79.48291968512524,-79.48306678133214,-79.48317168650318,-79.48322377592777,-79.4832566035611,-79.48342070738538,-79.48382750950664,-79.48387268775872,-79.48392268954326,-79.48398833326857,-79.4841477850285,-79.4842514443958,-79.48526696833407,-79.48540837907335,-79.48553884733708,-79.48566024449059,-79.48576798395332,-79.48586642636324,-79.48595179398767,-79.48602787313239,-79.48609639207298,-79.48618233221937,-79.48645905798554,-79.48690727294894,-79.48733841489015,-79.4882214738259,-79.48865321077183,-79.48915222256787,-79.48930444887259,-79.48945424818469,-79.48962284695668,-79.48970333495245,-79.48977599290963,-79.48987206000835,-79.48987755766544,-79.49001851027036,-79.49004922678867,-79.49012235021809,-79.49026383609709,-79.49033331205516,-79.49037067772534,-79.4905147306868,-79.49054825504922,-79.49061939589126,-79.49072368380581,-79.49084504889706,-79.49099719604119,-79.49113644337245,-79.49128981426449,-79.49144064769276,-79.49162287989193,-79.49184736247355,-79.49209596035313,-79.49232943747714,-79.49255883775311,-79.49277693281336,-79.49297952032531,-79.49315485733212,-79.49291063421458,-79.49277040338879,-79.4925243343052,-79.49250321392527,-79.49249470699723,-79.4925108876419,-79.49258020615676,-79.49272059911443,-79.49289197898757,-79.49305787415069,-79.49324853697986,-79.49345390740348,-79.49365812315274,-79.49384696803995,-79.49405861911286,-79.49424757625552,-79.4944106947362,-79.49454879192677,-79.49474632871281,-79.49492266192206,-79.49511957413809,-79.4953087138971,-79.49552312048853,-79.49572578134648,-79.49594090210914,-79.49612279684128,-79.49628022119198,-79.49639231565104,-79.4965262163627,-79.49664279609931,-79.49676325419276,-79.49686806578787,-79.49719140356373,-79.49726390833884,-79.4979480071683,-79.49827949070185,-79.4983787019449,-79.49862021794681,-79.49899572504859,-79.49930359719255,-79.49965857888816,-79.50000674593264,-79.50035704788885,-79.50040245072664,-79.50069231490416,-79.50104521803692,-79.50141750868143,-79.50155812247151,-79.50159562047968,-79.49684685602216],"lat":[43.61222976399194,43.61551587227427,43.61559723827819,43.61570676426922,43.61660232350849,43.61740522147806,43.61844955475912,43.61979026092745,43.62015306485195,43.62091423291297,43.62180396157383,43.62253850298347,43.62388344514069,43.62383751621444,43.62380363175079,43.62374282945266,43.62363702044484,43.62353613295188,43.62342164713888,43.62332072817974,43.62320188118496,43.6231014248466,43.62304690622708,43.62296995547424,43.62291095757876,43.62284694837175,43.62277232880258,43.62274302138561,43.62271344826157,43.62267017972851,43.62262702596909,43.6225837241849,43.62253172236463,43.62246143605489,43.62239579260087,43.62230265667034,43.62221866445341,43.6221481691205,43.62208657433845,43.62202969209797,43.622004603436,43.62200222658692,43.62200475064803,43.62197614944336,43.62190488659704,43.62156085045219,43.62119950479574,43.62108374933456,43.62091953756832,43.62074630827999,43.62063123310496,43.62055610735776,43.6205207901582,43.620507722623,43.62041528796584,43.62025864696941,43.62025024415382,43.62023290318218,43.62021126860596,43.62018187606131,43.62015624445837,43.61990855610948,43.61986091845999,43.6198176367024,43.61976523248585,43.61969914401238,43.61963293227216,43.61956645722471,43.61949544803505,43.61942433865392,43.6193309546673,43.6189204318531,43.61809799098464,43.61727532227414,43.61572927779957,43.61494262014975,43.61403982235991,43.61377177056479,43.61344958366352,43.61321316523691,43.61299817764538,43.61272448264056,43.61213602218238,43.61210008628294,43.61114295108047,43.61093630775272,43.61046007050609,43.60966507028449,43.60855845483992,43.60830238738973,43.60696729197851,43.60673809007552,43.60649597172358,43.60623169564625,43.60595864280288,43.60566808195425,43.6053907637892,43.60514513937759,43.60493557985964,43.60470834019506,43.60449525097222,43.60427788835587,43.60410542648891,43.6039734200301,43.6038276707845,43.60371781526076,43.60364360870901,43.60306407121831,43.60279215764202,43.60253226122765,43.60238794828398,43.60226630706975,43.60220341541334,43.60212781137544,43.60205764567309,43.60201489483845,43.60200357895804,43.60206010546752,43.6021168255192,43.60209701171153,43.60209950036512,43.60208878596961,43.60208677436504,43.60212943283975,43.60221677226976,43.6022148728458,43.6021946895726,43.60215227160393,43.60211425204863,43.60205856046417,43.60200271389103,43.60194703091961,43.60185940284683,43.60175335822561,43.60163780494153,43.60154504376427,43.60143396028271,43.60132301770901,43.60122087139128,43.60211201508013,43.60239653609788,43.60235610647916,43.60313014601996,43.60334759108179,43.60397650316352,43.60485032237773,43.60564223292472,43.60647527060336,43.60729921550265,43.60813218939649,43.60829932473023,43.60897846810493,43.60979797059779,43.61068065105251,43.61103816909878,43.61117819383852,43.61222976399194]}]],[[{"lng":[-79.48874566774727,-79.48843864924198,-79.48823258426253,-79.48791597188352,-79.48749674866882,-79.48708875855337,-79.48686752304681,-79.48670865582493,-79.48661542248503,-79.48665487089957,-79.48675405089658,-79.48676344551022,-79.48676470091411,-79.48702464933487,-79.48709537551078,-79.48708009162561,-79.48700559779078,-79.48694590546455,-79.4887334253124,-79.48969605939226,-79.49085090559777,-79.49184368829629,-79.49232674894277,-79.49410063401712,-79.49548168022565,-79.49673382832636,-79.49812865900168,-79.49834332751216,-79.50255931504039,-79.50755463182067,-79.50755473396974,-79.50757417424305,-79.50785127472827,-79.5081302495269,-79.50891828818975,-79.50950760783931,-79.50987748266603,-79.51098231170432,-79.51293505053584,-79.51323762133885,-79.51315536087145,-79.51311591622628,-79.51063206176994,-79.50848114691989,-79.50506890188572,-79.50227562242685,-79.50107875744079,-79.50101120434449,-79.49451886697167,-79.49317690497385,-79.48874566774727],"lat":[43.6258841213065,43.62561854989806,43.62543569213702,43.62520645097531,43.62511979488774,43.62507838655065,43.6250034409025,43.62491131639597,43.62467593689982,43.62458643808725,43.62448296665332,43.62443033864133,43.62442918498776,43.62421567540178,43.62410156458618,43.62399783793678,43.62391583277996,43.62388344514069,43.62253850298347,43.62180396157383,43.62091423291297,43.62015306485195,43.61979026092745,43.61844955475912,43.61740522147806,43.61660232350849,43.61570676426922,43.61559723827819,43.61344648900321,43.61090003543246,43.61090003676629,43.61100228485081,43.61175506076244,43.61296543901098,43.61458287358791,43.61531091098728,43.61579293688605,43.61792328670479,43.62018618901327,43.62046550035119,43.62048171435102,43.62048570227856,43.62097107461695,43.62140222432004,43.62208249530713,43.62267616457273,43.62292612703108,43.62293424320003,43.62448324505095,43.6247986390877,43.6258841213065]}]],[[{"lng":[-79.50755808168515,-79.50755473396974,-79.50755463182067,-79.50255931504039,-79.49834332751216,-79.49818675404019,-79.49684685602216,-79.50159562047968,-79.50155812247151,-79.50141750868143,-79.50104521803692,-79.50069231490416,-79.50040245072664,-79.50035704788885,-79.50000674593264,-79.49965857888816,-79.49930359719255,-79.49899572504859,-79.49862021794681,-79.4983787019449,-79.49827949070185,-79.4979480071683,-79.4984887408371,-79.49949334562832,-79.49964212994281,-79.50075225136777,-79.50187034754349,-79.50301899363764,-79.50413177255128,-79.50526301681009,-79.50629760315799,-79.50657703922784,-79.50655875692941,-79.50757600639908,-79.50757739841283,-79.50756911965306,-79.50755808168515],"lat":[43.6109175446305,43.61090003676629,43.61090003543246,43.61344648900321,43.61559723827819,43.61551587227427,43.61222976399194,43.61117819383852,43.61103816909878,43.61068065105251,43.60979797059779,43.60897846810493,43.60829932473023,43.60813218939649,43.60729921550265,43.60647527060336,43.60564223292472,43.60485032237773,43.60397650316352,43.60334759108179,43.60313014601996,43.60235610647916,43.60225500544405,43.60202514028675,43.60199108483225,43.6017535014494,43.60150709965414,43.60125199484032,43.60100091012682,43.60075464713208,43.6032622186165,43.60401772966965,43.60675918485854,43.61088276036018,43.61088844988436,43.61089266281148,43.6109175446305]}]],[[{"lng":[-79.36898666219723,-79.3653694258756,-79.36182587613639,-79.36167994434753,-79.36102966565109,-79.36073858361942,-79.36095145075473,-79.36018993545918,-79.36112839869676,-79.36239522557214,-79.36373659254434,-79.36373800336401,-79.36428682555319,-79.36489125404348,-79.36617748156483,-79.36742841280238,-79.3682863286095,-79.36911673102959,-79.36898666219723],"lat":[43.64393418432234,43.64518807288613,43.64652401120835,43.64670196043981,43.64773264404664,43.64720616999458,43.64641686211617,43.64529400396917,43.64495179425395,43.64447475049811,43.64396275162323,43.64396727285281,43.64376349075087,43.64353358542105,43.64304768500306,43.64258836204199,43.64387017507319,43.64356686593566,43.64393418432234]}]],[[{"lng":[-79.52171303334833,-79.52130682521991,-79.52022923995233,-79.51913271769878,-79.51803569021483,-79.51693844576398,-79.51601390592387,-79.51504776210602,-79.51393611762015,-79.51290011852122,-79.51238263731472,-79.51235500258962,-79.51185044978129,-79.51183809209152,-79.51076300262736,-79.51060422150761,-79.50969932816163,-79.50863608798618,-79.508621270001,-79.50760718587881,-79.50642116091629,-79.50526301681009,-79.50413177255128,-79.50301899363764,-79.50187034754349,-79.50075225136777,-79.49964212994281,-79.49949334562832,-79.4984887408371,-79.4979480071683,-79.49726390833884,-79.49719140356373,-79.49686806802882,-79.49738576225184,-79.49747108792258,-79.49756837134952,-79.49766669317765,-79.49778075257314,-79.49790379896172,-79.4979940927204,-79.49806731826533,-79.49814454616366,-79.49817272428557,-79.49817467242143,-79.49812334256073,-79.49804686193688,-79.49796840939047,-79.49798075927907,-79.49795410203424,-79.49792700913682,-79.49788827151538,-79.49782764919085,-79.49785165153018,-79.49798449445103,-79.49805279419836,-79.49811005429108,-79.49819818218923,-79.49827687082697,-79.49835166097913,-79.4983391498841,-79.49831008453926,-79.49838205016181,-79.49846743635446,-79.4985696400628,-79.49874104777447,-79.49884909330098,-79.49886230214057,-79.49897619271836,-79.49914194406058,-79.49929681165769,-79.49943080132688,-79.49955098132762,-79.49964142055231,-79.49972135985686,-79.49978201872111,-79.49982385497169,-79.49989339159657,-79.49991817360997,-79.49999036204002,-79.50008369735529,-79.50015189033319,-79.50022515059966,-79.50029519310461,-79.50034184672823,-79.50035870901291,-79.50041197358682,-79.50047057114942,-79.50046581762864,-79.50042777435185,-79.50037739144064,-79.50042498441128,-79.50046408188224,-79.50058587322864,-79.50070202832556,-79.50085213793734,-79.50090613863441,-79.50117116093628,-79.50126547669055,-79.50130298404513,-79.50142507675514,-79.50153513813014,-79.50159358029264,-79.5017006990481,-79.50182425919178,-79.50190704641713,-79.50195560165608,-79.50208235572038,-79.50219182256932,-79.50230877258484,-79.50249669644381,-79.50265513339926,-79.50266054580848,-79.50284342737628,-79.50300432441669,-79.5031435717201,-79.503201094017,-79.5032531944657,-79.50336690419783,-79.50355333097249,-79.50372047557424,-79.50390290058449,-79.50401024719497,-79.50410858944869,-79.50413247633976,-79.50420059087024,-79.50436907141525,-79.50453564904839,-79.50469690705474,-79.50484817582959,-79.50495331387033,-79.5050086170378,-79.50503167787191,-79.50505188855524,-79.50517202837176,-79.50532157144318,-79.50545308302746,-79.50555551784528,-79.50564735960945,-79.50578045078262,-79.50590487841387,-79.50603546152765,-79.50617408231115,-79.5062590006992,-79.50629783568809,-79.50630643408732,-79.50631252548096,-79.50630921263476,-79.50631168162792,-79.50634368405852,-79.50639989834734,-79.50650205680772,-79.50662937376914,-79.50673924648143,-79.50681736882984,-79.50691593362905,-79.50704921030383,-79.50717222756833,-79.50728626722241,-79.50740513375725,-79.50741044402528,-79.50753083307205,-79.50761869307235,-79.50770644181345,-79.50777958730133,-79.50779593565215,-79.50788084310389,-79.5079989305389,-79.50813029577917,-79.50825085207735,-79.50835068690861,-79.50842849982789,-79.50844495924694,-79.50852171469579,-79.50861744635293,-79.50870572612106,-79.5087850854751,-79.50884481083645,-79.50891989938067,-79.50900474342696,-79.50907942939513,-79.50915629578213,-79.50925411468593,-79.50936854317008,-79.50946088672289,-79.5095676470894,-79.50966939455274,-79.50974554042241,-79.50982096256637,-79.50990110015702,-79.50997954635105,-79.51006635968997,-79.51014282992853,-79.51021587262423,-79.5102868906836,-79.51035627477046,-79.51042330115713,-79.51045682322237,-79.51056099909158,-79.51062327903364,-79.51071103998099,-79.51079544604737,-79.51083534309961,-79.51083242122449,-79.51076454924363,-79.51072786792983,-79.51068590127851,-79.5105779170337,-79.51044828392052,-79.51032355714079,-79.5102395313104,-79.51012954066373,-79.51003836657827,-79.50993303373626,-79.50979829980577,-79.50976496067717,-79.5097482005016,-79.5097789827949,-79.50983813830833,-79.50998801899343,-79.51010920672287,-79.51023931441523,-79.51041080891673,-79.51056102500171,-79.51060929814565,-79.51063986036984,-79.51065993764601,-79.51066563396327,-79.51064596062099,-79.51061207512133,-79.51053290689646,-79.5104044510895,-79.51022137989425,-79.51001502587296,-79.50982134354466,-79.50974482648344,-79.50970515621491,-79.50968030202019,-79.50966872317146,-79.50968550626013,-79.50973066914737,-79.50980124147935,-79.50987229761382,-79.51004409951159,-79.51014846398921,-79.51024785139282,-79.5103152004325,-79.51035785469013,-79.5104340923519,-79.51050737062221,-79.51057490128302,-79.5106174041879,-79.51066542133185,-79.51069617172337,-79.51076130423859,-79.51081491728107,-79.51086295575918,-79.51089448251248,-79.51092480146549,-79.51093960089069,-79.51093109792644,-79.51090566145938,-79.51086037924874,-79.51071052427412,-79.51051582996547,-79.51029589590583,-79.51008862460782,-79.50987403222447,-79.50967031787815,-79.50946053438126,-79.50926321810557,-79.50905425202969,-79.50883484350659,-79.50874488866393,-79.50871106770366,-79.50867452574467,-79.50866618542821,-79.50867965250885,-79.50871394184558,-79.50877598816089,-79.50884974153816,-79.50905817436346,-79.50927011490067,-79.50946259626069,-79.50952678869449,-79.50957191037745,-79.50965704785727,-79.50981824682977,-79.50994401400393,-79.5100532271755,-79.51016729680188,-79.510244410114,-79.51034910023243,-79.51040720664817,-79.51042629712751,-79.51044291241215,-79.51046523925527,-79.5104685555668,-79.51049151928788,-79.51047184267378,-79.51041515774878,-79.51029874076006,-79.51014047902918,-79.50995673903573,-79.50976419133463,-79.50959464045937,-79.50948089906956,-79.50936841883745,-79.50931414123498,-79.50925469192923,-79.50919813985864,-79.50917880181808,-79.5091759213513,-79.5091924416642,-79.50922946799378,-79.50926915808604,-79.50934297050883,-79.50939938847553,-79.50946405933843,-79.50960022439213,-79.50996464278829,-79.51015748241997,-79.51049131297481,-79.51064932506243,-79.51100625532385,-79.51121538428117,-79.51213197712156,-79.51249674891235,-79.5128534294401,-79.51323825458088,-79.51343655904675,-79.51364537211057,-79.51389745361737,-79.51401311744122,-79.51411805126256,-79.51426551096351,-79.51444029554825,-79.51477169848043,-79.51513120712616,-79.51562626325509,-79.51591467547922,-79.51602224092829,-79.51623474041996,-79.51672397403127,-79.51679787728695,-79.51692469095744,-79.51697338959968,-79.51724458358206,-79.51730443097368,-79.51732366314552,-79.51732370671947,-79.51729822853399,-79.51725635439139,-79.51671846370245,-79.51648687404362,-79.51637875505087,-79.51607590085143,-79.51601703347363,-79.51593167653623,-79.51581575471388,-79.51571408616537,-79.51548457699417,-79.51544211592737,-79.51534049873621,-79.51529813980569,-79.51526103111097,-79.51519027721066,-79.5151568361135,-79.5150432130751,-79.51464986040934,-79.51461596844523,-79.5145567103722,-79.51424536538828,-79.5140904371529,-79.51352121876215,-79.51343370133701,-79.51339959639954,-79.51335734858719,-79.51332013019076,-79.51329526719488,-79.51328346569287,-79.51326399230173,-79.5132521908383,-79.51323569231685,-79.51321590895012,-79.51318754834431,-79.51313047747502,-79.51306308482064,-79.51281112483322,-79.51270916662725,-79.51244366871754,-79.51227611169104,-79.51218874610028,-79.51218066188619,-79.51217164118461,-79.51217484750339,-79.51227087936553,-79.51227070734609,-79.51226238392631,-79.51224559402588,-79.51219458817208,-79.51213521791423,-79.51168792757265,-79.51149019274851,-79.51129485524962,-79.51122174850943,-79.51115914151107,-79.51112276830058,-79.51109978932882,-79.51107424646713,-79.51106022677963,-79.51106019643382,-79.5110687611827,-79.51151847606849,-79.51150216952477,-79.51144257533434,-79.51143120543435,-79.51143665796752,-79.51148796514914,-79.51154735711366,-79.51163790299179,-79.51175350233233,-79.51190376266666,-79.51212172891707,-79.51226258638198,-79.51256852052965,-79.51291111242392,-79.51300359418353,-79.51306395306901,-79.51316585757196,-79.51335233066099,-79.51355281048195,-79.51375694012144,-79.51397719718652,-79.51418935882675,-79.51439337680843,-79.5145826813805,-79.51464560123375,-79.51471299699844,-79.51478088844091,-79.51487620336017,-79.51492268822035,-79.51491120859035,-79.51492508770725,-79.51495072716108,-79.51501294092014,-79.51507243076522,-79.51510079324069,-79.51512632209143,-79.51514595383874,-79.5152790553643,-79.51529021998275,-79.51529311327786,-79.51528486751837,-79.51525956004133,-79.51507798284811,-79.51503573289631,-79.51501830540933,-79.51500766910316,-79.5149958665026,-79.51497608188988,-79.51495947138085,-79.5149366844075,-79.51493909284089,-79.51495364430821,-79.51497022096126,-79.51499066085931,-79.51502171207679,-79.51508899671964,-79.51530129404334,-79.51537251868113,-79.51540913515012,-79.51542327798859,-79.51543146196424,-79.51546836615893,-79.5154740420929,-79.515485955578,-79.51551399481274,-79.51554176172408,-79.51557608952595,-79.51564371545392,-79.51573998459273,-79.51579673497248,-79.51583386218886,-79.51586181594844,-79.51587630742048,-79.51589547251825,-79.51588439240516,-79.51585632697028,-79.51581733539271,-79.51578920455374,-79.51576576067431,-79.51569230630439,-79.51567793366793,-79.51568094590934,-79.51570052609036,-79.51570295109303,-79.51567781381951,-79.51565224265723,-79.51563841460973,-79.51564389456692,-79.51567480905838,-79.51568877261973,-79.51568623621726,-79.51566897882374,-79.51564056566764,-79.5156100678284,-79.51557005650784,-79.51544558924417,-79.51529258990628,-79.51516227664021,-79.51508066273486,-79.51504911833712,-79.51503557144321,-79.51499886453939,-79.51496472541719,-79.51493321569406,-79.51489917887098,-79.51489405658529,-79.51490259171912,-79.51494939331484,-79.515088623528,-79.51522949815666,-79.51536347786738,-79.51546688980466,-79.51553607940981,-79.51558505083256,-79.51566566014712,-79.51578547224294,-79.51594398513014,-79.51608910456301,-79.516161085492,-79.51621531573286,-79.51626010015489,-79.51633692925047,-79.5163707213124,-79.51639609574697,-79.51643245612405,-79.51647776023796,-79.5165094475406,-79.51650610412665,-79.51647246800881,-79.51640711786487,-79.51636264912196,-79.51631436209233,-79.51625759508016,-79.51622329501305,-79.51626337645698,-79.51632014261017,-79.51639100290684,-79.51645832319166,-79.51652198442386,-79.51661165975203,-79.51671921163219,-79.51680375536918,-79.51690291008893,-79.51700040598018,-79.51710648814053,-79.51722274692455,-79.51731529666957,-79.51741358988804,-79.517563765766,-79.51773464579942,-79.51790443575449,-79.51805919799695,-79.51819143672408,-79.51827127650988,-79.51834456451823,-79.51842844783091,-79.5184876338113,-79.51855355069058,-79.51864708015356,-79.51869671652248,-79.5187523168848,-79.51881283524845,-79.51886291523346,-79.5188903068855,-79.51892922930254,-79.51896229815948,-79.51900192713251,-79.51902850182567,-79.51904080509138,-79.51912697019219,-79.51923810840464,-79.51964602029138,-79.51997766124103,-79.52033193269808,-79.52068009449981,-79.52129857549201,-79.52133369834193,-79.52171303334833],"lat":[43.59711926456993,43.5971995307201,43.59743764545102,43.59769359899455,43.59794062330015,43.59819195527098,43.59838259400087,43.59859059902794,43.59882830180275,43.59908039117446,43.59919770698205,43.59920391888349,43.59931887993869,43.59931871904305,43.59954786465883,43.59958180436205,43.59978606110942,43.60000633901192,43.60001064674424,43.60021346210245,43.60048612901787,43.60075464713208,43.60100091012682,43.60125199484032,43.60150709965414,43.6017535014494,43.60199108483225,43.60202514028675,43.60225500544405,43.60235610647916,43.60239653609788,43.60211201508013,43.60122078139923,43.60076847866598,43.60070208422084,43.60058624503296,43.6004615072768,43.6003324751962,43.600203561105,43.60006062322409,43.59992205208309,43.59977453132761,43.59963977924423,43.59949577037098,43.59938256897647,43.59928245027323,43.59919589885777,43.59910603960108,43.59896615588411,43.59883967964434,43.59871314038259,43.59859981672027,43.59850560953325,43.59842624589714,43.59839113479124,43.59836037966095,43.59827151607139,43.59815552192425,43.59804838863741,43.59793119622832,43.59781828740049,43.59778772533492,43.59787886865009,43.5979387252682,43.59792297244482,43.5978793807372,43.59781653911628,43.59780903268186,43.59781120928813,43.59774572657468,43.59765296308372,43.5975239194964,43.59740357756583,43.59727409558173,43.59713526829643,43.59700978718866,43.59692067825269,43.59688949594445,43.59675532093399,43.59661701232407,43.59648737587445,43.59637581019075,43.59624160681679,43.59610718654685,43.59595887214562,43.59583795179703,43.59575320005024,43.59563610959908,43.5955184923762,43.5953918011018,43.59534291363363,43.59530291685076,43.59522799624521,43.5951620037376,43.59507386103785,43.59498904876255,43.5948889995017,43.59486322975896,43.59485021832054,43.59482481265526,43.59479024696668,43.59477750988639,43.59473390329917,43.59469051211745,43.59465999962995,43.59463813061546,43.5945812777701,43.59453770149227,43.59448071991887,43.59442466811759,43.59440423837736,43.59440430928094,43.59436619517903,43.59434129609595,43.59431611326453,43.59430336331332,43.59429504341587,43.59427393695519,43.59424937138073,43.59421555091933,43.59419993455061,43.59424184932617,43.59424313642111,43.59424795011967,43.59424884153292,43.59426004842291,43.59425322584023,43.59425083450338,43.5942393101064,43.59421367887867,43.59413788384344,43.59404816382852,43.59395390544778,43.59388345940621,43.59381780880359,43.59372950684141,43.59362732130962,43.59351149394334,43.59345012872331,43.59342924986459,43.59344445998659,43.59347336827637,43.59354649539642,43.5936325234773,43.59372265749143,43.59379025333012,43.59386222735375,43.59395237128528,43.59403380897378,43.59411106195946,43.59414390443639,43.5941590712028,43.5941650077649,43.59416602836419,43.59415831378552,43.5941555536068,43.59412565278542,43.5940911335048,43.5940431738958,43.59404324324048,43.59399980446418,43.59398294733524,43.59397058976833,43.59395804148438,43.59395375383425,43.59393235687484,43.5939158940518,43.59389951439125,43.59387408136693,43.59384387671223,43.59379538026351,43.59378659288711,43.59375608690505,43.59373483063965,43.59376298897613,43.59380903519444,43.59386832846125,43.59390531670332,43.59391542565823,43.59389839555391,43.59386338951035,43.59381965452334,43.59379414039655,43.59378184140368,43.59376072816244,43.59374405054182,43.59376304769798,43.59378653645054,43.59380108446225,43.59379310495515,43.59378964552398,43.59379964447827,43.59378709329329,43.59376551350815,43.59374391238354,43.59372678157342,43.59371371518642,43.59376008357705,43.59371588430178,43.59364501050712,43.59351548883333,43.59337647506852,43.59330441967079,43.5932316080641,43.59308700544472,43.59303244563197,43.59295001905815,43.59286731025637,43.59283417716944,43.59282407985839,43.59282714729385,43.59281695665896,43.59278857703826,43.59275531291969,43.59272787175025,43.59268264239613,43.59266053831672,43.59264771628802,43.59266326350141,43.59267834648959,43.59267554127635,43.59265968201489,43.59266163957538,43.59267586188363,43.59266266684973,43.59263142087531,43.59259098534608,43.59257272466503,43.59256328095691,43.59257575257698,43.59256957748561,43.59255368824173,43.59253749519787,43.59251696590498,43.59247995962347,43.59245243588928,43.59242060424052,43.5923889456813,43.5923711601762,43.59235824577833,43.59235466481141,43.59236459335944,43.59239834059693,43.59239970099159,43.59238749317002,43.59236136445529,43.59232591169492,43.59224588574617,43.59216573115842,43.59209909490006,43.59203663356431,43.59197424406287,43.59191613064407,43.59182245651951,43.59171954008225,43.59159413559057,43.59150452463769,43.59140589577063,43.59130697469048,43.59121684222357,43.59111748697809,43.59103587750978,43.59095731658934,43.59088276227102,43.59081688083718,43.59075116404058,43.59072136010316,43.59066469108934,43.59063044799221,43.59059186602678,43.5905576328449,43.59053676630379,43.5904814896726,43.59044053864633,43.59038154776758,43.59031392267109,43.59026908753077,43.59022902514545,43.590198327056,43.59018578600331,43.59017950283646,43.59015526069647,43.59010825897985,43.59007749844612,43.59003307596863,43.58994416438429,43.58981123354955,43.58967775056731,43.5895486427644,43.5894421035842,43.58933058153107,43.5891653157734,43.58900853502815,43.58885574688343,43.58871633971275,43.58863111002852,43.58855463477396,43.58840630814638,43.58823501043153,43.58808573585142,43.58793559273562,43.58782550388439,43.58774208919984,43.58767206264771,43.58764284542114,43.5876368611662,43.58761288889649,43.58758967559682,43.58754839047172,43.58748904873282,43.58745728889825,43.58741674154276,43.58737644722984,43.58734542254093,43.58730993149368,43.58726588326969,43.58723961250718,43.58722245151346,43.58719722056237,43.58720647281804,43.58723149196577,43.58732136341964,43.58735042896553,43.58737758467879,43.58737580771557,43.58725271015386,43.58718544044127,43.58713147753062,43.58709597397103,43.58709405214111,43.5870967677078,43.58711354872643,43.58712855575298,43.58714792426393,43.5871949420061,43.58726473011819,43.58745358145533,43.58762938370766,43.58783836279127,43.58797714053262,43.58801004472661,43.58806240541096,43.58822179218753,43.58823625446403,43.5882424010522,43.58824753400302,43.58836808063116,43.58841836894054,43.58845462711515,43.58848613529202,43.58851290133981,43.58851685923158,43.58838375982788,43.58832674120827,43.58831633570434,43.58825389013133,43.58822611933946,43.58816649697115,43.58812448201122,43.58810515746173,43.58803006945548,43.58802051579752,43.58802819810771,43.58801864572293,43.58798665608419,43.58787320979889,43.5878457688567,43.58777677644913,43.58762312958301,43.58761818806151,43.58763542231606,43.58779791592698,43.5878634182809,43.58799104965649,43.58799441254921,43.58799396898445,43.58797991623837,43.58795242563097,43.58792509571801,43.58790243677914,43.58778065412589,43.5877579951838,43.58773527514369,43.58772151454816,43.58770764237779,43.58770239890501,43.5877015222159,43.5877792637724,43.5878049436551,43.58786000280489,43.58790733398896,43.58793770442782,43.5879466013728,43.58796448830158,43.5879960376479,43.58818192222448,43.58821792868189,43.58824482686683,43.58826711376348,43.58829345637484,43.5883151889963,43.58843548614998,43.58850942974463,43.58861041073936,43.58865897049262,43.58871666913817,43.58877921055886,43.58883292427086,43.58889119568759,43.58895852935652,43.58902604525635,43.58908467094682,43.59012604808233,43.5901618444129,43.59019257591964,43.59021043218848,43.59024201079666,43.59032369850963,43.59039648931066,43.59046518466979,43.59051620182669,43.59054516470292,43.59062461070737,43.59065795179694,43.59071594646066,43.59082843026432,43.59084763778903,43.59084392193019,43.59079573562588,43.59073964703219,43.59067473811727,43.59062337956401,43.59057664138313,43.59054339082155,43.59049653076278,43.59045398021402,43.59044579571459,43.59044667151448,43.59045646586502,43.59054322497335,43.59063844176758,43.59066079804259,43.59066547945661,43.59066131149482,43.5906486165341,43.59077992088768,43.59079379261003,43.59079412425753,43.59078087603506,43.5905979705249,43.59058011120624,43.5905621444463,43.59054853408076,43.59053920317312,43.5904918335205,43.59047778137006,43.59045955061234,43.59036038854036,43.59033772977159,43.59032396946878,43.59030574931611,43.59009840330665,43.59006692699692,43.59004010954157,43.59002232056812,43.59001358395612,43.59000948628785,43.59001486150763,43.59007154233581,43.59009947399682,43.59012245502496,43.59014964523384,43.59019485241094,43.59076705963301,43.5907896387792,43.59080779785198,43.59083066744454,43.59083552915166,43.59082697280242,43.59081434778653,43.59077508817053,43.59073531532457,43.59070428982041,43.59066414302008,43.59061481924232,43.59052495635285,43.59044379294216,43.5903264002967,43.59020886576774,43.59012739091867,43.59010458106377,43.59006761852034,43.59005392861821,43.59002696121923,43.58998670571319,43.58994622742914,43.58990089013467,43.5898690504583,43.5898373632783,43.58980142575383,43.58970730440692,43.5896129629261,43.58949590174173,43.5894416645975,43.58940078583662,43.58938238542798,43.58937286363246,43.58936674601419,43.5893827630793,43.58939007247508,43.5894025154892,43.58939760459422,43.58937942425145,43.58929783778041,43.58920737250671,43.58913044464442,43.58903998067807,43.58899490325729,43.58896350654646,43.588946110278,43.58892091261878,43.58889114504206,43.5888883842774,43.58890332071601,43.58892672476082,43.58894986619353,43.58898242063978,43.58901998517733,43.5890580520784,43.58910494705323,43.58911038261064,43.58910658552161,43.58907565927004,43.58894612504452,43.58889696169386,43.58887028453044,43.58886625538884,43.58888484775266,43.58892135773865,43.58896182412748,43.58903790604131,43.58914508391668,43.58922552627721,43.58932833446734,43.58943121259986,43.58953429233384,43.58966984523084,43.58978319927844,43.58989664621555,43.58999204281553,43.59007838971259,43.59017857744422,43.59027458602577,43.59033419723797,43.59038949688585,43.59045827818252,43.59052266970514,43.59055118446028,43.5905343806921,43.59050414808964,43.59047008722724,43.59044979792112,43.59041148988054,43.59038198892143,43.59038370335902,43.59044775358962,43.59052072100676,43.59060732897153,43.59068470461072,43.59071256549188,43.59071827877251,43.59071442097098,43.59068813500744,43.59065291057587,43.59063096408524,43.59061781577242,43.59060931795153,43.59061874861451,43.59061476100995,43.59060160207853,43.59057475497307,43.5909765580198,43.59118063674399,43.59210432195034,43.59291449089616,43.59375195825157,43.59458484425004,43.59610989214588,43.59622746468332,43.59711926456993]}]],[[{"lng":[-79.5196376430393,-79.51963723006801,-79.51499727860413,-79.50757739841283,-79.50757600639908,-79.50655875692941,-79.50657703922784,-79.50629760315799,-79.50526301681009,-79.50642116091629,-79.50760718587881,-79.508621270001,-79.50863608798618,-79.50969932816163,-79.51060422150761,-79.51076300262736,-79.51183809209152,-79.51185044978129,-79.51235500258962,-79.51238263731472,-79.51236900410142,-79.5133147289296,-79.51503491258244,-79.51544467317675,-79.5159946662926,-79.51675448886401,-79.51765569804266,-79.51861729911133,-79.51841081536068,-79.51877335843082,-79.51889406493599,-79.51963734103633,-79.5196376430393],"lat":[43.604727673127,43.60472784782134,43.60710021856585,43.61088844988436,43.61088276036018,43.60675918485854,43.60401772966965,43.6032622186165,43.60075464713208,43.60048612901787,43.60021346210245,43.60001064674424,43.60000633901192,43.59978606110942,43.59958180436205,43.59954786465883,43.59931871904305,43.59931887993869,43.59920391888349,43.59919770698205,43.5993156378533,43.60158028231128,43.60120195795681,43.60218752546933,43.60206593688257,43.60386596778722,43.603663498017,43.60345622252373,43.60295671733036,43.60287562568648,43.60284046110714,43.60472748917251,43.604727673127]}]],[[{"lng":[-79.51323762133885,-79.51293505053584,-79.51098231170432,-79.50987748266603,-79.50950760783931,-79.50891828818975,-79.5081302495269,-79.50785127472827,-79.50757417424305,-79.50755473396974,-79.50755808168515,-79.50756911965306,-79.50757739841283,-79.51499727860413,-79.51963723006801,-79.5196376430393,-79.52158304648044,-79.52419120947624,-79.52629307237137,-79.52669522350875,-79.52651904668451,-79.52645523800987,-79.5263913193932,-79.5263290425394,-79.52626676559441,-79.52620459069254,-79.52614362584528,-79.52608288063527,-79.5260439998689,-79.52600593392985,-79.52596848294544,-79.52593114179547,-79.52589460992699,-79.52585889508852,-79.5258236831188,-79.52578939807219,-79.52575582792507,-79.52572317694907,-79.52569042379403,-79.52565848547898,-79.52562786930521,-79.52559756726723,-79.52556787022347,-79.52553899799956,-79.52551073076813,-79.52548328057971,-79.52545664743607,-79.52543083133902,-79.52540562801849,-79.52538155374954,-79.52535716744408,-79.52533431314369,-79.52531156094487,-79.52529044509076,-79.52527004196034,-79.52524943452602,-79.52523055557661,-79.52521158226462,-79.52519424091133,-79.52517619458183,-79.52515885537515,-79.52514223110006,-79.52512631394551,-79.52511143376205,-79.52509715856561,-79.52508462749522,-79.52507177436148,-79.52505985607787,-79.52504874485864,-79.52503906574576,-79.52502867165475,-79.52502062671277,-79.52501269171769,-79.52500606885688,-79.52499966590631,-79.52499457509191,-79.52498980633041,-79.52498554042758,-79.52498280439043,-79.524980274824,-79.5249791751891,-79.52497879052029,-79.52497932509606,-79.52497954543055,-79.52498172204913,-79.52498440153785,-79.52498728530719,-79.52499149468549,-79.52499356135134,-79.52499634299227,-79.52499983960988,-79.5250041455204,-79.52500928419923,-79.52501513003534,-79.52502189513362,-79.52502917093831,-79.5250373660087,-79.52504647814762,-79.5250568181746,-79.52506736249569,-79.52507883390996,-79.52509161751965,-79.52510452674359,-79.52511895244875,-79.52513337817464,-79.52514933602428,-79.52516529951184,-79.52518207452931,-79.52520027953702,-79.52522022876984,-79.52551144866098,-79.52815269335164,-79.52826065113267,-79.52827574440603,-79.53205773840251,-79.52933773180214,-79.52936279755659,-79.53009772975052,-79.53015398535015,-79.53023191188031,-79.52830090028689,-79.52576887245891,-79.52201424239971,-79.52045454145127,-79.51668803745605,-79.51504383618514,-79.51323762133885],"lat":[43.62046550035119,43.62018618901327,43.61792328670479,43.61579293688605,43.61531091098728,43.61458287358791,43.61296543901098,43.61175506076244,43.61100228485081,43.61090003676629,43.6109175446305,43.61089266281148,43.61088844988436,43.60710021856585,43.60472784782134,43.604727673127,43.6037535240624,43.60252215739999,43.6014733392963,43.60127138405895,43.60144474511119,43.60148893329256,43.6015376210991,43.60158182895821,43.60162603678175,43.60167024588666,43.60172347271769,43.60176770019093,43.60180770851112,43.60184781734733,43.60188784407958,43.6019233711374,43.6019634096983,43.60200345878465,43.60204801542351,43.60208808292832,43.60212815964336,43.60216824820412,43.60220833543725,43.60224852319288,43.6022931390504,43.60233325787357,43.60237788557065,43.60241802282195,43.60246266894811,43.60250732560723,43.60255199279987,43.60259667052669,43.6026413561529,43.60268164528339,43.60272634143884,43.60277105735445,43.60281577458244,43.60286042289653,43.60290517042379,43.60294991531095,43.6029991835738,43.60304394953809,43.60308882657775,43.60312910341903,43.60317389045879,43.6032186867207,43.60326799318157,43.60330831086628,43.60335313743266,43.60339348542607,43.6034428314161,43.6034831873137,43.60352814472301,43.60357303058993,43.60361790722921,43.60366281418213,43.60370322147606,43.60375264785572,43.60379307491877,43.60384252106777,43.60388296921865,43.60392792493516,43.60397299041558,43.60401796853859,43.60406296511448,43.60410797091668,43.60415298858142,43.60420250326707,43.60424304104515,43.60428808638891,43.60433313436851,43.60437828947613,43.60442332691027,43.60446837357073,43.60451342945742,43.60456299686376,43.60460357286372,43.60464865906503,43.60469375712849,43.60473886178171,43.60478397829681,43.60482919666679,43.60487434085929,43.60491948768681,43.6049601454004,43.60500982219717,43.60504599738523,43.60509119429187,43.60513639119662,43.60518160786908,43.60522241355748,43.60526764076904,43.60531288642915,43.60535365351596,43.60600493545603,43.60541206817837,43.60538852236651,43.60541482295201,43.61440528967879,43.61502432891052,43.61508181500689,43.61680203498933,43.61694679261544,43.61714584169619,43.61752617999622,43.61800225957607,43.61873713473256,43.6190411219097,43.61977566800754,43.62010999196718,43.62046550035119]}]],[[{"lng":[-79.5196376430393,-79.51963734103633,-79.51889406493599,-79.51877335843082,-79.51841081536068,-79.51861729911133,-79.51765569804266,-79.51675448886401,-79.5159946662926,-79.51544467317675,-79.51503491258244,-79.5133147289296,-79.51236900410142,-79.51238263731472,-79.51290011852122,-79.51393611762015,-79.51504776210602,-79.51601390592387,-79.51693844576398,-79.51803569021483,-79.51913271769878,-79.52022923995233,-79.52130682521991,-79.52171303334833,-79.52248134616339,-79.52283030828757,-79.52316994785863,-79.52350581769001,-79.5238375875212,-79.52419803460234,-79.52622296597133,-79.52645680890598,-79.52669522350875,-79.52629307237137,-79.52419120947624,-79.52158304648044,-79.5196376430393],"lat":[43.604727673127,43.60472748917251,43.60284046110714,43.60287562568648,43.60295671733036,43.60345622252373,43.603663498017,43.60386596778722,43.60206593688257,43.60218752546933,43.60120195795681,43.60158028231128,43.5993156378533,43.59919770698205,43.59908039117446,43.59882830180275,43.59859059902794,43.59838259400087,43.59819195527098,43.59794062330015,43.59769359899455,43.59743764545102,43.5971995307201,43.59711926456993,43.59694906563956,43.59781796370872,43.59865073150451,43.59948344949297,43.60032961667973,43.60113564336796,43.60066656217902,43.60116928692677,43.60127138405895,43.6014733392963,43.60252215739999,43.6037535240624,43.604727673127]}]],[[{"lng":[-79.53249280714208,-79.53134288966577,-79.53034811342961,-79.52837448011874,-79.52760531471408,-79.52669522350875,-79.52645680890598,-79.52622296597133,-79.52419803460234,-79.5238375875212,-79.52350581769001,-79.52316994785863,-79.52283030828757,-79.52248134616339,-79.52171303334833,-79.52133369834193,-79.52129857549201,-79.52068009449981,-79.52033193269808,-79.51997766124103,-79.51964602029138,-79.51923810840464,-79.51912697019219,-79.51904080509138,-79.51902824168431,-79.51903087695884,-79.51904203221513,-79.51906401165664,-79.51915809249698,-79.51932307393768,-79.51935676011557,-79.51939776565349,-79.51943078157096,-79.51947768507408,-79.51952672475738,-79.51968017479398,-79.51982563736709,-79.51998181148193,-79.52014471690234,-79.52030466092086,-79.52042547058706,-79.52054944579642,-79.52073339469553,-79.52091723326716,-79.52094603765799,-79.52111260260666,-79.52130446771393,-79.52148868233856,-79.52166699846842,-79.52183683903434,-79.52201362333011,-79.52217417136222,-79.52234165513282,-79.52249445034649,-79.52266849325858,-79.52283669167417,-79.52301620887833,-79.52318544425819,-79.52332199502371,-79.52338393099596,-79.523475323453,-79.52358093352754,-79.52370015634588,-79.5237947931348,-79.52390084296117,-79.52401454338646,-79.52410840211918,-79.52419062200488,-79.52429185645752,-79.52439522740089,-79.52446607115299,-79.52453107059567,-79.52462674331463,-79.52476432223784,-79.52485746654135,-79.52494858431149,-79.52500220986934,-79.52506982505948,-79.52521856556842,-79.52534035555803,-79.52544706479289,-79.52560033032242,-79.52572584406145,-79.52583331450003,-79.52596035213843,-79.52606099033895,-79.52616795777897,-79.52625433657595,-79.52638815974601,-79.52654728400562,-79.52672220518642,-79.52689570444355,-79.52707308402509,-79.52722034761045,-79.52737160884203,-79.52753330082496,-79.52767674467165,-79.52782379482082,-79.52795832036331,-79.5280902775736,-79.5282330584742,-79.52838316855816,-79.52851562029447,-79.52862615723748,-79.52871171158382,-79.52877868861935,-79.52887609489933,-79.52897269359055,-79.52905918060813,-79.52914910915582,-79.52923279363327,-79.52932310982541,-79.52941565508922,-79.52952525269664,-79.52960927010047,-79.52970071916846,-79.52981786298653,-79.52994243861322,-79.53006864802909,-79.53020781817587,-79.53033911840141,-79.53046166663829,-79.53056563798279,-79.53065850127058,-79.53072289660226,-79.53076853939386,-79.53080089308148,-79.53082745590616,-79.53083292580042,-79.53083543446223,-79.5308665343932,-79.53084286570164,-79.53078694215897,-79.53071987161334,-79.5307005292911,-79.53065870768805,-79.53059945342191,-79.53055588745225,-79.53052495159119,-79.53052363770954,-79.53053559839175,-79.53054762432237,-79.53056560967352,-79.53060397440071,-79.53069718084889,-79.53082355915288,-79.53096398573346,-79.53109664438857,-79.53121656827875,-79.53136366857376,-79.53155989862252,-79.5319678832397,-79.53230918914279,-79.53268279676431,-79.53311978940528,-79.53353393366709,-79.53400425054042,-79.53238762780848,-79.53098002789743,-79.53204025167631,-79.5323823165822,-79.53249280714208],"lat":[43.59840604166732,43.59898109363434,43.59948152045122,43.60046013749575,43.60081490919369,43.60127138405895,43.60116928692677,43.60066656217902,43.60113564336796,43.60032961667973,43.59948344949297,43.59865073150451,43.59781796370872,43.59694906563956,43.59711926456993,43.59622746468332,43.59610989214588,43.59458484425004,43.59375195825157,43.59291449089616,43.59210432195034,43.59118063674399,43.5909765580198,43.59057475497307,43.59052057915979,43.59047560244397,43.59046224372153,43.59044902524207,43.59043223981745,43.59039836845559,43.59034929288124,43.59031831645355,43.5903007397781,43.59028334297807,43.59027047491375,43.59023195258759,43.59020673985613,43.59018625674582,43.59017036165886,43.59015442800226,43.59015149080997,43.59014859446313,43.59012846982402,43.59011284454736,43.5901087161939,43.5901018692534,43.59010885263778,43.59011123565261,43.59010003882127,43.59008873210394,43.59007751492135,43.59006608754317,43.59005925069007,43.59004322154771,43.59002296574134,43.590016137399,43.5900184572434,43.59000263961337,43.58999090056422,43.59001870723328,43.59001988786678,43.59000774881579,43.58997778114126,43.58993399255315,43.58990385453742,43.58987831630365,43.58987052610377,43.58986249552925,43.58984129714958,43.58982462734888,43.58983904516874,43.58980837663165,43.58975559855005,43.5897393699594,43.58979008405624,43.58983176975027,43.58982345960026,43.58981532994702,43.58978124048836,43.58974230204801,43.58974367871289,43.5897501569174,43.58974277371031,43.58971715330863,43.58971429058517,43.58974259490768,43.58971246661738,43.58965956733074,43.58962978509605,43.58961383212552,43.58961608689798,43.58961382198948,43.58961160682088,43.58960450224583,43.58958844681058,43.58956352337279,43.58952036030065,43.58945914912375,43.58937536104025,43.58930054189886,43.58922586197242,43.58916477949292,43.5890989683994,43.58901027958662,43.58889885367194,43.58878268748719,43.58866691290675,43.58854653673682,43.58842612037769,43.58832375259131,43.58823030656449,43.58814135686985,43.58806152798069,43.58798191839292,43.5879334872572,43.58789865492269,43.58788665845716,43.58788826072121,43.58788988386023,43.58789617461053,43.58790686509116,43.58789943854313,43.58788727195552,43.5878614592125,43.58788479244006,43.58794839439003,43.58802082768575,43.58807518218271,43.58810225902408,43.58812929780314,43.5881567040735,43.58820591183256,43.58824579283579,43.58823583850184,43.58819958115406,43.58817203701253,43.58818927964119,43.58822922937143,43.58828743583713,43.58834143201183,43.58839559884528,43.58847227195957,43.58852651624146,43.58854501379871,43.58855971523818,43.58858384515942,43.58857214678503,43.58856484950018,43.58853938406401,43.58850526551176,43.58884086722791,43.58976450996582,43.5905027503692,43.59141694673364,43.59242197694694,43.59335469483597,43.59435573786477,43.59473117737764,43.59505067646703,43.59746841638032,43.59820657541879,43.59840604166732]}]],[[{"lng":[-79.54864509935844,-79.54809843189888,-79.54705334214563,-79.54457409165927,-79.54409051201989,-79.54316176217445,-79.54162011534288,-79.53846230329681,-79.53249280714208,-79.5323823165822,-79.53204025167631,-79.53098002789743,-79.53238762780848,-79.53400425054042,-79.53353393366709,-79.53311978940528,-79.53268279676431,-79.53230918914279,-79.5319678832397,-79.53155989862252,-79.53136366857376,-79.53149195813747,-79.53162014769217,-79.53174608850536,-79.53188257589659,-79.53202683090035,-79.53217414906202,-79.53234523737143,-79.53251447304721,-79.53267463727808,-79.53282774624773,-79.5329639045336,-79.53309913651765,-79.53323590008918,-79.53336041732953,-79.5334730823549,-79.53356452953203,-79.53366619490689,-79.53375419861877,-79.53383105777202,-79.53389189958448,-79.53394601125382,-79.53397514143661,-79.53398946548128,-79.53399001163088,-79.53395207372566,-79.53394262013666,-79.53392550823934,-79.53391676232226,-79.5339167029541,-79.53392867128656,-79.53394267470709,-79.53397218471876,-79.53400105359663,-79.53401876394662,-79.53404516086027,-79.53414278672966,-79.53417848349486,-79.53424456648122,-79.5343670123497,-79.53447944418389,-79.53457881988977,-79.53465955471947,-79.5347648914376,-79.53488984779246,-79.53501939221343,-79.53515304934028,-79.53525788992917,-79.53534097376755,-79.53541066700006,-79.53548725786625,-79.53555530634718,-79.53565046751952,-79.53571290400134,-79.53577149524084,-79.5358632728758,-79.53599610652329,-79.53613417577036,-79.536288378671,-79.53631850168571,-79.5363169630609,-79.53636965975615,-79.53648476762721,-79.53659611807457,-79.53670898625712,-79.53682120715123,-79.53694939199397,-79.53709681509069,-79.53719830835699,-79.5373275848014,-79.53744237268836,-79.53756414497076,-79.53768822433859,-79.53780879745761,-79.53791735540545,-79.53801272933477,-79.53811434324911,-79.53820726390657,-79.53831324090793,-79.53843185225149,-79.53854608625502,-79.53860970725486,-79.53869732615517,-79.53880424145954,-79.53891628225689,-79.53902073291465,-79.5391384643477,-79.53925029334427,-79.5393462703997,-79.5394184737463,-79.53946809066197,-79.53953149220065,-79.53960636999658,-79.53966091671028,-79.53976432436131,-79.53978608020425,-79.53987883498655,-79.53994971638458,-79.54000960988115,-79.54009377827225,-79.54019346901707,-79.54025467589582,-79.54034561585013,-79.54042322177447,-79.54050175311139,-79.54056580704952,-79.54064456205764,-79.54072800714489,-79.54081758501663,-79.54092147735916,-79.54100258857544,-79.54109147706171,-79.54132266810538,-79.54135966548218,-79.5414047598268,-79.54142672197459,-79.54147064989469,-79.54155798563352,-79.54166144066899,-79.54176052665787,-79.54186971202854,-79.54197961179582,-79.54209307187712,-79.54220433949523,-79.54229111242601,-79.54235766835778,-79.54243291796315,-79.54249853492384,-79.54255230114704,-79.54260229580889,-79.54266474902808,-79.5427148518306,-79.54277577086471,-79.54282013614589,-79.54284143474227,-79.54288590828163,-79.54292873513764,-79.5429837055276,-79.54303042053344,-79.54307527608351,-79.54311472011808,-79.54315466168681,-79.54320357094926,-79.54321421572543,-79.54322994625667,-79.54325741832619,-79.54329834836592,-79.5433225402482,-79.54334357554102,-79.54336194119294,-79.54338428230669,-79.54339340123923,-79.54386995896456,-79.54463873249207,-79.5481456013685,-79.54784853627892,-79.54649519799941,-79.54639518012523,-79.54646811710884,-79.54666552814116,-79.5468708213259,-79.54709633181869,-79.54731774853479,-79.54754600178491,-79.54776955049238,-79.54799228546554,-79.54820807153324,-79.54846430859729,-79.54864509935844],"lat":[43.59020704293001,43.59053318086109,43.59113673015076,43.59236576280868,43.5926071640334,43.59307253553216,43.59384524707461,43.59542566314265,43.59840604166732,43.59820657541879,43.59746841638032,43.59505067646703,43.59473117737764,43.59435573786477,43.59335469483597,43.59242197694694,43.59141694673364,43.5905027503692,43.58976450996582,43.58884086722791,43.58850526551176,43.58848440836341,43.58846345976424,43.58844257215677,43.58840381551811,43.58836065738662,43.58831753840864,43.58828822772344,43.5882678951744,43.58823835372166,43.58821331259398,43.58818805370355,43.58816728385203,43.5881465334942,43.58812112474724,43.58810006488633,43.58806522943301,43.58802602390284,43.58797313976426,43.58792911478323,43.58787588212306,43.58781797199664,43.58780034126389,43.5877825206239,43.5877600221807,43.58766501270782,43.58763788491633,43.58761065889593,43.58758804126866,43.58756103396873,43.58754768420995,43.58753886164214,43.58753924014002,43.58755761476638,43.58759385063069,43.58762569681061,43.58781599461719,43.58790656421648,43.58796142473005,43.58795399267975,43.58795093319296,43.58793870402909,43.58796224452149,43.5879906014213,43.5880102075298,43.58803437339561,43.58804058746351,43.58805993537165,43.58808350548451,43.58811590613784,43.58811238641027,43.58808625173805,43.58809197200331,43.58812878059351,43.58814303444366,43.58815771336689,43.5881684169901,43.58816118315349,43.58815415575374,43.5881545414959,43.58815902288269,43.58815519658557,43.58813866612365,43.58810858417053,43.58808752371797,43.58808896031607,43.58806809569377,43.58802047061373,43.58795875428529,43.5878928022875,43.5878177525956,43.58771578560932,43.587640854547,43.58758838394608,43.58751325407572,43.58744686762906,43.5873716487506,43.58730532078456,43.58724816188749,43.58720916880812,43.58716111745596,43.58708991344203,43.5870189261756,43.58694827554277,43.586864187054,43.58680250689296,43.58673649528325,43.58665690484682,43.58659502607254,43.58651943018105,43.58645704891283,43.58639484375302,43.58630577863802,43.5862253658823,43.58617717498093,43.58617295182338,43.58614713018343,43.58606701597066,43.58595975481451,43.58587080804112,43.58578196942651,43.58568372709328,43.5855723611489,43.58550133468976,43.58542581890121,43.58533652491006,43.58524750860023,43.58516305320398,43.58507417494422,43.58497197606171,43.58487839846121,43.58463197294127,43.58454940277213,43.58452727942473,43.58448284394757,43.58440660562295,43.58432164539826,43.5842282368393,43.58414403605848,43.58404618614212,43.58395305608876,43.58385993504184,43.58377586148524,43.58365566108247,43.58352623602347,43.58339655308032,43.58325788874622,43.58314169702257,43.58302985524732,43.58291346426177,43.58279714207448,43.58267625131921,43.5825599995281,43.58245253885807,43.58234019299543,43.58222823258108,43.58212525334552,43.58202692998059,43.58191940914838,43.58182545790071,43.58173143765973,43.58164642593498,43.5815749417523,43.58148505551669,43.58140873738156,43.58133256888453,43.58127007517776,43.58120286705058,43.58113102757318,43.58106824628302,43.58101001674592,43.58098762747429,43.58138538475558,43.58527142783056,43.58699070176895,43.58730659561112,43.58881553391608,43.58892237819084,43.58897731880539,43.589110360643,43.58922999907507,43.58934998414525,43.58947882889828,43.5896122611837,43.58975022435067,43.58988808671956,43.59003036148122,43.59016414735581,43.59020704293001]}]],[[{"lng":[-79.54520634268948,-79.53259881319613,-79.52827574440603,-79.52826065113267,-79.52815269335164,-79.52551144866098,-79.52522022876984,-79.52520027953702,-79.52518207452931,-79.52516529951184,-79.52514933602428,-79.52513337817464,-79.52511895244875,-79.52510452674359,-79.52509161751965,-79.52507883390996,-79.52506736249569,-79.5250568181746,-79.52504647814762,-79.5250373660087,-79.52502917093831,-79.52502189513362,-79.52501513003534,-79.52500928419923,-79.5250041455204,-79.52499983960988,-79.52499634299227,-79.52499356135134,-79.52499149468549,-79.52498728530719,-79.52498440153785,-79.52498172204913,-79.52497954543055,-79.52497932509606,-79.52497879052029,-79.5249791751891,-79.524980274824,-79.52498280439043,-79.52498554042758,-79.52498980633041,-79.52499457509191,-79.52499966590631,-79.52500606885688,-79.52501269171769,-79.52502062671277,-79.52502867165475,-79.52503906574576,-79.52504874485864,-79.52505985607787,-79.52507177436148,-79.52508462749522,-79.52509715856561,-79.52511143376205,-79.52512631394551,-79.52514223110006,-79.52515885537515,-79.52517619458183,-79.52519424091133,-79.52521158226462,-79.52523055557661,-79.52524943452602,-79.52527004196034,-79.52529044509076,-79.52531156094487,-79.52533431314369,-79.52535716744408,-79.52538155374954,-79.52540562801849,-79.52543083133902,-79.52545664743607,-79.52548328057971,-79.52551073076813,-79.52553899799956,-79.52556787022347,-79.52559756726723,-79.52562786930521,-79.52565848547898,-79.52569042379403,-79.52572317694907,-79.52575582792507,-79.52578939807219,-79.5258236831188,-79.52585889508852,-79.52589460992699,-79.52593114179547,-79.52596848294544,-79.52600593392985,-79.5260439998689,-79.52608288063527,-79.52614362584528,-79.52620459069254,-79.52626676559441,-79.5263290425394,-79.5263913193932,-79.52645523800987,-79.52651904668451,-79.52669522350875,-79.52760531471408,-79.52837448011874,-79.53034811342961,-79.53134288966577,-79.53249280714208,-79.53846230329681,-79.54162011534288,-79.54177240908014,-79.54205801755921,-79.54254880993062,-79.54289547350457,-79.54328125503606,-79.54374016549541,-79.54415085569845,-79.54446595039222,-79.54484695311233,-79.54520634268948],"lat":[43.60160616495697,43.60444262990978,43.60541482295201,43.60538852236651,43.60541206817837,43.60600493545603,43.60535365351596,43.60531288642915,43.60526764076904,43.60522241355748,43.60518160786908,43.60513639119662,43.60509119429187,43.60504599738523,43.60500982219717,43.6049601454004,43.60491948768681,43.60487434085929,43.60482919666679,43.60478397829681,43.60473886178171,43.60469375712849,43.60464865906503,43.60460357286372,43.60456299686376,43.60451342945742,43.60446837357073,43.60442332691027,43.60437828947613,43.60433313436851,43.60428808638891,43.60424304104515,43.60420250326707,43.60415298858142,43.60410797091668,43.60406296511448,43.60401796853859,43.60397299041558,43.60392792493516,43.60388296921865,43.60384252106777,43.60379307491877,43.60375264785572,43.60370322147606,43.60366281418213,43.60361790722921,43.60357303058993,43.60352814472301,43.6034831873137,43.6034428314161,43.60339348542607,43.60335313743266,43.60330831086628,43.60326799318157,43.6032186867207,43.60317389045879,43.60312910341903,43.60308882657775,43.60304394953809,43.6029991835738,43.60294991531095,43.60290517042379,43.60286042289653,43.60281577458244,43.60277105735445,43.60272634143884,43.60268164528339,43.6026413561529,43.60259667052669,43.60255199279987,43.60250732560723,43.60246266894811,43.60241802282195,43.60237788557065,43.60233325787357,43.6022931390504,43.60224852319288,43.60220833543725,43.60216824820412,43.60212815964336,43.60208808292832,43.60204801542351,43.60200345878465,43.6019634096983,43.6019233711374,43.60188784407958,43.60184781734733,43.60180770851112,43.60176770019093,43.60172347271769,43.60167024588666,43.60162603678175,43.60158182895821,43.6015376210991,43.60148893329256,43.60144474511119,43.60127138405895,43.60081490919369,43.60046013749575,43.59948152045122,43.59898109363434,43.59840604166732,43.59542566314265,43.59384524707461,43.59417585939461,43.59476023282613,43.59576150182613,43.59650878058304,43.59733298515815,43.59837435208796,43.59929348360365,43.60000425644032,43.60084198772371,43.60160616495697]}]],[[{"lng":[-79.54891763996386,-79.54892530031216,-79.54892489859824,-79.54894715272044,-79.54894818490972,-79.54895041493921,-79.54895345614098,-79.54895711026052,-79.54896240100123,-79.54896840464401,-79.54897451042757,-79.54898214850483,-79.54899050164177,-79.54899496154493,-79.54899902102447,-79.54900205073386,-79.54900416104803,-79.54900515588122,-79.54900583603194,-79.54900498388987,-79.54900749603249,-79.54900939525818,-79.54901129448017,-79.54901237864729,-79.5490119344189,-79.54901066692813,-79.54900960374167,-79.54900670182748,-79.54900318700771,-79.54899885498088,-79.5489944208083,-79.54898855652361,-79.54898187115151,-79.54897518967384,-79.54896697593948,-79.54895804715973,-79.54894830723711,-79.54893773801794,-79.54892645981035,-79.54891426226277,-79.54890035844772,-79.54889744668736,-79.54888699916705,-79.54886564319138,-79.54876659368664,-79.54874431388322,-79.54872121693214,-79.54869709246869,-79.54861962015858,-79.54859264180419,-79.54856463594261,-79.54853602332597,-79.54850587246109,-79.54847501270284,-79.54837586946419,-79.54830195385017,-79.54827497617597,-79.54817348518718,-79.54804701979222,-79.54772569515202,-79.54735284998389,-79.54712460622909,-79.54696684130447,-79.54660053192886,-79.54617478562898,-79.54583798626958,-79.54524317796756,-79.54520634268948,-79.54484695311233,-79.54446595039222,-79.54415085569845,-79.54374016549541,-79.54328125503606,-79.54289547350457,-79.54254880993062,-79.54205801755921,-79.54177240908014,-79.54162011534288,-79.54316176217445,-79.54409051201989,-79.54457409165927,-79.54705334214563,-79.54809843189888,-79.54864509935844,-79.54865997413813,-79.5486969090545,-79.54875175631302,-79.54889334914898,-79.54895544173749,-79.54903601732921,-79.54912396925411,-79.54919798541795,-79.54933657533056,-79.5494197946104,-79.54947552687335,-79.54953513377592,-79.54956490866604,-79.54958857147756,-79.54960239916733,-79.54958097458842,-79.54959042315727,-79.54957740786355,-79.54955898017596,-79.54952677605297,-79.5494803613677,-79.54941240889687,-79.54929727271096,-79.54919865914728,-79.54908095650191,-79.54898173482266,-79.54885512804914,-79.5487561152085,-79.54870193546033,-79.54867869842641,-79.54864584141217,-79.54864617631263,-79.54866825528141,-79.54872035490524,-79.54879691505343,-79.54886750651551,-79.54896237357931,-79.54902033202264,-79.54910322089107,-79.54941249566144,-79.54972630020119,-79.55000740732306,-79.5503038770715,-79.55061992316278,-79.55092760120237,-79.55121854858257,-79.55151146557309,-79.55179885609816,-79.55209482506619,-79.55237217272175,-79.55266459845073,-79.55295529070423,-79.55323062478612,-79.55349367049739,-79.553700480149,-79.55379769073937,-79.55388599078934,-79.55394242011421,-79.55391357601737,-79.55384256234282,-79.55376407240502,-79.55372226066487,-79.55366043367572,-79.55358440258125,-79.5535162427344,-79.55347288844493,-79.5534899245757,-79.55350827353999,-79.55357717664528,-79.55365957054357,-79.55377454669429,-79.55390658286451,-79.55406715039658,-79.55428074738228,-79.55453965128294,-79.55481746487472,-79.55509396471821,-79.55537227022403,-79.55555710365752,-79.55568060653226,-79.55568886266791,-79.55576038287148,-79.55582188888104,-79.55586426658446,-79.55589336381854,-79.55592245905598,-79.55595960315574,-79.5559976721926,-79.55604589680132,-79.55608804304752,-79.55610822725176,-79.5561325770236,-79.55617224159371,-79.55624073148464,-79.55625716764719,-79.55624461931808,-79.55619645087054,-79.55614121617786,-79.55611739732417,-79.55613894129893,-79.55620593294874,-79.55629988370629,-79.55644444020847,-79.55666286145895,-79.55687445129549,-79.55711396365909,-79.55735956501654,-79.55759355807385,-79.55780372131647,-79.55800409079032,-79.55818134869352,-79.55832488754808,-79.55842921083072,-79.55853442508933,-79.55865017843307,-79.55880568085203,-79.55895309684341,-79.55908666681958,-79.55925157877225,-79.55929057030174,-79.55951359408986,-79.55961964024172,-79.55975264528921,-79.55977124176121,-79.55761521508379,-79.55711885357141,-79.55632325864856,-79.55526931909935,-79.55511806302732,-79.55507920187488,-79.5550397277849,-79.55500004935097,-79.55496057516757,-79.55492192157597,-79.55488224300503,-79.55484276868498,-79.55480319778171,-79.55476372337131,-79.55472475404113,-79.55468681739271,-79.55464805788768,-79.55460929833681,-79.55457043659081,-79.55453157479887,-79.55449281511038,-79.55445333817406,-79.55441365904777,-79.55437337262236,-79.55433389770423,-79.55429421844008,-79.55425402838033,-79.55421445682926,-79.55417498172694,-79.55413611947743,-79.55409725718195,-79.55405778194036,-79.55401810235195,-79.55397791548486,-79.5539376229042,-79.55389733595754,-79.55385694113446,-79.55381746561693,-79.55377707069897,-79.5537366757333,-79.55369628071993,-79.55365608996004,-79.55361591055888,-79.55357551540878,-79.55353512021101,-79.55349503713026,-79.5534538224761,-79.55341352928885,-79.55337324177562,-79.55333213128803,-79.55329173580954,-79.5532507331074,-79.55321043968743,-79.55317239354747,-79.55313505668188,-79.5530993599311,-79.55306356098656,-79.55302714909057,-79.55299134215188,-79.55295482801549,-79.55291749662236,-79.55287863290897,-79.5528398713014,-79.552801109648,-79.55276470321095,-79.55272726367467,-79.55268850188584,-79.55264913291566,-79.5526094516685,-79.55257057965031,-79.55253110839732,-79.55249152916234,-79.55245113266157,-79.55241318777146,-79.55237432346402,-79.5523333197083,-79.55229149288535,-79.55225120411032,-79.55221080733065,-79.55217041050327,-79.55212940652798,-79.55208686439754,-79.55204422372773,-79.55200168732659,-79.55196057519456,-79.55191946301397,-79.55187610925162,-79.55183417975026,-79.55179246034838,-79.55175053659703,-79.55170810203586,-79.55166382281766,-79.55161975370984,-79.55157935622461,-79.5515368171681,-79.55149335498041,-79.55145010878589,-79.55140664650253,-79.55136247496664,-79.55131922276993,-79.55127351883574,-79.55122709977724,-79.55118292805881,-79.55113569167416,-79.55109162201774,-79.5510430634338,-79.55099511182954,-79.55094930541497,-79.55090145961694,-79.55085432510486,-79.55080484691167,-79.55075465359468,-79.55065968101765,-79.55061172903959,-79.55056225062961,-79.55051042262036,-79.55045962205644,-79.5504086111992,-79.55035688521451,-79.55030455221163,-79.55025435846312,-79.55020620180538,-79.55015529288237,-79.55010448987811,-79.55005429595542,-79.5500032847489,-79.54995145625755,-79.54990054711723,-79.54984963793275,-79.5497979174562,-79.54974619094497,-79.54969304021665,-79.54964121146371,-79.54958877574322,-79.5495348076545,-79.54948165677052,-79.54941265368576,-79.54939078036645,-79.5493149387884,-79.54923909497066,-79.54916233384009,-79.54908495969873,-79.54900840272585,-79.54899971343529,-79.54891763996386],"lat":[43.61374692376985,43.61352187740226,43.61344085300453,43.61322508406583,43.61299095138495,43.61292796468124,43.61286948934748,43.61281102179425,43.61275248499623,43.61269404726928,43.61263561083814,43.61257719385868,43.61251878595586,43.61239281253415,43.61226224293181,43.61213625134825,43.61201024809035,43.61187963957416,43.61175361815517,43.61162757728132,43.61158259843767,43.61153761181277,43.61149262518745,43.61144753819334,43.61139802074402,43.61135299391685,43.61130796968292,43.61126292210631,43.61121786674856,43.61117280101599,43.6111277339862,43.61108264880065,43.61104196424314,43.610996868682,43.61095175366786,43.61090662957511,43.61085699410958,43.61081635026269,43.61077119633951,43.61072603074327,43.6106583381122,43.6106222025301,43.61058075002877,43.61049576867617,43.61016584264094,43.61009804363547,43.61003023424937,43.60996691288575,43.60977229291744,43.60970443421517,43.60964106353278,43.60957318406286,43.6095097861256,43.60944187810222,43.60918396768983,43.60900298590246,43.60893512712617,43.60867718669967,43.60830640202649,43.60746043845107,43.60657339942009,43.60601677586939,43.60568610169459,43.60481705735745,43.60387523848187,43.60310108948853,43.60168441245803,43.60160616495697,43.60084198772371,43.60000425644032,43.59929348360365,43.59837435208796,43.59733298515815,43.59650878058304,43.59576150182613,43.59476023282613,43.59417585939461,43.59384524707461,43.59307253553216,43.5926071640334,43.59236576280868,43.59113673015076,43.59053318086109,43.59020704293001,43.5902129031753,43.59022741554728,43.59024890696938,43.59028221226131,43.59027849944619,43.59027502122506,43.59025363222084,43.59026807498103,43.59033284923353,43.59041492498993,43.5905101551366,43.59060993551238,43.59071842950478,43.59089427224129,43.59109258563387,43.59130836602575,43.59149762166092,43.5916865022158,43.59187531408216,43.59205503896444,43.59224799673832,43.59243176900099,43.59261935344057,43.59282074076077,43.59301729456737,43.59321417287889,43.59342411666336,43.59361649632846,43.59380935514078,43.59400260681602,43.59420932961008,43.59441638383009,43.59462830524755,43.59484501889451,43.5949451048646,43.59500451521195,43.59508673912294,43.59512348357083,43.59515154227926,43.5952004785507,43.59522246494598,43.59525753894964,43.59526120948804,43.59528772294511,43.595309628355,43.59531781761664,43.59530352566851,43.59529816512024,43.59529291251662,43.59531442982296,43.59536323392044,43.59541192530227,43.59549192914977,43.59559878333665,43.59575893837958,43.59593129966326,43.59612596353127,43.59632481522957,43.59652249805901,43.59671073519501,43.59691679200186,43.5971098996203,43.59730266392744,43.59749983955562,43.5976925236319,43.59789452375804,43.59798935214402,43.59809761037035,43.59828752785337,43.59846870385522,43.59865020203453,43.59882750487153,43.5989780719641,43.59911130524733,43.59919559918239,43.59926671839343,43.59932872813343,43.59941776652487,43.59952362737665,43.59966931272868,43.5996829203035,43.59982785867324,43.5999771715494,43.60014883814733,43.60032474799662,43.60050074783006,43.60069476364644,43.60088429004789,43.6010379360711,43.60115099554739,43.60118275809928,43.60123257759567,43.60128709162476,43.601405074912,43.60149530415602,43.60156716296176,43.60170158696929,43.60184951495711,43.6020337584054,43.60220516166404,43.60238155006162,43.60256278001545,43.60270422932181,43.60283301794205,43.60295271772988,43.60304126204978,43.60316148030574,43.60325445503727,43.60337413509727,43.60348468905149,43.60361754654053,43.60376339167392,43.60390424095542,43.60406769679717,43.6042176921785,43.60435468540065,43.60450516971926,43.60461938063737,43.60474748970786,43.60477048658899,43.60490391884799,43.60497277146436,43.60503368169884,43.60504219801133,43.6069869331445,43.60743536713156,43.60816358761591,43.60912727434113,43.60926939679472,43.60930491413864,43.60934042372033,43.60937593070474,43.60941144025735,43.60944254911668,43.60947805605764,43.60951356556706,43.60954457276578,43.60958008224622,43.60962009917491,43.60965112699734,43.60968664547707,43.6097221639428,43.60975768110274,43.6097931982486,43.60982871667226,43.60986431603289,43.60989982280089,43.60993082079856,43.60996633012114,43.61000183684514,43.61003733709411,43.61006834407716,43.61010385334146,43.6101393703444,43.61017488733327,43.61021039655451,43.61024590317665,43.61027699231353,43.61031249115205,43.61034348897246,43.61037898648847,43.6104144956216,43.61044999310762,43.61048549057846,43.61052098803415,43.61055648805994,43.61058298606564,43.61061848347617,43.61065398087153,43.61068498112773,43.61072055814463,43.61075605678725,43.61078705441265,43.61082254268179,43.61085803998598,43.61088902851512,43.61092452708188,43.61096005408108,43.6110000911202,43.61103564783471,43.6110712032441,43.61110675088226,43.61114689726303,43.6111824435832,43.6112179795445,43.61125349609272,43.61128901392021,43.61132453173371,43.61135557828142,43.61139561388396,43.61143113165676,43.61146214065344,43.61149764675582,43.61153775418732,43.61156876184736,43.61160426919987,43.61163976618793,43.61167529421125,43.61171081057692,43.6117417987577,43.61177727757221,43.61180827477907,43.6118437716771,43.61187926855997,43.61191025666365,43.61194572634076,43.61197678370137,43.61200775234534,43.61204324009081,43.61207872782058,43.61210968606163,43.61214516340495,43.61217614232132,43.61220711863299,43.61223808845645,43.61227353596244,43.6123044850403,43.61233998170226,43.61237104015844,43.61240649795135,43.61243295631858,43.61246841407677,43.61249936175586,43.61253032106993,43.6125612492916,43.61259216843028,43.61262311603674,43.61265402477753,43.61268497364204,43.61271136451018,43.61274226412533,43.6127731909166,43.61279968078207,43.61283059069683,43.61285696980333,43.61288333982079,43.61293614844158,43.61296704788946,43.61299342688727,43.61301977606655,43.61304163717981,43.61306799667584,43.61309434707892,43.61311618868677,43.61314255847725,43.61317345515719,43.61319981583222,43.61322176677705,43.61324813647673,43.61327449578597,43.61330084470372,43.61332720526163,43.61335356579612,43.61337541494184,43.6134017650597,43.61342359601004,43.61344994478271,43.61347178475722,43.6134936052614,43.61351543611089,43.61355056896732,43.61355929350775,43.61358983844159,43.6136204733178,43.61365100647807,43.61368153180479,43.61371206745272,43.61371645821487,43.61374692376985]}]],[[{"lng":[-79.54888699916705,-79.54865457047759,-79.53205773840251,-79.52827574440603,-79.53259881319613,-79.54520634268948,-79.54524317796756,-79.54583798626958,-79.54617478562898,-79.54660053192886,-79.54696684130447,-79.54712460622909,-79.54735284998389,-79.54772569515202,-79.54804701979222,-79.54817348518718,-79.54827497617597,-79.54830195385017,-79.54837586946419,-79.54847501270284,-79.54850587246109,-79.54853602332597,-79.54856463594261,-79.54859264180419,-79.54861962015858,-79.54869709246869,-79.54872121693214,-79.54874431388322,-79.54876659368664,-79.54886564319138,-79.54888699916705],"lat":[43.61058075002877,43.61062605042161,43.61440528967879,43.60541482295201,43.60444262990978,43.60160616495697,43.60168441245803,43.60310108948853,43.60387523848187,43.60481705735745,43.60568610169459,43.60601677586939,43.60657339942009,43.60746043845107,43.60830640202649,43.60867718669967,43.60893512712617,43.60900298590246,43.60918396768983,43.60944187810222,43.6095097861256,43.60957318406286,43.60964106353278,43.60970443421517,43.60977229291744,43.60996691288575,43.61003023424937,43.61009804363547,43.61016584264094,43.61049576867617,43.61058075002877]}]],[[{"lng":[-79.52576887245891,-79.52830090028689,-79.53023191188031,-79.53015398535015,-79.53009772975052,-79.52936279755659,-79.52933773180214,-79.53205773840251,-79.54865457047759,-79.54888699916705,-79.54889744668736,-79.54890035844772,-79.54891426226277,-79.54892645981035,-79.54893773801794,-79.54894830723711,-79.54895804715973,-79.54896697593948,-79.54897518967384,-79.54898187115151,-79.54898855652361,-79.5489944208083,-79.54899885498088,-79.54900318700771,-79.54900670182748,-79.54900960374167,-79.54901066692813,-79.5490119344189,-79.54901237864729,-79.54901129448017,-79.54900939525818,-79.54900749603249,-79.54900498388987,-79.54900583603194,-79.54900515588122,-79.54900416104803,-79.54900205073386,-79.54899902102447,-79.54899496154493,-79.54899050164177,-79.54898214850483,-79.54897451042757,-79.54896840464401,-79.54896240100123,-79.54895711026052,-79.54895345614098,-79.54895041493921,-79.54894818490972,-79.54894715272044,-79.54892489859824,-79.54892530031216,-79.54891763996386,-79.54893690245331,-79.54899996390536,-79.54900878493848,-79.54901934870568,-79.54903051937632,-79.54904250515634,-79.54905531642767,-79.54906883676702,-79.5490839976936,-79.54909915259353,-79.5491150286669,-79.54913253928338,-79.54915065682583,-79.54915513967065,-79.54919823205803,-79.54926042191164,-79.54932199885852,-79.5493833655012,-79.54944412310485,-79.54948842948245,-79.54953223112165,-79.54957582245871,-79.54961809779711,-79.54966005466687,-79.54970211372687,-79.54974356587215,-79.5497848077288,-79.5498252345152,-79.54986483591183,-79.54990383038481,-79.54994293303255,-79.54998089999577,-79.55001907131026,-79.55005560812538,-79.55009204281532,-79.55012797271911,-79.55016347991265,-79.55019869476644,-79.55023267181284,-79.55026625218633,-79.55029879907704,-79.55033073900766,-79.55036716799954,-79.55040195653463,-79.55043675106209,-79.55047001324122,-79.5505032695096,-79.55053570854479,-79.55056814761882,-79.55059927053908,-79.55062924976343,-79.55065944737764,-79.55068882181583,-79.55071727685365,-79.55074419360369,-79.55077121847573,-79.55079824337993,-79.55082373592008,-79.55084830313123,-79.55087297845415,-79.55089612140719,-79.55091915631172,-79.55094076475851,-79.55096155811125,-79.55098224341688,-79.55104455107467,-79.55110501411662,-79.55115884869187,-79.55116557949844,-79.55122532561212,-79.55128446698792,-79.55134207611356,-79.55139957735857,-79.55145544419078,-79.55151141118901,-79.55156574592553,-79.55162029101223,-79.55167483625225,-79.55172846218584,-79.55178147313573,-79.55183428206993,-79.5518856550285,-79.55193632468566,-79.55198606917149,-79.55203591379981,-79.55208422827944,-79.55213243490148,-79.55217839989359,-79.55222497217335,-79.55226951079287,-79.55231455087684,-79.55267449072591,-79.55269342974123,-79.55272834869663,-79.55276174322036,-79.55279513205818,-79.55288477572311,-79.55264798764634,-79.55246270252596,-79.5519276992584,-79.54940508724314,-79.54545621600394,-79.54266511453955,-79.54065931908497,-79.5358318196164,-79.53322674259965,-79.52960887319895,-79.5285421986309,-79.52835982514462,-79.52826698301497,-79.52754038870968,-79.52673115004961,-79.5264417537546,-79.52633650441075,-79.52631074967486,-79.52628805218038,-79.52626219536519,-79.52624683054079,-79.52610976832653,-79.52597299552171,-79.52581494784748,-79.52580205118603,-79.52578581437535,-79.52577029273056,-79.52576887245891],"lat":[43.61800225957607,43.61752617999622,43.61714584169619,43.61694679261544,43.61680203498933,43.61508181500689,43.61502432891052,43.61440528967879,43.61062605042161,43.61058075002877,43.6106222025301,43.6106583381122,43.61072603074327,43.61077119633951,43.61081635026269,43.61085699410958,43.61090662957511,43.61095175366786,43.610996868682,43.61104196424314,43.61108264880065,43.6111277339862,43.61117280101599,43.61121786674856,43.61126292210631,43.61130796968292,43.61135299391685,43.61139802074402,43.61144753819334,43.61149262518745,43.61153761181277,43.61158259843767,43.61162757728132,43.61175361815517,43.61187963957416,43.61201024809035,43.61213625134825,43.61226224293181,43.61239281253415,43.61251878595586,43.61257719385868,43.61263561083814,43.61269404726928,43.61275248499623,43.61281102179425,43.61286948934748,43.61292796468124,43.61299095138495,43.61322508406583,43.61344085300453,43.61352187740226,43.61374692376985,43.61400823048312,43.61428818752424,43.61433781129297,43.61438295611285,43.6144326097073,43.61448236366923,43.61452753701591,43.61457722043332,43.6146224236035,43.61467212776612,43.61471734000968,43.61476257299935,43.61481231476126,43.61482137380711,43.61483542401054,43.61485421766874,43.61487300351395,43.61489628772797,43.61491515312998,43.61493822077292,43.6149567809223,43.61497983945717,43.61499387913386,43.61501691689583,43.61503995593907,43.6150584861949,43.61508151483854,43.61510444310439,43.61513195197625,43.61515495206326,43.61517345243836,43.61520094053417,43.6152284312097,43.61525590113757,43.61528336975816,43.61530633089294,43.61533837881749,43.61536133085962,43.61539326933191,43.615416200621,43.61544812093346,43.61547553246747,43.61550750199816,43.6155439517824,43.61557590056115,43.6156078298958,43.61564426021636,43.61568068016328,43.61571710010131,43.61574450119866,43.61578548101766,43.61581737143538,43.61585375247743,43.61589012185384,43.6159309727884,43.61596732401686,43.61600367523916,43.61604000702604,43.61608082814531,43.61611714955945,43.6161534515401,43.6161942532163,43.61623062575411,43.61626689793264,43.61630766980762,43.61639398003894,43.61648476792197,43.61656646960883,43.61657555706914,43.61666642581984,43.61675269578181,43.61683894629164,43.61692969647395,43.6170204259124,43.6171112466132,43.61720195657917,43.61728816811418,43.61737437962405,43.61746057945823,43.61754686149533,43.61763305092585,43.61772372320672,43.61780538441458,43.6178915349473,43.61797777674735,43.61806390911928,43.61815454117239,43.61824064373992,43.61833125504914,43.61841292846284,43.61850352032585,43.61912931624623,43.61917906781844,43.61927412243647,43.61936456665043,43.61945951184951,43.61966328494955,43.61971429952906,43.61975696376561,43.6198762167194,43.62047005715307,43.62136977667834,43.62200947252089,43.62246567386464,43.623551975052,43.62415340315924,43.624984804082,43.62520063056865,43.62471207783108,43.62444982078704,43.62279271343122,43.62091398364871,43.62029351708966,43.62006701684449,43.62001267199425,43.61996286762253,43.61990852144433,43.61987231480697,43.61955088176114,43.61906291284046,43.61817848547476,43.61813330846983,43.61807008411566,43.61800686898303,43.61800225957607]}]],[[{"lng":[-79.32946368069595,-79.32662220503474,-79.32525703116198,-79.32354355561013,-79.32165372298368,-79.32056228297756,-79.32018824031584,-79.32001254169437,-79.31991220990322,-79.31903001515812,-79.31890009774799,-79.31952249152323,-79.32194283841298,-79.32422631156331,-79.32842333075148,-79.32858376858472,-79.32904633688017,-79.32946368069595],"lat":[43.66153787958673,43.66220314725984,43.66251172321857,43.6629142861774,43.66333223753402,43.66358194291393,43.6636529626461,43.66368584787615,43.66363099090071,43.6616267702363,43.66134597643577,43.66120317613041,43.66068943759277,43.66016472866803,43.65922669965719,43.659503608184,43.66054587602085,43.66153787958673]}]],[[{"lng":[-79.37469640660493,-79.37305247992553,-79.37137161891411,-79.36898666219723,-79.36911673102959,-79.3682538919798,-79.36902843662375,-79.37050910336946,-79.37188203548726,-79.37200886563716,-79.37351904310654,-79.37404945274649,-79.37452192708247,-79.37469349390209,-79.37469640660493],"lat":[43.64192209744876,43.64247501898473,43.64305430661144,43.64393418432234,43.64356686593566,43.64226248360603,43.64198997470977,43.64145278756991,43.64093664160509,43.64088893698459,43.64032071380594,43.64111162110537,43.64180709015252,43.64192016559222,43.64192209744876]}]],[[{"lng":[-79.52831488885261,-79.52812225381,-79.52713220952212,-79.52747268896461,-79.52430989335728,-79.52396028975672,-79.51640930159489,-79.51627153645549,-79.51609841278982,-79.51555064422467,-79.51544187497308,-79.51521036496176,-79.51498824596146,-79.5147732592758,-79.51443537175253,-79.51416521482089,-79.51413379214469,-79.51410236954621,-79.51406991020301,-79.51403624005667,-79.51400235697488,-79.51396786965002,-79.51393174772583,-79.51389490202743,-79.51385785429764,-79.5138204044636,-79.51378182218428,-79.51374180965526,-79.51370179722048,-79.51366096754454,-79.51362380947778,-79.51358603405784,-79.51354804789084,-79.51350945752111,-79.51346993901913,-79.51342960326896,-79.51338795038664,-79.51334608231761,-79.51330360133993,-79.51326020969356,-79.5132601119638,-79.51325817511315,-79.51321619641092,-79.51317126372956,-79.51315536087145,-79.51323762133885,-79.51504383618514,-79.51668803745605,-79.52045454145127,-79.52201424239971,-79.52576887245891,-79.52577029273056,-79.52578581437535,-79.52580205118603,-79.52581494784748,-79.52597299552171,-79.52610976832653,-79.52624683054079,-79.52626219536519,-79.52628805218038,-79.52631074967486,-79.52633650441075,-79.5264417537546,-79.52673115004961,-79.52754038870968,-79.52826698301497,-79.52831488885261],"lat":[43.62458501939369,43.6245715562281,43.62479681837518,43.62559924365218,43.62630517754873,43.62552215976056,43.62715280083665,43.62718549060436,43.62677760761134,43.62548742251018,43.62524745358808,43.62465012629274,43.62416553703493,43.62358651770966,43.62289778506949,43.62232254845705,43.62224562190096,43.62216869533641,43.62210066738698,43.62202371158254,43.62195125406117,43.6218742876104,43.6217972998952,43.62172480381946,43.62165221508286,43.6215752100636,43.62150269136826,43.62143015406059,43.6213576167394,43.62128506877572,43.62121697958672,43.62114906239879,43.62108555349855,43.62101753566402,43.62095400680747,43.62089046730723,43.62082231957615,43.62075876011469,43.62069519266491,43.62062711228965,43.62062693097587,43.62062258476073,43.62056352487414,43.62049992548096,43.62048171435102,43.62046550035119,43.62010999196718,43.61977566800754,43.6190411219097,43.61873713473256,43.61800225957607,43.61800686898303,43.61807008411566,43.61813330846983,43.61817848547476,43.61906291284046,43.61955088176114,43.61987231480697,43.61990852144433,43.61996286762253,43.62001267199425,43.62006701684449,43.62029351708966,43.62091398364871,43.62279271343122,43.62444982078704,43.62458501939369]}]],[[{"lng":[-79.51770601429276,-79.51672158818093,-79.51388297358675,-79.51009179534293,-79.50887901726087,-79.50788361523205,-79.50715196395799,-79.50528515431044,-79.50546859646998,-79.50563719090599,-79.50285212702896,-79.50286217155919,-79.50284921273018,-79.50274500549565,-79.50270725034257,-79.50252088443339,-79.5023501561683,-79.50221442867681,-79.50200520346834,-79.50188588103168,-79.50177360014298,-79.5016765357184,-79.50151276846746,-79.50132759598839,-79.50112947476451,-79.50092548055034,-79.50075425920264,-79.50056632296713,-79.50037921364004,-79.50017350595341,-79.50002392803749,-79.49997332582608,-79.4997905465838,-79.49958023963011,-79.49940490007042,-79.49919680717508,-79.49905514193395,-79.49896562396859,-79.49891765715473,-79.49887636060112,-79.49880035428799,-79.49878654259578,-79.49881724224724,-79.49886897820939,-79.49888312510421,-79.49882415668941,-79.49878610303998,-79.49882393174828,-79.49886912128002,-79.49891771205581,-79.49890691809395,-79.49878266137034,-79.49861415234392,-79.49839341654777,-79.49812791586635,-79.49803807432976,-79.49796564671114,-79.49787900584461,-79.49774526878466,-79.49768996822412,-79.49757595532222,-79.49744098492125,-79.49733073260805,-79.49715472683481,-79.49687941128627,-79.49660290636479,-79.49636841022689,-79.4963205283499,-79.49626582020612,-79.49620530764773,-79.49606854625553,-79.49586550804892,-79.49563075132187,-79.49536555311647,-79.49511366188202,-79.49488596532471,-79.49450996171717,-79.49408214517926,-79.49365445243626,-79.49346469444809,-79.49328217300125,-79.49310988226658,-79.49307178208221,-79.49301001071211,-79.49288948662236,-79.49286649301884,-79.49248888367923,-79.49205873985409,-79.49146343274971,-79.49118296308619,-79.49105003976355,-79.49095543266601,-79.49063172266315,-79.49030453605481,-79.49013227258105,-79.48973788548214,-79.48947787336819,-79.48912055610198,-79.48887571783389,-79.48874566774727,-79.49317690497385,-79.49451886697167,-79.50101120434449,-79.50107875744079,-79.50227562242685,-79.50506890188572,-79.50848114691989,-79.51063206176994,-79.51311591622628,-79.51315536087145,-79.51317126372956,-79.51321619641092,-79.51325817511315,-79.5132601119638,-79.51326020969356,-79.51330360133993,-79.51334608231761,-79.51338795038664,-79.51342960326896,-79.51346993901913,-79.51350945752111,-79.51354804789084,-79.51358603405784,-79.51362380947778,-79.51366096754454,-79.51370179722048,-79.51374180965526,-79.51378182218428,-79.5138204044636,-79.51385785429764,-79.51389490202743,-79.51393174772583,-79.51396786965002,-79.51400235697488,-79.51403624005667,-79.51406991020301,-79.51410236954621,-79.51413379214469,-79.51416521482089,-79.51443537175253,-79.5147732592758,-79.51498824596146,-79.51521036496176,-79.51544187497308,-79.51555064422467,-79.51609841278982,-79.51627153645549,-79.51629626574679,-79.51647113159316,-79.51681453546546,-79.51697665503367,-79.51711673731594,-79.51741152669862,-79.51770601429276],"lat":[43.63078802559249,43.63100768865214,43.63164102147017,43.6324494740031,43.63270804216748,43.63260863106699,43.63233900595504,43.63271817160143,43.63319408132053,43.63358040566973,43.63418823789989,43.63419539111722,43.6341862192616,43.63412210950136,43.63409883956923,43.63398837266301,43.63387802034261,43.6337907217353,43.63366645125673,43.63360187253509,43.63354188704704,43.6334866019834,43.63346645073572,43.63346852383931,43.6334569236712,43.63346775139137,43.63349251181243,43.63354405900489,43.6335911156847,43.63363792799736,43.63367197344571,43.6336803114331,43.63370941976765,43.63375617063942,43.63379437812771,43.63380964999686,43.63377628229647,43.63363557404689,43.6335134156629,43.63333274116231,43.63308868602807,43.63286336195926,43.63265671676035,43.6325042704572,43.63243694046768,43.63234614489841,43.63222411663944,43.63207598872826,43.63187853591172,43.63166762458056,43.63150994593561,43.631368781239,43.63123144613286,43.63109801569369,43.6309684976522,43.63093130861314,43.63092135468327,43.63091571496107,43.63090495509918,43.6309132303267,43.63094774006831,43.6309864752317,43.63103003627463,43.63109523789936,43.63119964270415,43.63128611700368,43.63136855223234,43.63139492860982,43.63141221304763,43.63142041898099,43.63143662368472,43.63143395179998,43.631403855744,43.63132384686561,43.63123492045668,43.63111039370565,43.63095690673409,43.63071712580425,43.63046843281211,43.63015526893961,43.62980619170254,43.62940773744214,43.62913267076017,43.62875367765346,43.62846393087818,43.6284678586652,43.62822000170701,43.62807029356675,43.62758523580867,43.62740158136796,43.62726470482596,43.62708341349313,43.62664694756487,43.62634546647455,43.62629818035304,43.62620294849278,43.62619051029984,43.6261047683392,43.62598441397977,43.6258841213065,43.6247986390877,43.62448324505095,43.62293424320003,43.62292612703108,43.62267616457273,43.62208249530713,43.62140222432004,43.62097107461695,43.62048570227856,43.62048171435102,43.62049992548096,43.62056352487414,43.62062258476073,43.62062693097587,43.62062711228965,43.62069519266491,43.62075876011469,43.62082231957615,43.62089046730723,43.62095400680747,43.62101753566402,43.62108555349855,43.62114906239879,43.62121697958672,43.62128506877572,43.6213576167394,43.62143015406059,43.62150269136826,43.6215752100636,43.62165221508286,43.62172480381946,43.6217972998952,43.6218742876104,43.62195125406117,43.62202371158254,43.62210066738698,43.62216869533641,43.62224562190096,43.62232254845705,43.62289778506949,43.62358651770966,43.62416553703493,43.62465012629274,43.62524745358808,43.62548742251018,43.62677760761134,43.62718549060436,43.62724387528695,43.62765583151854,43.62853367368263,43.62893637151814,43.62932086897503,43.63003153928168,43.63078802559249]}]],[[{"lng":[-79.50274500549565,-79.49975529997326,-79.49955302901536,-79.49983601339748,-79.50009096417293,-79.4966598182033,-79.4956020107654,-79.49558919870913,-79.49341811868192,-79.49301726797775,-79.49277793085967,-79.49258273966782,-79.49242489174061,-79.48652065237785,-79.48644166848466,-79.48628004363451,-79.48547200856983,-79.48553098144615,-79.48548421562907,-79.48505103922575,-79.4846299897802,-79.48425196425443,-79.48524023368216,-79.48672479576177,-79.49035322889472,-79.49180510165861,-79.49286526013421,-79.49286649301884,-79.49288948662236,-79.49301001071211,-79.49307178208221,-79.49310988226658,-79.49328217300125,-79.49346469444809,-79.49365445243626,-79.49408214517926,-79.49450996171717,-79.49488596532471,-79.49511366188202,-79.49536555311647,-79.49563075132187,-79.49586550804892,-79.49606854625553,-79.49620530764773,-79.49626582020612,-79.4963205283499,-79.49636841022689,-79.49660290636479,-79.49687941128627,-79.49715472683481,-79.49733073260805,-79.49744098492125,-79.49757595532222,-79.49768996822412,-79.49774526878466,-79.49787900584461,-79.49796564671114,-79.49803807432976,-79.49812791586635,-79.49839341654777,-79.49861415234392,-79.49878266137034,-79.49890691809395,-79.49891771205581,-79.49886912128002,-79.49882393174828,-79.49878610303998,-79.49882415668941,-79.49888312510421,-79.49886897820939,-79.49881724224724,-79.49878654259578,-79.49880035428799,-79.49887636060112,-79.49891765715473,-79.49896562396859,-79.49905514193395,-79.49919680717508,-79.49940490007042,-79.49958023963011,-79.4997905465838,-79.49997332582608,-79.50002392803749,-79.50017350595341,-79.50037921364004,-79.50056632296713,-79.50075425920264,-79.50092548055034,-79.50112947476451,-79.50132759598839,-79.50151276846746,-79.5016765357184,-79.50177360014298,-79.50188588103168,-79.50200520346834,-79.50221442867681,-79.5023501561683,-79.50252088443339,-79.50270725034257,-79.50274500549565],"lat":[43.63412210950136,43.63477264568341,43.63501799833886,43.63544400168001,43.63616697580056,43.63687492122499,43.63569397016641,43.63567894805699,43.63615122653589,43.6352321386585,43.6346977686562,43.63423770758716,43.6342874776989,43.63558316524193,43.63559805300213,43.63517497433099,43.6332730181819,43.63323671097999,43.63324689380365,43.63228143863754,43.6313301857682,43.6303757209259,43.63014567622515,43.62983217406657,43.62904269212029,43.62870177571531,43.62846811247095,43.6284678586652,43.62846393087818,43.62875367765346,43.62913267076017,43.62940773744214,43.62980619170254,43.63015526893961,43.63046843281211,43.63071712580425,43.63095690673409,43.63111039370565,43.63123492045668,43.63132384686561,43.631403855744,43.63143395179998,43.63143662368472,43.63142041898099,43.63141221304763,43.63139492860982,43.63136855223234,43.63128611700368,43.63119964270415,43.63109523789936,43.63103003627463,43.6309864752317,43.63094774006831,43.6309132303267,43.63090495509918,43.63091571496107,43.63092135468327,43.63093130861314,43.6309684976522,43.63109801569369,43.63123144613286,43.631368781239,43.63150994593561,43.63166762458056,43.63187853591172,43.63207598872826,43.63222411663944,43.63234614489841,43.63243694046768,43.6325042704572,43.63265671676035,43.63286336195926,43.63308868602807,43.63333274116231,43.6335134156629,43.63363557404689,43.63377628229647,43.63380964999686,43.63379437812771,43.63375617063942,43.63370941976765,43.6336803114331,43.63367197344571,43.63363792799736,43.6335911156847,43.63354405900489,43.63349251181243,43.63346775139137,43.6334569236712,43.63346852383931,43.63346645073572,43.6334866019834,43.63354188704704,43.63360187253509,43.63366645125673,43.6337907217353,43.63387802034261,43.63398837266301,43.63409883956923,43.63412210950136]}]],[[{"lng":[-79.48553098144615,-79.48547200856983,-79.48628004363451,-79.48644166848466,-79.4858198693763,-79.48586412306211,-79.48136112631403,-79.48180932845339,-79.48187209634891,-79.48138910842187,-79.48084822219177,-79.48038676087432,-79.47984055810807,-79.47935022067281,-79.47883599266262,-79.47844368223605,-79.47810693201205,-79.47782793393783,-79.47748818013592,-79.47728092352607,-79.47720347585624,-79.47707439138748,-79.47696376888642,-79.4768605531889,-79.47681872304454,-79.47672320273536,-79.47670081280292,-79.47651540541982,-79.47632831008029,-79.47623928637428,-79.47616359218165,-79.47599802768437,-79.4758261620393,-79.4756909598234,-79.47548784175009,-79.47511682919856,-79.47516681224086,-79.47525644569902,-79.47534678261324,-79.47543805054778,-79.47552899078342,-79.47617808655534,-79.47625560191234,-79.47635546248739,-79.47645285882845,-79.47652672024016,-79.47660730037356,-79.47671989847858,-79.47684035300003,-79.47701262599733,-79.47718844064988,-79.47731401631329,-79.47764834083407,-79.47923922410104,-79.4815061441155,-79.48306405263598,-79.48425196425443,-79.4846299897802,-79.48505103922575,-79.48548421562907,-79.48553098144615],"lat":[43.63323671097999,43.6332730181819,43.63517497433099,43.63559805300213,43.63571494600773,43.63581707596339,43.63682020890521,43.63783736889011,43.63797944552032,43.63802254047419,43.6380873690508,43.63810373961119,43.63812798330176,43.6381259614067,43.63811461734141,43.63806438469946,43.63798788428219,43.63786714158586,43.63772749466292,43.63752677747608,43.637399716277,43.63720436124286,43.63698233581902,43.63674231477933,43.63643559603644,43.63626328315272,43.63622697635179,43.63595885318289,43.63563219357128,43.63547797106752,43.63534643145677,43.63503356149076,43.63474761353417,43.63443073697277,43.634099361169,43.63384676394473,43.63380242031522,43.6337180961934,43.63363828243369,43.63355397997044,43.63347858514063,43.63296502939596,43.63290304815864,43.63281435845605,43.63273013686595,43.63265901461073,43.63259707398318,43.63252205688637,43.63245164543058,43.63237742331223,43.63231675127425,43.63228232593588,43.63217425331393,43.63167322063286,43.63096500907638,43.63063452484054,43.6303757209259,43.6313301857682,43.63228143863754,43.63324689380365,43.63323671097999]}]],[[{"lng":[-79.49219584841696,-79.49199067625005,-79.49187469606196,-79.49172630830441,-79.49153908910432,-79.49142469772288,-79.49134791637896,-79.49119022693853,-79.49097142740914,-79.49077522789065,-79.4905804117373,-79.49044691929763,-79.49034909958525,-79.49021158987941,-79.49018340626371,-79.49008738821988,-79.48995755812153,-79.4898908169264,-79.4897871005892,-79.48964701424676,-79.48942542362207,-79.48927762540661,-79.48909445214507,-79.48898511612039,-79.4882009704381,-79.48793352082401,-79.48749384465742,-79.48708096080233,-79.48665240101491,-79.48633270045548,-79.48596386244182,-79.48569719864908,-79.4854286473109,-79.48520537767118,-79.4850532685889,-79.48497879133936,-79.48498543947235,-79.4849937336399,-79.48499474118087,-79.4850144083656,-79.48506040506035,-79.48510231802139,-79.48509231621672,-79.48511531353093,-79.48508888255171,-79.48503418894708,-79.48490949882833,-79.48482277075284,-79.48472390767667,-79.48466349576853,-79.48452235959614,-79.48438526560115,-79.4842390985078,-79.48404138219371,-79.48384168069563,-79.48360491936411,-79.48336521653033,-79.48323289753543,-79.48299990667208,-79.4824676457077,-79.48232037348801,-79.48187209634891,-79.48180932845339,-79.48136112631403,-79.48586412306211,-79.4858198693763,-79.48644166848466,-79.48652065237785,-79.49242489174061,-79.49258273966782,-79.49277793085967,-79.49301726797775,-79.49341811868192,-79.49390970367244,-79.49431772278155,-79.49510591726958,-79.49391796130924,-79.49375332966768,-79.49365922782242,-79.49359479589994,-79.49354960320751,-79.49344762545492,-79.49318536085683,-79.49376951512762,-79.49419000408587,-79.49421561620724,-79.49423051735816,-79.49423801921648,-79.49422998644255,-79.49421029019398,-79.49411385310152,-79.49404438408408,-79.49398262051487,-79.49393491178981,-79.49393243812291,-79.49393238464569,-79.49394306483481,-79.49395702580485,-79.49403519145321,-79.49412343287497,-79.49415754906104,-79.49422433819554,-79.49428013244282,-79.49431811228051,-79.49435669523169,-79.49442688053834,-79.49379617507326,-79.49219584841696],"lat":[43.64534172825509,43.64518139474917,43.64508084143096,43.64493935093029,43.64470723683607,43.64453918837445,43.64441664669646,43.64418942285367,43.64382636062011,43.64345909534542,43.64306934276289,43.64268039981089,43.6423775377587,43.64204705501499,43.64199267016097,43.64183836615037,43.64163401380461,43.6415476121152,43.64145171971799,43.64135075553367,43.64122629875781,43.64118833685102,43.64119941860704,43.64123398151099,43.64151176868827,43.64163425989796,43.64180407289513,43.64192463734411,43.64201798645928,43.64206326426279,43.64209888815326,43.64210435735636,43.64208729570017,43.64202582313636,43.64191128087428,43.64173025184355,43.64155930026436,43.64138386945434,43.6412217551638,43.64086634326278,43.64061039336906,43.64026877942376,43.64005250662249,43.6396296230798,43.63910706131825,43.63881826934711,43.63856896885616,43.63846879607836,43.63839996967819,43.63836316033836,43.6382801801101,43.63821534752083,43.63814139234199,43.63808475742589,43.63804610002224,43.63803395685527,43.63801277206933,43.63800201402972,43.6379944206948,43.6379603483856,43.63796289388412,43.63797944552032,43.63783736889011,43.63682020890521,43.63581707596339,43.63571494600773,43.63559805300213,43.63558316524193,43.6342874776989,43.63423770758716,43.6346977686562,43.6352321386585,43.63615122653589,43.63735075366898,43.6383195341238,43.6400225791662,43.64027735379893,43.64033207745603,43.64035856368977,43.64038823156195,43.64041302179559,43.64049692741867,43.64069070586758,43.64128813297799,43.64195100654398,43.64210437953684,43.6421585884052,43.64223070396833,43.64230261486328,43.64237896315594,43.64267476152661,43.64297550621625,43.64321333775904,43.6433837487794,43.64357276004956,43.64364036503114,43.6437080214343,43.64377122002358,43.64396129386942,43.64430462592868,43.6444040983999,43.64451759448351,43.64461285147864,43.64467636645053,43.64474439039814,43.64484883899091,43.64497114907471,43.64534172825509]}]],[[{"lng":[-79.50813396319322,-79.50804546365225,-79.50795189254239,-79.50785184565532,-79.50777418653867,-79.50770324668288,-79.5076733330077,-79.50769860912305,-79.50778194200959,-79.50788989389105,-79.50727636891426,-79.50528686957692,-79.50273610743326,-79.50299656918989,-79.50330683542919,-79.50365872114526,-79.50399440064285,-79.5043619968361,-79.50474595160711,-79.50511535353688,-79.50544448088968,-79.50429832703087,-79.50319586604903,-79.50203556894198,-79.49975745773325,-79.49940242016214,-79.49920416399245,-79.49879404778153,-79.49841250893726,-79.49808953950573,-79.49755375122341,-79.4971747563682,-79.49666742088021,-79.49624812360977,-79.49573484108913,-79.49543356342056,-79.49489820767165,-79.49454623399438,-79.49387670815035,-79.4936591583627,-79.49256097357822,-79.49195990078825,-79.49202621880279,-79.49226032582268,-79.49240817143513,-79.49261102232035,-79.49275914480889,-79.49278292570038,-79.49280706139572,-79.49280042600552,-79.49277482827982,-79.49269335513287,-79.49266040511559,-79.49254097162454,-79.49250697915751,-79.4924223437982,-79.49234797981605,-79.49219584841696,-79.49379617507326,-79.49442688053834,-79.49435669523169,-79.49431811228051,-79.49428013244282,-79.49422433819554,-79.49415754906104,-79.49412343287497,-79.49403519145321,-79.49395702580485,-79.49394306483481,-79.49393238464569,-79.49393243812291,-79.49393491178981,-79.49398262051487,-79.49404438408408,-79.49411385310152,-79.49421029019398,-79.49422998644255,-79.49423801921648,-79.49423051735816,-79.49421561620724,-79.49419000408587,-79.49376951512762,-79.49318536085683,-79.49344762545492,-79.49354960320751,-79.49359479589994,-79.49365922782242,-79.49375332966768,-79.49391796130924,-79.49510591726958,-79.49431772278155,-79.49390970367244,-79.49341811868192,-79.49558919870913,-79.4956020107654,-79.4966598182033,-79.50009096417293,-79.49983601339748,-79.49955302901536,-79.49975529997326,-79.50274500549565,-79.50284921273018,-79.50286217155919,-79.50297179518462,-79.50307457985588,-79.50311681674428,-79.50312933325084,-79.50312303746179,-79.50310346688337,-79.5030620914055,-79.50298450239808,-79.5028878107355,-79.50287490059634,-79.50287469065164,-79.50288122026755,-79.50288138091221,-79.50293335244133,-79.50314653701275,-79.50336689944446,-79.50355262843962,-79.503765000792,-79.50395602460732,-79.50417387914126,-79.50437006889744,-79.50476585513395,-79.50503300992995,-79.50516842085754,-79.50522619474107,-79.50527761110908,-79.50535553434422,-79.50539047022762,-79.50541005227929,-79.50538445435429,-79.50527774063468,-79.50514498064585,-79.5050149005001,-79.50493212714439,-79.50494969067888,-79.50502021210637,-79.5050784067478,-79.50517854722187,-79.50541677189339,-79.50571606337627,-79.50601553180654,-79.50631631006866,-79.50661872342066,-79.50692523427385,-79.50723801709017,-79.50752841990354,-79.50754751202126,-79.50776831028017,-79.50786778823911,-79.50791939089433,-79.50794337552844,-79.50805653794868,-79.50813396319322],"lat":[43.63933349925756,43.63953048019395,43.63970930063105,43.63990604063419,43.64009866181961,43.64030478380243,43.64053853773817,43.64077292226306,43.64100365376762,43.6411112878918,43.64124019832845,43.64168689667899,43.64224323607846,43.64289497862694,43.64363280258576,43.64441176995517,43.64522203133225,43.64602370721708,43.64697413026666,43.64780274338298,43.64868493106863,43.64893108760742,43.64920022004534,43.64945517068712,43.64997009525338,43.6500329498287,43.65005744285027,43.65006555976986,43.65004704444352,43.65002929753715,43.65001316321788,43.6499541678415,43.64987547895829,43.64978444309303,43.64966966350104,43.64959818211031,43.6494380077512,43.64932084597945,43.64910038589915,43.64902550213812,43.64865985153059,43.64845837724795,43.64837823355965,43.64809766662153,43.64785656073656,43.64743154742332,43.64711383697698,43.64690701265413,43.64672278822871,43.64645254818048,43.64620465328341,43.6458975079204,43.64582505667428,43.64567035596211,43.64560689300524,43.64552025681304,43.6454562612666,43.64534172825509,43.64497114907471,43.64484883899091,43.64474439039814,43.64467636645053,43.64461285147864,43.64451759448351,43.6444040983999,43.64430462592868,43.64396129386942,43.64377122002358,43.6437080214343,43.64364036503114,43.64357276004956,43.6433837487794,43.64321333775904,43.64297550621625,43.64267476152661,43.64237896315594,43.64230261486328,43.64223070396833,43.6421585884052,43.64210437953684,43.64195100654398,43.64128813297799,43.64069070586758,43.64049692741867,43.64041302179559,43.64038823156195,43.64035856368977,43.64033207745603,43.64027735379893,43.6400225791662,43.6383195341238,43.63735075366898,43.63615122653589,43.63567894805699,43.63569397016641,43.63687492122499,43.63616697580056,43.63544400168001,43.63501799833886,43.63477264568341,43.63412210950136,43.6341862192616,43.63419539111722,43.63430485230008,43.63440981277326,43.63449138489374,43.63461307722443,43.63473902422116,43.63481087475436,43.6349498654503,43.63510638600898,43.63525824525568,43.63537060242906,43.63546962280732,43.63561833305003,43.63573536248295,43.63585307057683,43.63594588382234,43.63593076544046,43.6359106921248,43.63590447046059,43.63589346790701,43.63585580988475,43.63581786794212,43.63582304653495,43.6358625497076,43.63595884291015,43.63604061744365,43.63618091248686,43.63635297148393,43.63650655409849,43.6366238374921,43.63684405422426,43.63704979699844,43.63725051790896,43.63743795060291,43.63767992455691,43.63778376847338,43.63787921298782,43.63790698048034,43.63796230285902,43.63801042885152,43.63803234647241,43.638018257225,43.63801318641004,43.63800813616783,43.63804814915144,43.63811975057944,43.63819105890521,43.63820031031649,43.63832481171206,43.63850615247155,43.63870946340381,43.63892582688779,43.63911193718476,43.63933349925756]}]],[[{"lng":[-79.5170166381468,-79.51694942708605,-79.51672083966987,-79.51646274160332,-79.51618934602945,-79.5159571528438,-79.51574057206821,-79.51549438880308,-79.51544969998966,-79.51524409105488,-79.51502931464053,-79.51483691020761,-79.51472250894706,-79.51470097744166,-79.51474541149645,-79.51479504675613,-79.51479926312554,-79.51486992128777,-79.5149109330348,-79.51494373131638,-79.51493701052496,-79.5149336914741,-79.5147853358186,-79.51446888415992,-79.51411603703548,-79.51376188855406,-79.51340466646329,-79.5130141258317,-79.51267867705387,-79.51251457315576,-79.51235588626227,-79.51203271479112,-79.51176624592894,-79.51156421987343,-79.51130422526347,-79.51102992237739,-79.51074568288433,-79.51042691173076,-79.51013849280527,-79.50985040437348,-79.50958086338345,-79.50936425440686,-79.50917135246277,-79.50890854928967,-79.50890701628059,-79.50858128472942,-79.50825783818078,-79.50796951176099,-79.50788989389105,-79.50778194200959,-79.50769860912305,-79.5076733330077,-79.50770324668288,-79.50777418653867,-79.50785184565532,-79.50795189254239,-79.50804546365225,-79.50813396319322,-79.50805653794868,-79.50794337552844,-79.50791939089433,-79.50786778823911,-79.50776831028017,-79.50754751202126,-79.50752841990354,-79.50723801709017,-79.50692523427385,-79.50661872342066,-79.50631631006866,-79.50601553180654,-79.50571606337627,-79.50541677189339,-79.50517854722187,-79.5050784067478,-79.50502021210637,-79.50494969067888,-79.50493212714439,-79.5050149005001,-79.50514498064585,-79.50527774063468,-79.50538445435429,-79.50541005227929,-79.50539047022762,-79.50535553434422,-79.50527761110908,-79.50522619474107,-79.50516842085754,-79.50503300992995,-79.50476585513395,-79.50437006889744,-79.50417387914126,-79.50395602460732,-79.503765000792,-79.50355262843962,-79.50336689944446,-79.50314653701275,-79.50293335244133,-79.50288138091221,-79.50288122026755,-79.50287469065164,-79.50287490059634,-79.5028878107355,-79.50298450239808,-79.5030620914055,-79.50310346688337,-79.50312303746179,-79.50312933325084,-79.50311681674428,-79.50307457985588,-79.50297179518462,-79.50286217155919,-79.50285212702896,-79.50563719090599,-79.50546859646998,-79.50528515431044,-79.50715196395799,-79.50788361523205,-79.50887901726087,-79.51009179534293,-79.51388297358675,-79.51672158818093,-79.51770601429276,-79.51775258033018,-79.51791720556822,-79.51877437603103,-79.51912180011533,-79.51979687164263,-79.52000314981862,-79.52002527345338,-79.5201803422004,-79.52034623150877,-79.52069457077711,-79.52073770460798,-79.52121285052564,-79.5216725923993,-79.52171567038312,-79.52208412817222,-79.5226607152035,-79.52342946305436,-79.52228891753369,-79.52115848853668,-79.52008823904022,-79.51896878609159,-79.51705078450611,-79.5170166381468],"lat":[43.64610299922774,43.64602560934476,43.64577949648551,43.6455194969698,43.64529539664349,43.64506723960346,43.64479886545008,43.64456602472415,43.64452493484678,43.64434672316876,43.64410078629305,43.64391374334502,43.64371862167889,43.64348878849376,43.64329122986848,43.64309832987637,43.64308488152727,43.64285615628732,43.64257753425596,43.64231689968489,43.64214568253217,43.64206462054919,43.641828638284,43.64168490305406,43.64159929742321,43.64150017063619,43.64140550382052,43.64131049242292,43.64121601659565,43.6411463652032,43.64107678408394,43.64092404255842,43.64073143902363,43.64056226939418,43.64045535903111,43.64045178578463,43.6405110970796,43.64061496809803,43.64071923388893,43.64081009013446,43.64092810378765,43.64110081974922,43.64122433300975,43.64129301145102,43.6412929914507,43.64129774344234,43.64128452013928,43.64119064553074,43.6411112878918,43.64100365376762,43.64077292226306,43.64053853773817,43.64030478380243,43.64009866181961,43.63990604063419,43.63970930063105,43.63953048019395,43.63933349925756,43.63911193718476,43.63892582688779,43.63870946340381,43.63850615247155,43.63832481171206,43.63820031031649,43.63819105890521,43.63811975057944,43.63804814915144,43.63800813616783,43.63801318641004,43.638018257225,43.63803234647241,43.63801042885152,43.63796230285902,43.63790698048034,43.63787921298782,43.63778376847338,43.63767992455691,43.63743795060291,43.63725051790896,43.63704979699844,43.63684405422426,43.6366238374921,43.63650655409849,43.63635297148393,43.63618091248686,43.63604061744365,43.63595884291015,43.6358625497076,43.63582304653495,43.63581786794212,43.63585580988475,43.63589346790701,43.63590447046059,43.6359106921248,43.63593076544046,43.63594588382234,43.63585307057683,43.63573536248295,43.63561833305003,43.63546962280732,43.63537060242906,43.63525824525568,43.63510638600898,43.6349498654503,43.63481087475436,43.63473902422116,43.63461307722443,43.63449138489374,43.63440981277326,43.63430485230008,43.63419539111722,43.63418823789989,43.63358040566973,43.63319408132053,43.63271817160143,43.63233900595504,43.63260863106699,43.63270804216748,43.6324494740031,43.63164102147017,43.63100768865214,43.63078802559249,43.6308688383288,43.63130316459985,43.63343022265298,43.63421358711199,43.63587907962793,43.6364083740923,43.63646492372717,43.6368608636915,43.63723668877574,43.63810107878779,43.6382096622949,43.63937285111382,43.64053124744315,43.64067142718621,43.64150906645244,43.64289861278371,43.64464172032458,43.64491963751114,43.64519317313889,43.64541796531213,43.64566011403081,43.64609894124573,43.64610299922774]}]],[[{"lng":[-79.5340258203241,-79.52814431354736,-79.52786322113016,-79.52684708335578,-79.52562521047324,-79.5245263947795,-79.52342946305436,-79.5226607152035,-79.52208412817222,-79.52171567038312,-79.5216725923993,-79.52121285052564,-79.52073770460798,-79.52069457077711,-79.52034623150877,-79.5201803422004,-79.52002527345338,-79.52000314981862,-79.51979687164263,-79.51912180011533,-79.51877437603103,-79.51791720556822,-79.51775258033018,-79.51770601429276,-79.51741152669862,-79.51711673731594,-79.51697665503367,-79.51681453546546,-79.51647113159316,-79.51629626574679,-79.51627153645549,-79.51640930159489,-79.52396028975672,-79.52430989335728,-79.52747268896461,-79.52713220952212,-79.52812225381,-79.52831488885261,-79.52835982514462,-79.5285421986309,-79.52855684013821,-79.52856923606267,-79.52858804291961,-79.52863265237741,-79.52883634797739,-79.52999162038222,-79.5310591540086,-79.53179185487394,-79.53231164457863,-79.53273109121689,-79.53293982339913,-79.53339966515668,-79.53379113437593,-79.5340258203241],"lat":[43.6384426296902,43.64361670336072,43.6436769979566,43.64390696157931,43.64418386385327,43.64439923706255,43.64464172032458,43.64289861278371,43.64150906645244,43.64067142718621,43.64053124744315,43.63937285111382,43.6382096622949,43.63810107878779,43.63723668877574,43.6368608636915,43.63646492372717,43.6364083740923,43.63587907962793,43.63421358711199,43.63343022265298,43.63130316459985,43.6308688383288,43.63078802559249,43.63003153928168,43.62932086897503,43.62893637151814,43.62853367368263,43.62765583151854,43.62724387528695,43.62718549060436,43.62715280083665,43.62552215976056,43.62630517754873,43.62559924365218,43.62479681837518,43.6245715562281,43.62458501939369,43.62471207783108,43.62520063056865,43.62536294729576,43.62552514508589,43.62564241479343,43.62578711305472,43.62629844503479,43.62902348759062,43.63147723633233,43.63320641455332,43.634487159309,43.63552346637751,43.63593133073319,43.63677451832161,43.6378150532192,43.6384426296902]}]],[[{"lng":[-79.55435520293256,-79.55411699295297,-79.55014137807228,-79.55006569568022,-79.54534629991291,-79.54356709343435,-79.54352765938579,-79.5408160900113,-79.53932303444194,-79.53697488580995,-79.53660404469224,-79.53601411968003,-79.5340262006625,-79.53379113437593,-79.53339966515668,-79.53293982339913,-79.53273109121689,-79.53231164457863,-79.53179185487394,-79.5310591540086,-79.52999162038222,-79.52883634797739,-79.52863265237741,-79.52858804291961,-79.52856923606267,-79.52855684013821,-79.5285421986309,-79.52960887319895,-79.53322674259965,-79.5358318196164,-79.54065931908497,-79.54266511453955,-79.54545621600394,-79.54940508724314,-79.5519276992584,-79.55246270252596,-79.55264798764634,-79.55288477572311,-79.5536725069295,-79.55413861406244,-79.55435520293256],"lat":[43.62295470837932,43.6230688129148,43.62578264433912,43.62583569702479,43.62908022919364,43.63037890199972,43.63040765637512,43.63271416042663,43.63396915792853,43.63599646655535,43.63630688275494,43.63679453382422,43.63844380484219,43.6378150532192,43.63677451832161,43.63593133073319,43.63552346637751,43.634487159309,43.63320641455332,43.63147723633233,43.62902348759062,43.62629844503479,43.62578711305472,43.62564241479343,43.62552514508589,43.62536294729576,43.62520063056865,43.624984804082,43.62415340315924,43.623551975052,43.62246567386464,43.62200947252089,43.62136977667834,43.62047005715307,43.6198762167194,43.61975696376561,43.61971429952906,43.61966328494955,43.62141102782979,43.62245676144374,43.62295470837932]}]],[[{"lng":[-79.55977124176121,-79.55982187782085,-79.56000947846651,-79.56011157859962,-79.56020915126713,-79.5604468533536,-79.56059080153507,-79.5608118469885,-79.56102381050073,-79.56106597487252,-79.56122740098192,-79.56144007758503,-79.56168131408351,-79.56185567301254,-79.56201727839803,-79.56212741641956,-79.56225505410893,-79.56241181431885,-79.56250501673256,-79.56263019555448,-79.56289886769756,-79.56314927149083,-79.5634172645004,-79.56368504860157,-79.56393833341157,-79.56407171504136,-79.56421612842374,-79.56446816645352,-79.56469424758434,-79.56486195564717,-79.56489450139139,-79.56484274562719,-79.56477165595069,-79.56474339260173,-79.56471575552503,-79.56471145623605,-79.56462729967026,-79.56459986833636,-79.56462165917809,-79.56465694817655,-79.5647931665751,-79.56503965835388,-79.56532097841767,-79.56553602461527,-79.56564611602072,-79.56577319283211,-79.56590949164037,-79.56618932707458,-79.5663910547788,-79.56663398888118,-79.56683539276486,-79.56698611521178,-79.56707298134616,-79.56716974486673,-79.56723271836894,-79.56725288198109,-79.56736657206601,-79.56725277983162,-79.56726277447123,-79.56726640502768,-79.5672751656242,-79.56728788201134,-79.56732243309469,-79.56732321205831,-79.56728963086844,-79.56721344091041,-79.56715848715696,-79.56706524520679,-79.56695129920567,-79.56684484312827,-79.56663723327171,-79.56654872043754,-79.56633134868491,-79.56613428070293,-79.5659645600945,-79.56577279083128,-79.5657255627992,-79.56561094886133,-79.56543343342916,-79.56527207926752,-79.56509401697551,-79.56487365282207,-79.5646602636146,-79.5644335819029,-79.56420516011843,-79.56404462570589,-79.5639379040166,-79.56395674185198,-79.56401215615644,-79.56407593544276,-79.56418296336928,-79.56436720662944,-79.56459473110927,-79.56485178895493,-79.56510499727332,-79.56534577884163,-79.56557790341184,-79.56579592818824,-79.56604632190398,-79.56629996290015,-79.56652589177267,-79.56676435991726,-79.56699045826326,-79.56724773365805,-79.56749314537808,-79.56770126118813,-79.56787981957747,-79.56794984813983,-79.56631766363888,-79.56244694942163,-79.55828355830769,-79.55475500595206,-79.55454710891527,-79.55435520293256,-79.55413861406244,-79.5536725069295,-79.55288477572311,-79.55279513205818,-79.55276174322036,-79.55272834869663,-79.55269342974123,-79.55267449072591,-79.55231455087684,-79.55226951079287,-79.55222497217335,-79.55217839989359,-79.55213243490148,-79.55208422827944,-79.55203591379981,-79.55198606917149,-79.55193632468566,-79.5518856550285,-79.55183428206993,-79.55178147313573,-79.55172846218584,-79.55167483625225,-79.55162029101223,-79.55156574592553,-79.55151141118901,-79.55145544419078,-79.55139957735857,-79.55134207611356,-79.55128446698792,-79.55122532561212,-79.55116557949844,-79.55115884869187,-79.55110501411662,-79.55104455107467,-79.55098224341688,-79.55096155811125,-79.55094076475851,-79.55091915631172,-79.55089612140719,-79.55087297845415,-79.55084830313123,-79.55082373592008,-79.55079824337993,-79.55077121847573,-79.55074419360369,-79.55071727685365,-79.55068882181583,-79.55065944737764,-79.55062924976343,-79.55059927053908,-79.55056814761882,-79.55053570854479,-79.5505032695096,-79.55047001324122,-79.55043675106209,-79.55040195653463,-79.55036716799954,-79.55033073900766,-79.55029879907704,-79.55026625218633,-79.55023267181284,-79.55019869476644,-79.55016347991265,-79.55012797271911,-79.55009204281532,-79.55005560812538,-79.55001907131026,-79.54998089999577,-79.54994293303255,-79.54990383038481,-79.54986483591183,-79.5498252345152,-79.5497848077288,-79.54974356587215,-79.54970211372687,-79.54966005466687,-79.54961809779711,-79.54957582245871,-79.54953223112165,-79.54948842948245,-79.54944412310485,-79.5493833655012,-79.54932199885852,-79.54926042191164,-79.54919823205803,-79.54915513967065,-79.54915065682583,-79.54913253928338,-79.5491150286669,-79.54909915259353,-79.5490839976936,-79.54906883676702,-79.54905531642767,-79.54904250515634,-79.54903051937632,-79.54901934870568,-79.54900878493848,-79.54899996390536,-79.54893690245331,-79.54891763996386,-79.54899971343529,-79.54900840272585,-79.54908495969873,-79.54916233384009,-79.54923909497066,-79.5493149387884,-79.54939078036645,-79.54941265368576,-79.54948165677052,-79.5495348076545,-79.54958877574322,-79.54964121146371,-79.54969304021665,-79.54974619094497,-79.5497979174562,-79.54984963793275,-79.54990054711723,-79.54995145625755,-79.5500032847489,-79.55005429595542,-79.55010448987811,-79.55015529288237,-79.55020620180538,-79.55025435846312,-79.55030455221163,-79.55035688521451,-79.5504086111992,-79.55045962205644,-79.55051042262036,-79.55056225062961,-79.55061172903959,-79.55065968101765,-79.55075465359468,-79.55080484691167,-79.55085432510486,-79.55090145961694,-79.55094930541497,-79.55099511182954,-79.5510430634338,-79.55109162201774,-79.55113569167416,-79.55118292805881,-79.55122709977724,-79.55127351883574,-79.55131922276993,-79.55136247496664,-79.55140664650253,-79.55145010878589,-79.55149335498041,-79.5515368171681,-79.55157935622461,-79.55161975370984,-79.55166382281766,-79.55170810203586,-79.55175053659703,-79.55179246034838,-79.55183417975026,-79.55187610925162,-79.55191946301397,-79.55196057519456,-79.55200168732659,-79.55204422372773,-79.55208686439754,-79.55212940652798,-79.55217041050327,-79.55221080733065,-79.55225120411032,-79.55229149288535,-79.5523333197083,-79.55237432346402,-79.55241318777146,-79.55245113266157,-79.55249152916234,-79.55253110839732,-79.55257057965031,-79.5526094516685,-79.55264913291566,-79.55268850188584,-79.55272726367467,-79.55276470321095,-79.552801109648,-79.5528398713014,-79.55287863290897,-79.55291749662236,-79.55295482801549,-79.55299134215188,-79.55302714909057,-79.55306356098656,-79.5530993599311,-79.55313505668188,-79.55317239354747,-79.55321043968743,-79.5532507331074,-79.55329173580954,-79.55333213128803,-79.55337324177562,-79.55341352928885,-79.5534538224761,-79.55349503713026,-79.55353512021101,-79.55357551540878,-79.55361591055888,-79.55365608996004,-79.55369628071993,-79.5537366757333,-79.55377707069897,-79.55381746561693,-79.55385694113446,-79.55389733595754,-79.5539376229042,-79.55397791548486,-79.55401810235195,-79.55405778194036,-79.55409725718195,-79.55413611947743,-79.55417498172694,-79.55421445682926,-79.55425402838033,-79.55429421844008,-79.55433389770423,-79.55437337262236,-79.55441365904777,-79.55445333817406,-79.55449281511038,-79.55453157479887,-79.55457043659081,-79.55460929833681,-79.55464805788768,-79.55468681739271,-79.55472475404113,-79.55476372337131,-79.55480319778171,-79.55484276868498,-79.55488224300503,-79.55492192157597,-79.55496057516757,-79.55500004935097,-79.5550397277849,-79.55507920187488,-79.55511806302732,-79.55526931909935,-79.55632325864856,-79.55711885357141,-79.55761521508379,-79.55977124176121],"lat":[43.60504219801133,43.60506534143684,43.60518473312558,43.60522202801471,43.60528636227899,43.60545139513967,43.6055162231264,43.60560002603673,43.60567480204527,43.60569333718423,43.60574938224132,43.60582857751076,43.6059441404112,43.60603635595443,43.60620501914237,43.6063639424764,43.60649598933382,43.60661507914398,43.60666576321534,43.6067123482561,43.60677874172782,43.60683140168313,43.60686627784961,43.60690565184092,43.60694034195725,43.60696002200849,43.60697984048144,43.60703260809716,43.60709846242103,43.60720859416439,43.60733953407358,43.60749651198908,43.60757708886666,43.60761409291398,43.60764624365411,43.60765087078779,43.60779537883634,43.60793915886027,43.60809697023333,43.60824144787365,43.60837828074232,43.60846689604907,43.60856044907077,43.6086351653648,43.60865905235808,43.60865614591616,43.60862634857604,43.60858034743357,43.60859638103978,43.60867144505291,43.60880009076993,43.60894601501413,43.60910464176066,43.60927248468554,43.60942180963306,43.60953656978339,43.60958543619364,43.60953656850293,43.60959331733262,43.60963837360937,43.6097600124942,43.6099087074205,43.6100802714348,43.61021531349022,43.6103859334599,43.61056511147534,43.61065894518747,43.61073429456388,43.61077787675111,43.61079004520743,43.61082345046754,43.6108674412158,43.61095473640029,43.61106479125821,43.61120219534084,43.61132140822082,43.61135232311665,43.61144540740621,43.61160971931743,43.61175181845896,43.61185310791938,43.61196286811603,43.61204570902686,43.61213747475716,43.61223362922989,43.6123711458189,43.61252743262227,43.61267620475232,43.6128209353373,43.61297486315115,43.61309773668686,43.61323508322612,43.61330995798686,43.61337620097737,43.61341097756802,43.61341400008158,43.61339431805361,43.6133745486528,43.61336868815422,43.61338528318716,43.61342421552001,43.61349022106427,43.61356507317294,43.61362231135456,43.61369290349042,43.61377202988773,43.613882383101,43.61393277232686,43.61499274846543,43.61750578074435,43.62023113030919,43.62263109666955,43.62277259180111,43.62295470837932,43.62245676144374,43.62141102782979,43.61966328494955,43.61945951184951,43.61936456665043,43.61927412243647,43.61917906781844,43.61912931624623,43.61850352032585,43.61841292846284,43.61833125504914,43.61824064373992,43.61815454117239,43.61806390911928,43.61797777674735,43.6178915349473,43.61780538441458,43.61772372320672,43.61763305092585,43.61754686149533,43.61746057945823,43.61737437962405,43.61728816811418,43.61720195657917,43.6171112466132,43.6170204259124,43.61692969647395,43.61683894629164,43.61675269578181,43.61666642581984,43.61657555706914,43.61656646960883,43.61648476792197,43.61639398003894,43.61630766980762,43.61626689793264,43.61623062575411,43.6161942532163,43.6161534515401,43.61611714955945,43.61608082814531,43.61604000702604,43.61600367523916,43.61596732401686,43.6159309727884,43.61589012185384,43.61585375247743,43.61581737143538,43.61578548101766,43.61574450119866,43.61571710010131,43.61568068016328,43.61564426021636,43.6156078298958,43.61557590056115,43.6155439517824,43.61550750199816,43.61547553246747,43.61544812093346,43.615416200621,43.61539326933191,43.61536133085962,43.61533837881749,43.61530633089294,43.61528336975816,43.61525590113757,43.6152284312097,43.61520094053417,43.61517345243836,43.61515495206326,43.61513195197625,43.61510444310439,43.61508151483854,43.6150584861949,43.61503995593907,43.61501691689583,43.61499387913386,43.61497983945717,43.6149567809223,43.61493822077292,43.61491515312998,43.61489628772797,43.61487300351395,43.61485421766874,43.61483542401054,43.61482137380711,43.61481231476126,43.61476257299935,43.61471734000968,43.61467212776612,43.6146224236035,43.61457722043332,43.61452753701591,43.61448236366923,43.6144326097073,43.61438295611285,43.61433781129297,43.61428818752424,43.61400823048312,43.61374692376985,43.61371645821487,43.61371206745272,43.61368153180479,43.61365100647807,43.6136204733178,43.61358983844159,43.61355929350775,43.61355056896732,43.61351543611089,43.6134936052614,43.61347178475722,43.61344994478271,43.61342359601004,43.6134017650597,43.61337541494184,43.61335356579612,43.61332720526163,43.61330084470372,43.61327449578597,43.61324813647673,43.61322176677705,43.61319981583222,43.61317345515719,43.61314255847725,43.61311618868677,43.61309434707892,43.61306799667584,43.61304163717981,43.61301977606655,43.61299342688727,43.61296704788946,43.61293614844158,43.61288333982079,43.61285696980333,43.61283059069683,43.61279968078207,43.6127731909166,43.61274226412533,43.61271136451018,43.61268497364204,43.61265402477753,43.61262311603674,43.61259216843028,43.6125612492916,43.61253032106993,43.61249936175586,43.61246841407677,43.61243295631858,43.61240649795135,43.61237104015844,43.61233998170226,43.6123044850403,43.61227353596244,43.61223808845645,43.61220711863299,43.61217614232132,43.61214516340495,43.61210968606163,43.61207872782058,43.61204324009081,43.61200775234534,43.61197678370137,43.61194572634076,43.61191025666365,43.61187926855997,43.6118437716771,43.61180827477907,43.61177727757221,43.6117417987577,43.61171081057692,43.61167529421125,43.61163976618793,43.61160426919987,43.61156876184736,43.61153775418732,43.61149764675582,43.61146214065344,43.61143113165676,43.61139561388396,43.61135557828142,43.61132453173371,43.61128901392021,43.61125349609272,43.6112179795445,43.6111824435832,43.61114689726303,43.61110675088226,43.6110712032441,43.61103564783471,43.6110000911202,43.61096005408108,43.61092452708188,43.61088902851512,43.61085803998598,43.61082254268179,43.61078705441265,43.61075605678725,43.61072055814463,43.61068498112773,43.61065398087153,43.61061848347617,43.61058298606564,43.61055648805994,43.61052098803415,43.61048549057846,43.61044999310762,43.6104144956216,43.61037898648847,43.61034348897246,43.61031249115205,43.61027699231353,43.61024590317665,43.61021039655451,43.61017488733327,43.6101393703444,43.61010385334146,43.61006834407716,43.61003733709411,43.61000183684514,43.60996633012114,43.60993082079856,43.60989982280089,43.60986431603289,43.60982871667226,43.6097931982486,43.60975768110274,43.6097221639428,43.60968664547707,43.60965112699734,43.60962009917491,43.60958008224622,43.60954457276578,43.60951356556706,43.60947805605764,43.60944254911668,43.60941144025735,43.60937593070474,43.60934042372033,43.60930491413864,43.60926939679472,43.60912727434113,43.60816358761591,43.60743536713156,43.6069869331445,43.60504219801133]}]],[[{"lng":[-79.37137161891411,-79.37179019883129,-79.37252750344544,-79.37037825157378,-79.370211350007,-79.36999922622907,-79.36994069947228,-79.369888533645,-79.36984213739673,-79.36981041679637,-79.36952779142362,-79.36949126371005,-79.36943957155121,-79.36937947968696,-79.36898666219723,-79.37137161891411],"lat":[43.64305430661144,43.64375351971363,43.64568182080398,43.64634912398306,43.64641876043878,43.6459565401769,43.64582967658902,43.64572090781763,43.64562563438684,43.64553966235385,43.6448152857096,43.64470673984398,43.64460697966937,43.6445116006543,43.64393418432234,43.64305430661144]}]],[[{"lng":[-79.56834332975475,-79.56832848064585,-79.5682679298247,-79.56821660775874,-79.56820603040245,-79.56817176598605,-79.56811765723755,-79.56806590926229,-79.56798850751215,-79.56793176702048,-79.56789472458109,-79.56785044036489,-79.56781460947217,-79.56780741604636,-79.56781909698987,-79.56785072954065,-79.56784004400285,-79.56770974320227,-79.56754094281864,-79.56732999091786,-79.56713703786639,-79.56695793859846,-79.56681876757364,-79.56668479682116,-79.5666019070935,-79.56655817505869,-79.56653340089117,-79.56649923775666,-79.56655690080071,-79.56660751741059,-79.56667355848693,-79.56672734272759,-79.56676446225761,-79.56681294126295,-79.56677856214954,-79.56672537702428,-79.56670478257175,-79.56671062320356,-79.56672394787363,-79.56672725551022,-79.56673426749128,-79.56675802758011,-79.56679785650626,-79.56681929799726,-79.56682061196329,-79.56682937469746,-79.56686300481738,-79.56691536769962,-79.566896201378,-79.56680796162357,-79.56664099194542,-79.56643644034189,-79.56620268417868,-79.56598302670747,-79.56579921561885,-79.56564236220319,-79.56557007364167,-79.56553761869492,-79.56557241585109,-79.56564479530114,-79.5657370909138,-79.56583052800978,-79.56589329702113,-79.5658888374015,-79.56585452696191,-79.56578629727547,-79.56575331188647,-79.56574279252393,-79.56570449592208,-79.5656661851429,-79.5656393535488,-79.56560098098849,-79.5655234540037,-79.56541721768859,-79.56525369263811,-79.56507510563773,-79.5648700456359,-79.56466863050871,-79.5644470797625,-79.5642509616966,-79.56407436955624,-79.5639252054783,-79.56379284729307,-79.56368756493242,-79.56359136858482,-79.56355320679359,-79.56354650282532,-79.56359420879234,-79.5635636332326,-79.56352721063602,-79.56348548465093,-79.56346167953569,-79.56343944702603,-79.56342967510425,-79.56345128890257,-79.56349378212542,-79.56355341432477,-79.56360663162828,-79.5636741341914,-79.56371983551767,-79.56374972196794,-79.56236236461663,-79.56098657342983,-79.56058163264392,-79.55992946234196,-79.55977156236239,-79.55961355478584,-79.5594554395766,-79.55929835447731,-79.55914105638864,-79.55898386564658,-79.55888563821536,-79.55882656722515,-79.55867019361791,-79.55851371230844,-79.55835794609158,-79.55820135686156,-79.55804569788067,-79.55789064426425,-79.55773476975094,-79.5575797179341,-79.5574245583107,-79.5573309278253,-79.55727031812253,-79.55724854810784,-79.55711597556089,-79.55696725825631,-79.55699129615529,-79.55435520293256,-79.55454710891527,-79.55475500595206,-79.55828355830769,-79.56244694942163,-79.56631766363888,-79.56794984813983,-79.56803059403883,-79.56817717135439,-79.56828320658033,-79.56834332975475],"lat":[43.61436089254843,43.61449573875192,43.61466602107032,43.61480050039573,43.61495340437587,43.61509250840331,43.61525395920088,43.61540634737762,43.61556750628534,43.61570632862544,43.61586340203331,43.61600697149077,43.61617756326095,43.61631700635402,43.61647928132633,43.6166147098376,43.61677211345256,43.61691010393184,43.61703401852481,43.61711689488033,43.6171819921997,43.61728336147928,43.61740314540764,43.61753199651781,43.61766148808407,43.61775582220432,43.61780925428371,43.61794385793664,43.61809761752039,43.61824687765403,43.61840074223994,43.61855004203693,43.61871254602688,43.61885718815983,43.61900088132585,43.61915325077165,43.61930602888329,43.61948173385593,43.61963493730502,43.61979251620573,43.61992322516121,43.62006305624661,43.62017608240372,43.62028446693696,43.62029798661936,43.62032060183985,43.62040204276159,43.62054673351174,43.62069961955051,43.62086055159127,43.62097998664437,43.62108103583453,43.62116812507657,43.62125539072761,43.62135660896625,43.62147166841689,43.62161488540321,43.62177201545449,43.62190748419236,43.62205701778277,43.62221571335947,43.62235650893384,43.62251483387978,43.62267231526932,43.62284301532399,43.62301319972127,43.62318832740759,43.62337282915251,43.6235478001442,43.62373195311477,43.62388465264184,43.62403270625585,43.62419836273558,43.62436356879579,43.62449663818921,43.6245754154443,43.62456833977674,43.62449829459715,43.62443690827126,43.62438052210365,43.62440981113314,43.62448904625431,43.62461791405536,43.6247471219541,43.62488103502734,43.62501999841943,43.62516844930806,43.62563724961111,43.62574948198401,43.62588405616137,43.62600956154043,43.62609928362998,43.62615301696592,43.62630151928019,43.62643682285927,43.62654088140609,43.6266677506766,43.6267764450195,43.62690332311087,43.6269714133736,43.62701688959071,43.62733702722699,43.62762587618314,43.62771079839084,43.62783770509063,43.62786272188551,43.62789223816839,43.62792625393823,43.62795137036992,43.62798539600378,43.6280149217117,43.628040689607,43.62804894690756,43.62807848248661,43.62811251755515,43.62814656142864,43.62818509576937,43.62821463950691,43.62824878175342,43.62828732447112,43.62832137629625,43.62835992761173,43.62838575218748,43.62839849032189,43.62840271658202,43.62843705153092,43.62847118249099,43.62853450084392,43.62295470837932,43.62277259180111,43.62263109666955,43.62023113030919,43.61750578074435,43.61499274846543,43.61393277232686,43.61399229792819,43.61411116212417,43.61421601507988,43.61436089254843]}]],[[{"lng":[-79.55620329442817,-79.55608871554207,-79.55188540608135,-79.54691427164551,-79.54481132378167,-79.54463737243576,-79.54418895032137,-79.54374569503338,-79.54356709343435,-79.54534629991291,-79.55006569568022,-79.55014137807228,-79.55411699295297,-79.55435520293256,-79.55699129615529,-79.5570081066671,-79.55700878102192,-79.55675904384466,-79.55615400936495,-79.55620329442817],"lat":[43.62872439650568,43.62869873335397,43.62987911547886,43.63101348481155,43.63210848327835,43.63186186095231,43.63127731399953,43.63058498570295,43.63037890199972,43.62908022919364,43.62583569702479,43.62578264433912,43.6230688129148,43.62295470837932,43.62853450084392,43.62857522263599,43.62857694155328,43.62852706729169,43.62861844907573,43.62872439650568]}]],[[{"lng":[-79.54481132378167,-79.54346204304744,-79.54128095170996,-79.53794298656659,-79.53629251042729,-79.53662875761047,-79.53732972048401,-79.53676332448755,-79.53535213350131,-79.53521469771535,-79.5350025266202,-79.53488150594931,-79.53426248146368,-79.5340262006625,-79.53601411968003,-79.53660404469224,-79.53697488580995,-79.53932303444194,-79.5408160900113,-79.54352765938579,-79.54356709343435,-79.54374569503338,-79.54418895032137,-79.54463737243576,-79.54481132378167],"lat":[43.63210848327835,43.63281101539366,43.63482802905369,43.63855798136015,43.64049427349023,43.64062460864039,43.64137643484915,43.64149521431321,43.64178780113482,43.64144837105051,43.64093244188612,43.64062022807022,43.63908166362302,43.63844380484219,43.63679453382422,43.63630688275494,43.63599646655535,43.63396915792853,43.63271416042663,43.63040765637512,43.63037890199972,43.63058498570295,43.63127731399953,43.63186186095231,43.63210848327835]}]],[[{"lng":[-79.53535213350131,-79.53451897747924,-79.53338798018424,-79.53054499814949,-79.52870497987983,-79.52681948389734,-79.52678553910542,-79.52654406881412,-79.52619300779014,-79.52613845499744,-79.52585098011889,-79.52555224679806,-79.52529783232339,-79.52501305888747,-79.52473342178104,-79.5244867419308,-79.5242516518453,-79.52397909918535,-79.52374085437752,-79.52345367862257,-79.52311046488667,-79.52280496648852,-79.52252130400973,-79.52239284964683,-79.52220722358805,-79.52186801114651,-79.52147479120885,-79.52109109493122,-79.5207546689377,-79.52039908684409,-79.52013984130014,-79.52003698889388,-79.51985621007,-79.51959899629489,-79.51931989028039,-79.51895424496577,-79.51860830248481,-79.51828874260242,-79.51797905777164,-79.51767462353553,-79.51740034095798,-79.51716246573615,-79.5170166381468,-79.51705078450611,-79.51896878609159,-79.52008823904022,-79.52115848853668,-79.52228891753369,-79.52342946305436,-79.5245263947795,-79.52562521047324,-79.52684708335578,-79.52786322113016,-79.52814431354736,-79.5340258203241,-79.5340262006625,-79.53426248146368,-79.53488150594931,-79.5350025266202,-79.53521469771535,-79.53535213350131],"lat":[43.64178780113482,43.64214296654048,43.64362496770382,43.64676835019537,43.64878500158049,43.65089177732042,43.65092932856901,43.65088570634867,43.6507910689772,43.65077236131931,43.65067413185148,43.65052624494984,43.65034733180823,43.6501681163199,43.64997546329071,43.64976063982889,43.64952355028593,43.64931739372292,43.64907567131004,43.64887391566381,43.64874345186389,43.64855937196283,43.64833965519693,43.64826597784232,43.64815546274657,43.64802054590078,43.64790743460777,43.6477854431089,43.64769547943092,43.64762786201045,43.64742195854573,43.64738011734415,43.64735977209401,43.64738344696509,43.64736182719453,43.64732108116885,43.64723998993798,43.64709640512125,43.64692135012987,43.64672826819707,43.64653566660775,43.6462804320747,43.64610299922774,43.64609894124573,43.64566011403081,43.64541796531213,43.64519317313889,43.64491963751114,43.64464172032458,43.64439923706255,43.64418386385327,43.64390696157931,43.6436769979566,43.64361670336072,43.6384426296902,43.63844380484219,43.63908166362302,43.64062022807022,43.64093244188612,43.64144837105051,43.64178780113482]}]],[[{"lng":[-79.52639002112068,-79.52545679715296,-79.5248953188419,-79.52468187636512,-79.52270158980099,-79.52227114795855,-79.52216945014062,-79.52208138069655,-79.52196918555575,-79.52188013416092,-79.52178588316679,-79.52162739487999,-79.5214941808438,-79.52031822761496,-79.52003341833458,-79.51845564479832,-79.51747459715163,-79.51629503483495,-79.51617888939526,-79.51573332361598,-79.51535303095901,-79.51531699000704,-79.51510744323865,-79.51501196588195,-79.51403316901555,-79.51321094166353,-79.51311461019851,-79.51279846531729,-79.51264943450779,-79.51244630441956,-79.51211878807108,-79.51201185238207,-79.51173653420587,-79.51135397125771,-79.51244251467043,-79.51358011706162,-79.5137439454935,-79.5142248691304,-79.51456226665111,-79.51579622210683,-79.5170166381468,-79.51716246573615,-79.51740034095798,-79.51767462353553,-79.51797905777164,-79.51828874260242,-79.51860830248481,-79.51895424496577,-79.51931989028039,-79.51959899629489,-79.51985621007,-79.52003698889388,-79.52013984130014,-79.52039908684409,-79.5207546689377,-79.52109109493122,-79.52147479120885,-79.52186801114651,-79.52220722358805,-79.52239284964683,-79.52252130400973,-79.52280496648852,-79.52311046488667,-79.52345367862257,-79.52374085437752,-79.52397909918535,-79.5242516518453,-79.5244867419308,-79.52473342178104,-79.52501305888747,-79.52529783232339,-79.52555224679806,-79.52585098011889,-79.52613845499744,-79.52621481823451,-79.52639002112068],"lat":[43.65135641965954,43.65237521172845,43.65299379097564,43.65322509003472,43.6554369768,43.65592220532282,43.65602441422518,43.65610879511808,43.65619286394097,43.65625472678766,43.65630761025774,43.65638207790042,43.65644336917363,43.6568918471963,43.65700068538166,43.65752045775594,43.6578634979542,43.65829838445722,43.65803581650597,43.65715214869577,43.65635484667969,43.65627786085935,43.65584294892076,43.65563006961077,43.65351491263966,43.6516584328819,43.65141853450363,43.6506266489215,43.65026012521394,43.64985679881509,43.64908717835448,43.64883813882044,43.64824482741692,43.64735746063914,43.64711498438957,43.64686872477742,43.64682584491585,43.64671498077217,43.64666535391331,43.64637522399003,43.64610299922774,43.6462804320747,43.64653566660775,43.64672826819707,43.64692135012987,43.64709640512125,43.64723998993798,43.64732108116885,43.64736182719453,43.64738344696509,43.64735977209401,43.64738011734415,43.64742195854573,43.64762786201045,43.64769547943092,43.6477854431089,43.64790743460777,43.64802054590078,43.64815546274657,43.64826597784232,43.64833965519693,43.64855937196283,43.64874345186389,43.64887391566381,43.64907567131004,43.64931739372292,43.64952355028593,43.64976063982889,43.64997546329071,43.6501681163199,43.65034733180823,43.65052624494984,43.65067413185148,43.65077236131931,43.65094447568409,43.65135641965954]}]],[[{"lng":[-79.5170166381468,-79.51579622210683,-79.51456226665111,-79.5142248691304,-79.5137439454935,-79.51358011706162,-79.51244251467043,-79.51135397125771,-79.510238939803,-79.5100134350196,-79.50916428910318,-79.50890101375342,-79.50799402511937,-79.50774908906243,-79.50658718761873,-79.50544448088968,-79.50511535353688,-79.50474595160711,-79.5043619968361,-79.50399440064285,-79.50365872114526,-79.50330683542919,-79.50299656918989,-79.50273610743326,-79.50528686957692,-79.50727636891426,-79.50788989389105,-79.50796951176099,-79.50825783818078,-79.50858128472942,-79.50890701628059,-79.50890854928967,-79.50917135246277,-79.50936425440686,-79.50958086338345,-79.50985040437348,-79.51013849280527,-79.51042691173076,-79.51074568288433,-79.51102992237739,-79.51130422526347,-79.51156421987343,-79.51176624592894,-79.51203271479112,-79.51235588626227,-79.51251457315576,-79.51267867705387,-79.5130141258317,-79.51340466646329,-79.51376188855406,-79.51411603703548,-79.51446888415992,-79.5147853358186,-79.5149336914741,-79.51493701052496,-79.51494373131638,-79.5149109330348,-79.51486992128777,-79.51479926312554,-79.51479504675613,-79.51474541149645,-79.51470097744166,-79.51472250894706,-79.51483691020761,-79.51502931464053,-79.51524409105488,-79.51544969998966,-79.51549438880308,-79.51574057206821,-79.5159571528438,-79.51618934602945,-79.51646274160332,-79.51672083966987,-79.51694942708605,-79.5170166381468],"lat":[43.64610299922774,43.64637522399003,43.64666535391331,43.64671498077217,43.64682584491585,43.64686872477742,43.64711498438957,43.64735746063914,43.64760399225061,43.64766415721569,43.64784212763843,43.64792871394,43.64809700891518,43.64816132675287,43.64842071379611,43.64868493106863,43.64780274338298,43.64697413026666,43.64602370721708,43.64522203133225,43.64441176995517,43.64363280258576,43.64289497862694,43.64224323607846,43.64168689667899,43.64124019832845,43.6411112878918,43.64119064553074,43.64128452013928,43.64129774344234,43.6412929914507,43.64129301145102,43.64122433300975,43.64110081974922,43.64092810378765,43.64081009013446,43.64071923388893,43.64061496809803,43.6405110970796,43.64045178578463,43.64045535903111,43.64056226939418,43.64073143902363,43.64092404255842,43.64107678408394,43.6411463652032,43.64121601659565,43.64131049242292,43.64140550382052,43.64150017063619,43.64159929742321,43.64168490305406,43.641828638284,43.64206462054919,43.64214568253217,43.64231689968489,43.64257753425596,43.64285615628732,43.64308488152727,43.64309832987637,43.64329122986848,43.64348878849376,43.64371862167889,43.64391374334502,43.64410078629305,43.64434672316876,43.64452493484678,43.64456602472415,43.64479886545008,43.64506723960346,43.64529539664349,43.6455194969698,43.64577949648551,43.64602560934476,43.64610299922774]}]],[[{"lng":[-79.51629503483495,-79.51627535850187,-79.51626272466473,-79.51622596096459,-79.51617041576111,-79.51611568614885,-79.51606095011407,-79.51600681880758,-79.51595209130716,-79.51589877775517,-79.51584547278156,-79.5157929856107,-79.51574122066499,-79.51570212755652,-79.51569035381399,-79.51567450859875,-79.51562209714854,-79.51555333147006,-79.51548405877705,-79.51541416619227,-79.51534355791343,-79.51527306045138,-79.51522243315353,-79.51520234756599,-79.51517092788274,-79.51513103211178,-79.51505900097129,-79.5149868588382,-79.51491400970144,-79.5148404448759,-79.51476760429883,-79.51469332371531,-79.51466354837969,-79.51461965644307,-79.51454466881698,-79.51412448003748,-79.51357562511156,-79.51203414171538,-79.51044911678629,-79.50983976987148,-79.50938897308703,-79.50911588000015,-79.50863356841791,-79.50823367265698,-79.50790198911939,-79.50753170396926,-79.50717759004998,-79.50681614787473,-79.50657470438577,-79.50630332872197,-79.50602108062,-79.50583420374815,-79.50570878917786,-79.50561554579875,-79.50544219947328,-79.50526907646439,-79.50505386724218,-79.50477199370756,-79.50458044742533,-79.50445467186459,-79.50415352679751,-79.50387458496515,-79.50378970077436,-79.50362442112383,-79.50344277058733,-79.50323250726316,-79.50304062497281,-79.50285192410803,-79.50258088302576,-79.50244773631172,-79.50229068082622,-79.50209768629031,-79.50191096963533,-79.50171035198196,-79.50153446140114,-79.50141055071278,-79.50093516299138,-79.50073103531072,-79.5005406003857,-79.49964987121908,-79.4994487074229,-79.49929717739276,-79.49923729524903,-79.49920416399245,-79.49940242016214,-79.49975745773325,-79.50203556894198,-79.50319586604903,-79.50429832703087,-79.50544448088968,-79.50658718761873,-79.50774908906243,-79.50799402511937,-79.50890101375342,-79.50916428910318,-79.5100134350196,-79.510238939803,-79.51135397125771,-79.51173653420587,-79.51201185238207,-79.51211878807108,-79.51244630441956,-79.51264943450779,-79.51279846531729,-79.51311461019851,-79.51321094166353,-79.51403316901555,-79.51501196588195,-79.51510744323865,-79.51531699000704,-79.51535303095901,-79.51573332361598,-79.51617888939526,-79.51629503483495],"lat":[43.65829838445722,43.65830839137668,43.65831479887211,43.65833349599752,43.65835077899921,43.65836816258386,43.65838995707377,43.65841626042517,43.65843355393784,43.65845986785635,43.65848168082564,43.65850350439033,43.65852506724816,43.6585428337449,43.65855168290613,43.65855975902205,43.65858680469235,43.65862182980541,43.65865252729705,43.65868762770883,43.65872271878035,43.65875331021221,43.65877848844104,43.65878848985761,43.65880194484729,43.6588190705742,43.65884964194715,43.65888471286745,43.65891527351975,43.65894582482506,43.65897188445961,43.65900242636807,43.65901554250478,43.65903297619907,43.65905900778607,43.6592201739439,43.65943169715823,43.66002584945793,43.66062193395436,43.65911478611409,43.65797896764849,43.65729106804519,43.65608272857664,43.65517262111742,43.65435351348208,43.65347079614551,43.65268281054189,43.65272318929047,43.65276954600944,43.65280650894593,43.65281182184231,43.65281838092394,43.65281224019955,43.65282002311235,43.65280875431538,43.65278848609862,43.65275416400646,43.65267846001907,43.65262185139732,43.65259319937928,43.65230119181651,43.6520498934397,43.65197226454786,43.65182156638271,43.6516795656666,43.65153277892207,43.6513952346237,43.65125314061627,43.65108755232384,43.65101829215948,43.65094421748843,43.65085157740062,43.65078611553606,43.65072047113005,43.65065965161503,43.65062201861424,43.65051226030171,43.65046448227567,43.65041706357447,43.65027484369559,43.65022719230833,43.65015318606558,43.65009838727161,43.65005744285027,43.6500329498287,43.64997009525338,43.64945517068712,43.64920022004534,43.64893108760742,43.64868493106863,43.64842071379611,43.64816132675287,43.64809700891518,43.64792871394,43.64784212763843,43.64766415721569,43.64760399225061,43.64735746063914,43.64824482741692,43.64883813882044,43.64908717835448,43.64985679881509,43.65026012521394,43.6506266489215,43.65141853450363,43.6516584328819,43.65351491263966,43.65563006961077,43.65584294892076,43.65627786085935,43.65635484667969,43.65715214869577,43.65803581650597,43.65829838445722]}]],[[{"lng":[-79.50445467186459,-79.50458044742533,-79.50477199370756,-79.50505386724218,-79.50526907646439,-79.50544219947328,-79.50561554579875,-79.50570878917786,-79.50583420374815,-79.50602108062,-79.50630332872197,-79.50657470438577,-79.50681614787473,-79.50717759004998,-79.50753170396926,-79.50790198911939,-79.50823367265698,-79.50863356841791,-79.50911588000015,-79.50938897308703,-79.50983976987148,-79.51044911678629,-79.50860718440914,-79.50824110388355,-79.50754899346833,-79.50718448821857,-79.50630734667176,-79.50557911583149,-79.50551320004266,-79.50539452313728,-79.50513504283948,-79.5048488089714,-79.50445765257072,-79.5041928331417,-79.50408417821522,-79.50374537392406,-79.50343702433103,-79.50307296016476,-79.50268515024408,-79.50244842942725,-79.50228163115484,-79.50190395173702,-79.50147066903212,-79.50140480706405,-79.5010438261163,-79.50060191944577,-79.50013507384992,-79.49976666933043,-79.49950961484748,-79.49943010193428,-79.49951887752411,-79.49955049178516,-79.49972530613859,-79.50001572348734,-79.50039502014012,-79.50080569907861,-79.50122399017798,-79.50160139485683,-79.50193109795114,-79.50224581901537,-79.5024991483245,-79.50262343116428,-79.50268111111482,-79.50278368544471,-79.50284852264214,-79.50286045621895,-79.50277005721783,-79.50247781631424,-79.50205871101448,-79.50151222449466,-79.50093483743824,-79.50033052274078,-79.49970601606631,-79.49906237553742,-79.49846000887786,-79.4978512412787,-79.49775679165658,-79.49724408524196,-79.49663605161861,-79.49637102170044,-79.49601181825291,-79.49543499155443,-79.49482633166717,-79.494255339559,-79.4936386957859,-79.49304693300851,-79.49247072142633,-79.49195110543288,-79.4914848234651,-79.49134718723886,-79.49122747934841,-79.49107444279196,-79.49095407273485,-79.49092136874768,-79.49086721950236,-79.49101270273471,-79.49127589002022,-79.49152573687866,-79.49176930403247,-79.49195990078825,-79.49256097357822,-79.4936591583627,-79.49387670815035,-79.49454623399438,-79.49489820767165,-79.49543356342056,-79.49573484108913,-79.49624812360977,-79.49666742088021,-79.4971747563682,-79.49755375122341,-79.49808953950573,-79.49841250893726,-79.49879404778153,-79.49920416399245,-79.49923729524903,-79.49929717739276,-79.4994487074229,-79.49964987121908,-79.5005406003857,-79.50073103531072,-79.50093516299138,-79.50141055071278,-79.50153446140114,-79.50171035198196,-79.50191096963533,-79.50209768629031,-79.50229068082622,-79.50244773631172,-79.50258088302576,-79.50285192410803,-79.50304062497281,-79.50323250726316,-79.50344277058733,-79.50362442112383,-79.50378970077436,-79.50387458496515,-79.50415352679751,-79.50445467186459],"lat":[43.65259319937928,43.65262185139732,43.65267846001907,43.65275416400646,43.65278848609862,43.65280875431538,43.65282002311235,43.65281224019955,43.65281838092394,43.65281182184231,43.65280650894593,43.65276954600944,43.65272318929047,43.65268281054189,43.65347079614551,43.65435351348208,43.65517262111742,43.65608272857664,43.65729106804519,43.65797896764849,43.65911478611409,43.66062193395436,43.66134526411374,43.66148001838313,43.66175913849023,43.66192091612219,43.66238215396263,43.66276431370493,43.66279946002184,43.66273939466369,43.66254245663124,43.66233157489218,43.66213291259907,43.66199882697279,43.66193889141773,43.66169140055603,43.66140370817293,43.66111528521686,43.66084014311202,43.66067041328989,43.66055120069919,43.66027609866374,43.65999585530732,43.65995898344737,43.65976520612723,43.65955236185285,43.65933909855941,43.65903719496487,43.65868264972121,43.65829892785753,43.65790841342391,43.65785031503451,43.65752394443836,43.65715858154107,43.6568664006869,43.65661973059674,43.65635056364263,43.6560764480828,43.65578361184077,43.65546807310752,43.65512922391448,43.65490129942375,43.65479403022895,43.65443970207671,43.65404437013205,43.65362583992928,43.65321047018212,43.65286897344291,43.65257982571161,43.65239262027007,43.65226802145369,43.65216548155799,43.65213478004372,43.65211282591294,43.65205540211572,43.65197997687937,43.65197873544746,43.65196299307043,43.65197300072187,43.6520055224981,43.65205030753224,43.65209222672161,43.65219682831557,43.65227932772734,43.65229820805653,43.65225439859716,43.65211626940665,43.65190677747867,43.65162146951259,43.65142160706271,43.65124898735318,43.65085978761314,43.65053403351786,43.65044358091689,43.65003768236766,43.64967492909628,43.64937674403922,43.64905587708105,43.6487080104639,43.64845837724795,43.64865985153059,43.64902550213812,43.64910038589915,43.64932084597945,43.6494380077512,43.64959818211031,43.64966966350104,43.64978444309303,43.64987547895829,43.6499541678415,43.65001316321788,43.65002929753715,43.65004704444352,43.65006555976986,43.65005744285027,43.65009838727161,43.65015318606558,43.65022719230833,43.65027484369559,43.65041706357447,43.65046448227567,43.65051226030171,43.65062201861424,43.65065965161503,43.65072047113005,43.65078611553606,43.65085157740062,43.65094421748843,43.65101829215948,43.65108755232384,43.65125314061627,43.6513952346237,43.65153277892207,43.6516795656666,43.65182156638271,43.65197226454786,43.6520498934397,43.65230119181651,43.65259319937928]}]],[[{"lng":[-79.51044911678629,-79.51203414171538,-79.51357562511156,-79.51412448003748,-79.51454466881698,-79.51461965644307,-79.51466354837969,-79.51469332371531,-79.51476760429883,-79.5148404448759,-79.51491400970144,-79.5149868588382,-79.51505900097129,-79.51513103211178,-79.51517092788274,-79.51520234756599,-79.51522243315353,-79.51527306045138,-79.51534355791343,-79.51541416619227,-79.51548405877705,-79.51555333147006,-79.51562209714854,-79.51567450859875,-79.51569035381399,-79.51570212755652,-79.51574122066499,-79.5157929856107,-79.51584547278156,-79.51589877775517,-79.51595209130716,-79.51600681880758,-79.51606095011407,-79.51611568614885,-79.51617041576111,-79.51622596096459,-79.51626272466473,-79.51627535850187,-79.51629503483495,-79.51653587096155,-79.51704783080808,-79.51710908489872,-79.51712729931832,-79.51733345926067,-79.51758652927231,-79.51807960868722,-79.51825024612675,-79.51839487246886,-79.51850015641554,-79.51915338493677,-79.51963934500705,-79.52016357304325,-79.52027526246138,-79.52064159685868,-79.52097721915729,-79.52098706288687,-79.52108229688653,-79.52137306853389,-79.52178006628118,-79.52193630751992,-79.52212309170821,-79.5222652219544,-79.52239188848093,-79.52250903220769,-79.52281845018241,-79.52311660944699,-79.52336200128772,-79.52337217037545,-79.52349209910633,-79.52367510945477,-79.52369780827367,-79.52369992557647,-79.52363667341848,-79.52333909917148,-79.52305158623845,-79.52237514555604,-79.52226048779485,-79.52217256325524,-79.52198191116908,-79.52158846880012,-79.52115546039762,-79.52056042141656,-79.51981560624387,-79.51948313663856,-79.51936096962203,-79.51896298557659,-79.51873775820947,-79.51863480108749,-79.51843862963776,-79.51826964439678,-79.51792824742424,-79.51760698353405,-79.51729891473147,-79.5169846944755,-79.51668184681141,-79.51637746703415,-79.51598903532688,-79.51579598367546,-79.51551212211862,-79.51496670508745,-79.51461899922337,-79.5142958171635,-79.51356560965075,-79.51313286623426,-79.51274098376209,-79.51239243654695,-79.51240835976863,-79.51243017163199,-79.5124220844705,-79.51234542066324,-79.51223978467466,-79.51208094329895,-79.5119562629073,-79.51185949201164,-79.51183366712964,-79.51179441995417,-79.51172875318355,-79.51162272969675,-79.51150124481293,-79.51147724891477,-79.51133438359541,-79.51126479484714,-79.51144869229859,-79.5117779211842,-79.51215497248666,-79.51251990482774,-79.51278896697283,-79.5129773094211,-79.51306649572277,-79.51307190405899,-79.51303959394866,-79.51294643317596,-79.51285994578311,-79.51279491207416,-79.51268517421857,-79.51257470431779,-79.51239544775189,-79.51220258451986,-79.51211651686428,-79.51200452362937,-79.51197481736587,-79.51195040958004,-79.51183102003569,-79.51167966653489,-79.51152667767417,-79.51148448772933,-79.51129428770795,-79.51107539702843,-79.51097278814454,-79.510567883048,-79.5102384316192,-79.50977413639471,-79.50928120469905,-79.50883311070596,-79.5083652259555,-79.50793257963228,-79.50749045378774,-79.50707878284166,-79.50659240977588,-79.50615687880072,-79.50575866885474,-79.50551320004266,-79.50557911583149,-79.50630734667176,-79.50718448821857,-79.50754899346833,-79.50824110388355,-79.50860718440914,-79.51044911678629],"lat":[43.66062193395436,43.66002584945793,43.65943169715823,43.6592201739439,43.65905900778607,43.65903297619907,43.65901554250478,43.65900242636807,43.65897188445961,43.65894582482506,43.65891527351975,43.65888471286745,43.65884964194715,43.6588190705742,43.65880194484729,43.65878848985761,43.65877848844104,43.65875331021221,43.65872271878035,43.65868762770883,43.65865252729705,43.65862182980541,43.65858680469235,43.65855975902205,43.65855168290613,43.6585428337449,43.65852506724816,43.65850350439033,43.65848168082564,43.65845986785635,43.65843355393784,43.65841626042517,43.65838995707377,43.65836816258386,43.65835077899921,43.65833349599752,43.65831479887211,43.65830839137668,43.65829838445722,43.65884622609094,43.66014943831696,43.6603032682768,43.66034851495137,43.66085989654399,43.66138097828811,43.66241828955332,43.66285269117325,43.66314722330764,43.6633781407899,43.6649038144386,43.6659500282589,43.66726679714778,43.66757890434544,43.66842101829508,43.66917721398037,43.66921785062457,43.66947564163249,43.67010522687294,43.67115040916408,43.6715441092778,43.6719562082261,43.67229571305627,43.67253599519517,43.67278506610597,43.67335628507517,43.67393635971772,43.67452025278595,43.67455189134715,43.67487958504483,43.67537967252876,43.6754384790564,43.67544390763551,43.67542868727404,43.67542484276893,43.67551123800997,43.67578155761978,43.67590610378079,43.67645427172571,43.67664084920111,43.67655006078227,43.67645011707017,43.67643341371827,43.67645077569859,43.67634744668547,43.67627375761578,43.67631379105882,43.67639189049884,43.67654359078451,43.67645102739337,43.67625970362298,43.67601222161115,43.6757739117455,43.67537373532444,43.67503658257753,43.67474449677457,43.67450640255506,43.67415928157094,43.67394063543556,43.67388293634319,43.67386684842315,43.67388033388463,43.6739031391457,43.67383963188526,43.67377999038245,43.6736397705394,43.67362524190639,43.67357692813138,43.67350969667994,43.67305039691294,43.67257220037225,43.67211163069219,43.67166837230118,43.67122114744787,43.67077869685186,43.67060273053669,43.67033665892227,43.6698856111268,43.66947004591039,43.66911729357923,43.66904946563719,43.66865142435903,43.66827234153954,43.66788755846811,43.66758568616073,43.66734753989519,43.66713615079295,43.66685158655338,43.66653887522573,43.66616185912325,43.66575224575651,43.6653376405927,43.66493124570766,43.66451143446502,43.66410090436777,43.66370329553843,43.66331467911566,43.66291616516938,43.66262558858147,43.66256586467942,43.66248338802989,43.66246049609656,43.66244667520746,43.66238210621422,43.66229911671734,43.66221610572052,43.66219764212486,43.66211405614019,43.66206169331161,43.66204235235963,43.6619605586724,43.66192925848645,43.66199522187957,43.66211491227285,43.6623071120562,43.66247654661295,43.66261052106906,43.66278028831797,43.66290544136606,43.66301170270239,43.66300601072789,43.66290169234148,43.66279946002184,43.66276431370493,43.66238215396263,43.66192091612219,43.66175913849023,43.66148001838313,43.66134526411374,43.66062193395436]}]],[[{"lng":[-79.51963934500705,-79.51915338493677,-79.51850015641554,-79.51839487246886,-79.51825024612675,-79.51807960868722,-79.51758652927231,-79.51733345926067,-79.51712729931832,-79.51710908489872,-79.51704783080808,-79.51653587096155,-79.51629503483495,-79.51747459715163,-79.51845564479832,-79.52003341833458,-79.52031822761496,-79.5214941808438,-79.52162739487999,-79.52178588316679,-79.52188013416092,-79.52196918555575,-79.52208138069655,-79.52216945014062,-79.52227114795855,-79.52270158980099,-79.52468187636512,-79.5248953188419,-79.52545679715296,-79.52639002112068,-79.52668624370722,-79.52672880555224,-79.52721768115423,-79.52791381667983,-79.52875491425567,-79.52922686316076,-79.52978012244171,-79.52981640327926,-79.53011854956212,-79.5301969518417,-79.53050276787569,-79.53086992783878,-79.53117937180411,-79.5312562019574,-79.53162283192583,-79.53202456044845,-79.53039694671671,-79.53020177971823,-79.52991175516796,-79.52964816941375,-79.52944231952405,-79.52929991259192,-79.52917233070472,-79.52902850780869,-79.52891668651536,-79.52876530563051,-79.52863192699476,-79.52860434494519,-79.52850298331532,-79.52815119351408,-79.52644224752855,-79.52487458764844,-79.52432403716729,-79.52416544541869,-79.52402123522214,-79.52394253580768,-79.52384172082019,-79.523583677247,-79.52346967352031,-79.52336005795759,-79.52324571516436,-79.52315540418476,-79.52287014797717,-79.5227645648757,-79.52256943346204,-79.52247485102684,-79.52239625023677,-79.52230790403091,-79.52222863023015,-79.522165961995,-79.52209774822674,-79.52204481700181,-79.52193801782545,-79.52189991929116,-79.52179338543924,-79.52156439773142,-79.52149669387313,-79.52141472914744,-79.52134881370166,-79.52127501950085,-79.52120099586219,-79.52112380247532,-79.52102885286266,-79.52094416216326,-79.5208654259003,-79.5207809450373,-79.52066313512817,-79.5205636697341,-79.52045275894065,-79.52035252031368,-79.52022070731041,-79.5200936032965,-79.51996672238747,-79.51982593623313,-79.51975861035238,-79.51963934500705],"lat":[43.6659500282589,43.6649038144386,43.6633781407899,43.66314722330764,43.66285269117325,43.66241828955332,43.66138097828811,43.66085989654399,43.66034851495137,43.6603032682768,43.66014943831696,43.65884622609094,43.65829838445722,43.6578634979542,43.65752045775594,43.65700068538166,43.6568918471963,43.65644336917363,43.65638207790042,43.65630761025774,43.65625472678766,43.65619286394097,43.65610879511808,43.65602441422518,43.65592220532282,43.6554369768,43.65322509003472,43.65299379097564,43.65237521172845,43.65135641965954,43.65201288969407,43.65210805024205,43.65326229661338,43.65487399550324,43.65680712967515,43.65785771492219,43.65920182142504,43.65926980368495,43.6599715306679,43.66016158248777,43.66081384417004,43.66171445045141,43.66240726611917,43.66262430314395,43.66342587722013,43.66427741221877,43.66464817453485,43.66469967714515,43.6647364562069,43.66473756642068,43.66475742306601,43.66474658854121,43.66473594463099,43.66471608948064,43.66469664607042,43.66467219227172,43.66464346900695,43.66462961080875,43.66459229752677,43.664457147603,43.66390391440366,43.6633749844757,43.66319233685221,43.66314518866741,43.66311632014742,43.66310180065203,43.6631049995885,43.66309725535659,43.66310028358645,43.6631078694322,43.6631288971892,43.66314582436467,43.66320515234332,43.66324429696453,43.66333179526567,43.66339358698907,43.66343758112772,43.6634994533382,43.66357503590654,43.66362823796092,43.66369487136588,43.66373469616848,43.6638233356491,43.66385885117887,43.66396999911396,43.6641741746297,43.66424081426433,43.66433427555098,43.66442803444426,43.66452610236873,43.66463767036959,43.66474919730366,43.66489659267631,43.66502602658696,43.66514203440199,43.66526705989082,43.66539156388673,43.66549371005221,43.66559129694878,43.66566660677296,43.66573241560867,43.66579387425804,43.66584624357256,43.66589843265453,43.66592456685635,43.6659500282589]}]],[[{"lng":[-79.37429554794018,-79.37252750344544,-79.37179019883129,-79.37137161891411,-79.37305247992553,-79.37423033714872,-79.37429457933328,-79.37429554794018],"lat":[43.64534465187621,43.64568182080398,43.64375351971363,43.64305430661144,43.64247501898473,43.64509382701684,43.64534265764439,43.64534465187621]}]],[[{"lng":[-79.53258859895767,-79.53235408068659,-79.53117937180411,-79.53086992783878,-79.53050276787569,-79.5301969518417,-79.53011854956212,-79.52981640327926,-79.52978012244171,-79.52922686316076,-79.52875491425567,-79.52791381667983,-79.52721768115423,-79.52672880555224,-79.52668624370722,-79.52639002112068,-79.52621481823451,-79.52613845499744,-79.52619300779014,-79.52654406881412,-79.52678553910542,-79.52687067094124,-79.52720888857191,-79.52757485166865,-79.52790185427817,-79.52824969634777,-79.52862625485976,-79.52899023383421,-79.52933955930916,-79.52965021204002,-79.52997633682786,-79.53032906431655,-79.53072905352086,-79.53083352501626,-79.53088780522201,-79.53094959207385,-79.53096917791623,-79.53098835443001,-79.53096466991637,-79.53101539420305,-79.53109692797636,-79.5312602145917,-79.53148898240318,-79.53197683930655,-79.53219176650576,-79.53227750832039,-79.53226741910026,-79.53221658396905,-79.53223796284169,-79.53231358525386,-79.53234060037671,-79.53239497474308,-79.53257035165402,-79.53287261561698,-79.53321335181019,-79.53358790809777,-79.53394271405686,-79.53429554898671,-79.53463612499031,-79.53496340488094,-79.53521428810824,-79.53549762463379,-79.53592308199237,-79.53625972252826,-79.53641633226371,-79.53659077789,-79.53676621734176,-79.53695182956122,-79.53714689930166,-79.53742510245668,-79.53751058215239,-79.53752930434769,-79.53751069940309,-79.5374732828478,-79.53743383348515,-79.53738250992656,-79.53733676442187,-79.5372858795627,-79.53722793408032,-79.5371573455793,-79.53715883446557,-79.53721799091734,-79.53731392499219,-79.53742512100409,-79.53752291192897,-79.53760391035367,-79.53767970957331,-79.5377718558731,-79.53785553393182,-79.53798551335424,-79.53815673873734,-79.53832616608828,-79.53852804073799,-79.5387017219706,-79.5388886994445,-79.53905265550755,-79.53921758663797,-79.53939100235728,-79.53957191462509,-79.53974866270026,-79.53989651475135,-79.54003413024522,-79.54014773149504,-79.54020698742596,-79.54020653718557,-79.54016253255068,-79.54009839207254,-79.54001788016694,-79.53993300014952,-79.53982786651694,-79.53973866863468,-79.53966429807231,-79.53964042599681,-79.53964528549412,-79.53974946458148,-79.53988567317302,-79.5400089539034,-79.54012097653077,-79.54021787700712,-79.5403284753705,-79.54045137438459,-79.54046784799411,-79.54081267336772,-79.54176825673915,-79.54222281102264,-79.54231472115633,-79.54276074350076,-79.54290936841055,-79.54321593055847,-79.54253114643492,-79.5418346511896,-79.54054470156149,-79.53905187551931,-79.53850586077903,-79.53559435948893,-79.53455895105076,-79.5334370160918,-79.53258859895767],"lat":[43.66212370934698,43.66217471040907,43.66240726611917,43.66171445045141,43.66081384417004,43.66016158248777,43.6599715306679,43.65926980368495,43.65920182142504,43.65785771492219,43.65680712967515,43.65487399550324,43.65326229661338,43.65210805024205,43.65201288969407,43.65135641965954,43.65094447568409,43.65077236131931,43.6507910689772,43.65088570634867,43.65092932856901,43.65093942798278,43.65095720000767,43.65094841225311,43.65094812359197,43.65101561798811,43.65108798206757,43.65116468403977,43.65114217336507,43.65107865471946,43.65099282902094,43.65095235490513,43.6508899820744,43.65082822045379,43.65076590359847,43.65067667694801,43.65057790575354,43.65035751111041,43.65015465974179,43.65007879398201,43.65003933252451,43.65001433474853,43.65001277335978,43.65003254389597,43.65017942777288,43.65037857480846,43.6506080884601,43.65081448359113,43.65105340343646,43.65131543505536,43.65160843974929,43.65186128636921,43.65207058619217,43.65213748130444,43.65211934957516,43.65211065285088,43.65211520467874,43.65213773429571,43.65215560460461,43.65224090911403,43.65241516440143,43.65249531311323,43.65249176200779,43.65260859998723,43.65260610457604,43.65257233003456,43.65255657211807,43.65257245144997,43.65261995882608,43.6527811458302,43.65289926677379,43.653021034475,43.65315141663185,43.65328587905898,43.65341140339488,43.65354586785931,43.65364430533378,43.65376068117807,43.65388146767108,43.65397967712546,43.65420474822766,43.65429102515447,43.65439136588012,43.65447380764974,43.65456507985795,43.65467414127617,43.65477413403158,43.65487892694306,43.65497451934076,43.65507520502099,43.65514491084217,43.65518758716982,43.65520367192722,43.65520589273221,43.65520378221649,43.65521488023882,43.65518998216637,43.65516519228783,43.65511799273722,43.65505273553287,43.65496460369523,43.65488534298178,43.65480118430222,43.6546894151595,43.65458138442947,43.65447729842167,43.65437286516765,43.6542773148455,43.6541637044979,43.65405883740893,43.6539630858927,43.65386311274167,43.65376378480492,43.65366032294497,43.65358513635294,43.65355536927839,43.65355694418686,43.65356737724111,43.65357311599564,43.65357902961331,43.65356259500282,43.65355371327484,43.65435498068199,43.65656872586666,43.65764595202705,43.6578497610789,43.65883235428142,43.65912681667968,43.65985098069816,43.65999537533669,43.6601080191669,43.66040230552754,43.66071630925146,43.66082644522469,43.66146442196925,43.66168079315866,43.66191395877772,43.66212370934698]}]],[[{"lng":[-79.54046784799411,-79.54045137438459,-79.5403284753705,-79.54021787700712,-79.54012097653077,-79.5400089539034,-79.53988567317302,-79.53974946458148,-79.53964528549412,-79.53964042599681,-79.53966429807231,-79.53973866863468,-79.53982786651694,-79.53993300014952,-79.54001788016694,-79.54009839207254,-79.54016253255068,-79.54020653718557,-79.54020698742596,-79.54014773149504,-79.54003413024522,-79.53989651475135,-79.53974866270026,-79.53957191462509,-79.53939100235728,-79.53921758663797,-79.53905265550755,-79.5388886994445,-79.5387017219706,-79.53852804073799,-79.53832616608828,-79.53815673873734,-79.53798551335424,-79.53785553393182,-79.5377718558731,-79.53767970957331,-79.53760391035367,-79.53752291192897,-79.53742512100409,-79.53731392499219,-79.53721799091734,-79.53715883446557,-79.5371573455793,-79.53722793408032,-79.5372858795627,-79.53733676442187,-79.53738250992656,-79.53743383348515,-79.5374732828478,-79.53751069940309,-79.53752930434769,-79.53751058215239,-79.53742510245668,-79.53714689930166,-79.53695182956122,-79.53676621734176,-79.53659077789,-79.53641633226371,-79.53625972252826,-79.53592308199237,-79.53549762463379,-79.53521428810824,-79.53496340488094,-79.53463612499031,-79.53429554898671,-79.53394271405686,-79.53358790809777,-79.53321335181019,-79.53287261561698,-79.53257035165402,-79.53239497474308,-79.53234060037671,-79.53231358525386,-79.53223796284169,-79.53221658396905,-79.53226741910026,-79.53227750832039,-79.53219176650576,-79.53197683930655,-79.53148898240318,-79.5312602145917,-79.53109692797636,-79.53101539420305,-79.53096466991637,-79.53098835443001,-79.53096917791623,-79.53094959207385,-79.53088780522201,-79.53083352501626,-79.53072905352086,-79.53032906431655,-79.52997633682786,-79.52965021204002,-79.52933955930916,-79.52899023383421,-79.52862625485976,-79.52824969634777,-79.52790185427817,-79.52757485166865,-79.52720888857191,-79.52687067094124,-79.52678553910542,-79.52681948389734,-79.52870497987983,-79.53054499814949,-79.53338798018424,-79.53451897747924,-79.53535213350131,-79.53579574790395,-79.53619067778956,-79.53677718023935,-79.53719096360646,-79.53762193829741,-79.5380664957212,-79.53852979485568,-79.539386260844,-79.53972082208576,-79.53995797078204,-79.54046784799411],"lat":[43.65355371327484,43.65356259500282,43.65357902961331,43.65357311599564,43.65356737724111,43.65355694418686,43.65355536927839,43.65358513635294,43.65366032294497,43.65376378480492,43.65386311274167,43.6539630858927,43.65405883740893,43.6541637044979,43.6542773148455,43.65437286516765,43.65447729842167,43.65458138442947,43.6546894151595,43.65480118430222,43.65488534298178,43.65496460369523,43.65505273553287,43.65511799273722,43.65516519228783,43.65518998216637,43.65521488023882,43.65520378221649,43.65520589273221,43.65520367192722,43.65518758716982,43.65514491084217,43.65507520502099,43.65497451934076,43.65487892694306,43.65477413403158,43.65467414127617,43.65456507985795,43.65447380764974,43.65439136588012,43.65429102515447,43.65420474822766,43.65397967712546,43.65388146767108,43.65376068117807,43.65364430533378,43.65354586785931,43.65341140339488,43.65328587905898,43.65315141663185,43.653021034475,43.65289926677379,43.6527811458302,43.65261995882608,43.65257245144997,43.65255657211807,43.65257233003456,43.65260610457604,43.65260859998723,43.65249176200779,43.65249531311323,43.65241516440143,43.65224090911403,43.65215560460461,43.65213773429571,43.65211520467874,43.65211065285088,43.65211934957516,43.65213748130444,43.65207058619217,43.65186128636921,43.65160843974929,43.65131543505536,43.65105340343646,43.65081448359113,43.6506080884601,43.65037857480846,43.65017942777288,43.65003254389597,43.65001277335978,43.65001433474853,43.65003933252451,43.65007879398201,43.65015465974179,43.65035751111041,43.65057790575354,43.65067667694801,43.65076590359847,43.65082822045379,43.6508899820744,43.65095235490513,43.65099282902094,43.65107865471946,43.65114217336507,43.65116468403977,43.65108798206757,43.65101561798811,43.65094812359197,43.65094841225311,43.65095720000767,43.65093942798278,43.65092932856901,43.65089177732042,43.64878500158049,43.64676835019537,43.64362496770382,43.64214296654048,43.64178780113482,43.64279289820823,43.64369834682167,43.64504743942618,43.64600713821955,43.64696255430169,43.64799925120835,43.64909010840206,43.65097385151491,43.6518515090732,43.65239475399525,43.65355371327484]}]],[[{"lng":[-79.53732972048401,-79.53815222694712,-79.53954747522982,-79.54038871432638,-79.54061843412572,-79.54081483609468,-79.54097075282728,-79.54172189521759,-79.54221837080114,-79.54250542861693,-79.54449654261673,-79.5450369302161,-79.54592911259637,-79.54645363435455,-79.54669963152983,-79.54685584892511,-79.54752732040758,-79.54753324484965,-79.54755897925142,-79.54815616314175,-79.54897210156464,-79.54995592009668,-79.55076526582539,-79.55089843262469,-79.55131933788577,-79.55148881103389,-79.5515091846263,-79.55156212052468,-79.55162496650614,-79.55171184164551,-79.55180149493391,-79.5519036296961,-79.55200558606548,-79.5521126420962,-79.55255865577089,-79.55324373047918,-79.55396554617469,-79.55447226187087,-79.55474020118686,-79.55491393820398,-79.55502177017516,-79.55508987161738,-79.55523618999526,-79.55383546645662,-79.5536756346755,-79.55229910796925,-79.54954386101683,-79.54910975351041,-79.54858616104347,-79.54754203508502,-79.5465405987287,-79.54640457243345,-79.54553724146068,-79.54533497993928,-79.54430013466683,-79.54321593055847,-79.54290936841055,-79.54276074350076,-79.54231472115633,-79.54222281102264,-79.54176825673915,-79.54081267336772,-79.54046784799411,-79.53995797078204,-79.53972082208576,-79.539386260844,-79.53852979485568,-79.5380664957212,-79.53762193829741,-79.53719096360646,-79.53677718023935,-79.53619067778956,-79.53579574790395,-79.53535213350131,-79.53676332448755,-79.53732972048401],"lat":[43.64137643484915,43.64119791391943,43.64088708452185,43.64068169049842,43.64063511242416,43.64058810851895,43.64054958956012,43.64037463288854,43.64025475694725,43.64020440534215,43.63977058199149,43.63966493618043,43.64164360998991,43.64287483776771,43.6434677836298,43.64380284760157,43.64530159021899,43.64537818333198,43.64546853133911,43.64683111434886,43.64879988170856,43.65100932810256,43.65286997277359,43.65320023742716,43.65416447546676,43.65456280521339,43.65461707592898,43.65470776765277,43.65480308598143,43.65489420773313,43.65497186149536,43.65504076126939,43.65509156450427,43.65515143433635,43.6553911379576,43.65575098336135,43.6561653020028,43.65643286240347,43.65659828842522,43.65672201339216,43.65681798864624,43.65688636524732,43.65702774723897,43.65738820920583,43.65742219498593,43.65771542867826,43.65838283707149,43.65847634981409,43.65859122972225,43.65882111122097,43.65907844227285,43.65911722163811,43.65931332490956,43.65934225798345,43.65957213789272,43.65985098069816,43.65912681667968,43.65883235428142,43.6578497610789,43.65764595202705,43.65656872586666,43.65435498068199,43.65355371327484,43.65239475399525,43.6518515090732,43.65097385151491,43.64909010840206,43.64799925120835,43.64696255430169,43.64600713821955,43.64504743942618,43.64369834682167,43.64279289820823,43.64178780113482,43.64149521431321,43.64137643484915]}]],[[{"lng":[-79.55968815538345,-79.5595756967455,-79.55941295545524,-79.55933979305942,-79.55917692451679,-79.55905690860803,-79.55685904513682,-79.55685914733213,-79.55669925548666,-79.55663571523024,-79.55245747279665,-79.55146941804556,-79.54776281236306,-79.54639615437685,-79.5450369302161,-79.54449654261673,-79.54250542861693,-79.54221837080114,-79.54172189521759,-79.54097075282728,-79.54081483609468,-79.54061843412572,-79.54038871432638,-79.53954747522982,-79.53815222694712,-79.53732972048401,-79.53662875761047,-79.53629251042729,-79.53794298656659,-79.54128095170996,-79.54346204304744,-79.54481132378167,-79.54691427164551,-79.55188540608135,-79.55608871554207,-79.55620329442817,-79.55968815538345],"lat":[43.63621185623738,43.6362294334537,43.63626789184403,43.63628371359259,43.63631892928149,43.63634901367403,43.63688364516287,43.63688364645325,43.63692249703237,43.63693771838479,43.63794028405182,43.63817091284746,43.63902426659308,43.63934006120969,43.63966493618043,43.63977058199149,43.64020440534215,43.64025475694725,43.64037463288854,43.64054958956012,43.64058810851895,43.64063511242416,43.64068169049842,43.64088708452185,43.64119791391943,43.64137643484915,43.64062460864039,43.64049427349023,43.63855798136015,43.63482802905369,43.63281101539366,43.63210848327835,43.63101348481155,43.62987911547886,43.62869873335397,43.62872439650568,43.63621185623738]}]],[[{"lng":[-79.58171516904399,-79.58047855425606,-79.57981733170156,-79.57789831468824,-79.57775977023104,-79.57746857438053,-79.57714509161899,-79.5768904328673,-79.57662340847348,-79.57514894264729,-79.57371003863125,-79.57341449303682,-79.57303359027092,-79.57077435520547,-79.56960390761917,-79.56956485946392,-79.5694660180758,-79.56929869954907,-79.5690784321545,-79.56890558571797,-79.56790874727795,-79.56722010644893,-79.56600544753614,-79.5657480455166,-79.56364495222425,-79.56230839497685,-79.56225247769565,-79.56054607936564,-79.56049538063959,-79.56045888687463,-79.56002894261306,-79.55978891651422,-79.55968815538345,-79.55620329442817,-79.55615400936495,-79.55675904384466,-79.55700878102192,-79.5570081066671,-79.55699129615529,-79.55696725825631,-79.55711597556089,-79.55724854810784,-79.55727031812253,-79.5573309278253,-79.5574245583107,-79.5575797179341,-79.55773476975094,-79.55789064426425,-79.55804569788067,-79.55820135686156,-79.55835794609158,-79.55851371230844,-79.55867019361791,-79.55882656722515,-79.55888563821536,-79.55898386564658,-79.55914105638864,-79.55929835447731,-79.5594554395766,-79.55961355478584,-79.55977156236239,-79.55992946234196,-79.56058163264392,-79.56098657342983,-79.56236236461663,-79.56374972196794,-79.56375739975702,-79.56377265745492,-79.56378850777524,-79.56382099345471,-79.56386623675431,-79.5638825363622,-79.56388894163652,-79.56389213957281,-79.56394361920601,-79.56409780256431,-79.56432488577636,-79.56452927547362,-79.56478183212653,-79.56501530080016,-79.56526543972012,-79.56551496074346,-79.56575178239959,-79.56599240430297,-79.56620638108106,-79.56643108205408,-79.56664894166028,-79.56685126970858,-79.56696082679844,-79.56706620861492,-79.56725735320282,-79.56734987407171,-79.56742718399718,-79.56747684729547,-79.5675127656494,-79.56750948772859,-79.56752313409669,-79.56759580872004,-79.56769132750156,-79.56783372517977,-79.56799183651167,-79.56816281024643,-79.56835593751465,-79.56857507344847,-79.56878626753745,-79.56896471861977,-79.56913651009953,-79.56935728683268,-79.56957920143344,-79.56978413181997,-79.5699923683604,-79.57020534119151,-79.57037862448921,-79.57055874432224,-79.57077881110914,-79.57096392553406,-79.57117632352079,-79.57138790359996,-79.57158915000977,-79.57180952527175,-79.57201590157491,-79.57206342745293,-79.57223136462362,-79.57245020991246,-79.57265348908436,-79.5728829168421,-79.57308555665291,-79.57328203154135,-79.57349122335198,-79.57368214209551,-79.57386164620428,-79.57402213854431,-79.57419031684756,-79.57433152404522,-79.57449764101311,-79.57466913542828,-79.57486392802774,-79.57505388285961,-79.57523603713875,-79.57540223770813,-79.57557007746932,-79.57572349417443,-79.57586810358605,-79.57602109194025,-79.57616877771459,-79.57632636275409,-79.57646894633714,-79.57660600860505,-79.57674296901185,-79.57688597823767,-79.57705616708373,-79.57722226509725,-79.57738223057534,-79.57754722099729,-79.57770566037154,-79.57787983942724,-79.57806361570859,-79.57824556478749,-79.57843895036717,-79.57861434889425,-79.57877934089838,-79.57898009961272,-79.57918656142371,-79.57937799350017,-79.57955452890263,-79.57969569529293,-79.579761358829,-79.57980925455797,-79.57991882155946,-79.58000466806944,-79.5801175108608,-79.58023146879165,-79.58033480405146,-79.58045050345152,-79.58057816950193,-79.58074630476624,-79.58083915744042,-79.58093599528586,-79.58109903177666,-79.58130725098701,-79.58134811608087,-79.58139989257705,-79.58151440475248,-79.58165417410466,-79.58171516904399],"lat":[43.62794790008505,43.62881043593942,43.62928392565154,43.63066925747148,43.63076205587519,43.63095197765903,43.63113258465668,43.6312554438685,43.63137814851383,43.6320351159365,43.63266091055624,43.63277884069674,43.63292712181875,43.63354722563713,43.63389716927492,43.6339057727196,43.63392704093759,43.63395645403363,43.63399870717424,43.63403705260396,43.63428562812413,43.63444362666615,43.63472105219219,43.63479434038926,43.6352991446112,43.6356200093316,43.6356328991325,43.6360256081859,43.63603397177131,43.63604251424787,43.63613612149328,43.63619611197578,43.63621185623738,43.62872439650568,43.62861844907573,43.62852706729169,43.62857694155328,43.62857522263599,43.62853450084392,43.62847118249099,43.62843705153092,43.62840271658202,43.62839849032189,43.62838575218748,43.62835992761173,43.62832137629625,43.62828732447112,43.62824878175342,43.62821463950691,43.62818509576937,43.62814656142864,43.62811251755515,43.62807848248661,43.62804894690756,43.628040689607,43.6280149217117,43.62798539600378,43.62795137036992,43.62792625393823,43.62789223816839,43.62786272188551,43.62783770509063,43.62771079839084,43.62762587618314,43.62733702722699,43.62701688959071,43.62703354998417,43.62706668949696,43.62714790781202,43.62724733941001,43.62736043448862,43.6274957612121,43.62761737039886,43.62777494778705,43.62788371014287,43.62798467064847,43.62804603710511,43.62811612022337,43.62815079932095,43.62816723396016,43.62817037419449,43.628178007189,43.62821248665701,43.62822900906276,43.62822269139729,43.62822100890438,43.62821473911087,43.62820827415288,43.62819164358338,43.62816145737392,43.62810083873105,43.62805698785155,43.62799944305508,43.62790095210946,43.62775286719786,43.62766280484732,43.62760896311808,43.62748834521808,43.62731391076021,43.62715815791274,43.62702501705839,43.62690112926504,43.6267955229616,43.62673525270331,43.62674689960381,43.62680314680334,43.62687731460957,43.62691158574015,43.62693236749227,43.62697544160705,43.62698254820602,43.62695370516116,43.62686585145018,43.62678708510454,43.62671331906462,43.62664802770431,43.62657866607734,43.62650929383079,43.62645329524076,43.6263750299985,43.62629658943082,43.62628368000244,43.62622717410791,43.62614438744747,43.62609741449555,43.62605976980959,43.62603979449531,43.62605575038835,43.6260223529449,43.62597072282586,43.62588744250309,43.62581283681346,43.62571141040559,43.62560514626223,43.6255216979478,43.62547882604142,43.62543615435845,43.62542952057754,43.62545888779074,43.62550597010809,43.62554857154648,43.62560899727253,43.62569181838215,43.62577474369896,43.62584851060164,43.62593158279248,43.62599637364341,43.62606559662169,43.62613481816673,43.6261816095348,43.62622873910314,43.62628031856001,43.62633191146648,43.62636547240342,43.62641245477746,43.62645513170721,43.62650692989975,43.62654520187182,43.62659261792709,43.62664431103481,43.62667787021229,43.62670737277209,43.62675945120063,43.62682043459944,43.62685863719873,43.6269144042771,43.62696023087994,43.62699233347624,43.62707921508984,43.62718380616042,43.6272618161708,43.62735325309457,43.62744005689571,43.62752701416508,43.62760061673468,43.62766580958078,43.62770297104517,43.62774018191276,43.62779621878378,43.62783031093599,43.62783982029379,43.62787647145355,43.62791840226123,43.62793814106073,43.62794790008505]}]],[[{"lng":[-79.58568526924452,-79.58513143883577,-79.5843581103838,-79.58366549299865,-79.58353459737246,-79.58342557959845,-79.58331534360349,-79.58319908663566,-79.58308518006444,-79.58291169288481,-79.58260978878184,-79.58198790679967,-79.58222838969995,-79.582220277811,-79.58219997856379,-79.58218552908485,-79.58214007788229,-79.58208849370452,-79.58193455607666,-79.58142375765739,-79.58080054865916,-79.58047743453808,-79.58021038837738,-79.58027560655087,-79.57957592226214,-79.57946718237315,-79.57875961695939,-79.57864009864629,-79.57728716385708,-79.57703067898221,-79.5766244264466,-79.57652826526058,-79.5763455496401,-79.57590034039561,-79.5755046040842,-79.57544649789163,-79.57528788057333,-79.575077354193,-79.57489080967142,-79.57469339237397,-79.57427861483099,-79.57400102999644,-79.57354032379065,-79.57302983639725,-79.57262701880795,-79.57127009168724,-79.57055849074615,-79.56945474363037,-79.56956485946392,-79.56960390761917,-79.57077435520547,-79.57303359027092,-79.57341449303682,-79.57371003863125,-79.57514894264729,-79.57662340847348,-79.5768904328673,-79.57714509161899,-79.57746857438053,-79.57775977023104,-79.57789831468824,-79.57981733170156,-79.58047855425606,-79.58171516904399,-79.58177085421308,-79.58189398631984,-79.58205544364087,-79.58221537179122,-79.58240469934263,-79.58259442145098,-79.58277677605521,-79.58295791189828,-79.5831414841079,-79.58329023244156,-79.58333109809841,-79.5835244931226,-79.58372932934418,-79.58389062941517,-79.58405837769398,-79.58407901155603,-79.5849559475474,-79.58512615336747,-79.58528254449169,-79.58537939025891,-79.58558010801374,-79.58570355828304,-79.58582159208217,-79.58591786122392,-79.58598766178562,-79.58601717337277,-79.58599334690707,-79.58592548654845,-79.58582634599496,-79.58571616218089,-79.58564903933627,-79.58555930170132,-79.5855354487143,-79.58539450969607,-79.58532677802506,-79.58526435303254,-79.58519067707803,-79.58513120945491,-79.58509901408428,-79.58506580347428,-79.58504514947838,-79.58505669348436,-79.58507783302727,-79.58510234956883,-79.58513545217976,-79.585170091167,-79.58520002410356,-79.58523558986857,-79.58528219564406,-79.58533645697169,-79.58540779314174,-79.58549212125132,-79.58560353348915,-79.5856912458475,-79.58574818105781,-79.58580186646009,-79.58582345732376,-79.58579598093526,-79.58574306756644,-79.58564917514664,-79.58558788495374,-79.58553015904943,-79.58549634463743,-79.58552303474896,-79.58558274326975,-79.58567811851012,-79.58578995257568,-79.58590412070753,-79.58597986457985,-79.58601308078109,-79.58603271079309,-79.58604672021211,-79.58604315350594,-79.58598215564713,-79.58592596402653,-79.58586446140352,-79.58579957394593,-79.58572927533741,-79.58567870687375,-79.58560687606501,-79.58552402344176,-79.58545424823953,-79.58539356147692,-79.58533327438012,-79.58529485919426,-79.58525295897455,-79.58525613419881,-79.58527504907056,-79.58527983197152,-79.5852934924622,-79.58533581957025,-79.58542179199125,-79.58548385506884,-79.58553620151839,-79.58560603077956,-79.5856806736862,-79.58568526924452],"lat":[43.63788340658488,43.63843486311833,43.63919964913595,43.63984831252636,43.63994571361732,43.64003438360395,43.6401096252389,43.64017569999722,43.64024180377436,43.64030266767812,43.6404249536303,43.64064238258405,43.64102354458541,43.64107286554292,43.64114048964209,43.64119846402517,43.64128792128819,43.64137298143447,43.6415196064018,43.64198164738287,43.64261062967698,43.64293393366568,43.64319887941157,43.64313415422172,43.64264285782679,43.64251538684874,43.64202938873298,43.64183885819374,43.6408496172472,43.6406842962774,43.64056221023716,43.64043498310235,43.64029758565412,43.64034038053703,43.64020608865508,43.64004332645717,43.63993323400184,43.63992160740889,43.639946377918,43.63990781772069,43.63962357903863,43.63930495216182,43.63885800971296,43.63842844809351,43.63816226689292,43.63787533933566,43.6375602799075,43.63496232333939,43.6339057727196,43.63389716927492,43.63354722563713,43.63292712181875,43.63277884069674,43.63266091055624,43.6320351159365,43.63137814851383,43.6312554438685,43.63113258465668,43.63095197765903,43.63076205587519,43.63066925747148,43.62928392565154,43.62881043593942,43.62794790008505,43.62795309212664,43.62795011886723,43.62793861881111,43.62792259849348,43.62794295110909,43.62798131256261,43.62803308554121,43.6280758409893,43.62813672063203,43.62817457299557,43.62818408165121,43.62823148921429,43.62828353928593,43.62834405172138,43.62839114061854,43.62840039836571,43.62866790935237,43.62871052604271,43.62877997787243,43.62881268400659,43.62891410147135,43.6290327468008,43.62915573602034,43.62927845584315,43.6294053491935,43.6295678426125,43.62972958614088,43.62987737170875,43.63004268444112,43.63021686256114,43.63033314963506,43.6304940774143,43.63055229604381,43.63071718111718,43.63082886937134,43.63094962539382,43.63108374529757,43.63120912893311,43.63135276437847,43.63148738511833,43.63163575450223,43.6317754303396,43.6319287281471,43.63207315564947,43.63221759940956,43.63235756111355,43.6325020556332,43.6326330266604,43.63275963326357,43.63289542674424,43.63302233938359,43.63313140851974,43.63323631169894,43.63332750834232,43.6334452404389,43.63353592592971,43.63363521641343,43.63375199404351,43.63385936477449,43.6339662283884,43.6340689943229,43.63418539753497,43.63430650768326,43.63441936449902,43.63451912673904,43.63461491926466,43.63470632428031,43.63482026336283,43.63493372696329,43.63506475865301,43.63519103117255,43.63531723415338,43.63543880865632,43.63555958263021,43.63567591497329,43.63578768051566,43.63591299733947,43.63603365608646,43.63614555681901,43.6362617854878,43.63635978345071,43.63646244426128,43.63657421963087,43.6366950920507,43.6368161451413,43.63695065825308,43.63707672718554,43.63719857990322,43.63722564545365,43.63727082512061,43.63738387547573,43.63748846388733,43.63758384423702,43.63769251745482,43.63779240488299,43.63787884866029,43.63788340658488]}]],[[{"lng":[-79.58027560655087,-79.58021038837738,-79.57928592790464,-79.57913385901082,-79.57892152914714,-79.57880343883924,-79.57866217420785,-79.57854902876009,-79.57847503722617,-79.57840626622179,-79.57831716745865,-79.5782278879843,-79.57814309647681,-79.57793591812,-79.57699564327174,-79.57681581437319,-79.57624629131334,-79.57528910017312,-79.57435124937462,-79.57410326292937,-79.57247604592077,-79.57150974658076,-79.57054284267475,-79.56885877211894,-79.56816714182692,-79.56791718574088,-79.56723975986856,-79.56716241963976,-79.56645395699016,-79.56580503037002,-79.56473997986599,-79.56406599890983,-79.56368396178426,-79.56300902825183,-79.56270198970346,-79.56233726809309,-79.56225247769565,-79.56230839497685,-79.56364495222425,-79.5657480455166,-79.56600544753614,-79.56722010644893,-79.56790874727795,-79.56890558571797,-79.5690784321545,-79.56929869954907,-79.5694660180758,-79.56956485946392,-79.56945474363037,-79.57055849074615,-79.57127009168724,-79.57262701880795,-79.57302983639725,-79.57354032379065,-79.57400102999644,-79.57427861483099,-79.57469339237397,-79.57489080967142,-79.575077354193,-79.57528788057333,-79.57544649789163,-79.5755046040842,-79.57590034039561,-79.5763455496401,-79.57652826526058,-79.5766244264466,-79.57703067898221,-79.57728716385708,-79.57864009864629,-79.57875961695939,-79.57946718237315,-79.57957592226214,-79.58027560655087],"lat":[43.64313415422172,43.64319887941157,43.64408949268802,43.64423865761676,43.64436789860862,43.64441144077529,43.64442318726336,43.64442178013969,43.64441185778565,43.64439299821348,43.64436038258976,43.64430075833589,43.6442589239898,43.64414373008305,43.64361917715961,43.64355392332493,43.64318422536932,43.64269752653573,43.64291097465487,43.64294388951797,43.64333326289622,43.64356424650055,43.64377730016596,43.64414790662354,43.64432378722578,43.6443746681703,43.64454631009489,43.64456550524698,43.64293399232308,43.64207596246869,43.63934800745352,43.63783195846764,43.6373690406374,43.63691999347209,43.63653201258977,43.63586243936157,43.6356328991325,43.6356200093316,43.6352991446112,43.63479434038926,43.63472105219219,43.63444362666615,43.63428562812413,43.63403705260396,43.63399870717424,43.63395645403363,43.63392704093759,43.6339057727196,43.63496232333939,43.6375602799075,43.63787533933566,43.63816226689292,43.63842844809351,43.63885800971296,43.63930495216182,43.63962357903863,43.63990781772069,43.639946377918,43.63992160740889,43.63993323400184,43.64004332645717,43.64020608865508,43.64034038053703,43.64029758565412,43.64043498310235,43.64056221023716,43.6406842962774,43.6408496172472,43.64183885819374,43.64202938873298,43.64251538684874,43.64264285782679,43.64313415422172]}]],[[{"lng":[-79.56225247769565,-79.56233726809309,-79.56270198970346,-79.56300902825183,-79.56368396178426,-79.56406599890983,-79.56473997986599,-79.56580503037002,-79.56645395699016,-79.56716241963976,-79.56623957150259,-79.56600186734548,-79.56585126938758,-79.56494521757938,-79.56400562394401,-79.56377497791361,-79.5636191694788,-79.56285071539016,-79.56260558803737,-79.5623834870299,-79.56183894279849,-79.56166844011705,-79.56162227394802,-79.5615769234127,-79.56153299868872,-79.56149010130713,-79.56144781943513,-79.56140624579261,-79.56136538769969,-79.56132545474135,-79.56131679883269,-79.56128644387786,-79.56124722660752,-79.56120964467721,-79.56117277828044,-79.56113662956113,-79.56110058102341,-79.56107044454828,-79.56104184115296,-79.56101395538759,-79.56098709475917,-79.56096063779981,-79.56093510070809,-79.5609119111853,-79.56091048562766,-79.56088648166362,-79.56086340277363,-79.56084103934165,-79.56082010367234,-79.56079947773448,-79.56077935762379,-79.5607601625731,-79.56072739848069,-79.56071899981255,-79.56070917564108,-79.56069863611792,-79.5606873812458,-79.5606751044393,-79.56066221140264,-79.5606486061083,-79.56063397365568,-79.5606188354929,-79.56060277760747,-79.56058518159433,-79.56056677329802,-79.56045888687463,-79.56049538063959,-79.56054607936564,-79.56225247769565],"lat":[43.6356328991325,43.63586243936157,43.63653201258977,43.63691999347209,43.6373690406374,43.63783195846764,43.63934800745352,43.64207596246869,43.64293399232308,43.64456550524698,43.64479482839283,43.64485486093491,43.64489348077609,43.64512975577638,43.64537460425005,43.64542571872693,43.64545976909225,43.64359081494445,43.64298900252345,43.64245499493916,43.64111106153392,43.6406902279984,43.64057703065454,43.64046393357628,43.64035535547652,43.6402422892349,43.64012914070136,43.64002059213936,43.63991205256732,43.63979902357028,43.63977190828999,43.63968591614358,43.63957289612438,43.63945989667646,43.63934690622204,43.63923383476772,43.63912085458236,43.63902145197867,43.6389220686671,43.63882260436093,43.63871874191994,43.63861938560818,43.63852004087077,43.63842513671855,43.6384206177143,43.63832129226689,43.63821747740614,43.63811367154791,43.63801429469601,43.63791051070486,43.63781123413308,43.63770746814972,43.63758543704768,43.63751331439792,43.63743667273992,43.63736002207173,43.63728336239316,43.63720668984309,43.63713442056504,43.63705773128022,43.63698553011315,43.63690882151871,43.63683210133851,43.63675986283995,43.63668311305182,43.63604251424787,43.63603397177131,43.6360256081859,43.6356328991325]}]],[[{"lng":[-79.5599102576369,-79.55964504728817,-79.55907101092984,-79.55874198599986,-79.55862184999772,-79.55832302998819,-79.55760995177606,-79.55692996926082,-79.5571730461071,-79.55663999887771,-79.55663571523024,-79.55669925548666,-79.55685914733213,-79.55685904513682,-79.55905690860803,-79.55917692451679,-79.55933979305942,-79.55941295545524,-79.5595756967455,-79.55968815538345,-79.55978891651422,-79.56002894261306,-79.56045888687463,-79.56056677329802,-79.56058518159433,-79.56060277760747,-79.5606188354929,-79.56063397365568,-79.5606486061083,-79.56066221140264,-79.5606751044393,-79.5606873812458,-79.56069863611792,-79.56070917564108,-79.56071899981255,-79.56072739848069,-79.5607601625731,-79.56077935762379,-79.56079947773448,-79.56082010367234,-79.56084103934165,-79.56086340277363,-79.56088648166362,-79.56091048562766,-79.5609119111853,-79.56093510070809,-79.56096063779981,-79.56098709475917,-79.56101395538759,-79.56104184115296,-79.56107044454828,-79.56110058102341,-79.56113662956113,-79.56117277828044,-79.56120964467721,-79.56124722660752,-79.56128644387786,-79.56131679883269,-79.56132545474135,-79.56136538769969,-79.56140624579261,-79.56144781943513,-79.56149010130713,-79.56153299868872,-79.5615769234127,-79.56162227394802,-79.56166844011705,-79.56183894279849,-79.5623834870299,-79.56260558803737,-79.56285071539016,-79.5636191694788,-79.56128170296464,-79.5599102576369],"lat":[43.64621392340366,43.64557197256757,43.64499697449928,43.64442902435665,43.64404446996575,43.64316299616772,43.64150004280768,43.63971597402328,43.63807696831196,43.6369680195739,43.63693771838479,43.63692249703237,43.63688364645325,43.63688364516287,43.63634901367403,43.63631892928149,43.63628371359259,43.63626789184403,43.6362294334537,43.63621185623738,43.63619611197578,43.63613612149328,43.63604251424787,43.63668311305182,43.63675986283995,43.63683210133851,43.63690882151871,43.63698553011315,43.63705773128022,43.63713442056504,43.63720668984309,43.63728336239316,43.63736002207173,43.63743667273992,43.63751331439792,43.63758543704768,43.63770746814972,43.63781123413308,43.63791051070486,43.63801429469601,43.63811367154791,43.63821747740614,43.63832129226689,43.6384206177143,43.63842513671855,43.63852004087077,43.63861938560818,43.63871874191994,43.63882260436093,43.6389220686671,43.63902145197867,43.63912085458236,43.63923383476772,43.63934690622204,43.63945989667646,43.63957289612438,43.63968591614358,43.63977190828999,43.63979902357028,43.63991205256732,43.64002059213936,43.64012914070136,43.6402422892349,43.64035535547652,43.64046393357628,43.64057703065454,43.6406902279984,43.64111106153392,43.64245499493916,43.64298900252345,43.64359081494445,43.64545976909225,43.64588965589673,43.64621392340366]}]],[[{"lng":[-79.5599102576369,-79.54897210156464,-79.54815616314175,-79.54755897925142,-79.54753324484965,-79.54752732040758,-79.54685584892511,-79.54669963152983,-79.54645363435455,-79.54592911259637,-79.5450369302161,-79.54639615437685,-79.54776281236306,-79.55146941804556,-79.55245747279665,-79.55663571523024,-79.55663999887771,-79.5571730461071,-79.55692996926082,-79.55760995177606,-79.55832302998819,-79.55862184999772,-79.55874198599986,-79.55907101092984,-79.55964504728817,-79.5599102576369],"lat":[43.64621392340366,43.64879988170856,43.64683111434886,43.64546853133911,43.64537818333198,43.64530159021899,43.64380284760157,43.6434677836298,43.64287483776771,43.64164360998991,43.63966493618043,43.63934006120969,43.63902426659308,43.63817091284746,43.63794028405182,43.63693771838479,43.6369680195739,43.63807696831196,43.63971597402328,43.64150004280768,43.64316299616772,43.64404446996575,43.64442902435665,43.64499697449928,43.64557197256757,43.64621392340366]}]],[[{"lng":[-79.3761519531329,-79.37429554794018,-79.37429457933328,-79.37423033714872,-79.37305247992553,-79.37469640660493,-79.37485582481258,-79.3751235303573,-79.37514152363384,-79.37515882064517,-79.37521800588456,-79.37523350832046,-79.37526207960774,-79.37528934147819,-79.37531637991701,-79.37534282430894,-79.37536751231418,-79.37539230256333,-79.37541568117575,-79.37543812097499,-79.37546491473316,-79.37548076900167,-79.37550085837657,-79.37552023240438,-79.37554207828863,-79.37572210146324,-79.37575618036054,-79.37579776249191,-79.37584348019801,-79.3761519531329],"lat":[43.64499061691141,43.64534465187621,43.64534265764439,43.64509382701684,43.64247501898473,43.64192209744876,43.64219001544853,43.64266201932683,43.64268478031012,43.64270303035894,43.64274888212491,43.64278060974672,43.64283952935432,43.64289392930695,43.64295282710876,43.64300721542652,43.6430660798022,43.64312494562544,43.64317929034406,43.64323812274456,43.64330610911524,43.64335134479867,43.64341014377521,43.64346893257966,43.64352325548325,43.64397150636436,43.64405751030485,43.64415721398393,43.64427038949262,43.64499061691141]}]],[[{"lng":[-79.56416964661734,-79.56149971212326,-79.56064065117411,-79.55761553339359,-79.55698040282049,-79.55573640352419,-79.55560347718936,-79.55523618999526,-79.55508987161738,-79.55502177017516,-79.55491393820398,-79.55474020118686,-79.55447226187087,-79.55396554617469,-79.55324373047918,-79.55255865577089,-79.5521126420962,-79.55200558606548,-79.5519036296961,-79.55180149493391,-79.55171184164551,-79.55162496650614,-79.55156212052468,-79.5515091846263,-79.55148881103389,-79.55131933788577,-79.55089843262469,-79.55076526582539,-79.54995592009668,-79.54897210156464,-79.5599102576369,-79.55994998285122,-79.56021694975168,-79.55996901382085,-79.55992703029176,-79.56015300030327,-79.56046098479059,-79.56070399160923,-79.56150700123435,-79.56170597533826,-79.56207004465671,-79.56257503698862,-79.56320398693032,-79.56416964661734],"lat":[43.6549601129401,43.65559854504676,43.65579936836144,43.65651308428173,43.65665810232264,43.65694854967938,43.65696937503694,43.65702774723897,43.65688636524732,43.65681798864624,43.65672201339216,43.65659828842522,43.65643286240347,43.6561653020028,43.65575098336135,43.6553911379576,43.65515143433635,43.65509156450427,43.65504076126939,43.65497186149536,43.65489420773313,43.65480308598143,43.65470776765277,43.65461707592898,43.65456280521339,43.65416447546676,43.65320023742716,43.65286997277359,43.65100932810256,43.64879988170856,43.64621392340366,43.64631002621687,43.64698998725204,43.6478669985642,43.64818199301708,43.64870804175088,43.64926798066721,43.64944496160372,43.64970803006612,43.64985798849941,43.65055798116465,43.65123597953618,43.65269799561626,43.6549601129401]}]],[[{"lng":[-79.56416964661734,-79.56320398693032,-79.56257503698862,-79.56207004465671,-79.56170597533826,-79.56150700123435,-79.56070399160923,-79.56046098479059,-79.56015300030327,-79.55992703029176,-79.55996901382085,-79.56021694975168,-79.55994998285122,-79.5599102576369,-79.56128170296464,-79.5636191694788,-79.56367646589652,-79.56377841354696,-79.5638286909598,-79.56387907289073,-79.56393006825149,-79.56398351684906,-79.56403625299376,-79.56409041741441,-79.56409572243217,-79.56414621740002,-79.56420130208264,-79.56425792296558,-79.56430923610606,-79.564360656586,-79.56441043683637,-79.56446011005221,-79.56450835526164,-79.56455650054366,-79.56460310781014,-79.56464982238882,-79.56469490168759,-79.56473916126563,-79.56478270267395,-79.5648253242896,-79.56486733769063,-79.56490842420546,-79.5649487974876,-79.5649885625366,-79.56502750998301,-79.56502811834361,-79.56506553268217,-79.56512397178237,-79.56518149330975,-79.56523840176291,-79.56528643729769,-79.56529449270656,-79.5653498711153,-79.56540361150826,-79.56545765875507,-79.5655097619495,-79.56556196970118,-79.56561346213218,-79.56566342150035,-79.56571235186136,-79.56576149386925,-79.56580828505307,-79.5658668001296,-79.56592602828361,-79.56598628863111,-79.56604715556581,-79.56610884267909,-79.56617124561599,-79.56623436224045,-79.56629850833939,-79.56634266312498,-79.5663632632149,-79.56642883884558,-79.56649533207458,-79.56656254598192,-79.56663047360151,-79.56669983209835,-79.56677378394342,-79.56683946844441,-79.56691056834575,-79.56712540597626,-79.56733953095068,-79.56755191626121,-79.56759119185304,-79.56776369586235,-79.56780154556753,-79.56781327305011,-79.56784816307426,-79.56797405189148,-79.5680640451541,-79.56815240584561,-79.5682408713586,-79.56832791085054,-79.56833239943394,-79.56833974567196,-79.56845458458321,-79.56851960018493,-79.56858307791057,-79.56864656052552,-79.56870850779381,-79.56877035050964,-79.56883076224273,-79.56889107195283,-79.56894995318386,-79.56900862763941,-79.56906668892142,-79.56912311473707,-79.56917953860668,-79.56920401230458,-79.56921810710232,-79.56923524919969,-79.56928943116651,-79.56934339960229,-79.56939593949326,-79.56944880223784,-79.56950002484332,-79.5695513544466,-79.56959448752902,-79.56829502944422,-79.56577303138776,-79.5651009850442,-79.56455504579812,-79.56400203024641,-79.56356098094999,-79.56480802293031,-79.56419395645321,-79.56416964661734],"lat":[43.6549601129401,43.65269799561626,43.65123597953618,43.65055798116465,43.64985798849941,43.64970803006612,43.64944496160372,43.64926798066721,43.64870804175088,43.64818199301708,43.6478669985642,43.64698998725204,43.64631002621687,43.64621392340366,43.64588965589673,43.64545976909225,43.64559101960152,43.64573633433464,43.64583607923368,43.64593573540148,43.64603539925332,43.64613509390455,43.64623036854979,43.64633007214182,43.64633914088981,43.64642979625442,43.64652951135236,43.64662483469208,43.6467155002589,43.6468016661009,43.64689231236517,43.64698745831099,43.6470781752706,43.64716880093268,43.64726390830592,43.64735451595801,43.64744510305358,43.64753576985726,43.6476308386516,43.64772589587812,43.64781644440198,43.64791157234002,43.64800660128564,43.64809712153084,43.64818763149575,43.64819214018188,43.64828263088388,43.64840048183757,43.64851823121922,43.64863597287368,43.64873559892535,43.6487537042356,43.6488670155973,43.64898471739632,43.64910242301959,43.64922019423614,43.64933787671823,43.64945555019701,43.64957320441093,43.64969543675544,43.64981308066496,43.64993069505195,43.65007555283094,43.6502248305498,43.6503651190927,43.65051000627874,43.65065481369822,43.65079963006161,43.65094454536185,43.6510848824713,43.6511844593622,43.65122972822402,43.65137017320702,43.65151504068326,43.65165541611094,43.65179589046808,43.65194070372982,43.65209016561383,43.65222611066372,43.65236653460081,43.65278341438688,43.6532002848108,43.65362163404895,43.65369864404737,43.65403847423246,43.65411096524161,43.65413811846307,43.65420607132398,43.65445079511798,43.65461396023598,43.65477269380375,43.65493133859188,43.65508546440167,43.6550945227145,43.65510811786669,43.65526718286549,43.65536702014461,43.65547133916738,43.65557115717176,43.65567104593282,43.65577534437147,43.65587962485607,43.6559839040297,43.65608375426215,43.65618801289547,43.65629226382011,43.65639649423824,43.65650081462345,43.65654676157855,43.65657331413666,43.65660503602912,43.6567047372284,43.65681343781164,43.65691770946513,43.65700839198763,43.65710355499775,43.65719421828106,43.65727262614411,43.65758799207367,43.6581389861115,43.6580350385416,43.65773300478238,43.65710599434666,43.6560380260753,43.65577201727552,43.65501704146044,43.6549601129401]}]],[[{"lng":[-79.5696160073097,-79.56920401230458,-79.56917953860668,-79.56912311473707,-79.56906668892142,-79.56900862763941,-79.56894995318386,-79.56889107195283,-79.56883076224273,-79.56877035050964,-79.56870850779381,-79.56864656052552,-79.56858307791057,-79.56851960018493,-79.56845458458321,-79.56833974567196,-79.56833239943394,-79.56832791085054,-79.5682408713586,-79.56815240584561,-79.5680640451541,-79.56797405189148,-79.56784816307426,-79.56781327305011,-79.56780154556753,-79.56776369586235,-79.56759119185304,-79.56755191626121,-79.56733953095068,-79.56712540597626,-79.56691056834575,-79.56683946844441,-79.56677378394342,-79.56669983209835,-79.56663047360151,-79.56656254598192,-79.56649533207458,-79.56642883884558,-79.5663632632149,-79.56634266312498,-79.56629850833939,-79.56623436224045,-79.56617124561599,-79.56610884267909,-79.56604715556581,-79.56598628863111,-79.56592602828361,-79.5658668001296,-79.56580828505307,-79.56576149386925,-79.56571235186136,-79.56566342150035,-79.56561346213218,-79.56556196970118,-79.5655097619495,-79.56545765875507,-79.56540361150826,-79.5653498711153,-79.56529449270656,-79.56528643729769,-79.56523840176291,-79.56518149330975,-79.56512397178237,-79.56506553268217,-79.56502811834361,-79.56502750998301,-79.5649885625366,-79.5649487974876,-79.56490842420546,-79.56486733769063,-79.5648253242896,-79.56478270267395,-79.56473916126563,-79.56469490168759,-79.56464982238882,-79.56460310781014,-79.56455650054366,-79.56450835526164,-79.56446011005221,-79.56441043683637,-79.564360656586,-79.56430923610606,-79.56425792296558,-79.56420130208264,-79.56414621740002,-79.56409572243217,-79.56409041741441,-79.56403625299376,-79.56398351684906,-79.56393006825149,-79.56387907289073,-79.5638286909598,-79.56377841354696,-79.56367646589652,-79.5636191694788,-79.56377497791361,-79.56400562394401,-79.56494521757938,-79.56585126938758,-79.56600186734548,-79.56623957150259,-79.56716241963976,-79.56734837463127,-79.56763201109271,-79.56782200886549,-79.56861702021827,-79.56917697760088,-79.56955798110631,-79.56984799360639,-79.57029698580051,-79.57213003254741,-79.57250614751537,-79.57289003072732,-79.57299696075732,-79.57282097836772,-79.57236502189402,-79.5710269805101,-79.57020198679325,-79.5696160073097],"lat":[43.65646099752285,43.65654676157855,43.65650081462345,43.65639649423824,43.65629226382011,43.65618801289547,43.65608375426215,43.6559839040297,43.65587962485607,43.65577534437147,43.65567104593282,43.65557115717176,43.65547133916738,43.65536702014461,43.65526718286549,43.65510811786669,43.6550945227145,43.65508546440167,43.65493133859188,43.65477269380375,43.65461396023598,43.65445079511798,43.65420607132398,43.65413811846307,43.65411096524161,43.65403847423246,43.65369864404737,43.65362163404895,43.6532002848108,43.65278341438688,43.65236653460081,43.65222611066372,43.65209016561383,43.65194070372982,43.65179589046808,43.65165541611094,43.65151504068326,43.65137017320702,43.65122972822402,43.6511844593622,43.6510848824713,43.65094454536185,43.65079963006161,43.65065481369822,43.65051000627874,43.6503651190927,43.6502248305498,43.65007555283094,43.64993069505195,43.64981308066496,43.64969543675544,43.64957320441093,43.64945555019701,43.64933787671823,43.64922019423614,43.64910242301959,43.64898471739632,43.6488670155973,43.6487537042356,43.64873559892535,43.64863597287368,43.64851823121922,43.64840048183757,43.64828263088388,43.64819214018188,43.64818763149575,43.64809712153084,43.64800660128564,43.64791157234002,43.64781644440198,43.64772589587812,43.6476308386516,43.64753576985726,43.64744510305358,43.64735451595801,43.64726390830592,43.64716880093268,43.6470781752706,43.64698745831099,43.64689231236517,43.6468016661009,43.6467155002589,43.64662483469208,43.64652951135236,43.64642979625442,43.64633914088981,43.64633007214182,43.64623036854979,43.64613509390455,43.64603539925332,43.64593573540148,43.64583607923368,43.64573633433464,43.64559101960152,43.64545976909225,43.64542571872693,43.64537460425005,43.64512975577638,43.64489348077609,43.64485486093491,43.64479482839283,43.64456550524698,43.64500929938886,43.64570097498093,43.64745498436349,43.64945404695561,43.65066202781087,43.65145799089588,43.65176697120291,43.65196496334971,43.65240602589107,43.65302628712779,43.65405281832948,43.65425697053112,43.65469299478189,43.65503802155899,43.65602998368444,43.65633896978967,43.65646099752285]}]],[[{"lng":[-79.57949051728477,-79.57749565499192,-79.57523367712508,-79.57428217448,-79.57254121451766,-79.57250614751537,-79.57213003254741,-79.57029698580051,-79.56984799360639,-79.56955798110631,-79.56917697760088,-79.56861702021827,-79.56782200886549,-79.56763201109271,-79.56734837463127,-79.56716241963976,-79.56723975986856,-79.56791718574088,-79.56816714182692,-79.56885877211894,-79.57054284267475,-79.57150974658076,-79.57247604592077,-79.57281703634138,-79.57290378554227,-79.57298411540853,-79.57306263173923,-79.57313544185625,-79.57317536240356,-79.57321977158968,-79.57326681032728,-79.57330862143225,-79.57335635222506,-79.57341216634595,-79.57344963009125,-79.57353722967366,-79.57361075236153,-79.57368285711128,-79.57375231734763,-79.57383425162053,-79.57394184508502,-79.57404188360685,-79.57411975102509,-79.57418301086196,-79.57441387071934,-79.57506690694322,-79.5752619816672,-79.57553900726843,-79.57642841433032,-79.57652057664183,-79.5766460316886,-79.57675737773684,-79.57683582391566,-79.57689781006393,-79.57694835260912,-79.57699030729397,-79.57704316869206,-79.57712110216507,-79.57727318241356,-79.57739399680543,-79.5774749484156,-79.57755746548061,-79.57761496020721,-79.57771116244051,-79.57790742606036,-79.57801575325313,-79.57809567483609,-79.57820066892792,-79.57883704387118,-79.57891996981027,-79.57921288920308,-79.57945003393705,-79.57949051728477],"lat":[43.65140255050166,43.65185944134693,43.65239857760573,43.65262076650814,43.65302222406303,43.65302628712779,43.65240602589107,43.65196496334971,43.65176697120291,43.65145799089588,43.65066202781087,43.64945404695561,43.64745498436349,43.64570097498093,43.64500929938886,43.64456550524698,43.64454631009489,43.6443746681703,43.64432378722578,43.64414790662354,43.64377730016596,43.64356424650055,43.64333326289622,43.64404877761744,43.64418048134977,43.64429401063722,43.64438051091527,43.6444489356945,43.64449444458244,43.64454901159336,43.64463070774116,43.64473025279769,43.64483887383182,43.64493859363738,43.6449885728121,43.64508877930314,43.64516171371768,43.64522112723616,43.64526700455519,43.64530853643497,43.64534588736408,43.64537414183188,43.64539311751242,43.64540290880733,43.64544179699142,43.64554455261736,43.64559649608714,43.64567637691395,43.64594860934966,43.64598576564589,43.64605484384153,43.64612824725642,43.64618773774807,43.64625152428684,43.64631066725773,43.6463697933032,43.64646497350555,43.64664148472379,43.6469990509209,43.64723019833791,43.64734823321518,43.64743027911862,43.64748950830936,43.64758522756235,43.64780381015775,43.64794469050983,43.64807180424048,43.64828465973203,43.64972417802616,43.64992784632216,43.65044919856011,43.65126341508713,43.65140255050166]}]],[[{"lng":[-79.59107290539899,-79.59085270686964,-79.59000325942128,-79.58100664311745,-79.57969989179217,-79.57945003393705,-79.57921288920308,-79.57891996981027,-79.57883704387118,-79.57820066892792,-79.57809567483609,-79.57801575325313,-79.57790742606036,-79.57771116244051,-79.57761496020721,-79.57755746548061,-79.5774749484156,-79.57739399680543,-79.57727318241356,-79.57712110216507,-79.57704316869206,-79.57699030729397,-79.57694835260912,-79.57689781006393,-79.57683582391566,-79.57675737773684,-79.5766460316886,-79.57652057664183,-79.57642841433032,-79.57553900726843,-79.5752619816672,-79.57506690694322,-79.57441387071934,-79.57418301086196,-79.57411975102509,-79.57404188360685,-79.57394184508502,-79.57383425162053,-79.57375231734763,-79.57368285711128,-79.57361075236153,-79.57353722967366,-79.57344963009125,-79.57341216634595,-79.57335635222506,-79.57330862143225,-79.57326681032728,-79.57321977158968,-79.57317536240356,-79.57313544185625,-79.57306263173923,-79.57298411540853,-79.57290378554227,-79.57281703634138,-79.57247604592077,-79.57410326292937,-79.57435124937462,-79.57528910017312,-79.57624629131334,-79.57681581437319,-79.57699564327174,-79.57793591812,-79.57814309647681,-79.5782278879843,-79.57831716745865,-79.57840626622179,-79.57847503722617,-79.57854902876009,-79.57866217420785,-79.57880343883924,-79.57892152914714,-79.57913385901082,-79.57928592790464,-79.58021038837738,-79.58047743453808,-79.58080054865916,-79.58142375765739,-79.58193455607666,-79.58208849370452,-79.58214007788229,-79.58218552908485,-79.58219997856379,-79.582220277811,-79.58222838969995,-79.58198790679967,-79.58260978878184,-79.58291169288481,-79.58308518006444,-79.58319908663566,-79.58331534360349,-79.58342557959845,-79.58353459737246,-79.58366549299865,-79.5843581103838,-79.58513143883577,-79.58568526924452,-79.58573521060273,-79.58580649262322,-79.58589727675519,-79.58600888300359,-79.58613357333473,-79.58627675474868,-79.58638886361243,-79.58646889879397,-79.5865730315907,-79.58666460954264,-79.58677426423975,-79.58686011113241,-79.58692714898299,-79.58694615580829,-79.58695923946298,-79.58700604205339,-79.58706113629944,-79.58714064522485,-79.58721331557707,-79.58728790859929,-79.58737017162539,-79.58746809694411,-79.5875823784434,-79.58772712785176,-79.58787647260999,-79.58800476068296,-79.58810033430849,-79.58820685165591,-79.58826073293312,-79.5882306978601,-79.58812411851305,-79.5880698507917,-79.58811454010566,-79.58816932562004,-79.58823137226419,-79.58834246436223,-79.58843017392583,-79.58854895255057,-79.58869166906908,-79.58886863165512,-79.58901156291745,-79.58916022752783,-79.58929630955498,-79.58939344680002,-79.58952501906377,-79.5896442246295,-79.5897851773349,-79.58992687271974,-79.5900588404705,-79.59020780278566,-79.59036748627641,-79.59056327329594,-79.59073646055822,-79.59092714105138,-79.59104049081775,-79.59107290539899],"lat":[43.64427744197021,43.6445125629611,43.64530984571304,43.65087495692715,43.65120449590634,43.65126341508713,43.65044919856011,43.64992784632216,43.64972417802616,43.64828465973203,43.64807180424048,43.64794469050983,43.64780381015775,43.64758522756235,43.64748950830936,43.64743027911862,43.64734823321518,43.64723019833791,43.6469990509209,43.64664148472379,43.64646497350555,43.6463697933032,43.64631066725773,43.64625152428684,43.64618773774807,43.64612824725642,43.64605484384153,43.64598576564589,43.64594860934966,43.64567637691395,43.64559649608714,43.64554455261736,43.64544179699142,43.64540290880733,43.64539311751242,43.64537414183188,43.64534588736408,43.64530853643497,43.64526700455519,43.64522112723616,43.64516171371768,43.64508877930314,43.6449885728121,43.64493859363738,43.64483887383182,43.64473025279769,43.64463070774116,43.64454901159336,43.64449444458244,43.6444489356945,43.64438051091527,43.64429401063722,43.64418048134977,43.64404877761744,43.64333326289622,43.64294388951797,43.64291097465487,43.64269752653573,43.64318422536932,43.64355392332493,43.64361917715961,43.64414373008305,43.6442589239898,43.64430075833589,43.64436038258976,43.64439299821348,43.64441185778565,43.64442178013969,43.64442318726336,43.64441144077529,43.64436789860862,43.64423865761676,43.64408949268802,43.64319887941157,43.64293393366568,43.64261062967698,43.64198164738287,43.6415196064018,43.64137298143447,43.64128792128819,43.64119846402517,43.64114048964209,43.64107286554292,43.64102354458541,43.64064238258405,43.6404249536303,43.64030266767812,43.64024180377436,43.64017569999722,43.6401096252389,43.64003438360395,43.63994571361732,43.63984831252636,43.63919964913595,43.63843486311833,43.63788340658488,43.63792903514008,43.63799752303753,43.63809316850446,43.63822507995973,43.63835706313869,43.63849395599948,43.63863937646502,43.63876189486664,43.63891180758014,43.63904347069767,43.63919336148333,43.63933404570987,43.63948340941733,43.63962767821307,43.63976296168161,43.63991207515861,43.64004328689303,43.64018839359527,43.64032432357488,43.64048728360539,43.6406414262917,43.64076416507284,43.64088260483963,43.64098350677064,43.64109337738351,43.64120748872104,43.64133469895381,43.6414486311409,43.6415663241204,43.64169198266902,43.64180328253388,43.64194214484947,43.64205072222557,43.64219093168026,43.64232231870837,43.64247672672966,43.64259933847845,43.64272242386582,43.6428187083716,43.64290641341586,43.64298919700932,43.64305854803375,43.64314133674372,43.64323255672932,43.64333770475441,43.64344270002514,43.64357956089762,43.64368033230618,43.6438079898584,43.64389534740108,43.64399643016001,43.64406636052467,43.64415401610931,43.64421938183163,43.64426452945352,43.64427744197021]}]],[[{"lng":[-79.60872409390845,-79.60633802510058,-79.60621896419357,-79.59845696818417,-79.59842960003142,-79.59839982614886,-79.59567036063528,-79.59371850521936,-79.5925453523666,-79.58876761360365,-79.5899508991521,-79.59018472568124,-79.59026625806236,-79.5903184700059,-79.59034339708074,-79.59038987710693,-79.59043492364266,-79.59047608692596,-79.59051479325892,-79.59053266537899,-79.59055114795089,-79.59056758862292,-79.59058371950636,-79.59059842197755,-79.59061312140767,-79.59062639242022,-79.59063955909092,-79.59065190988611,-79.5907542480267,-79.59086087820322,-79.59107648401493,-79.59114459442451,-79.59118403273304,-79.59119013017713,-79.59109331649651,-79.59089398078225,-79.59069536037265,-79.59049683869493,-79.59029832269987,-79.59010122950268,-79.58990342217213,-79.58970643550232,-79.58951005875328,-79.58931388603125,-79.58911822409323,-79.58892286951387,-79.58876320585506,-79.58872791920555,-79.58840676306701,-79.58808601475096,-79.58776567214902,-79.58732795749874,-79.58701333863186,-79.58599430034678,-79.58561315791323,-79.58537908941294,-79.58506502109992,-79.58445854674667,-79.58331552180672,-79.58235398552266,-79.58180750046726,-79.58077361702085,-79.58018053343582,-79.57949051728477,-79.57945003393705,-79.57969989179217,-79.58100664311745,-79.59000325942128,-79.59085270686964,-79.59107290539899,-79.591113727644,-79.59127505920638,-79.59145939898225,-79.59148637645353,-79.59162963046944,-79.59179936012009,-79.59201234244091,-79.59221131936845,-79.59244219479143,-79.59265828426474,-79.5928949320067,-79.59310859061492,-79.59329640094109,-79.59349802516863,-79.59366693957458,-79.59383534790487,-79.59401989640325,-79.59419565369289,-79.59437264849565,-79.59452633633187,-79.59470433841432,-79.5948601682169,-79.5950403316581,-79.59522916342277,-79.59540879858237,-79.5956182699516,-79.59582261940562,-79.59605476425308,-79.59626461455271,-79.59648958109872,-79.59671864353128,-79.59692744335294,-79.59713584457718,-79.59730918110587,-79.5975191155413,-79.59769940483514,-79.59789338510411,-79.59805528209164,-79.59822240388452,-79.59836712509806,-79.59848607308609,-79.59860409172836,-79.59870605224768,-79.59877704365414,-79.59882861567492,-79.59892433949213,-79.59906279391153,-79.59920808983053,-79.59941156155418,-79.59961176295161,-79.5997732470028,-79.59978524703489,-79.59981687305651,-79.6000255746477,-79.60026228168431,-79.60048427305217,-79.60072159367279,-79.60197846989168,-79.60226545469688,-79.60232481737788,-79.60247052439927,-79.6026777626073,-79.60292395415608,-79.60315034026448,-79.60333033734322,-79.60351740534344,-79.60372840452993,-79.6039613824917,-79.60418881044241,-79.60441623627641,-79.60462741253899,-79.60467412862805,-79.60477133642672,-79.60499423757852,-79.60518471132741,-79.60538569646002,-79.60559723114467,-79.60580502789107,-79.60605034348738,-79.60627604364831,-79.60646678375448,-79.60668859934978,-79.60694863249246,-79.60714885623621,-79.60734568779701,-79.60753341605054,-79.60767239010391,-79.60780164376418,-79.60789071876589,-79.60796252036666,-79.60802348230665,-79.60808086406117,-79.60815510610402,-79.60828023164636,-79.60840477553101,-79.60872409390845],"lat":[43.64644514671017,43.64861754130646,43.64872564503877,43.65577532915902,43.65580010937747,43.65582720063056,43.65827881215936,43.66007359370748,43.66114867478338,43.66461362139611,43.66681609051162,43.66713413768269,43.66722516412048,43.66727982056569,43.66731163527474,43.66737522306076,43.66743888316839,43.6675024053451,43.66757039824184,43.66760212590214,43.66763836212371,43.6676700721252,43.66770627933975,43.66773796790098,43.66777415745851,43.66780582836341,43.66783758800064,43.66786924755459,43.66804154884019,43.66821840398458,43.66858123496709,43.66867776660177,43.66874513813869,43.66875556568123,43.66878858011243,43.66885813917589,43.66892770671063,43.66900177615125,43.66906684324574,43.66914101963635,43.6692150968537,43.66928468281915,43.66935877700541,43.66943287337092,43.66950697570367,43.66957667048094,43.66964221458407,43.6696552819627,43.66976834267245,43.66988140751393,43.66999456648524,43.66891530016856,43.66745946694284,43.66601058311872,43.66488528873322,43.66439168865846,43.66369896197443,43.66235437348087,43.65990010693908,43.65777233449736,43.65659501345336,43.65431320338917,43.65306777963134,43.65140255050166,43.65126341508713,43.65120449590634,43.65087495692715,43.64530984571304,43.6445125629611,43.64427744197021,43.64429369887706,43.64438561853098,43.64445549628562,43.64446483085674,43.64453411198055,43.64459921807876,43.64463785044647,43.64468081096221,43.64471066081625,43.64468631550122,43.64463971761267,43.64457933309139,43.64450062582258,43.64439949295048,43.64429804726515,43.64418759299402,43.64409084018232,43.64399838997195,43.64388353945989,43.64377290312013,43.64368056974556,43.64357887137545,43.64345955771912,43.64337185769809,43.64327954328888,43.64320109837458,43.64314500540045,43.64309384456806,43.64306491472131,43.64305417433403,43.64302998062796,43.64305054826819,43.64305310641519,43.64306873701427,43.6430713132788,43.64308252754716,43.64310291159654,43.64310489756343,43.64308444210712,43.64309521908517,43.64313718722617,43.64319714806767,43.64327500614858,43.6433568954995,43.6434385467374,43.64352524025117,43.64359895429145,43.64368184414375,43.64373384932576,43.64378581407561,43.64382326086017,43.64382601848502,43.64383333755553,43.64385389833215,43.64387480170938,43.6438910233508,43.64391193328989,43.64411644840328,43.64437651849308,43.64442225498597,43.64451855885492,43.64461120378328,43.64468623025263,43.64471150254368,43.64470470014231,43.64465747436465,43.64457903325815,43.64449626914615,43.64447654156125,43.64446131455197,43.64444138765116,43.64443286585173,43.6444205493827,43.64446828075109,43.64457421968187,43.64472520707724,43.64482681122622,43.64482484513757,43.64480533204706,43.64476307396421,43.64472939129674,43.64468708503479,43.64466774942882,43.64468819386538,43.64475810821779,43.64484591554152,43.64493312864275,43.64504281846455,43.64518343612457,43.64532384333281,43.64547321060719,43.64563594738544,43.6458123926147,43.64603455795344,43.64618010813506,43.64644514671017]}]],[[{"lng":[-79.57346757167562,-79.57359367694364,-79.57370505704796,-79.57368351537474,-79.57365901693126,-79.57365452627624,-79.57363442060235,-79.57361155834799,-79.57358870046623,-79.57356737647665,-79.57354594592245,-79.57352626024176,-79.57350647020387,-79.57348820970203,-79.57347077162696,-79.57345333357968,-79.57343732280087,-79.57342141865223,-79.57340715062925,-79.57339359842214,-79.57338055529577,-79.57336863922649,-79.57335672753075,-79.57334727435097,-79.57333843036649,-79.57333040444189,-79.57332330318745,-79.57331609533254,-79.5733104278098,-79.57330557184851,-79.5733015382942,-79.57329811391843,-79.5732948984159,-79.57329321674131,-79.57329214423881,-79.57329189413279,-79.57329245983662,-79.57329384579985,-79.57329488141822,-79.57329584092763,-79.57329947211959,-79.57329747060699,-79.5732978209422,-79.57329949622209,-79.57330117149903,-79.57330387580122,-79.57330718713337,-79.57331213887699,-79.57331698835863,-79.5733226515066,-79.5733299550578,-79.57333786776486,-79.57334578046057,-79.57335543580433,-79.5733648822751,-79.57337597126653,-79.57348163244147,-79.57349172261073,-79.57350273302195,-79.57351527284511,-79.57352771685544,-79.57354179039288,-79.57355668624426,-79.57357229781798,-79.5736215121848,-79.57364791618795,-79.57367133750246,-79.57372290277907,-79.57372721888983,-79.57377497241686,-79.57378987199708,-79.57380334217457,-79.57381681018411,-79.57382781986219,-79.57383965181594,-79.57384933650727,-79.5738530433302,-79.57385871224692,-79.57386737008918,-79.57387531647524,-79.57388193360759,-79.57388742168182,-79.57388399260677,-79.57387985210515,-79.57387560718581,-79.57386982642018,-79.5738631254311,-79.57385662894782,-79.57384849650904,-79.57383975490521,-79.57382987997251,-79.57382031610678,-79.57380901617971,-79.57380830475839,-79.5737969984073,-79.57378585820541,-79.57378426492249,-79.5737706112326,-79.57375634408136,-79.57374125897618,-79.57372535591911,-79.57370873715904,-79.573691300451,-79.57367304792834,-79.57365407325553,-79.5736342849609,-79.57361296300095,-79.57355031354096,-79.57348673980911,-79.57342184582868,-79.57335602759082,-79.57328949392581,-79.57322235356759,-79.57315418022115,-79.57308437561383,-79.57303171994225,-79.57298069822455,-79.5729297789233,-79.5728794732574,-79.57282998784348,-79.57278213636059,-79.572734285039,-79.57259615970325,-79.57232737613381,-79.57215606480709,-79.57205552809188,-79.57200852753469,-79.57185646722792,-79.57168164737247,-79.57152296157048,-79.57150278096863,-79.5714724034293,-79.57143122008596,-79.57138983456221,-79.57134772691934,-79.57130409036057,-79.57126045610718,-79.57121507287661,-79.57116990327629,-79.57112391146168,-79.57107628617389,-79.57102876116809,-79.57097960507576,-79.57093055141262,-79.57086977460567,-79.57080859606539,-79.57074721098547,-79.57068449956178,-79.57062147722959,-79.57055702607651,-79.57049257311701,-79.57042679599731,-79.57035999233786,-79.57031817689773,-79.5702923711556,-79.57022403226418,-79.57018517743427,-79.57014489603634,-79.57010379233959,-79.5700618709122,-79.57001913389421,-79.56999902929508,-79.5699763948715,-79.56993212250602,-79.56988692560857,-79.56987305423405,-79.56984122441227,-79.56979449431748,-79.56974787118621,-79.56969981696739,-79.56965083821922,-79.56960115074408,-79.56959448752902,-79.5695513544466,-79.56950002484332,-79.56944880223784,-79.56939593949326,-79.56934339960229,-79.56928943116651,-79.56923524919969,-79.56921810710232,-79.56920401230458,-79.5696160073097,-79.57020198679325,-79.5710269805101,-79.57236502189402,-79.57282097836772,-79.57299696075732,-79.57289003072732,-79.57250614751537,-79.57254121451766,-79.57428217448,-79.57523367712508,-79.57749565499192,-79.57949051728477,-79.58018053343582,-79.58077361702085,-79.58180750046726,-79.58235398552266,-79.58331552180672,-79.58445854674667,-79.58506502109992,-79.58537908941294,-79.58561315791323,-79.58599430034678,-79.58701333863186,-79.58732795749874,-79.58776567214902,-79.58744584173692,-79.58732157122495,-79.58712600681953,-79.58707844909289,-79.58680719778388,-79.58648889425615,-79.58617069175585,-79.58585249144741,-79.58553479223426,-79.58521801761718,-79.58490124177936,-79.58480662838339,-79.58471191262481,-79.58461801128458,-79.58452421699839,-79.58443113346786,-79.58438019293536,-79.58433794756206,-79.58424486732984,-79.58415321501478,-79.58406166483611,-79.58397072450965,-79.58387978761846,-79.58378895286475,-79.58369883378047,-79.58361177653643,-79.58352012338212,-79.58343153417333,-79.58334366062526,-79.58325578694993,-79.58316791314732,-79.5830815730542,-79.58299533509178,-79.58290888885217,-79.58283983693326,-79.58282408221712,-79.58273927334083,-79.58265456871571,-79.58257036791683,-79.58248719687801,-79.58247215785434,-79.58240463557811,-79.58232197188933,-79.58224012242918,-79.58216472238691,-79.58215837877917,-79.58207745305164,-79.58199795666982,-79.58191753453657,-79.58183814226268,-79.58176007547519,-79.58168139874104,-79.58160404374522,-79.58152700286398,-79.58145046352327,-79.58137873977122,-79.58130701966553,-79.58123600772346,-79.58116581749051,-79.5810956233661,-79.5810262509642,-79.58095759425595,-79.5808896456553,-79.58082170240104,-79.58075539348526,-79.58068837245321,-79.58062267685743,-79.58055707958454,-79.58049291762967,-79.58042814200915,-79.58036490015503,-79.5803023718786,-79.5802397395107,-79.58017792509429,-79.58011692863165,-79.58005675398263,-79.57999718892037,-79.57993783212861,-79.57987908279445,-79.57982094515124,-79.57976373160405,-79.57970743828427,-79.57965114874435,-79.57959618069782,-79.57954121643363,-79.57948706411982,-79.57943291772149,-79.57936646869054,-79.57927517404769,-79.57923710175314,-79.57919831358863,-79.57916095310216,-79.5791242022013,-79.57908847959457,-79.57905275906919,-79.57901775039096,-79.57898335523214,-79.57895069846653,-79.57891712130609,-79.57888599837109,-79.57885416351476,-79.5788236540662,-79.57879395871353,-79.57876426331615,-79.57873538780191,-79.57870794207523,-79.57868070479415,-79.57865489914178,-79.57862980532019,-79.57859141343334,-79.5785532299808,-79.57851616340214,-79.57848022562025,-79.57844407714767,-79.57840957091771,-79.5783757764809,-79.57834198597276,-79.57830972535618,-79.57827757093872,-79.57824684415029,-79.57821622356865,-79.578186922226,-79.57815855132701,-79.57813027863766,-79.5781025172132,-79.57807638794135,-79.57805036890652,-79.57802442398862,-79.57799493993912,-79.57797394161645,-79.57795519117674,-79.57771190468286,-79.57759901487226,-79.57758410444434,-79.57754355392477,-79.57750218536043,-79.57745999875358,-79.57741719863546,-79.57741025277474,-79.5773747995229,-79.57733036544943,-79.57728582706488,-79.57724046657333,-79.57720583572687,-79.57719378080164,-79.57714678016036,-79.57709917417533,-79.57705145783657,-79.57700221170577,-79.57695225193844,-79.5769021878632,-79.57685069218294,-79.57679909432468,-79.57674688297243,-79.57577705784182,-79.57545466844327,-79.57403388363726,-79.57399812321611,-79.5739633897329,-79.5739294743607,-79.57389688839096,-79.57386368890911,-79.57383192108568,-79.57380107362594,-79.5737701239448,-79.57374071030665,-79.57371140116125,-79.57370915148071,-79.57368290578711,-79.57365584206011,-79.57362806255928,-79.57361754733485,-79.57360263936349,-79.57357639813856,-79.57356608745914,-79.57355250887824,-79.57352861530552,-79.57350564642616,-79.57348318886532,-79.57346757167562],"lat":[43.67313047354025,43.67310954235651,43.67308842726649,43.67305215018487,43.67300683413701,43.67299777603164,43.67295701583155,43.6729117201933,43.67286192357426,43.67281214609489,43.67276686831167,43.67271702125143,43.67266726290553,43.67262202467772,43.67257229568147,43.67252256668255,43.67247735652677,43.67242764666864,43.67237795723079,43.67232827672602,43.67228301358423,43.6722333534991,43.67217919243602,43.6721205610298,43.67206643825791,43.67201232569684,43.67195372364643,43.67189962129547,43.67184094712017,43.6717868741269,43.67172831036807,43.67167425524448,43.67161570169639,43.67155716729509,43.67150314152926,43.67144462499775,43.67139052966071,43.67133203355187,43.67131854338513,43.67127804607852,43.67122407902826,43.67117904372574,43.67113403778189,43.67109354940925,43.67105306103628,43.6710079944562,43.67096752650593,43.6709225780011,43.67087762821916,43.67083718962574,43.67079227047711,43.67075185996379,43.6707114494495,43.67066655965554,43.67062616828524,43.67058120636413,43.67001980618315,43.66994791560391,43.66987603650896,43.66980867753412,43.66973672630777,43.66966938647119,43.66959755586255,43.66952573418398,43.66928320260986,43.66915300217969,43.6690407686497,43.66879826635099,43.66877581504673,43.66856036136717,43.66848402972711,43.66840759022602,43.66833124071599,43.66825936156056,43.66818299163003,43.66810659490272,43.66807963495395,43.66803460533175,43.66796269682166,43.66788627839811,43.66780984338631,43.66773789531891,43.66768834115268,43.66763427707424,43.66758462270595,43.66753503919306,43.66748544419659,43.66743585175125,43.66738623889093,43.66733211739349,43.66728698382181,43.66723735309603,43.66718761068434,43.66718310077193,43.66713794933391,43.66709442034038,43.66708827905043,43.66703859728176,43.66698890785553,43.66693920821952,43.66688949837351,43.66683977959313,43.66679005060209,43.66674022140596,43.66669497424974,43.66664521590069,43.66659543840687,43.66645962543545,43.66632830192587,43.66618795982801,43.66605210718864,43.66591624557773,43.66577578529509,43.66564440416607,43.66550850159151,43.6654042303188,43.6653000694426,43.6651959098209,43.66509175783733,43.66498752605213,43.66488340466878,43.66477928326587,43.66445798465454,43.6639018100532,43.6635223930156,43.66328960358145,43.66317126920134,43.66283413189721,43.66240452868319,43.66194587008148,43.66188260330982,43.66179220285017,43.6616701600441,43.66154802467298,43.66143047131744,43.66130839779304,43.6611862342578,43.66107314093775,43.66095104819038,43.66083344618261,43.66071573371077,43.66059811249166,43.66047596981709,43.66035382839989,43.66019994307082,43.66003714062356,43.65987874657576,43.65971592488476,43.6595576003071,43.65939916779967,43.6592408252504,43.6590779650577,43.65891959302287,43.65882004705934,43.65876121071854,43.65860723041184,43.65851222247321,43.65841269563175,43.65831765952768,43.65822261317651,43.6581274665833,43.65808265472933,43.65803240996851,43.65793733415031,43.65784674778626,43.65781506692413,43.65775156403054,43.65766095844203,43.65756585313032,43.65747072988876,43.65738009609679,43.6572848623501,43.65727262614411,43.65719421828106,43.65710355499775,43.65700839198763,43.65691770946513,43.65681343781164,43.6567047372284,43.65660503602912,43.65657331413666,43.65654676157855,43.65646099752285,43.65633896978967,43.65602998368444,43.65503802155899,43.65469299478189,43.65425697053112,43.65405281832948,43.65302628712779,43.65302222406303,43.65262076650814,43.65239857760573,43.65185944134693,43.65140255050166,43.65306777963134,43.65431320338917,43.65659501345336,43.65777233449736,43.65990010693908,43.66235437348087,43.66369896197443,43.66439168865846,43.66488528873322,43.66601058311872,43.66745946694284,43.66891530016856,43.66999456648524,43.67010764086437,43.6701511152111,43.67022521532251,43.67023813048986,43.67033388951999,43.67045148112278,43.67056907308461,43.67068216315017,43.67080435061443,43.67092195758303,43.67103956365431,43.6710744003931,43.67110923578559,43.67114858221799,43.67117901785386,43.67121837426058,43.67124024846699,43.67125772932295,43.67129258458528,43.67133195849362,43.67137133359342,43.67141521720801,43.6714545997598,43.67149398350445,43.67153337604529,43.67157739751341,43.67161677090052,43.6716606832227,43.6717046043466,43.67174852540087,43.67179244638548,43.67183638631661,43.67188032744824,43.67192876696388,43.67196391900615,43.67197272571265,43.67201677439085,43.67206073427785,43.67211370241388,43.67215768119564,43.67216649672988,43.67220616851117,43.6722546544964,43.67230765155357,43.67235172649844,43.67235614883248,43.67240465619994,43.67245327126442,43.67251078682873,43.67255931305196,43.67261235669928,43.67266089169181,43.67271844509597,43.67276700028303,43.67282465372585,43.67287327480813,43.67291