/*
 *
 *    Copyright (c) 2015,2017-2018,2020,2022
 *      SMASH Team
 *
 *    GNU General Public License (GPLv3 or later)
 *
 */

#include "vir/test.h"  // This include has to be first

#include "smash/average.h"

#include <tuple>

using namespace smash;

TEST(average) {
  Average<double> avg;
  for (int i = 1; i <= 5; i++)
    avg.add(i);
  COMPARE(avg.average(), 3.0);
}

static void test_dedup_avg(std::initializer_list<double>&& a,
                           std::initializer_list<double>&& b,
                           std::initializer_list<double>&& expected_a,
                           std::initializer_list<double>&& expected_b) {
  const std::vector<double> x = a;
  const std::vector<double> y = b;
  std::vector<double> dedup_x;
  std::vector<double> dedup_y;
  std::tie(dedup_x, dedup_y) = dedup_avg(x, y);
  const std::vector<double> expected_x = expected_a;
  const std::vector<double> expected_y = expected_b;
  COMPARE(dedup_x.size(), dedup_y.size());
  COMPARE(dedup_x.size(), expected_x.size());
  COMPARE(dedup_y.size(), expected_y.size());
  for (size_t i = 0; i < dedup_x.size(); i++) {
    COMPARE(dedup_x[i], expected_x[i]);
    COMPARE(dedup_y[i], expected_y[i]);
  }
}

TEST(dedup_avg) {
  test_dedup_avg({1, 2, 2, 4, 5, 6, 7, 7, 9}, {1, 2, 3, 4, 5, 6, 7, 7, 9},
                 {1, 2, 4, 5, 6, 7, 9}, {1, 2.5, 4, 5, 6, 7, 9});
  test_dedup_avg({1, 2, 2, 4, 5, 6, 7, 7}, {1, 2, 3, 4, 5, 6, 7, 7},
                 {1, 2, 4, 5, 6, 7}, {1, 2.5, 4, 5, 6, 7});
  test_dedup_avg({1}, {2}, {1}, {2});
}

TEST(dedup_avg_pdg) {
  /// PDG data on K- p elastic cross section: momentum in lab frame.
  const std::vector<double> x = {
      0.03000,   0.05000,   0.06287,   0.07000,  0.07044,  0.07300,  0.08730,
      0.09000,   0.11000,   0.11000,   0.11210,  0.11300,  0.12262,  0.13000,
      0.13000,   0.13722,   0.14552,   0.15000,  0.15000,  0.15300,  0.15909,
      0.16269,   0.17000,   0.17000,   0.17470,  0.18768,  0.18916,  0.19000,
      0.19000,   0.19300,   0.20534,   0.21000,  0.21000,  0.21554,  0.22049,
      0.22500,   0.23000,   0.23000,   0.23300,  0.23500,  0.23944,  0.24500,
      0.24623,   0.25000,   0.25500,   0.26154,  0.26500,  0.27000,  0.27500,
      0.27618,   0.28227,   0.28500,   0.29000,  0.29290,  0.29300,  0.29500,
      0.30000,   0.30191,   0.30500,   0.31500,  0.32500,  0.33500,  0.34500,
      0.35000,   0.35012,   0.35500,   0.36500,  0.37500,  0.38500,  0.38700,
      0.38986,   0.39200,   0.39500,   0.40000,  0.40500,  0.41500,  0.42000,
      0.42500,   0.43400,   0.43407,   0.43500,  0.43600,  0.44500,  0.45500,
      0.45500,   0.46500,   0.47500,   0.49500,  0.51300,  0.51300,  0.51400,
      0.52228,   0.53400,   0.55400,   0.57300,  0.59700,  0.61000,  0.61700,
      0.62004,   0.63700,   0.64200,   0.65800,  0.66000,  0.67200,  0.67700,
      0.69900,   0.70000,   0.70300,   0.70800,  0.71900,  0.72500,  0.73000,
      0.74000,   0.74000,   0.74100,   0.75800,  0.75996,  0.76100,  0.76800,
      0.77300,   0.77700,   0.77700,   0.78000,  0.78500,  0.79300,  0.80200,
      0.80600,   0.80600,   0.81000,   0.82000,  0.82000,  0.83300,  0.83800,
      0.83800,   0.84999,   0.85300,   0.85300,  0.85600,  0.86000,  0.87000,
      0.87400,   0.87400,   0.87600,   0.89400,  0.89400,  0.89900,  0.90000,
      0.90400,   0.90400,   0.90500,   0.91600,  0.91600,  0.92200,  0.92500,
      0.93500,   0.93500,   0.93500,   0.94000,  0.94300,  0.94500,  0.95400,
      0.95400,   0.95500,   0.96000,   0.96500,  0.97000,  0.97000,  0.98000,
      0.99000,   0.99100,   0.99100,   1.00500,  1.02000,  1.02200,  1.02200,
      1.04000,   1.04400,   1.04400,   1.04500,  1.06000,  1.06100,  1.06100,
      1.08000,   1.08000,   1.08500,   1.10000,  1.10200,  1.10200,  1.11700,
      1.11700,   1.12500,   1.13400,   1.13400,  1.13800,  1.14000,  1.15000,
      1.15000,   1.15300,   1.15300,   1.16100,  1.16500,  1.17400,  1.17400,
      1.17900,   1.18000,   1.18300,   1.18300,  1.20100,  1.20500,  1.22000,
      1.22600,   1.22600,   1.23300,   1.24500,  1.25300,  1.26000,  1.26000,
      1.26300,   1.26300,   1.27600,   1.28500,  1.29600,  1.30000,  1.31600,
      1.31600,   1.32000,   1.32800,   1.34000,  1.35500,  1.36800,  1.36800,
      1.38000,   1.38300,   1.41500,   1.41500,  1.42300,  1.43300,  1.46200,
      1.46500,   1.48300,   1.51300,   1.51400,  1.53000,  1.53400,  1.54500,
      1.54600,   1.58400,   1.60600,   1.60600,  1.63400,  1.65200,  1.65300,
      1.68000,   1.68400,   1.70500,   1.70500,  1.73400,  1.73900,  1.74100,
      1.78400,   1.80000,   1.80000,   1.81500,  1.84300,  1.84300,  1.88400,
      1.93400,   1.93400,   1.98400,   2.00000,  2.03100,  2.03400,  2.08400,
      2.13400,   2.13500,   2.17500,   2.23400,  2.24000,  2.28400,  2.32500,
      2.33100,   2.37400,   2.41200,   2.51600,  2.66000,  2.66000,  2.98500,
      3.00000,   3.00000,   3.46000,   3.59000,  3.65000,  4.10000,  4.20000,
      4.60000,   5.00000,   5.50000,   6.00000,  7.20020,  9.00010,  10.12000,
      14.30000,  14.30000,  25.20000,  32.10000, 40.10000, 50.00000, 70.00000,
      100.00000, 140.00000, 147.00000, 175.00000};
  /// PDG data on K- p elastic cross section: cross section.
  const std::vector<double> y = {
      313.50, 103.60, 113.00, 44.800, 58.500, 187.00, 92.000, 71.500, 92.800,
      87.290, 82.000, 105.00, 59.400, 40.400, 79.220, 82.000, 49.000, 41.400,
      69.610, 108.00, 53.900, 98.000, 75.760, 32.800, 45.000, 73.000, 66.000,
      59.090, 53.300, 68.000, 37.000, 53.300, 60.490, 48.000, 41.000, 65.410,
      62.900, 55.690, 50.000, 55.740, 41.200, 53.240, 37.000, 51.500, 49.220,
      43.600, 47.710, 58.060, 48.750, 30.000, 44.900, 39.420, 38.270, 47.800,
      48.200, 41.220, 44.500, 44.200, 40.360, 37.020, 40.280, 37.840, 37.260,
      34.000, 33.500, 34.770, 34.210, 36.670, 33.890, 31.900, 34.700, 34.000,
      38.470, 38.900, 32.060, 32.590, 48.400, 31.190, 30.600, 32.800, 26.830,
      25.800, 28.830, 23.800, 30.320, 31.990, 23.100, 21.500, 26.500, 27.600,
      21.700, 35.000, 19.300, 19.100, 17.500, 17.700, 17.660, 18.600, 16.000,
      16.000, 17.230, 16.400, 12.100, 16.220, 15.600, 15.200, 14.200, 15.220,
      13.500, 14.200, 11.500, 14.070, 15.900, 14.000, 11.600, 16.930, 16.700,
      17.300, 15.200, 18.600, 18.300, 18.700, 17.900, 19.370, 20.500, 19.300,
      20.000, 20.450, 20.670, 18.700, 19.300, 19.630, 19.800, 19.530, 22.400,
      19.100, 19.390, 19.780, 19.500, 21.360, 20.100, 20.310, 21.070, 21.600,
      21.660, 22.020, 21.500, 20.900, 20.710, 21.340, 20.850, 20.400, 22.340,
      20.940, 20.120, 20.100, 19.980, 21.800, 21.010, 19.330, 20.640, 20.700,
      20.610, 21.270, 20.340, 20.400, 20.720, 22.400, 21.220, 21.400, 22.150,
      21.750, 20.800, 22.100, 22.150, 23.300, 21.500, 21.460, 22.220, 21.200,
      20.600, 20.560, 18.700, 18.740, 19.830, 18.300, 18.600, 18.770, 17.600,
      17.820, 17.890, 17.000, 17.750, 15.700, 17.000, 18.300, 17.300, 17.200,
      17.230, 15.300, 15.390, 16.500, 16.460, 14.660, 16.800, 15.900, 15.890,
      12.680, 13.890, 15.700, 11.300, 11.810, 13.110, 12.320, 11.870, 15.200,
      14.000, 10.870, 10.870, 11.440, 10.960, 11.570, 12.000, 10.200, 11.200,
      10.260, 9.7400, 14.400, 9.5300, 10.300, 10.300, 16.600, 10.500, 8.8300,
      8.8300, 8.4200, 8.6000, 9.1100, 9.1100, 8.3000, 7.7000, 7.7000, 8.1500,
      8.8000, 8.0600, 8.0600, 8.7000, 8.8600, 8.8600, 9.2000, 8.4000, 8.4000,
      8.1900, 8.9000, 9.0800, 9.0800, 9.5000, 8.5100, 8.5100, 9.0000, 8.1300,
      8.1300, 6.9500, 7.8600, 7.8600, 9.0000, 7.9000, 7.4800, 7.8000, 7.4600,
      7.3100, 7.1000, 7.4000, 7.1000, 7.2600, 7.9000, 6.8700, 6.2000, 7.3000,
      6.5000, 6.7600, 6.6000, 6.3200, 6.2100, 5.7000, 6.0400, 4.9500, 5.0600,
      4.9500, 4.9400, 4.4000, 4.6000, 4.3000, 4.5000, 4.2000, 3.8400, 4.1000,
      3.6200, 4.2300, 3.9500, 3.2400, 2.9600, 3.0100, 2.4600, 2.5600, 2.3300,
      2.5400, 2.5300, 2.5100, 2.5200, 2.7400, 2.5900};
  std::vector<double> dedup_x;
  dedup_x.reserve(x.size());
  std::vector<double> dedup_y;
  dedup_y.resize(y.size());
  std::tie(dedup_x, dedup_y) = dedup_avg(x, y);
  COMPARE(y[1], dedup_y[1]);
}
