/*
 *
 *    Copyright (c) 2020,2022
 *      SMASH Team
 *
 *    GNU General Public License (GPLv3 or later)
 *
 */

#ifndef SRC_INCLUDE_SMASH_CROSSSECTIONSBREMS_H_
#define SRC_INCLUDE_SMASH_CROSSSECTIONSBREMS_H_

#include <initializer_list>
#include <memory>

#include "interpolation.h"
#include "interpolation2D.h"

namespace smash {
// sqrt(s), k and theta lists are identical for all channels, so we need
// to define them only once

/// Center-of-mass energy.
const std::initializer_list<double> BREMS_SQRTS = {
    0.3,  0.31, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.4,  0.41,
    0.42, 0.43, 0.44, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5,  0.51, 0.52, 0.53,
    0.54, 0.55, 0.56, 0.57, 0.58, 0.59, 0.6,  0.61, 0.62, 0.63, 0.64, 0.65,
    0.66, 0.67, 0.68, 0.69, 0.7,  0.71, 0.72, 0.73, 0.74, 0.75, 0.76, 0.77,
    0.78, 0.79, 0.8,  0.81, 0.82, 0.83, 0.84, 0.85, 0.86, 0.87, 0.88, 0.89,
    0.9,  0.91, 0.92, 0.93, 0.94, 0.95, 0.96, 0.97, 0.98, 0.99, 1.0,  1.01,
    1.02, 1.03, 1.04, 1.05, 1.06, 1.07, 1.08, 1.09, 1.1,  1.11, 1.12, 1.13,
    1.14, 1.15, 1.16, 1.17, 1.18, 1.19, 1.2,  1.21, 1.22, 1.23, 1.24, 1.25,
    1.26, 1.27, 1.28, 1.29, 1.3,  1.31, 1.32, 1.33, 1.34, 1.35, 1.36, 1.37,
    1.38, 1.39, 1.4,  1.41, 1.42, 1.43, 1.44, 1.45, 1.46, 1.47, 1.48, 1.49,
    1.5,  1.51, 1.52, 1.53, 1.54, 1.55, 1.56, 1.57, 1.58, 1.59, 1.6,  1.61,
    1.62, 1.63, 1.64, 1.65, 1.66, 1.67, 1.68, 1.69, 1.7,  1.71, 1.72, 1.73,
    1.74, 1.75, 1.76, 1.77, 1.78, 1.79, 1.8,  1.81, 1.82, 1.83, 1.84, 1.85,
    1.86, 1.87, 1.88, 1.89, 1.9,  1.91, 1.92, 1.93, 1.94, 1.95, 1.96, 1.97,
    1.98, 1.99, 2.03, 2.06, 2.09, 2.12, 2.15, 2.18, 2.21, 2.24, 2.27, 2.3,
    2.33, 2.36, 2.39, 2.42, 2.45, 2.48, 2.51, 2.54, 2.57, 2.6,  2.63, 2.66,
    2.69, 2.72, 2.75, 2.78, 2.81, 2.84, 2.87, 2.9,  2.93, 2.96, 2.99, 3.02,
    3.05, 3.08, 3.11, 3.14, 3.17, 3.2,  3.23, 3.26, 3.29, 3.32, 3.35, 3.38,
    3.41, 3.44, 3.47, 3.5,  3.53, 3.56, 3.59, 3.62, 3.65, 3.68, 3.71, 3.74,
    3.77, 3.8,  3.83, 3.86, 3.89, 3.92, 3.95, 3.98, 4.01, 4.04, 4.07, 4.1,
    4.13, 4.16, 4.19, 4.22, 4.25, 4.28, 4.31, 4.34, 4.37, 4.4,  4.43, 4.46,
    4.49, 4.52, 4.55, 4.58, 4.61, 4.64, 4.67, 4.7,  4.73, 4.76, 4.79, 4.82,
    4.85, 4.88, 4.91, 4.94, 4.97, 5.0};

/// photon momentum
const std::initializer_list<double> BREMS_K = {
    0.001,      0.00107227, 0.00114976, 0.00123285, 0.00132194, 0.00141747,
    0.00151991, 0.00162975, 0.00174753, 0.00187382, 0.00200923, 0.00215443,
    0.00231013, 0.00247708, 0.00265609, 0.00284804, 0.00305386, 0.00327455,
    0.00351119, 0.00376494, 0.00403702, 0.00432876, 0.00464159, 0.00497702,
    0.0053367,  0.00572237, 0.00613591, 0.00657933, 0.0070548,  0.00756463,
    0.00811131, 0.00869749, 0.00932603, 0.01,       0.0107227,  0.0114976,
    0.0123285,  0.0132194,  0.0141747,  0.0151991,  0.0162975,  0.0174753,
    0.0187382,  0.0200923,  0.0215443,  0.0231013,  0.0247708,  0.0265609,
    0.0284804,  0.0305386,  0.0327455,  0.0351119,  0.0376494,  0.0403702,
    0.0432876,  0.0464159,  0.0497702,  0.053367,   0.0572237,  0.0613591,
    0.0657933,  0.070548,   0.0756463,  0.0811131,  0.0869749,  0.0932603,
    0.1,        0.107227,   0.114976,   0.123285,   0.132194,   0.141747,
    0.151991,   0.162975,   0.174753,   0.187382,   0.200923,   0.215443,
    0.231013,   0.247708,   0.265609,   0.284804,   0.305386,   0.327455,
    0.351119,   0.376494,   0.403702,   0.432876,   0.464159,   0.497702,
    0.53367,    0.572237,   0.613591,   0.657933,   0.70548,    0.756463,
    0.811131,   0.869749,   0.932603,   1.0};

/// theta angle with respect to collision axis of incoming pions
const std::initializer_list<double> BREMS_THETA = {
    0.0,      0.039767, 0.079534, 0.119301, 0.159068, 0.198835, 0.238602,
    0.278369, 0.318136, 0.357903, 0.39767,  0.437437, 0.477204, 0.516971,
    0.556738, 0.596505, 0.636272, 0.676039, 0.715806, 0.755573, 0.79534,
    0.835107, 0.874874, 0.914641, 0.954408, 0.994175, 1.03394,  1.07371,
    1.11348,  1.15324,  1.19301,  1.23278,  1.27254,  1.31231,  1.35208,
    1.39184,  1.43161,  1.47138,  1.51115,  1.55091,  1.59068,  1.63045,
    1.67021,  1.70998,  1.74975,  1.78951,  1.82928,  1.86905,  1.90882,
    1.94858,  1.98835,  2.02812,  2.06788,  2.10765,  2.14742,  2.18718,
    2.22695,  2.26672,  2.30649,  2.34625,  2.38602,  2.42579,  2.46555,
    2.50532,  2.54509,  2.58485,  2.62462,  2.66439,  2.70416,  2.74392,
    2.78369,  2.82346,  2.86322,  2.90299,  2.94276,  2.98252,  3.02229,
    3.06206,  3.10183,  3.14159};

/** @name Interpolation objects for π+- + π-+ -> π+- + π-+ + γ processes
// (opposite charge incoming pions, charged pions in final state)
 */
///@{
static std::unique_ptr<InterpolateDataLinear<double>>
    pipi_pipi_opp_interpolation = nullptr;
static std::unique_ptr<InterpolateData2DSpline>
    pipi_pipi_opp_dsigma_dk_interpolation = nullptr;
static std::unique_ptr<InterpolateData2DSpline>
    pipi_pipi_opp_dsigma_dtheta_interpolation = nullptr;
///@}

/// Total π+- + π-+ -> π+- + π-+ + γ cross section
const std::initializer_list<double> BREMS_PIPI_PIPI_OPP_SIG = {
    0.00747491, 0.0122921, 0.0176876, 0.0236189, 0.0301352, 0.0372133, 0.044843,
    0.05311,    0.0621347, 0.0720515, 0.0824852, 0.0940721, 0.106591,  0.120533,
    0.135286,   0.151444,  0.169051,  0.188232,  0.20997,   0.233411,  0.259348,
    0.287607,   0.319946,  0.355493,  0.395039,  0.438563,  0.488855,  0.544286,
    0.608247,   0.680442,  0.764123,  0.856338,  0.967263,  1.09267,   1.23912,
    1.40952,    1.6116,    1.85228,   2.13028,   2.45897,   2.85138,   3.30455,
    3.83744,    4.43706,   5.09203,   5.80154,   6.46221,   7.06256,   7.51544,
    7.78748,    7.84,      7.70954,   7.44853,   7.08494,   6.66813,   6.24972,
    5.81327,    5.41223,   5.03537,   4.69045,   4.37603,   4.09305,   3.83445,
    3.60231,    3.38701,   3.20009,   3.02559,   2.87479,   2.73184,   2.60424,
    2.49208,    2.391,     2.29629,   2.21057,   2.13476,   2.07019,   2.00961,
    1.95593,    1.90715,   1.86424,   1.8286,    1.79633,   1.77056,   1.74846,
    1.72886,    1.7192,    1.71162,   1.7047,    1.70337,   1.70558,   1.71363,
    1.72294,    1.73306,   1.74804,   1.7595,    1.77087,   1.77198,   1.76828,
    1.76118,    1.74163,   1.72122,   1.69628,   1.67209,   1.65474,   1.63878,
    1.63031,    1.62497,   1.62056,   1.62956,   1.63053,   1.64161,   1.64875,
    1.66546,    1.68474,   1.70011,   1.71996,   1.73777,   1.76191,   1.7844,
    1.80851,    1.83118,   1.86175,   1.88899,   1.91479,   1.94324,   1.975,
    2.00866,    2.0391,    2.07129,   2.1051,    2.13811,   2.17213,   2.20452,
    2.23961,    2.27368,   2.31333,   2.35377,   2.39195,   2.42749,   2.46805,
    2.50904,    2.54859,   2.58903,   2.63007,   2.66903,   2.71996,   2.75608,
    2.7978,     2.84993,   2.89137,   2.93571,   2.97935,   3.02777,   3.08136,
    3.12387,    3.17755,   3.22167,   3.27441,   3.32749,   3.36911,   3.41718,
    3.47981,    3.52576,   3.57691,   3.63248,   3.68847,   3.74877,   3.80069,
    3.8544,     3.91838,   4.13699,   4.31295,   4.50087,   4.69107,   4.87806,
    5.0714,     5.26738,   5.49042,   5.69159,   5.90579,   6.1394,    6.36685,
    6.5964,     6.83928,   7.0883,    7.32897,   7.59102,   7.86491,   8.1367,
    8.38902,    8.70122,   8.98157,   9.27895,   9.54127,   9.84217,   10.1746,
    10.4644,    10.8398,   11.1548,   11.4589,   11.8207,   12.2152,   12.537,
    12.9078,    13.2669,   13.6722,   14.0214,   14.3702,   14.7963,   15.204,
    15.5938,    16.0512,   16.4451,   16.8603,   17.3394,   17.7836,   18.2734,
    18.7134,    19.2399,   19.6287,   20.1323,   20.5788,   21.0816,   21.6743,
    22.1739,    22.7047,   23.239,    23.7949,   24.3387,   24.9575,   25.4845,
    26.0145,    26.6606,   27.2388,   27.7798,   28.5058,   29.2422,   29.7581,
    30.4349,    30.9674,   31.6934,   32.458,    33.1288,   33.7908,   34.5177,
    35.2798,    35.9836,   36.7349,   37.3355,   38.1506,   39.026,    39.7256,
    40.5225,    41.4248,   42.2532,   43.0392,   43.7769,   44.6876,   45.4984,
    46.4982,    47.4763,   48.1899,   49.0961,   50.0169,   50.8442,   51.8479,
    52.6948,    53.6507,   54.7555,   55.6072};

/// dSigma/dk for π+- + π-+ -> π+- + π-+ + γ
const std::initializer_list<double> BREMS_PIPI_PIPI_OPP_DIFF_SIG_K = {
    8.21892,     7.55436,   7.08325,     6.5781,      6.09324,   5.63109,
    5.19069,     4.83512,   4.49386,     4.18069,     3.86107,   3.54472,
    3.26334,     3.06083,   2.81386,     2.59901,     2.42298,   2.21835,
    2.0353,      1.88354,   1.71416,     1.57348,     1.44202,   1.31934,
    1.18342,     1.09911,   0.990126,    0.90147,     0.817643,  0.733823,
    0.656684,    0.593693,  0.53355,     0.466334,    0.407176,  0.356383,
    0.308854,    0.265089,  0.223695,    0.18406,     0.149187,  0.118495,
    0.0887508,   0.0609496, 0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         12.0932,   11.2315,
    10.3075,     9.50695,   8.90842,     8.26664,     7.65069,   7.16269,
    6.63942,     6.16799,   5.69578,     5.28756,     4.92964,   4.57022,
    4.19749,     3.90792,   3.62437,     3.35326,     3.09613,   2.84242,
    2.62767,     2.41633,   2.22526,     2.07125,     1.87684,   1.73486,
    1.57276,     1.4533,    1.31457,     1.20851,     1.09571,   0.996247,
    0.90271,     0.818416,  0.738742,    0.654625,    0.582607,  0.524047,
    0.455257,    0.399126,  0.346848,    0.298396,    0.251257,  0.210009,
    0.172676,    0.136956,  0.10419,     0.0743578,   0.0493235, 0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       16.3775,     15.0077,     13.539,    12.7819,
    12.0088,     11.1798,   10.2909,     9.65116,     8.98279,   8.26611,
    7.72377,     7.19368,   6.66298,     6.20895,     5.73195,   5.34176,
    4.93577,     4.55559,   4.2087,      3.89594,     3.60596,   3.32302,
    3.0635,      2.82884,   2.59663,     2.40031,     2.23773,   2.04249,
    1.84204,     1.70272,   1.54668,     1.42388,     1.30338,   1.17706,
    1.07011,     0.964406,  0.863808,    0.785652,    0.707461,  0.626414,
    0.560524,    0.495778,  0.428076,    0.380422,    0.319623,  0.274728,
    0.225655,    0.185894,  0.150083,    0.114296,    0.0836782, 0.0561258,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    20.1229,     19.0275,   17.853,      16.5936,     15.2432,   13.9577,
    13.1626,     12.3101,   11.3959,     10.502,      9.86797,   9.18811,
    8.47722,     7.9108,    7.30345,     6.76492,     6.29483,   5.83427,
    5.426,       5.01967,   4.6228,      4.2845,      3.96167,   3.65434,
    3.35982,     3.10127,   2.86687,     2.63606,     2.41712,   2.24651,
    2.05744,     1.85494,   1.72118,     1.59794,     1.44517,   1.3047,
    1.18919,     1.07333,   0.96802,     0.869883,    0.785168,  0.703772,
    0.613255,    0.553791,  0.488201,    0.422065,    0.363689,  0.31034,
    0.25952,     0.215816,  0.175599,    0.136166,    0.103137,  0.071412,
    0.0453632,   0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         26.0321,   23.2633,
    21.3804,     20.2239,   18.9838,     17.6541,     16.2283,   14.6994,
    13.888,      13.0334,   12.1171,     11.1345,     10.4356,   9.69597,
    8.90286,     8.31791,   7.74083,     7.15655,     6.6223,    6.07193,
    5.68404,     5.28481,   4.92001,     4.54994,     4.17832,   3.91329,
    3.60045,     3.29344,   3.03311,     2.79494,     2.55078,   2.34721,
    2.159,       1.98062,   1.82537,     1.68211,     1.51415,   1.35062,
    1.2592,      1.13703,   1.03117,     0.931895,    0.840453,  0.739926,
    0.659174,    0.580689,  0.518812,    0.442539,    0.383067,  0.327833,
    0.275109,    0.225719,  0.182997,    0.143326,    0.108028,  0.0762312,
    0.0478755,   0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       32.2824,     29.6796,     26.8886,   23.8959,
    22.109,      20.9189,   19.6428,     18.2744,     16.8072,   15.3022,
    14.4389,     13.5132,   12.5206,     11.4916,     10.8196,   10.0991,
    9.32649,     8.67673,   7.98981,     7.35281,     6.83576,   6.3111,
    5.86645,     5.43093,   5.05539,     4.68377,     4.31455,   3.98562,
    3.68021,     3.39246,   3.1342,      2.88499,     2.63969,   2.44199,
    2.22571,     2.03541,   1.85251,     1.69148,     1.55316,   1.41518,
    1.28424,     1.15924,   1.04721,     0.95177,     0.851051,  0.752764,
    0.673237,    0.585953,  0.520895,    0.451196,    0.38155,   0.325403,
    0.274062,    0.222836,  0.179596,    0.14016,     0.102731,  0.0707127,
    0.0425862,   0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    37.0284,     34.7085,   32.2211,     29.5538,     26.6938,   23.6882,
    22.5189,     21.265,    19.9206,     18.479,      16.9332,   15.6288,
    14.6841,     13.671,    12.5847,     11.652,      10.96,     10.2179,
    9.4795,      8.82022,   8.11331,     7.52568,     6.95654,   6.44944,
    5.98719,     5.54916,   5.10818,     4.67979,     4.33216,   4.05972,
    3.72281,     3.41966,   3.17808,     2.92274,     2.66815,   2.4193,
    2.23214,     2.04206,   1.85858,     1.71593,     1.53469,   1.39534,
    1.28853,     1.16563,   1.04303,     0.940386,    0.8412,    0.73657,
    0.657659,    0.583201,  0.504033,    0.440203,    0.372289,  0.313648,
    0.262533,    0.209861,  0.164351,    0.126059,    0.088552,  0.0574634,
    0.0319723,   0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         42.2917,   40.0725,
    37.693,      35.1415,   32.4057,     29.4721,     26.3265,   24.2605,
    22.9972,     21.6427,   20.1903,     18.633,      16.9655,   15.9411,
    14.8426,     13.6648,   12.4018,     11.6772,     10.9033,   10.0735,
    9.40313,     8.71924,   8.0495,      7.48476,     6.89827,   6.40842,
    5.91398,     5.48732,   5.05669,     4.6298,      4.35197,   4.00449,
    3.65316,     3.36285,   3.13557,     2.88746,     2.62404,   2.40643,
    2.24404,     2.01425,   1.83439,     1.66683,     1.52942,   1.39673,
    1.26117,     1.13112,   1.0197,      0.907854,    0.799641,  0.723257,
    0.621723,    0.542874,  0.476697,    0.411955,    0.345458,  0.289607,
    0.236087,    0.186966,  0.143277,    0.1049,      0.0717221, 0.0415828,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       47.3677,     45.2327,     42.9435,   40.4888,
    37.8568,     35.0345,   32.0083,     28.7634,     25.9616,   24.5937,
    23.127,      21.5542,   19.8678,     18.0596,     16.8322,   15.8079,
    14.7094,     13.5316,   12.6031,     11.7499,     10.835,    10.0495,
    9.35394,     8.61077,   8.01832,     7.38305,     6.86764,   6.34829,
    5.88151,     5.41289,   4.98182,     4.59099,     4.23937,   3.93322,
    3.63355,     3.32159,   3.11113,     2.78876,     2.5771,    2.34034,
    2.15626,     1.99251,   1.80225,     1.62065,     1.48724,   1.35118,
    1.21903,     1.08903,   0.963017,    0.879325,    0.775177,  0.679457,
    0.587712,    0.511333,  0.446785,    0.371578,    0.314802,  0.257349,
    0.205556,    0.157013,  0.116476,    0.0807934,   0.047909,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    51.9886,     49.8506,   47.5581,     45.0999,     42.4641,   39.6378,
    36.6072,     33.3576,   29.8732,     26.5045,     25.2348,   23.8733,
    22.4135,     20.8481,   19.1696,     17.8162,     16.734,    15.5735,
    14.3292,     13.2674,   12.3613,     11.3897,     10.5196,   9.8326,
    9.09597,     8.44949,   7.78184,     7.22476,     6.70579,   6.21143,
    5.71484,     5.30575,   4.91045,     4.53612,     4.1965,    3.80548,
    3.52958,     3.26543,   2.96992,     2.71524,     2.50187,   2.30418,
    2.08293,     1.89157,   1.73519,     1.56585,     1.43072,   1.27826,
    1.15055,     1.03312,   0.927438,    0.80546,     0.713191,  0.623076,
    0.543579,    0.465113,  0.395023,    0.330373,    0.268918,  0.215591,
    0.168289,    0.12516,   0.0848828,   0.0531451,   0.0270149, 0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         82.4192,   54.1486,
    52.0042,     49.7048,   47.2393,     44.5955,     41.7607,   38.721,
    35.4617,     31.9668,   28.4309,     27.028,      25.5237,   23.9107,
    22.1812,     20.3266,   18.7969,     17.6311,     16.3811,   15.0407,
    13.8882,     12.9565,   11.9574,     11.0307,     10.2987,   9.51382,
    8.84765,     8.17949,   7.57294,     6.99187,     6.4701,    5.95961,
    5.57438,     5.14235,   4.69344,     4.32125,     3.99017,   3.63212,
    3.37556,     3.09572,   2.83081,     2.60262,     2.37133,   2.16086,
    1.98209,     1.82764,   1.64209,     1.50015,     1.33076,   1.23436,
    1.06142,     0.954517,  0.848481,    0.753861,    0.655908,  0.563242,
    0.481845,    0.418659,  0.34924,     0.284478,    0.228893,  0.173051,
    0.130524,    0.0912968, 0.0552502,   0.0289287,   0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       126.444,     84.5258,     56.586,    54.3225,
    51.8953,     49.2928,   46.5022,     43.5099,     40.3013,   36.8609,
    33.1719,     29.5192,   27.9953,     26.3614,     24.6094,   22.7307,
    20.7163,     19.1734,   18.0806,     16.9088,     15.6523,   14.5302,
    13.5385,     12.4751,   11.5153,     10.7714,     9.97384,   9.33168,
    8.68855,     8.06336,   7.4284,      6.79687,     6.16081,   5.72609,
    5.31769,     4.92651,   4.52966,     4.11385,     3.839,     3.54405,
    3.24699,     2.94456,   2.72125,     2.48051,     2.31668,   2.05997,
    1.85997,     1.70503,   1.53199,     1.39129,     1.22848,   1.10734,
    0.983028,    0.877727,  0.76612,     0.671168,    0.577913,  0.495086,
    0.426219,    0.34925,   0.286893,    0.23363,     0.1793,    0.132668,
    0.0914246,   0.0573409, 0.0284295,   0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    170.27,      128.502,   83.7162,     58.7899,     56.4354,   53.9107,
    51.2036,     48.3008,   45.1883,     41.8508,     38.2722,   34.4349,
    30.9275,     29.3659,   27.6913,     25.8958,     23.9705,   21.906,
    20.3534,     19.0756,   17.7055,     16.2364,     15.0845,   14.1061,
    13.057,      12.1078,   11.2704,     10.3725,     9.6275,    8.84412,
    8.20518,     7.5859,    7.0459,      6.50933,     6.02957,   5.56388,
    5.12815,     4.74495,   4.36102,     4.01507,     3.67018,   3.37095,
    3.04592,     2.77699,   2.57417,     2.34971,     2.09086,   1.92635,
    1.74816,     1.56292,   1.43182,     1.27095,     1.14318,   1.01333,
    0.904716,    0.784358,  0.68602,     0.605398,    0.505145,  0.435741,
    0.356775,    0.292644,  0.232677,    0.181327,    0.133283,  0.091901,
    0.0552784,   0.0291745, 0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         213.613,   172.378,
    128.164,     80.7544,   62.51,       59.8833,     57.0668,   54.0467,
    50.8084,     47.3361,   43.6128,     39.6205,     35.3396,   31.9782,
    30.3399,     28.5832,   26.6996,     24.6799,     22.5142,   21.0664,
    19.7273,     18.2915,   16.7519,     15.6103,     14.5169,   13.3445,
    12.4009,     11.547,    10.664,      10.0014,     9.29093,   8.60989,
    7.89516,     7.34969,   6.77286,     6.16835,     5.72973,   5.23198,
    4.72896,     4.42047,   4.16642,     3.74768,     3.45845,   3.18848,
    2.88044,     2.59421,   2.42372,     2.21038,     1.97323,   1.79325,
    1.62656,     1.45692,   1.29381,     1.16945,     1.0301,    0.909261,
    0.808007,    0.702352,  0.609159,    0.51352,     0.430828,  0.36178,
    0.293721,    0.233623,  0.176196,    0.129955,    0.0879582, 0.0516859,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       251.02,      210.795,     167.663,   121.413,
    71.8216,     64.0437,   61.3941,     58.553,      55.5066,   52.24,
    48.7374,     44.9816,   40.9545,     36.6363,     33.8153,   31.9427,
    29.9349,     27.7819,   25.4733,     23.0423,     21.6887,   20.2372,
    18.6808,     17.0317,   15.9864,     14.8656,     13.6638,   12.7916,
    11.9048,     11.0227,   10.2197,     9.39815,     8.73806,   8.0851,
    7.54019,     6.92915,   6.28893,     5.89817,     5.45754,   4.99284,
    4.57014,     4.21329,   3.86657,     3.57396,     3.25065,   2.97705,
    2.69254,     2.46517,   2.24449,     2.03138,     1.84809,   1.63871,
    1.49078,     1.3417,    1.19612,     1.04723,     0.915094,  0.782077,
    0.700729,    0.609124,  0.516141,    0.428613,    0.362063,  0.286159,
    0.229434,    0.173024,  0.127253,    0.0814348,   0.0481178, 0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    299.263,     258.105,   213.972,     166.65,      115.908,   70.4009,
    67.5783,     64.5516,   61.3063,     57.8264,     54.095,    50.0939,
    45.8037,     41.2035,   36.2708,     34.1813,     32.3327,   30.3505,
    28.225,      25.946,    23.8667,     22.4753,     20.9834,   19.3837,
    17.8689,     16.7231,   15.4946,     14.2564,     13.3179,   12.3116,
    11.3653,     10.4394,   9.59902,     8.9057,      8.29775,   7.76399,
    7.1397,      6.57206,   6.14687,     5.62955,     5.09326,   4.70623,
    4.39079,     3.99313,   3.64318,     3.28376,     3.01132,   2.75837,
    2.52971,     2.30918,   2.06445,     1.87396,     1.70279,   1.49104,
    1.34772,     1.20552,   1.04373,     0.92822,     0.813463,  0.698067,
    0.60163,     0.513722,  0.430523,    0.354292,    0.281603,  0.224607,
    0.169711,    0.119997,  0.0760168,   0.0418391,   0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         339.902,   298.881,
    254.895,     207.73,    157.157,     102.929,     72.1482,   69.2146,
    66.069,      62.6961,   59.0794,     55.2014,     51.0431,   46.5843,
    41.8033,     37.4152,   35.5482,     33.5463,     31.3998,   29.0981,
    26.6301,     24.7653,   23.244,      21.6128,     19.8637,   18.4069,
    17.11,       15.7193,   14.5509,     13.6432,     12.6699,   11.7848,
    10.849,      9.99987,   9.1427,      8.46345,     7.82662,   7.36728,
    6.7913,      6.20095,   5.69943,     5.19872,     4.82473,   4.42107,
    4.05785,     3.65424,   3.35407,     3.08469,     2.80864,   2.56959,
    2.33087,     2.11263,   1.87225,     1.70358,     1.53235,   1.35258,
    1.19164,     1.0593,    0.925439,    0.805823,    0.696868,  0.595402,
    0.502555,    0.418566,  0.348731,    0.274821,    0.212832,  0.157057,
    0.110155,    0.0683044, 0.0397739,   0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       388.611,     346.805,     301.977,   253.91,
    202.37,      147.104,   87.8453,     75.5788,     72.4296,   69.0527,
    65.4319,     61.5494,   57.3863,     52.9224,     48.1358,   43.0034,
    39.4924,     37.2931,   34.9349,     32.4062,     29.6948,   26.7874,
    25.295,      23.7206,   22.0323,     20.222,      18.9153,   17.5297,
    16.0439,     14.9741,   13.9238,     12.8901,     12.0256,   11.127,
    10.4115,     9.65495,   8.88334,     8.11168,     7.34716,   6.80625,
    6.28809,     5.8369,    5.39827,     4.88013,     4.48263,   4.12556,
    3.78485,     3.42981,   3.12745,     2.81589,     2.60766,   2.33434,
    2.13041,     1.94086,   1.70997,     1.50991,     1.3471,    1.21289,
    1.06848,     0.93005,   0.810398,    0.691039,    0.576775,  0.49902,
    0.413546,    0.330912,  0.266269,    0.200273,    0.149215,  0.101433,
    0.0606961,   0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    433.851,     391.987,   347.096,     298.962,     247.349,   192.006,
    132.664,     83.4425,   80.0607,     76.4346,     72.5464,   68.3772,
    63.9068,     59.1132,   53.9733,     48.4618,     42.5521,   40.2947,
    38.054,      35.6514,   33.0751,     30.3127,     27.9296,   26.3715,
    24.7008,     22.9094,   21.1624,     19.599,      17.9227,   16.3343,
    15.3207,     14.2338,   13.2491,     12.2845,     11.3345,   10.4043,
    9.69738,     9.13334,   8.39458,     7.64156,     6.92647,   6.46943,
    5.9701,      5.48905,   4.94646,     4.59502,     4.1921,    3.79503,
    3.56141,     3.2253,    2.89649,     2.63578,     2.37034,   2.13252,
    1.9259,      1.70598,   1.51329,     1.35236,     1.22594,   1.05156,
    0.92465,     0.79189,   0.683924,    0.583478,    0.484146,  0.395163,
    0.319181,    0.253061,  0.188283,    0.135883,    0.0901514, 0.0510232,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         485.482,   442.614,
    396.647,     347.359,   294.509,     237.839,     177.074,   111.918,
    86.0606,     82.5104,   78.7036,     74.6218,     70.2449,   65.5518,
    60.5195,     55.1235,   49.3376,     44.6753,     42.2758,   39.7029,
    36.944,      33.9858,   30.8138,     28.647,      26.6497,   24.5081,
    22.2116,     20.7287,   19.4102,     17.9963,     16.7962,   15.6778,
    14.513,      13.6087,   12.6391,     11.6465,     10.5959,   9.79355,
    9.02799,     8.38113,   7.71837,     7.14925,     6.63656,   6.02286,
    5.58434,     5.04368,   4.67646,     4.23151,     3.89312,   3.4852,
    3.22029,     2.91186,   2.64487,     2.38302,     2.15615,   1.90499,
    1.74948,     1.50123,   1.35535,     1.19997,     1.06734,   0.897787,
    0.775433,    0.666994,  0.560335,    0.473984,    0.38343,   0.306022,
    0.238526,    0.174482,  0.122112,    0.0816211,   0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       537.492,     494.067,     447.504,   397.576,
    344.04,      286.635,   225.081,     159.079,     95.4149,   91.6293,
    87.5702,     83.2178,   78.5508,     73.5465,     68.1807,   62.427,
    56.2575,     49.6422,   46.3536,     43.6366,     40.7232,   37.5993,
    34.2497,     31.1468,   29.3724,     27.4698,     25.4297,   23.4397,
    21.9612,     20.3758,   18.7349,     17.4732,     16.1203,   14.895,
    13.7777,     12.7318,   11.8821,     11.0164,     10.1391,   9.35496,
    8.60223,     7.90747,   7.34027,     6.68416,     6.12431,   5.67481,
    5.12461,     4.7118,    4.33434,     3.94172,     3.57864,   3.26436,
    2.94807,     2.66275,   2.42135,     2.12586,     1.95678,   1.72048,
    1.52315,     1.34091,   1.19023,     1.03391,     0.899682,  0.766362,
    0.654286,    0.552492,  0.442555,    0.37646,     0.288795,  0.222011,
    0.161163,    0.108299,  0.067648,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    592.963,     548.295,   500.398,     449.039,     393.969,   334.919,
    271.602,     203.71,    130.91,      96.3988,     92.4778,   88.2734,
    83.7652,     78.9312,   73.7478,     68.1899,     62.2303,   55.8401,
    50.3682,     47.7739,   44.9921,     42.0092,     38.8109,   35.3813,
    32.9012,     30.7233,   28.388,      25.8839,     24.1088,   22.5859,
    20.9529,     19.483,    18.1143,     16.6537,     15.493,    14.2485,
    13.1952,     12.1206,   11.2393,     10.3347,     9.43404,   8.69615,
    8.01112,     7.38075,   6.87869,     6.07148,     5.59637,   5.27043,
    4.7735,      4.3655,    3.96385,     3.61095,     3.23574,   2.99303,
    2.68267,     2.39865,   2.19158,     1.93381,     1.72914,   1.51564,
    1.33638,     1.16841,   1.02554,     0.878947,    0.754,     0.635501,
    0.537086,    0.432377,  0.349677,    0.271742,    0.207893,  0.148593,
    0.0977084,   0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         655.605,   609.447,
    559.953,     506.883,   449.978,     388.96,      323.533,   253.377,
    178.151,     103.39,    99.4611,     95.2485,     90.7314,   85.888,
    80.6945,     75.1257,   69.1544,     62.7517,     55.8862,   52.246,
    49.2981,     46.1373,   42.748,      39.1137,     35.6643,   33.6316,
    31.452,      29.1149,   26.7977,     25.0506,     23.1772,   21.2253,
    19.8,        18.2718,   16.8775,     15.6212,     14.3713,   13.2272,
    12.2404,     11.486,    10.5394,     9.60279,     8.90272,   8.17571,
    7.54148,     6.96097,   6.33651,     5.83588,     5.4137,    4.9076,
    4.4241,      4.06985,   3.69064,     3.26396,     2.99099,   2.71925,
    2.40518,     2.16162,   1.92803,     1.70744,     1.50082,   1.35172,
    1.14971,     1.00816,   0.84802,     0.727574,    0.622062,  0.515718,
    0.411371,    0.32784,   0.250299,    0.18867,     0.132255,  0.0855243,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       731.851,     683.721,     632.113,   576.775,
    517.438,     453.813,   385.59,      312.437,     233.997,   149.888,
    111.313,     106.691,   101.736,     96.4232,     90.7261,   84.6173,
    78.0669,     71.0432,   63.512,      57.1225,     53.9861,   50.6229,
    47.0168,     43.15,     39.0038,     36.3661,     34.2091,   31.8962,
    29.4162,     27.5489,   25.8528,     24.0341,     22.3207,   20.647,
    18.8734,     17.61,     16.2553,     14.8958,     13.4541,   12.575,
    11.6664,     10.7199,   9.95574,     9.15346,     8.31931,   7.73351,
    7.05654,     6.49788,   5.88461,     5.412,       4.88565,   4.49321,
    4.02851,     3.58204,   3.25235,     2.94151,     2.68108,   2.41236,
    2.14985,     1.91233,   1.69668,     1.49433,     1.30259,   1.15133,
    0.986501,    0.851395,  0.696538,    0.592592,    0.485055,  0.389592,
    0.308087,    0.231932,  0.171,       0.11628,     0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    796.724,     747.313,   694.332,     637.521,     576.605,   511.286,
    441.247,     366.147,   285.619,     199.272,     118.569,   114.136,
    109.383,     104.287,   98.8223,     92.9627,     86.6797,   79.9426,
    72.7186,     64.9726,   60.8596,     57.1709,     53.2156,   48.9744,
    44.4268,     40.3397,   38.1251,     35.7503,     33.204,    30.6225,
    28.2803,     25.7689,   23.2851,     21.9835,     20.5878,   19.2021,
    17.8025,     16.451,    15.2385,     14.0471,     12.8796,   11.9439,
    11.0101,     10.0209,   9.24646,     8.5875,      7.79429,   7.14538,
    6.58934,     5.97773,   5.42979,     5.03156,     4.54178,   4.09569,
    3.67087,     3.33816,   3.03499,     2.67159,     2.42328,   2.16567,
    1.91048,     1.66842,   1.4746,      1.286,       1.10812,   0.935943,
    0.809117,    0.683643,  0.568892,    0.451047,    0.370796,  0.286931,
    0.21756,     0.15204,   0.103257,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         878.229,   826.699,
    771.446,     712.199,   648.67,      580.551,     507.509,   429.188,
    345.208,     255.158,   158.6,       123.579,     118.509,   113.072,
    107.242,     100.992,   94.289,      87.1021,     79.3957,   71.1325,
    64.6677,     61.2609,   57.6078,     53.6908,     49.4907,   44.9871,
    42.0468,     39.2964,   36.3472,     33.1848,     31.0946,   29.1462,
    27.0569,     25.1553,   23.2715,     21.3384,     19.8861,   18.3289,
    16.8776,     15.3788,   14.4902,     13.4757,     12.2893,   11.2541,
    10.3357,     9.50355,   8.67669,     7.99657,     7.32722,   6.65217,
    6.13685,     5.50027,   5.05685,     4.50918,     4.05056,   3.76657,
    3.33446,     3.01051,   2.72385,     2.4115,      2.12798,   1.89796,
    1.65126,     1.46225,   1.25452,     1.07837,     0.925962,  0.784008,
    0.650858,    0.543867,  0.435607,    0.344627,    0.262818,  0.194494,
    0.138205,    0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       978.527,     924.377,     866.314,   804.055,
    737.297,     665.714,   588.958,     506.656,     418.405,   323.777,
    222.31,      140.898,   135.176,     129.042,     122.464,   115.411,
    107.848,     99.7382,   91.0426,     81.7187,     71.7208,   67.9504,
    64.0164,     59.7981,   55.2749,     50.4249,     46.3331,   43.6577,
    40.789,      37.7129,   34.871,      32.5526,     30.0666,   27.6222,
    25.6369,     23.5081,   21.6796,     19.9221,     18.4516,   17.2665,
    15.9687,     14.5609,   13.604,      12.6086,     11.5599,   10.4887,
    9.61456,     8.86591,   8.20819,     7.4152,      6.66178,   6.17645,
    5.62469,     4.98693,   4.58108,     4.10616,     3.69997,   3.30412,
    3.00596,     2.67209,   2.36169,     2.11962,     1.84698,   1.65689,
    1.4472,      1.23449,   1.06042,     0.899472,    0.754349,  0.626997,
    0.516409,    0.409771,  0.326279,    0.246052,    0.175344,  0.131531,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1057.37,     1001.92,   942.474,     878.727,     810.372,   737.079,
    658.488,     574.218,   483.858,     386.968,     283.076,   171.675,
    148.623,     142.174,   135.259,     127.845,     119.894,   111.369,
    102.228,     92.426,    81.9159,     74.9234,     70.7342,   66.2422,
    61.4256,     56.261,    50.7231,     47.9559,     45.0263,   41.8851,
    38.5168,     35.9398,   33.1976,     30.2572,     28.1934,   26.1762,
    24.1446,     22.3065,   20.4239,     19.1462,     17.775,    16.3009,
    15.0415,     13.9558,   12.7105,     11.6985,     10.6926,   9.74513,
    9.09378,     8.27555,   7.5114,      6.84292,     6.23775,   5.61653,
    5.0706,      4.61492,   4.16425,     3.78681,     3.33153,   2.99797,
    2.70349,     2.39012,   2.11917,     1.83412,     1.58615,   1.39213,
    1.21533,     1.0228,    0.858601,    0.717346,    0.593893,  0.494095,
    0.38316,     0.296143,  0.223046,    0.159017,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1203.88,   1142.99,
    1077.69,     1007.67,   932.596,     852.095,     765.776,   673.218,
    573.972,     467.554,   353.445,     231.09,      156.982,   150.732,
    144.03,      136.844,   129.138,     120.876,     112.017,   102.517,
    92.331,      82.6404,   78.311,      73.6687,     68.691,    63.3535,
    57.6303,     53.2681,   49.8668,     46.2197,     42.309,    39.1639,
    36.5702,     33.789,    31.2351,     29.0308,     26.6671,   24.9018,
    23.1101,     21.2969,   19.3982,     17.9851,     16.6528,   15.4731,
    14.1711,     12.9394,   12.045,      11.0056,     10.2442,   9.34329,
    8.18067,     7.58302,   6.94531,     6.36483,     5.76482,   5.12896,
    4.61926,     4.20289,   3.74683,     3.34719,     2.96303,   2.68757,
    2.36298,     2.0861,    1.831,       1.59925,     1.36092,   1.1683,
    0.999263,    0.833773,  0.708477,    0.560626,    0.453268,  0.364951,
    0.283459,    0.201076,  0.205386,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1314.37,     1251.06,     1183.16,   1110.36,
    1032.31,     948.605,   858.856,     762.621,     659.432,   548.785,
    430.142,     302.924,   174.773,     168.076,     160.894,   153.193,
    144.936,     136.082,   126.588,     116.408,     105.493,   93.7883,
    87.5414,     82.5931,   77.2872,     71.5979,     65.4975,   59.6346,
    56.0754,     52.259,    48.1668,     44.1097,     41.1571,   37.9912,
    34.6988,     32.6156,   30.3817,     28.2225,     26.1498,   23.9873,
    21.797,      19.8263,   18.2172,     16.8276,     15.5277,   14.368,
    13.3432,     12.3308,   11.2013,     10.1944,     9.38966,   8.50425,
    7.58885,     7.04144,   6.31092,     5.75437,     5.24377,   4.66967,
    4.18469,     3.70525,   3.35878,     3.00095,     2.67573,   2.33352,
    2.07176,     1.77577,   1.54819,     1.33281,     1.14876,   0.955827,
    0.814594,    0.668472,  0.544374,    0.431604,    0.332505,  0.253458,
    0.18785,     0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1469.58,     1401.65,   1328.81,     1250.71,     1166.96,   1077.16,
    980.87,      877.621,   766.911,     648.2,       520.91,    384.421,
    238.069,     185.286,   177.49,      169.132,     160.169,   150.558,
    140.254,     129.204,   117.356,     104.651,     94.7975,   89.7199,
    84.2754,     78.4374,   72.1775,     65.4653,     61.3284,   57.5329,
    53.463,      49.0991,   45.7914,     42.548,      39.0703,   36.2681,
    33.6836,     31.0233,   28.9473,     26.7213,     24.4989,   22.1803,
    20.5388,     18.9161,   17.3935,     16.0481,     14.8409,   13.7143,
    12.5152,     11.3598,   10.3284,     9.32221,     8.69394,   7.89826,
    7.13748,     6.40033,   5.81412,     5.23474,     4.71511,   4.20452,
    3.8345,      3.38178,   3.0092,      2.62851,     2.30897,   2.04058,
    1.77113,     1.49904,   1.30318,     1.1045,      0.939074,  0.756939,
    0.638733,    0.509875,  0.409938,    0.314301,    0.235278,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1585.01,   1515.12,
    1440.18,     1359.81,   1273.65,     1181.25,     1082.18,   975.946,
    862.036,     739.895,   608.926,     468.493,     317.912,   207.897,
    199.594,     190.69,    181.144,     170.907,     159.93,    148.16,
    135.54,      122.008,   108.019,     102.468,     96.515,    90.132,
    83.2879,     75.949,    69.8812,     65.4743,     60.749,    55.6822,
    51.2857,     47.8186,   44.101,      40.5681,     37.7167,   34.6592,
    32.2413,     29.9188,   27.6606,     25.405,      23.3514,   21.3281,
    19.7091,     18.3021,   17.004,      15.2537,     13.983,    12.6835,
    11.7633,     10.5174,   9.77037,     8.81552,     8.02629,   7.28076,
    6.56913,     5.89498,   5.26931,     4.76222,     4.35156,   3.79032,
    3.31363,     2.97119,   2.64728,     2.30606,     2.00068,   1.73582,
    1.47541,     1.25958,   1.06386,     0.899147,    0.740117,  0.606238,
    0.481549,    0.378031,  0.293761,    0.291174,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1773.62,     1698.27,     1617.47,   1530.83,
    1437.93,     1338.32,   1231.51,     1116.98,     994.168,   862.486,
    721.287,     569.883,   407.539,     233.462,     220.639,   211.241,
    201.163,     190.357,   178.771,     166.347,     153.025,   138.74,
    123.423,     114.339,   107.802,     100.792,     93.2759,   85.2164,
    77.1262,     72.6475,   67.8452,     62.6959,     57.4023,   53.6,
    49.5229,     45.1511,   42.1337,     38.9274,     35.9263,   33.3412,
    30.718,      28.3903,   26.1812,     24.359,      22.411,    20.4362,
    18.8112,     17.2822,   15.7519,     14.3403,     13.1371,   12.1742,
    11.0589,     9.77504,   8.97845,     8.13262,     7.25566,   6.73015,
    5.99401,     5.36483,   4.81087,     4.36336,     3.83983,   3.37937,
    3.00787,     2.64259,   2.30174,     1.97312,     1.6715,    1.47073,
    1.25104,     1.04591,   0.881212,    0.721721,    0.588729,  0.467684,
    0.367883,    0.278229,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    2020.24,     1937.3,    1848.36,     1753.0,      1650.75,   1541.11,
    1423.55,     1297.49,   1162.32,     1017.38,     861.963,   695.318,
    516.631,     325.03,    245.86,      235.811,     225.036,   213.483,
    201.095,     187.811,   173.567,     158.294,     141.917,   128.491,
    121.604,     114.219,   106.3,       97.8087,     88.704,    82.7335,
    77.4428,     71.7698,   65.6868,     61.0251,     56.5973,   51.8496,
    47.7775,     43.9968,   40.0766,     37.547,      34.8346,   32.253,
    29.5307,     27.3443,   25.0915,     22.8573,     20.7745,   19.0024,
    17.5547,     16.1221,   14.8894,     13.7143,     12.3262,   11.1261,
    10.0593,     9.21971,   8.38297,     7.49441,     6.69928,   6.03342,
    5.47359,     4.9113,    4.35051,     3.84344,     3.37325,   3.0095,
    2.59012,     2.25351,   1.97314,     1.66633,     1.43249,   1.20151,
    1.0216,      0.846653,  0.686605,    0.55992,     0.446414,  0.345143,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         2158.63,   2073.72,
    1982.68,     1885.06,   1780.39,     1668.15,     1547.8,    1418.75,
    1280.38,     1132.01,   972.912,     802.319,     619.398,   423.258,
    275.445,     264.365,   252.484,     239.745,     226.085,   211.438,
    195.733,     178.892,   160.835,     141.902,     134.952,   127.499,
    119.508,     110.939,   101.751,     93.7361,     87.4797,   80.7712,
    73.5779,     67.3595,   62.6568,     57.6142,     52.859,    49.2479,
    45.3758,     41.9994,   38.653,      35.8005,     33.3174,   30.8836,
    28.3894,     26.0799,   23.5919,     21.2012,     20.1518,   18.501,
    16.7933,     15.4644,   13.9131,     12.7039,     11.6084,   10.3581,
    9.52424,     8.52264,   7.65689,     6.93718,     6.13512,   5.48339,
    4.83705,     4.32323,   3.84135,     3.42726,     2.94977,   2.57005,
    2.25931,     1.93361,   1.67266,     1.41273,     1.20459,   1.00759,
    0.805606,    0.663207,  0.541266,    0.42973,     0.420387,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       2488.37,     2393.5,      2291.77,   2182.69,
    2065.73,     1940.31,   1805.83,     1661.63,     1507.01,   1341.22,
    1163.44,     972.821,   768.424,     549.254,     314.246,   298.35,
    285.464,     271.646,   256.83,      240.944,     223.909,   205.643,
    186.057,     165.055,   152.885,     144.203,     134.894,   124.912,
    114.208,     103.518,   97.6688,     91.3969,     84.6718,   77.7635,
    72.683,      67.2353,   61.3939,     57.0503,     52.4125,   48.1836,
    44.6972,     41.1298,   37.8421,     34.6341,     31.7808,   29.0425,
    26.5527,     25.1443,   23.01,       20.7781,     18.8187,   17.1625,
    15.5978,     14.4369,   13.3422,     11.9452,     10.9019,   9.47186,
    8.78502,     7.82999,   7.03279,     6.25126,     5.59001,   4.94214,
    4.4568,      3.87467,   3.39828,     2.98125,     2.59729,   2.26789,
    1.93109,     1.6481,    1.38816,     1.16261,     0.969472,  0.798108,
    0.64651,     0.521888,  0.403124,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    2858.95,     2753.19,   2639.78,     2518.18,     2387.79,   2247.98,
    2098.06,     1937.31,   1764.94,     1580.12,     1381.94,   1169.44,
    941.579,     697.253,   435.27,      336.047,     322.323,   307.608,
    291.829,     274.91,    256.769,     237.316,     216.458,   194.092,
    176.193,     166.563,   156.238,     145.167,     133.296,   120.566,
    112.149,     104.366,   96.0202,     87.0715,     81.1705,   75.7664,
    69.9717,     64.6537,   59.3938,     54.0417,     50.7029,   47.1229,
    43.0692,     38.7887,   35.9594,     33.1613,     30.5637,   28.1537,
    25.8668,     23.6889,   21.548,      19.7157,     18.1694,   16.1727,
    14.8892,     13.5426,   12.2807,     11.021,      9.95506,   8.99094,
    8.10414,     7.24938,   6.34068,     5.72848,     5.0439,    4.47037,
    4.04821,     3.4591,    2.99812,     2.59709,     2.24559,   1.9433,
    1.6188,      1.35234,   1.16241,     0.960467,    0.789084,  0.642111,
    0.511148,    0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         3217.87,   3102.73,
    2979.28,     2846.9,    2704.96,     2552.75,     2389.55,   2214.56,
    2026.92,     1825.72,   1609.98,     1378.64,     1130.59,   864.617,
    579.42,      382.26,    366.817,     350.257,     332.501,   313.462,
    293.048,     271.157,   247.685,     222.517,     197.197,   187.12,
    176.314,     164.727,   152.303,     138.982,     128.108,   119.865,
    111.028,     101.551,   93.3988,     86.7347,     79.589,    72.9816,
    67.743,      62.1258,   58.0104,     54.0755,     49.7557,   45.0624,
    41.2541,     37.6972,   34.9868,     32.2319,     29.5998,   27.5185,
    24.1806,     22.2446,   20.4637,     18.7465,     16.8802,   15.0475,
    13.9221,     12.7152,   11.3232,     10.2013,     9.22833,   8.23311,
    7.35073,     6.61374,   5.75677,     5.14318,     4.59753,   4.00885,
    3.4934,      3.00707,   2.61446,     2.25386,     1.92199,   1.63935,
    1.36214,     1.14223,   0.954612,    0.770724,    0.624403,  0.481638,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       3669.0,      3541.87,     3405.55,   3259.38,
    3102.65,     2934.59,   2754.38,     2561.16,     2353.97,   2131.8,
    1893.59,     1638.15,   1364.26,     1070.57,     755.659,   440.308,
    423.198,     404.851,   385.179,     364.085,     341.467,   317.215,
    291.209,     263.324,   233.424,     217.716,     205.257,   191.897,
    177.572,     162.211,   147.441,     138.204,     128.298,   117.677,
    107.265,     100.077,   92.3688,     84.3312,     78.6486,   72.5554,
    67.013,      62.0658,   57.0489,     52.2697,     47.9776,   44.4593,
    40.8169,     37.2436,   34.1924,     31.3369,     28.5394,   25.9504,
    23.4875,     21.5966,   19.5927,     17.7398,     16.1754,   14.5155,
    13.0685,     11.7312,   10.5158,     9.41279,     8.4411,    7.54663,
    6.78606,     5.9005,    5.28894,     4.67237,     4.04637,   3.53772,
    3.08344,     2.65129,   2.28589,     1.93585,     1.61442,   1.37361,
    1.1441,      0.939576,  0.773062,    0.603511,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    4123.74,     3985.35,   3836.96,     3677.84,     3507.23,   3324.29,
    3128.13,     2917.79,   2692.25,     2450.41,     2191.1,    1913.05,
    1614.9,      1295.2,    952.403,     584.831,     484.008,   463.141,
    440.765,     416.773,   391.047,     363.461,     333.882,   302.165,
    268.156,     244.076,   230.88,      216.731,     201.558,   185.29,
    167.845,     157.072,   146.231,     134.606,     122.142,   114.255,
    106.342,     97.8585,   90.7759,     83.7569,     76.7758,   71.3232,
    65.5193,     60.3219,   54.9657,     50.4686,     46.4228,   42.9411,
    38.8556,     35.6676,   32.7824,     29.7125,     27.2372,   24.6136,
    22.6726,     20.663,    18.4562,     16.988,      14.957,    13.7312,
    12.0083,     11.0479,   9.73806,     8.7155,      7.9215,    6.89035,
    6.07029,     5.40128,   4.64577,     4.09854,     3.53633,   3.1048,
    2.65724,     2.28912,   1.95765,     1.63526,     1.35865,   1.16745,
    0.933054,    0.766591,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         4802.15,   4644.72,
    4475.92,     4294.93,   4100.85,     3892.75,     3669.6,    3430.34,
    3173.78,     2898.68,   2603.7,      2287.4,      1948.24,   1584.58,
    1194.63,     776.504,   542.96,      520.921,     497.289,   471.948,
    444.777,     415.642,   384.402,     350.903,     314.985,   282.071,
    267.282,     251.425,   234.422,     216.19,      196.64,    182.001,
    170.036,     157.207,   143.451,     132.933,     124.241,   114.921,
    106.063,     97.7373,   88.8102,     82.0167,     74.936,    69.2049,
    63.6753,     58.8696,   54.0134,     49.3082,     44.9549,   40.953,
    37.4593,     34.4919,   31.5504,     28.4787,     26.2156,   23.8459,
    21.2468,     19.4396,   17.4149,     15.6342,     14.1631,   12.7693,
    11.5777,     10.3201,   9.01415,     8.04621,     7.00958,   6.23148,
    5.50055,     4.92741,   4.22536,     3.62622,     3.09981,   2.68356,
    2.318,       1.98057,   1.66874,     1.3944,      1.15873,   0.932743,
    1.15234,     0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       5395.05,     5224.04,     5040.67,   4844.05,
    4633.23,     4407.16,   4164.76,     3904.84,     3626.14,   3327.29,
    3006.85,     2663.25,   2294.82,     1899.77,     1476.16,   1021.94,
    637.884,     612.034,   584.316,     554.595,     522.726,   488.554,
    451.912,     412.623,   370.494,     325.32,      307.362,   289.992,
    271.366,     251.394,   229.978,     210.915,     197.655,   183.437,
    168.191,     154.151,   143.493,     132.066,     120.815,   112.359,
    103.291,     95.2148,   87.4444,     80.4138,     74.3291,   67.9513,
    61.217,      57.1458,   52.699,      47.5647,     43.5192,   40.1368,
    36.9881,     33.245,    30.0251,     26.9576,     24.8097,   22.4339,
    20.3415,     18.2177,   16.144,      14.7221,     13.0667,   11.8238,
    10.4226,     9.20904,   8.18252,     7.30168,     6.42782,   5.64591,
    4.9154,      4.30153,   3.78062,     3.23535,     2.73897,   2.31748,
    2.0015,      1.67349,   1.39745,     1.16953,     0.938218,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    6286.32,     6091.4,    5882.4,      5658.29,     5417.99,   5160.33,
    4884.04,     4587.78,   4270.12,     3929.5,      3564.26,   3172.63,
    2752.7,      2302.42,   1819.6,      1301.89,     746.761,   689.439,
    659.871,     628.165,   594.168,     557.715,     518.627,   476.714,
    431.772,     383.583,   354.548,     334.805,     313.635,   290.935,
    266.595,     241.651,   227.107,     211.513,     194.792,   177.405,
    165.095,     151.897,   137.744,     128.522,     118.938,   109.646,
    101.303,     92.8825,   85.9008,     78.7047,     71.6219,   65.9838,
    61.0796,     56.1794,   51.0729,     46.0023,     41.8641,   38.6425,
    35.02,       31.6355,   28.5247,     25.8474,     23.2133,   21.1275,
    18.8453,     16.9409,   15.2581,     13.4938,     12.1075,   10.8742,
    9.60964,     8.51177,   7.62,        6.64417,     5.86978,   5.08632,
    4.40113,     3.82108,   3.24354,     2.85838,     2.40363,   2.03494,
    1.72457,     1.41395,   1.14614,     0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         7151.73,   6936.36,
    6705.42,     6457.79,   6192.26,     5907.55,     5602.26,   5274.9,
    4923.9,      4547.52,   4143.95,     3711.21,     3247.19,   2749.65,
    2216.15,     1644.09,   1030.69,     803.868,     769.514,   732.676,
    693.177,     650.823,   605.408,     556.711,     504.495,   448.506,
    404.77,      382.339,   358.286,     332.496,     304.841,   275.188,
    257.834,     242.407,   225.865,     208.127,     192.739,   177.083,
    160.296,     147.283,   135.671,     123.841,     115.804,   107.186,
    99.2584,     90.908,    83.1971,     75.8208,     69.3656,   63.9414,
    58.2578,     52.888,    49.0083,     45.0238,     40.0757,   37.1584,
    33.5218,     29.802,    26.7808,     24.3522,     22.0314,   19.5887,
    17.7197,     15.7873,   14.0958,     12.5893,     11.2113,   9.88838,
    8.82924,     7.71303,   6.81663,     5.95984,     5.21312,   4.47686,
    3.90605,     3.32277,   2.84936,     2.46571,     2.06837,   1.7279,
    1.42121,     0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       8117.6,      7878.87,     7622.89,   7348.41,
    7054.1,      6738.51,   6400.12,     6037.27,     5648.2,    5231.02,
    4783.68,     4304.02,   3789.7,      3238.2,      2646.85,   2012.77,
    1332.86,     902.946,   866.024,     826.434,     783.982,   738.463,
    689.655,     637.319,   581.201,     521.028,     462.875,   438.117,
    411.57,      383.103,   352.58,      319.851,     294.441,   274.896,
    253.938,     231.466,   214.097,     200.997,     186.95,    173.033,
    159.686,     145.375,   134.609,     123.822,     114.335,   105.145,
    96.5204,     87.9729,   81.4443,     73.9626,     66.7053,   61.7066,
    56.5308,     51.4717,   46.3091,     42.2318,     38.3837,   34.7479,
    31.039,      27.9207,   25.4505,     22.9121,     20.5944,   18.0889,
    16.1227,     14.5671,   12.9875,     11.4614,     10.1622,   8.9817,
    7.86132,     7.02137,   6.05555,     5.25772,     4.63411,   3.92551,
    3.43778,     2.87465,   2.45709,     2.03986,     1.69719,   1.7185,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    9035.27,     8776.15,   8498.32,     8200.4,      7880.96,   7538.43,
    7171.14,     6777.31,   6355.03,     5902.22,     5416.69,   4896.08,
    4337.84,     3739.26,   3097.42,     2409.19,     1671.23,   1010.72,
    970.71,      927.809,   881.809,     832.484,     779.594,   722.883,
    662.073,     596.868,   526.951,     495.644,     467.34,    436.99,
    404.448,     369.553,   337.687,     316.192,     293.144,   268.43,
    245.235,     228.196,   209.926,     191.794,     179.448,   166.209,
    153.363,     140.486,   128.868,     119.379,     110.274,   101.437,
    92.6265,     84.8178,   78.9737,     70.5748,     64.0351,   57.9319,
    52.7677,     48.091,    43.5264,     39.4431,     35.4448,   32.3015,
    28.9174,     25.8303,   23.4168,     20.8768,     18.7242,   16.8149,
    14.9476,     13.2372,   11.6042,     10.3665,     9.09974,   7.97088,
    6.92304,     6.19256,   5.27441,     4.59894,     3.94932,   3.4211,
    2.89253,     2.41885,   2.02325,     1.63164,     0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         9749.43,   9476.21,
    9183.24,     8869.1,    8532.26,     8171.07,     7783.79,   7368.51,
    6923.23,     6445.76,   5933.79,     5384.83,     4796.19,   4165.01,
    3488.21,     2762.51,   1984.36,     1149.98,     1042.83,   998.677,
    951.332,     900.565,   846.13,      787.761,     725.174,   658.064,
    586.104,     540.923,   510.522,     477.923,     442.968,   405.488,
    366.652,     345.655,   323.14,      298.998,     273.703,   256.981,
    239.051,     219.825,   203.421,     186.175,     169.868,   156.397,
    142.895,     132.729,   122.216,     111.899,     102.71,    94.177,
    86.1688,     79.8978,   72.9347,     65.7569,     59.7996,   54.4583,
    49.5573,     44.4301,   40.4678,     36.4851,     33.2413,   29.3603,
    26.3718,     23.2871,   21.3263,     18.6941,     16.8118,   14.9971,
    13.2545,     11.6373,   10.1702,     9.08784,     7.91835,   6.97022,
    6.05974,     5.24341,   4.56862,     3.90112,     3.33062,   2.77496,
    2.35451,     1.93603,   0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       10292.7,     10012.0,     9711.03,   9388.28,
    9042.21,     8671.13,   8273.24,     7846.59,     7389.1,    6898.56,
    6372.57,     5808.56,   5203.79,     4555.32,     3859.99,   3114.41,
    2314.94,     1457.7,    1136.66,     1088.28,     1036.39,   980.764,
    921.112,     857.15,    788.566,     715.025,     636.17,    574.904,
    544.81,      512.54,    477.939,     440.837,     401.054,   375.992,
    353.112,     328.579,   302.273,     280.286,     258.199,   234.516,
    217.041,     202.097,   186.64,      174.492,     161.467,   149.058,
    136.009,     124.923,   114.077,     104.178,     95.2511,   86.9895,
    79.9953,     74.3806,   67.0989,     60.3352,     54.3681,   48.8835,
    44.6204,     40.0218,   36.3285,     32.326,      29.7643,   26.1318,
    23.3398,     21.078,    18.4938,     16.6452,     14.762,    13.0599,
    11.4672,     10.1472,   8.94446,     7.78003,     6.75111,   5.92843,
    5.12986,     4.34694,   3.71964,     3.15741,     2.65884,   2.18871,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    10714.2,     10428.6,   10122.3,     9793.93,     9441.8,    9064.22,
    8659.36,     8225.23,   7759.73,     7260.6,      6725.39,   6151.5,
    5536.14,     4876.31,   4168.8,      3410.16,     2596.69,   1724.43,
    1186.96,     1139.08,   1087.74,     1032.69,     973.659,   910.363,
    842.492,     769.717,   691.683,     616.793,     582.464,   545.654,
    506.184,     463.862,   418.481,     386.387,     364.318,   340.654,
    315.28,      293.618,   274.66,      254.332,     235.743,   219.956,
    203.027,     187.747,   171.775,     158.318,     145.496,   134.719,
    124.005,     113.367,   103.531,     94.8817,     88.065,    80.0168,
    73.2478,     66.0031,   59.6182,     55.0009,     48.9677,   44.3934,
    39.3704,     35.8763,   31.6913,     28.8295,     25.4399,   22.8255,
    20.2066,     17.9431,   16.0246,     14.1943,     12.4903,   11.0113,
    9.62305,     8.38553,   7.3337,      6.43078,     5.50756,   4.78037,
    4.07011,     3.49028,   2.90928,     2.38573,     2.55517,   0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         10695.7,   10416.7,
    10117.5,     9796.74,   9452.76,     9083.92,     8688.43,   8264.35,
    7809.63,     7322.05,   6799.24,     6238.64,     5637.52,   4992.97,
    4301.84,     3560.76,   2766.12,     1914.06,     1183.22,   1138.3,
    1090.14,     1038.5,    983.123,     923.747,     860.079,   791.811,
    718.609,     640.116,   604.562,     570.161,     533.274,   493.721,
    451.31,      412.971,   385.744,     356.549,     325.245,   297.923,
    281.204,     263.278,   244.731,     227.455,     208.931,   193.185,
    178.652,     165.113,   152.997,     141.91,      131.453,   119.794,
    109.156,     100.61,    92.9248,     84.5051,     77.1351,   69.9494,
    64.1234,     57.8347,   51.6047,     47.2327,     42.387,    38.2148,
    34.0263,     30.4996,   27.1833,     24.2709,     21.4746,   19.1931,
    17.1218,     15.0323,   13.2622,     11.4833,     10.2112,   8.98542,
    7.83273,     6.81485,   5.83481,     5.05715,     4.32244,   3.70821,
    3.07645,     2.60181,   2.05225,     0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       10390.1,     10125.5,     9841.84,   9537.63,
    9211.44,     8861.68,   8486.64,     8084.49,     7653.29,   7190.92,
    6695.14,     6163.52,   5593.5,      4982.27,     4326.88,   3624.12,
    2870.57,     2062.57,   1196.18,     1129.08,     1080.66,   1028.75,
    973.077,     913.384,   849.378,     780.746,     707.153,   628.243,
    582.881,     551.54,    517.934,     481.898,     443.259,   404.276,
    381.475,     357.027,   330.811,     303.974,     285.877,   266.474,
    245.668,     231.019,   215.463,     199.377,     182.997,   167.347,
    156.881,     145.765,   134.048,     122.863,     112.328,   104.583,
    96.7644,     88.0699,   80.4279,     73.8416,     66.4375,   60.9082,
    55.4673,     49.6862,   44.8004,     39.3949,     35.5242,   31.9017,
    29.0866,     25.439,    22.6777,     19.9087,     17.648,    15.5673,
    13.6543,     11.9974,   10.5809,     9.19153,     8.10935,   6.9316,
    6.05175,     5.23273,   4.44596,     3.82711,     3.20691,   2.66653,
    2.15215,     0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    9758.42,     9515.09,   9254.17,     8974.4,      8674.41,   8352.74,
    8007.82,     7637.98,   7241.41,     6816.18,     6360.22,   5871.31,
    5347.07,     4784.94,   4182.19,     3535.88,     2842.86,   2099.77,
    1302.96,     1063.05,   1019.2,      972.178,     921.76,    867.698,
    809.729,     747.571,   680.921,     609.455,     557.22,    529.025,
    498.792,     466.375,   431.615,     394.343,     371.374,   348.869,
    324.737,     298.861,   280.164,     261.34,      241.156,   225.078,
    209.701,     193.871,   179.819,     164.83,      154.741,   144.023,
    133.257,     122.929,   113.356,     105.007,     96.4495,   88.7221,
    82.2542,     74.6863,   69.138,      62.9172,     56.4458,   51.3253,
    47.2436,     41.5982,   37.5666,     33.3797,     29.639,    26.5108,
    23.2499,     20.6007,   18.2198,     16.0654,     13.9989,   12.1999,
    10.7629,     9.30447,   8.13381,     7.07968,     6.14971,   5.29035,
    4.53218,     3.85807,   3.22128,     2.72721,     2.15656,   0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         9206.89,   8982.52,
    8741.92,     8483.95,   8207.32,     7910.71,     7592.66,   7251.63,
    6885.95,     6493.84,   6073.4,      5622.58,     5139.17,   4620.83,
    4065.03,     3469.07,   2830.04,     2144.82,     1410.09,   1023.7,
    981.857,     936.989,   888.877,     837.289,     781.973,   722.659,
    659.059,     590.862,   530.505,     504.511,     476.637,   446.75,
    414.702,     380.338,   354.472,     332.315,     308.557,   283.081,
    263.964,     247.782,   230.431,     214.978,     201.254,   186.537,
    174.061,     160.797,   149.336,     137.8,       129.884,   121.523,
    111.317,     103.169,   95.9369,     89.0152,     81.7925,   75.9184,
    69.9323,     63.5695,   57.9813,     52.6557,     47.6758,   42.9968,
    38.3794,     34.2008,   30.6198,     27.1731,     24.0252,   21.1964,
    18.4597,     16.3244,   14.2343,     12.3994,     10.7041,   9.42947,
    8.15779,     7.09594,   6.1147,      5.25617,     4.55148,   3.80022,
    3.185,       2.66854,   2.16165,     0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       8417.55,     8215.94,     7999.75,   7767.95,
    7519.39,     7252.87,   6967.08,     6660.65,     6332.07,   5979.74,
    5601.96,     5196.87,   4762.5,      4296.75,     3797.34,   3261.84,
    2687.63,     2071.93,   1411.74,     918.055,     882.71,    844.81,
    804.172,     760.596,   713.871,     663.77,      610.048,   552.444,
    492.193,     468.406,   442.899,     415.55,      386.224,   354.778,
    328.285,     308.9,     288.113,     265.824,     246.561,   231.887,
    216.153,     201.272,   189.763,     177.422,     165.217,   152.486,
    142.3,       134.038,   124.94,      115.126,     107.908,   101.015,
    94.2266,     85.9465,   80.1323,     74.308,      68.9434,   63.6065,
    58.1506,     52.4878,   48.1485,     44.0355,     39.2967,   35.0829,
    31.5722,     27.7746,   24.4129,     21.6112,     18.8817,   16.4259,
    14.3432,     12.4785,   10.8758,     9.4075,      8.15407,   7.05853,
    6.02166,     5.16588,   4.40934,     3.76658,     3.10269,   2.6255,
    2.11179,     1.65983,   0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    7574.67,     7397.0,    7206.48,     7002.2,      6783.16,   6548.28,
    6296.44,     6026.39,   5736.82,     5426.33,     5093.41,   4736.42,
    4353.63,     3943.18,   3503.07,     3031.15,     2525.13,   1982.54,
    1400.74,     835.88,    804.414,     770.674,     734.496,   695.704,
    654.107,     609.505,   561.68,      510.398,     455.411,   427.584,
    404.666,     380.091,   353.741,     325.486,     298.992,   283.079,
    266.017,     247.721,   229.761,     216.199,     201.656,   186.532,
    174.543,     161.687,   150.679,     141.356,     131.921,   122.834,
    115.231,     109.558,   102.672,     95.6298,     89.3412,   82.9214,
    76.7731,     71.1035,   66.9817,     63.1191,     57.5,      53.1011,
    48.2675,     44.39,     40.0014,     35.6838,     32.2327,   28.5362,
    25.446,      22.2759,   19.4084,     16.8901,     14.4346,   12.6248,
    10.8455,     9.39592,   8.06749,     6.96515,     6.00224,   5.05415,
    4.32972,     3.63823,   3.04788,     2.4968,      2.07587,   1.58574,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         6809.91,   6653.38,
    6485.53,     6305.56,   6112.58,     5905.65,     5683.76,   5445.85,
    5190.74,     4917.19,   4623.87,     4309.36,     3972.12,   3610.5,
    3222.76,     2806.99,   2361.17,     1883.14,     1370.57,   820.945,
    730.578,     700.12,    667.46,      632.44,      594.89,    554.626,
    511.452,     465.158,   415.518,     384.34,      365.313,   344.911,
    323.035,     299.577,   274.752,     259.255,     242.639,   224.822,
    205.892,     193.842,   180.92,      167.065,     157.66,    148.252,
    138.761,     129.862,   120.792,     113.825,     106.878,   100.954,
    94.6792,     88.1889,   82.9486,     78.4327,     73.4621,   67.9972,
    63.5346,     59.4603,   54.7882,     51.0982,     47.6436,   43.1848,
    39.5698,     36.6337,   32.6062,     29.492,      25.8606,   22.5146,
    19.7661,     17.2107,   14.759,      12.8279,     10.9769,   9.30809,
    8.01455,     6.86515,   5.88854,     4.9798,      4.2172,    3.51425,
    2.94874,     2.44771,   1.99327,     1.554,       0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       6214.85,     6074.39,     5923.77,   5762.27,
    5589.1,      5403.41,   5204.31,     4990.81,     4761.89,   4516.42,
    4253.22,     3970.99,   3668.37,     3343.88,     2995.94,   2622.85,
    2222.8,      1793.84,   1333.88,     840.682,     658.128,   631.562,
    603.075,     572.529,   539.777,     504.657,     466.999,   426.62,
    383.322,     349.298,   331.564,     312.548,     292.159,   270.295,
    246.852,     232.365,   219.204,     205.093,     189.961,   178.203,
    166.644,     154.251,   145.121,     137.299,     129.01,    120.859,
    112.12,      105.599,   98.8612,     93.0353,     87.0738,   81.1502,
    77.0321,     72.5906,   67.8205,     63.4038,     59.245,    56.3162,
    52.6607,     49.7283,   45.9537,     42.5511,     39.4157,   36.4928,
    33.0758,     29.8259,   26.6839,     23.4627,     20.4338,   17.7602,
    15.2531,     13.138,    11.1586,     9.40022,     8.04471,   6.88644,
    5.69732,     4.85321,   4.13125,     3.46981,     2.85444,   2.3749,
    1.91011,     1.51071,   0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    5578.73,     5454.85,   5322.03,     5179.6,      5026.88,   4863.12,
    4687.53,     4499.25,   4297.36,     4080.89,     3848.77,   3599.87,
    3332.99,     3046.82,   2739.97,     2410.94,     2058.14,   1679.84,
    1274.2,      839.248,   588.738,     566.034,     541.689,   515.584,
    487.593,     457.58,    425.397,     390.889,     353.887,   319.487,
    304.017,     287.43,    269.643,     250.572,     230.122,   214.109,
    200.7,       186.321,   170.904,     159.534,     150.738,   141.305,
    132.407,     124.213,   115.427,     109.036,     102.47,    96.0775,
    89.4586,     83.7497,   78.412,      75.1508,     70.383,    65.4196,
    61.6614,     58.0331,   55.2502,     51.8124,     49.2528,   46.1629,
    43.6731,     41.0064,   38.0682,     35.8435,     32.7417,   29.9038,
    26.8607,     23.6988,   20.9577,     18.6081,     15.8087,   13.5342,
    11.4286,     9.6594,    8.13416,     6.89093,     5.79515,   4.83047,
    4.049,       3.38228,   2.77065,     2.30389,     1.85718,   1.45092,
    1.08731,     0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         5023.1,    4913.68,
    4796.35,     4670.54,   4535.63,     4390.98,     4235.88,   4069.57,
    3891.23,     3700.01,   3494.97,     3275.12,     3039.37,   2786.59,
    2515.54,     2224.91,   1913.26,     1579.1,      1220.79,   836.586,
    535.69,      514.539,   491.86,      467.542,     441.466,   413.506,
    383.526,     351.378,   316.908,     280.033,     267.584,   254.235,
    239.922,     224.574,   208.116,     193.775,     183.13,    171.716,
    159.477,     148.208,   138.867,     128.851,     119.271,   112.04,
    104.286,     97.7267,   91.4123,     85.9751,     81.318,    76.8876,
    72.4484,     68.3449,   64.2414,     60.1861,     56.0776,   53.3285,
    50.6374,     47.3787,   45.921,      43.4774,     40.4371,   38.3609,
    35.988,      34.0952,   31.8123,     29.6394,     26.853,    24.4158,
    21.9278,     19.0089,   16.5558,     14.084,      11.9036,   9.98957,
    8.31975,     6.88093,   5.81205,     4.80655,     3.95994,   3.32153,
    2.72635,     2.21712,   1.81238,     1.42812,     1.05608,   0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       4480.62,     4385.09,     4282.65,   4172.81,
    4055.04,     3928.75,   3793.33,     3648.13,     3492.44,   3325.5,
    3146.49,     2954.54,   2748.72,     2528.03,     2291.39,   2037.65,
    1765.57,     1473.83,   1161.0,      825.573,     493.908,   474.865,
    454.445,     432.55,    409.072,     383.898,     356.904,   327.96,
    296.924,     263.645,   247.387,     234.781,     221.264,   206.77,
    191.229,     176.289,   166.497,     155.998,     144.74,    133.723,
    125.901,     117.514,   108.779,     102.247,     95.2435,   88.9681,
    83.4204,     78.2962,   74.4308,     70.3258,     65.9732,   62.2316,
    58.6436,     55.2075,   51.427,      48.7513,     46.4509,   43.7053,
    41.2778,     39.5489,   37.9803,     35.6804,     33.8103,   32.2074,
    30.4018,     28.8431,   26.7083,     24.5162,     22.0538,   19.7754,
    17.1546,     14.6759,   12.4102,     10.3445,     8.66238,   7.12097,
    5.93418,     4.87018,   4.00531,     3.28081,     2.69479,   2.192,
    1.74447,     1.38054,   1.02962,     0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    4034.22,     3949.68,   3859.04,     3761.85,     3657.63,   3545.88,
    3426.05,     3297.57,   3159.8,      3012.07,     2853.67,   2683.82,
    2501.7,      2306.41,   2097.02,     1872.49,     1631.73,   1373.57,
    1096.76,     799.945,   481.679,     428.852,     410.809,   391.463,
    370.719,     348.475,   324.624,     299.049,     271.626,   242.221,
    223.663,     212.255,   200.022,     186.906,     172.842,   157.995,
    150.203,     141.847,   132.887,     123.334,     115.719,   107.553,
    98.797,      93.0951,   87.4411,     81.8775,     76.9807,   71.9374,
    67.7454,     63.5202,   59.7786,     56.2542,     52.9256,   50.3415,
    47.4635,     44.8598,   42.667,      40.5092,     38.3714,   36.1667,
    34.4218,     32.9087,   31.4046,     30.4063,     28.7493,   27.6367,
    25.9152,     24.2118,   22.2891,     20.2297,     17.7514,   15.3176,
    13.0942,     10.9511,   8.94288,     7.41047,     6.10132,   4.95605,
    4.04324,     3.29875,   2.68905,     2.16853,     1.7223,    1.34877,
    1.0252,      0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         3714.29,   3637.58,
    3555.33,     3467.14,   3372.57,     3271.17,     3162.44,   3045.85,
    2920.83,     2786.78,   2643.05,     2488.92,     2323.66,   2146.46,
    1956.45,     1752.71,   1534.24,     1299.99,     1048.8,    779.468,
    490.668,     387.957,   372.0,       354.89,      336.543,   316.871,
    295.776,     273.158,   248.904,     222.898,     203.142,   193.497,
    183.154,     172.064,   160.172,     147.421,     138.526,   129.945,
    120.743,     110.877,   104.268,     98.0214,     91.324,    85.5896,
    80.0781,     74.4021,   69.8651,     65.0003,     61.3249,   57.5094,
    54.2046,     50.9766,   48.0018,     45.2547,     42.9369,   40.7307,
    38.1671,     36.632,    34.9753,     33.0743,     31.6917,   29.9675,
    29.0624,     27.8329,   26.5582,     25.8318,     24.3924,   23.4426,
    21.9223,     20.3244,   18.222,      16.0531,     13.9015,   11.563,
    9.52147,     7.78429,   6.3365,      5.13299,     4.12713,   3.32891,
    2.68553,     2.134,     1.70054,     1.31854,     1.00374,   0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       3443.88,     3373.68,     3298.41,   3217.69,
    3131.14,     3038.34,   2938.83,     2832.13,     2717.72,   2595.04,
    2463.5,      2322.45,   2171.2,      2009.03,     1835.13,   1648.67,
    1448.73,     1234.35,   1004.47,     757.977,     493.672,   349.81,
    336.223,     321.655,   306.034,     289.284,     271.324,   252.065,
    231.415,     209.273,   189.093,     179.543,     169.303,   158.323,
    146.55,      133.926,   125.035,     118.09,      110.642,   102.657,
    96.2886,     90.726,    84.7613,     79.1621,     73.9392,   68.3389,
    64.4725,     60.4502,   56.7083,     52.8714,     49.9241,   46.9478,
    43.845,      41.5817,   39.3088,     36.5913,     35.1492,   33.2442,
    31.261,      29.8498,   28.7015,     27.6053,     26.3158,   25.4317,
    24.6774,     23.6953,   22.882,      22.1384,     21.1373,   20.0101,
    18.4679,     16.6411,   14.4835,     12.3673,     10.1567,   8.3504,
    6.64782,     5.30408,   4.28016,     3.37366,     2.71488,   2.13012,
    1.6834,      1.31883,   0.978976,    0.690158,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    3123.61,     3061.2,    2994.28,     2922.52,     2845.57,   2763.06,
    2674.59,     2579.73,   2478.01,     2368.94,     2251.99,   2126.59,
    1992.12,     1847.94,   1693.34,     1527.56,     1349.81,   1159.21,
    954.83,      735.685,   500.703,     327.74,      314.946,   301.227,
    286.516,     270.743,   253.829,     235.693,     216.247,   195.395,
    173.774,     165.53,    156.69,      147.212,     137.048,   126.15,
    117.108,     110.563,   103.544,     96.018,      89.19,     83.3963,
    77.184,      71.3837,   66.9873,     62.2732,     58.8028,   55.594,
    52.0523,     48.1809,   45.2361,     42.6458,     40.7496,   38.4906,
    35.9462,     33.6009,   32.1139,     30.2973,     29.0677,   27.7733,
    26.3803,     25.3076,   24.0853,     23.2231,     22.5365,   21.7007,
    21.168,      20.5463,   20.0925,     19.175,      18.1601,   16.8649,
    15.0349,     12.8278,   10.8893,     8.79512,     7.03645,   5.58138,
    4.43389,     3.51717,   2.77248,     2.17299,     1.68089,   1.30666,
    0.990545,    0.72711,   0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         2923.77,   2866.2,
    2804.46,     2738.26,   2667.28,     2591.17,     2509.56,   2422.05,
    2328.22,     2227.6,    2119.72,     2004.04,     1880.0,    1746.99,
    1604.37,     1451.45,   1287.47,     1111.64,     923.112,   720.955,
    504.188,     303.479,   291.875,     279.434,     266.093,   251.788,
    236.45,      220.002,   202.367,     183.456,     163.18,    153.607,
    145.292,     136.375,   126.814,     116.562,     107.153,   101.231,
    94.8802,     88.071,    81.6175,     77.0191,     72.0884,   67.0406,
    63.0789,     58.831,    54.9811,     51.3828,     47.7987,   44.3754,
    41.4786,     39.128,    37.2799,     35.3367,     33.0851,   31.0191,
    29.499,      27.971,    26.5263,     25.335,      24.0256,   23.121,
    21.9381,     21.3175,   20.5518,     19.8498,     19.4536,   18.8758,
    18.6337,     18.2076,   17.6145,     16.7227,     15.1772,   13.5896,
    11.5458,     9.46265,   7.52666,     5.98271,     4.64415,   3.64281,
    2.85525,     2.2101,    1.71718,     1.30002,     0.982624,  0.702138,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       2614.71,     2564.2,      2510.04,   2451.97,
    2389.7,      2322.93,   2251.33,     2174.56,     2092.24,   2003.97,
    1909.33,     1807.84,   1699.02,     1582.34,     1457.22,   1323.06,
    1179.21,     1024.96,   859.561,     682.212,     492.045,   288.136,
    269.256,     257.86,    245.64,      232.537,     218.487,   203.422,
    187.268,     169.946,   151.373,     140.378,     132.814,   124.704,
    116.008,     106.683,   97.2634,     92.4244,     87.2356,   81.6719,
    75.9198,     71.8739,   67.5355,     62.8836,     58.9474,   54.7653,
    50.7275,     47.0934,   43.6665,     41.6828,     39.3418,   36.3914,
    34.2766,     32.5757,   30.8648,     28.6874,     26.974,    25.7372,
    24.4704,     23.1274,   21.9361,     20.8783,     20.3589,   19.3154,
    18.591,      18.3161,   17.6505,     17.1754,     17.3493,   16.9115,
    16.5285,     16.2347,   15.157,      13.9409,     12.151,    10.1602,
    8.14094,     6.40483,   4.94287,     3.84764,     2.95125,   2.2832,
    1.74683,     1.31988,   1.00239,     0.705973,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    2435.52,     2389.09,   2339.31,     2285.93,     2228.7,    2167.32,
    2101.52,     2030.95,   1955.29,     1874.16,     1787.17,   1693.89,
    1593.86,     1486.61,   1371.61,     1248.3,      1116.08,   974.301,
    822.277,     659.266,   484.476,     297.053,     246.925,   236.82,
    225.986,     214.368,   201.91,      188.553,     174.229,   158.871,
    142.403,     130.659,   123.98,      116.819,     109.14,    100.906,
    92.0774,     87.2202,   82.3987,     77.2287,     71.6851,   67.4277,
    63.0204,     58.2947,   54.6937,     51.2398,     47.6994,   44.4948,
    41.1136,     38.6687,   36.1344,     33.9005,     31.8139,   29.9106,
    28.5097,     26.9722,   25.49,       24.1012,     22.427,    21.1137,
    20.3997,     19.3746,   18.6603,     17.844,      17.0463,   16.4771,
    16.2868,     15.8307,   15.6597,     15.6289,     15.5542,   15.3461,
    14.8951,     14.0585,   12.8208,     10.7354,     8.70125,   6.94465,
    5.3409,      4.05689,   3.1011,      2.34222,     1.7618,    1.36241,
    1.00091,     0.723525,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         2270.14,   2227.53,
    2181.85,     2132.86,   2080.33,     2024.0,      1963.6,    1898.84,
    1829.4,      1754.94,   1675.1,      1589.49,     1497.69,   1399.26,
    1293.72,     1180.54,   1059.19,     929.072,     789.547,   639.94,
    479.521,     307.51,    231.09,      221.697,     211.626,   200.827,
    189.248,     176.832,   163.518,     149.243,     133.936,   121.235,
    115.295,     108.925,   102.094,     94.7705,     86.9172,   81.586,
    76.9357,     71.9493,   66.6025,     62.6551,     59.0651,   55.2156,
    51.5184,     47.8347,   43.9698,     41.6118,     39.0833,   36.7237,
    34.2474,     31.7072,   29.3689,     27.7298,     26.2436,   24.781,
    23.4086,     22.3307,   21.1562,     19.761,      18.6683,   17.8434,
    17.115,      16.2271,   15.7914,     15.1081,     14.9374,   14.4347,
    14.3586,     14.2972,   14.2485,     14.2401,     14.1263,   13.8432,
    12.8228,     11.3443,   9.54422,     7.49248,     5.76334,   4.33964,
    3.26467,     2.46652,   1.84357,     1.38562,     1.0139,    0.733563,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       2130.87,     2091.49,     2049.27,   2003.99,
    1955.44,     1903.38,   1847.56,     1787.71,     1723.53,   1654.71,
    1580.92,     1501.79,   1416.95,     1325.98,     1228.43,   1123.83,
    1011.67,     891.41,    762.457,     624.185,     475.921,   316.941,
    218.309,     209.567,   200.193,     190.141,     179.363,   167.806,
    155.413,     142.126,   127.877,     114.297,     108.887,   103.087,
    96.8671,     90.198,    83.0469,     77.5219,     73.2215,   68.6104,
    63.666,      59.3796,   55.5781,     51.5019,     47.8022,   44.7162,
    41.4072,     38.8564,   36.2696,     34.1586,     32.1912,   30.0315,
    27.797,      25.9894,   24.5446,     23.2247,     21.701,    20.3188,
    19.1085,     18.0082,   17.2345,     16.5161,     15.7336,   15.0082,
    14.5598,     14.0837,   13.6861,     13.1856,     13.1564,   13.0634,
    13.1201,     13.1858,   13.1788,     13.2561,     12.6357,   11.6949,
    10.0886,     8.14129,   6.20394,     4.67285,     3.49917,   2.60165,
    1.93454,     1.4419,    1.05336,     0.740321,    0.501465,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    2004.87,     1968.28,   1929.05,     1886.99,     1841.89,   1793.53,
    1741.67,     1686.07,   1626.45,     1562.52,     1493.97,   1420.46,
    1341.65,     1257.13,   1166.51,     1069.34,     965.153,   853.433,
    733.638,     605.187,   467.452,     319.764,     202.305,   194.636,
    186.413,     177.595,   168.141,     158.003,     147.132,   135.476,
    122.978,     109.576,   104.3,       98.7308,     92.7588,   86.3552,
    79.4889,     73.5002,   69.1506,     64.4868,     59.4859,   55.0436,
    51.7146,     48.145,    44.6285,     41.7263,     38.6144,   36.1172,
    33.8049,     31.7379,   29.9029,     28.0433,     26.1123,   24.5404,
    23.0286,     21.6082,   20.2368,     19.4145,     18.2599,   17.0564,
    16.1864,     15.446,    14.548,      14.067,      13.4462,   13.039,
    12.672,      12.2019,   12.0297,     11.8544,     12.0612,   12.1123,
    12.2041,     12.3393,   12.3768,     11.8262,     10.4511,   8.7979,
    6.81416,     5.11257,   3.75641,     2.77718,     2.05158,   1.50782,
    1.0789,      0.765539,  0.504591,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1882.83,   1849.03,
    1812.78,     1773.91,   1732.23,     1687.54,     1639.62,   1588.23,
    1533.13,     1474.05,   1410.71,     1342.78,     1269.94,   1191.85,
    1108.1,      1018.31,   922.023,     818.78,      708.077,   589.373,
    462.091,     325.611,   193.031,     185.679,     177.795,   169.341,
    160.277,     150.557,   140.135,     128.96,      116.977,   104.129,
    97.9197,     92.9499,   87.621,      81.907,      75.7801,   69.8482,
    65.5898,     61.0237,   56.1276,     51.4538,     48.427,    45.1815,
    41.8589,     39.7241,   37.435,      35.1933,     32.9804,   30.6426,
    28.2004,     26.0926,   24.4257,     23.2433,     21.9706,   20.2377,
    19.1531,     17.8268,   16.8415,     16.1201,     15.1107,   14.2555,
    13.4513,     13.2042,   12.4987,     11.9977,     11.6104,   11.3474,
    11.0704,     10.7826,   10.936,      10.9901,     11.0825,   11.4852,
    11.6553,     11.4908,   10.8193,     9.34472,     7.45128,   5.64922,
    4.07096,     2.98217,   2.16585,     1.56054,     1.11393,   0.784077,
    0.528856,    0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1748.95,     1718.07,     1684.95,   1649.43,
    1611.36,     1570.53,   1526.74,     1479.8,      1429.46,   1375.49,
    1317.61,     1255.55,   1189.01,     1117.65,     1041.15,   959.108,
    871.141,     776.817,   675.677,     567.228,     450.941,   326.251,
    192.549,     176.393,   169.017,     161.109,     152.628,   143.535,
    133.785,     123.331,   112.12,      100.1,       92.7736,   87.9267,
    82.7294,     77.1566,   71.1811,     64.9974,     61.3008,   57.3371,
    53.087,      48.6798,   46.1812,     43.5019,     40.6291,   38.0545,
    35.3292,     32.6967,   30.3845,     28.1093,     26.5466,   24.9166,
    23.2761,     21.8033,   20.4403,     19.2742,     18.1605,   17.0079,
    16.0386,     15.2817,   14.2797,     13.4453,     12.8091,   12.2274,
    11.6777,     11.0549,   10.8108,     10.5292,     10.1142,   10.0916,
    10.0648,     9.91957,   10.1741,     10.6697,     10.9836,   11.1603,
    10.8639,     9.72336,   8.04732,     6.12863,     4.45378,   3.23486,
    2.3248,      1.64812,   1.17356,     0.81417,     0.537629,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1685.17,     1655.75,   1624.2,      1590.37,     1554.1,    1515.21,
    1473.5,      1428.79,   1380.84,     1329.42,     1274.29,   1215.18,
    1151.79,     1083.82,   1010.94,     932.797,     849.004,   759.155,
    662.813,     559.508,   448.738,     329.963,     202.604,   166.618,
    159.84,      152.571,   144.778,     136.421,     127.46,    117.852,
    107.549,     96.5021,   88.418,      83.7533,     78.7515,   73.3882,
    67.6373,     61.4708,   57.8897,     54.3622,     50.5799,   46.5242,
    43.8696,     41.2149,   38.3683,     35.9497,     33.5489,   31.1728,
    29.4136,     27.5278,   25.5245,     23.5153,     22.2315,   20.8651,
    19.4118,     18.2182,   17.3533,     16.3709,     15.0095,   14.175,
    13.4328,     12.5729,   11.7719,     11.3683,     10.7736,   10.4251,
    10.0177,     9.63626,   9.45985,     9.14949,     9.09807,   9.2075,
    9.37286,     9.51461,   10.02,       10.4761,     10.5244,   10.0051,
    8.59599,     6.72678,   4.90894,     3.48262,     2.45847,   1.72757,
    1.23829,     0.846913,  0.567532,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1578.5,    1551.36,
    1522.26,     1491.06,   1457.6,      1421.72,     1383.25,   1342.0,
    1297.76,     1250.34,   1199.48,     1144.95,     1086.47,   1023.78,
    956.548,     884.46,    807.163,     724.28,      635.407,   540.112,
    437.929,     328.363,   210.878,     158.504,     152.243,   145.529,
    138.33,      130.611,   122.334,     113.459,     103.942,   93.7378,
    85.0589,     80.4796,   75.5693,     70.3041,     64.6585,   58.6048,
    54.8969,     51.893,    48.672,      45.2182,     42.3941,   39.686,
    36.7821,     34.3011,   32.0574,     29.679,      27.7357,   25.652,
    24.3134,     23.018,    21.3164,     19.7174,     18.5341,   17.4439,
    16.4103,     15.4057,   14.3227,     13.4412,     12.6908,   11.9238,
    11.3264,     10.7792,   10.1805,     9.77558,     9.51758,   8.99826,
    8.82765,     8.54501,   8.50661,     8.47161,     8.63717,   8.777,
    9.16819,     9.75246,   10.0885,     9.89422,     8.99844,   7.29748,
    5.3816,      3.82655,   2.68262,     1.88503,     1.30954,   0.903501,
    0.582914,    0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1509.71,     1484.1,      1456.64,   1427.2,
    1395.63,     1361.78,   1325.48,     1286.56,     1244.83,   1200.08,
    1152.1,      1100.64,   1045.48,     986.32,      922.889,   854.874,
    781.944,     703.743,   619.892,     529.98,      433.571,   330.194,
    219.347,     151.852,   145.909,     139.536,     132.703,   125.376,
    117.519,     109.095,   100.062,     90.3758,     81.1292,   76.9034,
    72.3723,     67.5137,   62.3039,     56.7177,     52.6091,   49.5681,
    46.3073,     42.8109,   39.9937,     37.6712,     35.1808,   32.7615,
    30.4831,     28.04,     26.3427,     24.6466,     23.0785,   21.5035,
    20.1296,     18.7797,   17.633,      16.5529,     15.5457,   14.6403,
    13.7855,     13.0758,   12.0565,     11.5231,     10.7185,   10.2742,
    9.62847,     9.08645,   8.81375,     8.47839,     8.32439,   7.92935,
    7.7346,      7.7196,    7.7893,      7.9783,      8.38407,   8.78759,
    9.37811,     9.65604,   9.17547,     7.83864,     5.93206,   4.22269,
    2.89912,     2.00075,   1.38647,     0.935048,    0.618794,  0.395423,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1447.18,     1422.98,   1397.04,     1369.21,     1339.38,   1307.39,
    1273.09,     1236.31,   1196.88,     1154.59,     1109.25,   1060.63,
    1008.49,     952.591,   892.65,      828.378,     759.46,    685.563,
    606.324,     521.36,    430.255,     332.567,     227.818,   146.95,
    141.174,     134.98,    128.339,     121.218,     113.582,   105.395,
    96.6157,     87.2021,   77.2035,     73.3991,     69.3199,   64.9458,
    60.2557,     55.2266,   50.9835,     48.009,      44.8197,   41.3998,
    38.3852,     36.0764,   33.6007,     31.224,      29.3669,   27.3755,
    25.5002,     23.5904,   21.9853,     20.6374,     19.4018,   18.1874,
    16.999,      15.8539,   14.8009,     13.9246,     13.0472,   12.3151,
    11.6108,     10.7888,   10.1351,     9.51641,     9.24432,   8.66913,
    8.27661,     7.90778,   7.54554,     7.44823,     7.28494,   7.23957,
    7.14411,     7.36864,   7.67661,     8.02047,     8.63118,   9.10488,
    9.23166,     8.30607,   6.48696,     4.69902,     3.18648,   2.1761,
    1.47875,     1.00263,   0.665898,    0.406923,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1414.29,   1390.95,
    1365.92,     1339.08,   1310.31,     1279.45,     1246.37,   1210.89,
    1172.85,     1132.07,   1088.33,     1041.43,     991.149,   937.229,
    879.413,     817.419,   750.944,     679.666,     603.236,   521.283,
    433.408,     339.182,   238.147,     143.733,     138.061,   131.978,
    125.456,     118.463,   110.964,     102.923,     94.3016,   85.0568,
    75.1438,     70.6913,   66.9733,     62.9865,     58.7117,   54.1279,
    49.81,       46.8144,   43.6023,     40.158,      36.9672,   34.9767,
    32.8423,     30.5971,   28.4705,     26.1902,     24.4897,   23.2444,
    21.9026,     20.4538,   18.9958,     17.5288,     16.4356,   15.3412,
    14.1324,     13.3403,   12.5337,     11.7843,     11.0588,   10.3251,
    9.67273,     9.19785,   8.74345,     8.28049,     7.89238,   7.57431,
    7.24315,     6.99591,   6.81031,     6.71975,     6.68483,   6.73592,
    6.91677,     7.3307,    7.84762,     8.45403,     8.92764,   8.53336,
    7.04252,     5.154,     3.52035,     2.35253,     1.58047,   1.05558,
    0.698052,    0.430097,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1347.22,     1325.33,     1301.86,   1276.69,
    1249.71,     1220.77,   1189.74,     1156.47,     1120.8,    1082.54,
    1041.53,     997.545,   950.385,     899.817,     845.595,   787.454,
    725.111,     658.263,   586.584,     509.725,     427.311,   338.942,
    244.187,     142.584,   134.846,     128.998,     122.727,   116.003,
    108.794,     101.063,   92.7732,     83.8846,     74.3537,   68.9904,
    65.3506,     61.4479,   57.2631,     52.7758,     48.2406,   45.4632,
    42.4851,     39.2917,   36.0353,     33.9112,     31.6337,   29.1915,
    27.3214,     25.3287,   23.6,        22.3355,     20.9859,   19.559,
    18.1607,     16.9118,   15.9293,     15.0018,     13.8122,   12.7561,
    12.0371,     11.4092,   10.6963,     10.0022,     9.3748,    8.89825,
    8.28103,     7.95249,   7.36756,     7.13153,     6.82584,   6.61702,
    6.42949,     6.26743,   6.24313,     6.30369,     6.39676,   6.6572,
    7.13236,     7.78868,   8.48986,     8.58436,     7.49068,   5.68392,
    3.89301,     2.62546,   1.72741,     1.13661,     0.743031,  0.461179,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1308.13,     1287.11,   1264.57,     1240.4,      1214.48,   1186.69,
    1156.89,     1124.94,   1090.68,     1053.94,     1014.55,   972.307,
    927.015,     878.45,    826.375,     770.537,     710.663,   646.463,
    577.623,     503.808,   424.659,     339.79,      248.788,   151.209,
    129.222,     123.792,   117.971,     111.728,     105.035,   97.8572,
    90.1612,     81.909,    73.0604,     67.0887,     63.6386,   59.9391,
    55.9722,     51.7187,   47.1577,     44.5162,     41.7594,   38.8033,
    35.6337,     33.315,    30.8746,     28.2579,     26.545,    24.9369,
    23.3223,     21.9052,   20.4118,     19.0857,     17.7316,   16.5597,
    15.3919,     14.2309,   13.2579,     12.491,      11.6814,   10.8211,
    10.2055,     9.47444,   9.04805,     8.56494,     7.96958,   7.56581,
    7.0844,      6.77558,   6.51275,     6.25226,     6.06425,   5.93852,
    5.89357,     5.84009,   5.90606,     6.12507,     6.51602,   7.15753,
    7.86517,     8.38885,   7.84667,     6.13011,     4.36944,   2.85987,
    1.89785,     1.22903,   0.777466,    0.496507,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1268.09,   1247.96,
    1226.37,     1203.23,   1178.41,     1151.8,      1123.27,   1092.68,
    1059.87,     1024.69,   986.976,     946.531,     903.164,   856.663,
    806.801,     753.336,   696.007,     634.536,     568.621,   497.944,
    422.158,     340.896,   253.761,     160.33,      125.254,   120.153,
    114.683,     108.818,   102.53,      95.7866,     88.5562,   80.8033,
    72.4901,     65.866,    62.2938,     58.4634,     54.3562,   49.9522,
    45.2299,     42.4223,   39.9312,     37.26,       34.3959,   32.4955,
    30.7418,     28.8614,   26.7969,     24.5598,     22.2987,   20.9708,
    19.5469,     18.3304,   17.064,      16.0402,     14.9073,   13.634,
    12.9874,     12.2787,   11.3841,     10.4245,     9.87464,   9.35721,
    8.79335,     8.10265,   7.70063,     7.28257,     6.8112,    6.44244,
    6.25179,     6.00266,   5.7159,      5.55833,     5.48675,   5.45737,
    5.42493,     5.65137,   6.00182,     6.46137,     7.22685,   7.8669,
    7.95543,     6.67043,   4.90025,     3.15014,     2.07361,   1.33031,
    0.850388,    0.526966,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1227.52,     1208.3,      1187.69,   1165.6,
    1141.9,      1116.5,    1089.26,     1060.05,     1028.73,   995.144,
    959.134,     920.521,   879.117,     834.722,     787.118,   736.074,
    681.341,     622.652,   559.723,     492.246,     419.892,   342.309,
    259.12,      169.919,   122.924,     117.93,      112.574,   106.832,
    100.674,     94.0719,   86.9924,     79.4012,     71.2615,   64.0406,
    60.9021,     57.5368,   53.9282,     50.0589,     45.91,     42.7114,
    39.9209,     36.9288,   33.7204,     31.3393,     29.3473,   27.2114,
    25.4235,     23.9624,   22.3956,     21.0001,     19.5139,   18.1486,
    16.7473,     15.5628,   14.4191,     13.5005,     12.5773,   11.7528,
    11.0746,     10.1749,   9.52282,     8.95844,     8.34988,   7.88989,
    7.37574,     7.04838,   6.57604,     6.30386,     5.94001,   5.69612,
    5.56106,     5.35046,   5.13147,     5.08957,     5.1407,    5.25168,
    5.46135,     5.84427,   6.57324,     7.31999,     7.8363,    7.05519,
    5.38566,     3.5462,    2.27932,     1.45253,     0.93096,   0.567857,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1217.09,     1198.22,   1177.98,     1156.27,     1133.0,    1108.05,
    1081.29,     1052.6,    1021.84,     988.851,     953.48,    915.554,
    874.886,     831.28,    784.522,     734.386,     680.626,   622.981,
    561.17,      494.892,   423.824,     347.621,     265.911,   178.295,
    119.024,     114.291,   109.217,     103.776,     97.9417,   91.6858,
    84.9778,     77.7851,   70.0725,     62.3934,     59.3093,   56.0022,
    52.4562,     48.6539,   44.5768,     41.4428,     39.2538,   36.9066,
    34.3898,     32.0267,   29.8207,     27.4552,     25.2824,   23.5516,
    21.6956,     20.2458,   18.8144,     17.5782,     16.424,    15.4309,
    14.4347,     13.2181,   12.3148,     11.6146,     10.805,    10.0706,
    9.56669,     8.85154,   8.18614,     7.77663,     7.26919,   6.78836,
    6.31271,     6.12555,   5.74602,     5.46843,     5.27276,   5.06119,
    4.85344,     4.8611,    4.79939,     4.9271,      5.04822,   5.43561,
    5.99035,     6.74833,   7.40363,     7.28718,     5.89356,   4.02466,
    2.52829,     1.57355,   0.976643,    0.607782,    0.352612,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1171.16,   1153.28,
    1134.12,     1113.56,   1091.52,     1067.89,     1042.55,   1015.38,
    986.241,     955.0,     921.501,     885.581,     847.066,   805.767,
    761.484,     714.0,     663.085,     608.491,     549.951,   487.18,
    419.874,     347.703,   270.316,     187.337,     118.674,   114.01,
    109.008,     103.646,   97.8956,     91.7299,     85.1186,   78.0295,
    70.4281,     62.2774,   59.0318,     55.7829,     52.2992,   48.5637,
    44.5583,     40.998,    38.4363,     35.6896,     32.7443,   30.1835,
    28.4947,     26.6838,   24.8529,     23.2517,     21.5349,   20.0248,
    18.5712,     17.2542,   16.0938,     14.991,      13.9024,   13.0299,
    12.1798,     11.355,    10.4842,     9.79455,     9.16136,   8.57889,
    8.06683,     7.51301,   6.91715,     6.6225,      6.26369,   5.82544,
    5.63645,     5.26226,   5.01014,     4.81444,     4.72696,   4.60196,
    4.50917,     4.54396,   4.71237,     4.94196,     5.41254,   6.17636,
    6.9542,      7.29105,   6.35972,     4.42618,     2.81269,   1.75634,
    1.07867,     0.656595,  0.376532,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1173.86,     1156.12,     1137.09,   1116.69,
    1094.82,     1071.36,   1046.21,     1019.24,     990.318,   959.31,
    926.06,      890.408,   852.179,     811.188,     767.234,   720.104,
    669.567,     615.379,   557.275,     494.971,     428.165,   356.532,
    279.721,     197.36,    116.858,     112.314,     107.441,   102.216,
    96.614,      90.6068,   84.1655,     77.2587,     69.8527,   61.9115,
    57.9476,     54.7516,   51.3246,     47.6499,     43.7097,   39.9931,
    37.7361,     35.316,    32.721,      30.1597,     28.1999,   26.0984,
    23.9449,     22.5985,   21.1548,     19.6845,     18.1792,   16.7798,
    15.6839,     14.6775,   13.7987,     12.8475,     11.8831,   10.9867,
    10.4174,     9.7374,    8.98321,     8.44932,     7.94494,   7.27758,
    6.88651,     6.48299,   6.08004,     5.78094,     5.39389,   5.08465,
    4.89459,     4.6508,    4.50264,     4.3796,      4.32705,   4.31063,
    4.40263,     4.66476,   4.93368,     5.57044,     6.44798,   7.21723,
    6.65945,     4.93604,   3.15003,     1.94143,     1.18425,   0.705612,
    0.403754,    0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1131.84,     1114.97,   1096.88,     1077.48,     1056.68,   1034.38,
    1010.47,     984.824,   957.328,     927.845,     896.231,   862.333,
    825.986,     787.011,   745.22,      700.409,     652.359,   600.837,
    545.592,     486.354,   422.835,     354.726,     281.695,   203.386,
    119.418,     110.624,   105.818,     100.666,     95.1406,   89.2164,
    82.864,      76.0526,   68.7489,     60.9174,     56.402,    53.4949,
    50.3776,     47.0351,   43.4511,     39.7689,     37.4798,   35.0252,
    32.3932,     29.6705,   27.8837,     25.9677,     23.9133,   22.4916,
    21.0055,     19.5716,   18.2947,     16.958,      15.6501,   14.4226,
    13.4853,     12.5331,   11.5842,     10.8017,     10.107,    9.37674,
    8.74825,     8.30025,   7.80169,     7.19515,     6.72452,   6.35387,
    5.87333,     5.5336,    5.24839,     4.92543,     4.7382,    4.53627,
    4.35501,     4.19152,   4.07751,     4.10597,     4.16081,   4.24161,
    4.62345,     5.09612,   5.88893,     6.73973,     6.86553,   5.47208,
    3.51807,     2.14777,   1.30944,     0.788162,    0.436565,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1139.5,    1122.64,
    1104.56,     1085.18,   1064.4,      1042.11,     1018.22,   992.599,
    965.126,     935.668,   904.081,     870.211,     833.893,   794.951,
    753.195,     708.421,   660.411,     608.932,     553.733,   494.544,
    431.078,     363.026,   290.055,     211.811,     127.913,   107.71,
    103.29,      98.5505,   93.4686,     88.0194,     82.1764,   75.9111,
    69.1931,     61.9896,   56.9442,     53.9508,     50.7411,   47.2995,
    43.6091,     39.652,    37.3646,     35.017,      32.4997,   29.8005,
    28.0406,     26.2249,   24.2779,     22.6176,     20.9382,   19.2714,
    17.9053,     16.4792,   15.4683,     14.4167,     13.4393,   12.4472,
    11.4498,     10.9618,   10.1833,     9.39093,     8.7788,    8.16458,
    7.63973,     7.01277,   6.49986,     6.11962,     5.87646,   5.47239,
    5.15219,     4.82435,   4.55832,     4.37737,     4.2258,    4.02117,
    3.92564,     3.92223,   3.92009,     4.063,       4.20316,   4.67762,
    5.40414,     6.29699,   6.85409,     6.00751,     3.9708,    2.39225,
    1.44614,     0.833528,  0.473461,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1138.55,     1121.91,     1104.06,   1084.93,
    1064.42,     1042.42,   1018.83,     993.537,     966.416,   937.336,
    906.154,     872.718,   836.866,     798.424,     757.203,   713.003,
    665.609,     614.79,    560.299,     501.869,     439.218,   372.038,
    300.003,     222.763,   139.941,     108.365,     103.94,    99.1944,
    94.106,      88.6499,   82.7995,     76.5264,     69.7998,   62.5872,
    56.812,      53.723,    50.4107,     46.859,      43.0507,   38.9671,
    36.5892,     34.5203,   32.3018,     29.923,      28.1158,   26.3662,
    24.49,       22.7043,   20.9021,     19.0661,     17.917,    16.6848,
    15.5267,     14.3133,   13.3767,     12.485,      11.7227,   10.8203,
    10.0221,     9.39369,   8.85191,     8.1658,      7.45556,   7.0343,
    6.62109,     6.14965,   5.72679,     5.33007,     5.0222,    4.74867,
    4.48368,     4.25421,   4.07494,     3.8798,      3.8008,    3.74254,
    3.73837,     3.84601,   3.96736,     4.34252,     4.91577,   5.84154,
    6.61182,     6.24444,   4.4832,      2.71953,     1.56364,   0.930376,
    0.520482,    0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1137.39,     1120.99,   1103.41,     1084.55,     1064.33,   1042.65,
    1019.4,      994.468,   967.738,     939.076,     908.342,   875.388,
    840.052,     802.162,   761.535,     717.971,     671.259,   621.171,
    567.464,     509.875,   448.125,     381.912,     310.914,   234.785,
    153.154,     110.494,   105.87,      100.912,     95.596,    89.8955,
    83.783,      77.2288,   70.2009,     62.6652,     56.0689,   53.4234,
    50.5867,     47.545,    44.2835,     40.7863,     38.0417,   35.6023,
    32.9866,     30.1819,   28.144,      26.4624,     24.6593,   22.939,
    21.2844,     19.5102,   18.1924,     16.7975,     15.6419,   14.4935,
    13.4606,     12.4176,   11.4429,     10.6513,     9.91985,   9.17879,
    8.39108,     8.10954,   7.42334,     7.00399,     6.5457,    6.08258,
    5.61897,     5.32208,   4.96659,     4.64505,     4.34662,   4.14123,
    3.90966,     3.81192,   3.6963,      3.60734,     3.60393,   3.57093,
    3.72507,     3.99964,   4.45083,     5.28678,     6.22813,   6.48556,
    5.01301,     3.04924,   1.77466,     1.02438,     0.568712,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1129.17,   1113.04,
    1095.75,     1077.2,    1057.31,     1035.99,     1013.13,   988.613,
    962.325,     934.138,   903.914,     871.505,     836.755,   799.493,
    759.538,     716.696,   670.758,     621.5,       568.682,   512.047,
    451.32,      386.203,   316.381,     241.514,     161.235,   108.192,
    103.868,     99.2312,   94.2597,     88.9289,     83.2129,   77.0839,
    70.5119,     63.4649,   56.5226,     53.6784,     50.6287,   47.3586,
    43.8522,     40.0924,   37.1101,     34.8463,     32.4188,   29.8159,
    27.5923,     25.7296,   23.7322,     21.9571,     20.6205,   19.1872,
    17.8178,     16.3838,   15.2825,     14.3358,     13.3529,   12.3501,
    11.6504,     10.7616,   9.85619,     9.31062,     8.62616,   7.88531,
    7.40385,     6.88816,   6.50073,     5.99589,     5.59986,   5.22435,
    4.92522,     4.67404,   4.38226,     4.07728,     3.84601,   3.76358,
    3.64136,     3.50399,   3.44448,     3.4639,      3.57603,   3.72118,
    4.16131,     4.8207,    5.81434,     6.45628,     5.48447,   3.50462,
    1.99307,     1.12314,   0.6278,      0.328559,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1112.5,      1096.8,      1079.98,   1061.93,
    1042.59,     1021.84,   999.6,       975.748,     950.173,   922.75,
    893.345,     861.815,   828.006,     791.754,     752.882,   711.201,
    666.508,     618.585,   567.198,     512.098,     453.017,   389.665,
    321.736,     248.897,   170.794,     108.284,     104.025,   99.4589,
    94.5626,     89.3123,   83.6827,     77.6462,     71.1734,   64.2329,
    56.7908,     53.8331,   50.7322,     47.4071,     43.8418,   40.0189,
    36.8246,     34.7881,   32.6045,     30.2631,     28.0637,   26.199,
    24.1997,     22.2992,   20.9552,     19.5142,     18.1419,   16.7472,
    15.4904,     14.3672,   13.2696,     12.1562,     11.4498,   10.6688,
    9.81439,     9.13206,   8.66068,     7.97248,     7.46753,   6.98152,
    6.49513,     5.93612,   5.51475,     5.23558,     4.8166,    4.57112,
    4.32661,     4.05661,   3.87967,     3.66532,     3.47696,   3.42068,
    3.33586,     3.31872,   3.3572,      3.57304,     3.83199,   4.41357,
    5.34199,     6.22115,   5.84818,     3.92052,     2.22423,   1.25304,
    0.689367,    0.358701,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1128.74,     1113.01,   1096.14,     1078.06,     1058.67,   1037.88,
    1015.59,     991.68,    966.047,     938.562,     909.09,    877.489,
    843.603,     807.27,    768.31,      726.535,     681.741,   633.709,
    582.207,     526.982,   467.767,     404.273,     336.19,    263.186,
    184.907,     111.347,   106.991,     102.32,      97.3124,   91.9425,
    86.1845,     80.0104,   73.3901,     66.2914,     58.6797,   55.0406,
    51.9446,     48.6248,   45.065,      41.2481,     37.759,    35.7124,
    33.518,      31.1649,   28.8023,     26.7382,     24.525,    22.2846,
    20.7792,     19.165,    17.8862,     16.8733,     15.7845,   14.6128,
    13.5062,     12.4735,   11.4807,     10.5944,     9.95617,   9.24073,
    8.61842,     7.92276,   7.27241,     6.8063,      6.35271,   6.00908,
    5.5492,      5.25126,   4.81066,     4.50393,     4.26821,   4.03609,
    3.82439,     3.58634,   3.43757,     3.3539,      3.23659,   3.20125,
    3.24866,     3.33597,   3.60997,     4.13454,     4.84605,   5.92289,
    6.10751,     4.38262,   2.5481,      1.401,       0.749687,  0.376612,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1149.48,   1133.61,
    1116.59,     1098.35,   1078.78,     1057.8,      1035.3,    1011.18,
    985.312,     957.577,   927.837,     895.948,     861.755,   825.09,
    785.776,     743.621,   698.419,     649.951,     597.98,    542.253,
    482.5,       418.427,   349.725,     276.057,     197.066,   112.367,
    107.502,     102.831,   97.8222,     92.4515,     86.6928,   80.5179,
    73.8968,     66.7972,   59.1845,     54.9119,     51.8887,   48.6469,
    45.1709,     41.4437,   37.7252,     35.5456,     33.2086,   30.7026,
    28.1744,     26.5706,   24.8508,     23.0072,     21.6164,   20.125,
    18.5485,     16.8884,   15.3799,     14.5694,     13.687,    12.7171,
    11.7735,     10.8604,   10.1046,     9.3433,      8.63498,   7.995,
    7.46406,     6.9145,    6.42867,     6.04349,     5.5141,    5.22496,
    4.92691,     4.52461,   4.28273,     4.02584,     3.69502,   3.61349,
    3.4425,      3.30545,   3.15943,     3.13314,     3.11886,   3.20405,
    3.41548,     3.74131,   4.4813,      5.54344,     6.17936,   4.86129,
    2.84134,     1.57131,   0.825938,    0.424487,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1143.43,     1127.75,     1110.93,   1092.89,
    1073.56,     1052.82,   1030.59,     1006.75,     981.189,   953.779,
    924.389,     892.875,   859.083,     822.85,      783.998,   742.338,
    697.667,     649.769,   598.408,     543.337,     484.285,   420.966,
    353.07,      280.269,   202.206,     118.501,     103.217,   99.0549,
    94.5919,     89.8064,   84.675,      79.1728,     73.273,    66.9468,
    60.1635,     55.5282,   52.4547,     49.1591,     45.6253,   41.8361,
    37.7878,     35.7266,   33.5166,     31.1468,     28.6058,   26.9094,
    25.0918,     23.1429,   21.736,      20.333,      18.8919,   17.4971,
    16.0667,     14.9941,   13.9098,     12.9639,     12.0013,   11.0259,
    10.1867,     9.44354,   8.73588,     8.05025,     7.49612,   7.07458,
    6.50101,     6.00174,   5.6441,      5.26632,     4.93897,   4.54407,
    4.17355,     3.98882,   3.78262,     3.5629,      3.41889,   3.22313,
    3.11746,     3.11634,   3.06921,     3.08214,     3.26213,   3.55162,
    4.11253,     5.0503,    6.04955,     5.29669,     3.26158,   1.74324,
    0.893764,    0.443541,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1149.27,     1133.75,   1117.1,      1099.25,     1080.11,   1059.58,
    1037.58,     1013.98,   988.677,     961.545,     932.453,   901.259,
    867.81,      831.944,   793.486,     752.249,     708.031,   660.618,
    609.779,     555.266,   496.813,     434.136,     366.929,   294.866,
    217.595,     134.739,   108.557,     104.039,     99.1954,   94.0014,
    88.432,      82.4602,   76.0567,     69.1905,     61.8282,   56.288,
    53.2004,     49.8898,   46.3399,     42.5334,     38.4519,   36.153,
    33.9895,     31.6696,   29.1821,     27.2942,     25.3914,   23.351,
    21.8937,     20.5959,   19.2133,     17.7744,     16.2315,   14.9442,
    13.6247,     12.7171,   11.8731,     11.1377,     10.3692,   9.61656,
    8.88389,     8.1666,    7.53786,     7.13606,     6.58746,   6.01642,
    5.66017,     5.20698,   4.92743,     4.55417,     4.26277,   3.96528,
    3.77175,     3.54305,   3.30439,     3.16395,     3.09919,   2.98512,
    2.95043,     3.01324,   3.10245,     3.3383,      3.77348,   4.62649,
    5.70226,     5.57724,   3.6612,      1.92338,     0.980943,  0.462396,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1141.19,   1125.91,
    1109.53,     1091.96,   1073.13,     1052.93,     1031.28,   1008.06,
    983.161,     956.464,   927.838,     897.143,     864.23,    828.938,
    791.096,     750.519,   707.009,     660.356,     610.33,    556.69,
    499.173,     437.5,     371.369,     300.46,      224.426,   142.897,
    106.682,     102.412,   97.8334,     92.9239,     87.6596,   82.0148,
    75.9621,     69.472,    62.5128,     56.6043,     53.4988,   50.1689,
    46.5983,     42.7697,   38.6644,     36.1092,     34.0109,   31.7609,
    29.3483,     27.457,    25.681,      23.7766,     22.0511,   20.4083,
    18.6803,     17.5483,   16.3346,     15.1754,     13.9544,   13.0681,
    12.1496,     11.2181,   10.4389,     9.67129,     8.91307,   8.26575,
    7.69097,     7.02504,   6.55265,     6.18975,     5.72869,   5.24851,
    4.96081,     4.57142,   4.29781,     3.9701,      3.72269,   3.50853,
    3.32653,     3.17918,   3.04371,     2.92683,     2.93535,   2.90878,
    2.97066,     3.17891,   3.55332,     4.2054,      5.3395,    5.80502,
    4.08907,     2.1577,    1.0724,      0.502857,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1145.9,      1130.71,     1114.42,   1096.95,
    1078.22,     1058.14,   1036.61,     1013.52,     988.762,   962.215,
    933.749,     903.226,   870.498,     835.404,     797.774,   757.425,
    714.16,      667.768,   618.023,     564.684,     507.49,    446.163,
    380.403,     309.892,   234.285,     153.214,     106.706,   102.484,
    97.9563,     93.1019,   87.8967,     82.3153,     76.3305,   69.9133,
    63.0323,     56.6527,   53.8115,     50.765,      47.4983,   43.9956,
    40.2397,     37.3207,   34.8887,     32.281,      29.4848,   27.3254,
    25.5562,     23.6593,   22.0394,     20.7573,     19.3826,   17.9638,
    16.4474,     15.3151,   14.278,      13.339,      12.3802,   11.4322,
    10.5959,     9.82389,   9.09138,     8.41166,     7.69026,   7.19544,
    6.70202,     6.26573,   5.70724,     5.37456,     4.91082,   4.63495,
    4.35199,     3.97212,   3.71402,     3.50473,     3.32455,   3.1923,
    3.04208,     2.88254,   2.89275,     2.80126,     2.88895,   2.99455,
    3.32234,     3.94011,   4.91133,     5.79326,     4.52359,   2.36958,
    1.16876,     0.530973,  0.215861,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1152.78,     1137.7,    1121.53,     1104.19,     1085.61,   1065.67,
    1044.3,      1021.38,   996.809,     970.459,     942.205,   911.91,
    879.425,     844.592,   807.242,     767.194,     724.25,    678.204,
    628.829,     575.887,   519.119,     458.248,     392.978,   322.991,
    247.947,     167.479,   110.818,     106.398,     101.658,   96.5753,
    91.1256,     85.282,    79.0161,     72.2974,     65.0931,   57.7825,
    54.8968,     51.8026,   48.4848,     44.9272,     41.1125,   37.9739,
    35.6097,     33.0747,   30.3565,     28.0375,     26.1981,   24.2257,
    22.4295,     21.0914,   19.6567,     18.3055,     16.9073,   15.6228,
    14.3835,     13.3936,   12.4602,     11.4101,     10.5692,   9.86419,
    8.99227,     8.38505,   7.85333,     7.42682,     6.73468,   6.39256,
    5.68709,     5.40024,   4.9244,      4.63332,     4.32703,   4.05263,
    3.68396,     3.49071,   3.36154,     3.07747,     2.97878,   2.83067,
    2.73844,     2.75141,   2.73117,     2.88455,     3.08858,   3.57867,
    4.50341,     5.59801,   4.88682,     2.66419,     1.23885,   0.527868,
    0.21238,     0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1141.28,   1126.47,
    1110.6,      1093.58,   1075.33,     1055.76,     1034.77,   1012.27,
    988.147,     962.276,   934.537,     904.792,     872.898,   838.699,
    802.028,     762.708,   720.546,     675.337,     626.861,   574.881,
    519.145,     459.382,   395.299,     326.585,     252.906,   173.902,
    108.096,     103.859,   99.3162,     94.4449,     89.2216,   83.6208,
    77.6153,     71.1757,   64.2708,     56.8669,     53.817,    50.7851,
    47.5341,     44.0481,   40.3102,     37.0935,     35.0315,   32.8205,
    30.4497,     28.2337,   26.4493,     24.5359,     22.6262,   21.0572,
    19.3748,     17.9956,   16.7351,     15.5065,     14.3206,   13.2739,
    12.3043,     11.4951,   10.7095,     9.96067,     9.23949,   8.6043,
    7.99885,     7.42021,   6.82527,     6.25623,     5.78913,   5.38216,
    5.06553,     4.58532,   4.29659,     4.0442,      3.74502,   3.43947,
    3.22119,     3.00121,   2.89427,     2.77174,     2.72198,   2.64854,
    2.64753,     2.71352,   2.91107,     3.30506,     4.08582,   5.19205,
    5.20884,     3.01138,   1.35431,     0.561742,    0.195663,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1117.92,     1103.58,     1088.21,   1071.73,
    1054.05,     1035.1,    1014.78,     992.991,     969.627,   944.574,
    917.711,     888.907,   858.021,     824.903,     789.392,   751.314,
    710.484,     666.704,   619.76,      569.424,     515.45,    457.575,
    395.518,     328.976,   257.626,     181.119,     107.507,   103.361,
    98.9147,     94.1473,   89.0353,     83.5539,     77.6763,   71.374,
    64.6163,     57.3702,   53.8335,     50.929,      47.8145,   44.475,
    40.8942,     37.4978,   35.2483,     32.8362,     30.2498,   27.7963,
    26.214,      24.5175,   22.7517,     21.299,      19.7413,   18.361,
    17.1317,     15.883,    14.6671,     13.6108,     12.7555,   11.7933,
    10.8368,     10.0027,   9.28444,     8.76968,     8.1522,    7.49718,
    6.92059,     6.33431,   5.81254,     5.39459,     4.9205,    4.64023,
    4.31257,     3.97483,   3.69943,     3.52443,     3.28677,   3.08076,
    2.8812,      2.76759,   2.59914,     2.554,       2.51126,   2.59966,
    2.69074,     3.04241,   3.72124,     4.78996,     5.31105,   3.37,
    1.47524,     0.579954,  0.189231,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1111.47,     1097.35,   1082.21,     1065.97,     1048.56,   1029.89,
    1009.87,     988.401,   965.383,     940.702,     914.238,   885.86,
    855.433,     822.806,   787.821,     750.308,     710.084,   666.954,
    620.706,     571.116,   517.943,     460.926,     399.79,    334.235,
    263.943,     188.571,   107.752,     102.385,     98.1596,   93.6288,
    88.7705,     83.5612,   77.9754,     71.9859,     65.5635,   58.6771,
    54.5206,     51.4336,   48.1236,     44.5743,     40.7686,   36.9602,
    34.8405,     32.5676,   30.1304,     27.6445,     25.9675,   24.1694,
    22.2414,     20.9596,   19.5942,     18.2118,     16.8449,   15.5305,
    14.5994,     13.5944,   12.5044,     11.54,       10.6766,   10.0734,
    9.30537,     8.61474,   8.02892,     7.46447,     6.88605,   6.32368,
    5.83061,     5.48266,   4.96537,     4.58201,     4.24389,   4.02473,
    3.76442,     3.38196,   3.22049,     3.0162,      2.80146,   2.65305,
    2.53127,     2.48249,   2.46618,     2.45039,     2.55436,   2.83905,
    3.33043,     4.27234,   5.16555,     3.68848,     1.65265,   0.614514,
    0.18568,     0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1078.14,   1064.57,
    1050.02,     1034.42,   1017.69,     999.749,     980.513,   959.887,
    937.77,      914.055,   888.626,     861.36,      832.123,   800.774,
    767.158,     731.114,   692.465,     651.022,     606.585,   558.936,
    507.844,     453.059,   394.316,     331.327,     263.786,   191.364,
    113.709,     99.2195,   95.1842,     90.8573,     86.2177,   81.2428,
    75.9084,     70.1885,   64.0552,     57.4787,     53.0481,   50.1983,
    47.1424,     43.8657,   40.3523,     36.5931,     34.4784,   32.2109,
    29.7795,     27.1724,   25.5551,     23.8254,     21.9706,   20.7123,
    19.4783,     18.2054,   16.9611,     15.6788,     14.6779,   13.6428,
    12.6608,     11.7168,   10.817,      10.1995,     9.3541,    8.59039,
    8.01919,     7.42309,   6.8476,      6.38086,     5.94949,   5.48352,
    5.01144,     4.68241,   4.22307,     3.87655,     3.70144,   3.33821,
    3.08191,     2.93737,   2.77016,     2.61031,     2.39579,   2.36247,
    2.34982,     2.36106,   2.36039,     2.58755,     3.01483,   3.87395,
    4.90487,     4.03122,   1.81677,     0.662258,    0.188327,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1062.06,     1048.84,     1034.66,   1019.46,
    1003.16,     985.676,   966.932,     946.835,     925.285,   902.177,
    877.4,       850.832,   822.344,     791.797,     759.043,   723.921,
    686.262,     645.881,   602.582,     556.154,     506.37,    452.989,
    395.75,      334.374,   268.563,     197.997,     122.33,    99.0461,
    95.0329,     90.7296,   86.1154,     81.1677,     75.8625,   70.1738,
    64.0741,     57.5335,   52.6067,     49.7799,     46.7488,   43.4987,
    40.0137,     36.2768,   34.0832,     31.9766,     29.7177,   27.2956,
    25.5142,     23.7231,   21.8026,     20.464,      19.2804,   18.0417,
    16.856,      15.5862,   14.5351,     13.4433,     12.5482,   11.6264,
    10.6866,     10.2066,   9.55346,     8.78003,     8.04489,   7.37429,
    6.87402,     6.35183,   5.90864,     5.4185,      5.03363,   4.59537,
    4.11503,     3.86889,   3.61125,     3.34575,     3.14705,   2.89395,
    2.72635,     2.49296,   2.40638,     2.23635,     2.17026,   2.23651,
    2.23598,     2.43151,   2.74967,     3.47962,     4.55702,   4.32334,
    2.04738,     0.733644,  0.193433,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1055.86,     1042.85,   1028.91,     1013.96,     997.924,   980.734,
    962.301,     942.536,   921.342,     898.617,     874.25,    848.122,
    820.106,     790.065,   757.853,     723.313,     686.277,   646.564,
    603.982,     558.322,   509.363,     456.865,     400.574,   340.215,
    275.493,     206.095,   131.681,     99.7023,     95.5244,   91.0447,
    86.2412,     81.0906,   75.5678,     69.6458,     63.2959,   56.4871,
    50.9824,     48.5499,   45.9415,     43.1447,     40.1458,   36.9301,
    34.527,      32.2821,   29.875,      27.294,      25.4969,   23.892,
    22.1711,     20.6788,   19.2945,     17.8425,     16.7902,   15.6619,
    14.589,      13.4569,   12.3924,     11.423,      10.7459,   10.1021,
    9.40346,     8.69094,   8.1241,      7.45262,     6.93576,   6.39062,
    5.90388,     5.41328,   4.98464,     4.64719,     4.22135,   3.87669,
    3.61645,     3.28288,   3.05305,     2.83911,     2.59503,   2.44435,
    2.31609,     2.14423,   2.11857,     2.11294,     2.11131,   2.22041,
    2.50711,     3.10505,   4.13883,     4.42123,     2.35918,   0.863523,
    0.22945,     0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1015.61,   1003.23,
    989.951,     975.717,   960.453,     944.087,     926.538,   907.721,
    887.543,     865.908,   842.709,     817.834,     791.161,   762.56,
    731.893,     699.009,   663.749,     625.94,      585.399,   541.929,
    495.317,     445.337,   391.744,     334.279,     272.661,   206.589,
    135.743,     96.8695,   93.0095,     88.8706,     84.4325,   79.6737,
    74.571,      69.0996,   63.2327,     56.9419,     51.2162,   48.5937,
    45.7816,     42.7663,   39.5331,     36.0663,     33.4773,   31.338,
    29.0441,     26.5844,   24.7929,     23.3662,     21.8363,   20.4212,
    19.1274,     17.7401,   16.6441,     15.4928,     14.359,    13.1747,
    12.3502,     11.5389,   10.7062,     9.99169,     9.3321,    8.693,
    8.06021,     7.5512,    7.01145,     6.38911,     5.96484,   5.51476,
    5.0597,      4.68402,   4.24901,     3.89419,     3.60255,   3.28159,
    3.05247,     2.72313,   2.60139,     2.35802,     2.22547,   2.13857,
    2.03613,     1.99108,   2.00927,     2.0882,      2.25782,   2.73378,
    3.62263,     4.44271,   2.70302,     0.948709,    0.27888,   0.0184032,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1008.07,     995.885,     982.82,    968.811,
    953.79,      937.683,   920.411,     901.892,     882.035,   860.742,
    837.911,     813.429,   787.179,     759.031,     728.849,   696.486,
    661.785,     624.575,   584.677,     541.895,     496.021,   446.833,
    394.089,     337.534,   276.892,     211.867,     142.144,   95.5352,
    91.762,      87.7162,   83.378,      78.7262,     73.7383,   68.39,
    62.6551,     56.5058,   50.4155,     47.9374,     45.2802,   42.4311,
    39.376,      36.1001,   33.408,      31.2739,     28.9856,   26.5319,
    24.5648,     23.0841,   21.4964,     19.9927,     18.6974,   17.3084,
    16.2288,     15.1597,   14.1691,     13.1932,     12.3692,   11.5583,
    10.6975,     9.94356,   9.28107,     8.66401,     8.08805,   7.63918,
    6.91527,     6.39028,   5.99331,     5.52329,     5.08662,   4.66398,
    4.21042,     3.8457,    3.61513,     3.2023,      3.07324,   2.67624,
    2.56471,     2.34082,   2.17558,     2.04458,     1.96791,   1.90645,
    1.87001,     1.93846,   2.08905,     2.44459,     3.33917,   4.28071,
    3.10149,     1.12392,   0.321738,    0.0352306,   0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1013.11,     1000.94,   987.903,     973.919,     958.925,   942.848,
    925.608,     907.122,   887.301,     866.047,     843.258,   818.821,
    792.618,     764.522,   734.395,     702.091,     667.453,   630.311,
    590.485,     547.782,   501.992,     452.893,     400.245,   343.793,
    283.262,     218.356,   148.759,     93.9412,     90.3547,   86.509,
    82.3854,     77.9638,   73.2227,     68.1389,     62.6878,   56.8427,
    50.5752,     48.0179,   45.284,      42.3525,     39.2091,   35.8386,
    32.9992,     31.0883,   29.0393,     26.8422,     24.8755,   23.3552,
    21.725,      20.1461,   18.9082,     17.5809,     16.3145,   15.0226,
    13.9482,     13.0755,   12.3056,     11.5739,     10.7241,   9.94951,
    9.26957,     8.52689,   7.92525,     7.45774,     6.88556,   6.34306,
    5.95777,     5.55294,   5.09243,     4.67472,     4.26572,   3.89327,
    3.53489,     3.26308,   2.93999,     2.69652,     2.53496,   2.29831,
    2.11293,     1.97575,   1.88891,     1.84546,     1.77608,   1.79018,
    1.91694,     2.24554,   2.93702,     4.00116,     3.47107,   1.33761,
    0.397911,    0.0572314, 0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1002.3,    990.386,
    977.613,     963.917,   949.231,     933.484,     916.598,   898.493,
    879.079,     858.262,   835.94,      812.006,     786.342,   758.823,
    729.315,     697.675,   663.749,     627.371,     588.363,   546.537,
    501.688,     453.598,   402.033,     346.741,     287.454,   223.882,
    155.715,     93.8185,   90.1632,     86.2437,     82.041,    77.5345,
    72.7024,     67.521,    61.9652,     56.008,      49.6202,   46.8679,
    44.4781,     41.9155,   39.1678,     36.2214,     33.4255,   31.3075,
    29.0366,     26.6014,   24.3736,     22.9492,     21.422,    19.8525,
    18.5326,     17.1173,   15.9028,     14.8128,     13.7848,   12.8833,
    12.0381,     11.2425,   10.6388,     9.95271,     9.07817,   8.275,
    7.81916,     7.41391,   6.81085,     6.34017,     5.94596,   5.51929,
    5.16387,     4.68123,   4.28869,     3.9218,      3.60652,   3.23712,
    3.00932,     2.69958,   2.4822,      2.27564,     2.1122,    1.98091,
    1.87168,     1.67869,   1.73013,     1.70138,     1.81142,   2.00756,
    2.63035,     3.7088,    3.69804,     1.6207,      0.469482,  0.085908,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1019.65,     1007.68,     994.854,   981.097,
    966.346,     950.529,   933.568,     915.383,     895.883,   874.973,
    852.553,     828.512,   802.734,     775.094,     745.455,   713.675,
    679.598,     643.058,   603.878,     561.867,     516.819,   468.516,
    416.722,     361.185,   301.635,     237.781,     169.312,   98.9268,
    94.9089,     90.6006,   85.981,      81.0275,     75.7161,   70.0208,
    63.914,      57.3658,   50.3444,     46.8598,     44.3798,   41.7206,
    38.8691,     35.8116,   32.8363,     31.0241,     29.0809,   26.9973,
    24.899,      23.3004,   21.5863,     19.7846,     18.5815,   17.2915,
    16.0515,     14.881,    13.7799,     12.9598,     12.0825,   11.1448,
    10.4405,     9.80003,   9.14489,     8.49519,     7.83579,   7.26448,
    6.81442,     6.39002,   5.90108,     5.44,        5.16297,   4.68604,
    4.38691,     3.96804,   3.58948,     3.28656,     2.96517,   2.74171,
    2.47266,     2.26353,   2.09835,     1.97268,     1.79153,   1.68826,
    1.65131,     1.60691,   1.66697,     1.88464,     2.3364,    3.29743,
    3.81533,     1.96002,   0.58206,     0.122064,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1017.12,     1005.27,   992.559,     978.932,     964.322,   948.655,
    931.856,     913.843,   894.528,     873.818,     851.611,   827.798,
    802.266,     774.888,   745.531,     714.053,     680.3,     644.108,
    605.3,       563.688,   519.068,     471.224,     419.923,   364.914,
    305.929,     242.682,   174.865,     102.146,     93.2868,   89.2476,
    84.9164,     80.2722,   75.2924,     69.9527,     64.2271,   58.0878,
    51.5048,     47.5683,   45.0413,     42.3316,     39.4261,   36.3106,
    33.0902,     31.2275,   29.2301,     27.0884,     24.8497,   23.3162,
    21.6718,     19.9086,   18.6333,     17.3093,     16.0477,   14.9742,
    13.8812,     12.9599,   12.0555,     11.2828,     10.5235,   9.75874,
    8.99244,     8.45489,   7.8478,      7.28558,     6.91216,   6.4539,
    6.03095,     5.56206,   5.12004,     4.64777,     4.36181,   4.00579,
    3.66051,     3.34033,   3.05113,     2.77371,     2.50219,   2.25745,
    2.12058,     1.88806,   1.78163,     1.61297,     1.58415,   1.58358,
    1.61113,     1.74364,   2.16911,     2.94369,     3.81843,   2.25116,
    0.704095,    0.165241,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1010.17,   998.49,
    985.968,     972.541,   958.144,     942.706,     926.153,   908.403,
    889.371,     868.963,   847.08,      823.616,     798.457,   771.479,
    742.551,     711.533,   678.274,     642.611,     604.37,    563.366,
    519.399,     472.255,   421.703,     367.498,     309.376,   247.054,
    180.228,     108.572,   92.0021,     88.2043,     84.132,    79.7654,
    75.0833,     70.0628,   64.6794,     58.9071,     52.7176,   48.4124,
    45.8187,     43.0375,   40.0553,     36.8577,     33.4289,   31.4076,
    29.3108,     27.0624,   24.6515,     23.076,      21.4387,   19.6831,
    18.4636,     17.3027,   16.1442,     15.1599,     14.1217,   13.21,
    12.2418,     11.2442,   10.4631,     9.88706,     8.94805,   8.34118,
    7.86252,     7.28579,   6.67439,     6.26323,     5.83602,   5.4438,
    5.10012,     4.67002,   4.33476,     4.01345,     3.68144,   3.34735,
    3.11238,     2.75311,   2.50441,     2.27942,     2.10589,   1.93063,
    1.76265,     1.64903,   1.53295,     1.50317,     1.52235,   1.60998,
    1.88729,     2.67564,   3.68646,     2.59001,     0.859421,  0.204799,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1015.44,     1003.82,     991.366,   978.011,
    963.691,     948.336,   931.872,     914.217,     895.287,   874.988,
    853.223,     829.885,   804.86,      778.027,     749.255,   718.403,
    685.322,     649.85,    611.814,     571.03,      527.299,   480.407,
    430.126,     376.212,   318.402,     256.413,     189.945,   118.674,
    94.1339,     90.1944,   85.9701,     81.4405,     76.5836,   71.3757,
    65.7915,     59.8037,   53.3832,     48.506,      45.997,    43.3068,
    40.4221,     37.3289,   34.0123,     31.8872,     29.8712,   27.7096,
    25.3917,     23.7005,   22.0419,     20.2634,     18.8954,   17.6532,
    16.3582,     15.1923,   13.9421,     12.9605,     11.9435,   11.1348,
    10.3552,     9.64816,   8.97076,     8.33753,     7.79596,   7.33501,
    6.71833,     6.08471,   5.79025,     5.42374,     5.1478,    4.79694,
    4.4628,      4.07581,   3.74878,     3.40102,     3.06818,   2.82864,
    2.50539,     2.22838,   2.04713,     1.8588,      1.76073,   1.59893,
    1.51695,     1.49767,   1.43353,     1.5479,      1.75185,   2.35785,
    3.41261,     3.04096,   1.06185,     0.27451,     0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1029.38,     1017.68,   1005.14,     991.686,     977.264,   961.801,
    945.22,      927.44,    908.376,     887.934,     866.015,   842.511,
    817.309,     790.286,   761.31,      730.24,      696.925,   661.202,
    622.897,     581.824,   537.783,     490.56,      439.923,   385.627,
    327.408,     264.981,   198.042,     126.267,     93.6135,   89.7323,
    85.5707,     81.1083,   76.3234,     71.1927,     65.6912,   59.7922,
    53.4668,     48.1589,   45.6696,     43.0005,     40.1384,   37.0694,
    33.7787,     31.5571,   29.6525,     27.6102,     25.4203,   23.6983,
    22.0901,     20.3657,   18.8866,     17.5532,     16.1449,   15.2277,
    14.2442,     13.1434,   11.9554,     11.2061,     10.4474,   9.70258,
    8.95041,     8.29666,   7.76279,     7.1776,      6.73901,   6.17912,
    5.72271,     5.49412,   5.11515,     4.59785,     4.35758,   4.10224,
    3.76484,     3.41466,   3.03475,     2.81184,     2.52922,   2.30406,
    2.0673,      1.88904,   1.74979,     1.61397,     1.47725,   1.4496,
    1.38359,     1.44386,   1.64379,     2.11937,     3.13944,   3.30841,
    1.29293,     0.333078,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1047.18,   1035.38,
    1022.72,     1009.15,   994.605,     979.005,     962.277,   944.341,
    925.108,     904.485,   882.372,     858.661,     833.237,   805.975,
    776.743,     745.398,   711.788,     675.749,     637.106,   595.671,
    551.24,      503.599,   452.515,     397.74,      339.006,   276.027,
    208.497,     136.087,   95.1432,     91.3547,     87.2925,   82.9368,
    78.2662,     73.2582,   67.8882,     62.1301,     55.9559,   50.2541,
    47.6417,     44.8404,   41.8367,     38.6159,     35.1623,   32.59,
    30.5197,     28.2997,   25.9193,     24.0949,     22.5969,   20.9906,
    19.529,      18.2389,   16.8556,     15.6532,     14.3869,   13.2733,
    12.163,      11.3001,   10.4746,     9.7864,      9.03706,   8.2981,
    7.67256,     7.25947,   6.68397,     6.17062,     5.82244,   5.4411,
    5.01175,     4.61179,   4.42832,     4.07741,     3.74509,   3.48703,
    3.13904,     2.83086,   2.54958,     2.29833,     2.11744,   1.91435,
    1.74601,     1.59939,   1.48174,     1.3936,      1.36135,   1.38863,
    1.52737,     1.92702,   2.84511,     3.46228,     1.53266,   0.423867,
    0.0827228,   0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1043.05,     1031.41,     1018.92,   1005.53,
    991.174,     975.778,   959.271,     941.57,      922.59,    902.239,
    880.417,     857.018,   831.928,     805.024,     776.177,   745.245,
    712.077,     676.512,   638.377,     597.487,     553.641,   506.627,
    456.215,     402.16,    344.198,     282.048,     215.406,   143.949,
    95.5545,     91.7482,   87.6667,     83.2903,     78.5977,   73.5659,
    68.1705,     62.3851,   56.1817,     49.9912,     47.414,    44.6506,
    41.6875,     38.5102,   35.1033,     32.4058,     30.4211,   28.293,
    26.011,      24.1075,   22.5994,     20.9822,     19.4381,   18.0966,
    16.658,      15.4718,   14.2816,     13.3577,     12.57,     11.55,
    10.4564,     9.94245,   9.26379,     8.49209,     7.82237,   7.26621,
    6.7682,      6.18128,   5.7256,      5.39179,     5.00372,   4.67347,
    4.28985,     4.09663,   3.76712,     3.59339,     3.23576,   2.92171,
    2.59974,     2.34104,   2.13106,     1.85139,     1.7107,    1.57909,
    1.42942,     1.37383,   1.31381,     1.33667,     1.45282,   1.76676,
    2.53615,     3.46878,   1.86499,     0.522477,    0.11365,   0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1043.4,      1031.88,   1019.53,     1006.28,     992.081,   976.851,
    960.52,      943.01,    924.234,     904.101,     882.513,   859.365,
    834.544,     807.929,   779.391,     748.791,     715.979,   680.796,
    643.071,     602.619,   559.244,     512.734,     462.863,   409.388,
    352.048,     290.565,   224.639,     153.948,     98.1713,   94.2806,
    90.1088,     85.6356,   80.839,      75.6958,     70.181,    64.2676,
    57.9268,     51.1278,   48.6518,     46.0088,     43.1747,   40.1358,
    36.8773,     34.0113,   31.859,      29.5512,     27.0766,   24.839,
    23.0699,     21.173,    19.3954,     18.181,      16.8788,   15.7297,
    14.6016,     13.5207,   12.4773,     11.603,      10.8041,   9.95167,
    9.21781,     8.60776,   7.79829,     7.16116,     6.66226,   6.19226,
    5.78492,     5.44728,   5.00478,     4.67277,     4.31755,   4.05672,
    3.77897,     3.4774,    3.18822,     2.93026,     2.59944,   2.37439,
    2.14888,     1.92671,   1.77174,     1.61361,     1.42165,   1.31695,
    1.28176,     1.28895,   1.32251,     1.61706,     2.26158,   3.36471,
    2.16484,     0.628377,  0.149605,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1062.38,   1050.71,
    1038.2,      1024.77,   1010.38,     994.952,     978.406,   960.664,
    941.64,      921.241,   899.367,     875.914,     850.765,   823.799,
    794.884,     763.879,   730.634,     694.986,     656.762,   615.776,
    571.827,     524.703,   474.173,     419.992,     361.895,   299.599,
    232.802,     161.177,   96.7544,     93.1069,     89.1958,   85.002,
    80.5052,     75.6834,   70.5132,     64.9693,     59.0248,   52.6506,
    49.5097,     46.61,     43.5007,     40.1668,     36.5919,   33.4044,
    31.4943,     29.446,    27.2498,     25.1581,     23.5535,   21.8329,
    20.1224,     18.9651,   17.7242,     16.4082,     15.0071,   13.7647,
    12.7919,     11.8139,   10.8227,     10.017,      9.28481,   8.65661,
    7.9268,      7.33756,   6.80142,     6.31908,     5.83572,   5.45095,
    5.02579,     4.7253,    4.40848,     4.07078,     3.76758,   3.481,
    3.19402,     2.97604,   2.68055,     2.40217,     2.18427,   1.98284,
    1.78683,     1.59587,   1.43052,     1.33416,     1.29292,   1.2325,
    1.30949,     1.49633,   2.00386,     3.15526,     2.52924,   0.779045,
    0.187482,    0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1062.93,     1051.4,      1039.03,   1025.77,
    1011.56,     996.314,   979.968,     962.441,     943.647,   923.496,
    901.887,     878.718,   853.874,     827.234,     798.669,   768.04,
    735.198,     699.982,   662.221,     621.731,     578.316,   531.762,
    481.844,     428.319,   370.926,     309.385,     243.397,   172.64,
    101.092,     97.1072,   92.8342,     88.2524,     83.3396,   78.0717,
    72.4231,     66.3662,   59.8717,     52.9078,     49.3428,   46.6257,
    43.7123,     40.5883,   37.2386,     34.0313,     32.1704,   30.1751,
    28.0355,     25.8829,   24.1855,     22.3656,     20.4556,   19.0268,
    17.4948,     16.1328,   14.9649,     13.8098,     12.7824,   11.8148,
    10.9584,     10.1365,   9.3736,      8.7999,      8.05983,   7.42146,
    6.96502,     6.43264,   5.81323,     5.40802,     4.98003,   4.6461,
    4.32498,     3.93844,   3.68563,     3.49541,     3.25545,   2.93245,
    2.73975,     2.4991,    2.24807,     1.95189,     1.77126,   1.57288,
    1.468,       1.2889,    1.23828,     1.17826,     1.20864,   1.3621,
    1.82873,     2.81417,   2.82104,     0.93903,     0.225343,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1094.69,     1082.88,   1070.21,     1056.62,     1042.05,   1026.43,
    1009.68,     991.724,   972.466,     951.817,     929.675,   905.933,
    880.476,     853.179,   823.909,     792.524,     758.87,    722.785,
    684.092,     642.603,   598.115,     550.412,     499.263,   444.416,
    385.606,     322.546,   254.929,     182.425,     104.682,   97.227,
    93.1153,     88.7064,   83.9789,     78.9098,     73.4744,   67.6461,
    61.3966,     54.6956,   50.6302,     47.7905,     44.7456,   41.4806,
    37.9797,     34.4137,   32.415,      30.2719,     27.9739,   25.6039,
    24.068,      22.4211,   20.6552,     19.3778,     18.0347,   16.6834,
    15.3761,     14.0926,   13.1572,     12.1899,     11.2283,   10.4008,
    9.61512,     8.73739,   8.07814,     7.47202,     6.91493,   6.43375,
    5.9268,      5.51012,   5.09875,     4.67821,     4.26537,   3.96626,
    3.68774,     3.433,     3.31701,     3.02401,     2.79334,   2.46735,
    2.23175,     2.02805,   1.80722,     1.62488,     1.43467,   1.32026,
    1.24896,     1.16394,   1.19534,     1.33319,     1.65418,   2.59213,
    3.07886,     1.14827,   0.284792,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1113.72,   1101.81,
    1089.04,     1075.35,   1060.67,     1044.93,     1028.04,   1009.94,
    990.535,     969.723,   947.407,     923.479,     897.821,   870.309,
    840.809,     809.177,   775.259,     738.89,      699.893,   658.077,
    613.24,      565.162,   513.61,      458.332,     399.06,    335.504,
    267.355,     194.281,   115.926,     100.213,     95.9675,   91.4156,
    86.5347,     81.3011,   75.6893,     69.6719,     63.2197,   56.3012,
    51.6989,     48.912,    45.9236,     42.7193,     39.2834,   35.5992,
    33.5308,     31.3283,   28.9666,     26.4342,     24.7569,   22.9756,
    21.0657,     19.6732,   18.2946,     16.9121,     15.6735,   14.39,
    13.3762,     12.3432,   11.4294,     10.5159,     9.61121,   8.95662,
    8.29826,     7.67493,   7.08148,     6.50564,     6.00566,   5.48579,
    5.07959,     4.65123,   4.31241,     3.99337,     3.70808,   3.46312,
    3.24405,     3.01339,   2.81014,     2.57065,     2.33047,   2.05201,
    1.82178,     1.62139,   1.47445,     1.31083,     1.22608,   1.15287,
    1.14579,     1.22974,   1.50605,     2.26552,     3.13575,   1.36533,
    0.352994,    0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1097.96,     1086.33,     1073.87,   1060.51,
    1046.18,     1030.81,   1014.34,     996.668,     977.725,   957.412,
    935.632,     912.278,   887.235,     860.384,     831.591,   800.718,
    767.614,     732.117,   694.055,     653.243,     609.481,   562.556,
    512.241,     458.289,   400.438,     338.407,     271.893,   200.572,
    124.097,     100.821,   96.6419,     92.1613,     87.3569,   82.2053,
    76.6814,     70.7583,   64.4071,     57.597,      52.4907,   49.5549,
    46.407,      43.0315,   39.4121,     35.5311,     33.3857,   31.3466,
    29.1601,     26.8155,   24.9997,     23.1505,     21.1676,   19.7137,
    18.384,      17.0295,   15.9007,     14.6925,     13.6226,   12.5116,
    11.5932,     10.6757,   9.776,       9.12133,     8.42596,   7.7557,
    7.15432,     6.47899,   6.01038,     5.49935,     5.12886,   4.5905,
    4.29499,     3.96374,   3.6585,      3.41445,     3.21925,   2.97289,
    2.80405,     2.55157,   2.33557,     2.06151,     1.85081,   1.67056,
    1.4961,      1.29969,   1.20356,     1.16551,     1.13246,   1.15711,
    1.37736,     2.07403,   3.15846,     1.67368,     0.422759,  0.0,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1123.97,     1112.13,   1099.44,     1085.84,     1071.24,   1055.6,
    1038.82,     1020.83,   1001.54,     980.86,      958.682,   934.902,
    909.403,     882.061,   852.744,     821.307,     787.599,   751.455,
    712.699,     671.142,   626.582,     578.801,     527.568,   472.632,
    413.726,     350.563,   282.836,     210.214,     132.344,   100.695,
    96.6091,     92.2281,   87.5304,     82.4933,     77.0922,   71.3008,
    65.0908,     58.432,    53.0405,     50.3454,     47.4556,   44.3568,
    41.0342,     37.4714,   34.9566,     32.6278,     30.1307,   27.4532,
    25.4625,     23.5878,   21.5775,     20.0812,     18.8448,   17.5403,
    16.3854,     15.147,    13.9935,     12.7815,     11.8255,   10.8759,
    10.0029,     9.3526,    8.58881,     7.79701,     7.34647,   6.71795,
    6.04658,     5.64796,   5.1761,      4.74316,     4.39692,   4.0589,
    3.6692,      3.39962,   3.15327,     3.00722,     2.81922,   2.62002,
    2.36041,     2.11352,   1.85019,     1.67576,     1.533,     1.3365,
    1.21054,     1.12583,   1.07239,     1.12567,     1.31214,   1.8421,
    2.93549,     1.94847,   0.522403,    0.0,         0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1142.91,   1130.99,
    1118.2,      1104.49,   1089.79,     1074.03,     1057.13,   1039.0,
    1019.57,     998.73,    976.386,     952.427,     926.737,   899.19,
    869.652,     837.98,    804.019,     767.603,     728.556,   686.688,
    641.793,     593.654,   542.036,     486.688,     427.34,    363.703,
    295.468,     222.301,   143.846,     103.796,     99.5983,   95.0975,
    90.2715,     85.0968,   79.548,      73.5983,     67.2186,   60.3778,
    54.3104,     51.3913,   48.2612,     44.905,      41.3061,   37.4472,
    34.7269,     32.4871,   30.0854,     27.5102,     25.6496,   24.1009,
    22.4403,     20.8642,   19.3491,     17.7245,     16.5368,   15.2741,
    14.1171,     12.9257,   11.9095,     10.9057,     10.0215,   9.3096,
    8.68998,     8.09509,   7.30881,     6.69121,     6.16553,   5.67191,
    5.28222,     4.83783,   4.43462,     4.04097,     3.68603,   3.48052,
    3.20115,     2.96935,   2.76608,     2.60067,     2.38151,   2.14927,
    1.94099,     1.70625,   1.50636,     1.33312,     1.19545,   1.14973,
    1.07428,     1.0948,    1.22845,     1.66506,     2.74757,   2.27549,
    0.622965,    0.0,       0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1124.22,     1112.61,     1100.17,   1086.83,
    1072.53,     1057.19,   1040.75,     1023.11,     1004.2,    983.927,
    962.186,     938.874,   913.877,     887.074,     858.334,   827.517,
    794.472,     759.04,    721.047,     680.309,     636.626,   589.787,
    539.563,     485.709,   427.963,     366.044,     299.651,   228.459,
    152.123,     105.182,   100.913,     96.3354,     91.4269,   86.1637,
    80.5202,     74.4688,   67.9801,     61.0224,     54.3966,   51.6093,
    48.6204,     45.4156,   41.9792,     38.2944,     35.4701,   33.2756,
    30.9225,     28.3994,   26.3422,     24.6321,     22.7984,   21.1107,
    19.6585,     18.1012,   16.9511,     15.7911,     14.6104,   13.3718,
    12.3307,     11.3217,   10.4815,     9.68639,     8.91034,   8.13063,
    7.42733,     6.84442,   6.30161,     5.80027,     5.34807,   4.85418,
    4.47224,     4.12112,   3.77152,     3.39859,     3.23235,   2.90044,
    2.83185,     2.57286,   2.393,       2.21729,     1.95875,   1.75772,
    1.52505,     1.40696,   1.22844,     1.15323,     1.04399,   1.05331,
    1.1562,      1.49543,   2.47188,     2.56441,     0.772587,  0.1681,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1143.58,     1131.87,   1119.32,     1105.86,     1091.42,   1075.95,
    1059.36,     1041.56,   1022.48,     1002.03,     980.09,    956.569,
    931.348,     904.304,   875.306,     844.213,     810.872,   775.122,
    736.788,     695.684,   651.61,      604.35,      553.676,   499.339,
    441.075,     378.601,   311.612,     239.781,     162.76,    107.451,
    103.13,      98.4963,   93.528,      88.2007,     82.4883,   76.3632,
    69.7954,     62.7529,   55.5388,     52.704,      49.6643,   46.405,
    42.9101,     39.1627,   36.0502,     33.7175,     31.2161,   28.534,
    26.3219,     24.7103,   22.9821,     21.2769,     19.7376,   18.0869,
    16.9932,     16.0181,   14.838,      13.4812,     12.4559,   11.5388,
    10.6659,     9.83027,   9.04737,     8.32763,     7.63312,   7.01743,
    6.46604,     5.84826,   5.41739,     4.92954,     4.48111,   4.16405,
    3.74823,     3.42688,   3.26549,     2.87995,     2.76273,   2.58815,
    2.39773,     2.22275,   2.01595,     1.74872,     1.53741,   1.37171,
    1.22452,     1.13272,   1.03206,     1.02792,     1.08481,   1.36269,
    2.17247,     2.81021,   0.930271,    0.211347,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1160.16,   1148.36,
    1135.7,      1122.13,   1107.59,     1091.99,     1075.26,   1057.32,
    1038.09,     1017.47,   995.358,     971.648,     946.225,   918.964,
    889.733,     858.39,    824.782,     788.744,     750.103,   708.669,
    664.241,     616.602,   565.52,      510.747,     452.015,   389.039,
    321.512,     249.105,   171.466,     107.752,     103.586,   99.118,
    94.3274,     89.1906,   83.6826,     77.7766,     71.4437,   64.6532,
    57.3719,     54.2377,   51.0425,     47.6164,     43.9427,   40.0035,
    36.6443,     34.4836,   32.1667,     29.6824,     27.3761,   25.5195,
    23.5287,     21.595,    20.1567,     18.6144,     17.2199,   15.8509,
    14.696,      13.7763,   12.723,      11.5502,     10.8021,   9.9868,
    9.07979,     8.30105,   7.68108,     7.05082,     6.50701,   6.04802,
    5.55752,     4.9531,    4.58807,     4.24618,     3.76746,   3.49586,
    3.21072,     2.87917,   2.63706,     2.51253,     2.36306,   2.24029,
    1.98138,     1.79862,   1.59803,     1.37586,     1.26454,   1.11508,
    1.03278,     0.997719,  1.03126,     1.27302,     1.97391,   2.88509,
    1.10874,     0.258352,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1194.64,     1182.59,     1169.68,   1155.83,
    1140.98,     1125.05,   1107.98,     1089.67,     1070.04,   1048.98,
    1026.41,     1002.21,   976.257,     948.429,     918.59,    886.595,
    852.287,     815.5,     776.055,     733.759,     688.406,   639.776,
    587.632,     531.719,   471.765,     407.479,     338.547,   264.634,
    185.379,     112.117,   107.573,     102.701,     97.4766,   91.8747,
    85.8681,     79.4274,   72.5212,     65.1159,     57.1755,   53.5735,
    50.4846,     47.1726,   43.6211,     39.813,      36.403,    34.5012,
    32.462,      30.2754,   28.1484,     26.4577,     24.645,    22.7694,
    21.1659,     19.4464,   17.9223,     16.5263,     15.1487,   13.8529,
    12.7714,     11.9101,   11.0041,     10.1333,     9.384,     8.63807,
    7.97616,     7.2039,    6.63216,     6.10621,     5.56086,   5.06329,
    4.64049,     4.22298,   3.88956,     3.49842,     3.25554,   2.9435,
    2.72867,     2.47069,   2.36173,     2.21184,     2.04099,   1.85145,
    1.60981,     1.39272,   1.25036,     1.11476,     1.00984,   0.994025,
    1.03305,     1.19072,   1.74571,     2.87272,     1.33617,   0.305459,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1173.23,     1161.48,   1148.89,     1135.39,     1120.91,   1105.39,
    1088.74,     1070.89,   1051.75,     1031.23,     1009.23,   985.634,
    960.333,     933.205,   904.116,     872.925,     839.48,    803.617,
    765.163,     723.931,   679.718,     632.31,      581.476,   526.969,
    468.522,     405.852,   338.653,     266.597,     189.334,   109.974,
    105.76,      101.241,   96.3956,     91.2,        85.6289,   79.6551,
    73.2497,     66.3814,   59.0167,     55.0773,     52.0781,   48.8622,
    45.4139,     41.7164,   38.096,      35.7723,     33.2807,   30.609,
    27.9898,     26.3534,   24.5987,     22.738,      21.1538,   19.4551,
    17.9402,     16.6556,   15.3883,     14.2869,     13.2041,   12.1844,
    11.1808,     10.2382,   9.53645,     8.71534,     8.06452,   7.41089,
    6.79794,     6.27548,   5.66641,     5.16407,     4.77527,   4.30635,
    4.03396,     3.61772,   3.26849,     2.94436,     2.72239,   2.52334,
    2.33087,     2.17875,   2.04847,     1.88619,     1.69313,   1.45701,
    1.29358,     1.13093,   1.06335,     0.971635,    0.977724,  1.10511,
    1.58716,     2.73007,   1.62494,     0.377529,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1224.41,   1212.23,
    1199.17,     1185.17,   1170.15,     1154.05,     1136.79,   1118.28,
    1098.43,     1077.15,   1054.33,     1029.86,     1003.62,   975.482,
    945.314,     912.965,   878.279,     841.085,     801.205,   758.441,
    712.588,     663.421,   610.701,     554.17,      493.555,   428.559,
    358.866,     284.136,   204.006,     118.085,     109.17,    104.597,
    99.6938,     94.4362,   88.7987,     82.7537,     76.2719,   69.3217,
    61.8693,     57.2241,   53.9314,     50.4006,     46.6147,   42.5552,
    38.4164,     36.2556,   33.9387,     31.4543,     28.8583,   26.8757,
    24.7498,     22.4702,   21.0946,     19.6751,     18.262,    16.9264,
    15.5886,     14.5213,   13.4459,     12.444,      11.4398,   10.4435,
    9.60636,     8.80707,   8.09075,     7.55263,     7.03147,   6.30507,
    5.78805,     5.35693,   4.78373,     4.40854,     3.95296,   3.60386,
    3.29609,     3.09904,   2.77121,     2.53455,     2.28563,   2.15871,
    2.05007,     1.88306,   1.6792,      1.50504,     1.26153,   1.15504,
    1.03774,     0.95674,   0.942614,    1.05743,     1.4134,    2.49722,
    1.92305,     0.431859,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1205.11,     1193.28,     1180.59,   1166.99,
    1152.41,     1136.77,   1120.0,      1102.02,     1082.74,   1062.06,
    1039.89,     1016.12,   990.636,     963.307,     934.002,   902.579,
    868.886,     832.758,   794.019,     752.48,      707.939,   660.18,
    608.969,     554.057,   495.177,     432.042,     364.344,   291.754,
    213.918,     130.457,   113.397,     108.492,     103.232,   97.5918,
    91.5442,     85.0597,   78.1065,     70.6508,     62.6563,   57.3988,
    54.2913,     50.9593,   47.3864,     43.5554,     39.4475,   37.1776,
    34.7615,     32.1708,   29.3929,     27.4184,     25.3181,   23.066,
    21.5178,     20.0093,   18.5566,     17.4189,     16.1983,   14.8835,
    13.6007,     12.6801,   11.7009,     10.6797,     9.93549,   9.06114,
    8.28098,     7.66105,   7.01572,     6.4108,      5.84184,   5.33428,
    4.85083,     4.50102,   4.11329,     3.65928,     3.38483,   3.06548,
    2.73668,     2.53774,   2.24748,     2.15985,     2.01947,   1.88198,
    1.69433,     1.53446,   1.33187,     1.18267,     1.03534,   0.939511,
    0.947389,    1.00722,   1.31672,     2.248,       2.18903,   0.55256,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1243.43,     1231.26,   1218.2,      1204.2,      1189.19,   1173.1,
    1155.84,     1137.33,   1117.49,     1096.21,     1073.4,    1048.93,
    1022.7,      994.576,   964.417,     932.078,     897.402,   860.22,
    820.351,     777.6,     731.761,     682.608,     629.904,   573.39,
    512.793,     447.817,   378.144,     303.437,     223.331,   137.436,
    111.949,     107.312,   102.34,      97.0092,     91.2927,   85.1631,
    78.5905,     71.5429,   63.9861,     58.3992,     55.2356,   51.8434,
    48.206,      44.3058,   40.1237,     37.6792,     35.3033,   32.7558,
    30.0242,     28.1021,   26.1726,     24.1037,     22.3475,   20.616,
    18.8776,     17.5257,   16.0838,     14.9977,     13.851,    12.7472,
    11.7014,     10.748,    10.0079,     9.24244,     8.46407,   7.7572,
    7.23087,     6.61233,   6.00733,     5.47264,     5.00867,   4.46721,
    4.11719,     3.78923,   3.33906,     3.09431,     2.78493,   2.46984,
    2.29127,     2.13123,   2.01025,     1.87131,     1.75149,   1.55532,
    1.29879,     1.19197,   1.02015,     0.942097,    0.916679,  0.958569,
    1.17041,     2.04088,   2.45619,     0.646534,    0.0,       0.0,
    0.0,         0.0,       0.0,         0.0,         1254.79,   1242.59,
    1229.5,      1215.46,   1200.41,     1184.27,     1166.97,   1148.41,
    1128.52,     1107.18,   1084.31,     1059.78,     1033.48,   1005.27,
    975.034,     942.608,   907.839,     870.558,     830.582,   787.717,
    741.755,     692.471,   639.625,     582.961,     522.201,   457.051,
    387.192,     312.285,   231.964,     145.839,     112.078,   107.543,
    102.68,      97.4663,   91.8754,     85.8805,     79.4523,   72.5596,
    65.1688,     59.1857,   55.9934,     52.5704,     48.9,      44.9644,
    40.7444,     37.9965,   35.5087,     32.8411,     29.9808,   27.9606,
    26.0782,     24.0598,   22.4716,     21.0705,     19.5921,   18.2352,
    16.7803,     15.4977,   14.1589,     13.0902,     12.0528,   11.1366,
    10.2213,     9.37629,   8.63271,     7.95771,     7.32626,   6.65145,
    6.17812,     5.55604,   5.06867,     4.61333,     4.17792,   3.7598,
    3.45932,     3.09075,   2.81,        2.55481,     2.24916,   2.07179,
    1.98546,     1.8644,    1.74377,     1.57009,     1.39851,   1.19818,
    1.02624,     0.965449,  0.902431,    0.919867,    1.1201,    1.78783,
    2.63068,     0.784778,  0.0,         0.0,         0.0,       0.0,
    0.0,         0.0,       1276.77,     1264.43,     1251.21,   1237.03,
    1221.83,     1205.53,   1188.05,     1169.31,     1149.21,   1127.67,
    1104.56,     1079.78,   1053.22,     1024.73,     994.186,   961.434,
    926.315,     888.658,   848.279,     804.982,     758.557,   708.776,
    655.398,     598.163,   536.791,     470.984,     400.422,   324.76,
    243.63,      156.637,   114.177,     109.61,      104.712,   99.4608,
    93.8298,     87.7919,   81.3176,     74.3755,     66.9316,   60.4199,
    57.1743,     53.6942,   49.9626,     45.9613,     41.6708,   38.7127,
    36.2498,     33.6088,   30.777,      28.6296,     26.7256,   24.684,
    22.8814,     21.2514,   19.505,      18.1819,     16.7631,   15.5809,
    14.3867,     13.2604,   12.1669,     11.2854,     10.402,    9.61404,
    8.93317,     8.05607,   7.39091,     6.8678,      6.18209,   5.74158,
    5.24183,     4.68261,   4.26317,     3.81844,     3.46094,   3.20919,
    2.87017,     2.60087,   2.33689,     2.09402,     1.93962,   1.81763,
    1.75426,     1.56849,   1.41923,     1.16675,     1.08314,   0.954815,
    0.897066,    0.892002,  1.01527,     1.59523,     2.67521,   0.986296,
    0.0,         0.0,       0.0,         0.0,         0.0,       0.0,
    1342.54,     1329.62,   1315.76,     1300.9,      1284.96,   1267.87,
    1249.55,     1229.91,   1208.84,     1186.25,     1162.03,   1136.06,
    1108.21,     1078.35,   1046.34,     1012.0,      975.19,    935.717,
    893.391,     848.006,   799.341,     747.16,      691.207,   631.211,
    566.879,     497.898,   423.932,     344.621,     259.578,   168.389,
    115.217,     110.662,   105.779,     100.542,     94.9276,   88.907,
    82.4513,     75.529,    68.1065,     61.1541,     57.9917,   54.6008,
    50.9648,     47.0661,   42.8856,     39.771,      37.3307,   34.7141,
    31.9084,     29.5949,   27.5866,     25.4332,     23.4038,   21.5477,
    19.5575,     18.3749,   17.2058,     16.044,      14.8327,   13.6286,
    12.4254,     11.4752,   10.67,       9.86604,     8.94293,   8.28838,
    7.63237,     6.97291,   6.41362,     5.76349,     5.28381,   4.89926,
    4.30443,     3.96765,   3.55976,     3.20312,     2.86287,   2.59211,
    2.38387,     2.10363,   1.91735,     1.8219,      1.72699,   1.58519,
    1.44159,     1.23354,   1.08485,     0.930562,    0.878387,  0.855159,
    0.986287,    1.43515,   2.57146,     1.16522,     0.231101,  0.0,
    0.0,         0.0,       0.0,         0.0,         1333.94,   1321.25,
    1307.64,     1293.05,   1277.41,     1260.63,     1242.64,   1223.35,
    1202.67,     1180.5,    1156.72,     1131.22,     1103.88,   1074.56,
    1043.13,     1009.42,   973.278,     934.524,     892.969,   848.411,
    800.633,     749.402,   694.468,     635.565,     572.406,   504.681,
    432.063,     354.196,   270.703,     181.175,     120.187,   115.27,
    109.998,     104.345,   98.2833,     91.7836,     84.8141,   77.341,
    69.3278,     61.3346,   58.2303,     54.9018,     51.3327,   47.5056,
    43.402,      40.1421,   37.7443,     35.1733,     32.4164,   30.0147,
    27.9901,     25.8193,   23.7628,     22.0127,     20.1361,   18.6514,
    17.1832,     16.0746,   15.158,      13.9827,     12.6916,   11.7169,
    10.7432,     9.8387,    9.16578,     8.40643,     7.57045,   6.98319,
    6.42987,     5.83788,   5.38528,     4.90232,     4.42814,   4.05851,
    3.63059,     3.33448,   2.8711,      2.66379,     2.40641,   2.13312,
    1.94856,     1.79795,   1.70252,     1.58566,     1.46385,   1.26833,
    1.12052,     0.96705,   0.899633,    0.868377,    0.931743,  1.2891,
    2.39321,     1.3985,    0.279982,    0.0,         0.0,       0.0,
    0.0,         0.0,       1314.8,      1302.38,     1289.06,   1274.78,
    1259.47,     1243.05,   1225.44,     1206.57,     1186.32,   1164.62,
    1141.35,     1116.39,   1089.63,     1060.94,     1030.17,   997.182,
    961.808,     923.878,   883.207,     839.597,     792.835,   742.693,
    688.928,     631.278,   569.461,     503.177,     432.103,   355.893,
    274.175,     186.551,   118.513,     113.899,     108.951,   103.646,
    97.958,      91.8585,   85.3182,     78.3053,     70.7856,   62.7714,
    59.6668,     56.3378,   52.7682,     48.9407,     44.8366,   41.291,
    38.6852,     35.891,    32.8948,     30.2794,     28.3377,   26.2558,
    24.1944,     22.42,     20.5174,     18.9682,     17.5022,   16.2804,
    15.2709,     14.1165,   12.8398,     11.973,      11.0618,   10.1294,
    9.32616,     8.58948,   7.93625,     7.14336,     6.60765,   5.862,
    5.42795,     4.95698,   4.56701,     4.15322,     3.66923,   3.34661,
    3.05263,     2.73966,   2.40202,     2.13958,     1.99447,   1.75285,
    1.64824,     1.62166,   1.43378,     1.28853,     1.13639,   0.969101,
    0.881975,    0.828627,  0.880405,    1.15609,     2.20679,   1.64858,
    0.335704,    0.0,       0.0,         0.0,         0.0,       0.0,
    1320.51,     1308.16,   1294.91,     1280.71,     1265.48,   1249.15,
    1231.64,     1212.86,   1192.73,     1171.14,     1147.99,   1123.17,
    1096.55,     1068.01,   1037.41,     1004.6,      969.416,   931.69,
    891.237,     847.861,   801.35,      751.478,     698.001,   640.661,
    579.176,     513.248,   442.555,     366.754,     285.475,   198.322,
    122.235,     117.4,     112.216,     106.657,     100.697,   94.3057,
    87.4525,     80.1041,   72.2247,     63.7758,     60.2215,   56.9263,
    53.3929,     49.6042,   45.5417,     41.8725,     39.3614,   36.6689,
    33.7817,     31.0458,   28.8852,     26.5684,     24.3107,   22.8472,
    21.278,      19.865,    18.5137,     17.0075,     15.3213,   13.9582,
    12.8504,     12.1617,   11.3134,     10.1714,     9.48769,   8.7827,
    8.01754,     7.32499,   6.67321,     6.14527,     5.51412,   5.09203,
    4.58392,     4.1637,    3.82934,     3.37929,     3.03259,   2.75852,
    2.46387,     2.19706,   1.97027,     1.84837,     1.59931,   1.58625,
    1.47998,     1.30429,   1.13806,     0.99014,     0.894137,  0.826628,
    0.855039,    1.07775,   1.94129,     1.9276,      0.413646,  0.0,
    0.0,         0.0,       0.0,         0.0,         1372.96,   1360.2,
    1346.51,     1331.82,   1316.08,     1299.2,      1281.11,   1261.7,
    1240.89,     1218.58,   1194.65,     1169.0,      1141.49,   1111.99,
    1080.36,     1046.45,   1010.08,     971.092,     929.281,   884.449,
    836.377,     784.831,   729.56,      670.294,     606.746,   538.605,
    465.54,      387.194,   303.187,     213.109,     125.848,   120.915,
    115.625,     109.953,   103.871,     97.3496,     90.3569,   82.8589,
    74.819,      66.1981,   61.8769,     58.3214,     54.509,    50.421,
    46.0377,     41.9141,   39.3652,     36.6321,     33.7015,   30.9128,
    29.1234,     27.2047,   25.1814,     23.3123,     21.3082,   19.6405,
    18.279,      16.9223,   15.6553,     14.4543,     13.3475,   12.3162,
    11.279,      10.2347,   9.56563,     8.939,       8.18944,   7.49731,
    6.9368,      6.2152,    5.65452,     5.18877,     4.70601,   4.22001,
    3.89018,     3.46417,   3.15553,     2.80038,     2.50854,   2.21393,
    1.96303,     1.80981,   1.6408,      1.5428,      1.44003,   1.33346,
    1.15231,     1.00792,   0.919292,    0.806828,    0.821664,  0.991119,
    1.73832,     2.18467,   0.494512,    0.0,         0.0,       0.0,
    0.0,         0.0,       1387.58,     1374.75,     1361.0,    1346.25,
    1330.44,     1313.49,   1295.31,     1275.81,     1254.91,   1232.5,
    1208.47,     1182.7,    1155.06,     1125.44,     1093.67,   1059.6,
    1023.07,     983.905,   941.907,     896.874,     848.586,   796.809,
    741.291,     681.76,    617.927,     549.481,     476.088,   397.392,
    313.008,     222.526,   126.208,     121.449,     116.346,   110.875,
    105.008,     98.7164,   91.9706,     84.7373,     76.9813,   68.6648,
    63.7602,     60.0029,   55.9741,     51.6542,     47.0221,   42.4496,
    39.8464,     37.0551,   34.0621,     31.0704,     29.2156,   27.2266,
    25.1046,     23.5948,   21.9758,     20.3683,     18.8088,   17.3006,
    16.1385,     14.875,    13.4912,     12.3811,     11.3988,   10.5419,
    9.78341,     8.99825,   8.31828,     7.70634,     6.96682,   6.38433,
    5.78798,     5.42829,   4.8034,      4.34328,     3.89602,   3.57612,
    3.25328,     2.83268,   2.57336,     2.26205,     2.03163,   1.78188,
    1.66032,     1.48597,   1.44784,     1.35579,     1.19297,   1.0439,
    0.874759,    0.821713,  0.797037,    0.936357,    1.55149,   2.41142,
    0.606983,    0.0,       0.0,         0.0,         0.0,       0.0,
    1378.72,     1366.06,   1352.5,      1337.95,     1322.35,   1305.63,
    1287.7,      1268.47,   1247.85,     1225.74,     1202.03,   1176.61,
    1149.35,     1120.12,   1088.78,     1055.18,     1019.15,   980.509,
    939.08,      894.657,   847.023,     795.948,     741.181,   682.456,
    619.487,     551.968,   479.569,     401.938,     318.698,   229.441,
    133.734,     121.211,   116.228,     110.885,     105.156,   99.0123,
    92.425,      85.3617,   77.788,      69.6669,     64.4693,   60.9047,
    57.0825,     52.9841,   48.5895,     44.018,      41.3811,   38.5537,
    35.5219,     32.35,     30.4262,     28.3634,     26.1516,   24.2888,
    22.3349,     20.5071,   19.0449,     17.5531,     16.3082,   15.0818,
    14.0383,     12.9498,   11.8356,     10.8779,     9.99427,   9.14061,
    8.4304,      7.86762,   7.15097,     6.46193,     5.86673,   5.43338,
    4.86788,     4.41821,   3.979,       3.61696,     3.15554,   2.8814,
    2.59879,     2.26164,   2.02416,     1.7527,      1.6065,    1.46203,
    1.44999,     1.348,     1.23923,     1.02734,     0.889173,  0.815645,
    0.785658,    0.900121,  1.37088,     2.4352,      0.733988,  0.0,
    0.0,         0.0,       0.0,         0.0,         1379.73,   1367.18,
    1353.72,     1339.29,   1323.82,     1307.23,     1289.43,   1270.36,
    1249.9,      1227.97,   1204.45,     1179.23,     1152.19,   1123.19,
    1092.1,      1058.77,   1023.02,     984.689,     943.589,   899.519,
    852.264,     801.594,   747.262,     689.004,     626.535,   559.553,
    487.729,     410.715,   328.136,     239.588,     144.642,   123.602,
    118.505,     113.04,    107.179,     100.895,     94.1568,   86.9316,
    79.1843,     70.8771,   65.0792,     61.4502,     57.559,    53.3866,
    48.9127,     44.1155,   41.5054,     38.786,      35.8701,   32.7434,
    30.7601,     28.6882,   26.4666,     24.5998,     22.7042,   20.8902,
    19.5646,     18.1529,   16.7379,     15.3234,     14.2244,   13.1102,
    11.9874,     11.13,     10.2493,     9.39282,     8.59793,   7.88418,
    7.22544,     6.51128,   5.97191,     5.54027,     4.93756,   4.50451,
    4.10558,     3.70049,   3.22299,     2.9285,      2.59742,   2.36332,
    2.01251,     1.82496,   1.60115,     1.44432,     1.39556,   1.3337,
    1.20617,     1.05891,   0.90987,     0.815221,    0.770279,  0.858981,
    1.22033,     2.43968,   0.889666,    0.0,         0.0,       0.0,
    0.0,         0.0,       1428.91,     1415.98,     1402.12,   1387.25,
    1371.32,     1354.23,   1335.9,      1316.26,     1295.19,   1272.6,
    1248.37,     1222.4,    1194.55,     1164.68,     1132.66,   1098.33,
    1061.51,     1022.03,   979.702,     934.312,     885.642,   833.454,
    777.495,     717.493,   653.154,     584.165,     510.191,   430.871,
    345.818,     254.619,   156.83,      126.497,     121.354,   115.84,
    109.928,     103.588,   96.7897,     89.5003,     81.6842,   73.3032,
    66.9734,     63.3492,   59.4631,     55.2961,     50.828,    46.037,
    42.9991,     40.0336,   36.8538,     33.4442,     31.2595,   29.1372,
    26.8617,     25.0345,   23.2926,     21.5025,     19.9535,   18.294,
    17.068,      15.7647,   14.4585,     13.2582,     12.2294,   11.2723,
    10.3565,     9.46668,   8.64033,     8.05141,     7.34355,   6.78137,
    6.10648,     5.61333,   5.01229,     4.57897,     4.22734,   3.66344,
    3.36136,     2.99506,   2.67607,     2.37983,     2.0336,    1.85089,
    1.63306,     1.4682,    1.36621,     1.28422,     1.23395,   1.06955,
    0.932101,    0.809833,  0.763226,    0.814611,    1.10581,   2.26475,
    1.08444,     0.0,       0.0,         0.0,         0.0,       0.0,
    1458.36,     1445.22,   1431.13,     1416.02,     1399.82,   1382.45,
    1363.82,     1343.85,   1322.43,     1299.47,     1274.85,   1248.44,
    1220.13,     1189.77,   1157.22,     1122.32,     1084.89,   1044.76,
    1001.73,     955.594,   906.119,     853.07,      796.186,   735.192,
    669.789,     599.661,   524.464,     443.833,     357.375,   264.669,
    165.263,     125.62,    120.516,     115.045,     109.177,   102.886,
    96.1403,     88.907,    81.1508,     72.8342,     66.1179,   62.6896,
    59.0134,     55.0716,   50.845,      46.3129,     43.3536,   40.6944,
    37.843,      34.7855,   32.3117,     29.8872,     27.2875,   25.2549,
    23.4847,     21.6396,   20.2171,     18.6917,     17.3279,   15.9019,
    14.768,      13.6501,   12.6343,     11.7058,     10.5941,   9.52569,
    9.05596,     8.18639,   7.48143,     6.80567,     6.20676,   5.70471,
    5.05965,     4.68045,   4.26711,     3.88351,     3.39898,   3.04067,
    2.75529,     2.426,     2.15063,     1.89395,     1.69269,   1.49916,
    1.33374,     1.29922,   1.20735,     1.08346,     0.947717,  0.827177,
    0.755975,    0.774106,  1.007,       2.04855,     1.29253,   0.0,
    0.0,         0.0,       0.0,         0.0,         1463.23,   1450.14,
    1436.11,     1421.07,   1404.93,     1387.64,     1369.09,   1349.2,
    1327.87,     1305.0,    1280.48,     1254.19,     1226.0,    1195.77,
    1163.35,     1128.6,    1091.33,     1051.37,     1008.52,   962.569,
    913.301,     860.474,   803.828,     743.09,      677.961,   608.126,
    533.244,     452.951,   366.855,     274.537,     175.548,   127.11,
    121.834,     116.177,   110.112,     103.608,     96.6336,   89.1554,
    81.1369,     72.5389,   65.1806,     62.0412,     58.6749,   55.0654,
    51.195,      47.0449,   43.9387,     41.1903,     38.2432,   35.0832,
    32.6023,     30.3507,   27.9363,     25.8741,     24.0767,   22.1507,
    20.751,      19.2501,   17.6053,     15.834,      14.8079,   13.7745,
    12.6841,     11.7697,   10.8613,     9.915,       9.06489,   8.28065,
    7.64426,     6.98182,   6.24204,     5.75112,     5.29447,   4.75109,
    4.28061,     3.8438,    3.49132,     3.13234,     2.75795,   2.43723,
    2.16573,     1.94973,   1.66505,     1.45049,     1.29713,   1.25853,
    1.224,       1.11172,   0.963297,    0.835392,    0.725955,  0.751887,
    0.948942,    1.81916,   1.5656,      0.0,         0.0,       0.0,
    0.0,         0.0,       1489.27,     1476.04,     1461.87,   1446.66,
    1430.36,     1412.89,   1394.14,     1374.05,     1352.5,    1329.4,
    1304.62,     1278.06,   1249.57,     1219.03,     1186.28,   1151.16,
    1113.5,      1073.12,   1029.83,     983.406,     933.628,   880.252,
    823.018,     761.649,   695.844,     625.284,     549.625,   468.499,
    381.509,     288.233,   188.216,     130.4,       125.194,   119.612,
    113.627,     107.209,   100.327,     92.9482,     85.036,    76.5519,
    68.6373,     65.0131,   61.1271,     56.9601,     52.4921,   47.7011,
    44.1386,     41.3287,   38.3158,     35.0851,     32.4878,   30.2779,
    27.9083,     25.8309,   24.1212,     22.288,      20.8721,   19.4072,
    17.9948,     16.5384,   15.3277,     14.1231,     12.9685,   11.9025,
    10.917,      10.0131,   8.99497,     8.48769,     7.69565,   6.99496,
    6.43246,     5.95295,   5.4791,      4.81562,     4.33293,   3.93417,
    3.54628,     3.18015,   2.87614,     2.52458,     2.16752,   1.94404,
    1.66429,     1.48462,   1.29461,     1.26795,     1.21194,   1.12417,
    0.982109,    0.852605,  0.755577,    0.727274,    0.875022,  1.61961,
    1.87134,     0.326904,  0.0,         0.0,         0.0,       0.0,
    1503.67,     1490.42,   1476.2,      1460.96,     1444.62,   1427.1,
    1408.31,     1388.17,   1366.56,     1343.4,      1318.56,   1291.93,
    1263.38,     1232.76,   1199.92,     1164.72,     1126.97,   1086.49,
    1043.09,     996.548,   946.645,     893.136,     835.759,   774.237,
    708.268,     637.532,   561.684,     480.354,     393.147,   299.638,
    199.372,     132.38,    126.97,      121.168,     114.948,   108.277,
    101.125,     93.4558,   85.2324,     76.4147,     67.7125,   64.4246,
    60.8991,     57.1189,   53.0654,     48.719,      45.2513,   42.6257,
    39.8104,     36.7917,   34.1181,     31.7839,     29.2811,   26.9507,
    25.0146,     22.9386,   21.3207,     19.717,      18.2524,   16.8232,
    15.5696,     14.3373,   13.2342,     12.158,      11.1594,   10.3444,
    9.2156,      8.55129,   7.89185,     7.19084,     6.60513,   6.06849,
    5.43742,     4.9841,    4.45047,     4.03613,     3.63487,   3.27607,
    2.89922,     2.52307,   2.24472,     1.98428,     1.72107,   1.44338,
    1.2831,      1.21741,   1.20207,     1.1066,      0.998776,  0.845029,
    0.756763,    0.71759,   0.831849,    1.43051,     2.01618,   0.3913,
    0.0,         0.0,       0.0,         0.0,         1502.21,   1489.08,
    1475.01,     1459.91,   1443.73,     1426.38,     1407.77,   1387.82,
    1366.42,     1343.48,   1318.88,     1292.51,     1264.23,   1233.9,
    1201.38,     1166.51,   1129.13,     1089.04,     1046.05,   999.96,
    950.536,     897.541,   840.716,     779.784,     714.449,   644.393,
    569.273,     488.725,   402.356,     309.745,     210.442,   135.227,
    129.832,     124.047,   117.844,     111.193,     104.062,   96.4148,
    88.2151,     79.4229,   70.1725,     66.645,      62.8626,   58.8067,
    54.4578,     49.7946,   45.8258,     42.9063,     39.7758,   36.419,
    33.6191,     31.6863,   29.6138,     27.5087,     25.5243,   23.3966,
    21.5389,     19.7,      18.2334,     17.0631,     15.9324,   14.7809,
    13.4563,     12.3632,   11.5026,     10.5304,     9.68455,   8.72625,
    8.04938,     7.4486,    6.66278,     6.01491,     5.51645,   5.03979,
    4.5773,      4.12046,   3.67355,     3.34305,     2.92571,   2.60472,
    2.3411,      2.03248,   1.72402,     1.55558,     1.2945,    1.17031,
    1.15742,     1.15239,   0.981187,    0.857899,    0.7609,    0.710917,
    0.793544,    1.27497,   2.20787,     0.480749,    0.0,       0.0,
    0.0,         0.0,       1496.47,     1483.5,      1469.6,    1454.69,
    1438.7,      1421.56,   1403.18,     1383.47,     1362.33,   1339.67,
    1315.38,     1289.32,   1261.38,     1231.43,     1199.31,   1164.86,
    1127.93,     1088.33,   1045.87,     1000.34,     951.517,   899.168,
    843.035,     782.846,   718.307,     649.104,     574.9,     495.334,
    410.017,     318.535,   220.442,     137.376,     131.929,   126.088,
    119.826,     113.111,   105.91,      98.1895,     89.9107,   81.0337,
    71.5152,     67.4293,   63.437,      59.1563,     54.5661,   49.6443,
    45.4544,     42.9776,   40.3217,     37.474,      34.7872,   32.6101,
    30.2757,     27.9385,   26.0348,     23.9936,     22.0966,   20.2214,
    18.7299,     17.7142,   16.3689,     14.7428,     13.6428,   12.5989,
    11.6027,     10.7064,   9.81649,     8.97129,     8.18569,   7.62755,
    6.92677,     6.14631,   5.67166,     5.15482,     4.66557,   4.19261,
    3.77428,     3.36783,   2.96436,     2.68778,     2.35206,   2.0561,
    1.77806,     1.56119,   1.27897,     1.19345,     1.13563,   1.09429,
    1.04016,     0.884559,  0.761829,    0.67425,     0.763659,  1.12463,
    2.23487,     0.580514,  0.0,         0.0,         0.0,       0.0,
    1548.34,     1534.99,   1520.67,     1505.33,     1488.87,   1471.22,
    1452.3,      1432.01,   1410.26,     1386.93,     1361.92,   1335.1,
    1306.34,     1275.5,    1242.44,     1206.99,     1168.97,   1128.2,
    1084.49,     1037.62,   987.369,     933.481,     875.699,   813.741,
    747.305,     676.068,   599.684,     517.779,     429.955,   335.784,
    234.808,     140.322,   134.786,     128.85,      122.485,   115.659,
    108.341,     100.494,   92.0797,     83.0573,     73.3829,   68.8792,
    65.0652,     60.9757,   56.5906,     51.8885,     47.5286,   44.7323,
    41.734,      38.5191,   35.4387,     33.1884,     30.7756,   28.2864,
    26.2589,     24.0848,   22.2358,     20.6295,     19.0565,   17.6073,
    16.2808,     15.0887,   13.8237,     12.6782,     11.8634,   10.9014,
    9.94125,     9.08299,   8.2655,      7.52307,     6.91933,   6.31222,
    5.8361,      5.20803,   4.77077,     4.24443,     3.83539,   3.43555,
    3.09033,     2.72238,   2.38813,     2.08096,     1.8748,    1.54349,
    1.32376,     1.14395,   1.08439,     1.09341,     1.05717,   0.907384,
    0.7701,      0.692157,  0.739706,    0.992784,    2.1946,    0.707372,
    0.0,         0.0,       0.0,         0.0,         1564.95,   1551.57,
    1537.22,     1521.83,   1505.32,     1487.63,     1468.66,   1448.31,
    1426.5,      1403.1,    1378.02,     1351.13,     1322.29,   1291.37,
    1258.21,     1222.66,   1184.54,     1143.66,     1099.83,   1052.83,
    1002.43,     948.394,   890.452,     828.322,     761.702,   690.268,
    613.671,     531.539,   443.471,     349.039,     247.783,   144.009,
    138.269,     132.114,   125.514,     118.437,     110.849,   102.712,
    93.9873,     84.6321,   74.6008,     69.3759,     65.4636,   61.2685,
    56.7703,     51.947,    47.2967,     44.6364,     41.7839,   38.7252,
    35.6604,     33.3844,   30.944,      28.3687,     26.392,    24.2724,
    22.4069,     20.8535,   19.363,      18.1697,     16.8254,   15.2915,
    14.1267,     13.037,    11.8557,     10.9208,     9.98678,   9.11674,
    8.4706,      7.78213,   6.95552,     6.53437,     5.90621,   5.32595,
    4.85531,     4.35859,   3.98375,     3.46753,     3.08885,   2.72295,
    2.43665,     2.15804,   1.86542,     1.60915,     1.38375,   1.18858,
    1.03132,     1.05506,   1.06292,     0.93726,     0.791667,  0.692618,
    0.705684,    0.923174,  2.02681,     0.865697,    0.0,       0.0,
    0.0,         0.0,       1569.09,     1555.72,     1541.38,   1526.01,
    1509.53,     1491.85,   1472.9,      1452.58,     1430.79,   1407.43,
    1382.38,     1355.51,   1326.71,     1295.82,     1262.7,    1227.19,
    1189.11,     1148.29,   1104.51,     1057.56,     1007.22,   953.251,
    895.376,     833.319,   766.777,     695.427,     618.92,    536.884,
    448.92,      354.599,   253.461,     145.014,     135.419,   129.64,
    123.443,     116.799,   109.674,     102.034,     93.8427,   85.059,
    75.6406,     70.146,    66.4172,     62.419,      58.1319,   53.5349,
    48.8422,     45.9151,   42.7766,     39.4112,     35.9419,   33.5764,
    31.0398,     28.32,     26.6742,     24.9533,     23.1946,   21.4425,
    19.7019,     18.3274,   16.94,       15.6288,     14.3031,   12.9623,
    11.7191,     11.0801,   10.3682,     9.5405,      8.70397,   7.83042,
    7.21379,     6.59056,   6.06211,     5.38655,     4.88093,   4.39707,
    3.92295,     3.58049,   3.16972,     2.79281,     2.48264,   2.17793,
    1.89089,     1.63327,   1.36188,     1.19381,     1.03442,   1.01696,
    1.03242,     0.909039,  0.770624,    0.691271,    0.691436,  0.851308,
    1.84056,     1.04415,   0.0,         0.0,         0.0,       0.0,
    1530.46,     1517.58,   1503.76,     1488.94,     1473.05,   1456.02,
    1437.75,     1418.16,   1397.16,     1374.64,     1350.49,   1324.6,
    1296.84,     1267.07,   1235.15,     1200.92,     1164.22,   1124.86,
    1082.66,     1037.41,   988.896,     936.872,     881.088,   821.273,
    757.135,     688.362,   614.619,     535.546,     450.76,    359.846,
    262.362,     157.833,   139.454,     133.347,     126.798,   119.776,
    112.246,     104.173,   95.5155,     86.2327,     76.2791,   70.1813,
    66.7465,     63.0635,   59.1144,     54.8798,     50.375,    47.3183,
    44.0407,     40.5262,   36.7791,     34.4993,     32.0547,   29.4335,
    27.537,      25.6278,   23.7279,     22.0187,     20.2323,   18.608,
    17.0464,     15.9235,   14.867,      13.8241,     12.5083,   11.3977,
    10.5777,     9.78245,   8.75593,     8.02098,     7.29544,   6.65356,
    6.04514,     5.52385,   5.10568,     4.51404,     4.07033,   3.6524,
    3.26208,     2.90099,   2.53789,     2.21798,     1.94212,   1.69043,
    1.41344,     1.25872,   1.00317,     0.984299,    1.00414,   0.954522,
    0.807867,    0.714539,  0.654814,    0.807376,    1.62723,   1.28384,
    0.0,         0.0,       0.0,         0.0,         1621.63,   1608.04,
    1593.46,     1577.83,   1561.07,     1543.1,      1523.83,   1503.17,
    1481.01,     1457.25,   1431.78,     1404.46,     1375.17,   1343.77,
    1310.09,     1273.99,   1235.27,     1193.75,     1149.24,   1101.5,
    1050.32,     995.437,   936.589,     873.488,     805.827,   733.277,
    655.483,     572.067,   482.624,     386.716,     283.878,   173.607,
    145.137,     138.747,   131.894,     124.546,     116.668,   108.22,
    99.1609,     89.4475,   79.0323,     71.9373,     68.2068,   64.2068,
    59.9178,     55.3187,   50.3873,     47.4176,     44.3939,   41.1516,
    37.6751,     35.142,    32.52,       29.7084,     27.5609,   25.4815,
    23.4826,     22.1198,   20.6598,     19.1163,     17.5229,   16.1272,
    14.7677,     13.4556,   12.7245,     11.6283,     10.5662,   9.81319,
    8.95864,     8.15149,   7.45383,     6.82155,     6.14485,   5.65203,
    5.11687,     4.59593,   4.14655,     3.69656,     3.32992,   2.93394,
    2.58821,     2.22433,   2.04322,     1.72656,     1.45375,   1.18968,
    1.02509,     0.973707,  1.00495,     0.968837,    0.799114,  0.708206,
    0.649852,    0.740445,  1.43254,     1.5117,      0.0,       0.0,
    0.0,         0.0,       1645.03,     1631.27,     1616.52,   1600.71,
    1583.74,     1565.56,   1546.06,     1525.15,     1502.73,   1478.69,
    1452.91,     1425.27,   1395.63,     1363.85,     1329.77,   1293.23,
    1254.05,     1212.04,   1166.99,     1118.69,     1066.9,    1011.36,
    951.808,     887.953,   819.485,     746.068,     667.345,   582.934,
    492.422,     395.369,   291.302,     179.715,     142.134,   136.351,
    130.15,      123.5,     116.371,     108.726,     100.528,   91.7386,
    82.3135,     75.055,    71.0519,     66.7596,     62.157,    57.2218,
    51.93,       48.6082,   45.4608,     42.086,      38.4673,   35.8929,
    33.3773,     30.6799,   28.3319,     26.0355,     23.7267,   22.1405,
    20.4396,     19.0564,   17.6179,     16.4366,     15.0889,   13.5273,
    12.7556,     11.8647,   10.8702,     9.98284,     9.17578,   8.33298,
    7.63155,     6.83375,   6.29457,     5.7171,      5.22601,   4.66233,
    4.17708,     3.74417,   3.36774,     3.01221,     2.58736,   2.33394,
    2.04361,     1.71763,   1.49111,     1.23513,     1.04765,   0.90692,
    0.940504,    0.975188,  0.82948,     0.697481,    0.644505,  0.700102,
    1.25363,     1.74331,   0.0,         0.0,         0.0,       0.0,
    1634.29,     1620.74,   1606.22,     1590.64,     1573.94,   1556.03,
    1536.83,     1516.24,   1494.16,     1470.49,     1445.11,   1417.89,
    1388.7,      1357.41,   1323.85,     1287.87,     1249.29,   1207.92,
    1163.56,     1115.99,   1064.99,     1010.3,      951.661,   888.783,
    821.36,      749.065,   671.546,     588.424,     499.295,   403.726,
    301.25,      191.368,   145.051,     139.068,     132.653,   125.775,
    118.399,     110.49,    102.009,     92.9161,     83.1657,   75.1738,
    71.2179,     66.9762,   62.4279,     57.551,      52.3216,   49.0744,
    46.3126,     43.3512,   40.1759,     37.3867,     34.5924,   31.5961,
    29.0755,     26.7831,   24.4077,     23.0069,     21.505,    19.868,
    18.1112,     16.5967,   15.1494,     13.9586,     12.9155,   11.9431,
    11.0082,     10.0399,   9.3107,      8.5603,      7.70919,   6.98914,
    6.48586,     5.83169,   5.23201,     4.80584,     4.28597,   3.8818,
    3.40135,     3.06411,   2.68697,     2.42256,     2.08025,   1.78703,
    1.49861,     1.2658,    1.09171,     0.867892,    0.893378,  0.916,
    0.840168,    0.723724,  0.644126,    0.690335,    1.12123,   1.94978,
    0.0,         0.0,       0.0,         0.0,         1667.11,   1653.37,
    1638.63,     1622.83,   1605.89,     1587.73,     1568.25,   1547.37,
    1524.97,     1500.96,   1475.21,     1447.61,     1418.0,    1386.26,
    1352.22,     1315.73,   1276.59,     1234.63,     1189.64,   1141.39,
    1089.66,     1034.19,   974.708,     910.929,     842.542,   769.212,
    690.583,     606.272,   515.868,     418.93,      314.988,   203.533,
    147.308,     141.237,   134.728,     127.748,     120.264,   112.239,
    103.634,     94.4072,   84.5136,     75.8554,     71.9127,   67.6851,
    63.152,      58.2914,   53.0794,     49.4334,     46.4248,   43.1988,
    39.7396,     36.9941,   34.514,      31.8546,     29.464,    27.2847,
    24.9478,     23.3198,   21.5854,     19.9627,     18.2796,   16.8377,
    15.4778,     14.4898,   13.321,      12.2493,     11.3945,   10.3169,
    9.4253,      8.73108,   7.77391,     7.20168,     6.53065,   5.98854,
    5.30921,     4.94052,   4.37521,     3.91046,     3.49012,   3.07511,
    2.77915,     2.41272,   2.09643,     1.8415,      1.55841,   1.28463,
    1.02806,     0.894934,  0.83899,     0.918803,    0.86255,   0.759369,
    0.637956,    0.659224,  0.979729,    2.0167,      0.0,       0.0,
    0.0,         0.0,       1702.33,     1688.34,     1673.34,   1657.25,
    1640.0,      1621.51,   1601.68,     1580.41,     1557.61,   1533.16,
    1506.95,     1478.84,   1448.69,     1416.37,     1381.72,   1344.56,
    1304.71,     1261.99,   1216.18,     1167.05,     1114.38,   1057.9,
    997.337,     932.399,   862.767,     788.104,     708.045,   622.201,
    530.152,     431.452,   325.619,     212.137,     145.712,   140.061,
    134.0,       127.502,   120.535,     113.063,     105.052,   96.4621,
    87.2512,     78.5407,   74.2467,     69.6425,     64.7055,   59.4117,
    53.7353,     49.7345,   46.8286,     43.7128,     40.3718,   37.5342,
    35.0025,     32.2878,   29.7215,     27.3672,     24.8428,   23.2577,
    21.6779,     20.0523,   18.3353,     17.034,      15.7786,   14.6045,
    13.4825,     12.3815,   11.2957,     10.3428,     9.50975,   8.60308,
    7.8363,      7.26892,   6.6345,      6.14516,     5.53092,   4.99199,
    4.48141,     3.93676,   3.63274,     3.18945,     2.81231,   2.47223,
    2.10809,     1.83279,   1.61215,     1.30673,     1.13468,   0.905249,
    0.789704,    0.914897,  0.877138,    0.73944,     0.628398,  0.611001,
    0.894165,    2.07483,   0.496406,    0.0,         0.0,       0.0,
    1695.36,     1681.56,   1666.76,     1650.9,      1633.88,   1615.64,
    1596.08,     1575.11,   1552.61,     1528.5,      1502.64,   1474.91,
    1445.18,     1413.3,    1379.12,     1342.47,     1303.16,   1261.02,
    1215.83,     1167.38,   1115.42,     1059.71,     999.974,   935.92,
    867.238,     793.592,   714.624,     629.948,     539.154,   441.798,
    337.407,     225.472,   150.453,     144.388,     137.885,   130.912,
    123.435,     115.417,   106.82,      97.6021,     87.7178,   77.9184,
    73.9494,     69.6937,   65.1303,     60.2372,     54.9905,   50.8619,
    47.8169,     44.552,    41.051,      38.0857,     35.6575,   33.0538,
    30.6525,     28.7044,   26.6155,     24.7599,     22.8542,   21.051,
    19.2516,     17.6859,   16.1517,     14.7732,     13.4453,   12.306,
    11.6561,     10.6294,   9.71374,     8.9847,      8.19458,   7.64004,
    6.68365,     6.14905,   5.53344,     5.00011,     4.48515,   4.05071,
    3.57515,     3.30242,   2.80587,     2.56442,     2.20249,   1.91523,
    1.61129,     1.36082,   1.12391,     0.898116,    0.786488,  0.880556,
    0.870911,    0.743939,  0.636058,    0.623987,    0.821829,  1.92613,
    0.60732,     0.0,       0.0,         0.0,         1687.6,    1673.95,
    1659.32,     1643.64,   1626.82,     1608.78,     1589.45,   1568.71,
    1546.48,     1522.63,   1497.07,     1469.66,     1440.27,   1408.75,
    1374.96,     1338.72,   1299.86,     1258.2,      1213.53,   1165.62,
    1114.26,     1059.18,   1000.13,     936.804,     868.904,   796.096,
    718.027,     634.316,   544.556,     448.309,     345.107,   234.446,
    151.089,     145.032,   138.537,     131.573,     124.105,   116.098,
    107.512,     98.3051,   88.4332,     78.0812,     74.2704,   70.1841,
    65.8026,     61.1044,   56.0666,     51.8387,     48.8005,   45.5427,
    42.0495,     38.8181,   36.0296,     33.0396,     30.2348,   28.1441,
    25.9023,     24.2376,   22.714,      21.0313,     19.1884,   17.7508,
    16.4729,     15.2812,   14.0605,     12.8692,     11.9205,   10.7388,
    9.84404,     8.96457,   8.15408,     7.42571,     6.89141,   6.28423,
    5.68076,     5.10524,   4.59691,     4.1512,      3.7269,    3.19321,
    2.95558,     2.56482,   2.30236,     1.91913,     1.64464,   1.40713,
    1.17532,     0.971273,  0.717004,    0.795022,    0.842545,  0.770056,
    0.641741,    0.617692,  0.767662,    1.7692,      0.713095,  0.0,
    0.0,         0.0,       1757.01,     1742.89,     1727.74,   1711.5,
    1694.08,     1675.4,    1655.38,     1633.91,     1610.89,   1586.2,
    1559.73,     1531.34,   1500.91,     1468.27,     1433.28,   1395.76,
    1355.52,     1312.38,   1266.12,     1216.52,     1163.33,   1106.3,
    1045.15,     979.581,   909.271,     833.881,     753.042,   666.361,
    573.415,     473.753,   366.889,     252.301,     156.444,   149.939,
    142.964,     135.484,   127.464,     118.865,     109.644,   99.7568,
    89.1551,     77.7872,   73.7571,     69.8593,     65.6798,   61.1983,
    56.3929,     52.1062,   49.0459,     45.7644,     42.2458,   38.963,
    36.3425,     33.5326,   30.8026,     28.8451,     26.7461,   24.7867,
    22.8372,     21.0114,   19.3393,     18.0171,     16.9233,   15.4292,
    14.0393,     12.8838,   11.88,       11.0164,     10.2309,   9.26744,
    8.40489,     7.57883,   6.93877,     6.304,       5.82517,   5.18815,
    4.72959,     4.24779,   3.75034,     3.33191,     2.9585,    2.53282,
    2.30067,     1.84528,   1.72381,     1.37837,     1.15202,   0.933888,
    0.725791,    0.765949,  0.862128,    0.768502,    0.659152,  0.589745,
    0.710946,    1.57323,   0.900553,    0.0,         0.0,       0.0,
    1733.88,     1720.06,   1705.25,     1689.36,     1672.32,   1654.05,
    1634.46,     1613.46,   1590.94,     1566.79,     1540.9,    1513.13,
    1483.36,     1451.43,   1417.2,      1380.5,      1341.14,   1298.94,
    1253.69,     1205.16,   1153.14,     1097.35,     1037.53,   973.385,
    904.607,     830.858,   751.779,     666.986,     576.065,   478.574,
    374.037,     261.946,   158.905,     152.511,     145.656,   138.305,
    130.424,     121.972,   112.91,      103.193,     92.7731,   81.6007,
    76.6111,     72.3151,   67.7087,     62.7693,     57.473,    52.6514,
    49.664,      46.4607,   43.0259,     39.678,      36.9752,   34.077,
    31.152,      29.0655,   26.8283,     24.9193,     23.2305,   21.5403,
    19.9073,     18.3474,   16.8561,     15.6081,     14.3768,   13.1413,
    12.2859,     11.2844,   10.2614,     9.27874,     8.48323,   7.81772,
    7.05998,     6.56369,   5.84103,     5.27431,     4.80093,   4.22851,
    3.82829,     3.43991,   2.98679,     2.69436,     2.36404,   2.05342,
    1.71178,     1.43979,   1.17199,     0.943813,    0.74055,   0.671323,
    0.82467,     0.778639,  0.662398,    0.600975,    0.677065,  1.36424,
    1.08736,     0.0,       0.0,         0.0,         1792.19,   1777.95,
    1762.68,     1746.3,    1728.75,     1709.93,     1689.74,   1668.1,
    1644.89,     1620.01,   1593.32,     1564.71,     1534.03,   1501.14,
    1465.86,     1428.04,   1387.49,     1344.0,      1297.37,   1247.37,
    1193.76,     1136.28,   1074.63,     1008.54,     937.667,   861.674,
    780.189,     692.815,   599.126,     498.667,     390.949,   275.445,
    159.745,     153.49,    146.784,     139.593,     131.882,   123.614,
    114.748,     105.242,   95.0489,     84.1191,     78.5255,   74.1733,
    69.5067,     64.5028,   59.1373,     54.0071,     50.9296,   47.6296,
    44.0911,     40.6024,   38.1085,     35.4343,     32.6209,   30.3068,
    27.8253,     25.6808,   23.9006,     22.0956,     20.3772,   18.6864,
    17.0715,     15.828,    14.6761,     13.5319,     12.3825,   11.2706,
    10.3121,     9.50275,   8.71905,     7.85459,     7.28206,   6.4895,
    5.94927,     5.37421,   4.82964,     4.32725,     4.0329,    3.4637,
    3.15244,     2.74598,   2.33233,     2.02111,     1.75596,   1.48516,
    1.27027,     0.942687,  0.756359,    0.664617,    0.80666,   0.816203,
    0.678451,    0.577564,  0.641712,    1.20671,     1.34616,   0.0,
    0.0,         0.0,       1801.56,     1787.35,     1772.11,   1755.77,
    1738.24,     1719.45,   1699.31,     1677.71,     1654.54,   1629.71,
    1603.08,     1574.52,   1543.9,      1511.07,     1475.86,   1438.11,
    1397.63,     1354.23,   1307.69,     1257.79,     1204.28,   1146.9,
    1085.38,     1019.41,   948.67,      872.822,     791.492,   704.285,
    610.775,     510.508,   402.994,     287.712,     164.097,   156.32,
    149.498,     142.183,   134.339,     125.929,     116.911,   107.24,
    96.8715,     85.7533,   79.5558,     75.2667,     70.6676,   65.7362,
    60.4484,     55.1309,   51.8332,     48.2972,     44.5056,   40.6773,
    38.3095,     35.7706,   33.0482,     30.7704,     28.3312,   26.0281,
    23.9895,     22.029,    20.6165,     19.13,       17.5868,   16.0497,
    14.5907,     13.5856,   12.8023,     11.615,      10.5121,   9.79629,
    8.95451,     8.08039,   7.41339,     6.67518,     6.16695,   5.50599,
    4.94903,     4.48566,   4.04901,     3.60799,     3.08168,   2.78594,
    2.43129,     2.11664,   1.78173,     1.5327,      1.26642,   1.05023,
    0.796802,    0.596423,  0.732234,    0.788154,    0.659704,  0.5905,
    0.597923,    1.0676,    1.52985,     0.0,         0.0,       0.0,
    1784.19,     1770.19,   1755.18,     1739.09,     1721.83,   1703.33,
    1683.49,     1662.21,   1639.4,      1614.94,     1588.71,   1560.59,
    1530.43,     1498.1,    1463.43,     1426.25,     1386.38,   1343.64,
    1297.8,      1248.66,   1195.96,     1139.45,     1078.86,   1013.89,
    944.231,     869.533,   789.436,     703.551,     611.46,    512.713,
    406.83,      293.295,   171.556,     154.66,      148.125,   141.118,
    133.604,     125.548,   116.909,     107.646,     97.7135,   87.0632,
    80.4303,     76.1362,   71.5319,     66.5947,     61.3008,   55.7659,
    52.4092,     48.8098,   44.9504,     40.896,      38.3579,   35.6363,
    32.7181,     30.5334,   28.2788,     26.0918,     24.1881,   22.2235,
    20.4917,     18.8925,   17.8438,     16.3985,     14.7795,   14.0473,
    13.0643,     11.8684,   10.6654,     9.70604,     8.98242,   8.31676,
    7.55781,     6.97187,   6.23327,     5.48966,     4.99795,   4.55051,
    3.99255,     3.69149,   3.21072,     2.76651,     2.46318,   2.14013,
    1.8335,      1.52585,   1.25529,     1.01024,     0.773765,  0.603656,
    0.681367,    0.804454,  0.688576,    0.587513,    0.584282,  0.94965,
    1.75188,     0.0,       0.0,         0.0,         1769.34,   1755.56,
    1740.79,     1724.96,   1707.98,     1689.77,     1670.24,   1649.3,
    1626.86,     1602.78,   1576.97,     1549.3,      1519.62,   1487.8,
    1453.68,     1417.09,   1377.86,     1335.8,      1290.69,   1242.33,
    1190.47,     1134.86,   1075.23,     1011.29,     942.738,   869.227,
    790.404,     705.884,   615.257,     518.08,      413.88,    302.15,
    182.346,     156.598,   150.339,     143.628,     136.432,   128.716,
    120.442,     111.571,   102.058,     91.8579,     84.4686,   79.4326,
    74.0326,     68.2424,   62.0337,     55.3764,     52.2859,   49.0739,
    45.6298,     41.9368,   39.2901,     36.4966,     33.5012,   30.9512,
    28.3478,     25.948,    24.4538,     22.8542,     21.164,    19.4365,
    17.9175,     16.554,    15.3301,     14.1046,     13.0804,   11.997,
    10.8506,     9.97894,   9.18208,     8.47223,     7.59742,   6.98661,
    6.30066,     5.81787,   5.13642,     4.70482,     4.12364,   3.72255,
    3.26033,     2.89338,   2.52916,     2.15687,     1.87513,   1.58776,
    1.3244,      1.11379,   0.820831,    0.612398,    0.633025,  0.771857,
    0.707875,    0.629112,  0.609852,    0.834763,    1.89581,   0.0,
    0.0,         0.0,       1865.27,     1850.81,     1835.3,    1818.68,
    1800.85,     1781.73,   1761.23,     1739.25,     1715.68,   1690.41,
    1663.31,     1634.25,   1603.09,     1569.69,     1533.86,   1495.45,
    1454.26,     1410.1,    1362.74,     1311.96,     1257.52,   1199.13,
    1136.53,     1069.41,   997.428,     920.25,      837.494,   748.757,
    653.608,     551.582,   442.184,     324.879,     199.097,   162.157,
    155.333,     148.016,   140.169,     131.756,     122.735,   113.062,
    102.689,     91.5674,   83.4184,     78.8706,     73.9941,   68.7652,
    63.1585,     57.1465,   53.5502,     50.0251,     46.2452,   42.1922,
    39.4895,     36.7992,   33.9145,     31.649,      29.4865,   27.2435,
    25.1603,     22.9455,   21.5264,     20.0574,     18.8425,   17.3436,
    15.5006,     14.4016,   13.2256,     12.1135,     11.2287,   10.3135,
    9.19808,     8.40478,   7.75643,     7.20573,     6.38184,   5.85685,
    5.27857,     4.7185,    4.19695,     3.69572,     3.35383,   2.92041,
    2.57207,     2.24541,   1.91961,     1.64798,     1.33497,   1.07507,
    0.820485,    0.634735,  0.592199,    0.769166,    0.711701,  0.599134,
    0.563805,    0.749105,  1.90041,     0.0,         0.0,       0.0,
    1872.19,     1857.75,   1842.27,     1825.66,     1807.86,   1788.77,
    1768.31,     1746.36,   1722.83,     1697.59,     1670.54,   1641.53,
    1610.42,     1577.06,   1541.29,     1502.94,     1461.82,   1417.72,
    1370.44,     1319.74,   1265.38,     1207.08,     1144.58,   1077.56,
    1005.69,     928.635,   846.007,     757.409,     662.408,   560.541,
    451.312,     334.19,    208.604,     162.067,     155.312,   148.07,
    140.303,     131.976,   123.047,     113.472,     103.206,   92.1972,
    83.6122,     79.2824,   74.6397,     69.6615,     64.3235,   58.5998,
    55.0627,     51.8515,   48.4082,     44.7161,     41.5281,   38.2917,
    34.8213,     32.1772,   29.8528,     27.4509,     25.5671,   23.5472,
    21.8611,     20.1066,   18.6487,     17.2399,     15.9836,   14.7684,
    13.5433,     12.3627,   11.3042,     10.2921,     9.3864,    8.5912,
    7.84425,     7.19558,   6.59705,     5.90031,     5.34749,   4.83599,
    4.28153,     3.85497,   3.41101,     2.95438,     2.62271,   2.27937,
    1.92774,     1.68169,   1.38197,     1.10158,     0.846181,  0.633487,
    0.488977,    0.702245,  0.738536,    0.605601,    0.548078,  0.697236,
    1.84847,     0.0,       0.0,         0.0,         1796.28,   1782.59,
    1767.91,     1752.17,   1735.29,     1717.2,      1697.79,   1676.98,
    1654.67,     1630.75,   1605.1,      1577.59,     1548.1,    1516.48,
    1482.57,     1446.21,   1407.22,     1365.41,     1320.58,   1272.52,
    1220.98,     1165.71,   1106.45,     1042.91,     974.78,    901.723,
    823.387,     739.389,   649.321,     552.745,     449.188,   338.148,
    219.084,     165.558,   158.779,     151.511,     143.718,   135.361,
    126.401,     116.793,   106.491,     95.4438,     86.0957,   81.3271,
    76.214,      70.7313,   64.8524,     58.5487,     54.5965,   51.3476,
    47.8639,     44.1285,   40.9224,     37.778,      34.4064,   31.9613,
    30.1155,     28.1366,   26.0219,     23.7543,     21.851,    19.8938,
    18.4265,     17.0433,   15.9655,     14.5175,     13.397,    12.6661,
    11.459,      10.632,    9.64206,     8.72892,     8.15802,   7.31144,
    6.72307,     5.97487,   5.43165,     4.843,       4.43046,   3.94153,
    3.44543,     3.08583,   2.68602,     2.36961,     1.92415,   1.73733,
    1.42571,     1.10701,   0.922365,    0.678027,    0.474281,  0.661051,
    0.773859,    0.634255,  0.554454,    0.662822,    1.67672,   0.0,
    0.0,         0.0,       1902.21,     1887.73,     1872.21,   1855.56,
    1837.71,     1818.57,   1798.05,     1776.04,     1752.44,   1727.14,
    1700.01,     1670.92,   1639.73,     1606.28,     1570.41,   1531.96,
    1490.72,     1446.5,    1399.09,     1348.25,     1293.74,   1235.29,
    1172.62,     1105.41,   1033.35,     956.079,     873.226,   784.385,
    689.124,     586.979,   477.452,     360.009,     234.08,    167.88,
    160.919,     153.454,   145.45,      136.868,     127.666,   117.798,
    107.218,     95.8728,   85.8745,     81.7464,     77.32,     72.5738,
    67.4845,     62.0274,   57.6149,     53.6164,     49.329,    44.7318,
    41.5149,     38.9696,   36.2404,     33.4969,     30.7206,   27.7436,
    26.1018,     24.3957,   22.6963,     20.9096,     19.3568,   17.8209,
    16.474,      15.1444,   13.9435,     12.96,       11.926,    10.7934,
    9.54627,     8.96928,   8.13035,     7.22404,     6.71675,   6.01702,
    5.47005,     4.94716,   4.50634,     3.96708,     3.54733,   3.0942,
    2.73429,     2.38846,   2.05804,     1.75895,     1.43554,   1.1559,
    0.929557,    0.672015,  0.456536,    0.58285,     0.73785,   0.638939,
    0.551848,    0.611687,  1.46368,     0.0,         0.0,       0.0,
    1898.78,     1884.41,   1869.01,     1852.49,     1834.77,   1815.78,
    1795.41,     1773.57,   1750.16,     1725.05,     1698.13,   1669.26,
    1638.3,      1605.11,   1569.52,     1531.35,     1490.43,   1446.56,
    1399.51,     1349.06,   1294.96,     1236.96,     1174.76,   1108.07,
    1036.56,     959.881,   877.661,     789.5,       694.967,   593.603,
    484.913,     368.368,   243.402,     168.112,     161.286,   153.967,
    146.119,     137.704,   128.681,     119.006,     108.631,   97.5069,
    86.97,       82.3053,   77.3035,     71.9402,     66.1893,   60.0229,
    55.6915,     52.6563,   49.4016,     45.9118,     42.7976,   39.9161,
    36.8263,     34.0978,   31.8875,     29.5175,     27.2386,   24.828,
    22.8307,     20.9306,   19.446,      17.9787,     16.432,    15.2091,
    14.0648,     12.7259,   11.65,       10.8897,     9.87683,   8.99519,
    8.35018,     7.54805,   6.72992,     6.28005,     5.65025,   5.04343,
    4.42566,     3.97181,   3.53724,     3.19904,     2.78981,   2.39919,
    2.10282,     1.76328,   1.46744,     1.21627,     0.937594,  0.686312,
    0.456004,    0.512757,  0.736509,    0.641664,    0.557767,  0.582438,
    1.30046,     0.783516,  0.0,         0.0,         1897.49,   1883.22,
    1867.92,     1851.52,   1833.92,     1815.06,     1794.84,   1773.15,
    1749.9,      1724.96,   1698.22,     1669.56,     1638.81,   1605.85,
    1570.51,     1532.61,   1491.97,     1448.4,      1401.67,   1351.57,
    1297.85,     1240.25,   1178.49,     1112.26,     1041.24,   965.093,
    883.443,     795.892,   702.014,     601.352,     493.415,   377.677,
    253.576,     169.291,   162.651,     155.531,     147.897,   139.711,
    130.933,     121.522,   111.43,      100.608,     89.7812,   85.1385,
    80.1603,     74.8223,   69.0986,     62.9612,     57.9956,   54.2167,
    50.1647,     45.8199,   42.3618,     39.8398,     37.1355,   34.3422,
    31.5245,     28.5031,   26.6789,     25.0518,     23.3302,   21.4976,
    19.8512,     18.2351,   16.9635,     15.6993,     14.4558,   13.2579,
    11.8995,     10.951,    10.0294,     9.13293,     8.45273,   7.74898,
    6.89232,     6.33138,   5.73642,     5.08549,     4.60754,   4.14409,
    3.60472,     3.32686,   2.85242,     2.4849,      2.12397,   1.81865,
    1.49303,     1.16566,   0.977502,    0.738998,    0.461906,  0.439922,
    0.735285,    0.66445,   0.535684,    0.592834,    1.1344,    0.958924,
    0.0,         0.0,       1995.57,     1980.89,     1965.16,   1948.29,
    1930.2,      1910.81,   1890.01,     1867.71,     1843.79,   1818.15,
    1790.66,     1761.18,   1729.57,     1695.67,     1659.33,   1620.36,
    1578.57,     1533.76,   1485.71,     1434.2,      1378.95,   1319.72,
    1256.21,     1188.1,    1115.08,     1036.77,     952.812,   862.782,
    766.246,     662.734,   551.741,     432.728,     305.113,   179.071,
    172.303,     165.045,   157.263,     148.918,     139.971,   130.376,
    120.089,     109.058,   97.2294,     90.7817,     85.4097,   79.6494,
    73.4728,     66.8499,   60.6077,     57.0059,     53.1437,   49.0024,
    44.9391,     42.0034,   38.8555,     35.5805,     33.1608,   30.5663,
    28.2406,     26.1822,   24.2304,     22.6406,     20.9094,   19.0204,
    17.5657,     16.1877,   14.7658,     13.9461,     12.8196,   11.8624,
    10.8286,     9.82269,   8.96475,     8.12615,     7.49795,   6.73089,
    6.05025,     5.43665,   4.985,       4.44696,     3.92132,   3.45699,
    3.12027,     2.63821,   2.33777,     1.9312,      1.64037,   1.3551,
    1.13632,     0.852587,  0.57705,     0.364495,    0.421769,  0.660512,
    0.569051,    0.50414,   0.675031,    1.74518,     0.0,       0.0,
    2038.69,     2023.95,   2008.15,     1991.21,     1973.04,   1953.56,
    1932.68,     1910.28,   1886.27,     1860.51,     1832.9,    1803.3,
    1771.55,     1737.51,   1701.01,     1661.87,     1619.9,    1574.9,
    1526.65,     1474.91,   1419.43,     1359.95,     1296.16,   1227.77,
    1154.43,     1075.79,   991.468,     901.053,     804.104,   700.148,
    588.68,      469.157,   340.996,     203.573,     178.069,   170.905,
    163.224,     154.987,   146.155,     136.685,     126.531,   115.643,
    103.968,     95.7544,   90.0564,     83.9467,     77.3955,   70.3708,
    62.8623,     59.2742,   55.4268,     51.3014,     46.8779,   44.029,
    40.979,      37.7087,   35.0795,     32.3974,     29.7865,   27.6199,
    25.3778,     23.553,    21.7476,     20.3135,     18.8111,   17.2651,
    16.0816,     14.6543,   13.3677,     12.3716,     11.3844,   10.336,
    9.35703,     8.50161,   7.81592,     7.07137,     6.3562,    5.79174,
    5.27912,     4.63701,   4.12735,     3.65876,     3.19936,   2.85377,
    2.47412,     2.07945,   1.78763,     1.47359,     1.1933,    0.912782,
    0.683964,    0.377064,  0.199131,    0.566761,    0.595706,  0.509356,
    0.541457,    1.52956,   0.0,         0.0,         2064.23,   2049.54,
    2033.79,     2016.89,   1998.78,     1979.35,     1958.53,   1936.19,
    1912.25,     1886.57,   1859.04,     1829.51,     1797.86,   1763.92,
    1727.52,     1688.49,   1646.64,     1601.77,     1553.66,   1502.07,
    1446.75,     1387.43,   1323.83,     1255.63,     1182.5,    1104.08,
    1020.0,      929.846,   833.174,     729.516,     618.366,   499.184,
    371.388,     234.358,   179.335,     172.174,     164.496,   156.262,
    147.434,     137.968,   127.818,     116.934,     105.264,   95.7179,
    90.6155,     85.1444,   79.2779,     72.9874,     66.2424,   61.9341,
    58.1153,     54.0206,   49.63,       46.2493,     43.0054,   39.527,
    36.6405,     34.0066,   31.2563,     29.1018,     26.7917,   24.8071,
    22.7348,     20.7933,   19.1572,     18.2429,     16.534,    15.1808,
    14.1951,     12.8189,   11.8402,     10.908,      9.84474,   8.95604,
    7.88583,     7.32586,   6.65835,     5.98067,     5.42011,   4.85194,
    4.34385,     3.84178,   3.37334,     3.05928,     2.59641,   2.24263,
    1.91233,     1.58369,   1.28226,     0.992634,    0.71775,   0.434853,
    0.190202,    0.335311,  0.625313,    0.534254,    0.49004,   1.01205,
    0.0,         0.0,       2151.19,     2136.12,     2119.97,   2102.66,
    2084.09,     2064.18,   2042.82,     2019.93,     1995.38,   1969.06,
    1940.84,     1910.57,   1878.12,     1843.33,     1806.02,   1766.01,
    1723.11,     1677.12,   1627.79,     1574.91,     1518.2,    1457.39,
    1392.19,     1322.28,   1247.31,     1166.93,     1080.74,   988.318,
    889.218,     782.957,   669.016,     546.842,     415.838,   275.367,
    188.565,     180.99,    172.867,     164.158,     154.82,    144.806,
    134.069,     122.556,   110.211,     98.376,      93.1375,   87.5206,
    81.4976,     75.0395,   68.1146,     63.1004,     59.536,    55.714,
    51.6158,     48.0008,   44.7297,     41.2223,     38.0601,   35.4491,
    32.6495,     30.2494,   27.7637,     25.7467,     23.8709,   22.012,
    20.1273,     18.5518,   17.2248,     15.9424,     14.449,    13.4296,
    12.0782,     11.2724,   10.2882,     9.46089,     8.46728,   7.69751,
    7.11945,     6.32476,   5.8074,      5.04516,     4.57102,   4.0808,
    3.53234,     3.20764,   2.75154,     2.36669,     1.98863,   1.6527,
    1.40545,     1.07834,   0.804189,    0.523089,    0.288418,  0.101066,
    0.536204,    0.551597,  0.487356,    0.668064,    1.25618,   0.0,
    2195.82,     2180.72,   2164.53,     2147.17,     2128.56,   2108.6,
    2087.2,      2064.26,   2039.65,     2013.27,     1984.98,   1954.65,
    1922.12,     1887.25,   1849.85,     1809.75,     1766.76,   1720.65,
    1671.22,     1618.21,   1561.37,     1500.42,     1435.07,   1365.0,
    1289.86,     1209.29,   1122.9,      1030.27,     930.94,    824.434,
    710.231,     587.775,   456.47,      315.675,     198.295,   190.363,
    181.858,     172.738,   162.96,      152.474,     141.231,   129.176,
    116.249,     102.388,   96.8791,     91.4301,     85.5874,   79.3224,
    72.6047,     66.7085,   62.6793,     58.3588,     53.7262,   49.4771,
    46.2251,     42.738,    39.3299,     36.7937,     34.0741,   31.6567,
    29.321,      26.8849,   24.3458,     22.4176,     20.8903,   19.4274,
    17.8697,     16.2436,   14.9987,     13.6932,     12.5653,   11.7075,
    10.8138,     9.75422,   8.96265,     8.04469,     7.38245,   6.67149,
    6.01033,     5.38062,   4.72125,     4.24704,     3.7657,    3.29024,
    2.92136,     2.51593,   2.10775,     1.78822,     1.44976,   1.16346,
    0.931902,    0.612054,  0.304364,    0.000821728, 0.300488,  0.551125,
    0.467821,    0.523234,  1.6351,      0.0,         2314.01,   2298.28,
    2281.41,     2263.33,   2243.93,     2223.14,     2200.84,   2176.93,
    2151.3,      2123.81,   2094.33,     2062.72,     2028.83,   1992.49,
    1953.53,     1911.75,   1866.94,     1818.91,     1767.4,    1712.16,
    1652.94,     1589.43,   1521.34,     1448.32,     1370.03,   1286.08,
    1196.07,     1099.55,   996.051,     885.075,     766.079,   638.484,
    501.668,     354.964,   201.16,      193.292,     184.855,   175.809,
    166.109,     155.708,   144.555,     132.596,     119.773,   106.023,
    98.6695,     93.3388,   87.6229,     81.494,      74.9221,   68.4203,
    64.2839,     59.8485,   55.0926,     50.4049,     47.6503,   44.6966,
    41.5437,     38.6495,   35.5461,     32.6546,     30.078,    27.6093,
    25.7181,     23.6728,   21.4523,     19.9482,     18.5936,   16.8798,
    15.6595,     14.5787,   13.3915,     12.1045,     11.1025,   9.94185,
    9.19842,     8.48808,   7.6493,      6.89404,     6.20248,   5.61904,
    5.04597,     4.50006,   3.98996,     3.4674,      3.05677,   2.66654,
    2.20161,     1.98715,   1.58947,     1.24249,     0.980336,  0.711032,
    0.407784,    0.0572473, -0.00303134, 0.572803,    0.475949,  0.463354,
    1.23369,     0.0,       2322.23,     2306.71,     2290.06,   2272.21,
    2253.07,     2232.55,   2210.55,     2186.95,     2161.65,   2134.52,
    2105.43,     2074.24,   2040.79,     2004.93,     1966.47,   1925.24,
    1881.02,     1833.62,   1782.78,     1728.27,     1669.82,   1607.15,
    1539.95,     1467.89,   1390.62,     1307.77,     1218.94,   1123.68,
    1021.54,     912.019,   794.582,     668.658,     533.635,   388.853,
    233.609,     198.133,   189.787,     180.838,     171.241,   160.952,
    149.918,     138.088,   125.402,     111.8,       102.521,   97.1122,
    91.3127,     85.0941,   78.4261,     71.2763,     67.4007,   63.3922,
    59.0941,     54.4853,   50.5619,     46.4037,     41.9449,   39.2681,
    36.8524,     34.3445,   31.8969,     29.3264,     27.1735,   24.9877,
    23.1646,     21.2262,   19.1965,     18.0239,     16.6364,   15.2734,
    14.0318,     12.7565,   11.7221,     10.6735,     9.84143,   8.82024,
    7.97609,     7.39156,   6.57425,     5.9149,      5.20353,   4.74535,
    4.21364,     3.56873,   3.28436,     2.80515,     2.39343,   2.01616,
    1.70419,     1.38239,   1.03602,     0.790233,    0.456598,  0.149121,
    -0.10778,    0.373831,  0.499032,    0.42932,     0.793912,  0.0,
    2406.75,     2390.92,   2373.94,     2355.74,     2336.22,   2315.29,
    2292.84,     2268.78,   2242.97,     2215.3,      2185.63,   2153.82,
    2119.71,     2083.13,   2043.91,     2001.85,     1956.76,   1908.4,
    1856.56,     1800.96,   1741.35,     1677.43,     1608.89,   1535.39,
    1456.59,     1372.09,   1281.48,     1184.33,     1080.15,   968.45,
    848.674,     720.242,   582.529,     434.864,     276.527,   208.541,
    199.929,     190.695,   180.793,     170.176,     158.792,   146.585,
    133.495,     119.46,    107.802,     101.872,     95.5147,   88.6975,
    81.3877,     73.5496,   68.807,      64.8889,     60.6878,   56.183,
    52.1007,     47.9725,   43.546,      40.3547,     37.8865,   35.2484,
    32.5538,     29.6645,   27.8048,     25.9763,     24.1288,   22.336,
    20.8115,     18.4977,   16.7847,     15.8589,     14.5868,   13.6701,
    12.3397,     11.258,    10.2858,     9.14851,     8.40026,   7.73102,
    6.74702,     6.2606,    5.53067,     4.84871,     4.30577,   3.8737,
    3.35903,     2.94361,   2.58637,     2.1173,      1.77611,   1.42725,
    1.15389,     0.88419,   0.561164,    0.24399,     -0.072689, 0.00625819,
    0.540181,    0.461343,  0.579909,    0.0,         2386.57,   2357.18,
    2340.76,     2323.16,   2304.28,     2284.03,     2262.33,   2239.05,
    2214.1,      2187.34,   2158.64,     2127.87,     2094.88,   2059.51,
    2021.58,     1980.9,    1937.29,     1890.53,     1840.38,   1786.61,
    1728.96,     1667.14,   1600.86,     1529.78,     1453.56,   1371.84,
    1284.21,     1190.25,   1089.5,      981.472,     865.634,   741.424,
    608.238,     465.427,   312.295,     215.714,     207.025,   197.707,
    187.717,     177.005,   165.518,     153.201,     139.995,   125.833,
    112.194,     106.514,   100.423,     93.8917,     86.8886,   79.3794,
    73.4988,     68.9022,   63.9735,     58.6886,     54.2853,   50.5911,
    46.6299,     42.9604,   39.8139,     36.4401,     33.8161,   31.1629,
    28.8729,     26.6765,   24.5308,     22.4017,     20.921,    19.4814,
    17.9535,     16.1573,   14.8131,     13.8822,     12.5888,   11.4829,
    10.6242,     9.40179,   8.71118,     8.00631,     7.06125,   6.45435,
    5.71616,     5.03952,   4.54614,     4.06025,     3.53895,   3.10327,
    2.64032,     2.30263,   1.89886,     1.58329,     1.23475,   0.94501,
    0.651479,    0.379985,  0.00339255,  -0.240815,   0.453681,  0.436522,
    0.427977,    1.50213,   2576.64,     2513.97,     2496.66,   2478.11,
    2458.21,     2436.88,   2414.0,      2389.47,     2363.17,   2334.97,
    2304.73,     2272.31,   2237.54,     2200.26,     2160.28,   2117.42,
    2071.46,     2022.18,   1969.33,     1912.67,     1851.91,   1786.76,
    1716.9,      1642.0,    1561.68,     1475.56,     1383.21,   1284.19,
    1178.01,     1064.16,   942.083,     811.183,     670.824,   520.321,
    358.941,     224.912,   215.758,     205.941,     195.416,   184.129,
    172.027,     159.051,   145.137,     130.217,     114.219,   107.956,
    101.735,     95.0639,   87.9109,     80.2408,     73.6352,   69.3533,
    64.7619,     59.8388,   55.3613,     51.9954,     48.3862,   44.7611,
    41.6862,     38.389,    35.4868,     32.6958,     29.9164,   27.1611,
    24.9935,     23.1971,   21.5661,     20.1039,     18.8386,   17.031,
    15.6245,     14.4474,   13.2316,     11.9881,     10.9363,   10.1966,
    9.19039,     8.1335,    7.3713,      6.66909,     5.92883,   5.44972,
    4.73729,     4.19622,   3.75577,     3.3218,      2.86124,   2.38985,
    2.05828,     1.66914,   1.33504,     1.03286,     0.696916,  0.433176,
    0.0926589,   -0.306624, 0.212605,    0.472443,    0.407469,  1.03492,
    2586.38,     2493.27,   2476.38,     2458.27,     2438.85,   2418.03,
    2395.71,     2371.77,   2346.1,      2318.58,     2289.06,   2257.42,
    2223.49,     2187.1,    2148.09,     2106.25,     2061.4,    2013.3,
    1961.73,     1906.42,   1847.13,     1783.54,     1715.36,   1642.26,
    1563.87,     1479.82,   1389.69,     1293.05,     1189.43,   1078.31,
    959.17,      831.416,   694.431,     547.546,     390.045,   227.792,
    218.983,     209.538,   199.41,      188.55,      176.905,   164.419,
    151.031,     136.675,   121.281,     113.232,     107.262,   100.86,
    93.9958,     86.6354,   79.3266,     74.1834,     68.6684,   62.7549,
    56.9679,     53.4598,   49.6982,     45.7294,     42.8387,   39.7392,
    36.8244,     34.157,    31.4932,     29.1016,     26.7649,   24.5907,
    22.9672,     21.3215,   19.0887,     17.4438,     16.5234,   15.3485,
    13.7357,     12.4515,   11.2524,     10.5577,     9.58963,   8.59095,
    7.68493,     6.98818,   6.23897,     5.6714,      5.01545,   4.58452,
    3.89161,     3.46412,   2.92309,     2.58945,     2.15983,   1.71416,
    1.45934,     1.06661,   0.829872,    0.509726,    0.205324,  -0.226256,
    -0.257264,   0.499614,  0.3985,      0.663495,    2792.43,   2659.99,
    2642.17,     2623.06,   2602.57,     2580.6,      2557.04,   2531.78,
    2504.69,     2475.65,   2444.51,     2411.11,     2375.3,    2336.91,
    2295.74,     2251.59,   2204.26,     2153.5,      2099.08,   2040.72,
    1978.14,     1911.05,   1839.1,      1761.96,     1679.24,   1590.54,
    1495.43,     1393.45,   1284.1,      1166.84,     1041.12,   906.303,
    761.747,     606.745,   440.541,     262.325,     227.153,   217.615,
    207.387,     196.42,    184.66,      172.051,     158.53,    144.032,
    128.487,     118.014,   111.458,     104.428,     96.8907,   88.8084,
    80.1421,     75.6211,   70.7993,     65.6291,     60.0852,   56.1759,
    52.0122,     47.5477,   44.5313,     41.5996,     38.4946,   35.2623,
    31.989,      30.0021,   27.8885,     25.6814,     23.6329,   21.721,
    20.0397,     18.4392,   16.9891,     15.6727,     14.319,    12.9649,
    11.9281,     10.8573,   9.86035,     8.8553,      7.98981,   7.33453,
    6.58044,     5.92878,   5.25833,     4.65767,     4.13593,   3.56929,
    3.06674,     2.76005,   2.34123,     1.90681,     1.58023,   1.23567,
    0.910253,    0.592572,  0.28416,     -0.0791845,  -0.502099, 0.380276,
    0.394992,    0.486323,  2797.47,     2633.53,     2616.15,   2597.52,
    2577.54,     2556.12,   2533.15,     2508.52,     2482.1,    2453.78,
    2423.42,     2390.86,   2355.94,     2318.5,      2278.36,   2235.32,
    2189.16,     2139.67,   2086.6,      2029.7,      1968.69,   1903.26,
    1833.11,     1757.89,   1677.24,     1590.75,     1498.01,   1398.58,
    1291.95,     1177.62,   1055.03,     923.58,      782.629,   631.493,
    469.434,     295.664,   228.826,     219.387,     209.266,   198.413,
    186.777,     174.299,   160.919,     146.573,     131.19,    118.99,
    112.79,      106.143,   99.0147,     91.3717,     83.1764,   77.6045,
    72.4172,     66.855,    60.8908,     56.6977,     52.7686,   48.5556,
    45.3437,     42.5586,   39.5903,     36.5649,     33.3208,   30.7447,
    28.1004,     26.4121,   24.5563,     22.4873,     20.7664,   19.1815,
    17.7029,     16.2347,   14.7271,     13.5423,     12.6801,   11.1282,
    10.1202,     9.06728,   8.50512,     7.62635,     6.80634,   6.15648,
    5.58854,     4.82541,   4.33149,     3.82154,     3.29052,   2.85128,
    2.38223,     2.0269,    1.62623,     1.31243,     0.91396,   0.658883,
    0.344647,    0.025101,  -0.521021,   0.0764884,   0.398372,  0.389332,
    2935.95,     2737.9,    2713.96,     2694.93,     2674.52,   2652.64,
    2629.17,     2604.01,   2577.04,     2548.11,     2517.09,   2483.83,
    2448.17,     2409.93,   2368.93,     2324.96,     2277.82,   2227.27,
    2173.06,     2114.94,   2052.62,     1985.8,      1914.14,   1837.31,
    1754.93,     1666.59,   1571.87,     1470.3,      1361.39,   1244.61,
    1119.4,      985.13,    841.16,      686.786,     521.256,   343.764,
    242.335,     232.576,   222.112,     210.892,     198.86,    185.96,
    172.127,     157.294,   141.389,     126.603,     119.64,    112.173,
    104.167,     95.5827,   86.3777,     79.7703,     74.7102,   69.2845,
    63.4666,     59.0119,   55.3801,     51.4859,     47.8434,   44.5151,
    40.9462,     38.3552,   35.6847,     32.6528,     29.3424,   27.605,
    25.9559,     23.7544,   21.6414,     19.6802,     18.035,    16.9296,
    15.6553,     14.3883,   12.8757,     11.4116,     10.6142,   9.63362,
    8.66819,     7.92295,   7.06888,     6.42383,     5.68899,   5.02833,
    4.40627,     3.91227,   3.43967,     2.9058,      2.51618,   2.14152,
    1.76742,     1.37176,   1.03037,     0.738892,    0.447059,  0.0454682,
    -0.331256,   -0.452023, 0.458327,    0.321375,    3040.84,   2835.62,
    2779.12,     2759.86,   2739.22,     2717.08,     2693.35,   2667.9,
    2640.61,     2611.34,   2579.96,     2546.32,     2510.24,   2471.56,
    2430.08,     2385.6,    2337.91,     2286.77,     2231.94,   2173.15,
    2110.1,      2042.5,    1970.01,     1892.29,     1808.95,   1719.58,
    1623.76,     1521.01,   1410.84,     1292.71,     1166.03,   1030.21,
    884.566,     728.4,     560.947,     381.393,     245.304,   235.313,
    224.601,     213.115,   200.798,     187.591,     173.43,    158.245,
    141.963,     124.851,   119.041,     112.811,     106.132,   98.9691,
    91.289,      84.6794,   79.6993,     74.3592,     68.6333,   63.3022,
    58.6685,     53.6999,   49.2084,     46.3414,     43.2673,   40.2034,
    37.0023,     33.9795,   31.0647,     28.8715,     26.9817,   24.9526,
    22.6443,     20.3381,   19.4866,     17.6338,     16.3537,   14.8711,
    13.2121,     11.7515,   11.0238,     10.0648,     8.94732,   8.33749,
    7.49141,     6.73273,   6.09374,     5.30983,     4.7162,    3.99479,
    3.59787,     3.13107,   2.61046,     2.24741,     1.83469,   1.42389,
    1.18711,     0.817868,  0.531477,    0.127522,    -0.186866, -0.69734,
    0.392709,    0.396926,  3194.85,     2978.95,     2886.87,   2867.07,
    2845.84,     2823.08,   2798.67,     2772.5,      2744.44,   2714.35,
    2682.08,     2647.48,   2610.39,     2570.61,     2527.96,   2482.22,
    2433.18,     2380.6,    2324.21,     2263.75,     2198.93,   2129.41,
    2054.88,     1974.95,   1889.25,     1797.36,     1698.83,   1593.18,
    1479.89,     1358.41,   1228.16,     1088.49,     938.727,   778.143,
    605.953,     421.321,   247.068,     237.496,     227.231,   216.225,
    204.423,     191.769,   178.2,       163.65,      148.049,   131.321,
    123.354,     116.649,   109.46,      101.751,     93.4842,   85.699,
    80.4456,     74.8125,   68.7723,     63.0949,     59.3967,   55.4312,
    51.3072,     47.7828,   44.0037,     40.8345,     38.1481,   35.357,
    32.5111,     29.8466,   27.3944,     25.1023,     23.1587,   21.9847,
    19.7504,     18.2035,   16.7041,     15.2905,     13.9007,   12.6393,
    11.5791,     10.4207,   9.58608,     8.58154,     7.4952,    7.05985,
    6.27504,     5.50247,   4.92336,     4.31705,     3.80538,   3.29468,
    2.85917,     2.32482,   1.98402,     1.5541,      1.26062,   0.917265,
    0.619649,    0.267145,  -0.114327,   -0.625224,   0.132752,  0.382192,
    3160.66,     2947.69,   2824.7,      2805.71,     2785.34,   2763.5,
    2740.08,     2714.97,   2688.04,     2659.17,     2628.21,   2595.01,
    2559.41,     2521.25,   2480.32,     2436.44,     2389.38,   2338.93,
    2284.82,     2226.81,   2164.61,     2097.91,     2026.39,   1949.7,
    1867.47,     1779.3,    1684.75,     1583.38,     1474.67,   1358.11,
    1233.13,     1099.12,   955.417,     801.332,     636.113,   458.953,
    268.991,     248.957,   238.499,     227.286,     215.262,   202.37,
    188.545,     173.722,   157.828,     140.784,     130.511,   123.534,
    116.052,     108.03,    99.4281,     90.5843,     85.148,    79.3187,
    73.0682,     66.5737,   62.0602,     57.2205,     52.0311,   48.9511,
    45.7779,     42.3323,   38.5665,     34.834,      32.0236,   29.6281,
    28.413,      26.523,    24.222,      22.0271,     20.8026,   19.0744,
    17.299,      15.9062,   14.2341,     13.1102,     11.9257,   10.8373,
    9.70016,     8.89934,   8.03064,     7.16789,     6.39696,   5.75775,
    5.15874,     4.51105,   3.96152,     3.34288,     2.92366,   2.5527,
    2.08982,     1.66633,   1.29255,     0.973168,    0.649479,  0.303915,
    -0.0330001,  -0.538441, -0.467675,   0.399001,    3319.9,    3095.87,
    2934.03,     2914.51,   2893.58,     2871.13,     2847.07,   2821.26,
    2793.59,     2763.92,   2732.11,     2697.99,     2661.42,   2622.2,
    2580.14,     2535.04,   2486.69,     2434.84,     2379.25,   2319.63,
    2255.71,     2187.17,   2113.68,     2034.88,     1950.38,   1859.77,
    1762.61,     1658.44,   1546.74,     1426.96,     1298.53,   1160.81,
    1013.15,     854.811,   685.032,     502.983,     307.777,   253.497,
    243.021,     231.788,   219.743,     206.827,     192.979,   178.129,
    162.206,     145.132,   132.587,     125.152,     117.18,    108.631,
    99.4652,     89.6367,   84.3821,     79.2612,     73.7702,   67.8824,
    63.6503,     59.3352,   54.7082,     50.9914,     47.3742,   43.6641,
    40.3373,     36.8194,   34.4455,     31.9369,     29.4698,   26.9767,
    24.4757,     22.8831,   21.1934,     19.5491,     17.9623,   16.1469,
    14.8103,     13.3233,   12.3124,     11.2567,     10.3466,   9.39055,
    8.37023,     7.3486,    6.70722,     5.98197,     5.2782,    4.77547,
    4.06971,     3.56333,   3.03427,     2.60851,     2.19716,   1.77407,
    1.38705,     1.01684,   0.723907,    0.419802,    0.0669591, -0.444873,
    -0.944241,   0.385646,  3513.96,     3276.68,     3071.48,   3051.3,
    3029.66,     3006.45,   2981.57,     2954.89,     2926.28,   2895.61,
    2862.71,     2827.45,   2789.63,     2749.08,     2705.6,    2658.97,
    2608.98,     2555.37,   2497.89,     2436.26,     2370.17,   2299.31,
    2223.32,     2141.85,   2054.48,     1960.8,      1860.36,   1752.65,
    1637.16,     1513.32,   1380.54,     1238.16,     1085.48,   921.779,
    746.244,     558.023,   356.2,       264.879,     253.835,   241.993,
    229.295,     215.679,   201.08,      185.425,     168.639,   150.64,
    135.529,     128.306,   120.562,     112.258,     103.354,   93.8058,
    87.5933,     82.3814,   76.7927,     70.8002,     66.1445,   61.8163,
    57.1753,     52.9053,   48.8035,     44.4931,     42.1147,   39.5645,
    36.3359,     32.7929,   30.3888,     28.0713,     26.1091,   23.9877,
    22.0015,     20.2355,   18.4133,     17.0535,     15.7769,   14.2359,
    12.9003,     11.808,    10.5864,     9.59128,     8.61629,   8.00505,
    7.05732,     6.37426,   5.53834,     4.90369,     4.28788,   3.70316,
    3.24726,     2.77723,   2.30952,     1.90436,     1.52223,   1.08523,
    0.832586,    0.457329,  0.168415,    -0.264438,   -0.947901, 0.209606,
    3594.67,     3352.13,   3125.93,     3088.37,     3066.76,   3043.6,
    3018.76,     2992.13,   2963.57,     2932.95,     2900.11,   2864.9,
    2827.15,     2786.67,   2743.26,     2696.72,     2646.82,   2593.3,
    2535.92,     2474.39,   2408.42,     2337.68,     2261.83,   2180.49,
    2093.28,     1999.76,   1899.49,     1791.97,     1676.68,   1553.05,
    1420.5,      1278.36,   1125.96,     962.534,     787.302,   599.407,
    397.933,     271.906,   261.126,     249.567,     237.173,   223.883,
    209.633,     194.353,   177.968,     160.4,       143.435,   135.788,
    127.587,     118.794,   109.365,     99.2548,     91.6495,   86.0033,
    79.9491,     73.4574,   68.2199,     63.9842,     59.4424,   54.9678,
    50.7011,     46.126,    42.7635,     39.3993,     36.3906,   33.4393,
    31.1311,     28.8581,   26.3287,     24.3304,     22.6047,   20.7711,
    18.9593,     17.1485,   16.0885,     14.6068,     13.0878,   11.9218,
    10.9722,     9.72889,   8.99778,     8.05649,     7.35903,   6.4349,
    5.78806,     5.04741,   4.48406,     4.00214,     3.43067,   2.79494,
    2.38398,     1.95261,   1.61103,     1.22701,     0.861201,  0.551772,
    0.179011,    -0.189003, -0.748468,   -0.270923,   3710.32,   3459.7,
    3225.98,     3153.9,    3132.04,     3108.59,     3083.46,   3056.5,
    3027.6,      2996.61,   2963.38,     2927.75,     2889.55,   2848.58,
    2804.65,     2757.55,   2707.04,     2652.89,     2594.82,   2532.55,
    2465.78,     2394.19,   2317.43,     2235.12,     2146.86,   2052.22,
    1950.74,     1841.93,   1725.25,     1600.14,     1466.0,    1322.15,
    1167.92,     1002.53,   825.194,     635.043,     431.149,   268.294,
    257.825,     246.6,     234.563,     221.656,     207.817,   192.977,
    177.065,     160.004,   141.709,     135.032,     128.045,   120.552,
    112.518,     103.903,   96.0178,     89.6299,     82.7803,   75.4358,
    69.1227,     64.8232,   60.2131,     55.629,      51.7341,   47.5577,
    44.229,      41.1682,   38.0452,     34.8454,     32.1901,   29.8025,
    27.3782,     25.211,    23.3864,     21.4766,     19.846,    18.3095,
    16.8166,     15.1923,   13.5927,     12.6608,     11.4934,   10.1897,
    9.3578,      8.54876,   7.53124,     6.74291,     6.02192,   5.36648,
    4.63665,     4.04725,   3.54359,     3.09749,     2.61532,   2.08684,
    1.71593,     1.2976,    0.964001,    0.609931,    0.27848,   -0.0997042,
    -0.621426,   -0.939213, 3878.21,     3616.41,     3372.26,   3262.13,
    3239.87,     3216.0,    3190.41,     3162.97,     3133.55,   3102.0,
    3068.16,     3031.89,   2992.99,     2951.28,     2906.56,   2858.61,
    2807.19,     2752.05,   2692.93,     2629.54,     2561.57,   2488.68,
    2410.53,     2326.73,   2236.87,     2140.52,     2037.2,    1926.42,
    1807.64,     1680.27,   1543.69,     1397.25,     1240.22,   1071.84,
    891.298,     697.706,   490.124,     288.128,     276.801,   264.656,
    251.632,     237.667,   222.694,     206.638,     189.421,   170.961,
    151.166,     141.373,   133.448,     124.951,     115.84,    106.071,
    96.9644,     91.7517,   86.1622,     80.1688,     74.1127,   68.9063,
    63.3237,     57.589,    53.7559,     49.6457,     45.5549,   41.4537,
    37.9379,     35.8015,   33.5493,     31.1796,     28.5478,   26.0794,
    24.3134,     22.5487,   20.3048,     18.6031,     17.0106,   15.4354,
    14.3602,     12.9749,   11.7972,     10.814,      9.9302,    8.4557,
    7.93302,     7.15271,   6.17016,     5.54806,     4.93795,   4.19971,
    3.64195,     3.15717,   2.65328,     2.14957,     1.84931,   1.41893,
    1.01006,     0.70709,   0.351663,    0.0211478,   -0.442963, -1.2852,
    3896.53,     3633.62,   3388.42,     3243.82,     3221.98,   3198.57,
    3173.46,     3146.54,   3117.67,     3086.72,     3053.53,   3017.95,
    2979.79,     2938.87,   2895.0,      2847.95,     2797.51,   2743.42,
    2685.42,     2623.23,   2556.55,     2485.04,     2408.37,   2326.16,
    2238.01,     2143.49,   2042.13,     1933.46,     1816.92,   1691.97,
    1557.99,     1414.32,   1260.27,     1095.09,     917.972,   728.054,
    524.411,     306.051,   279.659,     267.567,     254.6,     240.697,
    225.789,     209.803,   192.662,     174.283,     154.575,   142.914,
    135.529,     127.61,    119.119,     110.014,     100.562,   94.4874,
    87.9738,     80.9895,   73.7377,     69.4729,     64.8999,   59.9965,
    55.7498,     51.2668,   47.0335,     43.504,      39.9957,   37.4191,
    34.8325,     32.4736,   29.9998,     27.3919,     24.6697,   22.7586,
    21.1496,     19.6594,   17.9525,     16.0805,     14.9088,   13.3207,
    12.0359,     11.2218,   10.2147,     9.12296,     8.09035,   7.00956,
    6.4887,      5.80089,   5.12707,     4.38933,     3.80539,   3.26709,
    2.80089,     2.31001,   1.86004,     1.48263,     1.10131,   0.771289,
    0.400589,    0.120498,  -0.298471,   -1.12482,    3944.85,   3678.83,
    3430.74,     3250.54,   3228.97,     3205.84,     3181.05,   3154.46,
    3125.95,     3095.38,   3062.6,      3027.45,     2989.76,   2949.35,
    2906.02,     2859.56,   2809.74,     2756.31,     2699.03,   2637.61,
    2571.75,     2501.13,   2425.4,      2344.21,     2257.14,   2163.79,
    2063.68,     1956.35,   1841.25,     1717.84,     1585.51,   1443.62,
    1291.47,     1128.33,   953.397,     765.823,     564.693,   349.029,
    287.868,     275.6,     262.446,     248.34,      233.216,   216.998,
    199.608,     180.962,   160.968,     146.686,     138.94,    130.634,
    121.727,     112.177,   101.937,     95.4622,     88.9748,   82.0186,
    74.5597,     70.144,    65.807,      61.1566,     57.2371,   53.3558,
    49.3349,     45.5792,   41.6005,     38.8489,     35.9606,   33.2477,
    30.5597,     27.9303,   25.9842,     23.5366,     21.4042,   20.0685,
    18.0741,     16.5215,   15.3673,     14.0823,     12.749,    11.3954,
    10.4936,     9.1933,    8.42772,     7.48346,     6.8026,    5.99821,
    5.22613,     4.60632,   4.0312,      3.45362,     2.97547,   2.47865,
    2.03328,     1.62458,   1.17069,     0.80828,     0.516211,  0.164082,
    -0.277641,   -0.933436, 4216.28,     3931.54,     3665.99,   3438.87,
    3416.17,     3391.82,   3365.71,     3337.72,     3307.7,    3275.52,
    3241.0,      3204.0,    3164.32,     3121.77,     3076.15,   3027.23,
    2974.77,     2918.53,   2858.22,     2793.55,     2724.21,   2649.85,
    2570.13,     2484.64,   2392.97,     2294.68,     2189.29,   2076.28,
    1955.1,      1825.17,   1685.84,     1536.45,     1376.26,   1204.5,
    1020.32,     822.831,   611.07,      384.007,     284.096,   273.155,
    261.423,     248.844,   235.355,     220.892,     205.384,   188.755,
    170.924,     155.843,   147.999,     139.588,     130.57,    120.899,
    110.53,      102.505,   94.9638,     86.8777,     78.2073,   72.5334,
    67.7492,     62.6193,   58.3678,     54.6219,     50.6354,   46.9856,
    43.072,      39.658,    36.1165,     33.6335,     31.2002,   29.0548,
    27.5157,     25.2297,   22.3093,     20.8326,     18.9634,   17.1168,
    16.1651,     14.2,      12.9491,     12.103,      10.7064,   9.7509,
    8.71396,     7.9509,    6.92358,     6.28131,     5.41481,   4.79693,
    4.07361,     3.44519,   3.05981,     2.54515,     2.07799,   1.68561,
    1.2827,      0.918923,  0.590582,    0.246968,    -0.187512, -0.699759,
    4226.3,      3941.55,   3675.99,     3428.33,     3391.23,   3367.55,
    3342.16,     3314.94,   3285.75,     3254.44,     3220.88,   3184.89,
    3146.3,      3104.92,   3060.55,     3012.98,     2961.97,   2907.27,
    2848.61,     2785.72,   2718.28,     2645.97,     2568.44,   2485.3,
    2396.15,     2300.56,   2198.06,     2088.16,     1970.31,   1843.94,
    1708.45,     1563.16,   1407.37,     1240.33,     1061.21,   869.144,
    663.202,     442.376,   308.58,      296.242,     283.011,   268.824,
    253.613,     237.301,   219.811,     201.058,     180.948,   161.865,
    153.448,     144.423,   134.746,     124.37,      113.244,   104.76,
    98.1218,     91.0043,   83.3724,     77.0185,     71.5547,   65.696,
    60.4577,     56.1329,   51.4955,     48.2893,     45.0798,   41.8013,
    38.3537,     35.3246,   32.2825,     29.3491,     27.5719,   26.1043,
    23.884,      21.7864,   19.7172,     17.8795,     16.3291,   14.8027,
    13.709,      12.5825,   11.1205,     9.93343,     9.08761,   8.12863,
    7.25913,     6.33784,   5.72107,     5.02023,     4.37766,   3.7296,
    3.22343,     2.67931,   2.18396,     1.76419,     1.43059,   0.998611,
    0.623553,    0.312094,  -0.0733409,  -0.571626,   4438.45,   4138.81,
    3859.37,     3598.76,   3525.05,     3500.62,     3474.43,   3446.34,
    3416.23,     3383.93,   3349.31,     3312.18,     3272.37,   3229.68,
    3183.9,      3134.82,   3082.19,     3025.76,     2965.25,   2900.36,
    2830.79,     2756.19,   2676.2,      2590.43,     2498.46,   2399.84,
    2294.09,     2180.71,   2059.13,     1928.76,     1788.97,   1639.08,
    1478.36,     1306.02,   1121.23,     923.088,     710.623,   482.804,
    308.069,     295.635,   282.301,     268.004,     252.674,   236.236,
    218.61,      199.71,    179.444,     157.995,     150.155,   141.748,
    132.734,     123.069,   112.705,     104.085,     98.1936,   91.8766,
    85.1031,     78.9346,   73.8401,     68.3773,     63.0458,   58.6084,
    53.8503,     49.818,    45.8996,     42.4638,     39.4123,   36.1081,
    32.5486,     30.5669,   28.5474,     26.4005,     23.596,    22.968,
    20.5314,     18.6774,   16.9219,     15.7442,     13.7593,   12.4979,
    11.4271,     10.4922,   9.45203,     8.3952,      7.45267,   6.6811,
    6.02069,     5.31019,   4.50185,     3.97278,     3.35533,   2.77224,
    2.27075,     1.88712,   1.44606,     1.05564,     0.756685,  0.362583,
    -0.00771474, -0.48191,  4672.43,     4356.96,     4062.74,   3788.35,
    3674.53,     3649.34,   3622.34,     3593.38,     3562.33,   3529.04,
    3493.34,     3455.07,   3414.02,     3370.01,     3322.82,   3272.22,
    3217.96,     3159.78,   3097.4,      3030.51,     2958.78,   2881.87,
    2799.41,     2710.98,   2616.16,     2514.49,     2405.47,   2288.58,
    2163.24,     2028.84,   1884.72,     1730.19,     1564.5,    1386.83,
    1196.32,     992.039,   772.998,     538.128,     320.114,   307.827,
    294.651,     280.524,   265.375,     249.132,     231.715,   213.039,
    193.013,     171.541,   160.737,     151.15,      140.87,    129.847,
    118.027,     107.358,   101.259,     94.7188,     87.706,    81.0312,
    76.2154,     71.0515,   65.5716,     60.0496,     54.1286,   49.6932,
    46.3899,     43.3759,   40.9606,     38.1662,     34.9682,   31.9688,
    29.1183,     26.608,    24.2965,     22.6817,     20.7341,   18.8823,
    17.408,      15.9057,   14.5084,     13.0801,     12.0249,   10.5197,
    9.43495,     8.74982,   7.72053,     7.00801,     6.19874,   5.47604,
    4.71915,     4.11908,   3.47018,     2.91128,     2.42017,   1.99981,
    1.54727,     1.15989,   0.73773,     0.431169,    0.0903971, -0.328325,
    4888.14,     4557.7,    4249.55,     3962.15,     3806.21,   3780.32,
    3752.56,     3722.79,   3690.87,     3656.64,     3619.93,   3580.58,
    3538.38,     3493.13,   3444.61,     3392.59,     3336.8,    3276.99,
    3212.85,     3144.07,   3070.33,     2991.26,     2906.47,   2815.55,
    2718.07,     2613.54,   2501.45,     2381.27,     2252.4,    2114.22,
    1966.05,     1807.17,   1636.81,     1454.15,     1258.27,   1048.25,
    823.046,     581.568,   322.639,     306.617,     294.074,   280.625,
    266.204,     250.741,   234.16,      216.382,     197.318,   176.877,
    164.276,     154.7,     144.432,     133.422,     121.616,   109.844,
    103.559,     96.8193,   89.5926,     82.1465,     76.4916,   70.4281,
    63.9264,     60.1597,   56.1542,     52.3944,     49.1173,   45.6137,
    41.89,       38.5062,   36.0045,     32.8219,     29.5022,   27.3235,
    25.3452,     23.6879,   21.9866,     19.9007,     17.8606,   16.3509,
    15.1669,     13.3944,   12.2838,     11.3804,     10.1677,   9.02214,
    8.26571,     6.99989,   6.35853,     5.62887,     4.78685,   4.29964,
    3.60124,     3.01451,   2.64102,     2.10088,     1.61854,   1.18913,
    0.834009,    0.468133,  0.156373,    -0.28598,    4796.35,   4472.75,
    4170.97,     3889.52,   3700.51,     3675.75,     3649.2,    3620.74,
    3590.22,     3557.49,   3522.4,      3484.77,     3444.42,   3401.15,
    3354.76,     3305.02,   3251.68,     3194.49,     3133.16,   3067.4,
    2996.89,     2921.29,   2840.22,     2753.29,     2660.08,   2560.13,
    2452.96,     2338.05,   2214.83,     2082.71,     1941.03,   1789.12,
    1626.24,     1451.58,   1264.3,      1063.48,     848.151,   617.262,
    369.687,     316.251,   303.479,     289.783,     275.098,   259.352,
    242.467,     224.363,   204.95,      184.134,     169.457,   159.977,
    149.812,     138.913,   127.226,     114.694,     107.626,   100.219,
    92.2759,     83.759,    78.6367,     73.2866,     67.5499,   63.5584,
    59.7099,     55.6577,   51.5192,     47.2239,     44.0081,   40.5871,
    37.0264,     34.1473,   31.8741,     29.018,      26.9978,   24.5241,
    21.6475,     20.2609,   18.7002,     16.9624,     15.6481,   14.3635,
    12.6479,     11.482,    10.5787,     8.9364,      8.38097,   7.47092,
    6.57234,     5.82057,   5.10422,     4.41993,     3.77753,   3.25457,
    2.64293,     2.24646,   1.77855,     1.3612,      0.884094,  0.528984,
    0.157809,    -0.174974, 4866.51,     4538.26,     4232.13,   3946.63,
    3719.49,     3694.87,   3668.48,     3640.18,     3609.83,   3577.29,
    3542.4,      3504.98,   3464.87,     3421.85,     3375.73,   3326.27,
    3273.24,     3216.37,   3155.4,      3090.02,     3019.91,   2944.74,
    2864.14,     2777.71,   2685.03,     2585.66,     2479.11,   2364.85,
    2242.34,     2110.98,   1970.12,     1819.08,     1657.13,   1483.48,
    1297.27,     1097.61,   883.517,     653.954,     407.801,   317.653,
    305.214,     291.875,   277.572,     262.236,     245.792,   228.159,
    209.251,     188.978,   172.693,     163.417,     153.47,    142.805,
    131.368,     119.106,   111.283,     104.067,     96.3304,   88.0344,
    81.9714,     76.1285,   69.8634,     65.1235,     60.9693,   56.5921,
    52.4766,     48.0637,   44.4606,     40.7344,     37.9406,   34.7991,
    31.1929,     28.9407,   26.7617,     24.7563,     23.2328,   21.1731,
    18.8704,     17.2405,   15.9327,     14.4845,     13.3472,   11.9219,
    10.7013,     9.67992,   8.60903,     7.69131,     6.75393,   6.20089,
    5.23712,     4.57434,   3.94548,     3.3652,      2.80412,   2.20616,
    1.82265,     1.37471,   0.993181,    0.682734,    0.279196,  -0.134774,
    5107.87,     4763.29,   4441.94,     4142.24,     3866.85,   3841.58,
    3814.5,      3785.45,   3754.3,      3720.91,     3685.1,    3646.7,
    3605.53,     3561.38,   3514.05,     3463.29,     3408.86,   3350.5,
    3287.92,     3220.82,   3148.87,     3071.72,     2989.0,    2900.3,
    2805.18,     2703.2,    2593.84,     2476.58,     2350.85,   2216.03,
    2071.47,     1916.46,   1750.24,     1572.02,     1380.92,   1176.0,
    956.281,     720.68,    468.052,     336.893,     323.205,   308.527,
    292.789,     275.914,   257.819,     238.417,     217.612,   195.304,
    175.431,     166.069,   156.031,     145.268,     133.726,   121.351,
    113.283,     107.212,   100.703,     93.7235,     87.0352,   80.2729,
    73.0218,     67.0321,   62.1876,     56.9929,     52.9442,   48.6466,
    45.0068,     41.3591,   38.2611,     35.272,      32.8801,   30.363,
    27.8306,     25.4033,   23.083,      21.4239,     19.8228,   17.7763,
    16.6032,     15.0785,   13.7306,     12.3737,     10.9741,   9.93027,
    8.80734,     7.73574,   7.12712,     6.35076,     5.39955,   4.90067,
    4.1408,      3.41252,   2.96223,     2.39402,     1.88825,   1.48308,
    1.05409,     0.696356,  0.279743,    -0.102651,   5083.21,   4740.6,
    4421.09,     4123.12,   3845.23,     3788.79,     3762.42,   3734.14,
    3703.81,     3671.3,    3636.44,     3599.05,     3558.97,   3515.98,
    3469.9,      3420.48,   3367.49,     3310.67,     3249.74,   3184.41,
    3114.36,     3039.25,   2958.71,     2872.35,     2779.74,   2680.45,
    2573.98,     2459.82,   2337.4,      2206.14,     2065.39,   1914.47,
    1752.65,     1579.13,   1393.07,     1193.56,     979.642,   750.259,
    504.299,     339.489,   326.042,     311.623,     296.163,   279.585,
    261.809,     242.748,   222.31,      200.395,     178.692,   170.057,
    160.799,     150.871,   140.226,     128.812,     119.261,   111.498,
    103.174,     94.2482,   87.0098,     81.4683,     75.5263,   69.7635,
    64.5597,     58.9798,   54.9455,     51.0446,     47.2276,   43.3387,
    39.9024,     36.5863,   34.3506,     31.4595,     28.3585,   26.1291,
    24.3911,     22.4267,   20.3141,     18.5075,     16.7873,   15.5233,
    14.2045,     12.3113,   11.2556,     10.3275,     9.28495,   8.19437,
    7.29218,     6.51707,   5.69398,     5.0183,      4.28514,   3.70626,
    3.04573,     2.58633,   2.01271,     1.57575,     1.17373,   0.768464,
    0.42526,     0.0479329, 5356.66,     4995.25,     4658.2,    4343.86,
    4050.71,     3954.81,   3927.52,     3898.26,     3866.89,   3833.25,
    3797.18,     3758.5,    3717.02,     3672.55,     3624.87,   3573.73,
    3518.91,     3460.12,   3397.08,     3329.49,     3257.01,   3179.29,
    3095.96,     3006.61,   2910.8,      2808.06,     2697.9,    2579.78,
    2453.12,     2317.31,   2171.69,     2015.54,     1848.11,   1668.57,
    1476.07,     1269.65,   1048.31,     810.979,     556.495,   347.776,
    334.359,     319.973,   304.546,     288.005,     270.269,   251.251,
    230.859,     208.992,   185.546,     175.254,     164.741,   153.468,
    141.381,     128.419,   117.547,     110.928,     103.829,   96.2183,
    89.2005,     83.6447,   77.6874,     71.679,      66.4347,   60.8114,
    56.1345,     51.7771,   47.6441,     43.7608,     40.7662,   38.3106,
    35.0268,     32.1574,   29.8972,     27.0466,     24.9107,   22.7849,
    20.7503,     18.9074,   17.328,      15.7841,     14.3066,   12.909,
    11.5771,     10.5109,   9.75847,     8.45148,     7.52999,   6.77434,
    5.78165,     5.11761,   4.39533,     3.80403,     3.19693,   2.68211,
    2.13537,     1.66135,   1.25591,     0.854697,    0.481827,  0.069556,
    5633.0,      5252.69,   4898.01,     4567.24,     4258.76,   4120.27,
    4092.11,     4061.9,    4029.52,     3994.8,      3957.56,   3917.64,
    3874.83,     3828.92,   3779.7,      3726.92,     3670.33,   3609.65,
    3544.58,     3474.81,   3400.0,      3319.78,     3233.76,   3141.53,
    3042.63,     2936.59,   2822.88,     2700.95,     2570.22,   2430.03,
    2279.72,     2118.54,   1945.71,     1760.4,      1561.69,   1348.62,
    1120.15,     875.177,   612.496,     357.481,     343.816,   329.164,
    313.453,     296.606,   278.541,     259.172,     238.402,   216.131,
    192.251,     180.239,   170.383,     159.815,     148.483,   136.332,
    124.573,     116.137,   107.092,     97.393,      88.4326,   83.7728,
    78.7762,     73.4915,   68.4841,     63.1148,     58.212,    53.7195,
    49.1462,     44.6901,   41.1947,     38.9332,     36.0931,   33.2473,
    30.9021,     28.3122,   25.6167,     23.7491,     21.9548,   20.0318,
    17.8734,     16.1495,   14.7206,     13.476,      12.0093,   10.663,
    9.75055,     8.6759,    7.89772,     7.18584,     6.21574,   5.28223,
    4.65599,     3.96414,   3.31007,     2.78611,     2.24588,   1.76221,
    1.28284,     0.888216,  0.517794,    0.110512,    5703.96,   5318.88,
    4959.76,     4624.85,   4312.51,     4135.0,      4107.01,   4077.0,
    4044.82,     4010.32,   3973.32,     3933.65,     3891.11,   3845.49,
    3796.58,     3744.14,   3687.91,     3627.61,     3562.95,   3493.62,
    3419.29,     3339.58,   3254.11,     3162.46,     3064.19,   2958.82,
    2845.83,     2724.68,   2594.77,     2455.47,     2306.11,   2145.95,
    1974.22,     1790.08,   1592.63,     1380.92,     1153.9,    910.476,
    649.461,     369.583,   343.478,     329.348,     314.197,   297.952,
    280.532,     261.853,   241.825,     220.349,     197.321,   183.181,
    173.147,     162.388,   150.852,     138.482,     125.846,   118.362,
    110.337,     101.733,   92.8804,     87.3582,     81.437,    75.0879,
    69.3139,     63.1667,   57.5672,     53.1435,     48.9443,   46.4634,
    43.6058,     40.1254,   37.2139,     34.5194,     31.5855,   29.3341,
    26.6192,     23.9939,   22.0161,     20.0124,     18.3293,   16.6959,
    15.4293,     13.9621,   12.4975,     11.3181,     10.1482,   9.16209,
    8.11654,     7.19967,   6.41745,     5.40031,     4.89052,   4.09146,
    3.44125,     2.93362,   2.37332,     1.76781,     1.35595,   0.976644,
    0.592571,    0.195179,  5971.75,     5568.56,     5192.53,   4841.85,
    4514.81,     4290.36,   4261.64,     4230.83,     4197.81,   4162.39,
    4124.41,     4083.69,   4040.03,     3993.21,     3943.01,   3889.18,
    3831.46,     3769.57,   3703.21,     3632.05,     3555.75,   3473.94,
    3386.21,     3292.14,   3191.27,     3083.12,     2967.15,   2842.8,
    2709.46,     2566.48,   2413.18,     2248.79,     2072.52,   1883.52,
    1680.86,     1463.55,   1230.53,     980.682,     712.773,   425.504,
    357.933,     343.016,   327.02,      309.868,     291.477,   271.757,
    250.612,     227.938,   203.626,     186.951,     177.377,   167.111,
    156.102,     144.299,   131.642,     124.053,     116.202,   107.783,
    98.7568,     91.6292,   84.1321,     76.0932,     70.4498,   65.0786,
    59.9724,     56.4973,   52.7439,     48.3821,     44.0385,   40.8711,
    37.7979,     34.838,    32.7076,     30.4359,     27.8302,   25.1832,
    23.1991,     21.0388,   19.3883,     17.5896,     15.8991,   14.7377,
    13.0856,     11.9061,   10.4728,     9.33267,     8.37132,   7.37842,
    6.44042,     5.76498,   5.02353,     4.20064,     3.67145,   2.99093,
    2.36748,     1.82684,   1.49123,     1.07237,     0.61822,   0.245063,
    5932.95,     5532.85,   5159.71,     4811.74,     4487.21,   4226.1,
    4198.24,     4168.37,   4136.34,     4102.0,      4065.17,   4025.69,
    3983.35,     3937.95,   3889.27,     3837.07,     3781.1,    3721.08,
    3656.73,     3587.72,   3513.73,     3434.4,      3349.33,   3258.11,
    3160.3,      3055.42,   2942.96,     2822.38,     2693.08,   2554.43,
    2405.77,     2246.36,   2075.43,     1892.16,     1695.63,   1484.91,
    1258.95,     1016.67,   756.873,     478.305,     372.877,   357.272,
    340.54,      322.599,   303.361,     282.733,     260.615,   236.897,
    211.466,     191.353,   180.917,     169.726,     157.727,   144.861,
    131.065,     123.127,   116.216,     108.804,     100.858,   93.5658,
    86.0491,     77.9892,   72.4055,     67.9122,     63.1675,   58.6764,
    53.8608,     50.013,    46.0141,     42.4482,     38.8847,   35.5034,
    33.3577,     31.0334,   28.4033,     25.7649,     23.1408,   21.1321,
    19.8284,     17.6487,   16.1923,     14.714,      13.2352,   12.18,
    10.7547,     9.60235,   8.7062,      7.74767,     6.78119,   6.0066,
    5.20232,     4.46117,   3.80414,     3.14215,     2.57051,   2.08748,
    1.63226,     1.15377,   0.709206,    0.27307,     6239.84,   5818.52,
    5425.59,     5059.15,   4717.4,      4404.46,     4375.56,   4344.57,
    4311.33,     4275.7,    4237.49,     4196.52,     4152.59,   4105.48,
    4054.97,     4000.81,   3942.73,     3880.46,     3813.69,   3742.09,
    3665.32,     3582.99,   3494.73,     3400.08,     3298.59,   3189.76,
    3073.08,     2947.96,   2813.79,     2669.94,     2515.68,   2350.28,
    2172.93,     1982.76,   1778.84,     1560.19,     1325.74,   1074.34,
    804.78,      515.737,   366.827,     352.534,     337.208,   320.775,
    303.154,     284.259,   263.999,     242.275,     218.981,   198.09,
    187.652,     176.46,    164.459,     151.59,      137.792,   128.127,
    120.292,     111.89,    102.881,     95.4223,     88.5493,   81.1796,
    74.8007,     69.2984,   63.3984,     59.1873,     54.7288,   50.8751,
    46.9849,     44.0022,   40.9638,     37.8301,     34.2559,   31.1419,
    28.9964,     26.3182,   24.2462,     22.1093,     19.9897,   17.9817,
    16.2525,     14.8775,   13.6446,     12.4203,     11.2874,   9.94354,
    9.03005,     8.01746,   6.99442,     6.33297,     5.36386,   4.54168,
    3.83945,     3.32848,   2.65327,     2.15799,     1.59579,   1.22298,
    0.742336,    0.288587,  6319.93,     5893.37,     5495.56,   5124.56,
    4778.57,     4455.89,   4394.36,     4363.57,     4330.56,   4295.16,
    4257.2,      4216.5,    4172.85,     4126.05,     4075.87,   4022.07,
    3964.37,     3902.51,   3836.18,     3765.05,     3688.78,   3607.0,
    3519.31,     3425.28,   3324.46,     3216.35,     3100.43,   2976.13,
    2842.84,     2699.93,   2546.69,     2382.37,     2206.18,   2017.26,
    1814.68,     1597.47,   1364.55,     1114.81,     847.013,   559.866,
    371.457,     357.269,   342.055,     325.742,     308.25,    289.494,
    269.382,     247.817,   224.693,     201.878,     191.856,   181.11,
    169.588,     157.233,   143.985,     133.367,     125.141,   116.32,
    106.861,     98.5888,   91.4179,     83.7288,     77.0392,   72.2414,
    67.0968,     61.7797,   56.1183,     51.5792,     47.5231,   44.4086,
    41.4528,     38.3558,   34.8562,     31.4512,     29.2036,   27.1712,
    24.9382,     22.5158,   20.8579,     18.9722,     17.3742,   15.9312,
    13.7523,     12.8523,   11.7803,     10.5333,     9.10709,   8.15093,
    7.4162,      6.36639,   5.41628,     4.78969,     4.02551,   3.42298,
    2.8791,      2.29764,   1.83416,     1.19754,     0.811006,  0.443854,
    6450.9,      6015.46,   5609.36,     5230.64,     4877.44,   4548.05,
    4446.58,     4415.79,   4382.77,     4347.36,     4309.4,    4268.69,
    4225.04,     4178.23,   4128.05,     4074.23,     4016.53,   3954.66,
    3888.31,     3817.17,   3740.89,     3659.1,      3571.4,    3477.35,
    3376.52,     3268.39,   3152.45,     3028.13,     2894.83,   2751.9,
    2598.63,     2434.29,   2258.08,     2069.12,     1866.52,   1649.27,
    1416.32,     1166.54,   898.704,     611.514,     382.492,   367.804,
    352.055,     335.167,   317.059,     297.643,     276.823,   254.499,
    230.561,     204.894,   194.402,     183.357,     171.514,   158.814,
    145.197,     133.292,   124.629,     115.341,     105.381,   97.0258,
    91.7233,     86.0376,   80.2361,     74.8486,     69.0718,   64.1104,
    59.3329,     54.6349,   49.994,      45.8044,     41.7758,   38.9083,
    35.8027,     32.5658,   30.4782,     27.3057,     25.5804,   22.9877,
    20.9586,     18.9046,   17.6727,     15.9333,     14.3259,   13.1883,
    12.2072,     10.3806,   9.5854,      8.62077,     7.36261,   6.51669,
    5.77481,     5.00225,   4.24164,     3.59067,     2.94782,   2.35902,
    1.80603,     1.32822,   0.881409,    0.469945,    6515.29,   6075.76,
    5665.86,     5283.58,   4927.07,     4594.58,     4453.79,   4423.34,
    4390.68,     4355.67,   4318.12,     4277.86,     4234.69,   4188.4,
    4138.77,     4085.54,   4028.48,     3967.28,     3901.67,   3831.31,
    3755.87,     3674.98,   3588.24,     3495.24,     3395.51,   3288.57,
    3173.91,     3050.96,   2919.13,     2777.77,     2626.19,   2463.66,
    2289.38,     2102.51,   1902.13,     1687.27,     1456.89,   1209.86,
    944.971,     660.942,   394.432,     379.443,     363.371,   346.137,
    327.658,     307.843,   286.596,     263.814,     239.386,   213.192,
    199.444,     187.239,   174.151,     160.118,     145.07,    131.604,
    124.623,     117.138,   109.111,     101.052,     94.0057,   86.4497,
    78.8153,     73.761,    68.3413,     63.3863,     58.7491,   54.3034,
    50.3837,     46.6282,   43.0595,     40.3977,     37.5758,   34.1331,
    30.6856,     28.6693,   26.3823,     24.0838,     22.3181,   19.7775,
    17.5255,     16.635,    15.0642,     13.9377,     12.0844,   10.9852,
    9.80951,     8.72371,   7.73261,     6.83011,     6.01242,   5.06776,
    4.33323,     3.68849,   3.07886,     2.41089,     1.95736,   1.44894,
    0.973358,    0.495397,  6656.62,     6208.02,     5789.63,   5399.46,
    5035.58,     4696.22,   4513.65,     4483.25,     4450.64,   4415.68,
    4378.19,     4337.99,   4294.89,     4248.68,     4199.12,   4145.98,
    4089.0,      4027.91,   3962.4,      3892.15,     3816.83,   3736.06,
    3649.46,     3556.6,    3457.03,     3350.26,     3235.78,   3113.03,
    2981.4,      2840.26,   2688.92,     2526.64,     2352.64,   2166.06,
    1966.0,      1751.48,   1521.46,     1274.81,     1010.34,   726.763,
    422.687,     404.284,   386.521,     367.475,     347.052,   325.153,
    301.671,     276.493,   249.495,     220.546,     204.272,   192.739,
    180.374,     167.115,   152.897,     138.793,     131.098,   122.847,
    113.999,     104.893,   97.7869,     90.1671,     81.9995,   77.0817,
    71.8085,     66.5102,   61.3093,     56.1692,     51.9556,   47.766,
    43.8539,     40.2762,   36.8519,     33.7347,     32.1657,   29.9414,
    27.01,       24.4118,   22.4818,     20.7012,     18.5319,   17.4347,
    15.6588,     13.4149,   12.5963,     11.4289,     10.2165,   9.02285,
    8.18025,     6.93757,   6.16928,     5.19846,     4.6558,    3.82176,
    3.18267,     2.64107,   2.05124,     1.54464,     0.946572,  0.629517,
    6928.69,     6461.29,   6025.37,     5618.84,     5239.7,    4886.12,
    4657.56,     4626.36,   4592.91,     4557.03,     4518.57,   4477.33,
    4433.1,      4385.68,   4334.83,     4280.31,     4221.85,   4159.16,
    4091.95,     4019.87,   3942.59,     3859.72,     3770.86,   3675.58,
    3573.42,     3463.87,   3346.41,     3220.46,     3085.4,    2940.59,
    2785.3,      2618.8,    2440.27,     2248.83,     2043.56,   1823.45,
    1587.44,     1334.37,   1063.02,     772.049,     460.055,   401.439,
    384.359,     366.044,   346.405,     325.347,     302.767,   278.556,
    252.595,     224.757,   206.368,     194.928,     182.662,   169.509,
    155.405,     140.331,   133.52,      126.217,     118.386,   109.989,
    102.659,     94.8047,   86.3822,     79.7015,     72.9053,   66.4185,
    61.3753,     56.3323,   53.5346,     50.2854,     45.9738,   42.1464,
    38.7507,     36.1944,   32.9858,     29.6702,     27.4301,   26.2091,
    23.557,      21.6555,   19.6963,     17.7524,     15.7952,   14.3043,
    12.9746,     11.8042,   10.2118,     9.1235,      8.35196,   7.16067,
    6.32316,     5.4434,    4.80689,     4.05186,     3.28651,   2.73415,
    2.02829,     1.5879,    1.19595,     0.659286,    7140.44,   6658.68,
    6209.4,      5790.4,    5399.63,     5035.2,      4759.99,   4728.41,
    4694.54,     4658.23,   4619.29,     4577.54,     4532.77,   4484.76,
    4433.29,     4378.09,   4318.91,     4255.45,     4187.41,   4114.44,
    4036.21,     3952.32,   3862.36,     3765.91,     3662.49,   3551.59,
    3432.68,     3305.17,   3168.45,     3021.85,     2864.65,   2696.1,
    2515.36,     2321.57,   2113.76,     1890.94,     1652.02,   1395.83,
    1121.13,     826.574,   510.733,     410.152,     392.941,   374.486,
    354.697,     333.478,   310.726,     286.329,     260.17,    232.12,
    211.564,     201.175,   190.035,     178.089,     165.281,   151.547,
    141.143,     130.63,    119.356,     107.268,     100.698,   94.6699,
    88.2064,     82.1834,   76.0585,     69.7694,     64.4176,   58.679,
    54.4281,     49.9768,   46.0527,     42.4164,     39.275,    35.8492,
    33.0681,     30.6283,   28.1652,     26.5934,     23.8833,   21.5532,
    19.5305,     17.8736,   16.4693,     14.7192,     13.2147,   11.9012,
    10.9749,     9.52949,   8.36769,     7.44617,     6.65243,   5.79693,
    4.76529,     4.14007,   3.5247,      2.81354,     2.23762,   1.69797,
    1.17053,     0.675716,  7221.33,     6734.29,     6280.08,   5856.46,
    5461.41,     5092.98,   4775.25,     4743.9,      4710.28,   4674.23,
    4635.58,     4594.14,   4549.7,      4502.05,     4450.95,   4396.16,
    4337.42,     4274.42,   4206.88,     4134.45,     4056.79,   3973.52,
    3884.23,     3788.49,   3685.83,     3575.75,     3457.71,   3331.15,
    3195.44,     3049.92,   2893.88,     2726.57,     2547.16,   2354.79,
    2148.52,     1927.35,   1690.19,     1435.89,     1163.21,   870.824,
    557.311,     416.572,   399.62,      381.443,     361.952,   341.053,
    318.643,     294.614,   268.848,     241.22,      217.93,    206.345,
    193.924,     180.604,   166.322,     151.007,     140.85,    132.176,
    122.876,     112.903,   105.415,     98.5695,     91.2289,   84.221,
    77.2813,     69.9429,   64.4939,     58.6512,     55.2815,   52.131,
    47.9489,     43.7867,   40.1143,     36.9368,     34.2742,   32.0104,
    29.2185,     25.8873,   24.2278,     22.5588,     19.8117,   18.3096,
    16.698,      15.1774,   13.8085,     12.1463,     11.1349,   9.55732,
    8.77026,     7.60993,   6.83993,     5.9026,      5.04206,   4.30773,
    3.63194,     2.96117,   2.32844,     1.7791,      1.22438,   0.766525,
    7526.66,     7018.67,   6544.89,     6103.06,     5691.0,    5306.71,
    4948.32,     4903.07,   4868.57,     4831.57,     4791.89,   4749.35,
    4703.74,     4654.82,   4602.38,     4546.14,     4485.84,   4421.18,
    4351.85,     4277.51,   4197.79,     4112.32,     4020.66,   3922.39,
    3817.01,     3704.02,   3582.86,     3452.94,     3313.64,   3164.27,
    3004.1,      2832.36,   2648.21,     2450.75,     2239.02,   2011.99,
    1768.55,     1507.52,   1227.63,     927.51,      605.701,   419.665,
    403.384,     385.926,   367.207,     347.135,     325.612,   302.534,
    277.788,     251.254,   226.335,     214.204,     201.197,   187.25,
    172.294,     156.258,   144.048,     134.16,      123.559,   112.191,
    104.239,     98.5329,   92.4139,     86.3518,     80.4108,   74.0405,
    68.3354,     62.3279,   57.9503,     54.0156,     49.6392,   45.257,
    42.1524,     38.81,     35.7751,     33.6222,     30.8284,   27.1942,
    24.8315,     23.3622,   20.9236,     18.809,      17.1492,   15.522,
    14.1328,     12.6127,   11.3114,     10.18,       8.83316,   8.28804,
    6.94069,     6.22468,   5.17723,     4.45865,     3.65469,   2.93164,
    2.45612,     1.86071,   1.3823,      0.863139,    7707.01,   7186.88,
    6701.78,     6249.39,   5827.48,     5434.01,     5067.06,   4980.42,
    4945.75,     4908.58,   4868.72,     4825.99,     4780.16,   4731.02,
    4678.33,     4621.84,   4561.26,     4496.3,      4426.65,   4351.96,
    4271.88,     4186.01,   4093.93,     3995.2,      3889.34,   3775.82,
    3654.11,     3523.59,   3383.64,     3233.58,     3072.68,   2900.15,
    2715.15,     2516.78,   2304.07,     2075.99,     1831.43,   1569.2,
    1288.01,     986.504,   663.209,     433.522,     415.989,   397.189,
    377.031,     355.415,   332.238,     307.385,     280.736,   252.162,
    223.073,     212.168,   200.475,     187.936,     174.492,   160.077,
    148.072,     138.927,   129.121,     118.606,     109.308,   101.54,
    93.2102,     85.7289,   80.4577,     74.8056,     69.1349,   63.1641,
    58.4995,     54.6447,   51.1395,     47.6122,     43.4745,   40.1813,
    37.4005,     33.686,    30.7863,     28.5411,     25.9204,   23.6236,
    21.7438,     19.703,    17.6234,     16.2393,     14.8379,   12.7533,
    11.5681,     10.6127,   9.14455,     8.27242,     7.09872,   6.37162,
    5.40313,     4.58542,   3.85344,     3.22326,     2.50095,   1.99146,
    1.40946,     0.894394,  7867.76,     7336.88,     6841.78,   6380.04,
    5949.42,     5547.84,   5173.31,     5043.93,     5009.27,   4972.1,
    4932.25,     4889.52,   4843.71,     4794.58,     4741.9,    4685.41,
    4624.84,     4559.89,   4490.25,     4415.58,     4335.51,   4249.66,
    4157.6,      4058.89,   3953.04,     3839.54,     3717.85,   3587.35,
    3447.43,     3297.4,    3136.52,     2964.02,     2779.05,   2580.71,
    2368.04,     2140.0,    1895.49,     1633.3,      1352.16,   1050.71,
    727.466,     455.199,   435.947,     415.304,     393.169,   369.435,
    343.985,     316.696,   287.435,     256.06,      222.417,   210.714,
    199.571,     187.622,   174.809,     161.071,     148.851,   140.299,
    131.128,     121.295,   112.017,     104.443,     96.321,    88.4896,
    83.1846,     77.4963,   72.4883,     67.6991,     62.5438,   56.9939,
    52.2131,     47.9081,   44.8105,     41.2057,     36.9294,   34.0116,
    31.9401,     29.1147,   26.5101,     24.526,      22.0461,   19.2184,
    18.1902,     16.6841,   14.9299,     13.488,      12.2325,   10.5345,
    9.57141,     8.37532,   7.48374,     6.39327,     5.72196,   4.86207,
    4.02186,     3.2132,    2.63452,     2.07019,     1.57901,   0.994113,
    8111.25,     7563.92,   7053.48,     6577.45,     6133.5,    5719.47,
    5333.34,     5159.48,   5124.26,     5086.5,      5046.01,   5002.6,
    4956.05,     4906.13,   4852.61,     4795.22,     4733.68,   4667.69,
    4596.94,     4521.07,   4439.72,     4352.49,     4258.96,   4158.67,
    4051.13,     3935.81,   3812.17,     3679.59,     3537.43,   3384.99,
    3221.54,     3046.27,   2858.34,     2656.83,     2440.76,   2209.07,
    1960.64,     1694.25,   1408.62,     1102.34,     773.925,   454.76,
    437.217,     418.405,   398.235,     376.606,     353.414,   328.547,
    301.882,     273.291,   242.633,     227.108,     214.342,   200.653,
    185.975,     170.237,   155.258,     145.464,     134.964,   123.704,
    112.956,     106.012,   98.5648,     90.8499,     85.034,    78.7979,
    72.8154,     67.0326,   61.7524,     57.7878,     53.8367,   49.949,
    45.5284,     41.3292,   38.2337,     35.4424,     31.9793,   29.0844,
    27.0568,     24.8997,   22.0221,     21.0635,     18.8386,   16.7539,
    15.4489,     13.7936,   12.3565,     11.1035,     9.76898,   8.73218,
    7.78986,     6.67203,   5.8849,      4.91489,     4.22942,   3.46144,
    2.84806,     2.09319,   1.56296,     1.08943,     8003.29,   7463.56,
    6960.2,      6490.77,   6052.98,     5644.69,     5263.92,   5052.55,
    5018.44,     4981.87,   4942.65,     4900.6,      4855.51,   4807.16,
    4755.31,     4699.72,   4640.12,     4576.2,      4507.67,   4434.18,
    4355.38,     4270.89,   4180.29,     4083.15,     3978.98,   3867.29,
    3747.53,     3619.11,   3481.4,      3333.75,     3175.43,   3005.67,
    2823.63,     2628.45,   2419.15,     2194.74,     1954.1,    1696.07,
    1419.4,      1122.73,   804.628,     463.533,     436.72,    418.304,
    398.556,     377.381,   354.676,     330.33,      304.225,   276.233,
    246.219,     228.233,   215.291,     201.414,     186.535,   170.58,
    154.518,     145.639,   136.119,     125.912,     115.471,   108.644,
    101.324,     93.4755,   86.6288,     79.3234,     73.0454,   68.6139,
    63.8547,     58.7268,   54.4029,     52.0535,     46.9384,   41.1088,
    39.1132,     36.6834,   33.7572,     30.8223,     27.9098,   24.9539,
    23.1141,     21.5358,   19.4309,     17.4856,     15.7784,   14.2666,
    12.8301,     11.2443,   9.83268,     9.2586,      7.85814,   6.96365,
    5.85765,     5.12057,   4.31087,     3.53876,     2.95002,   2.28221,
    1.76336,     1.1541,    8396.81,     7830.44,     7302.21,   6809.6,
    6350.19,     5921.74,   5522.17,     5259.41,     5224.22,   5186.47,
    5146.0,      5102.61,   5056.08,     5006.18,     4952.69,   4895.32,
    4833.81,     4767.85,   4697.13,     4621.3,      4539.98,   4452.79,
    4359.3,      4259.06,   4151.57,     4036.31,     3912.72,   3780.2,
    3638.1,      3485.73,   3322.35,     3147.17,     2959.32,   2757.9,
    2541.93,     2310.34,   2062.02,     1795.76,     1510.25,   1204.11,
    875.846,     523.859,   453.347,     434.406,     414.097,   392.319,
    368.968,     343.93,    317.081,     288.293,     257.424,   236.411,
    223.101,     208.828,   193.525,     177.115,     159.52,    151.089,
    142.135,     132.534,   122.239,     114.203,     105.637,   96.4511,
    90.0456,     83.7793,   77.3294,     71.0992,     64.8089,   61.2314,
    57.3197,     52.8541,   48.4292,     43.9736,     39.0101,   36.3938,
    34.327,      31.5509,   28.5655,     26.0631,     23.7705,   21.3191,
    19.7241,     18.1716,   16.4143,     14.6049,     13.1722,   11.5629,
    10.4188,     9.15664,   8.34771,     7.09837,     6.0643,    5.3177,
    4.4613,      3.79777,   3.00181,     2.36876,     1.64907,   1.21801,
    8504.03,     7930.71,   7396.03,     6897.4,      6432.37,   5998.68,
    5594.22,     5286.78,   5251.83,     5214.35,     5174.16,   5131.07,
    5084.86,     5035.32,   4982.19,     4925.23,     4864.15,   4798.65,
    4728.42,     4653.12,   4572.37,     4485.79,     4392.95,   4293.4,
    4186.66,     4072.21,   3949.48,     3817.88,     3676.78,   3525.47,
    3363.24,     3189.27,   3002.74,     2802.72,     2588.26,   2358.29,
    2111.7,      1847.29,   1563.78,     1259.77,     933.799,   584.268,
    470.715,     450.588,   429.007,     405.866,     381.053,   354.446,
    325.917,     295.326,   262.524,     238.316,     226.009,   212.814,
    198.665,     183.494,   167.226,     156.775,     146.648,   135.789,
    124.146,     115.469,   106.795,     97.4938,     90.5252,   84.1813,
    77.8211,     73.2792,   68.409,      62.774,      56.9557,   52.7773,
    49.0882,     46.2003,   43.0945,     38.7361,     34.4758,   31.8937,
    28.6935,     27.6118,   24.153,      21.8434,     20.1249,   18.5707,
    16.9672,     14.9336,   13.787,      12.2044,     10.8833,   9.71033,
    8.23623,     7.34459,   6.44789,     5.43515,     4.47163,   3.96654,
    3.24287,     2.46807,   1.78116,     1.26643,     8748.08,   8157.88,
    7607.47,     7094.17,   6615.45,     6169.0,      5752.64,   5395.28,
    5359.83,     5321.81,   5281.04,     5237.32,     5190.45,   5140.19,
    5086.29,     5028.5,    4966.54,     4900.1,      4828.85,   4752.46,
    4670.55,     4582.71,   4488.53,     4387.55,     4279.26,   4163.15,
    4038.65,     3905.15,   3762.01,     3608.51,     3443.93,   3267.45,
    3078.22,     2875.32,   2657.75,     2424.45,     2174.3,    1906.07,
    1618.46,     1310.06,   979.373,     624.789,     467.006,   447.427,
    426.433,     403.922,   379.784,     353.901,     326.148,   296.39,
    264.481,     238.206,   226.476,     213.899,     200.412,   185.951,
    170.445,     159.424,   149.545,     138.951,     127.592,   118.687,
    110.319,     101.346,   93.3726,     85.8989,     78.052,    73.1659,
    67.9268,     62.9125,   57.6286,     53.4573,     49.4649,   46.2094,
    41.8333,     37.7367,   34.9239,     33.8669,     30.3441,   28.3549,
    25.1735,     23.065,    21.2119,     19.4838,     16.7637,   15.6334,
    13.6111,     12.7181,   10.9971,     9.64556,     8.64992,   7.59047,
    6.54852,     5.8594,    4.76227,     3.93444,     3.24007,   2.59653,
    1.95657,     1.3317,    8756.76,     8166.61,     7616.21,   7102.94,
    6624.24,     6177.8,    5761.46,     5373.17,     5327.76,   5290.45,
    5250.45,     5207.55,   5161.56,     5112.24,     5059.35,   5002.65,
    4941.85,     4876.65,   4806.74,     4731.78,     4651.4,    4565.22,
    4472.8,      4373.71,   4267.45,     4153.52,     4031.35,   3900.36,
    3759.89,     3609.28,   3447.78,     3274.61,     3088.93,   2889.83,
    2676.34,     2447.42,   2201.96,     1938.76,     1656.54,   1353.92,
    1029.43,     681.495,   484.008,     464.126,     442.807,   419.948,
    395.437,     369.154,   340.972,     310.753,     278.351,   248.626,
    235.875,     222.203,   207.544,     191.824,     174.969,   162.595,
    152.793,     142.282,   131.011,     122.236,     114.91,    107.054,
    99.1173,     91.1255,   82.5561,     76.236,      69.6926,   65.0724,
    60.9403,     56.076,    51.0176,     46.5797,     43.497,    40.3896,
    36.216,      33.6568,   29.7222,     28.8587,     25.624,    23.45,
    21.8318,     19.2378,   17.4656,     15.9799,     14.547,    12.7159,
    11.3539,     10.1715,   9.02911,     7.78037,     6.67315,   5.83738,
    4.85542,     4.1121,    3.4929,      2.71905,     2.10075,   1.50915,
    9280.81,     8654.9,    8071.17,     7526.8,      7019.11,   6545.63,
    6104.07,     5692.26,   5602.28,     5563.31,     5521.52,   5476.7,
    5428.65,     5377.13,   5321.88,     5262.64,     5199.12,   5131.0,
    5057.97,     4979.66,   4895.68,     4805.64,     4709.1,    4605.57,
    4494.57,     4375.54,   4247.91,     4111.06,     3964.31,   3806.96,
    3638.25,     3457.33,   3263.35,     3055.34,     2832.31,   2593.15,
    2336.71,     2061.74,   1766.9,      1450.75,     1111.76,   748.263,
    498.229,     478.666,   457.689,     435.197,     411.079,   385.218,
    357.488,     327.754,   295.871,     263.724,     249.362,   233.962,
    217.449,     199.743,   180.757,     165.958,     155.592,   144.476,
    132.557,     122.756,   115.29,      107.285,     99.1509,   91.2055,
    82.6859,     76.9535,   71.6418,     66.5694,     61.5054,   57.0817,
    52.7688,     49.055,    44.9256,     40.8265,     37.9722,   35.5769,
    32.2396,     29.6445,   26.1928,     24.1634,     21.4395,   19.986,
    18.06,       16.3352,   14.4174,     12.9652,     11.3611,   10.4775,
    9.10179,     7.82302,   7.04639,     6.05989,     5.20758,   4.20544,
    3.63062,     2.77247,   2.21122,     1.56051,     9483.65,   8843.92,
    8247.33,     7690.93,   7172.02,     6688.1,      6236.79,   5815.89,
    5682.64,     5643.28,   5601.07,     5555.81,     5507.28,   5455.25,
    5399.45,     5339.62,   5275.47,     5206.68,     5132.92,   5053.83,
    4969.02,     4878.09,   4780.58,     4676.03,     4563.92,   4443.71,
    4314.81,     4176.6,    4028.4,      3869.49,     3699.09,   3516.38,
    3320.47,     3110.4,    2885.14,     2643.61,     2384.63,   2106.93,
    1809.15,     1489.87,   1147.5,      780.395,     483.742,   466.529,
    448.072,     428.28,    407.059,     384.304,     359.904,   333.742,
    305.688,     275.607,   260.358,     244.413,     227.316,   208.984,
    189.327,     172.947,   162.877,     152.079,     140.5,     129.44,
    119.813,     109.491,   99.7652,     93.3513,     86.4739,   79.5717,
    72.3894,     66.7444,   62.6983,     58.6874,     54.5894,   49.3725,
    44.6625,     40.9453,   38.8594,     35.4469,     32.3371,   29.9946,
    27.8566,     24.4873,   22.4047,     20.707,      18.0961,   16.1714,
    14.9481,     13.3981,   12.0209,     10.7453,     9.48229,   8.35699,
    7.17827,     6.32778,   5.266,       4.36753,     3.63501,   2.92282,
    2.27564,     1.63507,   9773.11,     9113.89,     8499.11,   7925.76,
    7391.05,     6892.38,   6427.32,     5993.6,      5812.23,   5772.48,
    5729.86,     5684.16,   5635.15,     5582.61,     5526.26,   5465.85,
    5401.07,     5331.6,    5257.12,     5177.26,     5091.62,   4999.79,
    4901.33,     4795.75,   4682.55,     4561.16,     4431.0,    4291.43,
    4141.78,     3981.31,   3809.25,     3624.75,     3426.91,   3214.79,
    2987.33,     2743.43,   2481.91,     2201.48,     1900.8,    1578.38,
    1232.66,     861.96,    516.657,     496.891,     475.697,   452.971,
    428.603,     402.474,   374.456,     344.414,     312.201,   277.659,
    260.408,     245.226,   228.946,     211.49,      192.773,   175.411,
    164.111,     151.996,   139.004,     127.035,     119.715,   111.867,
    103.625,     95.8818,   87.5791,     81.6911,     77.6943,   72.8996,
    66.9613,     61.2105,   55.659,      51.1174,     46.6957,   42.2899,
    40.0552,     36.2799,   32.8867,     30.3272,     28.0887,   25.2971,
    23.5035,     20.8507,   19.061,      16.6793,     15.3911,   13.9036,
    12.3847,     10.7519,   9.89515,     8.5153,      7.30681,   6.23501,
    5.42311,     4.50217,   3.92564,     2.8984,      2.28498,   1.67366,
    10113.3,     9430.89,   8794.5,      8201.01,     7647.51,   7131.31,
    6649.91,     6200.95,   5969.04,     5928.61,     5885.25,   5838.76,
    5788.91,     5735.46,   5678.15,     5616.69,     5550.8,    5480.14,
    5404.37,     5323.13,   5236.02,     5142.61,     5042.45,   4935.06,
    4819.9,      4696.42,   4564.02,     4422.05,     4269.82,   4106.59,
    3931.56,     3743.88,   3542.65,     3326.86,     3095.49,   2847.39,
    2581.36,     2296.11,   1990.24,     1662.27,     1310.6,    933.515,
    536.261,     514.772,   491.729,     467.022,     440.529,   412.122,
    381.662,     349.0,     313.978,     276.425,     257.097,   243.795,
    229.531,     214.237,   197.837,     181.283,     169.087,   156.009,
    141.986,     128.254,   120.986,     113.192,     104.865,   97.1317,
    88.8393,     82.1068,   77.5305,     72.2409,     65.562,    59.6995,
    55.4856,     51.2559,   47.0378,     43.2194,     40.4031,   37.3785,
    34.0982,     30.9334,   27.8727,     25.7577,     22.7331,   22.0897,
    19.6686,     17.9353,   15.8846,     14.3181,     12.6602,   11.2718,
    9.56423,     8.77912,   7.76855,     6.42229,     5.6869,    4.54511,
    3.91593,     3.13882,   2.49378,     1.76307,     10029.7,   9353.31,
    8722.47,     8134.16,   7585.49,     7073.8,      6596.6,    6151.55,
    5879.5,      5839.95,   5797.54,     5752.07,     5703.31,   5651.03,
    5594.97,     5534.85,   5470.4,      5401.28,     5327.17,   5247.71,
    5162.5,      5071.13,   4973.16,     4868.11,     4755.47,   4634.69,
    4505.18,     4366.32,   4217.41,     4057.75,     3886.54,   3702.97,
    3506.13,     3295.06,   3068.74,     2826.06,     2565.85,   2286.83,
    1987.65,     1666.84,   1322.85,     954.007,     558.505,   501.036,
    480.177,     457.81,    433.826,     408.11,      380.535,   350.968,
    319.263,     285.268,   263.938,     250.258,     235.588,   219.859,
    202.993,     185.297,   174.58,      163.088,     150.766,   137.722,
    128.08,      117.741,   106.654,     100.102,     93.6225,   86.6385,
    79.0776,     71.5945,   66.783,      62.152,      58.6007,   53.8892,
    48.492,      44.6961,   41.0465,     38.1289,     35.2799,   31.5361,
    28.72,       26.8889,   24.5602,     22.0695,     19.721,    18.0584,
    16.3877,     14.4059,   13.0187,     11.737,      10.1179,   8.94121,
    7.83089,     6.87101,   5.97591,     4.92127,     4.01135,   3.20809,
    2.40429,     1.92594,   10384.9,     9684.13,     9030.6,    8421.11,
    7852.7,      7322.59,   6828.22,     6367.16,     6041.64,   6001.29,
    5958.02,     5911.63,   5861.88,     5808.54,     5751.35,   5690.02,
    5624.26,     5553.74,   5478.13,     5397.06,     5310.13,   5216.91,
    5116.96,     5009.78,   4894.86,     4771.64,     4639.51,   4497.83,
    4345.91,     4183.02,   4008.35,     3821.06,     3620.23,   3404.89,
    3173.99,     2926.4,    2660.92,     2376.26,     2071.02,   1743.73,
    1392.78,     1016.47,   612.959,     508.836,     488.079,   465.823,
    441.958,     416.369,   388.93,      359.509,     327.961,   294.134,
    269.817,     255.343,   239.823,     223.181,     205.337,   186.203,
    174.741,     163.128,   150.677,     137.325,     128.668,   119.863,
    110.421,     102.456,   94.4872,     86.894,      82.0593,   76.86,
    71.0069,     64.9877,   59.88,       54.9655,     50.2886,   46.337,
    42.5046,     38.8503,   35.5797,     32.7779,     29.2313,   26.5185,
    25.0339,     22.5873,   20.2473,     18.1428,     16.4522,   14.6477,
    13.1207,     12.3989,   10.3977,     9.27797,     8.3484,    7.21481,
    5.89224,     5.02061,   4.10806,     3.36492,     2.61114,   1.98667,
    10930.5,     10192.6,   9504.4,      8862.62,     8264.1,    7705.9,
    7185.34,     6699.86,   6312.01,     6270.14,     6225.23,   6177.09,
    6125.46,     6070.1,    6010.74,     5947.09,     5878.84,   5805.66,
    5727.19,     5643.05,   5552.83,     5456.09,     5352.36,   5241.13,
    5121.87,     4993.98,   4856.85,     4709.82,     4552.15,   4383.1,
    4201.82,     4007.45,   3799.03,     3575.54,     3335.91,   3078.96,
    2803.44,     2508.0,    2191.22,     1851.55,     1487.32,   1096.78,
    678.013,     522.754,   501.957,     479.658,     455.747,   430.108,
    402.616,     373.137,   341.529,     307.635,     280.501,   265.515,
    249.446,     232.215,   213.74,      193.929,     181.158,   169.394,
    156.781,     143.256,   133.181,     123.442,     112.999,   104.21,
    95.9417,     87.3934,   80.716,      73.556,      68.8698,   64.081,
    59.7143,     55.0826,   50.2005,     46.304,      43.156,    40.2507,
    36.3823,     32.304,    30.0448,     28.0864,     25.1472,   22.8776,
    20.9411,     18.7168,   17.205,      15.1979,     13.9979,   12.0744,
    10.9221,     9.24395,   8.2051,      7.27959,     6.05938,   5.3158,
    4.28677,     3.51112,   2.63064,     1.92182,     10462.1,   9756.71,
    9098.87,     8485.38,   7913.25,     7379.66,     6882.04,   6417.96,
    6003.33,     5964.04,   5921.92,     5876.74,     5828.31,   5776.37,
    5720.68,     5660.97,   5596.94,     5528.29,     5454.67,   5375.73,
    5291.09,     5200.33,   5103.01,     4998.66,     4886.77,   4766.79,
    4638.14,     4500.19,   4352.28,     4193.67,     4023.61,   3841.25,
    3645.71,     3436.05,   3211.23,     2970.17,     2711.68,   2434.51,
    2137.32,     1818.64,   1476.94,     1110.54,     717.667,   526.471,
    505.577,     483.172,   459.148,     433.388,     405.767,   376.149,
    344.392,     310.339,   280.509,     265.372,     249.141,   231.738,
    213.077,     193.067,   179.347,     167.947,     155.724,   142.617,
    132.785,     124.12,    114.829,     106.958,     100.144,   92.8407,
    85.7593,     78.1662,   72.6644,     67.3306,     61.474,    56.0555,
    52.6668,     47.5409,   43.1665,     40.6017,     37.4911,   33.9897,
    31.9369,     27.5165,   25.6284,     23.4274,     21.3787,   19.3716,
    17.3868,     15.8785,   14.0377,     12.4935,     11.4433,   9.76174,
    8.53343,     7.35866,   6.10021,     5.29092,     4.48069,   3.73212,
    2.84241,     2.13267,   11106.6,     10357.2,     9658.26,   9006.42,
    8398.53,     7831.6,    7302.88,     6809.8,      6349.95,   6283.93,
    6239.76,     6192.4,    6141.61,     6087.15,     6028.76,   5966.15,
    5899.01,     5827.02,   5749.83,     5667.06,     5578.31,   5483.15,
    5381.11,     5271.69,   5154.37,     5028.57,     4893.67,   4749.03,
    4593.94,     4427.63,   4249.31,     4058.1,      3853.08,   3633.24,
    3397.51,     3144.74,   2873.71,     2583.09,     2271.47,   1937.33,
    1579.04,     1194.86,   782.911,     536.129,     515.417,   493.208,
    469.394,     443.859,   416.479,     387.12,      355.64,    321.885,
    289.749,     274.442,   258.029,     240.431,     221.56,    201.326,
    186.086,     174.271,   161.601,     148.017,     137.375,   128.809,
    119.623,     111.107,   103.596,     95.5435,     87.4099,   78.7506,
    72.6575,     67.4262,   62.1354,     57.0382,     54.3306,   49.9929,
    45.2289,     41.6823,   39.3906,     33.9305,     32.1297,   28.4111,
    27.1311,     24.3799,   22.0269,     19.736,      18.2407,   15.7918,
    14.1429,     13.0897,   11.6007,     10.2057,     8.84318,   7.64729,
    6.49426,     5.44835,   4.69593,     3.98022,     2.94453,   2.29701,
    10993.5,     10252.2,   9560.82,     8916.07,     8314.75,   7753.96,
    7230.96,     6743.22,   6288.34,     6179.08,     6136.19,   6090.21,
    6040.9,      5988.04,   5931.35,     5870.56,     5805.38,   5735.49,
    5660.55,     5580.19,   5494.03,     5401.64,     5302.57,   5196.35,
    5082.44,     4960.31,   4829.35,     4688.92,     4538.35,   4376.9,
    4203.77,     4018.14,   3819.09,     3605.66,     3376.8,    3131.4,
    2868.27,     2586.13,   2283.59,     1959.19,     1611.35,   1238.37,
    838.432,     552.538,   530.101,     506.043,     480.246,   452.585,
    422.925,     391.121,   357.019,     320.453,     283.108,   268.873,
    253.609,     237.242,   219.693,     200.875,     185.822,   175.294,
    164.006,     151.902,   141.48,      133.197,     124.314,   114.999,
    105.412,     95.1329,   87.3226,     79.8698,     73.7338,   68.3982,
    63.5823,     58.7989,   53.9244,     49.9556,     46.627,    42.5846,
    37.9508,     35.9838,   33.017,      29.7034,     28.2685,   24.7512,
    22.1694,     20.573,    18.3198,     16.2185,     15.5512,   13.2613,
    11.534,      10.1569,   8.97404,     7.5943,      6.76938,   5.68611,
    4.77675,     3.89067,   3.2071,      2.41812,     11504.0,   10727.8,
    10003.9,     9328.71,   8699.09,     8111.9,      7564.28,   7053.56,
    6577.27,     6418.64,   6374.35,     6326.85,     6275.92,   6221.32,
    6162.77,     6099.98,   6032.66,     5960.47,     5883.07,   5800.07,
    5711.08,     5615.65,   5513.33,     5403.61,     5285.96,   5159.81,
    5024.55,     4879.51,   4723.99,     4557.23,     4378.41,   4186.68,
    3981.09,     3760.64,   3524.26,     3270.8,      2999.02,   2707.6,
    2395.12,     2060.06,   1700.78,     1315.54,     902.464,   560.674,
    538.96,      515.677,   490.711,     463.941,     435.237,   404.458,
    371.455,     336.067,   298.121,     280.625,     262.861,   243.813,
    223.389,     201.488,   183.785,     174.715,     164.99,    154.562,
    144.086,     134.102,   123.398,     113.107,     105.968,   98.3136,
    90.9,        83.3495,   76.519,      70.5185,     66.0702,   62.6219,
    56.6545,     51.4746,   47.6674,     43.4517,     40.6442,   36.1795,
    32.8127,     30.397,    27.9844,     25.0818,     22.6137,   20.6901,
    18.6297,     17.441,    15.195,      13.629,      12.0176,   10.4138,
    8.99949,     8.06003,   6.72988,     5.59611,     5.14968,   3.87969,
    3.15649,     2.38796,   11987.4,     11178.4,     10424.0,   9720.39,
    9064.21,     8452.28,   7881.57,     7349.34,     6852.98,   6641.62,
    6596.2,      6547.5,    6495.28,     6439.29,     6379.25,   6314.88,
    6245.85,     6171.83,   6092.46,     6007.36,     5916.1,    5818.26,
    5713.34,     5600.84,   5480.21,     5350.86,     5212.16,   5063.44,
    4903.97,     4732.98,   4549.63,     4353.03,     4142.23,   3916.19,
    3673.81,     3413.92,   3135.25,     2836.43,     2516.03,   2172.46,
    1804.07,     1409.06,   985.503,     586.746,     563.144,   537.837,
    510.701,     481.604,   450.404,     416.949,     381.077,   342.612,
    301.368,     282.607,   267.087,     250.446,     232.603,   213.47,
    195.651,     184.182,   171.884,     158.698,     145.823,   135.75,
    124.948,     113.903,   105.733,     96.9733,     89.6284,   83.3869,
    77.3879,     72.0841,   66.4285,     60.3967,     56.0677,   51.5976,
    46.242,      43.6926,   42.4239,     39.0887,     34.8303,   32.0162,
    28.1634,     26.4693,   22.662,      21.067,      19.7721,   17.2093,
    15.7154,     13.9855,   12.5812,     10.9978,     9.6982,    8.30792,
    7.16918,     6.09094,   5.02141,     4.26607,     3.37521,   2.63999,
    12042.7,     11230.2,   10472.4,     9765.75,     9106.69,   8492.05,
    7918.83,     7384.25,   6885.7,      6628.06,     6583.1,    6534.89,
    6483.19,     6427.76,   6368.33,     6304.6,      6236.26,   6162.99,
    6084.42,     6000.17,   5909.83,     5812.97,     5709.1,    5597.73,
    5478.31,     5350.26,   5212.96,     5065.74,     4907.87,   4738.6,
    4557.09,     4362.47,   4153.78,     3930.01,     3690.07,   3432.79,
    3156.91,     2861.1,    2543.91,     2203.8,      1839.11,   1448.07,
    1028.77,     586.479,   563.852,     539.59,      513.574,   485.678,
    455.766,     423.693,   389.301,     352.425,     312.883,   290.635,
    273.971,     256.103,   236.943,     216.399,     196.33,    185.468,
    173.82,      161.331,   148.555,     138.419,     127.551,   115.987,
    107.516,     98.4329,   90.7888,     85.1702,     79.2823,   73.3309,
    67.9318,     63.7184,   58.67,       53.8571,     51.5135,   46.3434,
    41.4876,     37.8997,   34.8127,     32.2614,     30.1243,   26.7544,
    24.561,      21.7202,   20.1569,     17.5572,     16.262,    14.5052,
    12.4073,     10.9539,   9.7902,      8.6704,      7.22805,   6.31415,
    5.34355,     4.42599,   3.44973,     2.78847,     12387.8,   11551.7,
    10772.1,     10045.0,   9366.91,     8734.5,      8144.73,   7594.7,
    7081.75,     6771.66,   6725.93,     6676.89,     6624.3,    6567.92,
    6507.46,     6442.63,   6373.12,     6298.59,     6218.66,   6132.97,
    6041.08,     5942.54,   5836.89,     5723.61,     5602.13,   5471.88,
    5332.21,     5182.45,   5021.87,     4849.68,     4665.05,   4467.08,
    4254.8,      4027.18,   3783.11,     3521.4,      3240.78,   2939.88,
    2617.23,     2271.27,   1900.3,      1502.53,     1076.01,   618.668,
    556.672,     534.328,   510.368,     484.677,     457.13,    427.592,
    395.919,     361.957,   325.541,     302.146,     286.311,   269.331,
    251.125,     231.603,   211.12,      197.493,     182.881,   167.213,
    150.759,     140.718,   129.95,      118.404,     111.044,   103.626,
    95.9597,     88.2934,   80.7933,     76.292,      71.0221,   64.2217,
    59.3117,     55.6977,   52.7003,     47.4415,     43.0836,   39.9259,
    36.5433,     33.8877,   30.7199,     27.5415,     24.7786,   22.7329,
    20.8213,     18.5654,   16.2804,     14.747,      12.7353,   11.3823,
    10.1133,     8.66829,   7.47203,     6.35584,     5.52347,   4.46801,
    3.69202,     2.63323,   12304.2,     11474.3,     10700.3,   9978.44,
    9305.24,     8677.42,   8091.92,     7545.87,     7036.62,   6682.59,
    6638.0,      6590.19,   6538.92,     6483.95,     6425.0,    6361.8,
    6294.03,     6221.36,   6143.44,     6059.89,     5970.3,    5874.23,
    5771.23,     5660.78,   5542.34,     5415.35,     5279.19,   5133.18,
    4976.62,     4808.74,   4628.73,     4435.72,     4228.76,   4006.84,
    3768.88,     3513.72,   3240.13,     2946.76,     2632.2,    2294.9,
    1933.22,     1545.41,   1129.57,     683.681,     574.195,   550.255,
    524.586,     497.061,   467.548,     435.901,     401.968,   365.582,
    326.567,     299.394,   283.699,     266.87,      248.825,   229.476,
    208.728,     195.417,   181.665,     166.918,     151.106,   141.69,
    132.102,     121.822,   113.02,      104.12,      95.4233,   88.8184,
    82.0698,     79.5828,   76.0414,     68.0904,     61.0935,   55.1476,
    51.2488,     47.2079,   43.5516,     40.0544,     36.0834,   34.3448,
    31.2929,     27.8856,   25.4597,     23.3545,     20.7324,   19.1062,
    17.511,      15.081,    13.4134,     11.5183,     10.4506,   9.05917,
    8.09346,     6.53773,   5.71765,     4.52648,     3.58825,   2.9497,
    12286.8,     11458.3,   10685.7,     9965.14,     9293.16,   8666.47,
    8082.02,     7536.96,   7028.63,     6631.13,     6587.21,   6540.11,
    6489.61,     6435.46,   6377.4,      6315.15,     6248.39,   6176.81,
    6100.06,     6017.76,   5929.51,     5834.89,     5733.43,   5624.63,
    5507.97,     5382.89,   5248.76,     5104.94,     4950.72,   4785.37,
    4608.06,     4417.93,   4214.07,     3995.48,     3761.09,   3509.76,
    3240.26,     2951.29,   2641.44,     2309.2,      1952.94,   1570.94,
    1161.34,     722.13,    568.642,     546.369,     522.486,   496.878,
    469.418,     439.975,   408.403,     374.55,      338.251,   309.457,
    292.773,     274.883,   255.7,       235.131,     213.076,   199.205,
    186.242,     172.342,   157.438,     147.924,     139.061,   129.558,
    118.798,     107.014,   95.7107,     92.1822,     88.3986,   81.4609,
    74.0262,     67.6317,   61.4906,     55.9936,     51.9682,   48.8653,
    45.4672,     40.3928,   37.6634,     35.1564,     31.1685,   28.2622,
    25.5582,     23.0332,   20.7086,     19.1195,     16.9047,   15.6171,
    13.4118,     12.2807,   10.316,      9.18309,     8.02608,   6.77058,
    5.79935,     4.65651,   3.93593,     2.99248,     13364.8,   12462.6,
    11621.1,     10836.4,   10104.6,     9422.07,     8785.56,   8191.95,
    7638.35,     7157.88,   7110.73,     7060.17,     7005.96,   6947.83,
    6885.5,      6818.67,   6747.01,     6670.16,     6587.77,   6499.42,
    6404.68,     6303.1,    6194.18,     6077.38,     5952.15,   5817.86,
    5673.87,     5519.48,   5353.92,     5176.41,     4986.06,   4781.96,
    4563.11,     4328.44,   4076.81,     3807.0,      3517.69,   3207.48,
    2874.84,     2518.17,   2135.72,     1725.63,     1285.91,   814.409,
    599.303,     574.905,   548.743,     520.69,      490.61,    458.357,
    423.772,     386.688,   346.925,     313.226,     296.732,   279.045,
    260.081,     239.746,   217.941,     203.048,     190.221,   176.466,
    161.718,     150.559,   140.389,     129.483,     119.252,   109.291,
    98.7756,     92.4067,   85.5777,     79.5708,     73.3558,   69.2949,
    64.6717,     58.8589,   53.5479,     49.6066,     46.6594,   41.4383,
    39.2998,     35.1066,   31.5827,     28.8107,     26.3927,   24.6072,
    21.2725,     19.7137,   17.6881,     15.4405,     14.117,    12.5501,
    11.1425,     9.16223,   8.30575,     7.26482,     5.75918,   4.69123,
    3.94296,     3.05131,   12692.3,     11836.6,     11038.6,   10294.4,
    9600.37,     8953.11,   8349.46,     7786.5,      7261.48,   6771.84,
    6716.0,      6668.91,   6618.41,     6564.26,     6506.2,    6443.94,
    6377.19,     6305.61,   6228.86,     6146.56,     6058.31,   5963.69,
    5862.23,     5753.43,   5636.77,     5511.69,     5377.56,   5233.74,
    5079.53,     4914.17,   4736.86,     4546.74,     4342.88,   4124.28,
    3889.89,     3638.56,   3369.07,     3080.1,      2770.25,   2438.01,
    2081.75,     1699.76,   1290.15,     850.945,     602.879,   577.764,
    550.834,     521.958,   490.996,     457.795,     422.195,   384.023,
    343.092,     305.958,   291.46,      275.914,     259.245,   241.371,
    222.205,     206.724,   193.17,      178.636,     163.053,   150.921,
    140.758,     129.861,   120.773,     113.758,     106.237,   97.4463,
    87.964,      81.8874,   76.7545,     69.8645,     62.8806,   58.0483,
    54.2012,     50.2153,   45.3776,     42.6308,     39.7354,   35.1948,
    33.1631,     29.5559,   27.7055,     24.7685,     21.7548,   20.2701,
    17.9173,     16.1309,   14.3544,     12.7084,     10.8174,   9.99739,
    8.3083,      7.42284,   6.02016,     4.85102,     4.36874,   3.25624,
    13650.4,     12728.9,   11869.5,     11068.0,     10320.6,   9623.49,
    8973.39,     8367.11,   7801.69,     7274.38,     7168.17,   7118.06,
    7064.33,     7006.72,   6944.95,     6878.71,     6807.68,   6731.53,
    6649.86,     6562.3,    6468.41,     6367.73,     6259.78,   6144.02,
    6019.9,      5886.81,   5744.11,     5591.09,     5427.01,   5251.07,
    5062.42,     4860.14,   4643.24,     4410.66,     4161.27,   3893.87,
    3607.13,     3299.68,   2970.01,     2616.51,     2237.47,   1831.03,
    1395.22,     927.922,   615.189,     589.797,     562.569,   533.374,
    502.069,     468.501,   432.508,     393.913,     352.53,    311.604,
    295.663,     278.571,   260.243,     240.59,      219.518,   203.667,
    192.873,     181.3,     168.89,      157.0,       145.597,   133.37,
    122.67,      115.053,   106.886,     99.3315,     91.4926,   85.3494,
    80.0199,     75.1975,   70.0589,     61.9748,     56.7058,   53.0695,
    48.0766,     41.9948,   38.8757,     36.1937,     33.1069,   29.8139,
    28.4359,     25.3891,   22.5651,     20.3705,     18.5457,   16.1338,
    14.7139,     13.0833,   11.1556,     10.0001,     8.81418,   7.30464,
    6.42428,     5.34706,   4.15384,     3.20154,     13292.9,   12396.7,
    11560.9,     10781.4,   10054.5,     9376.56,     8744.31,   8154.69,
    7604.8,      7091.97,   6943.42,     6895.38,     6843.87,   6788.64,
    6729.42,     6665.92,   6597.83,     6524.82,     6446.54,   6362.59,
    6272.58,     6176.07,   6072.58,     5961.61,     5842.62,   5715.04,
    5578.23,     5431.53,   5274.24,     5105.58,     4924.72,   4730.8,
    4522.87,     4299.91,   4060.83,     3804.48,     3529.6,    3234.86,
    2918.82,     2579.93,   2216.56,     1826.93,     1409.14,   961.154,
    614.187,     590.731,   565.58,      538.611,     509.693,   478.686,
    445.438,     409.787,   371.559,     330.853,     313.792,   295.498,
    275.882,     254.848,   232.295,     213.493,     200.805,   187.2,
    172.612,     159.851,   150.3,       140.059,     129.404,   118.803,
    107.435,     99.7716,   93.3388,     87.6955,     82.7145,   76.0231,
    68.1266,     63.4538,   58.4047,     52.9867,     47.6316,   44.1521,
    41.0723,     38.5322,   35.4264,     32.5752,     27.4411,   25.4533,
    23.4812,     20.9674,   19.0339,     16.2151,     15.0448,   13.4212,
    11.3267,     10.3753,   8.83716,     7.62718,     6.44877,   5.31899,
    4.21313,     3.43316,   14016.0,     13070.5,     12188.8,   11366.5,
    10599.6,     9884.38,   9217.38,     8595.33,     8015.21,   7474.18,
    7270.4,      7220.5,    7167.0,      7109.63,     7048.11,   6982.14,
    6911.41,     6835.57,   6754.25,     6667.05,     6573.55,   6473.29,
    6365.78,     6250.51,   6126.91,     5994.37,     5852.26,   5699.87,
    5536.48,     5361.27,   5173.4,      4971.96,     4755.96,   4524.35,
    4276.0,      4009.71,   3724.17,     3417.99,     3089.69,   2737.66,
    2360.19,     1955.45,   1521.45,     1056.09,     646.141,   620.674,
    593.366,     564.085,   532.688,     499.022,     462.923,   424.215,
    382.709,     338.205,   318.122,     299.443,     279.414,   257.937,
    234.908,     214.541,   201.979,     188.509,     174.066,   161.14,
    152.979,     144.227,   134.762,     124.25,      112.978,   103.366,
    94.6126,     86.4947,   79.4169,     73.1033,     67.3799,   63.8147,
    59.2899,     52.9724,   50.2844,     45.0648,     41.3896,   38.4501,
    34.6801,     32.267,    28.533,      25.4935,     23.8585,   21.7437,
    19.1776,     17.4314,   15.4491,     13.8913,     12.1134,   10.4552,
    9.22427,     7.93502,   6.3396,      5.49362,     4.50585,   3.51418,
    13998.9,     13054.9,   12174.6,     11353.6,     10587.9,   9873.82,
    9207.88,     8586.81,   8007.61,     7467.44,     7216.48,   7167.54,
    7115.06,     7058.79,   6998.46,     6933.76,     6864.39,   6790.0,
    6710.24,     6624.71,   6533.01,     6434.67,     6329.23,   6216.17,
    6094.94,     5964.95,   5825.56,     5676.1,      5515.84,   5344.0,
    5159.74,     4962.16,   4750.31,     4523.15,     4279.56,   4018.38,
    3738.32,     3438.02,   3116.02,     2770.75,     2400.53,   2003.55,
    1577.89,     1121.46,   670.871,     642.542,     612.165,   579.593,
    544.668,     507.218,   467.062,     424.004,     377.835,   328.328,
    306.536,     290.851,   274.033,     255.999,     236.662,   217.928,
    204.986,     191.109,   176.23,      161.604,     150.898,   139.418,
    127.636,     120.658,   113.176,     104.488,     94.5384,   86.404,
    82.6477,     77.614,    70.9791,     64.11,       58.1447,   55.0587,
    50.2272,     46.0394,   42.2403,     38.4403,     34.8123,   32.328,
    29.759,      27.1696,   24.0993,     21.6098,     19.9044,   17.6148,
    15.8949,     13.6616,   11.9048,     10.6642,     9.07306,   7.99949,
    7.14386,     5.70693,   4.76106,     3.69104,     14671.6,   13681.5,
    12758.1,     11896.9,   11093.8,     10344.9,     9646.38,   8994.95,
    8387.43,     7820.86,   7511.77,     7460.84,     7406.23,   7347.67,
    7284.89,     7217.56,   7145.37,     7067.96,     6984.95,   6895.95,
    6800.52,     6698.19,   6588.46,     6470.8,      6344.65,   6209.37,
    6064.32,     5908.79,   5742.01,     5563.19,     5371.44,   5165.83,
    4945.37,     4708.97,   4455.49,     4183.69,     3892.24,   3579.74,
    3244.65,     2885.35,   2500.08,     2086.97,     1644.0,    1169.03,
    659.724,     625.716,   599.509,     571.407,     541.275,   508.965,
    474.32,      437.172,   397.339,     354.627,     328.935,   309.966,
    289.627,     267.818,   244.433,     221.172,     209.444,   196.869,
    183.385,     169.496,   159.08,      147.912,     135.937,   126.616,
    116.666,     107.284,   99.057,      90.7943,     83.7144,   77.0991,
    71.8282,     65.875,    59.9647,     56.2734,     51.7552,   47.7388,
    43.5437,     39.5681,   37.1005,     34.0322,     29.2159,   28.0437,
    25.1089,     22.468,    20.2363,     18.7503,     15.8213,   14.2413,
    12.6652,     11.0691,   9.57573,     8.42454,     6.94823,   5.71637,
    4.77091,     3.97993,   15113.4,     14093.4,     13142.1,   12254.9,
    11427.5,     10655.9,   9936.29,     9265.17,     8639.29,   8055.59,
    7688.6,      7636.83,   7581.31,     7521.78,     7457.94,   7389.49,
    7316.1,      7237.4,    7153.02,     7062.54,     6965.51,   6861.48,
    6749.93,     6630.32,   6502.06,     6364.53,     6217.07,   6058.95,
    5889.4,      5707.6,    5512.66,     5303.63,     5079.5,    4839.16,
    4581.47,     4305.14,   4008.85,     3691.15,     3350.49,   2985.21,
    2593.53,     2173.54,   1723.21,     1240.33,     722.548,   637.246,
    611.213,     583.297,   553.365,     521.269,     486.854,   449.952,
    410.383,     367.954,   339.497,     320.328,     299.774,   277.735,
    254.102,     229.096,   215.809,     201.562,     186.285,   170.031,
    160.216,     149.691,   138.406,     128.916,     119.079,   109.545,
    101.534,     93.266,    86.3516,     79.2448,     72.5431,   65.8216,
    59.4664,     57.7836,   54.5186,     49.64,       44.464,    40.5425,
    36.8459,     33.6889,   30.197,      28.3573,     24.5059,   22.8,
    20.7158,     17.8074,   16.3096,     14.9853,     13.3966,   11.2854,
    10.0036,     8.77268,   7.23452,     6.14384,     5.00684,   3.98185,
    14649.2,     13661.6,   12740.6,     11881.6,     11080.6,   10333.5,
    9636.84,     8987.08,   8381.12,     7816.0,      7412.96,   7363.67,
    7310.83,     7254.16,   7193.4,      7128.25,     7058.39,   6983.48,
    6903.15,     6817.02,   6724.67,     6625.65,     6519.46,   6405.61,
    6283.52,     6152.62,   6012.25,     5861.74,     5700.35,   5527.3,
    5341.74,     5142.78,   4929.43,     4700.67,     4455.37,   4192.35,
    3910.32,     3607.91,   3283.65,     2935.95,     2563.12,   2163.35,
    1734.69,     1275.05,   782.195,     650.671,     623.793,   594.973,
    564.07,      530.934,   495.403,     457.304,     416.453,   372.648,
    340.981,     322.265,   302.197,     280.678,     257.604,   232.863,
    218.654,     204.437,   189.193,     172.847,     162.004,   150.995,
    139.19,      129.685,   120.362,     111.362,     105.308,   98.6873,
    88.8847,     79.2516,   73.7505,     68.8251,     64.5933,   59.8134,
    54.6402,     49.3978,   44.1657,     39.6046,     38.9986,   34.8874,
    31.0214,     28.2884,   26.4416,     23.2657,     20.6285,   18.7993,
    17.0629,     14.9016,   13.1369,     11.8262,     10.1179,   8.49724,
    7.15829,     6.22153,   5.15969,     4.05114,     14862.2,   13860.4,
    12926.2,     12054.9,   11242.4,     10484.6,     9777.85,   9118.76,
    8504.07,     7930.83,   7475.18,     7425.9,      7373.06,   7316.39,
    7255.64,     7190.49,   7120.63,     7045.73,     6965.41,   6879.29,
    6786.94,     6687.92,   6581.74,     6467.9,      6345.82,   6214.92,
    6074.56,     5924.06,   5762.68,     5589.64,     5404.1,    5205.14,
    4991.81,     4763.06,   4517.78,     4254.78,     3972.77,   3670.37,
    3346.13,     2998.45,   2625.65,     2225.9,      1797.27,   1337.66,
    844.833,     664.16,    636.409,     606.651,     574.744,   540.53,
    503.844,     464.507,   422.327,     377.099,     342.163,   324.778,
    306.137,     286.149,   264.716,     241.734,     225.995,   211.035,
    194.994,     177.794,   166.577,     156.192,     145.056,   133.443,
    121.142,     109.058,   104.129,     98.8446,     92.0079,   84.6696,
    77.7311,     71.1656,   65.5314,     61.6806,     56.092,    49.6664,
    45.5651,     42.7784,   39.3103,     35.1972,     32.376,    28.9576,
    25.97,       23.909,    22.0629,     19.6489,     17.0162,   15.0889,
    13.6607,     12.0783,   10.2767,     8.93388,     7.59351,   6.46948,
    5.46157,     4.2433,    15669.3,     14611.9,     13625.9,   12706.3,
    11848.7,     11048.8,   10302.9,     9607.25,     8958.47,   8353.44,
    7825.06,     7773.6,    7718.42,     7659.25,     7595.81,   7527.78,
    7454.84,     7376.63,   7292.76,     7202.83,     7106.4,    7003.01,
    6892.14,     6773.26,   6645.79,     6509.11,     6362.55,   6205.39,
    6036.88,     5856.2,    5662.45,     5454.71,     5231.95,   4993.09,
    4736.97,     4462.35,   4167.87,     3852.12,     3513.54,   3150.5,
    2761.22,     2343.81,   1896.24,     1416.32,     901.718,   663.472,
    636.418,     607.41,    576.306,     542.953,     507.191,   468.844,
    427.725,     383.635,   346.256,     328.552,     309.568,   289.212,
    267.386,     243.982,   227.32,      212.676,     196.975,   180.138,
    166.801,     154.376,   141.053,     131.708,     125.186,   118.194,
    110.725,     102.716,   93.2961,     83.045,      77.7539,   72.7382,
    68.4279,     61.7607,   56.4105,     53.3678,     48.9039,   43.2748,
    40.409,      36.5316,   33.3625,     29.1891,     26.8827,   24.2159,
    22.3484,     19.7676,   17.6664,     15.6972,     14.1641,   11.8181,
    10.4315,     8.99468,   7.58257,     6.74348,     5.578,     4.63854,
    15929.0,     14854.2,   13851.9,     12917.1,     12045.4,   11232.4,
    10474.2,     9767.04,   9107.59,     8492.58,     7919.02,   7855.5,
    7800.07,     7740.63,   7676.9,      7608.57,     7535.29,   7456.72,
    7372.47,     7282.13,   7185.27,     7081.4,      6970.03,   6850.61,
    6722.56,     6585.25,   6438.02,     6280.15,     6110.88,   5929.37,
    5734.74,     5526.05,   5302.28,     5062.33,     4805.05,   4529.17,
    4233.35,     3916.16,   3576.05,     3211.35,     2820.3,    2400.99,
    1951.38,     1469.27,   952.328,     662.715,     637.217,   609.876,
    580.56,      549.125,   515.418,     479.276,     440.521,   398.966,
    360.568,     342.367,   322.851,     301.924,     279.486,   255.426,
    236.407,     220.074,   202.56,      183.781,     169.802,   158.627,
    146.645,     135.144,   124.221,     112.509,     106.564,   100.7,
    93.9077,     86.4547,   79.5011,     72.9252,     68.88,     62.7661,
    56.2988,     51.2611,   47.7407,     45.2031,     40.7285,   37.4855,
    32.7768,     30.0504,   27.6147,     24.6698,     21.9664,   19.62,
    17.273,      16.1944,   14.4507,     12.4562,     10.7622,   9.63808,
    8.11573,     6.74069,   5.63449,     4.46347,     16477.5,   15365.5,
    14328.4,     13361.2,   12459.2,     11617.9,     10833.4,   10101.8,
    9419.44,     8783.09,   8189.63,     8074.21,     8017.73,   7957.17,
    7892.23,     7822.6,    7747.94,     7667.88,     7582.03,   7489.99,
    7391.29,     7285.46,   7171.98,     7050.3,      6919.82,   6779.92,
    6629.9,      6469.05,   6296.57,     6111.63,     5913.32,   5700.68,
    5472.67,     5228.18,   4966.03,     4684.93,     4383.52,   4060.32,
    3713.77,     3342.18,   2943.73,     2516.48,     2058.36,   1567.13,
    1040.41,     692.338,   664.248,     634.128,     601.831,   567.2,
    530.066,     490.249,   447.554,     401.774,     356.678,   338.32,
    318.636,     297.529,   274.896,     250.628,     232.447,   219.938,
    206.525,     192.143,   179.155,     167.466,     154.932,   142.924,
    132.259,     120.824,   111.234,     101.499,     94.0738,   87.7298,
    80.8973,     73.868,    69.1459,     63.8703,     58.2147,   52.622,
    48.5924,     43.3593,   41.0716,     37.4795,     34.2388,   31.6101,
    28.3763,     26.014,    22.1173,     21.022,      18.5835,   15.9924,
    14.281,      12.3309,   11.373,      9.58246,     8.44207,   6.87409,
    6.00759,     4.67624,   16389.6,     15284.4,     14253.7,   13292.4,
    12395.9,     11559.8,   10780.1,     10052.9,     9374.76,   8742.3,
    8152.47,     7988.39,   7933.14,     7873.89,     7810.37,   7742.25,
    7669.21,     7590.89,   7506.91,     7416.87,     7320.31,   7216.78,
    7105.76,     6986.73,   6859.09,     6722.23,     6575.47,   6418.11,
    6249.38,     6068.45,   5874.45,     5666.43,     5443.38,   5204.21,
    4947.75,     4672.76,   4377.9,      4061.72,     3722.7,    3359.18,
    2969.39,     2551.43,   2103.26,     1622.71,     1107.42,   715.608,
    686.562,     655.417,   622.021,     586.212,     547.815,   506.643,
    462.496,     415.159,   365.309,     347.625,     328.662,   308.329,
    286.527,     263.149,   242.634,     226.59,      209.386,   190.938,
    175.27,      164.024,   151.964,     140.256,     130.58,    120.204,
    111.212,     102.351,   94.0299,     86.0567,     80.7321,   76.6231,
    70.0229,     65.1258,   61.5513,     55.0749,     50.1653,   44.5806,
    40.5856,     37.0127,   34.6198,     32.6745,     29.243,    25.0085,
    23.7267,     21.1971,   18.9967,     16.6628,     14.8205,   13.1546,
    11.8696,     9.96484,   8.37473,     7.32432,     6.02432,   4.76604,
    16390.1,     15285.4,   14255.2,     13294.4,     12398.4,   11562.7,
    10783.4,     10056.6,   9378.75,     8746.61,     8157.07,   7944.73,
    7890.31,     7831.95,   7769.38,     7702.28,     7630.34,   7553.2,
    7470.48,     7381.79,   7286.68,     7184.7,      7075.36,   6958.11,
    6832.38,     6697.57,   6553.02,     6398.03,     6231.83,   6053.62,
    5862.53,     5657.63,   5437.93,     5202.35,     4949.74,   4678.88,
    4388.44,     4077.02,   3743.09,     3385.02,     3001.09,   2589.4,
    2147.96,     1674.62,   1167.08,     731.101,     702.061,   670.923,
    637.534,     601.732,   563.343,     522.179,     478.041,   430.713,
    379.965,     357.993,   336.93,      314.346,     290.129,   264.162,
    241.481,     227.215,   211.919,     195.517,     180.227,   168.47,
    155.862,     143.553,   135.048,     125.929,     116.279,   106.004,
    97.7561,     92.1681,   85.5961,     78.1132,     71.5054,   66.8172,
    64.5927,     55.0588,   49.6827,     45.8708,     42.4726,   38.3799,
    36.3726,     32.9416,   29.447,      26.932,      23.5138,   21.5392,
    19.255,      17.0957,   15.0791,     13.2568,     11.1899,   10.268,
    8.28288,     7.20597,   5.93312,     5.15669,     16369.2,   15266.2,
    14237.4,     13278.1,   12383.3,     11548.9,     10770.8,   10045.0,
    9368.19,     8736.99,   8148.31,     7889.15,     7835.52,   7778.03,
    7716.38,     7650.27,   7579.39,     7503.38,     7421.88,   7334.49,
    7240.79,     7140.31,   7032.58,     6917.05,     6793.18,   6660.36,
    6517.94,     6365.22,   6201.47,     6025.89,     5837.62,   5635.74,
    5419.27,     5187.16,   4938.27,     4671.4,      4385.24,   4078.4,
    3749.39,     3396.6,    3018.31,     2612.69,     2177.75,   1711.39,
    1211.32,     732.583,   703.757,     672.849,     639.707,   604.17,
    566.064,     525.205,   481.393,     434.415,     384.042,   359.328,
    338.757,     316.699,   293.048,     267.686,     244.128,   229.991,
    214.833,     198.58,    182.958,     171.977,     160.203,   148.091,
    139.142,     129.547,   120.391,     111.532,     102.961,   95.3782,
    88.0329,     81.0197,   72.4624,     65.1345,     61.826,    56.8843,
    52.2948,     47.6406,   44.4471,     40.3269,     36.7584,   34.9317,
    29.4502,     26.8262,   24.6949,     22.0271,     20.3372,   17.4803,
    15.3885,     14.0748,   12.286,      10.6552,     9.24454,   7.59833,
    6.32894,     5.29289,   17365.5,     16194.7,     15102.8,   14084.6,
    13134.9,     12249.3,   11423.3,     10653.0,     9934.65,   9264.71,
    8639.9,      8316.56,   8260.25,     8199.89,     8135.15,   8065.75,
    7991.32,     7911.52,   7825.95,     7734.19,     7635.8,    7530.31,
    7417.19,     7295.89,   7165.83,     7026.37,     6876.84,   6716.49,
    6544.56,     6360.2,    6162.52,     5950.56,     5723.27,   5479.56,
    5218.24,     4938.03,   4637.58,     4315.41,     3969.96,   3599.54,
    3202.35,     2776.46,   2319.8,      1830.13,     1305.07,   750.806,
    721.257,     689.571,   655.596,     619.166,     580.103,   538.217,
    493.304,     445.145,   393.506,     365.898,     346.243,   325.168,
    302.57,      278.338,   254.831,     243.157,     230.641,   217.219,
    202.421,     184.184,   164.63,      143.977,     135.879,   127.196,
    119.996,     114.881,   108.222,     97.9641,     88.3942,   80.4333,
    75.2071,     70.2858,   62.8081,     58.0663,     53.1906,   49.6454,
    45.9813,     41.3036,   37.6539,     33.2055,     30.5431,   27.7143,
    24.9589,     22.0998,   20.3762,     17.9296,     16.0256,   15.0223,
    12.4755,     10.9343,   8.90122,     7.74978,     6.25537,   5.47452,
    17694.1,     16500.9,   15388.2,     14350.4,     13382.6,   12480.0,
    11638.2,     10853.2,   10121.1,     9438.33,     8801.57,   8421.06,
    8364.58,     8304.02,   8239.08,     8169.44,     8094.78,   8014.71,
    7928.87,     7836.81,   7738.11,     7632.27,     7518.79,   7397.1,
    7266.62,     7126.71,   6976.68,     6815.82,     6643.33,   6458.38,
    6260.06,     6047.41,   5819.39,     5574.89,     5312.72,   5031.61,
    4730.18,     4406.97,   4060.4,      3688.78,     3290.31,   2863.05,
    2404.9,      1913.65,   1386.89,     822.071,     747.781,   714.156,
    678.101,     639.44,    597.986,     553.535,     505.873,   454.766,
    399.965,     367.184,   346.526,     324.375,     300.623,   275.154,
    248.897,     236.107,   222.392,     207.686,     192.135,   179.101,
    165.125,     150.14,    140.884,     131.493,     122.246,   113.825,
    105.097,     97.0959,   88.852,      80.8282,     74.1131,   68.4492,
    64.937,      58.7051,   53.0071,     49.6522,     46.9747,   42.0246,
    37.8611,     34.6811,   31.1025,     28.032,      25.5988,   23.2172,
    20.3775,     18.6525,   16.5765,     14.0032,     12.3896,   11.1038,
    9.75865,     7.86075,   6.36427,     5.5027,      18258.3,   17026.2,
    15877.2,     14805.6,   13806.2,     12874.1,     12004.9,   11194.3,
    10438.3,     9733.24,   9075.71,     8633.91,     8576.04,   8513.99,
    8447.46,     8376.12,   8299.62,     8217.6,      8129.64,   8035.34,
    7934.21,     7825.78,   7709.51,     7584.84,     7451.16,   7307.82,
    7154.12,     6989.31,   6812.59,     6623.11,     6419.92,   6202.06,
    5968.45,     5717.95,   5449.36,     5161.35,     4852.53,   4521.4,
    4166.33,     3785.6,    3377.36,     2939.62,     2470.24,   1966.95,
    1427.28,     848.605,   723.345,     693.801,     662.122,   628.153,
    591.73,      552.674,   510.796,     465.891,     417.741,   383.47,
    360.989,     336.884,   311.038,     283.323,     253.605,   239.559,
    224.993,     209.375,   192.629,     181.108,     168.99,    155.996,
    146.54,      137.303,   127.275,     116.181,     104.755,   97.1308,
    89.4042,     82.9089,   76.9087,     71.3408,     65.5261,   61.157,
    56.0456,     50.1626,   46.1262,     41.2469,     38.3926,   35.4802,
    32.9063,     28.3957,   26.047,      23.4765,     20.9411,   18.8996,
    16.9007,     14.6568,   13.3658,     11.1254,     9.22241,   8.04253,
    6.83952,     5.71241,   18492.0,     17244.5,     16081.1,   14996.2,
    13984.3,     13040.6,   12160.6,     11339.8,     10574.4,   9860.59,
    9194.87,     8696.55,   8638.72,     8576.71,     8510.22,   8438.92,
    8362.47,     8280.5,    8192.6,      8098.35,     7997.29,   7888.93,
    7772.73,     7648.14,   7514.54,     7371.29,     7217.69,   7052.98,
    6876.37,     6687.0,    6483.95,     6266.22,     6032.76,   5782.42,
    5513.99,     5226.17,   4917.54,     4586.61,     4231.77,   3851.28,
    3443.3,      3005.83,   2536.74,     2033.76,     1494.43,   916.12,
    737.864,     708.681,   677.389,     643.835,     607.857,   569.279,
    527.912,     483.557,   435.996,     398.592,     374.64,    348.958,
    321.419,     291.89,    260.227,     244.458,     230.069,   214.639,
    198.095,     186.663,   175.347,     163.212,     150.764,   137.616,
    124.558,     115.493,   105.784,     99.0124,     91.965,    86.1252,
    80.6635,     75.8372,   68.6571,     61.0836,     54.6921,   51.1746,
    48.4349,     42.3586,   38.1958,     36.2255,     32.8334,   29.1557,
    27.067,      23.6141,   21.6005,     19.0003,     16.9238,   14.9253,
    13.0985,     11.3348,   9.94501,     8.34336,     7.20578,   5.87597,
    19247.2,     17948.8,   16738.0,     15608.8,     14555.7,   13573.5,
    12657.6,     11803.4,   11006.7,     10263.7,     9570.87,   8998.85,
    8939.6,      8876.06,   8807.94,     8734.89,     8656.56,   8572.57,
    8482.51,     8385.94,   8282.4,      8171.37,     8052.32,   7924.66,
    7787.78,     7641.0,    7483.62,     7314.87,     7133.92,   6939.89,
    6731.84,     6508.76,   6269.55,     6013.06,     5738.04,   5443.13,
    5126.92,     4787.85,   4424.28,     4034.44,     3616.42,   3168.19,
    2687.57,     2172.22,   1619.63,     1027.1,      787.403,   753.632,
    717.42,      678.591,   636.955,     592.311,     544.441,   493.111,
    438.072,     394.483,   374.435,     352.938,     329.887,   305.171,
    278.668,     259.519,   241.525,     222.231,     201.543,   186.817,
    173.17,      158.536,   146.45,      135.465,     124.065,   115.921,
    107.188,     102.014,   96.85,       89.7656,     82.3476,   74.7301,
    68.9744,     63.1667,   57.6,        53.5377,     48.9585,   45.5781,
    39.6874,     35.9157,   33.4924,     30.8134,     27.5068,   25.4396,
    21.9528,     18.9221,   17.5214,     15.3532,     13.1903,   11.9929,
    9.85952,     8.59298,   7.12209,     5.80538,     18873.6,   17601.4,
    16414.9,     15308.4,   14276.5,     13314.1,     12416.6,   11579.6,
    10799.0,     10071.0,   9392.04,     8781.06,     8723.69,   8662.17,
    8596.22,     8525.49,   8449.65,     8368.33,     8281.14,   8187.64,
    8087.39,     7979.89,   7864.63,     7741.03,     7608.5,    7466.4,
    7314.02,     7150.64,   6975.44,     6787.59,     6586.16,   6370.17,
    6138.57,     5890.24,   5623.96,     5338.44,     5032.28,   4704.0,
    4351.99,     3974.55,   3569.83,     3135.86,     2670.53,   2171.57,
    1636.55,     1062.87,   778.789,     747.216,     713.361,   677.06,
    638.135,     596.397,   551.643,     503.655,     452.199,   407.34,
    386.201,     363.534,   339.229,     313.168,     285.223,   265.357,
    248.535,     230.496,   211.155,     196.368,     183.255,   169.194,
    157.977,     149.045,   139.468,     129.493,     118.799,   107.896,
    96.3328,     87.8714,   80.5077,     76.6959,     70.666,    63.8154,
    57.5269,     54.1203,   46.6343,     46.5125,     40.7393,   36.9729,
    33.731,      29.521,    27.0861,     25.6717,     22.0881,   20.4285,
    17.481,      15.7151,   13.9998,     11.935,      10.359,    8.75342,
    7.49912,     6.02511,   19363.3,     18058.0,     16840.7,   15705.4,
    14646.7,     13659.3,   12738.4,     11879.6,     11078.7,   10331.8,
    9635.17,     8985.52,   8897.66,     8835.52,     8768.88,   8697.43,
    8620.82,     8538.67,   8450.58,     8356.13,     8254.85,   8146.25,
    8029.8,      7904.94,   7771.06,     7627.49,     7473.56,   7308.5,
    7131.51,     6941.73,   6738.24,     6520.04,     6286.07,   6035.2,
    5766.19,     5477.74,   5168.45,     4836.81,     4481.2,    4099.89,
    3691.02,     3252.61,   2782.51,     2278.45,     1737.95,   1158.39,
    817.213,     783.148,   746.622,     707.455,     665.459,   620.427,
    572.141,     520.366,   464.849,     412.586,     387.863,   361.354,
    332.929,     302.45,    269.768,     250.645,     240.677,   229.989,
    218.529,     204.823,   189.157,     172.359,     157.429,   144.969,
    131.609,     124.028,   116.615,     108.812,     100.501,   91.4721,
    82.5638,     76.9309,   71.6532,     65.7757,     58.5077,   53.8285,
    50.9889,     44.6788,   41.6175,     38.0207,     34.5229,   32.473,
    28.1668,     25.5064,   22.9884,     20.1575,     18.0892,   15.6222,
    13.9602,     12.2514,   10.3787,     9.36373,     7.33609,   5.82475,
    19638.6,     18313.9,   17078.4,     15926.2,     14851.6,   13849.5,
    12914.9,     12043.3,   11230.5,     10472.4,     9765.44,   9106.11,
    8968.32,     8905.67,   8838.5,      8766.47,     8689.24,   8606.42,
    8517.62,     8422.41,   8320.31,     8210.83,     8093.45,   7967.58,
    7832.61,     7687.89,   7532.71,     7366.31,     7187.9,    6996.58,
    6791.45,     6571.48,   6335.62,     6082.72,     5811.54,   5520.76,
    5208.97,     4874.65,   4516.16,     4131.77,     3719.6,    3277.64,
    2803.75,     2295.6,    1750.74,     1166.5,      770.79,    741.211,
    709.494,     675.485,   639.018,     599.917,     557.989,   513.031,
    464.825,     416.519,   394.854,     371.623,     346.714,   320.004,
    291.364,     268.25,    250.712,     231.905,     211.74,    194.919,
    181.612,     167.343,   154.831,     146.046,     136.627,   127.123,
    117.069,     109.564,   103.41,      95.964,      87.7348,   79.3926,
    72.1363,     66.3972,   62.9997,     55.6561,     49.7153,   47.116,
    43.2917,     39.2713,   34.8933,     31.6905,     28.8553,   26.1106,
    23.3167,     21.4957,   18.8557,     16.4576,     14.488,    12.3812,
    11.1174,     9.5703,    7.68413,     6.40537,     20638.3,   19245.5,
    17946.7,     16735.4,   15605.7,     14552.2,     13569.7,   12653.4,
    11798.9,     11001.9,   10258.7,     9565.53,     9367.43,   9302.47,
    9232.81,     9158.12,   9078.03,     8992.16,     8900.08,   8801.34,
    8695.47,     8581.94,   8460.22,     8329.69,     8189.73,   8039.66,
    7878.75,     7706.2,    7521.19,     7322.8,      7110.08,   6881.98,
    6637.4,      6375.15,   6093.94,     5792.41,     5469.1,    5122.41,
    4750.67,     4352.07,   3924.66,     3466.37,     2974.95,   2448.02,
    1883.01,     1277.17,   810.505,     778.179,     743.517,   706.349,
    666.496,     623.763,   577.942,     528.809,     476.125,   420.192,
    397.468,     373.102,   346.975,     318.96,      288.921,   265.316,
    251.743,     237.188,   221.582,     205.267,     188.363,   170.238,
    154.712,     147.735,   140.254,     131.235,     121.179,   110.541,
    99.2557,     91.5982,   85.7161,     80.7454,     74.5315,   67.2762,
    61.4413,     57.9242,   52.9531,     48.0197,     43.8814,   39.2179,
    36.3302,     33.8993,   28.3979,     25.8349,     24.7626,   21.2714,
    18.8116,     16.3973,   14.5303,     12.8773,     11.015,    9.6652,
    7.73079,     6.70553,   20558.3,     19171.1,     17877.2,   16670.7,
    15545.4,     14496.0,   13517.3,     12604.6,     11753.4,   10959.6,
    10219.2,     9528.77,   9280.1,      9216.04,     9147.36,   9073.7,
    8994.73,     8910.05,   8819.25,     8721.89,     8617.49,   8505.55,
    8385.51,     8256.81,   8118.8,      7970.82,     7812.14,   7642.0,
    7459.56,     7263.93,   7054.17,     6829.25,     6588.08,   6329.48,
    6052.19,     5754.86,   5436.04,     5094.18,     4727.62,   4334.56,
    3913.11,     3461.19,   2976.62,     2457.02,     1899.88,   1302.48,
    787.923,     759.043,   728.075,     694.87,      659.265,   621.087,
    580.149,     536.254,   489.186,     438.716,     411.711,   384.968,
    356.292,     325.544,   292.574,     265.038,     251.006,   235.959,
    219.826,     203.845,   189.416,     173.945,     159.089,   150.049,
    140.357,     132.138,   124.589,     116.014,     106.244,   97.4121,
    89.1967,     80.694,    73.4861,     68.5056,     62.8646,   58.6267,
    53.555,      48.7229,   44.4271,     39.4738,     37.5776,   33.3263,
    29.0798,     27.2112,   23.8861,     21.7583,     19.2867,   16.9148,
    14.8708,     13.0383,   11.6133,     10.1363,     8.12747,   6.90267,
    21281.6,     19844.5,   18504.4,     17254.6,     16089.0,   15002.0,
    13988.2,     13042.8,   12161.1,     11338.8,     10571.9,   9856.75,
    9547.01,     9481.34,   9410.92,     9335.42,     9254.46,   9167.65,
    9074.56,     8974.75,   8867.72,     8752.96,     8629.91,   8497.97,
    8356.48,     8204.78,   8042.11,     7867.68,     7680.65,   7480.11,
    7265.07,     7034.49,   6787.25,     6522.14,     6237.87,   5933.06,
    5606.22,     5255.76,   4879.97,     4477.03,     4044.96,   3581.68,
    3084.91,     2552.24,   1981.08,     1368.64,     784.565,   755.19,
    723.693,     689.92,    653.706,     614.875,     573.238,   528.591,
    480.718,     429.386,   403.007,     380.417,     356.194,   330.22,
    302.37,      276.533,   260.991,     244.325,     226.456,   209.238,
    196.898,     183.667,   170.077,     160.093,     149.387,   137.335,
    123.932,     111.913,   103.064,     94.9506,     87.7469,   82.0292,
    76.1948,     69.5891,   62.5859,     57.4399,     53.2806,   48.5793,
    43.2549,     40.3087,   37.2439,     34.0919,     30.4548,   27.7882,
    24.457,      22.0959,   19.6289,     17.9853,     15.6777,   13.4107,
    11.4733,     9.92638,   8.15575,     6.78123,     21467.3,   20018.6,
    18667.4,     17407.4,   16232.3,     15136.3,     14114.3,   13161.1,
    12272.2,     11443.1,   10670.0,     9948.94,     9582.1,    9516.9,
    9446.99,     9372.03,   9291.65,     9205.47,     9113.05,   9013.96,
    8907.7,      8793.77,   8671.6,      8540.6,      8400.14,   8249.52,
    8088.02,     7914.85,   7729.16,     7530.06,     7316.57,   7087.65,
    6842.18,     6578.98,   6296.75,     5994.13,     5669.64,   5321.7,
    4948.61,     4548.57,   4119.61,     3659.65,     3166.46,   2637.62,
    2070.56,     1462.53,   826.368,     794.123,     759.546,   722.472,
    682.718,     640.09,    594.383,     545.372,     492.819,   436.469,
    406.616,     385.06,    361.947,     337.163,     310.587,   284.381,
    268.021,     250.48,    231.67,      212.713,     198.974,   184.242,
    168.625,     157.574,   145.724,     135.683,     128.09,    118.991,
    106.793,     96.6726,   90.4073,     81.9725,     73.8777,   70.5225,
    64.2685,     59.0622,   54.5589,     49.9357,     45.8807,   42.7467,
    38.7769,     35.3249,   30.291,      28.2588,     25.05,     23.4358,
    20.0906,     18.0861,   15.4376,     13.8355,     12.0843,   9.91241,
    8.43066,     7.17194,   21714.6,     20249.1,     18882.5,   17607.9,
    16419.3,     15310.7,   14276.9,     13312.7,     12413.5,   11574.9,
    10792.9,     10063.5,   9641.02,     9575.61,     9505.47,   9430.27,
    9349.63,     9263.16,   9170.44,     9071.03,     8964.42,   8850.12,
    8727.55,     8596.13,   8455.21,     8304.1,      8142.08,   7968.34,
    7782.05,     7582.3,    7368.11,     7138.44,     6892.18,   6628.12,
    6344.97,     6041.37,   5715.82,     5366.75,     4992.45,   4591.1,
    4160.74,     3699.29,   3204.49,     2673.93,     2105.03,   1495.01,
    840.91,      771.521,   740.512,     707.262,     671.609,   633.38,
    592.388,     548.434,   501.303,     450.766,     420.388,   400.24,
    378.637,     355.471,   330.632,     304.708,     285.232,   264.348,
    241.955,     218.575,   201.773,     183.758,     164.44,    152.96,
    141.209,     131.414,   125.684,     118.902,     109.027,   100.186,
    94.6696,     88.2354,   80.9755,     72.8114,     65.4298,   59.2468,
    55.3875,     52.058,    45.9083,     42.5353,     37.7458,   35.6082,
    32.81,       29.3946,   25.8967,     22.9931,     20.5147,   17.7972,
    15.6344,     13.7507,   12.4257,     10.3264,     9.03365,   7.02183};

/// dSigma/dtheta for π+- + π-+ -> π+- + π-+ + γ
const std::initializer_list<double> BREMS_PIPI_PIPI_OPP_DIFF_SIG_THETA = {
    0.0,         0.000205807, 0.000416519, 0.000634403, 0.00085428,
    0.00108768,  0.00132836,  0.00159196,  0.0018614,   0.00215746,
    0.00244745,  0.00275881,  0.00312114,  0.00345015,  0.00380256,
    0.00414097,  0.00453323,  0.00494294,  0.00536167,  0.00571321,
    0.0061138,   0.00646985,  0.00685511,  0.00727692,  0.00762749,
    0.00793374,  0.00827818,  0.00852658,  0.00881902,  0.00918588,
    0.00945323,  0.00965661,  0.00993551,  0.0101247,   0.0102038,
    0.0102638,   0.0105134,   0.0106249,   0.0106391,   0.0107496,
    0.0106255,   0.0106442,   0.0105768,   0.0104156,   0.0104346,
    0.0102927,   0.0100192,   0.00985772,  0.00971918,  0.00948562,
    0.00922236,  0.0089422,   0.00863652,  0.00827212,  0.00800609,
    0.00760861,  0.00718157,  0.00687594,  0.00648169,  0.00613385,
    0.00576442,  0.00526515,  0.00490205,  0.0045625,   0.00417125,
    0.00382139,  0.00345092,  0.00312909,  0.00278674,  0.00243635,
    0.00216563,  0.00186675,  0.00159596,  0.0013251,   0.0010815,
    0.000853999, 0.000627345, 0.00041666,  0.000205839, 1.37979e-08,
    0.0,         0.000374299, 0.000749184, 0.00114582,  0.00153967,
    0.00195288,  0.00239162,  0.00285078,  0.00333795,  0.00384404,
    0.00440084,  0.00490218,  0.00549615,  0.00613103,  0.00667245,
    0.00728951,  0.00788065,  0.00853942,  0.00918063,  0.00982789,
    0.0103445,   0.0109047,   0.0114745,   0.0121432,   0.0126279,
    0.0130591,   0.013365,    0.0140659,   0.0144423,   0.0147997,
    0.0151015,   0.0156581,   0.0157055,   0.0161505,   0.0163101,
    0.0164449,   0.0165592,   0.0165706,   0.0168402,   0.0168631,
    0.0167372,   0.0168039,   0.0167372,   0.0166021,   0.0164124,
    0.0162739,   0.016143,    0.0157329,   0.0156583,   0.015118,
    0.0148238,   0.0145216,   0.0141151,   0.0136583,   0.0132219,
    0.0125893,   0.0120709,   0.0115099,   0.0109778,   0.010269,
    0.00973253,  0.00916164,  0.00855902,  0.00782198,  0.00730271,
    0.0066993,   0.0060812,   0.00548278,  0.00495928,  0.00437862,
    0.00386448,  0.00335937,  0.00285739,  0.00238998,  0.00193669,
    0.00154091,  0.00113676,  0.000754518, 0.000374329, 2.48613e-08,
    0.0,         0.000580333, 0.00117929,  0.00176864,  0.00242615,
    0.0030267,   0.00373234,  0.00444579,  0.00515117,  0.00594127,
    0.00671447,  0.0075745,   0.00841101,  0.00921075,  0.0101553,
    0.0109961,   0.0118521,   0.0127748,   0.0135174,   0.0145377,
    0.0151365,   0.0160768,   0.0168222,   0.0174396,   0.0180652,
    0.0187813,   0.0193556,   0.019957,    0.0206927,   0.0206462,
    0.0215244,   0.0218662,   0.0220504,   0.0225616,   0.0225094,
    0.0228278,   0.0229102,   0.0233625,   0.0234139,   0.0234084,
    0.0231959,   0.0231088,   0.0229371,   0.0229789,   0.0228076,
    0.0226456,   0.0224829,   0.022066,    0.0217737,   0.0214568,
    0.0206682,   0.0202595,   0.0200911,   0.0195868,   0.0187457,
    0.0180967,   0.0173485,   0.0166901,   0.0159492,   0.0150736,
    0.0143155,   0.0136571,   0.0127729,   0.0117913,   0.0109628,
    0.0101547,   0.00919158,  0.00836063,  0.00756782,  0.00672058,
    0.00591415,  0.00516526,  0.00441561,  0.00372215,  0.003048,
    0.00237927,  0.00177668,  0.00117774,  0.000582671, 3.89952e-08,
    0.0,         0.000837671, 0.00168796,  0.00255385,  0.00343366,
    0.00434967,  0.00535217,  0.00628714,  0.00737125,  0.00846341,
    0.00955121,  0.0107189,   0.0118744,   0.0129554,   0.0141099,
    0.0151396,   0.0165597,   0.0177586,   0.0187894,   0.0197496,
    0.0208337,   0.0218817,   0.022755,    0.0235983,   0.024307,
    0.0248884,   0.0256886,   0.0266329,   0.0270434,   0.0274627,
    0.0281012,   0.0281854,   0.0283897,   0.0291977,   0.0294319,
    0.0294024,   0.029761,    0.0297622,   0.0298909,   0.0295232,
    0.0299219,   0.0299774,   0.0299346,   0.0295594,   0.0295912,
    0.0293259,   0.0290949,   0.0289175,   0.0284881,   0.0280023,
    0.0278246,   0.0270479,   0.0264041,   0.0257767,   0.0252386,
    0.0242426,   0.0236431,   0.0225927,   0.0215145,   0.0207954,
    0.0197399,   0.0189852,   0.0178468,   0.0164912,   0.0154298,
    0.0143656,   0.013046,    0.0119086,   0.0107271,   0.00963133,
    0.0084335,   0.00740071,  0.00631672,  0.00532724,  0.00435977,
    0.00342308,  0.00254822,  0.00167962,  0.000828692, 5.55785e-08,
    0.0,         0.0011343,   0.0022577,   0.00346106,  0.00469559,
    0.00586072,  0.00725032,  0.00860067,  0.0100356,   0.0114818,
    0.0130547,   0.0144479,   0.0158708,   0.0174863,   0.0189698,
    0.0205797,   0.0219269,   0.0231361,   0.0247673,   0.0257684,
    0.0271103,   0.0283678,   0.0289761,   0.0298359,   0.0309731,
    0.0321139,   0.0330052,   0.033495,    0.0339339,   0.0346568,
    0.0348025,   0.0352775,   0.0358957,   0.0361192,   0.0363595,
    0.0365654,   0.0362276,   0.0364989,   0.0371335,   0.036875,
    0.036943,    0.0368231,   0.0366469,   0.0364556,   0.0365892,
    0.036349,    0.0360601,   0.0356305,   0.0353875,   0.0348896,
    0.0341995,   0.0340552,   0.0332303,   0.0322316,   0.0318942,
    0.0313475,   0.030254,    0.0290242,   0.0282324,   0.0272891,
    0.0258603,   0.0248977,   0.0232571,   0.0220531,   0.0204991,
    0.018955,    0.0174032,   0.0158899,   0.014421,    0.0128123,
    0.0113689,   0.00994664,  0.00861194,  0.0072678,   0.00596392,
    0.00465345,  0.00344873,  0.00228126,  0.00113395,  7.55078e-08,
    0.0,         0.00148349,  0.00296627,  0.00443789,  0.00608332,
    0.0077722,   0.00937698,  0.0111147,   0.0131104,   0.014951,
    0.0167513,   0.018773,    0.0206981,   0.0224509,   0.0244141,
    0.0263663,   0.0282255,   0.0298017,   0.0314002,   0.0327971,
    0.0337675,   0.0355484,   0.0364998,   0.0374129,   0.0388679,
    0.0396153,   0.0399497,   0.0409289,   0.0416204,   0.0424654,
    0.0425442,   0.0423444,   0.0431362,   0.043649,    0.0437913,
    0.0432663,   0.0437827,   0.0439045,   0.043704,    0.0440129,
    0.0442064,   0.0438239,   0.0439871,   0.0440814,   0.0435673,
    0.0428967,   0.0436268,   0.0431543,   0.0431168,   0.0417788,
    0.0412945,   0.041417,    0.0409238,   0.0400406,   0.0394423,
    0.0384205,   0.0375684,   0.0366361,   0.0353524,   0.0338625,
    0.0330267,   0.031349,    0.0299237,   0.0280367,   0.0262968,
    0.0245184,   0.0224445,   0.0205993,   0.0187509,   0.0166622,
    0.0149008,   0.013106,    0.0111803,   0.00940476,  0.00771432,
    0.00604517,  0.0044917,   0.00297271,  0.00148156,  9.75258e-08,
    0.0,         0.00187252,  0.00375916,  0.00572885,  0.00771497,
    0.00973245,  0.0120552,   0.0142869,   0.0164106,   0.0190883,
    0.0212538,   0.0233579,   0.0257644,   0.0283523,   0.0308025,
    0.0326617,   0.034671,    0.0367946,   0.0384953,   0.0398762,
    0.0418043,   0.043337,    0.0445499,   0.0454889,   0.0474822,
    0.0472809,   0.0485898,   0.0488582,   0.0493133,   0.049715,
    0.0503902,   0.0498453,   0.0510519,   0.0504849,   0.0509343,
    0.0517064,   0.0512701,   0.0512349,   0.0516318,   0.0515047,
    0.0515,      0.0520108,   0.0516736,   0.0515645,   0.0517363,
    0.05153,     0.050725,    0.050772,    0.0502082,   0.0503964,
    0.0497495,   0.049375,    0.0489458,   0.0484335,   0.0470727,
    0.0461943,   0.0457208,   0.0447605,   0.0433858,   0.0417099,
    0.0403934,   0.0386938,   0.0368204,   0.0346714,   0.0327165,
    0.0302689,   0.0284753,   0.0260421,   0.0234613,   0.0211586,
    0.018853,    0.0167187,   0.0143015,   0.0119015,   0.00970817,
    0.00775263,  0.00572424,  0.00377031,  0.0018812,   1.24447e-07,
    0.0,         0.00231005,  0.00465118,  0.00706931,  0.00958742,
    0.0120352,   0.0147748,   0.0175419,   0.0205078,   0.023186,
    0.0260953,   0.0291206,   0.0318906,   0.0352639,   0.0378745,
    0.0405088,   0.0423969,   0.0446851,   0.0469772,   0.0485384,
    0.0502628,   0.0518127,   0.0531052,   0.0546896,   0.0560719,
    0.0562631,   0.0569086,   0.0569876,   0.0581876,   0.0588131,
    0.0587147,   0.0584418,   0.0592847,   0.0584735,   0.0589039,
    0.060083,    0.0589167,   0.0588431,   0.0594925,   0.058745,
    0.0597217,   0.0596649,   0.0588879,   0.0587033,   0.0591537,
    0.0597458,   0.0581476,   0.0585801,   0.0585377,   0.0584483,
    0.0581454,   0.0575688,   0.057007,    0.0569441,   0.0558967,
    0.0558341,   0.0542004,   0.0531436,   0.0513327,   0.0503852,
    0.0485771,   0.0466112,   0.0449687,   0.0426126,   0.0401046,
    0.0374592,   0.0349205,   0.0325398,   0.0291786,   0.0263624,
    0.0232751,   0.0204292,   0.0174966,   0.0148012,   0.0121853,
    0.00957452,  0.00707144,  0.0046647,   0.00230694,  1.55271e-07,
    0.0,         0.00279209,  0.00559562,  0.00866164,  0.0117865,
    0.0148404,   0.0178428,   0.0213729,   0.025029,    0.028283,
    0.031985,    0.0355373,   0.0393119,   0.042249,    0.0452225,
    0.0480521,   0.0512358,   0.0530631,   0.056136,    0.058219,
    0.0596924,   0.0604064,   0.0628189,   0.0635189,   0.0644925,
    0.0648878,   0.0657991,   0.0664024,   0.06636,     0.0661469,
    0.0671438,   0.0671911,   0.0671152,   0.0681195,   0.0674205,
    0.0678824,   0.0673895,   0.0668444,   0.0674696,   0.0676596,
    0.0668864,   0.0669619,   0.0671653,   0.0670104,   0.0674385,
    0.0675593,   0.0672378,   0.0673057,   0.0677555,   0.0670711,
    0.0672462,   0.0661306,   0.0663331,   0.0659838,   0.0649761,
    0.0645251,   0.0636729,   0.0625131,   0.0612909,   0.0596557,
    0.0577672,   0.0556972,   0.0538413,   0.0513822,   0.0487837,
    0.0458752,   0.0426975,   0.0386945,   0.0355166,   0.0320688,
    0.0287402,   0.0250103,   0.021494,    0.018062,    0.0148662,
    0.0115131,   0.00868608,  0.00567151,  0.00282378,  1.87737e-07,
    0.0,         0.00338847,  0.00685253,  0.0103406,   0.0141437,
    0.0177366,   0.021901,    0.0259636,   0.0299911,   0.0344003,
    0.0387006,   0.0428918,   0.0467546,   0.0502504,   0.0538883,
    0.0574897,   0.0600722,   0.0638707,   0.0665961,   0.069338,
    0.0699088,   0.071495,    0.0732947,   0.0738563,   0.0740293,
    0.075063,    0.0760458,   0.076024,    0.0769283,   0.0751737,
    0.0771988,   0.0758027,   0.076424,    0.0759353,   0.0758307,
    0.0764728,   0.0745397,   0.075887,    0.0760041,   0.0754467,
    0.0758547,   0.0760563,   0.076055,    0.0751042,   0.0760412,
    0.0763435,   0.0763751,   0.0759093,   0.0759698,   0.0773506,
    0.0762398,   0.0755278,   0.07642,     0.0751522,   0.0756281,
    0.0747821,   0.0744137,   0.0724593,   0.0711981,   0.0700502,
    0.0679494,   0.0664838,   0.0636717,   0.0609705,   0.0572843,
    0.0543108,   0.0505088,   0.0469932,   0.0428841,   0.0383167,
    0.0341041,   0.0298949,   0.0261411,   0.0219401,   0.0178069,
    0.014067,    0.0103724,   0.00683404,  0.00336039,  2.24559e-07,
    0.0,         0.00401216,  0.00813064,  0.0122746,   0.0164994,
    0.0214972,   0.0259791,   0.0307293,   0.0365415,   0.0411504,
    0.0455157,   0.0502493,   0.05591,     0.0608995,   0.0640365,
    0.0672343,   0.0704132,   0.0736545,   0.077253,    0.0781728,
    0.0806983,   0.0814757,   0.083429,    0.084671,    0.0859828,
    0.0856113,   0.0852683,   0.0868018,   0.0859677,   0.0843512,
    0.0855875,   0.0849404,   0.0858471,   0.0859266,   0.0856032,
    0.0851931,   0.0855534,   0.0853205,   0.085463,    0.0845213,
    0.0845597,   0.0835307,   0.0850464,   0.0845428,   0.0852687,
    0.0837268,   0.0853957,   0.0849073,   0.0849748,   0.086242,
    0.0861356,   0.0860422,   0.0861394,   0.085909,    0.0857127,
    0.0846068,   0.0853672,   0.0828874,   0.0824037,   0.080185,
    0.0791065,   0.0768365,   0.0743093,   0.0711439,   0.067631,
    0.0637655,   0.0593795,   0.0556726,   0.0511335,   0.0462913,
    0.0408426,   0.0356979,   0.031089,    0.0261037,   0.021268,
    0.0167271,   0.0122459,   0.00806815,  0.00400269,  2.68197e-07,
    0.0,         0.0047068,   0.00948102,  0.0145945,   0.0194743,
    0.0248902,   0.030762,    0.0366929,   0.0427093,   0.0484104,
    0.0539052,   0.0607997,   0.0645826,   0.0706028,   0.074163,
    0.0794003,   0.0823479,   0.0861206,   0.0880728,   0.0901628,
    0.0929324,   0.0941802,   0.0952609,   0.0963753,   0.0971006,
    0.0972031,   0.0962153,   0.0968612,   0.0970985,   0.0948159,
    0.0965376,   0.0955429,   0.0954778,   0.0945788,   0.0946666,
    0.0949178,   0.0937679,   0.0946085,   0.0928921,   0.0938344,
    0.0948503,   0.0939183,   0.0926702,   0.0929403,   0.0942464,
    0.0954598,   0.0956579,   0.0955347,   0.0958501,   0.0965063,
    0.0976964,   0.0958845,   0.0961898,   0.0965304,   0.0967552,
    0.0968294,   0.097278,    0.096167,    0.0947981,   0.09388,
    0.091008,    0.0889351,   0.0858696,   0.0835648,   0.0788763,
    0.0742711,   0.0703131,   0.0649703,   0.0598564,   0.0542655,
    0.0484113,   0.0426501,   0.0365888,   0.0308728,   0.0249667,
    0.019589,    0.0144653,   0.00959232,  0.00468983,  3.12899e-07,
    0.0,         0.00546156,  0.011045,    0.016907,    0.0227564,
    0.0288549,   0.0359138,   0.0426227,   0.0499702,   0.0559025,
    0.0640175,   0.0697216,   0.0757171,   0.0813391,   0.0874936,
    0.0913189,   0.0958596,   0.0990603,   0.102027,    0.104441,
    0.105691,    0.106409,    0.109058,    0.11093,     0.109969,
    0.108325,    0.109171,    0.108853,    0.109108,    0.107815,
    0.107006,    0.107551,    0.10716,     0.104736,    0.104651,
    0.105115,    0.104513,    0.103541,    0.104769,    0.103933,
    0.102696,    0.103246,    0.104033,    0.103982,    0.105189,
    0.106086,    0.105947,    0.105876,    0.107637,    0.107265,
    0.106608,    0.110215,    0.108812,    0.110186,    0.110152,
    0.109722,    0.108679,    0.108872,    0.108398,    0.106363,
    0.10381,     0.101935,    0.0983691,   0.0959844,   0.092071,
    0.0878414,   0.0817595,   0.076002,    0.0696018,   0.0636871,
    0.0569054,   0.0499568,   0.0428787,   0.0356658,   0.0293658,
    0.0226819,   0.0170303,   0.0110604,   0.0055231,   3.63549e-07,
    0.0,         0.00626956,  0.012888,    0.019477,    0.0265727,
    0.0338896,   0.0421523,   0.0496201,   0.0575043,   0.0653868,
    0.0739686,   0.081195,    0.0887963,   0.0936597,   0.100269,
    0.105302,    0.110028,    0.114312,    0.114838,    0.118273,
    0.121567,    0.123124,    0.123413,    0.121465,    0.124503,
    0.12366,     0.122185,    0.121319,    0.121094,    0.119118,
    0.117597,    0.119596,    0.116717,    0.115874,    0.115506,
    0.115244,    0.113848,    0.11399,     0.114585,    0.113924,
    0.114356,    0.114328,    0.114147,    0.11513,     0.115829,
    0.116951,    0.116545,    0.116972,    0.118623,    0.119052,
    0.120279,    0.123081,    0.121492,    0.12291,     0.124264,
    0.121742,    0.123554,    0.123389,    0.120595,    0.12133,
    0.119629,    0.116802,    0.113854,    0.110912,    0.105587,
    0.099669,    0.0947773,   0.0885058,   0.0814853,   0.0743021,
    0.0660472,   0.0575358,   0.0501907,   0.0422096,   0.0339862,
    0.0265993,   0.0195215,   0.0128099,   0.00634015,  4.23995e-07,
    0.0,         0.00730093,  0.0146084,   0.0225179,   0.0310467,
    0.0393665,   0.0481787,   0.0583116,   0.0670514,   0.0754966,
    0.0849838,   0.093867,    0.101047,    0.10804,     0.113727,
    0.120347,    0.125503,    0.13008,     0.130136,    0.133419,
    0.136001,    0.136895,    0.137836,    0.138189,    0.137894,
    0.136335,    0.137588,    0.137447,    0.134911,    0.133555,
    0.130702,    0.128417,    0.129973,    0.129938,    0.127877,
    0.125369,    0.126472,    0.125486,    0.124614,    0.128069,
    0.12498,     0.12558,     0.126341,    0.124734,    0.126001,
    0.12679,     0.128781,    0.129201,    0.13199,     0.132875,
    0.133173,    0.134762,    0.134284,    0.136319,    0.137023,
    0.137909,    0.137332,    0.140997,    0.13656,     0.136882,
    0.135947,    0.133433,    0.129128,    0.124383,    0.122206,
    0.114821,    0.108762,    0.102421,    0.0943173,   0.0852359,
    0.0761552,   0.0675174,   0.0573104,   0.0482876,   0.0394627,
    0.0307161,   0.0228213,   0.0148316,   0.00724836,  4.83537e-07,
    0.0,         0.00817756,  0.016903,    0.0260873,   0.0348874,
    0.0456447,   0.0558789,   0.0673443,   0.0771323,   0.087719,
    0.0975672,   0.107369,    0.117257,    0.126736,    0.130967,
    0.138689,    0.142983,    0.148657,    0.148989,    0.151302,
    0.152106,    0.155645,    0.154742,    0.154389,    0.154371,
    0.151003,    0.151433,    0.148383,    0.148706,    0.145695,
    0.146121,    0.144221,    0.141949,    0.14033,     0.137867,
    0.139036,    0.139369,    0.13848,     0.137644,    0.137343,
    0.139601,    0.136361,    0.138066,    0.139355,    0.138982,
    0.141587,    0.140269,    0.142048,    0.142517,    0.146925,
    0.145769,    0.148831,    0.149195,    0.15431,     0.15123,
    0.154096,    0.152753,    0.155323,    0.155161,    0.154005,
    0.152181,    0.149176,    0.147771,    0.142378,    0.13779,
    0.132483,    0.125078,    0.117734,    0.10779,     0.0983258,
    0.0880738,   0.0782166,   0.0670092,   0.0564346,   0.0448909,
    0.035271,    0.0257786,   0.0168059,   0.00830769,  5.56257e-07,
    0.0,         0.00946929,  0.0192187,   0.0291413,   0.0404458,
    0.0514303,   0.0642803,   0.0769868,   0.0885356,   0.0996757,
    0.113228,    0.125727,    0.133181,    0.140135,    0.150801,
    0.156725,    0.161814,    0.166754,    0.169806,    0.169482,
    0.171337,    0.173654,    0.172546,    0.172843,    0.169166,
    0.169317,    0.167129,    0.166768,    0.163596,    0.160777,
    0.159636,    0.157767,    0.157038,    0.156574,    0.154465,
    0.153035,    0.151399,    0.152249,    0.147519,    0.149835,
    0.15063,     0.153253,    0.150607,    0.151717,    0.153124,
    0.155379,    0.153771,    0.154942,    0.159556,    0.15937,
    0.163331,    0.163505,    0.167498,    0.168115,    0.170132,
    0.17078,     0.174037,    0.173035,    0.172173,    0.1722,
    0.170517,    0.169246,    0.163619,    0.161106,    0.157083,
    0.149897,    0.143737,    0.134709,    0.124492,    0.112107,
    0.101392,    0.0888714,   0.0762966,   0.0640561,   0.0515525,
    0.0403769,   0.0292547,   0.0191838,   0.00941481,  6.3408e-07,
    0.0,         0.0106883,   0.0217974,   0.0336008,   0.0460933,
    0.0594988,   0.0725281,   0.0868572,   0.101642,    0.115512,
    0.126829,    0.142309,    0.152784,    0.161246,    0.170365,
    0.176849,    0.182345,    0.186827,    0.19047,     0.193487,
    0.190697,    0.192339,    0.192325,    0.191573,    0.186267,
    0.189019,    0.188453,    0.184323,    0.179716,    0.179529,
    0.17621,     0.172879,    0.173063,    0.168101,    0.169535,
    0.166736,    0.165566,    0.164092,    0.162934,    0.166108,
    0.163134,    0.164797,    0.165018,    0.167431,    0.165519,
    0.170248,    0.170201,    0.173071,    0.173165,    0.174445,
    0.17998,     0.179376,    0.184233,    0.184894,    0.186086,
    0.191124,    0.192514,    0.193384,    0.190824,    0.194304,
    0.190368,    0.189185,    0.186383,    0.180314,    0.178666,
    0.169736,    0.160316,    0.152664,    0.142281,    0.129424,
    0.115011,    0.102272,    0.0871232,   0.0730232,   0.0591777,
    0.046195,    0.033276,    0.0218238,   0.0107301,   7.1138e-07,
    0.0,         0.012201,    0.02483,     0.0376112,   0.0523212,
    0.0672542,   0.0819751,   0.0986449,   0.11474,     0.129928,
    0.144944,    0.161384,    0.173807,    0.183548,    0.194835,
    0.200035,    0.204717,    0.209329,    0.211307,    0.21499,
    0.216669,    0.216202,    0.216059,    0.213026,    0.207975,
    0.208203,    0.207116,    0.199162,    0.199749,    0.194044,
    0.193299,    0.191259,    0.188201,    0.1859,      0.183704,
    0.182402,    0.181626,    0.180402,    0.18155,     0.179992,
    0.180598,    0.179789,    0.182946,    0.178992,    0.184235,
    0.184389,    0.188336,    0.188723,    0.193183,    0.193373,
    0.196829,    0.198074,    0.203142,    0.205504,    0.208933,
    0.21101,     0.210952,    0.21615,     0.213756,    0.21317,
    0.213704,    0.210928,    0.209018,    0.205671,    0.198272,
    0.192346,    0.181647,    0.173003,    0.160654,    0.14688,
    0.131325,    0.116356,    0.0996062,   0.0832434,   0.0673036,
    0.0518519,   0.03811,     0.0248767,   0.0120765,   8.0251e-07,
    0.0,         0.0137463,   0.0276594,   0.0430266,   0.0592626,
    0.0756919,   0.0943804,   0.112644,    0.132733,    0.150086,
    0.165336,    0.182053,    0.193518,    0.20551,     0.213426,
    0.228323,    0.230068,    0.236057,    0.234987,    0.234645,
    0.235269,    0.238881,    0.238613,    0.235883,    0.232891,
    0.225774,    0.228844,    0.224111,    0.219836,    0.212743,
    0.216088,    0.211621,    0.208885,    0.204235,    0.201835,
    0.19893,     0.203141,    0.196485,    0.19594,     0.197312,
    0.198378,    0.197413,    0.198824,    0.198371,    0.198242,
    0.20533,     0.204272,    0.207156,    0.209249,    0.211766,
    0.214735,    0.220514,    0.223993,    0.227638,    0.229303,
    0.232437,    0.236234,    0.236716,    0.242341,    0.238438,
    0.237274,    0.239935,    0.2384,      0.232598,    0.226728,
    0.220001,    0.207484,    0.197218,    0.182443,    0.166784,
    0.151213,    0.131293,    0.113672,    0.0941002,   0.0767764,
    0.0587752,   0.0431291,   0.0279074,   0.0136807,   9.11155e-07,
    0.0,         0.0155708,   0.0318049,   0.0481646,   0.0664729,
    0.0866899,   0.107567,    0.129747,    0.148419,    0.168192,
    0.189266,    0.208565,    0.223828,    0.233498,    0.246373,
    0.251169,    0.258258,    0.264677,    0.262264,    0.268546,
    0.264242,    0.26504,     0.263985,    0.255599,    0.259264,
    0.250885,    0.250051,    0.246048,    0.241544,    0.235683,
    0.236935,    0.228837,    0.229007,    0.224645,    0.224107,
    0.219518,    0.21662,     0.215141,    0.219673,    0.218325,
    0.213929,    0.21512,     0.216279,    0.220497,    0.21982,
    0.22119,     0.224645,    0.226881,    0.228591,    0.23401,
    0.236949,    0.240887,    0.244141,    0.24953,     0.251997,
    0.257267,    0.258083,    0.261767,    0.265686,    0.263133,
    0.268534,    0.265798,    0.261695,    0.260977,    0.249757,
    0.247039,    0.236136,    0.222154,    0.205467,    0.190263,
    0.169933,    0.149805,    0.128355,    0.108828,    0.0868744,
    0.0668729,   0.0483353,   0.0314067,   0.0153535,   1.01587e-06,
    0.0,         0.0171677,   0.034975,    0.0544189,   0.0761196,
    0.0989202,   0.121159,    0.147409,    0.169334,    0.192003,
    0.213605,    0.232792,    0.251074,    0.259605,    0.272084,
    0.277911,    0.28757,     0.29391,     0.29053,     0.291459,
    0.29906,     0.292364,    0.291581,    0.286551,    0.279387,
    0.28293,     0.273366,    0.268421,    0.266072,    0.259518,
    0.256546,    0.254037,    0.253293,    0.244149,    0.244973,
    0.244772,    0.241902,    0.24141,     0.238474,    0.236652,
    0.236469,    0.235853,    0.238139,    0.239334,    0.24032,
    0.243223,    0.24467,     0.247952,    0.252623,    0.255718,
    0.259425,    0.264425,    0.268245,    0.275315,    0.275974,
    0.28388,     0.291764,    0.294954,    0.296368,    0.298341,
    0.296369,    0.29566,     0.293856,    0.290493,    0.283803,
    0.273966,    0.26476,     0.246522,    0.231772,    0.213917,
    0.191947,    0.168834,    0.145954,    0.121683,    0.0988316,
    0.0760455,   0.0549755,   0.0352681,   0.0171366,   1.13722e-06,
    0.0,         0.0190692,   0.0396395,   0.0617479,   0.0860783,
    0.1108,      0.139884,    0.163086,    0.193814,    0.218717,
    0.239159,    0.26653,     0.278248,    0.294963,    0.309119,
    0.317532,    0.328757,    0.331279,    0.331222,    0.328387,
    0.330526,    0.323808,    0.324432,    0.315869,    0.316976,
    0.308579,    0.299568,    0.298424,    0.290138,    0.286334,
    0.284807,    0.275098,    0.278837,    0.273777,    0.268352,
    0.266451,    0.261979,    0.257965,    0.261038,    0.26264,
    0.258565,    0.258269,    0.261575,    0.261664,    0.265031,
    0.265429,    0.269139,    0.275245,    0.276745,    0.278531,
    0.287135,    0.292812,    0.298848,    0.299455,    0.305221,
    0.317446,    0.318395,    0.323233,    0.326716,    0.325452,
    0.326435,    0.329893,    0.327092,    0.324451,    0.316747,
    0.313859,    0.29035,     0.278559,    0.261677,    0.242806,
    0.216633,    0.191945,    0.165512,    0.13663,     0.110935,
    0.0850414,   0.0620004,   0.0397652,   0.0193866,   1.28592e-06,
    0.0,         0.021669,    0.0439397,   0.0678066,   0.09613,
    0.124748,    0.156146,    0.186267,    0.217009,    0.247541,
    0.272688,    0.295858,    0.318635,    0.330579,    0.350094,
    0.355532,    0.363396,    0.362062,    0.363115,    0.36469,
    0.356855,    0.359163,    0.358084,    0.353104,    0.348691,
    0.333527,    0.327981,    0.330903,    0.319939,    0.312345,
    0.308823,    0.303555,    0.307411,    0.302167,    0.293956,
    0.291354,    0.288832,    0.29076,     0.278883,    0.288334,
    0.287886,    0.28695,     0.285265,    0.288241,    0.29278,
    0.294305,    0.300008,    0.299329,    0.308126,    0.306856,
    0.316574,    0.319428,    0.326987,    0.338409,    0.339421,
    0.345546,    0.353139,    0.357727,    0.356043,    0.363893,
    0.366882,    0.367205,    0.368334,    0.362065,    0.358571,
    0.341551,    0.334594,    0.316786,    0.294666,    0.271507,
    0.245754,    0.218752,    0.187363,    0.156919,    0.125026,
    0.0965955,   0.0696531,   0.0446218,   0.0216522,   1.44754e-06,
    0.0,         0.0240834,   0.0498364,   0.0778355,   0.107712,
    0.140801,    0.17468,     0.210196,    0.247799,    0.278644,
    0.308786,    0.334272,    0.35804,     0.37685,     0.387739,
    0.395468,    0.405156,    0.403866,    0.406959,    0.402154,
    0.403418,    0.40105,     0.398989,    0.386463,    0.387381,
    0.370013,    0.370287,    0.36031,     0.354637,    0.354992,
    0.341396,    0.338795,    0.331128,    0.329282,    0.32085,
    0.321238,    0.313111,    0.317889,    0.313985,    0.310546,
    0.316067,    0.314124,    0.31426,     0.317223,    0.322801,
    0.325431,    0.32582,     0.333144,    0.338023,    0.340753,
    0.351312,    0.353051,    0.362016,    0.364199,    0.376166,
    0.386484,    0.382145,    0.393943,    0.406002,    0.405959,
    0.412289,    0.406471,    0.407915,    0.398871,    0.397017,
    0.387851,    0.369482,    0.358022,    0.337283,    0.312797,
    0.282433,    0.244398,    0.21324,     0.177815,    0.141537,
    0.109002,    0.0776689,   0.0500084,   0.0241214,   1.60734e-06,
    0.0,         0.0271983,   0.0554419,   0.0862661,   0.121299,
    0.162288,    0.198145,    0.237984,    0.276501,    0.313216,
    0.347804,    0.371226,    0.397173,    0.422139,    0.440816,
    0.452965,    0.453848,    0.458511,    0.453327,    0.451473,
    0.447628,    0.441882,    0.438235,    0.430717,    0.415382,
    0.410011,    0.410814,    0.398053,    0.391452,    0.38811,
    0.378856,    0.371496,    0.373327,    0.364525,    0.358893,
    0.35535,     0.354509,    0.344924,    0.345301,    0.352611,
    0.345048,    0.351236,    0.352398,    0.354344,    0.353337,
    0.359492,    0.362141,    0.371106,    0.37847,     0.379898,
    0.382444,    0.386843,    0.397666,    0.412476,    0.417753,
    0.420246,    0.429212,    0.435992,    0.444333,    0.451296,
    0.45162,     0.450248,    0.447315,    0.454588,    0.44419,
    0.430904,    0.41974,     0.403793,    0.372405,    0.345018,
    0.312601,    0.27863,     0.236893,    0.198639,    0.159066,
    0.121968,    0.0874944,   0.0556835,   0.0269023,   1.77591e-06,
    0.0,         0.0298613,   0.06263,     0.0979714,   0.135783,
    0.179588,    0.225379,    0.267192,    0.310252,    0.356766,
    0.391521,    0.418309,    0.448703,    0.473136,    0.48166,
    0.496417,    0.507351,    0.510277,    0.498816,    0.505782,
    0.49431,     0.491,       0.481629,    0.47463,     0.475327,
    0.455442,    0.447179,    0.443799,    0.431498,    0.426539,
    0.423097,    0.411597,    0.404827,    0.395339,    0.391629,
    0.394904,    0.384443,    0.384874,    0.37989,     0.380912,
    0.380408,    0.379201,    0.392128,    0.389473,    0.389697,
    0.394196,    0.39406,     0.406743,    0.413941,    0.417237,
    0.428644,    0.432927,    0.443,       0.451449,    0.461586,
    0.46997,     0.472913,    0.489569,    0.491928,    0.503702,
    0.510781,    0.506957,    0.50352,     0.501144,    0.496755,
    0.48955,     0.466272,    0.450578,    0.419852,    0.393251,
    0.356782,    0.315674,    0.271262,    0.223986,    0.179979,
    0.137581,    0.0986963,   0.062412,    0.0305861,   1.99509e-06,
    0.0,         0.033896,    0.0689554,   0.11013,     0.152765,
    0.202305,    0.25601,     0.309308,    0.351879,    0.402969,
    0.437965,    0.475256,    0.503684,    0.525936,    0.546654,
    0.559103,    0.563692,    0.573424,    0.562956,    0.55636,
    0.551334,    0.542878,    0.535575,    0.521992,    0.521442,
    0.511119,    0.505763,    0.487645,    0.476149,    0.475813,
    0.454374,    0.44984,     0.448216,    0.443841,    0.442114,
    0.433503,    0.430192,    0.421439,    0.432778,    0.415355,
    0.426786,    0.427014,    0.421797,    0.427338,    0.430102,
    0.437765,    0.447463,    0.451792,    0.453885,    0.463798,
    0.474326,    0.47633,     0.493871,    0.506157,    0.515124,
    0.530794,    0.526222,    0.539305,    0.546548,    0.554301,
    0.571537,    0.563113,    0.567573,    0.555116,    0.557258,
    0.541545,    0.525432,    0.501996,    0.475574,    0.436134,
    0.402124,    0.352494,    0.303126,    0.252233,    0.203207,
    0.15471,     0.110141,    0.0698363,   0.0338151,   2.25482e-06,
    0.0,         0.0374066,   0.0785128,   0.123561,    0.171941,
    0.230926,    0.286361,    0.343484,    0.403312,    0.455266,
    0.495316,    0.530619,    0.572194,    0.592412,    0.60856,
    0.617663,    0.634254,    0.627172,    0.636364,    0.61645,
    0.618745,    0.612101,    0.603002,    0.589725,    0.57117,
    0.563107,    0.554338,    0.546315,    0.527348,    0.53042,
    0.516446,    0.510841,    0.500632,    0.485136,    0.487801,
    0.48661,     0.470378,    0.473364,    0.485178,    0.470883,
    0.474303,    0.467611,    0.469982,    0.477046,    0.485652,
    0.492674,    0.489397,    0.500443,    0.508222,    0.514741,
    0.532395,    0.542596,    0.544006,    0.561923,    0.570352,
    0.582445,    0.588615,    0.597365,    0.612965,    0.612066,
    0.628163,    0.641878,    0.62907,     0.627188,    0.615001,
    0.606213,    0.58897,     0.565088,    0.541664,    0.496771,
    0.45243,     0.397997,    0.345559,    0.285379,    0.228591,
    0.175915,    0.124116,    0.0784121,   0.0379695,   2.48825e-06,
    0.0,         0.042488,    0.0895743,   0.137327,    0.193771,
    0.259076,    0.322152,    0.387299,    0.45047,     0.514362,
    0.566468,    0.602978,    0.635681,    0.6702,      0.676907,
    0.698501,    0.709604,    0.711779,    0.713793,    0.694955,
    0.686425,    0.687694,    0.66892,     0.652271,    0.643426,
    0.631814,    0.632134,    0.612921,    0.596258,    0.593216,
    0.582292,    0.564469,    0.571391,    0.542451,    0.543901,
    0.535513,    0.537208,    0.530505,    0.528353,    0.525415,
    0.528104,    0.526289,    0.536129,    0.537572,    0.544071,
    0.545437,    0.552507,    0.553689,    0.562386,    0.570807,
    0.584752,    0.592513,    0.61426,     0.621827,    0.632521,
    0.644223,    0.661466,    0.666755,    0.67487,     0.689532,
    0.705464,    0.687756,    0.713009,    0.705338,    0.691841,
    0.686654,    0.665322,    0.636368,    0.60379,     0.557078,
    0.510665,    0.450622,    0.389968,    0.322689,    0.259323,
    0.197606,    0.13788,     0.0879966,   0.0423561,   2.8019e-06,
    0.0,         0.0483162,   0.0985452,   0.155685,    0.22108,
    0.287929,    0.368398,    0.445521,    0.511029,    0.582591,
    0.642712,    0.687516,    0.712947,    0.755827,    0.767314,
    0.78373,     0.786457,    0.781879,    0.785184,    0.780981,
    0.767099,    0.766389,    0.740486,    0.741288,    0.724368,
    0.697771,    0.696828,    0.688847,    0.672436,    0.657442,
    0.652339,    0.637959,    0.624723,    0.629836,    0.611811,
    0.60233,     0.603467,    0.597597,    0.592939,    0.590913,
    0.593257,    0.601047,    0.599756,    0.592656,    0.604576,
    0.616402,    0.60563,     0.629609,    0.627576,    0.639564,
    0.65708,     0.668842,    0.694218,    0.697157,    0.70727,
    0.715836,    0.734669,    0.763877,    0.747822,    0.768241,
    0.78715,     0.779931,    0.791144,    0.797662,    0.791688,
    0.766508,    0.749927,    0.717404,    0.676213,    0.625426,
    0.581617,    0.504643,    0.440448,    0.367209,    0.287263,
    0.223307,    0.156945,    0.0971267,   0.0474334,   3.13165e-06,
    0.0,         0.0520592,   0.110666,    0.175953,    0.25164,
    0.332596,    0.413519,    0.495869,    0.580572,    0.659122,
    0.697028,    0.750864,    0.81096,     0.850778,    0.869385,
    0.874347,    0.885766,    0.890304,    0.874421,    0.893037,
    0.869834,    0.854738,    0.837601,    0.812245,    0.802258,
    0.790378,    0.796524,    0.777407,    0.753452,    0.741054,
    0.733764,    0.708056,    0.706822,    0.691066,    0.684333,
    0.671833,    0.672051,    0.660574,    0.663289,    0.667443,
    0.668409,    0.66492,     0.672138,    0.67571,     0.673314,
    0.687087,    0.696097,    0.694655,    0.714499,    0.731252,
    0.731456,    0.744276,    0.760108,    0.786256,    0.796306,
    0.811905,    0.817467,    0.835343,    0.859311,    0.875708,
    0.879376,    0.888725,    0.885473,    0.880483,    0.878867,
    0.872623,    0.836675,    0.810534,    0.775001,    0.708387,
    0.653676,    0.582957,    0.504247,    0.413052,    0.331604,
    0.251814,    0.175833,    0.111065,    0.0531674,   3.49807e-06,
    0.0,         0.0589771,   0.124437,    0.197514,    0.285237,
    0.368106,    0.471846,    0.566742,    0.651451,    0.737153,
    0.797963,    0.859421,    0.903823,    0.95818,     0.969864,
    1.00187,     0.987894,    0.990082,    0.990027,    0.990753,
    0.967417,    0.966789,    0.941841,    0.90835,     0.907815,
    0.875369,    0.880457,    0.873895,    0.85243,     0.833544,
    0.83064,     0.806811,    0.786098,    0.776518,    0.784129,
    0.760712,    0.752442,    0.753694,    0.752314,    0.742938,
    0.747623,    0.760114,    0.75245,     0.751973,    0.760977,
    0.769852,    0.781452,    0.796975,    0.792307,    0.814752,
    0.835466,    0.839013,    0.856504,    0.885931,    0.902344,
    0.903873,    0.923904,    0.928516,    0.955773,    0.982118,
    0.998206,    0.979725,    1.0042,      0.992498,    1.00014,
    0.958206,    0.950829,    0.906453,    0.875368,    0.81463,
    0.741807,    0.647707,    0.568914,    0.469974,    0.377904,
    0.283241,    0.19885,     0.125619,    0.0591862,   3.94476e-06,
    0.0,         0.0676277,   0.140581,    0.226374,    0.320447,
    0.427722,    0.541509,    0.630528,    0.746222,    0.838899,
    0.903255,    0.962729,    1.03738,     1.06257,     1.11068,
    1.11543,     1.12825,     1.13546,     1.13069,     1.10173,
    1.0922,      1.07604,     1.06518,     1.04786,     1.00056,
    1.0049,      0.986551,    0.963808,    0.951446,    0.94879,
    0.921232,    0.908727,    0.889052,    0.891126,    0.870156,
    0.865897,    0.875959,    0.857349,    0.845215,    0.850428,
    0.840345,    0.854812,    0.857955,    0.869073,    0.869959,
    0.876431,    0.873338,    0.902485,    0.899936,    0.929797,
    0.933335,    0.95341,     0.965112,    0.991013,    1.01275,
    1.01923,     1.02631,     1.06209,     1.08919,     1.09476,
    1.10216,     1.11654,     1.11248,     1.11559,     1.12394,
    1.08425,     1.05236,     1.04052,     0.987485,    0.91492,
    0.837308,    0.741779,    0.644913,    0.534019,    0.422615,
    0.317534,    0.224306,    0.14151,     0.066822,    4.41498e-06,
    0.0,         0.0753057,   0.162422,    0.25247,     0.360289,
    0.47817,     0.605802,    0.734089,    0.848791,    0.931313,
    1.04638,     1.10035,     1.15529,     1.21057,     1.22756,
    1.24481,     1.27781,     1.25991,     1.27687,     1.24683,
    1.24615,     1.23146,     1.22673,     1.17882,     1.1582,
    1.14492,     1.11323,     1.11357,     1.07885,     1.07267,
    1.05151,     1.02649,     1.02567,     1.02207,     0.996242,
    0.998031,    0.965755,    0.983271,    0.973391,    0.971398,
    0.960417,    0.98631,     0.974588,    0.975675,    0.988317,
    1.01201,     1.01073,     1.00896,     1.0249,      1.05371,
    1.06591,     1.08771,     1.07789,     1.11103,     1.13516,
    1.14729,     1.21104,     1.20509,     1.21576,     1.24802,
    1.24342,     1.26726,     1.2677,      1.26813,     1.27567,
    1.24023,     1.22074,     1.16682,     1.10641,     1.03393,
    0.954204,    0.846318,    0.7319,      0.609707,    0.486626,
    0.362358,    0.255727,    0.158226,    0.0755664,   4.95846e-06,
    0.0,         0.0843327,   0.179481,    0.28749,     0.411984,
    0.55294,     0.696803,    0.823952,    0.956365,    1.07728,
    1.17736,     1.27594,     1.34296,     1.37224,     1.41933,
    1.42567,     1.46345,     1.41386,     1.44839,     1.42,
    1.40835,     1.34867,     1.35583,     1.35511,     1.32766,
    1.29811,     1.27661,     1.26226,     1.24012,     1.21077,
    1.18615,     1.18847,     1.17845,     1.16653,     1.15546,
    1.12283,     1.12001,     1.12343,     1.13385,     1.12794,
    1.12393,     1.12,        1.12117,     1.13144,     1.13921,
    1.1479,      1.15505,     1.14832,     1.1801,      1.19581,
    1.22067,     1.24091,     1.23726,     1.27937,     1.29044,
    1.32461,     1.35418,     1.36762,     1.37699,     1.41103,
    1.39158,     1.44696,     1.42083,     1.42967,     1.4125,
    1.39536,     1.38304,     1.32575,     1.26756,     1.18964,
    1.07727,     0.975525,    0.835876,    0.696019,    0.549551,
    0.419557,    0.285612,    0.179158,    0.0847172,   5.58822e-06,
    0.0,         0.0972556,   0.201142,    0.326366,    0.464905,
    0.636969,    0.798935,    0.935867,    1.09816,     1.22922,
    1.37388,     1.44308,     1.50776,     1.54874,     1.58652,
    1.62247,     1.60722,     1.64064,     1.62147,     1.63345,
    1.59022,     1.60012,     1.5625,      1.56401,     1.51973,
    1.49793,     1.47157,     1.43117,     1.41354,     1.40014,
    1.38069,     1.34342,     1.34921,     1.32893,     1.30984,
    1.30817,     1.29261,     1.3025,      1.28313,     1.28206,
    1.27095,     1.28485,     1.26946,     1.28556,     1.30357,
    1.32871,     1.32999,     1.33966,     1.35715,     1.36339,
    1.38733,     1.43295,     1.4395,      1.47489,     1.47801,
    1.4938,      1.5211,      1.54282,     1.57712,     1.57392,
    1.59919,     1.59314,     1.64048,     1.62596,     1.59673,
    1.56244,     1.56472,     1.53845,     1.44675,     1.36236,
    1.24749,     1.0981,      0.950098,    0.79397,     0.628471,
    0.469429,    0.330046,    0.20393,     0.0974148,   6.31916e-06,
    0.0,         0.108664,    0.230878,    0.37385,     0.539903,
    0.725014,    0.912615,    1.08637,     1.27183,     1.42041,
    1.56765,     1.63029,     1.73716,     1.79963,     1.79747,
    1.84944,     1.851,       1.89702,     1.87557,     1.85085,
    1.84906,     1.82316,     1.76261,     1.77842,     1.72908,
    1.69206,     1.68969,     1.65464,     1.6574,      1.59443,
    1.59415,     1.5706,      1.5342,      1.54553,     1.52108,
    1.52177,     1.48268,     1.49712,     1.49428,     1.48904,
    1.46368,     1.47091,     1.50206,     1.48391,     1.51245,
    1.48391,     1.52944,     1.53351,     1.56848,     1.5603,
    1.56954,     1.61058,     1.65491,     1.67593,     1.70979,
    1.74395,     1.75086,     1.79932,     1.81098,     1.83218,
    1.82785,     1.87229,     1.8662,      1.87728,     1.8297,
    1.84786,     1.78424,     1.73056,     1.62724,     1.53819,
    1.4094,      1.26245,     1.08216,     0.90946,     0.724208,
    0.533431,    0.377533,    0.229868,    0.108761,    7.07173e-06,
    0.0,         0.124304,    0.264304,    0.422017,    0.618501,
    0.829383,    1.04867,     1.27135,     1.42849,     1.62452,
    1.74447,     1.88002,     1.97063,     2.03482,     2.09885,
    2.10194,     2.14891,     2.12948,     2.14433,     2.11503,
    2.08645,     2.04555,     2.04093,     2.00009,     2.01954,
    1.9421,      1.95744,     1.89524,     1.87927,     1.86227,
    1.83818,     1.82028,     1.7835,      1.77599,     1.77634,
    1.76288,     1.7328,      1.72078,     1.73846,     1.73393,
    1.71294,     1.73611,     1.73855,     1.74719,     1.77226,
    1.78862,     1.78783,     1.78787,     1.80214,     1.84962,
    1.83645,     1.88212,     1.88993,     1.92609,     1.93906,
    1.99793,     2.0469,      2.01969,     2.10574,     2.08834,
    2.10125,     2.13485,     2.14418,     2.14412,     2.16177,
    2.09867,     2.01941,     1.9513,      1.92356,     1.76374,
    1.61733,     1.45173,     1.25411,     1.03208,     0.821173,
    0.61881,     0.420675,    0.262871,    0.123597,    8.07606e-06,
    0.0,         0.139914,    0.299062,    0.486695,    0.715003,
    0.949898,    1.20669,     1.46556,     1.67443,     1.86672,
    2.04426,     2.1719,      2.2752,      2.36486,     2.40584,
    2.42636,     2.40266,     2.48787,     2.46984,     2.43708,
    2.43616,     2.39832,     2.36557,     2.298,       2.29232,
    2.27197,     2.26485,     2.20161,     2.17315,     2.13016,
    2.1545,      2.11263,     2.06037,     2.04193,     2.07964,
    2.03698,     2.02569,     2.02086,     2.02797,     2.00338,
    2.01645,     1.99213,     2.03729,     2.04901,     2.04672,
    2.05967,     2.04245,     2.07092,     2.0998,      2.11442,
    2.16587,     2.13765,     2.18748,     2.22704,     2.27014,
    2.27504,     2.32413,     2.34688,     2.38626,     2.37546,
    2.44057,     2.45805,     2.4441,      2.44372,     2.43345,
    2.36717,     2.3265,      2.2506,      2.13762,     2.03364,
    1.86614,     1.65037,     1.45583,     1.19244,     0.961461,
    0.708521,    0.486282,    0.298929,    0.140037,    9.15297e-06,
    0.0,         0.162182,    0.342321,    0.564213,    0.812931,
    1.10487,     1.40704,     1.6726,      1.91133,     2.1187,
    2.3332,      2.51804,     2.65531,     2.70502,     2.78516,
    2.77203,     2.81903,     2.81236,     2.85387,     2.80987,
    2.82892,     2.72966,     2.70463,     2.71601,     2.6478,
    2.65823,     2.60844,     2.55886,     2.51624,     2.53335,
    2.50315,     2.44414,     2.45396,     2.42858,     2.37246,
    2.36294,     2.36258,     2.3528,      2.32932,     2.36252,
    2.36305,     2.35154,     2.38819,     2.41001,     2.41775,
    2.41056,     2.40159,     2.45417,     2.46632,     2.50684,
    2.47395,     2.53449,     2.61923,     2.60138,     2.63704,
    2.67899,     2.6801,      2.71882,     2.7554,      2.77933,
    2.7602,      2.80448,     2.8295,      2.82393,     2.80211,
    2.7633,      2.70759,     2.62851,     2.52468,     2.33914,
    2.13595,     1.94282,     1.68188,     1.36387,     1.09933,
    0.818253,    0.553165,    0.342745,    0.16224,     1.04262e-05,
    0.0,         0.180313,    0.38877,     0.64126,     0.936041,
    1.27466,     1.56574,     1.91716,     2.23184,     2.47645,
    2.70949,     2.86519,     2.9937,      3.08652,     3.18735,
    3.24225,     3.25272,     3.27014,     3.25818,     3.2381,
    3.22975,     3.17464,     3.12935,     3.11783,     3.13895,
    3.03123,     2.99318,     3.01751,     2.97992,     2.91337,
    2.89601,     2.83908,     2.85586,     2.85716,     2.80782,
    2.84936,     2.74413,     2.77584,     2.72727,     2.78508,
    2.76177,     2.75391,     2.76209,     2.77052,     2.78689,
    2.81442,     2.83124,     2.84459,     2.82315,     2.89706,
    2.91454,     3.01591,     3.00812,     3.02749,     3.04788,
    3.0655,      3.14712,     3.15143,     3.19488,     3.20162,
    3.2477,      3.19101,     3.28177,     3.26524,     3.24958,
    3.14763,     3.11238,     3.00066,     2.86065,     2.66776,
    2.48203,     2.23513,     1.9196,      1.5983,      1.25701,
    0.929899,    0.639723,    0.394363,    0.181412,    1.18891e-05,
    0.0,         0.20449,     0.446498,    0.735015,    1.05801,
    1.4349,      1.8563,      2.20654,     2.5611,      2.87025,
    3.12349,     3.3201,      3.40487,     3.59252,     3.63662,
    3.79865,     3.7344,      3.72565,     3.74805,     3.81218,
    3.69883,     3.66062,     3.66237,     3.64884,     3.60212,
    3.54861,     3.51063,     3.48113,     3.39131,     3.42064,
    3.45667,     3.3425,      3.32508,     3.30276,     3.30855,
    3.26934,     3.24194,     3.23194,     3.25656,     3.29518,
    3.22305,     3.2212,      3.23886,     3.26193,     3.30209,
    3.31719,     3.28694,     3.33144,     3.35876,     3.37412,
    3.43793,     3.48698,     3.49846,     3.50389,     3.53437,
    3.63239,     3.64012,     3.65009,     3.67008,     3.77677,
    3.76109,     3.73086,     3.79611,     3.75102,     3.73553,
    3.69922,     3.58724,     3.49286,     3.34227,     3.11862,
    2.86667,     2.54075,     2.21041,     1.85323,     1.44717,
    1.07775,     0.735949,    0.445952,    0.205679,    1.34317e-05,
    0.0,         0.233297,    0.502014,    0.831068,    1.23037,
    1.67073,     2.13185,     2.54801,     2.95126,     3.23175,
    3.54825,     3.80599,     4.03573,     4.09445,     4.11344,
    4.28749,     4.31504,     4.3499,      4.38621,     4.35503,
    4.26458,     4.32155,     4.24905,     4.17985,     4.21822,
    4.05327,     4.08449,     4.05951,     4.00961,     3.99952,
    3.98535,     3.91479,     3.87938,     3.85961,     3.8497,
    3.81637,     3.8722,      3.75075,     3.77003,     3.79956,
    3.77864,     3.86154,     3.80713,     3.88313,     3.75594,
    3.85536,     3.86373,     3.93177,     3.96217,     4.00605,
    4.02875,     4.07833,     4.06541,     4.08174,     4.06352,
    4.12375,     4.17523,     4.22222,     4.31119,     4.25766,
    4.27488,     4.29714,     4.32831,     4.26621,     4.22855,
    4.2,         4.09208,     3.98418,     3.79373,     3.53473,
    3.30284,     2.93829,     2.5601,      2.11139,     1.66566,
    1.23123,     0.834463,    0.503555,    0.23238,     1.50916e-05,
    0.0,         0.255042,    0.558771,    0.956489,    1.39789,
    1.88804,     2.39712,     2.87982,     3.3082,      3.75649,
    4.06325,     4.30343,     4.47473,     4.65771,     4.76314,
    4.85525,     4.95609,     4.97442,     4.9882,      4.89044,
    5.00009,     4.87024,     4.8753,      4.88572,     4.78988,
    4.69717,     4.7559,      4.68738,     4.67911,     4.57897,
    4.5701,      4.49064,     4.50099,     4.5001,      4.42482,
    4.41889,     4.38457,     4.36535,     4.38427,     4.37893,
    4.36462,     4.36373,     4.36448,     4.44308,     4.38863,
    4.45735,     4.47378,     4.47978,     4.52352,     4.57544,
    4.58978,     4.62985,     4.71837,     4.72471,     4.74767,
    4.82234,     4.81829,     4.92689,     4.98585,     4.82943,
    4.81835,     5.00873,     4.88931,     4.95665,     4.84354,
    4.79485,     4.68487,     4.52049,     4.3177,      3.96039,
    3.68774,     3.37027,     2.90868,     2.43371,     1.86227,
    1.39391,     0.958613,    0.574111,    0.261143,    1.68027e-05,
    0.0,         0.285967,    0.641166,    1.06351,     1.5643,
    2.12922,     2.70817,     3.27737,     3.77778,     4.25871,
    4.56278,     4.85415,     5.09026,     5.27374,     5.38421,
    5.54563,     5.60068,     5.6426,      5.6045,      5.54067,
    5.66188,     5.5144,      5.48177,     5.45434,     5.48951,
    5.41309,     5.43816,     5.39567,     5.31232,     5.19974,
    5.18895,     5.2254,      5.15739,     5.18338,     5.14363,
    5.14127,     5.03014,     4.99746,     5.12042,     4.96148,
    5.01185,     5.01543,     4.99245,     5.07543,     5.12572,
    5.10882,     5.05618,     5.13155,     5.18244,     5.26793,
    5.27386,     5.27388,     5.30893,     5.32546,     5.51317,
    5.43291,     5.50961,     5.47208,     5.61126,     5.58658,
    5.65152,     5.53517,     5.69116,     5.593,       5.52881,
    5.3593,      5.33582,     5.12456,     4.84729,     4.52908,
    4.21608,     3.79183,     3.27921,     2.70244,     2.11401,
    1.56604,     1.05289,     0.625646,    0.286576,    1.85345e-05,
    0.0,         0.309192,    0.693076,    1.15244,     1.69778,
    2.348,       2.98146,     3.55089,     4.25441,     4.67269,
    5.05215,     5.44737,     5.65637,     5.92421,     5.96269,
    6.16647,     6.17675,     6.05553,     6.22687,     6.22302,
    6.15211,     6.14303,     6.15395,     6.20051,     6.013,
    5.97586,     6.03519,     5.95053,     5.90878,     5.80004,
    5.82645,     5.82346,     5.80074,     5.72143,     5.76372,
    5.71312,     5.56146,     5.69201,     5.68808,     5.59592,
    5.69273,     5.75941,     5.70852,     5.69648,     5.65784,
    5.68293,     5.71433,     5.77334,     5.80269,     5.91755,
    5.88283,     5.95538,     5.99703,     5.96342,     6.05471,
    6.1539,      6.15417,     6.26648,     6.22633,     6.21846,
    6.30408,     6.23638,     6.26858,     6.18207,     6.15253,
    5.94484,     5.87933,     5.64095,     5.38886,     5.11778,
    4.68098,     4.21162,     3.60939,     3.01836,     2.38896,
    1.71178,     1.16434,     0.69944,     0.30928,     2.00599e-05,
    0.0,         0.330093,    0.73624,     1.24762,     1.86602,
    2.50685,     3.27545,     3.88955,     4.50067,     5.04009,
    5.53479,     5.83464,     6.17388,     6.34874,     6.53731,
    6.70905,     6.75518,     6.78842,     6.85755,     6.77837,
    6.89128,     6.75576,     6.73909,     6.61665,     6.57853,
    6.65181,     6.53917,     6.51971,     6.61083,     6.35769,
    6.43347,     6.36941,     6.38525,     6.26463,     6.33141,
    6.30449,     6.09529,     6.24607,     6.22797,     6.10653,
    6.24755,     6.25977,     6.28281,     6.30727,     6.2716,
    6.24909,     6.26447,     6.37763,     6.34618,     6.49622,
    6.50396,     6.48226,     6.63083,     6.64453,     6.65043,
    6.72816,     6.70193,     6.69875,     6.75352,     6.76904,
    6.78087,     6.79247,     6.72866,     6.77097,     6.71399,
    6.52787,     6.37705,     6.31551,     5.94626,     5.49765,
    5.06584,     4.53412,     3.92669,     3.24588,     2.55787,
    1.85035,     1.24662,     0.726536,    0.327444,    2.08665e-05,
    0.0,         0.340139,    0.759766,    1.29504,     1.95452,
    2.6893,      3.42245,     4.17877,     4.78688,     5.34793,
    5.80686,     6.24312,     6.45511,     6.81093,     6.95464,
    7.14425,     7.17015,     7.08033,     7.3107,      7.38634,
    7.27145,     7.18113,     7.11925,     7.11453,     7.09086,
    7.12716,     6.90651,     6.96683,     6.96945,     6.88759,
    6.85346,     6.7532,      6.77574,     6.71235,     6.65855,
    6.71718,     6.65854,     6.62421,     6.69074,     6.66355,
    6.6941,      6.65157,     6.65543,     6.69018,     6.67729,
    6.74745,     6.71784,     6.77268,     6.88205,     6.85417,
    6.83379,     6.95234,     7.12011,     7.02163,     7.06486,
    7.1694,      7.06745,     7.15881,     7.26649,     7.22676,
    7.20534,     7.20748,     7.33219,     7.13099,     6.98426,
    6.91486,     6.77605,     6.43456,     6.19552,     5.88456,
    5.38479,     4.82206,     4.19705,     3.42115,     2.69506,
    1.94961,     1.29002,     0.760769,    0.338977,    2.17787e-05,
    0.0,         0.334639,    0.758796,    1.33328,     2.00982,
    2.76421,     3.52365,     4.25982,     5.04278,     5.53105,
    6.08274,     6.43467,     6.76714,     6.92348,     7.06996,
    7.32967,     7.39244,     7.41744,     7.4467,      7.49879,
    7.51032,     7.41962,     7.50128,     7.45664,     7.28608,
    7.34581,     7.20751,     7.22297,     7.18422,     7.12628,
    7.19838,     7.06999,     6.99198,     7.00926,     6.98598,
    6.92865,     6.87395,     6.90072,     6.90444,     6.90321,
    6.89847,     6.80255,     6.91083,     7.00661,     6.97829,
    6.96839,     7.00042,     7.07265,     7.09765,     7.18704,
    7.05761,     7.19418,     7.37268,     7.26206,     7.27096,
    7.3764,      7.38921,     7.46633,     7.52195,     7.39851,
    7.54089,     7.45893,     7.47573,     7.34725,     7.2662,
    7.05369,     6.94219,     6.7194,      6.45516,     5.93584,
    5.5085,      4.9768,      4.31626,     3.5249,      2.76459,
    2.02348,     1.32058,     0.758785,    0.343485,    2.14443e-05,
    0.0,         0.328139,    0.753122,    1.29514,     2.02323,
    2.80418,     3.62514,     4.3731,      5.05631,     5.64002,
    6.08417,     6.48744,     6.75643,     7.07896,     7.13962,
    7.30325,     7.44174,     7.54563,     7.52872,     7.56648,
    7.44661,     7.56369,     7.49818,     7.45821,     7.43944,
    7.33564,     7.39691,     7.32695,     7.28724,     7.21613,
    7.15659,     7.08594,     7.10187,     7.02696,     7.00275,
    7.03766,     6.95023,     7.01974,     6.96036,     7.02324,
    6.97444,     6.94729,     6.998,       7.01584,     6.97235,
    7.07514,     7.0623,      7.07412,     7.12193,     7.16144,
    7.23701,     7.31395,     7.3101,      7.40678,     7.3,
    7.39439,     7.49588,     7.50247,     7.53417,     7.44839,
    7.52009,     7.44378,     7.48787,     7.51998,     7.27416,
    7.1773,      7.06316,     6.76291,     6.62447,     6.09287,
    5.62028,     5.03542,     4.3291,      3.55173,     2.83466,
    2.00544,     1.31291,     0.753,       0.327939,    2.0858e-05,
    0.0,         0.308645,    0.716549,    1.28005,     1.97717,
    2.77526,     3.53068,     4.31853,     5.01558,     5.52785,
    5.96746,     6.35187,     6.68427,     7.04739,     7.19402,
    7.19377,     7.305,       7.40442,     7.36111,     7.40467,
    7.38174,     7.32398,     7.41842,     7.34281,     7.26149,
    7.27901,     7.17002,     7.37415,     7.05593,     7.05424,
    7.03015,     7.00763,     7.01336,     6.87941,     6.83153,
    6.88483,     6.8459,      6.882,       6.87923,     6.91728,
    6.89426,     6.95359,     6.8876,      6.8844,      6.88679,
    6.90412,     6.95085,     6.98551,     7.04378,     7.04092,
    7.15173,     7.24043,     7.14909,     7.15292,     7.17857,
    7.28215,     7.33322,     7.33463,     7.35773,     7.37617,
    7.27514,     7.28832,     7.38327,     7.26292,     7.28239,
    7.03917,     6.90059,     6.68605,     6.34952,     6.07243,
    5.53852,     4.99881,     4.24254,     3.6083,      2.76233,
    1.96092,     1.29279,     0.730043,    0.317153,    1.96707e-05,
    0.0,         0.291257,    0.68839,     1.23326,     1.87928,
    2.6462,      3.45187,     4.14033,     4.77706,     5.3539,
    5.91637,     6.1813,      6.46321,     6.72282,     6.87039,
    6.91762,     7.02994,     7.12976,     7.20698,     7.08771,
    7.17934,     7.184,       7.14552,     7.10751,     7.02465,
    6.99834,     6.93092,     6.84886,     6.8812,      6.85682,
    6.80107,     6.78531,     6.78755,     6.61594,     6.65062,
    6.6015,      6.66057,     6.54982,     6.57677,     6.67629,
    6.62045,     6.64436,     6.58748,     6.62018,     6.65194,
    6.82815,     6.73097,     6.77787,     6.83972,     6.88196,
    6.87874,     6.75709,     6.93738,     6.91552,     7.04478,
    6.96969,     7.13216,     7.13926,     7.11405,     7.15464,
    7.17843,     7.08807,     7.00519,     7.08004,     7.00543,
    6.91966,     6.57773,     6.58494,     6.2681,      5.78932,
    5.37993,     4.79468,     4.13931,     3.44775,     2.64473,
    1.89588,     1.22576,     0.683869,    0.293537,    1.84471e-05,
    0.0,         0.27204,     0.634122,    1.14901,     1.81791,
    2.55191,     3.3073,      3.96272,     4.57898,     5.0896,
    5.59422,     5.93464,     6.18594,     6.40177,     6.54517,
    6.65893,     6.67222,     6.80785,     6.79709,     6.73264,
    6.85954,     6.75074,     6.78676,     6.59674,     6.69841,
    6.66459,     6.56625,     6.66033,     6.48478,     6.44001,
    6.42256,     6.43432,     6.43375,     6.34143,     6.24976,
    6.31277,     6.39607,     6.37829,     6.33453,     6.24067,
    6.35706,     6.23032,     6.33379,     6.26407,     6.40031,
    6.3877,      6.41015,     6.38051,     6.50364,     6.42661,
    6.41672,     6.54115,     6.66759,     6.6603,      6.6687,
    6.71147,     6.67973,     6.64346,     6.75642,     6.75179,
    6.86144,     6.73899,     6.81379,     6.67703,     6.65045,
    6.55517,     6.37407,     6.24982,     5.96983,     5.67743,
    5.14113,     4.55949,     3.98212,     3.28409,     2.54464,
    1.81097,     1.15006,     0.63925,     0.272657,    1.71934e-05,
    0.0,         0.246333,    0.592416,    1.08199,     1.71355,
    2.44433,     3.13274,     3.75971,     4.40618,     4.91143,
    5.35179,     5.67009,     5.85505,     6.02407,     6.23277,
    6.26295,     6.29679,     6.38762,     6.35917,     6.41698,
    6.33222,     6.38919,     6.45189,     6.32917,     6.26467,
    6.23286,     6.29485,     6.19282,     6.21572,     6.0694,
    6.13596,     5.9967,      5.96549,     5.94927,     5.90343,
    5.95435,     5.8076,      5.88609,     5.87899,     5.85503,
    5.93073,     5.91093,     5.94744,     5.98301,     6.02267,
    5.99813,     5.9444,      6.00693,     5.9719,      6.11086,
    6.09148,     6.14535,     6.13518,     6.25792,     6.22895,
    6.23772,     6.30907,     6.27818,     6.3328,      6.32088,
    6.37672,     6.43665,     6.36464,     6.19056,     6.3334,
    6.19751,     6.02924,     5.82845,     5.51864,     5.39711,
    4.8581,      4.3603,      3.83522,     3.12629,     2.36567,
    1.71075,     1.08045,     0.606098,    0.250304,    1.5624e-05,
    0.0,         0.227255,    0.553184,    1.01238,     1.63612,
    2.30298,     2.94386,     3.57358,     4.14917,     4.62158,
    4.9278,      5.27918,     5.46955,     5.60744,     5.76644,
    5.97429,     5.89764,     5.97432,     5.94424,     6.0385,
    5.93179,     5.96934,     5.97379,     5.89167,     5.86807,
    5.81592,     5.73323,     5.74749,     5.73795,     5.67301,
    5.64084,     5.67195,     5.57079,     5.54952,     5.54417,
    5.50547,     5.57095,     5.52306,     5.45085,     5.44025,
    5.49991,     5.48247,     5.54879,     5.48138,     5.54166,
    5.53233,     5.57313,     5.68732,     5.69539,     5.7278,
    5.7241,      5.74368,     5.75102,     5.79919,     5.86214,
    5.89208,     5.93926,     5.85911,     5.94633,     5.88905,
    5.93052,     5.98538,     5.97875,     5.99903,     5.84939,
    5.85922,     5.66215,     5.37282,     5.2997,      5.01465,
    4.67465,     4.1623,      3.51615,     2.98113,     2.3065,
    1.62452,     1.01816,     0.55136,     0.229908,    1.40222e-05,
    0.0,         0.205894,    0.5137,      0.958973,    1.52904,
    2.13586,     2.79787,     3.40247,     3.85801,     4.32826,
    4.72311,     4.95187,     5.12429,     5.23588,     5.40286,
    5.38048,     5.55102,     5.56074,     5.63481,     5.52713,
    5.57388,     5.42232,     5.51878,     5.45959,     5.49262,
    5.38188,     5.42801,     5.3964,      5.3268,      5.25857,
    5.30092,     5.27147,     5.20208,     5.16457,     5.13005,
    5.11698,     5.0544,      5.08483,     5.1157,      5.04988,
    5.12566,     5.11668,     5.07261,     5.11756,     5.11224,
    5.13568,     5.07394,     5.23861,     5.17593,     5.30356,
    5.25453,     5.31287,     5.40015,     5.42164,     5.4572,
    5.51805,     5.52043,     5.53235,     5.59219,     5.54231,
    5.60069,     5.60657,     5.49092,     5.5856,      5.41516,
    5.43814,     5.33767,     5.17076,     4.97123,     4.73533,
    4.37615,     3.90414,     3.37418,     2.79301,     2.13971,
    1.51759,     0.949019,    0.509975,    0.210319,    1.27665e-05,
    0.0,         0.191025,    0.473128,    0.889601,    1.42855,
    2.03875,     2.60082,     3.19875,     3.68403,     4.09325,
    4.33084,     4.65088,     4.84337,     4.95254,     5.08935,
    5.15739,     5.17609,     5.12543,     5.15338,     5.14688,
    5.12516,     5.11698,     5.0641,      5.08682,     5.11173,
    4.97532,     5.07248,     4.94298,     4.88734,     4.91881,
    4.8178,      4.88387,     4.89786,     4.78887,     4.80183,
    4.81407,     4.70119,     4.69026,     4.69049,     4.71301,
    4.68594,     4.75809,     4.69432,     4.77533,     4.69365,
    4.76863,     4.7675,      4.80084,     4.85305,     4.92298,
    4.88185,     4.87296,     5.01208,     5.06098,     5.08871,
    5.1015,      5.16826,     5.18154,     5.12117,     5.18566,
    5.21507,     5.16883,     5.25781,     5.18508,     5.09205,
    5.06395,     4.95346,     4.81826,     4.60965,     4.42925,
    4.09396,     3.64026,     3.20918,     2.59306,     2.03269,
    1.44158,     0.88522,     0.477938,    0.194452,    1.18534e-05,
    0.0,         0.176078,    0.437973,    0.836359,    1.36198,
    1.93243,     2.50078,     3.03074,     3.45118,     3.80034,
    4.09316,     4.34322,     4.46435,     4.63649,     4.75774,
    4.69349,     4.84352,     4.81918,     4.79547,     4.83174,
    4.84531,     4.79763,     4.74899,     4.75598,     4.652,
    4.69957,     4.63727,     4.58538,     4.5575,      4.53979,
    4.53496,     4.44439,     4.44589,     4.47276,     4.36012,
    4.35396,     4.36487,     4.38707,     4.36353,     4.36841,
    4.37695,     4.38666,     4.3802,      4.39862,     4.36715,
    4.41438,     4.44423,     4.43429,     4.41832,     4.50586,
    4.50985,     4.57851,     4.53019,     4.56354,     4.66529,
    4.70531,     4.69205,     4.79719,     4.7344,      4.7929,
    4.72146,     4.81958,     4.81473,     4.77829,     4.84171,
    4.71762,     4.65355,     4.48531,     4.37196,     4.10575,
    3.81664,     3.42755,     3.04011,     2.48228,     1.93511,
    1.35293,     0.851628,    0.445614,    0.178455,    1.07822e-05,
    0.0,         0.164979,    0.416739,    0.789254,    1.27129,
    1.8191,      2.31625,     2.84658,     3.25474,     3.56873,
    3.8476,      4.07603,     4.26132,     4.29966,     4.42878,
    4.48191,     4.48994,     4.50654,     4.53541,     4.50711,
    4.4477,      4.45969,     4.41864,     4.40092,     4.32205,
    4.3485,      4.30167,     4.22598,     4.18338,     4.2217,
    4.17571,     4.14595,     4.11099,     4.10581,     4.05566,
    4.07231,     4.06988,     4.02282,     4.06913,     4.0355,
    4.00197,     4.0101,      4.02662,     4.04115,     4.0874,
    4.06405,     4.05382,     4.12428,     4.11875,     4.26399,
    4.23825,     4.19414,     4.24579,     4.29333,     4.28321,
    4.36061,     4.4086,      4.40114,     4.44596,     4.48763,
    4.44633,     4.47894,     4.52906,     4.51921,     4.44216,
    4.44054,     4.35639,     4.23086,     4.08892,     3.93112,
    3.59177,     3.22943,     2.85523,     2.38011,     1.82992,
    1.26844,     0.800374,    0.418057,    0.166194,    1.00491e-05,
    0.0,         0.152374,    0.394687,    0.749403,    1.21939,
    1.72766,     2.21787,     2.688,       3.08475,     3.43195,
    3.61528,     3.83933,     3.95902,     4.06023,     4.1066,
    4.15348,     4.13779,     4.20777,     4.20733,     4.17823,
    4.14222,     4.12189,     4.10067,     4.07017,     4.06378,
    4.02535,     3.99014,     3.99893,     3.88813,     3.89447,
    3.87008,     3.84087,     3.78027,     3.78664,     3.76556,
    3.7688,      3.67016,     3.75627,     3.71845,     3.70453,
    3.72065,     3.78183,     3.70486,     3.79566,     3.7428,
    3.79676,     3.83901,     3.82087,     3.85803,     3.91088,
    3.96548,     3.91753,     3.98249,     3.99048,     4.01864,
    4.02728,     4.12263,     4.12715,     4.14917,     4.13818,
    4.22951,     4.23231,     4.23107,     4.19984,     4.13682,
    4.20448,     4.08353,     3.92227,     3.83246,     3.66232,
    3.4121,      3.0719,      2.71101,     2.23586,     1.7268,
    1.22633,     0.752395,    0.391535,    0.155109,    9.29632e-06,
    0.0,         0.142796,    0.369695,    0.72043,     1.14955,
    1.63776,     2.11962,     2.56815,     2.88464,     3.17388,
    3.38008,     3.59028,     3.73813,     3.80504,     3.84797,
    3.84778,     3.95241,     3.90485,     3.92894,     3.93604,
    3.90594,     3.8708,      3.80406,     3.80283,     3.80307,
    3.64833,     3.7018,      3.69383,     3.67035,     3.64069,
    3.56316,     3.56082,     3.58187,     3.55598,     3.48413,
    3.44461,     3.47012,     3.47059,     3.43621,     3.4927,
    3.50119,     3.4885,      3.47446,     3.51873,     3.51772,
    3.50052,     3.53879,     3.52063,     3.58059,     3.6549,
    3.61668,     3.65454,     3.6945,      3.73149,     3.74264,
    3.80873,     3.81813,     3.85261,     3.91289,     3.8902,
    3.89643,     3.83505,     3.93231,     3.92418,     3.84947,
    3.86537,     3.8139,      3.76901,     3.59019,     3.44144,
    3.16061,     2.88521,     2.55498,     2.12366,     1.66713,
    1.16284,     0.712258,    0.375351,    0.144731,    8.71346e-06,
    0.0,         0.134158,    0.348372,    0.679929,    1.09361,
    1.55403,     1.99912,     2.42472,     2.78563,     3.03126,
    3.27227,     3.40938,     3.47548,     3.53795,     3.64151,
    3.65987,     3.69678,     3.73675,     3.65098,     3.67076,
    3.67283,     3.58949,     3.5662,      3.55027,     3.47499,
    3.58425,     3.46974,     3.4098,      3.36806,     3.35556,
    3.36282,     3.30179,     3.29968,     3.27518,     3.28622,
    3.28128,     3.28143,     3.23976,     3.22291,     3.25751,
    3.25423,     3.21402,     3.22637,     3.26549,     3.25911,
    3.27117,     3.33069,     3.34583,     3.37921,     3.33399,
    3.40989,     3.4208,      3.4807,      3.46211,     3.51251,
    3.48894,     3.49186,     3.60382,     3.59963,     3.67141,
    3.63341,     3.64528,     3.73139,     3.65822,     3.72225,
    3.66978,     3.56074,     3.51221,     3.43344,     3.27139,
    3.0659,      2.78289,     2.41609,     2.03986,     1.57076,
    1.10497,     0.685451,    0.352425,    0.136804,    8.11923e-06,
    0.0,         0.12671,     0.329848,    0.643829,    1.05018,
    1.50279,     1.93549,     2.28416,     2.62487,     2.8857,
    3.09413,     3.22965,     3.30541,     3.35501,     3.41114,
    3.44849,     3.45943,     3.47146,     3.43015,     3.45387,
    3.38645,     3.40053,     3.41993,     3.32459,     3.33507,
    3.28379,     3.21191,     3.23296,     3.1781,      3.17563,
    3.14156,     3.11963,     3.08585,     3.0681,      3.0505,
    2.98675,     3.01549,     2.98299,     3.03325,     2.95029,
    3.03071,     3.01598,     3.02129,     3.00067,     3.02763,
    3.06146,     3.03898,     3.07657,     3.0943,      3.08701,
    3.13224,     3.174,       3.23817,     3.20565,     3.245,
    3.32203,     3.35793,     3.35064,     3.40929,     3.44268,
    3.42586,     3.40486,     3.48184,     3.4891,      3.45238,
    3.36253,     3.3726,      3.38051,     3.19483,     3.06441,
    2.87368,     2.66091,     2.32044,     1.92642,     1.48991,
    1.06803,     0.655071,    0.339629,    0.129367,    7.66115e-06,
    0.0,         0.118339,    0.31361,     0.622695,    1.02331,
    1.44499,     1.83966,     2.23342,     2.53108,     2.74571,
    2.88393,     3.03481,     3.13344,     3.22159,     3.21386,
    3.21801,     3.25893,     3.24473,     3.21185,     3.2742,
    3.23171,     3.14108,     3.14852,     3.13783,     3.11793,
    3.0588,      3.00184,     3.01263,     3.00874,     2.9858,
    2.90331,     2.94045,     2.88446,     2.91093,     2.86579,
    2.85716,     2.85771,     2.83561,     2.7943,      2.83593,
    2.84792,     2.77074,     2.84532,     2.85442,     2.86812,
    2.88332,     2.878,       2.84393,     2.93253,     2.95159,
    2.97282,     3.00212,     3.02551,     3.08884,     3.07481,
    3.14374,     3.1351,      3.15152,     3.18545,     3.20859,
    3.20758,     3.2293,      3.2424,      3.22503,     3.2582,
    3.21027,     3.22733,     3.16291,     3.07873,     2.93674,
    2.74071,     2.5256,      2.21711,     1.8673,      1.43354,
    1.01188,     0.636016,    0.32641,     0.12389,     7.23838e-06,
    0.0,         0.113524,    0.302371,    0.59088,     0.972178,
    1.39491,     1.77177,     2.08529,     2.40533,     2.61381,
    2.74393,     2.88428,     2.968,       3.0081,      3.068,
    3.08693,     3.10831,     3.0518,      3.08049,     3.03712,
    3.00693,     2.98377,     2.9512,      2.96162,     2.91828,
    2.89634,     2.88089,     2.81606,     2.85423,     2.7647,
    2.7125,      2.7504,      2.68152,     2.68152,     2.66785,
    2.69651,     2.67408,     2.67063,     2.6203,      2.66031,
    2.70297,     2.64255,     2.60923,     2.67077,     2.70279,
    2.67984,     2.67868,     2.74775,     2.70726,     2.78677,
    2.81987,     2.82853,     2.84815,     2.87194,     2.9125,
    2.9123,      2.95763,     2.98558,     3.00391,     3.03996,
    3.03669,     3.03432,     3.07156,     3.07538,     3.08124,
    3.07998,     3.01093,     2.93234,     2.90351,     2.79227,
    2.60558,     2.39926,     2.1437,      1.79434,     1.40212,
    0.982108,    0.609451,    0.308628,    0.117374,    6.85736e-06,
    0.0,         0.107272,    0.287332,    0.572186,    0.929244,
    1.32405,     1.68327,     2.03261,     2.27732,     2.5078,
    2.59789,     2.75757,     2.81815,     2.88659,     2.90765,
    2.89966,     2.91015,     2.91612,     2.89963,     2.85316,
    2.9088,      2.82459,     2.79164,     2.77818,     2.72518,
    2.68087,     2.6922,      2.65512,     2.62189,     2.61768,
    2.60971,     2.62433,     2.53898,     2.53298,     2.49597,
    2.51393,     2.51817,     2.47896,     2.47951,     2.50772,
    2.45335,     2.50058,     2.50581,     2.4734,      2.53916,
    2.5225,      2.57904,     2.53742,     2.58324,     2.62508,
    2.61795,     2.66844,     2.65473,     2.68289,     2.71023,
    2.71792,     2.80709,     2.78767,     2.81945,     2.8656,
    2.87553,     2.91386,     2.96532,     2.93795,     2.91358,
    2.87918,     2.88898,     2.87254,     2.77879,     2.67174,
    2.49965,     2.32148,     2.03637,     1.75055,     1.3612,
    0.952492,    0.585449,    0.29973,     0.112301,    6.53126e-06,
    0.0,         0.103396,    0.282027,    0.562565,    0.908325,
    1.27762,     1.64211,     1.94365,     2.22262,     2.41799,
    2.5534,      2.63451,     2.71678,     2.74934,     2.76576,
    2.79868,     2.74707,     2.71883,     2.71307,     2.71834,
    2.67441,     2.66674,     2.65895,     2.60649,     2.61189,
    2.6079,      2.51886,     2.49465,     2.4811,      2.48281,
    2.43552,     2.42466,     2.41703,     2.41311,     2.40686,
    2.38101,     2.3588,      2.37097,     2.35868,     2.35128,
    2.33948,     2.33579,     2.32781,     2.39109,     2.36305,
    2.39354,     2.41957,     2.46277,     2.46223,     2.45064,
    2.48786,     2.48106,     2.54949,     2.56211,     2.55528,
    2.59539,     2.61587,     2.67545,     2.64725,     2.73834,
    2.70395,     2.75885,     2.79405,     2.78512,     2.8139,
    2.7544,      2.7459,      2.69948,     2.62149,     2.55616,
    2.38842,     2.20456,     1.97913,     1.66877,     1.30262,
    0.913789,    0.573568,    0.289834,    0.108007,    6.27741e-06,
    0.0,         0.0989603,   0.27184,     0.542596,    0.877079,
    1.25977,     1.58513,     1.8785,      2.11107,     2.28187,
    2.4134,      2.50656,     2.54413,     2.58852,     2.59805,
    2.64147,     2.67252,     2.59789,     2.61535,     2.5766,
    2.57329,     2.53988,     2.5258,      2.51018,     2.45644,
    2.43756,     2.39016,     2.39081,     2.3846,      2.33127,
    2.32134,     2.29332,     2.26747,     2.26576,     2.21569,
    2.25303,     2.25835,     2.23931,     2.24623,     2.21815,
    2.24267,     2.20744,     2.24206,     2.21347,     2.25534,
    2.25288,     2.27281,     2.29439,     2.28852,     2.33064,
    2.37095,     2.38209,     2.40687,     2.38043,     2.43986,
    2.49633,     2.5048,      2.51778,     2.54049,     2.54559,
    2.58578,     2.58109,     2.63197,     2.64112,     2.6505,
    2.63341,     2.60177,     2.57883,     2.51188,     2.41534,
    2.31857,     2.13133,     1.88155,     1.60449,     1.25482,
    0.899403,    0.55085,     0.279359,    0.103641,    6.01204e-06,
    0.0,         0.0961487,   0.259932,    0.526319,    0.86424,
    1.21338,     1.53056,     1.8059,      2.03193,     2.22296,
    2.32401,     2.40542,     2.45544,     2.48949,     2.49862,
    2.49433,     2.4978,      2.52122,     2.47394,     2.431,
    2.44629,     2.366,       2.41741,     2.36706,     2.36099,
    2.29465,     2.30461,     2.21137,     2.26136,     2.25582,
    2.19657,     2.18402,     2.17126,     2.15455,     2.14105,
    2.11179,     2.11814,     2.11744,     2.11568,     2.0947,
    2.09496,     2.10515,     2.12154,     2.13636,     2.10342,
    2.15601,     2.19264,     2.20885,     2.1894,      2.20321,
    2.21245,     2.29393,     2.27585,     2.31136,     2.32835,
    2.35008,     2.39483,     2.41093,     2.4262,      2.4328,
    2.45568,     2.49159,     2.4909,      2.50149,     2.51468,
    2.5269,      2.49637,     2.46585,     2.43627,     2.33851,
    2.21177,     2.04717,     1.81433,     1.54701,     1.22292,
    0.876562,    0.547443,    0.272397,    0.101618,    5.83805e-06,
    0.0,         0.0922394,   0.253961,    0.518238,    0.822457,
    1.15713,     1.50208,     1.77652,     1.96155,     2.1166,
    2.26985,     2.3154,      2.33044,     2.34868,     2.38656,
    2.36893,     2.3816,      2.39723,     2.36717,     2.37012,
    2.30356,     2.32658,     2.29503,     2.2304,      2.21033,
    2.15349,     2.16171,     2.12031,     2.1471,      2.12935,
    2.12642,     2.12323,     2.09049,     2.08916,     2.02417,
    2.0482,      2.02895,     2.02097,     2.01395,     2.00431,
    1.99608,     2.03447,     2.03038,     2.02006,     2.03478,
    2.04839,     2.08797,     2.06779,     2.08915,     2.11369,
    2.13817,     2.15123,     2.15512,     2.17767,     2.21688,
    2.19521,     2.2464,      2.31145,     2.31028,     2.36593,
    2.35538,     2.34804,     2.41092,     2.39746,     2.38131,
    2.36503,     2.39183,     2.37266,     2.36437,     2.25195,
    2.10239,     1.99403,     1.76442,     1.50279,     1.18768,
    0.845411,    0.524969,    0.267629,    0.0985828,   5.6597e-06,
    0.0,         0.0888333,   0.245818,    0.494644,    0.808521,
    1.13961,     1.46303,     1.71396,     1.90153,     2.04897,
    2.15887,     2.22959,     2.25647,     2.2752,      2.28442,
    2.31415,     2.27186,     2.28115,     2.25161,     2.22216,
    2.23114,     2.1902,      2.17217,     2.16887,     2.14783,
    2.12315,     2.08756,     2.05222,     2.06543,     2.00831,
    2.01896,     1.98915,     1.9754,      1.96388,     1.96665,
    1.9503,      1.92319,     1.89075,     1.92772,     1.92782,
    1.93309,     1.94568,     1.94144,     1.93481,     1.94003,
    1.94093,     1.9625,      1.99513,     1.97665,     1.99988,
    2.07008,     2.0467,      2.04662,     2.07186,     2.1382,
    2.12903,     2.17176,     2.20662,     2.22478,     2.22075,
    2.23905,     2.2621,      2.28871,     2.31719,     2.30894,
    2.3121,      2.28791,     2.30651,     2.26058,     2.1744,
    2.06971,     1.93219,     1.728,       1.46428,     1.15558,
    0.836385,    0.514545,    0.259702,    0.0961299,   5.56352e-06,
    0.0,         0.0855248,   0.24127,     0.48841,     0.79135,
    1.11136,     1.40233,     1.65942,     1.82671,     1.98601,
    2.06567,     2.12932,     2.15802,     2.19887,     2.16984,
    2.16788,     2.19048,     2.19478,     2.13964,     2.16205,
    2.12183,     2.09351,     2.06952,     2.03781,     2.06338,
    2.01567,     2.00497,     1.95145,     1.95427,     1.92728,
    1.92434,     1.88172,     1.88451,     1.88588,     1.87145,
    1.86126,     1.83944,     1.84728,     1.84053,     1.84049,
    1.82184,     1.84542,     1.81939,     1.83923,     1.87263,
    1.87071,     1.89518,     1.88169,     1.90749,     1.9346,
    1.9856,      1.99185,     1.98858,     1.99283,     2.0295,
    2.05397,     2.08443,     2.09031,     2.09882,     2.12536,
    2.152,       2.17868,     2.20254,     2.21182,     2.22448,
    2.25747,     2.21587,     2.22451,     2.16363,     2.1064,
    2.01086,     1.85504,     1.6874,      1.43949,     1.13131,
    0.81528,     0.513185,    0.257531,    0.0944051,   5.39663e-06,
    0.0,         0.0840017,   0.237054,    0.469512,    0.767711,
    1.07809,     1.35561,     1.60651,     1.76532,     1.90508,
    2.01643,     2.06504,     2.09011,     2.12238,     2.09805,
    2.11636,     2.11828,     2.08704,     2.08079,     2.03294,
    2.04885,     2.03309,     1.99325,     1.95038,     1.94884,
    1.93024,     1.94426,     1.87398,     1.89147,     1.87227,
    1.85841,     1.84006,     1.79777,     1.79295,     1.77686,
    1.80236,     1.77731,     1.78243,     1.78899,     1.77599,
    1.78365,     1.77616,     1.78762,     1.78318,     1.79913,
    1.80993,     1.8231,      1.8303,      1.8542,      1.85137,
    1.86368,     1.90376,     1.94656,     1.93736,     1.95293,
    1.95327,     1.98212,     1.99169,     2.04622,     2.04082,
    2.07901,     2.09809,     2.10709,     2.09907,     2.13045,
    2.14661,     2.14856,     2.13311,     2.13224,     2.03052,
    1.9401,      1.8096,      1.63938,     1.41048,     1.11131,
    0.808653,    0.501587,    0.253354,    0.0911536,   5.29236e-06,
    0.0,         0.0812489,   0.232564,    0.464571,    0.774035,
    1.0701,      1.34436,     1.56546,     1.74544,     1.86657,
    1.93802,     1.97394,     2.01687,     2.03978,     2.03465,
    2.02111,     2.03791,     2.01603,     2.01377,     1.98654,
    1.98304,     1.927,       1.90358,     1.91546,     1.86607,
    1.85819,     1.83331,     1.82702,     1.79118,     1.80479,
    1.80174,     1.73151,     1.76236,     1.77941,     1.75943,
    1.73126,     1.70369,     1.68552,     1.70995,     1.70534,
    1.71302,     1.70902,     1.74556,     1.71218,     1.75237,
    1.71375,     1.77155,     1.76902,     1.76246,     1.78025,
    1.82965,     1.80546,     1.82207,     1.83029,     1.8708,
    1.89096,     1.92169,     1.95384,     1.96019,     1.98762,
    1.99291,     2.00734,     2.03188,     2.06589,     2.08057,
    2.08254,     2.04952,     2.08289,     2.04866,     1.97311,
    1.89817,     1.75508,     1.59655,     1.36831,     1.0982,
    0.798787,    0.494351,    0.250444,    0.090616,    5.19705e-06,
    0.0,         0.0781139,   0.228085,    0.460159,    0.746033,
    1.03609,     1.30513,     1.52616,     1.68412,     1.81211,
    1.88611,     1.94767,     1.95713,     1.95107,     1.99791,
    1.97587,     1.9977,      1.92165,     1.94734,     1.87317,
    1.87849,     1.8676,      1.87096,     1.80556,     1.80573,
    1.80399,     1.78336,     1.77413,     1.75207,     1.72927,
    1.71995,     1.70776,     1.69062,     1.68689,     1.68918,
    1.65748,     1.64964,     1.64911,     1.65561,     1.65478,
    1.68774,     1.66255,     1.66479,     1.67801,     1.67707,
    1.6749,      1.67269,     1.69047,     1.71448,     1.73225,
    1.76271,     1.74615,     1.74059,     1.8103,      1.79755,
    1.81944,     1.88706,     1.88243,     1.88486,     1.91809,
    1.91395,     1.93476,     2.00506,     1.99594,     1.99749,
    1.97076,     2.00473,     1.99706,     1.98523,     1.90789,
    1.84222,     1.72541,     1.57188,     1.33528,     1.07997,
    0.784839,    0.489421,    0.250588,    0.0906567,   5.16439e-06,
    0.0,         0.0771569,   0.221947,    0.449438,    0.73357,
    1.01181,     1.27975,     1.4887,      1.63722,     1.77205,
    1.83477,     1.88974,     1.89552,     1.9179,      1.9204,
    1.89025,     1.89916,     1.85676,     1.85128,     1.85287,
    1.85065,     1.81285,     1.78939,     1.80245,     1.74723,
    1.73904,     1.70921,     1.72027,     1.68798,     1.67039,
    1.67363,     1.67115,     1.62087,     1.61856,     1.6291,
    1.65197,     1.64035,     1.60644,     1.60544,     1.5942,
    1.59737,     1.58607,     1.60963,     1.64378,     1.61805,
    1.62163,     1.6529,      1.64913,     1.68227,     1.67244,
    1.69557,     1.71599,     1.6972,      1.73249,     1.73698,
    1.75055,     1.80614,     1.8117,      1.82775,     1.81169,
    1.89417,     1.89483,     1.90684,     1.93295,     1.90031,
    1.93655,     1.91897,     1.93747,     1.9022,      1.84949,
    1.7998,      1.68763,     1.53625,     1.31366,     1.07419,
    0.764997,    0.485376,    0.246718,    0.0899065,   5.1409e-06,
    0.0,         0.0762039,   0.217332,    0.451036,    0.721979,
    1.00485,     1.2482,      1.46068,     1.59536,     1.71729,
    1.78827,     1.80272,     1.85528,     1.83789,     1.84767,
    1.82409,     1.82544,     1.84066,     1.80389,     1.79244,
    1.7729,      1.72585,     1.74262,     1.75242,     1.68792,
    1.70446,     1.68795,     1.68654,     1.64327,     1.61561,
    1.61973,     1.63457,     1.60188,     1.59635,     1.584,
    1.55948,     1.58302,     1.5517,      1.56849,     1.56811,
    1.56528,     1.54937,     1.57551,     1.56199,     1.57004,
    1.58699,     1.60885,     1.5979,      1.61745,     1.6371,
    1.65253,     1.67405,     1.66948,     1.70161,     1.71702,
    1.7341,      1.72363,     1.75143,     1.78049,     1.78414,
    1.814,       1.82949,     1.85037,     1.86212,     1.84225,
    1.85794,     1.88797,     1.8852,      1.85568,     1.83062,
    1.76244,     1.65341,     1.47858,     1.29685,     1.05881,
    0.764059,    0.483096,    0.246458,    0.0903704,   5.08823e-06,
    0.0,         0.0751488,   0.215575,    0.442289,    0.706605,
    1.0041,      1.22979,     1.41911,     1.5656,      1.68592,
    1.74237,     1.7827,      1.81261,     1.82009,     1.79708,
    1.79559,     1.80178,     1.76548,     1.75489,     1.73345,
    1.71566,     1.69833,     1.68668,     1.69146,     1.67639,
    1.63366,     1.62623,     1.61037,     1.61446,     1.59552,
    1.55992,     1.55813,     1.56444,     1.55284,     1.55351,
    1.54776,     1.52683,     1.51498,     1.50696,     1.55129,
    1.53616,     1.54855,     1.53607,     1.56016,     1.54755,
    1.52654,     1.55814,     1.55726,     1.57498,     1.5905,
    1.60113,     1.61445,     1.62008,     1.66476,     1.67408,
    1.6893,      1.69002,     1.71723,     1.71669,     1.75912,
    1.75098,     1.8059,      1.77356,     1.82857,     1.83726,
    1.83663,     1.82701,     1.84445,     1.80841,     1.74717,
    1.69429,     1.60859,     1.50013,     1.2817,      1.04371,
    0.770981,    0.481105,    0.245587,    0.0907822,   5.19015e-06,
    0.0,         0.0741891,   0.213103,    0.437959,    0.704214,
    0.975313,    1.23219,     1.38644,     1.55236,     1.6131,
    1.69982,     1.74123,     1.76012,     1.7449,      1.75507,
    1.76389,     1.75645,     1.74532,     1.7245,      1.67736,
    1.65643,     1.66642,     1.62763,     1.62013,     1.62754,
    1.61214,     1.59299,     1.59036,     1.59323,     1.5435,
    1.53759,     1.53376,     1.51806,     1.50677,     1.52957,
    1.51422,     1.51129,     1.48966,     1.48122,     1.51468,
    1.5063,      1.48908,     1.51749,     1.48048,     1.54001,
    1.52393,     1.52624,     1.5215,      1.54846,     1.55686,
    1.56284,     1.59363,     1.59462,     1.6077,      1.64492,
    1.6318,      1.64069,     1.65442,     1.67692,     1.71317,
    1.72292,     1.75114,     1.77183,     1.76631,     1.78396,
    1.80384,     1.788,       1.77181,     1.76105,     1.74588,
    1.69661,     1.59295,     1.44857,     1.25587,     1.03486,
    0.764098,    0.483283,    0.248162,    0.0918465,   5.21345e-06,
    0.0,         0.0728358,   0.211549,    0.439296,    0.704586,
    0.965464,    1.20256,     1.40262,     1.53323,     1.59383,
    1.64694,     1.71701,     1.68798,     1.73011,     1.71296,
    1.69403,     1.69744,     1.6892,      1.68878,     1.67005,
    1.65224,     1.65425,     1.60598,     1.60833,     1.57477,
    1.56282,     1.54862,     1.53468,     1.53922,     1.50242,
    1.52062,     1.52402,     1.49764,     1.49809,     1.48278,
    1.48805,     1.47807,     1.46655,     1.47226,     1.44286,
    1.45701,     1.47679,     1.48592,     1.49929,     1.50357,
    1.49324,     1.48965,     1.48644,     1.53958,     1.51961,
    1.55049,     1.55297,     1.56354,     1.59501,     1.55872,
    1.6154,      1.64888,     1.60968,     1.64149,     1.66803,
    1.70059,     1.70763,     1.7011,      1.74081,     1.72601,
    1.75714,     1.76042,     1.75624,     1.74038,     1.7017,
    1.64851,     1.55693,     1.44594,     1.25787,     1.03324,
    0.756704,    0.471705,    0.248081,    0.0920738,   5.34471e-06,
    0.0,         0.0730876,   0.213336,    0.431673,    0.700618,
    0.961129,    1.18016,     1.3611,      1.47517,     1.59068,
    1.61284,     1.6756,      1.68133,     1.68704,     1.66623,
    1.66511,     1.64794,     1.64741,     1.6266,      1.6218,
    1.60981,     1.59114,     1.57161,     1.58594,     1.55842,
    1.553,       1.53927,     1.50852,     1.51685,     1.45949,
    1.49854,     1.49284,     1.48696,     1.47265,     1.45267,
    1.45602,     1.45101,     1.44897,     1.44719,     1.44501,
    1.45761,     1.46384,     1.47845,     1.45802,     1.47723,
    1.44625,     1.49132,     1.48155,     1.50204,     1.47544,
    1.52033,     1.53119,     1.52201,     1.55655,     1.56328,
    1.57021,     1.56468,     1.57725,     1.61616,     1.66019,
    1.63879,     1.66475,     1.67189,     1.69971,     1.70755,
    1.71692,     1.71289,     1.72233,     1.7114,      1.68733,
    1.62774,     1.55765,     1.41205,     1.23696,     1.01757,
    0.754723,    0.483458,    0.253095,    0.0938403,   5.35455e-06,
    0.0,         0.0719115,   0.212063,    0.431711,    0.697367,
    0.950575,    1.17896,     1.34058,     1.45951,     1.54238,
    1.62242,     1.65398,     1.65217,     1.65061,     1.64081,
    1.64088,     1.63323,     1.60343,     1.58979,     1.59324,
    1.57091,     1.56816,     1.55251,     1.52818,     1.54023,
    1.50238,     1.5025,      1.51588,     1.47481,     1.46797,
    1.47689,     1.46566,     1.43047,     1.43663,     1.44071,
    1.43531,     1.41751,     1.44735,     1.43342,     1.44775,
    1.44887,     1.44047,     1.43952,     1.4486,      1.42636,
    1.44799,     1.4441,      1.45298,     1.47748,     1.4738,
    1.48853,     1.51083,     1.48876,     1.50232,     1.53908,
    1.5498,      1.55898,     1.57208,     1.60419,     1.6052,
    1.64761,     1.64963,     1.65369,     1.67199,     1.66417,
    1.67229,     1.71676,     1.6947,      1.68045,     1.65857,
    1.5924,      1.54571,     1.41455,     1.24108,     1.01715,
    0.764505,    0.491558,    0.259453,    0.0976793,   5.56048e-06,
    0.0,         0.0718545,   0.2103,      0.433819,    0.712814,
    0.956123,    1.17895,     1.33205,     1.44805,     1.53138,
    1.56723,     1.58297,     1.61163,     1.61527,     1.57697,
    1.60033,     1.60365,     1.60508,     1.60213,     1.56239,
    1.54957,     1.55424,     1.50573,     1.52803,     1.51112,
    1.49153,     1.49606,     1.46633,     1.48188,     1.45957,
    1.44243,     1.43005,     1.42597,     1.42778,     1.43104,
    1.40348,     1.43668,     1.40165,     1.42985,     1.41126,
    1.41695,     1.40521,     1.43292,     1.44281,     1.44018,
    1.41335,     1.46138,     1.42838,     1.4445,      1.48011,
    1.47039,     1.49523,     1.50564,     1.50591,     1.50265,
    1.52948,     1.52649,     1.56121,     1.55447,     1.60998,
    1.6223,      1.60673,     1.61164,     1.6357,      1.67425,
    1.66237,     1.66409,     1.66218,     1.65587,     1.62684,
    1.60546,     1.52806,     1.40694,     1.2367,      1.03329,
    0.765904,    0.497638,    0.263811,    0.09932,     5.75714e-06,
    0.0,         0.0732427,   0.214032,    0.435551,    0.70037,
    0.947825,    1.16165,     1.32605,     1.43223,     1.50207,
    1.56301,     1.57882,     1.60135,     1.60295,     1.61356,
    1.56343,     1.55951,     1.55571,     1.52511,     1.54481,
    1.54599,     1.51013,     1.49394,     1.49851,     1.46085,
    1.48528,     1.49047,     1.43636,     1.42812,     1.44,
    1.43575,     1.42796,     1.42615,     1.41177,     1.41414,
    1.40709,     1.40439,     1.40992,     1.39248,     1.40297,
    1.42361,     1.40746,     1.40804,     1.42633,     1.40714,
    1.4175,      1.43373,     1.41483,     1.42579,     1.44677,
    1.43911,     1.46774,     1.46062,     1.49842,     1.5084,
    1.50044,     1.5164,      1.52936,     1.5228,      1.56464,
    1.56299,     1.57324,     1.60437,     1.63301,     1.63519,
    1.63053,     1.65164,     1.67238,     1.66185,     1.62172,
    1.62175,     1.50542,     1.40202,     1.2387,      1.03727,
    0.779816,    0.502964,    0.268367,    0.103484,    6.00561e-06,
    0.0,         0.0727515,   0.214424,    0.440094,    0.695266,
    0.938281,    1.1609,      1.30682,     1.42672,     1.49528,
    1.54364,     1.57665,     1.55444,     1.57636,     1.56373,
    1.55063,     1.55088,     1.5463,      1.5152,      1.49701,
    1.50451,     1.50751,     1.48438,     1.47567,     1.46428,
    1.45719,     1.45614,     1.4371,      1.4457,      1.41347,
    1.44016,     1.4059,      1.40884,     1.39786,     1.44355,
    1.40257,     1.41811,     1.38946,     1.41499,     1.40454,
    1.41384,     1.41327,     1.43701,     1.40304,     1.41454,
    1.40357,     1.43727,     1.42309,     1.44279,     1.45741,
    1.46433,     1.44268,     1.47767,     1.45962,     1.48219,
    1.48923,     1.50517,     1.50344,     1.5364,      1.53628,
    1.56784,     1.54462,     1.56492,     1.58766,     1.58237,
    1.63345,     1.62458,     1.63881,     1.64061,     1.59938,
    1.56313,     1.51507,     1.40664,     1.25074,     1.05614,
    0.793245,    0.524104,    0.278708,    0.106774,    6.27454e-06,
    0.0,         0.0715712,   0.218955,    0.444368,    0.703381,
    0.938817,    1.16555,     1.29414,     1.38828,     1.48594,
    1.52395,     1.54028,     1.52862,     1.57739,     1.54117,
    1.55217,     1.52465,     1.51174,     1.50349,     1.4956,
    1.48512,     1.5105,      1.47532,     1.42782,     1.46592,
    1.44433,     1.45109,     1.43449,     1.43605,     1.42308,
    1.44436,     1.41096,     1.43829,     1.42332,     1.38954,
    1.39497,     1.39581,     1.38236,     1.40261,     1.38776,
    1.39135,     1.3932,      1.38801,     1.40077,     1.40494,
    1.41539,     1.43247,     1.4325,      1.43788,     1.4183,
    1.43067,     1.44793,     1.47531,     1.46106,     1.47869,
    1.48671,     1.47351,     1.49204,     1.51165,     1.53783,
    1.53091,     1.52495,     1.56787,     1.58394,     1.58201,
    1.61225,     1.61116,     1.59477,     1.64848,     1.61004,
    1.57215,     1.52823,     1.40774,     1.26459,     1.07629,
    0.823756,    0.533789,    0.290621,    0.11367,     6.58307e-06,
    0.0,         0.0746646,   0.22066,     0.450043,    0.706671,
    0.953073,    1.14205,     1.28438,     1.41677,     1.46407,
    1.49774,     1.51031,     1.53941,     1.55697,     1.51402,
    1.53546,     1.51264,     1.48176,     1.48103,     1.47631,
    1.45596,     1.4823,      1.47263,     1.43195,     1.47237,
    1.44778,     1.42883,     1.43769,     1.40791,     1.42211,
    1.41775,     1.41935,     1.41523,     1.39221,     1.4191,
    1.42976,     1.40952,     1.37589,     1.42218,     1.39567,
    1.4095,      1.4043,      1.418,       1.42241,     1.44043,
    1.41113,     1.40233,     1.42501,     1.40291,     1.42597,
    1.43713,     1.43341,     1.46186,     1.46686,     1.46584,
    1.48163,     1.4821,      1.49759,     1.50356,     1.52409,
    1.53382,     1.53556,     1.56711,     1.52792,     1.57969,
    1.60536,     1.59562,     1.60669,     1.62897,     1.62646,
    1.55954,     1.50555,     1.41023,     1.27866,     1.07968,
    0.839393,    0.553922,    0.307161,    0.119686,    7.18489e-06,
    0.0,         0.0750056,   0.22223,     0.450789,    0.703176,
    0.970045,    1.15719,     1.30301,     1.40903,     1.45823,
    1.50096,     1.5204,      1.54806,     1.50692,     1.50461,
    1.51883,     1.5104,      1.49261,     1.4873,      1.48082,
    1.4771,      1.44333,     1.43851,     1.43932,     1.46131,
    1.43339,     1.43844,     1.42999,     1.42793,     1.42494,
    1.41297,     1.40615,     1.3946,      1.40026,     1.41325,
    1.39261,     1.37855,     1.40746,     1.41933,     1.40227,
    1.39448,     1.40924,     1.41373,     1.41033,     1.41319,
    1.43818,     1.40559,     1.43505,     1.44069,     1.42662,
    1.42439,     1.45155,     1.46917,     1.45089,     1.45519,
    1.45768,     1.46323,     1.47307,     1.48689,     1.48996,
    1.52244,     1.51447,     1.54503,     1.56973,     1.57485,
    1.61834,     1.6047,      1.5954,      1.60945,     1.60065,
    1.58856,     1.54138,     1.43306,     1.29938,     1.1031,
    0.847775,    0.578652,    0.316668,    0.130435,    7.69212e-06,
    0.0,         0.0758013,   0.227461,    0.464073,    0.718816,
    0.973791,    1.14822,     1.3166,      1.4008,      1.45472,
    1.49857,     1.52728,     1.51136,     1.51608,     1.51415,
    1.50139,     1.49127,     1.45747,     1.47603,     1.45856,
    1.45962,     1.446,       1.44216,     1.43658,     1.43846,
    1.43564,     1.41119,     1.4369,      1.41195,     1.41208,
    1.40554,     1.40059,     1.3856,      1.42049,     1.418,
    1.39375,     1.4095,      1.38475,     1.44523,     1.4219,
    1.4068,      1.40766,     1.41444,     1.43357,     1.42582,
    1.41934,     1.42771,     1.43704,     1.41306,     1.42615,
    1.44326,     1.42961,     1.44452,     1.45462,     1.44608,
    1.46699,     1.47472,     1.4952,      1.50494,     1.50169,
    1.49402,     1.50981,     1.54353,     1.54044,     1.5918,
    1.60929,     1.6211,      1.62275,     1.63301,     1.64213,
    1.59701,     1.56755,     1.46169,     1.33737,     1.13497,
    0.893516,    0.605616,    0.34397,     0.139694,    8.33575e-06,
    0.0,         0.0795501,   0.229667,    0.469503,    0.724184,
    0.962681,    1.17202,     1.31627,     1.41352,     1.472,
    1.50752,     1.48453,     1.47937,     1.49919,     1.51462,
    1.51303,     1.49139,     1.46378,     1.47086,     1.47686,
    1.44219,     1.45823,     1.44695,     1.44125,     1.43778,
    1.41997,     1.4379,      1.43149,     1.42199,     1.43826,
    1.40291,     1.42308,     1.41974,     1.42343,     1.43221,
    1.44196,     1.42309,     1.40519,     1.40707,     1.41262,
    1.41437,     1.4302,      1.43599,     1.43434,     1.46679,
    1.41839,     1.41331,     1.41341,     1.42471,     1.42887,
    1.45044,     1.4548,      1.44144,     1.44831,     1.4481,
    1.45741,     1.45138,     1.47209,     1.47502,     1.49928,
    1.54442,     1.54714,     1.56042,     1.53641,     1.60233,
    1.57524,     1.59628,     1.61806,     1.6309,      1.61499,
    1.61745,     1.59099,     1.50023,     1.36926,     1.17165,
    0.916672,    0.636791,    0.36082,     0.150971,    9.13376e-06,
    0.0,         0.0805041,   0.237709,    0.478358,    0.721665,
    0.992696,    1.19359,     1.32825,     1.42825,     1.46013,
    1.49818,     1.51363,     1.49341,     1.52513,     1.48141,
    1.50422,     1.4894,      1.46678,     1.45736,     1.46174,
    1.42293,     1.45373,     1.44611,     1.45896,     1.43715,
    1.43098,     1.41576,     1.43192,     1.42596,     1.41466,
    1.40863,     1.44511,     1.41371,     1.40928,     1.4256,
    1.40953,     1.401,       1.41682,     1.41204,     1.43393,
    1.43129,     1.42272,     1.43592,     1.40306,     1.41566,
    1.4172,      1.42092,     1.44519,     1.4374,      1.43229,
    1.44516,     1.44885,     1.44652,     1.41944,     1.45015,
    1.45309,     1.48903,     1.46445,     1.47033,     1.47873,
    1.54071,     1.52514,     1.51804,     1.56085,     1.57861,
    1.60422,     1.62262,     1.66815,     1.67131,     1.6582,
    1.63603,     1.62722,     1.54362,     1.41048,     1.22082,
    0.97757,     0.673536,    0.393774,    0.16491,     1.02113e-05,
    0.0,         0.0853244,   0.246526,    0.491118,    0.740161,
    0.994088,    1.19416,     1.33503,     1.4155,      1.4773,
    1.48027,     1.49994,     1.49916,     1.48512,     1.50468,
    1.47011,     1.47492,     1.49264,     1.45718,     1.4565,
    1.45419,     1.44101,     1.44705,     1.42931,     1.4505,
    1.42116,     1.43636,     1.44695,     1.45486,     1.43451,
    1.42433,     1.43743,     1.4393,      1.42498,     1.42998,
    1.42597,     1.42377,     1.42483,     1.44766,     1.42861,
    1.41729,     1.42714,     1.4332,      1.4151,      1.44671,
    1.43887,     1.43812,     1.44069,     1.43799,     1.4366,
    1.42696,     1.42647,     1.45034,     1.46517,     1.46938,
    1.48673,     1.49306,     1.47215,     1.48913,     1.54637,
    1.53149,     1.57315,     1.54116,     1.55912,     1.60802,
    1.61579,     1.62708,     1.66646,     1.68475,     1.70206,
    1.68646,     1.65016,     1.60719,     1.46678,     1.27224,
    1.01933,     0.718102,    0.42503,     0.18105,     1.12882e-05,
    0.0,         0.0872712,   0.24927,     0.502593,    0.786858,
    1.02938,     1.23052,     1.36386,     1.43268,     1.48028,
    1.50794,     1.50436,     1.51904,     1.50909,     1.47121,
    1.49394,     1.47687,     1.47104,     1.46299,     1.44648,
    1.46865,     1.45805,     1.44196,     1.42957,     1.42987,
    1.42406,     1.43013,     1.45055,     1.42433,     1.42297,
    1.44088,     1.44823,     1.43998,     1.42243,     1.40938,
    1.44664,     1.43742,     1.42342,     1.41103,     1.43124,
    1.45755,     1.41336,     1.42991,     1.43309,     1.44355,
    1.4481,      1.44703,     1.45949,     1.43435,     1.44234,
    1.45667,     1.48641,     1.45309,     1.47181,     1.46506,
    1.48661,     1.48243,     1.49977,     1.5304,      1.52116,
    1.50483,     1.55437,     1.5556,      1.58657,     1.61358,
    1.61522,     1.67942,     1.68397,     1.6837,      1.71564,
    1.72038,     1.69932,     1.61971,     1.50388,     1.31177,
    1.07206,     0.767915,    0.461205,    0.198847,    1.2553e-05,
    0.0,         0.0932656,   0.269093,    0.516812,    0.8006,
    1.03383,     1.2389,      1.37056,     1.44668,     1.48608,
    1.49192,     1.54017,     1.48838,     1.50548,     1.49865,
    1.48898,     1.4707,      1.47693,     1.47654,     1.45545,
    1.45489,     1.42844,     1.42189,     1.42893,     1.42635,
    1.44467,     1.43563,     1.43674,     1.44649,     1.42555,
    1.41688,     1.43187,     1.44405,     1.40057,     1.42718,
    1.43052,     1.44315,     1.43483,     1.41813,     1.42941,
    1.42608,     1.44187,     1.43993,     1.45753,     1.44467,
    1.42261,     1.42819,     1.47524,     1.43024,     1.45343,
    1.43643,     1.46001,     1.46008,     1.44224,     1.47406,
    1.45629,     1.4794,      1.48887,     1.50963,     1.51068,
    1.52468,     1.57539,     1.54566,     1.58157,     1.6127,
    1.66763,     1.68387,     1.70476,     1.73072,     1.76006,
    1.7577,      1.75107,     1.68403,     1.593,       1.38443,
    1.13653,     0.817629,    0.496888,    0.219302,    1.39225e-05,
    0.0,         0.0968876,   0.277308,    0.54363,     0.815869,
    1.07201,     1.23404,     1.40373,     1.45997,     1.49798,
    1.50106,     1.50336,     1.52715,     1.51838,     1.46685,
    1.46337,     1.4766,      1.46699,     1.46263,     1.44531,
    1.44508,     1.43314,     1.45911,     1.4134,      1.43873,
    1.45999,     1.41323,     1.41048,     1.40155,     1.42484,
    1.42499,     1.43773,     1.4227,      1.42796,     1.41302,
    1.42591,     1.43781,     1.40281,     1.42061,     1.41325,
    1.43195,     1.44471,     1.42982,     1.43898,     1.40967,
    1.43805,     1.40367,     1.40372,     1.4134,      1.43468,
    1.4817,      1.46711,     1.42603,     1.48394,     1.47281,
    1.48293,     1.46559,     1.50213,     1.50402,     1.50188,
    1.56782,     1.55921,     1.57139,     1.60183,     1.63538,
    1.69792,     1.71078,     1.70737,     1.7277,      1.80492,
    1.80737,     1.78269,     1.74568,     1.62777,     1.46465,
    1.19346,     0.878359,    0.52977,     0.239186,    1.5162e-05,
    0.0,         0.101531,    0.28891,     0.552694,    0.850459,
    1.08062,     1.29137,     1.39247,     1.46913,     1.51748,
    1.52582,     1.53392,     1.52825,     1.53276,     1.5274,
    1.48028,     1.47614,     1.47714,     1.44456,     1.47176,
    1.44148,     1.44826,     1.4222,      1.41847,     1.43406,
    1.43224,     1.40558,     1.41287,     1.39139,     1.41144,
    1.42584,     1.41808,     1.41378,     1.39797,     1.41615,
    1.41542,     1.41835,     1.41232,     1.4239,      1.42639,
    1.39238,     1.38463,     1.42869,     1.41407,     1.42863,
    1.39648,     1.42284,     1.41679,     1.40945,     1.41332,
    1.41805,     1.43744,     1.41986,     1.44736,     1.43978,
    1.45798,     1.50139,     1.46387,     1.48992,     1.49162,
    1.54437,     1.55405,     1.58839,     1.63166,     1.66964,
    1.7177,      1.67296,     1.77994,     1.79481,     1.81731,
    1.85435,     1.82435,     1.78224,     1.66896,     1.50975,
    1.24273,     0.918243,    0.568198,    0.258959,    1.64895e-05,
    0.0,         0.10643,     0.302262,    0.559799,    0.856823,
    1.10762,     1.26874,     1.4048,      1.47776,     1.51321,
    1.51913,     1.51868,     1.49793,     1.53591,     1.48403,
    1.49696,     1.46878,     1.48508,     1.45379,     1.44274,
    1.42244,     1.42735,     1.44142,     1.41376,     1.40883,
    1.39117,     1.41145,     1.41548,     1.39367,     1.43077,
    1.39532,     1.38823,     1.40106,     1.41454,     1.37299,
    1.38474,     1.3831,      1.39,        1.37465,     1.37566,
    1.37635,     1.3819,      1.40989,     1.39399,     1.39903,
    1.38169,     1.37931,     1.41305,     1.38025,     1.41218,
    1.35321,     1.3923,      1.41667,     1.43837,     1.4447,
    1.45889,     1.46947,     1.44785,     1.49798,     1.54172,
    1.54238,     1.58461,     1.57037,     1.61346,     1.63795,
    1.70198,     1.7378,      1.78631,     1.83067,     1.87143,
    1.87755,     1.84544,     1.8332,      1.71025,     1.52747,
    1.29732,     0.964792,    0.599495,    0.275834,    1.75338e-05,
    0.0,         0.113131,    0.299953,    0.580048,    0.879218,
    1.13412,     1.29568,     1.41873,     1.47288,     1.50628,
    1.5574,      1.52152,     1.5353,      1.51804,     1.49714,
    1.45946,     1.46791,     1.46122,     1.45847,     1.42409,
    1.42586,     1.40548,     1.40543,     1.40856,     1.40599,
    1.39293,     1.39079,     1.3872,      1.37284,     1.38431,
    1.38804,     1.38163,     1.36665,     1.37131,     1.37464,
    1.3645,      1.35628,     1.35503,     1.35765,     1.35833,
    1.36604,     1.35138,     1.35866,     1.3737,      1.3616,
    1.35723,     1.39112,     1.39914,     1.35965,     1.38291,
    1.37106,     1.37952,     1.405,       1.42138,     1.4244,
    1.44749,     1.44391,     1.47406,     1.46767,     1.49315,
    1.51539,     1.55219,     1.58007,     1.63173,     1.63878,
    1.68029,     1.72122,     1.7845,      1.82003,     1.84257,
    1.90358,     1.88178,     1.83075,     1.77236,     1.59264,
    1.33662,     0.985711,    0.618691,    0.283248,    1.8085e-05,
    0.0,         0.115067,    0.308906,    0.595513,    0.880012,
    1.13104,     1.317,       1.41199,     1.50257,     1.51915,
    1.5219,      1.50968,     1.50158,     1.48039,     1.47591,
    1.46534,     1.44443,     1.4418,      1.42278,     1.40639,
    1.38916,     1.39099,     1.40059,     1.37446,     1.37096,
    1.36122,     1.39486,     1.36808,     1.35242,     1.38405,
    1.34792,     1.34466,     1.3473,      1.35336,     1.34882,
    1.33642,     1.34218,     1.34631,     1.35244,     1.32361,
    1.35823,     1.3163,      1.34019,     1.30326,     1.33427,
    1.33096,     1.33331,     1.3338,      1.34907,     1.34155,
    1.35079,     1.38293,     1.38371,     1.3885,      1.38639,
    1.40566,     1.41802,     1.45907,     1.45454,     1.49861,
    1.51671,     1.52887,     1.54425,     1.62436,     1.66353,
    1.68566,     1.73885,     1.82006,     1.85142,     1.84243,
    1.86704,     1.87861,     1.84713,     1.77267,     1.59411,
    1.32055,     0.997129,    0.625432,    0.289895,    1.81121e-05,
    0.0,         0.119666,    0.322164,    0.607416,    0.896668,
    1.14503,     1.32222,     1.41325,     1.49168,     1.51521,
    1.49759,     1.51677,     1.49695,     1.472,       1.47833,
    1.45863,     1.42731,     1.42492,     1.42539,     1.40659,
    1.38028,     1.37783,     1.39077,     1.36607,     1.35982,
    1.36743,     1.33976,     1.32762,     1.33088,     1.33498,
    1.34805,     1.32341,     1.29934,     1.33217,     1.3264,
    1.32254,     1.32391,     1.31779,     1.31093,     1.28681,
    1.30316,     1.3264,      1.29089,     1.30783,     1.29675,
    1.29751,     1.30583,     1.30506,     1.34438,     1.31825,
    1.33925,     1.34988,     1.35112,     1.36784,     1.36949,
    1.37731,     1.34534,     1.43329,     1.46019,     1.49177,
    1.49234,     1.52994,     1.52965,     1.56501,     1.61196,
    1.68038,     1.7001,      1.74171,     1.8119,      1.84993,
    1.87889,     1.8531,      1.80272,     1.75174,     1.59492,
    1.32284,     0.987671,    0.62521,     0.28335,     1.81601e-05,
    0.0,         0.122082,    0.327725,    0.611204,    0.909286,
    1.12944,     1.30472,     1.42574,     1.46077,     1.50625,
    1.52158,     1.47869,     1.48815,     1.47198,     1.45184,
    1.43593,     1.43894,     1.41725,     1.39879,     1.3484,
    1.36465,     1.3611,      1.34879,     1.31741,     1.32612,
    1.35491,     1.33013,     1.3366,      1.32118,     1.31127,
    1.31142,     1.32449,     1.30248,     1.29964,     1.30806,
    1.274,       1.30061,     1.28657,     1.28092,     1.275,
    1.29409,     1.27651,     1.29264,     1.30178,     1.27643,
    1.28272,     1.27589,     1.29426,     1.30423,     1.32412,
    1.28437,     1.2973,      1.34289,     1.33153,     1.30631,
    1.33892,     1.38354,     1.38664,     1.4296,      1.43924,
    1.46856,     1.50678,     1.48754,     1.56031,     1.61254,
    1.62793,     1.72922,     1.74853,     1.77454,     1.8197,
    1.82469,     1.83623,     1.82137,     1.74912,     1.55741,
    1.30399,     0.975933,    0.612646,    0.278259,    1.76319e-05,
    0.0,         0.130589,    0.337913,    0.625967,    0.899641,
    1.15044,     1.30674,     1.40365,     1.47452,     1.498,
    1.48375,     1.52827,     1.46918,     1.4905,      1.45431,
    1.4079,      1.42202,     1.37824,     1.39139,     1.3323,
    1.34717,     1.36645,     1.32511,     1.34155,     1.33577,
    1.30904,     1.299,       1.30313,     1.30508,     1.29966,
    1.30586,     1.28053,     1.27959,     1.27812,     1.28251,
    1.27609,     1.24681,     1.25733,     1.27228,     1.25309,
    1.25154,     1.28925,     1.25515,     1.25257,     1.27113,
    1.25901,     1.27358,     1.25133,     1.27,        1.29377,
    1.29059,     1.29394,     1.27074,     1.34735,     1.33961,
    1.33253,     1.38336,     1.38557,     1.38459,     1.41514,
    1.44082,     1.50583,     1.50794,     1.52948,     1.60127,
    1.59092,     1.65006,     1.68292,     1.74494,     1.79061,
    1.7967,      1.79558,     1.77994,     1.67729,     1.54616,
    1.25345,     0.952865,    0.5982,      0.264317,    1.69666e-05,
    0.0,         0.132833,    0.347241,    0.629479,    0.918164,
    1.14268,     1.33397,     1.43209,     1.44231,     1.51121,
    1.51087,     1.48457,     1.49011,     1.45146,     1.42328,
    1.40334,     1.39988,     1.38351,     1.35066,     1.3489,
    1.3346,      1.33205,     1.3208,      1.30527,     1.31553,
    1.26763,     1.2814,      1.27961,     1.28349,     1.27126,
    1.28913,     1.28721,     1.27345,     1.26252,     1.23581,
    1.26337,     1.26765,     1.23379,     1.23794,     1.24421,
    1.25276,     1.25031,     1.22498,     1.23965,     1.25124,
    1.2382,      1.24633,     1.26117,     1.24285,     1.27373,
    1.30093,     1.28484,     1.2448,      1.28828,     1.32596,
    1.36703,     1.33,        1.37065,     1.3707,      1.44716,
    1.43041,     1.46945,     1.48755,     1.53864,     1.54203,
    1.59791,     1.62251,     1.68972,     1.70151,     1.73636,
    1.7677,      1.74945,     1.71657,     1.65214,     1.4868,
    1.24378,     0.92285,     0.563638,    0.254072,    1.59789e-05,
    0.0,         0.136236,    0.347719,    0.641301,    0.910321,
    1.14905,     1.31976,     1.43519,     1.46046,     1.4869,
    1.47368,     1.46858,     1.45738,     1.43477,     1.38394,
    1.40514,     1.36886,     1.34897,     1.31734,     1.3587,
    1.3109,      1.33268,     1.30299,     1.32299,     1.29738,
    1.29282,     1.27409,     1.28335,     1.29845,     1.25237,
    1.28419,     1.2676,      1.24242,     1.26877,     1.27296,
    1.24724,     1.23505,     1.25955,     1.25384,     1.26311,
    1.22718,     1.24602,     1.235,       1.22954,     1.24685,
    1.22761,     1.27472,     1.2471,      1.23273,     1.24808,
    1.26034,     1.25546,     1.28987,     1.28975,     1.31758,
    1.34944,     1.34186,     1.35712,     1.38894,     1.37744,
    1.40266,     1.42749,     1.46546,     1.48113,     1.55543,
    1.60139,     1.58562,     1.65185,     1.67416,     1.70777,
    1.73574,     1.7365,      1.7016,      1.59531,     1.45022,
    1.22306,     0.886271,    0.546257,    0.240038,    1.49663e-05,
    0.0,         0.136167,    0.366535,    0.654885,    0.937455,
    1.14797,     1.32458,     1.3998,      1.47047,     1.46927,
    1.48784,     1.45938,     1.45301,     1.43242,     1.39926,
    1.39094,     1.36793,     1.33742,     1.34964,     1.32808,
    1.32073,     1.30151,     1.31484,     1.319,       1.29024,
    1.29977,     1.2991,      1.27043,     1.27233,     1.26742,
    1.27417,     1.27209,     1.26623,     1.24986,     1.2559,
    1.23821,     1.24973,     1.25938,     1.26965,     1.2421,
    1.23729,     1.25802,     1.23264,     1.23869,     1.2115,
    1.24787,     1.23557,     1.28088,     1.25232,     1.2528,
    1.22324,     1.27934,     1.26889,     1.23737,     1.30751,
    1.34084,     1.33055,     1.33385,     1.36447,     1.36121,
    1.4147,      1.44694,     1.45974,     1.49998,     1.51419,
    1.558,       1.59166,     1.60733,     1.65648,     1.66763,
    1.71005,     1.71312,     1.65105,     1.55614,     1.40814,
    1.17265,     0.859202,    0.521679,    0.22898,     1.41496e-05,
    0.0,         0.141093,    0.362717,    0.665216,    0.937304,
    1.18546,     1.32396,     1.42365,     1.44909,     1.46544,
    1.4996,      1.45639,     1.41879,     1.44325,     1.40397,
    1.34544,     1.38008,     1.3318,      1.32323,     1.35111,
    1.3225,      1.28001,     1.31899,     1.29523,     1.31726,
    1.29462,     1.24893,     1.27548,     1.28294,     1.28408,
    1.26762,     1.27353,     1.25513,     1.24103,     1.24478,
    1.25051,     1.25823,     1.2617,      1.22275,     1.27437,
    1.23788,     1.2371,      1.25885,     1.22501,     1.24188,
    1.25293,     1.22719,     1.24833,     1.25451,     1.25008,
    1.2622,      1.25157,     1.27894,     1.29043,     1.30438,
    1.28446,     1.34614,     1.366,       1.3664,      1.34499,
    1.40594,     1.42271,     1.44701,     1.46653,     1.52316,
    1.53441,     1.58093,     1.61572,     1.63031,     1.63295,
    1.64594,     1.63468,     1.63181,     1.52769,     1.35668,
    1.13951,     0.837625,    0.494447,    0.215106,    1.31634e-05,
    0.0,         0.145952,    0.372393,    0.679761,    0.96013,
    1.16466,     1.34883,     1.42968,     1.47158,     1.47364,
    1.46597,     1.45179,     1.439,       1.42507,     1.40068,
    1.39092,     1.35294,     1.3696,      1.36087,     1.35296,
    1.31521,     1.31632,     1.3113,      1.29074,     1.27545,
    1.29893,     1.28402,     1.28392,     1.28565,     1.26902,
    1.30058,     1.24915,     1.28318,     1.27129,     1.27937,
    1.28075,     1.27269,     1.2762,      1.23561,     1.22571,
    1.24356,     1.26737,     1.25973,     1.2774,      1.28198,
    1.22856,     1.26357,     1.25633,     1.25275,     1.23179,
    1.27305,     1.27269,     1.28042,     1.30055,     1.28974,
    1.30656,     1.34508,     1.33829,     1.35626,     1.35593,
    1.38701,     1.43197,     1.44659,     1.46761,     1.47989,
    1.50391,     1.53724,     1.59475,     1.59801,     1.5993,
    1.63252,     1.62565,     1.60009,     1.51617,     1.33937,
    1.1192,      0.807094,    0.474912,    0.199428,    1.24292e-05,
    0.0,         0.149463,    0.381311,    0.686283,    0.968341,
    1.20531,     1.33657,     1.42857,     1.47011,     1.48004,
    1.47284,     1.44559,     1.43676,     1.4175,      1.42532,
    1.4022,      1.36416,     1.35383,     1.33411,     1.35884,
    1.33185,     1.32684,     1.31656,     1.32662,     1.31908,
    1.30028,     1.31357,     1.28805,     1.28873,     1.27537,
    1.28635,     1.29965,     1.27982,     1.27307,     1.26218,
    1.28471,     1.26199,     1.27634,     1.26399,     1.27705,
    1.26588,     1.26491,     1.25391,     1.26169,     1.2752,
    1.275,       1.28039,     1.29552,     1.2685,      1.27836,
    1.27591,     1.31405,     1.29333,     1.28124,     1.33322,
    1.31832,     1.34074,     1.37094,     1.3517,      1.38348,
    1.39703,     1.43174,     1.46103,     1.4749,      1.47587,
    1.51675,     1.52113,     1.53739,     1.59445,     1.61005,
    1.63207,     1.60687,     1.56285,     1.46963,     1.33458,
    1.08816,     0.791207,    0.46331,     0.192857,    1.16023e-05,
    0.0,         0.154664,    0.394154,    0.689285,    0.971907,
    1.21206,     1.34506,     1.42894,     1.46148,     1.48954,
    1.47056,     1.48061,     1.45545,     1.42439,     1.42121,
    1.37086,     1.38326,     1.37166,     1.35372,     1.31891,
    1.31485,     1.34251,     1.33465,     1.31091,     1.29769,
    1.27953,     1.30845,     1.32088,     1.3121,      1.30765,
    1.30089,     1.29251,     1.31006,     1.2841,      1.32119,
    1.29054,     1.29166,     1.2803,      1.27953,     1.28636,
    1.30342,     1.27495,     1.27583,     1.29019,     1.29136,
    1.28151,     1.30943,     1.27683,     1.2991,      1.27224,
    1.30373,     1.32435,     1.29931,     1.31595,     1.30904,
    1.35985,     1.31609,     1.35871,     1.36205,     1.38597,
    1.41023,     1.41497,     1.42864,     1.46248,     1.48589,
    1.4661,      1.54382,     1.5322,      1.58857,     1.5779,
    1.5964,      1.58751,     1.53387,     1.45133,     1.31794,
    1.06582,     0.771558,    0.451721,    0.180888,    1.09761e-05,
    0.0,         0.159203,    0.392119,    0.703914,    1.00472,
    1.21934,     1.32824,     1.43534,     1.46969,     1.48714,
    1.48065,     1.43696,     1.44639,     1.43585,     1.41673,
    1.41307,     1.38063,     1.36393,     1.37408,     1.31703,
    1.32128,     1.35546,     1.35289,     1.33924,     1.32166,
    1.33252,     1.30302,     1.33916,     1.31947,     1.29382,
    1.339,       1.3143,      1.31204,     1.31582,     1.30052,
    1.29051,     1.31315,     1.30724,     1.2875,      1.3094,
    1.27978,     1.31636,     1.3013,      1.28897,     1.28066,
    1.30261,     1.29026,     1.3118,      1.28109,     1.30455,
    1.31918,     1.32106,     1.32797,     1.31705,     1.35051,
    1.33939,     1.3551,      1.32344,     1.38117,     1.37879,
    1.40989,     1.41659,     1.42725,     1.45989,     1.46932,
    1.49279,     1.54966,     1.55579,     1.53816,     1.60381,
    1.58306,     1.56843,     1.55124,     1.454,       1.2867,
    1.06112,     0.747753,    0.433137,    0.177588,    1.03037e-05,
    0.0,         0.163408,    0.407156,    0.719667,    1.02077,
    1.22218,     1.37703,     1.47532,     1.51197,     1.48307,
    1.47905,     1.48941,     1.43794,     1.42131,     1.42345,
    1.42959,     1.40539,     1.36778,     1.35775,     1.35484,
    1.35108,     1.36439,     1.32973,     1.32381,     1.35776,
    1.33044,     1.35805,     1.35611,     1.35122,     1.34647,
    1.33342,     1.3178,      1.32787,     1.33247,     1.32807,
    1.31675,     1.33336,     1.32487,     1.31495,     1.31643,
    1.3437,      1.31241,     1.29886,     1.31854,     1.33824,
    1.31702,     1.32268,     1.33401,     1.29621,     1.33215,
    1.33962,     1.35347,     1.34451,     1.30255,     1.36098,
    1.34501,     1.33342,     1.3874,      1.38399,     1.39685,
    1.39123,     1.42105,     1.43719,     1.44666,     1.45779,
    1.48775,     1.51882,     1.58324,     1.55263,     1.56683,
    1.58308,     1.55762,     1.51334,     1.42276,     1.29443,
    1.05639,     0.767111,    0.426231,    0.167246,    9.67319e-06,
    0.0,         0.167164,    0.418943,    0.745716,    1.04516,
    1.25718,     1.40415,     1.46892,     1.50125,     1.531,
    1.4948,      1.4851,      1.43565,     1.44466,     1.43827,
    1.39786,     1.36692,     1.36595,     1.38152,     1.37067,
    1.38193,     1.36004,     1.35787,     1.36537,     1.37118,
    1.34523,     1.35934,     1.37408,     1.34565,     1.36286,
    1.365,       1.35945,     1.36108,     1.34885,     1.34725,
    1.34701,     1.35161,     1.33669,     1.33957,     1.34486,
    1.34483,     1.32814,     1.36178,     1.3362,      1.34763,
    1.30645,     1.34698,     1.34485,     1.33807,     1.33629,
    1.3315,      1.33476,     1.34168,     1.33092,     1.37737,
    1.37694,     1.38613,     1.35457,     1.41721,     1.35673,
    1.42103,     1.45632,     1.46684,     1.46861,     1.47813,
    1.49782,     1.5198,      1.52245,     1.55729,     1.55578,
    1.58129,     1.558,       1.50729,     1.42106,     1.29229,
    1.05176,     0.737067,    0.420656,    0.164032,    9.55659e-06,
    0.0,         0.167482,    0.425542,    0.749988,    1.04331,
    1.26695,     1.40984,     1.50933,     1.49645,     1.52381,
    1.51351,     1.50855,     1.49693,     1.47166,     1.46443,
    1.38498,     1.42509,     1.38885,     1.41581,     1.40145,
    1.39521,     1.37141,     1.37879,     1.36129,     1.36684,
    1.35303,     1.3835,      1.38916,     1.35242,     1.37209,
    1.3604,      1.38038,     1.37894,     1.36638,     1.3459,
    1.36678,     1.36595,     1.36087,     1.37574,     1.37751,
    1.37522,     1.3579,      1.36585,     1.37687,     1.34243,
    1.36977,     1.36911,     1.37168,     1.34542,     1.34684,
    1.39196,     1.36998,     1.37792,     1.39391,     1.38784,
    1.35147,     1.41339,     1.39532,     1.38424,     1.40677,
    1.42317,     1.43748,     1.45297,     1.47027,     1.46027,
    1.51718,     1.51014,     1.53475,     1.54724,     1.57997,
    1.56585,     1.57292,     1.5207,      1.42769,     1.26973,
    1.03621,     0.720041,    0.410177,    0.157433,    9.05449e-06,
    0.0,         0.174169,    0.440156,    0.766789,    1.07678,
    1.29224,     1.42025,     1.51722,     1.54973,     1.53192,
    1.54365,     1.516,       1.45712,     1.47143,     1.44807,
    1.4308,      1.39005,     1.41887,     1.42461,     1.40453,
    1.38716,     1.39911,     1.40169,     1.37971,     1.38487,
    1.37735,     1.3696,      1.37898,     1.39537,     1.37164,
    1.38233,     1.39092,     1.39086,     1.396,       1.3659,
    1.3709,      1.40142,     1.40789,     1.39419,     1.38875,
    1.37344,     1.39465,     1.39517,     1.37938,     1.36881,
    1.37408,     1.39483,     1.39659,     1.37231,     1.38731,
    1.39871,     1.35323,     1.38047,     1.40339,     1.38364,
    1.40004,     1.37737,     1.41843,     1.43513,     1.41062,
    1.43897,     1.45568,     1.47144,     1.47842,     1.49389,
    1.51542,     1.54259,     1.54116,     1.54421,     1.55094,
    1.57846,     1.54917,     1.49516,     1.44071,     1.26516,
    1.06061,     0.732137,    0.410452,    0.152043,    8.89134e-06,
    0.0,         0.176433,    0.445427,    0.779284,    1.06989,
    1.31171,     1.45409,     1.50726,     1.53965,     1.5723,
    1.53262,     1.51327,     1.52571,     1.48509,     1.46384,
    1.4366,      1.43573,     1.41184,     1.43591,     1.37773,
    1.42952,     1.39934,     1.40624,     1.40804,     1.39174,
    1.3908,      1.43505,     1.41822,     1.41395,     1.39553,
    1.414,       1.40442,     1.42514,     1.41107,     1.38922,
    1.39282,     1.43757,     1.41645,     1.42245,     1.42424,
    1.42261,     1.41631,     1.40088,     1.43468,     1.4223,
    1.41092,     1.42152,     1.42012,     1.39163,     1.41257,
    1.38575,     1.43013,     1.42213,     1.40601,     1.39925,
    1.39061,     1.42312,     1.40903,     1.40127,     1.41524,
    1.45336,     1.44592,     1.47452,     1.49995,     1.49947,
    1.49246,     1.50988,     1.53227,     1.53538,     1.57907,
    1.54876,     1.55333,     1.51031,     1.44107,     1.28589,
    1.05762,     0.723081,    0.414481,    0.154558,    8.40924e-06,
    0.0,         0.175752,    0.454536,    0.791269,    1.11298,
    1.33772,     1.46429,     1.53315,     1.55952,     1.57335,
    1.54696,     1.51478,     1.51095,     1.48458,     1.47379,
    1.47565,     1.48008,     1.43933,     1.43149,     1.40642,
    1.41034,     1.44224,     1.38358,     1.43656,     1.43775,
    1.43196,     1.44534,     1.42937,     1.44823,     1.43371,
    1.47882,     1.44109,     1.42539,     1.44646,     1.43704,
    1.41928,     1.41422,     1.47074,     1.43146,     1.42666,
    1.42785,     1.44754,     1.41572,     1.41036,     1.42397,
    1.41603,     1.4627,      1.42143,     1.44162,     1.42433,
    1.42765,     1.44553,     1.40822,     1.41256,     1.44233,
    1.42494,     1.42317,     1.42084,     1.44886,     1.48876,
    1.45524,     1.45112,     1.46743,     1.47248,     1.50762,
    1.52486,     1.53719,     1.55914,     1.59738,     1.5666,
    1.5842,      1.56537,     1.50193,     1.4257,      1.28296,
    1.04194,     0.730067,    0.408146,    0.152112,    8.13735e-06,
    0.0,         0.189422,    0.47173,     0.803499,    1.11207,
    1.34811,     1.49565,     1.58888,     1.59064,     1.58471,
    1.55924,     1.56217,     1.5431,      1.57512,     1.47878,
    1.50846,     1.46558,     1.43206,     1.48762,     1.4256,
    1.44921,     1.46196,     1.4613,      1.43932,     1.45466,
    1.43248,     1.46922,     1.45199,     1.43058,     1.4687,
    1.49081,     1.45101,     1.44035,     1.45406,     1.48369,
    1.44702,     1.47484,     1.43775,     1.44186,     1.47796,
    1.47579,     1.45916,     1.45593,     1.44876,     1.43208,
    1.46299,     1.46545,     1.43434,     1.46436,     1.43943,
    1.45342,     1.42428,     1.43752,     1.45772,     1.43371,
    1.46319,     1.48202,     1.45585,     1.4551,      1.47886,
    1.47214,     1.48777,     1.49511,     1.49681,     1.51646,
    1.5031,      1.5149,      1.53573,     1.58216,     1.59601,
    1.60128,     1.55092,     1.53354,     1.44061,     1.29106,
    1.07052,     0.746577,    0.403645,    0.146137,    7.88662e-06,
    0.0,         0.191282,    0.480256,    0.82755,     1.13169,
    1.39598,     1.50213,     1.57039,     1.60175,     1.6168,
    1.58735,     1.57811,     1.56396,     1.50792,     1.50712,
    1.51429,     1.51497,     1.47512,     1.47361,     1.45873,
    1.44945,     1.47742,     1.42296,     1.4774,      1.50442,
    1.48452,     1.45092,     1.45336,     1.48868,     1.52121,
    1.50241,     1.4706,      1.4902,      1.50881,     1.45922,
    1.45878,     1.47066,     1.49128,     1.45955,     1.52437,
    1.47466,     1.47752,     1.52459,     1.46458,     1.49053,
    1.48164,     1.50423,     1.4936,      1.46943,     1.49634,
    1.48614,     1.459,       1.47321,     1.44704,     1.46374,
    1.49142,     1.4473,      1.47854,     1.48313,     1.46985,
    1.47484,     1.46412,     1.50656,     1.51136,     1.5157,
    1.56151,     1.55031,     1.56704,     1.57036,     1.59176,
    1.59425,     1.57957,     1.53514,     1.45696,     1.30701,
    1.06314,     0.752377,    0.403141,    0.14881,     7.78645e-06,
    0.0,         0.195736,    0.49961,     0.84227,     1.17026,
    1.37384,     1.50904,     1.56743,     1.61957,     1.64945,
    1.58331,     1.56453,     1.56877,     1.53144,     1.5434,
    1.52247,     1.50434,     1.47945,     1.47846,     1.48838,
    1.48875,     1.48874,     1.44806,     1.4443,      1.49001,
    1.50817,     1.48329,     1.5124,      1.5057,      1.51426,
    1.52343,     1.48829,     1.53408,     1.48265,     1.54299,
    1.48181,     1.50353,     1.54173,     1.49555,     1.51807,
    1.53444,     1.51115,     1.51026,     1.49932,     1.50307,
    1.52395,     1.51214,     1.47972,     1.50861,     1.50594,
    1.47424,     1.53304,     1.4901,      1.4782,      1.50857,
    1.49438,     1.50201,     1.51464,     1.48016,     1.50174,
    1.50543,     1.5339,      1.4928,      1.55016,     1.53949,
    1.54775,     1.5894,      1.57878,     1.56811,     1.57421,
    1.59325,     1.60825,     1.5534,      1.4536,      1.30096,
    1.0414,      0.734117,    0.394351,    0.147059,    7.51975e-06,
    0.0,         0.197114,    0.497818,    0.874061,    1.20461,
    1.41082,     1.56085,     1.63678,     1.64406,     1.62329,
    1.62972,     1.59394,     1.5815,      1.53046,     1.52386,
    1.51543,     1.52655,     1.54611,     1.52874,     1.50956,
    1.52843,     1.50856,     1.48714,     1.51617,     1.50422,
    1.50808,     1.51314,     1.54061,     1.52709,     1.5195,
    1.5429,      1.52196,     1.50947,     1.53384,     1.53821,
    1.54258,     1.52202,     1.5559,      1.54845,     1.57595,
    1.5464,      1.55367,     1.53668,     1.53644,     1.52355,
    1.55083,     1.52177,     1.54942,     1.54207,     1.52565,
    1.53285,     1.51688,     1.55547,     1.51559,     1.48155,
    1.51852,     1.55775,     1.54034,     1.53798,     1.51323,
    1.50764,     1.50368,     1.51515,     1.54444,     1.55557,
    1.57747,     1.57639,     1.5833,      1.60829,     1.60067,
    1.62372,     1.60197,     1.57061,     1.46666,     1.31701,
    1.08446,     0.756783,    0.408589,    0.142192,    7.64063e-06,
    0.0,         0.203949,    0.519619,    0.882313,    1.20358,
    1.47824,     1.611,       1.65006,     1.68659,     1.64854,
    1.64077,     1.6406,      1.58101,     1.57665,     1.55572,
    1.53268,     1.52228,     1.55179,     1.5721,      1.53998,
    1.52447,     1.51368,     1.54056,     1.55451,     1.53233,
    1.53846,     1.545,       1.54465,     1.56154,     1.56827,
    1.55366,     1.55446,     1.53158,     1.57473,     1.57054,
    1.52865,     1.52572,     1.57789,     1.55649,     1.59531,
    1.597,       1.56349,     1.55706,     1.53835,     1.57052,
    1.56897,     1.54464,     1.58162,     1.54948,     1.54234,
    1.57116,     1.5123,      1.54139,     1.55127,     1.52323,
    1.50001,     1.52195,     1.56332,     1.52599,     1.565,
    1.51071,     1.53075,     1.54208,     1.53682,     1.5628,
    1.59088,     1.57627,     1.60311,     1.60814,     1.64126,
    1.62442,     1.61186,     1.58136,     1.48673,     1.33983,
    1.10081,     0.764799,    0.412269,    0.141271,    7.30343e-06,
    0.0,         0.208577,    0.52685,     0.918035,    1.24118,
    1.4941,      1.60479,     1.65991,     1.67277,     1.67085,
    1.66164,     1.651,       1.63599,     1.59064,     1.56518,
    1.58487,     1.56751,     1.53702,     1.56918,     1.52969,
    1.54624,     1.5485,      1.57396,     1.54404,     1.55565,
    1.58979,     1.56034,     1.59958,     1.60974,     1.53355,
    1.61569,     1.60276,     1.57989,     1.58082,     1.61293,
    1.60801,     1.6128,      1.577,       1.58182,     1.60283,
    1.60065,     1.61212,     1.57949,     1.56353,     1.60108,
    1.54478,     1.60682,     1.58511,     1.58153,     1.57093,
    1.56276,     1.58554,     1.5667,      1.57159,     1.52186,
    1.56362,     1.5686,      1.55794,     1.53839,     1.54649,
    1.54347,     1.54961,     1.56943,     1.59365,     1.56658,
    1.57394,     1.59054,     1.59098,     1.64716,     1.63292,
    1.61127,     1.61706,     1.58932,     1.50133,     1.34346,
    1.09415,     0.768538,    0.423667,    0.138933,    7.3433e-06,
    0.0,         0.219079,    0.54454,     0.927007,    1.26118,
    1.47071,     1.64587,     1.68261,     1.68627,     1.70133,
    1.67722,     1.64106,     1.65214,     1.60917,     1.60748,
    1.61035,     1.57901,     1.58777,     1.56805,     1.56909,
    1.60678,     1.58207,     1.6067,      1.55091,     1.60505,
    1.60078,     1.587,       1.54386,     1.58448,     1.61288,
    1.64237,     1.59523,     1.61274,     1.62795,     1.612,
    1.65065,     1.62733,     1.62235,     1.63799,     1.60969,
    1.6255,      1.6186,      1.63255,     1.60732,     1.5932,
    1.59179,     1.62076,     1.60387,     1.57585,     1.61137,
    1.59503,     1.5838,      1.59701,     1.58079,     1.58274,
    1.58374,     1.54825,     1.58045,     1.5777,      1.59433,
    1.59475,     1.58598,     1.58895,     1.59157,     1.62195,
    1.6066,      1.60014,     1.65573,     1.63528,     1.63458,
    1.65419,     1.67065,     1.63911,     1.51284,     1.37772,
    1.1275,      0.799602,    0.422752,    0.140959,    7.04252e-06,
    0.0,         0.213622,    0.554905,    0.940794,    1.28178,
    1.51915,     1.65961,     1.71868,     1.73187,     1.75131,
    1.70429,     1.66648,     1.6626,      1.62728,     1.57798,
    1.60641,     1.60151,     1.60387,     1.63115,     1.56251,
    1.59935,     1.61364,     1.61418,     1.61103,     1.63498,
    1.6534,      1.64102,     1.61114,     1.65834,     1.62921,
    1.63222,     1.63319,     1.63732,     1.65796,     1.67528,
    1.65407,     1.69912,     1.63917,     1.623,       1.7042,
    1.63679,     1.64721,     1.64307,     1.65249,     1.66022,
    1.64289,     1.64395,     1.63843,     1.61637,     1.6304,
    1.61564,     1.58882,     1.61879,     1.58641,     1.61803,
    1.57055,     1.59083,     1.60569,     1.60542,     1.57965,
    1.57846,     1.6132,      1.59844,     1.61708,     1.59184,
    1.63423,     1.60717,     1.61713,     1.65287,     1.64463,
    1.68853,     1.65064,     1.61432,     1.53558,     1.36313,
    1.13456,     0.805484,    0.427514,    0.139907,    7.10322e-06,
    0.0,         0.223539,    0.560744,    0.96606,     1.30022,
    1.57018,     1.68364,     1.75432,     1.76001,     1.72686,
    1.74653,     1.71402,     1.66076,     1.68121,     1.63057,
    1.61326,     1.63159,     1.63515,     1.63369,     1.63981,
    1.64049,     1.6307,      1.63623,     1.60306,     1.65579,
    1.6828,      1.62873,     1.66149,     1.64767,     1.62958,
    1.65734,     1.69866,     1.71549,     1.67103,     1.67967,
    1.67821,     1.71204,     1.71275,     1.68136,     1.64841,
    1.68436,     1.66718,     1.6893,      1.68381,     1.67627,
    1.65966,     1.66929,     1.66282,     1.6654,      1.65048,
    1.63167,     1.66883,     1.64007,     1.61746,     1.64545,
    1.62355,     1.64579,     1.62736,     1.61059,     1.63609,
    1.64308,     1.58364,     1.62177,     1.64022,     1.633,
    1.65452,     1.66893,     1.6249,      1.65584,     1.67233,
    1.6903,      1.6645,      1.64254,     1.5314,      1.41177,
    1.16067,     0.803938,    0.433862,    0.138958,    6.9935e-06,
    0.0,         0.222354,    0.576702,    0.98796,     1.35411,
    1.55866,     1.71172,     1.78978,     1.79628,     1.78306,
    1.76878,     1.71852,     1.69629,     1.69722,     1.6832,
    1.68115,     1.62377,     1.67427,     1.63571,     1.67853,
    1.6482,      1.6581,      1.63028,     1.64024,     1.63782,
    1.65742,     1.66145,     1.65818,     1.68305,     1.71124,
    1.74067,     1.68414,     1.69674,     1.70021,     1.72706,
    1.76672,     1.7313,      1.76403,     1.70139,     1.73336,
    1.69636,     1.73165,     1.67991,     1.73453,     1.7503,
    1.66855,     1.69358,     1.68129,     1.68031,     1.66667,
    1.64769,     1.70359,     1.64532,     1.68012,     1.71541,
    1.68731,     1.64281,     1.64184,     1.66468,     1.63206,
    1.63297,     1.63214,     1.64241,     1.66167,     1.65253,
    1.66886,     1.68024,     1.71405,     1.66126,     1.71124,
    1.69193,     1.68541,     1.63395,     1.58545,     1.44273,
    1.17587,     0.817979,    0.44348,     0.145806,    7.12425e-06,
    0.0,         0.226216,    0.593809,    1.02216,     1.37462,
    1.62116,     1.73815,     1.80966,     1.7927,      1.80957,
    1.77282,     1.73159,     1.73697,     1.69235,     1.70427,
    1.68412,     1.64598,     1.67239,     1.6326,      1.69307,
    1.63932,     1.64652,     1.67554,     1.70271,     1.69498,
    1.67044,     1.69632,     1.74219,     1.6819,      1.68126,
    1.72171,     1.7426,      1.72799,     1.74712,     1.72993,
    1.75791,     1.77024,     1.75161,     1.75562,     1.76447,
    1.76838,     1.72752,     1.73869,     1.73156,     1.74492,
    1.72356,     1.74546,     1.70171,     1.7371,      1.72894,
    1.69135,     1.67532,     1.6763,      1.65309,     1.69409,
    1.67519,     1.6958,      1.64102,     1.65776,     1.66077,
    1.65431,     1.66962,     1.65395,     1.68763,     1.68291,
    1.66825,     1.67437,     1.67084,     1.72655,     1.72622,
    1.67908,     1.74399,     1.70687,     1.60479,     1.45081,
    1.2117,      0.843795,    0.445177,    0.14796,     6.98552e-06,
    0.0,         0.23541,     0.603017,    1.04073,     1.3996,
    1.62588,     1.77114,     1.81839,     1.84666,     1.79575,
    1.80829,     1.74098,     1.7468,      1.7461,      1.7163,
    1.7103,      1.69469,     1.71231,     1.6815,      1.70418,
    1.68166,     1.67333,     1.70495,     1.74057,     1.70737,
    1.71819,     1.77054,     1.76223,     1.75933,     1.74592,
    1.75659,     1.7495,      1.74147,     1.77876,     1.74045,
    1.79267,     1.77514,     1.78851,     1.75327,     1.81041,
    1.7899,      1.82176,     1.79804,     1.78558,     1.77226,
    1.76478,     1.79942,     1.74624,     1.78499,     1.70186,
    1.72594,     1.7339,      1.70202,     1.72515,     1.72313,
    1.71746,     1.69101,     1.69053,     1.72892,     1.68354,
    1.67856,     1.66737,     1.6585,      1.70134,     1.69202,
    1.71215,     1.71014,     1.70488,     1.72588,     1.74831,
    1.78343,     1.7474,      1.71586,     1.61907,     1.44666,
    1.21807,     0.867149,    0.45369,     0.141668,    7.18327e-06,
    0.0,         0.239957,    0.623438,    1.07232,     1.44914,
    1.68604,     1.78699,     1.85932,     1.87556,     1.8929,
    1.78018,     1.80019,     1.76445,     1.76269,     1.72719,
    1.70957,     1.74392,     1.73136,     1.7218,      1.71997,
    1.6763,      1.70237,     1.7477,      1.72092,     1.76588,
    1.73691,     1.73593,     1.74952,     1.79028,     1.71207,
    1.79444,     1.79213,     1.80591,     1.83895,     1.80809,
    1.78494,     1.83942,     1.82165,     1.81094,     1.83182,
    1.79719,     1.79555,     1.81522,     1.80882,     1.81576,
    1.81762,     1.7931,      1.77593,     1.78229,     1.7674,
    1.79432,     1.73809,     1.7598,      1.78021,     1.72397,
    1.73774,     1.72505,     1.71755,     1.71248,     1.73245,
    1.70089,     1.70466,     1.68339,     1.71335,     1.70736,
    1.70556,     1.72549,     1.73741,     1.78124,     1.76743,
    1.76046,     1.75263,     1.74978,     1.63489,     1.49847,
    1.22925,     0.850732,    0.464948,    0.145581,    7.05881e-06,
    0.0,         0.244991,    0.642777,    1.11695,     1.45045,
    1.72677,     1.82511,     1.86959,     1.88565,     1.87971,
    1.87867,     1.80833,     1.76447,     1.77,        1.74213,
    1.77711,     1.73546,     1.77837,     1.74015,     1.70519,
    1.72272,     1.75865,     1.76229,     1.7965,      1.77114,
    1.81073,     1.77749,     1.84134,     1.82911,     1.7824,
    1.8556,      1.8302,      1.8288,      1.84082,     1.84069,
    1.83974,     1.85688,     1.82391,     1.8251,      1.85439,
    1.83548,     1.88425,     1.81328,     1.80646,     1.80851,
    1.82293,     1.81303,     1.82275,     1.82781,     1.79998,
    1.79277,     1.79759,     1.78404,     1.77776,     1.80319,
    1.75387,     1.71962,     1.73247,     1.75784,     1.75444,
    1.73773,     1.72147,     1.72402,     1.74748,     1.73791,
    1.70251,     1.73092,     1.7659,      1.74715,     1.78386,
    1.81325,     1.78686,     1.75009,     1.68956,     1.50929,
    1.2794,      0.885365,    0.472712,    0.147945,    6.98207e-06,
    0.0,         0.252763,    0.661099,    1.10651,     1.50275,
    1.7435,      1.87682,     1.9708,      1.94363,     1.94411,
    1.91224,     1.865,       1.81357,     1.81254,     1.79191,
    1.78674,     1.74113,     1.74639,     1.72432,     1.73305,
    1.76149,     1.75394,     1.78666,     1.8062,      1.8072,
    1.82514,     1.82095,     1.80953,     1.81305,     1.83149,
    1.85484,     1.89461,     1.89004,     1.84445,     1.90922,
    1.86171,     1.89095,     1.87018,     1.86352,     1.87286,
    1.83894,     1.88603,     1.88058,     1.88416,     1.85313,
    1.84248,     1.80589,     1.84633,     1.84443,     1.82186,
    1.85106,     1.837,       1.80252,     1.80727,     1.76545,
    1.7904,      1.77832,     1.75473,     1.77801,     1.72473,
    1.7223,      1.75841,     1.7314,      1.75332,     1.75785,
    1.77022,     1.78221,     1.78167,     1.79569,     1.82165,
    1.81463,     1.81167,     1.80421,     1.7074,      1.53099,
    1.28682,     0.928828,    0.4781,      0.150259,    7.25403e-06,
    0.0,         0.259186,    0.676803,    1.13245,     1.54588,
    1.78209,     1.89826,     1.94108,     1.94207,     1.93134,
    1.93331,     1.84856,     1.84387,     1.8237,      1.78418,
    1.80236,     1.76023,     1.74855,     1.81363,     1.77004,
    1.79615,     1.80508,     1.83907,     1.85463,     1.88606,
    1.81797,     1.8696,      1.8362,      1.87245,     1.86123,
    1.88958,     1.9145,      1.92623,     1.94174,     1.96377,
    1.88102,     1.92297,     1.96054,     1.9174,      1.92712,
    1.88339,     1.92069,     1.94025,     1.9178,      1.9301,
    1.89036,     1.88922,     1.90587,     1.89216,     1.86082,
    1.85005,     1.87334,     1.84158,     1.85372,     1.82279,
    1.81772,     1.78825,     1.84093,     1.79138,     1.80987,
    1.79291,     1.79903,     1.72008,     1.79929,     1.78171,
    1.79393,     1.82856,     1.79584,     1.82084,     1.81487,
    1.81373,     1.80827,     1.81567,     1.72064,     1.57105,
    1.3142,      0.914493,    0.482022,    0.156088,    7.1919e-06,
    0.0,         0.259419,    0.677444,    1.1632,      1.57351,
    1.78467,     1.91056,     1.94347,     1.97063,     1.9715,
    1.91485,     1.91044,     1.85925,     1.82541,     1.85435,
    1.7951,      1.84274,     1.79623,     1.81815,     1.79444,
    1.82241,     1.8405,      1.8244,      1.84958,     1.89272,
    1.85922,     1.87726,     1.89202,     1.86391,     1.91626,
    1.90674,     1.94718,     1.92524,     1.92825,     1.93492,
    1.95749,     1.93081,     1.98737,     1.95561,     1.9727,
    1.93516,     1.93776,     1.92086,     1.88261,     1.9588,
    1.90517,     1.98521,     1.93553,     1.93868,     1.88619,
    1.86746,     1.89234,     1.88641,     1.85924,     1.86923,
    1.83938,     1.82501,     1.86274,     1.84632,     1.81356,
    1.8042,      1.77449,     1.77314,     1.81216,     1.80827,
    1.81102,     1.83236,     1.82528,     1.85576,     1.84427,
    1.85782,     1.86592,     1.82055,     1.76287,     1.61533,
    1.34542,     0.952481,    0.484897,    0.15618,     7.15535e-06,
    0.0,         0.269733,    0.707544,    1.19369,     1.56696,
    1.86524,     1.97624,     2.02187,     2.00214,     2.02031,
    1.94377,     1.96093,     1.90634,     1.86587,     1.85547,
    1.84558,     1.86879,     1.83467,     1.84221,     1.85609,
    1.84182,     1.86272,     1.85622,     1.83104,     1.86614,
    1.8937,      1.96372,     1.90249,     1.915,       1.88573,
    1.95924,     1.90703,     1.992,       2.01033,     1.94676,
    1.9831,      1.97175,     1.94084,     1.99369,     1.9869,
    1.97548,     1.92225,     1.99224,     1.96941,     2.02297,
    1.95763,     1.90839,     1.95389,     1.95171,     1.89295,
    1.94665,     1.90751,     1.91129,     1.91386,     1.9501,
    1.8936,      1.87891,     1.85121,     1.85595,     1.85451,
    1.84777,     1.80966,     1.8286,      1.83409,     1.83099,
    1.86511,     1.84419,     1.83868,     1.88379,     1.87514,
    1.92634,     1.87852,     1.85423,     1.809,       1.65072,
    1.36364,     0.965768,    0.507202,    0.152578,    7.27669e-06,
    0.0,         0.276845,    0.717018,    1.23551,     1.61429,
    1.8549,      2.00947,     2.05748,     2.04499,     2.02631,
    1.987,       1.92559,     1.9334,      1.93484,     1.90843,
    1.88614,     1.89615,     1.85528,     1.8845,      1.85146,
    1.8619,      1.91322,     1.89487,     1.88415,     1.91437,
    1.94944,     1.91008,     1.97771,     2.00132,     1.98416,
    1.95754,     2.00336,     1.9745,      2.01848,     2.02307,
    2.01627,     2.00427,     2.01162,     2.01778,     1.97325,
    2.04384,     1.98675,     2.00264,     2.00779,     2.00087,
    1.98869,     2.01349,     1.97029,     1.98704,     1.97097,
    1.97953,     1.93082,     1.931,       1.95779,     1.85346,
    1.89582,     1.84818,     1.86351,     1.84374,     1.82177,
    1.87488,     1.86589,     1.84889,     1.82587,     1.8604,
    1.84375,     1.87714,     1.86433,     1.91201,     1.89918,
    1.91044,     1.92923,     1.91526,     1.83717,     1.65141,
    1.3968,      1.00764,     0.516267,    0.158004,    7.17937e-06,
    0.0,         0.282223,    0.716759,    1.25413,     1.6701,
    1.9065,      2.07876,     2.05422,     2.1046,      2.07575,
    2.02356,     1.99835,     1.9751,      1.93438,     1.9181,
    1.9465,      1.89384,     1.88093,     1.91569,     1.92571,
    1.87626,     1.91387,     1.90534,     1.93636,     1.96705,
    1.92713,     2.02078,     2.02091,     2.01805,     1.99237,
    1.99699,     2.01826,     1.99201,     2.02058,     2.0468,
    2.0486,      2.06853,     2.05722,     2.02164,     2.07706,
    2.0737,      2.03988,     2.02765,     2.0763,      2.01465,
    2.06357,     2.02972,     1.99152,     2.02037,     1.99208,
    1.96751,     1.94186,     1.9295,      1.95708,     1.95274,
    1.91942,     1.93629,     1.91355,     1.90741,     1.89164,
    1.8714,      1.84122,     1.90495,     1.89192,     1.85765,
    1.90828,     1.88897,     1.92955,     1.87653,     1.90407,
    1.96063,     1.94253,     1.91304,     1.83047,     1.6943,
    1.4284,      1.01593,     0.526791,    0.157991,    7.26578e-06,
    0.0,         0.286733,    0.745292,    1.28079,     1.67969,
    1.97373,     2.07831,     2.13318,     2.12361,     2.09956,
    2.05081,     2.02397,     1.97022,     1.95604,     1.9633,
    1.93416,     1.94434,     1.91873,     1.93605,     1.91177,
    1.95884,     1.94576,     1.9578,      1.92363,     1.9629,
    1.98181,     1.98506,     2.03446,     2.0364,      2.04713,
    1.96721,     2.11576,     2.05542,     2.09596,     2.06082,
    2.0906,      2.09563,     2.10855,     2.04328,     2.11318,
    2.12062,     2.06014,     2.12096,     2.08044,     2.06946,
    2.05671,     2.08089,     2.04896,     2.0778,      2.07575,
    2.02418,     2.01405,     1.99546,     1.99166,     1.98482,
    1.94681,     1.92158,     1.92937,     1.93307,     1.92464,
    1.89377,     1.91569,     1.90738,     1.89514,     1.88324,
    1.91335,     1.94319,     1.93919,     1.92197,     1.96639,
    1.97857,     1.95565,     1.93386,     1.87999,     1.72603,
    1.44411,     1.03592,     0.538278,    0.162774,    7.17158e-06,
    0.0,         0.291304,    0.769362,    1.30849,     1.72733,
    1.96469,     2.10079,     2.131,       2.11344,     2.16261,
    2.06228,     2.02085,     2.01374,     1.97413,     1.9782,
    1.95885,     1.95606,     1.9471,      1.94069,     1.97642,
    1.96594,     1.93513,     1.98065,     1.98274,     2.04011,
    2.00195,     2.0537,      2.0209,      2.03531,     2.04422,
    2.09859,     2.04504,     2.07662,     2.08592,     2.08998,
    2.1339,      2.08829,     2.10083,     2.15138,     2.1295,
    2.09226,     2.15289,     2.11284,     2.13637,     2.10093,
    2.12472,     2.14104,     2.04582,     2.11703,     2.08494,
    2.0648,      2.04115,     1.97915,     2.04845,     1.98996,
    1.98052,     2.0206,      1.96471,     1.94941,     1.93053,
    1.92648,     1.93079,     1.93435,     1.90347,     1.9239,
    1.91759,     1.95676,     1.95902,     1.96357,     1.97536,
    2.01227,     1.98598,     1.97593,     1.947,       1.78116,
    1.48358,     1.05666,     0.555889,    0.169836,    7.30707e-06,
    0.0,         0.298547,    0.790497,    1.33603,     1.75427,
    2.02056,     2.17166,     2.18356,     2.17563,     2.1548,
    2.09925,     2.07382,     2.04759,     2.01719,     2.01617,
    2.01126,     1.9594,      1.95743,     2.02292,     1.97399,
    1.98038,     2.01087,     2.04849,     2.03341,     2.07512,
    2.02066,     2.08294,     2.09104,     2.08164,     2.15038,
    2.13054,     2.10883,     2.1197,      2.17491,     2.20768,
    2.10407,     2.14449,     2.18742,     2.1836,      2.19927,
    2.21692,     2.18107,     2.15498,     2.12874,     2.1295,
    2.13377,     2.10296,     2.10689,     2.16757,     2.10278,
    2.08591,     2.07457,     2.02098,     2.02486,     2.04926,
    2.01847,     1.97287,     1.98696,     2.0127,      2.00712,
    1.94149,     1.92892,     1.95461,     1.97163,     1.92794,
    1.9408,      1.9944,      1.98527,     1.98929,     2.0198,
    2.02078,     2.02339,     2.0186,      1.93239,     1.77393,
    1.51893,     1.07601,     0.573981,    0.170081,    7.69238e-06,
    0.0,         0.305897,    0.791636,    1.35021,     1.80705,
    2.06608,     2.19081,     2.21403,     2.1962,      2.1974,
    2.15144,     2.12599,     2.06276,     2.03684,     2.02142,
    2.02789,     2.01402,     2.00988,     1.97407,     2.01551,
    2.05603,     2.02156,     2.02544,     2.07497,     2.07882,
    2.06956,     2.13726,     2.09739,     2.12229,     2.15085,
    2.13991,     2.17628,     2.13882,     2.23099,     2.15267,
    2.2049,      2.19118,     2.20472,     2.22588,     2.19787,
    2.15601,     2.1901,      2.16042,     2.18449,     2.17229,
    2.16857,     2.19517,     2.15702,     2.14943,     2.12449,
    2.15486,     2.11099,     2.11579,     2.06295,     2.09351,
    2.04165,     2.02818,     2.02823,     1.98268,     1.98809,
    2.00195,     1.9823,      1.98406,     1.99747,     1.99323,
    1.96163,     1.9625,      1.99427,     2.03159,     2.05967,
    2.08151,     2.0453,      2.06522,     1.96123,     1.78274,
    1.55307,     1.12098,     0.581348,    0.173009,    7.4362e-06,
    0.0,         0.315832,    0.805561,    1.38605,     1.85247,
    2.11032,     2.23369,     2.28023,     2.26355,     2.24598,
    2.18395,     2.16883,     2.14584,     2.0672,      1.99045,
    2.03941,     2.00579,     2.04043,     1.96622,     2.04594,
    2.0643,      2.08174,     2.13038,     2.10159,     2.1103,
    2.14352,     2.16699,     2.14522,     2.15682,     2.19733,
    2.20749,     2.19602,     2.23485,     2.1614,      2.25616,
    2.19478,     2.21533,     2.24638,     2.24664,     2.28642,
    2.20352,     2.22366,     2.24328,     2.20831,     2.25719,
    2.19213,     2.23144,     2.19599,     2.1497,      2.13041,
    2.13353,     2.1606,      2.14478,     2.09543,     2.09042,
    2.09162,     2.09574,     2.05153,     2.0386,      1.9987,
    2.00133,     2.00487,     2.0505,      2.00357,     2.01702,
    2.02445,     2.04305,     2.00875,     2.05518,     2.07855,
    2.07732,     2.13661,     2.06681,     2.00671,     1.85653,
    1.54342,     1.13105,     0.584349,    0.179685,    7.31148e-06,
    0.0,         0.310644,    0.845166,    1.41458,     1.8974,
    2.15289,     2.27279,     2.29873,     2.30336,     2.27081,
    2.23544,     2.18586,     2.1806,      2.1027,      2.06813,
    2.06423,     2.04864,     2.00691,     2.06643,     2.05079,
    2.10067,     2.12703,     2.12223,     2.12281,     2.0851,
    2.16751,     2.2133,      2.13803,     2.18673,     2.17886,
    2.23758,     2.25676,     2.20701,     2.2169,      2.23394,
    2.27059,     2.30417,     2.30362,     2.29148,     2.21103,
    2.27079,     2.26565,     2.27918,     2.28753,     2.26181,
    2.25674,     2.27145,     2.19415,     2.18739,     2.19901,
    2.1992,      2.18445,     2.18411,     2.12671,     2.13531,
    2.12262,     2.10242,     2.08152,     2.05926,     2.05925,
    2.05973,     2.04273,     2.02438,     2.01612,     2.05887,
    2.0376,      2.04162,     2.04502,     2.05444,     2.05749,
    2.08897,     2.15657,     2.08651,     2.06552,     1.90409,
    1.61543,     1.15512,     0.615457,    0.183143,    7.57033e-06,
    0.0,         0.315495,    0.844775,    1.46302,     1.95137,
    2.18106,     2.2563,      2.37044,     2.32458,     2.33941,
    2.21124,     2.2035,      2.1549,      2.17276,     2.13673,
    2.1182,      2.11312,     2.08992,     2.1032,      2.09084,
    2.12073,     2.13225,     2.18545,     2.16608,     2.19418,
    2.18203,     2.22316,     2.21165,     2.19424,     2.26347,
    2.25806,     2.2225,      2.26623,     2.30809,     2.28483,
    2.30591,     2.32326,     2.29226,     2.30675,     2.29208,
    2.32134,     2.28175,     2.27812,     2.31797,     2.27978,
    2.24848,     2.26614,     2.26837,     2.24629,     2.22186,
    2.20416,     2.2214,      2.20699,     2.18114,     2.16825,
    2.1243,      2.1233,      2.12741,     2.14213,     2.0683,
    2.08318,     2.06545,     2.08734,     2.0685,      2.0748,
    2.04821,     2.05915,     2.10715,     2.13646,     2.1119,
    2.17972,     2.16502,     2.16714,     2.17367,     1.92963,
    1.66715,     1.17614,     0.634832,    0.182112,    7.97128e-06,
    0.0,         0.33303,     0.885297,    1.48345,     1.97158,
    2.20443,     2.37197,     2.36463,     2.35916,     2.29706,
    2.26733,     2.22812,     2.1812,      2.1806,      2.16629,
    2.16237,     2.14428,     2.07322,     2.12009,     2.18171,
    2.18893,     2.17346,     2.12459,     2.16478,     2.21572,
    2.2215,      2.20685,     2.24313,     2.27006,     2.28073,
    2.2896,      2.30204,     2.33734,     2.32756,     2.32794,
    2.34126,     2.29605,     2.34719,     2.35562,     2.32332,
    2.31504,     2.32066,     2.34156,     2.39981,     2.30952,
    2.33368,     2.32406,     2.32778,     2.31994,     2.25456,
    2.25663,     2.24636,     2.21576,     2.21081,     2.17018,
    2.22423,     2.13796,     2.16261,     2.11782,     2.13436,
    2.10281,     2.08267,     2.0662,      2.11725,     2.11449,
    2.09407,     2.11958,     2.12214,     2.13414,     2.19676,
    2.16131,     2.1397,      2.19064,     2.11567,     1.96826,
    1.68003,     1.22555,     0.659199,    0.182745,    8.15299e-06,
    0.0,         0.335863,    0.89716,     1.52674,     1.9881,
    2.23955,     2.34126,     2.48083,     2.37788,     2.29684,
    2.28003,     2.2525,      2.21001,     2.21342,     2.19184,
    2.11332,     2.20206,     2.17981,     2.16575,     2.16175,
    2.1944,      2.22708,     2.18838,     2.29179,     2.27064,
    2.26678,     2.25361,     2.2838,      2.30281,     2.2882,
    2.36145,     2.31722,     2.36046,     2.35773,     2.37749,
    2.38029,     2.35281,     2.417,       2.41029,     2.36347,
    2.38512,     2.36824,     2.37234,     2.3482,      2.36884,
    2.36233,     2.41503,     2.36891,     2.37538,     2.30774,
    2.32459,     2.29898,     2.21823,     2.24598,     2.25513,
    2.1752,      2.23089,     2.19975,     2.16477,     2.17404,
    2.14099,     2.13471,     2.12217,     2.1429,      2.1495,
    2.13983,     2.146,       2.14667,     2.16306,     2.19609,
    2.23251,     2.22786,     2.24786,     2.17536,     2.01268,
    1.7268,      1.26988,     0.647723,    0.190114,    8.04756e-06,
    0.0,         0.346307,    0.91214,     1.57351,     2.01334,
    2.30461,     2.46212,     2.51114,     2.4164,      2.40576,
    2.33144,     2.31151,     2.26794,     2.22382,     2.21131,
    2.22243,     2.21701,     2.19824,     2.1994,      2.22528,
    2.23703,     2.23696,     2.24354,     2.2435,      2.29555,
    2.30271,     2.28827,     2.36094,     2.32226,     2.3732,
    2.43022,     2.36848,     2.38088,     2.40881,     2.38579,
    2.41782,     2.43068,     2.44332,     2.41144,     2.42484,
    2.34711,     2.48107,     2.41216,     2.4497,      2.38667,
    2.40892,     2.37075,     2.37648,     2.33028,     2.35743,
    2.3374,      2.3012,      2.31673,     2.26133,     2.28251,
    2.23445,     2.21521,     2.22673,     2.25371,     2.16268,
    2.14888,     2.17646,     2.15077,     2.20788,     2.13635,
    2.15411,     2.17286,     2.18054,     2.19523,     2.24222,
    2.21779,     2.29051,     2.24551,     2.20286,     2.08678,
    1.75965,     1.28534,     0.683268,    0.196426,    8.18136e-06,
    0.0,         0.351357,    0.934671,    1.60634,     2.04963,
    2.36067,     2.45121,     2.47781,     2.47506,     2.40437,
    2.34402,     2.35108,     2.33226,     2.28602,     2.26738,
    2.22979,     2.21997,     2.25842,     2.24631,     2.20335,
    2.21947,     2.2719,      2.30329,     2.27675,     2.33768,
    2.36359,     2.34022,     2.32775,     2.42271,     2.35216,
    2.4103,      2.45471,     2.44764,     2.41027,     2.47502,
    2.48696,     2.45777,     2.46764,     2.49411,     2.39499,
    2.36884,     2.46273,     2.45126,     2.44839,     2.42687,
    2.4039,      2.41124,     2.42265,     2.39001,     2.405,
    2.36489,     2.33581,     2.36075,     2.32135,     2.35044,
    2.27406,     2.25266,     2.29878,     2.25881,     2.21738,
    2.19765,     2.1551,      2.15557,     2.18987,     2.18647,
    2.13923,     2.19017,     2.21349,     2.20071,     2.25085,
    2.27016,     2.27825,     2.32745,     2.2304,      2.07043,
    1.82552,     1.30783,     0.705355,    0.2016,      8.31776e-06,
    0.0,         0.360215,    0.964877,    1.60834,     2.12223,
    2.43137,     2.52734,     2.51174,     2.53985,     2.49275,
    2.41492,     2.36418,     2.29577,     2.29638,     2.26549,
    2.28697,     2.23971,     2.26973,     2.27991,     2.29456,
    2.28355,     2.32379,     2.25899,     2.35119,     2.35353,
    2.36086,     2.41179,     2.38023,     2.39358,     2.44631,
    2.4606,      2.45702,     2.54991,     2.51456,     2.54426,
    2.49797,     2.52563,     2.48653,     2.49514,     2.57467,
    2.53453,     2.47125,     2.48264,     2.54841,     2.46503,
    2.4691,      2.43497,     2.46053,     2.38766,     2.45758,
    2.40203,     2.35138,     2.41814,     2.36432,     2.40103,
    2.3295,      2.30147,     2.23558,     2.23931,     2.28978,
    2.25826,     2.21018,     2.21339,     2.23438,     2.20035,
    2.18502,     2.18569,     2.24126,     2.2354,      2.28581,
    2.30225,     2.32842,     2.31414,     2.29442,     2.12739,
    1.84302,     1.35065,     0.706218,    0.207607,    8.33406e-06,
    0.0,         0.369299,    0.996483,    1.64057,     2.14397,
    2.4544,      2.53139,     2.55299,     2.52311,     2.49827,
    2.45991,     2.36929,     2.40676,     2.32937,     2.31123,
    2.26365,     2.35209,     2.3076,      2.31008,     2.30201,
    2.316,       2.34204,     2.305,       2.35567,     2.35163,
    2.43469,     2.47417,     2.46201,     2.3732,      2.46223,
    2.42779,     2.48271,     2.49308,     2.48083,     2.52658,
    2.54024,     2.62325,     2.49211,     2.5169,      2.56124,
    2.53563,     2.51858,     2.53586,     2.56423,     2.5231,
    2.45943,     2.57485,     2.60818,     2.4828,      2.46172,
    2.47811,     2.44823,     2.39775,     2.36617,     2.33799,
    2.37256,     2.33163,     2.32505,     2.33687,     2.31917,
    2.29225,     2.29188,     2.23092,     2.27365,     2.31948,
    2.23944,     2.25241,     2.27994,     2.29009,     2.35806,
    2.3451,      2.39676,     2.401,       2.37194,     2.20206,
    1.84747,     1.35869,     0.739194,    0.21423,     8.38729e-06,
    0.0,         0.374078,    1.00139,     1.72939,     2.20289,
    2.5481,      2.63646,     2.60661,     2.59352,     2.45525,
    2.46234,     2.41333,     2.35351,     2.34604,     2.29802,
    2.32946,     2.3208,      2.3477,      2.32864,     2.30788,
    2.33808,     2.36676,     2.37606,     2.44238,     2.39779,
    2.4043,      2.44366,     2.42289,     2.45816,     2.50955,
    2.55989,     2.55655,     2.53662,     2.51775,     2.55395,
    2.53714,     2.54206,     2.55886,     2.56721,     2.59607,
    2.59426,     2.59031,     2.58155,     2.5346,      2.55322,
    2.58347,     2.54207,     2.50054,     2.50825,     2.47418,
    2.50343,     2.42748,     2.42601,     2.45634,     2.39541,
    2.39296,     2.34908,     2.3276,      2.34915,     2.28047,
    2.29401,     2.30254,     2.27579,     2.30457,     2.25752,
    2.27458,     2.31908,     2.2932,      2.35032,     2.33039,
    2.43287,     2.4064,      2.41325,     2.36787,     2.22089,
    1.90703,     1.4302,      0.745795,    0.220775,    8.48299e-06,
    0.0,         0.381475,    1.04463,     1.74894,     2.27914,
    2.55171,     2.64443,     2.69008,     2.62231,     2.58048,
    2.48959,     2.45928,     2.40607,     2.41665,     2.3806,
    2.39222,     2.40125,     2.38283,     2.40378,     2.40918,
    2.34088,     2.37296,     2.4543,      2.43094,     2.46756,
    2.45206,     2.51066,     2.55379,     2.47603,     2.57705,
    2.50117,     2.49432,     2.50882,     2.57296,     2.60224,
    2.69682,     2.57354,     2.60095,     2.636,       2.60559,
    2.65842,     2.56899,     2.58781,     2.62262,     2.58969,
    2.57645,     2.51612,     2.57018,     2.57223,     2.52287,
    2.52763,     2.49922,     2.47954,     2.45956,     2.48262,
    2.4704,      2.41057,     2.37159,     2.38249,     2.38208,
    2.32152,     2.33205,     2.2773,      2.29262,     2.30201,
    2.30212,     2.32647,     2.36121,     2.36277,     2.40861,
    2.454,       2.45384,     2.44769,     2.42701,     2.25141,
    1.97983,     1.43177,     0.782049,    0.220789,    8.89692e-06,
    0.0,         0.38782,     1.07209,     1.79967,     2.26669,
    2.60086,     2.73577,     2.72283,     2.66185,     2.66026,
    2.5665,      2.50165,     2.44502,     2.43731,     2.41949,
    2.41001,     2.3964,      2.43531,     2.4172,      2.41465,
    2.42173,     2.42803,     2.49037,     2.47531,     2.38679,
    2.51542,     2.49553,     2.55108,     2.55955,     2.66397,
    2.59188,     2.63209,     2.56859,     2.63774,     2.65612,
    2.67882,     2.65424,     2.64507,     2.69713,     2.73034,
    2.67281,     2.6559,      2.66761,     2.67581,     2.63067,
    2.61643,     2.60415,     2.63498,     2.60632,     2.55468,
    2.54344,     2.5177,      2.47238,     2.55219,     2.41869,
    2.49643,     2.43668,     2.41343,     2.40266,     2.34932,
    2.3622,      2.3291,      2.32057,     2.34974,     2.33394,
    2.33869,     2.36407,     2.4224,      2.38476,     2.40581,
    2.48023,     2.52363,     2.47284,     2.46889,     2.29798,
    2.01842,     1.49419,     0.798427,    0.226889,    9.0515e-06,
    0.0,         0.397069,    1.09115,     1.81752,     2.34447,
    2.60293,     2.71268,     2.77517,     2.68329,     2.68332,
    2.58495,     2.55461,     2.54273,     2.48633,     2.45362,
    2.44852,     2.42518,     2.37625,     2.48628,     2.46645,
    2.47627,     2.54922,     2.44248,     2.51798,     2.52727,
    2.54491,     2.55429,     2.60871,     2.56763,     2.59661,
    2.62549,     2.67403,     2.68643,     2.6894,      2.72607,
    2.70908,     2.69676,     2.71805,     2.69405,     2.72642,
    2.73506,     2.68353,     2.71571,     2.695,       2.70181,
    2.65949,     2.67071,     2.61599,     2.65138,     2.59534,
    2.61097,     2.53599,     2.55925,     2.51224,     2.49413,
    2.46373,     2.48681,     2.43637,     2.4073,      2.4442,
    2.38515,     2.30773,     2.36637,     2.36355,     2.38551,
    2.38165,     2.36263,     2.43405,     2.39835,     2.43688,
    2.4988,      2.54192,     2.51563,     2.48076,     2.34858,
    2.02798,     1.49401,     0.813433,    0.23905,     8.87985e-06,
    0.0,         0.409197,    1.09598,     1.91349,     2.37604,
    2.67894,     2.76603,     2.7751,      2.80748,     2.71278,
    2.582,       2.56334,     2.51241,     2.5145,      2.47733,
    2.4698,      2.43654,     2.44659,     2.43181,     2.48816,
    2.51511,     2.48598,     2.51304,     2.54141,     2.53052,
    2.5767,      2.59551,     2.57248,     2.62622,     2.6835,
    2.60612,     2.71497,     2.68531,     2.74091,     2.70353,
    2.73358,     2.75004,     2.76357,     2.81174,     2.81805,
    2.81042,     2.77562,     2.744,       2.72547,     2.7358,
    2.6603,      2.73188,     2.64036,     2.72168,     2.60764,
    2.60677,     2.65556,     2.60542,     2.59075,     2.55258,
    2.61339,     2.52172,     2.52809,     2.48838,     2.43484,
    2.49994,     2.41266,     2.45216,     2.40383,     2.38714,
    2.3947,      2.41328,     2.42823,     2.49877,     2.5186,
    2.57417,     2.56041,     2.57191,     2.49282,     2.41149,
    2.09141,     1.58367,     0.842117,    0.248205,    9.15661e-06,
    0.0,         0.40569,     1.1448,      1.89032,     2.44062,
    2.75903,     2.85524,     2.84539,     2.81504,     2.68612,
    2.67712,     2.66004,     2.63303,     2.49282,     2.49229,
    2.54914,     2.50182,     2.53026,     2.52416,     2.56327,
    2.51092,     2.57273,     2.5169,      2.58181,     2.59165,
    2.68702,     2.63695,     2.66841,     2.73088,     2.74148,
    2.70542,     2.76354,     2.705,       2.83443,     2.77429,
    2.73729,     2.79075,     2.73796,     2.8475,      2.77943,
    2.74088,     2.80705,     2.7794,      2.77122,     2.82317,
    2.76859,     2.72888,     2.72422,     2.69567,     2.6969,
    2.66536,     2.66811,     2.64112,     2.55326,     2.62151,
    2.53079,     2.57183,     2.52963,     2.51328,     2.48612,
    2.42221,     2.42108,     2.39121,     2.45709,     2.44004,
    2.48447,     2.44699,     2.508,       2.54553,     2.57348,
    2.61198,     2.59189,     2.62624,     2.5639,      2.45176,
    2.13946,     1.58521,     0.867786,    0.240017,    9.32657e-06,
    0.0,         0.42299,     1.1777,      1.93348,     2.48642,
    2.72132,     2.95012,     2.87029,     2.82234,     2.76861,
    2.76768,     2.65701,     2.61397,     2.55896,     2.56101,
    2.55405,     2.52841,     2.49761,     2.57096,     2.50171,
    2.52278,     2.60059,     2.60046,     2.63491,     2.67735,
    2.62295,     2.64111,     2.71447,     2.76725,     2.76152,
    2.77401,     2.77706,     2.74651,     2.75477,     2.7662,
    2.80371,     2.7897,      2.81071,     2.81875,     2.84564,
    2.85606,     2.80493,     2.89129,     2.77333,     2.81936,
    2.75624,     2.78694,     2.81274,     2.75836,     2.73829,
    2.71436,     2.65013,     2.63833,     2.69782,     2.63162,
    2.60499,     2.56652,     2.61621,     2.49935,     2.54227,
    2.51416,     2.49721,     2.50561,     2.43878,     2.46348,
    2.45348,     2.52426,     2.48335,     2.55698,     2.54829,
    2.63088,     2.68835,     2.66775,     2.64502,     2.50253,
    2.19932,     1.61684,     0.899967,    0.252934,    9.59164e-06,
    0.0,         0.438312,    1.19682,     2.01052,     2.57299,
    2.84838,     2.97659,     3.0047,      2.87713,     2.8073,
    2.71331,     2.72105,     2.604,       2.62245,     2.5644,
    2.55177,     2.57784,     2.58487,     2.5782,      2.62291,
    2.57507,     2.64086,     2.65414,     2.60219,     2.67906,
    2.72712,     2.67172,     2.71041,     2.75933,     2.80761,
    2.77837,     2.82225,     2.85787,     2.85981,     2.8334,
    2.80781,     2.91698,     2.92181,     2.93427,     2.875,
    2.88451,     2.93309,     2.83994,     2.89253,     2.83631,
    2.84633,     2.79397,     2.81313,     2.79127,     2.78304,
    2.77387,     2.74997,     2.73553,     2.71113,     2.68886,
    2.61906,     2.595,       2.61668,     2.58875,     2.53521,
    2.5784,      2.52466,     2.50855,     2.50174,     2.48452,
    2.48735,     2.49878,     2.55507,     2.58046,     2.60563,
    2.70879,     2.73006,     2.72271,     2.6254,      2.52881,
    2.22894,     1.67015,     0.900527,    0.258403,    9.81069e-06,
    0.0,         0.427633,    1.19705,     2.03407,     2.60811,
    2.90873,     2.9323,      2.95045,     2.90656,     2.84466,
    2.8328,      2.72493,     2.70072,     2.621,       2.64635,
    2.62297,     2.58201,     2.66468,     2.65531,     2.6163,
    2.65781,     2.67719,     2.72044,     2.65443,     2.72402,
    2.78192,     2.78347,     2.75688,     2.73696,     2.81962,
    2.91675,     2.80005,     2.88572,     2.91258,     2.93815,
    2.91402,     2.89435,     2.94779,     2.93834,     2.8853,
    2.89077,     2.95683,     2.90558,     2.92238,     2.84332,
    2.88475,     2.8672,      2.86668,     2.83637,     2.83655,
    2.80176,     2.77148,     2.82615,     2.76047,     2.75363,
    2.71199,     2.64784,     2.61099,     2.62147,     2.62066,
    2.56607,     2.54579,     2.4845,      2.52458,     2.56887,
    2.54611,     2.59068,     2.55173,     2.59003,     2.65315,
    2.67492,     2.76295,     2.77921,     2.74501,     2.58041,
    2.31054,     1.70648,     0.935143,    0.274294,    9.80874e-06,
    0.0,         0.44956,     1.22332,     2.10564,     2.64215,
    2.92564,     3.05879,     3.07227,     2.94853,     2.91382,
    2.7717,      2.77077,     2.76504,     2.71163,     2.71586,
    2.66571,     2.66822,     2.67064,     2.6682,      2.68434,
    2.66436,     2.69491,     2.68764,     2.74887,     2.75892,
    2.80495,     2.83335,     2.83956,     2.85368,     2.84836,
    2.9341,      2.93087,     2.95225,     2.93186,     2.93218,
    2.9713,      3.00709,     2.94214,     2.94785,     2.96511,
    2.98057,     2.9381,      2.87159,     2.91049,     2.92753,
    2.96034,     2.89073,     2.91589,     2.94349,     2.86593,
    2.88933,     2.80645,     2.84866,     2.77126,     2.74512,
    2.72529,     2.74637,     2.68683,     2.67573,     2.65087,
    2.62233,     2.61502,     2.54327,     2.56505,     2.54882,
    2.63173,     2.58476,     2.57707,     2.63704,     2.68309,
    2.7309,      2.8305,      2.79389,     2.78119,     2.65438,
    2.31345,     1.75132,     0.973649,    0.269564,    9.94319e-06,
    0.0,         0.457332,    1.26954,     2.11318,     2.6407,
    3.03188,     3.08828,     3.09352,     2.98635,     2.9361,
    2.89975,     2.80785,     2.75744,     2.70577,     2.78238,
    2.69673,     2.65654,     2.70381,     2.63331,     2.69274,
    2.68549,     2.76066,     2.76271,     2.86406,     2.77762,
    2.85655,     2.83766,     2.87801,     2.80115,     2.92768,
    2.91725,     2.96109,     2.93537,     2.90473,     2.9708,
    2.96244,     3.03449,     3.0206,      3.01386,     3.01611,
    3.05013,     3.01242,     2.98918,     2.93472,     3.0193,
    2.91683,     2.97053,     2.88974,     2.9061,      2.9409,
    2.9252,      2.83147,     2.86109,     2.84019,     2.78056,
    2.76955,     2.78247,     2.72286,     2.63223,     2.61509,
    2.65919,     2.67267,     2.6449,      2.61782,     2.64729,
    2.64383,     2.61055,     2.64049,     2.64181,     2.6952,
    2.79062,     2.84119,     2.84563,     2.87138,     2.73289,
    2.38422,     1.80826,     0.97905,     0.276267,    1.02527e-05,
    0.0,         0.473176,    1.30884,     2.17405,     2.7981,
    3.06335,     3.2163,      3.11392,     3.02458,     2.94495,
    2.94149,     2.89059,     2.76187,     2.84116,     2.76097,
    2.68893,     2.71698,     2.67812,     2.70599,     2.73876,
    2.73587,     2.74528,     2.78336,     2.78999,     2.87888,
    2.8623,      2.94066,     2.9631,      2.9885,      2.99482,
    2.92095,     3.01907,     2.99322,     2.97056,     3.0248,
    3.0533,      3.07032,     3.03168,     3.08971,     3.05344,
    3.1154,      3.14805,     3.07526,     3.02039,     2.99111,
    2.95907,     3.04289,     2.96745,     2.93223,     2.95068,
    2.92379,     2.88756,     2.8661,      2.8856,      2.8399,
    2.84579,     2.80087,     2.76901,     2.72511,     2.66322,
    2.72232,     2.65539,     2.66425,     2.64522,     2.68331,
    2.64386,     2.74031,     2.70669,     2.77622,     2.76397,
    2.83838,     2.88922,     2.87865,     2.82277,     2.76514,
    2.42499,     1.84951,     0.996616,    0.278688,    1.09086e-05,
    0.0,         0.47516,     1.33859,     2.25431,     2.81795,
    3.0971,      3.12522,     3.19733,     3.11221,     2.99778,
    2.88473,     2.89084,     2.83511,     2.80504,     2.73258,
    2.82506,     2.77065,     2.6949,      2.72097,     2.80781,
    2.81739,     2.7885,      2.82375,     2.85832,     2.87223,
    2.88619,     2.8984,      2.99274,     2.95744,     3.03026,
    2.98085,     3.04974,     3.1202,      2.98432,     3.04417,
    3.05385,     3.08566,     3.1805,      3.03833,     3.07907,
    3.11207,     3.11946,     3.08945,     3.04527,     3.16811,
    3.11634,     3.06252,     3.033,       2.95152,     2.99893,
    2.9878,      2.96672,     2.92408,     2.91747,     2.86972,
    2.82291,     2.79827,     2.74075,     2.75281,     2.72632,
    2.68701,     2.66068,     2.71204,     2.70699,     2.69764,
    2.67782,     2.72886,     2.73765,     2.78453,     2.80428,
    2.87056,     2.87308,     2.92589,     2.90247,     2.80774,
    2.467,       1.87854,     1.02212,     0.294595,    1.0953e-05,
    0.0,         0.498191,    1.36258,     2.27605,     2.8634,
    3.14735,     3.18451,     3.24003,     3.14921,     3.05234,
    2.98184,     2.93417,     2.86686,     2.8375,      2.78187,
    2.83366,     2.82154,     2.7836,      2.84439,     2.8622,
    2.79878,     2.79165,     2.81466,     2.87243,     2.92247,
    2.95283,     2.96021,     3.02082,     2.92844,     3.09494,
    3.08969,     3.12113,     3.05129,     3.11976,     3.17297,
    3.10638,     3.12181,     3.14776,     3.09999,     3.18159,
    3.13085,     3.12515,     3.15387,     3.1173,      3.10636,
    3.1397,      3.11794,     3.08835,     3.05774,     3.00387,
    2.95233,     2.96836,     2.98199,     2.95603,     2.9535,
    2.87853,     2.84169,     2.87209,     2.7601,      2.77928,
    2.82932,     2.72743,     2.72548,     2.73973,     2.70927,
    2.71133,     2.7751,      2.76165,     2.80668,     2.85609,
    2.86224,     2.91327,     3.01082,     2.99785,     2.8612,
    2.51369,     1.96631,     1.07281,     0.304977,    1.11522e-05,
    0.0,         0.497338,    1.39535,     2.31806,     2.98367,
    3.22477,     3.27911,     3.23897,     3.19638,     3.13496,
    3.05954,     2.96408,     2.96824,     2.88127,     2.87295,
    2.86637,     2.82851,     2.84921,     2.86335,     2.87371,
    2.78683,     2.87991,     2.95702,     2.89666,     2.9507,
    2.98714,     3.04365,     3.05803,     3.03385,     3.03248,
    3.06026,     3.13655,     3.12507,     3.17612,     3.18011,
    3.21229,     3.13419,     3.19491,     3.1567,      3.18022,
    3.19548,     3.18408,     3.14764,     3.17349,     3.16368,
    3.15111,     3.14071,     3.09189,     3.07424,     3.02836,
    3.04098,     3.03336,     2.95395,     2.97493,     2.94008,
    2.93688,     2.87608,     2.82296,     2.85955,     2.81038,
    2.79706,     2.75927,     2.78467,     2.80362,     2.7358,
    2.80108,     2.78527,     2.82151,     2.82726,     2.88857,
    2.93916,     2.98079,     3.05228,     3.02168,     2.89866,
    2.55147,     1.99009,     1.10776,     0.314449,    1.09497e-05,
    0.0,         0.518659,    1.43015,     2.36148,     3.02595,
    3.26915,     3.38522,     3.32824,     3.28057,     3.19527,
    3.11443,     2.97518,     2.98818,     2.94601,     2.95102,
    2.88337,     2.84867,     2.90317,     2.84439,     2.9031,
    2.85244,     2.92724,     2.98828,     2.92448,     3.00788,
    3.04822,     3.02192,     3.05917,     3.06606,     3.09371,
    3.13076,     3.1934,      3.09605,     3.1764,      3.16019,
    3.27281,     3.26447,     3.25173,     3.28372,     3.26649,
    3.2695,      3.21202,     3.24092,     3.2242,      3.19462,
    3.14204,     3.14767,     3.1891,      3.09421,     3.1311,
    3.13933,     3.03922,     3.04235,     3.04528,     2.96552,
    2.98651,     2.87765,     2.94553,     2.87715,     2.8862,
    2.85101,     2.85003,     2.82834,     2.847,       2.79335,
    2.81995,     2.79054,     2.83191,     2.88135,     2.85774,
    2.95526,     3.04214,     3.01083,     3.07215,     2.98854,
    2.62535,     1.99423,     1.12323,     0.317586,    1.14565e-05,
    0.0,         0.529475,    1.40565,     2.425,       3.04093,
    3.37122,     3.42382,     3.43751,     3.28331,     3.20315,
    3.09983,     3.05402,     3.01875,     2.99358,     2.95741,
    2.92608,     2.87224,     2.89997,     2.92604,     2.9185,
    3.01046,     3.02757,     3.02908,     3.02679,     3.05439,
    3.11969,     3.10874,     3.09654,     3.16831,     3.14318,
    3.21862,     3.14883,     3.20682,     3.25746,     3.25077,
    3.31788,     3.32659,     3.27802,     3.25036,     3.22586,
    3.27363,     3.18286,     3.25991,     3.29665,     3.26621,
    3.254,       3.26021,     3.22018,     3.15927,     3.11154,
    3.13383,     3.09443,     3.10614,     3.13616,     3.04187,
    2.99687,     3.04149,     2.94034,     2.9279,      2.85859,
    2.86859,     2.86449,     2.85654,     2.8356,      2.84151,
    2.79282,     2.79289,     2.88274,     2.88702,     2.93643,
    2.95882,     3.07971,     3.14608,     3.13103,     3.01142,
    2.74905,     2.0195,      1.1341,      0.332191,    1.21918e-05,
    0.0,         0.533594,    1.4907,      2.51278,     3.10912,
    3.38753,     3.47657,     3.45761,     3.37032,     3.35454,
    3.14195,     3.14351,     3.04902,     3.02067,     2.96125,
    2.92118,     2.98634,     2.95381,     2.95385,     2.9321,
    2.98149,     2.99462,     3.04737,     3.11674,     3.05343,
    3.17896,     3.192,       3.14632,     3.14763,     3.16008,
    3.27836,     3.25629,     3.22824,     3.32345,     3.28488,
    3.33584,     3.2469,      3.39644,     3.31454,     3.33921,
    3.28618,     3.29329,     3.34514,     3.31181,     3.34102,
    3.27406,     3.27863,     3.32518,     3.26872,     3.25067,
    3.22452,     3.19417,     3.14342,     3.15859,     3.06743,
    3.08636,     2.96285,     3.0267,      2.9693,      2.91738,
    2.93927,     2.96559,     2.86788,     2.84688,     2.89646,
    2.91918,     2.92141,     2.90718,     2.95389,     3.01066,
    3.03636,     3.15674,     3.21005,     3.20046,     3.06318,
    2.75644,     2.13406,     1.16903,     0.340639,    1.1991e-05,
    0.0,         0.540121,    1.53406,     2.56943,     3.20892,
    3.49048,     3.48568,     3.50658,     3.40207,     3.23757,
    3.22191,     3.19067,     3.07404,     3.05538,     2.9737,
    2.93443,     3.07037,     2.9877,      2.99432,     3.03316,
    3.06114,     3.01346,     3.10209,     3.0914,      3.16728,
    3.18354,     3.1498,      3.24008,     3.195,       3.27127,
    3.30051,     3.32427,     3.32903,     3.27456,     3.34179,
    3.40753,     3.37187,     3.4126,      3.36778,     3.36247,
    3.34635,     3.33806,     3.4089,      3.3356,      3.39215,
    3.35512,     3.40106,     3.24115,     3.34083,     3.27038,
    3.27606,     3.16454,     3.19116,     3.11028,     3.06013,
    3.10964,     3.03418,     3.07978,     3.02098,     2.95425,
    2.96139,     2.86507,     2.90726,     2.8856,      2.9101,
    2.95291,     2.92775,     2.99516,     2.98899,     3.08088,
    3.13037,     3.11312,     3.20792,     3.24172,     3.06885,
    2.79934,     2.16303,     1.21727,     0.33674,     1.23652e-05,
    0.0,         0.557871,    1.56641,     2.55964,     3.26744,
    3.48423,     3.58754,     3.50403,     3.51405,     3.37108,
    3.27641,     3.19439,     3.12703,     3.16809,     3.05679,
    3.00767,     3.04609,     3.02903,     3.08792,     3.06584,
    3.00099,     3.16254,     3.13891,     3.15576,     3.2025,
    3.23824,     3.24659,     3.17426,     3.27802,     3.26477,
    3.34738,     3.34075,     3.32272,     3.30995,     3.42882,
    3.40156,     3.44863,     3.37969,     3.33464,     3.38908,
    3.49926,     3.40622,     3.39118,     3.34382,     3.4077,
    3.42221,     3.39011,     3.28154,     3.27593,     3.35675,
    3.2454,      3.34553,     3.18695,     3.21597,     3.19567,
    3.16158,     3.04271,     3.12467,     3.03135,     2.97055,
    2.99302,     3.01645,     2.93628,     2.96856,     3.0349,
    2.91265,     2.955,       3.00638,     3.0276,      3.0997,
    3.0687,      3.21684,     3.24377,     3.26104,     3.15144,
    2.83993,     2.1818,      1.25374,     0.351306,    1.30363e-05,
    0.0,         0.604357,    1.71212,     2.81282,     3.45604,
    3.8344,      3.84568,     3.71094,     3.68639,     3.51991,
    3.46131,     3.32541,     3.23232,     3.20447,     3.28142,
    3.1923,      3.2933,      3.19158,     3.26486,     3.19515,
    3.17732,     3.3528,      3.30359,     3.29931,     3.31136,
    3.42498,     3.40625,     3.40216,     3.43097,     3.45676,
    3.38122,     3.5518,      3.48394,     3.53268,     3.64744,
    3.61043,     3.63628,     3.62503,     3.64416,     3.64455,
    3.57914,     3.65757,     3.64653,     3.65908,     3.59474,
    3.57076,     3.59567,     3.52645,     3.50454,     3.47748,
    3.54887,     3.49076,     3.35782,     3.42023,     3.37396,
    3.27126,     3.31025,     3.2715,      3.25693,     3.18979,
    3.21552,     3.18711,     3.11415,     3.18226,     3.0642,
    3.09478,     3.10455,     3.17225,     3.29579,     3.28762,
    3.30276,     3.43311,     3.47992,     3.49571,     3.47141,
    3.13491,     2.48891,     1.35683,     0.389824,    1.32143e-05,
    0.0,         0.647344,    1.85189,     2.95329,     3.63544,
    3.99714,     4.03576,     3.9966,      3.82459,     3.68549,
    3.56458,     3.51326,     3.43635,     3.3076,      3.3623,
    3.34966,     3.30174,     3.31695,     3.30566,     3.3757,
    3.40237,     3.42271,     3.45985,     3.52493,     3.50154,
    3.51196,     3.62752,     3.56633,     3.6196,      3.60451,
    3.6707,      3.78307,     3.69663,     3.82418,     3.78075,
    3.6945,      3.72627,     3.75795,     3.67188,     3.82121,
    3.80773,     3.66792,     3.79413,     3.79884,     3.68078,
    3.64935,     3.75081,     3.79412,     3.6864,      3.68557,
    3.73009,     3.47961,     3.48651,     3.59025,     3.45918,
    3.35523,     3.45779,     3.38868,     3.35462,     3.30842,
    3.24199,     3.27931,     3.28327,     3.20804,     3.23217,
    3.22827,     3.19032,     3.30028,     3.37508,     3.36899,
    3.44085,     3.54079,     3.63692,     3.67073,     3.56687,
    3.2648,      2.65722,     1.49611,     0.421035,    1.49595e-05,
    0.0,         0.686518,    1.9556,      3.19649,     3.90112,
    4.27178,     4.26377,     4.18815,     3.99555,     3.8214,
    3.82581,     3.63827,     3.51522,     3.41449,     3.45213,
    3.48958,     3.49403,     3.48944,     3.42547,     3.45097,
    3.45976,     3.59964,     3.5217,      3.52483,     3.65797,
    3.60078,     3.71191,     3.68662,     3.70207,     3.87102,
    3.72266,     3.79182,     3.94666,     3.88927,     3.85737,
    3.8747,      3.83011,     3.91985,     3.93129,     3.86668,
    3.97046,     3.79213,     3.86587,     3.93243,     3.91871,
    3.92976,     3.85207,     3.83299,     3.83919,     3.84637,
    3.73015,     3.74203,     3.6742,      3.57549,     3.64098,
    3.5303,      3.49754,     3.47068,     3.52,        3.45963,
    3.45238,     3.39281,     3.36,        3.34216,     3.28767,
    3.35334,     3.43542,     3.40183,     3.46303,     3.54665,
    3.67649,     3.68128,     3.79398,     3.74722,     3.84085,
    3.51627,     2.79117,     1.60806,     0.463657,    1.51417e-05,
    0.0,         0.725491,    2.04703,     3.34831,     4.16619,
    4.41222,     4.44894,     4.20694,     4.1347,      4.12311,
    3.91843,     3.7656,      3.755,       3.64221,     3.62398,
    3.59435,     3.57004,     3.59271,     3.60256,     3.73756,
    3.67631,     3.66512,     3.5963,      3.74187,     3.74921,
    3.79954,     3.81268,     3.95272,     3.82472,     3.97673,
    4.05446,     3.99422,     3.97924,     4.04294,     3.97048,
    4.07953,     4.08878,     4.03575,     4.09235,     4.12337,
    4.1264,      4.09023,     4.10153,     4.12084,     4.03937,
    4.03914,     4.01168,     4.00798,     3.9302,      3.94318,
    3.97271,     3.91486,     3.79974,     3.78102,     3.7344,
    3.70069,     3.67938,     3.71244,     3.62655,     3.51064,
    3.53604,     3.50572,     3.58907,     3.44293,     3.44704,
    3.49792,     3.45588,     3.56755,     3.52349,     3.62375,
    3.81818,     3.89029,     4.02173,     4.04109,     3.9661,
    3.67692,     2.9437,      1.70112,     0.489098,    1.64416e-05,
    0.0,         0.786692,    2.22462,     3.56495,     4.31186,
    4.6257,      4.59095,     4.58637,     4.26257,     4.15105,
    4.00933,     3.96424,     3.8338,      3.73295,     3.74339,
    3.78264,     3.59396,     3.65595,     3.76988,     3.83462,
    3.81883,     3.81815,     3.84176,     3.8392,      3.94119,
    3.99547,     4.04522,     4.10968,     4.05204,     4.06723,
    4.02576,     4.12218,     4.10725,     4.14406,     4.2158,
    4.15984,     4.25002,     4.2252,      4.26814,     4.21972,
    4.21492,     4.22377,     4.20332,     4.24949,     4.19003,
    4.22929,     4.1548,      4.25982,     4.16607,     4.11655,
    4.02432,     4.02591,     4.05173,     3.99521,     3.90738,
    3.80262,     3.81059,     3.76237,     3.73437,     3.7353,
    3.72518,     3.68714,     3.6351,      3.63219,     3.63681,
    3.57718,     3.62106,     3.71456,     3.7159,      3.84021,
    3.92181,     3.96122,     4.1239,      4.20821,     4.25128,
    3.95374,     3.17883,     1.8787,      0.527508,    1.77333e-05,
    0.0,         0.831477,    2.3526,      3.81394,     4.54895,
    4.74712,     4.6932,      4.8099,      4.45476,     4.37244,
    4.13954,     4.09627,     4.0367,      3.91069,     3.91393,
    3.85648,     3.79444,     3.9071,      3.8556,      3.88193,
    3.89923,     3.87407,     3.85415,     4.02813,     4.11986,
    4.16586,     4.13688,     4.10372,     4.30477,     4.27052,
    4.19354,     4.35966,     4.30534,     4.25496,     4.32511,
    4.43786,     4.32879,     4.35893,     4.38603,     4.38243,
    4.42235,     4.49827,     4.44837,     4.46801,     4.37139,
    4.36368,     4.29837,     4.29403,     4.33262,     4.21283,
    4.09861,     4.19559,     4.17052,     4.01714,     4.0378,
    4.092,       3.9767,      3.88113,     3.84055,     3.85625,
    3.88358,     3.79308,     3.72799,     3.71514,     3.75632,
    3.75402,     3.74665,     3.74867,     3.90896,     4.03814,
    4.0699,      4.20155,     4.40505,     4.46602,     4.38226,
    4.19828,     3.36615,     2.01983,     0.574443,    1.86709e-05,
    0.0,         0.879343,    2.52703,     3.98321,     4.7805,
    5.09751,     4.9248,      4.89469,     4.67236,     4.49074,
    4.32638,     4.22396,     4.16127,     4.18026,     4.09731,
    3.99205,     3.93721,     4.0431,      3.99863,     4.03822,
    4.15226,     4.07435,     4.14648,     4.16723,     4.23638,
    4.28451,     4.25035,     4.29942,     4.3395,      4.43102,
    4.38165,     4.45196,     4.56576,     4.4602,      4.53714,
    4.52125,     4.51598,     4.55193,     4.55219,     4.6295,
    4.50048,     4.63708,     4.52165,     4.51904,     4.57224,
    4.40488,     4.53901,     4.45836,     4.45285,     4.38634,
    4.28338,     4.32831,     4.29445,     4.16498,     4.2553,
    4.08941,     4.11222,     4.02839,     4.04959,     3.89958,
    3.98472,     3.95193,     3.83547,     3.82295,     3.85098,
    3.81718,     3.86723,     4.01982,     4.04662,     4.14651,
    4.25141,     4.41829,     4.55774,     4.64854,     4.56755,
    4.39678,     3.58699,     2.1251,      0.614223,    1.99549e-05,
    0.0,         0.93101,     2.70359,     4.27459,     5.06889,
    5.24287,     5.22028,     5.03824,     4.77659,     4.75012,
    4.46615,     4.40546,     4.27952,     4.23833,     4.11496,
    4.12851,     4.19518,     4.16829,     4.14685,     4.12096,
    4.2783,      4.19952,     4.33312,     4.30353,     4.43849,
    4.30939,     4.45278,     4.42674,     4.59121,     4.58993,
    4.55791,     4.59748,     4.71353,     4.6507,      4.71542,
    4.6671,      4.77097,     4.69393,     4.74334,     4.74425,
    4.81844,     4.79128,     4.73296,     4.64289,     4.73106,
    4.70544,     4.68201,     4.74097,     4.63253,     4.60712,
    4.59949,     4.65244,     4.54978,     4.38411,     4.3641,
    4.27886,     4.28557,     4.29165,     4.19747,     4.13942,
    4.11567,     4.00382,     4.03094,     4.02818,     3.98476,
    4.02225,     4.09077,     4.10702,     4.14384,     4.2857,
    4.40699,     4.60487,     4.75539,     4.84122,     4.86071,
    4.71142,     3.81365,     2.3118,      0.685079,    2.14834e-05,
    0.0,         1.00014,     2.90566,     4.53369,     5.3669,
    5.52065,     5.45949,     5.28409,     4.91585,     4.88664,
    4.76911,     4.53282,     4.4496,      4.34906,     4.37551,
    4.23759,     4.28621,     4.34176,     4.20688,     4.39065,
    4.37112,     4.36272,     4.38856,     4.50339,     4.45552,
    4.58371,     4.6375,      4.65058,     4.67329,     4.7081,
    4.79209,     4.80459,     4.88827,     4.79576,     4.81673,
    4.93473,     4.84695,     4.90052,     4.96651,     4.84896,
    4.91017,     4.84401,     4.93609,     4.98912,     4.89858,
    4.86894,     4.84683,     4.8716,      4.85049,     4.73135,
    4.71897,     4.60505,     4.64061,     4.55671,     4.54229,
    4.48979,     4.37264,     4.40258,     4.28308,     4.31145,
    4.22782,     4.13996,     4.13815,     4.19197,     4.14279,
    4.24202,     4.25647,     4.33692,     4.32246,     4.43419,
    4.54616,     4.8264,      4.867,       5.10349,     5.19372,
    4.87314,     4.03792,     2.50504,     0.725698,    2.31606e-05,
    0.0,         1.06144,     3.07578,     4.73853,     5.66127,
    5.75286,     5.72844,     5.54167,     5.21145,     5.05072,
    4.8205,      4.73997,     4.64291,     4.50971,     4.38412,
    4.37878,     4.42574,     4.36688,     4.48863,     4.54752,
    4.51541,     4.56458,     4.6986,      4.66423,     4.56661,
    4.77258,     4.73774,     4.71529,     4.75308,     4.71834,
    4.95597,     5.01407,     5.01946,     5.02318,     5.04911,
    5.00318,     5.05321,     5.09099,     5.10117,     5.12121,
    5.01148,     5.2045,      4.9449,      5.07046,     5.13326,
    5.03342,     5.12654,     4.9338,      4.89261,     4.98518,
    4.88726,     4.78659,     4.80469,     4.7146,      4.64882,
    4.56579,     4.62833,     4.5846,      4.50919,     4.49236,
    4.4092,      4.33965,     4.33066,     4.32977,     4.34627,
    4.34305,     4.41907,     4.45186,     4.40345,     4.55984,
    4.70102,     4.94103,     5.12519,     5.33667,     5.34555,
    5.19092,     4.29113,     2.61821,     0.78243,     2.49371e-05,
    0.0,         1.1207,      3.26186,     4.97841,     5.91146,
    5.96395,     5.99262,     5.69097,     5.42732,     5.23568,
    5.00726,     4.88288,     4.79027,     4.63178,     4.66572,
    4.54854,     4.63094,     4.61169,     4.50849,     4.6355,
    4.77134,     4.78597,     4.71957,     4.70181,     4.82424,
    4.92351,     4.94543,     5.06369,     5.02936,     5.10261,
    5.17844,     5.14674,     5.15956,     5.16041,     5.28227,
    5.26254,     5.3079,      5.32971,     5.2501,      5.27267,
    5.23192,     5.24817,     5.32889,     5.28862,     5.09671,
    5.19661,     5.28237,     5.18688,     4.99555,     5.10443,
    5.07498,     4.96476,     4.91374,     4.88096,     4.83937,
    4.86116,     4.68757,     4.6748,      4.66568,     4.63457,
    4.59745,     4.47422,     4.4122,      4.45164,     4.47636,
    4.50062,     4.59048,     4.62027,     4.72643,     4.77476,
    5.01262,     5.14592,     5.27445,     5.5597,      5.65556,
    5.35764,     4.58328,     2.81194,     0.812182,    2.56811e-05,
    0.0,         1.19899,     3.44183,     5.32474,     6.16502,
    6.40013,     6.19812,     5.95445,     5.53107,     5.46364,
    5.29144,     5.04635,     4.87244,     4.98734,     4.75528,
    4.75189,     4.60686,     4.70245,     4.84076,     4.79688,
    4.82297,     4.82801,     4.93063,     4.89293,     4.90351,
    5.10524,     5.05553,     5.24157,     5.2539,      5.2537,
    5.32994,     5.34462,     5.42297,     5.441,       5.34,
    5.36524,     5.44932,     5.41193,     5.52497,     5.54245,
    5.41601,     5.49485,     5.46276,     5.47942,     5.55175,
    5.35564,     5.33395,     5.44269,     5.31783,     5.32389,
    5.15877,     5.11941,     5.23018,     5.28922,     5.01024,
    5.07946,     4.95518,     4.84982,     4.79134,     4.81674,
    4.69527,     4.642,       4.67695,     4.66429,     4.58722,
    4.62185,     4.71378,     4.79055,     4.7772,      4.99668,
    5.08345,     5.28859,     5.54335,     5.8231,      5.88825,
    5.73778,     4.84742,     3.03368,     0.894654,    2.71939e-05,
    0.0,         1.28099,     3.71392,     5.50857,     6.50011,
    6.71145,     6.5413,      6.14576,     5.84188,     5.53431,
    5.36513,     5.15456,     5.17667,     4.92073,     5.03522,
    4.81404,     4.85068,     4.83092,     4.9248,      4.9042,
    5.01747,     5.05011,     5.0806,      5.11526,     5.17946,
    5.1406,      5.38312,     5.20351,     5.22561,     5.47514,
    5.33271,     5.45296,     5.56899,     5.58741,     5.5676,
    5.60577,     5.6686,      5.66355,     5.59076,     5.65064,
    5.57972,     5.57525,     5.66482,     5.47922,     5.66406,
    5.6387,      5.52088,     5.43079,     5.5882,      5.49585,
    5.41709,     5.42409,     5.29289,     5.23096,     5.20149,
    5.06353,     5.13738,     5.04787,     5.03168,     4.9334,
    4.82221,     4.81617,     4.84412,     4.79031,     4.76442,
    4.80311,     4.79026,     4.86085,     5.03468,     5.1736,
    5.33372,     5.50295,     5.69215,     6.0042,      6.16008,
    5.9584,      5.11368,     3.15608,     0.958218,    2.97715e-05,
    0.0,         1.36021,     3.9179,      5.87793,     6.87086,
    6.9034,      6.55107,     6.3771,      6.20056,     5.81266,
    5.5755,      5.35974,     5.23133,     5.24868,     5.05018,
    5.10365,     4.99318,     5.05338,     5.12608,     5.1287,
    5.10058,     5.11981,     5.25002,     5.3491,      5.29199,
    5.33309,     5.41497,     5.52483,     5.55633,     5.65872,
    5.63093,     5.57097,     5.73746,     5.70865,     5.73893,
    5.63446,     5.74884,     5.92487,     5.89764,     5.90334,
    5.68055,     5.86968,     5.90069,     5.7956,      5.845,
    5.80433,     5.75916,     5.61902,     5.67279,     5.64755,
    5.74695,     5.51541,     5.53004,     5.45926,     5.41171,
    5.29327,     5.30322,     5.22397,     5.18094,     5.10335,
    5.08251,     4.97789,     5.05478,     4.89651,     4.94829,
    5.02875,     5.03289,     5.04346,     5.09443,     5.26482,
    5.49253,     5.85664,     6.06673,     6.28158,     6.45129,
    6.29242,     5.51231,     3.39219,     1.01111,     3.08795e-05,
    0.0,         1.44949,     4.16539,     6.10878,     7.08221,
    7.23284,     6.95444,     6.6842,      6.30925,     6.10638,
    5.86407,     5.64705,     5.50469,     5.21531,     5.31149,
    5.24946,     5.25809,     5.12708,     5.26319,     5.30996,
    5.19824,     5.37781,     5.42839,     5.46066,     5.61342,
    5.6219,      5.70735,     5.59811,     5.71785,     5.77072,
    5.9182,      5.89182,     5.79414,     5.90634,     6.00372,
    5.96034,     5.91665,     5.94274,     6.09238,     6.03968,
    5.89121,     6.02507,     5.97005,     6.09602,     5.9939,
    5.94939,     5.95377,     5.90913,     5.80414,     5.8011,
    5.8314,      5.86505,     5.67554,     5.60867,     5.63974,
    5.59697,     5.55001,     5.39513,     5.28853,     5.19556,
    5.26193,     5.21257,     5.10938,     5.06655,     5.19672,
    5.13083,     5.15791,     5.20848,     5.33284,     5.47699,
    5.60431,     5.84783,     6.22527,     6.31222,     6.69407,
    6.67186,     5.6232,      3.70286,     1.11819,     3.32857e-05,
    0.0,         1.56744,     4.45754,     6.51239,     7.48362,
    7.51617,     7.22557,     6.9749,      6.65164,     6.20364,
    5.96078,     5.76917,     5.64591,     5.45943,     5.54685,
    5.42538,     5.3528,      5.40967,     5.41281,     5.58574,
    5.55513,     5.54992,     5.51259,     5.59835,     5.74623,
    5.7258,      5.87629,     5.85527,     5.90669,     5.98196,
    5.88969,     6.16009,     6.02634,     6.13736,     6.25544,
    6.12489,     6.31766,     6.34206,     6.27616,     6.25689,
    6.3063,      6.08027,     6.20576,     6.14283,     6.12788,
    6.12476,     6.13064,     6.16303,     6.03735,     6.05681,
    6.01077,     5.93152,     5.91149,     5.94425,     5.68391,
    5.69269,     5.60945,     5.6123,      5.48415,     5.33568,
    5.27647,     5.25442,     5.31606,     5.23766,     5.2469,
    5.32607,     5.31104,     5.43122,     5.46412,     5.77338,
    5.82079,     6.23753,     6.5472,      6.82344,     7.09677,
    6.98685,     6.02742,     3.82852,     1.16228,     3.67595e-05,
    0.0,         1.64696,     4.65607,     6.91312,     7.90687,
    7.90051,     7.58714,     7.20489,     6.81512,     6.51667,
    6.22512,     5.94857,     5.8288,      5.63209,     5.69512,
    5.62513,     5.49614,     5.64524,     5.64624,     5.61525,
    5.64684,     5.80236,     5.80071,     5.84179,     5.76084,
    5.88574,     6.02476,     6.06547,     6.11174,     6.17688,
    6.22897,     6.32074,     6.32504,     6.27494,     6.40287,
    6.30141,     6.4491,      6.22056,     6.48008,     6.45813,
    6.47853,     6.43929,     6.49441,     6.54793,     6.38266,
    6.29846,     6.29877,     6.20786,     6.19382,     6.2099,
    6.16477,     6.05216,     6.08436,     5.98813,     5.85591,
    5.91202,     5.80553,     5.68352,     5.65018,     5.59849,
    5.55744,     5.57568,     5.43591,     5.45937,     5.421,
    5.46888,     5.47265,     5.58424,     5.66035,     5.96132,
    6.01192,     6.50542,     6.64715,     7.16656,     7.26136,
    7.14919,     6.32659,     4.09223,     1.26619,     3.82926e-05,
    0.0,         1.70605,     4.84182,     7.17812,     8.23015,
    8.14436,     7.90736,     7.48375,     7.03437,     6.5898,
    6.37329,     6.24228,     6.04301,     5.91783,     5.93192,
    5.76172,     5.85859,     5.73751,     5.85966,     5.71611,
    5.83103,     5.82683,     6.00834,     5.9071,      6.06862,
    6.17806,     6.2503,      6.32948,     6.2897,      6.28369,
    6.40093,     6.43896,     6.48078,     6.54134,     6.70957,
    6.46013,     6.5859,      6.56501,     6.52993,     6.65369,
    6.71081,     6.63237,     6.646,       6.59848,     6.68446,
    6.55873,     6.63946,     6.65003,     6.46528,     6.40621,
    6.42481,     6.48915,     6.16417,     6.22124,     6.03828,
    5.92572,     5.86756,     5.98075,     5.85708,     5.76558,
    5.71114,     5.70018,     5.69609,     5.64391,     5.61393,
    5.60885,     5.57122,     5.68553,     5.90613,     6.01233,
    6.41368,     6.638,       6.89169,     7.55547,     7.88858,
    7.68547,     6.73081,     4.44904,     1.38083,     3.94785e-05,
    0.0,         1.8145,      5.27328,     7.63586,     8.5807,
    8.5299,      8.13316,     7.77626,     7.32681,     6.90294,
    6.45958,     6.40458,     6.19111,     6.17134,     6.02855,
    5.8417,      5.95334,     6.00867,     6.07616,     6.10335,
    6.0743,      6.09819,     6.13789,     6.16189,     6.35546,
    6.35887,     6.30143,     6.45038,     6.63515,     6.47879,
    6.55284,     6.77324,     6.70944,     6.71883,     6.89692,
    6.71183,     6.84994,     6.75156,     6.87708,     6.81545,
    6.8152,      6.9057,      6.85768,     6.79562,     6.77431,
    6.61299,     6.86343,     6.61822,     6.58047,     6.66436,
    6.7173,      6.53397,     6.47316,     6.54023,     6.42156,
    6.22565,     6.31824,     6.04857,     6.06742,     6.0175,
    6.06248,     5.87089,     5.65844,     5.76822,     5.84672,
    5.8093,      5.78465,     5.94549,     6.1814,      6.34391,
    6.47628,     6.916,       7.24606,     7.46203,     7.90075,
    7.96562,     7.20594,     4.66227,     1.48772,     4.29113e-05,
    0.0,         1.98327,     5.45979,     8.00683,     8.92057,
    8.9528,      8.42088,     7.83145,     7.54511,     7.07937,
    6.67806,     6.53103,     6.45838,     6.29472,     6.30848,
    6.22267,     6.1313,      6.19612,     6.30219,     6.20731,
    6.14371,     6.41782,     6.21889,     6.44792,     6.51906,
    6.47163,     6.71509,     6.53676,     6.61634,     6.73505,
    6.81605,     6.88887,     6.85523,     6.90836,     7.04961,
    6.96098,     7.08055,     6.97351,     7.09759,     7.17164,
    7.01255,     7.14157,     6.98437,     7.0022,      7.12782,
    6.98556,     6.9341,      6.94765,     6.97832,     6.62109,
    6.88234,     6.71937,     6.59686,     6.66414,     6.48906,
    6.44314,     6.36288,     6.35905,     6.18013,     6.00431,
    6.09092,     6.00968,     6.05085,     5.96031,     6.06066,
    5.95321,     5.99987,     6.19901,     6.35788,     6.5764,
    6.70541,     7.12295,     7.41352,     7.87616,     8.32157,
    8.38238,     7.54052,     5.00569,     1.53277,     4.69103e-05,
    0.0,         2.03699,     5.78609,     8.39815,     9.27899,
    9.13784,     8.76435,     8.14776,     7.71208,     7.20116,
    7.03514,     6.6914,      6.45712,     6.36509,     6.38444,
    6.34149,     6.25358,     6.36971,     6.44609,     6.29292,
    6.41054,     6.5087,      6.48495,     6.70952,     6.69406,
    6.82038,     6.67755,     6.76671,     6.98554,     6.98416,
    6.83966,     6.98217,     7.2926,      7.01518,     7.14755,
    7.1394,      7.25287,     7.27775,     7.33363,     7.19422,
    7.33161,     7.32574,     7.15907,     7.36217,     7.35102,
    7.15522,     7.09624,     7.14951,     7.01035,     6.829,
    7.03826,     7.06772,     6.94904,     6.72192,     6.75921,
    6.77848,     6.46157,     6.49576,     6.47939,     6.27591,
    6.24801,     6.19017,     6.04354,     6.18986,     6.16509,
    6.29547,     6.26166,     6.43935,     6.43358,     6.66602,
    7.06922,     7.38589,     7.73541,     8.16244,     8.66652,
    8.80759,     7.74924,     5.31409,     1.67412,     4.94689e-05,
    0.0,         2.22718,     6.07801,     8.82133,     9.73352,
    9.57867,     8.99564,     8.46225,     7.96783,     7.71506,
    7.28106,     6.98818,     6.82303,     6.71089,     6.49311,
    6.45221,     6.52601,     6.52313,     6.50508,     6.46449,
    6.64038,     6.54052,     6.8194,      6.73181,     6.82633,
    6.83193,     7.05063,     6.90948,     7.0301,      7.24818,
    7.27911,     7.31795,     7.22843,     7.46309,     7.56681,
    7.52792,     7.47434,     7.62376,     7.42018,     7.47153,
    7.41383,     7.50848,     7.48913,     7.49082,     7.42611,
    7.34623,     7.51411,     7.22187,     7.37644,     7.26834,
    7.13514,     7.31076,     7.00669,     6.70003,     7.03539,
    6.76585,     6.7143,      6.62284,     6.66909,     6.52499,
    6.40928,     6.35263,     6.42675,     6.32058,     6.33725,
    6.38393,     6.44524,     6.56958,     6.63656,     6.83806,
    7.02088,     7.61331,     8.02571,     8.54031,     8.9396,
    9.09801,     8.32282,     5.62777,     1.82178,     5.25467e-05,
    0.0,         2.34157,     6.51713,     9.32927,     10.2146,
    9.83847,     9.4828,      8.85278,     8.27463,     7.87171,
    7.50528,     7.17852,     6.96635,     6.83537,     6.79082,
    6.7126,      6.56798,     6.69879,     6.6026,      6.78869,
    6.70192,     6.97291,     7.02746,     7.16055,     7.0391,
    7.21371,     7.16201,     7.17117,     7.37607,     7.39016,
    7.32898,     7.41521,     7.52108,     7.75153,     7.58491,
    7.58471,     7.70592,     7.5498,      7.80368,     7.73844,
    7.594,       7.68071,     7.77686,     7.78222,     7.72567,
    7.51573,     7.6551,      7.59397,     7.32746,     7.41956,
    7.41703,     7.26124,     7.15249,     7.09244,     7.13347,
    7.02639,     7.04004,     6.94881,     6.87847,     6.85978,
    6.68717,     6.58473,     6.64186,     6.58109,     6.53271,
    6.56202,     6.53553,     6.58875,     6.91166,     7.06385,
    7.52907,     7.81417,     8.40743,     8.93426,     9.53362,
    9.46272,     8.60958,     5.899,       1.92248,     5.58034e-05,
    0.0,         2.45892,     6.92956,     9.77577,     10.5032,
    10.3269,     9.72669,     9.16841,     8.42095,     7.97328,
    7.68777,     7.39094,     7.23867,     6.95241,     6.92889,
    6.74019,     6.90387,     6.76042,     6.93869,     6.98002,
    6.92678,     7.11462,     7.12577,     7.27184,     7.14329,
    7.38642,     7.53399,     7.4984,      7.70763,     7.52499,
    7.63468,     7.61849,     7.76269,     8.04293,     7.81877,
    7.80723,     7.80982,     7.93157,     7.86849,     7.96399,
    7.98055,     7.96757,     8.13439,     8.06146,     7.78176,
    7.93604,     7.71201,     7.80121,     7.93297,     7.75598,
    7.63807,     7.58368,     7.54815,     7.39871,     7.33661,
    7.15976,     7.21586,     7.21221,     7.0087,      6.97504,
    6.90187,     6.72293,     6.74016,     6.70733,     6.65826,
    6.74499,     6.83077,     6.98309,     7.2161,      7.32008,
    7.78704,     8.053,       8.64932,     9.19486,     9.69512,
    9.92362,     9.00813,     6.3185,      2.07416,     5.85275e-05,
    0.0,         2.56806,     7.35919,     10.2081,     10.8546,
    10.7076,     9.96573,     9.3947,      8.67678,     8.26297,
    7.94441,     7.7745,      7.48951,     7.23959,     7.32782,
    7.10012,     7.09272,     6.96422,     7.15387,     7.10708,
    7.35192,     7.36775,     7.30818,     7.35921,     7.54514,
    7.36077,     7.4122,      7.78478,     8.07904,     7.85544,
    7.70699,     7.73479,     8.03172,     7.76851,     8.01699,
    7.94898,     8.08648,     8.09889,     8.04912,     8.25072,
    8.3037,      8.23203,     8.22052,     8.22886,     8.11845,
    8.25644,     8.11496,     7.98376,     8.04182,     7.95969,
    7.67551,     7.73598,     7.6466,      7.72073,     7.4624,
    7.5524,      7.32956,     7.1515,      7.15937,     7.14608,
    7.11449,     6.97315,     6.89439,     6.88405,     6.83617,
    7.06999,     6.81027,     7.12404,     7.30327,     7.56112,
    7.77607,     8.3373,      8.83279,     9.56383,     10.133,
    10.3706,     9.6011,      6.62418,     2.15072,     6.19724e-05,
    0.0,         2.73309,     7.82624,     10.7353,     11.3094,
    11.308,      10.3525,     9.57218,     9.23244,     8.50703,
    8.31936,     7.79954,     7.60468,     7.51345,     7.24353,
    7.2975,      7.24149,     7.34045,     7.23615,     7.30026,
    7.40768,     7.54443,     7.47328,     7.63882,     7.61865,
    7.68759,     7.94401,     7.78465,     8.05663,     8.19031,
    8.1658,      8.20874,     8.3085,      8.38577,     8.40246,
    8.24753,     8.24761,     8.33075,     8.4627,      8.60674,
    8.3979,      8.34003,     8.41087,     8.31303,     8.51135,
    8.33043,     8.3262,      8.12992,     8.12126,     8.08039,
    8.17607,     8.0316,      8.10751,     7.83876,     7.93964,
    7.62546,     7.5626,      7.49195,     7.57904,     7.39603,
    7.23882,     7.23799,     7.17981,     7.12886,     7.08475,
    7.14807,     7.21402,     7.41726,     7.5426,      7.79412,
    8.28856,     8.66613,     9.43422,     9.95477,     10.5434,
    10.8844,     9.97501,     7.01191,     2.3087,      6.55509e-05,
    0.0,         2.96882,     8.15002,     11.163,      11.8889,
    11.5421,     10.7123,     10.0862,     9.40669,     8.9818,
    8.30834,     7.98941,     7.799,       7.62362,     7.59493,
    7.42576,     7.60017,     7.4926,      7.40829,     7.47382,
    7.70902,     7.57303,     7.99318,     7.90092,     7.96435,
    8.1004,      8.11365,     8.30742,     8.04043,     8.38272,
    8.41612,     8.35161,     8.53789,     8.49252,     8.60087,
    8.62194,     8.50479,     8.75045,     8.53081,     8.64992,
    8.63693,     8.56288,     8.67372,     8.54276,     8.62255,
    8.54251,     8.52013,     8.6338,      8.26702,     8.4839,
    8.54189,     8.17121,     8.03211,     8.12066,     8.0961,
    7.94569,     7.69626,     7.64504,     7.73459,     7.36761,
    7.54359,     7.38203,     7.20701,     7.27583,     7.30464,
    7.25871,     7.52912,     7.67225,     7.98506,     7.98795,
    8.55224,     9.04907,     9.6271,      10.1937,     11.0415,
    10.9733,     10.5347,     7.31555,     2.47603,     7.07978e-05,
    0.0,         3.11167,     8.52601,     11.8504,     12.2518,
    11.8463,     11.2191,     10.4937,     9.68334,     9.29376,
    8.83965,     8.36776,     8.07761,     7.84583,     7.91066,
    7.72396,     7.76316,     7.6402,      7.71671,     7.68297,
    7.78738,     7.70256,     8.09267,     8.10819,     8.14314,
    8.13273,     8.28308,     8.51318,     8.54887,     8.4929,
    8.62367,     8.58279,     8.87267,     8.76386,     9.00117,
    8.95716,     8.83152,     8.73107,     9.06284,     8.82628,
    9.08255,     8.62271,     8.92899,     8.73063,     8.90444,
    8.88754,     8.79391,     8.97431,     8.72762,     8.60437,
    8.60205,     8.44556,     8.25754,     8.32011,     8.46852,
    8.08751,     8.0659,      7.88684,     7.72371,     7.93347,
    7.54762,     7.72144,     7.54474,     7.58901,     7.33311,
    7.63222,     7.5573,      7.64101,     7.92071,     8.3271,
    8.65345,     9.20059,     9.82764,     10.4759,     11.1322,
    11.5479,     10.9844,     7.7935,      2.60995,     7.37439e-05,
    0.0,         3.33466,     8.89087,     12.1916,     12.7821,
    12.4493,     11.5194,     10.9369,     10.0692,     9.35204,
    8.79355,     8.61618,     8.31715,     8.22894,     8.06981,
    8.0717,      7.94854,     7.7412,      7.98501,     7.86524,
    8.09439,     8.02209,     8.23791,     8.02855,     8.16933,
    8.4126,      8.53366,     8.65545,     8.4014,      8.85631,
    8.82715,     8.94703,     8.90579,     8.85369,     9.13713,
    9.17592,     9.25208,     8.93617,     9.00198,     9.00389,
    9.20004,     9.20161,     9.24654,     8.99243,     9.03803,
    9.0515,      8.92731,     8.89562,     8.84866,     8.83066,
    8.90402,     8.56967,     8.61372,     8.6316,      8.33684,
    8.26438,     8.37017,     8.22971,     8.20302,     8.02308,
    7.93468,     7.78224,     7.87713,     7.6601,      7.96391,
    7.7979,      7.83862,     8.15972,     8.18004,     8.61815,
    9.01153,     9.42581,     10.1782,     11.0769,     11.625,
    12.2732,     11.3194,     8.23671,     2.78515,     8.14007e-05,
    0.0,         3.50751,     9.37445,     12.6521,     13.3709,
    12.7305,     11.5638,     10.9898,     10.3023,     9.7438,
    9.31807,     8.75374,     8.6708,      8.31341,     8.24885,
    8.12857,     8.06628,     7.99958,     8.07651,     8.16293,
    8.24008,     8.48706,     8.62446,     8.48908,     8.75489,
    8.63105,     8.66464,     9.15728,     8.79387,     8.99926,
    8.94986,     8.94604,     9.06871,     9.09571,     9.32256,
    9.39883,     9.33582,     9.4049,      9.32001,     9.43402,
    9.36148,     9.50656,     9.38957,     9.39307,     9.29722,
    9.30534,     9.11591,     9.23831,     9.34882,     9.07885,
    8.97261,     9.04586,     8.86438,     8.78425,     8.63862,
    8.72797,     8.63217,     8.44205,     8.20547,     8.21176,
    8.00472,     8.02085,     7.99997,     7.9172,      8.05215,
    7.94328,     8.00378,     8.16758,     8.59174,     8.71699,
    9.01393,     9.79191,     10.5067,     11.2603,     12.0055,
    12.5624,     12.0627,     8.77612,     2.9767,      8.49675e-05,
    0.0,         3.73951,     9.97758,     13.1558,     13.9265,
    13.3063,     12.2193,     11.2899,     10.5437,     10.0931,
    9.41337,     9.01452,     8.70299,     8.57447,     8.55587,
    8.51905,     8.32134,     8.35088,     8.45791,     8.36116,
    8.31664,     8.55843,     8.63001,     8.67654,     8.81264,
    8.96769,     9.05927,     8.95615,     9.11282,     9.15855,
    9.35258,     9.32067,     9.47873,     9.40702,     9.43975,
    9.7626,      9.71306,     9.53147,     9.65691,     9.70764,
    9.58319,     9.58752,     9.55998,     9.60067,     9.86673,
    9.62432,     9.28297,     9.51175,     9.27263,     9.2846,
    9.10621,     9.11083,     8.89081,     8.92467,     8.80873,
    8.73964,     8.47066,     8.66795,     8.429,       8.56908,
    8.4626,      8.22821,     8.04912,     8.24431,     8.19161,
    8.23124,     8.08823,     8.31427,     8.65744,     8.8877,
    9.57966,     10.1491,     10.7388,     11.7664,     12.6276,
    13.2006,     12.5406,     9.11882,     3.15937,     8.86661e-05,
    0.0,         3.88539,     10.4585,     13.6156,     14.3917,
    13.8192,     12.7843,     11.6163,     10.6659,     10.2626,
    9.867,       9.184,       9.15617,     8.92825,     8.75562,
    8.6235,      8.60019,     8.61595,     8.41782,     8.61725,
    8.967,       8.80799,     9.06895,     9.04725,     9.16183,
    9.23489,     9.34507,     9.25577,     9.46233,     9.3823,
    9.4898,      9.39801,     9.68157,     9.77688,     9.63934,
    9.85139,     9.83784,     10.0103,     9.89834,     9.84653,
    9.87324,     9.98582,     9.93764,     9.88746,     9.74855,
    9.9732,      9.88559,     9.7533,      9.55558,     9.57781,
    9.74198,     9.33812,     9.39038,     9.38536,     9.30141,
    8.9908,      8.85204,     8.81172,     8.80696,     8.56836,
    8.52058,     8.61372,     8.50306,     8.42707,     8.29797,
    8.2759,      8.57209,     8.6631,      8.76809,     9.13256,
    9.72433,     10.4964,     11.1211,     12.2067,     13.169,
    13.5331,     13.0766,     9.81429,     3.29737,     9.43928e-05,
    0.0,         4.09459,     11.1812,     14.4663,     14.816,
    14.0498,     13.23,       12.1381,     11.143,      10.6836,
    9.81025,     9.4813,      9.24349,     8.81112,     9.03499,
    8.82809,     8.75189,     8.757,       8.73818,     8.78618,
    8.73387,     9.1112,      9.32278,     9.22602,     9.54644,
    9.17402,     9.14751,     9.47312,     9.67486,     9.63858,
    9.72373,     9.75002,     9.88079,     10.1311,     9.93125,
    9.87098,     10.4612,     10.0621,     10.1199,     10.4138,
    10.1346,     10.3895,     10.0827,     10.1929,     10.0954,
    10.1329,     10.0442,     9.96477,     9.80481,     9.95357,
    9.81314,     9.68596,     9.52046,     9.2301,      9.45648,
    9.16039,     9.29993,     9.06484,     9.08035,     8.91231,
    8.73008,     8.67879,     8.67595,     8.51207,     8.77942,
    8.70498,     8.72935,     8.8581,      8.92597,     9.32224,
    9.99448,     10.5772,     11.2366,     12.426,      13.2671,
    13.9884,     13.7334,     10.099,      3.60466,     0.000100437,
    0.0,         4.36119,     11.3936,     15.1683,     15.7414,
    14.5932,     13.4283,     12.2208,     11.7438,     10.8035,
    10.1764,     9.9276,      9.53604,     9.33123,     9.05547,
    9.02011,     9.04707,     8.9739,      9.24206,     9.00248,
    9.00729,     9.3783,      9.08402,     9.42637,     9.60336,
    9.85988,     9.84955,     9.93063,     9.99026,     10.1505,
    10.1424,     10.1348,     9.95635,     10.3002,     10.0813,
    10.1326,     10.1091,     10.4574,     10.5593,     10.5814,
    10.6125,     10.4038,     10.4897,     10.3424,     10.5551,
    10.4721,     10.1476,     10.3631,     10.2391,     10.219,
    9.84836,     10.0589,     9.79841,     9.84593,     9.59705,
    9.66142,     9.51317,     9.41143,     9.10543,     9.0022,
    8.97629,     8.9469,      8.74885,     8.85053,     8.8421,
    8.88976,     8.97502,     9.07559,     9.56861,     9.72314,
    10.2113,     11.103,      11.7966,     12.8651,     13.8566,
    15.074,      14.012,      10.6898,     3.65618,     0.000107977,
    0.0,         4.58915,     12.2312,     15.5907,     15.9464,
    15.0593,     13.6816,     12.9255,     12.0467,     11.0955,
    10.5731,     10.4544,     9.77754,     9.51653,     9.46217,
    9.25388,     9.35331,     9.28952,     9.31484,     9.22414,
    9.53594,     9.69031,     9.61691,     9.68818,     9.4927,
    9.72985,     9.96175,     10.3533,     10.2353,     10.1173,
    10.3945,     10.5474,     10.1727,     10.6002,     10.5197,
    10.2697,     10.5925,     10.8728,     10.624,      10.8181,
    10.7777,     10.8975,     10.6131,     10.5539,     10.5319,
    10.803,      10.406,      10.4488,     10.6591,     10.412,
    10.2094,     10.2157,     10.0708,     9.87116,     9.95027,
    9.84504,     9.90377,     9.67261,     9.40978,     9.49113,
    9.14222,     9.18986,     8.95061,     9.04353,     9.00775,
    8.99489,     9.17453,     9.25859,     9.72468,     10.0967,
    10.5147,     11.1779,     12.2022,     13.141,      14.4925,
    15.0879,     14.6225,     11.2476,     3.89821,     0.000108364,
    0.0,         4.79506,     12.7019,     16.4086,     16.543,
    15.5203,     14.4083,     13.398,      12.1189,     11.5193,
    10.7542,     10.3381,     9.88837,     9.6954,      9.72476,
    9.53633,     9.59209,     9.3437,      9.67512,     9.57145,
    9.51492,     9.85307,     9.68787,     9.93303,     9.91458,
    9.87415,     9.81055,     10.0567,     10.5097,     10.7501,
    10.4855,     10.7546,     10.8734,     11.0272,     10.7883,
    10.7656,     10.6528,     11.1864,     10.8233,     11.0667,
    11.1229,     11.084,      11.0353,     10.7995,     11.0467,
    11.1272,     11.0602,     10.7439,     10.7476,     10.7169,
    10.6848,     10.4357,     10.408,      10.1067,     10.2397,
    10.0578,     10.0293,     9.88946,     9.86376,     9.61448,
    9.41411,     9.34819,     9.38426,     9.25578,     9.34683,
    9.33879,     9.35293,     9.47485,     10.0431,     10.2657,
    10.9373,     11.6761,     12.5604,     13.7471,     14.8211,
    15.7075,     15.5107,     11.9268,     4.34918,     0.000117669,
    0.0,         5.10712,     13.2299,     16.9974,     17.2496,
    16.1499,     14.7337,     13.59,       12.4138,     11.835,
    11.1486,     10.6598,     10.2067,     10.0201,     9.99867,
    9.82161,     9.63037,     9.57565,     9.65008,     9.88726,
    10.3195,     9.95015,     9.81844,     9.98832,     10.5659,
    10.3711,     10.4929,     10.3667,     10.5857,     10.8981,
    10.7131,     10.7925,     11.0477,     11.201,      11.0846,
    11.3147,     11.0243,     11.3026,     10.9918,     11.2412,
    11.3558,     11.1178,     11.2462,     11.1298,     11.472,
    11.0669,     11.0774,     10.8795,     10.9234,     10.9497,
    10.8994,     10.5946,     10.6407,     10.3778,     10.4647,
    10.1932,     10.4884,     10.1095,     9.95967,     9.89803,
    9.75765,     9.5195,      9.54827,     9.80817,     9.24865,
    9.47584,     9.79646,     9.81835,     10.3173,     10.5759,
    10.9977,     12.0675,     13.0381,     14.1378,     15.585,
    16.0683,     16.1073,     12.4036,     4.54893,     0.000122798,
    0.0,         5.3261,      13.8279,     17.5217,     17.7371,
    16.5582,     15.2146,     13.7204,     12.9927,     12.1293,
    11.3858,     10.7047,     10.6087,     10.1889,     10.036,
    9.89061,     9.87388,     9.89412,     10.0182,     10.0928,
    10.0305,     10.234,      10.2817,     10.5698,     10.2526,
    10.7793,     10.9887,     10.7955,     11.0374,     11.1639,
    11.269,      11.1943,     10.962,      11.4854,     11.3571,
    11.5307,     11.6449,     11.3543,     11.5406,     11.6385,
    11.3878,     11.4413,     11.58,       11.5852,     11.4032,
    11.6701,     11.3331,     11.2789,     11.1481,     11.0735,
    11.1406,     10.9298,     11.0188,     10.804,      10.537,
    10.6009,     10.4089,     10.1463,     10.1659,     9.95133,
    9.99909,     9.8583,      9.8448,      9.71222,     9.74599,
    9.74241,     9.87896,     10.0978,     10.5887,     10.9744,
    11.4735,     12.2535,     13.1964,     14.4826,     15.5144,
    16.8222,     16.8493,     13.2363,     4.72943,     0.000131398,
    0.0,         5.66626,     14.5125,     18.2352,     18.4855,
    17.4256,     15.7587,     14.2955,     13.1747,     12.2796,
    11.7457,     11.256,      10.6808,     10.5788,     10.3598,
    10.3007,     10.1034,     10.2588,     10.2435,     10.2312,
    10.2567,     10.5849,     10.5815,     10.8458,     10.668,
    10.7691,     10.8967,     11.1673,     11.3273,     11.3259,
    11.4865,     11.3905,     11.6498,     11.6881,     11.9645,
    11.4555,     11.7382,     11.6537,     11.8212,     11.5506,
    11.8368,     11.6208,     11.678,      11.8761,     11.8924,
    11.9378,     11.6015,     11.4473,     11.6856,     11.2844,
    11.3398,     11.3959,     11.1004,     10.96,       10.9441,
    10.9273,     10.897,      10.3983,     10.5001,     10.5892,
    10.0304,     10.0746,     10.1243,     9.86107,     10.1763,
    9.80993,     9.86743,     10.5044,     10.8494,     11.0726,
    11.5686,     12.4885,     13.4373,     14.9149,     16.0883,
    17.7432,     17.6976,     13.7078,     5.04034,     0.000137749,
    0.0,         5.97327,     15.1635,     19.1074,     19.0694,
    17.8128,     16.4818,     14.6187,     13.4128,     12.4628,
    12.1216,     11.5126,     11.2582,     10.861,      10.7365,
    10.4932,     10.4704,     10.562,      10.7218,     10.8065,
    10.465,      10.7222,     10.9753,     10.7642,     10.8329,
    11.1526,     11.2644,     11.4969,     11.2264,     11.5379,
    11.585,      11.8281,     11.6268,     11.9914,     12.075,
    11.3921,     11.9075,     11.9978,     12.2667,     11.8391,
    12.1321,     12.1717,     11.7694,     12.0184,     12.2442,
    12.031,      11.9133,     12.053,      11.5578,     11.9443,
    11.8572,     11.4834,     11.4956,     11.162,      11.3764,
    10.9579,     11.2118,     10.5681,     10.6598,     10.5075,
    10.4243,     10.2578,     10.0413,     10.1317,     10.4146,
    10.1873,     10.247,      10.8454,     10.8706,     11.4153,
    12.416,      12.9283,     13.7668,     15.4538,     16.7158,
    18.4251,     18.4396,     13.7861,     5.39005,     0.000147139,
    0.0,         6.29418,     15.7647,     19.6737,     19.9923,
    18.0762,     16.628,      14.9504,     14.0509,     13.0025,
    12.3302,     11.6373,     11.1932,     11.2619,     10.7341,
    10.5904,     10.7013,     11.0981,     10.7673,     10.7225,
    11.0178,     10.9496,     11.0371,     11.2166,     11.2178,
    11.4495,     11.5073,     11.5734,     11.6457,     11.8851,
    11.7686,     12.0703,     11.9609,     12.169,      12.3183,
    11.919,      12.3034,     12.1932,     12.7097,     12.1257,
    12.6085,     12.3652,     12.3531,     12.1347,     12.4542,
    12.1367,     12.5765,     12.016,      12.0101,     11.9879,
    11.9666,     11.87,       11.9182,     11.4904,     11.5285,
    11.5173,     11.3204,     11.1004,     10.8851,     10.6678,
    10.5789,     10.7558,     10.2283,     10.2952,     10.407,
    10.5293,     10.6024,     10.9084,     10.8991,     11.5832,
    12.5998,     13.5767,     14.3468,     15.8536,     17.3619,
    19.1363,     18.866,      14.8745,     5.62084,     0.000156162,
    0.0,         6.67518,     16.6059,     20.3777,     20.6872,
    19.0229,     17.2415,     15.9231,     14.3645,     13.2786,
    12.5768,     11.9244,     11.7329,     11.312,      11.1889,
    10.9816,     11.0717,     11.0978,     10.8306,     11.1182,
    11.1317,     11.5124,     11.7232,     11.2466,     11.627,
    11.6005,     11.7142,     11.5211,     12.0806,     12.1656,
    11.9706,     12.704,      12.7756,     12.5909,     12.6837,
    12.8048,     12.9436,     12.6665,     12.6523,     13.0317,
    12.6794,     12.7165,     12.458,      12.5315,     12.7747,
    12.6355,     12.5157,     12.1837,     12.3277,     12.3904,
    12.2078,     12.1482,     12.2199,     11.8174,     11.6624,
    11.842,      11.4992,     11.1735,     11.2917,     11.2592,
    10.9963,     10.861,      10.8742,     10.6207,     10.687,
    10.7873,     10.8291,     11.0694,     11.4921,     11.8708,
    13.0091,     13.8871,     14.8927,     16.4502,     17.8132,
    19.6053,     19.5982,     15.8061,     5.94405,     0.000163383,
    0.0,         7.05615,     17.4052,     21.2957,     20.9455,
    19.378,      17.5644,     15.9376,     14.7139,     13.7331,
    12.747,      12.2137,     11.9699,     11.5492,     11.375,
    11.3373,     11.2829,     11.1856,     11.6408,     11.4244,
    11.0875,     11.4143,     11.9742,     11.8554,     11.8359,
    12.1121,     11.8805,     12.2616,     12.3948,     12.2482,
    12.4828,     12.6133,     12.6166,     12.4653,     12.6806,
    12.7839,     13.0845,     12.9686,     13.1191,     12.667,
    13.0231,     13.0906,     12.847,      13.1771,     12.9384,
    13.0266,     12.901,      12.8473,     12.545,      12.7149,
    12.8342,     12.2498,     11.9172,     12.1822,     12.2351,
    11.8704,     12.0144,     11.4264,     11.4596,     11.4209,
    11.1583,     11.0284,     10.9754,     11.1689,     10.8616,
    11.1171,     11.1117,     11.3228,     11.9091,     12.3685,
    13.0167,     13.89,       14.976,      16.954,      18.4528,
    20.4708,     20.3374,     16.5154,     6.29742,     0.000170037,
    0.0,         7.35903,     18.1281,     22.1304,     22.0352,
    20.3744,     18.2145,     16.4955,     14.8935,     14.3708,
    13.1909,     12.6321,     12.3337,     11.7383,     11.6747,
    11.5098,     11.4365,     11.6556,     11.3554,     11.356,
    11.5719,     11.7732,     11.9196,     12.1221,     12.159,
    12.3325,     12.4288,     12.6586,     12.5106,     12.7283,
    12.3368,     12.8228,     13.0708,     13.4306,     13.2762,
    13.2085,     13.2939,     13.2561,     13.2948,     13.3651,
    13.0229,     13.1367,     13.2265,     13.3175,     13.565,
    13.0188,     12.6724,     12.8725,     13.0657,     12.8322,
    13.0453,     12.9404,     12.4557,     12.6459,     12.1831,
    11.8574,     12.375,      11.9678,     11.8498,     11.6008,
    11.775,      11.4176,     11.1944,     11.2626,     11.1632,
    11.1407,     11.5849,     11.8943,     12.2265,     12.4835,
    13.4711,     14.512,      15.4273,     17.1514,     19.2238,
    21.0673,     21.3877,     17.1141,     6.53867,     0.000180288,
    0.0,         7.77576,     18.8426,     23.1133,     22.5135,
    20.3634,     18.8387,     16.5185,     15.5356,     14.3438,
    13.9008,     12.8875,     12.6006,     12.0236,     11.9711,
    11.6308,     11.7222,     12.0703,     11.9313,     11.7963,
    12.0231,     12.1874,     11.9191,     12.2272,     12.6728,
    12.5086,     12.523,      12.9704,     12.7346,     12.8711,
    12.9933,     13.1246,     13.1881,     13.6148,     13.2121,
    13.3143,     13.6775,     13.6929,     13.73,       13.5375,
    13.5621,     13.5827,     13.4458,     13.3152,     13.6574,
    13.6538,     13.6796,     13.1784,     13.2659,     13.4038,
    12.982,      13.1488,     13.1193,     12.7339,     12.467,
    12.7078,     12.4476,     12.3783,     11.9165,     11.975,
    11.769,      11.6089,     11.3124,     11.9025,     11.5362,
    11.4286,     11.3198,     11.747,      12.1427,     12.8433,
    13.3626,     14.6481,     16.1277,     17.7482,     19.4177,
    21.7631,     22.0946,     17.9204,     6.81891,     0.000190313,
    0.0,         8.04558,     19.6987,     23.8949,     23.1366,
    21.48,       19.3162,     17.3636,     16.2705,     14.7889,
    13.7695,     13.4204,     12.8825,     12.4371,     12.2451,
    12.1986,     12.023,      12.0287,     12.1425,     11.9067,
    12.4142,     12.0478,     12.3233,     12.6685,     12.8758,
    12.8754,     13.1622,     13.0876,     12.953,      13.1607,
    13.5589,     13.6122,     13.6231,     14.0306,     13.4389,
    13.7831,     13.8571,     13.7443,     13.7435,     13.9178,
    14.2026,     14.2671,     13.929,      13.797,      14.0731,
    13.5905,     13.7837,     13.6502,     13.4291,     13.4743,
    13.3394,     13.5266,     12.9314,     12.7733,     13.1457,
    12.952,      12.736,      12.5291,     12.4644,     12.3484,
    12.2734,     11.9956,     11.8288,     11.7828,     11.8161,
    11.7291,     11.8606,     12.2822,     12.482,      13.3048,
    14.1847,     15.0675,     16.3703,     18.3121,     20.4983,
    21.8782,     23.1032,     18.6114,     7.31797,     0.00019905,
    0.0,         8.65507,     20.6692,     24.3355,     23.9866,
    22.1608,     19.9017,     18.0256,     16.3447,     14.9169,
    14.2629,     13.5841,     13.2999,     12.7959,     12.9508,
    12.2568,     12.2017,     12.4456,     12.5603,     12.8029,
    12.5546,     12.6042,     12.762,      12.5303,     13.1639,
    13.0665,     13.2142,     13.3265,     13.5494,     13.728,
    13.5325,     13.988,      13.7648,     13.8598,     14.1492,
    14.1016,     14.2961,     14.2046,     14.2221,     14.345,
    14.1724,     14.1111,     14.5605,     14.3186,     14.1517,
    14.0378,     13.8693,     14.1845,     14.11,       13.8764,
    13.7809,     13.8466,     13.6018,     13.143,      13.1814,
    13.228,      12.886,      12.6879,     12.6419,     12.3957,
    12.3054,     12.1983,     12.1026,     12.0189,     11.8526,
    12.0282,     11.9887,     12.5228,     13.0506,     13.2782,
    14.2866,     15.59,       17.1078,     18.3506,     20.4474,
    22.8391,     23.4569,     19.4816,     7.58037,     0.000213896,
    0.0,         8.98715,     21.2039,     25.6187,     24.7183,
    22.4812,     20.2522,     18.3874,     16.5043,     15.4564,
    14.7757,     13.8852,     13.2149,     13.0637,     12.7138,
    12.5695,     12.5898,     12.4469,     12.5587,     12.2182,
    12.6513,     13.1079,     12.8879,     13.0882,     13.0887,
    13.4176,     13.5981,     13.711,      13.7513,     13.8094,
    14.3326,     14.766,      14.4491,     13.8094,     14.5929,
    14.4127,     14.0264,     14.4226,     14.4886,     14.6224,
    14.5777,     14.6936,     14.6351,     14.4376,     14.4331,
    14.2163,     14.4628,     14.4595,     13.9157,     14.3244,
    13.7838,     13.5942,     13.9381,     13.8881,     13.6953,
    13.4019,     13.3039,     13.1544,     12.9663,     12.5506,
    12.6815,     12.6627,     11.9362,     11.9727,     12.355,
    12.2379,     12.5096,     12.6969,     13.3984,     13.772,
    14.8687,     15.5597,     17.5516,     18.8431,     21.4844,
    23.558,      24.8363,     20.3279,     7.93331,     0.00022558,
    0.0,         9.36386,     22.3659,     26.235,      24.9593,
    22.795,      20.7682,     18.4585,     17.0768,     15.6877,
    15.1389,     14.2099,     13.587,      13.4613,     13.1312,
    12.6089,     12.9698,     13.163,      13.1037,     13.0686,
    13.0138,     13.2122,     13.1881,     13.6169,     13.2264,
    13.6853,     13.645,      14.1648,     14.5246,     14.1365,
    14.7206,     14.4109,     14.4783,     14.3425,     14.5669,
    14.3874,     14.4658,     14.6466,     14.9283,     14.8294,
    14.628,      14.948,      14.814,      15.2105,     15.252,
    14.1611,     14.7202,     14.5386,     14.4232,     14.5813,
    14.1714,     14.0832,     14.2857,     13.6059,     13.6981,
    13.6335,     13.5316,     13.468,      13.3826,     13.0903,
    13.2053,     12.2945,     12.6032,     12.7331,     12.4275,
    12.742,      12.9303,     12.944,      13.336,      13.9914,
    14.8412,     16.4048,     17.6925,     19.6096,     22.2683,
    24.1245,     25.2169,     21.1497,     8.79741,     0.000230445,
    0.0,         9.91539,     23.1007,     27.4127,     26.4635,
    23.5499,     21.4797,     18.9381,     17.74,       16.3737,
    15.1338,     14.6227,     14.1178,     13.5446,     13.6085,
    13.3223,     13.359,      12.7998,     13.2836,     13.4134,
    13.1987,     13.2212,     13.86,       13.7173,     13.8959,
    14.2209,     14.166,      13.9419,     14.3103,     14.6968,
    14.7382,     14.8515,     14.834,      14.552,      14.9207,
    15.2799,     15.2969,     15.2665,     15.0829,     15.5525,
    15.0562,     15.004,      15.4664,     15.0739,     15.1193,
    15.1999,     14.8518,     14.955,      14.7109,     15.1482,
    14.6082,     14.5775,     13.9852,     14.6307,     13.9905,
    14.0473,     13.9534,     13.7955,     13.4796,     13.2218,
    13.1521,     13.037,      12.8131,     13.2601,     12.8034,
    13.1766,     13.2095,     13.0255,     13.6397,     14.4326,
    15.4383,     16.8563,     18.3919,     20.8596,     22.657,
    24.9703,     26.4562,     22.3661,     9.06511,     0.000238113,
    0.0,         10.4158,     24.1313,     28.5719,     26.9704,
    24.5929,     21.9891,     19.8188,     18.2625,     16.6114,
    15.5763,     15.0098,     14.3455,     13.8941,     13.5323,
    13.6371,     13.4072,     13.1474,     13.5119,     13.6126,
    13.4084,     13.7761,     14.0702,     14.1774,     13.7935,
    14.0256,     14.4614,     14.6923,     14.7303,     14.6768,
    14.9665,     14.9037,     15.3041,     15.2484,     15.4609,
    15.3914,     15.6535,     15.546,      15.0459,     15.3468,
    15.4037,     15.4475,     15.5994,     15.6711,     15.6138,
    15.8725,     15.5021,     15.2172,     14.9838,     15.143,
    14.8865,     14.7949,     14.6561,     14.6095,     14.2859,
    14.3123,     14.1233,     13.5588,     13.786,      13.5287,
    13.7176,     13.1187,     13.3756,     13.1556,     13.0263,
    12.9497,     13.5871,     13.8697,     14.0426,     15.0686,
    15.7661,     17.5715,     18.5705,     20.9823,     23.6967,
    25.4874,     27.0533,     23.5751,     9.3849,      0.000260538,
    0.0,         10.9722,     25.1094,     29.4127,     28.4464,
    25.6241,     22.4107,     20.3618,     18.9265,     16.9306,
    16.2309,     15.1779,     14.569,      14.1555,     13.8234,
    13.8102,     13.8517,     13.8067,     13.8381,     13.6404,
    13.812,      14.217,      14.5453,     14.2748,     14.4647,
    14.3304,     15.1109,     14.9709,     15.2211,     15.053,
    15.1091,     15.4213,     15.5702,     15.5556,     15.8382,
    16.018,      15.524,      16.0531,     15.8481,     15.9803,
    15.9277,     15.6274,     16.0747,     15.7781,     15.933,
    15.7437,     15.541,      15.7294,     15.5095,     15.4452,
    15.598,      15.3593,     15.3715,     14.777,      14.7536,
    14.8354,     14.343,      14.306,      14.1311,     13.8403,
    13.5843,     13.6182,     13.3539,     13.4187,     13.0134,
    13.7382,     13.9103,     14.0645,     14.5746,     15.1262,
    16.0829,     17.1191,     19.1131,     21.4557,     24.8489,
    26.4256,     28.3565,     24.3324,     9.67915,     0.000268572,
    0.0,         11.4461,     26.1534,     30.7358,     29.0693,
    25.8414,     23.1342,     20.4813,     18.4705,     17.6973,
    16.3906,     15.5397,     15.2268,     14.2729,     14.234,
    14.1457,     13.5866,     14.3167,     14.1897,     14.2019,
    14.112,      14.4567,     14.4452,     14.5736,     14.425,
    14.8203,     14.9839,     15.026,      15.2985,     15.4035,
    15.6571,     15.2801,     15.9314,     15.7045,     15.8171,
    15.9148,     16.4539,     16.0953,     15.5342,     16.2058,
    15.9774,     15.9965,     16.3899,     16.0601,     16.0938,
    15.9046,     15.7503,     16.0571,     16.215,      15.3296,
    15.3744,     15.627,      15.3192,     15.2713,     15.2597,
    14.77,       14.573,      14.5058,     14.5443,     14.3108,
    13.9976,     13.8705,     13.8677,     13.7614,     13.3965,
    13.6501,     13.9703,     14.6229,     14.7159,     15.3462,
    16.4917,     17.7208,     19.3323,     21.77,       24.7001,
    27.7392,     29.3476,     25.3029,     10.5239,     0.000288854,
    0.0,         11.9215,     27.3735,     31.2445,     29.6447,
    26.7689,     23.4785,     21.2839,     19.3918,     17.9578,
    16.5324,     15.9628,     15.4475,     14.7022,     14.359,
    14.5019,     14.3639,     14.5436,     14.1979,     14.7789,
    14.4955,     14.863,      15.2042,     14.9432,     15.1142,
    15.5995,     15.7045,     15.5379,     15.5464,     16.3311,
    15.8286,     15.7214,     16.2552,     16.413,      16.0111,
    16.6489,     16.805,      16.7018,     16.4161,     16.94,
    16.4362,     16.4764,     16.8231,     16.5575,     16.4809,
    16.6478,     16.6183,     16.2017,     16.1477,     15.7506,
    16.0253,     16.1332,     15.6229,     15.6214,     15.2611,
    15.1914,     15.0123,     14.5025,     14.4986,     14.6779,
    14.0502,     14.3095,     14.3741,     13.8911,     13.8207,
    13.857,      14.1932,     14.6759,     15.1517,     16.0257,
    16.8821,     17.8888,     20.5287,     22.6655,     25.4517,
    28.4443,     30.738,      25.5951,     11.2779,     0.000299316,
    0.0,         12.4481,     28.7374,     32.7007,     31.0541,
    27.3243,     23.9837,     21.5759,     19.8482,     18.5826,
    17.1562,     16.0771,     15.7933,     14.9767,     15.0592,
    14.6892,     14.606,      14.662,      14.7034,     14.5845,
    14.6433,     14.7001,     14.9325,     14.9214,     14.9968,
    15.6788,     15.5788,     15.6728,     15.7276,     15.8383,
    16.5413,     16.5703,     16.7301,     16.5555,     16.1521,
    16.7255,     16.5698,     16.1969,     16.601,      17.208,
    17.1001,     16.9944,     16.9242,     16.5589,     17.226,
    16.9813,     16.6604,     16.6683,     16.1934,     16.4968,
    16.0242,     16.3398,     15.8642,     15.9377,     15.599,
    15.6931,     15.6164,     15.0298,     14.9172,     14.8974,
    14.9499,     14.3172,     14.2416,     14.3009,     14.3214,
    13.9183,     14.482,      14.9378,     15.5144,     16.5252,
    17.0932,     18.556,      21.0662,     22.9351,     26.3258,
    29.2841,     30.9436,     27.4461,     11.4334,     0.00030771,
    0.0,         13.0198,     29.8295,     33.1474,     31.9463,
    28.1036,     25.023,      21.8902,     20.1767,     18.6543,
    17.4943,     16.7668,     15.9705,     15.5304,     15.3632,
    14.7789,     14.8438,     14.8012,     15.1905,     14.9793,
    15.4376,     15.4583,     15.4169,     15.3021,     15.8995,
    15.7969,     15.7894,     15.8312,     16.5014,     16.4614,
    16.7801,     16.2372,     16.9557,     17.0292,     16.8026,
    16.9218,     17.2549,     17.1278,     17.0672,     17.7207,
    16.9747,     16.8818,     17.2875,     17.4443,     17.2539,
    17.092,      17.142,      17.2291,     16.4506,     16.9394,
    17.1897,     16.5484,     16.2634,     16.4045,     16.1263,
    15.9881,     15.5064,     15.378,      15.756,      15.1552,
    14.6919,     15.1487,     14.7854,     14.6131,     14.3369,
    14.5079,     14.8905,     15.4107,     15.6779,     16.4785,
    17.969,      19.5254,     21.1907,     23.7024,     27.4726,
    29.6672,     32.26,       27.9015,     12.261,      0.000321301,
    0.0,         13.6996,     30.4892,     34.497,      32.4875,
    28.9003,     25.2929,     22.822,      20.2584,     18.9572,
    17.8714,     16.9382,     16.0299,     15.5553,     15.4147,
    15.0852,     15.2762,     15.3439,     15.3575,     15.6562,
    15.6988,     15.3679,     15.4462,     15.8538,     15.7046,
    16.4722,     16.3555,     16.6707,     16.7648,     16.8776,
    16.6045,     17.272,      17.2719,     16.8892,     17.0733,
    17.555,      17.4302,     17.7287,     17.4097,     17.4119,
    17.707,      17.458,      17.5157,     17.6394,     17.3664,
    17.413,      17.433,      17.815,      17.1407,     16.5293,
    17.4507,     16.6336,     16.9749,     16.7379,     16.6259,
    16.5391,     15.7623,     15.8895,     15.6722,     15.4148,
    15.2244,     14.9335,     15.1813,     14.8967,     15.0203,
    14.7847,     15.6316,     15.5908,     15.6281,     17.1925,
    18.0012,     19.7383,     21.9857,     24.3727,     28.2196,
    30.9487,     34.0089,     29.5924,     12.9269,     0.000343851,
    0.0,         14.2697,     31.9179,     35.3656,     33.4951,
    29.8526,     26.2031,     23.3327,     21.6685,     19.5102,
    18.4532,     17.4449,     16.7684,     16.0484,     15.9984,
    15.6134,     15.5242,     15.4472,     15.5799,     15.3419,
    15.8693,     16.0452,     16.2434,     16.6471,     16.1389,
    16.613,      16.7556,     16.9854,     16.6519,     16.7008,
    17.2757,     17.3915,     17.3031,     17.8784,     18.041,
    18.0603,     17.5899,     17.917,      18.2239,     17.4001,
    17.9371,     17.8059,     17.8055,     17.8557,     18.1343,
    18.0971,     17.8019,     17.5507,     18.0166,     17.4515,
    17.1186,     17.323,      17.1081,     16.7751,     16.6821,
    16.9378,     16.5264,     16.2,        15.6669,     15.7795,
    15.3384,     15.526,      14.9473,     15.2336,     14.9551,
    15.3054,     15.4047,     15.8496,     16.3754,     17.2201,
    18.6544,     20.3802,     21.7945,     25.4816,     28.2382,
    31.5445,     34.0913,     30.3551,     13.2899,     0.00035976,
    0.0,         14.9397,     33.3994,     36.8571,     34.214,
    30.3608,     27.2764,     23.9734,     21.692,      20.1827,
    18.7111,     17.7493,     16.8932,     16.5281,     16.2087,
    15.8886,     15.8966,     16.3247,     15.8566,     15.681,
    16.1562,     16.2554,     16.5459,     17.091,      16.6244,
    16.7739,     17.3578,     17.5008,     17.461,      17.8522,
    17.8417,     18.045,      17.8604,     17.5441,     17.7982,
    17.8497,     17.4816,     18.2288,     18.2857,     17.9721,
    18.3266,     17.9506,     18.6471,     18.5858,     18.1893,
    17.8644,     18.4127,     17.5594,     17.8541,     17.8613,
    17.3984,     17.53,       17.4558,     17.2601,     16.9151,
    16.7514,     16.185,      16.6052,     16.0926,     15.931,
    15.4481,     15.6699,     15.249,      15.1286,     15.6012,
    15.7967,     16.2354,     15.9268,     16.9345,     17.5447,
    19.0172,     20.9444,     22.7924,     25.772,      28.8125,
    32.3152,     34.8865,     31.8827,     14.1985,     0.000371747,
    0.0,         15.7893,     34.3559,     38.2871,     35.5536,
    31.412,      27.8169,     25.0084,     21.9756,     20.3019,
    19.1,        18.0261,     17.2176,     17.0962,     16.6549,
    16.2107,     16.2092,     16.5552,     16.4043,     16.3194,
    16.5748,     16.6599,     16.8516,     17.02,       17.4979,
    17.159,      17.3052,     18.0499,     17.8852,     17.7735,
    16.9523,     17.8745,     18.7217,     18.0009,     18.2419,
    18.7473,     18.0507,     18.4184,     18.9404,     18.6416,
    18.7739,     18.7009,     18.3456,     18.7063,     19.057,
    18.3012,     18.9234,     18.2058,     18.1169,     18.3076,
    18.0483,     17.892,      17.4234,     17.0136,     17.4805,
    17.2645,     17.2607,     16.6783,     16.7352,     16.1652,
    16.4373,     16.0803,     15.4338,     15.7915,     15.9628,
    16.3044,     16.3955,     16.4178,     17.4534,     17.4752,
    19.3423,     21.4172,     23.6501,     25.475,      29.3673,
    34.285,      36.5757,     33.288,      14.6687,     0.000392835,
    0.0,         16.3294,     35.9045,     40.0612,     36.2365,
    32.8204,     28.5052,     25.4776,     22.2387,     20.7258,
    19.2413,     18.2154,     17.7317,     17.0203,     16.7074,
    16.4818,     16.7574,     16.8683,     16.5903,     16.6332,
    16.6543,     16.5388,     16.818,      17.4744,     17.6335,
    17.615,      17.8475,     17.9336,     18.324,      17.9828,
    18.1653,     18.5059,     18.897,      18.6945,     18.6865,
    18.8433,     19.3899,     18.2916,     19.5052,     19.1815,
    19.2855,     19.0516,     19.0117,     19.2522,     18.8392,
    18.8895,     19.1067,     18.7468,     19.0335,     18.7001,
    18.5447,     18.0342,     17.8962,     17.9817,     18.1033,
    17.4514,     17.4084,     17.1072,     16.5805,     16.7966,
    16.2586,     16.5144,     16.2924,     16.1459,     15.9352,
    16.3209,     15.9495,     16.9099,     17.5277,     18.2862,
    19.9808,     21.2923,     24.1734,     27.2815,     30.6313,
    34.3614,     38.3251,     34.1936,     15.3,        0.000423738,
    0.0,         16.9312,     37.4419,     40.4765,     37.7569,
    32.8768,     28.3496,     25.736,      23.2714,     21.4866,
    19.9016,     18.8374,     18.3133,     17.7453,     17.1076,
    16.7611,     16.4432,     16.7342,     16.6988,     17.356,
    17.0176,     17.5911,     17.7678,     17.276,      18.0712,
    18.0406,     17.986,      17.7552,     18.502,      18.6012,
    18.8466,     18.9452,     19.188,      18.6909,     18.8007,
    19.0366,     19.4931,     18.9386,     19.6375,     19.4312,
    18.791,      19.1393,     19.8034,     20.0534,     18.7453,
    19.6769,     19.487,      19.1027,     19.39,       18.7232,
    18.5973,     18.6237,     18.4207,     18.1648,     17.7908,
    18.0223,     17.7936,     17.4971,     17.1406,     17.2269,
    16.7755,     17.0282,     16.7653,     16.4752,     16.4948,
    16.4528,     16.5175,     17.0605,     18.2814,     18.8209,
    20.7549,     21.9574,     24.6107,     27.0623,     31.2545,
    35.4188,     39.0027,     35.8785,     16.1093,     0.000433258,
    0.0,         18.1294,     38.2769,     41.3209,     38.6919,
    34.0706,     29.2097,     26.6382,     23.7625,     21.7121,
    20.3198,     19.7231,     18.351,      18.0645,     17.5856,
    17.2736,     17.1499,     17.003,      16.9708,     17.5815,
    17.2844,     17.1411,     18.2762,     17.6831,     18.1129,
    17.9793,     18.7048,     19.3049,     18.6299,     18.9844,
    19.2311,     19.0847,     19.4836,     19.2973,     19.598,
    19.1953,     19.2482,     19.8175,     19.7899,     20.3438,
    20.142,      19.8361,     19.2708,     19.818,      19.8242,
    19.3814,     19.2067,     19.1964,     19.4354,     19.6705,
    19.2156,     18.8411,     19.0393,     19.027,      18.6276,
    18.3322,     18.466,      17.8697,     18.1006,     17.4827,
    16.8925,     16.6825,     16.9727,     16.6047,     17.0267,
    16.8342,     17.4191,     17.7911,     18.481,      19.2586,
    20.4931,     21.7663,     25.033,      28.9359,     32.0329,
    37.0664,     39.6562,     36.9098,     16.9975,     0.000455571,
    0.0,         19.0065,     39.6883,     43.7622,     39.2666,
    33.891,      30.3063,     26.63,       24.0892,     22.515,
    20.6344,     19.6017,     18.9244,     18.2152,     18.1662,
    17.5742,     17.8632,     17.5121,     17.3824,     17.5125,
    17.8906,     17.6442,     17.7603,     17.9316,     18.5904,
    18.7223,     18.4662,     18.7331,     19.0141,     19.4904,
    19.9764,     19.653,      19.7529,     19.9948,     20.0188,
    19.6935,     19.8256,     19.9186,     20.278,      20.441,
    20.4032,     20.3138,     20.076,      20.1951,     20.1054,
    20.3369,     20.0924,     20.1094,     20.1688,     19.7351,
    19.9212,     19.5632,     19.4494,     18.9706,     18.3792,
    18.4448,     18.5311,     18.1841,     18.0842,     17.8849,
    17.8056,     17.692,      17.036,      16.5735,     17.2203,
    17.2342,     17.3254,     17.7442,     18.4677,     19.1272,
    20.986,      22.7594,     25.4427,     28.828,      32.5769,
    36.9641,     41.9137,     38.412,      17.3821,     0.00048181,
    0.0,         19.4509,     41.5468,     44.5405,     40.2788,
    35.4954,     31.2738,     27.709,      24.6436,     22.7902,
    21.5296,     20.3537,     19.1126,     18.5192,     18.4975,
    17.7935,     17.8929,     17.7751,     18.2025,     18.0329,
    17.9477,     18.1132,     18.8809,     18.0179,     18.987,
    18.702,      19.6042,     19.6407,     20.0003,     19.652,
    19.9409,     19.8848,     20.2413,     20.545,      20.2945,
    20.8383,     20.234,      19.7261,     21.0428,     20.401,
    20.6689,     20.8729,     20.928,      20.2619,     20.2542,
    20.4331,     20.9844,     20.7412,     20.4247,     19.8221,
    19.6993,     18.9347,     19.9019,     19.5779,     19.2391,
    18.8097,     18.7039,     18.6098,     18.8303,     17.9625,
    18.0217,     18.0276,     17.6022,     17.5258,     16.8725,
    17.7114,     18.2415,     18.2335,     19.0946,     20.3031,
    20.8487,     23.1625,     26.1632,     30.0214,     33.9243,
    38.9143,     43.0694,     39.2176,     18.4132,     0.00050639,
    0.0,         20.7432,     43.0474,     45.4531,     42.1893,
    36.1512,     31.0353,     28.4813,     25.3822,     23.096,
    21.3903,     20.4421,     19.0227,     19.206,      18.6172,
    18.0321,     18.263,      18.4631,     18.2641,     18.7825,
    18.8709,     19.0943,     18.8452,     19.4517,     19.7841,
    19.4052,     19.5546,     19.9034,     20.3945,     19.9445,
    20.4256,     20.259,      20.235,      20.5343,     21.297,
    20.4968,     20.7395,     21.405,      21.0349,     20.5133,
    21.1214,     20.9857,     20.6219,     20.9976,     20.9399,
    20.2891,     20.48,       20.7904,     20.5,        20.4043,
    20.0865,     19.8539,     20.1705,     19.8207,     19.8907,
    19.6591,     19.2675,     18.5578,     18.6921,     18.5535,
    18.3663,     17.9524,     17.8635,     17.6161,     18.0757,
    17.8273,     18.051,      18.3404,     19.1671,     20.7194,
    21.8356,     24.1805,     26.681,      31.1334,     34.5234,
    40.254,      44.6555,     41.4166,     19.331,      0.000519248,
    0.0,         21.4451,     44.0935,     47.1903,     41.8898,
    37.5233,     32.1803,     29.1643,     25.871,      23.4667,
    22.0885,     21.0573,     20.0138,     19.1539,     19.3881,
    18.6336,     18.6988,     19.0198,     18.8243,     18.5713,
    18.783,      19.1828,     18.9336,     18.9329,     19.5432,
    19.7237,     19.6811,     20.0416,     20.5323,     20.226,
    20.6307,     20.5913,     20.5765,     21.0088,     20.9304,
    20.8995,     21.2155,     21.2391,     21.0379,     21.302,
    21.3278,     21.4008,     21.4159,     21.154,      21.0409,
    21.574,      21.7671,     20.9683,     20.6364,     21.1343,
    20.529,      20.7024,     20.3128,     20.4543,     20.0348,
    19.7623,     19.267,      19.7061,     19.0545,     18.789,
    18.6538,     18.3058,     18.4159,     18.3199,     17.8655,
    17.8587,     18.7423,     19.0077,     19.7289,     21.14,
    21.8391,     24.8123,     27.3283,     31.5723,     35.4982,
    40.3509,     45.8066,     42.2888,     20.0854,     0.000554527,
    0.0,         22.1612,     45.942,      48.7105,     44.0936,
    38.0783,     33.1369,     29.4953,     26.5005,     24.1645,
    22.4128,     21.2679,     20.4158,     19.8982,     19.2609,
    18.6974,     18.8339,     18.822,      18.9903,     19.5389,
    19.5559,     19.1424,     18.8473,     19.62,       20.1543,
    20.1727,     20.3901,     20.9581,     21.1977,     21.2041,
    21.2652,     21.9394,     20.8229,     21.6259,     21.5258,
    21.6898,     21.1491,     21.5399,     21.9052,     21.9047,
    21.8955,     21.4253,     22.239,      20.9841,     21.9567,
    21.3188,     21.8297,     21.3801,     21.3725,     21.1182,
    20.8165,     21.3061,     20.5649,     20.8605,     20.3616,
    19.9306,     19.9532,     19.7566,     19.6461,     19.2128,
    18.96,       18.4675,     18.373,      18.527,      18.623,
    18.302,      18.629,      19.5146,     20.1411,     21.2274,
    22.6809,     25.1278,     27.8746,     31.6541,     36.0726,
    41.5085,     46.3194,     43.9095,     20.9418,     0.000575481,
    0.0,         22.9718,     46.4385,     49.6784,     44.6114,
    39.4524,     33.2835,     30.2161,     26.7317,     24.6363,
    22.8719,     21.9208,     20.5886,     20.0183,     19.8746,
    19.2037,     19.4077,     19.3692,     19.3748,     19.6523,
    19.6186,     19.4494,     19.3178,     19.9005,     19.7203,
    21.2684,     20.0507,     20.6003,     21.0485,     21.3865,
    21.4852,     21.3529,     21.7551,     21.6174,     21.6359,
    22.1392,     22.3289,     22.5507,     22.473,      22.4008,
    22.401,      22.096,      22.1526,     22.3181,     22.329,
    21.7232,     22.1439,     21.9459,     21.5737,     21.9233,
    21.5318,     21.1786,     20.7007,     20.6392,     20.858,
    20.3043,     20.1713,     20.1942,     20.2928,     19.5794,
    19.366,      18.7602,     18.7906,     18.295,      18.2886,
    18.5138,     18.9691,     19.4067,     20.5441,     21.3671,
    23.2515,     25.6862,     28.3858,     32.401,      36.2769,
    43.1613,     48.8629,     45.9644,     21.6766,     0.000607966,
    0.0,         23.7899,     48.7566,     51.2691,     45.5302,
    39.2163,     34.4785,     30.503,      27.6601,     25.15,
    23.5104,     22.7016,     21.6188,     20.5314,     20.1845,
    19.7649,     19.7647,     19.3404,     19.6892,     19.8216,
    20.2277,     20.4505,     20.4784,     20.6297,     20.9838,
    20.8761,     21.3867,     21.4987,     21.4427,     21.0738,
    21.7049,     21.6017,     22.1702,     22.1786,     22.0111,
    22.1958,     22.146,      22.8559,     22.3349,     22.8145,
    22.7005,     22.0274,     22.643,      22.3547,     22.3039,
    22.7359,     22.1788,     22.7785,     21.7357,     22.2255,
    21.4617,     21.2257,     21.315,      21.0428,     20.9675,
    20.5059,     20.8257,     20.225,      20.0437,     20.1744,
    19.6095,     19.7758,     19.7726,     19.2072,     18.9917,
    19.4181,     19.9438,     19.8061,     20.4837,     22.3008,
    23.7685,     26.6793,     28.4012,     32.7113,     38.3741,
    44.458,      50.4126,     47.7474,     22.5697,     0.000628198,
    0.0,         25.1533,     50.6646,     53.4984,     46.8059,
    40.4846,     35.2731,     31.3793,     28.3659,     25.5107,
    23.6844,     22.4965,     21.6973,     21.5425,     20.7602,
    20.182,      19.8323,     19.9849,     19.4801,     19.9276,
    20.3128,     19.83,       20.8263,     21.0643,     21.0411,
    20.9548,     20.9849,     21.7677,     21.8188,     22.1129,
    21.9892,     21.993,      22.8178,     23.1811,     22.817,
    22.6623,     22.7525,     22.8868,     22.9386,     23.2369,
    22.8548,     22.6945,     22.8113,     22.9631,     23.1012,
    22.5493,     22.801,      22.8951,     22.2104,     22.3445,
    21.6357,     22.3154,     22.1669,     21.3642,     21.7594,
    20.9489,     21.3838,     20.7341,     20.5523,     20.4804,
    19.7243,     19.7774,     19.0655,     19.7438,     19.3274,
    19.3743,     19.9106,     20.5369,     21.1419,     22.5419,
    24.2363,     26.7074,     29.9141,     33.8987,     38.9724,
    45.4486,     50.9883,     48.3509,     24.269,      0.000655807,
    0.0,         25.973,      52.5645,     54.6317,     48.973,
    42.1354,     36.8142,     31.259,      28.6218,     26.2986,
    24.0503,     23.005,      21.9899,     20.8123,     20.6442,
    20.8048,     20.3789,     20.5245,     20.3169,     20.9399,
    20.6241,     20.9275,     21.297,      21.3028,     21.4393,
    22.0933,     22.083,      22.5974,     22.2879,     22.7286,
    22.4208,     22.3042,     23.035,      23.1808,     23.3662,
    23.1072,     23.3668,     22.9452,     23.6874,     23.5838,
    23.43,       23.4358,     23.5068,     23.1557,     23.3014,
    23.0648,     22.8439,     23.0643,     22.5709,     22.2521,
    23.2495,     22.04,       22.0834,     22.0322,     22.0966,
    21.9034,     21.6884,     20.5893,     20.8599,     20.8424,
    20.474,      19.7172,     20.3978,     19.9742,     19.9176,
    19.6255,     20.2965,     21.0804,     21.5153,     23.0369,
    24.5089,     27.2488,     30.0037,     34.8628,     40.5411,
    46.2938,     53.282,      50.1677,     24.8709,     0.000678893,
    0.0,         27.4823,     54.7351,     55.8649,     50.303,
    43.3768,     37.1562,     32.4688,     28.9567,     26.6614,
    24.644,      23.2263,     22.5224,     21.4026,     21.3918,
    20.8506,     20.9499,     20.8406,     20.9455,     21.0816,
    20.88,       21.431,      21.5477,     21.8188,     21.926,
    21.6343,     22.3368,     22.2326,     22.2031,     23.3418,
    23.313,      23.1574,     22.697,      23.5269,     23.4721,
    23.4834,     23.106,      23.9323,     23.5159,     23.6312,
    23.6561,     23.7255,     23.7521,     23.1697,     23.7424,
    23.5453,     23.5074,     23.8602,     23.1149,     23.3487,
    23.0332,     23.2152,     23.0432,     22.5367,     22.4807,
    22.874,      22.2625,     21.6263,     21.4472,     20.8107,
    20.342,      20.4695,     19.7861,     19.9811,     20.5891,
    20.5003,     20.3912,     21.0206,     22.5032,     23.2193,
    25.2827,     27.9516,     30.9252,     35.1612,     41.2919,
    47.9567,     53.3312,     52.1391,     26.0826,     0.000695202,
    0.0,         28.3533,     55.595,      57.5162,     50.4576,
    44.6752,     38.1791,     33.98,       29.1856,     26.8943,
    25.1467,     23.6227,     22.7255,     21.792,      22.211,
    21.5019,     20.9553,     21.2378,     21.1922,     21.5586,
    21.4178,     21.493,      22.4356,     22.4779,     22.7865,
    22.5841,     22.6551,     22.9214,     23.3152,     23.0978,
    23.5668,     24.3295,     23.5321,     24.1694,     24.2523,
    24.2071,     24.5971,     24.269,      23.792,      24.389,
    24.5048,     24.4343,     24.1466,     24.4709,     24.3708,
    24.359,      24.4053,     23.6477,     23.6324,     23.9739,
    23.618,      23.1492,     23.2655,     22.9051,     22.9848,
    22.4809,     22.8652,     21.8802,     21.6803,     21.8081,
    21.067,      21.0297,     20.6023,     20.3781,     20.4408,
    20.7411,     20.6831,     21.942,      22.4556,     24.4814,
    25.9683,     27.6108,     32.3081,     36.2877,     41.3751,
    49.0453,     57.3134,     55.0339,     26.9032,     0.000746743,
    0.0,         29.8906,     56.8097,     59.7394,     52.658,
    44.8447,     38.8848,     35.0251,     30.6396,     27.3647,
    25.2535,     24.2401,     22.9692,     21.7021,     21.5449,
    21.8491,     21.3208,     21.4909,     21.3886,     22.2684,
    21.9012,     22.3869,     22.746,      22.7621,     22.8599,
    22.758,      22.9187,     23.5599,     23.5132,     23.3501,
    24.2499,     23.7931,     24.7461,     24.6562,     24.1627,
    24.4907,     24.936,      25.3459,     24.6832,     24.1899,
    25.0645,     24.6165,     24.7105,     25.0109,     24.3987,
    24.9728,     24.7061,     24.4273,     24.1037,     24.2468,
    23.8376,     24.2928,     23.4331,     23.3016,     23.3284,
    22.8233,     23.044,      22.6183,     21.8069,     21.8017,
    21.5757,     21.4699,     21.2125,     20.7132,     21.4929,
    21.2753,     21.4509,     21.462,      22.9241,     23.9264,
    26.2754,     28.4474,     32.6663,     36.8562,     42.3835,
    49.6034,     58.2266,     55.8583,     28.5645,     0.000777184,
    0.0,         31.0518,     59.7328,     60.9655,     53.5849,
    45.8835,     39.8674,     34.5993,     30.9244,     28.1589,
    26.3688,     24.7207,     23.1933,     22.5379,     22.7786,
    22.0647,     21.6469,     21.803,      22.0701,     21.9304,
    22.6516,     22.5329,     22.9191,     23.0046,     22.9757,
    22.94,       23.3392,     24.2235,     23.9276,     24.0154,
    24.1912,     24.978,      24.1315,     24.331,      24.8386,
    25.5021,     25.5556,     24.9793,     25.9219,     24.9488,
    24.3441,     25.315,      24.9172,     25.3097,     24.5415,
    25.0062,     25.0796,     24.9853,     24.797,      24.9569,
    25.0365,     23.9183,     23.515,      24.0452,     23.9295,
    23.178,      23.3553,     22.2608,     22.1136,     22.2582,
    21.8543,     21.3808,     21.5243,     21.1648,     21.3765,
    21.6745,     21.4152,     22.6479,     23.6877,     24.4329,
    26.7144,     29.3928,     32.4417,     37.3663,     44.1197,
    51.3384,     59.9932,     58.3031,     29.8857,     0.000814377,
    0.0,         32.1361,     61.951,      62.7645,     55.4385,
    47.4029,     41.1399,     35.3852,     30.807,      28.5779,
    26.5731,     25.1537,     24.1771,     23.0156,     22.2649,
    22.4707,     22.696,      22.2021,     22.4394,     22.364,
    22.5866,     22.6067,     22.5606,     23.0001,     22.932,
    23.9583,     23.7912,     24.2457,     24.6057,     25.0832,
    24.5705,     25.2753,     24.8808,     25.1484,     25.2694,
    25.9479,     25.5233,     25.7675,     25.3949,     25.2842,
    25.8066,     25.475,      25.6463,     25.3097,     25.4713,
    25.0989,     25.6231,     24.9042,     24.7033,     25.2089,
    24.7624,     24.414,      24.7549,     24.3158,     23.4047,
    23.2421,     23.3847,     23.1757,     22.6538,     23.1632,
    22.3398,     22.3534,     21.924,      21.6332,     21.4099,
    21.7809,     22.1988,     23.0222,     23.718,      24.7422,
    26.726,      29.9404,     33.4419,     38.6475,     45.6283,
    53.2559,     60.149,      60.1629,     30.7399,     0.000846442,
    0.0,         33.1075,     63.7146,     63.5448,     56.4654,
    47.5906,     41.0356,     36.1647,     32.3025,     29.0643,
    26.8869,     26.1441,     24.3369,     23.6038,     23.2924,
    22.3719,     22.2506,     22.9309,     22.5156,     22.5673,
    23.1324,     23.6331,     23.5411,     23.684,      24.201,
    24.9288,     24.5251,     24.8446,     24.7952,     25.2782,
    25.4845,     25.3524,     25.4689,     25.7728,     26.4155,
    26.3423,     25.8027,     26.4665,     25.9939,     26.2516,
    26.1055,     26.0755,     25.4611,     26.0755,     25.2794,
    26.0399,     25.7755,     25.7528,     25.1377,     25.1252,
    25.6788,     24.8743,     24.7306,     24.3593,     24.5739,
    24.1847,     24.0588,     23.6731,     23.2697,     22.8744,
    22.8002,     22.2389,     22.3807,     22.0426,     22.2691,
    22.9061,     22.1452,     23.2097,     24.1633,     26.0982,
    28.3974,     30.143,      35.3669,     40.3764,     46.1243,
    53.6791,     62.8294,     62.245,      31.2468,     0.000893153,
    0.0,         34.1187,     65.0076,     65.4637,     57.1754,
    48.6555,     41.9196,     37.2921,     32.8628,     29.7953,
    27.7637,     26.261,      24.5071,     24.0395,     23.4125,
    23.0596,     22.8424,     23.2706,     22.8671,     23.2337,
    23.5733,     24.1141,     24.0353,     24.3222,     24.3583,
    24.2105,     24.6353,     25.7271,     24.9192,     24.9725,
    25.1463,     26.1432,     25.5103,     25.5063,     26.0542,
    25.3824,     26.3047,     26.3112,     27.0734,     27.0187,
    26.2437,     26.4975,     26.9358,     25.9651,     26.4242,
    25.8959,     26.3318,     26.4497,     26.4995,     25.7976,
    26.188,      25.5866,     24.9085,     24.7425,     24.7148,
    24.3782,     24.5449,     24.0969,     23.2724,     23.4123,
    22.9255,     22.1948,     22.6539,     22.2284,     22.3678,
    22.4628,     22.4188,     24.2159,     24.5676,     26.1629,
    28.2421,     31.4904,     34.2571,     40.2344,     47.1288,
    55.7044,     64.8237,     63.8208,     33.0234,     0.000938221,
    0.0,         35.6335,     67.9785,     67.8939,     58.2508,
    49.7906,     42.708,      37.2446,     34.0716,     30.2715,
    28.3564,     26.5691,     24.8251,     24.7546,     23.863,
    23.9194,     23.5584,     23.8144,     23.6499,     23.6798,
    23.9703,     24.3372,     24.8987,     24.6162,     25.3048,
    24.7743,     25.418,      25.2682,     25.565,      25.639,
    26.9508,     26.5335,     27.0771,     26.8907,     26.7315,
    26.9966,     27.2298,     27.1625,     27.0927,     27.1302,
    27.3026,     27.1574,     26.735,      26.7687,     26.9875,
    26.7282,     25.9983,     26.5748,     26.0519,     26.0455,
    26.3135,     25.5043,     25.8325,     26.4635,     25.6334,
    25.0977,     25.0794,     24.5616,     24.0166,     23.9642,
    23.7603,     23.7965,     22.8598,     22.8344,     22.9919,
    23.1223,     22.9321,     23.2753,     24.3196,     26.6443,
    28.9214,     31.5429,     35.3769,     41.6598,     46.8854,
    57.9261,     65.6353,     65.5663,     34.5953,     0.000961878,
    0.0,         36.8623,     69.7438,     69.063,      59.9691,
    50.9248,     43.6146,     37.5259,     34.4658,     31.4381,
    28.7641,     26.7205,     25.7304,     25.1689,     24.6961,
    24.1827,     23.7824,     23.739,      23.4506,     24.1398,
    24.3852,     24.9454,     25.0759,     25.5669,     25.0934,
    25.7853,     25.517,      26.7929,     26.3888,     26.1044,
    26.8321,     25.9345,     26.4707,     26.7662,     27.4709,
    27.2554,     27.3102,     27.8975,     27.6114,     27.3543,
    26.9617,     28.1306,     27.7689,     27.7723,     26.9321,
    27.0129,     27.6033,     27.112,      26.8508,     27.5266,
    26.9151,     26.963,      26.165,      25.9733,     25.454,
    25.4891,     25.4113,     24.8356,     23.5984,     24.8533,
    23.849,      23.2799,     23.4827,     23.1759,     23.4285,
    23.6984,     23.315,      24.2511,     25.8098,     26.9653,
    29.6123,     32.3169,     36.3615,     42.3669,     48.1968,
    57.9153,     67.3611,     68.1383,     36.0073,     0.000978777,
    0.0,         38.8355,     70.8817,     70.9784,     61.3735,
    52.4027,     44.8078,     38.4132,     34.837,      31.5741,
    29.4632,     26.7659,     25.9963,     25.355,      24.926,
    24.5118,     23.9356,     24.1877,     24.5854,     24.6633,
    24.3323,     24.7892,     24.8277,     25.4692,     26.2391,
    26.2974,     26.1317,     26.6254,     27.3455,     27.4842,
    27.4037,     27.3478,     27.4562,     27.086,      28.0957,
    27.9599,     27.7979,     28.0946,     28.409,      27.7848,
    28.0169,     28.0712,     27.746,      27.4425,     28.0133,
    27.6028,     28.1627,     27.5542,     27.209,      27.4522,
    27.4782,     27.0231,     26.4063,     26.9094,     25.6972,
    26.5616,     25.8889,     24.3738,     25.1049,     25.1963,
    24.2589,     23.849,      24.1476,     23.4821,     23.3203,
    23.808,      24.2372,     25.1473,     26.3876,     27.4957,
    29.7291,     32.6354,     37.9043,     42.8712,     50.6817,
    59.4085,     68.9889,     70.904,      37.5811,     0.00102523,
    0.0,         40.2248,     75.3077,     74.3326,     63.9912,
    53.3257,     45.6943,     40.2439,     35.7339,     31.9344,
    30.1667,     27.6313,     26.9599,     25.8449,     25.4636,
    25.1119,     25.0694,     24.4428,     24.6268,     25.2136,
    25.0234,     24.6588,     25.2398,     25.6506,     26.4175,
    27.0891,     26.7353,     26.8363,     27.2772,     26.9689,
    27.9124,     27.4369,     27.2485,     27.2965,     28.2492,
    28.7009,     28.2813,     27.6891,     28.2371,     28.9346,
    28.4156,     28.7785,     28.6611,     28.8725,     28.4794,
    28.3665,     28.2888,     28.2929,     27.963,      28.2963,
    27.7966,     27.3552,     27.754,      27.2292,     26.7609,
    26.5887,     25.9538,     25.0905,     25.856,      24.8545,
    24.5476,     25.1554,     24.4778,     24.2671,     24.3099,
    24.7825,     24.7623,     25.0289,     26.6151,     28.0525,
    30.313,      33.4377,     37.9386,     43.2786,     51.452,
    60.8978,     71.36,       72.1128,     39.1396,     0.00107145,
    0.0,         40.8896,     76.2407,     76.1731,     64.664,
    55.3471,     46.8257,     40.0898,     35.6466,     32.3343,
    29.1889,     28.5838,     26.8283,     26.0835,     25.0286,
    24.9065,     25.4416,     25.1688,     25.283,      25.7415,
    25.7439,     26.1185,     26.5438,     26.4883,     26.2528,
    26.997,      27.8536,     27.0935,     27.0572,     27.9037,
    28.289,      28.3855,     27.987,      28.4671,     28.1234,
    28.2367,     28.8395,     29.3384,     29.1314,     28.5523,
    28.3875,     29.0874,     28.36,       28.5183,     28.5066,
    28.8019,     28.3628,     28.3982,     28.6271,     27.9166,
    27.9261,     27.537,      28.4855,     26.8575,     27.5476,
    26.5134,     25.9992,     26.5286,     25.6757,     25.7787,
    25.8289,     24.9366,     24.3529,     24.7349,     24.3353,
    25.6814,     25.0143,     25.6211,     27.1115,     28.3517,
    30.7174,     34.4015,     37.2345,     45.7022,     51.0605,
    62.9223,     73.9609,     75.2656,     40.3128,     0.0011431,
    0.0,         43.4033,     78.6611,     78.3512,     67.0728,
    55.4385,     47.2231,     41.192,      37.091,      33.3391,
    30.7224,     28.2065,     27.2598,     26.9629,     26.1961,
    26.2043,     25.5439,     25.5644,     26.0045,     26.023,
    26.3131,     26.0132,     26.7401,     26.2134,     26.7747,
    26.9713,     26.8156,     27.4612,     28.328,      28.0495,
    28.3577,     29.296,      28.9033,     29.4331,     28.7152,
    28.9607,     29.1722,     29.251,      28.9944,     29.329,
    29.6655,     29.3468,     29.4981,     29.0564,     29.5399,
    29.1482,     28.8844,     28.8965,     28.9721,     28.6433,
    28.2658,     28.568,      27.7176,     27.6097,     27.2115,
    27.1922,     26.742,      26.7637,     26.3121,     26.1046,
    26.1359,     25.771,      25.264,      25.6887,     24.6091,
    24.7675,     25.7872,     26.1726,     27.2905,     28.5019,
    31.9756,     34.3559,     39.4123,     44.7546,     52.462,
    64.7081,     74.0159,     76.9609,     41.4433,     0.00116176,
    0.0,         45.5193,     81.334,      79.4993,     68.1017,
    56.6892,     48.4058,     41.4524,     37.1864,     33.9292,
    31.1937,     29.1669,     28.0402,     27.0836,     26.8045,
    26.0203,     26.3696,     25.9476,     27.0511,     26.381,
    26.3438,     27.3249,     27.3796,     27.536,      27.6968,
    28.417,      27.6524,     28.247,      28.659,      28.8707,
    28.9339,     29.1846,     29.8459,     28.8155,     29.6471,
    29.5534,     29.4075,     29.5636,     29.1036,     29.8642,
    29.8717,     29.5979,     30.171,      30.1084,     29.5556,
    29.388,      29.8768,     29.7461,     30.0934,     29.4092,
    29.3352,     28.6969,     28.098,      28.2398,     28.2242,
    27.5947,     27.7119,     27.2795,     26.0985,     26.6137,
    25.6494,     26.1502,     25.1746,     25.291,      25.5375,
    25.1745,     25.6914,     27.0389,     27.2676,     29.4393,
    32.2352,     34.9165,     39.9741,     47.1047,     54.4419,
    65.8707,     77.2327,     80.0698,     44.0395,     0.00121734,
    0.0,         46.9813,     83.367,      79.7005,     69.9626,
    59.0702,     48.8925,     42.7708,     37.836,      34.2368,
    31.5802,     29.518,      28.4885,     27.2227,     26.3929,
    26.6094,     26.8342,     26.245,      26.3964,     26.7521,
    25.9778,     27.1667,     27.4278,     28.5467,     28.2985,
    28.0635,     28.8009,     28.7556,     28.7569,     29.214,
    29.4056,     29.2997,     30.0447,     30.4024,     30.6912,
    29.8187,     30.3083,     30.1225,     30.7826,     30.4003,
    30.992,      30.8993,     30.4605,     30.3037,     30.8708,
    30.6141,     29.5402,     30.1792,     29.4241,     29.4843,
    29.5329,     29.6505,     29.1165,     28.7814,     28.0087,
    28.0223,     27.7006,     27.2389,     27.4635,     26.1157,
    27.1219,     26.6273,     25.969,      25.7233,     25.7567,
    25.9065,     26.4602,     26.9841,     28.3707,     31.0181,
    32.6932,     35.7635,     40.7216,     47.6823,     57.0601,
    66.5407,     76.5274,     82.9094,     44.9347,     0.00125805,
    0.0,         48.4253,     86.1165,     84.1411,     70.2858,
    59.0359,     50.7344,     43.5334,     38.0848,     34.5815,
    32.2969,     30.3694,     28.5791,     27.5603,     27.3257,
    27.2011,     27.0545,     27.0658,     27.3791,     27.0578,
    26.8765,     27.9782,     28.0769,     28.6793,     28.9848,
    29.0129,     28.5175,     29.5498,     29.9827,     29.702,
    30.2792,     30.8679,     29.5758,     30.2744,     30.7946,
    30.453,      30.5261,     30.8587,     30.0709,     31.152,
    30.6863,     30.9499,     30.8855,     30.8707,     30.616,
    29.9786,     30.309,      30.6782,     30.0165,     30.325,
    30.0875,     30.0421,     30.039,      29.6169,     28.7829,
    28.3537,     28.9921,     27.8642,     27.788,      26.5971,
    26.6838,     26.9206,     26.0668,     26.111,      26.1125,
    26.7975,     27.0231,     28.057,      27.9552,     30.5517,
    33.0065,     36.4991,     41.8135,     48.4726,     56.89,
    66.8721,     81.0428,     84.1732,     46.2379,     0.00130236,
    0.0,         49.5467,     88.6113,     86.101,      72.7695,
    61.5304,     51.2403,     44.0584,     39.618,      34.838,
    32.7763,     30.8813,     29.4768,     28.4975,     28.0481,
    27.6754,     27.3516,     27.2999,     27.1913,     27.7892,
    28.0822,     28.3682,     28.6855,     28.5305,     29.0044,
    28.7532,     29.4144,     28.801,      29.5033,     29.6935,
    30.0905,     29.8698,     30.3665,     31.0502,     30.9585,
    31.5764,     31.5389,     31.4678,     31.6298,     31.2748,
    31.3012,     31.5404,     30.9571,     31.2074,     30.9285,
    31.1395,     30.8884,     30.4542,     30.9719,     30.679,
    30.2819,     29.267,      29.5332,     29.479,      30.1113,
    29.4009,     29.1326,     27.587,      27.4349,     27.8387,
    27.3082,     27.7701,     26.7388,     26.4227,     26.5898,
    27.115,      27.7116,     28.075,      29.3271,     31.3606,
    33.1857,     37.6016,     41.9765,     49.8747,     57.5936,
    71.1199,     82.7151,     86.4076,     49.64,       0.00135168,
    0.0,         51.5765,     91.3042,     87.1432,     74.473,
    61.4584,     52.1324,     46.3206,     39.852,      35.9049,
    33.0672,     30.9042,     29.5874,     28.6845,     28.2759,
    27.8822,     28.0824,     27.4461,     27.8015,     28.2853,
    27.8339,     28.7817,     28.9652,     29.3956,     30.1724,
    30.3027,     29.9376,     30.974,      31.1771,     31.3012,
    31.2371,     31.6521,     31.202,      30.9932,     31.0645,
    31.3214,     31.433,      31.5394,     31.8957,     32.0548,
    31.8014,     31.8371,     31.6782,     31.6847,     31.3624,
    31.509,      31.576,      31.7427,     30.5543,     31.0854,
    32.1186,     30.7504,     30.3257,     30.0617,     30.2399,
    29.8475,     29.2759,     29.3469,     28.8248,     27.9493,
    27.3806,     27.2629,     27.5266,     27.0371,     28.0585,
    27.52,       27.6745,     28.4483,     29.7566,     31.1495,
    34.813,      37.3163,     43.5464,     48.6212,     59.2647,
    71.7152,     84.5636,     88.5278,     50.748,      0.00139542,
    0.0,         53.7564,     93.1854,     88.9668,     75.3089,
    62.6985,     53.9856,     46.0161,     40.9658,     35.8914,
    33.7869,     31.2282,     29.757,      28.9144,     28.5302,
    27.9035,     27.3429,     27.9385,     28.4166,     29.1025,
    28.796,      28.8731,     28.5341,     28.9864,     30.2357,
    29.711,      30.3188,     30.4484,     30.2272,     31.3202,
    31.4729,     32.4727,     31.9314,     31.1467,     32.1314,
    32.3653,     32.0397,     32.2973,     31.7338,     32.7167,
    32.2146,     32.5026,     31.8374,     32.3152,     33.4636,
    32.3485,     32.1276,     32.2406,     31.8021,     32.4303,
    31.9269,     30.4776,     31.5059,     30.502,      30.2463,
    30.8329,     29.9898,     28.904,      28.951,      28.8761,
    27.6324,     28.4736,     27.7766,     27.5578,     27.082,
    27.997,      28.1466,     28.5948,     30.1629,     33.3042,
    35.053,      39.1337,     42.7079,     51.0839,     61.3306,
    74.5736,     87.5456,     93.359,      52.4983,     0.00145447,
    0.0,         57.4522,     96.9321,     91.5383,     77.7031,
    64.5387,     55.2032,     46.847,      41.5603,     38.209,
    34.0045,     32.4398,     31.0136,     29.7583,     28.767,
    29.0232,     28.6146,     28.7174,     29.2538,     28.954,
    28.5165,     29.7051,     29.7322,     30.1578,     31.047,
    30.8117,     31.3178,     31.8781,     31.0583,     32.0897,
    31.7885,     32.43,       32.6122,     32.3882,     32.5975,
    32.1888,     32.4208,     32.5,        33.0741,     32.2481,
    32.6749,     33.1183,     33.0444,     33.0983,     32.535,
    32.8768,     32.4713,     31.4251,     31.947,      32.4248,
    31.2368,     30.9579,     31.1235,     30.8634,     31.2586,
    30.2903,     30.7074,     29.3803,     29.3889,     27.9682,
    29.212,      28.4492,     27.7661,     28.3165,     27.8875,
    28.626,      29.3,        29.3996,     30.5243,     33.1335,
    35.2331,     38.6997,     46.0355,     52.0251,     62.1899,
    77.9987,     89.2395,     94.707,      55.1827,     0.00152485,
    0.0,         57.3852,     99.7329,     93.8354,     78.3921,
    66.1678,     55.4845,     47.5989,     41.8471,     39.3885,
    35.1478,     32.1465,     30.6266,     30.2194,     29.3304,
    29.1503,     28.9228,     28.9861,     29.3434,     28.975,
    29.2431,     30.0644,     30.6827,     30.2127,     30.5042,
    31.8672,     32.0749,     32.4077,     31.5801,     32.0586,
    32.0307,     33.085,      33.0326,     33.0304,     32.8966,
    33.4169,     32.7319,     33.466,      33.0321,     32.3579,
    33.2094,     33.6946,     33.6732,     33.5429,     32.4647,
    33.5041,     32.9046,     33.3846,     32.3886,     32.0848,
    32.1034,     31.9753,     30.9411,     31.8601,     30.848,
    31.0093,     31.3324,     30.4375,     29.2508,     30.2918,
    29.1561,     28.4449,     28.4515,     28.3802,     28.4771,
    27.4068,     29.4184,     30.1604,     31.0721,     32.2627,
    36.1837,     41.1953,     44.966,      53.3216,     63.0819,
    77.8875,     92.3359,     96.6072,     57.2352,     0.00155217,
    0.0,         60.4438,     102.789,     95.6035,     79.6237,
    67.272,      56.4614,     48.9639,     43.0147,     38.5176,
    35.6601,     32.6373,     31.0844,     30.2205,     29.7442,
    30.0422,     29.4065,     29.2436,     29.6652,     29.1087,
    29.808,      30.8948,     30.9809,     31.2915,     30.0744,
    31.8307,     32.3148,     32.2739,     32.3199,     33.1061,
    32.8407,     33.3011,     33.0478,     32.6439,     33.4167,
    33.974,      33.3994,     33.2927,     34.7274,     33.8992,
    34.3174,     34.4822,     32.9792,     34.4708,     33.2306,
    33.8387,     33.3968,     33.987,      32.7669,     33.1136,
    33.3182,     33.3692,     32.2389,     31.73,       31.6376,
    31.6961,     31.3047,     30.7272,     31.0565,     30.2252,
    30.4536,     29.2213,     29.3717,     28.8627,     28.69,
    29.2069,     29.9748,     30.6424,     31.4864,     34.0349,
    35.9805,     41.6741,     46.9789,     53.4661,     63.788,
    79.1896,     94.0583,     99.7471,     58.4638,     0.00162509,
    0.0,         62.8041,     105.33,      97.9743,     83.0982,
    68.2429,     57.1373,     48.7105,     43.2692,     38.8036,
    36.1065,     34.5456,     32.1394,     30.9679,     29.7287,
    29.4842,     30.4743,     29.7956,     30.2757,     30.3746,
    30.3316,     31.1685,     32.2759,     31.9405,     32.3424,
    32.843,      32.0366,     32.0406,     32.2584,     34.7684,
    34.3841,     33.0984,     32.889,      34.183,      32.806,
    34.5361,     33.7522,     35.0949,     34.2605,     33.7292,
    34.8646,     34.0812,     34.107,      35.1383,     34.4687,
    34.2777,     34.5624,     33.8086,     33.3478,     33.361,
    33.1379,     32.3808,     32.9565,     32.7183,     31.9848,
    32.3187,     31.5664,     31.3768,     30.8742,     30.8924,
    29.7606,     30.5367,     29.8089,     29.3034,     29.2352,
    29.5892,     30.0582,     31.0013,     32.0147,     34.9241,
    37.052,      42.536,      47.7072,     54.2364,     66.4352,
    81.8807,     95.6642,     102.93,      61.5006,     0.00172906,
    0.0,         64.9668,     109.051,     101.104,     84.4313,
    69.3278,     59.5822,     50.7318,     44.9502,     39.4697,
    36.762,      34.2105,     32.4266,     31.9948,     30.4877,
    30.6083,     31.5436,     30.3014,     30.2796,     30.9037,
    31.3211,     31.1514,     32.1904,     31.8961,     31.7213,
    32.0276,     32.952,      33.253,      33.5698,     33.258,
    33.122,      33.8656,     33.8427,     34.4004,     34.4769,
    34.1374,     34.2113,     35.1139,     34.4105,     35.2895,
    35.4108,     35.0122,     34.5913,     34.6576,     34.3645,
    34.9677,     34.7597,     34.9711,     33.6308,     34.2292,
    33.6283,     33.166,      33.7533,     33.3362,     32.9595,
    32.1099,     32.4854,     31.4979,     31.9314,     30.6737,
    31.0134,     30.6956,     30.1703,     29.9399,     29.9665,
    29.466,      29.6666,     30.7631,     33.1859,     35.2291,
    38.0285,     43.537,      47.8927,     56.2631,     67.9246,
    82.2279,     98.427,      105.833,     63.345,      0.0017489,
    0.0,         68.0244,     111.344,     102.267,     85.795,
    70.363,      59.947,      51.3823,     44.8115,     40.3744,
    37.2984,     34.4392,     33.0062,     32.3117,     31.081,
    30.8999,     31.0646,     30.3055,     31.0691,     32.1102,
    31.1607,     31.3323,     32.0451,     33.1747,     33.3703,
    33.5832,     34.2149,     33.2287,     33.9576,     34.7731,
    34.3054,     34.4925,     35.0772,     33.8183,     34.8365,
    35.2321,     34.7637,     35.7537,     34.8794,     35.1268,
    35.8692,     35.6286,     36.0293,     35.3971,     35.9657,
    35.5339,     35.0892,     35.645,      34.6251,     35.1887,
    35.1345,     33.2522,     34.1337,     33.8491,     34.2811,
    33.2241,     33.0331,     31.0317,     31.3708,     31.614,
    30.8371,     31.4295,     30.7396,     30.7793,     29.8907,
    30.9762,     31.3705,     31.5111,     33.8642,     35.4489,
    38.5082,     41.8976,     48.3733,     56.6482,     67.5554,
    82.3988,     100.166,     109.925,     65.1861,     0.00185782,
    0.0,         70.0618,     113.969,     105.89,      87.8617,
    72.8943,     60.4391,     52.2422,     46.6503,     40.551,
    38.8297,     36.0645,     33.8238,     32.1756,     31.644,
    31.0702,     31.2668,     31.3515,     31.0499,     32.1017,
    32.4122,     32.7211,     33.0142,     33.4614,     33.3573,
    32.8604,     34.8633,     34.4861,     34.7378,     34.774,
    34.4846,     35.1325,     35.087,      35.2932,     36.275,
    36.3328,     35.8824,     36.002,      36.7491,     35.822,
    36.0177,     35.4213,     36.1831,     36.163,      35.3537,
    35.0498,     35.9295,     35.1046,     35.8752,     35.4004,
    34.9153,     35.088,      33.7507,     34.334,      33.5452,
    33.0404,     33.2637,     32.5186,     33.2232,     32.236,
    31.7227,     31.0496,     30.743,      30.8069,     30.9671,
    30.5739,     31.149,      31.8803,     33.4254,     35.5408,
    40.0262,     43.2342,     48.8458,     58.0182,     70.093,
    85.5542,     102.256,     112.76,      68.3339,     0.00188589,
    0.0,         72.7442,     116.866,     107.729,     89.3928,
    74.215,      61.8251,     53.0877,     46.811,      41.831,
    37.9642,     35.4228,     34.0257,     33.6579,     32.3669,
    32.2659,     31.3711,     31.3149,     31.8431,     31.7856,
    32.1012,     33.2237,     32.523,      33.155,      33.7967,
    34.3562,     35.3056,     34.9609,     34.3848,     35.1007,
    35.1613,     34.991,      35.7292,     37.3811,     35.8052,
    35.1214,     36.892,      36.4388,     36.4077,     36.6461,
    36.6167,     37.4363,     36.765,      36.5739,     36.1289,
    36.3084,     35.6431,     35.0029,     35.9682,     36.1826,
    35.6073,     34.924,      34.5898,     34.5536,     33.2751,
    33.5225,     33.6526,     32.6191,     32.873,      33.0435,
    31.6621,     31.8899,     31.9696,     31.7286,     31.8763,
    30.9045,     31.6156,     33.5327,     35.1473,     35.9817,
    39.5443,     44.5662,     50.6678,     59.5575,     72.4223,
    88.0371,     105.39,      115.361,     69.8664,     0.00195524,
    0.0,         74.8364,     120.847,     111.669,     93.4425,
    75.6532,     62.5785,     55.0702,     47.1519,     42.3506,
    38.5787,     36.9895,     34.5364,     33.5095,     32.6784,
    32.5861,     32.2908,     32.4114,     32.057,      33.3701,
    33.2672,     33.3768,     33.9719,     35.1165,     33.415,
    33.6952,     34.7275,     34.4382,     35.453,      36.5206,
    36.4321,     35.8744,     36.2806,     37.1474,     37.0982,
    36.9096,     37.1718,     37.1603,     36.9818,     37.6108,
    36.8875,     36.8922,     36.3282,     36.4559,     37.0324,
    36.9647,     35.9473,     36.6183,     37.0814,     36.8423,
    35.4836,     35.4017,     34.5167,     34.7471,     34.5399,
    34.3829,     34.4398,     33.2806,     33.3175,     33.3455,
    32.9732,     31.8092,     32.7868,     30.5899,     31.4387,
    31.4387,     31.5502,     33.4101,     35.3176,     36.7171,
    39.6251,     45.6218,     52.2923,     59.4464,     72.8055,
    89.3869,     106.916,     117.96,      72.9616,     0.00205999};

/** @name Interpolation objects for π+ + π+ -> π+ + π+ + γ and
    or π- + π- -> π- + π- + γ processes (same charge incoming pions)
 */
///@{
static std::unique_ptr<InterpolateDataLinear<double>>
    pipi_pipi_same_interpolation = nullptr;
static std::unique_ptr<InterpolateData2DSpline>
    pipi_pipi_same_dsigma_dk_interpolation = nullptr;
static std::unique_ptr<InterpolateData2DSpline>
    pipi_pipi_same_dsigma_dtheta_interpolation = nullptr;
///@}

/// Total π+ + π+ -> π+ + π+ + γ or π- + π- -> π- + π- + γ cross section
const std::initializer_list<double> BREMS_PIPI_PIPI_SAME_SIG = {
    0.000969136, 0.00233271, 0.0044325, 0.00734116, 0.0111208, 0.0158084,
    0.0213911,   0.0279234,  0.0353036, 0.0436295,  0.052903,  0.0628584,
    0.0735025,   0.0850656,  0.0971299, 0.109618,   0.122981,  0.136364,
    0.150406,    0.164681,   0.179279,  0.193688,   0.208438,  0.22276,
    0.237478,    0.252381,   0.267283,  0.28126,    0.295054,  0.309158,
    0.322475,    0.335719,   0.348017,  0.360952,   0.372779,  0.383829,
    0.395392,    0.406154,   0.415441,  0.424573,   0.43445,   0.442656,
    0.450083,    0.458042,   0.465637,  0.471963,   0.477361,  0.483862,
    0.48904,     0.49278,    0.497885,  0.501573,   0.505073,  0.508422,
    0.51032,     0.512307,   0.514865,  0.516576,   0.517638,  0.518468,
    0.519457,    0.519139,   0.519025,  0.519197,   0.519436,  0.517729,
    0.516813,    0.515962,   0.513931,  0.511979,   0.510757,  0.509431,
    0.507328,    0.505094,   0.503198,  0.500842,   0.497925,  0.495024,
    0.492382,    0.488794,   0.485695,  0.483221,   0.479006,  0.476568,
    0.472572,    0.469219,   0.466201,  0.462231,   0.45996,   0.454188,
    0.451276,    0.44794,    0.444003,  0.44046,    0.435907,  0.432249,
    0.427568,    0.423377,   0.41992,   0.41625,    0.411312,  0.40796,
    0.402374,    0.400231,   0.394959,  0.390681,   0.387765,  0.382141,
    0.378615,    0.373983,   0.370397,  0.364742,   0.361468,  0.357118,
    0.352876,    0.347501,   0.343018,  0.338776,   0.334663,  0.330169,
    0.325828,    0.320871,   0.318045,  0.312695,   0.30858,   0.30318,
    0.300002,    0.294108,   0.290207,  0.285444,   0.28073,   0.276583,
    0.271697,    0.267954,   0.263332,  0.260014,   0.255004,  0.249476,
    0.245746,    0.240914,   0.235927,  0.232454,   0.227968,  0.224168,
    0.219563,    0.214924,   0.209572,  0.206347,   0.201492,  0.197879,
    0.19377,     0.189777,   0.184646,  0.181022,   0.17686,   0.172101,
    0.168189,    0.164418,   0.160048,  0.156346,   0.151701,  0.147425,
    0.144319,    0.14077,    0.136744,  0.132802,   0.129487,  0.126027,
    0.122473,    0.12003,    0.105734,  0.0967245,  0.0881767, 0.0806414,
    0.0736338,   0.0686662,  0.0625226, 0.059212,   0.0558711, 0.0545981,
    0.0534943,   0.0548203,  0.0581519, 0.0618153,  0.0671825, 0.0737499,
    0.0840423,   0.0953125,  0.107564,  0.124422,   0.140641,  0.158166,
    0.180187,    0.203672,   0.230239,  0.258719,   0.293487,  0.324736,
    0.360223,    0.402352,   0.44618,   0.493426,   0.543468,  0.59611,
    0.651562,    0.710552,   0.775447,  0.843043,   0.916221,  0.993683,
    1.06823,     1.15544,    1.24577,   1.33637,    1.43103,   1.54083,
    1.63871,     1.75356,    1.86784,   1.98408,    2.11542,   2.25387,
    2.38954,     2.53789,    2.68264,   2.83624,    2.99077,   3.15083,
    3.3512,      3.52451,    3.70659,   3.89256,    4.07861,   4.28465,
    4.5158,      4.72078,    4.96091,   5.20374,    5.44222,   5.70749,
    5.95233,     6.21446,    6.50105,   6.78375,    7.08858,   7.36244,
    7.66918,     7.99648,    8.30407,   8.69105,    8.9521,    9.36577,
    9.74388,     10.0771,    10.4683,   10.8733,    11.2799,   11.7025,
    12.1981,     12.5713,    12.993,    13.5243,    13.9254,   14.4026,
    14.9083,     15.4436,    15.9466,   16.4654,    16.9964,   17.5169};

/// dSigma/dk for π+ + π+ -> π+ + π+ + γ or π- + π- -> π- + π- + γ
const std::initializer_list<double> BREMS_PIPI_PIPI_SAME_DIFF_SIG_K = {
    1.13979,      1.04659,      0.979941,     0.908479,    0.83872,
    0.770901,     0.705981,     0.652866,     0.603964,    0.562142,
    0.518838,     0.476759,     0.44213,      0.407105,    0.372979,
    0.34146,      0.317269,     0.292839,     0.266165,    0.243206,
    0.225638,     0.202312,     0.184841,     0.167223,    0.150936,
    0.137622,     0.123759,     0.112898,     0.10015,     0.0899651,
    0.0796331,    0.0706138,    0.0625697,    0.055136,    0.0486205,
    0.0418899,    0.0358479,    0.030785,     0.0262595,   0.0219495,
    0.0184843,    0.0145459,    0.011485,     0.00851845,  0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    2.44203,      2.26485,      2.07486,      1.91343,     1.79814,
    1.67452,      1.55125,      1.44125,      1.32329,     1.22997,
    1.14222,      1.05758,      0.974501,     0.897372,    0.820782,
    0.75883,      0.700411,     0.646694,     0.594185,    0.553557,
    0.509808,     0.465262,     0.428707,     0.383673,    0.360669,
    0.327134,     0.29477,      0.270599,     0.247425,    0.220887,
    0.199153,     0.18017,      0.160743,     0.144414,    0.128903,
    0.11469,      0.101656,     0.088936,     0.0779744,   0.0684238,
    0.058489,     0.0507143,    0.0431115,    0.0361823,   0.0297984,
    0.0245324,    0.0192732,    0.0149049,    0.0107182,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    4.34443,      3.98176,      3.59287,      3.37867,     3.15856,
    2.92255,      2.66948,      2.50803,      2.3411,      2.1621,
    2.01658,      1.8711,       1.72482,      1.59761,     1.46456,
    1.36331,      1.26013,      1.17346,      1.07653,     0.970024,
    0.905472,     0.845697,     0.782112,     0.711432,    0.646664,
    0.592561,     0.551094,     0.50128,      0.459228,    0.423741,
    0.383087,     0.341153,     0.310843,     0.286066,    0.258271,
    0.229178,     0.205452,     0.185923,     0.160483,    0.144832,
    0.127256,     0.111369,     0.0970118,    0.0840225,   0.0716981,
    0.061162,     0.0515938,    0.04309,      0.0346649,   0.0278917,
    0.0215989,    0.0158473,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    6.64339,      6.27407,      5.87805,      5.45342,     4.9981,
    4.5613,       4.27751,      3.97322,      3.64694,     3.33249,
    3.12853,      2.90983,      2.6839,       2.53439,     2.37408,
    2.1984,       2.00642,      1.83391,      1.71449,     1.58423,
    1.44184,      1.33642,      1.23938,      1.14235,     1.05048,
    0.968663,     0.893326,     0.811331,     0.74795,     0.685984,
    0.629986,     0.569578,     0.523337,     0.475506,    0.430021,
    0.389175,     0.351461,     0.316755,     0.283821,    0.252747,
    0.22221,      0.199247,     0.176341,     0.15456,     0.135478,
    0.118699,     0.101429,     0.0861132,    0.0734342,   0.0604568,
    0.0500852,    0.0408197,    0.0318675,    0.0243346,   0.0172691,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    10.1469,      9.08346,      8.34663,      7.87697,     7.37338,
    6.8334,       6.25439,      5.63354,      5.3044,      4.95778,
    4.5861,       4.18756,      3.93808,      3.67542,     3.39379,
    3.16561,      2.93487,      2.70643,      2.5122,      2.31095,
    2.15823,      1.99832,      1.84148,      1.70091,     1.57277,
    1.4134,       1.29657,      1.21619,      1.13843,     1.02821,
    0.933555,     0.84938,      0.788101,     0.715633,    0.657874,
    0.60271,      0.541381,     0.495124,     0.44304,     0.398154,
    0.353399,     0.314342,     0.281466,     0.251363,    0.220476,
    0.19323,      0.170227,     0.14718,      0.128172,    0.10845,
    0.0910237,    0.0762279,    0.0625771,    0.0503829,   0.0394871,
    0.0304993,    0.0220017,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    14.4294,      13.2614,      12.009,       10.666,      9.84556,
    9.28209,      8.6779,       8.03005,      7.33538,     6.62719,
    6.27094,      5.88895,      5.47935,      5.05133,     4.72874,
    4.38283,      4.01193,      3.73286,      3.44014,     3.16736,
    2.94335,      2.71749,      2.53207,      2.33765,     2.13893,
    1.97547,      1.8336,       1.7138,       1.57128,     1.43851,
    1.32053,      1.20671,      1.10316,      1.01284,     0.93512,
    0.854703,     0.768439,     0.692888,     0.630339,    0.578834,
    0.521763,     0.464957,     0.418316,     0.375954,    0.331028,
    0.297105,     0.259782,     0.228871,     0.198107,    0.171987,
    0.147959,     0.124747,     0.104818,     0.0861584,   0.0718372,
    0.0566973,    0.0440518,    0.0335621,    0.0236529,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    18.6076,      17.426,       16.1591,      14.8007,     13.344,
    11.8126,      11.1937,      10.5301,      9.81858,     9.05562,
    8.23751,      7.57047,      7.12534,      6.64805,     6.13627,
    5.68835,      5.34192,      4.97045,      4.60622,     4.29435,
    3.95995,      3.66254,      3.36552,      3.1033,      2.86657,
    2.64954,      2.43532,      2.25096,      2.07299,     1.90615,
    1.75286,      1.61661,      1.48146,      1.36392,     1.24026,
    1.13874,      1.05212,      0.955642,     0.872292,    0.787298,
    0.717625,     0.645361,     0.57758,      0.521166,    0.46521,
    0.417619,     0.370255,     0.328459,     0.288152,    0.253981,
    0.217441,     0.187417,     0.159599,     0.135012,    0.113033,
    0.0929097,    0.0753497,    0.0592522,    0.0449929,   0.0332303,
    0.0225935,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    23.4318,      22.1361,      20.7468,      19.257,      17.6596,
    15.9467,      14.1101,      12.9233,      12.221,      11.468,
    10.6606,      9.79474,      8.86792,      8.36321,     7.82202,
    7.24173,      6.61949,      6.22654,      5.80651,     5.35614,
    4.98944,      4.61475,      4.26728,      4.02572,     3.76058,
    3.43151,      3.11926,      2.92081,      2.7125,      2.49523,
    2.29601,      2.08845,      1.91599,      1.7804,      1.6309,
    1.4871,       1.35948,      1.24477,      1.144,       1.03709,
    0.934157,     0.85819,      0.774072,     0.694694,    0.625022,
    0.554363,     0.505229,     0.443839,     0.39057,     0.34883,
    0.307145,     0.263319,     0.228994,     0.192555,    0.164466,
    0.138942,     0.112178,     0.0927096,    0.0728535,   0.0566916,
    0.0424096,    0.0297751,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    27.3532,      26.1528,      24.8656,      23.4854,     22.0054,
    20.4185,      18.7169,      16.8924,      15.2967,     14.4569,
    13.5564,      12.5909,      11.5555,      10.4453,     9.72304,
    9.14034,      8.51552,      7.84555,      7.28709,     6.7564,
    6.18735,      5.7142,       5.30963,      4.87787,     4.5701,
    4.24009,      3.92264,      3.58955,      3.35943,     3.11026,
    2.81772,      2.59919,      2.40997,      2.23088,     2.03766,
    1.87953,      1.72877,      1.57375,      1.45087,     1.32282,
    1.18279,      1.10121,      0.980451,     0.893532,    0.813839,
    0.73415,      0.655109,     0.587401,     0.514995,    0.457723,
    0.401324,     0.353697,     0.309988,     0.264622,    0.230689,
    0.193944,     0.163534,     0.135441,     0.108763,    0.0883267,
    0.0672923,    0.050579,     0.0359384,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    32.6315,      31.3067,      29.8861,      28.3628,     26.7295,
    24.9781,      23.1002,      21.0865,      18.9273,     16.8315,
    15.9824,      15.0718,      14.0955,      13.0487,     11.9262,
    11.037,       10.3434,      9.5997,       8.80226,     8.14264,
    7.60218,      7.02266,      6.48891,      6.04212,     5.56304,
    5.17819,      4.78851,      4.43302,      4.08261,     3.77474,
    3.47322,      3.22403,      2.99541,      2.75814,     2.46429,
    2.28698,      2.11324,      1.92988,      1.79601,     1.60307,
    1.48875,      1.34711,      1.2334,       1.10613,     1.01193,
    0.917675,     0.80504,      0.736079,     0.651516,    0.59298,
    0.523742,     0.460065,     0.402764,     0.353914,    0.304638,
    0.261574,     0.221623,     0.184569,     0.152711,    0.12582,
    0.0987375,    0.0763006,    0.0583782,    0.0411301,   0.0272607,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    53.9377,      35.7563,      34.3221,      32.7842,     31.1351,
    29.3669,      27.4709,      25.4379,      23.2579,     20.9205,
    18.5541,      17.5978,      16.5725,      15.4731,     14.2942,
    13.0301,      12.0024,      11.2393,      10.4211,     9.54371,
    8.82361,      8.28698,      7.71158,      7.14866,     6.64296,
    6.10071,      5.67255,      5.25378,      4.88021,     4.52725,
    4.1665,       3.79515,      3.52954,      3.26839,     3.01988,
    2.80221,      2.55501,      2.34228,      2.14407,     1.95378,
    1.79995,      1.64953,      1.49144,      1.35817,     1.2355,
    1.11438,      1.01717,      0.903204,     0.815472,    0.73378,
    0.650222,     0.58315,      0.513861,     0.448201,    0.393711,
    0.339529,     0.286498,     0.244214,     0.204667,    0.170519,
    0.136817,     0.110235,     0.085122,     0.0627795,   0.045433,
    0.0289655,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    85.1671,      57.163,       38.483,       36.9314,     35.2676,
    33.4835,      31.5706,      29.5194,      27.32,       24.9616,
    22.4328,      19.936,       18.9549,      17.903,      16.7751,
    15.5656,      14.2688,      13.2254,      12.4226,     11.5618,
    10.6388,      9.83688,      9.15349,      8.42071,     7.76547,
    7.26924,      6.73715,      6.25578,      5.75865,     5.3242,
    4.91254,      4.51201,      4.11566,      3.88652,     3.59753,
    3.27925,      3.00501,      2.74833,      2.52804,     2.33489,
    2.12704,      1.94209,      1.79583,      1.63391,     1.50459,
    1.33321,      1.21013,      1.09646,      1.0001,      0.890384,
    0.799089,     0.707387,     0.631502,     0.558113,    0.490325,
    0.424273,     0.367841,     0.313661,     0.264555,    0.224477,
    0.183935,     0.148399,     0.116502,     0.0918532,   0.067864,
    0.0475221,    0.0311366,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    118.784,      89.9913,      59.1177,      41.8935,     40.1828,
    38.3484,      36.3815,      34.2725,      32.011,      29.5861,
    26.9859,      24.1979,      21.658,       20.5627,     19.3883,
    18.129,       16.7787,      15.3308,      14.2053,     13.2485,
    12.2226,      11.1225,      10.3145,      9.67334,     8.98588,
    8.36135,      7.80729,      7.21318,      6.69041,     6.13799,
    5.6822,       5.23819,      4.86481,      4.48976,     4.12659,
    3.81393,      3.52147,      3.23299,      2.97152,     2.66495,
    2.52559,      2.28054,      2.11023,      1.92686,     1.73145,
    1.57701,      1.44631,      1.31494,      1.16589,     1.06876,
    0.949659,     0.845888,     0.757272,     0.668908,    0.595084,
    0.510885,     0.450459,     0.389531,     0.336055,    0.283815,
    0.236612,     0.195456,     0.155327,     0.122504,    0.0952066,
    0.0690428,    0.0484605,    0.0563958,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    153.203,      123.593,      91.8427,      57.798,      44.7161,
    42.8607,      40.8712,      38.738,       36.4506,     33.9979,
    31.3679,      28.5479,      25.5241,      23.1224,     21.8924,
    20.5735,      19.1593,      17.6428,      16.0168,     14.987,
    14.0569,      13.0597,      11.9903,      11.1677,     10.3688,
    9.51215,      8.80835,      8.16188,      7.49266,     6.98405,
    6.43868,      5.99463,      5.53337,      5.13333,     4.74295,
    4.39152,      4.04488,      3.73414,      3.47007,     3.22017,
    2.9145,       2.66606,      2.41778,      2.23601,     2.02875,
    1.84613,      1.67622,      1.5084,       1.38226,     1.23929,
    1.11712,      0.99852,      0.879863,     0.813435,    0.711921,
    0.61888,      0.546401,     0.474957,     0.410699,    0.345803,
    0.288637,     0.245195,     0.201406,     0.161488,    0.126118,
    0.0951254,    0.069147,     0.0479224,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    185.543,      155.688,      123.675,      89.3482,     52.5409,
    46.7525,      44.7673,      42.6387,      40.3562,     37.9087,
    35.2844,      32.4705,      29.4532,      26.2178,     24.152,
    22.8306,      21.4137,      19.8944,      18.2653,     16.5506,
    15.6186,      14.6193,      13.5478,      12.4103,     11.6093,
    10.7504,      9.82941,      9.25882,      8.69552,     8.09507,
    7.45864,      6.81564,      6.34614,      5.86519,     5.41308,
    5.01694,      4.64985,      4.22351,      3.89446,     3.62219,
    3.33174,      3.00755,      2.76495,      2.53003,     2.32722,
    2.12063,      1.91991,      1.76623,      1.59776,     1.43543,
    1.30516,      1.1779,       1.05248,      0.934791,    0.829086,
    0.735997,     0.645349,     0.563696,     0.485884,    0.420401,
    0.359423,     0.296383,     0.250726,     0.202936,    0.162066,
    0.125801,     0.0942969,    0.0683422,    0.0457393,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    216.947,      187.178,      155.258,      121.031,     84.3307,
    51.4212,      49.4108,      47.2551,      44.9436,     42.465,
    39.8074,      36.9576,      33.902,       30.6255,     27.1122,
    25.4888,      24.0106,      22.4257,      20.7262,     18.9038,
    17.2681,      16.246,       15.1501,      13.975,      12.8605,
    12.0131,      11.1044,      10.2014,      9.58415,     8.92227,
    8.29372,      7.67404,      7.0594,       6.4683,      5.97158,
    5.55849,      5.12629,      4.71651,      4.36633,     4.03657,
    3.72503,      3.42868,      3.15512,      2.89183,     2.60528,
    2.40665,      2.17859,      1.99195,      1.83091,     1.66886,
    1.50793,      1.33379,      1.20857,      1.08858,     0.968365,
    0.854178,     0.757165,     0.661095,     0.582623,    0.496541,
    0.426819,     0.362402,     0.300221,     0.251834,    0.200962,
    0.161925,     0.123424,     0.0923802,    0.0647266,   0.0413237,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    247.641,      217.88,       185.967,      151.748,     115.057,
    75.7136,      53.3605,      51.1861,      48.8545,     46.3545,
    43.6738,      40.7993,      37.7172,      34.4122,     30.8685,
    27.609,       26.1911,      24.6707,      23.0405,     21.2924,
    19.418,       18.0295,      16.9209,      15.7322,     14.4575,
    13.4342,      12.5542,      11.6107,      10.777,      10.0732,
    9.3185,       8.61789,      7.87568,      7.31447,     6.79384,
    6.33331,      5.8532,       5.32693,      4.91928,     4.55526,
    4.1724,       3.77683,      3.46804,      3.16521,     2.92042,
    2.70486,      2.49507,      2.27149,      2.03542,     1.86087,
    1.67755,      1.5062,       1.37484,      1.2331,      1.10228,
    0.987308,     0.857241,     0.764666,     0.661692,    0.583614,
    0.504983,     0.428798,     0.364024,     0.297676,    0.25005,
    0.199743,     0.158249,     0.121099,     0.0863427,   0.0595133,
    0.0644462,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    276.856,      247.116,      215.227,      181.033,     144.368,
    105.054,      62.8982,      54.1469,      51.8749,     49.4386,
    46.8263,      44.0252,      41.0216,      37.801,      34.3477,
    30.6448,      28.1822,      26.7263,      25.1653,     23.4914,
    21.6966,      19.772,       18.56,        17.2739,     15.8948,
    14.4161,      13.6007,      12.7452,      11.8278,     11.0211,
    10.1887,      9.37019,      8.68757,      7.97698,     7.40168,
    6.8122,       6.28235,      5.83202,      5.44493,     4.94857,
    4.61711,      4.29657,      3.91918,      3.59142,     3.2796,
    2.99879,      2.76418,      2.52881,      2.25551,     2.09045,
    1.89401,      1.69328,      1.53125,      1.39065,     1.23155,
    1.12291,      0.986538,     0.867851,     0.777247,    0.668647,
    0.580361,     0.502105,     0.4277,       0.356216,    0.295866,
    0.240107,     0.190703,     0.148249,     0.111612,    0.0797274,
    0.0522842,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    308.487,      278.567,      246.484,      212.082,     175.194,
    135.641,      93.2285,      58.079,       55.7924,     53.3406,
    50.7116,      47.8926,      44.8699,      41.6287,     38.1533,
    34.4268,      30.4309,      28.8368,      27.2461,     25.5404,
    23.7115,      21.7504,      20.0158,      18.79,       17.4757,
    16.0664,      14.8145,      13.9344,      12.9908,     12.0264,
    11.1494,      10.209,       9.426,        8.70017,     8.04439,
    7.46982,      6.96617,      6.50124,      5.99933,     5.5128,
    5.06769,      4.68849,      4.28312,      3.97065,     3.62656,
    3.30709,      3.05544,      2.7624,       2.50359,     2.28513,
    2.10453,      1.91171,      1.71856,      1.56111,     1.40055,
    1.23184,      1.11738,      0.988798,     0.861272,    0.767382,
    0.658851,     0.579173,     0.492133,     0.422269,    0.351971,
    0.288854,     0.230501,     0.18502,      0.140002,    0.102573,
    0.0707611,    0.0449678,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    343.344,      313.006,      280.475,      245.594,     208.191,
    168.086,      125.082,      78.9706,      60.6551,     58.1163,
    55.394,       52.4751,      49.3451,      45.989,      42.3903,
    38.5316,      34.394,       31.0941,      29.4726,     27.7338,
    25.8694,      23.8703,      21.7267,      20.3013,     19.0035,
    17.6119,      16.1197,      15.0042,      13.9423,     12.8037,
    11.8679,      11.0164,      10.136,       9.51947,     8.85834,
    8.27335,      7.65826,      7.02237,      6.41713,     5.90877,
    5.46252,      5.05677,      4.67426,      4.2707,      3.93895,
    3.5844,       3.32197,      3.06267,      2.78797,     2.5646,
    2.30408,      2.10388,      1.89821,      1.70955,     1.54378,
    1.37458,      1.24223,      1.10203,      0.979705,    0.867497,
    0.757192,     0.666385,     0.563823,     0.491374,    0.407802,
    0.34097,      0.2728,       0.219099,     0.172462,    0.128394,
    0.0916196,    0.0623153,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    368.226,      338.245,      306.099,      271.629,     234.668,
    195.036,      152.54,       106.973,      63.0421,     60.6605,
    58.1068,      55.3686,      52.4325,      49.2842,     45.9083,
    42.2885,      38.4072,      34.2453,      32.0495,     30.1772,
    28.1696,      26.0169,      23.7086,      21.5591,     20.2885,
    18.926,       17.4651,      16.0517,      15.0438,     13.9632,
    12.8469,      12.0087,      11.1098,      10.2838,     9.51815,
    8.79062,      8.17745,      7.57645,      6.99582,     6.5081,
    6.01733,      5.48968,      5.10401,      4.65672,     4.29201,
    3.96993,      3.63086,      3.30898,      3.01536,     2.76525,
    2.51408,      2.28731,      2.07815,      1.89181,     1.70883,
    1.51339,      1.36884,      1.21868,      1.09355,     0.962319,
    0.85179,      0.741983,     0.647392,     0.548986,    0.472564,
    0.392289,     0.329829,     0.261443,     0.207743,    0.160689,
    0.115475,     0.0821504,    0.0511642,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    398.314,      368.227,      335.965,      301.372,     264.278,
    224.504,      181.856,      136.126,      87.0909,     63.8644,
    61.2589,      58.465,       55.4693,      52.257,      48.8126,
    45.1193,      41.1591,      36.9127,      33.2858,     31.5939,
    29.7797,      27.8344,      25.7485,      23.5119,     21.8786,
    20.4349,      18.8869,      17.227,       16.0649,     15.0772,
    14.0181,      13.0174,      12.044,       11.0056,     10.2078,
    9.35228,      8.75907,      8.1863,       7.54177,     6.91939,
    6.43553,      5.94248,      5.44801,      4.99782,     4.688,
    4.2297,       3.94149,      3.51907,      3.24429,     2.9856,
    2.77226,      2.47779,      2.29526,      2.05416,     1.85732,
    1.68554,      1.49302,      1.35644,      1.18784,     1.06525,
    0.946892,     0.828567,     0.720834,     0.615621,    0.533728,
    0.446691,     0.371533,     0.306333,     0.245227,    0.193222,
    0.141271,     0.102508,     0.0687877,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    420.261,      390.679,      358.959,      324.946,     288.476,
    249.37,       207.438,      162.476,      114.264,     66.3498,
    63.8282,      61.1243,      58.225,       55.1162,     51.7828,
    48.2084,      44.3757,      40.2661,      35.8594,     33.5396,
    31.6697,      29.6646,      27.5147,      25.2093,     22.9984,
    21.5997,      20.0998,      18.4915,      16.9425,     15.9666,
    14.9203,      13.8199,      12.8932,      11.8996,     11.0129,
    10.2367,      9.45486,      8.7185,       8.00898,     7.3495,
    6.83733,      6.32941,      5.74564,      5.46954,     4.98379,
    4.53561,      4.19369,      3.83964,      3.53845,     3.22399,
    2.94402,      2.6888,       2.47445,      2.24083,     2.01376,
    1.81692,      1.64591,      1.47141,      1.30506,     1.15154,
    1.03301,      0.911634,     0.803162,     0.69266,     0.595349,
    0.502409,     0.422142,     0.349293,     0.285558,    0.225135,
    0.16946,      0.124672,     0.0868643,    0.0551281,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    438.982,      410.195,      379.328,      346.23,      310.74,
    272.685,      231.88,       188.126,      141.21,      90.9041,
    67.7958,      64.9681,      61.936,       58.6849,     55.1987,
    51.4607,      47.4525,      43.1547,      38.5462,     34.6794,
    32.9009,      30.9939,      28.9491,      26.7564,     24.4054,
    22.7819,      21.3744,      19.8652,      18.247,      17.0199,
    15.9006,      14.7004,      13.6316,      12.6365,     11.5814,
    10.8141,      9.99133,      9.31036,      8.61492,     7.97342,
    7.33384,      6.7562,       6.28443,      5.8016,      5.30124,
    4.89767,      4.49336,      4.1469,       3.78057,     3.44704,
    3.14719,      2.87754,      2.63707,      2.38526,     2.18048,
    1.97385,      1.78981,      1.60286,      1.45921,     1.27835,
    1.1434,       0.987931,     0.888671,     0.761189,    0.662726,
    0.563713,     0.483592,     0.391326,     0.325714,    0.25501,
    0.203693,     0.154324,     0.108607,     0.0721998,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    462.769,      434.226,      403.621,      370.804,     335.616,
    297.884,      257.426,      214.044,      167.527,     117.648,
    71.0053,      68.2516,      65.299,       62.1329,     58.7381,
    55.0979,      51.1947,      47.0094,      42.5216,     37.7095,
    35.348,       33.2973,      31.0984,      28.7405,     26.2123,
    23.8927,      22.4802,      20.9656,      19.3416,     17.7834,
    16.6363,      15.4064,      14.1685,      13.3674,     12.5085,
    11.5939,      10.6183,      9.75661,      9.12552,     8.49053,
    7.85178,      7.16033,      6.55745,      6.18813,     5.69444,
    5.14914,      4.76393,      4.42135,      4.04079,     3.66882,
    3.43596,      3.08605,      2.80547,      2.56259,     2.3624,
    2.1276,       1.88873,      1.72843,      1.5129,      1.39593,
    1.21918,      1.08911,      0.977614,     0.832447,    0.728867,
    0.622362,     0.539108,     0.441941,     0.363942,    0.296529,
    0.23415,      0.179791,     0.129816,     0.0901084,   0.0555468,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    492.03,       463.234,      432.358,      399.25,      363.749,
    325.683,      284.866,      241.1,        194.17,      143.849,
    89.8912,      70.249,       67.3047,      64.1476,     60.7623,
    57.1324,      53.2402,      49.0667,      44.5916,     39.7931,
    36.0836,      34.2188,      32.2192,      30.0751,     27.7761,
    25.311,       23.7064,      22.2071,      20.5994,     18.8755,
    17.5898,      16.3378,      14.9953,      13.9547,     13.0221,
    12.0517,      11.223,       10.3345,      9.70936,     9.05644,
    8.35244,      7.62881,      6.90281,      6.44344,     5.96414,
    5.4893,       5.09933,      4.60087,      4.19897,     3.91,
    3.63239,      3.30395,      3.01008,      2.75216,     2.50258,
    2.25815,      2.0374,       1.83812,      1.6997,      1.46336,
    1.35689,      1.17994,      1.04498,      0.924844,    0.797484,
    0.70438,      0.589825,     0.50074,      0.409141,    0.333191,
    0.270092,     0.206525,     0.157367,     0.112264,    0.0724445,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    510.618,      482.368,      452.076,      419.596,     384.768,
    347.424,      307.38,       264.443,      218.403,     169.036,
    116.101,      73.6343,      70.6752,      67.5021,     64.0998,
    60.4516,      56.5398,      52.3452,      47.8475,     43.0248,
    37.8536,      35.9561,      33.9787,      31.8583,     29.5847,
    27.1468,      25.0129,      23.4662,      21.8077,     20.0293,
    18.4619,      17.323,       16.1019,      14.8858,     13.8501,
    12.7395,      11.8061,      10.9203,      10.0745,     9.26601,
    8.55686,      7.89087,      7.41946,      6.87175,     6.24488,
    5.72866,      5.30142,      4.91639,      4.54076,     4.1147,
    3.76985,      3.47689,      3.18899,      2.91904,     2.6424,
    2.41291,      2.1977,       1.9763,       1.75656,     1.59105,
    1.41028,      1.26519,      1.1346,       0.979324,    0.870489,
    0.745282,     0.652831,     0.54981,      0.457639,    0.381775,
    0.304968,     0.238961,     0.184376,     0.131416,    0.0911675,
    0.0683087,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    538.588,      510.121,      479.596,      446.866,     411.77,
    374.138,      333.786,      290.518,      244.124,     194.376,
    141.034,      83.8365,      72.1276,      68.9768,     65.5984,
    61.9758,      58.0914,      53.9262,      49.4601,     44.6712,
    39.5363,      36.265,       34.4858,      32.5781,     30.5326,
    28.3392,      25.9873,      24.4301,      22.7718,     20.9936,
    19.087,       17.9938,      16.841,       15.6048,     14.5495,
    13.4663,      12.4121,      11.5589,      10.6695,     9.92991,
    9.16116,      8.42566,      7.77573,      7.19775,     6.62153,
    6.15023,      5.69283,      5.19291,      4.72249,     4.36375,
    4.02381,      3.68657,      3.3346,       3.11775,     2.77829,
    2.58335,      2.32836,      2.10345,      1.90032,     1.69454,
    1.52234,      1.35386,      1.21013,      1.0673,      0.938979,
    0.819161,     0.700484,     0.5972,       0.504317,    0.419542,
    0.347624,     0.274057,     0.211969,     0.157909,    0.108561,
    0.0713034,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    557.953,      529.888,      499.795,      467.527,     432.927,
    395.826,      356.045,      313.388,      267.649,     218.604,
    166.015,      109.626,      75.3795,      72.2807,     68.958,
    65.3951,      61.5748,      57.4783,      53.0859,     48.376,
    43.3257,      38.5494,      36.5725,      34.4527,     32.1797,
    29.7425,      27.1291,      25.1738,      23.6863,     22.0912,
    20.3808,      18.9297,      17.6582,      16.2947,     15.0754,
    14.0704,      12.9928,      12.0755,      11.1234,     10.316,
    9.54005,      8.79541,      8.05762,      7.50061,     6.94085,
    6.40201,      5.92257,      5.42854,      4.87411,     4.5799,
    4.21681,      3.90586,      3.53753,      3.242,       2.97111,
    2.74,         2.45841,      2.20871,      2.01296,     1.80816,
    1.61866,      1.44759,      1.30534,      1.12545,     1.00832,
    0.88502,      0.772952,     0.655538,     0.550839,    0.465565,
    0.381033,     0.308914,     0.242685,     0.183624,    0.129367,
    0.0888818,    0.108969,     0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    565.576,      538.369,      509.195,      477.914,     444.371,
    408.405,      369.839,      328.486,      284.145,     236.599,
    185.618,      130.952,      75.8843,      73.0009,     69.9091,
    66.5939,      63.0391,      59.2274,      55.1402,     50.7577,
    46.0584,      41.0196,      38.3223,      36.1813,     33.8855,
    31.4239,      28.7843,      26.2364,      24.6395,     22.9271,
    21.0909,      19.2969,      18.1115,      16.8405,     15.5043,
    14.4484,      13.3163,      12.3139,      11.4567,     10.6117,
    9.86462,      9.13664,      8.45342,      7.90031,     7.34791,
    6.71379,      6.15164,      5.71559,      5.31492,     4.83831,
    4.39829,      4.07541,      3.75628,      3.42203,     3.09474,
    2.83272,      2.55779,      2.34178,      2.14229,     1.94606,
    1.70479,      1.54927,      1.39794,      1.23623,     1.07665,
    0.944343,     0.822165,     0.708612,     0.609911,    0.507073,
    0.418034,     0.341338,     0.274009,     0.206368,    0.153744,
    0.105141,     0.0661839,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    561.54,       535.823,      508.248,      478.679,     446.974,
    412.978,      376.525,      337.437,      295.525,     250.584,
    202.395,      150.723,      95.3176,      75.1932,     72.0233,
    68.6242,      64.9796,      61.0715,      56.8811,     52.3878,
    47.5697,      42.4035,      38.4221,      36.4218,     34.277,
    31.9773,      29.5113,      26.8671,      25.1942,     23.6435,
    21.9809,      20.198,       18.933,       17.7195,     16.4184,
    15.2642,      14.1358,      12.9728,      12.055,      11.0709,
    10.2235,      9.34969,      8.77746,      8.16929,     7.52574,
    7.00011,      6.49244,      5.98875,      5.47819,     4.99598,
    4.63406,      4.28648,      3.88705,      3.60763,     3.24741,
    2.97144,      2.7363,       2.46952,      2.21871,     2.04469,
    1.80638,      1.6398,       1.48335,      1.28665,     1.15031,
    0.998292,     0.889822,     0.756866,     0.652408,    0.555335,
    0.461401,     0.376014,     0.300989,     0.233894,    0.176532,
    0.127204,     0.0814602,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    593.171,      566.988,      538.912,      508.808,     476.529,
    441.916,      404.802,      365.006,      322.334,     276.579,
    227.516,      174.908,      118.499,      77.3043,     74.253,
    70.9811,      67.4728,      63.711,       59.6773,     55.3521,
    50.7143,      45.7414,      40.5965,      38.4727,     36.1954,
    33.7535,      31.1351,      28.3275,      26.0634,     24.5055,
    22.835,       21.0438,      19.4934,      18.2765,     16.9716,
    15.7053,      14.6218,      13.46,        12.5328,     11.6387,
    10.7756,      9.91851,      9.16756,      8.43938,     7.75993,
    7.1697,       6.66616,      6.15906,      5.69343,     5.24584,
    4.81198,      4.48918,      4.09523,      3.689,       3.33456,
    3.04879,      2.79337,      2.58253,      2.30774,     2.09731,
    1.90719,      1.73293,      1.52841,      1.37462,     1.21483,
    1.06998,      0.933789,     0.819723,     0.697082,    0.595397,
    0.503172,     0.408091,     0.332492,     0.259899,    0.199716,
    0.144146,     0.0988283,    0.108911,     0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    604.358,      578.603,      550.986,      521.373,     489.62,
    455.573,      419.065,      379.919,      337.944,     292.935,
    244.674,      192.925,      137.436,      77.937,      73.6954,
    70.6381,      67.3599,      63.8448,      60.0756,     56.0341,
    51.7005,      47.0537,      42.0711,      39.0692,     36.8718,
    34.5155,      31.989,       29.2799,      26.561,      25.06,
    23.4505,      21.7247,      19.9635,      18.8096,     17.5722,
    16.2454,      15.1275,      13.9341,      12.8153,     11.8486,
    10.9178,      10.2649,      9.55433,      8.77268,     8.09604,
    7.48058,      6.96582,      6.4245,       5.88633,     5.41266,
    5.01753,      4.61087,      4.19537,      3.86792,     3.51734,
    3.21798,      2.94768,      2.65239,      2.42261,     2.21092,
    1.9873,       1.78315,      1.60538,      1.46699,     1.27448,
    1.13411,      0.994158,     0.885681,     0.761464,    0.647789,
    0.542263,     0.45457,      0.361101,     0.290903,    0.225882,
    0.164594,     0.116556,     0.0736421,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    610.657,      585.676,      558.89,       530.168,     499.371,
    466.348,      430.938,      392.97,       352.257,     308.603,
    261.793,      211.601,      157.782,      100.073,     76.1724,
    73.0533,      69.7087,      66.1224,      62.2769,     58.1535,
    53.7322,      48.9913,      43.9078,      39.7775,     37.7473,
    35.5705,      33.2363,      30.7334,      28.0496,     26.2218,
    24.5694,      22.7977,      20.8979,      19.3846,     17.9227,
    16.3552,      15.2174,      14.3096,      13.3493,     12.4835,
    11.5551,      10.6291,      9.64903,      9.03398,     8.3921,
    7.73874,      7.23811,      6.71529,      6.15664,     5.64222,
    5.16055,      4.82442,      4.36355,      4.02144,     3.69088,
    3.34186,      3.05347,      2.79278,      2.55717,     2.31769,
    2.09616,      1.88482,      1.68358,      1.52347,     1.34994,
    1.19628,      1.07044,      0.911653,     0.795607,    0.689892,
    0.580408,     0.487658,     0.409249,     0.32446,     0.247394,
    0.187024,     0.131854,     0.0893394,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    613.054,      588.884,      562.967,      535.178,     505.38,
    473.428,      439.168,      402.431,      363.04,      320.802,
    275.512,      226.949,      174.876,      119.04,      77.0123,
    74.033,       70.8385,      67.413,       63.74,       59.8016,
    55.5785,      51.0503,      46.1948,      41.0961,     38.9791,
    36.7091,      34.275,       31.665,       28.8664,     26.556,
    24.9574,      23.2433,      21.4054,      19.7727,     18.4665,
    17.066,       15.7504,      14.7647,      13.7079,     12.7424,
    11.7664,      10.9382,      10.2208,      9.40823,     8.5206,
    7.9635,       7.42396,      6.89629,      6.3051,      5.76742,
    5.36571,      4.99192,      4.56853,      4.06649,     3.82364,
    3.47162,      3.18042,      2.85388,      2.64929,     2.39513,
    2.20175,      1.97887,      1.767,        1.57745,     1.42862,
    1.25659,      1.10455,      0.980762,     0.840208,    0.73692,
    0.62335,      0.526853,     0.438679,     0.354535,    0.279362,
    0.210363,     0.15124,      0.104702,     0.103965,    0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    629.086,      605.157,      579.497,      551.984,     522.482,
    490.848,      456.928,      420.557,      381.558,     339.739,
    294.899,      246.819,      195.264,      139.983,     80.7066,
    76.6206,      73.2869,      69.7124,      65.8795,     61.7696,
    57.3627,      52.6373,      47.5705,      42.1374,     39.0888,
    36.992,       34.7436,      32.3328,      29.7477,     27.141,
    25.5243,      23.7909,      21.9322,      20.0397,     18.7829,
    17.4353,      15.9904,      15.0017,      13.9474,     12.9789,
    12.1684,      11.3192,      10.4712,      9.61478,     8.79392,
    8.16198,      7.59474,      6.96601,      6.3993,      5.91441,
    5.5157,       5.11136,      4.66649,      4.30146,     3.9446,
    3.57156,      3.27732,      3.02005,      2.75541,     2.4845,
    2.24648,      2.04316,      1.84,         1.63847,     1.48105,
    1.32072,      1.14624,      1.02092,      0.876752,    0.772885,
    0.667398,     0.563634,     0.463751,     0.384068,    0.302289,
    0.236336,     0.174692,     0.120255,     0.0762111,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    635.702,      612.28,       587.165,      560.236,     531.36,
    500.398,      467.197,      431.598,      393.426,     352.496,
    308.607,      261.547,      211.086,      156.978,     98.9597,
    76.9102,      73.751,       70.3635,      66.7312,     62.8364,
    58.6601,      54.182,       49.3803,      44.2315,     40.1535,
    38.0488,      35.7921,      33.3722,      30.7775,     27.9953,
    26.1375,      24.4127,      22.5633,      20.5803,     19.247,
    18.0155,      16.6951,      15.5648,      14.4939,     13.3723,
    12.3918,      11.3405,      10.5367,      9.7028,      8.90419,
    8.24524,      7.876,        7.29046,      6.70194,     6.16703,
    5.69062,      5.2274,       4.7929,       4.35789,     4.04128,
    3.70091,      3.43516,      3.11381,      2.85529,     2.60122,
    2.33511,      2.1096,       1.89601,      1.72704,     1.54274,
    1.36124,      1.2269,       1.07106,      0.944592,    0.819671,
    0.701654,     0.595157,     0.499734,     0.417098,    0.331123,
    0.256531,     0.19242,      0.140062,     0.091923,    0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    639.907,      617.076,      592.594,      566.344,     538.196,
    508.014,      475.652,      440.95,       403.74,      363.842,
    321.06,       275.186,      225.998,      173.254,     116.699,
    77.5807,      74.457,       71.1076,      67.5162,     63.6652,
    59.5359,      55.1082,      50.3606,      45.2698,     40.1476,
    38.0988,      35.902,       33.5464,      31.0206,     28.3123,
    26.1769,      24.6575,      23.0282,      21.2812,     19.7371,
    18.4217,      17.0113,      15.7022,      14.6544,     13.5309,
    12.554,       11.5637,      10.7267,      9.96654,     9.27131,
    8.57709,      7.93615,      7.35578,      6.82872,     6.31483,
    5.73379,      5.33695,      4.8971,       4.56242,     4.10646,
    3.77627,      3.50352,      3.21275,      2.89081,     2.66148,
    2.42281,      2.15046,      1.97204,      1.79025,     1.60213,
    1.42506,      1.26224,      1.12435,      0.976079,    0.865662,
    0.738931,     0.63812,      0.527953,     0.442771,    0.354995,
    0.286791,     0.214116,     0.155879,     0.106982,    0.0646889,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    648.124,      625.693,      601.641,      575.85,      548.196,
    518.543,      486.748,      452.654,      416.097,     376.898,
    334.866,      289.796,      241.469,      189.65,      134.086,
    78.4493,      75.4969,      72.3312,      68.9367,     65.2969,
    61.3941,      57.2092,      52.7219,      47.9104,     42.7511,
    39.9782,      37.7449,      35.3502,      32.7824,     30.029,
    27.3794,      25.7134,      23.927,       22.0115,     20.1339,
    18.8393,      17.4511,      16.0033,      14.9773,     13.8771,
    12.9095,      12.0851,      11.2113,      10.2955,     9.49334,
    8.8675,       8.18727,      7.51789,      6.96727,     6.46425,
    5.92913,      5.41133,      4.99862,      4.67095,     4.26624,
    3.88756,      3.60026,      3.28617,      3.08769,     2.74852,
    2.48133,      2.28647,      2.03693,      1.81767,     1.66236,
    1.47156,      1.32755,      1.18477,      1.04252,     0.896922,
    0.773963,     0.674838,     0.565016,     0.463962,    0.387526,
    0.308487,     0.239601,     0.175895,     0.120613,    0.0770835,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    653.28,       631.242,      607.612,      582.273,     555.104,
    525.972,      494.734,      461.238,      425.322,     386.81,
    345.515,      301.236,      253.757,      202.847,     148.258,
    89.7233,      73.8909,      70.8734,      67.6379,     64.1686,
    60.4485,      56.4596,      52.1825,      47.5963,     42.6786,
    39.154,       37.1568,      35.0153,      32.719,      30.2567,
    27.6166,      25.9527,      24.2732,      22.4723,     20.5413,
    19.1786,      17.7878,      16.2965,      15.2088,     14.1886,
    13.1476,      12.2288,      11.256,       10.5096,     9.72524,
    8.97621,      8.29152,      7.68807,      7.04419,     6.50134,
    6.04974,      5.64501,      5.23695,      4.77989,     4.36512,
    3.98956,      3.67449,      3.40385,      3.06137,     2.82639,
    2.5613,       2.33625,      2.09805,      1.92293,     1.7351,
    1.53744,      1.38484,      1.19923,      1.0828,      0.942204,
    0.812443,     0.707605,     0.599827,     0.499795,    0.412376,
    0.330171,     0.25771,      0.196834,     0.138518,    0.0881423,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    649.3,        628.101,      605.371,      580.998,     554.863,
    526.84,       496.791,      464.571,      430.023,     392.978,
    353.256,      310.663,      264.992,      216.02,      163.51,
    107.205,      75.7124,      72.6524,      69.3714,     65.8532,
    62.0807,      58.0357,      53.6983,      49.0474,     44.0605,
    39.5017,      37.4997,      35.3531,      33.0514,     30.5834,
    27.937,       26.02,        24.5082,      22.8872,     21.149,
    19.6402,      18.2393,      16.7371,      15.4943,     14.5421,
    13.5211,      12.5184,      11.4501,      10.6682,     9.95016,
    9.16777,      8.37829,      7.7438,       7.16742,     6.62962,
    6.1251,       5.62195,      5.20366,      4.91209,     4.43693,
    4.01611,      3.75139,      3.40772,      3.11442,     2.84917,
    2.6471,       2.37792,      2.17153,      1.94617,     1.77927,
    1.59017,      1.41328,      1.24854,      1.12832,     0.987828,
    0.862345,     0.744586,     0.626463,     0.530009,    0.438913,
    0.361416,     0.281981,     0.214165,     0.155797,    0.102703,
    0.129344,     0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    651.608,      630.94,       608.778,      585.014,     559.533,
    532.211,      502.914,      471.5,        437.816,     401.698,
    362.969,      321.442,      276.913,      229.167,     177.97,
    123.073,      76.6895,      73.7296,      70.5559,     67.1528,
    63.5038,      59.5911,      55.3957,      50.897,      46.0732,
    40.9008,      38.702,       36.5512,      34.245,      31.7721,
    29.1205,      26.7585,      25.1116,      23.3457,     21.4522,
    19.7445,      18.5283,      17.2242,      15.9081,     14.7924,
    13.5962,      12.5982,      11.6819,      10.813,      10.008,
    9.31751,      8.70011,      8.1125,       7.5323,      6.97281,
    6.40735,      5.81868,      5.30211,      4.90191,     4.54261,
    4.19694,      3.87734,      3.48817,      3.25608,     2.96543,
    2.67642,      2.43659,      2.22482,      1.97804,     1.81506,
    1.63787,      1.4619,       1.29694,      1.15227,     1.01317,
    0.899115,     0.780428,     0.657271,     0.56312,     0.464831,
    0.380154,     0.304569,     0.235585,     0.172324,    0.118664,
    0.0736583,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    655.608,      635.326,      613.579,      590.261,     565.257,
    538.446,      509.697,      478.871,      445.818,     410.375,
    372.372,      331.622,      287.927,      241.074,     190.835,
    136.966,      79.2037,      73.2748,      70.2384,     66.9825,
    63.4914,      59.7479,      55.7339,      51.4299,     46.8148,
    41.8662,      38.7765,      36.5788,      34.2223,     31.6955,
    28.9861,      26.2211,      24.7374,      23.1464,     21.4405,
    19.6656,      18.3983,      17.0395,      15.5825,     14.5997,
    13.5757,      12.5967,      11.7428,      10.8875,     10.2049,
    9.48725,      8.74903,      8.07504,      7.43354,     6.8365,
    6.40686,      5.97527,      5.47346,      5.00227,     4.64656,
    4.25316,      3.91081,      3.56267,      3.30322,     3.01085,
    2.77132,      2.46302,      2.29926,      2.04948,     1.86259,
    1.69336,      1.47575,      1.33862,      1.19165,     1.05112,
    0.930411,     0.809002,     0.691218,     0.587188,    0.494936,
    0.409993,     0.33017,      0.254028,     0.190751,    0.132198,
    0.0871037,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    654.073,      634.345,      613.192,      590.51,      566.19,
    540.111,      512.148,      482.164,      450.013,     415.538,
    378.573,      338.936,      296.434,      250.861,     201.994,
    149.596,      93.4118,      72.8148,      69.9468,     66.8715,
    63.5739,      60.0381,      56.2467,      52.1813,     47.8222,
    43.148,       39.3901,      37.243,       34.9406,     32.4719,
    29.8248,      26.9864,      25.2522,      23.6811,     21.9964,
    20.19,        18.9273,      17.7301,      16.4463,     15.2401,
    14.0266,      12.8054,      12.0949,      11.3331,     10.5121,
    9.64907,      9.0236,       8.37677,      7.72205,     7.06892,
    6.51239,      6.02851,      5.52293,      5.1443,      4.69901,
    4.32653,      3.94879,      3.66585,      3.2827,      3.07923,
    2.77471,      2.5872,       2.37727,      2.10897,     1.91199,
    1.75144,      1.5355,       1.37316,      1.23431,     1.09257,
    0.954522,     0.848273,     0.725841,     0.608367,    0.524384,
    0.427198,     0.353845,     0.272801,     0.207668,    0.152198,
    0.098119,     0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    658.362,      639.035,      618.312,      596.091,     572.264,
    546.715,      519.319,      489.944,      458.446,     424.672,
    388.457,      349.624,      307.986,      263.338,     215.464,
    164.13,       109.087,      74.2974,      71.3478,     68.185,
    64.7937,      61.1573,      57.258,       53.077,      48.5939,
    43.7867,      39.1395,      37.1509,      35.0186,     32.7322,
    30.2805,      27.6517,      25.593,       23.99,       22.2711,
    20.428,       18.988,       17.8843,      16.7009,     15.5865,
    14.6043,      13.5511,      12.5552,      11.5093,     10.5723,
    9.65457,      8.99681,      8.37777,      7.71899,     7.14077,
    6.59917,      6.03038,      5.55917,      5.19318,     4.77192,
    4.43419,      4.06606,      3.66257,      3.43649,     3.08472,
    2.83637,      2.60765,      2.38311,      2.13557,     1.9168,
    1.78205,      1.58631,      1.41982,      1.28058,     1.11979,
    0.980819,     0.87249,      0.752019,     0.636234,    0.546801,
    0.460455,     0.370049,     0.297141,     0.223077,    0.164399,
    0.111426,     0.100433,     0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    652.339,      633.703,      613.721,      592.294,     569.319,
    544.683,      518.267,      489.942,      459.571,     427.004,
    392.083,      354.64,       314.49,       271.438,     225.276,
    175.777,      122.701,      75.1768,      72.181,      68.9687,
    65.5243,      61.831,       57.8708,      53.6243,     49.071,
    44.1887,      38.9535,      36.6914,      34.67,       32.5026,
    30.1786,      27.6866,      25.4381,      23.9946,     22.4467,
    20.787,       19.1859,      17.8921,      16.5048,     15.1234,
    14.1585,      13.124,       12.259,       11.4938,     10.7076,
    9.91091,      9.09657,      8.25795,      7.69883,     7.19194,
    6.71944,      6.18739,      5.78278,      5.26441,     4.79922,
    4.48389,      4.08653,      3.72857,      3.39922,     3.18338,
    2.93007,      2.64837,      2.42713,      2.1994,      2.01226,
    1.83922,      1.63175,      1.46014,      1.30774,     1.14825,
    1.02637,      0.899022,     0.783329,     0.670142,    0.568994,
    0.478619,     0.392109,     0.310517,     0.2443,      0.181279,
    0.12617,      0.0804288,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    662.842,      644.303,      624.424,      603.108,     580.253,
    555.745,      529.467,      501.289,      471.075,     438.678,
    403.939,      366.69,       326.749,      283.921,     237.999,
    188.758,      135.958,      79.3425,      72.0416,     69.0101,
    65.7595,      62.2741,      58.5367,      54.5293,     50.2322,
    45.6246,      40.684,       37.5585,      35.4325,     33.153,
    30.7087,      28.0877,      25.3736,      23.9304,     22.3829,
    20.7236,      18.9914,      17.9615,      16.8572,     15.673,
    14.6405,      13.5526,      12.5045,      11.5982,     10.6865,
    9.98358,      9.26994,      8.60348,      8.03445,     7.49033,
    6.74835,      6.22854,      5.74532,      5.30013,     4.91813,
    4.50752,      4.14258,      3.83228,      3.54244,     3.2022,
    2.92075,      2.68479,      2.42355,      2.24565,     2.03998,
    1.83652,      1.66906,      1.50088,      1.32098,     1.17807,
    1.05359,      0.925874,     0.802875,     0.69248,     0.585099,
    0.49334,      0.410835,     0.32733,      0.260199,    0.193857,
    0.138427,     0.0918075,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    657.139,      639.156,      619.874,      599.199,     577.029,
    553.257,      527.767,      500.435,      471.128,     439.703,
    406.007,      369.876,      331.134,      289.591,     245.047,
    197.284,      146.069,      91.1526,      70.7108,     67.8064,
    64.6921,      61.3527,      57.772,       53.9326,     49.8157,
    45.4012,      40.6677,      36.951,       35.0425,     32.9962,
    30.8019,      28.4491,      25.9263,      24.3253,     22.8592,
    21.2871,      19.6015,      18.3407,      17.1196,     15.8104,
    14.6968,      13.6418,      12.5503,      11.6897,     10.767,
    9.953,        9.10719,      8.49207,      7.89535,     7.3599,
    6.85886,      6.3481,       5.82326,      5.32818,     5.04791,
    4.62924,      4.22498,      3.8421,       3.56674,     3.2991,
    2.98272,      2.77658,      2.48993,      2.25284,     2.07273,
    1.87046,      1.67965,      1.52169,      1.36665,     1.21015,
    1.08688,      0.943856,     0.83569,      0.712898,    0.61525,
    0.514782,     0.434434,     0.350888,     0.276425,    0.211892,
    0.154403,     0.10338,      0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    637.606,      620.633,      602.434,      582.92,      561.995,
    539.558,      515.5,        489.703,      462.041,     432.381,
    400.577,      366.475,      329.909,      290.699,     248.657,
    203.576,      155.237,      103.405,      71.4792,     68.664,
    65.6454,      62.4086,      58.9379,      55.2164,     51.2259,
    46.9471,      42.359,       37.9623,      35.9843,     33.8634,
    31.5892,      29.1506,      26.5358,      24.5862,     23.1321,
    21.573,       19.9011,      18.4681,      17.2084,     15.8576,
    14.6641,      13.7135,      12.6943,      11.7416,     10.7402,
    10.0758,      9.54274,      8.82985,      8.0672,      7.46392,
    6.85142,      6.28332,      5.84425,      5.37666,     5.03869,
    4.6584,       4.26381,      3.86738,      3.60037,     3.33648,
    3.07735,      2.75774,      2.5397,       2.26385,     2.12133,
    1.88382,      1.70132,      1.54699,      1.38583,     1.24436,
    1.11346,      0.984279,     0.859009,     0.736559,    0.644155,
    0.537578,     0.449081,     0.372089,     0.292228,    0.228211,
    0.166956,     0.114858,     0.119194,     0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    648.696,      631.74,       613.559,      594.063,     573.159,
    550.744,      526.709,      500.937,      473.303,     443.671,
    411.898,      377.83,       341.299,      302.128,     260.126,
    215.089,      166.797,      115.015,      70.6152,     67.9602,
    65.1134,      62.0608,      58.7876,      55.2779,     51.5146,
    47.4793,      43.1523,      38.5127,      36.3182,     34.1781,
    31.8833,      29.4226,      26.7842,      24.4862,     23.0554,
    21.5211,      19.876,       18.3642,      17.2512,     16.0578,
    14.844,       13.7979,      12.6761,      11.8303,     11.1273,
    10.3135,      9.37049,      8.64542,      8.08264,     7.461,
    6.86514,      6.33882,      5.8887,       5.51146,     5.0771,
    4.64842,      4.27744,      3.89607,      3.60001,     3.36692,
    3.07093,      2.8207,       2.53938,      2.35475,     2.11373,
    1.95453,      1.73718,      1.59571,      1.41736,     1.286,
    1.14137,      1.00431,      0.885676,     0.765833,    0.663425,
    0.552957,     0.468716,     0.384369,     0.311672,    0.243915,
    0.183561,     0.126023,     0.0806736,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    640.66,       624.274,      606.704,      587.864,     567.662,
    546.001,      522.774,      497.869,      471.163,     442.528,
    411.824,      378.9,        343.597,      305.743,     265.154,
    221.631,      174.963,      124.922,      71.265,      67.2997,
    64.5098,      61.5184,      58.3108,      54.8713,     51.1833,
    47.2288,      42.9885,      38.4418,      35.731,      33.7783,
    31.6844,      29.4393,      27.0319,      24.6123,     23.2737,
    21.8384,      20.2993,      18.7,         17.4131,     16.0331,
    14.5534,      13.7903,      12.9884,      12.0811,     11.039,
    10.0341,      9.32807,      8.67377,      8.16975,     7.63179,
    7.05996,      6.46392,      5.92843,      5.49152,     5.11776,
    4.7454,       4.3294,       3.97292,      3.64499,     3.39137,
    3.10804,      2.80416,      2.57027,      2.36367,     2.18166,
    1.98362,      1.78613,      1.59684,      1.43455,     1.29364,
    1.15652,      1.00115,      0.898356,     0.785718,    0.674457,
    0.589336,     0.491026,     0.40731,      0.32609,     0.257195,
    0.19704,      0.139311,     0.0933559,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    635.122,      619.252,      602.236,      583.991,     564.426,
    543.448,      520.954,      496.834,      470.971,     443.24,
    413.504,      381.619,      347.43,       310.77,      271.46,
    229.311,      184.114,      135.652,      83.6878,     68.0403,
    65.1788,      62.1105,      58.8205,      55.2927,     51.51,
    47.454,       43.1048,      38.4413,      35.0942,     33.3904,
    31.5634,      29.6044,      27.5038,      25.2514,     23.6065,
    21.9389,      20.1508,      18.2334,      17.0931,     15.9943,
    14.8161,      13.8802,      12.9862,      12.0731,     11.2956,
    10.4646,      9.77142,      9.01976,      8.1524,      7.44007,
    6.94513,      6.39521,      5.96058,      5.59677,     5.20804,
    4.79385,      4.34523,      4.05712,      3.71166,     3.40738,
    3.12204,      2.91242,      2.59519,      2.38037,     2.16675,
    1.96382,      1.80274,      1.62797,      1.46717,     1.31297,
    1.17786,      1.04437,      0.909913,     0.82233,     0.709982,
    0.605972,     0.513302,     0.424959,     0.343738,    0.275631,
    0.209755,     0.154514,     0.102847,     0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    623.035,      607.815,      591.495,      573.996,     555.232,
    535.112,      513.538,      490.405,      465.6,       439.003,
    410.484,      379.904,      347.113,      311.953,     274.252,
    233.827,      190.48,       144.001,      94.1622,     67.9813,
    65.1997,      62.2171,      59.0189,      55.5896,     51.9124,
    47.9696,      43.7418,      39.2084,      35.1687,     33.3277,
    31.3537,      29.2371,      26.9675,      24.5339,     22.8646,
    21.5524,      20.1453,      18.6366,      17.419,      16.3241,
    15.1501,      14.0397,      12.9831,      11.8501,     11.102,
    10.3159,      9.61679,      8.90632,      8.28224,     7.6548,
    7.07141,      6.49683,      5.94878,      5.46464,     5.07773,
    4.81645,      4.40042,      4.06178,      3.75192,     3.4348,
    3.17613,      2.92961,      2.66801,      2.42726,     2.20526,
    2.03639,      1.80862,      1.63728,      1.50223,     1.33456,
    1.18079,      1.06862,      0.961555,     0.829115,    0.72338,
    0.615921,     0.535234,     0.444278,     0.360736,    0.291995,
    0.226187,     0.163565,     0.111166,     0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    614.138,      599.402,      583.602,      566.659,     548.492,
    529.011,      508.124,      485.726,      461.71,      435.959,
    408.346,      378.738,      346.991,      312.948,     276.446,
    237.306,      195.338,      150.336,      102.083,     66.0117,
    63.4713,      60.7472,      57.8263,      54.6943,     51.336,
    47.7349,      43.8737,      39.7333,      35.3994,     33.5955,
    31.6612,      29.5871,      27.3631,      24.9784,     23.024,
    21.6809,      20.2408,      18.6965,      17.3221,     16.2113,
    15.0202,      13.8807,      12.9662,      11.9856,     11.1446,
    10.3155,      9.54055,      8.79724,      8.20991,     7.67956,
    7.10518,      6.59916,      6.15046,      5.61879,     5.2125,
    4.79258,      4.42861,      4.01491,      3.70499,     3.38909,
    3.178,        2.90292,      2.66158,      2.43858,     2.23138,
    2.05007,      1.82998,      1.68203,      1.50144,     1.35105,
    1.21969,      1.06191,      0.95969,      0.852872,    0.750281,
    0.634493,     0.540589,     0.459757,     0.381724,    0.297784,
    0.234039,     0.178392,     0.122901,     0.0800052,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    614.398,      599.971,      584.502,      567.915,     550.129,
    531.058,      510.609,      488.682,      465.17,      439.96,
    412.927,      383.941,      352.86,       319.532,     283.797,
    245.478,      204.391,      160.335,      113.094,     67.2242,
    64.6164,      61.8201,      58.8218,      55.6068,     52.1595,
    48.463,       44.4994,      40.2494,      35.6922,     33.4226,
    31.5707,      29.5848,      27.4555,      25.1723,     23.0343,
    21.7611,      20.396,       18.9322,      17.4941,     16.4039,
    15.2349,      14.0237,      13.1058,      12.1217,     11.1938,
    10.3128,      9.48168,      8.79896,      8.17356,     7.62651,
    7.11954,      6.61304,      6.10913,      5.60237,     5.18353,
    4.84081,      4.4831,       4.0614,       3.73345,     3.46649,
    3.19342,      2.93272,      2.67983,      2.46108,     2.22978,
    2.04168,      1.8555,       1.69421,      1.52581,     1.33834,
    1.23402,      1.09954,      0.980896,     0.877091,    0.752015,
    0.656394,     0.564753,     0.473869,     0.389177,    0.318566,
    0.249021,     0.185908,     0.136217,     0.0894282,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    614.38,       600.229,      585.055,      568.785,     551.339,
    532.632,      512.573,      491.065,      468.002,     443.273,
    416.756,      388.323,      357.836,      325.145,     290.092,
    252.505,      212.202,      168.987,      122.649,     72.9617,
    64.7848,      62.0223,      59.0602,      55.884,      52.4782,
    48.8264,      44.9106,      40.7118,      36.2096,     33.4101,
    31.7332,      29.935,       28.0069,      25.9394,     23.7487,
    22.29,        20.7259,      19.0487,      17.2689,     16.236,
    15.1284,      13.9408,      13.0312,      12.101,      11.188,
    10.3891,      9.5698,       8.90921,      8.2738,      7.80528,
    7.25241,      6.64556,      6.20064,      5.75717,     5.26147,
    4.77486,      4.41002,      4.04335,      3.71528,     3.49737,
    3.18917,      2.93819,      2.70199,      2.44852,     2.25055,
    2.06273,      1.89668,      1.69603,      1.54956,     1.39845,
    1.25176,      1.08863,      0.992003,     0.877022,    0.772917,
    0.662668,     0.574233,     0.480727,     0.401274,    0.33581,
    0.265493,     0.200849,     0.145774,     0.0993427,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    616.246,      602.275,      587.295,      571.233,     554.009,
    535.541,      515.738,      494.504,      471.736,     447.322,
    421.143,      393.073,      362.975,      330.701,     296.095,
    258.988,      219.199,      176.535,      130.788,     81.735,
    63.6202,      61.0431,      58.2799,      55.3169,     52.1398,
    48.7331,      45.0802,      41.1634,      36.9635,     33.6462,
    31.8824,      29.991,       27.9631,      25.7885,     23.4568,
    21.8691,      20.3701,      18.7627,      17.0392,     15.9565,
    14.9755,      13.9237,      13.0035,      12.1149,     11.1999,
    10.5059,      9.76175,      9.03777,      8.28181,     7.75072,
    7.19995,      6.64009,      6.09407,      5.62015,     5.2018,
    4.76921,      4.43587,      4.15013,      3.79931,     3.4999,
    3.19201,      2.98994,      2.70211,      2.444,       2.25822,
    2.07489,      1.91637,      1.73684,      1.55276,     1.40415,
    1.2723,       1.12354,      1.00102,      0.890471,    0.784426,
    0.68876,      0.586327,     0.500215,     0.422679,    0.341422,
    0.277635,     0.215863,     0.161187,     0.106546,    0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    586.436,      573.439,      559.504,      544.561,     528.539,
    511.359,      492.937,      473.184,      452.003,     429.292,
    404.939,      378.827,      350.827,      320.804,     288.612,
    254.092,      217.079,      177.39,       134.833,     89.2007,
    62.864,       60.3608,      57.6768,      54.7987,     51.7127,
    48.4036,      44.8554,      41.0508,      36.9712,     33.1797,
    31.4795,      29.6564,      27.7016,      25.6056,     23.358,
    21.6731,      20.3274,      18.8843,      17.3371,     16.1438,
    15.1705,      14.1269,      13.1236,      12.1758,     11.1595,
    10.3817,      9.57715,      8.91287,      8.27253,     7.65032,
    7.03249,      6.54413,      6.06144,      5.60819,     5.22055,
    4.85975,      4.46464,      4.1479,       3.87625,     3.47182,
    3.2018,       2.98845,      2.75106,      2.4862,      2.29184,
    2.0825,       1.90934,      1.73551,      1.54751,     1.42588,
    1.26731,      1.16249,      1.00154,      0.90108,     0.797728,
    0.690695,     0.610051,     0.511446,     0.435257,    0.355347,
    0.279466,     0.22443,      0.165315,     0.116251,    0.0741842,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    591.839,      578.967,      565.165,      550.365,     534.496,
    517.48,       499.234,      479.67,       458.692,     436.198,
    412.078,      386.215,      358.483,      328.747,     296.863,
    262.673,      226.014,      186.704,      144.554,     99.3585,
    63.9464,      61.3969,      58.6632,      55.732,      52.5889,
    49.2186,      45.6048,      41.7299,      37.5749,     33.1296,
    31.4902,      29.7323,      27.8473,      25.8262,     23.6589,
    21.8027,      20.4819,      19.0658,      17.5473,     16.1897,
    15.1346,      14.0032,      12.9205,      12.1015,     11.2234,
    10.4149,      9.60233,      8.89935,      8.2936,      7.71032,
    7.12149,      6.57018,      6.05232,      5.5958,      5.24442,
    4.84466,      4.57489,      4.16867,      3.79914,     3.49535,
    3.24662,      2.99409,      2.74766,      2.48503,     2.30768,
    2.09465,      1.91303,      1.75379,      1.56045,     1.43275,
    1.26383,      1.15924,      1.03769,      0.919091,    0.798007,
    0.6987,       0.612178,     0.527073,     0.445102,    0.373466,
    0.300886,     0.237791,     0.177692,     0.127886,    0.0823724,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    572.885,      560.637,      547.503,      533.42,      518.319,
    502.127,      484.764,      466.147,      446.185,     424.78,
    401.828,      377.218,      350.828,      322.532,     292.191,
    259.658,      224.773,      187.367,      147.258,     104.25,
    61.7333,      59.396,       56.8897,      54.2023,     51.3208,
    48.2309,      44.9178,      41.3653,      37.556,      33.4714,
    31.3875,      29.7235,      27.9393,      26.0261,     23.9746,
    21.9851,      20.6109,      19.1374,      17.5574,     16.0328,
    15.0396,      13.9746,      12.8694,      12.088,      11.2501,
    10.443,       9.66487,      8.92022,      8.29903,     7.70267,
    7.14964,      6.55974,      5.99712,      5.56706,     5.25915,
    4.78567,      4.42715,      4.16333,      3.88182,     3.58441,
    3.21811,      3.03132,      2.72389,      2.54324,     2.31705,
    2.11935,      1.90616,      1.75761,      1.60509,     1.43632,
    1.29836,      1.18736,      1.04331,      0.931832,    0.807338,
    0.719998,     0.635761,     0.543811,     0.454945,    0.381837,
    0.309671,     0.240966,     0.188717,     0.132707,    0.0912724,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    582.69,       570.465,      557.358,      543.303,     528.232,
    512.073,      494.745,      476.165,      456.243,     434.881,
    411.975,      387.413,      361.077,      332.838,     302.557,
    270.089,      235.274,      197.943,      157.914,     114.992,
    68.9688,      61.2759,      58.6021,      55.735,      52.6608,
    49.3644,      45.8298,      42.0397,      37.9757,     33.6181,
    30.8745,      29.1954,      27.395,       25.4644,     23.3944,
    21.2078,      20.0095,      18.7246,      17.3468,     15.8806,
    14.85,        13.745,       12.56,        11.8432,     11.1437,
    10.4156,      9.68161,      8.92568,      8.29732,     7.67094,
    7.138,        6.56434,      5.98023,      5.663,       5.24674,
    4.87768,      4.55657,      4.18758,      3.84681,     3.52671,
    3.21977,      3.04221,      2.75279,      2.51634,     2.31926,
    2.13242,      1.91848,      1.74591,      1.57582,     1.44352,
    1.30035,      1.1683,       1.07802,      0.928524,    0.835238,
    0.732724,     0.640331,     0.554863,     0.465505,    0.392724,
    0.319901,     0.254365,     0.199611,     0.146568,    0.100337,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    561.421,      549.837,      537.415,      524.096,     509.814,
    494.5,        478.08,       460.473,      441.593,     421.349,
    399.642,      376.366,      351.409,      324.647,     295.952,
    265.183,      232.19,       196.813,      158.88,      118.205,
    74.5903,      59.0985,      56.7186,      54.1668,     51.4305,
    48.4966,      45.3505,      41.9771,      38.36,       34.4814,
    31.3944,      29.6055,      27.6875,      25.6307,     23.4254,
    21.0607,      19.7018,      18.4801,      17.1702,     15.7655,
    14.8048,      13.8903,      12.9096,      11.9554,     10.975,
    9.9942,       9.40938,      8.7823,       8.28622,     7.75957,
    7.16109,      6.62856,      6.22579,      5.7457,      5.25742,
    4.81567,      4.48585,      4.21957,      3.86001,     3.48207,
    3.25406,      3.00905,      2.73076,      2.53144,     2.29194,
    2.09625,      1.94336,      1.75666,      1.62793,     1.44892,
    1.29856,      1.18193,      1.07287,      0.941968,    0.836642,
    0.743464,     0.643715,     0.560612,     0.475343,    0.394178,
    0.334086,     0.266415,     0.211791,     0.154437,    0.107323,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    571.984,      560.336,      547.847,      534.455,     520.094,
    504.697,      488.186,      470.482,      451.499,     431.144,
    409.318,      385.915,      360.821,      333.913,     305.06,
    274.122,      240.949,      205.378,      167.236,     126.339,
    82.4851,      58.5597,      56.1897,      53.6484,     50.9235,
    48.0016,      44.8686,      41.5092,      37.907,      34.0445,
    30.5679,      29.0971,      27.52,        25.829,      24.0157,
    22.0714,      20.5405,      19.2075,      17.7782,     16.2456,
    15.0343,      13.9847,      12.8592,      11.9397,     11.2355,
    10.4804,      9.77087,      9.01587,      8.3955,      7.7884,
    7.13659,      6.50038,      6.05718,      5.62203,     5.21097,
    4.83765,      4.41624,      4.13157,      3.82476,     3.53611,
    3.29835,      3.00867,      2.75438,      2.57532,     2.30459,
    2.12953,      1.94124,      1.76328,      1.59358,     1.46629,
    1.32209,      1.19718,      1.05391,      0.96741,     0.849503,
    0.75329,      0.665265,     0.567764,     0.50114,     0.420989,
    0.336774,     0.283387,     0.218589,     0.163331,    0.117142,
    0.0767093,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    551.678,      540.615,      528.752,      516.031,     502.392,
    487.766,      472.084,      455.268,      437.237,     417.904,
    397.172,      374.943,      351.107,      325.549,     298.144,
    268.758,      237.249,      203.462,      167.234,     128.387,
    86.7338,      56.1216,      54.0084,      51.7426,     49.3131,
    46.7079,      43.9145,      40.9192,      37.7075,     34.2637,
    30.6811,      29.0715,      27.3456,      25.495,      23.5106,
    21.3828,      19.6921,      18.5832,      17.3941,     16.1191,
    14.946,       13.927,       12.8343,      11.8452,     11.1712,
    10.4485,      9.69695,      8.89857,      8.25312,     7.71509,
    7.20684,      6.69519,      6.23346,      5.73419,     5.23999,
    4.92024,      4.55232,      4.12425,      3.78744,     3.46967,
    3.25835,      3.00468,      2.76935,      2.57259,     2.32454,
    2.11659,      1.94032,      1.76568,      1.6244,      1.44407,
    1.34355,      1.18944,      1.05259,      0.959824,    0.863435,
    0.774738,     0.657719,     0.585679,     0.501436,    0.424103,
    0.351767,     0.287583,     0.227748,     0.1717,      0.125236,
    0.0841667,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    556.782,      545.792,      534.007,      521.371,     507.822,
    493.294,      477.716,      461.012,      443.101,     423.895,
    403.302,      381.22,       357.543,      332.154,     304.931,
    275.74,       244.44,       210.878,      174.89,      136.302,
    94.9251,      56.6286,      54.5299,      52.2795,     49.8665,
    47.2791,      44.5048,      41.5299,      38.34,       34.9197,
    31.2521,      29.2784,      27.4756,      25.5426,     23.47,
    21.2475,      19.273,       18.231,       17.1137,     15.9157,
    14.7826,      13.9823,      13.1242,      12.2056,     11.2296,
    10.1831,      9.45884,      8.98134,      8.3978,      7.66258,
    6.98323,      6.36119,      5.9342,       5.53763,     5.14318,
    4.76777,      4.45302,      4.15139,      3.73376,     3.4897,
    3.26132,      3.00105,      2.76535,      2.5186,      2.35646,
    2.13185,      1.9189,       1.76552,      1.64882,     1.50019,
    1.33668,      1.20663,      1.07648,      0.962617,    0.872615,
    0.770262,     0.68284,      0.594883,     0.502889,    0.435408,
    0.359253,     0.293578,     0.236252,     0.181553,    0.133176,
    0.0895751,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    532.116,      521.831,      510.803,      498.977,     486.298,
    472.701,      458.123,      442.49,       425.728,     407.755,
    388.483,      367.818,      345.659,      321.899,     296.423,
    269.105,      239.813,      208.404,      174.725,     138.612,
    99.8897,      58.3689,      54.5519,      52.2619,     49.8064,
    47.1735,      44.3503,      41.323,       38.077,      34.5964,
    30.8643,      28.6091,      27.0184,      25.3127,     23.4838,
    21.5227,      19.532,       18.415,       17.2174,     15.9332,
    14.6104,      13.7292,      12.7844,      11.7713,     10.9982,
    10.1807,      9.40516,      8.73096,      8.07667,     7.6221,
    7.12867,      6.58724,      6.12757,      5.71401,     5.35346,
    4.93755,      4.50719,      4.11457,      3.78722,     3.51379,
    3.23847,      2.9787,       2.75259,      2.53375,     2.27877,
    2.14666,      1.95732,      1.82545,      1.62367,     1.48326,
    1.35661,      1.20973,      1.08441,      0.985274,    0.878406,
    0.783914,     0.687337,     0.60411,      0.515054,    0.43849,
    0.374097,     0.308858,     0.240992,     0.190193,    0.142572,
    0.09789,      0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    521.032,      511.095,      500.439,      489.013,     476.762,
    463.625,      449.539,      434.435,      418.239,     400.873,
    382.252,      362.285,      340.876,      317.919,     293.303,
    266.908,      238.606,      208.258,      175.717,     140.825,
    103.411,      63.2927,      52.5861,      50.4551,     48.17,
    45.7198,      43.0926,      40.2755,      37.2548,     34.0158,
    30.5427,      28.0059,      26.4731,      24.8296,     23.0673,
    21.1776,      19.1514,      18.0633,      16.9875,     15.834,
    14.5971,      13.6644,      12.7011,      11.6683,     10.9351,
    10.2529,      9.55249,      8.91443,      8.23572,     7.62491,
    7.00128,      6.50628,      6.00856,      5.51066,     5.18766,
    4.82522,      4.47392,      4.15522,      3.79466,     3.47968,
    3.15607,      2.93045,      2.75423,      2.51068,     2.33507,
    2.12491,      1.91143,      1.7754,       1.60904,     1.48727,
    1.31791,      1.2064,       1.10352,      0.988882,    0.898596,
    0.778852,     0.684008,     0.600335,     0.519754,    0.450969,
    0.380706,     0.308082,     0.256402,     0.199668,    0.149244,
    0.105066,     0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    533.217,      523.17,       512.397,      500.845,     488.459,
    475.177,      460.935,      445.665,      429.291,     411.733,
    392.907,      372.72,       351.074,      327.864,     302.977,
    276.291,      247.676,      216.994,      184.094,     148.817,
    110.991,      70.4304,      52.4814,      50.39,       48.1474,
    45.7428,      43.1644,      40.3997,      37.4352,     34.2564,
    30.8479,      27.9694,      26.4877,      24.899,      23.1955,
    21.3689,      19.4103,      18.1419,      17.0689,     15.9184,
    14.6847,      13.6729,      12.6999,      11.6565,     10.8221,
    10.1128,      9.36243,      8.77027,      8.13532,     7.62159,
    7.09634,      6.50513,      5.97367,      5.63698,     5.18244,
    4.76334,      4.43358,      4.12949,      3.83446,     3.52773,
    3.27404,      2.93961,      2.77624,      2.54919,     2.32261,
    2.1064,       1.91683,      1.79618,      1.65226,     1.50209,
    1.34106,      1.22756,      1.08771,      0.986835,    0.871454,
    0.787418,     0.698596,     0.601676,     0.533126,    0.461035,
    0.385,        0.32076,      0.257891,     0.200828,    0.156867,
    0.110583,     0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    517.36,       507.77,       497.488,      486.463,     474.64,
    461.964,      448.371,      433.796,      418.168,     401.41,
    383.441,      364.174,      343.514,      321.361,     297.608,
    272.138,      244.827,      215.542,      184.141,     150.471,
    114.368,      75.6551,      51.7062,      49.744,      47.6399,
    45.3837,      42.9645,      40.3705,      37.589,      34.6065,
    31.4085,      28.3117,      26.7836,      25.145,      23.388,
    21.5041,      19.484,       17.9889,      16.8944,     15.7207,
    14.4622,      13.4049,      12.4999,      11.5295,     10.6703,
    9.987,        9.25436,      8.68583,      8.10849,     7.54925,
    6.97633,      6.50303,      6.03955,      5.60385,     5.1498,
    4.73371,      4.44303,      4.09217,      3.76978,     3.54134,
    3.33151,      3.02496,      2.77093,      2.50846,     2.32777,
    2.09525,      1.92398,      1.77055,      1.62664,     1.47886,
    1.33216,      1.21905,      1.09465,      0.984432,    0.898936,
    0.785448,     0.705294,     0.612666,     0.532852,    0.453163,
    0.393824,     0.331876,     0.272002,     0.217126,    0.164278,
    0.121216,     0.0844204,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    518.705,      509.228,      499.066,      488.169,     476.485,
    463.957,      450.524,      436.119,      420.674,     404.112,
    386.354,      367.312,      346.894,      325.0,       301.525,
    276.353,      249.361,      220.42,       189.386,     156.11,
    120.429,      82.1696,      51.7379,      49.7387,     47.5951,
    45.2965,      42.8319,      40.1891,      37.3553,     34.3167,
    31.0586,      27.565,       26.1401,      24.6345,     23.0201,
    21.289,       19.4328,      17.8654,      16.8225,     15.7043,
    14.5052,      13.4023,      12.5048,      11.5423,     10.623,
    9.95156,      9.23164,      8.59873,      7.98057,     7.41432,
    6.8965,       6.46751,      6.08123,      5.62601,     5.15355,
    4.69008,      4.3687,       4.14263,      3.75608,     3.4368,
    3.28439,      2.96644,      2.73482,      2.5205,      2.30138,
    2.15266,      1.96664,      1.79031,      1.63648,     1.47648,
    1.36787,      1.22734,      1.10674,      0.996511,    0.87325,
    0.787197,     0.695058,     0.625863,     0.535294,    0.465267,
    0.401684,     0.336941,     0.275399,     0.220768,    0.169903,
    0.126534,     0.0875336,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    500.317,      491.339,      481.713,      471.391,     460.323,
    448.455,      435.729,      422.084,      407.453,     391.764,
    374.941,      356.903,      337.562,      316.822,     294.584,
    270.738,      245.17,       217.753,      188.355,     156.833,
    123.033,      86.7902,      51.5748,      49.5316,     47.3406,
    44.9913,      42.4723,      39.7712,      36.8749,     33.7693,
    30.4392,      26.8685,      25.1631,      23.8081,     22.3551,
    20.7971,      19.1266,      17.5613,      16.6279,     15.6271,
    14.5539,      13.4934,      12.6674,      11.7816,     10.8581,
    10.1058,      9.29907,      8.62021,      8.0591,      7.49294,
    6.95118,      6.42985,      5.94004,      5.48809,     5.07281,
    4.74519,      4.31621,      3.98393,      3.69828,     3.48016,
    3.23003,      2.93346,      2.736,        2.50951,     2.33699,
    2.12145,      1.94735,      1.7832,       1.64749,     1.46703,
    1.37303,      1.21634,      1.09247,      1.00211,     0.898669,
    0.794445,     0.705008,     0.622671,     0.555555,    0.477735,
    0.400678,     0.337218,     0.278551,     0.230451,    0.174177,
    0.130495,     0.0909875,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    489.461,      480.816,      471.546,      461.607,     450.948,
    439.52,       427.266,      414.126,      400.037,     384.929,
    368.73,       351.36,       332.734,      312.763,     291.348,
    268.386,      243.765,      217.364,      189.055,     158.7,
    126.152,      91.2513,      53.8286,      49.2496,     47.1195,
    44.8356,      42.3865,      39.7605,      36.9448,     33.9255,
    30.6881,      27.2166,      25.1335,      23.7865,     22.342,
    20.7932,      19.1325,      17.4179,      16.4473,     15.4065,
    14.2905,      13.1238,      12.3168,      11.4515,     10.5237,
    9.81011,      9.0646,       8.36787,      7.80166,     7.2424,
    6.84833,      6.39969,      5.85728,      5.44579,     5.09396,
    4.64831,      4.32809,      4.06005,      3.72996,     3.3813,
    3.16993,      2.92047,      2.6993,       2.50647,     2.27938,
    2.06358,      1.95414,      1.79206,      1.61589,     1.43919,
    1.34995,      1.23531,      1.10863,      1.00023,     0.918971,
    0.80783,      0.706187,     0.626301,     0.54668,     0.474659,
    0.409018,     0.347366,     0.289364,     0.233561,    0.182549,
    0.135869,     0.0973728,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    493.052,      484.44,       475.207,      465.306,     454.69,
    443.306,      431.1,        418.012,      403.978,     388.929,
    372.794,      355.492,      336.939,      317.046,     295.716,
    272.844,      248.319,      222.021,      193.824,     163.588,
    131.167,      96.4038,      59.128,       48.5696,     46.5498,
    44.3841,      42.0619,      39.5718,      36.9018,     34.0388,
    30.9689,      27.6772,      25.3,         23.9774,     22.5592,
    21.0386,      19.4081,      17.6597,      16.5629,     15.467,
    14.2919,      13.0318,      12.2189,      11.4081,     10.5386,
    9.83359,      9.14668,      8.44381,      7.8242,      7.17206,
    6.87578,      6.52411,      5.93406,      5.46587,     5.15344,
    4.70981,      4.35993,      4.02757,      3.66296,     3.42375,
    3.16608,      2.97352,      2.74017,      2.49246,     2.27689,
    2.11479,      1.90734,      1.77099,      1.60587,     1.47745,
    1.32853,      1.21555,      1.12074,      1.00127,     0.907851,
    0.795039,     0.7246,       0.62897,      0.568731,    0.476911,
    0.417554,     0.347923,     0.291632,     0.238849,    0.190169,
    0.140304,     0.102901,     0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    478.735,      470.5,        461.669,      452.199,     442.046,
    431.159,      419.485,      406.967,      393.544,     379.152,
    363.72,       347.172,      329.428,      310.403,     290.002,
    268.127,      244.671,      219.52,       192.551,     163.634,
    132.626,      99.3783,      63.7274,      47.8147,     45.8802,
    43.806,       41.5818,      39.1969,      36.6396,     33.8975,
    30.9573,      27.8046,      25.1434,      23.7925,     22.3441,
    20.791,       19.1256,      17.3399,      16.1964,     15.2395,
    14.2135,      13.1134,      12.206,       11.3323,     10.3954,
    9.64271,      9.00145,      8.32505,      7.84747,     7.33538,
    6.87015,      6.3844,       5.97492,      5.52934,     5.03132,
    4.65428,      4.28641,      3.92935,      3.68052,     3.447,
    3.26466,      2.9314,       2.69883,      2.49321,     2.25339,
    2.09511,      1.92506,      1.77124,      1.63058,     1.45537,
    1.32776,      1.22217,      1.11549,      1.01882,     0.905619,
    0.805754,     0.732781,     0.63074,      0.552624,    0.4933,
    0.411313,     0.353957,     0.294578,     0.247122,    0.196578,
    0.152371,     0.10851,      0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    476.898,      468.776,      460.066,      450.728,     440.714,
    429.977,      418.464,      406.118,      392.881,     378.687,
    363.467,      347.148,      329.649,      310.885,     290.765,
    269.192,      246.059,      221.255,      194.658,     166.139,
    135.559,      102.769,      67.6093,      46.2491,     44.479,
    42.5809,      40.5457,      38.3634,      36.0233,     33.5142,
    30.8237,      27.9388,      25.1745,      23.8534,     22.4369,
    20.918,       19.2894,      17.543,       16.2378,     15.2499,
    14.1905,      13.0546,      12.153,       11.4232,     10.6407,
    9.86574,      9.11548,      8.31099,      7.80632,     7.3133,
    6.84074,      6.35784,      5.9405,       5.51129,     5.02521,
    4.59285,      4.23805,      3.99606,      3.6562,      3.38342,
    3.13271,      2.84475,      2.68545,      2.48295,     2.25764,
    2.12204,      1.91921,      1.7319,       1.62164,     1.45725,
    1.36162,      1.2117,       1.10468,      0.994179,    0.905055,
    0.803766,     0.70626,      0.637939,     0.55834,     0.481708,
    0.421728,     0.366202,     0.302048,     0.24787,     0.195748,
    0.151649,     0.115646,     0.0807946,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    450.717,      443.186,      435.111,      426.453,     417.169,
    407.214,      396.539,      385.093,      372.82,      359.66,
    345.549,      330.418,      314.194,      296.797,     278.143,
    258.14,       236.693,      213.695,      189.036,     162.594,
    134.241,      103.84,       71.2412,      46.0824,     44.3148,
    42.4195,      40.3872,      38.208,       35.8714,     33.3659,
    30.6793,      27.7986,      24.7363,      23.3841,     21.9342,
    20.3796,      18.7125,      16.925,       15.5125,     14.6888,
    13.8056,      12.8585,      11.9991,      11.2984,     10.5471,
    9.76703,      8.99428,      8.16568,      7.64043,     7.21819,
    6.69948,      6.08774,      5.65377,      5.30541,     5.0017,
    4.64441,      4.24131,      3.88879,      3.69212,     3.44204,
    3.15199,      2.85408,      2.64183,      2.45904,     2.26961,
    2.08226,      1.90431,      1.70138,      1.57722,     1.47993,
    1.35444,      1.21441,      1.10079,      0.986989,    0.913107,
    0.825353,     0.713104,     0.649671,     0.555262,    0.487526,
    0.435183,     0.374522,     0.312353,     0.256527,    0.20519,
    0.162469,     0.119329,     0.0857724,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    463.688,      456.026,      447.812,      439.003,     429.558,
    419.43,       408.571,      396.926,      384.441,     371.052,
    356.697,      341.303,      324.798,      307.1,       288.122,
    267.773,      245.954,      222.558,      197.471,     170.571,
    141.727,      110.798,      77.6347,      46.6453,     44.7884,
    42.7973,      40.6623,      38.373,       35.9183,     33.2861,
    30.4638,      27.4375,      24.1924,      22.712,      21.4664,
    20.1308,      18.6987,      17.1631,      15.7453,     14.8514,
    13.8929,      12.8651,      11.8715,      11.1153,     10.3044,
    9.48701,      8.9472,       8.36838,      7.77628,     7.16355,
    6.59624,      6.1298,       5.70398,      5.32224,     4.89377,
    4.51362,      4.27014,      3.92361,      3.66721,     3.42926,
    3.12397,      2.85759,      2.6507,       2.44332,     2.2314,
    2.05348,      1.91723,      1.7791,       1.6139,      1.46425,
    1.32508,      1.22016,      1.0972,       0.989396,    0.911671,
    0.816036,     0.721575,     0.649259,     0.566007,    0.501855,
    0.427597,     0.365682,     0.316221,     0.255626,    0.20305,
    0.163095,     0.122659,     0.0924555,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    458.636,      451.129,      443.081,      434.45,      425.196,
    415.273,      404.633,      393.224,      380.991,     367.873,
    353.808,      338.726,      322.554,      305.213,     286.619,
    266.682,      245.304,      222.38,       197.8,       171.444,
    143.183,      112.88,       80.387,       45.5456,     43.0837,
    41.2882,      39.363,       37.2985,      35.085,      32.7114,
    30.1663,      27.4373,      24.511,       22.821,      21.6382,
    20.3699,      19.0099,      17.5516,      16.0745,     15.1434,
    14.1449,      13.0744,      11.9759,      11.2006,     10.3693,
    9.47798,      8.87493,      8.2342,       7.62962,     7.10042,
    6.57492,      6.14768,      5.67001,      5.12071,     4.81303,
    4.57434,      4.12834,      3.92779,      3.65947,     3.38166,
    3.10222,      2.8205,       2.63787,      2.42625,     2.2786,
    2.06107,      1.8951,       1.74539,      1.57027,     1.45282,
    1.30851,      1.19496,      1.08704,      1.00734,     0.906324,
    0.818908,     0.715345,     0.658929,     0.566593,    0.487476,
    0.439338,     0.37226,      0.30764,      0.263477,    0.205556,
    0.163971,     0.13207,      0.0956737,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    442.6,        435.482,      427.849,      419.665,     410.889,
    401.48,       391.39,       380.571,      368.97,      356.531,
    343.193,      328.891,      313.555,      297.112,     279.479,
    260.573,      240.3,        218.563,      195.254,     170.261,
    143.462,      114.726,      83.9129,      50.8735,     43.4589,
    41.6596,      39.7303,      37.6615,      35.4432,     33.0647,
    30.5142,      27.7794,      24.847,       22.819,      21.5821,
    20.2558,      18.8336,      17.3086,      15.6735,     14.8154,
    13.9254,      12.9711,      11.9479,      11.1931,     10.3983,
    9.54604,      8.88267,      8.22371,      7.58614,     7.10019,
    6.58973,      6.15413,      5.69169,      5.21488,     4.83984,
    4.55836,      4.17103,      3.83636,      3.61296,     3.41959,
    3.07362,      2.83176,      2.60395,      2.41544,     2.20712,
    2.05229,      1.86332,      1.73146,      1.58721,     1.44481,
    1.28815,      1.21781,      1.10104,      0.978201,    0.890487,
    0.806092,     0.734829,     0.661849,     0.566341,    0.496241,
    0.432043,     0.37692,      0.31571,      0.267517,    0.215517,
    0.170917,     0.133201,     0.0976286,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    434.301,      427.413,      420.028,      412.108,     403.617,
    394.511,      384.748,      374.279,      363.053,     351.017,
    338.11,       324.271,      309.431,      293.519,     276.457,
    258.163,      238.546,      217.511,      194.956,     170.772,
    144.839,      117.033,      87.217,       55.2463,     43.2073,
    41.4045,      39.4713,      37.3985,      35.1759,     32.7926,
    30.2371,      27.497,       24.5588,      22.2476,     21.0635,
    19.7939,      18.4325,      16.9727,      15.4074,     14.4247,
    13.5312,      12.5731,      11.5458,      10.8137,     10.1184,
    9.3728,       8.7785,       8.24048,      7.6646,      7.0552,
    6.40175,      6.01309,      5.61548,      5.1542,      4.78151,
    4.58616,      4.17419,      3.84343,      3.62067,     3.37838,
    3.01728,      2.81566,      2.61289,      2.41617,     2.19781,
    2.05079,      1.88775,      1.71331,      1.61666,     1.43624,
    1.31073,      1.18092,      1.10038,      1.00624,     0.882176,
    0.808503,     0.717802,     0.640194,     0.572602,    0.497423,
    0.439851,     0.380227,     0.320438,     0.268335,    0.219685,
    0.1764,       0.139558,     0.101171,     0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    422.762,      416.145,      409.051,      401.443,     393.286,
    384.54,       375.161,      365.105,      354.321,     342.759,
    330.361,      317.067,      302.812,      287.527,     271.138,
    253.564,      234.72,       214.514,      192.848,     169.617,
    144.706,      117.995,      89.3542,      58.6433,     42.4413,
    40.6773,      38.7858,      36.7576,      34.5828,     32.2509,
    29.7505,      27.0694,      24.1945,      21.651,      20.5707,
    19.4124,      18.1704,      16.8387,      15.4107,     14.348,
    13.4482,      12.4834,      11.4488,      10.6598,     9.9835,
    9.25835,      8.65417,      8.16364,      7.63766,     7.09669,
    6.51744,      6.04958,      5.58995,      5.24451,     4.89779,
    4.55005,      4.15376,      3.79699,      3.54488,     3.29778,
    3.05507,      2.8639,       2.64679,      2.41115,     2.18621,
    2.03981,      1.85182,      1.74576,      1.5666,      1.42489,
    1.31861,      1.17253,      1.0768,       0.973734,    0.874187,
    0.799314,     0.711274,     0.652499,     0.562367,    0.498075,
    0.438369,     0.375048,     0.326839,     0.273903,    0.221564,
    0.180741,     0.140645,     0.10678,      0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    434.895,      428.132,      420.881,      413.106,     404.769,
    395.829,      386.243,      375.965,      364.943,     353.125,
    340.454,      326.866,      312.297,      296.674,     279.923,
    261.961,      242.701,      222.049,      199.904,     176.16,
    150.699,      123.398,      94.1249,      62.7358,     41.524,
    39.8916,      38.1412,      36.2643,      34.2517,     32.0938,
    29.7799,      27.2987,      24.6383,      21.9804,     20.8292,
    19.5949,      18.2713,      16.852,       15.3302,     14.1416,
    13.2865,      12.3697,      11.3865,      10.5941,     10.0001,
    9.3631,       8.7386,       8.1653,       7.55058,     7.04787,
    6.5445,       6.01173,      5.44445,      5.08214,     4.77616,
    4.44614,      4.10849,      3.7773,       3.48609,     3.28695,
    3.05077,      2.81246,      2.52191,      2.37497,     2.19477,
    2.03479,      1.8935,       1.73242,      1.55972,     1.40898,
    1.33786,      1.17755,      1.08419,      0.965307,    0.910658,
    0.807681,     0.708936,     0.641456,     0.571004,    0.506633,
    0.434639,     0.376266,     0.330367,     0.272037,    0.226816,
    0.18282,      0.1431,       0.105238,     0.081338,    0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    412.006,      405.685,      398.907,      391.64,      383.847,
    375.491,      366.531,      356.923,      346.622,     335.575,
    323.731,      311.03,       297.412,      282.81,      267.152,
    250.363,      232.36,       213.056,      192.358,     170.163,
    146.365,      120.847,      93.4845,      64.1448,     39.8935,
    38.3644,      36.7249,      34.9668,      33.0817,     31.0603,
    28.8929,      26.5688,      24.0768,      21.4047,     20.2996,
    19.1886,      17.9974,      16.7201,      15.3505,     14.1406,
    13.2856,      12.3688,      11.3857,      10.4971,     9.83688,
    9.12898,      8.43184,      7.88664,      7.30205,     6.79888,
    6.32124,      5.88767,      5.50472,      5.08764,     4.63613,
    4.32992,      4.03244,      3.74205,      3.46901,     3.26452,
    3.00246,      2.77715,      2.58946,      2.39479,     2.14856,
    1.9947,       1.86692,      1.67918,      1.56075,     1.4476,
    1.30952,      1.16521,      1.0789,       0.97222,     0.877609,
    0.802972,     0.728194,     0.628138,     0.569949,    0.494408,
    0.437358,     0.380482,     0.323732,     0.275296,    0.231066,
    0.186446,     0.146044,     0.11009,      0.0847113,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    402.518,      396.436,      389.914,      382.92,      375.421,
    367.381,      358.759,      349.514,      339.601,     328.972,
    317.574,      305.353,      292.249,      278.197,     263.13,
    246.975,      229.651,      211.076,      191.159,     169.802,
    146.902,      122.346,      96.0166,      67.7841,     40.1919,
    38.6672,      37.0323,      35.2793,      33.3996,     31.384,
    29.2228,      26.9054,      24.4205,      21.7561,     20.3822,
    19.2526,      18.0414,      16.7426,      15.35,       14.0225,
    13.1626,      12.2406,      11.2519,      10.3093,     9.71664,
    9.08111,      8.42084,      7.91717,      7.37709,     6.88132,
    6.42608,      5.96178,      5.50888,      5.08134,     4.69189,
    4.33484,      4.00987,      3.7614,       3.44865,     3.21158,
    2.95971,      2.70699,      2.53572,      2.33994,     2.12915,
    1.94682,      1.81303,      1.67328,      1.54701,     1.41019,
    1.29138,      1.18045,      1.07867,      0.968232,    0.876193,
    0.813179,     0.709445,     0.628517,     0.554015,    0.509443,
    0.43051,      0.38404,      0.319893,     0.279733,    0.232835,
    0.189422,     0.148404,     0.112369,     0.0817751,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    391.298,      385.478,      379.237,      372.545,     365.369,
    357.675,      349.424,      340.578,      331.092,     320.921,
    310.014,      298.32,       285.78,       272.334,     257.916,
    242.457,      225.88,       208.105,      189.046,     168.609,
    146.695,      123.198,      98.003,       70.9869,     42.0185,
    38.9534,      37.2604,      35.4451,      33.4986,     31.4114,
    29.1733,      26.7736,      24.2004,      21.4412,     19.8223,
    18.7541,      17.6087,      16.3805,      15.0635,     13.7193,
    12.9786,      12.1844,      11.3328,      10.4465,     9.80273,
    9.1124,       8.37217,      7.82475,      7.24983,     6.69341,
    6.19473,      5.70948,      5.37916,      5.00465,     4.55916,
    4.28623,      4.08573,      3.76561,      3.42482,     3.15318,
    2.92952,      2.71519,      2.5288,       2.32491,     2.14376,
    1.95729,      1.80386,      1.63392,      1.56516,     1.40946,
    1.3084,       1.19756,      1.06973,      0.986113,    0.88121,
    0.790633,     0.710299,     0.648519,     0.569474,    0.514913,
    0.443342,     0.386946,     0.326093,     0.274055,    0.240941,
    0.187671,     0.151144,     0.118669,     0.0846903,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    393.987,      388.177,      381.948,      375.269,     368.107,
    360.427,      352.193,      343.363,      333.895,     323.743,
    312.857,      301.185,      288.669,      275.249,     260.858,
    245.428,      228.883,      211.142,      192.119,     171.721,
    149.849,      126.397,      101.249,      74.2846,     45.3713,
    38.339,       36.7237,      34.9917,      33.1345,     31.1431,
    29.0077,      26.7181,      24.263,       21.6304,     19.8034,
    18.7427,      17.6054,      16.3858,      15.0781,     13.6759,
    12.8403,      11.9801,      11.0578,      10.0689,     9.50684,
    8.9355,       8.32287,      7.78479,      7.23589,     6.67867,
    6.17975,      5.66293,      5.35203,      5.00796,     4.58956,
    4.21954,      3.90104,      3.63208,      3.40786,     3.18959,
    2.93777,      2.69135,      2.50213,      2.28148,     2.14393,
    1.97909,      1.79601,      1.67458,      1.52897,     1.39717,
    1.28784,      1.14553,      1.06523,      0.952243,    0.87794,
    0.790428,     0.696665,     0.642422,     0.568055,    0.48488,
    0.426859,     0.38224,      0.329285,     0.284615,    0.23885,
    0.196369,     0.154793,     0.119253,     0.0927617,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    382.784,      377.195,      371.203,      364.778,     357.889,
    350.502,      342.581,      334.087,      324.98,      315.215,
    304.743,      293.516,      281.476,      268.567,     254.725,
    239.882,      223.967,      206.901,      188.603,     168.982,
    147.943,      125.383,      101.193,      75.2556,     47.4433,
    36.8704,      35.4326,      33.8909,      32.2378,     30.4652,
    28.5645,      26.5265,      24.3412,      21.9979,     20.071,
    18.9328,      17.7123,      16.4037,      15.0005,     13.4958,
    12.6482,      11.9234,      11.1462,      10.3128,     9.58154,
    8.8385,       8.04176,      7.49058,      7.0506,      6.57767,
    6.06073,      5.50642,      5.23785,      4.97208,     4.63883,
    4.30593,      3.99105,      3.61441,      3.28768,     3.06147,
    2.92369,      2.69597,      2.51036,      2.30462,     2.10191,
    1.91193,      1.76825,      1.64553,      1.46762,     1.38483,
    1.26335,      1.17255,      1.05919,      0.946294,    0.855625,
    0.790523,     0.710913,     0.6294,       0.553771,    0.504416,
    0.446126,     0.38354,      0.328687,     0.279525,    0.229177,
    0.193237,     0.156315,     0.123595,     0.095122,    0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    374.405,      369.022,      363.249,      357.06,      350.423,
    343.306,      335.675,      327.493,      318.719,     309.311,
    299.223,      288.407,      276.808,      264.372,     251.037,
    236.738,      221.405,      204.965,      187.336,     168.434,
    148.166,      126.432,      103.129,      78.1408,     51.3471,
    37.3366,      35.8028,      34.1581,      32.3945,     30.5035,
    28.4758,      26.3016,      23.9703,      21.4705,     19.2566,
    18.2746,      17.2216,      16.0925,      14.8818,     13.5836,
    12.6405,      11.854,       11.0106,      10.1063,     9.43284,
    8.86449,      8.25506,      7.6909,       7.16556,     6.60225,
    6.10962,      5.58483,      5.2411,       4.93098,     4.56823,
    4.19377,      3.85108,      3.62027,      3.39668,     3.1245,
    2.8791,       2.69889,      2.46108,      2.27857,     2.09403,
    1.93056,      1.80631,      1.66273,      1.5075,      1.38001,
    1.26576,      1.16568,      1.05319,      0.935186,    0.84359,
    0.784362,     0.680041,     0.615241,     0.563106,    0.497339,
    0.436801,     0.378725,     0.331648,     0.274306,    0.238936,
    0.195343,     0.158629,     0.120704,     0.0941142,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    369.343,      364.085,      358.447,      352.402,     345.92,
    338.97,       331.517,      323.526,      314.957,     305.769,
    295.917,      285.353,      274.026,      261.88,      248.856,
    234.891,      219.917,      203.86,       186.644,     168.183,
    148.388,      127.162,      104.403,      79.9986,     53.8308,
    36.5291,      35.0897,      33.5463,      31.8913,     30.1168,
    28.214,       26.1736,      23.9859,      21.64,       19.3189,
    18.2861,      17.1787,      15.9913,      14.718,      13.3527,
    12.3263,      11.615,       10.8523,      10.0345,     9.34768,
    8.78596,      8.18365,      7.57156,      6.96741,     6.3196,
    5.94395,      5.60652,      5.26092,      4.89903,     4.47359,
    4.03556,      3.83807,      3.59541,      3.32141,     3.05036,
    2.86524,      2.65545,      2.42854,      2.26376,     2.09161,
    1.88545,      1.75438,      1.60167,      1.49084,     1.37339,
    1.27902,      1.14235,      1.04936,      0.94416,     0.863029,
    0.777914,     0.690253,     0.626967,     0.568554,    0.497937,
    0.433206,     0.380037,     0.32814,      0.287105,    0.240103,
    0.194825,     0.16092,      0.121807,     0.0972909,   0.075045,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    367.093,      361.911,      356.356,      350.398,     344.011,
    337.161,      329.817,      321.941,      313.497,     304.443,
    294.734,      284.323,      273.16,       261.191,     248.356,
    234.594,      219.837,      204.014,      187.047,     168.855,
    149.347,      128.43,       106.001,      81.9514,     56.1637,
    35.537,       34.1824,      32.7299,      31.1724,     29.5024,
    27.7116,      25.7915,      23.7326,      21.5249,     19.1577,
    18.1164,      17.021,       15.8463,      14.5868,     13.2363,
    12.1254,      11.4518,      10.7296,      9.95513,     9.22407,
    8.59774,      7.92615,      7.28933,      6.84402,     6.36654,
    5.97442,      5.60717,      5.21553,      4.79761,     4.41112,
    4.03333,      3.84761,      3.59928,      3.28472,     3.08245,
    2.80034,      2.64355,      2.42584,      2.27213,     2.0631,
    1.8942,       1.78664,      1.63593,      1.47706,     1.37362,
    1.23237,      1.12278,      1.04156,      0.959094,    0.876525,
    0.777953,     0.69872,      0.619289,     0.570973,    0.498556,
    0.427024,     0.384166,     0.322616,     0.285227,    0.238543,
    0.201657,     0.156714,     0.12785,      0.10137,     0.0761816,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    364.544,      359.465,      354.018,      348.179,     341.917,
    335.202,      328.003,      320.283,      312.005,     303.129,
    293.612,      283.407,      272.464,      260.73,      248.149,
    234.658,      220.193,      204.682,      188.05,      170.216,
    151.093,      130.588,      108.602,      85.0266,     59.7476,
    35.986,       34.5278,      32.9642,      31.2876,     29.4899,
    27.5622,      25.4953,      23.2789,      20.9024,     18.3542,
    17.228,       16.3067,      15.3188,      14.2595,     13.1237,
    12.0679,      11.3909,      10.665,       9.88668,     9.13734,
    8.58319,      7.989,        7.38147,      6.9299,      6.4457,
    6.01804,      5.63201,      5.22351,      4.79428,     4.41252,
    4.08383,      3.81141,      3.56215,      3.34792,     3.05716,
    2.8218,       2.6217,       2.42285,      2.23057,     2.06711,
    1.88478,      1.73673,      1.62105,      1.45404,     1.33779,
    1.23048,      1.14661,      1.02535,      0.929918,    0.861803,
    0.742091,     0.685393,     0.614659,     0.560491,    0.506525,
    0.437722,     0.381236,     0.330984,     0.285345,    0.236975,
    0.205802,     0.156224,     0.129957,     0.0994488,   0.0745784,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    364.564,      359.52,       354.113,      348.315,     342.097,
    335.431,      328.282,      320.617,      312.398,     303.585,
    294.135,      284.003,      273.138,      261.487,     248.995,
    235.6,        221.237,      205.837,      189.323,     171.616,
    152.629,      132.27,       110.439,      87.0313,     61.9317,
    35.0182,      33.5209,      32.0889,      30.5534,     28.907,
    27.1416,      25.2486,      23.2188,      21.0423,     18.7086,
    17.3874,      16.4438,      15.4321,      14.3472,     13.1839,
    12.0245,      11.3526,      10.6322,      9.85974,     9.07298,
    8.51875,      7.92446,      7.28735,      6.75767,     6.18972,
    5.73605,      5.45905,      5.15802,      4.82335,     4.45716,
    4.05157,      3.72369,      3.45372,      3.30077,     3.02054,
    2.79663,      2.58197,      2.35844,      2.22115,     2.09446,
    1.89338,      1.74832,      1.59958,      1.45801,     1.36069,
    1.22485,      1.12158,      1.03679,      0.94631,     0.834574,
    0.763053,     0.684399,     0.615995,     0.563038,    0.496976,
    0.429503,     0.374553,     0.322189,     0.279983,    0.237284,
    0.191802,     0.159192,     0.133319,     0.10499,     0.0769523,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    351.097,      346.309,      341.176,      335.672,     329.77,
    323.441,      316.655,      309.379,      301.577,     293.211,
    284.24,       274.621,      264.307,      253.248,     241.389,
    228.673,      215.039,      200.419,      184.742,     167.933,
    149.909,      130.582,      109.859,      87.6382,     63.8114,
    38.2627,      33.4978,      32.1039,      30.6093,     29.0067,
    27.2883,      25.4457,      23.47,        21.3514,     19.0798,
    17.5676,      16.6102,      15.5836,      14.4827,     13.3024,
    12.0406,      11.289,       10.4831,      9.61901,     8.69245,
    8.26187,      7.80097,      7.30676,      6.87983,     6.43797,
    5.97485,      5.50365,      5.02631,      4.71203,     4.3889,
    4.08815,      3.80553,      3.53289,      3.23167,     3.00019,
    2.77015,      2.53073,      2.3283,       2.16018,     2.03716,
    1.87135,      1.67891,      1.57406,      1.46468,     1.34997,
    1.20356,      1.13732,      0.997673,     0.933671,    0.834416,
    0.747293,     0.686885,     0.630054,     0.537984,    0.490037,
    0.425977,     0.377869,     0.322315,     0.273996,    0.242561,
    0.193433,     0.159689,     0.127959,     0.102266,    0.0784558,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    345.92,       341.261,      336.265,      330.907,     325.163,
    319.003,      312.399,      305.316,      297.723,     289.58,
    280.849,      271.487,      261.448,      250.684,     239.142,
    226.766,      213.496,      199.266,      184.008,     167.648,
    150.105,      131.295,      111.125,      89.4971,     66.3066,
    41.4401,      33.5324,      32.1044,      30.5732,     28.9313,
    27.1708,      25.283,       23.2589,      21.0884,     18.7611,
    17.0461,      16.153,       15.1953,      14.1685,     13.0674,
    11.8868,      11.1374,      10.4087,      9.62745,     8.78971,
    8.22299,      7.66702,      7.07087,      6.6262,      6.21999,
    5.80432,      5.45638,      5.08331,      4.66525,     4.24627,
    4.04425,      3.79177,      3.47396,      3.23899,     3.01062,
    2.76299,      2.49834,      2.32414,      2.1541,      1.99141,
    1.84887,      1.70677,      1.52072,      1.46018,     1.35458,
    1.19372,      1.11157,      1.00258,      0.929061,    0.833642,
    0.763831,     0.673741,     0.618303,     0.545052,    0.495316,
    0.42752,      0.370346,     0.315217,     0.281868,    0.23829,
    0.19762,      0.157078,     0.131906,     0.100091,    0.0814856,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    338.676,      334.153,      329.304,      324.105,     318.529,
    312.551,      306.141,      299.267,      291.897,     283.994,
    275.52,       266.433,      256.69,       246.243,     235.041,
    223.029,      210.149,      196.338,      181.53,      165.651,
    148.624,      130.368,      110.791,      89.8005,     67.2926,
    43.1581,      32.3858,      31.0312,      29.5787,     28.0212,
    26.3512,      24.5604,      22.6403,      20.5814,     18.3737,
    16.5513,      15.7302,      14.8496,      13.9054,     12.893,
    11.8074,      11.0397,      10.354,       9.61885,     8.83055,
    8.26474,      7.75924,      7.21721,      6.76019,     6.35155,
    5.91507,      5.48328,      5.02029,      4.63415,     4.23718,
    3.9345,       3.64586,      3.41259,      3.18162,     2.96189,
    2.74844,      2.50599,      2.31797,      2.12207,     1.97172,
    1.84643,      1.69315,      1.51635,      1.43597,     1.336,
    1.20172,      1.10374,      0.983705,     0.895526,    0.820947,
    0.782307,     0.680109,     0.594065,     0.548752,    0.490233,
    0.429052,     0.368192,     0.323359,     0.285247,    0.238707,
    0.193321,     0.161533,     0.132836,     0.10389,     0.0826297,
    0.0,          0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    338.271,      333.806,      329.019,      323.885,     318.381,
    312.478,      306.15,       299.363,      292.087,     284.285,
    275.918,      266.947,      257.328,      247.014,     235.954,
    224.095,      211.379,      197.744,      183.124,     167.447,
    150.637,      132.613,      113.286,      92.5616,     70.3401,
    46.5127,      32.7809,      31.4019,      29.9233,     28.3378,
    26.6377,      24.8147,      22.8601,      20.7641,     18.5167,
    16.452,       15.6199,      14.7277,      13.771,      12.7452,
    11.6453,      10.8349,      10.1983,      9.51577,     8.7839,
    8.17125,      7.62643,      7.04223,      6.52901,     6.10302,
    5.64624,      5.27214,      4.88157,      4.53789,     4.19629,
    3.86022,      3.53738,      3.33739,      3.11839,     2.89616,
    2.6884,       2.47254,      2.27539,      2.15985,     1.94311,
    1.81613,      1.66996,      1.51003,      1.40706,     1.28665,
    1.1872,       1.10932,      0.99747,      0.888722,    0.800405,
    0.754033,     0.680331,     0.594485,     0.535284,    0.489158,
    0.426829,     0.373097,     0.31827,      0.283342,    0.22932,
    0.198357,     0.162096,     0.127971,     0.100231,    0.0810307,
    0.0653582,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    329.81,       325.493,      320.864,      315.901,     310.578,
    304.871,      298.752,      292.19,       285.154,     277.61,
    269.52,       260.846,      251.545,      241.572,     230.878,
    219.412,      207.116,      193.932,      179.796,     164.637,
    148.384,      130.955,      112.268,      92.2292,     70.7427,
    47.7035,      31.489,       30.2487,      28.9187,     27.4926,
    25.9635,      24.3238,      22.5657,      20.6805,     18.659,
    16.6059,      15.7732,      14.8803,      13.9229,     12.8963,
    11.7955,      10.8943,      10.2217,      9.50045,     8.7271,
    8.09812,      7.64124,      7.15134,      6.65438,     6.17384,
    5.65857,      5.24545,      4.84016,      4.4824,      4.14817,
    3.84364,      3.54222,      3.28902,      3.08761,     2.91004,
    2.65288,      2.46104,      2.28873,      2.07536,     1.9586,
    1.72593,      1.63139,      1.50495,      1.36541,     1.26543,
    1.17628,      1.06301,      1.00165,      0.897101,    0.806724,
    0.727351,     0.668072,     0.598823,     0.537192,    0.472716,
    0.409427,     0.361114,     0.319376,     0.271871,    0.236377,
    0.195645,     0.155693,     0.128009,     0.104318,    0.0816431,
    0.0655992,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    330.83,       326.517,      321.893,      316.934,     311.618,
    305.917,      299.804,      293.249,      286.221,     278.685,
    270.604,      261.939,      252.648,      242.685,     232.003,
    220.548,      208.266,      195.096,      180.974,     165.832,
    149.596,      132.186,      113.518,      93.501,      72.0374,
    49.0227,      29.8749,      28.7421,      27.5273,     26.2248,
    24.8282,      23.3306,      21.7248,      20.003,      18.1567,
    16.177,       15.3475,      14.5212,      13.6352,     12.6851,
    11.6664,      10.7557,      10.0974,      9.39152,     8.63465,
    7.95535,      7.4669,       6.94315,      6.4113,      5.94139,
    5.43753,      5.07778,      4.78478,      4.47274,     4.14042,
    3.83143,      3.53229,      3.27448,      3.05284,     2.88007,
    2.68205,      2.42691,      2.2781,       2.1146,      1.91971,
    1.7856,       1.61727,      1.5159,       1.34721,     1.27009,
    1.15223,      1.07943,      0.97279,      0.886426,    0.826852,
    0.72457,      0.653765,     0.583237,     0.553454,    0.468747,
    0.427327,     0.365337,     0.314584,     0.278537,    0.230643,
    0.197893,     0.162326,     0.130406,     0.104674,    0.0837476,
    0.0767045,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    314.994,      310.951,      306.616,      301.967,     296.982,
    291.637,      285.906,      279.761,      273.171,     266.106,
    258.529,      250.406,      241.695,      232.354,     222.339,
    211.6,        200.085,      187.737,      174.497,     160.301,
    145.078,      128.756,      111.254,      92.4865,     72.3632,
    50.7857,      30.0255,      28.8626,      27.6157,     26.2787,
    24.845,       23.3078,      21.6594,      19.892,      17.9968,
    15.9646,      14.9744,      14.1619,      13.2907,     12.3565,
    11.3548,      10.4267,      9.89037,      9.31523,     8.69853,
    8.07076,      7.50749,      6.90352,      6.28883,     5.90174,
    5.48668,      5.12369,      4.80543,      4.48178,     4.16592,
    3.8781,       3.6265,       3.30714,      2.99332,     2.74711,
    2.62428,      2.46708,      2.26985,      2.09185,     1.93132,
    1.73544,      1.60498,      1.49072,      1.37037,     1.26378,
    1.14419,      1.04914,      0.987014,     0.877923,    0.801147,
    0.724836,     0.647364,     0.58415,      0.520548,    0.469098,
    0.410174,     0.362225,     0.320685,     0.258194,    0.232873,
    0.204866,     0.157522,     0.133577,     0.105137,    0.0838651,
    0.0669579,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    300.143,      296.351,      292.285,      287.926,     283.251,
    278.239,      272.864,      267.101,      260.922,     254.296,
    247.191,      239.572,      231.403,      222.644,     213.252,
    203.181,      192.382,      180.802,      168.386,     155.073,
    140.798,      125.49,       109.077,      91.4777,     72.6064,
    52.3712,      30.6738,      29.1402,      27.9044,     26.5793,
    25.1585,      23.6349,      22.0013,      20.2496,     18.3714,
    16.3573,      15.1731,      14.3171,      13.3994,     12.4153,
    11.36,        10.3077,      9.74936,      9.15072,     8.50881,
    7.84382,      7.31178,      6.74128,      6.12955,     5.8138,
    5.47913,      5.11101,      4.70317,      4.31434,     4.05073,
    3.75167,      3.40053,      3.20652,      3.05636,     2.77875,
    2.53399,      2.35732,      2.19281,      2.02849,     1.87825,
    1.74706,      1.62338,      1.47791,      1.35129,     1.28422,
    1.12094,      1.05711,      0.947897,     0.87857,     0.790974,
    0.724531,     0.637265,     0.579673,     0.513899,    0.456323,
    0.415355,     0.360166,     0.316264,     0.268885,    0.230042,
    0.197344,     0.159458,     0.129447,     0.102837,    0.0787044,
    0.0653286,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    292.513,      288.847,      284.916,      280.701,     276.181,
    271.335,      266.138,      260.566,      254.591,     248.185,
    241.315,      233.949,      226.051,      217.582,     208.5,
    198.763,      188.322,      177.126,      165.122,     152.249,
    138.447,      123.647,      107.777,      90.761,      72.5149,
    52.9502,      31.9716,      28.0134,      26.8687,     25.6414,
    24.3253,      22.9142,      21.401,       19.7786,     18.0388,
    16.1734,      14.9202,      14.1181,      13.2581,     12.336,
    11.3472,      10.2894,      9.7184,       9.10616,     8.44967,
    7.74574,      7.24293,      6.70471,      6.12759,     5.72422,
    5.32566,      4.9364,       4.61046,      4.27471,     4.01398,
    3.73398,      3.43231,      3.1776,       2.96236,     2.78087,
    2.54829,      2.33918,      2.17453,      1.99754,     1.83653,
    1.70025,      1.58728,      1.49817,      1.31546,     1.2216,
    1.10814,      1.01175,      0.938969,     0.865578,    0.794447,
    0.701769,     0.660668,     0.568352,     0.518917,    0.453763,
    0.409091,     0.357301,     0.314884,     0.27429,     0.230933,
    0.191312,     0.159849,     0.128342,     0.103527,    0.0801145,
    0.0688928,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    298.395,      294.676,      290.687,      286.411,     281.825,
    276.908,      271.636,      265.983,      259.921,     253.421,
    246.452,      238.978,      230.965,      222.373,     213.159,
    203.28,       192.687,      181.328,      169.149,     156.089,
    142.085,      127.07,       110.969,      93.7049,     75.193,
    55.3433,      34.0592,      27.5386,      26.4512,     25.2852,
    24.035,       22.6944,      21.2569,      19.7156,     18.0628,
    16.2906,      14.9292,      14.1035,      13.2182,     12.2689,
    11.251,       10.1595,      9.50665,      8.87665,     8.20112,
    7.47677,      7.06576,      6.67847,      6.2632,      5.79545,
    5.28605,      4.79812,      4.54757,      4.27895,     3.99614,
    3.69706,      3.40901,      3.15883,      2.96542,     2.71565,
    2.53245,      2.36235,      2.14336,      1.99537,     1.85243,
    1.73111,      1.63655,      1.4546,       1.32648,     1.22391,
    1.10807,      1.02931,      0.93038,      0.850393,    0.775097,
    0.697141,     0.645798,     0.559175,     0.516795,    0.46284,
    0.417346,     0.351895,     0.30698,      0.266765,    0.222643,
    0.19275,      0.156812,     0.136257,     0.106028,    0.0811875,
    0.0664795,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    292.154,      288.546,      284.678,      280.53,      276.082,
    271.313,      266.199,      260.716,      254.836,     248.532,
    241.771,      234.523,      226.75,       218.416,     209.479,
    199.897,      189.622,      178.605,      166.791,     154.124,
    140.541,      125.976,      110.359,      93.6139,     75.6582,
    56.405,       35.7603,      26.9565,      25.9132,     24.7944,
    23.5948,      22.3085,      20.9292,      19.4503,     17.8645,
    16.1641,      14.7298,      13.9435,      13.1003,     12.1962,
    11.2268,      10.1873,      9.50307,      8.90613,     8.26606,
    7.57972,      7.09679,      6.6629,       6.19766,     5.74733,
    5.29415,      4.82503,      4.58442,      4.32642,     4.00245,
    3.64879,      3.41752,      3.16619,      2.88886,     2.76103,
    2.61185,      2.39848,      2.17434,      2.00521,     1.81678,
    1.67238,      1.56331,      1.46339,      1.35728,     1.21228,
    1.12336,      1.00862,      0.916836,     0.866712,    0.761607,
    0.689958,     0.630851,     0.582636,     0.512144,    0.465159,
    0.394525,     0.347122,     0.304623,     0.270729,    0.227185,
    0.178906,     0.151902,     0.128233,     0.101342,    0.0779875,
    0.0629119,    0.0,          0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    288.705,      285.168,      281.376,      277.31,      272.95,
    268.275,      263.262,      257.886,      252.123,     245.943,
    239.316,      232.21,       224.591,      216.421,     207.661,
    198.267,      188.195,      177.395,      165.814,     153.397,
    140.082,      125.805,      110.496,      94.0809,     76.4794,
    57.6059,      37.3685,      26.2933,      25.2517,     24.1349,
    22.9375,      21.6534,      20.2766,      18.8003,     17.2172,
    15.5198,      13.9931,      13.3677,      12.6972,     11.9782,
    11.2072,      10.3805,      9.69714,      9.07898,     8.41614,
    7.7054,       7.13691,      6.64042,      6.10805,     5.64896,
    5.26753,      4.85854,      4.54024,      4.20629,     3.93181,
    3.6636,       3.37989,      3.11126,      2.95881,     2.70405,
    2.46402,      2.33003,      2.1546,       1.96439,     1.78477,
    1.65154,      1.54356,      1.43103,      1.30869,     1.23581,
    1.13661,      1.01076,      0.938725,     0.864422,    0.746446,
    0.685831,     0.608786,     0.571404,     0.513312,    0.455407,
    0.38639,      0.33988,      0.300424,     0.259084,    0.228863,
    0.186783,     0.153768,     0.128123,     0.101628,    0.0843698,
    0.0619447,    0.0526649,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    286.954,      283.473,      279.742,      275.74,      271.449,
    266.848,      261.915,      256.625,      250.952,     244.87,
    238.349,      231.356,      223.857,      215.817,     207.196,
    197.951,      188.039,      177.41,       166.013,     153.793,
    140.689,      126.639,      111.573,      95.4179,     78.0957,
    59.5217,      39.6053,      26.3113,      25.2865,     24.1877,
    23.0095,      21.7461,      20.3914,      18.9388,     17.3812,
    15.7111,      14.0506,      13.3198,      12.5361,     11.6958,
    10.7948,      9.82866,      9.08424,      8.55342,     7.98423,
    7.37391,      6.84519,      6.39726,      5.91696,     5.48862,
    5.16746,      4.82309,      4.51412,      4.19586,     3.86766,
    3.52297,      3.27696,      3.05549,      2.84305,     2.64262,
    2.45478,      2.27945,      2.12036,      1.93863,     1.76797,
    1.72294,      1.53926,      1.43275,      1.29539,     1.21069,
    1.06997,      0.984662,     0.928574,     0.830695,    0.755426,
    0.700485,     0.610681,     0.547013,     0.493656,    0.451038,
    0.397178,     0.33815,      0.300496,     0.252347,    0.220356,
    0.186491,     0.154663,     0.120149,     0.100175,    0.0742332,
    0.0629832,    0.0565341,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    272.787,      269.523,      266.024,      262.271,     258.247,
    253.932,      249.305,      244.344,      239.025,     233.321,
    227.205,      220.647,      213.615,      206.074,     197.989,
    189.32,       180.024,      170.056,      159.368,     147.907,
    135.619,      122.442,      108.313,      93.1625,     76.9175,
    59.4986,      40.8208,      26.0998,      25.1013,     24.0306,
    22.8825,      21.6514,      20.3314,      18.9159,     17.3982,
    15.7708,      14.0258,      13.29,        12.5033,     11.6597,
    10.7551,      9.78513,      8.96801,      8.41803,     7.8283,
    7.19595,      6.65025,      6.26522,      5.85238,     5.43783,
    5.06901,      4.67354,      4.36881,      4.09217,     3.83126,
    3.5836,       3.32117,      3.04154,      2.8655,      2.68125,
    2.48555,      2.27602,      2.09074,      1.9028,      1.73216,
    1.6224,       1.4784,       1.39991,      1.27312,     1.1565,
    1.07362,      0.985336,     0.892069,     0.815057,    0.745418,
    0.688846,     0.62686,      0.551474,     0.486852,    0.446656,
    0.398638,     0.33523,      0.304562,     0.264623,    0.212052,
    0.185526,     0.146695,     0.124783,     0.0975174,   0.081495,
    0.0617769,    0.0543875,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    278.189,      274.882,      271.335,      267.533,     263.456,
    259.084,      254.396,      249.37,       243.98,      238.2,
    232.003,      225.359,      218.234,      210.594,     202.401,
    193.617,      184.199,      174.099,      163.27,      151.657,
    139.206,      125.855,      111.539,      96.1889,     79.7291,
    62.0798,      43.1551,      25.9752,      24.989,      23.9315,
    22.7976,      21.5818,      20.2782,      18.8803,     17.3813,
    15.7741,      14.0507,      13.2286,      12.4877,     11.6932,
    10.8413,      9.92784,      9.08795,      8.52553,     7.92247,
    7.27583,      6.6808,       6.29028,      5.87153,     5.44626,
    5.1165,       4.76291,      4.42251,      4.08465,     3.77229,
    3.50858,      3.28448,      3.09767,      2.84936,     2.616,
    2.43773,      2.23126,      2.10176,      1.92194,     1.74881,
    1.61529,      1.48884,      1.40316,      1.28245,     1.13574,
    1.06221,      0.933315,     0.893186,     0.802371,    0.750661,
    0.677026,     0.602417,     0.554466,     0.493733,    0.438354,
    0.388232,     0.331304,     0.296133,     0.248411,    0.215281,
    0.180593,     0.138809,     0.117794,     0.0934451,   0.0758887,
    0.0642164,    0.0516589,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    266.237,      263.108,      259.752,      256.154,     252.295,
    248.158,      243.722,      238.966,      233.865,     228.396,
    222.532,      216.244,      209.502,      202.272,     194.52,
    186.208,      177.295,      167.738,      157.49,      146.502,
    134.719,      122.085,      108.538,      94.0122,     78.4364,
    61.735,       43.8267,      25.4201,      24.4465,     23.4026,
    22.2832,      21.0828,      19.7958,      18.4157,     16.9359,
    15.3492,      13.6478,      12.721,       12.0061,     11.2395,
    10.4176,      9.53621,      8.68509,      8.19967,     7.67917,
    7.12106,      6.57021,      6.20785,      5.81929,     5.40629,
    5.03638,      4.63974,      4.29734,      4.02226,     3.72476,
    3.39981,      3.12176,      2.92517,      2.76357,     2.60078,
    2.40639,      2.20718,      2.04885,      1.88011,     1.7369,
    1.60268,      1.47554,      1.35453,      1.24132,     1.15202,
    1.05316,      0.96096,      0.881724,     0.814546,    0.738687,
    0.657076,     0.602703,     0.533899,     0.483426,    0.434296,
    0.37399,      0.334765,     0.294995,     0.238594,    0.210998,
    0.181554,     0.15001,      0.11958,      0.0919327,   0.0772024,
    0.0568543,    0.0454201,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    254.158,      251.213,      248.056,      244.67,      241.04,
    237.147,      232.973,      228.497,      223.698,     218.551,
    213.033,      207.117,      200.772,      193.97,      186.675,
    178.853,      170.467,      161.474,      151.831,     141.491,
    130.404,      118.516,      105.769,      92.1003,     77.444,
    61.7286,      44.8775,      26.8085,      24.5454,     23.4706,
    22.3181,      21.0823,      19.7573,      18.3364,     16.8129,
    15.1793,      13.4276,      12.4067,      11.7772,     11.1022,
    10.3784,      9.60227,      8.79687,      8.28781,     7.74195,
    7.15665,      6.54559,      6.13821,      5.70139,     5.233,
    4.91377,      4.58441,      4.25591,      3.94724,     3.64273,
    3.43033,      3.20243,      2.9577,       2.73287,     2.52205,
    2.34612,      2.15941,      1.99962,      1.87305,     1.73649,
    1.58266,      1.48238,      1.33464,      1.26307,     1.13797,
    1.07692,      0.981729,     0.865576,     0.783972,    0.711356,
    0.656442,     0.584889,     0.534211,     0.497559,    0.416117,
    0.37982,      0.339012,     0.293114,     0.250252,    0.210445,
    0.172325,     0.144968,     0.113364,     0.0948002,   0.0744866,
    0.0568387,    0.0477953,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    255.083,      252.138,      248.979,      245.593,     241.962,
    238.068,      233.893,      229.416,      224.616,     219.469,
    213.95,       208.032,      201.686,      194.882,     187.586,
    179.762,      171.374,      162.379,      152.734,     142.392,
    131.303,      119.412,      106.662,      92.9907,     78.3313,
    62.6126,      45.7578,      27.6851,      23.5031,     22.5417,
    21.5108,      20.4054,      19.2202,      17.9492,     16.5865,
    15.1252,      13.5583,      12.4607,      11.7888,     11.0685,
    10.2961,      9.46781,      8.57972,      8.06948,     7.54119,
    6.97473,      6.36732,      6.00933,      5.64059,     5.2452,
    4.91715,      4.58661,      4.25622,      3.97373,     3.67837,
    3.44955,      3.19981,      2.91302,      2.68713,     2.52304,
    2.39514,      2.1882,       1.99632,      1.86354,     1.71256,
    1.59319,      1.44512,      1.33976,      1.18646,     1.15244,
    1.03173,      0.939471,     0.862488,     0.776945,    0.715548,
    0.646558,     0.589594,     0.5445,       0.468717,    0.427209,
    0.362621,     0.330515,     0.287074,     0.240818,    0.210781,
    0.182022,     0.142138,     0.117546,     0.0870706,   0.0705155,
    0.0550272,    0.044692,     0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    248.058,      245.227,      242.193,      238.938,     235.449,
    231.708,      227.696,      223.394,      218.782,     213.836,
    208.532,      202.846,      196.748,      190.21,      183.199,
    175.682,      167.621,      158.978,      149.71,      139.773,
    129.117,      117.691,      105.44,       92.3028,     78.2166,
    63.1124,      46.9167,      29.5505,      23.5437,     22.5428,
    21.4695,      20.3186,      19.0846,      17.7614,     16.3425,
    14.8212,      13.1898,      11.9744,      11.3946,     10.7729,
    10.1063,      9.3915,       8.62505,      8.08197,     7.55082,
    6.98127,      6.37057,      5.96629,      5.58167,     5.16925,
    4.82597,      4.49913,      4.16088,      3.87187,     3.56196,
    3.36975,      3.16087,      2.85326,      2.60631,     2.4634,
    2.29771,      2.11627,      1.95722,      1.84884,     1.67333,
    1.57578,      1.41585,      1.29668,      1.19761,     1.12905,
    1.01928,      0.956614,     0.844151,     0.76334,     0.700965,
    0.644221,     0.568333,     0.535863,     0.466193,    0.406255,
    0.361973,     0.330755,     0.275944,     0.240253,    0.195716,
    0.166647,     0.136982,     0.111756,     0.0918877,   0.0701009,
    0.0553,       0.0477713,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    246.572,      243.773,      240.771,      237.553,     234.102,
    230.401,      226.433,      222.179,      217.616,     212.725,
    207.479,      201.855,      195.824,      189.357,     182.423,
    174.988,      167.015,      158.467,      149.3,       139.472,
    128.932,      117.632,      105.514,      92.5209,     78.5888,
    63.6498,      47.6312,      30.455,       22.6685,     21.7885,
    20.8449,      19.833,       18.7481,      17.5847,     16.3373,
    14.9998,      13.5655,      12.3309,      11.6657,     10.9524,
    10.1875,      9.36733,      8.48791,      7.92614,     7.46305,
    6.96649,      6.43404,      6.01148,      5.61483,     5.18952,
    4.82859,      4.5065,       4.16459,      3.89298,     3.60175,
    3.33631,      3.05957,      2.87823,      2.69098,     2.49902,
    2.30463,      2.09779,      1.88937,      1.72417,     1.58897,
    1.50117,      1.43358,      1.30159,      1.18076,     1.10855,
    0.998447,     0.958737,     0.858451,     0.780508,    0.705566,
    0.649767,     0.589073,     0.516865,     0.453463,    0.404626,
    0.371624,     0.320154,     0.273447,     0.24073,     0.205355,
    0.169243,     0.133347,     0.10649,      0.0813048,   0.0645426,
    0.0517888,    0.0458582,    0.0,          0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    245.623,      242.857,      239.891,      236.71,      233.299,
    229.642,      225.721,      221.516,      217.007,     212.173,
    206.989,      201.431,      195.471,      189.08,      182.227,
    174.879,      167.0,        158.552,      149.493,     139.78,
    129.364,      118.196,      106.221,      93.3802,     79.6115,
    64.8479,      49.0173,      32.0427,      22.4447,     21.5572,
    20.6054,      19.5849,      18.4907,      17.3173,     16.0592,
    14.7101,      13.2636,      11.9257,      11.3038,     10.6369,
    9.92179,      9.15504,      8.33287,      7.72749,     7.24634,
    6.73043,      6.17724,      5.78508,      5.49108,     5.17583,
    4.82964,      4.44974,      4.04239,      3.76059,     3.47108,
    3.22177,      2.97543,      2.75639,      2.54936,     2.41521,
    2.26328,      2.08592,      1.88744,      1.78293,     1.6873,
    1.52321,      1.38791,      1.2858,       1.22513,     1.0853,
    1.02196,      0.92114,      0.838155,     0.775303,    0.693162,
    0.611795,     0.583327,     0.509287,     0.458022,    0.408684,
    0.346713,     0.323041,     0.271093,     0.2288,      0.190951,
    0.160788,     0.145082,     0.10371,      0.0794925,   0.061882,
    0.0529807,    0.043838,     0.0425264,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    231.26,       228.698,      225.952,      223.007,     219.85,
    216.464,      212.834,      208.941,      204.767,     200.291,
    195.492,      190.346,      184.828,      178.912,     172.567,
    165.765,      158.47,       150.649,      142.262,     133.27,
    123.627,      113.287,      102.201,      90.3128,     77.5658,
    63.8976,      49.2417,      33.5266,      22.8531,     21.9314,
    20.9431,      19.8833,      18.747,       17.5285,     16.222,
    14.8211,      13.3189,      11.8243,      11.245,      10.6239,
    9.95783,      9.24365,      8.47786,      7.85143,     7.36467,
    6.84274,      6.2831,       5.82403,      5.47013,     5.09065,
    4.71428,      4.36122,      3.98263,      3.72292,     3.47801,
    3.27356,      3.08783,      2.85481,      2.59668,     2.35535,
    2.19028,      2.07102,      1.91192,      1.74104,     1.62284,
    1.46461,      1.36441,      1.24812,      1.15805,     1.07684,
    1.02406,      0.901116,     0.835699,     0.753057,    0.697137,
    0.62146,      0.557646,     0.497457,     0.444432,    0.413347,
    0.348628,     0.324242,     0.268997,     0.233812,    0.188948,
    0.168736,     0.123792,     0.105606,     0.0840929,   0.0596813,
    0.0496574,    0.0332418,    0.0355172,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    233.203,      230.629,      227.869,      224.909,     221.736,
    218.333,      214.684,      210.771,      206.576,     202.077,
    197.254,      192.081,      186.535,      180.589,     174.212,
    167.375,      160.043,      152.182,      143.753,     134.714,
    125.022,      114.63,       103.487,      91.5384,     78.7264,
    64.9886,      50.2579,      34.4627,      22.0072,     21.166,
    20.264,       19.2968,      18.2597,      17.1476,     15.9552,
    14.6766,      13.3057,      11.8356,      11.2402,     10.6041,
    9.92212,      9.19083,      8.40669,      7.71709,     7.19935,
    6.64419,      6.04892,      5.56148,      5.26739,     4.95204,
    4.63312,      4.34299,      4.03189,      3.7453,      3.45777,
    3.1991,       2.96642,      2.72567,      2.47249,     2.35787,
    2.23261,      2.08367,      1.86909,      1.73651,     1.58893,
    1.45075,      1.36803,      1.30616,      1.15187,     1.07851,
    0.985551,     0.895762,     0.825648,     0.755381,    0.668849,
    0.617554,     0.569619,     0.50932,      0.448856,    0.391565,
    0.358444,     0.304646,     0.265951,     0.226403,    0.18159,
    0.164288,     0.130083,     0.0995566,    0.0801514,   0.0585804,
    0.0416824,    0.0353946,    0.0342548,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    232.448,      229.895,      227.158,      224.223,     221.076,
    217.701,      214.083,      210.203,      206.042,     201.581,
    196.798,      191.668,      186.169,      180.271,     173.948,
    167.167,      159.897,      152.101,      143.742,     134.778,
    125.167,      114.862,      103.811,      91.9622,     79.2569,
    65.6334,      51.0254,      35.3617,      21.2722,     20.4699,
    19.6096,      18.6872,      17.6981,      16.6375,     15.5003,
    14.2808,      12.9733,      11.5713,      10.9128,     10.3138,
    9.67161,      8.98296,      8.24454,      7.5621,      7.0856,
    6.57466,      6.0268,       5.52529,      5.19588,     4.84267,
    4.48613,      4.21548,      3.92528,      3.66561,     3.42212,
    3.19248,      2.9897,       2.77958,      2.56074,     2.34876,
    2.15992,      2.01728,      1.85984,      1.7182,      1.60462,
    1.47074,      1.3495,       1.26153,      1.15465,     1.05922,
    0.949438,     0.890137,     0.778532,     0.747672,    0.66583,
    0.633209,     0.546895,     0.505259,     0.444517,    0.389753,
    0.331643,     0.308606,     0.26521,      0.214975,    0.182283,
    0.150618,     0.124268,     0.100404,     0.0732324,   0.0525333,
    0.0385496,    0.0263974,    0.0334252,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    226.324,      223.865,      221.229,      218.403,     215.372,
    212.122,      208.638,      204.901,      200.894,     196.598,
    191.992,      187.052,      181.756,      176.077,     169.987,
    163.457,      156.456,      148.948,      140.898,     132.266,
    123.011,      113.086,      102.444,      91.0337,     78.7982,
    65.6786,      51.6108,      36.5264,      21.2742,     20.4395,
    19.5446,      18.5849,      17.5559,      16.4526,     15.2695,
    14.0009,      12.6407,      11.1821,      10.4657,     9.93766,
    9.37147,      8.76436,      8.11338,      7.48216,     7.07955,
    6.64786,      6.18496,      5.71138,      5.30188,     4.86279,
    4.40623,      4.12949,      3.83275,      3.55476,     3.29858,
    3.06016,      2.88346,      2.697,        2.50112,     2.28429,
    2.08956,      1.9921,       1.82439,      1.70937,     1.58269,
    1.43264,      1.33559,      1.25051,      1.14388,     1.04144,
    0.947428,     0.880996,     0.804774,     0.736702,    0.660521,
    0.602631,     0.549144,     0.4838,       0.442561,    0.383815,
    0.338686,     0.299538,     0.254404,     0.220936,    0.18127,
    0.150172,     0.114036,     0.0943663,    0.0690727,   0.0528095,
    0.0318043,    0.0271963,    0.0317236,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    218.426,      216.077,      213.559,      210.858,     207.963,
    204.858,      201.529,      197.959,      194.132,     190.027,
    185.626,      180.908,      175.848,      170.422,     164.604,
    158.366,      151.677,      144.505,      136.814,     128.568,
    119.725,      110.244,      100.077,      89.1757,     77.4865,
    64.9526,      51.5129,      37.1019,      21.6495,     20.1592,
    19.3322,      18.4455,      17.4947,      16.4752,     15.382,
    14.2099,      12.953,       11.6052,      10.7657,     10.1605,
    9.51152,      8.81565,      8.06949,      7.31349,     6.93106,
    6.521,        6.0813,       5.62104,      5.24732,     4.84658,
    4.41689,      4.11864,      3.80587,      3.531,       3.33286,
    3.11471,      2.85958,      2.62577,      2.45916,     2.28025,
    2.09325,      1.91855,      1.77788,      1.65615,     1.55228,
    1.44721,      1.31001,      1.20027,      1.12261,     1.01254,
    0.957668,     0.863531,     0.786136,     0.720416,    0.660557,
    0.596464,     0.529017,     0.497052,     0.428004,    0.376603,
    0.342447,     0.289298,     0.252952,     0.209848,    0.181768,
    0.151707,     0.113494,     0.0957794,    0.0791004,   0.0443481,
    0.0367679,    0.0225586,    0.0273497,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    216.062,      213.756,      211.283,      208.632,     205.788,
    202.74,       199.47,       195.965,      192.206,     188.176,
    183.855,      179.221,      174.252,      168.924,     163.211,
    157.086,      150.517,      143.474,      135.922,     127.824,
    119.141,      109.831,      99.8477,      89.1429,     77.6644,
    65.3565,      52.1591,      38.008,       22.8342,     19.8066,
    19.004,       18.1433,      17.2204,      16.2308,     15.1697,
    14.032,       12.812,       11.5038,      10.6096,     10.0386,
    9.42632,      8.7698,       8.06584,      7.311,       6.89303,
    6.44804,      5.9709,       5.45928,      5.12962,     4.77977,
    4.40463,      4.12986,      3.85749,      3.5638,      3.24469,
    2.92999,      2.8153,       2.67397,      2.45673,     2.27112,
    2.10911,      1.90202,      1.78895,      1.67207,     1.51312,
    1.39292,      1.30337,      1.19759,      1.11036,     1.01184,
    0.921103,     0.865102,     0.790426,     0.710697,    0.651359,
    0.593095,     0.540558,     0.465984,     0.420616,    0.381962,
    0.344161,     0.2926,       0.2351,       0.217771,    0.168164,
    0.138453,     0.118408,     0.0884276,    0.0665034,   0.0468975,
    0.0306013,    0.0192524,    0.0224889,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    215.98,       213.689,      211.231,      208.597,     205.772,
    202.742,      199.494,      196.011,      192.276,     188.272,
    183.978,      179.373,      174.436,      169.143,     163.466,
    157.379,      150.853,      143.855,      136.351,     128.305,
    119.677,      110.426,      100.506,      89.8699,     78.4646,
    66.2352,      53.122,       39.0611,      23.984,      19.4079,
    18.6022,      17.7382,      16.8119,      15.8185,     14.7534,
    13.6114,      12.3868,      11.0737,      10.1109,     9.5935,
    9.03866,      8.44372,      7.80579,      7.12176,     6.71387,
    6.31869,      5.89496,      5.44061,      5.09203,     4.73768,
    4.35773,      4.04669,      3.74604,      3.44642,     3.22843,
    2.9952,       2.798,        2.58993,      2.39218,     2.21355,
    2.06386,      1.91555,      1.78746,      1.65466,     1.49854,
    1.38767,      1.2934,       1.18017,      1.09294,     1.01496,
    0.940746,     0.847916,     0.774935,     0.696465,    0.652362,
    0.586607,     0.529958,     0.463731,     0.421036,    0.364491,
    0.322558,     0.278914,     0.242735,     0.218445,    0.164176,
    0.141434,     0.115642,     0.0927729,    0.0592904,   0.0492885,
    0.0275584,    0.018953,     0.0171219,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    205.442,      203.294,      200.99,       198.52,      195.872,
    193.032,      189.987,      186.721,      183.22,      179.466,
    175.441,      171.124,      166.496,      161.533,     156.212,
    150.506,      144.388,      137.827,      130.792,     123.249,
    115.161,      106.489,      97.1895,      87.2181,     76.5262,
    65.0615,      52.7684,      39.5868,      25.4527,     19.6618,
    18.8439,      17.9669,      17.0265,      16.0181,     14.9369,
    13.7776,      12.5344,      11.2014,      10.1253,     9.59487,
    9.02609,      8.41621,      7.76226,      7.06104,     6.56046,
    6.09449,      5.59485,      5.0591,       4.72914,     4.44751,
    4.14553,      3.9086,       3.70307,      3.48629,     3.29499,
    3.08987,      2.79794,      2.4812,       2.35653,     2.21033,
    2.02947,      1.90053,      1.78316,      1.65719,     1.50597,
    1.42545,      1.28455,      1.16823,      1.07918,     0.977908,
    0.930752,     0.828321,     0.774286,     0.691761,    0.640285,
    0.560074,     0.514789,     0.464232,     0.417234,    0.375716,
    0.314031,     0.277313,     0.231026,     0.202455,    0.154993,
    0.132434,     0.111887,     0.0738138,    0.0603435,   0.0379476,
    0.023132,     0.0151063,    0.0140699,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    205.139,      203.003,      200.713,      198.258,     195.625,
    192.802,      189.775,      186.529,      183.048,     179.316,
    175.315,      171.024,      166.423,      161.489,     156.199,
    150.527,      144.445,      137.923,      130.93,      123.431,
    115.391,      106.77,       97.525,       87.6125,     76.9836,
    65.5866,      53.3659,      40.2621,      26.2114,     19.0291,
    18.2589,      17.4331,      16.5475,      15.598,      14.5798,
    13.4881,      12.3175,      11.0622,      9.95612,     9.44866,
    8.90452,      8.32106,      7.69543,      7.0246,      6.53716,
    6.12528,      5.68363,      5.21007,      4.86601,     4.57821,
    4.26962,      3.97582,      3.6925,       3.3887,      3.17416,
    2.94632,      2.74109,      2.53082,      2.3417,      2.15542,
    1.99674,      1.86085,      1.72717,      1.58958,     1.46803,
    1.30642,      1.27742,      1.15598,      1.07762,     0.999204,
    0.886821,     0.847059,     0.767312,     0.675735,    0.63241,
    0.574171,     0.489883,     0.463304,     0.405311,    0.356593,
    0.317986,     0.272776,     0.224476,     0.201176,    0.157691,
    0.132398,     0.104723,     0.0850982,    0.0573363,   0.0398005,
    0.0226234,    0.0136726,    0.0103902,    0.0,         0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    203.539,      201.443,      199.195,      196.785,     194.201,
    191.431,      188.46,       185.274,      181.858,     178.195,
    174.268,      170.056,      165.541,      160.699,     155.507,
    149.94,       143.97,       137.57,       130.706,     123.347,
    115.456,      106.994,      97.9211,      88.1924,     77.7607,
    66.5751,      54.5811,      41.7204,      27.9303,     19.4353,
    18.6279,      17.7621,      16.8338,      15.8384,     14.771,
    13.6265,      12.3993,      11.0834,      9.83475,     9.33572,
    8.80063,      8.22686,      7.61163,      6.95195,     6.44874,
    6.06028,      5.64374,      5.1971,       4.82873,     4.51826,
    4.18536,      3.88399,      3.63218,      3.36216,     3.15174,
    2.93727,      2.76105,      2.59547,      2.36401,     2.11829,
    1.94556,      1.80777,      1.69002,      1.57224,     1.43241,
    1.33182,      1.2507,       1.19168,      1.0689,      0.980287,
    0.90637,      0.795254,     0.740403,     0.677141,    0.614516,
    0.577623,     0.488042,     0.458333,     0.406701,    0.358844,
    0.304276,     0.271089,     0.237228,     0.197019,    0.160446,
    0.13971,      0.103862,     0.0874982,    0.0528234,   0.0261274,
    0.0167863,    0.00683009,   0.00945033,   0.0168395,   0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    195.077,      193.082,      190.944,      188.651,     186.192,
    183.556,      180.729,      177.697,      174.447,     170.962,
    167.225,      163.217,      158.92,       154.313,     149.373,
    144.076,      138.395,      132.305,      125.774,     118.771,
    111.263,      103.211,      94.5779,      85.3207,     75.3945,
    64.751,       53.3383,      41.1009,      27.9791,     18.5625,
    17.8457,      17.077,       16.2528,      15.3691,     14.4214,
    13.4053,      12.3158,      11.1475,      9.94721,     9.42176,
    8.85834,      8.2542,       7.6064,       6.91178,     6.35609,
    5.97,         5.556,        5.11208,      4.73852,     4.45525,
    4.15151,      3.847,        3.56187,      3.25614,     3.04111,
    2.84351,      2.66343,      2.49195,      2.32259,     2.14791,
    1.98043,      1.81451,      1.6694,       1.60265,     1.45338,
    1.30812,      1.21327,      1.15525,      1.05822,     0.944325,
    0.891194,     0.77989,      0.740874,     0.666066,    0.62031,
    0.560601,     0.500298,     0.44917,      0.394536,    0.356019,
    0.312003,     0.265685,     0.237645,     0.20097,     0.157032,
    0.132551,     0.10282,      0.0746459,    0.0537203,   0.0327786,
    0.0142603,    0.00726491,   0.0061876,    0.00814569,  0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    188.326,      186.421,      184.378,      182.187,     179.838,
    177.319,      174.619,      171.723,      168.618,     165.288,
    161.718,      157.89,       153.785,      149.384,     144.665,
    139.604,      134.178,      128.359,      122.121,     115.431,
    108.258,      100.566,      92.3187,      83.4753,     73.9928,
    63.825,       52.9224,      41.2319,      28.6966,     18.396,
    17.6636,      16.8783,      16.0363,      15.1334,     14.1652,
    13.1271,      12.0139,      10.8203,      9.54049,     9.06456,
    8.586,        8.07285,      7.52262,      6.93263,     6.41012,
    6.03339,      5.62944,      5.1963,       4.80635,     4.51665,
    4.20602,      3.88033,      3.55448,      3.20509,     2.98395,
    2.82156,      2.64359,      2.44886,      2.27152,     2.10171,
    1.9755,       1.81883,      1.63239,      1.54786,     1.40062,
    1.31396,      1.22034,      1.122,        1.02966,     0.947621,
    0.866302,     0.818381,     0.737118,     0.664523,    0.608696,
    0.541824,     0.486432,     0.450606,     0.387103,    0.327602,
    0.292731,     0.257573,     0.230372,     0.185834,    0.154831,
    0.118468,     0.0956062,    0.0640167,    0.0454721,   0.0274366,
    0.0131013,    0.00039439,   0.000584185,  0.00801396,  0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    194.952,      192.983,      190.871,      188.607,     186.179,
    183.576,      180.785,      177.792,      174.582,     171.141,
    167.451,      163.494,      159.251,      154.702,     149.824,
    144.594,      138.985,      132.971,      126.523,     119.608,
    112.194,      104.244,      95.7195,      86.579,      76.7779,
    66.2685,      54.9996,      42.9163,      29.9598,     17.9884,
    17.2856,      16.5319,      15.7239,      14.8574,     13.9283,
    12.932,       11.8638,      10.7183,      9.49009,     8.92986,
    8.44829,      7.93192,      7.37823,      6.78452,     6.22846,
    5.84215,      5.42792,      4.98375,      4.57328,     4.31155,
    4.03091,      3.74526,      3.5302,       3.29959,     3.04434,
    2.76469,      2.5473,       2.43946,      2.27086,     2.0388,
    1.9217,       1.80463,      1.64895,      1.49562,     1.41172,
    1.31362,      1.20776,      1.10791,      1.00996,     0.927495,
    0.867956,     0.775086,     0.711178,     0.658983,    0.593055,
    0.530534,     0.486378,     0.435456,     0.391756,    0.349158,
    0.303534,     0.251985,     0.217255,     0.177934,    0.145373,
    0.116314,     0.0802763,    0.0694283,    0.0487958,   0.0232954,
    0.00844983,   0.000146545,  -0.000560227, 0.000783618, 0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    188.342,      186.456,      184.435,      182.267,     179.942,
    177.45,       174.777,      171.911,      168.838,     165.543,
    162.01,       158.221,      154.159,      149.803,     145.132,
    140.124,      134.754,      128.996,      122.822,     116.201,
    109.102,      101.49,       93.328,       84.5761,     75.1917,
    65.129,       54.3392,      42.7697,      30.364,      17.6217,
    16.9475,      16.2246,      15.4494,      14.6182,     13.7269,
    12.7713,      11.7465,      10.6477,      9.46951,     8.82117,
    8.31634,      7.77502,      7.19459,      6.57221,     5.97754,
    5.66898,      5.33812,      4.98335,      4.62138,     4.3213,
    3.99954,      3.66057,      3.41709,      3.15602,     2.93243,
    2.75509,      2.56997,      2.38319,      2.21168,     2.06915,
    1.92835,      1.78882,      1.66119,      1.51146,     1.34436,
    1.23226,      1.18041,      1.08665,      0.985258,    0.923706,
    0.8341,       0.776209,     0.72299,      0.646585,    0.560125,
    0.516865,     0.461493,     0.432846,     0.380823,    0.325502,
    0.282152,     0.261052,     0.21377,      0.187139,    0.135655,
    0.10997,      0.0884033,    0.0588015,    0.0391889,   0.0178274,
    0.00848642,   -0.012198,    -0.00853066,  -0.00213345, 0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    185.806,      183.96,       181.982,      179.86,      177.585,
    175.145,      172.529,      169.724,      166.716,     163.491,
    160.033,      156.325,      152.349,      148.086,     143.515,
    138.613,      133.357,      127.721,      121.678,     115.199,
    108.25,       100.8,        92.8117,      84.2458,     75.0609,
    65.2122,      54.6518,      43.3282,      31.1863,     18.1669,
    16.7738,      16.0332,      15.2391,      14.3877,     13.4746,
    12.4956,      11.4459,      10.3202,      9.11328,     8.41814,
    7.97499,      7.49981,      6.9903,       6.44396,     5.88496,
    5.57082,      5.23398,      4.87279,      4.49281,     4.17142,
    3.82681,      3.4573,       3.29713,      3.13764,     2.95049,
    2.72317,      2.50016,      2.34188,      2.18299,     2.03632,
    1.89121,      1.74535,      1.60563,      1.43631,     1.34808,
    1.2938,       1.17965,      1.07142,      0.991651,    0.900522,
    0.840927,     0.778174,     0.707053,     0.638259,    0.568204,
    0.517078,     0.465092,     0.425108,     0.376622,    0.336515,
    0.290865,     0.247267,     0.20885,      0.180951,    0.142484,
    0.117198,     0.0835458,    0.0633257,    0.0344858,   0.0211796,
    0.00354186,   -0.01147,     -0.0108971,   -0.00182315, 0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    177.374,      175.632,      173.765,      171.762,     169.615,
    167.313,      164.845,      162.198,      159.359,     156.316,
    153.053,      149.554,      145.802,      141.778,     137.465,
    132.839,      127.879,      122.561,      116.858,     110.743,
    104.186,      97.1558,      89.6171,      81.5337,     72.8661,
    63.5721,      53.6065,      42.9206,      31.4625,     19.1764,
    16.6635,      15.9395,      15.1631,      14.3306,     13.438,
    12.4809,      11.4546,      10.3541,      9.1741,      8.42371,
    8.01211,      7.57075,      7.09751,      6.59006,     6.04593,
    5.66924,      5.26705,      4.83579,      4.37337,     4.16112,
    3.9383,       3.69937,      3.43263,      3.14476,     2.87875,
    2.70219,      2.51902,      2.37252,      2.20419,     1.9835,
    1.81534,      1.69025,      1.53325,      1.41107,     1.32112,
    1.24881,      1.17841,      1.11264,      1.01162,     0.908762,
    0.85678,      0.750864,     0.683846,     0.617431,    0.573091,
    0.494989,     0.466622,     0.421794,     0.367022,    0.321725,
    0.268584,     0.236549,     0.208931,     0.169742,    0.1308,
    0.10036,      0.07595,      0.0544041,    0.0257654,   0.0141396,
    0.00107906,   -0.0106819,   -0.0206226,   -0.00980493, 0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    172.661,      170.972,      169.161,      167.22,      165.138,
    162.905,      160.512,      157.945,      155.193,     152.242,
    149.077,      145.684,      142.046,      138.145,     133.962,
    129.477,      124.667,      119.51,       113.98,      108.051,
    101.693,      94.8758,      87.5658,      79.7275,     71.3228,
    62.3107,      52.6473,      42.2855,      31.1749,     19.2615,
    15.7439,      15.1253,      14.462,       13.7508,     12.9882,
    12.1705,      11.2937,      10.3535,      9.34538,     8.5776,
    8.10626,      7.60085,      7.05892,      6.47782,     5.85473,
    5.4891,       5.13344,      4.75208,      4.34315,     4.08271,
    3.8267,       3.5522,       3.32396,      3.10091,     2.86946,
    2.65475,      2.42613,      2.27836,      2.1223,      1.97173,
    1.81886,      1.6654,       1.58166,      1.47372,     1.34716,
    1.22436,      1.12247,      1.03734,      0.970666,    0.905035,
    0.806251,     0.745207,     0.692295,     0.604323,    0.546094,
    0.499044,     0.454539,     0.398329,     0.363612,    0.30712,
    0.271897,     0.232024,     0.198076,     0.155375,    0.132595,
    0.106483,     0.075058,     0.0487855,    0.0263333,   0.00744197,
    -0.00600382,  -0.0180515,   -0.0180614,   -0.0162113,  0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    176.046,      174.336,      172.503,      170.537,     168.429,
    166.169,      163.745,      161.146,      158.36,      155.372,
    152.168,      148.733,      145.049,      141.099,     136.864,
    132.323,      127.453,      122.232,      116.633,     110.63,
    104.193,      97.2901,      89.8889,      81.9528,     73.4431,
    64.3185,      54.5345,      44.0435,      32.7942,     20.732,
    15.9991,      15.3566,      14.6676,      13.9289,     13.1367,
    12.2874,      11.3766,      10.4,         9.35285,     8.49727,
    8.02348,      7.51546,      6.97072,      6.38661,     5.7603,
    5.36972,      5.02345,      4.65215,      4.25402,     3.99841,
    3.77074,      3.52662,      3.3041,       3.08607,     2.85533,
    2.6372,       2.4033,       2.23693,      2.06701,     1.91635,
    1.7642,       1.61801,      1.5297,       1.43071,     1.31212,
    1.19753,      1.12325,      1.06251,      0.946303,    0.890867,
    0.818562,     0.733854,     0.667914,     0.59793,     0.540659,
    0.495193,     0.431973,     0.400268,     0.355917,    0.321459,
    0.258267,     0.224605,     0.191322,     0.163292,    0.128843,
    0.0930474,    0.0635897,    0.0499172,    0.0251269,   0.00536533,
    -0.012208,    -0.0240647,   -0.0226029,   -0.0258864,  0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    170.948,      169.298,      167.529,      165.632,     163.597,
    161.416,      159.078,      156.57,       153.881,     150.997,
    147.906,      144.591,      141.036,      137.224,     133.137,
    128.755,      124.056,      119.017,      113.614,     107.821,
    101.609,      94.9482,      87.806,       80.1476,     71.9358,
    63.1305,      53.6889,      43.565,       32.7095,     21.0695,
    15.385,       14.7678,      14.1061,      13.3965,     12.6356,
    11.8198,      10.945,       10.007,       9.00117,     8.12298,
    7.69049,      7.22674,      6.72948,      6.19629,     5.62457,
    5.27623,      5.01697,      4.73898,      4.44089,     4.14414,
    3.83619,      3.50599,      3.26127,      3.08452,     2.89493,
    2.65964,      2.40735,      2.23257,      2.06592,     1.91145,
    1.77107,      1.68678,      1.56155,      1.41937,     1.28535,
    1.20825,      1.11107,      0.977466,     0.934676,    0.881315,
    0.794501,     0.720538,     0.673765,     0.584144,    0.536138,
    0.494922,     0.433437,     0.399228,     0.3405,      0.300655,
    0.270518,     0.220667,     0.190561,     0.154549,    0.124847,
    0.0977804,    0.0593418,    0.0435931,    0.0207329,   0.00940246,
    -0.0183576,   -0.0262005,   -0.0282689,   -0.0268636,  0.0,
    0.0,          0.0,          0.0,          0.0,         0.0,
    172.81,       171.151,      169.371,      167.464,     165.418,
    163.225,      160.873,      158.351,      155.647,     152.747,
    149.638,      146.304,      142.73,       138.897,     134.787,
    130.38,       125.654,      120.587,      115.154,     109.328,
    103.081,      96.383,       89.2006,      81.4992,     73.2412,
    64.3864,      54.8918,      44.7109,      33.7944,     22.0889,
    15.2575,      14.6594,      14.0181,      13.3305,     12.5931,
    11.8025,      10.9548,      10.0458,      9.07108,     8.15137,
    7.7001,       7.21622,      6.69737,      6.14103,     5.54448,
    5.13969,      4.86004,      4.56019,      4.23867,     3.96701,
    3.72539,      3.4663,       3.22061,      2.9939,      2.7508,
    2.58349,      2.41379,      2.22313,      2.01541,     1.8798,
    1.75639,      1.65,         1.53055,      1.41181,     1.30971,
    1.19904,      1.11651,      1.01558,      0.971724,    0.85,
    0.771646,     0.7141,       0.64091,      0.59184,     0.521868,
    0.468975,     0.434806,     0.367072,     0.342976,    0.300425,
    0.267795,     0.224956,     0.183761,     0.142419,    0.123104,
    0.0923225,    0.0698596,    0.0378334,    0.0262921,   -0.00374401,
    -0.0156949,   -0.0236329,   -0.0383532,   -0.0261562,  -0.0147699,
    0.0,          0.0,          0.0,          0.0,         0.0,
    163.649,      162.098,      160.435,      158.651,     156.739,
    154.688,      152.49,       150.132,      147.604,     144.893,
    141.986,      138.869,      135.527,      131.944,     128.101,
    123.981,      119.563,      114.826,      109.747,     104.3,
    98.4598,      92.1975,      85.4826,      78.2825,     70.5621,
    62.2837,      53.407,       43.8889,      33.6829,     22.7394,
    15.2763,      14.6527,      13.9841,      13.267,      12.4982,
    11.6738,      10.7899,      9.84205,      8.82571,     7.81446,
    7.4467,       7.05238,      6.62955,      6.17617,     5.69002,
    5.28294,      4.95637,      4.60621,      4.23073,     3.9254,
    3.69465,      3.44723,      3.19521,      2.94722,     2.68132,
    2.49067,      2.3084,       2.14696,      1.99371,     1.8473,
    1.70013,      1.58462,      1.47108,      1.35886,     1.24531,
    1.136,        1.07217,      0.983558,     0.918767,    0.830484,
    0.76243,      0.712796,     0.644476,     0.579216,    0.526533,
    0.477981,     0.438221,     0.377126,     0.330901,    0.294315,
    0.252574,     0.200481,     0.181221,     0.138428,    0.110237,
    0.087223,     0.0538754,    0.0356792,    0.0062537,   -0.00708369,
    -0.0166324,   -0.0366875,   -0.037705,    -0.0306626,  -0.0193391,
    0.0,          0.0,          0.0,          0.0,         0.0,
    158.442,      156.95,       155.349,      153.633,     151.793,
    149.819,      147.704,      145.435,      143.002,     140.394,
    137.597,      134.597,      131.382,      127.933,     124.236,
    120.271,      116.02,       111.461,      106.573,     101.332,
    95.7124,      89.6864,      83.2249,      76.2964,     68.8673,
    60.9012,      52.3595,      43.2005,      33.3796,     22.8489,
    14.6662,      14.0897,      13.4715,      12.8087,     12.098,
    11.3359,      10.5187,      9.64251,      8.70298,     7.70179,
    7.32773,      6.92665,      6.49657,      6.03542,     5.54094,
    5.14061,      4.89299,      4.62747,      4.34277,     4.04499,
    3.73653,      3.40578,      3.10281,      2.90969,     2.70261,
    2.48441,      2.25197,      2.07118,      1.93615,     1.81267,
    1.69174,      1.57119,      1.454,        1.34237,     1.23145,
    1.16176,      1.05712,      0.948094,     0.922463,    0.826532,
    0.747477,     0.688181,     0.646388,     0.591148,    0.510395,
    0.46448,      0.403836,     0.379523,     0.32999,     0.277945,
    0.262016,     0.211214,     0.175458,     0.140024,    0.11947,
    0.0810765,    0.0457908,    0.0298241,    0.00957702,  -0.0128245,
    -0.0196498,   -0.038028,    -0.0428507,   -0.0432913,  -0.0237097,
    0.0,          0.0,          0.0,          0.0,         0.0,
    158.395,      156.909,      155.315,      153.607,     151.775,
    149.81,       147.704,      145.445,      143.024,     140.427,
    137.642,      134.656,      131.455,      128.022,     124.341,
    120.394,      116.162,      111.624,      106.758,     101.54,
    95.9455,      89.9465,      83.5139,      76.6165,     69.2206,
    61.2902,      52.7867,      43.6687,      33.8918,     23.4083,
    14.2614,      13.71,        13.1187,      12.4847,     11.8048,
    11.0759,      10.2942,      9.45607,      8.55737,     7.59371,
    7.16832,      6.76915,      6.34114,      5.8822,      5.39009,
    4.94549,      4.64086,      4.31422,      3.96397,     3.64137,
    3.4092,       3.16025,      2.91869,      2.76696,     2.60426,
    2.45247,      2.30348,      2.13964,      1.95887,     1.81448,
    1.69897,      1.57789,      1.4588,       1.34811,     1.24961,
    1.16869,      1.03405,      0.958803,     0.904793,    0.798557,
    0.729663,     0.7029,       0.612475,     0.551596,    0.521404,
    0.456577,     0.414346,     0.36912,      0.314787,    0.260896,
    0.230763,     0.193945,     0.165822,     0.135626,    0.110302,
    0.0773479,    0.055597,     0.0296075,    0.00357574,  -0.0180904,
    -0.0317303,   -0.0459,      -0.0489304,   -0.045429,   -0.0264922,
    0.0,          0.0,          0.0,          0.0,         0.0,
    150.547,      149.153,      147.659,      146.057,     144.339,
    142.498,      140.522,      138.405,      136.134,     133.699,
    131.088,      128.289,      125.287,      122.068,     118.616,
    114.915,      110.947,      106.692,      102.129,     97.2371,
    91.9912,      86.3662,      80.3347,      73.8673,     66.9326,
    59.4967,      51.5234,      42.9739,      33.8065,     23.9767,
    14.4515,      13.8839,      13.2752,      12.6225,     11.9227,
    11.1723,      10.3677,      9.50494,      8.57982,     7.58785,
    7.12344,      6.75672,      6.36349,      5.94184,     5.48972,
    5.05867,      4.77101,      4.46256,      4.13182,     3.80411,
    3.54412,      3.26534,      2.9811,       2.80572,     2.61766,
    2.42895,      2.23809,      2.0612,       1.92204,     1.79213,
    1.67502,      1.55701,      1.43926,      1.32879,     1.21866,
    1.12509,      1.03383,      0.945536,     0.880489,    0.806334,
    0.71218,      0.669715,     0.596268,     0.55426,     0.515107,
    0.478203,     0.418321,     0.370691,     0.321106,    0.284244,
    0.240829,     0.196919,     0.167378,     0.133615,    0.11027,
    0.0749765,    0.0502423,    0.0293525,    0.00222881,  -0.0153475,
    -0.0403788,   -0.0423408,   -0.055646,    -0.0508041,  -0.0422829,
    0.0,          0.0,          0.0,          0.0,         0.0,
    150.098,      148.713,      147.228,      145.635,     143.928,
    142.097,      140.133,      138.028,      135.771,     133.35,
    130.755,      127.972,      124.988,      121.788,     118.357,
    114.678,      110.733,      106.503,      101.968,     97.1042,
    91.8893,      86.2976,      80.3018,      73.8727,     66.979,
    59.5871,      51.6611,      43.1622,      34.0491,     24.2774,
    13.8754,      13.3516,      12.7899,      12.1876,     11.5418,
    10.8494,      10.1069,      9.31071,      8.457,       7.5416,
    7.02089,      6.63502,      6.22127,      5.77762,     5.3019,
    4.8351,       4.58709,      4.32116,      4.03601,     3.74058,
    3.488,        3.21716,      2.92879,      2.7659,      2.59123,
    2.41435,      2.23798,      2.07624,      1.97874,     1.84664,
    1.65905,      1.52489,      1.41068,      1.29543,     1.19257,
    1.09724,      1.02614,      0.949585,     0.864717,    0.786772,
    0.703854,     0.666681,     0.61089,      0.524655,    0.504208,
    0.432708,     0.398084,     0.366369,     0.326556,    0.265326,
    0.226305,     0.199268,     0.163492,     0.133552,    0.0951778,
    0.0642842,    0.0420521,    0.0204069,    -0.00255083, -0.0285124,
    -0.0312027,   -0.0526119,   -0.0574395,   -0.0591463,  -0.0457661,
    0.0,          0.0,          0.0,          0.0,         0.0,
    151.339,      149.952,      148.464,      146.869,     145.159,
    143.325,      141.359,      139.251,      136.99,      134.566,
    131.967,      129.18,       126.191,      122.987,     119.551,
    115.866,      111.916,      107.68,       103.137,     98.2668,
    93.0444,      87.4445,      81.4399,      75.0013,     68.0975,
    60.6948,      52.757,       44.2457,      35.1192,     25.3332,
    14.84,        13.4514,      12.8869,      12.2817,     11.6326,
    10.9367,      10.1905,      9.39034,      8.53237,     7.6124,
    7.023,        6.61822,      6.18418,      5.71877,     5.21973,
    4.70358,      4.45527,      4.18901,      3.90351,     3.60363,
    3.39894,      3.17947,      2.94414,      2.73925,     2.5236,
    2.32497,      2.17269,      2.01632,      1.8809,      1.74523,
    1.62361,      1.49117,      1.35516,      1.25981,     1.20001,
    1.11842,      1.01578,      0.930765,     0.851623,    0.785248,
    0.721168,     0.654327,     0.595229,     0.549594,    0.503532,
    0.410425,     0.41156,      0.328997,     0.30829,     0.255732,
    0.223644,     0.197527,     0.154108,     0.126606,    0.0925264,
    0.064986,     0.0325507,    0.0223901,    -0.015805,   -0.0246912,
    -0.037123,    -0.0568041,   -0.0625113,   -0.0647284,  -0.0430769,
    0.0,          0.0,          0.0,          0.0,         0.0,
    143.67,       142.364,      140.964,      139.463,     137.853,
    136.126,      134.275,      132.29,       130.162,     127.88,
    125.433,      122.809,      119.996,      116.979,     113.744,
    110.276,      106.556,      102.568,      98.2922,     93.707,
    88.7904,      83.5185,      77.8656,      71.8042,     65.3048,
    58.3356,      50.8629,      42.8501,      34.2582,     25.0454,
    15.1668,      12.9837,      12.461,       11.9005,     11.2995,
    10.6551,      9.9641,       9.22316,      8.42868,     7.57678,
    6.97743,      6.59619,      6.1874,       5.74906,     5.27905,
    4.77507,      4.50402,      4.22181,      3.9192,      3.59472,
    3.38078,      3.15674,      2.91651,      2.66491,     2.39636,
    2.17379,      2.12035,      2.05399,      1.88981,     1.72709,
    1.60681,      1.48138,      1.35185,      1.25657,     1.17977,
    1.09515,      0.986835,     0.903819,     0.875451,    0.778963,
    0.719021,     0.6432,       0.597274,     0.536409,    0.500888,
    0.42398,      0.386159,     0.36141,      0.289446,    0.239445,
    0.226269,     0.17348,      0.156905,     0.112538,    0.0927015,
    0.0598297,    0.0463729,    0.0109692,    -0.0114997,  -0.0308213,
    -0.0517334,   -0.0645702,   -0.0683537,   -0.0675832,  -0.0608434,
    0.0,          0.0,          0.0,          0.0,         0.0,
    143.732,      142.434,      141.043,      139.551,     137.952,
    136.237,      134.398,      132.426,      130.312,     128.044,
    125.613,      123.007,      120.212,      117.215,     114.001,
    110.555,      106.86,       102.898,      98.6496,     94.0943,
    89.2097,      83.9722,      78.3562,      72.3343,     65.8772,
    58.9535,      51.5295,      43.5689,      35.0331,     25.8803,
    16.0662,      13.0029,      12.4592,      11.8763,     11.2513,
    10.5811,      9.8624,       9.09182,      8.26555,     7.37956,
    6.72045,      6.36,         5.9735,       5.55908,     5.1147,
    4.63821,      4.37299,      4.12278,      3.85449,     3.56681,
    3.3158,       3.05526,      2.77589,      2.59364,     2.43979,
    2.28071,      2.13823,      1.98547,      1.82953,     1.66858,
    1.5465,       1.43406,      1.33727,      1.24022,     1.14457,
    1.05507,      0.975532,     0.905972,     0.851267,    0.762425,
    0.7024,       0.653989,     0.560389,     0.526864,    0.491196,
    0.429755,     0.369097,     0.356732,     0.279636,    0.250421,
    0.208487,     0.177796,     0.146795,     0.120401,    0.0848676,
    0.0538871,    0.0397697,    0.00846093,   -0.0189117,  -0.0360042,
    -0.0550465,   -0.063197,    -0.0674504,   -0.0696626,  -0.0591521,
    0.0,          0.0,          0.0,          0.0,         0.0,
    136.994,      135.766,      134.45,       133.039,     131.526,
    129.904,      128.164,      126.298,      124.298,     122.153,
    119.853,      117.387,      114.743,      111.908,     108.868,
    105.608,      102.112,      98.3639,      94.3448,     90.0354,
    85.4145,      80.4596,      75.1467,      69.4499,     63.3413,
    56.7913,      49.7679,      42.237,       34.1619,     25.5032,
    16.2187,      12.5042,      12.0084,      11.4767,     10.9066,
    10.2953,      9.63987,      8.93704,      8.18341,     7.37533,
    6.71803,      6.37178,      6.00051,      5.6024,      5.17553,
    4.71781,      4.39858,      4.10268,      3.78539,     3.44517,
    3.20896,      2.99213,      2.75964,      2.5606,      2.37441,
    2.18499,      2.08897,      1.98601,      1.83447,     1.6701,
    1.57833,      1.47045,      1.33708,      1.19221,     1.08685,
    1.02734,      0.968254,     0.878608,     0.816852,    0.770393,
    0.69347,      0.612541,     0.55319,      0.51744,     0.457345,
    0.393706,     0.383378,     0.32943,      0.285621,    0.251881,
    0.207977,     0.175061,     0.141786,     0.101353,    0.0866193,
    0.057901,     0.0274161,    0.0100799,    -0.0186857,  -0.022622,
    -0.0565729,   -0.0704823,   -0.0763906,   -0.0762047,  -0.0664564,
    0.0,          0.0,          0.0,          0.0,         0.0,
    133.684,      132.497,      131.225,      129.86,      128.397,
    126.828,      125.146,      123.342,      121.408,     119.334,
    117.111,      114.726,      112.169,      109.428,     106.488,
    103.336,      99.9559,      96.3317,      92.4455,     88.2785,
    83.8104,      79.0194,      73.8822,      68.3737,     62.4671,
    56.1337,      49.3425,      42.0606,      34.2525,     25.88,
    16.9026,      12.4942,      11.9867,      11.4426,     10.8591,
    10.2334,      9.56258,      8.84323,      8.07189,     7.24481,
    6.52294,      6.17006,      5.79167,      5.38593,     4.95088,
    4.48438,      4.18639,      3.95449,      3.70583,     3.43921,
    3.2085,       2.98597,      2.74735,      2.5488,      2.38097,
    2.20112,      2.0709,       1.93128,      1.79476,     1.65127,
    1.53946,      1.42916,      1.3297,       1.22218,     1.12725,
    1.05213,      0.9581,       0.82969,      0.77891,     0.726213,
    0.663251,     0.611942,     0.547885,     0.515706,    0.454947,
    0.408132,     0.354839,     0.324485,     0.274851,    0.24552,
    0.205243,     0.178421,     0.128162,     0.113741,    0.0924885,
    0.0493387,    0.0230367,    0.00166329,   -0.0280148,  -0.0465621,
    -0.0585516,   -0.0659906,   -0.0844807,   -0.079671,   -0.0742628,
    0.0,          0.0,          0.0,          0.0,         0.0,
    133.07,       131.895,      130.635,      129.285,     127.836,
    126.284,      124.618,      122.833,      120.919,     118.866,
    116.665,      114.304,      111.774,      109.06,      106.15,
    103.03,       99.6844,      96.097,       92.2504,     88.1258,
    83.7032,      78.9609,      73.8759,      68.4235,     62.577,
    56.308,       49.586,       42.3781,      34.6494,     26.3622,
    17.476,       12.3228,      11.8236,      11.2884,     10.7145,
    10.0991,      9.43918,      8.73163,      7.97294,     7.15942,
    6.40703,      6.09335,      5.757,        5.39634,     5.00961,
    4.59494,      4.27341,      4.00762,      3.72262,     3.41702,
    3.1745,       2.97092,      2.75264,      2.54883,     2.3641,
    2.16602,      2.00807,      1.84397,      1.73894,     1.65232,
    1.52243,      1.38889,      1.3042,       1.20513,     1.10054,
    1.00541,      0.956938,     0.84108,      0.797895,    0.751508,
    0.645631,     0.59767,      0.557925,     0.502104,    0.461408,
    0.413232,     0.351635,     0.322392,     0.279954,    0.239794,
    0.198736,     0.1859,       0.131872,     0.0998769,   0.0684161,
    0.0474682,    0.0270287,    -0.00225808,  -0.0299796,  -0.0513211,
    -0.0608877,   -0.0762634,   -0.0819236,   -0.0837125,  -0.0771209,
    -0.0549773,   0.0,          0.0,          0.0,         0.0,
    134.279,      133.096,      131.826,      130.465,     129.006,
    127.441,      125.764,      123.964,      122.035,     119.967,
    117.749,      115.371,      112.821,      110.086,     107.154,
    104.01,       100.639,      97.0243,      93.1483,     88.9922,
    84.5358,      79.7573,      74.6335,      69.1393,     63.2482,
    56.9313,      50.1579,      42.8951,      35.1073,     26.7568,
    17.8027,      11.8287,      11.3683,      10.8746,     10.3453,
    9.77766,      9.16903,      8.51643,      7.81666,     7.06633,
    6.32074,      5.99474,      5.64518,      5.27036,     4.86846,
    4.43751,      4.0987,       3.84837,      3.57994,     3.29211,
    3.05849,      2.87893,      2.68639,      2.49546,     2.31545,
    2.12243,      1.97703,      1.83439,      1.73341,     1.65388,
    1.52076,      1.36806,      1.25838,      1.15887,     1.06936,
    0.988844,     0.969278,     0.845442,     0.793181,    0.714799,
    0.624034,     0.574283,     0.538994,     0.481623,    0.442154,
    0.387768,     0.348952,     0.305237,     0.267254,    0.230031,
    0.194645,     0.140822,     0.124548,     0.105327,    0.0696711,
    0.0380373,    0.0248816,    -0.0134072,   -0.0362252,  -0.0399382,
    -0.0616631,   -0.0801982,   -0.0895243,   -0.0918959,  -0.080798,
    -0.0518821,   0.0,          0.0,          0.0,         0.0,
    127.205,      126.095,      124.905,      123.628,     122.26,
    120.792,      119.219,      117.531,      115.722,     113.782,
    111.702,      109.472,      107.08,       104.515,     101.766,
    98.8169,      95.6553,      92.2651,      88.63,       84.7321,
    80.5525,      76.071,       71.2655,      66.1127,     60.5876,
    54.6632,      48.3107,      41.4991,      34.1952,     26.3635,
    17.9658,      11.6088,      11.1655,      10.69,       10.1803,
    9.63366,      9.04755,      8.41908,      7.7452,      7.02261,
    6.26164,      5.94491,      5.60529,      5.24113,     4.85065,
    4.43195,      4.09144,      3.86681,      3.62595,     3.36769,
    3.12904,      2.92435,      2.70486,      2.48296,     2.27636,
    2.05483,      1.91628,      1.80349,      1.69431,     1.58661,
    1.46057,      1.32064,      1.22807,      1.14304,     1.06512,
    0.977679,     0.903825,     0.831348,     0.778002,    0.716146,
    0.633396,     0.598521,     0.531647,     0.464132,    0.433341,
    0.357832,     0.336989,     0.302054,     0.254022,    0.217184,
    0.180371,     0.154607,     0.119148,     0.0820962,   0.0683055,
    0.0419241,    0.0175281,    -0.0100066,   -0.0322681,  -0.0550318,
    -0.0741854,   -0.0857308,   -0.0992668,   -0.0988436,  -0.0906207,
    -0.0629113,   0.0,          0.0,          0.0,         0.0,
    130.95,       129.808,      128.583,      127.269,     125.861,
    124.351,      122.731,      120.995,      119.133,     117.136,
    114.996,      112.7,        110.239,      107.6,       104.77,
    101.735,      98.4814,      94.9925,      91.2514,     87.2401,
    82.9388,      78.3267,      73.3812,      68.0784,     62.3924,
    56.2954,      49.7579,      42.7478,      35.2312,     27.1714,
    18.5291,      11.2232,      10.8035,      10.3535,     9.87088,
    9.35344,      8.7986,       8.20366,      7.56572,     6.88169,
    6.14822,      5.82979,      5.51809,      5.18387,     4.82549,
    4.44121,      4.09193,      3.83375,      3.5569,      3.26005,
    2.99091,      2.79667,      2.58839,      2.38623,     2.2463,
    2.09625,      1.95441,      1.81268,      1.68286,     1.56856,
    1.45825,      1.34875,      1.24886,      1.14849,     1.05,
    0.968931,     0.893125,     0.825088,     0.756119,    0.696972,
    0.62759,      0.586714,     0.500564,     0.466253,    0.423696,
    0.377424,     0.328733,     0.29364,      0.243952,    0.21135,
    0.18162,      0.153881,     0.123234,     0.0959149,   0.063146,
    0.0358804,    -0.000797425, -0.0109238,   -0.0384534,  -0.0577831,
    -0.0719955,   -0.0908834,   -0.0932468,   -0.0979841,  -0.0978142,
    -0.07482,     0.0,          0.0,          0.0,         0.0,
    119.635,      118.608,      117.506,      116.325,     115.058,
    113.7,        112.244,      110.683,      109.008,     107.213,
    105.288,      103.224,      101.01,       98.6372,     96.0924,
    93.3637,      90.4377,      87.3004,      83.9363,     80.3291,
    76.4612,      72.3138,      67.8667,      63.0982,     57.9851,
    52.5025,      46.6236,      40.3199,      33.5607,     26.313,
    18.5415,      11.2673,      10.8234,      10.3474,     9.83703,
    9.28975,      8.70292,      8.07368,      7.39897,     6.6755,
    5.89975,      5.55786,      5.276,        4.97377,     4.6497,
    4.30221,      3.97313,      3.74019,      3.49042,     3.2226,
    2.9707,       2.80183,      2.62075,      2.43141,     2.25997,
    2.07615,      1.92498,      1.79875,      1.67938,     1.57683,
    1.45407,      1.30947,      1.18672,      1.09177,     1.04987,
    0.944098,     0.886604,     0.829914,     0.75727,     0.681863,
    0.620016,     0.581071,     0.514918,     0.474659,    0.425392,
    0.358687,     0.332633,     0.300613,     0.249789,    0.219238,
    0.172886,     0.140158,     0.102347,     0.0887763,   0.0492997,
    0.0263418,    0.00677424,   -0.0140469,   -0.0431933,  -0.0590721,
    -0.0871476,   -0.0916734,   -0.0998423,   -0.105748,   -0.0990305,
    -0.0839491,   0.0,          0.0,          0.0,         0.0,
    124.992,      123.923,      122.776,      121.547,     120.229,
    118.815,      117.3,        115.675,      113.932,     112.064,
    110.06,       107.912,      105.608,      103.138,     100.49,
    97.6497,      94.6046,      91.3393,      87.8381,     84.0839,
    80.0584,      75.742,       71.1136,      66.1507,     60.8292,
    55.1232,      49.0047,      42.4441,      35.4094,     27.8663,
    19.7781,      11.4886,      11.0244,      10.5266,     9.99291,
    9.42062,      8.80698,      8.14899,      7.44346,     6.68693,
    5.87573,      5.47486,      5.18831,      4.88105,     4.55159,
    4.19832,      3.85466,      3.64295,      3.41593,     3.17251,
    2.92816,      2.74448,      2.54753,      2.34237,     2.23628,
    2.12253,      1.9824,       1.81223,      1.64757,     1.51218,
    1.40015,      1.32727,      1.23324,      1.12589,     1.00743,
    0.923623,     0.872962,     0.812125,     0.733962,    0.663583,
    0.614486,     0.571135,     0.522828,     0.469894,    0.409587,
    0.379325,     0.313893,     0.284076,     0.252343,    0.191747,
    0.170511,     0.142025,     0.101422,     0.088199,    0.0428989,
    0.0192242,    -0.000485838, -0.0257089,   -0.0463289,  -0.0576673,
    -0.0897218,   -0.0973485,   -0.10667,     -0.111094,   -0.105205,
    -0.086174,    0.0,          0.0,          0.0,         0.0,
    121.735,      120.698,      119.586,      118.393,     117.115,
    115.744,      114.274,      112.697,      111.007,     109.194,
    107.251,      105.167,      102.933,      100.537,     97.9676,
    95.2128,      92.259,       89.0916,      85.6954,     82.0538,
    78.149,       73.962,       69.4724,      64.6583,     59.4964,
    53.9614,      48.0264,      41.6625,      34.8388,     27.5218,
    19.6761,      11.2635,      10.5349,      10.1042,     9.64242,
    9.14724,      8.61628,      8.04694,      7.43646,     6.78186,
    6.07996,      5.63772,      5.30932,      4.95718,     4.5796,
    4.17473,      3.76838,      3.5799,       3.37781,     3.1611,
    2.93487,      2.75186,      2.55564,      2.34523,     2.19298,
    2.03228,      1.87466,      1.72837,      1.58079,     1.45562,
    1.34151,      1.26021,      1.18482,      1.10746,     1.01173,
    0.90667,      0.838715,     0.792555,     0.727948,    0.662088,
    0.606035,     0.541314,     0.504361,     0.449474,    0.407606,
    0.366948,     0.295121,     0.263519,     0.250496,    0.195327,
    0.155974,     0.130519,     0.102184,     0.0736351,   0.0462375,
    0.0127569,    -0.00868302,  -0.0320561,   -0.0451221,  -0.0770632,
    -0.0871075,   -0.105134,    -0.113493,    -0.10848,    -0.113617,
    -0.0940259,   0.0,          0.0,          0.0,         0.0,
    117.551,      116.556,      115.489,      114.345,     113.119,
    111.803,      110.393,      108.881,      107.26,      105.521,
    103.657,      101.658,      99.5145,      97.2162,     94.7518,
    92.1093,      89.2758,      86.2376,      82.9798,     79.4866,
    75.741,       71.7246,      67.418,       62.8002,     57.8487,
    52.5393,      46.8463,      40.7418,      34.1961,     27.1775,
    19.6516,      11.5818,      10.2042,      9.78284,     9.33105,
    8.8466,       8.32715,      7.77016,      7.17291,     6.53251,
    5.84582,      5.39756,      5.11231,      4.80645,     4.47849,
    4.12682,      3.7539,       3.56006,      3.35222,     3.12935,
    2.89151,      2.72853,      2.55378,      2.36641,     2.20759,
    2.04301,      1.88356,      1.75047,      1.61334,     1.50126,
    1.38232,      1.25854,      1.16139,      1.08244,     0.987598,
    0.895276,     0.846038,     0.805685,     0.716899,    0.658976,
    0.583361,     0.524953,     0.493657,     0.443881,    0.381568,
    0.351943,     0.328663,     0.280239,     0.219176,    0.178259,
    0.1765,       0.128132,     0.103367,     0.082398,    0.0411692,
    0.0100906,    -0.00825442,  -0.0301387,   -0.0513763,  -0.0731982,
    -0.0885969,   -0.105269,    -0.110525,    -0.122388,   -0.118809,
    -0.0926944,   0.0,          0.0,          0.0,         0.0,
    116.984,      115.999,      114.942,      113.81,      112.595,
    111.293,      109.896,      108.399,      106.793,     105.071,
    103.225,      101.246,      99.1233,      96.8474,     94.407,
    91.7902,      88.9843,      85.9757,      82.7496,     79.2904,
    75.5812,      71.6039,      67.3392,      62.7663,     57.863,
    52.6053,      46.9676,      40.9225,      34.4406,     27.4902,
    20.0376,      12.0463,      10.0303,      9.63051,     9.20179,
    8.74209,      8.24916,      7.72062,      7.15388,     6.54618,
    5.89457,      5.41733,      5.11642,      4.79377,     4.44781,
    4.07684,      3.67906,      3.46153,      3.24276,     3.00818,
    2.75665,      2.60403,      2.4496,       2.28401,     2.12204,
    1.95238,      1.79157,      1.69061,      1.58269,     1.47253,
    1.35837,      1.25597,      1.14456,      1.02391,     0.966605,
    0.895225,     0.824658,     0.767541,     0.705951,    0.629913,
    0.582258,     0.532329,     0.461692,     0.435635,    0.388841,
    0.33881,      0.304432,     0.269373,     0.23607,     0.193027,
    0.160224,     0.129682,     0.0892866,    0.0633945,   0.0428321,
    0.0061503,    -0.0165869,   -0.0302532,   -0.0585708,  -0.0815517,
    -0.0925164,   -0.11196,     -0.123125,    -0.120789,   -0.124017,
    -0.10389,     0.0,          0.0,          0.0,         0.0,
    112.518,      111.579,      110.571,      109.491,     108.332,
    107.09,       105.758,      104.33,       102.799,     101.157,
    99.3958,      97.5078,      95.4834,      93.3126,     90.985,
    88.4892,      85.813,       82.9434,      79.8664,     76.5671,
    73.0293,      69.2359,      65.1683,      60.8068,     56.1301,
    51.1154,      45.7383,      39.9726,      33.7903,     27.1611,
    20.0529,      12.431,       9.85493,      9.44627,     9.00808,
    8.53823,      8.03441,      7.49419,      6.91493,     6.29381,
    5.6278,       5.12751,      4.87494,      4.60411,     4.3137,
    4.00232,      3.66842,      3.45487,      3.25134,     3.03311,
    2.7991,       2.607,        2.41205,      2.203,       2.03924,
    1.88818,      1.7383,       1.6477,       1.55055,     1.45622,
    1.35308,      1.21801,      1.1089,       1.04333,     0.972001,
    0.905446,     0.838771,     0.765188,     0.703638,    0.63999,
    0.599147,     0.510878,     0.463296,     0.422816,    0.376032,
    0.315101,     0.285652,     0.254546,     0.221058,    0.179315,
    0.135848,     0.122689,     0.0744373,    0.0474776,   0.0275433,
    0.00311178,   -0.023776,    -0.0433156,   -0.0595582,  -0.0821923,
    -0.0988291,   -0.112552,    -0.122729,    -0.131933,   -0.125465,
    -0.114519,    0.0,          0.0,          0.0,         0.0,
    108.212,      107.318,      106.36,       105.333,     104.231,
    103.05,       101.784,      100.426,      98.9695,     97.4081,
    95.7339,      93.9387,      92.0138,      89.9497,     87.7365,
    85.3634,      82.8187,      80.0902,      77.1645,     74.0273,
    70.6635,      67.0565,      63.1889,      59.0417,     54.5949,
    49.8267,      44.7139,      39.2316,      33.3532,     27.0499,
    20.2911,      13.0439,      9.97243,      9.54985,     9.09673,
    8.61087,      8.08989,      7.53127,      6.93227,     6.28999,
    5.60129,      5.04276,      4.78089,      4.50011,     4.19903,
    3.87619,      3.53002,      3.31852,      3.14045,     2.94952,
    2.74479,      2.56782,      2.39163,      2.20272,     2.04845,
    1.91169,      1.76541,      1.61365,      1.45092,     1.3826,
    1.3223,       1.22546,      1.12966,      1.04337,     0.988056,
    0.909555,     0.810447,     0.747686,     0.690795,    0.610184,
    0.57428,      0.531578,     0.479227,     0.394604,    0.364904,
    0.325378,     0.288404,     0.258938,     0.226087,    0.155716,
    0.166171,     0.104416,     0.095996,     0.0493812,   0.0292085,
    -0.00346806,  -0.0234786,   -0.0547235,   -0.0719212,  -0.0923959,
    -0.107187,    -0.116211,    -0.129412,    -0.136846,   -0.133772,
    -0.120914,    0.0,          0.0,          0.0,         0.0,
    107.371,      106.489,      105.543,      104.529,     103.441,
    102.275,      101.024,      99.6833,      98.2455,     96.7038,
    95.0507,      93.2781,      91.3775,      89.3394,     87.1541,
    84.8109,      82.2983,      79.6041,      76.7152,     73.6176,
    70.2961,      66.7346,      62.9157,      58.8208,     54.43,
    49.7219,      44.6735,      39.2603,      33.4559,     27.232,
    20.5584,      13.4024,      9.78554,      9.38236,     8.95005,
    8.48649,      7.98944,      7.45646,      6.88497,     6.27217,
    5.6151,       5.04226,      4.78881,      4.51703,     4.22562,
    3.91315,      3.5781,       3.34558,      3.15504,     2.95073,
    2.73165,      2.54806,      2.37588,      2.19127,     2.01475,
    1.84332,      1.65951,      1.56625,      1.46759,     1.37735,
    1.28432,      1.20639,      1.1237,       1.03192,     0.972306,
    0.906789,     0.821731,     0.763058,     0.659451,    0.627445,
    0.551942,     0.496717,     0.457058,     0.40738,     0.38458,
    0.318858,     0.27385,      0.237059,     0.21929,     0.168683,
    0.136031,     0.114836,     0.0752697,    0.0456078,   0.0108271,
    -0.00702722,  -0.027677,    -0.0492157,   -0.0727053,  -0.091384,
    -0.108568,    -0.122202,    -0.135046,    -0.146445,   -0.142621,
    -0.126708,    0.0,          0.0,          0.0,         0.0,
    105.212,      104.352,      103.43,       102.442,     101.382,
    100.246,      99.0275,      97.7209,      96.32,       94.8177,
    93.207,       91.4798,      89.6278,      87.642,      85.5127,
    83.2295,      80.7812,      78.1561,      75.3412,     72.323,
    69.0866,      65.6163,      61.8952,      57.9052,     53.6269,
    49.0394,      44.1204,      38.8458,      33.1901,     27.1257,
    20.623,       13.6504,      9.54951,      9.15801,     8.73822,
    8.28809,      7.80543,      7.28789,      6.73294,     6.1379,
    5.49985,      4.90776,      4.67129,      4.41773,     4.14585,
    3.85433,      3.54173,      3.29778,      3.09674,     2.88118,
    2.65003,      2.46028,      2.29666,      2.12122,     1.97387,
    1.86291,      1.74393,      1.62468,      1.49769,     1.38085,
    1.26294,      1.16065,      1.06046,      0.97668,     0.911513,
    0.847541,     0.768439,     0.712464,     0.628404,    0.592996,
    0.525678,     0.499264,     0.438595,     0.421504,    0.366115,
    0.331149,     0.26681,      0.242172,     0.198382,    0.169418,
    0.147909,     0.107071,     0.0844397,    0.0393455,   0.017157,
    -0.00288135,  -0.0276081,   -0.0628568,   -0.0794983,  -0.100924,
    -0.108507,    -0.126725,    -0.144607,    -0.140398,   -0.146144,
    -0.126008,    -0.103361,    0.0,          0.0,         0.0,
    101.854,      101.029,      100.143,      99.1937,     98.1756,
    97.0839,      95.9133,      94.6582,      93.3123,     91.8692,
    90.3218,      88.6625,      86.8834,      84.9757,     82.9301,
    80.7366,      78.3847,      75.8628,      73.1587,     70.2591,
    67.15,        63.8162,      60.2414,      56.4084,     52.2983,
    47.8912,      43.1656,      38.0985,      32.6652,     26.8393,
    20.5924,      13.894,       9.40486,      9.04221,     8.65337,
    8.23642,      7.78933,      7.30994,      6.79591,     6.24473,
    5.65371,      5.06452,      4.79711,      4.51037,     4.20291,
    3.87323,      3.51973,      3.24967,      3.0607,      2.85808,
    2.64081,      2.45035,      2.28664,      2.11109,     1.95119,
    1.82521,      1.69013,      1.56309,      1.43076,     1.33038,
    1.2459,       1.15216,      1.05355,      0.977538,    0.899314,
    0.825217,     0.769544,     0.689953,     0.653664,    0.607397,
    0.521089,     0.45038,      0.450798,     0.393773,    0.342762,
    0.3072,       0.277816,     0.231771,     0.190991,    0.170877,
    0.129543,     0.0905343,    0.0644,       0.0426622,   0.0132478,
    -0.0167198,   -0.0366951,   -0.0618276,   -0.0814521,  -0.100548,
    -0.123857,    -0.137326,    -0.143886,    -0.147656,   -0.144134,
    -0.142837,    -0.105593,    0.0,          0.0,         0.0,
    101.418,      100.598,      99.7188,      98.7758,     97.7646,
    96.6804,      95.5179,      94.2713,      92.9346,     91.5013,
    89.9645,      88.3166,      86.5496,      84.6549,     82.6232,
    80.4448,      78.1089,      75.6042,      72.9186,     70.0388,
    66.9509,      63.6399,      60.0895,      56.2826,     52.2006,
    47.8236,      43.1303,      38.0978,      32.7017,     26.9155,
    20.7112,      14.0586,      9.05406,      8.71276,     8.3468,
    7.95439,      7.53362,      7.08244,      6.59866,     6.07992,
    5.52369,      4.93916,      4.68442,      4.41126,     4.11837,
    3.8043,       3.46754,      3.20054,      3.03409,     2.8556,
    2.66422,      2.47764,      2.30206,      2.1138,      1.93529,
    1.79727,      1.64927,      1.54274,      1.44695,     1.35198,
    1.2562,       1.15565,      1.04939,      0.978182,    0.911061,
    0.842204,     0.737972,     0.714896,     0.661752,    0.577069,
    0.528043,     0.494774,     0.415554,     0.394098,    0.325571,
    0.289772,     0.260857,     0.223194,     0.195041,    0.147889,
    0.141109,     0.104318,     0.0457696,    0.0311903,   0.0134478,
    -0.0213937,   -0.0431784,   -0.0571138,   -0.0823552,  -0.100611,
    -0.127384,    -0.134989,    -0.150537,    -0.151569,   -0.159488,
    -0.150204,    -0.111509,    0.0,          0.0,         0.0,
    100.703,      99.8952,      99.0292,      98.1007,     97.105,
    96.0374,      94.8927,      93.6652,      92.349,      90.9377,
    89.4244,      87.8017,      86.0618,      84.1961,     82.1956,
    80.0506,      77.7505,      75.2842,      72.6396,     69.804,
    66.7634,      63.5031,      60.0072,      56.2586,     52.2392,
    47.9292,      43.3078,      38.3524,      33.0389,     27.3415,
    21.2322,      14.6815,      9.20317,      8.83885,     8.44821,
    8.02933,      7.58018,      7.09858,      6.58216,     6.02843,
    5.43468,      4.79803,      4.54924,      4.305,       4.04311,
    3.76229,      3.46118,      3.19103,      2.99487,     2.78454,
    2.55901,      2.36038,      2.22671,      2.08337,     1.93664,
    1.80315,      1.66002,      1.54368,      1.43824,     1.32975,
    1.21833,      1.12576,      1.04498,      0.979204,    0.893452,
    0.785992,     0.746598,     0.689635,     0.626923,    0.5724,
    0.532306,     0.477042,     0.430194,     0.386848,    0.351351,
    0.296058,     0.259247,     0.210703,     0.175685,    0.14988,
    0.123199,     0.0779,       0.0619193,    0.0328286,   0.00190681,
    -0.0218731,   -0.0624745,   -0.0737957,   -0.0934111,  -0.109664,
    -0.125961,    -0.141988,    -0.160629,    -0.162545,   -0.160942,
    -0.155838,    -0.124268,    0.0,          0.0,         0.0,
    96.789,       96.0187,      95.1927,      94.307,      93.3573,
    92.339,       91.2471,      90.0763,      88.8208,     87.4747,
    86.0313,      84.4835,      82.8239,      81.0444,     79.1362,
    77.0902,      74.8963,      72.5439,      70.0214,     67.3167,
    64.4165,      61.3067,      57.9722,      54.3967,     50.5628,
    46.4518,      42.0438,      37.3172,      32.249,      26.8146,
    20.9874,      14.7391,      8.99516,      8.63759,     8.25417,
    7.84305,      7.40222,      6.92953,      6.42267,     5.87919,
    5.29644,      4.67157,      4.40434,      4.17854,     3.93642,
    3.67681,      3.39843,      3.1382,       2.95654,     2.76176,
    2.5529,       2.35224,      2.19874,      2.03414,     1.86725,
    1.74347,      1.61075,      1.50647,      1.42247,     1.32751,
    1.21844,      1.10667,      0.991735,     0.924593,    0.871787,
    0.831996,     0.757672,     0.680307,     0.603889,    0.540553,
    0.510049,     0.462545,     0.42449,      0.359932,    0.336811,
    0.279489,     0.255635,     0.215572,     0.183584,    0.149878,
    0.109451,     0.0787453,    0.0561317,    0.0337079,   0.00689681,
    -0.023805,    -0.0465705,   -0.0671091,   -0.0988454,  -0.125893,
    -0.131433,    -0.153727,    -0.155077,    -0.175555,   -0.16314,
    -0.15888,     -0.127641,    0.0,          0.0,         0.0,
    96.8316,      96.0655,      95.2439,      94.363,      93.4184,
    92.4055,      91.3195,      90.155,       88.9063,     87.5673,
    86.1316,      84.5922,      82.9415,      81.1715,     79.2736,
    77.2386,      75.0564,      72.7166,      70.2077,     67.5175,
    64.6329,      61.5398,      58.2231,      54.6668,     50.8535,
    46.7646,      42.3802,      37.679,       32.638,      27.2327,
    21.4368,      15.222,       8.99573,      8.64492,     8.26877,
    7.86543,      7.43294,      6.9692,       6.47195,     5.93876,
    5.36704,      4.754,        4.43742,      4.18952,     3.92369,
    3.63866,      3.33302,      3.04223,      2.87402,     2.69366,
    2.50026,      2.31236,      2.19038,      2.05958,     1.92084,
    1.79154,      1.65289,      1.51477,      1.37726,     1.2593,
    1.19442,      1.11493,      1.01676,      0.923033,    0.838259,
    0.779991,     0.748361,     0.684996,     0.620056,    0.563805,
    0.498903,     0.458497,     0.413056,     0.359202,    0.322324,
    0.278977,     0.248213,     0.216736,     0.155718,    0.155264,
    0.112794,     0.0784504,    0.0375023,    0.0122049,   -0.00723023,
    -0.0270779,   -0.0622993,   -0.0761772,   -0.102197,   -0.11468,
    -0.141665,    -0.148863,    -0.164848,    -0.172223,   -0.174598,
    -0.167815,    -0.142513,    0.0,          0.0,         0.0,
    90.6838,      89.971,       89.2068,      88.3873,     87.5086,
    86.5664,      85.5561,      84.4728,      83.3112,     82.0657,
    80.7301,      79.2981,      77.7625,      76.116,      74.3505,
    72.4574,      70.4275,      68.2509,      65.917,      63.4144,
    60.731,       57.8536,      54.7683,      51.4601,     47.9128,
    44.1091,      40.0305,      35.6572,      30.9678,     25.9396,
    20.548,       14.7667,      8.56766,      8.1791,      7.83854,
    7.47337,      7.08182,      6.66197,      6.21177,     5.72905,
    5.21143,      4.65641,      4.33926,      4.11362,     3.87167,
    3.61224,      3.33405,      3.05401,      2.87823,     2.68973,
    2.48762,      2.28191,      2.14199,      1.99196,     1.83108,
    1.70492,      1.56987,      1.44967,      1.35472,     1.25457,
    1.15228,      1.06385,      1.00746,      0.929964,    0.842313,
    0.761344,     0.710541,     0.655493,     0.60339,     0.553389,
    0.495585,     0.430946,     0.383458,     0.365695,    0.312206,
    0.256312,     0.223571,     0.194705,     0.173799,    0.141038,
    0.11984,      0.0821131,    0.0386852,    0.0173664,   -0.0133723,
    -0.0400509,   -0.063391,    -0.0890864,   -0.106273,   -0.123629,
    -0.132812,    -0.157814,    -0.17262,     -0.178328,   -0.179471,
    -0.170771,    -0.148239,    0.0,          0.0,         0.0,
    95.1582,      94.4112,      93.6103,      92.7515,     91.8306,
    90.8432,      89.7845,      88.6492,      87.4318,     86.1265,
    84.7269,      83.2261,      81.6169,      79.8914,     78.0411,
    76.0572,      73.9299,      71.6488,      69.2029,     66.5803,
    63.7681,      60.7527,      57.5193,      54.0523,     50.3348,
    46.3486,      42.0743,      37.4912,      32.5768,     27.3073,
    21.6569,      15.5983,      9.10174,      8.21173,     7.87662,
    7.51729,      7.132,        6.71886,      6.27587,     5.80086,
    5.29152,      4.74538,      4.39739,      4.16414,     3.91404,
    3.64587,      3.35831,      3.05742,      2.87033,     2.66971,
    2.4546,       2.22895,      2.09489,      1.95115,     1.79701,
    1.68546,      1.57087,      1.4579,       1.35576,     1.24868,
    1.14585,      1.05073,      0.987853,     0.927613,    0.866597,
    0.793633,     0.716863,     0.651485,     0.592813,    0.522628,
    0.471509,     0.442225,     0.396347,     0.345165,    0.295438,
    0.256791,     0.226245,     0.191804,     0.167928,    0.119797,
    0.0980448,    0.0669648,    0.0369929,    0.00891417,  -0.0204241,
    -0.0479184,   -0.0594809,   -0.0915019,   -0.115544,   -0.125258,
    -0.146625,    -0.15834,     -0.173255,    -0.185838,   -0.178946,
    -0.180437,    -0.156722,    0.0,          0.0,         0.0,
    90.9546,      90.2506,      89.4957,      88.6862,     87.8182,
    86.8875,      85.8896,      84.8195,      83.6721,     82.4418,
    81.1225,      79.708,       78.1912,      76.5648,     74.8208,
    72.9508,      70.9457,      68.7957,      66.4903,     64.0182,
    61.3676,      58.5254,      55.4778,      52.2099,     48.7059,
    44.9487,      40.9199,      36.6,         31.9679,     27.0011,
    21.6753,      15.9646,      9.84126,      8.48962,     8.12415,
    7.73227,      7.31207,      6.86151,      6.37838,     5.86034,
    5.30486,      4.70924,      4.31647,      4.09543,     3.85842,
    3.60427,      3.33176,      3.03956,      2.84362,     2.63648,
    2.41437,      2.17621,      2.03892,      1.8957,      1.74213,
    1.64234,      1.54816,      1.44328,      1.32013,     1.19542,
    1.13218,      1.05988,      0.96467,      0.886417,    0.823959,
    0.766276,     0.67827,      0.618851,     0.59692,     0.541487,
    0.492781,     0.434291,     0.377835,     0.338008,    0.310483,
    0.268259,     0.221104,     0.181235,     0.164824,    0.113691,
    0.0872585,    0.0605444,    0.0405557,    0.00398859,  -0.0238138,
    -0.0211823,   -0.0751806,   -0.0967716,   -0.115314,   -0.128789,
    -0.155474,    -0.166276,    -0.174839,    -0.189366,   -0.200468,
    -0.185949,    -0.164216,    0.0,          0.0,         0.0,
    92.2647,      91.552,       90.7878,      89.9683,     89.0896,
    88.1474,      87.1372,      86.0539,      84.8923,     83.6468,
    82.3113,      80.8792,      79.3437,      77.6972,     75.9317,
    74.0386,      72.0088,      69.8322,      67.4983,     64.9958,
    62.3124,      59.4351,      56.3499,      53.0417,     49.4944,
    45.6908,      41.6123,      37.2391,      32.5498,     27.5216,
    22.1301,      16.3489,      10.15,        8.3159,      7.96052,
    7.57946,      7.17086,      6.73273,      6.26294,     5.7592,
    5.21905,      4.63987,      4.21855,      3.98837,     3.74154,
    3.47688,      3.19309,      2.8888,       2.71058,     2.53668,
    2.35021,      2.15027,      2.032,        1.91733,     1.79438,
    1.67002,      1.53908,      1.40615,      1.29529,     1.17759,
    1.10958,      1.03912,      0.980485,     0.925532,    0.876121,
    0.757196,     0.667516,     0.612664,     0.573126,    0.508816,
    0.456064,     0.441465,     0.368915,     0.340811,    0.287953,
    0.249532,     0.242745,     0.173006,     0.139751,    0.129528,
    0.0865628,    0.0523681,    0.0375244,    -0.00108011, -0.0243586,
    -0.0569402,   -0.0780482,   -0.0887733,   -0.117367,   -0.127364,
    -0.157701,    -0.16601,     -0.174493,    -0.191225,   -0.200216,
    -0.190643,    -0.169545,    0.0,          0.0,         0.0,
    88.5768,      87.8969,      87.1679,      86.3862,     85.548,
    84.6493,      83.6856,      82.6522,      81.5442,     80.3561,
    79.0821,      77.7161,      76.2513,      74.6807,     72.9966,
    71.1908,      69.2545,      67.1783,      64.952,      62.5648,
    60.0051,      57.2604,      54.3174,      51.1617,     47.7779,
    44.1497,      40.2592,      36.0875,      31.6144,     26.818,
    21.675,       16.1603,      10.2471,      8.04007,     7.69734,
    7.32984,      6.93579,      6.51325,      6.06018,     5.57437,
    5.05346,      4.49489,      4.06082,      3.84508,     3.61375,
    3.36571,      3.09973,      2.81454,      2.64914,     2.50319,
    2.34669,      2.17888,      2.02849,      1.8742,      1.70876,
    1.60985,      1.541,        1.45976,      1.31599,     1.16183,
    1.10405,      1.04661,      0.946695,     0.862778,    0.81101,
    0.73604,      0.660601,     0.600032,     0.565579,    0.515207,
    0.460083,     0.428664,     0.380306,     0.330824,    0.284014,
    0.238488,     0.222811,     0.174064,     0.155776,    0.108995,
    0.067761,     0.0423566,    0.0235009,    0.00519845,  -0.0400053,
    -0.0527223,   -0.0813729,   -0.0977192,   -0.123077,   -0.132828,
    -0.162321,    -0.191081,    -0.199567,    -0.1973,     -0.200323,
    -0.190273,    -0.178208,    0.0,          0.0,         0.0,
    88.2989,      87.6215,      86.8951,      86.1162,     85.281,
    84.3855,      83.4252,      82.3956,      81.2915,     80.1077,
    78.8383,      77.4772,      76.0176,      74.4527,     72.7746,
    70.9753,      69.0459,      66.9771,      64.7588,     62.3801,
    59.8296,      57.0948,      54.1623,      51.0179,     47.6463,
    44.031,       40.1544,      35.9977,      31.5406,     26.7614,
    21.6368,      16.1419,      10.2498,      7.61643,     7.30801,
    6.97731,      6.62271,      6.24248,      5.83477,     5.39761,
    4.92884,      4.4262,       4.00862,      3.81628,     3.61004,
    3.3889,       3.15178,      2.89752,      2.7149,      2.55081,
    2.37485,      2.18618,      2.03628,      1.8948,      1.74309,
    1.61607,      1.50351,      1.38397,      1.28231,     1.1733,
    1.10055,      1.02954,      0.945536,     0.860026,    0.779276,
    0.722851,     0.665193,     0.60099,      0.544754,    0.515558,
    0.444003,     0.40439,      0.34471,      0.322648,    0.268992,
    0.24306,      0.202354,     0.171402,     0.136327,    0.0956033,
    0.0701694,    0.0487356,    0.00666399,   -0.0128606,  -0.023116,
    -0.0544528,   -0.0869919,   -0.105413,    -0.127806,   -0.15894,
    -0.168382,    -0.179665,    -0.197365,    -0.198064,   -0.202529,
    -0.214929,    -0.176682,    0.0,          0.0,         0.0,
    87.6967,      87.0318,      86.3189,      85.5544,     84.7347,
    83.8558,      82.9133,      81.9028,      80.8192,     79.6572,
    78.4114,      77.0754,      75.643,       74.107,      72.46,
    70.694,       68.8004,      66.7699,      64.5927,     62.2581,
    59.7549,      57.0707,      54.1926,      51.1064,     47.7973,
    44.2489,      40.4442,      36.3645,      31.99,       27.2993,
    22.2697,      16.8766,      11.0937,      8.04446,     7.70628,
    7.34366,      6.95483,      6.5379,       6.09085,     5.61149,
    5.09748,      4.54633,      4.0605,       3.85951,     3.64399,
    3.41289,      3.1651,       2.8994,       2.69469,     2.51605,
    2.32451,      2.11911,      1.96985,      1.84727,     1.71584,
    1.5905,       1.47022,      1.34125,      1.24516,     1.14361,
    1.06904,      0.998445,     0.930328,     0.860596,    0.794165,
    0.723825,     0.654535,     0.591768,     0.536073,    0.474324,
    0.437497,     0.393739,     0.356048,     0.326462,    0.291944,
    0.246023,     0.21413,      0.152816,     0.131441,    0.118951,
    0.0809931,    0.039331,     0.0112695,    -0.00921663, -0.0346798,
    -0.068224,    -0.0938637,   -0.102667,    -0.136554,   -0.148997,
    -0.163677,    -0.18233,     -0.200564,    -0.202707,   -0.209008,
    -0.215873,    -0.192664,    0.0,          0.0,         0.0,
    87.646,       86.9814,      86.2688,      85.5047,     84.6854,
    83.8069,      82.8649,      81.8548,      80.7717,     79.6103,
    78.365,       77.0297,      75.598,       74.0627,     72.4165,
    70.6513,      68.7586,      66.7291,      64.5529,     62.2195,
    59.7174,      57.0345,      54.1577,      51.0731,     47.7655,
    44.2188,      40.4159,      36.3381,      31.9657,     27.2772,
    22.25,        16.8594,      11.0793,      7.6012,      7.29553,
    6.96776,      6.61631,      6.23945,      5.83537,     5.40208,
    4.93748,      4.4393,       3.97001,      3.77618,     3.56835,
    3.34549,      3.10653,      2.8503,       2.64781,     2.48168,
    2.30353,      2.11251,      1.96596,      1.85131,     1.72836,
    1.6057,       1.48539,      1.35639,      1.26728,     1.17792,
    1.08935,      0.997353,     0.917705,     0.839804,    0.774924,
    0.706193,     0.652282,     0.633124,     0.539075,    0.474374,
    0.440218,     0.389689,     0.353912,     0.302007,    0.2678,
    0.235096,     0.201855,     0.158493,     0.129428,    0.092664,
    0.0706539,    0.035969,     0.0161182,    -0.020366,   -0.0462192,
    -0.06667,     -0.0863025,   -0.10602,     -0.146508,   -0.156668,
    -0.182623,    -0.178391,    -0.198767,    -0.207625,   -0.217068,
    -0.212559,    -0.202735,    -0.168617,    0.0,         0.0,
    80.6118,      80.008,       79.3606,      78.6664,     77.922,
    77.1238,      76.268,       75.3502,      74.3662,     73.3111,
    72.1796,      70.9665,      69.6656,      68.2708,     66.7752,
    65.1714,      63.4518,      61.6079,      59.6307,     57.5107,
    55.2374,      52.7999,      50.1862,      47.3836,     44.3785,
    41.1563,      37.7011,      33.9963,      30.0237,     25.7641,
    21.1966,      16.299,       11.0475,      7.47567,     7.18026,
    6.8635,       6.52384,      6.15965,      5.76913,     5.35039,
    4.90138,      4.41993,      3.93953,      3.74614,     3.53876,
    3.3164,       3.07797,      2.82232,      2.61084,     2.44404,
    2.26519,      2.07341,      1.91861,      1.80283,     1.67868,
    1.56262,      1.46661,      1.36366,      1.24654,     1.1194,
    1.04628,      1.00455,      0.92671,      0.834479,    0.764393,
    0.705521,     0.65177,      0.586159,     0.539392,    0.502012,
    0.431201,     0.389804,     0.332661,     0.305873,    0.270095,
    0.235536,     0.178786,     0.16123,      0.118699,    0.0984726,
    0.065354,     0.0292286,    0.00498338,   -0.0247551,  -0.0561214,
    -0.0718457,   -0.08861,     -0.121361,    -0.142195,   -0.162845,
    -0.177993,    -0.183582,    -0.206816,    -0.213263,   -0.233062,
    -0.219423,    -0.206251,    -0.169659,    0.0,         0.0,
    79.405,       78.8217,      78.1964,      77.5258,     76.8068,
    76.0358,      75.2091,      74.3226,      73.3721,     72.3529,
    71.26,        70.0882,      68.8316,      67.4843,     66.0396,
    64.4905,      62.8294,      61.0483,      59.1385,     57.0907,
    54.8949,      52.5404,      50.0157,      47.3086,     44.4059,
    41.2934,      37.9559,      34.3773,      30.54,       26.4255,
    22.0135,      17.2828,      12.2102,      7.19966,     6.9255,
    6.63152,      6.3163,       5.97829,      5.61586,     5.22724,
    4.81053,      4.36371,      3.8846,       3.63348,     3.42914,
    3.21004,      2.9751,       2.72318,      2.48659,     2.35354,
    2.21088,      2.05791,      1.89907,      1.7483,      1.58664,
    1.42279,      1.3503,       1.27257,      1.18604,     1.09023,
    0.997826,     0.919103,     0.846503,     0.783254,    0.728165,
    0.667363,     0.588946,     0.54906,      0.503444,    0.448216,
    0.40933,      0.372841,     0.315705,     0.286046,    0.259118,
    0.209256,     0.17539,      0.13682,      0.104149,    0.0682578,
    0.0346061,    0.0135789,    -0.0149031,   -0.0453862,  -0.0681647,
    -0.0883782,   -0.107714,    -0.149772,    -0.164719,   -0.17743,
    -0.202008,    -0.213584,    -0.226982,    -0.239996,   -0.241189,
    -0.239836,    -0.233345,    -0.205767,    0.0,         0.0,
    77.842,       77.2775,      76.6723,      76.0232,     75.3273,
    74.581,       73.7809,      72.9229,      72.0029,     71.0164,
    69.9587,      68.8245,      67.6083,      66.3042,     64.9059,
    63.4066,      61.7989,      60.075,       58.2265,     56.2445,
    54.1192,      51.8403,      49.3967,      46.7765,     43.967,
    40.9545,      37.7242,      34.2605,      30.5465,     26.5641,
    22.2939,      17.7151,      12.8054,      7.54086,     6.57115,
    6.30579,      6.02125,      5.71614,      5.38899,     5.0382,
    4.66206,      4.25873,      3.82625,      3.53167,     3.33793,
    3.13018,      2.90742,      2.66857,      2.41322,     2.28343,
    2.14426,      1.99503,      1.83501,      1.71096,     1.57806,
    1.43556,      1.33434,      1.23385,      1.13902,     1.06819,
    0.992892,     0.916842,     0.84102,      0.778718,    0.71073,
    0.638316,     0.578729,     0.521591,     0.476915,    0.439209,
    0.382039,     0.323666,     0.301035,     0.271072,    0.220616,
    0.199477,     0.161419,     0.128859,     0.0825178,   0.0524986,
    0.0314532,    -0.00983029,  -0.0351146,   -0.0530354,  -0.0826967,
    -0.105526,    -0.130433,    -0.158031,    -0.169848,   -0.19908,
    -0.199884,    -0.232068,    -0.237958,    -0.248517,   -0.262047,
    -0.262824,    -0.258588,    -0.228877,    0.0,         0.0,
    77.9932,      77.4367,      76.84,        76.2001,     75.5141,
    74.7784,      73.9896,      73.1438,      72.2369,     71.2644,
    70.2216,      69.1035,      67.9046,      66.619,      65.2405,
    63.7625,      62.1776,      60.4781,      58.6559,     56.7019,
    54.6068,      52.3602,      49.9513,      47.3683,     44.5986,
    41.6288,      38.4444,      35.0298,      31.3685,     27.4426,
    23.2329,      18.7191,      13.879,       8.68916,     6.61655,
    6.36386,      6.09291,      5.80238,      5.49085,     5.15681,
    4.79862,      4.41456,      4.00273,      3.65004,     3.42562,
    3.18497,      2.92693,      2.65025,      2.35357,     2.18777,
    2.05172,      1.90582,      1.74939,      1.63531,     1.52838,
    1.41373,      1.31821,      1.23077,      1.13852,     1.0558,
    0.967099,     0.869457,     0.765958,     0.718331,    0.669465,
    0.621219,     0.577841,     0.527184,     0.465999,    0.401794,
    0.365547,     0.316251,     0.286875,     0.238384,    0.190171,
    0.160344,     0.154619,     0.102172,     0.0661719,   0.0554026,
    0.0122689,    -0.0197294,   -0.0356173,   -0.0518539,  -0.0829023,
    -0.113012,    -0.137215,    -0.167018,    -0.173387,   -0.211205,
    -0.227529,    -0.24289,     -0.254311,    -0.265328,   -0.27834,
    -0.283266,    -0.279993,    -0.267749,    0.0,         0.0,
    74.698,       74.1747,      73.6136,      73.0119,     72.3667,
    71.6749,      70.9331,      70.1377,      69.2848,     68.3702,
    67.3896,      66.3381,      65.2107,      64.0017,     62.7054,
    61.3154,      59.8249,      58.2268,      56.5131,     54.6756,
    52.7053,      50.5926,      48.3273,      45.8982,     43.2936,
    40.5007,      37.5061,      34.295,       30.8518,     27.1599,
    23.2011,      18.9562,      14.4046,      9.52404,     6.51617,
    6.2723,       6.01081,      5.73043,      5.42978,     5.1074,
    4.76172,      4.39107,      3.99362,      3.6071,      3.40331,
    3.1848,       2.9505,       2.69926,      2.42987,     2.2435,
    2.1207,       1.98902,      1.84782,      1.70899,     1.56988,
    1.42072,      1.30013,      1.22208,      1.13839,     1.05983,
    0.977237,     0.891799,     0.80157,      0.741125,    0.683474,
    0.612039,     0.564459,     0.522552,     0.460304,    0.398877,
    0.366459,     0.318317,     0.28532,      0.226091,    0.192917,
    0.165876,     0.136505,     0.0899093,    0.0632705,   0.0344804,
    -7.74125e-06, -0.0265927,   -0.0460567,   -0.0960974,  -0.114763,
    -0.136281,    -0.146026,    -0.180657,    -0.195061,   -0.227576,
    -0.241327,    -0.244814,    -0.280936,    -0.274005,   -0.297669,
    -0.300615,    -0.297014,    -0.280683,    -0.234999,   0.0,
    74.1948,      73.6827,      73.1336,      72.5448,     71.9134,
    71.2364,      70.5105,      69.7322,      68.8976,     68.0026,
    67.043,       66.0141,      64.9108,      63.7278,     62.4592,
    61.099,       59.6405,      58.0766,      56.3997,     54.6016,
    52.6735,      50.6061,      48.3893,      46.0123,     43.4635,
    40.7306,      37.8001,      34.6578,      31.2885,     27.6757,
    23.8017,      19.6479,      15.1938,      10.4179,     6.44007,
    6.18894,      5.91965,      5.63091,      5.32129,     4.98931,
    4.63333,      4.25162,      3.84233,      3.40346,     3.21226,
    3.02092,      2.81575,      2.59575,      2.35986,     2.15541,
    2.02128,      1.87746,      1.72324,      1.58437,     1.48354,
    1.37544,      1.2691,       1.18743,      1.09986,     1.02354,
    0.95074,      0.875862,     0.798972,     0.72759,     0.658575,
    0.612757,     0.561163,     0.501954,     0.460291,    0.392732,
    0.364204,     0.310011,     0.259449,     0.21969,     0.176562,
    0.150669,     0.112597,     0.0669855,    0.0477638,   0.0204432,
    -0.00691221,  -0.0423255,   -0.0617101,   -0.101144,   -0.11769,
    -0.141214,    -0.160778,    -0.191959,    -0.208344,   -0.241567,
    -0.255448,    -0.264604,    -0.282041,    -0.300094,   -0.312857,
    -0.316851,    -0.313025,    -0.303787,    -0.263533,   0.0,
    71.52,        71.0349,      70.5147,      69.957,      69.3589,
    68.7177,      68.0301,      67.2928,      66.5022,     65.6545,
    64.7455,      63.7709,      62.7258,      61.6051,     60.4035,
    59.1151,      57.7335,      56.2521,      54.6637,     52.9604,
    51.1341,      49.1758,      47.0759,      44.8243,     42.41,
    39.8212,      37.0454,      34.0689,      30.8773,     27.4551,
    23.7856,      19.8508,      15.6318,      11.1078,     6.36504,
    6.13036,      5.87873,      5.60892,      5.3196,      5.00938,
    4.67674,      4.32006,      3.9376,       3.52751,     3.28409,
    3.09095,      2.88384,      2.66177,      2.42365,     2.19408,
    2.08186,      1.96154,      1.83252,      1.69688,     1.56636,
    1.4264,       1.27905,      1.21345,      1.14312,     1.05937,
    0.959582,     0.866875,     0.804227,     0.733169,    0.650902,
    0.584858,     0.526785,     0.478795,     0.41678,     0.385686,
    0.355739,     0.306757,     0.247374,     0.201832,    0.195694,
    0.140041,     0.0939671,    0.0863928,    0.046541,    0.0230238,
    -0.0206589,   -0.0447422,   -0.0857832,   -0.109905,   -0.137807,
    -0.15694,     -0.177827,    -0.209104,    -0.22359,    -0.245357,
    -0.256369,    -0.286853,    -0.301518,    -0.303664,   -0.323007,
    -0.332119,    -0.332617,    -0.319846,    -0.292358,   0.0,
    73.3437,      72.8553,      72.3316,      71.7701,     71.168,
    70.5224,      69.8301,      69.0878,      68.2918,     67.4384,
    66.5232,      65.5419,      64.4897,      63.3615,     62.1518,
    60.8546,      59.4636,      57.9722,      56.3729,     54.6581,
    52.8194,      50.8478,      48.7337,      46.4668,     44.0361,
    41.4297,      38.635,       35.6383,      32.4251,     28.9796,
    25.2852,      21.3237,      17.076,       12.5213,     7.63746,
    6.5079,       6.22775,      5.92735,      5.60524,     5.25986,
    4.88951,      4.4924,       4.0666,       3.61002,     3.29848,
    3.11679,      2.92196,      2.71306,      2.48906,     2.24887,
    2.11137,      1.96858,      1.81548,      1.65131,     1.54849,
    1.44173,      1.32725,      1.20625,      1.07687,     0.965313,
    0.9254,       0.878298,     0.779765,     0.686672,    0.640136,
    0.587065,     0.527948,     0.4865,       0.434503,    0.388891,
    0.350919,     0.301801,     0.266555,     0.220201,    0.175886,
    0.141432,     0.096292,     0.0615914,    0.0340129,   -0.00442873,
    -0.0413274,   -0.0465571,   -0.0988881,   -0.114356,   -0.15993,
    -0.173394,    -0.197292,    -0.220333,    -0.245443,   -0.255052,
    -0.279961,    -0.302625,    -0.318544,    -0.329361,   -0.348158,
    -0.346299,    -0.33959,     -0.350954,    -0.32847,    0.0,
    74.6744,      74.1818,      73.6536,      73.0872,     72.4798,
    71.8286,      71.1303,      70.3816,      69.5787,     68.7178,
    67.7947,      66.8049,      65.7436,      64.6055,     63.3852,
    62.0768,      60.6737,      59.1693,      57.5562,     55.8265,
    53.9717,      51.983,       49.8505,      47.5639,     45.1121,
    42.4831,      39.6641,      36.6413,      33.4002,     29.9248,
    26.1982,      22.2023,      17.9177,      13.3234,     8.39711,
    6.29403,      6.04677,      5.78165,      5.49737,     5.19255,
    4.86569,      4.51522,      4.13941,      3.73645,     3.39551,
    3.20653,      3.00389,      2.78661,      2.55363,     2.30381,
    2.14325,      2.0053,       1.85738,      1.69877,     1.58217,
    1.47495,      1.35997,      1.24736,      1.13314,     1.01511,
    0.958492,     0.897783,     0.832836,     0.763215,    0.670495,
    0.586976,     0.531221,     0.479507,     0.430698,    0.384395,
    0.337243,     0.295027,     0.245105,     0.21946,     0.161912,
    0.12039,      0.0881974,    0.0531092,    0.0217769,   0.00767638,
    -0.0609791,   -0.0738094,   -0.112015,    -0.127841,   -0.152748,
    -0.186463,    -0.208294,    -0.233754,    -0.262577,   -0.279818,
    -0.297742,    -0.304398,    -0.3219,      -0.352785,   -0.360561,
    -0.365765,    -0.369848,    -0.369956,    -0.346587,   0.0,
    76.6816,      75.7458,      75.215,       74.6459,     74.0357,
    73.3814,      72.6798,      71.9274,      71.1208,     70.2558,
    69.3283,      68.3338,      67.2674,      66.1239,     64.8978,
    63.5831,      62.1734,      60.6619,      59.041,      57.3031,
    55.4395,      53.4413,      51.2987,      49.0012,     46.5377,
    43.8962,      41.0638,      38.0267,      34.7701,     31.2781,
    27.5338,      23.5189,      19.2139,      14.5977,     9.64799,
    6.53467,      6.27487,      5.99629,      5.69759,     5.37729,
    5.03385,      4.66559,      4.27072,      3.84731,     3.43873,
    3.26353,      3.07568,      2.87425,      2.65827,     2.42667,
    2.24015,      2.0888,       1.92652,      1.7525,      1.61955,
    1.52059,      1.41448,      1.30136,      1.18096,     1.05186,
    0.953943,     0.855492,     0.7986,       0.760326,    0.692854,
    0.617906,     0.566312,     0.506905,     0.449263,    0.409366,
    0.363656,     0.297403,     0.258123,     0.208022,    0.15717,
    0.124454,     0.0846356,    0.0661099,    0.0196508,   -0.0227557,
    -0.0478475,   -0.0658332,   -0.122357,    -0.140827,   -0.173536,
    -0.186864,    -0.226994,    -0.240437,    -0.261626,   -0.290142,
    -0.317946,    -0.333281,    -0.350915,    -0.355896,   -0.375206,
    -0.384567,    -0.397784,    -0.387318,    -0.374163,   -0.322597,
    81.0897,      79.1176,      78.5724,      77.9878,     77.361,
    76.689,       75.9683,      75.1955,      74.367,      73.4785,
    72.5258,      71.5043,      70.409,       69.2344,     67.9751,
    66.6247,      65.1767,      63.6241,      61.9592,     60.1741,
    58.26,        56.2075,      54.0067,      51.6468,     49.1165,
    46.4032,      43.4939,      40.3743,      37.0293,     33.4425,
    29.5965,      25.4726,      21.0507,      16.3092,     11.2251,
    7.0039,       6.72134,      6.41836,      6.09349,     5.74514,
    5.37161,      4.97109,      4.54162,      4.08112,     3.58734,
    3.38933,      3.19203,      2.98047,      2.75363,     2.51039,
    2.29766,      2.15278,      1.99743,      1.83085,     1.68138,
    1.57329,      1.45738,      1.34131,      1.2441,      1.13986,
    1.05136,      0.968259,     0.883265,     0.796467,    0.71804,
    0.643776,     0.575746,     0.507983,     0.442276,    0.384846,
    0.35058,      0.294709,     0.250792,     0.211705,    0.17941,
    0.115931,     0.0969617,    0.0309885,    0.0168647,   -0.0209503,
    -0.0571235,   -0.0901925,   -0.118383,    -0.14504,    -0.191148,
    -0.200832,    -0.22815,     -0.268303,    -0.276282,   -0.30977,
    -0.315663,    -0.338453,    -0.374138,    -0.375148,   -0.388843,
    -0.398216,    -0.407044,    -0.419182,    -0.390419,   -0.35459,
    83.9681,      80.941,       80.3913,      79.8019,     79.1699,
    78.4923,      77.7657,      76.9865,      76.1511,     75.2553,
    74.2947,      73.2647,      72.1603,      70.9761,     69.7063,
    68.3447,      66.8848,      65.3193,      63.6407,     61.8408,
    59.9108,      57.8413,      55.6223,      53.243,      50.6916,
    47.9559,      45.0225,      41.8771,      38.5044,     34.888,
    31.0102,      26.8522,      22.3937,      17.613,      12.4868,
    7.20601,      6.92405,      6.62171,      6.29752,     5.9499,
    5.57716,      5.17749,      4.74893,      4.2894,      3.79666,
    3.52654,      3.31813,      3.09466,      2.85504,     2.5981,
    2.35057,      2.21426,      2.0681,       1.91138,     1.75238,
    1.62593,      1.49035,      1.34827,      1.26541,     1.17657,
    1.08861,      1.00248,      0.90939,      0.807872,    0.712964,
    0.631504,     0.584237,     0.538428,     0.461234,    0.406463,
    0.350692,     0.298671,     0.259766,     0.21798,     0.163415,
    0.125718,     0.0726839,    0.0550464,    0.00248794,  -0.0129885,
    -0.0741873,   -0.091072,    -0.128063,    -0.150518,   -0.183673,
    -0.213109,    -0.249196,    -0.28554,     -0.284372,   -0.320401,
    -0.333503,    -0.363075,    -0.382726,    -0.396531,   -0.413779,
    -0.424932,    -0.420504,    -0.433135,    -0.412932,   -0.386487,
    89.1307,      84.903,       84.3333,      83.7224,     83.0674,
    82.3651,      81.612,       80.8044,      79.9386,     79.0101,
    78.0145,      76.947,       75.8024,      74.575,      73.259,
    71.8478,      70.3347,      68.7122,      66.9724,     65.1069,
    63.1066,      60.9618,      58.6619,      56.1959,     53.5516,
    50.7162,      47.676,       44.416,       40.9204,     37.1722,
    33.1532,      28.8437,      24.2227,      19.2679,     13.9549,
    8.25798,      7.13504,      6.83184,      6.50674,     6.15814,
    5.78435,      5.38355,      4.95378,      4.49295,     3.99883,
    3.67512,      3.48354,      3.27812,      3.05786,     2.82167,
    2.56842,      2.40176,      2.22363,      2.03263,     1.82782,
    1.69839,      1.56085,      1.41336,      1.314,       1.2175,
    1.11864,      1.02426,      0.92705,      0.854421,    0.776525,
    0.692937,     0.623042,     0.563843,     0.494389,    0.448376,
    0.395976,     0.331273,     0.282466,     0.235445,    0.180747,
    0.141167,     0.100223,     0.0654548,    0.0251426,   -0.0164644,
    -0.0367024,   -0.105259,    -0.136744,    -0.159922,   -0.197134,
    -0.223564,    -0.247165,    -0.29312,     -0.31353,    -0.327084,
    -0.351485,    -0.372184,    -0.387408,    -0.426177,   -0.428866,
    -0.432663,    -0.445189,    -0.445953,    -0.455814,   -0.424192,
    94.6376,      89.091,       88.5009,      87.8682,     87.1897,
    86.4622,      85.6821,      84.8457,      83.9488,     82.9871,
    81.9559,      80.8501,      79.6645,      78.3931,     77.0299,
    75.5682,      74.0009,      72.3202,      70.5181,     68.5858,
    66.5139,      64.2922,      61.91,        59.3556,     56.6166,
    53.6796,      50.5304,      47.1537,      43.5329,     39.6504,
    35.4874,      31.0235,      26.237,       21.1046,     15.6013,
    9.70035,      7.43762,      7.12824,      6.7965,      6.44079,
    6.05936,      5.65038,      5.21184,      4.7416,      4.23739,
    3.84241,      3.65239,      3.44864,      3.23016,     2.99589,
    2.7447,       2.55244,      2.36516,      2.16435,     1.94902,
    1.80627,      1.6759,       1.5361,       1.4105,      1.2881,
    1.1615,       1.06655,      0.964724,     0.888775,    0.810518,
    0.740731,     0.662811,     0.573875,     0.5133,      0.459193,
    0.397658,     0.321239,     0.306621,     0.235633,    0.17857,
    0.156781,     0.103269,     0.0420071,    0.00677727,  -0.0196715,
    -0.0631737,   -0.091018,    -0.136572,    -0.167738,   -0.209603,
    -0.251043,    -0.251848,    -0.283404,    -0.325812,   -0.353538,
    -0.35936,     -0.391904,    -0.407293,    -0.424336,   -0.436819,
    -0.457959,    -0.459976,    -0.484313,    -0.468455,   -0.454035,
    95.3998,      88.9663,      88.1841,      87.5665,     86.9044,
    86.1944,      85.4331,      84.6167,      83.7414,     82.8028,
    81.7964,      80.7172,      79.5601,      78.3193,     76.9889,
    75.5623,      74.0327,      72.3924,      70.6337,     68.7479,
    66.7257,      64.5575,      62.2325,      59.7395,     57.0664,
    54.2001,      51.1266,      47.831,       44.2973,     40.5082,
    36.4453,      32.0887,      27.4173,      22.4084,     17.0374,
    11.2783,      7.98294,      7.65694,      7.30738,     6.93256,
    6.53065,      6.0997,       5.6376,       5.14211,     4.61081,
    4.11786,      3.89025,      3.64619,      3.3845,      3.10389,
    2.803,        2.5882,       2.42478,      2.24956,     2.06168,
    1.89927,      1.75019,      1.59034,      1.45678,     1.35456,
    1.24495,      1.13911,      1.02664,      0.939222,    0.857171,
    0.78031,      0.702449,     0.630708,     0.568961,    0.495158,
    0.393066,     0.336478,     0.295868,     0.249003,    0.209664,
    0.148861,     0.119767,     0.0637858,    0.000536039, -0.0221724,
    -0.065426,    -0.111423,    -0.160042,    -0.182754,   -0.200449,
    -0.247163,    -0.280315,    -0.313017,    -0.340347,   -0.359838,
    -0.377923,    -0.416788,    -0.432482,    -0.459421,   -0.445741,
    -0.460443,    -0.482256,    -0.490691,    -0.496111,   -0.464339,
    106.089,      98.9215,      96.9419,      96.2701,     95.5497,
    94.7772,      93.9489,      93.0608,      92.1085,     91.0873,
    89.9924,      88.8183,      87.5594,      86.2095,     84.762,
    83.21,        81.5458,      79.7613,      77.8478,     75.7961,
    73.5961,      71.2371,      68.7076,      65.9954,     63.0871,
    59.9687,      56.6249,      53.0394,      49.1949,     45.0725,
    40.6521,      35.9124,      30.8301,      25.3805,     19.5372,
    13.2715,      8.52246,      8.17357,      7.79946,     7.39831,
    6.96818,      6.50696,      6.01241,      5.48212,     4.91351,
    4.31523,      4.08841,      3.84519,      3.58439,     3.30475,
    3.00489,      2.75669,      2.58559,      2.40212,     2.20539,
    2.02745,      1.88085,      1.72364,      1.56951,     1.43786,
    1.2967,       1.17848,      1.0637,       0.968785,    0.889442,
    0.820168,     0.753543,     0.660001,     0.584553,    0.525766,
    0.450439,     0.38675,      0.332402,     0.269497,    0.234448,
    0.187669,     0.114106,     0.0639676,    -0.00170035, -0.0214646,
    -0.0757554,   -0.10401,     -0.14861,     -0.173773,   -0.211295,
    -0.259303,    -0.273073,    -0.296363,    -0.339255,   -0.378948,
    -0.410825,    -0.424196,    -0.430824,    -0.469694,   -0.481582,
    -0.497486,    -0.489194,    -0.505322,    -0.514466,   -0.502244,
    112.27,       104.695,      101.443,      100.751,     100.01,
    99.2144,      98.3616,      97.4472,      96.4667,     95.4153,
    94.288,       93.0792,      91.783,       90.3932,     88.9029,
    87.3049,      85.5915,      83.7542,      81.7841,     79.6717,
    77.4066,      74.9779,      72.3736,      69.5811,     66.5867,
    63.376,       59.9333,      56.2418,      52.2835,     48.0391,
    43.488,       38.6081,      33.3754,      27.7646,     21.7484,
    15.2973,      9.20521,      8.8386,       8.44551,     8.024,
    7.57203,      7.08741,      6.56775,      6.01055,     5.41308,
    4.77242,      4.46978,      4.21607,      3.94403,     3.65233,
    3.33955,      3.046,        2.85127,      2.64247,     2.41858,
    2.20521,      2.05625,      1.89653,      1.72899,     1.57545,
    1.41081,      1.29322,      1.21464,      1.12347,     1.01433,
    0.900875,     0.78295,      0.703304,     0.628671,    0.550242,
    0.507916,     0.430551,     0.346987,     0.293863,    0.238706,
    0.191754,     0.139728,     0.0985042,    0.0259078,   -0.00153777,
    -0.0481238,   -0.0999855,   -0.150573,    -0.188014,   -0.240285,
    -0.250641,    -0.286137,    -0.337572,    -0.363925,   -0.393216,
    -0.407904,    -0.426978,    -0.461181,    -0.46194,    -0.497776,
    -0.495557,    -0.521236,    -0.515119,    -0.525104,   -0.515841,
    119.189,      111.149,      106.496,      105.78,      105.013,
    104.191,      103.309,      102.363,      101.349,     100.261,
    99.0953,      97.845,       96.5044,      95.0669,     93.5255,
    91.8727,      90.1005,      88.2002,      86.1626,     83.9777,
    81.6349,      79.1229,      76.4292,      73.541,      70.444,
    67.1231,      63.5623,      59.7442,      55.6501,     51.2602,
    46.553,       41.5057,      36.0936,      30.2903,     24.0677,
    17.3954,      10.2409,      9.46276,      9.04213,     8.59111,
    8.10749,      7.58892,      7.03288,      6.43665,     5.79734,
    5.11182,      4.71133,      4.45076,      4.17135,     3.87175,
    3.5505,       3.22092,      3.02605,      2.8171,      2.59305,
    2.36129,      2.21295,      2.05388,      1.88332,     1.72944,
    1.56594,      1.41914,      1.30878,      1.18969,     1.0591,
    0.940536,     0.860433,     0.767614,     0.669258,    0.591351,
    0.521943,     0.453835,     0.385841,     0.31329,     0.247096,
    0.196175,     0.127693,     0.0954229,    0.0488275,   0.00294004,
    -0.0493956,   -0.0887183,   -0.158306,    -0.18137,    -0.223017,
    -0.25481,     -0.298629,    -0.333119,    -0.358783,   -0.405019,
    -0.426285,    -0.449094,    -0.469839,    -0.496893,   -0.512751,
    -0.528998,    -0.532973,    -0.558926,    -0.537298,   -0.543398,
    132.365,      123.424,      116.96,       116.182,     115.347,
    114.452,      113.492,      112.463,      111.36,      110.177,
    108.909,      107.548,      106.09,       104.526,     102.849,
    101.051,      99.1228,      97.0555,      94.8387,     92.4617,
    89.9129,      87.1799,      84.2494,      81.1072,     77.7378,
    74.125,       70.2511,      66.0972,      61.6432,     56.8672,
    51.7461,      46.2549,      40.3669,      34.0534,     27.2836,
    20.0246,      12.241,       10.065,       9.63117,     9.16598,
    8.66717,      8.13232,      7.55881,      6.94385,     6.28446,
    5.57741,      5.0723,       4.79492,      4.49749,     4.17857,
    3.83661,      3.46993,      3.24848,      3.02772,     2.791,
    2.53718,      2.35385,      2.16678,      1.9662,      1.82102,
    1.68602,      1.54601,      1.41417,      1.2738,      1.15133,
    1.02744,      0.939362,     0.839529,     0.726383,    0.662329,
    0.581499,     0.493516,     0.424183,     0.380541,    0.287966,
    0.224065,     0.183596,     0.129806,     0.054869,    -0.0028423,
    -0.039876,    -0.0963065,   -0.135793,    -0.193449,   -0.232214,
    -0.258125,    -0.299675,    -0.345506,    -0.381084,   -0.413898,
    -0.441559,    -0.471266,    -0.492597,    -0.503014,   -0.52518,
    -0.544876,    -0.565595,    -0.555907,    -0.567021,   -0.555301,
    141.374,      131.827,      123.571,      122.759,     121.888,
    120.953,      119.952,      118.878,      117.726,     116.491,
    115.167,      113.748,      112.225,      110.593,     108.843,
    106.966,      104.953,      102.795,      100.482,     98.0006,
    95.3403,      92.4877,      89.429,       86.1492,     82.6324,
    78.8614,      74.818,       70.4823,      65.8333,     60.8483,
    55.5031,      49.7715,      43.6258,      37.036,      29.9699,
    22.3932,      14.2689,      10.5989,      10.165,      9.69983,
    9.20102,      8.66616,      8.09265,      7.4777,      6.8183,
    6.11125,      5.51523,      5.22377,      4.91123,     4.57612,
    4.21678,      3.83148,      3.5537,       3.30459,     3.03748,
    2.75107,      2.53785,      2.34444,      2.13706,     1.96797,
    1.82262,      1.66687,      1.50259,      1.32645,     1.21913,
    1.11746,      1.02208,      0.918859,     0.805062,    0.722118,
    0.642901,     0.559418,     0.474234,     0.388306,    0.315683,
    0.281394,     0.220272,     0.142149,     0.0800495,   0.0171771,
    -0.0415836,   -0.0933219,   -0.136449,    -0.171586,   -0.224773,
    -0.264401,    -0.307825,    -0.356092,    -0.376363,   -0.39681,
    -0.451505,    -0.488285,    -0.51739,     -0.529662,   -0.544976,
    -0.547828,    -0.584624,    -0.582544,    -0.609097,   -0.599708,
    156.798,      146.201,      136.318,      134.679,     133.734,
    132.721,      131.635,      130.471,      129.222,     127.883,
    126.448,      124.908,      123.258,      121.488,     119.59,
    117.555,      115.373,      113.033,      110.524,     107.834,
    104.95,       101.857,      98.54,        94.9838,     91.1706,
    87.0818,      82.6975,      77.9964,      72.9556,     67.5505,
    61.7547,      55.5401,      48.8764,      41.7312,     34.0695,
    25.8542,      17.0452,      11.5414,      11.0861,     10.5979,
    10.0744,      9.51301,      8.91109,      8.26567,     7.57361,
    6.83154,      6.1111,       5.76206,      5.3878,      4.98649,
    4.55617,      4.09476,      3.74703,      3.48818,     3.21062,
    2.913,        2.68296,      2.50763,      2.31964,     2.12967,
    1.94157,      1.73987,      1.62452,      1.5166,      1.38832,
    1.245,        1.11408,      0.983926,     0.874034,    0.765358,
    0.675942,     0.63144,      0.499738,     0.442461,    0.371744,
    0.287181,     0.202092,     0.153101,     0.090341,    0.0404608,
    -0.0297941,   -0.0616221,   -0.11463,     -0.165977,   -0.204099,
    -0.289823,    -0.304706,    -0.3468,      -0.382751,   -0.42023,
    -0.46062,     -0.485875,    -0.532361,    -0.541174,   -0.570472,
    -0.572244,    -0.607218,    -0.610689,    -0.606172,   -0.603113,
    167.228,      155.926,      145.386,      142.118,     141.135,
    140.08,       138.949,      137.737,      136.437,     135.042,
    133.548,      131.945,      130.226,      128.383,     126.407,
    124.288,      122.016,      119.58,       116.967,     114.166,
    111.163,      107.942,      104.489,      100.786,     96.8155,
    92.558,       87.993,       83.098,       77.8492,     72.2212,
    66.1864,      59.7155,      52.777,       45.337,      37.3594,
    28.8053,      19.6329,      12.3,         11.8019,     11.2679,
    10.6952,      10.0811,      9.42269,      8.71667,     7.95963,
    7.14788,      6.27746,      5.95123,      5.60946,     5.24299,
    4.85003,      4.42868,      4.06043,      3.79337,     3.50702,
    3.19997,      2.92712,      2.7237,       2.50558,     2.29014,
    2.11152,      1.92,         1.77791,      1.65354,     1.51005,
    1.34671,      1.21187,      1.09116,      0.974024,    0.857099,
    0.746087,     0.663917,     0.563698,     0.495732,    0.425308,
    0.332186,     0.242414,     0.152572,     0.113731,    0.028519,
    0.000823414,  -0.0553366,   -0.115396,    -0.185181,   -0.237037,
    -0.26167,     -0.303183,    -0.358135,    -0.40312,    -0.433435,
    -0.493602,    -0.494585,    -0.53278,     -0.547551,   -0.570478,
    -0.593456,    -0.629171,    -0.619735,    -0.61779,    -0.624385,
    177.325,      165.354,      154.191,      149.143,     148.127,
    147.037,      145.868,      144.615,      143.271,     141.831,
    140.286,      138.629,      136.853,      134.949,     132.906,
    130.717,      128.368,      125.851,      123.151,     120.256,
    117.152,      113.824,      110.255,      106.428,     102.325,
    97.9253,      93.2075,      88.1488,      82.7245,     76.9082,
    70.6716,      63.9843,      56.8137,      49.1249,     40.8804,
    32.0402,      22.5611,      13.3367,      12.8164,     12.2585,
    11.6603,      11.0188,      10.3309,      9.59341,     8.80257,
    7.95458,      7.0453,       6.56756,      6.16737,     5.73826,
    5.27813,      4.78476,      4.32677,      4.07182,     3.79843,
    3.50529,      3.21817,      3.00484,      2.77609,     2.53759,
    2.34426,      2.13696,      1.937,        1.74271,     1.55877,
    1.40669,      1.2769,       1.17661,      1.06894,     0.962091,
    0.867753,     0.712763,     0.621277,     0.544533,    0.443163,
    0.368094,     0.289595,     0.22079,      0.164054,    0.0676842,
    0.0191284,    -0.0547328,   -0.113669,    -0.157722,   -0.216377,
    -0.297841,    -0.324543,    -0.350196,    -0.422619,   -0.453884,
    -0.495012,    -0.509482,    -0.545559,    -0.575297,   -0.609551,
    -0.623233,    -0.634495,    -0.643125,    -0.66124,    -0.657474,
    197.224,      183.89,       171.455,      164.14,      163.03,
    161.84,       160.564,      159.196,      157.729,     156.156,
    154.469,      152.661,      150.722,      148.642,     146.413,
    144.022,      141.458,      138.709,      135.762,     132.601,
    129.212,      125.579,      121.682,      117.504,     113.024,
    108.22,       103.07,       97.5465,      91.6242,     85.274,
    78.4649,      71.1637,      63.3348,      54.9402,     45.939,
    36.2872,      25.9379,      14.8407,      13.5168,     12.9221,
    12.2844,      11.6006,      10.8674,      10.0813,     9.23829,
    8.33439,      7.36517,      6.81225,      6.48223,     6.12837,
    5.74893,      5.34207,      4.9159,       4.591,       4.24262,
    3.86906,      3.48111,      3.2518,       3.00592,     2.74226,
    2.52454,      2.29561,      2.08705,      1.92841,     1.76496,
    1.61822,      1.46464,      1.30883,      1.16387,     1.02843,
    0.926398,     0.786615,     0.679007,     0.601032,    0.498903,
    0.394778,     0.321827,     0.232939,     0.18405,     0.092568,
    0.042439,     -0.0416356,   -0.0913276,   -0.12991,    -0.191093,
    -0.264904,    -0.315107,    -0.359847,    -0.43129,    -0.474699,
    -0.505659,    -0.510218,    -0.57378,     -0.585427,   -0.626666,
    -0.631008,    -0.64838,     -0.678672,    -0.692037,   -0.690541,
    211.567,      197.271,      183.939,      174.285,     173.123,
    171.876,      170.539,      169.106,      167.569,     165.921,
    164.154,      162.26,       160.228,      158.05,      155.714,
    153.209,      150.523,      147.643,      144.556,     141.244,
    137.694,      133.887,      129.805,      125.428,     120.735,
    115.702,      110.306,      104.52,       98.3156,     91.6629,
    84.5294,      76.8804,      68.6787,      59.8842,     50.4542,
    40.3427,      29.5005,      17.8747,      14.6058,     13.9834,
    13.316,       12.6004,      11.833,       11.0102,     10.128,
    9.18195,      8.16756,      7.44239,      7.04816,     6.62543,
    6.17216,      5.68613,      5.16498,      4.86612,     4.57193,
    4.25648,      3.91823,      3.61806,      3.30313,     2.96545,
    2.71575,      2.48186,      2.25028,      2.0777,      1.89394,
    1.73695,      1.56835,      1.38596,      1.23009,     1.10841,
    1.02833,      0.906952,     0.7693,       0.654138,    0.554732,
    0.446169,     0.367489,     0.304272,     0.200615,    0.135785,
    0.0557083,    -0.0124084,   -0.0560894,   -0.137581,   -0.202957,
    -0.245629,    -0.323268,    -0.373278,    -0.427667,   -0.455325,
    -0.50103,     -0.551428,    -0.589866,    -0.603726,   -0.637754,
    -0.67882,     -0.693156,    -0.667624,    -0.698447,   -0.696653,
    224.919,      209.746,      195.594,      183.439,     182.235,
    180.945,      179.561,      178.078,      176.487,     174.781,
    172.952,      170.991,      168.888,      166.633,     164.215,
    161.622,      158.842,      155.861,      152.664,     149.237,
    145.562,      141.621,      137.396,      132.865,     128.006,
    122.797,      117.211,      111.222,      104.799,     97.9127,
    90.5285,      82.6107,      74.1207,      65.0171,     55.2557,
    44.7888,      33.5655,      21.5311,      16.1915,     15.5344,
    14.8299,      14.0744,      13.2643,      12.3957,     11.4643,
    10.4656,      9.39474,      8.48205,      7.98919,     7.4607,
    6.89403,      6.2864,       5.63487,      5.2252,      4.88533,
    4.52089,      4.13011,      3.82683,      3.54315,     3.23896,
    2.97424,      2.73036,      2.4722,       2.26457,     2.04195,
    1.86995,      1.69569,      1.53651,      1.37609,     1.22612,
    1.09151,      0.959605,     0.836461,     0.739756,    0.623771,
    0.543363,     0.405933,     0.318475,     0.261764,    0.176034,
    0.0821896,    0.00864913,   -0.0477664,   -0.148941,   -0.202835,
    -0.242726,    -0.314784,    -0.356532,    -0.43666,    -0.485744,
    -0.541629,    -0.543835,    -0.599237,    -0.632124,   -0.650527,
    -0.667384,    -0.703429,    -0.71219,     -0.751535,   -0.718746,
    243.415,      226.989,      211.67,       197.384,     195.239,
    193.874,      192.41,       190.841,      189.158,     187.353,
    185.419,      183.344,      181.119,      178.734,     176.176,
    173.434,      170.493,      167.34,       163.959,     160.333,
    156.446,      152.277,      147.807,      143.015,     137.876,
    132.365,      126.457,      120.121,      113.327,     106.043,
    98.2321,      89.8567,      80.8761,      71.2465,     60.9209,
    49.8492,      37.9773,      25.2475,      17.5272,     16.7985,
    16.0172,      15.1794,      14.281,       13.3177,     12.2848,
    11.1773,      9.98973,      8.87013,      8.4171,      7.93133,
    7.41046,      6.85194,      6.25306,      5.7501,      5.31013,
    4.83836,      4.3325,       3.99414,      3.78181,     3.55414,
    3.27718,      2.93957,      2.57756,      2.40003,     2.2369,
    2.03564,      1.80887,      1.66087,      1.51855,     1.34006,
    1.20045,      1.07054,      0.918653,     0.825316,    0.69367,
    0.544283,     0.510726,     0.388381,     0.291087,    0.187287,
    0.132851,     0.0253187,    -0.0614555,   -0.123219,   -0.194456,
    -0.242043,    -0.325099,    -0.377375,    -0.421982,   -0.488553,
    -0.519395,    -0.545233,    -0.607245,    -0.638677,   -0.667687,
    -0.68681,     -0.708863,    -0.739722,    -0.745157,   -0.764981,
    261.367,      243.714,      227.251,      211.898,     207.566,
    206.125,      204.58,       202.923,      201.146,     199.241,
    197.198,      195.007,      192.658,      190.14,      187.439,
    184.543,      181.438,      178.109,      174.539,     170.71,
    166.606,      162.204,      157.485,      152.424,     146.998,
    141.18,       134.941,      128.251,      121.078,     113.387,
    105.139,      96.296,       86.8136,      76.6458,     65.7433,
    54.0529,      41.5177,      28.0765,      17.7793,     17.0887,
    16.3481,      15.5541,      14.7026,      13.7897,     12.8107,
    11.761,       10.6354,      9.44294,      8.94209,     8.40504,
    7.82918,      7.21171,      6.54961,      6.01256,     5.66913,
    5.30087,      4.906,        4.52893,      4.18895,     3.8244,
    3.47585,      3.20509,      2.91477,      2.68749,     2.47562,
    2.25151,      2.01376,      1.79534,      1.57997,     1.44613,
    1.31345,      1.18237,      1.04337,      0.863712,    0.776157,
    0.65726,      0.535468,     0.401517,     0.34033,     0.231018,
    0.164464,     0.0523128,    -0.0174953,   -0.083354,   -0.169789,
    -0.241531,    -0.304134,    -0.388314,    -0.424005,   -0.474036,
    -0.53248,     -0.588346,    -0.609582,    -0.641071,   -0.682466,
    -0.717956,    -0.72668,     -0.748205,    -0.755966,   -0.775453,
    298.0,        277.834,      259.027,      241.487,     234.252,
    232.637,      230.904,      229.047,      227.056,     224.92,
    222.63,       220.175,      217.542,      214.719,     211.692,
    208.447,      204.967,      201.235,      197.233,     192.943,
    188.342,      183.409,      178.119,      172.447,     166.365,
    159.844,      152.851,      145.353,      137.313,     128.692,
    119.449,      109.537,      98.9085,      87.5122,     75.2923,
    62.1894,      48.1395,      33.0743,      19.0974,     18.3643,
    17.5781,      16.7352,      15.8313,      14.8621,     13.8229,
    12.7085,      11.5137,      10.2325,      9.62462,     9.098,
    8.53332,      7.92783,      7.27858,      6.66067,     6.2065,
    5.7195,       5.19731,      4.71703,      4.42277,     4.10724,
    3.77851,      3.48537,      3.17105,      2.91069,     2.68972,
    2.45794,      2.21738,      1.97522,      1.73112,     1.55614,
    1.38816,      1.21384,      1.1065,       0.963685,    0.831145,
    0.740389,     0.614606,     0.483581,     0.372442,    0.271114,
    0.208498,     0.116471,     -0.00397982,  -0.0657164,  -0.159617,
    -0.253311,    -0.294715,    -0.365207,    -0.407023,   -0.489507,
    -0.530639,    -0.586515,    -0.631395,    -0.670949,   -0.68975,
    -0.728799,    -0.753333,    -0.76466,     -0.791958,   -0.811038,
    312.257,      291.157,      271.48,       253.129,     243.147,
    241.497,      239.728,      237.831,      235.796,     233.615,
    231.276,      228.768,      226.079,      223.195,     220.104,
    216.788,      213.233,      209.422,      205.334,     200.952,
    196.252,      191.213,      185.81,       180.016,     173.804,
    167.143,      160.0,        152.341,      144.129,     135.323,
    125.881,      115.756,      104.9,        93.2593,     80.7773,
    67.3932,      53.0419,      37.6535,      21.153,      20.0926,
    19.2503,      18.3472,      17.3788,      16.3404,     15.227,
    14.0331,      12.753,       11.3803,      10.5568,     9.94754,
    9.29428,      8.59382,      7.84273,      7.08742,     6.6352,
    6.1503,       5.63035,      5.10278,      4.76746,     4.40791,
    4.02238,      3.7627,       3.48585,      3.19717,     2.89913,
    2.59593,      2.32234,      2.07281,      1.88632,     1.72204,
    1.55858,      1.36685,      1.24152,      1.09496,     0.943669,
    0.804217,     0.666343,     0.547447,     0.426976,    0.31988,
    0.235155,     0.0861774,    0.0508257,    -0.0398366,  -0.123037,
    -0.22167,     -0.313053,    -0.360054,    -0.438547,   -0.51038,
    -0.552769,    -0.62848,     -0.648159,    -0.681361,   -0.733864,
    -0.749793,    -0.784318,    -0.801969,    -0.790312,   -0.819802,
    336.683,      313.92,       292.691,      272.892,     259.634,
    257.888,      256.015,      254.008,      251.855,     249.547,
    247.072,      244.418,      241.573,      238.522,     235.25,
    231.742,      227.98,       223.946,      219.621,     214.984,
    210.011,      204.679,      198.961,      192.83,      186.257,
    179.208,      171.65,       163.545,      154.855,     145.537,
    135.546,      124.832,      113.344,      101.026,     87.8182,
    73.6555,      58.4693,      42.1857,      24.7253,     20.9871,
    20.1252,      19.2011,      18.2102,      17.1477,     16.0085,
    14.7869,      13.477,       12.0724,      11.1119,     10.5285,
    9.90298,      9.23226,      8.51307,      7.7419,      7.22313,
    6.67513,      6.08752,      5.45745,      5.09778,     4.72333,
    4.32182,      3.97782,      3.62623,      3.30091,     3.09548,
    2.87331,      2.6166,       2.36154,      2.16815,     1.9462,
    1.69935,      1.52589,      1.35256,      1.17269,     1.0081,
    0.899086,     0.782499,     0.620693,     0.504206,    0.365245,
    0.262518,     0.16111,      0.0649686,    -0.0168988,  -0.111966,
    -0.222012,    -0.316578,    -0.346792,    -0.430172,   -0.473177,
    -0.575203,    -0.60572,     -0.627164,    -0.707933,   -0.73463,
    -0.775391,    -0.788002,    -0.829206,    -0.827826,   -0.827924,
    350.314,      326.681,      304.641,      284.087,     267.692,
    265.924,      264.03,       261.998,      259.819,     257.483,
    254.978,      252.292,      249.412,      246.324,     243.013,
    239.462,      235.655,      231.572,      227.195,     222.501,
    217.468,      212.072,      206.285,      200.08,      193.427,
    186.293,      178.643,      170.441,      161.645,     152.215,
    142.102,      131.259,      119.632,      107.165,     93.7972,
    79.4632,      64.0933,      47.6126,      29.9409,     23.4348,
    22.4883,      21.4735,      20.3853,      19.2185,     17.9673,
    16.6257,      15.1872,      13.6447,      12.4027,     11.689,
    10.9238,      10.1032,      9.22341,      8.27999,     7.68098,
    7.12951,      6.5382,       5.90415,      5.50804,     5.14924,
    4.76451,      4.4209,       4.08479,      3.73055,     3.39669,
    3.03871,      2.8353,       2.62322,      2.3154,      2.05341,
    1.88345,      1.68704,      1.48845,      1.28943,     1.09878,
    0.982067,     0.803641,     0.678687,     0.556944,    0.419249,
    0.323755,     0.17244,      0.116355,     -0.00433266, -0.0916754,
    -0.166174,    -0.251955,    -0.358849,    -0.424635,   -0.491359,
    -0.581052,    -0.625466,    -0.694185,    -0.699777,   -0.781604,
    -0.798698,    -0.799484,    -0.855754,    -0.865354,   -0.860803,
    400.781,      373.679,      348.405,      324.833,     303.191,
    301.203,      299.07,       296.784,      294.332,     291.704,
    288.885,      285.862,      282.621,      279.146,     275.42,
    271.424,      267.14,       262.546,      257.62,      252.338,
    246.675,      240.602,      234.09,       227.107,     219.621,
    211.592,      202.984,      193.754,      183.857,     173.244,
    161.864,      149.663,      136.579,      122.55,      107.507,
    91.3764,      74.0805,      55.5347,      35.6486,     25.3217,
    24.2395,      23.079,       21.8347,      20.5005,     19.0698,
    17.5358,      15.8908,      14.1271,      12.5845,     11.9599,
    11.2902,      10.572,       9.80195,      8.97624,     8.33279,
    7.76287,      7.15177,      6.4965,       5.9921,      5.55322,
    5.08262,      4.65465,      4.26349,      3.84405,     3.5364,
    3.21059,      2.96945,      2.73939,      2.45199,     2.18039,
    2.0231,       1.80267,      1.58233,      1.41028,     1.27596,
    1.06107,      0.906623,     0.790611,     0.598979,    0.441822,
    0.394854,     0.265318,     0.108043,     0.0482024,   -0.0448717,
    -0.12677,     -0.244935,    -0.323741,    -0.436265,   -0.506973,
    -0.590404,    -0.618785,    -0.669405,    -0.693599,   -0.776126,
    -0.806896,    -0.850008,    -0.864371,    -0.896804,   -0.89102,
    416.383,      388.276,      362.064,      337.618,     314.821,
    310.194,      308.029,      305.708,      303.219,     300.55,
    297.689,      294.62,       291.33,       287.802,     284.019,
    279.963,      275.614,      270.95,       265.949,     260.587,
    254.837,      248.672,      242.061,      234.973,     227.372,
    219.222,      210.483,      201.113,      191.065,     180.291,
    168.739,      156.352,      143.069,      128.827,     113.555,
    97.1802,      79.6215,      60.794,       40.6059,     27.0903,
    26.019,       24.8703,      23.6386,      22.3179,     20.9018,
    19.3833,      17.7551,      16.0092,      14.272,      13.5092,
    12.6912,      11.8141,      10.8736,      9.86516,     9.05547,
    8.4377,       7.77527,      7.06498,      6.46245,     5.96781,
    5.43743,      4.96584,      4.61587,      4.24061,     3.93101,
    3.61929,      3.2874,       2.93285,      2.62785,     2.34054,
    2.18786,      1.9606,       1.69985,      1.50718,     1.38237,
    1.16207,      1.01651,      0.864596,     0.718282,    0.571988,
    0.440014,     0.226976,     0.184597,     0.0979681,   -0.0459877,
    -0.145941,    -0.266688,    -0.306489,    -0.416687,   -0.483391,
    -0.539812,    -0.617587,    -0.678915,    -0.737379,   -0.748843,
    -0.848903,    -0.844986,    -0.888055,    -0.905352,   -0.918859,
    431.464,      402.354,      375.206,      349.887,     326.274,
    318.532,      316.337,      313.983,      311.459,     308.752,
    305.85,       302.738,      299.401,      295.823,     291.987,
    287.873,      283.462,      278.733,      273.661,     268.223,
    262.392,      256.14,       249.435,      242.247,     234.538,
    226.273,      217.41,       207.907,      197.717,     186.791,
    175.075,      162.512,      149.042,      134.598,     119.11,
    102.503,      84.6962,      65.6022,      45.1283,     28.3276,
    27.2107,      26.0131,      24.7289,      23.3519,     21.8754,
    20.2922,      18.5946,      16.7743,      14.8224,     14.0288,
    13.2234,      12.3599,      11.434,       10.4411,     9.56299,
    8.93181,      8.25502,      7.52931,      6.88288,     6.41656,
    5.91654,      5.4013,       4.91487,      4.3933,      3.99763,
    3.65291,      3.36996,      3.15471,      2.91294,     2.64663,
    2.36379,      2.10534,      1.88855,      1.69775,     1.49095,
    1.2793,       1.08487,      0.979471,     0.757551,    0.596805,
    0.5154,       0.359635,     0.24531,      0.101782,    -0.0416215,
    -0.138882,    -0.204754,    -0.298572,    -0.347233,   -0.483005,
    -0.563743,    -0.631415,    -0.692998,    -0.786492,   -0.802464,
    -0.841968,    -0.886387,    -0.895176,    -0.930325,   -0.965425,
    472.149,      440.274,      410.547,      382.824,     356.969,
    345.33,       342.974,      340.448,      337.739,     334.835,
    331.72,       328.381,      324.8,        320.961,     316.844,
    312.43,       307.696,      302.621,      297.178,     291.343,
    285.085,      278.376,      271.181,      263.467,     255.195,
    246.326,      236.815,      226.617,      215.682,     203.957,
    191.385,      177.904,      163.449,      147.949,     131.329,
    113.508,      94.3988,      73.9089,      51.9382,     30.6027,
    29.3952,      28.1005,      26.7122,      25.2235,     23.6273,
    21.9157,      20.0805,      18.1126,      16.0024,     14.9619,
    14.1179,      13.213,       12.2426,      11.2022,     10.2052,
    9.52619,      8.79814,      8.01749,      7.27623,     6.81076,
    6.31165,      5.79083,      5.36189,      4.90196,     4.45102,
    4.00528,      3.59881,      3.2943,       3.01834,     2.78104,
    2.49797,      2.24233,      2.09599,      1.7907,      1.56458,
    1.36826,      1.17891,      1.00347,      0.854032,    0.701922,
    0.558319,     0.420842,     0.309442,     0.129969,    0.0552626,
    -0.0887283,   -0.179424,    -0.275961,    -0.383016,   -0.478619,
    -0.583981,    -0.617887,    -0.707618,    -0.760074,   -0.808734,
    -0.901656,    -0.91035,     -0.951685,    -0.957461,   -0.96333,
    505.349,      471.226,      439.403,      409.725,     382.047,
    366.315,      363.834,      361.175,      358.322,     355.264,
    351.985,      348.468,      344.698,      340.655,     336.32,
    331.671,      326.687,      321.343,      315.612,     309.467,
    302.878,      295.813,      288.237,      280.114,     271.404,
    262.064,      252.049,      241.311,      229.796,     217.45,
    204.211,      190.015,      174.794,      158.473,     140.972,
    122.206,      102.084,      80.5083,      57.3731,     32.566,
    30.2839,      29.0672,      27.7625,      26.3635,     24.8634,
    23.255,       21.5302,      19.6809,      17.6979,     16.4206,
    15.4638,      14.4378,      13.3377,      12.158,      10.9563,
    10.2774,      9.54944,      8.76886,      7.96209,     7.41747,
    6.83349,      6.20731,      5.73058,      5.22771,     4.77421,
    4.4245,       4.05242,      3.66419,      3.31449,     3.07997,
    2.72909,      2.3496,       2.20028,      1.98492,     1.74508,
    1.49825,      1.31429,      1.16369,      0.941327,    0.785396,
    0.646359,     0.478869,     0.319861,     0.192037,    0.0502333,
    -0.063124,    -0.137374,    -0.301138,    -0.366059,   -0.468645,
    -0.548088,    -0.655039,    -0.743145,    -0.789506,   -0.852007,
    -0.871112,    -0.906349,    -0.94031,     -0.966788,   -1.01112,
    527.426,      491.846,      458.665,      427.72,      398.861,
    378.942,      376.422,      373.719,      370.821,     367.713,
    364.381,      360.809,      356.977,      352.87,      348.465,
    343.742,      338.677,      333.247,      327.424,     321.18,
    314.485,      307.306,      299.609,      291.355,     282.505,
    273.015,      262.839,      251.928,      240.229,     227.684,
    214.232,      199.808,      184.342,      167.758,     149.976,
    130.909,      110.463,      88.5405,      65.0334,     39.8275,
    33.7687,      32.2893,      30.7031,      29.0022,     27.1784,
    25.2228,      23.1258,      20.8774,      18.4664,     16.8208,
    15.887,       14.8857,      13.8121,      12.6608,     11.4264,
    10.6838,      9.91548,      9.0916,       8.20817,     7.67917,
    7.13579,      6.55314,      6.10049,      5.65446,     5.20893,
    4.83142,      4.42859,      4.02094,      3.61542,     3.32169,
    2.96943,      2.56123,      2.33461,      2.09312,     1.8328,
    1.59175,      1.42892,      1.24904,      1.0646,      0.87251,
    0.650749,     0.542571,     0.365202,     0.276985,    0.092913,
    -0.050903,    -0.131373,    -0.269771,    -0.383237,   -0.464127,
    -0.553987,    -0.655627,    -0.677612,    -0.746844,   -0.849168,
    -0.87547,     -0.93195,     -0.967452,    -1.01385,    -1.02571,
    575.372,      536.539,      500.324,      466.55,      435.052,
    409.704,      407.0,        404.1,        400.992,     397.658,
    394.084,      390.251,      386.142,      381.735,     377.01,
    371.944,      366.511,      360.686,      354.44,      347.742,
    340.561,      332.86,       324.603,      315.749,     306.255,
    296.076,      285.16,       273.456,      260.906,     247.449,
    233.02,       217.547,      200.957,      183.167,     164.092,
    143.639,      121.707,      98.1909,      72.975,      45.9367,
    35.719,       34.2285,      32.6303,      30.9166,     29.0791,
    27.1087,      24.996,       22.7306,      20.3015,     18.3647,
    17.3269,      16.214,       15.0207,      13.7411,     12.3691,
    11.5102,      10.7322,      9.89794,      9.00342,     8.35238,
    7.73008,      7.06281,      6.53478,      6.06015,     5.55654,
    5.05981,      4.52719,      4.2035,       3.87467,     3.54374,
    3.19998,      2.85015,      2.5671,       2.26401,     1.97934,
    1.79136,      1.53842,      1.26764,      1.14153,     0.974564,
    0.820456,     0.59784,      0.427525,     0.338095,    0.165099,
    -0.0128484,   -0.13373,     -0.235488,    -0.383632,   -0.459498,
    -0.553552,    -0.637526,    -0.736787,    -0.797574,   -0.859525,
    -0.923253,    -0.965289,    -1.02783,     -1.02604,    -1.05362,
    633.718,      590.891,      550.949,      513.7,       478.961,
    447.151,      444.212,      441.061,      437.683,     434.06,
    430.175,      426.01,       421.543,      416.754,     411.619,
    406.112,      400.208,      393.877,      387.088,     379.809,
    372.003,      363.634,      354.66,       345.037,     334.719,
    323.655,      311.792,      299.071,      285.431,     270.806,
    255.123,      238.307,      220.276,      200.942,     180.21,
    157.98,       134.144,      108.585,      81.1796,     51.7932,
    36.6653,      35.2347,      33.7007,      32.0559,     30.2922,
    28.401,       26.3732,      24.1988,      21.8673,     19.768,
    18.6902,      17.5346,      16.2954,      14.9666,     13.5419,
    12.5057,      11.6369,      10.7053,      9.70628,     8.92266,
    8.22926,      7.48575,      6.91303,      6.49609,     6.04902,
    5.57128,      5.05907,      4.57212,      4.06624,     3.71896,
    3.38661,      3.09546,      2.8193,       2.4773,      2.06605,
    1.89535,      1.73414,      1.43841,      1.26541,     1.04624,
    0.886821,     0.688265,     0.543362,     0.367118,    0.190069,
    0.0571249,    -0.0931141,   -0.182374,    -0.32116,    -0.412865,
    -0.584425,    -0.675581,    -0.731608,    -0.824416,   -0.873234,
    -0.949681,    -0.991994,    -1.0171,      -1.02561,    -1.08973,
    648.877,      605.084,      564.243,      526.154,     490.633,
    457.505,      451.117,      447.967,      444.59,      440.969,
    437.086,      432.922,      428.458,      423.671,     418.538,
    413.033,      407.132,      400.803,      394.018,     386.742,
    378.94,       370.574,      361.604,      351.986,     341.672,
    330.613,      318.755,      306.04,       292.407,     277.787,
    262.112,      245.303,      227.28,       207.954,     187.232,
    165.012,      141.187,      115.639,      88.2458,     58.8725,
    39.5391,      37.928,       36.2005,      34.3481,     32.3619,
    30.2322,      27.9485,      25.4998,      22.8742,     20.2938,
    19.2425,      18.1153,      16.9066,      15.6105,     14.2208,
    13.1586,      12.3963,      11.5789,      10.7025,     9.8728,
    9.08311,      8.23636,      7.43998,      6.75643,     6.02349,
    5.63707,      5.30292,      4.90201,      4.44955,     4.06295,
    3.68227,      3.30864,      2.93629,      2.58727,     2.30851,
    2.05978,      1.88956,      1.67627,      1.39389,     1.20131,
    0.892148,     0.765426,     0.577221,     0.399047,    0.251366,
    0.164718,     -0.076082,    -0.151745,    -0.313259,   -0.430524,
    -0.535512,    -0.604599,    -0.736706,    -0.792742,   -0.887799,
    -0.944454,    -0.995595,    -1.08282,     -1.09922,    -1.12167,
    702.326,      654.888,      610.648,      569.389,     530.911,
    495.027,      483.921,      480.576,      476.99,      473.144,
    469.02,       464.599,      459.857,      454.774,     449.323,
    443.477,      437.21,       430.49,       423.283,     415.557,
    407.271,      398.387,      388.861,      378.647,     367.694,
    355.95,       343.357,      329.854,      315.375,     299.85,
    283.203,      265.353,      246.213,      225.69,      203.683,
    180.087,      154.785,      127.654,      98.5628,     67.3692,
    42.4462,      40.6613,      38.7474,      36.6952,     34.4947,
    32.1351,      29.605,       26.8921,      23.9832,     20.864,
    19.8298,      18.7486,      17.5893,      16.3461,     15.0132,
    13.8417,      12.9781,      12.052,       11.0589,     10.1532,
    9.43229,      8.65926,      7.90906,      7.3268,      6.70247,
    6.15211,      5.61437,      5.13952,      4.7253,      4.34174,
    3.96618,      3.59815,      3.21612,      2.84246,     2.61337,
    2.29271,      1.94528,      1.72684,      1.45795,     1.26176,
    1.07413,      0.807564,     0.602237,     0.466941,    0.333863,
    0.130792,     0.00773079,   -0.148716,    -0.281811,   -0.421573,
    -0.491114,    -0.642122,    -0.67971,     -0.80993,    -0.879063,
    -0.971011,    -1.00549,     -1.09392,     -1.09576,    -1.16025,
    748.947,      698.351,      651.166,      607.16,      566.12,
    527.847,      511.691,      508.178,      504.411,     500.373,
    496.042,      491.399,      486.419,      481.08,      475.356,
    469.217,      462.635,      455.577,      448.009,     439.895,
    431.193,      421.863,      411.859,      401.132,     389.629,
    377.296,      364.071,      349.89,       334.684,     318.38,
    300.897,      282.151,      262.05,       240.497,     217.386,
    192.604,      166.032,      137.54,       106.988,     74.2286,
    43.4984,      41.8406,      40.063,       38.157,      36.1133,
    33.9218,      31.572,       29.0523,      26.3506,     23.4536,
    21.9633,      20.6511,      19.244,       17.7353,     16.1176,
    14.6389,      13.7687,      12.8357,      11.8352,     10.8647,
    10.1217,      9.32488,      8.50647,      7.86968,     7.18688,
    6.6049,       6.09939,      5.56976,      5.02186,     4.54454,
    4.1456,       3.79828,      3.44741,      3.08122,     2.79455,
    2.43361,      2.19144,      1.95994,      1.57826,     1.35491,
    1.15051,      0.958454,     0.72686,      0.538759,    0.375917,
    0.212198,     0.0567473,    -0.0924944,   -0.231917,   -0.371185,
    -0.47527,     -0.628588,    -0.74038,     -0.792892,   -0.882252,
    -0.973617,    -1.05883,     -1.11332,     -1.14934,    -1.1649,
    759.087,      707.899,      660.159,      615.638,     574.117,
    535.395,      514.591,      511.116,      507.39,      503.395,
    499.111,      494.518,      489.593,      484.311,     478.648,
    472.576,      466.065,      459.083,      451.597,     443.57,
    434.963,      425.733,      415.837,      405.226,     393.847,
    381.647,      368.564,      354.537,      339.495,     323.367,
    306.073,      287.529,      267.645,      246.324,     223.462,
    198.948,      172.663,      144.478,      114.256,     81.8504,
    47.1027,      45.0915,      43.1607,      41.0904,     38.8705,
    36.4901,      33.9377,      31.2009,      28.2662,     25.1195,
    23.2978,      21.9666,      20.5392,      19.0086,     17.3674,
    15.7232,      14.7125,      13.6288,      12.4667,     11.2925,
    10.5344,      9.7215,       8.85007,      8.23005,     7.56521,
    6.91849,      6.31429,      5.73565,      5.32097,     4.89293,
    4.46331,      4.01886,      3.57479,      3.19937,     2.95176,
    2.5978,       2.2456,       2.00811,      1.7551,      1.46706,
    1.28364,      1.00202,      0.844192,     0.623407,    0.417773,
    0.26535,      0.0909644,    -0.105152,    -0.251247,   -0.396744,
    -0.528145,    -0.640574,    -0.71791,     -0.820322,   -0.962578,
    -0.994033,    -1.05754,     -1.11673,     -1.14814,    -1.16568,
    798.77,       744.905,      694.669,      647.819,     604.126,
    563.378,      536.941,      533.357,      529.514,     525.394,
    520.976,      516.239,      511.159,      505.712,     499.872,
    493.609,      486.894,      479.694,      471.973,     463.694,
    454.817,      445.299,      435.092,      424.149,     412.414,
    399.831,      386.339,      371.872,      356.359,     339.725,
    321.889,      302.764,      282.258,      260.269,     236.691,
    211.409,      184.3,        155.232,      124.063,     90.6422,
    54.8058,      47.961,       45.8603,      43.6077,     41.1924,
    38.6025,      35.8254,      32.8477,      29.6548,     26.2311,
    24.019,       22.7011,      21.2879,      19.7725,     18.1477,
    16.4096,      15.3389,      14.1909,      12.9599,     11.6399,
    10.8383,      9.98037,      9.06048,      8.45684,     7.8694,
    7.30114,      6.83909,      6.3413,       5.79067,     5.23488,
    4.75395,      4.283,        3.81616,      3.35528,     3.09403,
    2.83628,      2.51276,      2.22571,      1.92824,     1.64918,
    1.40809,      1.07734,      0.923628,     0.66877,     0.511792,
    0.31577,      0.148233,     -0.0182356,   -0.204223,   -0.343008,
    -0.431183,    -0.610662,    -0.706048,    -0.793148,   -0.950916,
    -1.00923,     -1.05982,     -1.12855,     -1.16814,    -1.25101,
    902.167,      841.209,      784.358,      731.34,      681.894,
    635.78,       601.005,      597.001,      592.707,     588.103,
    583.166,      577.872,      572.196,      566.109,     559.583,
    552.585,      545.081,      537.035,      528.408,     519.157,
    509.237,      498.601,      487.196,      474.967,     461.854,
    447.794,      432.717,      416.551,      399.216,     380.629,
    360.699,      339.328,      316.413,      291.841,     265.495,
    237.244,      206.951,      174.469,      139.64,      102.294,
    62.2494,      49.6228,      47.6235,      45.4796,     43.1808,
    40.7159,      38.0729,      35.2389,      32.2,        28.9416,
    26.3934,      24.8186,      23.1299,      21.3192,     19.3776,
    17.2957,      16.1413,      15.0633,      13.9075,     12.6681,
    11.7876,      10.9148,      9.97887,      9.3319,      8.76913,
    8.18487,      7.65416,      7.08511,      6.32171,     5.53717,
    4.99612,      4.47501,      3.99453,      3.76024,     3.407,
    3.01406,      2.70151,      2.30083,      2.02697,     1.75037,
    1.52468,      1.21895,      0.96964,      0.782695,    0.541378,
    0.339147,     0.138475,     0.0136091,    -0.182917,   -0.324201,
    -0.448095,    -0.577115,    -0.703869,    -0.886045,   -0.932827,
    -1.03936,     -1.08532,     -1.17682,     -1.18811,    -1.24751,
    936.366,      873.118,      814.133,      759.122,     707.821,
    659.975,      618.741,      614.667,      610.298,     605.614,
    600.591,      595.205,      589.429,      583.237,     576.596,
    569.476,      561.842,      553.655,      544.878,     535.465,
    525.373,      514.551,      502.947,      490.504,     477.163,
    462.857,      447.517,      431.069,      413.432,     394.521,
    374.242,      352.499,      329.184,      304.184,     277.377,
    248.634,      217.813,      184.764,      149.328,     111.33,
    70.5864,      52.3077,      50.1249,      47.7842,     45.2745,
    42.5833,      39.6977,      36.6035,      33.2857,     29.7282,
    26.7566,      25.352,       23.846,       22.2312,     20.4996,
    18.643,       17.2925,      16.0713,      14.7619,     13.3578,
    12.3504,      11.454,       10.4929,      9.68626,     8.97035,
    8.21518,      7.69904,      7.14561,      6.52172,     5.84787,
    5.39098,      4.90551,      4.38064,      3.94896,     3.57949,
    3.25798,      2.91401,      2.58524,      2.17412,     1.87985,
    1.58703,      1.30767,      1.0702,       0.859695,    0.571578,
    0.482485,     0.268797,     0.0812029,    -0.190285,   -0.255125,
    -0.436359,    -0.585869,    -0.691556,    -0.871161,   -0.926282,
    -1.03364,     -1.09596,     -1.13958,     -1.20407,    -1.26573,
    960.276,      895.518,      835.123,      778.8,       726.272,
    677.284,      631.598,      625.748,      621.363,     616.66,
    611.618,      606.211,      600.413,      594.197,     587.531,
    580.383,      572.719,      564.501,      555.69,      546.241,
    536.109,      525.246,      513.597,      501.107,     487.713,
    473.352,      457.953,      441.442,      423.737,     404.752,
    384.396,      362.568,      339.163,      314.067,     287.157,
    258.302,      227.362,      194.186,      158.613,     120.469,
    79.5677,      55.8422,      53.5927,      51.1807,     48.5944,
    45.8211,      42.8475,      39.6589,      36.2399,     32.5739,
    29.1534,      27.594,       25.9219,      24.1291,     22.2066,
    20.1452,      18.6008,      17.3731,      16.0567,     14.6452,
    13.4723,      12.4412,      11.3355,      10.338,      9.47906,
    8.55803,      7.94472,      7.32363,      6.80117,     6.29374,
    5.7856,       5.24417,      4.64864,      4.25777,     3.88507,
    3.36771,      3.09473,      2.66705,      2.43668,     2.05536,
    1.67463,      1.47104,      1.13747,      1.02565,     0.755722,
    0.523796,     0.283503,     0.0608236,    -0.0950761,  -0.254265,
    -0.456312,    -0.583264,    -0.685433,    -0.791212,   -0.975064,
    -1.08851,     -1.14216,     -1.1896,      -1.23448,    -1.30299,
    1008.34,      940.33,       876.901,      817.747,     762.581,
    711.132,      663.15,       651.715,      647.196,     642.351,
    637.155,      631.584,      625.611,      619.206,     612.337,
    604.973,      597.076,      588.609,      579.53,      569.794,
    559.355,      548.162,      536.16,       523.29,      509.491,
    494.694,      478.827,      461.814,      443.572,     424.011,
    403.037,      380.547,      356.432,      330.574,     302.847,
    273.117,      241.238,      207.055,      170.402,     131.1,
    88.9575,      58.9736,      56.5559,      53.9635,     51.1838,
    48.2031,      45.0071,      41.5801,      37.9054,     33.9652,
    29.9516,      28.4147,      26.7668,      24.9998,     23.1051,
    21.0735,      19.3459,      17.9801,      16.5156,     14.9453,
    13.6985,      12.8486,      11.9372,      11.0713,     10.3539,
    9.58469,      8.77432,      7.90943,      7.23715,     6.68467,
    6.155,        5.61441,      5.07818,      4.53521,     4.04372,
    3.77449,      3.27016,      2.93805,      2.53731,     2.16749,
    1.84103,      1.56607,      1.27904,      1.12688,     0.775638,
    0.586237,     0.345437,     0.12549,      -0.0108953,  -0.240479,
    -0.459704,    -0.558611,    -0.70663,     -0.895813,   -0.962807,
    -1.00034,     -1.13815,     -1.17483,     -1.2831,     -1.34722,
    1138.76,      1061.84,      990.106,      923.203,     860.809,
    802.621,      748.355,      729.731,      724.687,     719.279,
    713.48,       707.262,      700.595,      693.446,     685.78,
    677.56,       668.746,      659.295,      649.162,     638.296,
    626.644,      614.151,      600.755,      586.39,      570.988,
    554.472,      536.763,      517.775,      497.414,     475.581,
    452.171,      427.069,      400.153,      371.291,     340.344,
    307.161,      271.579,      233.427,      192.516,     148.65,
    101.613,      62.0733,      59.6494,      57.0503,     54.2634,
    51.275,       48.0708,      44.6349,      40.9507,     37.0004,
    32.7645,      30.9283,      29.1155,      27.1716,     25.0872,
    22.8522,      20.8876,      19.5639,      18.1445,     16.6225,
    15.2065,      14.0935,      12.9001,      11.7469,     10.9561,
    10.108,       9.32309,      8.54755,      7.89648,     7.39715,
    6.77075,      6.03522,      5.37585,      4.80758,     4.37503,
    3.94589,      3.57219,      3.10757,      2.72089,     2.37604,
    1.98762,      1.6407,       1.48103,      1.14178,     0.947468,
    0.678582,     0.347498,     0.210941,     -0.00647286, -0.198583,
    -0.372649,    -0.528396,    -0.650386,    -0.824333,   -0.93703,
    -1.01761,     -1.16189,     -1.26142,     -1.29735,    -1.37762,
    1171.6,       1092.48,      1018.7,       949.888,     885.716,
    825.869,      770.055,      744.94,       739.848,     734.388,
    728.533,      722.256,      715.524,      708.307,     700.567,
    692.268,      683.37,       673.829,      663.597,     652.627,
    640.864,      628.251,      614.726,      600.224,     584.673,
    567.999,      550.12,       530.949,      510.393,     488.351,
    464.716,      439.373,      412.198,      383.06,      351.816,
    318.314,      282.39,       243.871,      202.568,     158.281,
    110.793,      64.6437,      62.13,        59.4347,     56.5446,
    53.4456,      50.1227,      46.5596,      42.7391,     38.6424,
    34.2497,      31.9457,      30.0133,      27.9413,     25.7196,
    23.3373,      21.1438,      19.9844,      18.7411,     17.408,
    16.0713,      14.9584,      13.7651,      12.5449,     11.7771,
    10.9537,      10.0063,      8.93243,      8.01317,     7.45547,
    6.9015,       6.35869,      5.77528,      5.21557,     4.76942,
    4.13996,      3.73032,      3.31069,      2.88066,     2.52127,
    2.19745,      1.8307,       1.59457,      1.29286,     0.943094,
    0.649896,     0.510668,     0.261084,     0.077349,    -0.207967,
    -0.338,       -0.546251,    -0.690171,    -0.788067,   -0.965079,
    -1.01129,     -1.13007,     -1.30581,     -1.35931,    -1.37126,
    1232.63,      1149.48,      1071.93,      999.602,     932.152,
    869.247,      810.583,      778.024,      772.769,     767.134,
    761.091,      754.612,      747.665,      740.216,     732.228,
    723.664,      714.48,       704.633,      694.073,     682.751,
    670.611,      657.593,      643.635,      628.668,     612.619,
    595.41,       576.958,      557.172,      535.957,     513.208,
    488.815,      462.659,      434.614,      404.541,     372.295,
    337.719,      300.644,      260.89,       218.263,     172.555,
    123.544,      70.9917,      66.8385,      63.9765,     60.9078,
    57.6173,      54.089,       50.3057,      46.2491,     41.8992,
    37.235,       34.4795,      32.5285,      30.4364,     28.1932,
    25.7878,      23.3709,      22.0749,      20.6853,     19.1952,
    17.6413,      16.3545,      14.9746,      13.495,      12.4013,
    11.2398,      10.1619,      9.25383,      8.3892,      7.8279,
    7.19759,      6.46633,      5.94275,      5.50082,     4.98902,
    4.43815,      3.97277,      3.52934,      3.086,       2.69761,
    2.30903,      1.95796,      1.60543,      1.33773,     1.08952,
    0.844037,     0.574922,     0.295408,     0.0753319,   -0.0915852,
    -0.283943,    -0.487657,    -0.695571,    -0.793425,   -0.896977,
    -1.13457,     -1.23625,     -1.28336,     -1.33969,    -1.4593,
    1282.61,      1196.04,      1115.31,      1040.01,     969.794,
    904.307,      843.235,      803.037,      797.661,     791.896,
    785.715,      779.087,      771.98,       764.359,     756.188,
    747.426,      738.032,      727.958,      717.156,     705.573,
    693.154,      679.837,      665.557,      650.246,     633.828,
    616.223,      597.347,      577.106,      555.402,     532.13,
    507.177,      480.419,      451.729,      420.964,     387.977,
    352.606,      314.678,      274.01,       230.402,     183.643,
    133.505,      79.7438,      68.9643,      66.0592,     62.9441,
    59.6039,      56.0224,      52.182,       48.064,      43.6485,
    38.9138,      35.6354,      33.4923,      31.1942,     28.7301,
    26.0879,      23.2548,      21.9325,      20.5288,     19.0237,
    17.4099,      16.2809,      15.0804,      13.7932,     12.8723,
    11.9653,      11.0538,      10.2322,      9.35663,     8.46204,
    7.56367,      6.8182,       6.16809,      5.59897,     5.05968,
    4.58105,      4.13762,      3.68961,      3.2423,      2.89538,
    2.49892,      2.1055,       1.74419,      1.41206,     1.13562,
    0.894538,     0.672958,     0.416667,     0.182089,    -0.0625864,
    -0.289822,    -0.464886,    -0.681962,    -0.867253,   -0.968716,
    -0.992721,    -1.20389,     -1.27014,     -1.29987,    -1.43252,
    1351.05,      1259.88,      1174.86,      1095.57,     1021.62,
    952.652,      888.335,      839.505,      833.938,     827.969,
    821.568,      814.705,      807.346,      799.455,     790.994,
    781.921,      772.193,      761.761,      750.576,     738.582,
    725.722,      711.932,      697.146,      681.291,     664.291,
    646.062,      626.515,      605.556,      583.082,     558.984,
    533.145,      505.438,      475.729,      443.873,     409.715,
    373.088,      333.815,      291.703,      246.548,     198.13,
    146.212,      90.5432,      72.5572,      69.4969,     66.2155,
    62.6969,      58.9241,      54.8786,      50.5407,     45.8894,
    40.9019,      37.1669,      35.1706,      33.03,       30.7347,
    28.2735,      25.6345,      23.9518,      22.3244,     20.5794,
    18.7083,      17.458,       16.2422,      14.9385,     13.8545,
    12.8099,      11.7252,      10.7443,      9.69238,     8.8519,
    7.96963,      7.17172,      6.56226,      6.24093,     5.65953,
    5.01411,      4.39443,      3.9299,       3.59279,     3.03486,
    2.60758,      2.34514,      2.03512,      1.52487,     1.3122,
    1.02788,      0.564328,     0.485573,     0.175659,    -0.05916,
    -0.217875,    -0.460152,    -0.620034,    -0.822028,   -0.927539,
    -1.09659,     -1.15765,     -1.30723,     -1.40824,    -1.4426,
    1455.72,      1357.37,      1265.65,      1180.11,     1100.34,
    1025.94,      956.558,      897.062,      891.144,     884.798,
    877.994,      870.698,      862.874,      854.486,     845.491,
    835.846,      825.504,      814.414,      802.523,     789.773,
    776.102,      761.442,      745.723,      728.868,     710.795,
    691.416,      670.637,      648.356,      624.464,     598.846,
    571.377,      541.923,      510.34,       476.474,     440.161,
    401.224,      359.474,      314.706,      266.702,     215.23,
    160.038,      100.857,      74.646,       71.5935,     68.3205,
    64.811,       61.0478,      57.0127,      52.6859,     48.0465,
    43.0718,      38.8645,      36.6868,      34.3516,     31.8477,
    29.1629,      26.284,       24.3951,      22.7837,     21.056,
    19.2033,      17.8194,      16.5527,      15.1945,     14.033,
    12.9802,      11.8714,      11.1065,      10.2863,     9.4759,
    8.6175,       7.94168,      7.22646,      6.47045,     5.88859,
    5.28072,      4.68477,      4.35467,      3.65537,     3.219,
    2.8172,       2.40046,      2.09112,      1.78291,     1.44919,
    1.16054,      0.754807,     0.459435,     0.206231,    0.0275945,
    -0.20358,     -0.308483,    -0.630444,    -0.76727,    -0.939666,
    -1.06692,     -1.24372,     -1.35292,     -1.41873,    -1.50423,
    1542.43,      1438.19,      1340.96,      1250.3,      1165.74,
    1086.88,      1013.33,      944.743,      936.907,     930.288,
    923.192,      915.582,      907.423,      898.673,     889.292,
    879.233,      868.446,      856.881,      844.479,     831.181,
    816.922,      801.633,      785.239,      767.66,      748.81,
    728.599,      706.926,      683.688,      658.77,      632.052,
    603.402,      572.682,      539.742,      504.422,     466.549,
    425.939,      382.395,      335.703,      285.638,     231.954,
    174.391,      112.667,      77.8058,      74.6518,     71.2699,
    67.6436,      63.7553,      59.5859,      55.1153,     50.3215,
    45.1813,      40.3948,      38.0248,      35.4837,     32.7588,
    29.8371,      26.7042,      24.7471,      23.5012,     22.1654,
    20.7329,      19.2532,      17.7019,      16.0385,     14.6855,
    13.6925,      12.6276,      11.632,       10.5764,     9.81056,
    9.11505,      8.26052,      7.37603,      6.64187,     6.06255,
    5.56608,      5.07737,      4.39076,      3.96675,     3.51978,
    2.97018,      2.47622,      2.24183,      1.88195,     1.44653,
    1.21513,      0.896494,     0.724087,     0.357482,    0.0125673,
    -0.154619,    -0.380906,    -0.612691,    -0.763734,   -0.950282,
    -1.02003,     -1.17806,     -1.31469,     -1.46794,    -1.49623,
    1528.91,      1425.75,      1329.54,      1239.82,     1156.14,
    1078.11,      1005.33,      937.457,      922.669,     916.241,
    909.348,      901.958,      894.033,      885.535,     876.423,
    866.653,      856.177,      844.943,      832.898,     819.982,
    806.133,      791.283,      775.36,       758.287,     739.979,
    720.348,      699.299,      676.728,      652.527,     626.576,
    598.75,       568.913,      536.92,       502.615,     465.83,
    426.388,      384.095,      338.745,      290.118,     237.978,
    182.069,      122.119,      80.8828,      77.6575,     74.199,
    70.4907,      66.5144,      62.2507,      57.6789,     52.7766,
    47.5202,      42.2293,      39.9625,      37.5318,     34.9255,
    32.1308,      29.1342,      26.8439,      25.3013,     23.6471,
    21.8734,      20.2768,      18.8767,      17.3754,     15.8798,
    14.4734,      12.9654,      11.9975,      11.1189,     10.3585,
    9.65258,      8.7516,       7.76962,      7.16887,     6.55728,
    5.93355,      5.29189,      4.78043,      4.24213,     3.74361,
    3.27734,      2.82226,      2.43199,      2.05347,     1.61887,
    1.20894,      1.01602,      0.635206,     0.390914,    0.117418,
    -0.26856,     -0.295885,    -0.615221,    -0.712256,   -0.889681,
    -1.1045,      -1.21906,     -1.34852,     -1.50759,    -1.49963,
    1700.55,      1585.7,       1478.59,      1378.7,      1285.54,
    1198.66,      1117.63,      1042.07,      1018.01,     1010.97,
    1003.42,      995.319,      986.635,      977.324,     967.339,
    956.633,      945.153,      932.844,      919.645,     905.492,
    890.317,      874.045,      856.596,      837.887,     817.826,
    796.315,      773.25,       748.517,      721.998,     693.562,
    663.07,       630.376,      595.318,      557.727,     517.42,
    474.199,      427.855,      378.162,      324.878,     267.743,
    206.479,      140.788,      87.6015,      84.0951,     80.3354,
    76.3038,      71.981,       67.3457,      62.3755,     57.0461,
    51.3316,      45.204,       42.8092,      40.3412,     37.6949,
    34.8573,      31.8146,      29.101,       27.0665,     24.885,
    22.5459,      20.478,       18.9866,      17.3874,     15.8964,
    14.9666,      13.9695,      13.0508,      12.1353,     11.1055,
    9.95449,      9.1203,       8.47327,      7.57811,     6.76638,
    6.08392,      5.34421,      4.8822,       4.36195,     3.80115,
    3.37809,      3.02142,      2.41716,      2.22522,     1.76518,
    1.39146,      1.11391,      0.762009,     0.447379,    0.205101,
    -0.0527301,   -0.207446,    -0.510581,    -0.69394,    -0.916602,
    -1.08684,     -1.22759,     -1.32001,     -1.44374,    -1.58435,
    1732.1,       1615.23,      1506.23,      1404.58,     1309.78,
    1221.37,      1138.92,      1062.03,      1029.81,     1022.77,
    1015.23,      1007.15,      998.476,      989.178,     979.209,
    968.519,      957.056,      944.766,      931.586,     917.455,
    902.302,      886.054,      868.632,      849.951,     829.92,
    808.442,      785.411,      760.716,      734.236,     705.843,
    675.397,      642.752,      607.747,      570.212,     529.965,
    486.81,       440.536,      390.917,      337.713,     280.664,
    219.492,      153.9,        92.7597,      88.9397,     84.8438,
    80.4518,      75.7424,      70.6927,      65.2781,     59.4721,
    53.2466,      46.5712,      43.746,       41.4427,     38.973,
    36.3248,      33.4852,      30.7704,      28.7512,     26.586,
    24.2643,      22.0332,      20.3673,      18.5811,     16.8047,
    15.781,       14.6832,      13.6317,      12.6007,     11.5039,
    10.3414,      9.48786,      8.96446,      8.10067,     7.27559,
    6.70424,      5.86076,      5.08077,      4.72915,     4.20929,
    3.57709,      3.11345,      2.64202,      2.33735,     1.89926,
    1.55055,      1.13185,      0.84508,      0.599964,    0.289909,
    -0.0159803,   -0.227488,    -0.400299,    -0.6604,     -0.855246,
    -1.08755,     -1.27622,     -1.38808,     -1.43374,    -1.59861,
    1817.78,      1695.08,      1580.66,      1473.94,     1374.41,
    1281.6,       1195.04,      1114.31,      1072.77,     1065.47,
    1057.65,      1049.26,      1040.26,      1030.62,     1020.28,
    1009.19,      997.296,      984.545,      970.874,     956.214,
    940.495,      923.64,       905.567,      886.188,     865.408,
    843.127,      819.235,      793.617,      766.148,     736.693,
    705.11,       671.244,      634.931,      595.994,     554.243,
    509.474,      461.47,       409.998,      354.805,     295.624,
    232.166,      164.122,      92.4458,      89.0142,     85.3346,
    81.3891,      77.1585,      72.6221,      67.7579,     62.5422,
    56.9495,      50.9527,      47.4832,      44.8127,     41.9491,
    38.8787,      35.5863,      32.3103,      30.1675,     27.8697,
    25.406,       22.9778,      21.5958,      20.114,      18.5332,
    17.1691,      15.7065,      14.3875,      13.2786,     12.1674,
    11.181,       10.2042,      9.28575,      8.44456,     7.63216,
    6.87572,      6.15058,      5.52348,      4.98282,     4.45209,
    3.91867,      3.35367,      2.91509,      2.34051,     2.03966,
    1.61328,      1.32256,      1.01318,      0.711406,    0.378202,
    0.0514687,    -0.306978,    -0.419297,    -0.754624,   -0.925815,
    -1.05641,     -1.26951,     -1.38174,     -1.46731,    -1.5836,
    1940.41,      1809.36,      1687.14,      1573.16,     1466.86,
    1367.72,      1275.26,      1189.04,      1136.29,     1128.64,
    1120.42,      1111.62,      1102.18,      1092.06,     1081.2,
    1069.57,      1057.09,      1043.71,      1029.36,     1013.97,
    997.475,      979.785,      960.818,      940.479,     918.671,
    895.287,      870.213,      843.327,      814.498,     783.585,
    750.439,      714.897,      676.787,      635.922,     592.104,
    545.12,       494.74,       440.72,       382.796,     320.685,
    254.086,      182.674,      106.102,      94.9466,     90.8742,
    86.5076,      81.8254,      76.8049,      71.4215,     65.6491,
    59.4595,      52.8226,      48.526,       45.6326,     42.5301,
    39.2033,      35.6362,      31.9171,      30.1805,     28.3184,
    26.3217,      24.2088,      22.6704,      21.0208,     19.252,
    17.8495,      16.3964,      15.0399,      13.9845,     12.8554,
    11.6568,      10.4879,      9.54564,      8.6224,      7.72237,
    7.01265,      6.39011,      5.87072,      5.35272,     4.66555,
    4.02553,      3.55532,      3.02578,      2.59664,     2.07802,
    1.81799,      1.37951,      1.09925,      0.574104,    0.29946,
    0.0613693,    -0.136267,    -0.385063,    -0.726143,   -0.857438,
    -1.23015,     -1.25949,     -1.43591,     -1.57855,    -1.60205,
    1982.62,      1848.88,      1724.16,      1607.84,     1499.36,
    1398.19,      1303.84,      1215.84,      1153.56,     1145.88,
    1137.64,      1128.81,      1119.35,      1109.19,     1098.31,
    1086.64,      1074.12,      1060.7,       1046.32,     1030.89,
    1014.34,      996.603,      977.582,      957.186,     935.316,
    911.865,      886.72,       859.757,      830.846,     799.846,
    766.606,      730.963,      692.744,      651.764,     607.822,
    560.704,      510.181,      456.008,      397.919,     335.632,
    268.844,      197.229,      120.439,      100.381,     96.1039,
    91.5174,      86.5994,      81.3261,      75.6717,     69.6086,
    63.1074,      56.1363,      51.3028,      48.6827,     45.8733,
    42.8609,      39.6308,      36.1673,      33.6489,     31.0381,
    28.2386,      25.2367,      23.602,       21.9829,     20.2467,
    18.7761,      17.303,       15.8497,      14.7304,     13.5322,
    12.284,       11.0193,      10.0496,      9.24253,     8.62065,
    7.4781,       6.69614,      6.09674,      5.44904,     4.82441,
    4.33703,      3.81054,      3.2414,       2.76016,     2.20885,
    1.87149,      1.4471,       1.1051,       0.82956,     0.39048,
    0.138649,     -0.182551,    -0.378605,    -0.603052,   -0.91456,
    -1.06262,     -1.26963,     -1.39872,     -1.54087,    -1.69227,
    1989.12,      1855.0,       1729.92,      1613.27,     1504.49,
    1403.03,      1308.41,      1220.17,      1149.37,     1141.79,
    1133.68,      1124.97,      1115.64,      1105.63,     1094.89,
    1083.38,      1071.04,      1057.81,      1043.63,     1028.41,
    1012.1,       994.607,      975.852,      955.74,      934.176,
    911.053,      886.259,      859.673,      831.166,     800.599,
    767.823,      732.678,      694.994,      654.586,     611.257,
    564.798,      514.981,      461.564,      404.287,     342.871,
    277.016,      206.402,      130.684,      102.285,     98.0111,
    93.4286,      88.515,       83.2462,      77.5967,     71.539,
    65.0434,      58.0785,      52.558,       49.6233,     46.4764,
    43.1021,      39.484,       35.6044,      33.2467,     31.1294,
    28.859,       26.4245,      24.5149,      22.6357,     20.6207,
    19.0092,      17.5452,      16.0571,      15.102,      14.0779,
    12.8075,      11.4606,      10.5182,      9.63211,     8.88919,
    8.08621,      7.33582,      6.60633,      5.80883,     5.19884,
    4.59095,      4.02239,      3.4522,       2.93151,     2.52793,
    2.0426,       1.61069,      1.14254,      0.915358,    0.50104,
    0.187597,     -0.0457379,   -0.343786,    -0.625425,   -0.823645,
    -1.2,         -1.22168,     -1.43326,     -1.58173,    -1.68165,
    2113.08,      1970.52,      1837.56,      1713.57,     1597.93,
    1490.09,      1389.51,      1295.71,      1212.0,      1204.05,
    1195.53,      1186.39,      1176.59,      1166.08,     1154.81,
    1142.73,      1129.77,      1115.88,      1100.98,     1085.01,
    1067.89,      1049.52,      1029.83,      1008.72,     986.076,
    961.799,      935.768,      907.856,      877.927,     845.835,
    811.424,      774.526,      734.962,      692.538,     647.049,
    598.272,      545.97,       489.888,      429.754,     365.273,
    296.133,      221.997,      142.502,      103.832,     99.6339,
    95.1326,      90.306,       85.1306,      79.5811,     73.6307,
    67.2502,      60.4086,      54.4897,      51.7152,     48.7401,
    45.5501,      42.1295,      38.4617,      35.7874,     33.4585,
    30.9613,      28.2836,      26.1071,      24.0821,     21.9108,
    20.1199,      18.6176,      17.0084,      15.6426,     14.1782,
    13.2389,      12.3669,      11.286,       10.172,      9.12976,
    8.34711,      7.59311,      6.85289,      6.4352,      5.5686,
    4.91388,      4.08573,      3.6806,       3.19687,     2.73174,
    2.14777,      1.75248,      1.37943,      1.06145,     0.612264,
    0.269845,     -0.0680414,   -0.350309,    -0.546949,   -0.728089,
    -1.07791,     -1.26203,     -1.42393,     -1.56509,    -1.71267,
    2203.02,      2054.48,      1915.95,      1786.76,     1666.28,
    1553.91,      1449.12,      1351.4,       1260.25,     1246.82,
    1238.12,      1228.79,      1218.79,      1208.07,     1196.58,
    1184.25,      1171.03,      1156.86,      1141.66,     1125.36,
    1107.89,      1089.15,      1069.06,      1047.52,     1024.42,
    999.648,      973.089,      944.611,      914.075,     881.332,
    846.222,      808.576,      768.209,      724.924,     678.512,
    628.745,      575.382,      518.163,      456.808,     391.02,
    320.477,      244.836,      163.729,      114.789,     109.894,
    104.646,      99.0177,      92.9831,      86.5124,     79.574,
    72.1343,      64.1569,      56.6608,      53.6144,     50.3478,
    46.8451,      43.0894,      39.0622,      36.1447,     33.9982,
    31.6967,      29.2288,      27.195,       25.4578,     23.5951,
    21.7052,      19.8093,      17.7765,      16.4817,     15.2028,
    13.9,         12.5309,      11.4987,      10.5357,     9.77749,
    8.96882,      8.14855,      7.35339,      6.28491,     5.69516,
    5.12575,      4.52768,      3.89487,      3.24965,     2.86361,
    2.43374,      1.91164,      1.42856,      1.11888,     0.598667,
    0.332547,     0.0316785,    -0.186562,    -0.640855,   -0.759071,
    -1.12222,     -1.26716,     -1.43189,     -1.57871,    -1.74731,
    2270.02,      2116.95,      1974.21,      1841.08,     1716.93,
    1601.15,      1493.16,      1392.46,      1298.54,     1276.06,
    1267.19,      1257.68,      1247.49,      1236.56,     1224.83,
    1212.26,      1198.79,      1184.33,      1168.84,     1152.22,
    1134.4,       1115.3,       1094.81,      1072.85,     1049.29,
    1024.04,      996.955,      967.917,      936.781,     903.394,
    867.595,      829.209,      788.048,      743.913,     696.589,
    645.844,      591.432,      533.089,      470.528,     403.447,
    331.518,      254.391,      171.69,       112.641,     108.217,
    103.473,      98.3857,      92.9312,      87.0826,     80.8112,
    74.0867,      66.8761,      59.5035,      56.5659,     53.4161,
    50.0386,      46.4171,      42.5338,      39.3243,     36.9281,
    34.3588,      31.6038,      29.0542,      26.7777,     24.3367,
    22.0707,      20.3184,      18.4394,      17.1246,     15.9156,
    14.6955,      13.4384,      12.3349,      11.2617,     10.321,
    9.402,        8.49434,      7.53493,      6.83178,     6.20981,
    5.3599,       4.63161,      4.02841,      3.6173,      3.04666,
    2.31783,      2.04878,      1.60795,      1.19175,     0.946452,
    0.393905,     0.0945319,    -0.254012,    -0.555897,   -0.697426,
    -0.906549,    -1.28411,     -1.37921,     -1.59635,    -1.72063,
    2489.7,       2321.6,       2164.84,      2018.64,     1882.3,
    1755.14,      1636.55,      1525.96,      1422.82,     1388.57,
    1378.96,      1368.66,      1357.61,      1345.77,     1333.06,
    1319.44,      1304.84,      1289.18,      1272.39,     1254.38,
    1235.08,      1214.38,      1192.18,      1168.38,     1142.86,
    1115.49,      1086.15,      1054.69,      1020.95,     984.773,
    945.983,      904.39,       859.791,      811.969,     760.691,
    705.707,      646.75,       583.532,      515.745,     443.06,
    365.122,      281.551,      191.941,      117.851,     113.383,
    108.592,      103.456,      97.9475,      92.0414,     85.7086,
    78.918,       71.6367,      63.8293,      60.1469,     56.4,
    52.3823,      48.0743,      43.4549,      39.5771,     37.2635,
    34.7828,      32.1227,      29.6236,      27.5704,     25.3689,
    23.218,       21.5997,      19.8644,      18.2661,     16.684,
    15.2996,      14.1413,      12.9545,      11.7185,     10.8107,
    9.82832,      8.7599,       7.93536,      7.18969,     6.30968,
    5.61252,      4.91649,      4.28346,      3.70528,     3.17702,
    2.75035,      2.1555,       1.72676,      1.30952,     0.786323,
    0.522783,     0.211445,     -0.0363499,   -0.528405,   -0.803153,
    -0.990703,    -1.15014,     -1.38437,     -1.59599,    -1.82963,
    2551.14,      2378.98,      2218.42,      2068.69,     1929.04,
    1798.81,      1677.35,      1564.08,      1458.45,     1413.42,
    1403.76,      1393.4,       1382.29,      1370.38,     1357.6,
    1343.91,      1329.22,      1313.48,      1296.59,     1278.49,
    1259.07,      1238.26,      1215.93,      1192.0,      1166.34,
    1138.82,      1109.31,      1077.67,      1043.75,     1007.37,
    968.363,      926.538,      881.69,       833.601,     782.037,
    726.746,      667.46,       603.89,       535.725,     462.634,
    384.261,      300.224,      210.115,      125.287,     120.31,
    114.974,      109.253,      103.117,      96.539,      89.4851,
    81.9214,      73.8111,      65.1146,      60.9945,     57.5163,
    53.7868,      49.7878,      45.4997,      41.4749,     38.7817,
    35.8939,      32.7974,      29.8585,      27.8319,     25.6588,
    23.4491,      21.9052,      20.2497,      18.844,      17.6315,
    16.3678,      15.0718,      13.5818,      11.8803,     10.897,
    9.96365,      8.85041,      8.16051,      7.17732,     6.47006,
    5.861,        5.25454,      4.50342,      4.00288,     3.38948,
    2.74113,      2.37125,      1.92047,      1.43047,     0.949728,
    0.663837,     0.241889,     -0.0661863,   -0.550978,   -0.80188,
    -0.976414,    -1.27424,     -1.43813,     -1.55167,    -1.83335,
    2620.65,      2443.95,      2279.15,      2125.46,     1982.13,
    1848.46,      1723.8,       1607.54,      1499.11,     1442.94,
    1433.18,      1422.71,      1411.49,      1399.45,     1386.55,
    1372.71,      1357.87,      1341.96,      1324.9,      1306.61,
    1287.0,       1265.97,      1243.41,      1219.23,     1193.3,
    1165.5,       1135.69,      1103.72,      1069.44,     1032.69,
    993.28,       951.022,      905.71,       857.124,     805.026,
    749.163,      689.263,      625.035,      556.165,     482.317,
    403.133,      318.227,      227.185,      131.151,     126.138,
    120.763,      114.999,      108.819,      102.193,     95.087,
    87.468,       79.2984,      70.5385,      65.431,      61.484,
    57.2518,      52.7137,      47.8476,      43.1162,     40.6876,
    38.0836,      35.2913,      32.4409,      30.214,      27.8262,
    25.2822,      23.2616,      21.0949,      19.4204,     18.423,
    17.1926,      15.4471,      13.8877,      12.7161,     11.8642,
    10.9283,      9.29109,      8.52418,      7.73035,     6.96802,
    6.23931,      5.50133,      4.8461,       4.29862,     3.60227,
    2.98787,      2.59448,      2.02527,      1.39649,     1.11825,
    0.58392,      0.270744,     -0.152795,    -0.504465,   -0.643865,
    -0.986758,    -1.29124,     -1.42729,     -1.56625,    -1.79783,
    2821.93,      2631.42,      2453.76,      2288.06,     2133.54,
    1989.43,      1855.03,      1729.69,      1612.79,     1541.99,
    1531.58,      1520.43,      1508.47,      1495.65,     1481.9,
    1467.15,      1451.34,      1434.39,      1416.21,     1396.72,
    1375.82,      1353.41,      1329.38,      1303.61,     1275.99,
    1246.36,      1214.59,      1180.53,      1144.01,     1104.85,
    1062.85,      1017.83,      969.545,      917.775,     862.263,
    802.739,      738.914,      670.476,      597.093,     518.406,
    434.033,      343.562,      246.554,      142.535,     128.246,
    122.943,      117.257,      111.16,       104.622,     97.6122,
    90.0954,      82.0354,      73.3929,      67.6629,     63.6091,
    59.2623,      54.6014,      49.6036,      44.4136,     41.9771,
    39.3645,      36.5631,      33.5918,      31.1213,     28.4723,
    25.6318,      24.0442,      22.4795,      20.8518,     19.2026,
    17.5063,      16.041,       14.4685,      12.7787,     11.6383,
    10.8592,      10.1111,      9.20717,      8.10161,     7.21587,
    6.63666,      5.72982,      5.07703,      4.32362,     3.85101,
    3.1389,       2.50863,      2.14046,      1.44775,     1.33092,
    0.903288,     0.501062,     -0.000566156, -0.317971,   -0.703171,
    -1.13877,     -1.24115,     -1.45828,     -1.61887,    -1.80583,
    2925.96,      2728.43,      2544.21,      2372.41,     2212.18,
    2062.76,      1923.41,      1793.44,      1672.24,     1588.06,
    1577.44,      1566.05,      1553.84,      1540.75,     1526.71,
    1511.66,      1495.52,      1478.21,      1459.65,     1439.76,
    1418.42,      1395.54,      1371.01,      1344.7,      1316.5,
    1286.25,      1253.82,      1219.05,      1181.76,     1141.78,
    1098.91,      1052.94,      1003.65,      950.799,     894.127,
    833.359,      768.2,        698.332,      623.414,     543.083,
    456.946,      364.584,      265.547,      159.354,     133.289,
    127.603,      121.505,      114.966,      107.955,     100.438,
    92.3766,      83.7331,      74.465,       67.995,      64.2368,
    60.2069,      55.8858,      51.2525,      46.2843,     43.8075,
    41.3176,      38.6479,      35.7851,      33.1569,     30.3685,
    27.3786,      25.4457,      23.6818,      21.7683,     19.6452,
    17.4752,      16.6651,      15.6926,      14.1568,     12.6808,
    11.2776,      10.3911,      9.60294,      8.74579,     7.79914,
    7.04734,      6.20423,      5.33814,      4.5806,      3.93879,
    3.42633,      2.88353,      2.21512,      1.81465,     1.32392,
    0.838668,     0.39342,      0.0400102,    -0.330415,   -0.607138,
    -0.931814,    -1.26086,     -1.4606,      -1.71251,    -1.8526,
    2932.46,      2734.71,      2550.29,      2378.3,      2217.9,
    2068.31,      1928.8,       1798.69,      1677.36,     1582.45,
    1571.97,      1560.73,      1548.68,      1535.75,     1521.9,
    1507.04,      1491.11,      1474.03,      1455.72,     1436.08,
    1415.02,      1392.44,      1368.22,      1342.26,     1314.42,
    1284.57,      1252.56,      1218.24,      1181.44,     1141.98,
    1099.67,      1054.3,       1005.65,      953.487,     897.553,
    837.576,      773.265,      704.307,      630.365,     551.08,
    466.065,      374.906,      277.159,      172.349,     135.722,
    130.409,      124.712,      118.603,      112.052,     105.028,
    97.4968,      89.4211,      80.7617,      73.8319,     69.7014,
    65.2723,      60.5232,      55.4309,      49.9706,     46.7058,
    43.711,       40.4997,      37.0563,      34.1797,     31.264,
    28.1376,      26.0613,      24.3883,      22.6017,     20.7325,
    18.7283,      17.2866,      15.7946,      14.5225,     13.3777,
    12.4836,      11.1073,      9.96717,      9.08724,     8.23464,
    7.36886,      6.2089,       5.5777,       4.86593,     4.27778,
    3.81689,      2.93016,      2.4841,       2.04322,     1.35418,
    1.00823,      0.513767,     0.132787,     -0.363828,   -0.576904,
    -0.899434,    -1.3183,      -1.48161,     -1.75291,    -1.77905,
    3009.64,      2806.71,      2617.46,      2440.96,     2276.36,
    2122.85,      1979.69,      1846.17,      1721.65,     1613.17,
    1602.62,      1591.3,       1579.17,      1566.16,     1552.21,
    1537.25,      1521.21,      1504.02,      1485.58,     1465.8,
    1444.6,       1421.86,      1397.49,      1371.35,     1343.32,
    1313.26,      1281.04,      1246.48,      1209.43,     1169.7,
    1127.1,       1081.42,      1032.44,      979.92,      923.605,
    863.219,      798.469,      729.04,       654.594,     574.768,
    489.172,      397.392,      298.978,      193.452,     144.937,
    138.813,      132.246,      125.205,      117.655,     109.559,
    100.878,      91.5695,      81.5884,      73.1657,     69.1426,
    64.8287,      60.2031,      55.2431,      49.9248,     46.4796,
    43.6212,      40.5561,      37.2695,      34.6631,     32.2221,
    29.6047,      27.2937,      25.1576,      22.8854,     20.9936,
    18.965,       17.7736,      16.665,       15.2861,     13.8625,
    12.4836,      11.4148,      10.4491,      9.49124,     8.42045,
    7.66634,      6.9045,       6.00645,      5.00484,     4.24905,
    3.46471,      3.08744,      2.50835,      2.06374,     1.59976,
    1.02151,      0.554219,     0.0867681,    -0.284686,   -0.597584,
    -0.815138,    -1.22016,     -1.43577,     -1.6551,     -1.91744,
    3348.96,      3122.78,      2911.84,      2715.13,     2531.67,
    2360.58,      2201.01,      2052.2,       1913.42,     1784.0,
    1769.61,      1757.1,       1743.69,      1729.31,     1713.89,
    1697.36,      1679.63,      1660.62,      1640.24,     1618.39,
    1594.95,      1569.82,      1542.88,      1513.99,     1483.01,
    1449.79,      1414.17,      1375.98,      1335.03,     1291.11,
    1244.03,      1193.54,      1139.4,       1081.35,     1019.11,
    952.367,      880.801,      804.063,      721.779,     633.55,
    538.944,      437.501,      328.727,      212.092,     146.578,
    140.687,      134.37,       127.596,      120.333,     112.546,
    104.195,      95.2409,      85.6397,      76.8238,     72.9151,
    68.7239,      64.2298,      59.411,       54.2439,     50.3553,
    47.1782,      43.7715,      40.1187,      36.938,      33.9851,
    30.8189,      28.4285,      26.9213,      25.3053,     23.1026,
    20.7041,      18.8091,      17.006,       15.8922,     14.8092,
    13.5596,      12.2582,      11.0254,      10.0022,     9.04615,
    7.76003,      6.90787,      6.21349,      5.15071,     4.37373,
    4.02896,      3.24761,      2.8996,       2.10396,     1.79379,
    1.26257,      0.683909,     0.244041,     -0.0768683,  -0.530012,
    -0.778613,    -1.05734,     -1.43989,     -1.80183,    -1.82794,
    3438.18,      3206.13,      2989.71,      2787.88,     2599.66,
    2424.12,      2260.41,      2107.74,      1965.35,     1832.56,
    1805.85,      1793.21,      1779.66,      1765.13,     1749.55,
    1732.84,      1714.92,      1695.71,      1675.11,     1653.03,
    1629.34,      1603.95,      1576.72,      1547.52,     1516.22,
    1482.65,      1446.65,      1408.05,      1366.67,     1322.29,
    1274.7,       1223.68,      1168.97,      1110.3,      1047.4,
    979.95,       907.626,      830.074,      746.919,     657.754,
    562.145,      459.627,      349.701,      231.83,      153.209,
    147.515,      141.409,      134.863,      127.843,     120.316,
    112.245,      103.591,      94.3111,      84.9723,     79.9126,
    74.4872,      68.6698,      62.432,       55.7434,     50.9058,
    47.8654,      44.6052,      41.1094,      38.1183,     35.6299,
    32.9617,      30.3986,      28.1263,      25.6898,     23.8292,
    21.9974,      19.942,       17.6872,      16.333,      15.2171,
    13.9437,      12.7221,      11.5129,      10.222,      9.08125,
    8.66797,      7.38369,      6.56059,      5.71344,     4.784,
    4.04727,      3.6676,       2.65175,      2.28055,     1.87991,
    1.27249,      0.782065,     0.291568,     -0.0522279,  -0.553634,
    -0.830554,    -1.16563,     -1.48748,     -1.61364,    -1.93756,
    3439.33,      3207.48,      2991.25,      2789.6,      2601.54,
    2426.15,      2262.58,      2110.04,      1967.78,     1835.11,
    1796.32,      1783.95,      1770.69,      1756.46,     1741.21,
    1724.86,      1707.32,      1688.52,      1668.36,     1646.74,
    1623.56,      1598.71,      1572.05,      1543.48,     1512.83,
    1479.97,      1444.74,      1406.96,      1366.45,     1323.02,
    1276.44,      1226.5,       1172.95,      1115.53,     1053.96,
    987.945,      917.155,      841.249,      759.858,     672.585,
    579.005,      478.662,      371.067,      255.698,     166.111,
    159.219,      151.829,      143.905,      135.408,     126.297,
    116.528,      106.052,      94.8197,      82.8681,     78.7225,
    74.2774,      69.511,       64.4001,      58.9199,     54.1789,
    50.6711,      46.9099,      42.8768,      39.4468,     37.0517,
    34.4836,      31.8289,      29.2327,      26.4488,     24.0502,
    21.7094,      20.0413,      18.9709,      17.5158,     15.7913,
    14.6548,      13.3138,      11.8126,      10.649,      9.62851,
    8.43738,      7.63112,      6.70698,      5.55645,     5.17261,
    4.59072,      3.86525,      3.25705,      2.24958,     2.10002,
    1.50848,      0.85558,      0.520422,     -0.153968,   -0.405191,
    -0.71068,     -1.03933,     -1.3914,      -1.67859,    -1.9395,
    3562.2,       3322.04,      3098.07,      2889.19,     2694.39,
    2512.72,      2343.29,      2185.28,      2037.92,     1900.5,
    1848.75,      1836.1,       1822.54,      1807.99,     1792.4,
    1775.67,      1757.74,      1738.52,      1717.9,      1695.79,
    1672.09,      1646.67,      1619.42,      1590.2,      1558.86,
    1525.26,      1489.24,      1450.61,      1409.18,     1364.77,
    1317.14,      1266.07,      1211.31,      1152.6,      1089.64,
    1022.13,      949.745,      872.126,      788.899,     699.656,
    603.965,      501.358,      391.336,      273.363,     169.369,
    162.528,      155.193,      147.328,      138.894,     129.85,
    120.154,      109.756,      98.6072,      86.6525,     81.7059,
    77.2148,      72.3992,      67.2355,      61.6987,     56.5735,
    52.8189,      48.7929,      44.476,       40.4669,     37.5465,
    34.4152,      31.377,       29.5384,      27.567,      25.3149,
    22.8134,      20.7566,      19.3537,      17.8456,     16.2255,
    14.7968,      13.5069,      12.4353,      11.0615,     9.73547,
    8.96977,      7.96321,      7.06181,      6.04234,     5.44984,
    4.6736,       3.90921,      3.31525,      2.7399,      2.09231,
    1.63663,      1.02307,      0.545369,     0.0117561,   -0.250129,
    -0.628264,    -1.02544,     -1.52811,     -1.66341,    -1.86154,
    3834.17,      3575.43,      3334.12,      3109.07,     2899.19,
    2703.46,      2520.92,      2350.68,      2191.91,     2043.84,
    1975.29,      1961.83,      1947.4,       1931.92,     1915.33,
    1897.54,      1878.46,      1858.0,       1836.06,     1812.54,
    1787.32,      1760.28,      1731.28,      1700.19,     1666.84,
    1631.09,      1592.76,      1551.66,      1507.58,     1460.32,
    1409.65,      1355.31,      1297.05,      1234.57,     1167.58,
    1095.75,      1018.73,      936.146,      847.59,      752.635,
    650.817,      541.642,      424.576,      299.051,     175.204,
    168.347,      160.995,      153.112,      144.66,      135.596,
    125.877,      115.456,      104.282,      92.3006,     85.8808,
    80.5752,      74.8861,      68.7859,      62.2448,     56.1347,
    52.8127,      49.2505,      45.431,       41.7441,     39.3209,
    36.7226,      33.9881,      31.6094,      29.0588,     26.6514,
    24.381,       22.2272,      20.4678,      18.689,      16.9143,
    15.5355,      14.2269,      12.8839,      11.7182,     10.6049,
    9.36426,      8.17173,      7.47598,      6.79724,     5.61594,
    4.76366,      4.09069,      3.46785,      2.81427,     2.15165,
    1.6817,       1.20018,      0.501222,     0.119424,    -0.35489,
    -0.682029,    -1.0383,      -1.44456,     -1.71039,    -2.02706,
    4050.85,      3777.34,      3522.26,      3284.36,     3062.5,
    2855.6,       2662.64,      2482.68,      2314.85,     2158.34,
    2073.02,      2058.94,      2043.84,      2027.66,     2010.3,
    1991.69,      1971.74,      1950.34,      1927.4,      1902.8,
    1876.42,      1848.14,      1817.81,      1785.29,     1750.42,
    1713.03,      1672.93,      1629.94,      1583.85,     1534.42,
    1481.42,      1424.59,      1363.65,      1298.31,     1228.25,
    1153.12,      1072.57,      986.188,      893.569,     794.256,
    687.766,      573.581,      451.143,      319.858,     179.085,
    169.697,      162.466,      154.712,      146.398,     137.483,
    127.924,      117.674,      106.684,      94.8988,     88.158,
    83.4462,      78.394,       72.9766,      67.1678,     61.2672,
    57.3304,      53.1092,      48.5829,      44.0249,     41.458,
    38.7056,      35.7542,      33.1823,      30.4321,     27.9002,
    25.7785,      23.5302,      21.2049,      19.1091,     17.604,
    16.1323,      14.6532,      13.2061,      11.919,      10.6956,
    9.57283,      8.60047,      7.68832,      6.83361,     5.75359,
    5.23433,      4.55925,      3.5059,       2.99582,     2.48199,
    1.71168,      1.34131,      0.62539,      -0.0151099,  -0.218086,
    -0.677888,    -1.07695,     -1.48193,     -1.68872,    -1.88524,
    4161.41,      3880.63,      3618.77,      3374.55,     3146.8,
    2934.39,      2736.3,       2551.56,      2379.28,     2218.6,
    2117.24,      2103.03,      2087.8,       2071.47,     2053.95,
    2035.17,      2015.04,      1993.45,      1970.29,     1945.47,
    1918.85,      1890.31,      1859.7,       1826.88,     1791.69,
    1753.96,      1713.5,       1670.12,      1623.6,      1573.72,
    1520.24,      1462.89,      1401.39,      1335.45,     1264.75,
    1188.94,      1107.64,      1020.48,      927.012,     826.792,
    719.328,      604.099,      480.543,      348.057,     205.997,
    182.103,      174.366,      166.07,       157.174,     147.635,
    137.408,      126.44,       114.681,      102.071,     93.6047,
    87.892,       81.7664,      75.1981,      68.1552,     60.7213,
    57.5354,      54.1191,      50.456,       46.5476,     43.5207,
    40.2751,      36.7949,      33.7945,      30.6723,     27.7628,
    25.5987,      23.4366,      22.0774,      20.4951,     18.4252,
    16.5511,      14.864,       13.7764,      12.3938,     11.1498,
    10.2687,      9.29867,      7.9444,       7.25799,     6.32619,
    5.34818,      4.8219,       3.73018,      3.20093,     2.50387,
    2.06156,      1.30591,      0.765749,     0.298668,    -0.218743,
    -0.50939,     -1.02022,     -1.35775,     -1.68037,    -1.90723,
    4247.65,      3960.91,      3693.5,       3444.11,     3211.53,
    2994.63,      2792.34,      2603.69,      2427.75,     2263.67,
    2147.22,      2132.84,      2117.42,      2100.88,     2083.15,
    2064.14,      2043.75,      2021.9,       1998.46,     1973.32,
    1946.37,      1917.48,      1886.49,      1853.27,     1817.64,
    1779.44,      1738.48,      1694.56,      1647.47,     1596.97,
    1542.82,      1484.76,      1422.5,       1355.75,     1284.17,
    1207.42,      1125.12,      1036.87,      942.245,     840.782,
    731.987,      615.33,       490.242,      356.115,     212.294,
    174.496,      167.444,      159.883,      151.775,     143.082,
    133.76,       123.764,      113.046,      101.553,     93.4232,
    88.8815,      84.0115,      78.7896,      73.1903,     67.1864,
    62.8115,      58.291,       53.4438,      48.2463,     44.8098,
    41.3221,      37.5824,      34.8819,      32.3477,     29.7204,
    27.2284,      24.6282,      23.3462,      21.8784,     19.7921,
    17.6814,      15.5546,      14.6334,      13.3154,     11.9766,
    10.9445,      9.83302,      8.69121,      7.5414,      6.55952,
    5.7124,       4.67706,      4.31249,      3.31074,     2.59825,
    2.06797,      1.41199,      0.998274,     0.341375,    0.0380651,
    -0.729779,    -0.973889,    -1.15334,     -1.81084,    -1.9691,
    4334.17,      4041.76,      3769.08,      3514.76,     3277.58,
    3056.39,      2850.11,      2657.73,      2478.31,     2310.99,
    2178.04,      2163.65,      2148.23,      2131.69,     2113.96,
    2094.94,      2074.55,      2052.69,      2029.24,     2004.1,
    1977.15,      1948.25,      1917.25,      1884.02,     1848.39,
    1810.18,      1769.21,      1725.28,      1678.18,     1627.67,
    1573.51,      1515.44,      1453.17,      1386.4,      1314.81,
    1238.04,      1155.72,      1067.46,      972.816,     871.333,
    762.515,      645.835,      520.721,      386.567,     242.717,
    189.878,      181.617,      172.76,       163.263,     153.08,
    142.161,      130.452,      117.897,      104.435,     94.1765,
    89.3578,      84.1909,      78.6506,      72.7099,     66.3399,
    62.2338,      58.4172,      54.3248,      49.9367,     46.6701,
    43.4945,      40.0894,      36.8718,      33.622,      30.2897,
    27.8221,      25.1761,      23.6531,      22.0893,     20.3504,
    18.6474,      17.0811,      15.3859,      13.7345,     12.2572,
    11.0803,      10.1235,      8.93179,      7.62564,     6.58506,
    5.81474,      4.98742,      4.42046,      3.45409,     2.82615,
    2.10316,      1.49523,      0.947953,     0.50076,     -0.171558,
    -0.502746,    -1.00897,     -1.26532,     -1.73944,    -2.08386,
    4441.69,      4142.05,      3862.6,       3601.98,     3358.93,
    3132.25,      2920.85,      2723.71,      2539.84,     2368.38,
    2218.38,      2203.82,      2188.21,      2171.47,     2153.52,
    2134.28,      2113.64,      2091.51,      2067.78,     2042.34,
    2015.06,      1985.81,      1954.44,      1920.81,     1884.74,
    1846.07,      1804.61,      1760.15,      1712.47,     1661.35,
    1606.54,      1547.76,      1484.74,      1417.16,     1344.7,
    1267.0,       1183.69,      1094.36,      998.568,     895.857,
    785.723,      667.629,      541.001,      405.223,     259.632,
    192.199,      184.494,      176.232,      167.374,     157.875,
    147.69,       136.769,      125.058,      112.501,     101.568,
    95.5692,      89.1366,      82.2391,      74.8432,     66.9128,
    62.0683,      58.2717,      54.2008,      49.8357,     46.6485,
    43.8248,      40.7972,      37.8333,      34.8553,     31.6895,
    29.3002,      26.7382,      24.6449,      22.5184,     20.8136,
    19.144,       17.6902,      15.8504,      14.224,      12.9961,
    11.6179,      10.4402,      8.97812,      8.01257,     7.21821,
    6.31265,      5.41409,      4.45139,      3.68464,     2.8935,
    2.37108,      1.72634,      1.03076,      0.63447,     -0.0326178,
    -0.512675,    -1.03393,     -1.17351,     -1.49686,    -1.94802,
    4483.69,      4181.63,      3899.93,      3637.21,     3392.2,
    3163.7,       2950.61,      2751.87,      2566.53,     2393.68,
    2232.48,      2214.05,      2198.56,      2181.96,     2164.15,
    2145.06,      2124.59,      2102.63,      2079.1,      2053.86,
    2026.79,      1997.78,      1966.66,      1933.29,     1897.52,
    1859.16,      1818.02,      1773.92,      1726.62,     1675.91,
    1621.54,      1563.23,      1500.71,      1433.67,     1361.79,
    1284.71,      1202.07,      1113.45,      1018.42,     916.531,
    807.277,      690.127,      564.51,       429.816,     285.388,
    203.895,      195.527,      186.554,      176.932,     166.616,
    155.553,      143.692,      130.973,      117.335,     104.917,
    99.8267,      94.3688,      88.5165,      82.2413,     75.5125,
    70.0089,      65.1321,      59.9028,      54.2957,     49.6857,
    45.6112,      41.2423,      38.003,       36.0437,     33.9428,
    31.3499,      28.5434,      25.9741,      23.3671,     21.6588,
    19.9374,      17.8228,      16.1568,      14.7379,     13.4439,
    12.2675,      10.9478,      9.74452,      8.55939,     7.41704,
    6.48828,      5.52752,      4.76845,      3.85826,     3.27741,
    2.66751,      1.56373,      1.1921,       0.56712,     0.0715917,
    -0.255409,    -0.918166,    -1.23707,     -1.53712,    -1.9435,
    4772.4,       4450.42,      4150.14,      3870.1,      3608.93,
    3365.37,      3138.22,      2926.38,      2728.82,     2544.57,
    2372.74,      2339.14,      2322.81,      2305.29,     2286.51,
    2266.37,      2244.77,      2221.62,      2196.79,     2170.17,
    2141.62,      2111.01,      2078.18,      2042.99,     2005.25,
    1964.78,      1921.39,      1874.87,      1824.98,     1771.49,
    1714.13,      1652.62,      1586.68,      1515.96,     1440.14,
    1358.83,      1271.65,      1178.17,      1077.93,     970.452,
    855.204,      731.628,      599.121,      457.039,     304.688,
    204.027,      195.94,       187.269,      177.971,     168.001,
    157.311,      145.848,      133.557,      120.377,     107.326,
    101.447,      95.1436,      88.3847,      81.1372,     73.3661,
    67.474,       63.3309,      58.8883,      54.1248,     50.3945,
    47.662,       44.732,       41.7267,      38.7154,     35.4864,
    32.162,       28.6257,      26.2279,      24.4052,     22.2543,
    20.0448,      19.1238,      17.3831,      15.2459,     13.6702,
    12.1212,      11.3653,      9.82835,      8.8127,      7.62508,
    6.6078,       5.74257,      4.97745,      4.01828,     3.05408,
    2.41142,      1.85812,      1.4128,       0.649412,    0.195075,
    -0.401848,    -0.777091,    -1.15875,     -1.66997,    -1.93478,
    5126.76,      4780.75,      4458.05,      4157.1,      3876.43,
    3614.69,      3370.58,      3142.93,      2930.61,     2732.61,
    2547.95,      2496.93,      2479.58,      2460.98,     2441.04,
    2419.65,      2396.72,      2372.13,      2345.77,     2317.5,
    2287.19,      2254.68,      2219.83,      2182.46,     2142.38,
    2099.42,      2053.34,      2003.94,      1950.96,     1894.16,
    1833.26,      1767.95,      1697.92,      1622.83,     1542.31,
    1455.98,      1363.41,      1264.14,      1157.71,     1043.58,
    921.204,      789.984,      649.281,      498.41,      336.636,
    213.787,      205.239,      196.073,      186.245,     175.706,
    164.406,      152.289,      139.297,      125.365,     110.688,
    105.215,      99.3472,      93.0549,      86.3078,     79.0731,
    72.9756,      68.6079,      63.9245,      58.9027,     54.5146,
    51.1552,      47.553,       43.64,        39.3256,     34.6993,
    32.1545,      30.3096,      28.4171,      26.4567,     23.9666,
    21.1089,      19.1858,      17.2604,      15.4298,     14.0111,
    12.8732,      11.5735,      10.2405,      9.17688,     8.12531,
    6.92203,      6.24052,      5.24261,      4.32959,     3.7023,
    2.7413,       2.04884,      1.54132,      0.867368,    0.38593,
    -0.221602,    -0.717593,    -1.11707,     -1.55847,    -2.03247,
    5107.71,      4763.11,      4441.74,      4142.04,     3862.53,
    3601.86,      3358.76,      3132.04,      2920.61,     2723.42,
    2539.52,      2473.76,      2456.71,      2438.44,     2418.84,
    2397.82,      2375.29,      2351.12,      2325.21,     2297.43,
    2267.64,      2235.7,       2201.45,      2164.72,     2125.34,
    2083.12,      2037.84,      1989.29,      1937.23,     1881.41,
    1821.56,      1757.38,      1688.56,      1614.77,     1535.65,
    1450.81,      1359.84,      1262.29,      1157.69,     1045.54,
    925.278,      796.327,      658.057,      509.794,     350.817,
    214.446,      206.315,      197.596,      188.248,     178.224,
    167.475,      155.95,       143.592,      130.341,     116.132,
    109.262,      102.54,       95.3328,      87.6044,     79.3174,
    72.277,       68.3223,      64.0818,      59.5349,     55.157,
    51.467,       47.5103,      43.5145,      40.1888,     36.6228,
    33.7521,      31.2193,      28.8361,      26.6719,     24.5365,
    22.3859,      20.5975,      18.8275,      17.067,      15.1667,
    13.7664,      12.2976,      10.7563,      9.71388,     8.50988,
    7.14389,      6.29603,      5.36022,      4.7005,      3.85843,
    2.91336,      2.14159,      1.54786,      1.04932,     0.277016,
    -0.168978,    -0.576441,    -1.24055,     -1.64526,    -1.98631,
    5360.54,      4998.88,      4661.58,      4347.03,     4053.67,
    3780.08,      3524.94,      3286.99,      3065.07,     2858.12,
    2665.1,       2580.58,      2562.94,      2544.02,     2523.74,
    2502.0,       2478.68,      2453.67,      2426.86,     2398.11,
    2367.29,      2334.23,      2298.79,      2260.79,     2220.04,
    2176.34,      2129.49,      2079.25,      2025.38,     1967.62,
    1905.68,      1839.27,      1768.06,      1691.7,      1609.83,
    1522.03,      1427.89,      1326.95,      1218.72,     1102.66,
    978.215,      844.777,      701.696,      548.274,     383.766,
    226.339,      217.451,      207.921,      197.703,     186.745,
    174.997,      162.399,      148.89,       134.405,     118.874,
    111.365,      105.164,      98.5143,      91.3843,     83.739,
    76.6105,      72.2388,      67.5512,      62.5248,     57.5268,
    53.4181,      49.0124,      44.5206,      41.5313,     38.326,
    35.2106,      32.1425,      29.2463,      26.8234,     24.5811,
    22.5678,      20.7592,      18.8703,      16.7769,     15.0822,
    13.7996,      12.3841,      11.097,       9.87446,     8.79598,
    7.96812,      6.77202,      5.62912,      4.83003,     3.89943,
    3.01121,      2.31861,      1.71906,      1.10822,     0.413303,
    -0.0705126,   -0.595992,    -1.15255,     -1.56832,    -2.08002,
    5510.03,      5138.16,      4791.35,      4467.92,     4166.28,
    3884.98,      3622.63,      3377.96,      3149.78,     2936.99,
    2738.53,      2636.12,      2618.19,      2598.96,     2578.34,
    2556.24,      2532.53,      2507.11,      2479.85,     2450.63,
    2419.29,      2385.69,      2349.66,      2311.02,     2269.59,
    2225.17,      2177.54,      2126.47,      2071.7,      2012.98,
    1950.02,      1882.5,       1810.11,      1732.48,     1649.24,
    1559.99,      1464.29,      1361.67,      1251.63,     1133.65,
    1007.14,      871.48,       726.021,      570.05,      402.808,
    226.274,      217.685,      208.474,      198.598,     188.008,
    176.653,      164.478,      151.422,      137.423,     122.413,
    114.125,      108.036,      101.507,      94.5061,     86.9994,
    79.4813,      74.3247,      68.7954,      62.8666,     56.9976,
    53.542,       49.8367,      45.8807,      42.4013,     38.6705,
    35.4204,      32.8615,      30.3901,      28.4648,     26.2706,
    23.7092,      21.8008,      19.9838,      17.6761,     16.1822,
    14.5061,      13.1568,      11.8959,      10.5527,     9.2752,
    8.00594,      6.95586,      6.23268,      5.00279,     4.22586,
    3.22848,      2.61168,      1.97793,      1.13337,     0.50216,
    -0.00166266,  -0.661612,    -1.28401,     -1.57,       -1.77019,
    5556.94,      5182.23,      4832.81,      4506.93,     4203.0,
    3919.57,      3655.23,      3408.71,      3178.8,      2964.39,
    2764.43,      2645.09,      2627.34,      2608.3,      2587.89,
    2566.0,       2542.53,      2517.37,      2490.38,     2461.45,
    2430.42,      2397.16,      2361.49,      2323.24,     2282.22,
    2238.25,      2191.09,      2140.53,      2086.31,     2028.18,
    1965.84,      1899.0,       1827.33,      1750.48,     1668.07,
    1579.71,      1484.97,      1383.37,      1274.44,     1157.63,
    1032.39,      898.087,      754.082,      599.671,     434.1,
    256.565,      233.221,      222.661,      211.337,     199.195,
    186.176,      172.215,      157.246,      141.195,     123.984,
    114.321,      108.863,      103.011,      96.736,      90.0074,
    82.9616,      77.876,       72.4229,      66.5757,     60.4548,
    56.3832,      52.0174,      47.3361,      44.2591,     41.1119,
    37.9551,      34.9659,      31.8743,      29.0698,     26.3199,
    23.9973,      22.146,       20.5199,      18.5807,     17.0919,
    15.44,        13.4836,      11.8901,      11.0351,     9.76448,
    8.49961,      7.17752,      6.49349,      5.28987,     4.26485,
    3.46774,      2.68871,      1.92962,      1.21359,     0.676525,
    0.125301,     -0.401927,    -1.1302,      -1.78728,    -1.82503,
    5838.57,      5444.74,      5077.44,      4734.91,     4415.46,
    4117.53,      3839.69,      3580.58,      3338.92,     3113.56,
    2903.38,      2762.17,      2743.67,      2723.84,     2702.57,
    2679.77,      2655.32,      2629.1,       2600.99,     2570.84,
    2538.52,      2503.86,      2466.69,      2426.84,     2384.11,
    2338.29,      2289.16,      2236.49,      2180.0,      2119.43,
    2054.48,      1984.84,      1910.17,      1830.1,      1744.25,
    1652.19,      1553.48,      1447.63,      1334.14,     1212.44,
    1081.95,      942.029,      791.995,      631.119,     458.617,
    273.649,      233.499,      223.912,      213.633,     202.61,
    190.791,      178.118,      164.529,      149.958,     134.334,
    123.69,       117.299,      110.446,      103.098,     95.2189,
    86.7705,      81.1903,      75.3037,      68.9917,     62.2236,
    57.9333,      53.4415,      48.6252,      45.398,      42.3275,
    39.0031,      35.3497,      31.619,       29.463,      27.2539,
    25.2951,      23.5352,      21.9267,      19.6389,     17.6287,
    15.9143,      14.3292,      12.6176,      11.0505,     9.81722,
    8.62791,      7.10195,      6.56437,      5.49621,     4.65844,
    3.755,        2.94412,      2.04038,      1.47209,     0.868627,
    0.0645113,    -0.45563,     -1.12216,     -1.5433,     -2.01124,
    6392.52,      5960.73,      5558.03,      5182.5,      4832.27,
    4505.63,      4201.02,      3916.93,      3651.99,     3404.91,
    3174.48,      3002.61,      2982.53,      2961.0,      2937.91,
    2913.15,      2886.6,       2858.14,      2827.61,     2794.88,
    2759.79,      2722.16,      2681.81,      2638.54,     2592.15,
    2542.4,       2489.06,      2431.87,      2370.54,     2304.78,
    2234.27,      2158.66,      2077.58,      1990.65,     1897.44,
    1797.49,      1690.31,      1575.4,       1452.17,     1320.04,
    1178.37,      1026.45,      863.557,      688.891,     501.602,
    300.779,      239.054,      229.175,      218.581,     207.222,
    195.042,      181.981,      167.977,      152.961,     136.859,
    125.099,      119.03,       112.522,      105.544,     98.0621,
    90.0393,      84.233,       78.3946,      72.1343,     65.4216,
    60.8959,      56.4418,      51.6659,      47.5274,     43.4372,
    39.378,       36.5767,      33.5758,      31.3497,     28.9008,
    25.7778,      23.3507,      21.9347,      20.2027,     18.1407,
    16.1899,      14.801,       13.2798,      11.7946,     10.2434,
    9.04657,      8.19022,      6.65573,      5.91431,     4.64465,
    3.9094,       2.94689,      2.45486,      1.50383,     0.812937,
    0.113649,     -0.24141,     -0.993847,    -1.45964,    -1.93814,
    6275.06,      5851.77,      5457.02,      5088.87,     4745.53,
    4425.32,      4126.71,      3848.22,      3588.5,      3346.28,
    3120.39,      2933.95,      2914.64,      2893.92,     2871.71,
    2847.9,       2822.36,      2794.98,      2765.62,     2734.14,
    2700.38,      2664.19,      2625.38,      2583.76,     2539.14,
    2491.29,      2439.98,      2384.97,      2325.97,     2262.72,
    2194.9,       2122.17,      2044.19,      1960.57,     1870.91,
    1774.77,      1671.68,      1561.15,      1442.62,     1315.53,
    1179.25,      1033.13,      876.445,      708.438,     528.289,
    335.122,      256.825,      245.56,       233.481,     220.53,
    206.642,      191.751,      175.784,      158.663,     140.304,
    126.006,      119.994,      113.547,      106.634,     99.2223,
    91.2745,      85.4837,      80.0228,      74.1672,     67.8885,
    62.9128,      58.0822,      52.9026,      48.9618,     45.6189,
    42.0449,      38.3247,      34.3356,      31.6909,     29.0357,
    26.9437,      24.7361,      22.4359,      20.4617,     18.5153,
    16.7405,      15.4071,      13.615,       11.9712,     10.9778,
    9.24627,      8.32763,      6.95075,      5.4869,      5.13301,
    4.07285,      3.09816,      2.324,        1.65856,     0.983961,
    0.178156,     -0.300054,    -1.01621,     -1.4652,     -1.88325,
    6670.4,       6220.23,      5800.39,      5408.84,     5043.7,
    4703.15,      4385.57,      4089.39,      3813.16,     3555.56,
    3315.32,      3099.43,      3079.09,      3057.28,     3033.9,
    3008.82,      2981.93,      2953.1,       2922.18,     2889.03,
    2853.49,      2815.37,      2774.51,      2730.68,     2683.69,
    2633.31,      2579.28,      2521.35,      2459.24,     2392.63,
    2321.21,      2244.63,      2162.52,      2074.47,     1980.05,
    1878.82,      1770.27,      1653.87,      1529.07,     1395.24,
    1251.74,      1097.87,      932.885,      755.974,     566.278,
    362.873,      262.462,      251.859,      240.49,      228.299,
    215.227,      201.21,       186.181,      170.065,     152.784,
    137.568,      129.912,      121.703,      112.9,       103.461,
    93.3404,      86.6394,      81.2954,      75.5653,     69.421,
    64.6261,      60.3106,      55.6832,      51.4819,     47.5866,
    43.4099,      40.279,       36.9259,      33.667,      30.2484,
    28.2284,      26.069,       23.4166,      21.427,      19.6603,
    17.8488,      15.6352,      14.0544,      12.2993,     10.8324,
    9.89155,      8.6195,       7.36708,      6.45392,     5.5134,
    4.10034,      3.31911,      2.57739,      1.62414,     1.16692,
    0.347749,     -0.416624,    -0.800145,    -1.34979,    -1.98583,
    6596.23,      6151.45,      5736.68,      5349.85,     4989.1,
    4652.66,      4338.89,      4046.27,      3773.37,     3518.87,
    3281.52,      3060.16,      3030.67,      3009.42,     2986.63,
    2962.19,      2935.99,      2907.9,       2877.78,     2845.47,
    2810.84,      2773.7,       2733.88,      2691.18,     2645.39,
    2596.3,       2543.65,      2487.21,      2426.68,     2361.78,
    2292.19,      2217.57,      2137.56,      2051.76,     1959.77,
    1861.12,      1755.35,      1641.93,      1520.32,     1389.92,
    1250.1,       1100.17,      939.404,      767.022,     582.183,
    383.986,      267.774,      257.177,      245.813,     233.629,
    220.564,      206.554,      191.533,      175.425,     158.154,
    141.939,      134.484,      126.49,       117.919,     108.729,
    98.8738,      91.3247,      85.2276,      78.6898,     71.6796,
    66.0513,      61.3089,      56.2236,      51.6174,     47.6529,
    43.402,       39.9927,      36.459,       33.7693,     31.3039,
    28.7179,      26.1528,      24.3705,      22.64,       20.5898,
    17.9001,      16.2055,      14.7378,      13.8327,     11.0484,
    10.0333,      9.35901,      7.78995,      6.40587,     5.34363,
    4.59023,      3.26621,      2.61098,      1.76012,     1.25679,
    0.479506,     -0.404205,    -0.913036,    -1.26872,    -1.86103,
    7379.66,      6881.25,      6416.42,      5982.92,     5578.62,
    5201.59,      4849.96,      4522.04,      4216.21,     3930.99,
    3665.0,       3416.94,      3365.22,      3341.65,     3316.38,
    3289.29,      3260.24,      3229.09,      3195.69,     3159.87,
    3121.47,      3080.29,      3036.13,      2988.79,     2938.02,
    2883.58,      2825.21,      2762.62,      2695.51,     2623.55,
    2546.38,      2463.64,      2374.93,      2279.8,      2177.79,
    2068.41,      1951.13,      1825.38,      1690.53,     1545.94,
    1390.91,      1224.66,      1046.41,      855.269,     650.318,
    430.555,      281.46,       269.641,      256.969,     243.38,
    228.809,      213.186,      196.433,      178.47,      159.208,
    140.081,      133.154,      125.727,      117.763,     109.224,
    100.068,      92.8579,      87.6151,      81.9934,     75.9655,
    70.1347,      64.5057,      58.47,        53.0898,     49.1346,
    44.8936,      42.0785,      39.4603,      36.3243,     32.7716,
    30.2094,      27.8535,      24.9637,      22.492,      20.4009,
    18.5198,      16.4507,      15.4094,      13.427,      11.9134,
    10.3704,      9.3319,       7.85271,      6.7129,      5.76263,
    4.53913,      3.90221,      2.86023,      2.0772,      1.29066,
    0.409527,     -0.147126,    -0.77839,     -1.36033,    -1.9204,
    7308.32,      6815.07,      6355.09,      5926.1,      5526.01,
    5152.91,      4804.94,      4480.43,      4177.78,     3895.54,
    3632.31,      3386.83,      3316.8,       3293.77,     3269.08,
    3242.59,      3214.2,       3183.75,      3151.11,     3116.1,
    3078.56,      3038.31,      2995.15,      2948.88,     2899.26,
    2846.05,      2789.0,       2727.82,      2662.23,     2591.89,
    2516.47,      2435.6,       2348.89,      2255.91,     2156.21,
    2049.3,       1934.67,      1811.75,      1679.96,     1538.63,
    1387.1,       1224.61,      1050.38,      863.562,     663.241,
    448.443,      283.077,      271.937,      259.992,     247.183,
    233.449,      218.722,      202.931,      185.999,     167.843,
    148.578,      141.499,      133.908,      125.768,     117.041,
    107.682,      98.998,       91.5289,      83.52,       74.9324,
    67.944,       63.5784,      58.8972,      54.3034,     50.4307,
    46.2783,      42.7505,      39.3244,      36.1282,     33.1011,
    30.606,       28.3239,      25.8282,      23.5912,     21.6377,
    19.4968,      18.1493,      16.1015,      14.2841,     12.5341,
    11.3963,      9.71376,      8.38323,      6.83312,     6.03519,
    4.65163,      4.07919,      3.00666,      2.11371,     1.35815,
    0.458478,     -0.00814913,  -0.651986,    -1.2435,     -1.83065,
    7204.68,      6718.78,      6265.58,      5842.95,     5448.8,
    5081.22,      4738.4,       4418.69,      4120.54,     3842.47,
    3583.15,      3341.3,       3254.16,      3231.73,     3207.68,
    3181.89,      3154.24,      3124.59,      3092.8,      3058.71,
    3022.16,      2982.97,      2940.94,      2895.88,     2847.56,
    2795.74,      2740.19,      2680.61,      2616.74,     2548.25,
    2474.8,       2396.05,      2311.61,      2221.07,     2123.98,
    2019.88,      1908.25,      1788.56,      1660.21,     1522.59,
    1375.03,      1216.8,       1047.14,      865.216,     670.146,
    460.978,      280.764,      270.367,      259.218,     247.264,
    234.446,      220.701,      205.963,      190.16,      173.215,
    155.046,      146.032,      137.222,      127.775,     117.645,
    106.783,      97.5149,      92.2906,      86.6887,     80.682,
    74.6951,      69.2078,      63.3238,      57.5176,     53.2918,
    48.7605,      45.2287,      42.2138,      38.8549,     35.1026,
    32.0127,      29.4126,      26.9514,      24.6139,     22.5014,
    20.2453,      18.0948,      16.1146,      14.5431,     12.8805,
    11.7642,      9.93636,      8.86592,      7.62447,     5.99097,
    5.37311,      4.20652,      3.37776,      2.38121,     1.4959,
    0.511707,     0.162694,     -0.752669,    -1.37721,    -1.70329,
    7758.72,      7234.95,      6746.51,      6290.97,     5866.14,
    5469.93,      5100.43,      4755.84,      4434.47,     4134.76,
    3855.25,      3594.58,      3481.08,      3457.23,     3431.65,
    3404.22,      3374.81,      3343.27,      3309.46,     3273.2,
    3234.32,      3192.64,      3147.93,      3100.0,      3048.61,
    2993.5,       2934.41,      2871.05,      2803.11,     2730.26,
    2652.14,      2568.38,      2478.57,      2382.26,     2279.0,
    2168.27,      2049.54,      1922.24,      1785.73,     1639.35,
    1482.4,       1314.11,      1133.65,      940.154,     732.673,
    510.198,      297.935,      286.414,      274.06,      260.814,
    246.61,       231.38,       215.05,       197.539,     178.762,
    158.629,      148.235,      139.314,      129.747,     119.49,
    108.491,      98.3648,      92.5428,      86.3001,     79.6063,
    73.157,       68.5361,      63.5812,      58.4203,     54.0538,
    49.3718,      45.6177,      42.655,       39.6675,     36.789,
    33.6235,      30.1434,      27.4741,      25.0396,     22.8269,
    20.915,       18.9269,      16.7088,      14.8424,     13.326,
    11.7296,      10.5319,      8.97387,      7.70783,     6.83281,
    5.33025,      4.40976,      3.39611,      2.5777,      1.73518,
    0.890384,     -0.0447498,   -0.758673,    -1.19026,    -1.8715,
    7580.12,      7068.95,      6592.21,      6147.63,     5732.99,
    5346.3,       4985.68,      4649.36,      4335.7,      4043.19,
    3770.4,       3515.98,      3386.23,      3363.26,     3338.63,
    3312.22,      3283.89,      3253.53,      3220.96,     3186.05,
    3148.61,      3108.47,      3065.42,      3019.26,     2969.77,
    2916.7,       2859.8,       2798.78,      2733.36,     2663.2,
    2587.98,      2507.32,      2420.83,      2328.09,     2228.65,
    2122.03,      2007.69,      1885.1,       1753.64,     1612.69,
    1461.55,      1299.48,      1125.71,      939.375,     739.575,
    525.336,      301.188,      289.808,      277.606,     264.521,
    250.491,      235.448,      219.317,      202.02,      183.473,
    163.586,      152.083,      143.108,      133.484,     123.164,
    112.098,      101.409,      95.9762,      90.1508,     83.9044,
    77.4314,      71.7121,      65.5795,      59.1175,     55.8108,
    52.2651,      48.1889,      43.4914,      39.3772,     37.3161,
    34.7127,      31.3112,      28.1674,      25.3068,     23.3011,
    20.4637,      18.9496,      17.3689,      15.3485,     13.9462,
    12.5641,      11.0019,      9.58133,      7.9794,      6.67971,
    5.71431,      4.62137,      3.61073,      2.73358,     2.15091,
    0.7868,       0.365722,     -0.561447,    -1.0076,     -1.66661,
    8161.14,      7610.38,      7096.77,      6617.75,     6171.03,
    5754.39,      5365.85,      5003.5,       4665.57,     4350.41,
    4056.48,      3782.38,      3622.46,      3598.03,     3571.83,
    3543.73,      3513.6,       3481.3,       3446.66,     3409.52,
    3369.69,      3326.99,      3281.2,       3232.1,      3179.45,
    3123.0,       3062.47,      2997.56,      2927.97,     2853.34,
    2773.32,      2687.52,      2595.52,      2496.87,     2391.08,
    2277.66,      2156.04,      2025.63,      1885.79,     1735.85,
    1575.07,      1402.68,      1217.82,      1019.61,     807.069,
    579.171,      334.804,      307.441,      294.214,     280.031,
    264.823,      248.516,      231.031,      212.282,     192.178,
    170.622,      157.221,      147.921,      137.948,     127.255,
    115.789,      104.024,      97.6128,      90.7378,     83.366,
    75.7863,      71.9134,      67.7606,      63.3076,     58.4838,
    53.3085,      48.8306,      45.853,       42.4618,     38.0156,
    34.2035,      32.2815,      29.2239,      25.758,      24.2624,
    21.9233,      19.9289,      18.2372,      16.1958,     14.4337,
    12.9394,      11.027,       9.97212,      8.49358,     6.96951,
    5.99904,      4.89,         3.72153,      2.77166,     1.83502,
    1.12487,      0.339727,     -0.491037,    -1.1052,     -1.70282};

/// dSigma/dtheta for π+ + π+ -> π+ + π+ + γ or π- + π- -> π- + π- + γ
const std::initializer_list<double> BREMS_PIPI_PIPI_SAME_DIFF_SIG_THETA = {
    0.0,          1.85249e-05,  3.97272e-05,  6.58032e-05,  9.81839e-05,
    0.000138641,  0.000186201,  0.000244324,  0.0003192,    0.00039294,
    0.000486333,  0.000570778,  0.000665129,  0.00077228,   0.00086821,
    0.000963126,  0.00105114,   0.00113714,   0.00120613,   0.00125426,
    0.00130265,   0.00133459,   0.00133947,   0.00133473,   0.00132697,
    0.00127379,   0.00124744,   0.00118407,   0.00112514,   0.00106095,
    0.000976254,  0.000895428,  0.000818215,  0.000748558,  0.000689036,
    0.000630762,  0.000565561,  0.000535489,  0.000499762,  0.000494456,
    0.000482595,  0.000510754,  0.000528565,  0.000575861,  0.000610571,
    0.00069162,   0.000747312,  0.000819264,  0.000897701,  0.000979885,
    0.00105715,   0.00112864,   0.00117413,   0.00124817,   0.00128964,
    0.00133141,   0.00135425,   0.00134499,   0.00134494,   0.00130512,
    0.00126857,   0.00119751,   0.0011262,    0.00105688,   0.000950187,
    0.000861468,  0.0007667,    0.000657196,  0.000573126,  0.000484464,
    0.000392377,  0.000315639,  0.00024914,   0.000186287,  0.000139745,
    9.95869e-05,  6.5573e-05,   3.942e-05,    1.8583e-05,   1.21299e-09,
    0.0,          4.74749e-05,  0.000100796,  0.000165927,  0.000244744,
    0.000346329,  0.000464962,  0.000613701,  0.000788281,  0.000976857,
    0.00120167,   0.00141038,   0.00162856,   0.00190363,   0.00209417,
    0.00232986,   0.00253423,   0.00275903,   0.00289501,   0.00301932,
    0.00312615,   0.00317415,   0.00319537,   0.00318777,   0.003127,
    0.00306889,   0.00291585,   0.00279365,   0.00262299,   0.0024661,
    0.00228639,   0.0021384,    0.00197598,   0.00179572,   0.00165134,
    0.00153119,   0.00141912,   0.00131065,   0.0012579,    0.00122772,
    0.00124257,   0.00127723,   0.00132039,   0.0013976,    0.00150815,
    0.00162667,   0.00179778,   0.00197598,   0.00216333,   0.00230124,
    0.0025056,    0.00263407,   0.00276258,   0.00295801,   0.00303502,
    0.00313152,   0.00319043,   0.00318222,   0.00315716,   0.0030621,
    0.00297493,   0.00286664,   0.00271775,   0.00250558,   0.00233304,
    0.0021083,    0.0018763,    0.00166698,   0.00142768,   0.00119897,
    0.000999002,  0.000791036,  0.000604709,  0.000463808,  0.000346238,
    0.000245845,  0.000165163,  0.000100036,  4.65993e-05,  3.07045e-09,
    0.0,          9.13739e-05,  0.000194646,  0.000323743,  0.000488005,
    0.000686687,  0.000926822,  0.00120879,   0.0015465,    0.00195227,
    0.00233158,   0.00277613,   0.00318437,   0.00359926,   0.00408504,
    0.00456412,   0.00483985,   0.00525162,   0.0055126,    0.0057325,
    0.00589101,   0.00594356,   0.00601756,   0.00596077,   0.00583519,
    0.00576243,   0.0054783,    0.00523017,   0.00494588,   0.00464803,
    0.00430092,   0.00399943,   0.00371887,   0.00340248,   0.0031071,
    0.00285613,   0.00273044,   0.00257096,   0.002467,     0.0023835,
    0.00237506,   0.00242925,   0.00255256,   0.00269708,   0.00289778,
    0.00311216,   0.00335856,   0.00366641,   0.00402204,   0.00431916,
    0.0046314,    0.00492286,   0.00525104,   0.00542934,   0.00567003,
    0.00579083,   0.00589962,   0.00596764,   0.00593676,   0.00593157,
    0.00574503,   0.00548807,   0.00520082,   0.00488777,   0.0044912,
    0.00410393,   0.00369201,   0.00321452,   0.00275269,   0.00233569,
    0.00191958,   0.0015746,    0.00122676,   0.000922823,  0.000689564,
    0.000472343,  0.000320256,  0.000195833,  9.17627e-05,  5.98183e-09,
    0.0,          0.00015422,   0.000329788,  0.000547788,  0.000830882,
    0.00115766,   0.00157986,   0.0021056,    0.00264801,   0.00325247,
    0.00402069,   0.00467645,   0.0054601,    0.00616584,   0.00688972,
    0.00759466,   0.00809255,   0.00869244,   0.00904516,   0.00943119,
    0.00965312,   0.00978771,   0.00973455,   0.00971751,   0.00947396,
    0.009278,     0.00893874,   0.00854602,   0.00795752,   0.0075396,
    0.00700718,   0.00658887,   0.00602323,   0.00560695,   0.00515156,
    0.00482243,   0.00449658,   0.00428497,   0.00410691,   0.0040174,
    0.00407809,   0.00413773,   0.0042726,    0.00449716,   0.00485168,
    0.00523938,   0.00561341,   0.00598797,   0.0065654,    0.00706634,
    0.00748401,   0.00800616,   0.00847567,   0.00868588,   0.00935791,
    0.00947902,   0.00978173,   0.009849,     0.00974475,   0.00958162,
    0.00952419,   0.00912535,   0.00868483,   0.00812794,   0.00760982,
    0.00692575,   0.00619723,   0.00545811,   0.00471684,   0.00399888,
    0.00331132,   0.00264765,   0.00207884,   0.00157972,   0.0011797,
    0.000816807,  0.000553811,  0.000330515,  0.000157084,  1.01256e-08,
    0.0,          0.000239776,  0.000517617,  0.000863395,  0.00129154,
    0.0018184,    0.00248652,   0.00328938,   0.00411176,   0.00516494,
    0.00621212,   0.00729204,   0.00845969,   0.0095639,    0.0107479,
    0.011565,     0.0124928,    0.0132822,    0.013789,     0.0144212,
    0.014442,     0.0147577,    0.0146096,    0.0148343,    0.0142194,
    0.0138932,    0.0132855,    0.0126002,    0.011861,     0.0113596,
    0.0106474,    0.00971498,   0.00905332,   0.00835652,   0.00773205,
    0.00722026,   0.00688888,   0.00654495,   0.00635199,   0.00612989,
    0.00627487,   0.00631948,   0.00647087,   0.00693099,   0.0073294,
    0.00775486,   0.00838358,   0.00907258,   0.00976904,   0.0104042,
    0.0112653,    0.0120057,    0.0126581,    0.0133818,    0.0136789,
    0.014191,     0.0146117,    0.0145564,    0.0145307,    0.0147088,
    0.0144107,    0.0138272,    0.0132224,    0.0125862,    0.0115751,
    0.0106143,    0.009631,     0.00834577,   0.00730958,   0.0061834,
    0.00518138,   0.00416731,   0.00332786,   0.00250552,   0.00184795,
    0.00127233,   0.000855583,  0.000516856,  0.000240107,  1.58578e-08,
    0.0,          0.00035069,   0.00074253,   0.00124457,   0.00184947,
    0.0026476,    0.00360262,   0.0048004,    0.00603673,   0.0074601,
    0.00920991,   0.0107642,    0.0123338,    0.0138915,    0.0153881,
    0.0167811,    0.0179161,    0.0189148,    0.0200719,    0.0202392,
    0.0206365,    0.0207983,    0.0206145,    0.0203221,    0.0196647,
    0.019383,     0.0184713,    0.0175896,    0.0167027,    0.0157387,
    0.0147586,    0.0137962,    0.0127568,    0.0120344,    0.0112684,
    0.0105075,    0.00985521,   0.00955881,   0.00909134,   0.00886456,
    0.0089544,    0.00906875,   0.00961882,   0.00990142,   0.0103647,
    0.0110508,    0.0118889,    0.0127873,    0.01375,      0.0145897,
    0.0156817,    0.0165499,    0.017784,     0.0185354,    0.0194025,
    0.0199464,    0.0203194,    0.0206013,    0.0207984,    0.0207012,
    0.0202806,    0.0199625,    0.0189718,    0.0176495,    0.0165149,
    0.0152045,    0.0137957,    0.0122847,    0.0107793,    0.00900284,
    0.0076183,    0.0060887,    0.00483595,   0.00364456,   0.00264458,
    0.0018834,    0.0012118,    0.00072814,   0.000344413,  2.25237e-08,
    0.0,          0.000472515,  0.00102771,   0.00170269,   0.00260535,
    0.00372387,   0.00510687,   0.00668406,   0.00841575,   0.0104448,
    0.01248,      0.0147291,    0.0168122,    0.0191898,    0.0213061,
    0.0230455,    0.0245603,    0.025907,     0.0271311,    0.027501,
    0.0279735,    0.0280161,    0.0277512,    0.0273596,    0.0265299,
    0.0258525,    0.0246912,    0.0238539,    0.0224068,    0.0210664,
    0.0198966,    0.0182645,    0.0176564,    0.0159869,    0.0150212,
    0.0141966,    0.0134082,    0.0126529,    0.0124913,    0.0123855,
    0.0120968,    0.0121969,    0.0127892,    0.0133332,    0.0140023,
    0.0149479,    0.0159396,    0.0173136,    0.0184333,    0.0197488,
    0.0207505,    0.0223896,    0.0236118,    0.0248224,    0.025872,
    0.0264631,    0.0272001,    0.0278859,    0.0280425,    0.0279329,
    0.0274072,    0.026535,     0.0256888,    0.0246137,    0.0229052,
    0.021068,     0.0190074,    0.0169223,    0.0147117,    0.0125502,
    0.0105504,    0.0085091,    0.00669602,   0.00508603,   0.00367454,
    0.00260863,   0.0016863,    0.00102497,   0.000476521,  3.06452e-08,
    0.0,          0.000631832,  0.00136588,  