import os.path
from unittest import TestCase
from unittest.case import SkipTest

import numpy as np

from aspire.nufft import Plan, backend_available
from aspire.utils.types import complex_type, utest_tolerance

DATA_DIR = os.path.join(os.path.dirname(__file__), "saved_test_data")


class SimTestCase(TestCase):
    def setUp(self):
        self.fourier_pts = np.array(
            [
                [
                    0.88952655922411,
                    0.35922344760724,
                    -0.17107966400962,
                    -0.70138277562649,
                ],
                [
                    1.87089316522016,
                    1.99362869011803,
                    2.11636421501590,
                    2.23909973991377,
                ],
                [
                    -3.93035749861843,
                    -3.36417300942290,
                    -2.79798852022738,
                    -2.23180403103185,
                ],
            ]
        )

        # Note this data was stored as order=F
        self.vol = np.load(os.path.join(DATA_DIR, "nfft_volume.npy"))

        # Setup a 2D slice for testing 2d many
        self.plane = self.vol[0]
        self.vol = self.vol

        self.recip_space = np.array(
            [
                -0.05646675 + 1.503746j,
                1.677600 + 0.6610926j,
                0.9124417 - 0.7394574j,
                -0.9136836 - 0.5491410j,
            ]
        )

        self.recip_space_plane = np.array(
            [
                -0.14933796 + 0.00324942j,
                -0.29726508 + 0.07601329j,
                -0.26276074 + 0.12634184j,
                -0.09722212 + 0.12028122j,
            ],
            dtype=np.complex128,
        )

        self.adjoint_vol = np.array(
            [
                [
                    [
                        9.59495207e-01 - 1.06291322e00j,
                        4.96001394e-01 + 1.60922425e00j,
                        -2.78391527e00 - 2.18820002e00j,
                        4.53910627e00 + 2.52530574e00j,
                        -4.67770008e00 - 2.29928671e00j,
                        3.12346140e00 + 1.42107788e00j,
                        -8.35006613e-01 - 2.08392594e-01j,
                        -8.61811880e-01 - 7.44775341e-01j,
                    ],
                    [
                        6.95609379e-01 + 1.21528480e00j,
                        -1.98693653e00 + 6.99805141e-02j,
                        3.52261924e00 - 1.86188461e00j,
                        -4.40227232e00 + 3.06290826e00j,
                        3.95080650e00 - 2.99804116e00j,
                        -2.24070507e00 + 1.82530148e00j,
                        1.33509882e-01 - 3.53690847e-01j,
                        1.24778990e00 - 5.47387355e-01j,
                    ],
                    [
                        -1.33450724e00 + 3.51172971e-01j,
                        6.52415482e-01 - 2.28012935e00j,
                        2.41584016e-01 + 4.38776505e00j,
                        -8.09711177e-01 - 5.38236472e00j,
                        8.45459714e-01 + 4.54628775e00j,
                        -5.28057500e-01 - 2.26402471e00j,
                        1.96011929e-01 - 1.93622742e-01j,
                        -7.29420741e-02 + 1.50428070e00j,
                    ],
                    [
                        -5.67875578e-02 - 1.41461172e00j,
                        2.25953038e00 + 1.66189272e00j,
                        -4.40529657e00 - 1.78799885e00j,
                        5.16833043e00 + 1.66557527e00j,
                        -4.04985184e00 - 1.17019140e00j,
                        1.71256643e00 + 3.71267910e-01j,
                        4.75782819e-01 + 4.08631004e-01j,
                        -1.36051666e00 - 7.80558742e-01j,
                    ],
                    [
                        1.52683244e00 + 2.54429643e-01j,
                        -2.77915719e00 + 1.70545912e00j,
                        3.73687261e00 - 3.39672886e00j,
                        -3.72370373e00 + 3.80423857e00j,
                        2.54051450e00 - 2.75141067e00j,
                        -6.99543332e-01 + 9.63394139e-01j,
                        -8.40657096e-01 + 4.72767394e-01j,
                        1.30811192e00 - 8.36125573e-01j,
                    ],
                    [
                        -6.96113136e-01 + 1.64956662e00j,
                        -5.26556786e-01 - 3.67785088e00j,
                        1.48197487e00 + 5.05701967e00j,
                        -1.66956394e00 - 4.84516408e00j,
                        1.14523279e00 + 3.01572872e00j,
                        -3.84547327e-01 - 5.39120190e-01j,
                        -1.06716909e-01 - 1.23019252e00j,
                        1.13925960e-01 + 1.45913141e00j,
                    ],
                    [
                        -1.65131058e00 - 1.34345510e00j,
                        3.98453418e00 + 1.15295791e00j,
                        -5.32538828e00 - 9.22639650e-01j,
                        4.79627574e00 + 6.42693834e-01j,
                        -2.66107919e00 - 2.54914428e-01j,
                        1.57096046e-01 - 1.91604893e-01j,
                        1.33309475e00 + 5.11125022e-01j,
                        -1.19311268e00 - 5.41755365e-01j,
                    ],
                    [
                        2.15238025e00 - 1.33910059e00j,
                        -2.98313181e00 + 3.42296122e00j,
                        3.21543283e00 - 4.39927175e00j,
                        -2.52080224e00 + 3.69649112e00j,
                        1.09283808e00 - 1.79697223e00j,
                        3.80379526e-01 - 1.31632250e-01j,
                        -1.13883648e00 + 1.03630617e00j,
                        9.00187963e-01 - 6.40239871e-01j,
                    ],
                ],
                [
                    [
                        1.00527776e00 - 5.13762619e-01j,
                        -1.06516046e00 + 9.44749860e-01j,
                        -3.74638887e-01 - 1.72273382e00j,
                        2.55919554e00 + 2.63463313e00j,
                        -4.17049264e00 - 3.19294288e00j,
                        4.23169231e00 + 2.93055160e00j,
                        -2.74981604e00 - 1.78183369e00j,
                        6.69091601e-01 + 2.36886467e-01j,
                    ],
                    [
                        -4.43494995e-03 + 1.28361760e00j,
                        -5.72958286e-01 - 1.26784927e00j,
                        2.08902195e00 + 1.35249726e-01j,
                        -3.86184881e00 + 1.40214629e00j,
                        4.83226894e00 - 2.40070878e00j,
                        -4.27330560e00 + 2.33513458e00j,
                        2.34288389e00 - 1.39461706e00j,
                        -5.45623939e-02 + 2.67186433e-01j,
                    ],
                    [
                        -1.24973045e00 - 5.99696381e-01j,
                        1.36088033e00 - 2.43608722e-01j,
                        -9.52580668e-01 + 2.30536579e00j,
                        3.86913171e-01 - 4.48884773e00j,
                        9.39514192e-03 + 5.44887089e00j,
                        -1.57026838e-01 - 4.50913493e00j,
                        1.83309097e-01 + 2.14961395e00j,
                        -2.27116838e-01 + 2.90782995e-01j,
                    ],
                    [
                        1.08834945e00 - 9.36268763e-01j,
                        -4.64172267e-02 + 1.44241240e00j,
                        -2.13412256e00 - 2.01769468e00j,
                        4.18583978e00 + 2.44571181e00j,
                        -4.84645323e00 - 2.40092266e00j,
                        3.71171852e00 + 1.70306561e00j,
                        -1.48897402e00 - 5.46694434e-01j,
                        -5.04690873e-01 - 5.29476600e-01j,
                    ],
                    [
                        4.70487024e-01 + 1.35942338e00j,
                        -1.56067010e00 - 3.85532107e-01j,
                        3.09971530e00 - 1.38419516e00j,
                        -4.24832288e00 + 2.86210374e00j,
                        4.20716570e00 - 3.18144563e00j,
                        -2.80462870e00 + 2.25309584e00j,
                        7.03656405e-01 - 7.53980068e-01j,
                        9.76456992e-01 - 3.89055724e-01j,
                    ],
                    [
                        -1.41110761e00 + 4.44961532e-04j,
                        8.84935439e-01 - 1.65935394e00j,
                        2.55794780e-02 + 3.84488082e00j,
                        -7.36388646e-01 - 5.26702665e00j,
                        9.18576248e-01 + 4.96571513e00j,
                        -6.55447805e-01 - 2.99622702e00j,
                        2.78406473e-01 + 4.39557390e-01j,
                        -7.39259240e-02 + 1.29735355e00j,
                    ],
                    [
                        3.81671614e-01 - 1.32889127e00j,
                        1.58058028e00 + 1.58984228e00j,
                        -3.89515019e00 - 1.74369179e00j,
                        5.15400777e00 + 1.70039677e00j,
                        -4.55267076e00 - 1.32030127e00j,
                        2.43497521e00 + 5.99419596e-01j,
                        -5.31180397e-02 + 2.12374954e-01j,
                        -1.27882407e00 - 7.22662250e-01j,
                    ],
                    [
                        1.21850146e00 + 6.79879859e-01j,
                        -2.40966255e00 + 1.13184483e00j,
                        3.51667778e00 - 3.03504998e00j,
                        -3.82515261e00 + 3.87428727e00j,
                        2.95991217e00 - 3.19306165e00j,
                        -1.23035006e00 + 1.49432506e00j,
                        -4.85024387e-01 + 1.56037510e-01j,
                        1.29651185e00 - 8.62558411e-01j,
                    ],
                ],
                [
                    [
                        -6.86739810e-02 - 3.70300523e-01j,
                        -1.12063667e00 + 3.67172563e-01j,
                        1.15194738e00 - 8.20547146e-01j,
                        2.30641371e-01 + 1.83610960e00j,
                        -2.25827067e00 - 3.06022668e00j,
                        3.69690088e00 + 3.80242117e00j,
                        -3.69736000e00 - 3.47632660e00j,
                        2.33927662e00 + 2.06262453e00j,
                    ],
                    [
                        2.01642994e-01 + 3.70247627e-01j,
                        1.96353819e-01 - 1.30446538e00j,
                        4.53478904e-01 + 1.30789388e00j,
                        -2.17336227e00 - 3.70174142e-01j,
                        4.13698351e00 - 8.78905523e-01j,
                        -5.15569565e00 + 1.67455149e00j,
                        4.48187718e00 - 1.64139406e00j,
                        -2.36823829e00 + 9.75806650e-01j,
                    ],
                    [
                        -5.81347916e-01 - 2.32063891e-01j,
                        1.17258879e00 + 7.75790302e-01j,
                        -1.38408986e00 + 1.39407859e-01j,
                        1.27539559e00 - 2.28945062e00j,
                        -1.04103685e00 + 4.49286635e00j,
                        7.92773779e-01 - 5.38882066e00j,
                        -5.00737218e-01 + 4.36485624e00j,
                        1.14958382e-01 - 1.98652802e00j,
                    ],
                    [
                        7.29282363e-01 - 4.52505342e-01j,
                        -1.20273832e00 + 7.93312198e-01j,
                        1.54394858e-01 - 1.47157901e00j,
                        1.94926847e00 + 2.37210665e00j,
                        -3.86096638e00 - 3.07488034e00j,
                        4.41411827e00 + 3.07399461e00j,
                        -3.30379016e00 - 2.15968977e00j,
                        1.25586744e00 + 6.65608991e-01j,
                    ],
                    [
                        1.90552895e-02 + 1.06915710e00j,
                        -3.09587185e-01 - 1.39877126e00j,
                        1.59020799e00 + 5.31672910e-01j,
                        -3.38289188e00 + 1.00170896e00j,
                        4.67727196e00 - 2.24529659e00j,
                        -4.58255956e00 + 2.49582912e00j,
                        2.97612347e00 - 1.73789535e00j,
                        -6.65754916e-01 + 5.68583883e-01j,
                    ],
                    [
                        -1.10512212e00 - 5.61257766e-01j,
                        1.39417670e00 + 1.40277052e-01j,
                        -1.09240921e00 + 1.64935310e00j,
                        5.15335771e-01 - 3.93863575e00j,
                        -4.05116064e-02 + 5.35922652e00j,
                        -1.68556483e-01 - 4.96502808e00j,
                        1.97980361e-01 + 2.89906966e00j,
                        -2.11603565e-01 - 3.27366385e-01j,
                    ],
                    [
                        1.08436259e00 - 8.19418542e-01j,
                        -4.91742770e-01 + 1.28937713e00j,
                        -1.47149005e00 - 1.84499961e00j,
                        3.71149228e00 + 2.33217769e00j,
                        -4.86159529e00 - 2.44529713e00j,
                        4.21030384e00 + 1.93770771e00j,
                        -2.16832530e00 - 8.82399267e-01j,
                        -3.38178272e-02 - 2.65736902e-01j,
                    ],
                    [
                        3.16520474e-01 + 1.38975107e00j,
                        -1.18046123e00 - 7.77589784e-01j,
                        2.65147692e00 - 8.75141081e-01j,
                        -3.99619580e00 + 2.55830010e00j,
                        4.34457240e00 - 3.26013709e00j,
                        -3.30268788e00 + 2.63977923e00j,
                        1.30455644e00 - 1.19406495e00j,
                        5.99181898e-01 - 1.46683240e-01j,
                    ],
                ],
                [
                    [
                        -1.14690604e00 - 6.55982045e-01j,
                        -7.61877600e-02 + 3.55636826e-01j,
                        1.20901310e00 - 2.00000544e-01j,
                        -1.22094263e00 + 6.93962155e-01j,
                        -6.09734088e-02 - 1.94494143e00j,
                        1.88257130e00 + 3.45125794e00j,
                        -3.12826290e00 - 4.33627181e00j,
                        3.09113744e00 + 3.92393143e00j,
                    ],
                    [
                        9.66772110e-01 - 5.62132977e-01j,
                        -8.86573691e-02 - 4.64356502e-01j,
                        -3.89631175e-01 + 1.29267510e00j,
                        -3.38373773e-01 - 1.33825087e00j,
                        2.23291184e00 + 6.34137453e-01j,
                        -4.33589184e00 + 3.01830087e-01j,
                        5.36260619e00 - 9.02672832e-01j,
                        -4.57468874e00 + 9.36837573e-01j,
                    ],
                    [
                        -3.46552912e-02 + 8.77227736e-01j,
                        5.75097904e-01 + 3.88747592e-01j,
                        -1.06960345e00 - 9.36002431e-01j,
                        1.40630326e00 - 3.68793491e-02j,
                        -1.61490180e00 + 2.22573853e00j,
                        1.70466892e00 - 4.39420448e00j,
                        -1.57567238e00 + 5.20339199e00j,
                        1.11080277e00 - 4.12233180e00j,
                    ],
                    [
                        -4.77117802e-01 - 4.03353091e-01j,
                        -8.58071418e-01 + 3.40926482e-01j,
                        1.29295881e00 - 6.38936287e-01j,
                        -2.70315705e-01 + 1.50151708e00j,
                        -1.70246236e00 - 2.71460524e00j,
                        3.43475426e00 + 3.65753839e00j,
                        -3.88353239e00 - 3.66740932e00j,
                        2.84186794e00 + 2.53121897e00j,
                    ],
                    [
                        4.13935133e-01 + 1.91862212e-02j,
                        1.58938989e-01 - 1.11256935e00j,
                        1.51412250e-01 + 1.41656053e00j,
                        -1.61152180e00 - 6.94873844e-01j,
                        3.61678197e00 - 5.62385992e-01j,
                        -5.00981705e00 + 1.56003783e00j,
                        4.84170820e00 - 1.76676213e00j,
                        -3.05633917e00 + 1.22261111e00j,
                    ],
                    [
                        -3.68756035e-01 + 9.67688559e-02j,
                        1.04259718e00 + 7.42450956e-01j,
                        -1.36540790e00 - 2.71328949e-01j,
                        1.31688861e00 - 1.61379599e00j,
                        -1.08460739e00 + 3.95042963e00j,
                        8.28904293e-01 - 5.32883334e00j,
                        -5.60234996e-01 + 4.84644558e00j,
                        2.15138284e-01 - 2.73403160e00j,
                    ],
                    [
                        3.77708446e-01 - 4.04286837e-01j,
                        -1.20978632e00 + 6.72220800e-01j,
                        5.94782335e-01 - 1.24748100e00j,
                        1.31949469e00 + 2.10261059e00j,
                        -3.43231473e00 - 2.90232772e00j,
                        4.45469659e00 + 3.13743296e00j,
                        -3.78238955e00 - 2.48019282e00j,
                        1.87472801e00 + 1.10009206e00j,
                    ],
                    [
                        1.01017126e-01 + 7.78134174e-01j,
                        -1.33664623e-01 - 1.42870484e00j,
                        1.14223139e00 + 8.77057529e-01j,
                        -2.86816696e00 + 5.68988817e-01j,
                        4.40608703e00 - 2.00090277e00j,
                        -4.76005765e00 + 2.57308644e00j,
                        3.54421407e00 - 2.05479278e00j,
                        -1.32004566e00 + 9.08083244e-01j,
                    ],
                ],
                [
                    [
                        -1.19693802e00 - 8.37659192e-01j,
                        9.82544543e-01 + 7.99667114e-01j,
                        2.12745665e-01 - 3.06952942e-01j,
                        -1.26720324e00 + 1.79372368e-02j,
                        1.27399573e00 - 5.68062405e-01j,
                        -1.36520458e-01 + 2.04380403e00j,
                        -1.43610866e00 - 3.79398813e00j,
                        2.47754805e00 + 4.77884847e00j,
                    ],
                    [
                        1.37027493e00 - 7.33349695e-01j,
                        -9.85936092e-01 + 3.64218035e-01j,
                        -4.49749388e-02 + 5.30742295e-01j,
                        5.78226817e-01 - 1.24912294e00j,
                        2.27921976e-01 + 1.36182410e00j,
                        -2.26039545e00 - 9.24914566e-01j,
                        4.44803437e00 + 3.17060417e-01j,
                        -5.44633949e00 + 1.04899212e-01j,
                    ],
                    [
                        2.11958802e-02 + 1.56743729e00j,
                        1.84832990e-01 - 7.82154407e-01j,
                        -5.34383815e-01 - 5.44107076e-01j,
                        9.44784225e-01 + 1.07649788e00j,
                        -1.42901784e00 - 6.63403417e-02j,
                        1.96353752e00 - 2.10851642e00j,
                        -2.36054182e00 + 4.19009950e00j,
                        2.31865367e00 - 4.89744616e00j,
                    ],
                    [
                        -1.34003630e00 - 6.86373228e-01j,
                        3.24895547e-01 + 4.49565576e-01j,
                        9.64714575e-01 - 2.02920439e-01j,
                        -1.35862702e00 + 4.77988325e-01j,
                        3.97288895e-01 - 1.53058694e00j,
                        1.39316335e00 + 3.03383419e00j,
                        -2.91449701e00 - 4.17638493e00j,
                        3.26973739e00 + 4.16621589e00j,
                    ],
                    [
                        1.13612993e00 - 7.75676715e-01j,
                        -3.47738184e-01 - 1.50800701e-01j,
                        -3.44924195e-01 + 1.12276592e00j,
                        7.98513156e-04 - 1.41526438e00j,
                        1.61989135e00 + 8.76240200e-01j,
                        -3.79001165e00 + 7.29661392e-02j,
                        5.23450231e00 - 8.22498277e-01j,
                        -4.98000205e00 + 1.01464231e00j,
                    ],
                    [
                        9.82081652e-02 + 1.18428473e00j,
                        4.15240091e-01 + 4.11146167e-02j,
                        -9.49263839e-01 - 9.11332966e-01j,
                        1.32807927e00 + 3.92423672e-01j,
                        -1.55529479e00 + 1.54695244e00j,
                        1.66884293e00 - 3.87355644e00j,
                        -1.60824204e00 + 5.17457589e00j,
                        1.24863649e00 - 4.61678918e00j,
                    ],
                    [
                        -8.55486958e-01 - 4.30877302e-01j,
                        -5.17055458e-01 + 3.41054350e-01j,
                        1.30932287e00 - 5.07586690e-01j,
                        -6.93001128e-01 + 1.20488803e00j,
                        -1.12116352e00 - 2.35509086e00j,
                        3.05927648e00 + 3.43763539e00j,
                        -3.94328058e00 - 3.75824680e00j,
                        3.28435866e00 + 2.93580625e00j,
                    ],
                    [
                        6.24453613e-01 - 3.27379432e-01j,
                        4.83087834e-02 - 8.50467248e-01j,
                        -4.94758327e-02 + 1.43962642e00j,
                        -1.10247505e00 - 9.80946116e-01j,
                        3.04970566e00 - 2.15314311e-01j,
                        -4.73211907e00 + 1.37389910e00j,
                        5.06101686e00 - 1.83155260e00j,
                        -3.68297927e00 + 1.45651998e00j,
                    ],
                ],
                [
                    [
                        1.45172967e-02 - 2.52473566e-01j,
                        1.08514959e00 + 1.03064337e00j,
                        -7.98430024e-01 - 9.06752353e-01j,
                        -3.33986032e-01 + 2.26982498e-01j,
                        1.29326203e00 + 1.72901265e-01j,
                        -1.31353744e00 + 4.45166408e-01j,
                        3.62856223e-01 - 2.12643999e00j,
                        9.25507595e-01 + 4.07516392e00j,
                    ],
                    [
                        5.89742753e-01 - 7.73943338e-02j,
                        -1.47678703e00 + 5.21650586e-01j,
                        9.65729129e-01 - 1.72604500e-01j,
                        1.92903248e-01 - 5.65710900e-01j,
                        -7.56546708e-01 + 1.17585046e00j,
                        -1.21492852e-01 - 1.38134596e00j,
                        2.24865056e00 + 1.23862124e00j,
                        -4.46492401e00 - 9.63680776e-01j,
                    ],
                    [
                        -1.90671351e-01 + 9.17355109e-01j,
                        2.18587299e-01 - 1.54156135e00j,
                        -3.07006596e-01 + 6.61321961e-01j,
                        4.60215090e-01 + 6.91055492e-01j,
                        -8.02692856e-01 - 1.19446064e00j,
                        1.45289056e00 + 1.73099620e-01j,
                        -2.31229890e00 + 1.93336247e00j,
                        2.99077089e00 - 3.88086799e00j,
                    ],
                    [
                        -9.82997628e-01 - 7.12115983e-01j,
                        1.19059339e00 + 8.69386113e-01j,
                        -1.70463468e-01 - 4.59409111e-01j,
                        -1.04440320e00 + 4.34838398e-02j,
                        1.40039579e00 - 3.15062367e-01j,
                        -5.38090846e-01 + 1.55611180e00j,
                        -1.02300090e00 - 3.31798821e00j,
                        2.31057833e00 + 4.61538081e00j,
                    ],
                    [
                        1.24557384e00 - 6.68927635e-01j,
                        -1.20144702e00 + 5.61758232e-01j,
                        2.53079382e-01 + 2.61576682e-01j,
                        5.32295497e-01 - 1.09903134e00j,
                        -1.44642749e-01 + 1.39787596e00j,
                        -1.61000874e00 - 1.07574759e00j,
                        3.89221010e00 + 4.57527557e-01j,
                        -5.34275747e00 + 5.08276913e-02j,
                    ],
                    [
                        4.20663810e-02 + 1.53251690e00j,
                        9.70652325e-02 - 1.12048173e00j,
                        -4.28103935e-01 - 1.87422129e-01j,
                        8.28382608e-01 + 1.06256744e00j,
                        -1.28522392e00 - 5.04174225e-01j,
                        1.80308946e00 - 1.44346012e00j,
                        -2.25320291e00 + 3.70365917e00j,
                        2.35790078e00 - 4.89880206e00j,
                    ],
                    [
                        -1.41689925e00 - 6.80331646e-01j,
                        7.04097167e-01 + 5.36255746e-01j,
                        6.41772911e-01 - 2.46588248e-01j,
                        -1.38093921e00 + 3.30885169e-01j,
                        7.89014931e-01 - 1.16244156e00j,
                        8.74482339e-01 + 2.59380000e00j,
                        -2.59703728e00 - 3.92172364e00j,
                        3.34045318e00 + 4.29113964e00j,
                    ],
                    [
                        1.23049156e00 - 9.14803621e-01j,
                        -6.10242043e-01 + 1.63470674e-01j,
                        -2.13165068e-01 + 8.91996052e-01j,
                        2.28053402e-01 - 1.42408717e00j,
                        1.05893002e00 + 1.09151982e00j,
                        -3.18608795e00 - 1.81912305e-01j,
                        4.96196127e00 - 6.90929626e-01j,
                        -5.24024691e00 + 1.05551861e00j,
                    ],
                ],
                [
                    [
                        1.78709504e00 + 1.29732516e00j,
                        4.37647820e-02 + 3.35656379e-01j,
                        -9.36656013e-01 - 1.20382137e00j,
                        6.02471592e-01 + 9.74462792e-01j,
                        4.33663009e-01 - 1.19831017e-01j,
                        -1.28654231e00 - 3.66252860e-01j,
                        1.34236572e00 - 3.26731891e-01j,
                        -6.17690504e-01 + 2.18600250e00j,
                    ],
                    [
                        -1.43303874e00 + 7.90404658e-01j,
                        -6.99697163e-01 + 2.59124953e-02j,
                        1.54410214e00 - 2.92114249e-01j,
                        -9.08785605e-01 - 4.94808137e-03j,
                        -3.48199400e-01 + 5.66982267e-01j,
                        9.19670534e-01 - 1.07591940e00j,
                        1.76178954e-02 + 1.39914261e00j,
                        -2.19099200e00 - 1.56969880e00j,
                    ],
                    [
                        -1.07463663e-01 - 1.07404599e00j,
                        3.38631059e-01 - 9.67884234e-01j,
                        -4.49275418e-01 + 1.47309057e00j,
                        3.96832361e-01 - 5.21434035e-01j,
                        -3.55018453e-01 - 8.22922554e-01j,
                        6.48194731e-01 + 1.28816502e00j,
                        -1.47761511e00 - 2.86505291e-01j,
                        2.65097874e00 - 1.69747412e00j,
                    ],
                    [
                        5.53393873e-01 + 1.00847165e-01j,
                        9.19483430e-01 + 8.91220923e-01j,
                        -1.01249527e00 - 1.02009925e00j,
                        2.15663730e-02 + 4.33638015e-01j,
                        1.09349255e00 + 1.31574054e-01j,
                        -1.41984431e00 + 1.54276424e-01j,
                        6.94896309e-01 - 1.57444998e00j,
                        5.95902533e-01 + 3.55524211e00j,
                    ],
                    [
                        8.99818365e-02 + 1.99498021e-01j,
                        -1.37128322e00 + 4.92694114e-01j,
                        1.22485446e00 - 3.44206994e-01j,
                        -1.35245368e-01 - 3.46072260e-01j,
                        -7.14542694e-01 + 1.04199923e00j,
                        2.78638644e-01 - 1.36768080e00j,
                        1.57623647e00 + 1.29207580e00j,
                        -3.91446984e00 - 1.01795571e00j,
                    ],
                    [
                        -1.99597598e-01 + 4.32928337e-01j,
                        1.91369132e-01 - 1.54080456e00j,
                        -2.70751683e-01 + 1.02247056e00j,
                        4.06411455e-01 + 3.34808417e-01j,
                        -6.84140048e-01 - 1.19169421e00j,
                        1.23940913e00 + 6.07994555e-01j,
                        -2.05432179e00 + 1.29866291e00j,
                        2.82183863e00 - 3.43902791e00j,
                    ],
                    [
                        -6.38728452e-01 - 5.35269372e-01j,
                        1.29328443e00 + 8.87060798e-01j,
                        -5.39667874e-01 - 6.05276163e-01j,
                        -7.45678467e-01 + 1.24620115e-01j,
                        1.42378307e00 - 1.47616582e-01j,
                        -8.85612133e-01 + 1.12005302e00j,
                        -5.79112274e-01 - 2.80932039e00j,
                        2.05324718e00 + 4.33891167e00j,
                    ],
                    [
                        1.01111712e00 - 5.11400520e-01j,
                        -1.33579061e00 + 7.00509088e-01j,
                        5.61159931e-01 - 1.16719331e-02j,
                        3.86996353e-01 - 9.00231126e-01j,
                        -3.97853241e-01 + 1.38455374e00j,
                        -1.00851224e00 - 1.21040043e00j,
                        3.26775103e00 + 6.16641265e-01j,
                        -5.08567035e00 - 3.21184000e-02j,
                    ],
                ],
                [
                    [
                        3.00412828e00 + 3.25590973e00j,
                        -1.46340680e00 - 1.42240975e00j,
                        -7.10549219e-02 - 4.36242075e-01j,
                        7.55877906e-01 + 1.34984210e00j,
                        -4.02928826e-01 - 1.00171494e00j,
                        -5.06559002e-01 - 9.22344084e-03j,
                        1.24767279e00 + 5.55987162e-01j,
                        -1.36341089e00 + 2.13298035e-01j,
                    ],
                    [
                        -3.83337126e00 + 1.10383247e00j,
                        1.37299822e00 - 5.28719166e-01j,
                        7.99380664e-01 + 5.92153634e-02j,
                        -1.56922672e00 + 5.28361327e-02j,
                        8.19255719e-01 + 1.61442781e-01j,
                        5.03667462e-01 - 5.33765873e-01j,
                        -1.06353914e00 + 9.53222202e-01j,
                        8.58820677e-02 - 1.41692091e00j,
                    ],
                    [
                        6.35958847e-01 - 3.43433614e00j,
                        -1.54718782e-02 + 9.22328347e-01j,
                        -5.04320536e-01 + 9.86325819e-01j,
                        6.62655843e-01 - 1.36467744e00j,
                        -4.51475205e-01 + 3.69893121e-01j,
                        2.22505710e-01 + 9.33733510e-01j,
                        -4.86203979e-01 - 1.35698918e00j,
                        1.50185417e00 + 4.09668465e-01j,
                    ],
                    [
                        2.36354603e00 + 1.77374623e00j,
                        -4.19688560e-01 - 8.31064350e-02j,
                        -8.17957640e-01 - 1.06196592e00j,
                        8.12828116e-01 + 1.13370454e00j,
                        1.14433792e-01 - 3.74586440e-01j,
                        -1.10962912e00 - 3.15900501e-01j,
                        1.41932078e00 + 9.17108271e-04j,
                        -8.69026429e-01 + 1.58113060e00j,
                    ],
                    [
                        -2.13061310e00 + 1.08673060e00j,
                        -1.85770902e-01 - 1.62137261e-01j,
                        1.46484193e00 - 2.90837347e-01j,
                        -1.20690356e00 + 1.31439126e-01j,
                        4.96030747e-04 + 4.00097339e-01j,
                        8.85527228e-01 - 9.53581085e-01j,
                        -4.02252430e-01 + 1.32801662e00j,
                        -1.51290500e00 - 1.52250072e00j,
                    ],
                    [
                        -4.93376895e-02 - 1.81859258e00j,
                        2.75774308e-01 - 5.15265309e-01j,
                        -4.25576718e-01 + 1.50637960e00j,
                        4.16692659e-01 - 8.95752391e-01j,
                        -3.50768710e-01 - 4.75946441e-01j,
                        5.21425192e-01 + 1.29528746e00j,
                        -1.19254452e00 - 7.05949858e-01j,
                        2.30175055e00 - 1.10894167e00j,
                    ],
                    [
                        1.15986178e00 + 4.79957893e-01j,
                        6.31713873e-01 + 6.79893421e-01j,
                        -1.13431986e00 - 1.06844775e00j,
                        3.70131865e-01 + 6.36540729e-01j,
                        8.23535124e-01 + 1.99615224e-02j,
                        -1.43815632e00 - 3.68479566e-02j,
                        9.85503003e-01 - 1.07666914e00j,
                        2.36591262e-01 + 2.99179398e00j,
                    ],
                    [
                        -4.83278354e-01 + 5.35405301e-01j,
                        -1.14268550e00 + 3.88782017e-01j,
                        1.39982381e00 - 4.72811154e-01j,
                        -4.80945951e-01 - 1.21118001e-01j,
                        -5.62945787e-01 + 8.74090053e-01j,
                        5.55461782e-01 - 1.32420148e00j,
                        9.47476817e-01 + 1.33834936e00j,
                        -3.28600548e00 - 1.08079454e00j,
                    ],
                ],
            ]
        )

        self.adjoint_plane = np.array(
            [
                [
                    0.28276817 + 0.15660023j,
                    -0.21264103 + 0.1325063j,
                    -0.00666806 - 0.1822552j,
                    0.10494716 + 0.0590003j,
                    -0.06044544 + 0.02754128j,
                    0.01542389 - 0.02038304j,
                    -0.02641121 + 0.01316627j,
                    0.0215158 - 0.05030556j,
                ],
                [
                    0.54833966 + 0.33850828j,
                    -0.4958592 + 0.28852081j,
                    -0.01713981 - 0.49898315j,
                    0.37897714 + 0.18819081j,
                    -0.28442755 + 0.19914825j,
                    -0.02887101 - 0.27192077j,
                    0.18678311 + 0.08113525j,
                    -0.11320052 + 0.08151796j,
                ],
                [
                    0.69768533 + 0.51111915j,
                    -0.74630965 + 0.36269529j,
                    0.0271063 - 0.78358501j,
                    0.63172175 + 0.3637445j,
                    -0.56446735 + 0.35344828j,
                    -0.04125834 - 0.59543717j,
                    0.47733585 + 0.21476558j,
                    -0.35533761 + 0.27220962j,
                ],
                [
                    0.65654 + 0.5830565j,
                    -0.83763171 + 0.31681567j,
                    0.10977528 - 0.89399695j,
                    0.73395449 + 0.50935065j,
                    -0.77541578 + 0.40262967j,
                    0.00588976 - 0.84221936j,
                    0.70430209 + 0.37900244j,
                    -0.62136058 + 0.41564458j,
                ],
                [
                    0.45551939 + 0.50553654j,
                    -0.7169248 + 0.18720528j,
                    0.17640627 - 0.77381197j,
                    0.63754892 + 0.54242801j,
                    -0.8065859 + 0.32588577j,
                    0.08788936 - 0.88691676j,
                    0.75201985 + 0.49518815j,
                    -0.78544962 + 0.43340316j,
                ],
                [
                    0.2040729 + 0.31334696j,
                    -0.4437996 + 0.05144511j,
                    0.17955537 - 0.48801972j,
                    0.40159172 + 0.4344763j,
                    -0.63566113 + 0.17655863j,
                    0.14807687 - 0.70699674j,
                    0.60199882 + 0.49236828j,
                    -0.75763457 + 0.32461718j,
                ],
                [
                    0.01713837 + 0.10888565j,
                    -0.15671647 - 0.02906575j,
                    0.11958363 - 0.1813168j,
                    0.14886852 + 0.23912486j,
                    -0.34893683 + 0.03792622j,
                    0.14555865 - 0.39746133j,
                    0.33976368 + 0.36217065j,
                    -0.54640878 + 0.15861484j,
                ],
                [
                    -0.05256636 - 0.00471347j,
                    0.01359614 - 0.04566816j,
                    0.04580664 + 0.01237401j,
                    -0.01350782 + 0.06196736j,
                    -0.09008373 - 0.0351467j,
                    0.08978393 - 0.11075246j,
                    0.09570568 + 0.1731567j,
                    -0.25950948 + 0.02134457j,
                ],
            ],
            dtype=np.complex128,
        )

    def tearDown(self):
        pass

    def _testTransform(self, backend, dtype):
        if not backend_available(backend):
            raise SkipTest

        plan = Plan(self.vol.shape, self.fourier_pts.astype(dtype), backend=backend)
        result = plan.transform(self.vol.astype(dtype))
        self.assertTrue(np.allclose(result, self.recip_space))

    def _testTransformMany(self, backend, dtype, ntransforms=3):
        if not backend_available(backend):
            raise SkipTest

        plan = Plan(
            self.plane.shape,
            self.fourier_pts[:2].astype(dtype),
            backend=backend,
            ntransforms=ntransforms,
        )

        # Note, this is how (cu)finufft transform wants it for now.
        # Can be refactored as part of row major cleanup.
        batch = np.empty((ntransforms, *self.plane.shape), dtype)
        for i in range(ntransforms):
            batch[i] = self.plane

        result = plan.transform(batch)

        for r in range(ntransforms):
            self.assertTrue(np.allclose(result[r], self.recip_space_plane))

    def _testAdjoint(self, backend, dtype):
        if not backend_available(backend):
            raise SkipTest

        complex_dtype = complex_type(dtype)

        plan = Plan(self.vol.shape, self.fourier_pts.astype(dtype), backend=backend)

        # Test Adjoint
        result = plan.adjoint(self.recip_space.astype(complex_dtype))

        self.assertTrue(
            np.allclose(result, self.adjoint_vol, atol=utest_tolerance(dtype))
        )

    def _testAdjointMany(self, backend, dtype, ntransforms=2):
        if not backend_available(backend):
            raise SkipTest

        complex_dtype = complex_type(dtype)

        plan = Plan(
            self.plane.shape,
            self.fourier_pts[:2].astype(dtype),
            backend=backend,
            ntransforms=ntransforms,
        )

        batch = np.empty(
            (ntransforms, *self.recip_space_plane.shape), dtype=complex_dtype
        )
        for i in range(ntransforms):
            batch[i] = self.recip_space_plane

        # Test Adjoint
        result = plan.adjoint(batch)

        for r in range(ntransforms):
            self.assertTrue(
                np.allclose(result[r], self.adjoint_plane, atol=utest_tolerance(dtype))
            )

    # TODO: This list could be done better, as some sort of matrix
    #    once there are no raise exceptions, but more pressing things...

    def testTransform0_32(self):
        self._testTransform("cufinufft", np.float32)

    def testTransformMany0_32(self):
        self._testTransformMany("cufinufft", np.float32)

    def testAdjoint0_32(self):
        self._testAdjoint("cufinufft", np.float32)

    def testAdjointMany0_32(self):
        self._testAdjointMany("cufinufft", np.float32)

    def testTransformMany1_32(self):
        self._testTransformMany("finufft", np.float32)

    def testTransform1_32(self):
        self._testTransform("finufft", np.float32)

    def testAdjoint1_32(self):
        self._testAdjoint("finufft", np.float32)

    def testAdjointMany1_32(self):
        self._testAdjointMany("finufft", np.float32)

    def testTransform2_32(self):
        self._testTransform("pynfft", np.float32)

    def testAdjoint2_32(self):
        self._testAdjoint("pynfft", np.float32)

    def testTransform0_64(self):
        self._testTransform("cufinufft", np.float64)

    def testTransformMany0_64(self):
        self._testTransformMany("cufinufft", np.float64)

    def testTransformMany1_64(self):
        self._testTransformMany("finufft", np.float64)

    def testAdjoint0_64(self):
        self._testAdjoint("cufinufft", np.float64)

    def testAdjointMany0_64(self):
        self._testAdjointMany("cufinufft", np.float64)

    def testTransform1_64(self):
        self._testTransform("finufft", np.float64)

    def testAdjoint1_64(self):
        self._testAdjoint("finufft", np.float64)

    def testAdjointMany1_64(self):
        self._testAdjointMany("finufft", np.float64)

    def testTransform2_64(self):
        self._testTransform("pynfft", np.float64)

    def testAdjoint2_64(self):
        self._testAdjoint("pynfft", np.float64)
