import logging
from itertools import product
from unittest import TestCase

import numpy as np

from aspire.utils import grid_2d, grid_3d, utest_tolerance

logger = logging.getLogger(__name__)


class LegacyGridsTestCase(TestCase):
    def setUp(self):
        # Keys are of form (n,shift,normalized)
        self.ns = [3, 4]
        self.shifts = [False, True]
        self.norms = [False, True]

        self.legacy_references_2d = {
            (3, False, False): {
                "x": np.array(
                    [[-1.0, -1.0, -1.0], [-0.0, -0.0, -0.0], [1.0, 1.0, 1.0]],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [[-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0]],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [-2.3561945, -3.1415925, 2.3561945],
                        [-1.5707964, -3.1415925, 1.5707964],
                        [-0.7853982, -0.0, 0.7853982],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [1.4142135, 1.0, 1.4142135],
                        [1.0, 0.0, 1.0],
                        [1.4142135, 1.0, 1.4142135],
                    ],
                    dtype=np.float32,
                ),
            },
            (3, False, True): {
                "x": np.array(
                    [
                        [-0.6666667, -0.6666667, -0.6666667],
                        [-0.0, -0.0, -0.0],
                        [0.6666667, 0.6666667, 0.6666667],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [-0.6666667, -0.0, 0.6666667],
                        [-0.6666667, -0.0, 0.6666667],
                        [-0.6666667, -0.0, 0.6666667],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [-2.3561945, -3.1415925, 2.3561945],
                        [-1.5707964, -3.1415925, 1.5707964],
                        [-0.7853982, -0.0, 0.7853982],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [0.94280905, 0.6666667, 0.94280905],
                        [0.6666667, 0.0, 0.6666667],
                        [0.94280905, 0.6666667, 0.94280905],
                    ],
                    dtype=np.float32,
                ),
            },
            (3, True, False): {
                "x": np.array(
                    [[-1.0, -1.0, -1.0], [-0.0, -0.0, -0.0], [1.0, 1.0, 1.0]],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [[-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0]],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [-2.3561945, -3.1415925, 2.3561945],
                        [-1.5707964, -3.1415925, 1.5707964],
                        [-0.7853982, -0.0, 0.7853982],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [1.4142135, 1.0, 1.4142135],
                        [1.0, 0.0, 1.0],
                        [1.4142135, 1.0, 1.4142135],
                    ],
                    dtype=np.float32,
                ),
            },
            (3, True, True): {
                "x": np.array(
                    [
                        [-0.6666667, -0.6666667, -0.6666667],
                        [-0.0, -0.0, -0.0],
                        [0.6666667, 0.6666667, 0.6666667],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [-0.6666667, -0.0, 0.6666667],
                        [-0.6666667, -0.0, 0.6666667],
                        [-0.6666667, -0.0, 0.6666667],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [-2.3561945, -3.1415925, 2.3561945],
                        [-1.5707964, -3.1415925, 1.5707964],
                        [-0.7853982, -0.0, 0.7853982],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [0.94280905, 0.6666667, 0.94280905],
                        [0.6666667, 0.0, 0.6666667],
                        [0.94280905, 0.6666667, 0.94280905],
                    ],
                    dtype=np.float32,
                ),
            },
            (4, False, False): {
                "x": np.array(
                    [
                        [-2.0, -2.0, -2.0, -2.0],
                        [-1.0, -1.0, -1.0, -1.0],
                        [0.0, 0.0, 0.0, 0.0],
                        [1.0, 1.0, 1.0, 1.0],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [-2.0, -1.0, 0.0, 1.0],
                        [-2.0, -1.0, 0.0, 1.0],
                        [-2.0, -1.0, 0.0, 1.0],
                        [-2.0, -1.0, 0.0, 1.0],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [-2.3561945, -2.6779451, 3.1415925, 2.6779451],
                        [-2.0344439, -2.3561945, 3.1415925, 2.3561945],
                        [-1.5707964, -1.5707964, 0.0, 1.5707964],
                        [-1.1071488, -0.7853982, 0.0, 0.7853982],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [2.828427, 2.236068, 2.0, 2.236068],
                        [2.236068, 1.4142135, 1.0, 1.4142135],
                        [2.0, 1.0, 0.0, 1.0],
                        [2.236068, 1.4142135, 1.0, 1.4142135],
                    ],
                    dtype=np.float32,
                ),
            },
            (4, False, True): {
                "x": np.array(
                    [
                        [-1.0, -1.0, -1.0, -1.0],
                        [-0.5, -0.5, -0.5, -0.5],
                        [0.0, 0.0, 0.0, 0.0],
                        [0.5, 0.5, 0.5, 0.5],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [-1.0, -0.5, 0.0, 0.5],
                        [-1.0, -0.5, 0.0, 0.5],
                        [-1.0, -0.5, 0.0, 0.5],
                        [-1.0, -0.5, 0.0, 0.5],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [-2.3561945, -2.6779451, 3.1415925, 2.6779451],
                        [-2.0344439, -2.3561945, 3.1415925, 2.3561945],
                        [-1.5707964, -1.5707964, 0.0, 1.5707964],
                        [-1.1071488, -0.7853982, 0.0, 0.7853982],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [1.4142135, 1.118034, 1.0, 1.118034],
                        [1.118034, 0.70710677, 0.5, 0.70710677],
                        [1.0, 0.5, 0.0, 0.5],
                        [1.118034, 0.70710677, 0.5, 0.70710677],
                    ],
                    dtype=np.float32,
                ),
            },
            (4, True, False): {
                "x": np.array(
                    [
                        [-1.5, -1.5, -1.5, -1.5],
                        [-0.5, -0.5, -0.5, -0.5],
                        [0.5, 0.5, 0.5, 0.5],
                        [1.5, 1.5, 1.5, 1.5],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [-1.5, -0.5, 0.5, 1.5],
                        [-1.5, -0.5, 0.5, 1.5],
                        [-1.5, -0.5, 0.5, 1.5],
                        [-1.5, -0.5, 0.5, 1.5],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [-2.3561945, -2.819842, 2.819842, 2.3561945],
                        [-1.8925469, -2.3561945, 2.3561945, 1.8925469],
                        [-1.2490457, -0.7853982, 0.7853982, 1.2490457],
                        [-0.7853982, -0.32175055, 0.32175055, 0.7853982],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [2.1213202, 1.5811388, 1.5811388, 2.1213202],
                        [1.5811388, 0.70710677, 0.70710677, 1.5811388],
                        [1.5811388, 0.70710677, 0.70710677, 1.5811388],
                        [2.1213202, 1.5811388, 1.5811388, 2.1213202],
                    ],
                    dtype=np.float32,
                ),
            },
            (4, True, True): {
                "x": np.array(
                    [
                        [-1.0, -1.0, -1.0, -1.0],
                        [-0.33333334, -0.33333334, -0.33333334, -0.33333334],
                        [0.33333334, 0.33333334, 0.33333334, 0.33333334],
                        [1.0, 1.0, 1.0, 1.0],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [-1.0, -0.33333334, 0.33333334, 1.0],
                        [-1.0, -0.33333334, 0.33333334, 1.0],
                        [-1.0, -0.33333334, 0.33333334, 1.0],
                        [-1.0, -0.33333334, 0.33333334, 1.0],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [-2.3561945, -2.819842, 2.819842, 2.3561945],
                        [-1.8925469, -2.3561945, 2.3561945, 1.8925469],
                        [-1.2490457, -0.7853982, 0.7853982, 1.2490457],
                        [-0.7853982, -0.32175055, 0.32175055, 0.7853982],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [1.4142135, 1.0540925, 1.0540925, 1.4142135],
                        [1.0540925, 0.47140452, 0.47140452, 1.0540925],
                        [1.0540925, 0.47140452, 0.47140452, 1.0540925],
                        [1.4142135, 1.0540925, 1.0540925, 1.4142135],
                    ],
                    dtype=np.float32,
                ),
            },
        }

        self.legacy_references_3d = {
            (3, False, False): {
                "x": np.array(
                    [
                        [[-1.0, -1.0, -1.0], [-1.0, -1.0, -1.0], [-1.0, -1.0, -1.0]],
                        [[-0.0, -0.0, -0.0], [-0.0, -0.0, -0.0], [-0.0, -0.0, -0.0]],
                        [[1.0, 1.0, 1.0], [1.0, 1.0, 1.0], [1.0, 1.0, 1.0]],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [[-1.0, -1.0, -1.0], [-0.0, -0.0, -0.0], [1.0, 1.0, 1.0]],
                        [[-1.0, -1.0, -1.0], [-0.0, -0.0, -0.0], [1.0, 1.0, 1.0]],
                        [[-1.0, -1.0, -1.0], [-0.0, -0.0, -0.0], [1.0, 1.0, 1.0]],
                    ],
                    dtype=np.float32,
                ),
                "z": np.array(
                    [
                        [[-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0]],
                        [[-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0]],
                        [[-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0]],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [
                            [-2.3561945, -2.3561945, -2.3561945],
                            [-3.1415925, -3.1415925, -3.1415925],
                            [2.3561945, 2.3561945, 2.3561945],
                        ],
                        [
                            [-1.5707964, -1.5707964, -1.5707964],
                            [-3.1415925, -3.1415925, -3.1415925],
                            [1.5707964, 1.5707964, 1.5707964],
                        ],
                        [
                            [-0.7853982, -0.7853982, -0.7853982],
                            [-0.0, -0.0, -0.0],
                            [0.7853982, 0.7853982, 0.7853982],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "theta": np.array(
                    [
                        [
                            [2.186276, 1.5707964, 0.95531666],
                            [2.3561945, 1.5707964, 0.7853982],
                            [2.186276, 1.5707964, 0.95531666],
                        ],
                        [
                            [2.3561945, 1.5707964, 0.7853982],
                            [3.1415927, 1.5707964, 0.0],
                            [2.3561945, 1.5707964, 0.7853982],
                        ],
                        [
                            [2.186276, 1.5707964, 0.95531666],
                            [2.3561945, 1.5707964, 0.7853982],
                            [2.186276, 1.5707964, 0.95531666],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [
                            [1.7320508, 1.4142135, 1.7320508],
                            [1.4142135, 1.0, 1.4142135],
                            [1.7320508, 1.4142135, 1.7320508],
                        ],
                        [
                            [1.4142135, 1.0, 1.4142135],
                            [1.0, 0.0, 1.0],
                            [1.4142135, 1.0, 1.4142135],
                        ],
                        [
                            [1.7320508, 1.4142135, 1.7320508],
                            [1.4142135, 1.0, 1.4142135],
                            [1.7320508, 1.4142135, 1.7320508],
                        ],
                    ],
                    dtype=np.float32,
                ),
            },
            (3, False, True): {
                "x": np.array(
                    [
                        [
                            [-0.6666667, -0.6666667, -0.6666667],
                            [-0.6666667, -0.6666667, -0.6666667],
                            [-0.6666667, -0.6666667, -0.6666667],
                        ],
                        [[-0.0, -0.0, -0.0], [-0.0, -0.0, -0.0], [-0.0, -0.0, -0.0]],
                        [
                            [0.6666667, 0.6666667, 0.6666667],
                            [0.6666667, 0.6666667, 0.6666667],
                            [0.6666667, 0.6666667, 0.6666667],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [
                            [-0.6666667, -0.6666667, -0.6666667],
                            [-0.0, -0.0, -0.0],
                            [0.6666667, 0.6666667, 0.6666667],
                        ],
                        [
                            [-0.6666667, -0.6666667, -0.6666667],
                            [-0.0, -0.0, -0.0],
                            [0.6666667, 0.6666667, 0.6666667],
                        ],
                        [
                            [-0.6666667, -0.6666667, -0.6666667],
                            [-0.0, -0.0, -0.0],
                            [0.6666667, 0.6666667, 0.6666667],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "z": np.array(
                    [
                        [
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                        ],
                        [
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                        ],
                        [
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [
                            [-2.3561945, -2.3561945, -2.3561945],
                            [-3.1415925, -3.1415925, -3.1415925],
                            [2.3561945, 2.3561945, 2.3561945],
                        ],
                        [
                            [-1.5707964, -1.5707964, -1.5707964],
                            [-3.1415925, -3.1415925, -3.1415925],
                            [1.5707964, 1.5707964, 1.5707964],
                        ],
                        [
                            [-0.7853982, -0.7853982, -0.7853982],
                            [-0.0, -0.0, -0.0],
                            [0.7853982, 0.7853982, 0.7853982],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "theta": np.array(
                    [
                        [
                            [2.186276, 1.5707964, 0.95531666],
                            [2.3561945, 1.5707964, 0.7853982],
                            [2.186276, 1.5707964, 0.95531666],
                        ],
                        [
                            [2.3561945, 1.5707964, 0.7853982],
                            [3.1415927, 1.5707964, 0.0],
                            [2.3561945, 1.5707964, 0.7853982],
                        ],
                        [
                            [2.186276, 1.5707964, 0.95531666],
                            [2.3561945, 1.5707964, 0.7853982],
                            [2.186276, 1.5707964, 0.95531666],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [
                            [1.1547005, 0.94280905, 1.1547005],
                            [0.94280905, 0.6666667, 0.94280905],
                            [1.1547005, 0.94280905, 1.1547005],
                        ],
                        [
                            [0.94280905, 0.6666667, 0.94280905],
                            [0.6666667, 0.0, 0.6666667],
                            [0.94280905, 0.6666667, 0.94280905],
                        ],
                        [
                            [1.1547005, 0.94280905, 1.1547005],
                            [0.94280905, 0.6666667, 0.94280905],
                            [1.1547005, 0.94280905, 1.1547005],
                        ],
                    ],
                    dtype=np.float32,
                ),
            },
            (3, True, False): {
                "x": np.array(
                    [
                        [[-1.0, -1.0, -1.0], [-1.0, -1.0, -1.0], [-1.0, -1.0, -1.0]],
                        [[-0.0, -0.0, -0.0], [-0.0, -0.0, -0.0], [-0.0, -0.0, -0.0]],
                        [[1.0, 1.0, 1.0], [1.0, 1.0, 1.0], [1.0, 1.0, 1.0]],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [[-1.0, -1.0, -1.0], [-0.0, -0.0, -0.0], [1.0, 1.0, 1.0]],
                        [[-1.0, -1.0, -1.0], [-0.0, -0.0, -0.0], [1.0, 1.0, 1.0]],
                        [[-1.0, -1.0, -1.0], [-0.0, -0.0, -0.0], [1.0, 1.0, 1.0]],
                    ],
                    dtype=np.float32,
                ),
                "z": np.array(
                    [
                        [[-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0]],
                        [[-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0]],
                        [[-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0], [-1.0, -0.0, 1.0]],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [
                            [-2.3561945, -2.3561945, -2.3561945],
                            [-3.1415925, -3.1415925, -3.1415925],
                            [2.3561945, 2.3561945, 2.3561945],
                        ],
                        [
                            [-1.5707964, -1.5707964, -1.5707964],
                            [-3.1415925, -3.1415925, -3.1415925],
                            [1.5707964, 1.5707964, 1.5707964],
                        ],
                        [
                            [-0.7853982, -0.7853982, -0.7853982],
                            [-0.0, -0.0, -0.0],
                            [0.7853982, 0.7853982, 0.7853982],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "theta": np.array(
                    [
                        [
                            [2.186276, 1.5707964, 0.95531666],
                            [2.3561945, 1.5707964, 0.7853982],
                            [2.186276, 1.5707964, 0.95531666],
                        ],
                        [
                            [2.3561945, 1.5707964, 0.7853982],
                            [3.1415927, 1.5707964, 0.0],
                            [2.3561945, 1.5707964, 0.7853982],
                        ],
                        [
                            [2.186276, 1.5707964, 0.95531666],
                            [2.3561945, 1.5707964, 0.7853982],
                            [2.186276, 1.5707964, 0.95531666],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [
                            [1.7320508, 1.4142135, 1.7320508],
                            [1.4142135, 1.0, 1.4142135],
                            [1.7320508, 1.4142135, 1.7320508],
                        ],
                        [
                            [1.4142135, 1.0, 1.4142135],
                            [1.0, 0.0, 1.0],
                            [1.4142135, 1.0, 1.4142135],
                        ],
                        [
                            [1.7320508, 1.4142135, 1.7320508],
                            [1.4142135, 1.0, 1.4142135],
                            [1.7320508, 1.4142135, 1.7320508],
                        ],
                    ],
                    dtype=np.float32,
                ),
            },
            (3, True, True): {
                "x": np.array(
                    [
                        [
                            [-0.6666667, -0.6666667, -0.6666667],
                            [-0.6666667, -0.6666667, -0.6666667],
                            [-0.6666667, -0.6666667, -0.6666667],
                        ],
                        [[-0.0, -0.0, -0.0], [-0.0, -0.0, -0.0], [-0.0, -0.0, -0.0]],
                        [
                            [0.6666667, 0.6666667, 0.6666667],
                            [0.6666667, 0.6666667, 0.6666667],
                            [0.6666667, 0.6666667, 0.6666667],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [
                            [-0.6666667, -0.6666667, -0.6666667],
                            [-0.0, -0.0, -0.0],
                            [0.6666667, 0.6666667, 0.6666667],
                        ],
                        [
                            [-0.6666667, -0.6666667, -0.6666667],
                            [-0.0, -0.0, -0.0],
                            [0.6666667, 0.6666667, 0.6666667],
                        ],
                        [
                            [-0.6666667, -0.6666667, -0.6666667],
                            [-0.0, -0.0, -0.0],
                            [0.6666667, 0.6666667, 0.6666667],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "z": np.array(
                    [
                        [
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                        ],
                        [
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                        ],
                        [
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                            [-0.6666667, -0.0, 0.6666667],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [
                            [-2.3561945, -2.3561945, -2.3561945],
                            [-3.1415925, -3.1415925, -3.1415925],
                            [2.3561945, 2.3561945, 2.3561945],
                        ],
                        [
                            [-1.5707964, -1.5707964, -1.5707964],
                            [-3.1415925, -3.1415925, -3.1415925],
                            [1.5707964, 1.5707964, 1.5707964],
                        ],
                        [
                            [-0.7853982, -0.7853982, -0.7853982],
                            [-0.0, -0.0, -0.0],
                            [0.7853982, 0.7853982, 0.7853982],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "theta": np.array(
                    [
                        [
                            [2.186276, 1.5707964, 0.95531666],
                            [2.3561945, 1.5707964, 0.7853982],
                            [2.186276, 1.5707964, 0.95531666],
                        ],
                        [
                            [2.3561945, 1.5707964, 0.7853982],
                            [3.1415927, 1.5707964, 0.0],
                            [2.3561945, 1.5707964, 0.7853982],
                        ],
                        [
                            [2.186276, 1.5707964, 0.95531666],
                            [2.3561945, 1.5707964, 0.7853982],
                            [2.186276, 1.5707964, 0.95531666],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [
                            [1.1547005, 0.94280905, 1.1547005],
                            [0.94280905, 0.6666667, 0.94280905],
                            [1.1547005, 0.94280905, 1.1547005],
                        ],
                        [
                            [0.94280905, 0.6666667, 0.94280905],
                            [0.6666667, 0.0, 0.6666667],
                            [0.94280905, 0.6666667, 0.94280905],
                        ],
                        [
                            [1.1547005, 0.94280905, 1.1547005],
                            [0.94280905, 0.6666667, 0.94280905],
                            [1.1547005, 0.94280905, 1.1547005],
                        ],
                    ],
                    dtype=np.float32,
                ),
            },
            (4, False, False): {
                "x": np.array(
                    [
                        [
                            [-2.0, -2.0, -2.0, -2.0],
                            [-2.0, -2.0, -2.0, -2.0],
                            [-2.0, -2.0, -2.0, -2.0],
                            [-2.0, -2.0, -2.0, -2.0],
                        ],
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-1.0, -1.0, -1.0, -1.0],
                            [-1.0, -1.0, -1.0, -1.0],
                            [-1.0, -1.0, -1.0, -1.0],
                        ],
                        [
                            [0.0, 0.0, 0.0, 0.0],
                            [0.0, 0.0, 0.0, 0.0],
                            [0.0, 0.0, 0.0, 0.0],
                            [0.0, 0.0, 0.0, 0.0],
                        ],
                        [
                            [1.0, 1.0, 1.0, 1.0],
                            [1.0, 1.0, 1.0, 1.0],
                            [1.0, 1.0, 1.0, 1.0],
                            [1.0, 1.0, 1.0, 1.0],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [
                            [-2.0, -2.0, -2.0, -2.0],
                            [-1.0, -1.0, -1.0, -1.0],
                            [0.0, 0.0, 0.0, 0.0],
                            [1.0, 1.0, 1.0, 1.0],
                        ],
                        [
                            [-2.0, -2.0, -2.0, -2.0],
                            [-1.0, -1.0, -1.0, -1.0],
                            [0.0, 0.0, 0.0, 0.0],
                            [1.0, 1.0, 1.0, 1.0],
                        ],
                        [
                            [-2.0, -2.0, -2.0, -2.0],
                            [-1.0, -1.0, -1.0, -1.0],
                            [0.0, 0.0, 0.0, 0.0],
                            [1.0, 1.0, 1.0, 1.0],
                        ],
                        [
                            [-2.0, -2.0, -2.0, -2.0],
                            [-1.0, -1.0, -1.0, -1.0],
                            [0.0, 0.0, 0.0, 0.0],
                            [1.0, 1.0, 1.0, 1.0],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "z": np.array(
                    [
                        [
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                        ],
                        [
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                        ],
                        [
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                        ],
                        [
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                            [-2.0, -1.0, 0.0, 1.0],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [
                            [-2.3561945, -2.3561945, -2.3561945, -2.3561945],
                            [-2.6779451, -2.6779451, -2.6779451, -2.6779451],
                            [3.1415925, 3.1415925, 3.1415925, 3.1415925],
                            [2.6779451, 2.6779451, 2.6779451, 2.6779451],
                        ],
                        [
                            [-2.0344439, -2.0344439, -2.0344439, -2.0344439],
                            [-2.3561945, -2.3561945, -2.3561945, -2.3561945],
                            [3.1415925, 3.1415925, 3.1415925, 3.1415925],
                            [2.3561945, 2.3561945, 2.3561945, 2.3561945],
                        ],
                        [
                            [-1.5707964, -1.5707964, -1.5707964, -1.5707964],
                            [-1.5707964, -1.5707964, -1.5707964, -1.5707964],
                            [0.0, 0.0, 0.0, 0.0],
                            [1.5707964, 1.5707964, 1.5707964, 1.5707964],
                        ],
                        [
                            [-1.1071488, -1.1071488, -1.1071488, -1.1071488],
                            [-0.7853982, -0.7853982, -0.7853982, -0.7853982],
                            [0.0, 0.0, 0.0, 0.0],
                            [0.7853982, 0.7853982, 0.7853982, 0.7853982],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "theta": np.array(
                    [
                        [
                            [2.186276, 1.9106333, 1.5707964, 1.2309594],
                            [2.300524, 1.9913307, 1.5707964, 1.150262],
                            [2.3561945, 2.0344439, 1.5707964, 1.1071488],
                            [2.300524, 1.9913307, 1.5707964, 1.150262],
                        ],
                        [
                            [2.300524, 1.9913307, 1.5707964, 1.150262],
                            [2.526113, 2.186276, 1.5707964, 0.95531666],
                            [2.6779451, 2.3561945, 1.5707964, 0.7853982],
                            [2.526113, 2.186276, 1.5707964, 0.95531666],
                        ],
                        [
                            [2.3561945, 2.0344439, 1.5707964, 1.1071488],
                            [2.6779451, 2.3561945, 1.5707964, 0.7853982],
                            [3.1415927, 3.1415927, 1.5707964, 0.0],
                            [2.6779451, 2.3561945, 1.5707964, 0.7853982],
                        ],
                        [
                            [2.300524, 1.9913307, 1.5707964, 1.150262],
                            [2.526113, 2.186276, 1.5707964, 0.95531666],
                            [2.6779451, 2.3561945, 1.5707964, 0.7853982],
                            [2.526113, 2.186276, 1.5707964, 0.95531666],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [
                            [3.4641016, 3.0, 2.828427, 3.0],
                            [3.0, 2.4494898, 2.236068, 2.4494898],
                            [2.828427, 2.236068, 2.0, 2.236068],
                            [3.0, 2.4494898, 2.236068, 2.4494898],
                        ],
                        [
                            [3.0, 2.4494898, 2.236068, 2.4494898],
                            [2.4494898, 1.7320508, 1.4142135, 1.7320508],
                            [2.236068, 1.4142135, 1.0, 1.4142135],
                            [2.4494898, 1.7320508, 1.4142135, 1.7320508],
                        ],
                        [
                            [2.828427, 2.236068, 2.0, 2.236068],
                            [2.236068, 1.4142135, 1.0, 1.4142135],
                            [2.0, 1.0, 0.0, 1.0],
                            [2.236068, 1.4142135, 1.0, 1.4142135],
                        ],
                        [
                            [3.0, 2.4494898, 2.236068, 2.4494898],
                            [2.4494898, 1.7320508, 1.4142135, 1.7320508],
                            [2.236068, 1.4142135, 1.0, 1.4142135],
                            [2.4494898, 1.7320508, 1.4142135, 1.7320508],
                        ],
                    ],
                    dtype=np.float32,
                ),
            },
            (4, False, True): {
                "x": np.array(
                    [
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-1.0, -1.0, -1.0, -1.0],
                            [-1.0, -1.0, -1.0, -1.0],
                            [-1.0, -1.0, -1.0, -1.0],
                        ],
                        [
                            [-0.5, -0.5, -0.5, -0.5],
                            [-0.5, -0.5, -0.5, -0.5],
                            [-0.5, -0.5, -0.5, -0.5],
                            [-0.5, -0.5, -0.5, -0.5],
                        ],
                        [
                            [0.0, 0.0, 0.0, 0.0],
                            [0.0, 0.0, 0.0, 0.0],
                            [0.0, 0.0, 0.0, 0.0],
                            [0.0, 0.0, 0.0, 0.0],
                        ],
                        [
                            [0.5, 0.5, 0.5, 0.5],
                            [0.5, 0.5, 0.5, 0.5],
                            [0.5, 0.5, 0.5, 0.5],
                            [0.5, 0.5, 0.5, 0.5],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-0.5, -0.5, -0.5, -0.5],
                            [0.0, 0.0, 0.0, 0.0],
                            [0.5, 0.5, 0.5, 0.5],
                        ],
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-0.5, -0.5, -0.5, -0.5],
                            [0.0, 0.0, 0.0, 0.0],
                            [0.5, 0.5, 0.5, 0.5],
                        ],
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-0.5, -0.5, -0.5, -0.5],
                            [0.0, 0.0, 0.0, 0.0],
                            [0.5, 0.5, 0.5, 0.5],
                        ],
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-0.5, -0.5, -0.5, -0.5],
                            [0.0, 0.0, 0.0, 0.0],
                            [0.5, 0.5, 0.5, 0.5],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "z": np.array(
                    [
                        [
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                        ],
                        [
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                        ],
                        [
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                        ],
                        [
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                            [-1.0, -0.5, 0.0, 0.5],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [
                            [-2.3561945, -2.3561945, -2.3561945, -2.3561945],
                            [-2.6779451, -2.6779451, -2.6779451, -2.6779451],
                            [3.1415925, 3.1415925, 3.1415925, 3.1415925],
                            [2.6779451, 2.6779451, 2.6779451, 2.6779451],
                        ],
                        [
                            [-2.0344439, -2.0344439, -2.0344439, -2.0344439],
                            [-2.3561945, -2.3561945, -2.3561945, -2.3561945],
                            [3.1415925, 3.1415925, 3.1415925, 3.1415925],
                            [2.3561945, 2.3561945, 2.3561945, 2.3561945],
                        ],
                        [
                            [-1.5707964, -1.5707964, -1.5707964, -1.5707964],
                            [-1.5707964, -1.5707964, -1.5707964, -1.5707964],
                            [0.0, 0.0, 0.0, 0.0],
                            [1.5707964, 1.5707964, 1.5707964, 1.5707964],
                        ],
                        [
                            [-1.1071488, -1.1071488, -1.1071488, -1.1071488],
                            [-0.7853982, -0.7853982, -0.7853982, -0.7853982],
                            [0.0, 0.0, 0.0, 0.0],
                            [0.7853982, 0.7853982, 0.7853982, 0.7853982],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "theta": np.array(
                    [
                        [
                            [2.186276, 1.9106333, 1.5707964, 1.2309594],
                            [2.300524, 1.9913307, 1.5707964, 1.150262],
                            [2.3561945, 2.0344439, 1.5707964, 1.1071488],
                            [2.300524, 1.9913307, 1.5707964, 1.150262],
                        ],
                        [
                            [2.300524, 1.9913307, 1.5707964, 1.150262],
                            [2.526113, 2.186276, 1.5707964, 0.95531666],
                            [2.6779451, 2.3561945, 1.5707964, 0.7853982],
                            [2.526113, 2.186276, 1.5707964, 0.95531666],
                        ],
                        [
                            [2.3561945, 2.0344439, 1.5707964, 1.1071488],
                            [2.6779451, 2.3561945, 1.5707964, 0.7853982],
                            [3.1415927, 3.1415927, 1.5707964, 0.0],
                            [2.6779451, 2.3561945, 1.5707964, 0.7853982],
                        ],
                        [
                            [2.300524, 1.9913307, 1.5707964, 1.150262],
                            [2.526113, 2.186276, 1.5707964, 0.95531666],
                            [2.6779451, 2.3561945, 1.5707964, 0.7853982],
                            [2.526113, 2.186276, 1.5707964, 0.95531666],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [
                            [1.7320508, 1.5, 1.4142135, 1.5],
                            [1.5, 1.2247449, 1.118034, 1.2247449],
                            [1.4142135, 1.118034, 1.0, 1.118034],
                            [1.5, 1.2247449, 1.118034, 1.2247449],
                        ],
                        [
                            [1.5, 1.2247449, 1.118034, 1.2247449],
                            [1.2247449, 0.8660254, 0.70710677, 0.8660254],
                            [1.118034, 0.70710677, 0.5, 0.70710677],
                            [1.2247449, 0.8660254, 0.70710677, 0.8660254],
                        ],
                        [
                            [1.4142135, 1.118034, 1.0, 1.118034],
                            [1.118034, 0.70710677, 0.5, 0.70710677],
                            [1.0, 0.5, 0.0, 0.5],
                            [1.118034, 0.70710677, 0.5, 0.70710677],
                        ],
                        [
                            [1.5, 1.2247449, 1.118034, 1.2247449],
                            [1.2247449, 0.8660254, 0.70710677, 0.8660254],
                            [1.118034, 0.70710677, 0.5, 0.70710677],
                            [1.2247449, 0.8660254, 0.70710677, 0.8660254],
                        ],
                    ],
                    dtype=np.float32,
                ),
            },
            (4, True, False): {
                "x": np.array(
                    [
                        [
                            [-1.5, -1.5, -1.5, -1.5],
                            [-1.5, -1.5, -1.5, -1.5],
                            [-1.5, -1.5, -1.5, -1.5],
                            [-1.5, -1.5, -1.5, -1.5],
                        ],
                        [
                            [-0.5, -0.5, -0.5, -0.5],
                            [-0.5, -0.5, -0.5, -0.5],
                            [-0.5, -0.5, -0.5, -0.5],
                            [-0.5, -0.5, -0.5, -0.5],
                        ],
                        [
                            [0.5, 0.5, 0.5, 0.5],
                            [0.5, 0.5, 0.5, 0.5],
                            [0.5, 0.5, 0.5, 0.5],
                            [0.5, 0.5, 0.5, 0.5],
                        ],
                        [
                            [1.5, 1.5, 1.5, 1.5],
                            [1.5, 1.5, 1.5, 1.5],
                            [1.5, 1.5, 1.5, 1.5],
                            [1.5, 1.5, 1.5, 1.5],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [
                            [-1.5, -1.5, -1.5, -1.5],
                            [-0.5, -0.5, -0.5, -0.5],
                            [0.5, 0.5, 0.5, 0.5],
                            [1.5, 1.5, 1.5, 1.5],
                        ],
                        [
                            [-1.5, -1.5, -1.5, -1.5],
                            [-0.5, -0.5, -0.5, -0.5],
                            [0.5, 0.5, 0.5, 0.5],
                            [1.5, 1.5, 1.5, 1.5],
                        ],
                        [
                            [-1.5, -1.5, -1.5, -1.5],
                            [-0.5, -0.5, -0.5, -0.5],
                            [0.5, 0.5, 0.5, 0.5],
                            [1.5, 1.5, 1.5, 1.5],
                        ],
                        [
                            [-1.5, -1.5, -1.5, -1.5],
                            [-0.5, -0.5, -0.5, -0.5],
                            [0.5, 0.5, 0.5, 0.5],
                            [1.5, 1.5, 1.5, 1.5],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "z": np.array(
                    [
                        [
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                        ],
                        [
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                        ],
                        [
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                        ],
                        [
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                            [-1.5, -0.5, 0.5, 1.5],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [
                            [-2.3561945, -2.3561945, -2.3561945, -2.3561945],
                            [-2.819842, -2.819842, -2.819842, -2.819842],
                            [2.819842, 2.819842, 2.819842, 2.819842],
                            [2.3561945, 2.3561945, 2.3561945, 2.3561945],
                        ],
                        [
                            [-1.8925469, -1.8925469, -1.8925469, -1.8925469],
                            [-2.3561945, -2.3561945, -2.3561945, -2.3561945],
                            [2.3561945, 2.3561945, 2.3561945, 2.3561945],
                            [1.8925469, 1.8925469, 1.8925469, 1.8925469],
                        ],
                        [
                            [-1.2490457, -1.2490457, -1.2490457, -1.2490457],
                            [-0.7853982, -0.7853982, -0.7853982, -0.7853982],
                            [0.7853982, 0.7853982, 0.7853982, 0.7853982],
                            [1.2490457, 1.2490457, 1.2490457, 1.2490457],
                        ],
                        [
                            [-0.7853982, -0.7853982, -0.7853982, -0.7853982],
                            [-0.32175055, -0.32175055, -0.32175055, -0.32175055],
                            [0.32175055, 0.32175055, 0.32175055, 0.32175055],
                            [0.7853982, 0.7853982, 0.7853982, 0.7853982],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "theta": np.array(
                    [
                        [
                            [2.186276, 1.8022738, 1.339319, 0.95531666],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.186276, 1.8022738, 1.339319, 0.95531666],
                        ],
                        [
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.701082, 2.186276, 0.95531666, 0.44051075],
                            [2.701082, 2.186276, 0.95531666, 0.44051075],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                        ],
                        [
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.701082, 2.186276, 0.95531666, 0.44051075],
                            [2.701082, 2.186276, 0.95531666, 0.44051075],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                        ],
                        [
                            [2.186276, 1.8022738, 1.339319, 0.95531666],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.186276, 1.8022738, 1.339319, 0.95531666],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [
                            [2.598076, 2.1794493, 2.1794493, 2.598076],
                            [2.1794496, 1.6583124, 1.6583124, 2.1794496],
                            [2.1794496, 1.6583124, 1.6583124, 2.1794496],
                            [2.598076, 2.1794493, 2.1794493, 2.598076],
                        ],
                        [
                            [2.1794496, 1.6583124, 1.6583124, 2.1794496],
                            [1.6583124, 0.8660254, 0.8660254, 1.6583124],
                            [1.6583124, 0.8660254, 0.8660254, 1.6583124],
                            [2.1794496, 1.6583124, 1.6583124, 2.1794496],
                        ],
                        [
                            [2.1794496, 1.6583124, 1.6583124, 2.1794496],
                            [1.6583124, 0.8660254, 0.8660254, 1.6583124],
                            [1.6583124, 0.8660254, 0.8660254, 1.6583124],
                            [2.1794496, 1.6583124, 1.6583124, 2.1794496],
                        ],
                        [
                            [2.598076, 2.1794493, 2.1794493, 2.598076],
                            [2.1794496, 1.6583124, 1.6583124, 2.1794496],
                            [2.1794496, 1.6583124, 1.6583124, 2.1794496],
                            [2.598076, 2.1794493, 2.1794493, 2.598076],
                        ],
                    ],
                    dtype=np.float32,
                ),
            },
            (4, True, True): {
                "x": np.array(
                    [
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-1.0, -1.0, -1.0, -1.0],
                            [-1.0, -1.0, -1.0, -1.0],
                            [-1.0, -1.0, -1.0, -1.0],
                        ],
                        [
                            [-0.33333334, -0.33333334, -0.33333334, -0.33333334],
                            [-0.33333334, -0.33333334, -0.33333334, -0.33333334],
                            [-0.33333334, -0.33333334, -0.33333334, -0.33333334],
                            [-0.33333334, -0.33333334, -0.33333334, -0.33333334],
                        ],
                        [
                            [0.33333334, 0.33333334, 0.33333334, 0.33333334],
                            [0.33333334, 0.33333334, 0.33333334, 0.33333334],
                            [0.33333334, 0.33333334, 0.33333334, 0.33333334],
                            [0.33333334, 0.33333334, 0.33333334, 0.33333334],
                        ],
                        [
                            [1.0, 1.0, 1.0, 1.0],
                            [1.0, 1.0, 1.0, 1.0],
                            [1.0, 1.0, 1.0, 1.0],
                            [1.0, 1.0, 1.0, 1.0],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "y": np.array(
                    [
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-0.33333334, -0.33333334, -0.33333334, -0.33333334],
                            [0.33333334, 0.33333334, 0.33333334, 0.33333334],
                            [1.0, 1.0, 1.0, 1.0],
                        ],
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-0.33333334, -0.33333334, -0.33333334, -0.33333334],
                            [0.33333334, 0.33333334, 0.33333334, 0.33333334],
                            [1.0, 1.0, 1.0, 1.0],
                        ],
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-0.33333334, -0.33333334, -0.33333334, -0.33333334],
                            [0.33333334, 0.33333334, 0.33333334, 0.33333334],
                            [1.0, 1.0, 1.0, 1.0],
                        ],
                        [
                            [-1.0, -1.0, -1.0, -1.0],
                            [-0.33333334, -0.33333334, -0.33333334, -0.33333334],
                            [0.33333334, 0.33333334, 0.33333334, 0.33333334],
                            [1.0, 1.0, 1.0, 1.0],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "z": np.array(
                    [
                        [
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                        ],
                        [
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                        ],
                        [
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                        ],
                        [
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                            [-1.0, -0.33333334, 0.33333334, 1.0],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "phi": np.array(
                    [
                        [
                            [-2.3561945, -2.3561945, -2.3561945, -2.3561945],
                            [-2.819842, -2.819842, -2.819842, -2.819842],
                            [2.819842, 2.819842, 2.819842, 2.819842],
                            [2.3561945, 2.3561945, 2.3561945, 2.3561945],
                        ],
                        [
                            [-1.8925469, -1.8925469, -1.8925469, -1.8925469],
                            [-2.3561945, -2.3561945, -2.3561945, -2.3561945],
                            [2.3561945, 2.3561945, 2.3561945, 2.3561945],
                            [1.8925469, 1.8925469, 1.8925469, 1.8925469],
                        ],
                        [
                            [-1.2490457, -1.2490457, -1.2490457, -1.2490457],
                            [-0.7853982, -0.7853982, -0.7853982, -0.7853982],
                            [0.7853982, 0.7853982, 0.7853982, 0.7853982],
                            [1.2490457, 1.2490457, 1.2490457, 1.2490457],
                        ],
                        [
                            [-0.7853982, -0.7853982, -0.7853982, -0.7853982],
                            [-0.32175055, -0.32175055, -0.32175055, -0.32175055],
                            [0.32175055, 0.32175055, 0.32175055, 0.32175055],
                            [0.7853982, 0.7853982, 0.7853982, 0.7853982],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "theta": np.array(
                    [
                        [
                            [2.186276, 1.8022738, 1.339319, 0.95531666],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.186276, 1.8022738, 1.339319, 0.95531666],
                        ],
                        [
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.701082, 2.186276, 0.95531666, 0.44051075],
                            [2.701082, 2.186276, 0.95531666, 0.44051075],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                        ],
                        [
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.701082, 2.186276, 0.95531666, 0.44051075],
                            [2.701082, 2.186276, 0.95531666, 0.44051075],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                        ],
                        [
                            [2.186276, 1.8022738, 1.339319, 0.95531666],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.3298666, 1.8770738, 1.264519, 0.81172615],
                            [2.186276, 1.8022738, 1.339319, 0.95531666],
                        ],
                    ],
                    dtype=np.float32,
                ),
                "r": np.array(
                    [
                        [
                            [1.7320508, 1.4529663, 1.4529663, 1.7320508],
                            [1.4529663, 1.1055416, 1.1055416, 1.4529663],
                            [1.4529663, 1.1055416, 1.1055416, 1.4529663],
                            [1.7320508, 1.4529663, 1.4529663, 1.7320508],
                        ],
                        [
                            [1.4529663, 1.1055416, 1.1055416, 1.4529663],
                            [1.1055416, 0.57735026, 0.57735026, 1.1055416],
                            [1.1055416, 0.57735026, 0.57735026, 1.1055416],
                            [1.4529663, 1.1055416, 1.1055416, 1.4529663],
                        ],
                        [
                            [1.4529663, 1.1055416, 1.1055416, 1.4529663],
                            [1.1055416, 0.57735026, 0.57735026, 1.1055416],
                            [1.1055416, 0.57735026, 0.57735026, 1.1055416],
                            [1.4529663, 1.1055416, 1.1055416, 1.4529663],
                        ],
                        [
                            [1.7320508, 1.4529663, 1.4529663, 1.7320508],
                            [1.4529663, 1.1055416, 1.1055416, 1.4529663],
                            [1.4529663, 1.1055416, 1.1055416, 1.4529663],
                            [1.7320508, 1.4529663, 1.4529663, 1.7320508],
                        ],
                    ],
                    dtype=np.float32,
                ),
            },
        }

    def test_grid_2d(self):
        for k in product(self.ns, self.shifts, self.norms):
            for j in self.legacy_references_2d[k]:
                logger.debug(f"{k}, {j}")
                a = self.legacy_references_2d[k][j]
                b = grid_2d(*k, indexing="xy")[j]
                if j != "phi":
                    assert np.allclose(a, b, atol=utest_tolerance(np.float32))

    def test_grid_3d(self):
        for k in product(self.ns, self.shifts, self.norms):
            for j in self.legacy_references_3d[k]:
                logger.debug(f"{k}, {j}")
                a = self.legacy_references_3d[k][j]
                b = grid_3d(*k, indexing="xyz")[j]
                if j == "phi":
                    continue
                elif j == "theta":
                    a, b = np.mod(a, np.pi), np.mod(b, np.pi)
                    assert np.allclose(a, b, atol=utest_tolerance(np.float32))
                else:
                    assert np.allclose(a, b, atol=utest_tolerance(np.float32))
