"""
==================================
Relion Projection Interoperability
==================================

In this tutorial we compare projections generated by Relion
with projections generated by ASPIRE's ``Simulation`` class.
Both sets of projections are generated using a downsampled
volume map of a 70S Ribosome, absent of noise and CTF corruption.
"""

import os

import numpy as np

from aspire.source import RelionSource, Simulation
from aspire.volume import Volume

# %%
# Load Relion Projections
# -----------------------
# We load the Relion projections as a ``RelionSource`` and view the images.

starfile = os.path.join(os.path.dirname(os.getcwd()), "data", "rln_proj_65.star")
rln_src = RelionSource(starfile)
rln_src.images[:].show(colorbar=False)

# %%
# .. note::
#    The projections above were generated in Relion using the following command::
#
#        relion_project --i clean70SRibosome_vol_65p.mrc --nr_uniform 3000 --angpix 5
#
#    For this tutorial we take a subset of these projections consisting of the first 5 images.

# %%
# Generate Projections using ``Simulation``
# -----------------------------------------
# Using the metadata associated with the ``RelionSource`` and the same volume
# we generate an analogous set of projections with ASPIRE's ``Simulation`` class.

# Load the volume from file as a ``Volume`` object.
filepath = os.path.join(
    os.path.dirname(os.getcwd()), "data", "clean70SRibosome_vol_65p.mrc"
)
vol = Volume.load(filepath, dtype=rln_src.dtype)

# Create a ``Simulation`` source using metadata from the RelionSource projections.
# Note, for odd resolution Relion projections are shifted from ASPIRE projections
# by 1 pixel in x and y.
sim_src = Simulation(
    n=rln_src.n,
    vols=vol,
    offsets=-np.ones((rln_src.n, 2), dtype=rln_src.dtype),
    amplitudes=rln_src.amplitudes,
    angles=rln_src.angles,
    dtype=rln_src.dtype,
)

sim_src.images[:].show(colorbar=False)

# %%
# Comparing the Projections
# -------------------------
# We will take a few different approaches to comparing the two sets of projection images.

# %%
# Visual Comparison
# ^^^^^^^^^^^^^^^^^
# We'll first look at a side-by-side of the two sets of images to confirm visually that
# the projections are taken from the same viewing angles.

rln_src.images[:].show(colorbar=False)
sim_src.images[:].show(colorbar=False)

# %%
# Fourier Ring Correlation
# ^^^^^^^^^^^^^^^^^^^^^^^^
# Additionally, we can compare the two sets of images using the FRC. Note that the images
# are tightly correlated up to a high resolution of 2 pixels.
rln_src.images[:].frc(sim_src.images[:], cutoff=0.143, plot=True)

# %%
# Relative Error
# ^^^^^^^^^^^^^^
# As Relion and ASPIRE differ in methods of generating projections, the pixel intensity of
# the images may not correspond perfectly. So we begin by first normalizing the two sets of projections.
# We then check that the relative error with respect to the frobenius norm is less than 3%.

# Work with numpy arrays.
rln_np = rln_src.images[:].asnumpy()
sim_np = sim_src.images[:].asnumpy()

# Normalize images.
rln_np = (rln_np - np.mean(rln_np)) / np.std(rln_np)
sim_np = (sim_np - np.mean(sim_np)) / np.std(sim_np)

# Assert that error is less than 3%.
error = np.linalg.norm(rln_np - sim_np, axis=(1, 2)) / np.linalg.norm(
    rln_np, axis=(1, 2)
)
assert all(error < 0.03)
print(f"Relative per-image error: {error}")
