"""
ASPIRE Configuration
====================

This tutorial reviews the default ASPIRE configuration
and common patterns for overriding.
"""

# %%
# Default Configuration
# ---------------------
# ASPIRE uses the `confuse library`_ for managing configuration.
# While this document should cover common uses,
# advanced users and developers should consider reviewing their documentation.
#
# The ASPIRE package ships with ``config_default.yaml``.
# This represents a base configuration.
# The shipped configuration for this version of ASPIRE is:
#
# .. literalinclude:: ../../../src/aspire/config_default.yaml


# %%
# System Overrides
# ----------------
# From here we can override with a custom config file in your home dir,
# specifically ``$HOME/.config/ASPIRE/config.yaml`` on most Linux platforms.
# Items in this file will take precedence over the default configuration.
# For other platforms, refer to the `confuse` documentation.
#
# As an example, suppose you want to change the ``ray`` ``temp_dir`` variable
# when working on a specific machine.
# By creating ``$HOME/.config/ASPIRE/config.yaml`` with the following contents
# on that machine, ASPIRE's configuration utility will overload
# the ``temp_dir`` directory from a ``/tmp/ray`` folder to ``/scratch/tmp/ray``.
#
#     .. code-block:: yaml
#
#       ray:
#         temp_dir: /scratch/tmp/ray
#

# %%
# Override Configuration Directory
# --------------------------------
# Users may specify a directory containing the configuration file.
# This is done by using the enviornment variable ``ASPIREDIR``
# If you wanted a file in your working directory to take
# precedence over system-overrides, we can create a local ``config.yaml``.
#
# Suppose you want to store ASPIRE logs at ``/tmp/my_proj/aspire_logs``
# when working on a specific project. Create the following ``config.yaml``.
#
#     .. code-block:: yaml
#
#       logging:
#         log_dir: /tmp/my_proj/aspire_logs
#
# This directory must then be set before invoking any code.
#
#     .. code-block:: bash
#
#       export ASPIREDIR=$PWD
#
# Similarly, you could specify any directory you like that might contain a configuration.
# This allows you to store configurations for reuse.

# %%
# In-Code Overrides
# -----------------
# You can also specify your own file from an arbitrary location from within Python code.
# For precise behavior refer to the confuse documentation.
#
#     .. code-block:: python
#
#       aspire.config.set_file('/path/to/some_experimental_config.yaml')
#
#
# Or simply set specific variables as needed.
# Here we will disable progress bars displayed by
# ``aspire.utils.trange`` and ``aspire.utils.tqdm``.
#

import time

from aspire import config
from aspire.utils import trange

# Progress bars are displayed by default.

print("Disabling progress bars")
config["logging"]["tqdm_disable"] = True

for _ in trange(3):
    time.sleep(1)
print("Done Loop 1\n")

print("Re-enabling progress bars")
config["logging"]["tqdm_disable"] = False

for _ in trange(3):
    time.sleep(1)
print("Done Loop 2\n")


# %%
# Resolution
# ----------
# ASPIRE logs the ``config_dir()`` for your system at startup,
# along with the configuration sources and resolved configuration at import time.
# This should give an accurate snapshot of the configuration before any in-code overrides.
# To view these as saved in your log, you will need to locate your `log_dir`.
# If you are not sure where it is, we can ask the config:

import aspire

print(aspire.config["logging"]["log_dir"].as_filename())

# %%
# You can also resolve the config in code
print(aspire.config.dump())

# %%
# Logging preferences
# ---------
# The "Logging" section contains options for controlling the verbosity and destination of log messages generated by ASPIRE.
#
# By default, the ``log_dir`` is a directory called ``logs`` within the current working directory from which ASPIRE was invoked, but
# it can be changed to an absolute path (for instance, to go to the same folder as a custom configuration file stored elsewhere, such
# as in the example above):
#
#    .. code-block:: yaml
#
#      logging:
#          log_dir=/tmp/my_proj/aspire_logs
#
# ASPIRE directs output streams to both the console and to a log file during each session. The logging verbosity for both of these
# can be specified individually via ``console_level`` and ``log_file_level``. These levels are passed through to Python's logging
# system, so the labels are the same: ``"DEBUG", "INFO", "WARNING", "ERROR", "CRITICAL"``. In general, it may make sense to have the log file
# store more verbose output than the console. By default, only ``INFO`` and above level messages are printed to the terminal, while
# detailed ``DEBUG`` messages are saved to disk:
#
#    .. code-block:: yaml
#
#      logging:
#          console_level: INFO
#          log_file_level: DEBUG
#
# The following modification would save all ``DEBUG`` messages to the log file, but only print ``ERROR`` and ``CRITICAL`` messages
# in the terminal:
#
#    .. code-block:: yaml
#
#      logging:
#          console_level: ERROR
#          log_file_level: DEBUG
#
# By default, the filename of the log file is ``aspire-{%Y-%m-%dT%H-%M-%S.%f}.log``. The prefix before the timestamp can be customized
# via the ``log_prefix`` option.
#
# More details on logging behavior in Python can be found in the `Python logging HOWTO`_.

# %%
# Advanced logging options
# ---------
# Users with sophisticated knowledge of Python's ``logging`` library are able to directly work with the ``logging.conf`` file stored in
# ``src/aspire`` in the source.
#
# .. literalinclude:: ../../../src/aspire/logging.conf
#
# This file exposes options that ASPIRE's ``config.yaml`` does not, for example the precise formatting options for log messages. However,
# values from ASPIRE's configuration are passed through this file, so caution is needed when modifying it.


# %%
# .. _Python logging HOWTO: https://docs.python.org/3/howto/logging.html

# %%
# .. _confuse library: https://confuse.readthedocs.io/en/latest/index.html
