function test_tstar()

% set up parameters
noise = 0.000001 ;
spol = 80;
fast_true = 30;
tlag_true = 1.0 ;

fref = 10 ;

% generate synthetics

[trN,trE]=msac_splitwave(fast_true,tlag_true,...
   'spol',spol,'noise',noise,'dfreq',0.2);

% apply the tstar value
[trF,trS]=msac_rotate(trN,trE,fast_true) ;

figure

trS.x1 = zeros(1,trS.npts) ;
trS.x1(250) = 1 ;
trSA = msac_apply_tstar_operator(trS,fref,0.1) ;

% plot([trSA.b:trSA.delta:trSA.e],trS.x1,'k-'); 
% hold on 

x1ref = max(trSA.x1) ;
plot([trSA.b:trSA.delta:trSA.e],trSA.x1./x1ref,'k-'); 
hold on ;

trSA = msac_apply_tstar_operator(trS,fref,0.5) ;
plot([trSA.b:trSA.delta:trSA.e],trSA.x1./x1ref,'k-')

trSA = msac_apply_tstar_operator(trS,fref,4.0) ;
plot([trSA.b:trSA.delta:trSA.e],trSA.x1./x1ref,'k-')

axis([-2 12 -0.02 1.02])

end

% msac_apply_tstar_operator - attenuate a seismic trace using a causal
%                             t* operator.
%
% [tr_out] = msac_apply_tstar_operator(tr,fref,tstar)
%
% Usage: 
%     [tr_out] = msac_apply_tstar_operator(tr,fref,tstar)                    
%         Apply a causal t* operator (tstar, at a reference frequency fref)
%         to trace tr, returning its attenuated equivalent (tr_out)
%
% REF: Matheney, M. P. & Nowack, R. L. Geophys J Int 123, 1–15 (1995).
%
% 2022 version with proper negative frequency handling.
%
function [tr_out] = msac_apply_tstar_operator(tr,fref,tstar)

%  extract some useful information
   t = [0:tr.npts-1].*tr.delta ;
   y = tr.x1 ;
   Fs = 1/tr.delta ;

%  zero pad trace then FFT
   n = 2^nextpow2(tr.npts) ;
   Y = fft(y,n) ;

%  frequency vector
   f = Fs*(0:(n/2))/n ;

%  angular frequency
   om = 2.*pi.*f ;
   omref = 2*pi*fref ;

%  create (causal) t* multiplier
   D = zeros(1,n/2+1) ;
   D(1) = complex(1,0) ;
   for ii=2:1+n/2
      D(ii) = exp ( complex( -(1/2)*om(ii)*tstar , ...
                              (1/pi)*om(ii)*tstar*log(om(ii)/omref) ) ) ;
   end
% Note that the imaginary part is positive (cf. Shearer; Matheney and
% Nowack). This follows Durand after Muller (JoG, 1983), allowing the use 
% of the Nyquist frequency as the reference. Changing the sign of this 
% changes the sense of the phase shift, depending on the reference 
% frequency.


%  apply to FD signal
   for ii = 2:1+n/2
      Y(ii) = Y(ii).*D(ii) ;
   end
   
%  restore symmetry
   Y(2+n/2:length(Y)) = conj(Y(n/2:-1:2)) ;
      
%  inverse FFT
   X = ifft(Y) ;

%  generate output trace
   tr_out = tr ;
   tr_out.x1 = real(X(1:tr.npts)) ;
end
