%  MSAC_TSHIFT.M
%
%  [trs] = msac_tshift(tr,tsh,varargin) ;
%
%  Positive numbers move the trace
%  *later* in time. Shift introduces zeros at the end (or beginning), 
%  unless 'cyclic' is specified. Two options are available:
%
%  'int' : PCHIP interpolated shift.
%  'qnd' : quick and dirty nearest sample shift. 
%
%  (c) James Wookey, 2011
%
%  Version 1.0
%

function [trs] = msac_tshift(tr,tshift,varargin) ;

%  check that h1 and h2 are single traces only
   if (length(tr)~=1),error('Input must be single trace.') ;, end;
   
   iinterpshift = 1 ;
   icyclic = 0 ;
   if length(varargin)>0
%  ** parse optional arguments   
      for iarg = 1: length(varargin)
         switch lower(varargin{iarg})
         case 'qnd'
            iinterpshift = 0 ;
         case 'int'
            iinterpshift = 1 ;
         case 'cyclic'
            icyclic = 1 ;
         otherwise
            error('Unrecognised optional argument.') ;
         end
      end
   end   
   
   sshift = -(tshift/tr.delta) ;
   if sshift > tr.npts
      error('MSAC_TSHIFT: Requested shift is longer than the trace');
   end
   
   trs = tr;
   xx = zeros(1,tr.npts*3) ;
   if icyclic
      xx(1:tr.npts) = tr.x1 ;
      xx(tr.npts+1:tr.npts*2) = tr.x1 ;
      xx(tr.npts*2+1:tr.npts*3) = tr.x1 ;
   else
      xx(1:tr.npts) = 0 ;
      xx(tr.npts+1:tr.npts*2) = tr.x1 ;
      xx(tr.npts*2+1:tr.npts*3) = 0 ;
   end

   if (iinterpshift)
%  ** construct a timebase
      trtime = tr.b+[0:1:tr.npts-1] .* tr.delta ;
      tt = tr.b+[-tr.npts:1:tr.npts*2-1] .* tr.delta ;
%  ** interpolate the signal onto a shifted time base
      trs.x1 = pchip(tt,xx,trtime-tshift) ;
   else
%  ** easy-mode
      if abs(round(sshift)) < 10
         warning('MSAC_TSHIFT: Shift < 10 samples, consider using interpolated mode.')
      end
      trs.x1 = xx(tr.npts+1+round(sshift):tr.npts*2+round(sshift)) ;
   end


return

