%% Synthetic example of dt* stacking
clear

%mode = 'pos_dts1';
% Uncomment to measure negative dt* synthetics.
mode = 'neg_dts1'; 

data_path = ['../data/synthetics/data/', mode];
data = readtable([data_path, '/spol_snrs_used.txt'],'Format','%f%f%f');
%Set-up measurement parameters
ref_frame_rots = -90:90;
dts = 0:0.05:4;
[DSTARG,REFROTG] = meshgrid(dts,ref_frame_rots) ;
% Generate 100 synthetics with random source polarisation 
n = 100;
all_difr = cell(1,n);
spols = data.spol;
for i = 1:n
    fname_grid = [data_path, sprintf('/difr_grids/SYN_%03i_difr_grid.txt',i)];
    % if isfile(fname_grid)
    %     difr_grid = readmatrix(fname_grid);
    % else
        fnameN = sprintf('%s/SYN_%03i.BHN', data_path, i);
        fnameE = sprintf('%s/SYN_%03i.BHE', data_path, i);
        fnameZ = sprintf('%s/SYN_%03i.BHZ', data_path, i);
        trN = msac_read(fnameN);
        trE = msac_read(fnameE);
        trZ = msac_read(fnameZ); 
        %bandpass filter data
        trN = msac_bpfilt(trN, 0.01, 0.3, 2,2);
        trE = msac_bpfilt(trE, 0.01, 0.3, 2,2);
        fref = 1/(2*trN.delta);
        % Measure difr grid for each synthetic
        difr_grid = dtstar_fast_gridsearch(trN, trE, trN.a, trN.f, ref_frame_rots, dts, fref);
        writematrix(difr_grid, fname_grid,'Delimiter','space');
    % end
    all_difr{i} = difr_grid;
end

% Now do difr stacking
[N,edges,bin] = histcounts(spols, 0:10:360) ;
wgt = 1./N(bin) ;
difr = all_difr{1}*0; % Hack to mae sure stacking grid is right size
for i=1:n
    difr = difr + all_difr{i}.*wgt(i) ;
end   

difr_stack = difr ./sum(wgt) ;
stack_out = ['../data/synthetics/Synthetic_', mode, '_difr_stack.txt'];
writematrix(difr_stack, stack_out,'Delimiter','space')


nsamps = 10000;

[bstrap_min, bstrap_fast, bstrap_dts, rej_frac] = bootstrap_results(all_difr, data.spol, nsamps, mode);
sorted = sort(bstrap_min);
crit_val = sorted(nsamps*.95);

%%
    fig_stack = figure();
    
    [min_val,minind] = min(difr_stack(:)) ; %I is the index maximun Here tu can change the function to max or min
    [itstar, ifast] = ind2sub(size(difr_stack),minind) ; %I_row is the row index and I_col is the column index
    dts_best = DSTARG(itstar,ifast);
    fast_best = REFROTG(itstar,ifast);

    fasts = -90:90;
    dts = 0:0.05:4;
    % make the grids
    pcolor(DSTARG,REFROTG,difr_stack);
    colormap(flipud(parula))
    shading interp
    c = colorbar;
    c.Label.String = 'dIFr';
    sorted = sort(bstrap_min);
    crit_val = sorted(9500);
    clim([min_val, crit_val]);
    hold on
    plot(dts_best, fast_best, 'kx', 'MarkerSize', 12, 'LineWidth',3)
    plot(dtstar_true, fast_true, 'rx','MarkerSize', 14, 'LineWidth',3)
    xlabel('\Delta t*')
    ylabel('Reference frame rotation')
    title(sprintf('SYN Test stacked L1 Norm, N = %i \n \\Delta t* = %4.2f s, \\phi = %4.2f%c', ...
        n, dts_best, fast_best, char(0176)))
    set(gca,'FontSize',20)
    %%

function [difr] = dtstar_fast_gridsearch(trN, trE, wbeg, wend, s_fast, s_dtstar, fref)
% Performs the grid search over fast direction and differential attenuation 
%
% Parameters:
% -----------
%   trN - msac struct containing North component
%   trE - msac struct containing East component
%   wbeg - window start time (from SWS analysis)
%   wend - window end time (from SWS analysis)
%   s_fast - search range in (assumed) fast polarisation directions
%   s_dtstar - search range in dt*
%   fref - reference frequency. We advise using the nyquist frequency
%          following Muller (1984)
%
% Returns:
% -----------
%   difr - array that hold difference in instantaneous frequency as a
%          function of (s_fast, s_dtstar)

n = length(s_fast);
m = length(s_dtstar);
ifrF = zeros(n,m);
ifrS = zeros(n,m);

for i=1:length(s_fast)
   for j=1:length(s_dtstar)
      [trF,trS]=msac_rotate(trN,trE,s_fast(i)) ;
      % Impose apparent fast (S1) phase is reference phase (i.e. less
      % attenuated)
      trF = msac_apply_tstar_operator(trF,fref,s_dtstar(j)) ;  
      ifrF(i,j)=msac_ifa_wwind(trF,wbeg,wend) ;
      ifrS(i,j)=msac_ifa_wwind(trS,wbeg,wend) ;
   end
end
difr = abs(ifrF - ifrS);
end

function [bstrap_min, bstrap_fast, bstrap_dts, rej_frac] = bootstrap_results(all_difr, spols, nsamp, mode)
% Function that bootstraps our instantaneous frequency grid stacking
% process. For efficiency we read in already exisiting difr grids.
% 
% Parameters:
% -----------
% all_difr - cell array holding measured difr grids to bootstrap sample
% spols - source polarisation of waveforms corresponding to difr grids
% nsamp - number of bootstrap samples
% station - station code
%
% Returns:
% -----------
% bstrap_min - array of min(difr) for bootstrapped stacks
% bstrap_fast - array of best fitting fast directions for bootstrapped stacks
% bstrap_dts - array of best fitting dt*'s for bootstrapped stacks
% rej_frac - number of samples rejected for having an inadequete source
%            polarisation coverage

    fasts = -90:90;
    dts = 0:0.05:4;
    bstrap_fast = zeros(1,nsamp) ;
    bstrap_dts = zeros(1,nsamp) ;
    bstrap_min = zeros(1,nsamp);   
    % make grids of dt* and fast direction
    [DSTARG,FASTG] = meshgrid(dts,fasts) ;
    bs_fname = sprintf('../data/synthetics/Synthetic_%s_bootstraps.txt', mode);
    bs_fID = fopen(bs_fname, 'w');
    fprintf(bs_fID,'fast,dtstar,difr_min\n');
    nrej = 0 ;
    n = length(all_difr);
    for isamp=1:nsamp
        % generate random index
        ind = randi(n,1,n) ;
        bs_difr = FASTG.*0 ;
        [N,edges,bin] = histcounts(spols(ind),0:10:360) ;
        wgt = 1./N(bin) ;
        bzmin = edges(min(bin));
        bzmax = edges(max(bin));
        
        if abs(bzmax - bzmin) > 40
            % Is there a spread of more than 40 degrees in spol coverage
            for i=1:n
                bs_difr = bs_difr + all_difr{ind(i)}.*wgt(i) ;
            end   
           
            bs_difr = bs_difr ./ sum(wgt);   
            [minval,minind] = min(bs_difr(:)) ; 
            [itstar, ifast] = ind2sub(size(bs_difr),minind);
            
            bstrap_fast(isamp)  = FASTG(itstar,ifast) ;
            bstrap_dts(isamp) = DSTARG(itstar,ifast) ;
            bstrap_min(isamp)  = minval ;
            % now write bootstrapping results to a textfile
            fprintf(bs_fID, '%3i,%4.3f,%6.4f \n', bstrap_fast(isamp), bstrap_dts(isamp), bstrap_min(isamp));
        else
            nrej = nrej + 1;
            continue
        end
    end
    rej_frac = 100*(nrej / nsamp);
    fprintf(' %i (%4.2f %%) of %i samples rejected in bootstrapping \n', nrej, rej_frac, nsamp)
    fclose(bs_fID);
 
end