%% Script for measuring dt* at a single station of interest.
%  Difference in instantnaeous frequency grids are
%  stacking with added backzmithal weighting
close all; clear

station = 'FURI';
file = '../data/furi/FURI_sks_splitting_final.txt';

data = readtable(file, 'Format','%s%s%s%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f%f');
[difr_stack, all_difr, phi, dtstar, n] = meas_dtstar_stat(data, station);

nsamps = 10000;
[bstrap_min, bstrap_fast, bstrap_dts, rej_frac] = bootstrap_results(all_difr, data.SPOL, nsamps, station);
sorted = sort(bstrap_min);
crit_val = sorted(nsamps*.95);

function [difr_stack, all_difr, fast_best, dts_best, n] = meas_dtstar_stat(data, station)
% Function to find waveforms we have measured shear-wave splitting for and
% if a corresponding difr grid does not exist in data/furi/difr_grids/ then
% meausre the difference in instantaneous frequency (difr) for a range of 
% reference frame rotation angles and dt*'s
% Parameters:
% -----------
%   data - Matlab Table containing summarised shear-wave splitting results
%   station - station name as string
 

fasts = -90:90;
dts = 0:0.05:4;
n = height(data);
all_difr = cell(1,n); % use cell array to store difr surfaces
% make the grids
[DSTARG,FASTG] = meshgrid(dts,fasts) ;
for idx=1:n
    date = char(data.DATE(idx));
    time = char(data.TIME(idx));
    fileid = sprintf('%s_%s_%s',station, date, time);
    difr_fname = sprintf('../data/furi/difr_grids/%s_difr.txt', fileid);
    if isfile(difr_fname)
        difr_grid = readmatrix(difr_fname);
        all_difr{idx} = difr_grid;
    else
        % We don't have/ don't want to read the difr grids so
        % Read waveform data
        trN = msac_read(['../data/furi/data/', fileid, '.BHN']);
        trE = msac_read(['../data/furi/data/', fileid, '.BHE']);
        % we dont need BHZ component, but its good to see if its there
        trZ = msac_read(['../data/furi/data/', fileid, '.BHZ']);
        wbeg = data.WBEG(idx);
        wend = data.WEND(idx);
        fref = 1/(2*trN.delta);
        difr_grid = dtstar_fast_gridsearch(trN, trE,wbeg, wend, fasts, dts, fref);
        all_difr{idx} = difr_grid;
        writematrix(difr_grid, difr_fname, 'Delimiter','space')
    end
end

% do source polarisation weighting of the sum of the difr grids
ind = 1:n ;
[N, ~, bin] = histcounts(data.SPOL(ind),0:10:360) ;
wgt = 1./N(bin) ;
difr = FASTG.*0 ;
for i=ind
    difr = difr + all_difr{i}.*wgt(i) ;
end   

difr_stack = difr ./ sum(wgt) ;
% Write stack to file
file_out = sprintf('../data/furi/%s_difr_stack.txt', station);
writematrix(difr_stack,file_out, 'Delimiter','space')


[~,minind] = min(difr_stack(:)) ;
[itstar, ifast] = ind2sub(size(difr_stack),minind) ;

dts_best = DSTARG(itstar,ifast);
fast_best = FASTG(itstar,ifast);

end

function [difr] = dtstar_fast_gridsearch(trN, trE, wbeg, wend, s_fast, s_dtstar, fref)
% Performs the grid search over fast direction and differential attenuation 
%
% Parameters:
% -----------
%   trN - msac struct containing North component
%   trE - msac struct containing East component
%   wbeg - window start time (from SWS analysis)
%   wend - window end time (from SWS analysis)
%   s_fast - search range in (assumed) fast polarisation directions
%   s_dtstar - search range in dt*
%   fref - reference frequency. We advise using the nyquist frequency
%          following Muller (1984)
%
% Returns:
% -----------
%   difr - array that hold difference in instantaneous frequency as a
%          function of (s_fast, s_dtstar)

n = length(s_fast);
m = length(s_dtstar);
ifrF = zeros(n,m);
ifrS = zeros(n,m);

for i=1:length(s_fast)
   for j=1:length(s_dtstar)
      [trF,trS]=msac_rotate(trN,trE,s_fast(i)) ;
      % Impose apparent fast (S1) phase is reference phase (i.e. less
      % attenuated)
      trF = msac_apply_tstar_operator(trF,fref,s_dtstar(j)) ;  
      ifrF(i,j)=msac_ifa_wwind(trF,wbeg,wend) ;
      ifrS(i,j)=msac_ifa_wwind(trS,wbeg,wend) ;
   end
end
difr = abs(ifrF - ifrS);
end

function [bstrap_min, bstrap_fast, bstrap_dts, rej_frac] = bootstrap_results(all_difr, spols, nsamp, station)
% Function that bootstraps our instantaneous frequency grid stacking
% process. For efficiency we read in already exisiting difr grids.
% 
% Parameters:
% -----------
% all_difr - cell array holding measured difr grids to bootstrap sample
% spols - source polarisation of waveforms corresponding to difr grids
% nsamp - number of bootstrap samples
% station - station code
%
% Returns:
% -----------
% bstrap_min - array of min(difr) for bootstrapped stacks
% bstrap_fast - array of best fitting fast directions for bootstrapped stacks
% bstrap_dts - array of best fitting dt*'s for bootstrapped stacks
% rej_frac - number of samples rejected for having an inadequete source
%            polarisation coverage

    fasts = -90:90;
    dts = 0:0.05:4;
    bstrap_fast = zeros(1,nsamp) ;
    bstrap_dts = zeros(1,nsamp) ;
    bstrap_min = zeros(1,nsamp);   
    % make grids of dt* and fast direction
    [DSTARG,FASTG] = meshgrid(dts,fasts) ;
    bs_fname = sprintf('../data/furi/%s_bootstrap_final.txt', station);
    bs_fID = fopen(bs_fname, 'w');
    fprintf(bs_fID,'fast,dtstar,difr_min\n');
    nrej = 0 ;
    n = length(all_difr);
    for isamp=1:nsamp
        % generate random index
        ind = randi(n,1,n) ;
        bs_difr = FASTG.*0 ;
        [N,edges,bin] = histcounts(spols(ind),0:10:360) ;
        wgt = 1./N(bin) ;
        bzmin = edges(min(bin));
        bzmax = edges(max(bin));
        
        if abs(bzmax - bzmin) > 40
            % Is there a spread of more than 40 degrees in spol coverage
            for i=1:n
                bs_difr = bs_difr + all_difr{ind(i)}.*wgt(i) ;
            end   
           
            bs_difr = bs_difr ./ sum(wgt);   
            [minval,minind] = min(bs_difr(:)) ; 
            [itstar, ifast] = ind2sub(size(bs_difr),minind);
            
            bstrap_fast(isamp)  = FASTG(itstar,ifast) ;
            bstrap_dts(isamp) = DSTARG(itstar,ifast) ;
            bstrap_min(isamp)  = minval ;
            % now write bootstrapping results to a textfile
            fprintf(bs_fID, '%3i,%4.3f,%6.4f \n', bstrap_fast(isamp), bstrap_dts(isamp), bstrap_min(isamp));
        else
            nrej = nrej + 1;
            continue
        end
    end
    rej_frac = 100*(nrej / nsamp);
    fprintf(' %i (%4.2f %%) of %i samples rejected in bootstrapping \n', nrej, rej_frac, nsamp)
    fclose(bs_fID);
 
end