%% Figure 3, 4, 5 for Asplet et al., 2023
clc; clear; close all
% THis script 
%% Figure 3 - Example synthetics where dt*=0,1 and -1
% Synthetics are generate using a Gabor wavelet, see text of Asplet et al.,
% (2023) for description

% set up parameters
noise = 0.001 ;
spol = 70;
fast_true = 30;
tlag_true = 1.5 ;
tstar = 1;


% generate synthetics
[trN,trE,trZ]=msac_splitwave2(fast_true,tlag_true,'spol',spol,'noise',noise) ;
fref= 1/(2*trN.delta);
samps = [1:length(trN.x1)]*trN.delta + trN.b;
[trR, trT] = msac_rotate(trN, trE, fast_true);
fast = -90:90 ;

% apply the tstar value
[trF,trS]=msac_rotate(trN,trE,fast_true) ;
trSA = msac_apply_tstar_operator(trS,fref,tstar) ;
% apply negative dtstar 
trFA = msac_apply_tstar_operator(trF,fref,tstar) ; 
[trNpos,trEpos]=msac_rotate(trF,trSA,-1*fast_true);


fig3 = figure('Position',[1000 500 800 900],'DefaultAxesFontSize',16);
subplot(311)
plot(samps, trF.x1, 'LineWidth',2)
hold on
plot(samps, trS.x1, 'LineWidth',2)

xlabel('Time [s]', 'FontSize',16)
hold off
xlim([-10,10])
ylim([-1,1])
title(['\Deltat* = 0s, \phi = 30', char(176), ', \deltat = 1.0s'], 'FontSize',20)
legend('Fast trace', 'Slow trace')
text(-9.5,0.8,'(a)','FontSize',20)
subplot(312)
plot(samps, trF.x1, 'LineWidth', 2)
hold on
plot(samps, trSA.x1, 'LineWidth',2)
hold off
xlabel('Time [s]', 'FontSize',16)
title(['\Deltat* = 1s, \phi = 30', char(176), ', \deltat = 1.0s'], 'FontSize',20)
xlim([-10,10])
ylim([-1,1])
text(-9.5,0.8,'(b)','FontSize',20)
subplot(313)
plot(samps, trFA.x1, 'LineWidth', 2)
hold on
plot(samps, trS.x1, 'LineWidth',2)
hold off
xlabel('Time [s]', 'FontSize',16)
title(['\Deltat* = -1s, \phi = 30', char(176), ', \deltat = 1.0s'], 'FontSize',20)
xlim([-10,10])
ylim([-1,1])
text(-9.5,0.8,'(c)','FontSize',20)

exportgraphics(fig3, '../figs/synth_dts_on_off_paper.png','Resolution',500)

%% Figure 4 - Frequency content as a function of rotation angle

fast_true = 30;
tlag_true = 1.5;
spol = 70;
tstar = 1;
noise =0.01;
[trN,trE,trZ]=msac_splitwave2(fast_true,tlag_true,'spol',spol,'noise',noise);
fref= 1/(2*trN.delta);
% apply the tstar value
[trF,trS]=msac_rotate(trN,trE,fast_true) ;
trSA = msac_apply_tstar_operator(trS,fref,tstar) ;
[trN_attn,trE_attn]=msac_rotate(trF,trSA,-1*fast_true) ;
% filter traces
trN = msac_bpfilt(trN, 0.03, 0.5, 2,2);
trE = msac_bpfilt(trE, 0.03, 0.5, 2,2);
trN_attn = msac_bpfilt(trN_attn, 0.03, 0.5, 2,2);
trE_attn = msac_bpfilt(trE_attn, 0.03, 0.5, 2,2);

n = 200;
npts = pow2(nextpow2(trN.npts));
fasts = linspace(-90,90, n);
trf_spec = zeros(n, floor(npts/2));
trs_spec = zeros(n, floor(npts/2));
trf_spec_attn = zeros(n, floor(npts/2));
trs_spec_attn = zeros(n, floor(npts/2));
trf_ifr = zeros(n, 1);
trs_ifr = zeros(n, 1);
trf_ifr_attn = zeros(n, 1);
trs_ifr_attn = zeros(n, 1);
%Loop over fast directions
for i = 1:n
    fast = fasts(i);
    [tr1, tr2] = msac_rotate(trN, trE, fast);
    [tr1_attn, tr2_attn] = msac_rotate(trN_attn, trE_attn, fast);
    fs= 1/tr1.delta;
    
    f = fs*(0:npts-1)/npts ;
    tr1_fft = fft(tr1.x1,npts);
    tr2_fft = fft(tr2.x1,npts);
    tr1_attn_fft = fft(tr1_attn.x1,npts);
    tr2_attn_fft = fft(tr2_attn.x1,npts);
    tr1_am = abs(tr1_fft);
    tr2_am = abs(tr2_fft);
    tr1_attn_am = abs(tr1_attn_fft);
    tr2_attn_am = abs(tr2_attn_fft);
    trf_spec(i,:) = tr1_am(1:floor(npts/2));
    trs_spec(i,:) = tr2_am(1:floor(npts/2));
    trf_spec_attn(i,:) = tr1_attn_am(1:floor(npts/2));
    trs_spec_attn(i,:) = tr2_attn_am(1:floor(npts/2));
    % Also measure instantanteous frequency to show what it measures for
    % each synthetic.
    trf_ifr(i) = msac_ifa_wwind(tr1, tr1.a, tr1.f);
    trs_ifr(i) = msac_ifa_wwind(tr2, tr2.a, tr2.f);
    trf_ifr_attn(i) = msac_ifa_wwind(tr1_attn, tr1_attn.a, tr1_attn.f);
    trs_ifr_attn(i) = msac_ifa_wwind(tr2_attn, tr2_attn.a, tr2_attn.f);
end

fig4 = figure("Position",[800,1000, 800, 850]);
set(gca,'FontSize',14)
[X, Y] = meshgrid(f(1:floor(npts/2)), fasts);
% Make a 2x2 grid of f-theta "spectograms" 
% Left column is un-attenuated (i.e. not dt*) 
% right column has dt* = 1s applied
grey = [150/255, 150/255, 150/255];

subplot(2,2,1)

pcolor(X, Y, trf_spec);
shading interp
colormap parula 
hold on
plot(trf_ifr, fasts, 'k-', 'LineWidth',2)
ylabel('Reference Frame rotation')
C = colorbar;
clim([0,30]);
C.Label.String = 'Amplitude';
C.Label.Rotation = 270; 
C.Label.Position(1) = 3;
axis([0,1,-90,90]);
yticks([-90,-60,-30,0,30,60,90]) ;
title('(a) Apparent Fast Trace, \Delta t* = 0s')
xlabel('Frequency [Hz]')
yline(30,'LineWidth',2, 'Color', grey);
yline(70,'--','LineWidth',2, 'Color',grey);

legend('', 'f_{inst}', 'Source Polarisation', 'Fast polarisation', 'Location','southeast')

%apparent slow traces are on second (bottom) row
subplot(2,2,3)

pcolor(X, Y, trs_spec);
shading interp
colormap parula 
hold on
plot(trs_ifr, fasts, 'k-', 'LineWidth',2)
xlabel('Frequency [Hz]')
ylabel('Reference Frame rotation')
C = colorbar;
clim([0,30]);
C.Label.String = 'Amplitude';
C.Label.Rotation = 270; 
C.Label.Position(1) = 3;
title('(c) Apparent Slow Trace, \Delta t* = 0s')
axis([0,1,-90,90]);
yticks([-90,-60,-30,0,30,60,90])
yline(30,'LineWidth',2, 'Color',grey);
yline(70,'--','LineWidth',2, 'Color',grey);
% Now plot dt* = 1 traces

subplot(2,2,2) %Top right panel
pcolor(X, Y, trf_spec_attn);
shading interp
colormap parula 
hold on 
plot(trf_ifr_attn, fasts, 'k-', 'LineWidth',2)
xlabel('Frequency [Hz]')
ylabel('Reference Frame rotation')
C = colorbar;
clim([0,30]);
C.Label.String = 'Amplitude';
C.Label.Rotation = 270;
C.Label.Position(1) = 3;
axis([0,1,-90,90]);
yticks([-90,-60,-30,0,30,60,90]) ;
title('(b) Apparent Fast Trace, \Delta t* = 1s')
yline(30,'LineWidth',2, 'Color',grey);
yline(70,'--','LineWidth',2, 'Color',grey);
subplot(2,2,4)

pcolor(X, Y, trs_spec_attn);
shading interp
colormap parula 
hold on 
plot(trs_ifr_attn, fasts, 'k-', 'LineWidth',2)
xlabel('Frequency [Hz]')
ylabel('Reference Frame rotation')
C = colorbar;
clim([0,30]);
C.Label.String = 'Amplitude';
C.Label.Rotation = 270;
C.Label.Position(1) = 3;
title('(d) Apparent Slow Trace, \Delta t* = 1s')
axis([0,1,-90,90]);
yticks([-90,-60,-30,0,30,60,90])
yline(30,'LineWidth',2 , 'Color',grey);
yline(70,'--','LineWidth',2, 'Color',grey);
exportgraphics(fig4, '../figs/synth_rotation_frequency_cbar.png','Resolution',500)

%% Figure 5 
% 3 panels show:
%   (a) instantaneous frequency of apparent fast/slow
%   (b) difference in instantaneous frequency (difr)
%   (c) second eigenvalue of trace covariance matrix (lambda2)
% For synthetics vs. rotation angle

% Make fresh synthetic
fast_true = 30;
tlag_true = 1.5;
spol = 70;
tstar = 1;
noise = 0.01;
[trN,trE,trZ]=msac_splitwave2(fast_true,tlag_true,'spol',spol,'noise',noise);
fref= 1;% 1/(2*trN.delta);
% apply the tstar value
[trF,trS]=msac_rotate(trN,trE,fast_true) ;
trSA = msac_apply_tstar_operator(trS,fref,tstar) ;
[trN,trE]=msac_rotate(trF,trSA,-1*fast_true) ;
% filter traces
trN = msac_bpfilt(trN, 0.03, 0.5, 2,2);
trE = msac_bpfilt(trE, 0.03, 0.5, 2,2);

n=200;
ref_frame_rot = linspace(-90,90,n);
ifrF = zeros(1,n);
ifrFc = zeros(1,n);
ifrS = zeros(1,n);

lam2 = zeros(1,n);
lam2c = zeros(1,n);

 % Loop over refernece frame rotations
for i=1:length(ref_frame_rot)
   [trF,trS]=msac_rotate(trN,trE,ref_frame_rot(i)) ;
   trS = msac_tshift(trS,-tlag_true,'int') ;
   ifrF(i)=msac_ifa_wwind(trF,trF.a,trF.f) ;
   ifrS(i)=msac_ifa_wwind(trS,trS.a,trS.f) ;
   M = cov(trF.x1,trS.x1) ;
   E = eig(M) ;
   lam2(i) = min(E); 

   trF = msac_apply_tstar_operator(trF,fref,tstar+0.02) ;

   ifrFc(i)=msac_ifa_wwind(trF,trF.a,trF.f) ;
   M = cov(trF.x1,trS.x1) ;
   E = eig(M) ;  
   lam2c(i) = min(E) ;
end
% Calculate dIFR and predict spol = max(dIFR)
dIFr = abs(ifrF-ifrS) ;
dIFrc = abs(ifrFc-ifrS) ;

idx_min = find(lam2==min(lam2));

fig5 = figure("Position",[1000 800 600 1600]) ;
h = tiledlayout(3,1, 'TileSpacing', 'compact', 'Padding', 'compact');
nexttile
plot(ref_frame_rot,ifrF,'b-','LineWidth',2) ;
hold on 
plot(ref_frame_rot,ifrFc,'b--','LineWidth',2) ;
plot(ref_frame_rot,ifrS,'r-','LineWidth',2) ;
text(-87,0.205,'(a)','FontSize',14)


xline(ref_frame_rot(idx_min), 'k--', 'LineWidth',2)
xline(fast_true, 'k-','LineWidth',2) ;
ylabel('f (Hz)', 'FontSize',12)
legend('S1','Corrected S1','S2','Apparent \phi_f','True \phi_f', 'FontSize',14,'Location','southwest')

xlim([ref_frame_rot(1) ref_frame_rot(end)]) ;
set(gca,'FontSize',12)
xticks([-90,-60, -30, 0, 30, 60, 90])
xticklabels([])
nexttile
plot(ref_frame_rot,dIFr,'-', 'Color','#77AC30', 'LineWidth',2) ;
hold on
plot(ref_frame_rot,dIFrc,'g--','Color','#77AC30','LineWidth',2) ;
text(-87,0.085,'(b)','FontSize',14)

xline(ref_frame_rot(idx_min), 'k--', 'LineWidth',2)
xline(fast_true, 'k-','LineWidth',2) ;
ylabel('| \Deltaf | (Hz) ', 'FontSize',12)
aax=axis() ;
xlim([ref_frame_rot(1) ref_frame_rot(end)]) ;
legend('Observed','Corrected','Apparent \phi_f','True \phi_f', 'Location','southwest','FontSize',14)

set(gca,'FontSize',12)
xticks([-90,-60, -30, 0, 30, 60, 90])
xticklabels([])
nexttile
semilogy(ref_frame_rot,lam2,'r-','LineWidth',2)
hold on
semilogy(ref_frame_rot,lam2c,'r--','LineWidth',2)
text(-87,0.007,'(c)','FontSize',14)

xlabel(sprintf('Reference frame rotation (%s)', char(176)), 'FontSize',12)
ylabel('\lambda _2', 'FontSize',12)
idx_min = find(lam2==min(lam2));
xline(ref_frame_rot(idx_min), 'k--', 'LineWidth',2)
xline(fast_true, 'k-','LineWidth',2) ;
legend('Observed','Corrected','Apparent \phi_f','True \phi_f', 'FontSize',14,'Location','southwest')
xticks([-90,-60, -30, 0, 30, 60, 90])
xlim([-90,90])
set(gca,'FontSize',12)
exportgraphics(fig5, '../figs/synth_ifr_v_rot.png','Resolution',500)

%% Bonus figure - Visually show synthetics over a range of rotation angles
% This figure shows a synthetic shear-wave (left) panel roatated through
% reference frame rotation angles 0 - 90 degrees with no dt* 
% The right panel shows the instantaneous frequency of the apparent fast
% and slow shear-waves at each rotation angle.

% Make fresh synthetic
fast_true = 30;
tlag_true = 1.5;
spol = 60;
noise =0.01;

[trN,trE,trZ]=msac_splitwave2(fast_true,tlag_true,'spol',spol,'noise',noise) ;

% uncomment to apply t* 
% tstar=1;
% fref= 1/(2*trN.delta);
% [trF,trS]=msac_rotate(trN,trE,fast_true) ;
% trSA = msac_apply_tstar_operator(trS,fref,tstar) ;
% trFA = msac_apply_tstar_operator(trF,fref,tstar) ;
% [trN, trE]=msac_rotate(trF, trSA, -fast_true);

% Filter traces
trN = msac_bpfilt(trN, 0.03, 0.5, 2,2);
trE = msac_bpfilt(trE, 0.03, 0.5, 2,2);
samps = [0:length(trN.x1)-1]*trN.delta + trN.b;
fig4 = figure('Position',[1000 500 900 1500]);
tb = axtoolbar('default');
tb.Visible = 'off';

theta=0:1:90;
ifr1 = zeros(length(theta),1);
ifr2 = zeros(length(theta),1);

for i=1:length(theta)
    [tr1, tr2] = msac_rotate(trN, trE, theta(i));
    ifr1(i) = msac_ifa_wwind(tr1, tr1.a, tr1.f);
    ifr2(i) = msac_ifa_wwind(tr2, tr2.a, tr2.f);
    subplot(1,2,1)
    if rem(theta(i), 15) == 0 
        plot(samps, 7*(tr1.x1/max(abs(tr1.x1)))+theta(i), "LineWidth", 2, "Color","b");
        hold on
        plot(samps, 7*(tr2.x1/max(abs(tr2.x1)))+theta(i), "LineWidth", 2, "Color", "r");
    end
    [trN, trE] = msac_rotate(tr1, tr2, -theta(i));
end
hold off
set(gca,'FontSize',18)
xlabel('Time [s]')
ylabel('Reference frame rotation [°]')
axis([-10,10, -8, 98])
text(-9.7, 32, 'Fast direction', 'FontSize',18, 'FontWeight','bold')
text(-9,95,'(a)','FontSize',20)

subplot(1,2,2)
plot(ifr1, theta, "LineWidth", 2, "Color","b")
hold on
plot(ifr2, theta, "LineWidth", 2, "Color","r")
set(gca,'FontSize',18)
xlabel('Instantaneous Frequency [Hz]')
axis([0.1,0.25, -8, 98])
grid on
set(gca, 'GridLineWidth', 2)
text(0.105,95,'(b)','FontSize',20)
legend('Apparent fast trace', 'Apparent slow trace')
