// This file is part of the Acts project.
//
// Copyright (C) 2017-2018 CERN for the benefit of the Acts project
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#pragma once

#include "Acts/Digitization/PlanarModuleCluster.hpp"
#include "Acts/Geometry/TrackingGeometry.hpp"
#include "Acts/Utilities/Logger.hpp"
#include "ActsExamples/EventData/GeometryContainers.hpp"
#include "ActsExamples/EventData/SimHit.hpp"
#include "ActsExamples/Framework/DataHandle.hpp"
#include "ActsExamples/Framework/ProcessCode.hpp"
#include "ActsExamples/Framework/WriterT.hpp"

#include <cstdint>
#include <memory>
#include <mutex>
#include <string>
#include <vector>

class TFile;
class TTree;
namespace Acts {
class PlanarModuleCluster;
class TrackingGeometry;
}  // namespace Acts

namespace ActsExamples {
struct AlgorithmContext;

/// @class RootPlanarClusterWriter
///
/// Write out a planar cluster collection into a root file
/// to avoid immense long vectors, each cluster is one entry
/// in the root file for optimised data writing speed
/// The event number is part of the written data.
///
/// A common file can be provided for the writer to attach his TTree,
/// this is done by setting the Config::rootFile pointer to an existing file
///
/// Safe to use from multiple writer threads - uses a std::mutex lock.
class RootPlanarClusterWriter
    : public WriterT<GeometryIdMultimap<Acts::PlanarModuleCluster>> {
 public:
  struct Config {
    /// Which cluster collection to write.
    std::string inputClusters;
    /// Which simulated (truth) hits collection to use.
    std::string inputSimHits;
    std::string filePath = "";          ///< path of the output file
    std::string fileMode = "RECREATE";  ///< file access mode
    std::string treeName = "clusters";  ///< name of the output tree
    /// Tracking geometry required to access global-to-local transforms.
    std::shared_ptr<const Acts::TrackingGeometry> trackingGeometry;
  };

  /// Constructor with
  /// @param config configuration struct
  /// @param level logging level
  RootPlanarClusterWriter(const Config& config, Acts::Logging::Level level);

  /// Virtual destructor
  ~RootPlanarClusterWriter() override;

  /// End-of-run hook
  ProcessCode finalize() override;

  /// Get readonly access to the config parameters
  const Config& config() const { return m_cfg; }

 protected:
  /// This implementation holds the actual writing method
  /// and is called by the WriterT<>::write interface
  ///
  /// @param ctx The Algorithm context with per event information
  /// @param clusters is the data to be written out
  ProcessCode writeT(
      const AlgorithmContext& ctx,
      const GeometryIdMultimap<Acts::PlanarModuleCluster>& clusters) override;

 private:
  Config m_cfg;                    ///< the configuration object
  std::mutex m_writeMutex;         ///< protect multi-threaded writes
  TFile* m_outputFile{nullptr};    ///< the output file
  TTree* m_outputTree{nullptr};    ///< the output tree
  int m_eventNr = 0;               ///< the event number of
  int m_volumeID = 0;              ///< volume identifier
  int m_layerID = 0;               ///< layer identifier
  int m_surfaceID = 0;             ///< surface identifier
  float m_x = 0;                   ///< global x
  float m_y = 0;                   ///< global y
  float m_z = 0;                   ///< global z
  float m_t = 0;                   ///< global t
  float m_lx = 0;                  ///< local lx
  float m_ly = 0;                  ///< local ly
  float m_cov_lx = 0;              ///< local covariance lx
  float m_cov_ly = 0;              ///< local covariance ly
  std::vector<int> m_cell_IDx;     ///< cell ID in lx
  std::vector<int> m_cell_IDy;     ///< cell ID in ly
  std::vector<float> m_cell_lx;    ///< local cell position x
  std::vector<float> m_cell_ly;    ///< local cell position y
  std::vector<float> m_cell_data;  ///< local cell position y

  // (optional) the truth position
  std::vector<float> m_t_gx;          ///< truth position global x
  std::vector<float> m_t_gy;          ///< truth position global y
  std::vector<float> m_t_gz;          ///< truth position global z
  std::vector<float> m_t_gt;          ///< truth time t
  std::vector<float> m_t_lx;          ///< truth position local x
  std::vector<float> m_t_ly;          ///< truth position local y
  std::vector<uint64_t> m_t_barcode;  ///< associated truth particle barcode

  ReadDataHandle<SimHitContainer> m_inputSimHits{this, "InputSimHits"};
};

}  // namespace ActsExamples
