#!/usr/bin/env python3
# coding: utf-8

"""\
python3 generate.py -o ./trs_res ../tests/test-metadata-CWL-validated.yml
"""

import argparse
import datetime
import json
from pathlib import Path
from typing import Any

import yaml


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser()
    parser.add_argument("-o", required=True, help="Location of the output directory")
    parser.add_argument("metadata_locations", nargs="+", help="Location of the Yevis metadata files (only local file)")
    args = parser.parse_args()
    return args


def service_info() -> Any:
    service_info = {
        "id": "<FIX ME>",
        "name": "<FIX ME>",
        "type": {
            "group": "yevis",
            "artifact": "yevis",
            "version": "2.0.1"
        },
        "description": "The GA4GH TRS API generated by Yevis (https://github.com/sapporo-wes/yevis-cli)",
        "organization": {
            "name": "<FIX ME>",
            "url": "<FIX ME>"
        },
        "createdAt": datetime.datetime.now().isoformat(),
        "updatedAt": datetime.datetime.now().isoformat(),
        "version": datetime.datetime.now().strftime("%Y%m%d%H%M%S"),
    }

    return service_info


def tool_classes() -> Any:
    return [{
        "id": "workflow",
        "name": "Workflow",
        "description": "A computational workflow",
    }]


def descriptor(meta: Any) -> Any:
    return {
        "url": [f["url"] for f in meta["workflow"]["files"] if f["type"] == "primary"][0],
    }


def files(meta: Any) -> Any:
    files = []
    for f in meta["workflow"]["files"]:
        if f["type"] == "primary":
            files.append({
                "path": f["url"],
                "file_type": "PRIMARY_DESCRIPTOR"
            })
        elif f["type"] == "secondary":
            files.append({
                "path": f["url"],
                "file_type": "SECONDARY_DESCRIPTOR"
            })

    return files


def tests(meta: Any) -> Any:
    tests = []
    for t in meta["workflow"]["testing"]:
        tests.append({
            "content": json.dumps(t),
        })

    return tests


def version(out_dir: Path, meta: Any) -> Any:
    version_obj = {
        "author": [a["name"] for a in meta["authors"]],
        "name": meta["workflow"]["name"],
        "url": "<FIX ME>",
        "id": meta["version"],
        "descriptor_type": meta["workflow"]["language"]["type"],
        "verified": False,
        "verified_source": [],
    }
    _id = meta["id"]
    _version = meta["version"]
    _des_type = meta["workflow"]["language"]["type"]
    dump(out_dir, f"tools/{_id}/versions/{_version}", version_obj)
    dump(out_dir, f"tools/{_id}/versions/{_version}/containerfile", [])
    dump(out_dir, f"tools/{_id}/versions/{_version}/{_des_type}/descriptor", descriptor(meta))
    dump(out_dir, f"tools/{_id}/versions/{_version}/{_des_type}/files", files(meta))
    dump(out_dir, f"tools/{_id}/versions/{_version}/{_des_type}/tests", tests(meta))

    return version_obj


def versions(out_dir: Path, meta_vec: Any) -> Any:
    versions = [version(out_dir, meta) for meta in meta_vec]
    _id = meta_vec[0]["id"]
    dump(out_dir, f"tools/{_id}/versions", versions)

    return versions


def tool(out_dir: Path, meta_vec: Any) -> Any:
    tool_obj = {
        "url": "<FIX ME>",
        "id": meta_vec[0]["id"],
        "organization": ", ".join([a["name"] for a in meta_vec[0]["authors"]]),
        "name": meta_vec[0]["workflow"]["name"],
        "toolclass": tool_classes()[0],
        "description": meta_vec[0]["workflow"]["readme"],
        "has_checker": True,
        "checker_url": "https://github.com/ddbj/yevis-cli",
        "versions": versions(out_dir, meta_vec),
    }
    _id = meta_vec[0]["id"]
    dump(out_dir, f"tools/{_id}", tool_obj)

    return tool_obj


def tools(out_dir: Path, meta_vec: Any) -> Any:
    tools = {}  # type: ignore
    for meta in meta_vec:
        if meta["id"] in tools:
            tools[meta["id"]].append(meta)
        else:
            tools[meta["id"]] = [meta]

    tools_obj = [tool(out_dir, tool_meta_vec) for tool_meta_vec in tools.values()]
    dump(out_dir, "tools", tools_obj)


def dump(out_dir: Path, path: str, obj: Any) -> None:
    dst = out_dir.joinpath(f"{path}/index.json").resolve()
    dst.parent.mkdir(parents=True, exist_ok=True)
    with dst.open("w") as f:
        json.dump(obj, f, indent=2)


def main() -> None:
    args = parse_args()
    out_dir = Path(args.o).resolve()
    metadata_locs = args.metadata_locations
    meta_vec = []
    for metadata_loc in metadata_locs:
        with open(metadata_loc, "r") as f:
            metadata = yaml.safe_load(f)
        meta_vec.append(metadata)
    dump(out_dir, "service-info", service_info())
    dump(out_dir, "toolClasses", tool_classes())
    tools(out_dir, meta_vec)


if __name__ == "__main__":
    main()
