from collections import OrderedDict
import os, sys, math, yaml
from time import sleep
from yamlordereddictloader import SafeDumper as safe_dumper_ordered
from yamlordereddictloader import SafeLoader as safe_loader_ordered


arch_support_list = ["compute", "sram", "dram"]
arch_attr_list = ["template", "technology", "frequency"]
memory_list = ["sram", "dram"]
compute_list = ["compute"]
onchip_list = ["compute", "sram"]
offchip_list = ["dram"]
memory_action_list = ["rd", "wr"]


def yaml_load(file):
    return yaml.load(open(file), Loader=safe_loader_ordered)


class bcolors:
    HEADER = '\033[95m'
    OKBLUE = '\033[94m'
    OKCYAN = '\033[96m'
    OKGREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'
    UNDERLINE = '\033[4m'
    yellow = "#FFCA68"
    orange = "#FFA668"
    green = "#71DBBA"
    # gray = "#B8B8B8"
    gray = '#9E9E9E'
    blue = "#6D9CF5"
    red = "#FD661F"
    brown = "#9E7446"
    cactus = "#9CC424"


class Variable:
    def __init__(
        self, 
        name: str = "var", 
        byte_per_word: float = 1.0, 
        addr_word_min: int = 0, 
        addr_word_max: int = 100000, 
        action: list = memory_action_list
    ) -> None:

        assert isinstance(name, str), "ERROR: Varibale name requires 'string' type."
        assert "-" not in name, "ERROR: '-' is not allowed in Variable name."
        assert isinstance(byte_per_word, float) or isinstance(byte_per_word, int), "ERROR: Varibale name requires 'float' or 'int' type."
        assert isinstance(addr_word_min, int), "ERROR: Varibale min word adress requires 'int' type."
        assert isinstance(addr_word_max, int), "ERROR: Varibale max word adress requires 'int' type."
        assert isinstance(action, list), "ERROR: Varibale action requires 'set' type."

        self.name = name.lower()
        self.size = float(byte_per_word)
        self.address = (int(addr_word_min), int(addr_word_max))
        self.action = [x.lower() for x in action]
    
    def action_check(self, action: str) -> str:
        assert action is not None, "ERROR: no input action type."
        action = action.lower()
        assert action in self.action, "ERROR: invalid action type."
        return action
        

def prune(input_list: list) -> list:
    l = []

    for e in input_list:
        e = e.strip() # remove the leading and trailing characters, here space
        if e != '' and e != ' ':
            l.append(e)

    return l


def file_integrity_check(file_list: list):
    for file in file_list:
        assert os.path.exists(file), "ERROR: " + file + " does not exist."

def file_clean(file_list: list):
    file_exist = False
    for file in file_list:
        if os.path.exists(file):
            file_exist = True

    if file_exist is True:
        # print(bcolors.WARNING + "Deleting old files in 5 seconds..." + bcolors.ENDC)
        # sleep(5)
        print(bcolors.WARNING + "Deleting old files..." + bcolors.ENDC)
        for file in file_list:
            if os.path.exists(file):
                os.remove(file)


def repeated_key_check(full_dict: OrderedDict, key:str, val: OrderedDict):
    key_index = list(full_dict.keys()).index(key)
    key_list = list(full_dict.keys())[0 : key_index]
    for key in key_list:
        if full_dict[key] == val:
            return True, key
    return False, None


def type_list(arch_cnfg: OrderedDict, type_arch: str):
    out_list = []
    for key in arch_cnfg.keys():
        if isinstance(arch_cnfg[key], OrderedDict):
            if type_arch == arch_cnfg[key]["type"]:
                out_list.append(key)
    return out_list


def create_folder(directory):
    """
    Checks the existence of a directory, if does not exist, create a new one
    :param directory: path to directory under concern
    :return: None
    """
    try:
        if not os.path.exists(directory):
            os.makedirs(directory)
    except OSError:
        print (bcolors.FAIL + "ERROR: Creating directory: " +  directory + bcolors.ENDC)
        sys.exit()

    
def yaml_overwrite(file, content):
    """
    if file exists at filepath, overwite the file, if not, create a new file
    :param filepath: string that specifies the destination file path
    :param content: yaml string that needs to be written to the destination file
    :return: None
    """
    if os.path.exists(file):
        os.remove(file)
    create_folder(os.path.dirname(file))
    out_file = open(file, 'a')
    out_file.write(yaml.dump( content, default_flow_style= False, Dumper=safe_dumper_ordered))

    
def subdir_validate(path: str, subdir_list: list):
    for subdir in subdir_list:
        if not os.path.exists(path + "/" + subdir):
            create_folder(path + "/" + subdir)


# The following oneD_linear_interpolation and oneD_quadratic_interpolation are adapted from accelergy
# ===============================================================
# useful helper functions that are commonly used in estimators
# ===============================================================
def oneD_linear_interpolation(desired_x, known):
    """
    utility function that performs 1D linear interpolation with a known energy value
    :param desired_x: integer value of the desired attribute/argument
    :param known: list of dictionary [{x: <value>, y: <energy>}]
    :return energy value with desired attribute/argument
    """
    # assume E = ax + c where x is a hardware attribute
    ordered_list = []
    if known[1]['x'] < known[0]['x']:
        ordered_list.append(known[1])
        ordered_list.append(known[0])
    else:
        ordered_list = known

    slope = (known[1]['y'] - known[0]['y']) / (known[1]['x'] - known[0]['x'])
    desired_energy = slope * (desired_x - ordered_list[0]['x']) + ordered_list[0]['y']
    return desired_energy


def oneD_quadratic_interpolation(desired_x, known):
    """
    utility function that performs 1D linear interpolation with a known energy value
    :param desired_x: integer value of the desired attribute/argument
    :param known: list of dictionary [{x: <value>, y: <energy>}]
    :return energy value with desired attribute/argument
    """
    # assume E = ax^2 + c where x is a hardware attribute
    ordered_list = []
    if known[1]['x'] < known[0]['x']:
        ordered_list.append(known[1])
        ordered_list.append(known[0])
    else:
        ordered_list = known

    slope = (known[1]['y'] - known[0]['y']) / (known[1]['x']**2 - known[0]['x']**2)
    desired_energy = slope * (desired_x**2 - ordered_list[0]['x']**2) + ordered_list[0]['y']
    return desired_energy


def round2power(input, base=2, mode='floor', min_power=0):
    power = math.log(input, base)
    if mode.lower() == 'floor':
        return max(2**math.floor(power), 2**min_power)
    elif mode.lower() == 'ceil' or mode.lower() == 'ceiling':
        return max(2**math.ceil(power), 2**min_power)
    else:
        raise NotImplementedError
    
