from collections import OrderedDict
from tkinter.tix import Tree
from utils import prune, bcolors, yaml_load
import os, csv
from component import ComputeComponent


linear_interpolation_keywords = ["acc", "add", "reg", "rng", "shifter"]
quadratic_interpolation_keywords = ["multiplier"]
pure_comb_keywords = ['and_bitwise', 'or_bitwise', 'xor_bitwise', 'xnor_bitwise', 'comparator', 'multiplexer', 'shifter', 'multiplier', 'adder']


def power_flat_report(
    report=None
):
    """
    all outputs have the unit of mW
    """
    file = open(report, "r")
    for entry in file:
        elems = entry.strip().split(' ')
        elems = prune(elems)
        if len(elems) >= 6:
            if elems[0] == "Total" and elems[1] == "Dynamic" and elems[2] == "Power" and elems[3] == "=":
                dynamic = float(elems[4])
                unit = str(elems[5])
                if unit == "nW":
                    dynamic /= 1000000.0
                elif unit == "uW":
                    dynamic /= 1000.0
                elif unit == "mW":
                    dynamic *= 1.0
                else:
                    print("Unknown unit for dynamic power:" + unit)

            if elems[0] == "Cell" and elems[1] == "Leakage" and elems[2] == "Power" and elems[3] == "=":
                leakage = float(elems[4])
                unit = str(elems[5])
                if unit == "nW":
                    leakage /= 1000000.0
                elif unit == "uW":
                    leakage /= 1000.0
                elif unit == "mW":
                    leakage *= 1.0
                else:
                    print("Unknown unit for leakage power:" + unit)
    file.close()
    return dynamic, leakage

def power_hier_report(
    report=None
):
    """
    output dynamic and leakage ratio of register
    """
    dynamic = -1
    leakage = -1
    name = ''
    file = open(report, "r")
    for entry in file:
        elems = entry.strip().split(' ')
        elems = prune(elems)
        if len(elems) >= 3:
            if elems[0] == 'Design' and elems[1] == ':':
                name = elems[2]
                continue

            if elems[0] == "U_REG" and 'register' in elems[1]:
                switch = float(elems[2])
                internal = float(elems[3])
                dynamic = (switch + internal) 
                leakage = float(elems[4]) 
                continue
            if name != '' and elems[0] == name and len(elems) == 6:
                switch_t = float(elems[1])
                internal_t = float(elems[2])
                dynamic_t = (switch_t + internal_t) 
                leakage_t = float(elems[3]) 

    assert dynamic != -1 and leakage != -1, f"extracting from {report} failed"
    d_ratio = dynamic / dynamic_t
    l_ratio = leakage / leakage_t
    assert d_ratio > 0 and d_ratio <= 1, f'dynamic ratio for file {report} wrong'
    assert l_ratio > 0 and l_ratio <= 1, f'leakage ratio for file {report} wrong'
    file.close()
    return d_ratio, l_ratio

def power_hier_report_norm(
    report=None
):
    """
    output dynamic and leakage ratio of normalizer
    """
    dynamic = -1
    leakage = -1
    name = ''
    file = open(report, "r")
    for entry in file:
        elems = entry.strip().split(' ')
        elems = prune(elems)
        if len(elems) >= 3:
            if elems[0] == 'Design' and elems[1] == ':':
                name = elems[2]
                continue

            if 'multiplication_normaliser' in elems[1]:
                switch = float(elems[2])
                internal = float(elems[3])
                dynamic = (switch + internal) 
                leakage = float(elems[4]) 
                continue
            if name != '' and elems[0] == name and len(elems) == 6:
                switch_t = float(elems[1])
                internal_t = float(elems[2])
                dynamic_t = (switch_t + internal_t) 
                leakage_t = float(elems[3]) 

    assert dynamic != -1 and leakage != -1, f"extracting from {report} failed"
    d_ratio = dynamic / dynamic_t
    l_ratio = leakage / leakage_t
    assert d_ratio > 0 and d_ratio <= 1, f'dynamic ratio for file {report} wrong'
    assert l_ratio > 0 and l_ratio <= 1, f'leakage ratio for file {report} wrong'
    file.close()
    return d_ratio, l_ratio


def area_flat_report(
    report=None
):
    """
    output has the unit of mm^2
    """
    file = open(report, "r")
    for entry in file:
        elems = entry.strip().split(' ')
        elems = prune(elems)
        if len(elems) >= 3:
            if str(elems[0]) == "Total" and str(elems[1]) == "cell" and str(elems[2]) == "area:":
                area = float(elems[3])

            if str(elems[0]) == "Total" and str(elems[1]) == "area:":
                if str(elems[2]) != "undefined":
                    if area < float(elems[2]):
                        area = float(elems[2])
                    
    area /= 1000000.0
    file.close()
    return area

def area_hier_report(
    report=None
):
    """
    output area ratio of register
    """
    file = open(report, "r")
    name = ''
    for entry in file:
        elems = entry.strip().split(' ')
        elems = prune(elems)
        if len(elems) >= 3:
            if elems[0] == 'Design' and elems[1] == ':':
                name = elems[2]
                continue
            if str(elems[0]) == "U_REG":
                area = float(elems[1])
                continue
            if str(elems[0]) == name and name != '':
                area_t = float(elems[1])
                    
    area_ratio = area / area_t
    assert area_ratio > 0 and area_ratio <= 1, f'area ratio for {report} wrong'
    file.close()
    return area_ratio

def area_hier_report_norm(
    report=None
):
    """
    output area ratio of normalizer
    """
    file = open(report, "r")
    name = ''
    for entry in file:
        elems = entry.strip().split(' ')
        elems = prune(elems)
        if len(elems) >= 3:
            if elems[0] == 'Design' and elems[1] == ':':
                name = elems[2]
                continue
            if 'norm' in str(elems[0]):
                area = float(elems[1])
                continue
            if str(elems[0]) == name and name != '':
                area_t = float(elems[1])
                    
    area_ratio = area / area_t
    assert area_ratio > 0 and area_ratio <= 1, f'area ratio for {report} wrong'
    file.close()
    return area_ratio


def extract(technology, frequency):
    """technology in nm, frequency in MHz"""
    file_dir = os.path.dirname(os.path.abspath(__file__))
    rpt_dir = file_dir + "/rpt/"
    csv_dir = file_dir + "/csv/"
    param_yml = file_dir + '/rtl/param.yaml'
    param_dict = yaml_load(param_yml)
    for module in os.listdir(rpt_dir):
        interpolation = "linear"
        for keyword in linear_interpolation_keywords:
            if keyword in module:
                interpolation = "linear"
        for keyword in quadratic_interpolation_keywords:
            if keyword in module:
                interpolation = "quadratic"
        
        module_rpt_dir = rpt_dir + "/" + module + "/"
        if os.path.isdir(module_rpt_dir):
            # ======extract synthesis rpt=======
            dynamic_d_ratio = 0
            leakage_d_ratio = 0
            area_d_ratio = 0
            # parsing hierarchical design (for purely comb logic)
            if module in pure_comb_keywords:
                # print(f'{module} is pure comb logic')
                module_d_rpt_area   = module_rpt_dir + module + "_hier_area.txt"
                module_d_rpt_power  = module_rpt_dir + module + "_hier_power.txt"
                assert os.path.exists(module_d_rpt_area) and os.path.exists(module_d_rpt_power)
                dynamic_d_ratio, leakage_d_ratio = power_hier_report(module_d_rpt_power)
                area_d_ratio = area_hier_report(module_d_rpt_area)
            # parsing flattend design
            module_rpt_area     = module_rpt_dir + module + "_flat_area.txt"
            module_rpt_power    = module_rpt_dir + module + "_flat_power.txt"
            assert os.path.exists(module_rpt_area) and os.path.exists(module_rpt_power), \
                bcolors.FAIL + "Synthesis area or power report for module <" + module + "> does not exist." + bcolors.ENDC
            dynamic, leakage = power_flat_report(module_rpt_power)
            area = area_flat_report(module_rpt_area)
            dynamic, leakage, area = dynamic*(1-dynamic_d_ratio), leakage*(1-leakage_d_ratio), area*(1-area_d_ratio)

            assert dynamic > 0, f'dynamic power={dynamic} for {module} is negative'
            assert leakage > 0, f'leakage power={leakage} for {module} is negative'
            assert area > 0, f'area={area} for {module} is negative'

            # parsing parameter dict
            header_ = []
            default_ = []
            if module in param_dict.keys() and param_dict[module] != None:
                for item_ in param_dict[module].items():
                    header_.append(item_[0].lower())
                    default_.append(item_[1])

            # generate csv table without parameters
            module_csv = csv_dir + module + ".csv"
            with open(module_csv, 'w') as f:
                csvwriter = csv.writer(f)
                header = ["technology", "frequency", "dynamic", "leakage", "area", "num_instances", "interpolation"] + header_
                csvwriter.writerow(header)
                content = [technology, frequency, dynamic, leakage, area, 1, interpolation] + default_
                csvwriter.writerow(content)
    
    ## =========== generating normalizer_mulbf16 ===========
    # top level
    module_bfmul = 'multiplierbf16'
    module_rpt_dir = rpt_dir + "/" + module_bfmul + "/"
    module_hier_rpt_power  = module_rpt_dir + module_bfmul + "_hier_power.txt"
    module_hier_rpt_area  = module_rpt_dir + module_bfmul + "_hier_area.txt"
    assert os.path.exists(module_hier_rpt_power) and os.path.exists(module_hier_rpt_area)
    dynamic_n_ratio, leakage_n_ratio = power_hier_report_norm(module_hier_rpt_power)
    area_n_ratio = area_hier_report_norm(module_hier_rpt_area)

    module_flat_rpt_power  = module_rpt_dir + module_bfmul + "_flat_power.txt"
    module_flat_rpt_area  = module_rpt_dir + module_bfmul + "_flat_area.txt"
    assert os.path.exists(module_flat_rpt_power) and os.path.exists(module_flat_rpt_area)
    dynamic, leakage = power_flat_report(module_flat_rpt_power)
    area = area_flat_report(module_flat_rpt_area)
    dynamic, leakage, area = dynamic*(dynamic_n_ratio), leakage*(leakage_n_ratio), area*(area_n_ratio)


    # generate csv table without parameters
    module = 'normalizer_mulbf16'
    module_csv = csv_dir + module + ".csv"
    with open(module_csv, 'w') as f:
        csvwriter = csv.writer(f)
        header = ["technology", "frequency", "dynamic", "leakage", "area", "num_instances", "interpolation"]
        csvwriter.writerow(header)
        content = [technology, frequency, dynamic, leakage, area, 1, 'linear']
        csvwriter.writerow(content)
    ## =========== generating normalizer_mulbf16 ===========


if __name__ == "__main__":
    extract(32, 400)

