from collections import OrderedDict
import os, csv
from utils import bcolors, oneD_linear_interpolation, oneD_quadratic_interpolation


compute_skip_list = ["technology", "frequency", "interpolation", "dynamic", "leakage", "area"]
compute_interpolation_list = ["linear", "quadratic"]
compute_attr_list = ["class", "switching"]


class ComputeComponent:
    def __init__(self, attr: OrderedDict) -> None:
        self.attr = attr
        self.type = attr["class"]
        self.skip = compute_skip_list
        self.table = os.getcwd() + "/implement/compute/csv/" + self.type + ".csv"
        assert os.path.exists(self.table), \
            bcolors.FAIL + "Requested type table <" + self.table + "> does not exist." + bcolors.ENDC

    def query(self, technology: float, frequency: float):
        with open(self.table) as csv_file:
            reader = csv.DictReader(csv_file)
            technology_flag = False
            for row in reader:
                interpolation = row["interpolation"]
                assert interpolation in compute_interpolation_list, \
                    bcolors.FAIL + "Unrecognized interpolation <" + interpolation + "> in <" + self.table + ">." + bcolors.ENDC

                if row["technology"] == str(technology):
                    technology_flag = True
                    dynamic_power = float(row["dynamic"]) # in mW
                    leakage_power = float(row["leakage"]) # in mW
                    area = float(row["area"]) # in mm^2

                    # scaling the dynamic power based on input running frequency
                    dynamic_power *= (frequency / float(row["frequency"])) # in mW
                    switching = self.attr["switching"] if "switching" in self.attr.keys() else 1.
                    dynamic_power *= switching # in mW

                    for attr_key in self.attr.keys():
                        assert attr_key in row.keys() or attr_key in compute_attr_list, \
                            bcolors.FAIL + "Unrecognized attribute <" + attr_key + "> in <" + self.table + ">." + bcolors.ENDC

                    for attr_key in row.keys():
                        if attr_key not in self.skip:
                            if attr_key not in self.attr.keys():
                                self.attr[attr_key] = 1
                            if interpolation == "linear":
                                dynamic_power   = oneD_linear_interpolation(    self.attr[attr_key], [{'x': 0, 'y': 0}, {'x': float(row[attr_key]), 'y': dynamic_power}])
                                leakage_power   = oneD_linear_interpolation(    self.attr[attr_key], [{'x': 0, 'y': 0}, {'x': float(row[attr_key]), 'y': leakage_power}])
                                area            = oneD_linear_interpolation(    self.attr[attr_key], [{'x': 0, 'y': 0}, {'x': float(row[attr_key]), 'y': area}])
                            else:
                                dynamic_power   = oneD_quadratic_interpolation( self.attr[attr_key], [{'x': 0, 'y': 0}, {'x': float(row[attr_key]), 'y': dynamic_power}])
                                leakage_power   = oneD_quadratic_interpolation( self.attr[attr_key], [{'x': 0, 'y': 0}, {'x': float(row[attr_key]), 'y': leakage_power}])
                                area            = oneD_quadratic_interpolation( self.attr[attr_key], [{'x': 0, 'y': 0}, {'x': float(row[attr_key]), 'y': area}])

            assert technology_flag is True, \
                bcolors.FAIL + "Unrecognized technology in <" + self.table + ">." + bcolors.ENDC

        return dynamic_power, leakage_power, area

