from architecture import ArchBuilder
from performance import WorkloadPerf
from cost import WorkloadCost
from utils import bcolors, repeated_key_check, yaml_load, subdir_validate
import os, pyfiglet, argparse, glob
from configuration import ConfigChecker


def parse_commandline_args():
    """parse command line inputs"""
    parser = argparse.ArgumentParser(
        description="Archer is a trace-based simulator for architectural design space exploration of domain specific accelerators.")
    parser.add_argument("-name", "--run_name", type=str, default="example_systolic",
                        help = "Name of the run directory, which contains input configurations.")
    return parser.parse_args()


def launch(run_name: str):
    
    ascii_banner = pyfiglet.figlet_format("Archer")
    print(bcolors.HEADER + ascii_banner + bcolors.ENDC)

    # following are required fields
    required_cnfg = ["evaluate performance", "evaluate cost"]
    required_yaml = ["program", "architecture", "workload"]
    required_path = ["cost", "performance"]

    # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
    # Set up configurations
    # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
    path = os.getcwd()
    
    # validate input configurations
    cnfg_path = os.path.join(path, "runs/" + run_name + "/input/")
    assert os.path.exists(cnfg_path), bcolors.FAIL + "Direcroty for run <" + run_name + "> does not exist in " + path + "/runs/" + bcolors.ENDC
    cnfg_dict = {file: os.path.join(cnfg_path, file + ".yaml") for file in required_yaml}
    ConfigChecker(cnfg_dict).check()

    # create output dirs
    outp_path = os.path.join(path, "runs/" + run_name + "/output/")
    subdir_validate(outp_path, required_path)

    # load input configurations
    prog_cnfg = yaml_load(cnfg_dict["program"])["program"]
    eval_perf = prog_cnfg[required_cnfg[0]]
    eval_cost = prog_cnfg[required_cnfg[1]]
    arch_cnfg = yaml_load(cnfg_dict["architecture"])["architecture"]
    work_cnfg = yaml_load(cnfg_dict["workload"])["workload"]

    # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
    # Process problems in workloads sequentially
    # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
    work_perf = WorkloadPerf(arch_cnfg, outp_path, required = (eval_perf is False and eval_cost is True))
    work_cost = WorkloadCost(arch_cnfg, outp_path)
    if eval_perf is True or eval_cost is True:
        for prob_name in work_cnfg.keys():
            prob_cnfg = work_cnfg[prob_name]
            arch_inst = ArchBuilder(arch_cnfg, prob_name, prob_cnfg, outp_path)

            repeated, repeated_prob = repeated_key_check(work_cnfg, prob_name, prob_cnfg)
            if repeated is True:
                print(bcolors.OKCYAN + "Skipping evaluating problem <" + prob_name + "> as a repetition of problem " + repeated_prob + bcolors.ENDC)
                if eval_perf is True:
                    work_perf.record(prob_name, work_perf.disc[repeated_prob])
                if eval_cost is True:
                    work_cost.record(prob_name, work_cost.disc[repeated_prob])
            else:
                print(bcolors.OKBLUE + "Evaluating problem <" + prob_name + ">..." + bcolors.ENDC)
                if eval_perf is True:
                    arch_inst.eval_perf()
                    work_perf.record(prob_name, arch_inst.arch.perf.disc)
                else:
                    if eval_cost is True:
                        print(bcolors.WARNING + "Skipping performance evaluation for problem <" + prob_name + ">..." + bcolors.ENDC)
                        work_perf.disc = yaml_load(arch_inst.arch.perf_copy)
                        arch_inst.arch.perf.disc = work_perf.disc[prob_name]
                if eval_cost is True:
                    arch_inst.eval_cost()
                    work_cost.record(prob_name, arch_inst.arch.cost.disc)
            print()

    elif eval_perf is False and eval_cost is False:
        print("No evaluation is performed.")

    if eval_perf is True:
        work_perf.publish()
        print("Performance evaluation results locate in: ", arch_inst.arch.perf_path)

    if eval_cost is True:
        work_cost.publish(work_perf.disc)
        print("Cost        evaluation results locate in: ", arch_inst.arch.cost_path)

    print()

    fileList = glob.glob(outp_path + '/cost/*.flag')
    for filePath in fileList:
        try:
            os.remove(filePath)
        except:
            print("Error while deleting file : ", filePath)


if __name__ == "__main__":
    args = parse_commandline_args()
    launch(args.run_name)

