from collections import OrderedDict
import os
from .schedule import dataflow
from utils import bcolors, type_list, arch_support_list, arch_attr_list, memory_action_list
from performance import ProblemPerf
from cost import ProblemCost
from implement import CostEstimator


class architecture:
    def __init__(self, arch_cnfg: OrderedDict, prob_name: str, prob_cnfg: OrderedDict, outp_path: str) -> None:
        self.arch_cnfg = arch_cnfg
        self.perf = ProblemPerf(self.arch_cnfg)
        self.cost = ProblemCost(self.arch_cnfg)

        self.perf_path = outp_path + "/performance/" # the absolute address of architecture and implementation performance
        self.perf_copy = outp_path + "/performance/workloadperf.yaml"
        self.cost_path = outp_path + "/cost/" # the absolute address of architecture and implementation cost
        self.cost_copy = outp_path + "/cost/workloadcost.yaml"
        
        self.prob_name = prob_name
        self.prob_cnfg = prob_cnfg

        for cnfg in arch_attr_list:
            assert cnfg in self.arch_cnfg.keys(), bcolors.FAIL + "Archtecture configuration <" + cnfg + "> is missing." + bcolors.ENDC

    def eval_perf(self):
        """evaluate the architecture performance"""
        perf_dict = dataflow(self.arch_cnfg, self.prob_cnfg)
        compute_name_list = type_list(self.arch_cnfg, "compute")
        for c in compute_name_list:
            self.perf.record(c, perf_dict[c])
        
        sram_name_list = type_list(self.arch_cnfg, "sram")
        for s in sram_name_list:
            self.perf.record(s, perf_dict[s])
        
        dram_name_list = type_list(self.arch_cnfg, "dram")
        for d in dram_name_list:
            self.perf.record(d, perf_dict[d])
        
        self.perf.publish()

    def eval_cost(self):
        """evaluate the implementation cost"""
        for type_name in arch_support_list:
            type_name_list = type_list(self.arch_cnfg, type_name)
            for item in type_name_list:
                item_cost = CostEstimator(item, self.arch_cnfg, self.cost_path).estimate()
                self.cost.record(item, item_cost)
        self.cost.publish(self.perf.disc)

