function [x,nr,nrit,r] = solve_LS(A,rhs,tol,maxit,linsolver,x0,P1,P2,Ptype,cycl)
% this function solves a linear system A(x)=rhs 
% by the method linsolver, using maxit steps, tolerance tol, initial guess
% x0, and Ptype-preconditioner P=P1*P2; Here A, P1, P2 can be inserted as 
% numerical matrices or function handles.
% 
% Copyright (C) 2016-2024 Patrick Kürschner
%
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are
% met:
%
% 1. Redistributions of source code must retain the above copyright notice,
%    this list of conditions and the following disclaimer.
%
% 2. Redistributions in binary form must reproduce the above copyright
%    notice, this list of conditions and the following disclaimer in the
%    documentation and/or other materials provided with the distribution.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
% IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
% THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
% PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
% CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
% EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
% PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
% PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
% LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
% NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
% SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


[n,s]=size(rhs); 
x=zeros(n,s);
fl=zeros(s,1);
nr=zeros(s,1);
hist1=zeros(maxit+1,s); 

if strcmp(linsolver,'exact')
    if ~isnumeric(A)
        if isempty(P1)% || ~isfield(opts,'P1'))
            P1=A(speye(n));
            %     warning('cannot do exact solve without matrix');
            %     return
        end
    else, P1=A;
    end
else
    if isnumeric(A), Op=@(x) A*x; else Op=A; end
end

if nargin<7 || (isempty(P1) && isempty(P2))
    Ptype='right';
else
    if nargin<9 || isempty(Ptype)
            Ptype='right';
    end
end
if nargin<10 
    cycl=[];
else
    if (~isempty(cycl) && strcmp(linsolver,'gmres')),     maxit=maxit/cycl; end
end

switch linsolver
    case 'exact'
        x = apply_prec2(rhs,P1,[],0);
%         x = Op\rhs;
        nrit=0;
        r=rhs-P1*x;
        nr=norm(r); %/norm(rhs);
    case 'gmres'
        for k=1:s
            if ~isempty(P1)
                if strcmp(Ptype, 'left')
                    [x(:,k), fl(k),nr(k), nii, rvec]  = ...
                        gmres(Op, rhs(:,k),cycl,tol/s,maxit, @(x,t) apply_prec2(x,P1,P2,0),[],x0);
                elseif strcmp(Ptype, 'right')
                    [x(:,k), fl(k),nr(k), nii, rvec]  = ...
                        gmres(@(x) Op(apply_prec2(x,P1,P2,0)), rhs(:,k),cycl,tol/s,maxit, [],[],x0);
                    x(:,k) =  apply_prec2(x(:,k),P1,P2,0);
                elseif   strcmp(Ptype, 'center')
                    [x(:,k), fl(k),nr(k), nii, rvec]  = ...
                        gmres(@(x) apply_prec2(Op(apply_prec2(x,P2,[],0)),P1,[],0), apply_prec2(rhs(:,k),P1,[],0),cycl, tol/s,maxit, [],[],x0);
                    x(:,k) =  apply_prec2(x(:,k),P2,[],0);
                end
            else
                [x(:,k), fl(k),nr(k), nii, rvec] = ...
                    gmres(Op, rhs(:,k), cycl,tol/s,maxit,[],[],x0);
            end
            nrit(k)=length(rvec);
            hist1(1:length(rvec),k)=rvec;
        end
        r = (rhs-Op(x)); % true res
        %           nrit = max(nrit);
        nrit = ceil(sum(nrit)/s);  %alternative iteration coun
    case 'bicgstab'
        for k=1:s
            if ~isempty(P1)
                if strcmp(Ptype, 'left')
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        bicgstab(Op, rhs(:,k), tol/s,maxit, @(x,t) apply_prec2(x,P1,P2,0),[],x0);
                elseif strcmp(Ptype, 'right')
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        bicgstab(@(x) Op(apply_prec2(x,P1,P2,0)), rhs(:,k), tol/s,maxit, [],[],x0);
                    x(:,k) =  apply_prec2(x(:,k),P1,P2,0);
                elseif   strcmp(Ptype, 'center')
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        bicgstab(@(x) apply_prec2(Op(apply_prec2(x,P2,[],0)),P1,[],0), apply_prec2(rhs(:,k),P1,[],0), tol/s,maxit, [],[],x0);
                    x(:,k) =  apply_prec2(x(:,k),P2,[],0);
                end
            else
                [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                    bicgstab(Op, rhs(:,k), tol/s,maxit,[],[],x0);
            end
            hist1(1:length(rvec),k)=rvec;
        end
%         fl
        r = (rhs-Op(x));
        nrit = ceil(sum(nrit)/s);  %alternative iteration counting.
    case 'bicgstabl'
        for k=1:s
            if ~isempty(P1)
                if strcmp(Ptype, 'left')
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        bicgstabl(Op, rhs(:,k), tol/s,maxit, @(x,t) apply_prec2(x,P1,P2,0),[],x0);
                elseif strcmp(Ptype, 'right')
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        bicgstabl(@(x) Op(apply_prec2(x,P1,P2,0)), rhs(:,k), tol/s,maxit, [],[],x0);
                    x(:,k) =  apply_prec2(x(:,k),P1,P2,0);
                elseif   strcmp(Ptype, 'center')
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        bicgstabl(@(x) apply_prec2(Op(apply_prec2(x,P2,[],0)),P1,[],0), apply_prec2(rhs(:,k),P1,[],0), tol/s,maxit, [],[],x0);
                    x(:,k) =  apply_prec2(x(:,k),P2,[],0);
                end
            else
                [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                    bicgstabl(Op, rhs(:,k), tol/s,maxit,[],[],x0);
            end
            hist1(1:length(rvec),k)=rvec;
        end
%         fl
        r = (rhs-Op(x));
        nrit = ceil(sum(nrit)/s);  %alternative iteration counting.   
    case 'tfqmr'
        for k=1:s
            if ~isempty(P1)
                if strcmp(Ptype, 'left')
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        tfqmr(Op, rhs(:,k), tol/s,maxit, @(x,t) apply_prec2(x,P1,P2,0),[],x0);
                elseif strcmp(Ptype, 'right')
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        tfqmr(@(x) Op(apply_prec2(x,P1,P2,0)), rhs(:,k), tol/s,maxit, [],[],x0);
                    x(:,k) =  apply_prec2(x(:,k),P1,P2,0);
                elseif   strcmp(Ptype, 'center')
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        tfqmr(@(x) apply_prec2(Op(apply_prec2(x,P2,[],0)),P1,[],0), apply_prec2(rhs(:,k),P1,[],0), tol/s,maxit, [],[],x0);
                    x(:,k) =  apply_prec2(x(:,k),P2,[],0);
                end
            else
                [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                    tfqmr(Op, rhs(:,k), tol/s,maxit,[],[],x0);
            end
            hist1(1:length(rvec),k)=rvec;
        end
        r = (rhs-Op(x));
        nrit = ceil(sum(nrit)/s);
    case 'minres'
        if isnumeric(A), Ops=@(x) -A*x; else Ops=@(x) -A(x); end 
        for k=1:s
            if ~isempty(P1)
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        minres(Ops, -rhs(:,k), tol/s,maxit, P1,P1',x0);
            else
                [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                    minres(Ops, -rhs(:,k), tol/s,maxit,[],[],x0);
            end
            hist1(1:length(rvec),k)=rvec;
        end
        r = (rhs-Op(x));
        nrit = ceil(sum(nrit)/s);
     case 'pcg'
        if isnumeric(A), Ops=@(x) -A*x; else Ops=@(x) -A(x); end 
        for k=1:s
            if ~isempty(P1)
                    [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                        pcg(Ops, -rhs(:,k), tol/s,maxit, P1,P1',x0);
            else
                [x(:,k), fl(k),nr(k), nrit(k), rvec]  = ...
                    pcg(Ops, -rhs(:,k), tol/s,maxit,[],[],x0);
            end
            hist1(1:length(rvec),k)=rvec;
        end
        r = (rhs-Op(x));
        nrit = ceil(sum(nrit)/s);      
end