function [sA,sB] = pseudominmax(a,b)
% heuristic selection of shifts for Sylvester-ADI
% as described in
% Benner/Li/Truhar: On the ADI method for Sylvester equations,
%                   J. Comput. Appl. Math., 2009

% INPUT: a,b: vectors of Ritz/Eigenvalues w.r.t matrices A, B 
% OUTPUT: sA,sB: vectors with Sylvester-ADI shifts, s.t. product of
%                 spectralradii  is pseudo-minimized



% how many are Ritz-values given?
J=length(a);
L=length(b);
rval=zeros(J,L);

%evaluate Sylvester-ADI rational function on Ritz values
for iA=1:J,
    for iB=1:L,
        fi1=abs( (a-a(iA))./(a+b(iB)) );
        fi2=abs( (b-b(iB))./(b+a(iA)) );
        rval(iA,iB)=max(fi1)*max(fi2);
    end
end
%find minimum
[val,idx] = min(rval);
[~, j] = min(val);
i = idx(j);

sA=a(i); sB=b(j);

if i~=J % move to the end
    tmp=a(J); a(J)=a(i); a(i)=tmp;
end
if j~=L% move to the end
    tmp=b(L); b(L)=b(j); b(j)=tmp;
end
% ... repeat for the rest
for ishift=2:J
    kwa=J-ishift+1;
    kwb=L-ishift+1;
    rval=zeros(kwa,kwb);
     %evaluate Sylvester-ADI rational function
    for iA=1:kwa
        for iB=1:kwb
            fi1=abs( (a-a(iA))./(a+b(iB)) );
            fi2=abs( (b-b(iB))./(b+a(iA)) );
            rval(iA,iB)=max(fi1)*max(fi2);
        end
    end
    %find minimum again
    [val,idx] = min(rval);
    [~, j] = min(val);
    i = idx(j);

    sA=[sA; a(i)]; sB=[sB; b(j)];
    if i~=kwa  % move to the end
        tmp=a(kwa); a(kwa)=a(i); a(i)=tmp;
    end
    if j~=kwb% move  to the end
        tmp=b(kwb); b(kwb)=b(j); b(j)=tmp;
    end

end


