import { Component, Input, ViewChild } from '@angular/core';
import { RawSimulationLog, StructuredLogLevel } from '../../../../simulation-logs-datamodel';
import {
  CombineArchiveLog,
  SedDocumentLog,
  SedTaskLog,
  SedReportLog,
  SedPlot2DLog,
  SedPlot3DLog,
  SimulationRunLogStatus,
  SimulationRunStatus,
  SimulationStatusToSimulationLogStatus as statusConverter,
} from '@biosimulations/datamodel/common';
import { TocSection, TocSectionsContainerDirective } from '@biosimulations/shared/ui';
import { ScrollService } from '@biosimulations/shared/angular';
import { FormatService } from '@biosimulations/shared/services';
import { Observable } from 'rxjs';

interface StatusCount {
  color: string;
  label: string;
  count: number;
}

type StatusCountsMap = Map<SimulationRunLogStatus, StatusCount>;

@Component({
  selector: 'biosimulations-simulation-log',
  templateUrl: './simulation-log.component.html',
  styleUrls: ['./simulation-log.component.scss'],
})
export class SimulationLogComponent {
  public constructor(private scrollService: ScrollService) {}

  @Input()
  public status!: SimulationRunStatus;

  @Input()
  public rawLog!: RawSimulationLog;

  public StructuredLogLevel = StructuredLogLevel;
  public _structuredLog!: CombineArchiveLog;
  public structuredLogLevel!: StructuredLogLevel;
  public numSedDocuments = 0;
  public numTasks = 0;
  public numReports = 0;
  public numPlots = 0;
  public sedDocumentStatusCounts!: StatusCount[];
  public taskStatusCounts!: StatusCount[];
  public reportStatusCounts!: StatusCount[];
  public plotStatusCounts!: StatusCount[];
  public logHasSedDocuments = false;
  public logHasTasks = false;
  public logHasReports = false;
  public logHasPlots = false;

  public taskLogs: { doc: SedDocumentLog; task: SedTaskLog }[] = [];
  public reportLogs: { doc: SedDocumentLog; report: SedReportLog }[] = [];
  public plotLogs: {
    doc: SedDocumentLog;
    plot: SedPlot2DLog | SedPlot3DLog;
  }[] = [];

  public duration = 'N/A';

  public tocSections!: Observable<TocSection[]>;

  private scrollOffset = 64 + 32 + 32 + 32;

  @Input()
  public set structuredLog(value: CombineArchiveLog | undefined) {
    value
      ? (this._structuredLog = value)
      : (this._structuredLog = {
          status: SimulationRunLogStatus.UNKNOWN,
          exception: null,
          output: null,
          skipReason: null,
          duration: null,
          sedDocuments: [],
        });

    this.processStructuredLog(value);
  }

  public get structuredLog(): CombineArchiveLog | undefined {
    return this._structuredLog;
  }

  private processStructuredLog(log: CombineArchiveLog | undefined): void {
    let level: StructuredLogLevel = StructuredLogLevel.None;
    this.numSedDocuments = 0;
    this.numTasks = 0;
    this.numReports = 0;
    this.numPlots = 0;

    const sedDocumentStatusCountsMap = this.initStatusCountsMap();
    const taskStatusCountsMap = this.initStatusCountsMap();
    const reportStatusCountsMap = this.initStatusCountsMap();
    const plotStatusCountsMap = this.initStatusCountsMap();

    this.logHasSedDocuments = false;
    this.logHasTasks = false;
    this.logHasReports = false;
    this.logHasPlots = false;

    if (log && log.status && log.status !== SimulationRunLogStatus.UNKNOWN) {
      level = Math.max(level, StructuredLogLevel.CombineArchive);

      if (log?.sedDocuments) {
        this.logHasSedDocuments = true;
        this.numSedDocuments = log.sedDocuments.length;
        log.sedDocuments.sort(this.locationComparator);

        for (const docLog of log.sedDocuments) {
          level = Math.max(level, StructuredLogLevel.SedDocument);
          (sedDocumentStatusCountsMap.get(docLog.status) as StatusCount).count++;

          if (docLog?.tasks) {
            this.logHasTasks = true;
            this.numTasks += docLog.tasks.length;
            docLog.tasks.sort(this.idComparator);

            for (const taskLog of docLog.tasks) {
              level = Math.max(level, StructuredLogLevel.SedTaskOutput);
              (taskStatusCountsMap.get(taskLog.status) as StatusCount).count++;
              break;
            }
          }

          if (docLog?.outputs) {
            this.logHasReports = true;
            this.logHasPlots = true;
            docLog.outputs.sort(this.idComparator);

            for (const outputLog of docLog.outputs) {
              level = Math.max(level, StructuredLogLevel.SedTaskOutput);

              if (outputLog && 'dataSets' in outputLog && (outputLog as SedReportLog).dataSets) {
                const reportLog = outputLog as SedReportLog;
                this.numReports++;
                (reportStatusCountsMap.get(reportLog.status) as StatusCount).count++;
                if (reportLog.dataSets) {
                  level = Math.max(level, StructuredLogLevel.SedDataSetCurveSurface);
                }
              }

              if (outputLog && 'curves' in outputLog && (outputLog as SedPlot2DLog).curves) {
                this.numPlots++;
                const plot2dLog = outputLog as SedPlot2DLog;
                (plotStatusCountsMap.get(plot2dLog.status) as StatusCount).count++;
                if (plot2dLog.curves) {
                  level = Math.max(level, StructuredLogLevel.SedDataSetCurveSurface);
                }
              }

              if (outputLog && 'surfaces' in outputLog && (outputLog as SedPlot3DLog).surfaces) {
                this.numPlots++;
                const plot3dLog = outputLog as SedPlot3DLog;
                (plotStatusCountsMap.get(plot3dLog.status) as StatusCount).count++;
                if (plot3dLog.surfaces) {
                  level = Math.max(level, StructuredLogLevel.SedDataSetCurveSurface);
                }
              }
            }
          }
        }
      }
    }

    if (level >= StructuredLogLevel.SedTaskOutput) {
      const tasks: { doc: SedDocumentLog; task: SedTaskLog }[] = [];
      const reports: { doc: SedDocumentLog; report: SedReportLog }[] = [];
      const plots: {
        doc: SedDocumentLog;
        plot: SedPlot2DLog | SedPlot3DLog;
      }[] = [];

      log?.sedDocuments?.forEach((docLog: SedDocumentLog): void => {
        docLog?.tasks?.forEach((taskLog: SedTaskLog): void => {
          tasks.push({ doc: docLog, task: taskLog });
        });

        docLog?.outputs?.forEach((outputLog: SedReportLog | SedPlot2DLog | SedPlot3DLog): void => {
          if ('dataSets' in outputLog) {
            reports.push({ doc: docLog, report: outputLog });
          } else {
            plots.push({ doc: docLog, plot: outputLog });
          }
        });
      });

      this.taskLogs = tasks;
      this.reportLogs = reports;
      this.plotLogs = plots;
    } else {
      this.taskLogs = [];
      this.reportLogs = [];
      this.plotLogs = [];
    }

    this.sedDocumentStatusCounts = this.convertStatusCountsMapToArray(sedDocumentStatusCountsMap);
    this.taskStatusCounts = this.convertStatusCountsMapToArray(taskStatusCountsMap);
    this.reportStatusCounts = this.convertStatusCountsMapToArray(reportStatusCountsMap);
    this.plotStatusCounts = this.convertStatusCountsMapToArray(plotStatusCountsMap);

    this.structuredLogLevel = level;

    this.duration = log === undefined || log.duration === null ? 'N/A' : FormatService.formatDuration(log.duration);
  }

  private initStatusCountsMap(): StatusCountsMap {
    const statusCounts: StatusCountsMap = new Map<SimulationRunLogStatus, StatusCount>();
    statusCounts.set(SimulationRunLogStatus.QUEUED, {
      color: SimulationRunLogStatus.QUEUED,
      label: 'Queued',
      count: 0,
    });
    statusCounts.set(SimulationRunLogStatus.RUNNING, {
      color: SimulationRunLogStatus.RUNNING,
      label: 'Running',
      count: 0,
    });
    statusCounts.set(SimulationRunLogStatus.SUCCEEDED, {
      color: SimulationRunLogStatus.SUCCEEDED,
      label: 'Succeeded',
      count: 0,
    });
    statusCounts.set(SimulationRunLogStatus.SKIPPED, {
      color: SimulationRunLogStatus.SKIPPED,
      label: 'Skipped',
      count: 0,
    });
    statusCounts.set(SimulationRunLogStatus.FAILED, {
      color: SimulationRunLogStatus.FAILED,
      label: 'Failed',
      count: 0,
    });
    statusCounts.set(SimulationRunLogStatus.UNKNOWN, {
      color: SimulationRunLogStatus.SUCCEEDED,
      label: 'Unknown',
      count: 0,
    });
    return statusCounts;
  }

  private convertStatusCountsMapToArray(map: StatusCountsMap): StatusCount[] {
    const order = [
      SimulationRunLogStatus.QUEUED,
      SimulationRunLogStatus.RUNNING,
      SimulationRunLogStatus.SUCCEEDED,
      SimulationRunLogStatus.SKIPPED,
      SimulationRunLogStatus.FAILED,
      SimulationRunLogStatus.UNKNOWN,
    ];

    const array: StatusCount[] = [];

    order.forEach((key: SimulationRunLogStatus): void => {
      const statusCount = map.get(key) as StatusCount;
      if (statusCount.count === 0) {
        statusCount.color = SimulationRunLogStatus.SUCCEEDED;
      } else {
        array.push(statusCount);
      }
    });

    return array;
  }

  public locationComparator(a: any, b: any): number {
    return a.location.localeCompare(b.location, undefined, { numeric: true });
  }

  public idComparator(a: any, b: any): number {
    return a.id.localeCompare(b.id, undefined, { numeric: true });
  }

  public getStatus(): SimulationRunLogStatus {
    return statusConverter(this.status);
  }

  @ViewChild(TocSectionsContainerDirective)
  public set tocSectionsContainer(container: TocSectionsContainerDirective) {
    if (container) {
      setTimeout(() => {
        this.tocSections = container.sections$;
      });
    }
  }

  public scrollToElement(id: string): void {
    const scrollTarget = document.getElementById(id) as HTMLElement;
    this.scrollService.scrollToElement(scrollTarget, this.scrollOffset);
  }

  public downloadRawLog(): void {
    this.downloadLog(this.rawLog, 'log.txt', 'text/plain;charset=UTF-8');
  }

  public downloadStructuredLog(): void {
    this.downloadLog(JSON.stringify(this.structuredLog, null, 2), 'log.json', 'application/json');
  }

  private downloadLog(log: string, fileName: string, mimeType: string): void {
    const blob = new Blob([log], { type: mimeType });
    const a = document.createElement('a');
    a.href = URL.createObjectURL(blob);
    a.download = fileName;
    a.click();
  }
}
