import { permissions, OptionalAuth } from '@biosimulations/auth/nest';
import {
  Project,
  ProjectFilterQueryItem,
  ProjectInput,
  ProjectSummary,
  ProjectSummaryQueryResults,
} from '@biosimulations/datamodel/api';
import {
  Body,
  Controller,
  Delete,
  Get,
  NotFoundException,
  Post,
  Put,
  Query,
  Req,
  HttpCode,
  HttpStatus,
} from '@nestjs/common';
import {
  ApiNoContentResponse,
  ApiTags,
  ApiOperation,
  ApiQuery,
  ApiBody,
  ApiOkResponse,
  ApiCreatedResponse,
  ApiNotFoundResponse,
  ApiPayloadTooLargeResponse,
  ApiBadRequestResponse,
  ApiConflictResponse,
  ApiUnauthorizedResponse,
  ApiForbiddenResponse,
} from '@nestjs/swagger';
import { ProjectId, ProjectIdParam } from './id.decorator';
import { ProjectModel } from './project.model';
import { ProjectsService } from './projects.service';
import { ErrorResponseDocument } from '@biosimulations/datamodel/api';
import { Request } from 'express';
import { AuthToken } from '@biosimulations/auth/common';
import { scopes } from '@biosimulations/auth/common';

@ApiTags('Projects')
@Controller('projects')
export class ProjectsController {
  public constructor(private service: ProjectsService) {}

  @Get()
  @ApiOperation({
    summary: 'Get a list of the published projects',
    description: 'Get a list of information about each published project',
  })
  @ApiOkResponse({
    description: 'List of information about each published project',
    type: [Project],
  })
  public async getProjects(): Promise<Project[]> {
    const projects = await this.service.getProjects();
    return projects.map((proj) => this.returnProject(proj));
  }

  @Get('summary')
  @ApiOperation({
    summary: 'Get a summary of each published project',
    description: 'Get a list of summaries of each published project',
  })
  @ApiOkResponse({
    description: 'List of information about each published project',
    type: [ProjectSummary],
  })
  public async getProjectSummaries(): Promise<ProjectSummary[]> {
    return this.service.getProjectSummaries();
  }

  @Get('summary_filtered')
  @ApiOperation({
    summary: 'Get a summary of each published project according to filter criteria',
    description: 'Get a list of summaries of each published project',
  })
  @ApiQuery({
    name: 'pageSize',
    description: 'maximum number of records to return.',
    required: false,
    type: Number,
  })
  @ApiQuery({
    name: 'pageIndex',
    description: 'page to return when using pagination (using zero index).',
    required: false,
    type: Number,
  })
  @ApiQuery({
    name: 'searchText',
    description: 'search text for query',
    required: false,
    type: String,
  })
  @ApiOkResponse({
    description: 'Query results of each matched public project with summary statistics',
    type: ProjectSummaryQueryResults,
  })
  public async getProjectSummaries_filtered(
    @Query('pageSize')
    pageSize = 25,
    @Query('pageIndex')
    pageIndex = 0,
    @Query('searchText')
    searchText = '',
    @Query('filters')
    filtersJSON = '',
  ): Promise<ProjectSummaryQueryResults> {
    let filters: ProjectFilterQueryItem[] | undefined;
    if (filtersJSON && filtersJSON.length > 0) {
      filters = JSON.parse(filtersJSON) as ProjectFilterQueryItem[];
    } else {
      filters = [];
    }
    if (!searchText || searchText.length < 1) {
      return this.service.getProjectSummariesWithoutSearch(pageSize, pageIndex, filters);
    } else {
      return this.service.searchProjectSummaries(pageSize, pageIndex, searchText, filters);
    }
  }

  @Get(':projectId')
  @ApiOperation({
    summary: 'Get a published project',
    description: 'Get information about a published project',
  })
  @ApiOkResponse({
    description: 'Information about the project',
    type: Project,
  })
  @ApiNotFoundResponse({
    type: ErrorResponseDocument,
    description: 'No project has the requested id',
  })
  @ProjectIdParam()
  public async getProject(@ProjectId('projectId') projectId: string): Promise<Project> {
    const proj = await this.service.getProject(projectId);

    if (proj) {
      return this.returnProject(proj);
    }
    throw new NotFoundException(`Project with id ${projectId} not found.`);
  }

  @ApiOperation({
    summary: 'Get a summary of a project',
    description: 'Returns a summary of the project',
  })
  @ProjectIdParam()
  @ApiOkResponse({
    description: 'A summary of the project was successfully retrieved',
    type: ProjectSummary,
  })
  @ApiNotFoundResponse({
    description: 'No project could be found with requested id',
    type: ErrorResponseDocument,
  })
  @Get(':projectId/summary')
  public async getProjectSummary(@ProjectId('projectId') projectId: string): Promise<ProjectSummary> {
    return this.service.getProjectSummary(projectId);
  }

  @Post('validate')
  @ApiOperation({
    summary: 'Validate a project for publication',
    description:
      'Check whether a project is valid for publication (e.g, succeeded and provides the [minimum required metadata](https://docs.biosimulations.org/concepts/conventions/simulation-project-metadata/). Returns 204 (No Content) for a publishable run, or a 400 (Bad Input) for a run that cannot be published. 400 errors include diagnostic information which describe why the run cannot be published.',
  })
  @ApiQuery({
    name: 'validateSimulationResultsData',
    description:
      'Whether to validate the data (e.g., numerical simulation results) for each SED-ML report and plot for each SED-ML document. Default: false.',
    required: false,
    type: Boolean,
  })
  @ApiQuery({
    name: 'validateIdAvailable',
    description: 'Whether to validate that the id is available. Default: false.',
    required: false,
    type: Boolean,
  })
  @ApiQuery({
    name: 'validateSimulationRunNotPublished',
    description: 'Whether to validate that the simulation run has not yet been published. Default: false.',
    required: false,
    type: Boolean,
  })
  @ApiBody({
    description: 'Information about the project.',
    type: ProjectInput,
  })
  @ApiPayloadTooLargeResponse({
    type: ErrorResponseDocument,
    description: 'The payload is too large. The payload must be less than the server limit.',
  })
  @ApiBadRequestResponse({
    type: ErrorResponseDocument,
    description: 'The project is not valid.',
  })
  @ApiNoContentResponse({
    description: 'The project is valid.',
  })
  @HttpCode(HttpStatus.NO_CONTENT)
  public async validateProject(
    @Body() projectInput: ProjectInput,
    @Query('validateSimulationResultsData')
    validateSimulationResultsData = 'false',
    @Query('validateIdAvailable')
    validateIdAvailable = 'false',
    @Query('validateSimulationRunNotPublished')
    validateSimulationRunNotPublished = 'false',
  ): Promise<void> {
    await this.service.validateProject(
      projectInput,
      ['true', '1'].includes(validateSimulationResultsData.toLowerCase()),
      ['true', '1'].includes(validateIdAvailable.toLowerCase()),
      ['true', '1'].includes(validateSimulationRunNotPublished.toLowerCase()),
    );
    return;
  }

  @Post(':projectId')
  @ApiOperation({
    summary: 'Publish a simulation run',
    description: 'Publish a simulation run',
  })
  @ProjectIdParam()
  @ApiBody({
    description: 'Information about the simulation run to publish',
    type: ProjectInput,
  })
  @ApiPayloadTooLargeResponse({
    type: ErrorResponseDocument,
    description: 'The payload is too large. The payload must be less than the server limit.',
  })
  @ApiBadRequestResponse({
    type: ErrorResponseDocument,
    description:
      "The simulation run is not valid for publication (e.g., run didn't succeed or metadata doesn't meet minimum requirements)",
  })
  @ApiConflictResponse({
    type: ErrorResponseDocument,
    description:
      'The project could not be saved because another project already has the same id or simulation run. The `PUT` method can be used to modify projects.',
  })
  @ApiCreatedResponse({
    description: 'The simulation run was successfully published',
  })
  @OptionalAuth()
  public async createProject(
    @ProjectId('projectId') projectId: string,
    @Body() project: ProjectInput,
    @Req() req: Request,
  ): Promise<void> {
    const user = req?.user as AuthToken;
    await this.service.createProject(project, user);
    return;
  }

  @Put(':projectId')
  @ApiOperation({
    summary: 'Update a published simulation run',
    description: 'Update a published simulation run',
  })
  @ApiNotFoundResponse({
    type: ErrorResponseDocument,
    description: 'No project has the requested id',
  })
  @ApiBody({
    description: 'Updated information about the publication of the simulation run',
    type: ProjectInput,
  })
  @ApiPayloadTooLargeResponse({
    type: ErrorResponseDocument,
    description: 'The payload is too large. The payload must be less than the server limit.',
  })
  @ApiBadRequestResponse({
    type: ErrorResponseDocument,
    description:
      "The simulation run is not valid for publication (e.g., run didn't succeed or metadata doesn't meet minimum requirements)",
  })
  @ApiUnauthorizedResponse({
    type: ErrorResponseDocument,
    description: 'A valid authorization was not provided',
  })
  @ApiForbiddenResponse({
    type: ErrorResponseDocument,
    description: 'The account does not have permission to modify the requested project',
  })
  @permissions()
  @ApiConflictResponse({
    type: ErrorResponseDocument,
    description: 'The project could not be saved because another project already has the same id or simulation run.',
  })
  @ApiOkResponse({
    description: 'The information about the publication of the simulation run was successfully updated',
  })
  @ProjectIdParam()
  public async updateProject(
    @ProjectId('projectId') projectId: string,
    @Body() project: ProjectInput,
    @Req() req: Request,
  ): Promise<void> {
    const user = req?.user as AuthToken;
    await this.service.updateProject(projectId, project, user);
    return;
  }

  // @Delete()
  @ApiOperation({
    summary: 'Delete all published projects',
    description: 'Delete all published projects',
  })
  @ApiNoContentResponse({
    description: 'All published projects were successfully deleted',
  })
  @permissions(scopes.projects.delete.id)
  @HttpCode(HttpStatus.NO_CONTENT)
  public async deleteProjects(): Promise<void> {
    return this.service.deleteProjects();
  }

  @Delete(':projectId')
  @ApiOperation({
    summary: 'Delete a published project',
    description: 'Delete a published project',
  })
  @ApiNotFoundResponse({
    type: ErrorResponseDocument,
    description: 'No project has the requested id',
  })
  @ApiNoContentResponse({
    description: 'The project was successfully deleted',
  })
  @permissions(scopes.projects.delete.id)
  @ProjectIdParam()
  @HttpCode(HttpStatus.NO_CONTENT)
  public async deleteProject(@ProjectId('projectId') projectId: string): Promise<void> {
    const res = await this.service.deleteProject(projectId);
    return res;
  }

  private returnProject(projectModel: ProjectModel): Project {
    const project: Project = {
      id: projectModel.id,
      simulationRun: projectModel.simulationRun,
      created: projectModel.created.toISOString(),
      updated: projectModel.updated.toISOString(),
    };

    return project;
  }
}
