// LDA_macro.ijm
// 
// This Fiji macro was designed to determine the cell viability 
// in Live Dead Assays.
// 
// Author: Dr. Yannic Kerkhoff, ykerkhoff@zedat.fu-berlin.de.
// January 2024
// License: Creative Commons Attribution 4.0 International
// 
// Copyright 2024 Yannic Kerkhoff, 
// Freie Universität Berlin / Zuse Institute Berlin
// 
// The Creative Commons Attribution license allows re-distribution and 
// re-use of a licensed work on the condition that the creator is 
// appropriately credited.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

//Initialize the Analysis Parameters
counter = 0;
prom1 = 20;
prom2 = 20;
C1 = 2;
C2 = 1;
form = newArray("txt", "csv");
filt = newArray("Mean", "Median","Gaussian Blur");
rad = 0.5;
cut = 17;
params = newArray("Live Prominence", "Dead Prominence", "Live Channel", "Dead Channel", "Filter", "Radius");

//Ask for the input folder, output folder, and file suffix
#@ File (label = "Input directory", style = "directory") input
#@ File (label = "Output directory", style = "directory") output
#@ String (label = "File suffix", value = "lif") suffix

//Create Dialog Box
Dialog.create("Analysis Parameters");
Dialog.addNumber("Live Prominence:", prom1);
Dialog.addNumber("Dead Prominence:", prom2);
Dialog.addNumber("Live Channel:", C1);
Dialog.addNumber("Dead Channel:", C2);
Dialog.addChoice("Filter Option:", filt);
Dialog.addNumber("Filter Radius:", rad);
Dialog.addChoice("Export Format", form);
Dialog.addNumber("Cut Prefix:", cut);
Dialog.show();

//Update the Analysis Parameters from the Dialog Box Values
prom1 = Dialog.getNumber();
prom2 = Dialog.getNumber();
C1 = Dialog.getNumber();
C2 = Dialog.getNumber();
filt = Dialog.getChoice();
rad = Dialog.getNumber();
form = Dialog.getChoice();

cut = Dialog.getNumber();

//Array of the Analysis Parameter Values for the Summary Data Table
vals = newArray(""+prom1+"", ""+prom2+"", ""+C1+"", ""+C2+"", filt, ""+rad+"");

//Create the Summary Data Table
Table.create("Summary."+form+"");
Table.setColumn("Image");
Table.setColumn("Live Cells");
Table.setColumn("Dead Cells");
Table.setColumn("Cell Viability [%]");

//Start the Batch Processing
processFolder(input);

//Add the Analysis Parameter Values to the Summary Data Table after Image Analysis
selectWindow("Summary."+form+"");
Table.setColumn("Analysis Parameter", params);
Table.setColumn("Parameter Value", vals);
Table.update();
				
//save the summary data table when the analysis is complete 
selectWindow("Summary."+form+"");
saveAs("Results", "" + output + File.separator + "Summary."+form+"");

// function to scan folders/subfolders/files to find files with correct suffix
function processFolder(input) {
	run("Bio-Formats Macro Extensions");

	list = getFileList(input);
	list = Array.sort(list);

	for (i = 0; i < list.length; i++) {
		if(File.isDirectory(input + File.separator + list[i]))
			{processFolder(input + File.separator + list[i]);
}
		if(endsWith(list[i], "."+suffix))
{
			setBatchMode(true);
			id = input + File.separator + list[i];
			Ext.setId(id);
			Ext.getSeriesCount(seriesCount);
			for (k=0; k<seriesCount; k++) {
				
				run("Bio-Formats Importer", "open=["+id+"] color_mode=Default view=Hyperstack stack_order=XYCZT series_"+(k+1));
				
				//get image name and rename with short form
				title = getTitle();
				name = substring(title, cut);
				rename(name);
				
				//apply a filter to reduce noise
				run(""+filt+"...", "radius="+rad+" stack");
				run("Split Channels");
				
				//find living cells based on find maxima function
				selectWindow("C"+C1+"-"+name+"");
				run("Find Maxima...", "prominence="+prom1+" strict output=Count");
				run("Find Maxima...", "prominence="+prom1+" strict output=[Single Points]");
				run("Maximum...", "radius=0.5");
				
				//find dead cells based on find maxima function
				selectWindow("C"+C2+"-"+name+"");
				run("Find Maxima...", "prominence="+prom2+" strict output=Count");
				run("Find Maxima...", "prominence="+prom2+" strict output=[Single Points]");
				run("Maximum...", "radius=0.5");
				
				//calculate cell viability from live and dead cell counts
				selectWindow("Results");
				cell_array = Table.getColumn("Count");
				live_frac = (cell_array[0]/(cell_array[0] + cell_array[1])*100);
				
				//prepare the validation image
				run("Merge Channels...", "c1=[C"+C2+"-"+name+"] c2=[C"+C1+"-"+name+"] c5=[C"+C1+"-"+name+" Maxima] c6=[C"+C2+"-"+name+" Maxima] create keep ignore");
				run("RGB Color");
				run("Enhance Contrast", "saturated=0.1");
				saveAs("tif", "" + output + File.separator + name + " " + cell_array[0] + " Live; " + cell_array[1] + " Dead; " + live_frac + "% Viability.tif");
					
				//prepare the summary data table	
				selectWindow("Summary."+form+"");
				Table.set("Image", counter, name);
				Table.set("Live Cells", counter, cell_array[0]);
				Table.set("Dead Cells", counter, cell_array[1]);
				Table.set("Cell Viability [%]", counter, live_frac);
				Table.update();
				
				//close the data table of the find maxima functions
				if (isOpen("Results")) {selectWindow("Results");run("Close");}
				close("*");
				
				counter++;
			}
		}
	}	
}
