#!/usr/bin/env python

import os, sys, numpy as np, nibabel as nb, xlsxwriter
import argparse
np.set_printoptions(threshold=sys.maxsize) #Allows full matrix to be visable instead of truncated 
path = os.getcwd()


"""
Calculate Connectivity Probability 

Usage: calculateConnectivity.py -r <text file containing full paths to ROIs> -m <full path to network matrix generated by FSL probtrackx> -s <subject> -f <output file name>

Functions to calculate symmetric connectivity for node pairs weighted by target or seed voxels. 
The probability of tracts from Node A reaching Node B is represented by the following equation: 
CPa = K1 / N1
Where K1 represents the connections from Node A successfully reaching Node B and N1 represents the number of seed voxels x 5000.

The inverse relationship of tracts for Node B reaching Node A is presented by the following eqation: 
CPb = K2 / N2
Where K2 represents the connections from Node B successfully reaching Node A and N2 represents the number of target voxels x 5000. 

Thus, the total connectivity probability is therefore represented by the following binomial: 
CPab = (K1 + K2) / (N1 + N2).

This code will calculate all three values and output an excel spreadsheet with all final calculations.


Author: Holly J. Freeman -- February 7, 2023 
"""


#------------------- Functions -------------------

def parse_args ():

    parser = argparse.ArgumentParser(description="Calculates connectivity probability between node pairs and generates excel spreadsheet for a single subject.")
    
    #------------------- Required Arguments -------------------
    parser.add_argument('-r', '--roi_file', help="Directory containing all DICOMs to sort by series description.", type=str, required=True)
    parser.add_argument('-m', '--matrix', help="", type=str, required=True)
    parser.add_argument('-s', '--subject', help="Subject ID", type=str, required=True)
    parser.add_argument('-f', '--file_name', help="Output file name.", type=str, required=True)
    #$# this currently returns None, should return the parser obj or the value retruned by parser.parse_args()
    
def voxel_sizes (roi_input):
    """ 
    Returns the non-zero voxel size 
    (total count and volume mm^3) for the given input roi. By default only returns non-zero voxel count
    """

    #Load data
    print(":::: VOXEL SIZES ::::")
    nii = nb.load(roi_input)
    img = nii.get_fdata()
    
    #Grab dimensions 
    voxel_dims = (nii.header["pixdim"])[1:4]
    
    #Compute volume
    nonzero_voxel_count = np.count_nonzero(img)
    voxel_volume = np.prod(voxel_dims)
    nonzero_voxel_volume = nonzero_voxel_count * voxel_volume

    print("NON ZERO VOXELS = {}".format(nonzero_voxel_count))
    print("VOLUME NON ZERO VOXELS = {} mm^3".format(nonzero_voxel_volume))

    return nonzero_voxel_count

def write_excel_head (headers_list,row,column):
    """
    Writes the header row into the spreadsheet.
    """
    #Write header row 
    for header in headers_list:
        worksheet.write(row, column, header)
        column += 1

def write_excel_oneway (roi_row, roi_column, label, vox, streamlines, weighted_vox):
    """
    Function to write individual ROI statistics into designated space in the spreadsheet. 
    """
    #Populate list to look through writer
    items_to_write=(label, vox, streamlines, weighted_vox)

    for x in items_to_write: 
        worksheet.write(roi_row, roi_column, x)
        roi_column += 1

def oneway_calc (matrix, index, roi_list, roi, voxels, row, column, header_list): #$# HEADERS should be passed as an argument
    """
    Main function to calculate all connectivity statistics for each relationship. 
    Contains calls to additional functions to perform targeted calculations. 
    """

    single_seed_array = matrix[index]
    single_seed_array_list = single_seed_array.tolist() #$# Unused
    TARGET2SEED_ARRAY = [column[index] for column in matrix]

    for target in roi_list: 

        ##SEED TO TARGET CALCULATIONS
        print(":::: SEED-2-TARGET RELATIONSHIP ::::")
        print("CURRENT SEED ROW::: ", row)
        direction = "CPa"
        target_basename = os.path.basename(target)
        seed_basename = os.path.basename(roi)
        label = (seed_basename + ".2." + target_basename)
        target_position = roi_list.index(target)

        RAW_SEED_VOXELS = (voxels)
        SEED2TARGET_STREAMLINES = single_seed_array[target_position]
        WEIGHTED_SEED_VOXELS = (RAW_SEED_VOXELS * 5000)
        print("RELATIONSHIP =", label)
        print("SEED STREAMLINES=", SEED2TARGET_STREAMLINES)
        print("SEED VOXELS =", WEIGHTED_SEED_VOXELS)
        print()

        write_excel_oneway(row, column, label, RAW_SEED_VOXELS, SEED2TARGET_STREAMLINES, WEIGHTED_SEED_VOXELS)
        oneway_connectivity_probability(header_list, row, SEED2TARGET_STREAMLINES, WEIGHTED_SEED_VOXELS, direction)

        ##TARGET TO SEED CALCULATIONS
        print(":::: TARGET-2-SEED RELATIONSHIP ::::")
        column = 4
        direction = "CPb"
        TARGET2SEED_STREAMLINES = TARGET2SEED_ARRAY[target_position]
        inverse_label = (target_basename + ".2." + seed_basename)
        targ_nonzero_vox_count=voxel_sizes(target)
        weighted_seed_voxels=(targ_nonzero_vox_count * 5000)
        print("RELATIONSHIP =", inverse_label)
        print("WEIGHTED SEED VOXELS =", weighted_seed_voxels)
        print("TARGET STREAMLINES=", TARGET2SEED_STREAMLINES)
        print()

        write_excel_oneway(row, column, inverse_label, targ_nonzero_vox_count, TARGET2SEED_STREAMLINES, weighted_seed_voxels)
        oneway_connectivity_probability(header_list, row, TARGET2SEED_STREAMLINES, weighted_seed_voxels, direction)
        round_connectivity_probability(HEADERS, row, SEED2TARGET_STREAMLINES, TARGET2SEED_STREAMLINES, WEIGHTED_SEED_VOXELS, weighted_seed_voxels)

        column = 0
        row += 1
        print()

def oneway_connectivity_probability (headers_list,row,streamlines,voxel_weight,direction): #$# worksheet should be passed as an argument
    """
    Calculates the one-way statistical probability of connectivity between identified seed and target.
    """
    if direction == 'CPa': 
        column = headers_list.index('CPa')
        SEED2TARG_CP = (streamlines / voxel_weight)
        if np.isnan(float(SEED2TARG_CP)) == False: 
            worksheet.write(row,column, SEED2TARG_CP)
        else: 
            worksheet.write(row,column, '0')
    
    if direction == 'CPb':
        column = headers_list.index('CPb')
        TARG2SEED_CP = (streamlines / voxel_weight)
        if np.isnan(float(TARG2SEED_CP)) == False: 
            worksheet.write(row,column, TARG2SEED_CP)
        else: 
            worksheet.write(row,column, '0')


def round_connectivity_probability (headers_list, row, seed_streamline, target_streamline, seed_vox_value, targ_vox_value): #$# worksheet should be passed as an argument
    """
    Calculates the overall, weighted statistical probability of connectivity between seed to target, and target to seed.
    """
    column = headers_list.index('CPab')
    TOTAL_STREAMLINE = (seed_streamline + target_streamline)
    TOTAL_WEIGHTED_VOX = (seed_vox_value + targ_vox_value)
    TOTAL_CP = (TOTAL_STREAMLINE / TOTAL_WEIGHTED_VOX)
    print(TOTAL_CP)

    if np.isnan(float(TOTAL_CP)) == False: 
        worksheet.write(row,column,TOTAL_CP)
    else: 
        worksheet.write(row,column, '0')

    column = headers_list.index('CPab Percent')
    PERCENTAGE = (TOTAL_CP * 100) 
    print('Percentage Connectivity Probability: ', PERCENTAGE)

    if np.isnan(float(PERCENTAGE)) == False: 
        worksheet.write(row,column,PERCENTAGE)
    else:
        worksheet.write(row,column, '0')


def main ():
    #------------------- Argument Handling -------------------
    args = parse_args()
    ROI_FILE = args.roi_file
    NETWORK_MATRIX = args.matrix
    SUBJECT = args.subject
    NAME_FILE = args.file_name

    #------------------- Excel Configuration -------------------
    #workbook = xlsxwriter.Workbook(NAME_FILE, {'nan_inf_to_errors': True})
    workbook = xlsxwriter.Workbook(NAME_FILE)
    worksheet = workbook.add_worksheet(SUBJECT)
    HEADERS = ['SEED2TARGET','NON-ZERO VOX (SEED)','K1','N1','TARGET2SEED','NON-ZERO VOX (TARGET)','K2','N2','CPa','CPb','CPab', 'CPab Percent']

    ##Load network matrix 
    PROBTRACKX_MATRIX=np.loadtxt(NETWORK_MATRIX)

    #Read in network file and populate list
    with open(ROI_FILE) as f: 
        ROI_LIST = [line.rstrip() for line in f]

    NUM_ROI = (len(ROI_LIST))
    print("Number of ROIS = " + str(NUM_ROI))


    #Set header row
    worksheet_row = 0
    worksheet_column = 0
    write_excel_head(HEADERS,worksheet_row,worksheet_column)

    #Progress row past header 
    worksheet_row = 1
    worksheet_column = 0

    for roi_index, roi in enumerate(ROI_LIST, start=0):
        print(":::: INFORMATION ::::")
        print("ROI = ", roi)
        print("INDEX", roi_index)
        print()

        #Grab voxel sizes 
        nonzero_voxel_count = voxel_sizes(roi)
        print(nonzero_voxel_count)

        #Calculate Probability Statistics 
        oneway_calc(PROBTRACKX_MATRIX, roi_index, ROI_LIST, roi, nonzero_voxel_count, worksheet_row, worksheet_column, HEADERS)

        worksheet_row += NUM_ROI
        worksheet_column += 0
        print("-----------------------------------------------------")
        print()

    workbook.close()

if __name__ == '__main__':
    main()